//+------------------------------------------------------------------+
//| Cache-friendly OHLC data structure                               |
//+------------------------------------------------------------------+
class COHLCData
{
private:
   int m_capacity;
   int m_size;
   
   // Structure of arrays (SoA) layout for better cache locality
   datetime m_time[];
   double m_open[];
   double m_high[];
   double m_low[];
   double m_close[];
   long m_volume[];
   
public:
   // Constructor
   COHLCData(int capacity = 1000)
   {
      m_capacity = capacity;
      m_size = 0;
      
      // Allocate arrays
      ArrayResize(m_time, m_capacity);
      ArrayResize(m_open, m_capacity);
      ArrayResize(m_high, m_capacity);
      ArrayResize(m_low, m_capacity);
      ArrayResize(m_close, m_capacity);
      ArrayResize(m_volume, m_capacity);
   }
   
   // Add a new bar
   bool Add(datetime time, double open, double high, double low, double close, long volume)
   {
      if(m_size >= m_capacity)
         return false;
         
      m_time[m_size] = time;
      m_open[m_size] = open;
      m_high[m_size] = high;
      m_low[m_size] = low;
      m_close[m_size] = close;
      m_volume[m_size] = volume;
      
      m_size++;
      return true;
   }
   
   // Get bar data by index
   bool GetBar(int index, datetime &time, double &open, double &high, double &low, double &close, long &volume)
   {
      if(index < 0 || index >= m_size)
         return false;
         
      time = m_time[index];
      open = m_open[index];
      high = m_high[index];
      low = m_low[index];
      close = m_close[index];
      volume = m_volume[index];
      
      return true;
   }
   
   // Get size
   int Size()
   {
      return m_size;
   }
   
   // Process all high values (example of cache-friendly operation)
   double CalculateAverageHigh()
   {
      if(m_size == 0)
         return 0.0;
         
      double sum = 0.0;
      for(int i = 0; i < m_size; i++)
      {
         sum += m_high[i];
      }
      
      return sum / m_size;
   }
   
   // Process all low values (example of cache-friendly operation)
   double CalculateAverageLow()
   {
      if(m_size == 0)
         return 0.0;
         
      double sum = 0.0;
      for(int i = 0; i < m_size; i++)
      {
         sum += m_low[i];
      }
      
      return sum / m_size;
   }
};