//+------------------------------------------------------------------+
//| Pre-allocation example for high-frequency trading                |
//+------------------------------------------------------------------+
class CHFTSystem
{
private:
   // Pre-allocated arrays for price data
   double m_bidPrices[];
   double m_askPrices[];
   datetime m_times[];
   
   // Pre-allocated arrays for calculations
   double m_tempArray1[];
   double m_tempArray2[];
   double m_tempArray3[];
   
   // Pre-allocated string buffers
   string m_logMessages[];
   int m_logIndex;
   
   int m_capacity;
   int m_dataIndex;
   
public:
   // Constructor
   CHFTSystem(int capacity = 10000)
   {
      m_capacity = capacity;
      m_dataIndex = 0;
      m_logIndex = 0;
      
      // Pre-allocate all arrays
      ArrayResize(m_bidPrices, m_capacity);
      ArrayResize(m_askPrices, m_capacity);
      ArrayResize(m_times, m_capacity);
      
      ArrayResize(m_tempArray1, m_capacity);
      ArrayResize(m_tempArray2, m_capacity);
      ArrayResize(m_tempArray3, m_capacity);
      
      ArrayResize(m_logMessages, 1000);  // Pre-allocate log buffer
      
      Print("HFT system initialized with capacity for ", m_capacity, " data points");
   }
   
   // Add price data
   void AddPriceData(double bid, double ask)
   {
      // Use modulo to create a circular buffer effect
      int index = m_dataIndex % m_capacity;
      
      m_bidPrices[index] = bid;
      m_askPrices[index] = ask;
      m_times[index] = TimeCurrent();
      
      m_dataIndex++;
   }
   
   // Log a message without allocating new strings
   void Log(string message)
   {
      int index = m_logIndex % 1000;
      m_logMessages[index] = message;
      m_logIndex++;
   }
   
   // Perform calculations using pre-allocated arrays
   double CalculateSpread(int lookback = 100)
   {
      int available = MathMin(m_dataIndex, m_capacity);
      int count = MathMin(lookback, available);
      
      if(count <= 0)
         return 0.0;
         
      double sumSpread = 0.0;
      
      for(int i = 0; i < count; i++)
      {
         int index = (m_dataIndex - 1 - i + m_capacity) % m_capacity;
         sumSpread += m_askPrices[index] - m_bidPrices[index];
      }
      
      return sumSpread / count;
   }
};