//+------------------------------------------------------------------+
//|                                                     logistic.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<np.mqh>
namespace logistic
{
//+------------------------------------------------------------------+
//| matrix to vector                                                 |
//+------------------------------------------------------------------+
void matrixToArray(matrix &f,double &fv[])
  {
   if(fv.Size() != uint(f.Cols()*f.Rows()))
     {
      ArrayResize(fv,int(f.Cols()*f.Rows()));
     }

   int ii = 0;
   for(ulong i = 0; i<f.Rows(); i++)
      for(ulong j =0; j<f.Cols(); j++)
         fv[ii++] = f[i][j];
  }
//+------------------------------------------------------------------+
//| vector to matrix                                                 |
//+------------------------------------------------------------------+
void arrayToMatrix(double  &fv[],matrix &f, ulong rows,ulong cols)
  {
   if(f.Cols()*f.Rows() != cols*rows)
      f.Resize(rows,cols);

   int ii = 0;
   for(ulong i = 0; i<f.Rows(); i++)
      for(ulong j =0; j<f.Cols(); j++)
         f[i][j] = fv[ii++];
  }
//+------------------------------------------------------------------+
//|  function and gradient calculation object                        |
//+------------------------------------------------------------------+
class CFg:public CNDimensional_Grad
  {
private:
   matrix            m_preds;
   vector            m_targs;
   ulong             m_nclasses,m_samples,m_features;
   double            loss_gradient(matrix &coef,double &gradients[]);
   void              weight_intercept_raw(matrix &coef,matrix &x, matrix &wghts,vector &intcept,matrix &rpreds);
   void              weight_intercept(matrix &coef,matrix &wghts,vector &intcept);
   double            l2_penalty(matrix &wghts,double strenth);
   void              sum_exp_minus_max(ulong index,matrix &rp,vector &pr);
   void              closs_grad_halfbinmial(double y_true,double raw, double &inout_1,double &intout_2);
public:
   //--- constructor, destructor
                     CFg(matrix &predictors,vector &targets, ulong num_classes)
     {
      m_preds = predictors;
      vector classes = np::unique(targets);
      np::sort(classes);
      vector checkclasses = np::arange(classes.Size());
      if(checkclasses.Compare(classes,1.e-1))
        {
         double classv[];
         np::vecAsArray(classes,classv);
         m_targs = targets;
         for(ulong i = 0; i<targets.Size(); i++)
            m_targs[i] = double(ArrayBsearch(classv,m_targs[i]));
        }
      else
         m_targs = targets;

      m_nclasses = num_classes;
      m_features = m_preds.Cols();
      m_samples = m_preds.Rows();
     }
                    ~CFg(void) {}

   virtual void      Grad(double &x[],double &func,double &grad[],CObject &obj);
   virtual void      Grad(CRowDouble &x,double &func,CRowDouble &grad,CObject &obj);
  };
//+------------------------------------------------------------------+
//| this function is not used                                        |
//+------------------------------------------------------------------+
void CFg::Grad(double &x[],double &func,double &grad[],CObject &obj)
  {
   matrix coefficients;
   arrayToMatrix(x,coefficients,m_nclasses>2?m_nclasses:m_nclasses-1,m_features+1);
   func=loss_gradient(coefficients,grad);
   return;
  }
//+------------------------------------------------------------------+
//| get function value and gradients                                 |
//+------------------------------------------------------------------+
void CFg::Grad(CRowDouble &x,double &func,CRowDouble &grad,CObject &obj)
  {
   double xarray[],garray[];
   x.ToArray(xarray);
   Grad(xarray,func,garray,obj);
   grad = garray;
   return;
  }
//+------------------------------------------------------------------+
//| loss gradient                                                    |
//+------------------------------------------------------------------+
double CFg::loss_gradient(matrix &coef,double &gradients[])
  {
   matrix weights;
   vector intercept;
   vector losses;
   matrix gradpointwise;
   matrix rawpredictions;
   matrix gradient;
   double loss;
   double l2reg;

//calculate weights intercept and raw predictions
   weight_intercept_raw(coef,m_preds,weights,intercept,rawpredictions);
   gradpointwise = matrix::Zeros(m_samples,rawpredictions.Cols());
   losses = vector::Zeros(m_samples);
   double sw_sum = double(m_samples);
//loss gradient calculations
   if(m_nclasses>2)
     {
      double max_value, sum_exps;
      vector p(rawpredictions.Cols()+2);
      //---
      for(ulong i = 0; i< m_samples; i++)
        {
         sum_exp_minus_max(i,rawpredictions,p);
         max_value = p[rawpredictions.Cols()];
         sum_exps = p[rawpredictions.Cols()+1];
         losses[i] = log(sum_exps) + max_value;
         //---
         for(ulong k  = 0; k<rawpredictions.Cols(); k++)
           {
            if(ulong(m_targs[i]) == k)
               losses[i] -= rawpredictions[i][k];
            p[k]/=sum_exps;
            gradpointwise[i][k] = p[k] - double(int(ulong(m_targs[i])==k));
           }
        }
     }
   else
     {
      for(ulong i = 0; i<m_samples; i++)
        {
         closs_grad_halfbinmial(m_targs[i],rawpredictions[i][0],losses[i],gradpointwise[i][0]);
        }
     }
//---
   loss = losses.Sum()/sw_sum;
   l2reg = 1.0 / (1.0 * sw_sum);
   loss += l2_penalty(weights,l2reg);
   gradpointwise/=sw_sum;
//---
   if(m_nclasses>2)
     {
      gradient = gradpointwise.Transpose().MatMul(m_preds) + l2reg*weights;
      gradient.Resize(gradient.Rows(),gradient.Cols()+1);
      vector gpsum = gradpointwise.Sum(0);
      gradient.Col(gpsum,m_features);
     }
   else
     {
      gradient = m_preds.Transpose().MatMul(gradpointwise) + l2reg*weights.Transpose();
      gradient.Resize(gradient.Rows()+1,gradient.Cols());
      vector gpsum = gradpointwise.Sum(0);
      gradient.Row(gpsum,m_features);
     }
//---
   matrixToArray(gradient,gradients);
//---
   return loss;
  }
//+------------------------------------------------------------------+
//|  weight intercept raw preds                                      |
//+------------------------------------------------------------------+
void CFg::weight_intercept_raw(matrix &coef,matrix &x,matrix &wghts,vector &intcept,matrix &rpreds)
  {
   weight_intercept(coef,wghts,intcept);
   matrix intceptmat = np::vectorAsRowMatrix(intcept,x.Rows());
   rpreds = (x.MatMul(wghts.Transpose()))+intceptmat;
  }
//+------------------------------------------------------------------+
//| weight intercept                                                 |
//+------------------------------------------------------------------+
void CFg::weight_intercept(matrix &coef,matrix &wghts,vector &intcept)
  {
   intcept = coef.Col(m_features);
   wghts = np::sliceMatrixCols(coef,0,m_features);
  }
//+------------------------------------------------------------------+
//|  sum exp minus max                                               |
//+------------------------------------------------------------------+
void CFg::sum_exp_minus_max(ulong index,matrix &rp,vector &pr)
  {
   double mv = rp[index][0];
   double s_exps = 0.0;

   for(ulong k = 1; k<rp.Cols(); k++)
     {
      if(mv<rp[index][k])
         mv=rp[index][k];
     }

   for(ulong k = 0; k<rp.Cols(); k++)
     {
      pr[k] = exp(rp[index][k] - mv);
      s_exps += pr[k];
     }

   pr[rp.Cols()] = mv;
   pr[rp.Cols()+1] = s_exps;
  }
//+------------------------------------------------------------------+
//|  l2 penalty                                                      |
//+------------------------------------------------------------------+
double CFg::l2_penalty(matrix &wghts,double strenth)
  {
   double norm2_v;
   if(wghts.Rows()==1)
     {
      matrix nmat = (wghts).MatMul(wghts.Transpose());
      norm2_v = nmat[0][0];
     }
   else
      norm2_v = wghts.Norm(MATRIX_NORM_FROBENIUS);

   return 0.5*strenth*norm2_v;
  }
//+------------------------------------------------------------------+
//|   closs_grad_half_binomial                                       |
//+------------------------------------------------------------------+
void CFg::closs_grad_halfbinmial(double y_true,double raw, double &inout_1,double &inout_2)
  {
   if(raw <= -37.0)
     {
      inout_2 = exp(raw);
      inout_1 = inout_2 - y_true * raw;
      inout_2 -= y_true;
     }
   else
      if(raw <= -2.0)
        {
         inout_2 = exp(raw);
         inout_1 = log1p(inout_2) - y_true * raw;
         inout_2 = ((1.0 - y_true) * inout_2 - y_true) / (1.0 + inout_2);
        }
      else
         if(raw <= 18.0)
           {
            inout_2 = exp(-raw);
            // log1p(exp(x)) = log(1 + exp(x)) = x + log1p(exp(-x))
            inout_1 = log1p(inout_2) + (1.0 - y_true) * raw;
            inout_2 = ((1.0 - y_true) - y_true * inout_2) / (1.0 + inout_2);
           }
         else
           {
            inout_2 = exp(-raw);
            inout_1 = inout_2 + (1.0 - y_true) * raw;
            inout_2 = ((1.0 - y_true) - y_true * inout_2) / (1.0 + inout_2);
           }
  }
//+------------------------------------------------------------------+
//| logistic regression implementation                               |
//+------------------------------------------------------------------+
class Clogit
  {
public:
                     Clogit(void);
                    ~Clogit(void);
   bool              fit(matrix &predictors, vector &targets);
   double            predict(vector &preds);
   vector            predict(matrix &preds);
   vector            proba(vector &preds);
   matrix            probas(matrix &preds);
   double            coeffAt(ulong index);
private:
   ulong             m_nsamples;
   ulong             m_nfeatures;
   bool              m_trained;
   matrix            m_train_preds;
   vector            m_train_targs;
   matrix            m_coefs;
   vector            m_bias;
   vector            m_classes;
   double            m_xin[];
   CFg               *m_gradfunc;
   CObject           m_dummy;
   vector            predictProba(double &in);
  };
//+------------------------------------------------------------------+
//| constructor                                                      |
//+------------------------------------------------------------------+
Clogit::Clogit(void)
  {
  }
//+------------------------------------------------------------------+
//| destructor                                                       |
//+------------------------------------------------------------------+
Clogit::~Clogit(void)
  {
   if(CheckPointer(m_gradfunc) == POINTER_DYNAMIC)
      delete m_gradfunc;
  }
//+------------------------------------------------------------------+
//| fit a model to a dataset                                         |
//+------------------------------------------------------------------+
bool Clogit::fit(matrix &predictors, vector &targets)
  {
   m_trained = false;
   m_classes = np::unique(targets);
   np::sort(m_classes);

   if(predictors.Rows()!=targets.Size() || m_classes.Size()<2)
     {
      Print(__FUNCTION__," ",__LINE__," invalid inputs ");
      return m_trained;
     }

   m_train_preds = predictors;
   m_train_targs = targets;

   m_nfeatures = m_train_preds.Cols();
   m_nsamples = m_train_preds.Rows();

   m_coefs = matrix::Zeros(m_classes.Size()>2?m_classes.Size():m_classes.Size()-1,m_nfeatures+1);

   matrixToArray(m_coefs,m_xin);

   m_gradfunc = new CFg(m_train_preds,m_train_targs,m_classes.Size());
//---
   CMinLBFGSStateShell state;
   CMinLBFGSReportShell rep;
   CNDimensional_Rep   frep;
//---
   CAlglib::MinLBFGSCreate(m_xin.Size(),m_xin.Size()>=5?5:m_xin.Size(),m_xin,state);
//---
   CAlglib::MinLBFGSOptimize(state,m_gradfunc,frep,true,m_dummy);
//---
   CAlglib::MinLBFGSResults(state,m_xin,rep);
//---
   if(rep.GetTerminationType()>0)
     {
      m_trained = true;
      arrayToMatrix(m_xin,m_coefs,m_classes.Size()>2?m_classes.Size():m_classes.Size()-1,m_nfeatures+1);
      m_bias = m_coefs.Col(m_nfeatures);
      m_coefs = np::sliceMatrixCols(m_coefs,0,m_nfeatures);
     }
   else
      Print(__FUNCTION__," ", __LINE__, " failed to train the model ", rep.GetTerminationType());

   delete m_gradfunc;


   return m_trained;
  }

//+------------------------------------------------------------------+
//| get probability for single sample                                |
//+------------------------------------------------------------------+
vector Clogit::proba(vector &preds)
  {
   vector predicted;

   if(!m_trained)
     {
      Print(__FUNCTION__," ", __LINE__," no trained model available ");
      predicted.Fill(EMPTY_VALUE);
      return predicted;
     }

   predicted = ((preds.MatMul(m_coefs.Transpose())));
   predicted += m_bias;

   if(predicted.Size()>1)
     {
      if(!predicted.Activation(predicted,AF_SOFTMAX))
        {
         Print(__FUNCTION__," ", __LINE__," errror ", GetLastError());
         predicted.Fill(EMPTY_VALUE);
         return predicted;
        }
     }
   else
     {
      predicted = predictProba(predicted[0]);
     }

   return predicted;
  }
//+------------------------------------------------------------------+
//|  get probability for binary classification                       |
//+------------------------------------------------------------------+
vector Clogit::predictProba(double &in)
  {
   vector out(2);

   double n = 1.0/(1.0+exp(-1.0*in));

   out[0] = 1.0 - n;
   out[1] = n;

   return out;
  }
//+------------------------------------------------------------------+
//| get probabilities for multiple samples                           |
//+------------------------------------------------------------------+
matrix Clogit::probas(matrix &preds)
  {
   matrix output(preds.Rows(),m_classes.Size());
   vector rowin,rowout;
   for(ulong i = 0; i<preds.Rows(); i++)
     {
      rowin = preds.Row(i);
      rowout = proba(rowin);
      if(rowout.Max() == EMPTY_VALUE || !output.Row(rowout,i))
        {
         Print(__LINE__," probas error ", GetLastError());
         output.Fill(EMPTY_VALUE);
         break;
        }
     }

   return output;
  }
//+------------------------------------------------------------------+
//| get probability for single sample                                |
//+------------------------------------------------------------------+
double Clogit::predict(vector &preds)
  {
   vector prob = proba(preds);
   if(prob.Max() == EMPTY_VALUE)
     {
      Print(__LINE__," predict error ");
      return EMPTY_VALUE;
     }

   return m_classes[prob.ArgMax()];
  }
//+------------------------------------------------------------------+
//| get probability for single sample                                |
//+------------------------------------------------------------------+
vector Clogit::predict(matrix &preds)
  {
   matrix probs = probas(preds);
   vector out(preds.Rows());
   if(probs.Max() == EMPTY_VALUE)
     {
      Print(__LINE__," predict error ");
      out.Fill(EMPTY_VALUE);
      return out;
     }

   vector rowmax = probs.ArgMax(1);
   for(ulong i = 0; i<rowmax.Size(); i++)
      out[i] = m_classes[ulong(rowmax[i])];

   return out;
  }
//+------------------------------------------------------------------+
//|  get model coefficient at specific index                         |
//+------------------------------------------------------------------+
double Clogit::coeffAt(ulong index)
  {
   if(index<(m_coefs.Rows()))
     {
      return (m_coefs.Row(index)).Sum();
     }
   else
     {
      return 0.0;
     }
  }
}
//+------------------------------------------------------------------+
