//+------------------------------------------------------------------+
//|                                                  Gating_Demo.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#property script_show_inputs
#include<imodel.mqh>
#include<multilayerperceptron.mqh>
#include<gatedreg.mqh>
//--- input parameters
input int      NumSamples=100;
input int      NumModels=3;
input int      NumReplications=1000;
input double   DifficultyFactor=0.5;

//+------------------------------------------------------------------+
//|  normal(rngstate)                                                |
//+------------------------------------------------------------------+
double normal(CHighQualityRandStateShell &state)
  {
   return CAlglib::HQRndNormal(state);
  }
//+------------------------------------------------------------------+
//|Multilayer perceptron                                             |
//+------------------------------------------------------------------+
class CMLPC:public IModel  
{
private:
   CMlp              *m_mlfn;
   double             m_learningrate;
   double             m_tolerance;
   double             m_alfa;
   double             m_beyta;
   uint               m_epochs;
   ulong              m_in,m_out;
   ulong              m_hl1,m_hl2;

public:
                     CMLPC(ulong ins, ulong outs,ulong numhl1,ulong numhl2);
                    ~CMLPC(void);
   void              setParams(double alpha_, double beta_,double learning_rate, double tolerance, uint num_epochs);
   bool              train(matrix &predictors,matrix&targets);
   double            forecast(vector &predictors);
  };
//+------------------------------------------------------------------+
//| constructor                                                      |
//+------------------------------------------------------------------+
CMLPC::CMLPC(ulong ins, ulong outs,ulong numhl1,ulong numhl2)
  {
   m_in = ins;
   m_out = outs;
   m_alfa = 0.3;
   m_beyta = 0.01;
   m_learningrate=0.001;
   m_tolerance=1.e-8;
   m_epochs= 1000;
   m_hl1 = numhl1;
   m_hl2 = numhl2;
   m_mlfn = new CMlp();
  }
//+------------------------------------------------------------------+
//| destructor                                                       |
//+------------------------------------------------------------------+
CMLPC::~CMLPC(void)
  {
   if(CheckPointer(m_mlfn) == POINTER_DYNAMIC)
      delete m_mlfn;
  }
//+------------------------------------------------------------------+
//| set other hyperparameters of the i_model                           |
//+------------------------------------------------------------------+
void CMLPC::setParams(double alpha_, double beta_,double learning_rate, double tolerance, uint num_epochs)
  {
   m_alfa = alpha_;
   m_beyta = beta_;
   m_learningrate=learning_rate;
   m_tolerance=tolerance;
   m_epochs= num_epochs;
  }
//+------------------------------------------------------------------+
//| fit a i_model to the data                                          |
//+------------------------------------------------------------------+
bool CMLPC::train(matrix &predictors,matrix &targets)
  {
   if(m_in != predictors.Cols() || m_out != targets.Cols())
     {
      Print(__FUNCTION__, " failed training due to invalid training data");
      return false;
     }

   return m_mlfn.fit(predictors,targets,m_alfa,m_beyta,m_hl1,m_hl2,m_epochs,m_learningrate,m_tolerance);
  }
//+------------------------------------------------------------------+
//| make a prediction with the trained i_model                         |
//+------------------------------------------------------------------+
double CMLPC::forecast(vector &predictors)
  {
   vector predr= m_mlfn.predict(predictors);
   return predr[0]; 
  }
//+------------------------------------------------------------------+
//| clean up dynamic array pointers                                  |
//+------------------------------------------------------------------+
void cleanup(IModel* &array[])
  {
   for(uint i = 0; i<array.Size(); i++)
      if(CheckPointer(array[i])==POINTER_DYNAMIC)
         delete array[i];
  }
//+------------------------------------------------------------------+
//| global variables                                                 |
//+------------------------------------------------------------------+
int nreplications, nsamps,models_count, divisor, nreps_done, max_gates;
matrix xdata, xbad_data, xtainted_data, testmatrix[];
double errr, errrdiff, errr1, errr2 , std_factor;
vector computed_errr_raw;
double computed_errr_afterfact ;
double computed_errr_original ;
double computed_errr_random ;
double computed_errr_ratio ;
//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
   CHighQualityRandStateShell rngstate;
   CHighQualityRand::HQRndRandomize(rngstate.GetInnerObj());
//---
   nsamps = NumSamples ;
   models_count = NumModels ;
   nreplications = NumReplications ;
   std_factor = DifficultyFactor ;

   if((nsamps <= 3) || (models_count <= 0)  || (nreplications <= 0) || (std_factor < 0.0))
     {
      Alert(" Invalid inputs ");
      return;
     }

   divisor = 1 ;
   
   IModel* models[];
   
    max_gates = (models_count > 2) ? models_count : 2 ;
    vector gates(max_gates);
    vector contenders(models_count);
    matrix all_gates(nsamps,max_gates);
    matrix all_contenders(nsamps,models_count);
    vector all_trueval(nsamps);
    
    ArrayResize(models,models_count);
    ArrayResize(testmatrix,10);
    
    for(int i = 0; i<models_count; i++)
       models[i] = new CMLPC(2,1,2,0);
    
    xdata = matrix::Zeros(nsamps,3);
    xbad_data = matrix::Zeros(nsamps,3);
    xtainted_data = matrix::Zeros(nsamps,3);
   

   for(uint i = 0; i<testmatrix.Size(); i++)
      testmatrix[i] = matrix::Zeros(nsamps,3);
      
   computed_errr_raw = vector::Zeros(models_count);
   computed_errr_afterfact = 0.0 ;
   computed_errr_original = 0.0 ;
   computed_errr_random = 0.0 ;
   computed_errr_ratio = 0.0 ;
   matrix preds,targs;
   vector testrow,testpreds;
   
   CGatedReg gate;
   
   for(int i_rep=0 ; i_rep<nreplications ; i_rep++)
     {
      nreps_done = i_rep + 1 ;
      //---
      xdata.Random(0.0,1.0);
      for(int i=0; i<nsamps ; i++)
         xdata[i][2] = xdata[i][0] - xdata[i][1] + std_factor*normal(rngstate) ;


      if(models_count >= 4)
        {
         xbad_data = xdata;
         for(int i = 0; i<nsamps; i++)
            xbad_data[i][2] = normal(rngstate) ;
        }

      if(models_count >= 5)
        {
         xtainted_data = xdata;
         for(int i = 0; i<nsamps; i++)
            xtainted_data[i][2] *= 1000.0 ;  
        }

      for(int i=0 ; i<10 ; i++)         // Build a testmatrix dataset
        {
         testmatrix[i].Random(0.0,1.0);
         for(int j=0,z=0; j<nsamps; j++)
           testmatrix[i][j][2] = testmatrix[i][j][0] - testmatrix[i][j][1] + std_factor*normal(rngstate);
        }
        
       
      for(int i_model=0 ; i_model<models_count ; i_model++)
        {
         
         if(i_model == 3)
           {
            targs = np::sliceMatrixCols(xbad_data,2);
            preds = np::sliceMatrixCols(xbad_data,0,2);
           }
         else
            if(i_model == 4)
              {
               targs = np::sliceMatrixCols(xtainted_data,2);
               preds = np::sliceMatrixCols(xtainted_data,0,2);
              }
            else
              {
               targs = np::sliceMatrixCols(xdata,2);
               preds = np::sliceMatrixCols(xdata,0,2);
              }

         if(!models[i_model].train(preds,targs))
           {
            Print(" failed to train i_model at shift ", i_model);
            cleanup(models);
            return;
           }
         errr = 0.0 ;
         for(int i=0 ; i<10 ; i++)
           {
            
            for(int j=0; j<nsamps; j++)
              {
               testrow = testmatrix[i].Row(j);
               testpreds = np::sliceVector(testrow,0,2);
               errrdiff = models[i_model].forecast(testpreds) - testrow[2];
               errr += errrdiff * errrdiff;
              }
           }
         computed_errr_raw[i_model] += errr / (10 * nsamps) ;
        }  
       
       for(int i = 0; i<nsamps; i++)
        {
         all_trueval[i] = xdata[i][2];
         testrow = xdata.Row(i);
         testpreds = np::sliceVector(testrow,0,2);
         for(int imodel = 0; imodel<models_count; imodel++)
            all_contenders[i][imodel] = models[imodel].forecast(testpreds);
        }
       
       if(!gate.fit(all_contenders,all_contenders,all_trueval))
        {
         Print(" failed to train after effect oracle ");
         cleanup(models);
         return;
        } 
        
       errr = 0.0 ;
       for(int i=0 ; i<10 ; i++)
        {
         for(int z=0;z<nsamps;z++)
           {
            testrow = testmatrix[i].Row(z);
            testpreds = np::sliceVector(testrow,0,2); 
            for(int imodel = 0; imodel<models_count;imodel++)
               contenders[imodel] = models[imodel].forecast(testpreds);
            gates = contenders;
            errrdiff = gate.predict(gates,contenders) - testrow[2];
            errr += errrdiff*errrdiff;
           }
        }
       computed_errr_afterfact += errr/double(10*nsamps);
       
       all_gates = np::sliceMatrixCols(xdata,0,2);
       
       if(!gate.fit(all_gates,all_contenders,all_trueval))
        {
         Print(" failed to train OG vars as gates oracle ");
         cleanup(models);
         return;
        }
        
       errr = 0.0;
       for(int i=0 ; i<10 ; i++)
        {
         for(int z=0;z<nsamps;z++)
           {
            testrow = testmatrix[i].Row(z);
            testpreds = np::sliceVector(testrow,0,2); 
            for(int imodel = 0; imodel<models_count;imodel++)
               contenders[imodel] = models[imodel].forecast(testpreds);
            gates = testpreds;
            errrdiff = gate.predict(gates,contenders) - testrow[2];
            errr += errrdiff*errrdiff;
           }
        }
        computed_errr_original += errr/double(10*nsamps);
        
        
        all_gates = matrix::Zeros(nsamps,1);
        gates = vector::Zeros(1);
        all_gates.Random(0.0,1.0);
         
        if(!gate.fit(all_gates,all_contenders,all_trueval))
         {
          Print(" failed to train random vals as gates oracle ");
          cleanup(models);
          return;
         }
         
       errr = 0.0;
       for(int i=0 ; i<10 ; i++)
        {
         for(int z=0;z<nsamps;z++)
           {
            testrow = testmatrix[i].Row(z);
            testpreds = np::sliceVector(testrow,0,2); 
            for(int imodel = 0; imodel<models_count;imodel++)
               contenders[imodel] = models[imodel].forecast(testpreds);
            gates[0] = normal(rngstate);
            errrdiff = gate.predict(gates,contenders) - testrow[2];
            errr += errrdiff*errrdiff;
           }
        }
        computed_errr_random += errr/double(10*nsamps);
        
        for (int i=0 ; i<nsamps ; i++) 
        {
         errr1 = all_contenders[i][0] - all_trueval[i] ;
         errr2 = all_contenders[i][1] - all_trueval[i] ;
         all_gates[i][0] = log ( (fabs(errr1)+1.e-60) / (fabs(errr2)+1.e-60) ) ;
        }
        
        errr = 0.0;
       for(int i=0 ; i<10 ; i++)
        {
         for(int z=0;z<nsamps;z++)
           {
            testrow = testmatrix[i].Row(z);
            testpreds = np::sliceVector(testrow,0,2); 
            for(int imodel = 0; imodel<models_count;imodel++)
               contenders[imodel] = models[imodel].forecast(testpreds);
            errr1 = contenders[0] - testrow[2] ;
            errr2 = contenders[1] - testrow[2] ;
            gates[0] = log ( (fabs(errr1)+1.e-60) / (fabs(errr2)+1.e-60) ) ;
            errrdiff = gate.predict(gates,contenders) - testrow[2];
            errr += errrdiff*errrdiff;
           }
        }
        computed_errr_ratio += errr/double(10*nsamps);
      }
      
     errr = 0.0 ;
      PrintFormat ( "%5d    replications completed.", nreps_done) ;
      for (int imodel=0 ; imodel<models_count ; imodel++) {
         errr += computed_errr_raw[imodel] / nreps_done ;
         }
      PrintFormat ( " ++++++ Mean raw error = %8.8lf", errr / models_count ) ;
      PrintFormat ( "Component error = %8.8lf", computed_errr_afterfact / nreps_done );
      PrintFormat ( "Original error = %8.8lf", computed_errr_original / nreps_done );
      PrintFormat ( "Random error = %8.8lf", computed_errr_random / nreps_done ) ;
      PrintFormat ( "Ratio error = %8.8lf", computed_errr_ratio / nreps_done );
      cleanup(models);
      
      
   
  }
//+------------------------------------------------------------------+
