//+------------------------------------------------------------------+
//|                                                       oracle.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<qsort.mqh>
#include<imodel.mqh>
//+------------------------------------------------------------------+
//| Tabulated combination of component model outputs                 |
//+------------------------------------------------------------------+
class COracle
{
 private:
  ulong m_ncases;
  ulong m_nin;
  ulong m_ncats;
  uint m_nmodels;
  matrix m_thresh;
  ulong m_tally[];
 public:
  COracle(void);
  ~COracle(void);
  bool fit(matrix &predictors, vector &targets, IModel* &models[],ulong ncats);
  double predict(vector &inputs,IModel* &models[]);

};
//+------------------------------------------------------------------+
//| constructor                                                      |
//+------------------------------------------------------------------+
COracle::COracle(void)
{

}
//+------------------------------------------------------------------+
//| destructor                                                       |
//+------------------------------------------------------------------+
COracle::~COracle(void)
{

}
//+------------------------------------------------------------------+
//| fit an oracle                                                    |
//+------------------------------------------------------------------+
bool COracle::fit(matrix &predictors,vector &targets,IModel *&models[],ulong ncats)
{
 if(predictors.Rows()!=targets.Size())
  {
   Print(__FUNCTION__," ",__LINE__," invalid inputs ");
   return false;
  }
  
 m_ncases = predictors.Rows();
 m_nin = predictors.Cols();
 m_nmodels = models.Size();
 
 m_ncats = ncats;
 ulong nthresh = m_ncats - 1;
 ulong nbins = 1;
 
 nbins = (ulong)pow(m_ncats,m_nmodels);
 
 m_thresh = matrix::Zeros(m_nmodels,nthresh);
 
 ZeroMemory(m_tally);
 
 if(ArrayResize(m_tally,int(nbins))<0)
  {
   Print(__FUNCTION__," ", __LINE__," error ", GetLastError());
   return false;
  }
  
 matrix outputs(m_ncases,m_nmodels);
 matrix bins(nbins,m_nmodels);
 
 bins.Fill(0.0);
 
 vector inrow;
 for(ulong icase=0;icase<m_ncases; icase++)
  {
   inrow=predictors.Row(icase);
   for(uint imodel =0; imodel<m_nmodels; imodel++)
       outputs[icase][imodel] = models[imodel].forecast(inrow);
  }
 
 double frac;
 for(uint imodel =0; imodel<m_nmodels; imodel++)
     {
      inrow = outputs.Col(imodel);
      qsortd(0,long(m_ncases-1),inrow);
      for(ulong i = 0; i<nthresh; i++)
       { 
        frac = double(i+1)/double(ncats);
        m_thresh[imodel][i] = inrow[ulong(frac*(m_ncases-1))];
       }
     }
  vector outrow;  
  ulong ibin,index, klow, khigh, ibest, k;
  k = 0;
  double diff,best;
  for(ulong icase=0;icase<m_ncases; icase++)
  {
   inrow = predictors.Row(icase);
   outrow = outputs.Row(icase);
   ibin = 0;
   index = 1;
   for(uint imodel =0; imodel<m_nmodels; imodel++)
     {
      if(outrow[imodel] <= m_thresh[imodel][0])
          k = 0;
      else
        if(outrow[imodel] > m_thresh[imodel][nthresh-1])
            k = nthresh;
        else
         {
          klow = 0;
          khigh = nthresh-1;
          while(true)
           {
            k = (klow+khigh)/2;
            if(k == klow)
             {
              k = khigh;
              break;
             }
            if(outrow[imodel]<=m_thresh[imodel][k])
              khigh = k;
            else
              klow = k;
           }
         }
        ibin += k * index;
        index *= ncats;
     }
   best = DBL_MAX;
   for(uint imodel =0; imodel<m_nmodels; imodel++)
     {
      diff = fabs(outrow[imodel] - targets[icase]);
      if(diff<best)
       {
        best = diff;
        k = imodel;
       }
     }
   bins[ibin][k]+=1.0;  
  }
  
  for(ibin =0; ibin<nbins; ibin++)
   {
    k = 0;
    ibest = 0;
    for(uint imodel = 0; imodel<m_nmodels; imodel++)
      {
       if(bins[ibin][imodel] > double(ibest))
        {
         ibest = ulong(bins[ibin][imodel]);
         k = ulong(imodel);
        }
      }
    m_tally[ibin] = k;
   }
   
  return true;
}
//+------------------------------------------------------------------+
//| make a prediction                                                |
//+------------------------------------------------------------------+
double COracle::predict(vector &inputs,IModel* &models[])
{
 ulong k, klow, khigh, ibin, index, nthresh ;
 nthresh = m_ncats -1;
 k = 0;
 ibin = 0;
 index = 1;
 vector otk(m_nmodels);
 
 for(uint imodel = 0; imodel<m_nmodels; imodel++)
   {
    otk[imodel] = models[imodel].forecast(inputs);
    
    if(otk[imodel]<m_thresh[imodel][0])
       k = 0;
    else
      if(otk[imodel]>m_thresh[imodel][nthresh-1])
         k = nthresh - 1;
      else
        {
         klow=0;
         khigh = nthresh -1 ;
         while(true)
          {
           k = (klow + khigh) / 2;
           if(k == klow)
            {
             k = khigh;
             break;
            }
           if(otk[imodel] <= m_thresh[imodel][k])
             khigh = k;
           else
             klow = k;
          }
        }
      ibin += k*index;
      index *= m_ncats; 
   }
   
  return otk[ulong(m_tally[ibin])];
}
//+------------------------------------------------------------------+
