//+------------------------------------------------------------------+
//|                                                      NewsTrading |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                            https://www.mql5.com/en/users/kaaiblo |
//+------------------------------------------------------------------+
#include <Trade\Trade.mqh>
#include <Trade\OrderInfo.mqh>
#include <Trade\SymbolInfo.mqh>
#include "RiskManagement.mqh"
#include "TimeManagement.mqh"
#include "Sessions.mqh"

//+------------------------------------------------------------------+
//|TradeManagement class                                             |
//+------------------------------------------------------------------+
class CTradeManagement:CRiskManagement
  {
private:
   CTrade            Trade;//Trade class object
   CSymbolProperties CSymbol;//SymbolProperties class object
   CTimeManagement   CTime;//TimeManagement class object
   CSessions         CTS;//Sessions class object
   bool              TradeResult;//boolean to store trade result
   double            mySL;//double variable to store Stoploss
   double            myTP;//double variable to store Takeprofit
   uint              myDeviation;//store price deviation for stop orders
   double            myOpenPrice;//store open price for stop orders
   //--- Will retrieve if there are any open trades
   bool              OpenTrade(ENUM_POSITION_TYPE Type,ulong Magic,string COMMENT=NULL);
   //--- Will retrieve if there are any deals
   bool              OpenedDeal(ENUM_DEAL_TYPE Type,ulong Magic,string COMMENT=NULL);
   //--- Will retrieve if there are any open orders
   bool              OpenOrder(ENUM_ORDER_TYPE Type,ulong Magic,string COMMENT=NULL);
   //--  Check if trade is valid
   bool              Valid_Trade(ENUM_POSITION_TYPE Type,double Price,double SL,double TP);
   //-- Check if stop order is valid
   bool              Valid_Order(ENUM_ORDER_TYPE Type,double Price,double SL,double TP);
   //--- Will attempt to open buy trade
   bool              Buy(double SL,double TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt to open sell trade
   bool              Sell(double SL,double TP,ulong Magic,string COMMENT=NULL);

   //--- class to set and retrieve an order's properties
   class OrderSettings
     {
   private:
      struct TradeProperties
        {
         //store open-price,take-profit,stop-loss for stop orders
         double         Open,Take,Stop;
        } myTradeProp;
   public:
                     OrderSettings() {}
      //--- Set order properties
      void           Set(double myOpen,double myTake,double myStop)
        {
         //--- Set open-price
         myTradeProp.Open=myOpen;
         //--- Set take-profit
         myTradeProp.Take=myTake;
         //--- Set stop-loss
         myTradeProp.Stop=myStop;
        }
      TradeProperties Get()
        {
         //--- retrieve order properties
         return myTradeProp;
        }
     };

   //--- Declare variables for different order types
   OrderSettings     myBuyStop,mySellStop,myBuyTrade,mySellTrade;

   //--- Will set buy-stop order properties
   void              SetBuyStop(int SL,int TP);
   //--- Will set buy position properties
   void              SetBuyTrade(int SL,int TP,double OP);
   //--- Will set sell-stop order properties
   void              SetSellStop(int SL,int TP);
   //--- Will set sell position properties
   void              SetSellTrade(int SL,int TP,double OP);

public:
   //--- Class constructor
                     CTradeManagement(uint deviation,string SYMBOL=NULL)
                     :myDeviation(deviation)//Assign deviation value
     {
      //--- Set symbol name
      CSymbol.SetSymbolName(SYMBOL);
     }
   //--- Class destructor
                    ~CTradeManagement(void)
     {
     }
   //--- Will attempt to open buy trade
   bool              Buy(int SL,int TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt to open sell trade
   bool              Sell(int SL,int TP,ulong Magic,string COMMENT=NULL);
   /*This function will delete a pending order if the previous opposing pending order is
   opened into a position, this function is used when trading with StopOrdersType(STOP ORDERS)*/
   void              FundamentalMode(string COMMENT_COMMON);
   /* Function will attempt to re-adjust stop-losses or take-profit values that have
   been changed due to slippage on an order when opening.
   */
   void              SlippageReduction(int SL,int TP,string COMMENT_COMMON);
   //--- This function will open both buy-stop and sell-stop orders for StopOrdersType(STOP ORDERS)
   bool              OpenStops(int SL,int TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt to open a sell-stop order
   bool              OpenSellStop(int SL,int TP,ulong Magic,string COMMENT=NULL);
   //--- Will attempt to open a buy-stop order
   bool              OpenBuyStop(int SL,int TP,ulong Magic,string COMMENT=NULL);
   //--- Function will attempt to close all trades depending on the position comment
   void              CloseTrades(string COMMENT_COMMON);
  };

//+------------------------------------------------------------------+
//|This function will delete a pending order if the previous opposing|
//|pending order is opened into a position, this function is used    |
//|when trading with StopOrdersType(STOP ORDERS)                     |
//+------------------------------------------------------------------+
void CTradeManagement::FundamentalMode(string COMMENT_COMMON)
  {
//--- Iterate through all open positions if Orders are more than zero
   for(int P=0; P<PositionsTotal()&&OrdersTotal()>0; P++)
     {
      //--- Check if Position ticket is above zero
      if(PositionGetTicket(P)>0)
        {
         //--- Check if the Position's Symbol,Magic,Type,Comment is correct
         if(PositionGetString(POSITION_SYMBOL)==CSymbol.GetSymbolName()&&
            StringFind(PositionGetString(POSITION_COMMENT),COMMENT_COMMON)>=0)
           {
            //--- Iterate through all open orders
            for(int O=0; O<OrdersTotal(); O++)
              {
               //--- Check if Order ticket is above zero
               if(OrderGetTicket(O)>0)
                 {
                  //--- Check if the Order's Symbol,Magic,Comment is correct
                  if(OrderGetString(ORDER_SYMBOL)==CSymbol.GetSymbolName()
                     &&OrderGetInteger(ORDER_MAGIC)==PositionGetInteger(POSITION_MAGIC)
                     &&StringFind(OrderGetString(ORDER_COMMENT),COMMENT_COMMON)>=0
                     &&OrderGetString(ORDER_COMMENT)==PositionGetString(POSITION_COMMENT))
                    {
                     //--- Identify Position type
                     switch(int(PositionGetInteger(POSITION_TYPE)))
                       {
                        /* In the case that the Position type is a buy and if the corresponding order type is
                        a sell-stop then delete this order*/
                        case  POSITION_TYPE_BUY:
                           if(OrderGetInteger(ORDER_TYPE)==ORDER_TYPE_SELL_STOP)
                             {
                              //--- Delete the sell-stop order
                              Trade.OrderDelete(OrderGetTicket(O));
                             }
                           break;
                        /* In the case that the Position type is a sell and if the corresponding order type is
                        a buy-stop then delete this order*/
                        case POSITION_TYPE_SELL:
                           if(OrderGetInteger(ORDER_TYPE)==ORDER_TYPE_BUY_STOP)
                             {
                              //--- Delete the sell-stop order
                              Trade.OrderDelete(OrderGetTicket(O));
                             }
                           break;
                        default:
                           break;
                       }
                    }
                 }
              }
           }
        }
     }
  }

//+------------------------------------------------------------------+
//|Function will attempt to re-adjust stop-losses or take-profit     |
//|values that have been changed due to slippage on an order when    |
//|opening.                                                          |
//+------------------------------------------------------------------+
void CTradeManagement::SlippageReduction(int SL,int TP,string COMMENT_COMMON)
  {
//--- Iterate through all open positions
   for(int i=0; i<PositionsTotal(); i++)
     {
      ulong ticket = PositionGetTicket(i);
      //--- Check if Position ticket is above zero
      if(ticket>0)
        {
         //--- Check if the Position's Symbol,Comment is correct
         if(PositionGetString(POSITION_SYMBOL)==CSymbol.GetSymbolName()
            &&StringFind(PositionGetString(POSITION_COMMENT),COMMENT_COMMON)>=0)
           {
            //--- Identify Position type
            switch(int(PositionGetInteger(POSITION_TYPE)))
              {
               /*In the case that the Position type is a buy we set the buy trade properties
               into the variable myBuyTrade, the variable will have the expected trade properties
               for this position such as the stop-loss and take-profit values. In the variables
               mySL and myTP we store the actual stop-loss and take-profit values from the open
               position and compare the expected values from myBuyTrade with the actual values from
               mySL and myTP respectively, if these values don't match, the open position is modified
               to reflect the expected stop-loss and take-profit values from the myBuyTrade variable*/
               case  POSITION_TYPE_BUY:
                  //--- set expect buy trade properties
                  SetBuyTrade(SL,TP,PositionGetDouble(POSITION_PRICE_OPEN));
                  //--- assign sl price
                  mySL = PositionGetDouble(POSITION_SL);
                  //--- assign tp price
                  myTP = PositionGetDouble(POSITION_TP);
                  //--- Normalize sl price
                  CSymbol.NormalizePrice(mySL);
                  mySL = double(DoubleToString(mySL,CSymbol.Digits()));
                  //--- Normalize tp price
                  CSymbol.NormalizePrice(myTP);
                  myTP = double(DoubleToString(myTP,CSymbol.Digits()));
                  //--- check if expected properties match actual trade properties
                  if((myBuyTrade.Get().Stop!=mySL||
                      myBuyTrade.Get().Take!=myTP)
                     &&Valid_Trade(POSITION_TYPE_BUY,myBuyTrade.Get().Open,
                                   myBuyTrade.Get().Stop,myBuyTrade.Get().Take))
                    {
                     //--- Modify position to respect expected properties
                     Trade.PositionModify(ticket,myBuyTrade.Get().Stop,myBuyTrade.Get().Take);
                    }
                  break;
               /*In the case that the Position type is a sell we set the sell trade properties
               into the variable mySellTrade, the variable will have the expected trade properties
               for this position such as the stop-loss and take-profit values. In the variables
               mySL and myTP we store the actual stop-loss and take-profit values from the open
               position and compare the expected values from mySellTrade with the actual values from
               mySL and myTP respectively, if these values don't match, the open position is modified
               to reflect the expected stop-loss and take-profit values from the mySellTrade variable*/
               case POSITION_TYPE_SELL:
                  //--- set expect sell trade properties
                  SetSellTrade(SL,TP,PositionGetDouble(POSITION_PRICE_OPEN));
                  //--- assign sl price
                  mySL = PositionGetDouble(POSITION_SL);
                  //--- assign tp price
                  myTP = PositionGetDouble(POSITION_TP);
                  //--- Normalize sl price
                  CSymbol.NormalizePrice(mySL);
                  mySL = double(DoubleToString(mySL,CSymbol.Digits()));
                  //--- Normalize tp price
                  CSymbol.NormalizePrice(myTP);
                  myTP = double(DoubleToString(myTP,CSymbol.Digits()));
                  //--- check if expected properties match actual trade properties
                  if((mySellTrade.Get().Stop!=mySL||
                      mySellTrade.Get().Take!=myTP)
                     &&Valid_Trade(POSITION_TYPE_SELL,mySellTrade.Get().Open,
                                   mySellTrade.Get().Stop,mySellTrade.Get().Take))
                    {
                     //--- Modify position to respect expected properties
                     Trade.PositionModify(ticket,mySellTrade.Get().Stop,mySellTrade.Get().Take);
                    }
                  break;
               default:
                  break;
              }
           }
        }
     }
  }

//+------------------------------------------------------------------+
//|Will retrieve if there are any open trades                        |
//+------------------------------------------------------------------+
bool CTradeManagement::OpenTrade(ENUM_POSITION_TYPE Type,ulong Magic,string COMMENT=NULL)
  {
//--- Iterate through all open positions
   for(int i=0; i<PositionsTotal(); i++)
     {
      //--- Check if Position ticket is above zero
      if(PositionGetTicket(i)>0)
        {
         //--- Check if the Position's Symbol,Magic,Type,Comment is correct
         if(PositionGetString(POSITION_SYMBOL)==CSymbol.GetSymbolName()&&PositionGetInteger(POSITION_MAGIC)==Magic
            &&PositionGetInteger(POSITION_TYPE)==Type&&PositionGetString(POSITION_COMMENT)==COMMENT)
           {
            //--- Return true when there is an open position
            return true;
           }
        }
     }
//--- No open positions found.
   return false;
  }

//+------------------------------------------------------------------+
//|Will retrieve if there are any open orders                        |
//+------------------------------------------------------------------+
bool CTradeManagement::OpenOrder(ENUM_ORDER_TYPE Type,ulong Magic,string COMMENT=NULL)
  {
//--- Iterate through all open orders
   for(int i=0; i<OrdersTotal(); i++)
     {
      //--- Check if Order ticket is above zero
      if(OrderGetTicket(i)>0)
        {
         //--- Check if the Order's Symbol,Magic,Type,Comment is correct
         if(OrderGetString(ORDER_SYMBOL)==CSymbol.GetSymbolName()&&OrderGetInteger(ORDER_MAGIC)==Magic
            &&OrderGetInteger(ORDER_TYPE)==Type&&OrderGetString(ORDER_COMMENT)==COMMENT)
           {
            //--- Return true when there is an open order
            return true;
           }
        }
     }
//--- No open orders found.
   return false;
  }

//+------------------------------------------------------------------+
//|Will retrieve if there are any deals                              |
//+------------------------------------------------------------------+
bool CTradeManagement::OpenedDeal(ENUM_DEAL_TYPE Type,ulong Magic,string COMMENT=NULL)
  {
//--- Check History starting from 2 minutes ago
   if(HistorySelect(CTime.TimeMinusOffset(TimeTradeServer(),CTime.MinutesS(2)),TimeTradeServer()))
     {
      //--- Iterate through all history deals
      for(int i=0; i<HistoryDealsTotal(); i++)
        {
         //--- Assign history deal ticket
         ulong ticket = HistoryDealGetTicket(i);
         //--- Check if ticket is more than zero
         if(ticket>0)
           {
            //--- Check if the Deal's Symbol,Magic,Type,Comment is correct
            if(HistoryDealGetString(ticket,DEAL_SYMBOL)==CSymbol.GetSymbolName()&&
               HistoryDealGetInteger(ticket,DEAL_MAGIC)==Magic&&HistoryDealGetInteger(ticket,DEAL_TYPE)==Type
               &&HistoryDealGetString(ticket,DEAL_COMMENT)==COMMENT)
              {
               //--- Return true when there are any deals
               return true;
              }
           }
        }
     }
//--- No deals found.
   return false;
  }

//+------------------------------------------------------------------+
//|Check if a trade is valid                                         |
//+------------------------------------------------------------------+
bool CTradeManagement::Valid_Trade(ENUM_POSITION_TYPE Type,double Price,double SL,double TP)
  {
//--- Identify Position type
   switch(Type)
     {
      /*In the case that the Position type is a buy, the TP(take-profit) cannot
      be less than the Price(open-price) or the TP has to be equal to zero. The
      SL(stop-loss) cannot be more than the Price or the SL has to be equal to zero.
      The price difference between the Price and SL should be more than or equal to
      the Symbol's stoplevel or the SL has to be equal to zero.
      The price difference between the TP and Price should be more than or equal to
      the Symbol's stoplevel or the TP has to be equal to zero.
      */
      case  POSITION_TYPE_BUY:
         if((Price<TP||TP==0)&&(Price>SL||SL==0)&&
            ((int((Price-SL)/CSymbol.Point())>=CSymbol.StopLevel())
             ||(SL==0))&&
            ((int((TP-Price)/CSymbol.Point())>=CSymbol.StopLevel())
             ||(TP==0))&&
            Price>0
           )
           {
            //--- Trade properties are valid.
            return true;
           }
         break;
      /*In the case that the Position type is a sell, the TP(take-profit) cannot
      be more than the Price(open-price) or the TP has to be equal to zero. The
      SL(stop-loss) cannot be less than the Price or the SL has to be equal to zero.
      The price difference between the Price and TP should be more than or equal to
      the Symbol's stoplevel or the TP has to be equal to zero.
      The price difference between the SL and Price should be more than or equal to
      the Symbol's stoplevel or the SL has to be equal to zero.
      */
      case POSITION_TYPE_SELL:
         if((Price>TP||TP==0)&&(Price<SL||SL==0)&&
            ((int((Price-TP)/CSymbol.Point())>=CSymbol.StopLevel())
             ||(TP==0))&&
            ((int((SL-Price)/CSymbol.Point())>=CSymbol.StopLevel())
             ||(SL==0))&&
            Price>0
           )
           {
            //--- Trade properties are valid.
            return true;
           }
         break;
      default://Unknown
         return false;
         break;
     }
//--- Trade properties are not valid.
   Print("Something went wrong, SL/TP/Open-Price is incorrect!");
   return false;
  }

//+------------------------------------------------------------------+
//|Check if stop order is valid                                      |
//+------------------------------------------------------------------+
bool CTradeManagement::Valid_Order(ENUM_ORDER_TYPE Type,double Price,double SL,double TP)
  {
//--- Identify Order type
   switch(Type)
     {
      /*In the case that the Order type is a buy-stop, the TP(take-profit) cannot
       be less than the Price(open-price) or the TP has to be equal to zero. The
       SL(stop-loss) cannot be more than the Price or the SL has to be equal to zero.
       The Price should be more than the current Ask price.
       The price difference between the Price and SL should be more than or equal to
       the Symbol's stoplevel or the SL has to be equal to zero.
       The price difference between the TP and Price should be more than or equal to
       the Symbol's stoplevel or the TP has to be equal to zero.
       The myDeviation(Price deviation) should be more than or equal to the Symbol's
       stoplevel.
       */
      case  ORDER_TYPE_BUY_STOP:
         if((Price<TP||TP==0)&&(Price>SL||SL==0)&&(Price>CSymbol.Ask())&&
            ((int((Price-SL)/CSymbol.Point())>=CSymbol.StopLevel())
             ||(SL==0))&&
            ((int((TP-Price)/CSymbol.Point())>=CSymbol.StopLevel())
             ||(TP==0))&&
            myDeviation>=uint(CSymbol.StopLevel())
           )
           {
            //--- Order properties are valid.
            return true;
           }
         break;
      /*In the case that the Order type is a sell-stop, the TP(take-profit) cannot
       be more than the Price(open-price) or the TP has to be equal to zero. The
       SL(stop-loss) cannot be less than the Price or the SL has to be equal to zero.
       The Price should be less than the current Bid price.
       The price difference between the Price and TP should be more than or equal to
       the Symbol's stoplevel or the TP has to be equal to zero.
       The price difference between the SL and Price should be more than or equal to
       the Symbol's stoplevel or the SL has to be equal to zero.
       The myDeviation(Price deviation) should be more than or equal to the Symbol's
       stoplevel.
      */
      case ORDER_TYPE_SELL_STOP:
         if((Price>TP||TP==0)&&(Price<SL||SL==0)&&(Price<CSymbol.Bid())&&
            ((int((Price-TP)/CSymbol.Point())>=CSymbol.StopLevel())
             ||(TP==0))&&
            ((int((SL-Price)/CSymbol.Point())>=CSymbol.StopLevel())
             ||(SL==0))&&
            myDeviation>=uint(CSymbol.StopLevel())
           )
           {
            //--- Order properties are valid.
            return true;
           }
         break;
      default://Other
         return false;
         break;
     }
//--- Order properties are not valid.
   Print("Something went wrong, SL/TP/Deviation/Open-Price is incorrect!");
   return false;
  }

//+------------------------------------------------------------------+
//|Will attempt open buy trade                                       |
//+------------------------------------------------------------------+
bool CTradeManagement::Buy(double SL,double TP,ulong Magic,string COMMENT=NULL)
  {
//--- Normalize the SL Price
   CSymbol.NormalizePrice(SL);
//--- Normalize the TP Price
   CSymbol.NormalizePrice(TP);
//--- Set the order type for Risk management calculation
   SetOrderType(ORDER_TYPE_BUY);
//--- Set open price for Risk management calculation
   OpenPrice = CSymbol.Ask();
//--- Set close price for Risk management calculation
   ClosePrice = SL;
//--- Set Trade magic number
   Trade.SetExpertMagicNumber(Magic);
//--- Check if there are any open trades or opened deals already
   if(!OpenTrade(POSITION_TYPE_BUY,Magic,COMMENT)&&!OpenedDeal(DEAL_TYPE_BUY,Magic,COMMENT)
//--- check if trade is valid
      &&Valid_Trade(POSITION_TYPE_BUY,OpenPrice,SL,TP))
     {
      //--- Iterate through the Lot-sizes if they're more than max-lot
      for(double i=Volume();i>=CSymbol.LotsMin();i-=CSymbol.LotsMax())
        {
         //--- normalize Lot-size
         NormalizeLotsize(i);
         //--- Open trade with a Lot-size not more than max-lot
         TradeResult = Trade.Buy((i>CSymbol.LotsMax())?CSymbol.LotsMax():i,CSymbol.GetSymbolName(),OpenPrice,SL,TP,COMMENT);
         //--- Check if trade failed.
         if(!TradeResult)
           {
            return TradeResult;
           }
        }
     }
   else
     {
      //--- Trade failed because there is an open trade or opened deal
      return false;
     }
//--- Return trade result.
   return TradeResult;
  }

//+------------------------------------------------------------------+
//|Will attempt open buy trade with integer SL & TP                  |
//+------------------------------------------------------------------+
bool CTradeManagement::Buy(int SL,int TP,ulong Magic,string COMMENT=NULL)
  {
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL=CSymbol.Bid()-mySL;
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP+=CSymbol.Ask();
//--- Open trade
   return Buy((SL==0)?0:mySL,(TP==0)?0:myTP,Magic,COMMENT);
  }

//+------------------------------------------------------------------+
//|Will attempt open sell trade                                      |
//+------------------------------------------------------------------+
bool CTradeManagement::Sell(double SL,double TP,ulong Magic,string COMMENT=NULL)
  {
//--- Normalize the SL Price
   CSymbol.NormalizePrice(SL);
//--- Normalize the TP Price
   CSymbol.NormalizePrice(TP);
//--- Set the order type for Risk management calculation
   SetOrderType(ORDER_TYPE_SELL);
//--- Set open price for Risk management calculation
   OpenPrice = CSymbol.Bid();
//--- Set close price for Risk management calculation
   ClosePrice = SL;
//--- Set Trade magic number
   Trade.SetExpertMagicNumber(Magic);
//--- Check if there are any open trades or opened deals already
   if(!OpenTrade(POSITION_TYPE_SELL,Magic,COMMENT)&&!OpenedDeal(DEAL_TYPE_SELL,Magic,COMMENT)
//--- check if trade is valid
      &&Valid_Trade(POSITION_TYPE_SELL,OpenPrice,SL,TP))
     {
      //--- Iterate through the Lot-sizes if they're more than max-lot
      for(double i=Volume();i>=CSymbol.LotsMin();i-=CSymbol.LotsMax())
        {
         //--- normalize Lot-size
         NormalizeLotsize(i);
         //--- Open trade with a Lot-size not more than max-lot
         TradeResult = Trade.Sell((i>CSymbol.LotsMax())?CSymbol.LotsMax():i,CSymbol.GetSymbolName(),CSymbol.Bid(),SL,TP,COMMENT);
         //--- Check if trade failed.
         if(!TradeResult)
           {
            return TradeResult;
           }
        }
     }
   else
     {
      //--- Trade failed because there is an open trade or opened deal
      return false;
     }
//--- Return trade result.
   return TradeResult;
  }

//+------------------------------------------------------------------+
//|Will attempt open sell trade with integer SL & TP                 |
//+------------------------------------------------------------------+
bool CTradeManagement::Sell(int SL,int TP,ulong Magic,string COMMENT=NULL)
  {
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL+=CSymbol.Ask();
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP=CSymbol.Bid()-myTP;
//--- Open trade
   return Sell((SL==0)?0:mySL,(TP==0)?0:myTP,Magic,COMMENT);
  }

//+------------------------------------------------------------------+
//|Will set buy-stop order properties                                |
//+------------------------------------------------------------------+
void CTradeManagement::SetBuyStop(int SL,int TP)
  {
//-- Get Open-price
   myOpenPrice=CSymbol.Ask()+myDeviation*CSymbol.Point();
   CSymbol.NormalizePrice(myOpenPrice);
   NormalizeDouble(myOpenPrice,CSymbol.Digits());
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL=myOpenPrice-mySL;
//--- Normalize the SL Price
   CSymbol.NormalizePrice(mySL);
   NormalizeDouble(mySL,CSymbol.Digits());
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP+=myOpenPrice;
//--- Normalize the TP Price
   CSymbol.NormalizePrice(myTP);
   NormalizeDouble(myTP,CSymbol.Digits());
//--- Set BuyStop properties
   myBuyStop.Set(myOpenPrice,(TP==0)?0:myTP,(SL==0)?0:mySL);
  }

//+------------------------------------------------------------------+
//|Will set buy position properties                                  |
//+------------------------------------------------------------------+
void CTradeManagement::SetBuyTrade(int SL,int TP,double OP)
  {
//-- Get Open-price
   myOpenPrice=OP;
   CSymbol.NormalizePrice(myOpenPrice);
   myOpenPrice = double(DoubleToString(myOpenPrice,CSymbol.Digits()));
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL=myOpenPrice-mySL;
//--- Normalize the SL Price
   CSymbol.NormalizePrice(mySL);
   mySL = double(DoubleToString(mySL,CSymbol.Digits()));
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP+=myOpenPrice;
//--- Normalize the TP Price
   CSymbol.NormalizePrice(myTP);
   myTP = double(DoubleToString(myTP,CSymbol.Digits()));
//--- Set Buy trade properties
   myBuyTrade.Set(myOpenPrice,(TP==0)?0:myTP,(SL==0)?0:mySL);
  }

//+------------------------------------------------------------------+
//|Will set sell-stop order properties                               |
//+------------------------------------------------------------------+
void CTradeManagement::SetSellStop(int SL,int TP)
  {
//-- Get Open-price
   myOpenPrice=CSymbol.Bid()-myDeviation*CSymbol.Point();
   CSymbol.NormalizePrice(myOpenPrice);
   NormalizeDouble(myOpenPrice,CSymbol.Digits());
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL+=myOpenPrice;
//--- Normalize the SL Price
   CSymbol.NormalizePrice(mySL);
   NormalizeDouble(mySL,CSymbol.Digits());
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP=myOpenPrice-myTP;
//--- Normalize the TP Price
   CSymbol.NormalizePrice(myTP);
   NormalizeDouble(myTP,CSymbol.Digits());
//--- Set SellStop properties
   mySellStop.Set(myOpenPrice,(TP==0)?0:myTP,(SL==0)?0:mySL);
  }

//+------------------------------------------------------------------+
//|Will set sell position properties                                 |
//+------------------------------------------------------------------+
void CTradeManagement::SetSellTrade(int SL,int TP,double OP)
  {
//-- Get Open-price
   myOpenPrice=OP;
   CSymbol.NormalizePrice(myOpenPrice);
   myOpenPrice = double(DoubleToString(myOpenPrice,CSymbol.Digits()));
//--- Get SL value
   mySL=SL*CSymbol.Point();
   mySL+=myOpenPrice;
//--- Normalize the SL Price
   CSymbol.NormalizePrice(mySL);
   mySL = double(DoubleToString(mySL,CSymbol.Digits()));
//--- Get TP value
   myTP=TP*CSymbol.Point();
   myTP=myOpenPrice-myTP;
//--- Normalize the TP Price
   CSymbol.NormalizePrice(myTP);
   myTP = double(DoubleToString(myTP,CSymbol.Digits()));
//--- Set Sell trade properties
   mySellTrade.Set(myOpenPrice,(TP==0)?0:myTP,(SL==0)?0:mySL);
  }

//+------------------------------------------------------------------+
//|This function will open both buy-stop and sell-stop orders for    |
//|StopOrdersType(STOP ORDERS)                                       |
//+------------------------------------------------------------------+
bool CTradeManagement::OpenStops(int SL,int TP,ulong Magic,string COMMENT=NULL)
  {
//--- Set buy-stop properties
   SetBuyStop(SL,TP);
//--- Set sell-stop properties
   SetSellStop(SL,TP);
//--- Set the order type for Risk management calculation
   SetOrderType(ORDER_TYPE_BUY);
//--- Set open price for Risk management calculation
   OpenPrice = myBuyStop.Get().Open;
//--- Set close price for Risk management calculation
   ClosePrice = myBuyStop.Get().Stop;
//--- Set Trade magic number
   Trade.SetExpertMagicNumber(Magic);
//--- Check if there are any open trades or opened deals or canceled deals already
   if(!OpenOrder(ORDER_TYPE_BUY_STOP,Magic,COMMENT)&&!OpenedDeal(DEAL_TYPE_BUY,Magic,COMMENT)
      &&!OpenedDeal(DEAL_TYPE_BUY_CANCELED,Magic,COMMENT)
//--- Check if the buy-stop properties are valid
      &&Valid_Order(ORDER_TYPE_BUY_STOP,myBuyStop.Get().Open,myBuyStop.Get().Stop,myBuyStop.Get().Take)
      &&!OpenOrder(ORDER_TYPE_SELL_STOP,Magic,COMMENT)
      &&!OpenedDeal(DEAL_TYPE_SELL,Magic,COMMENT)&&!OpenedDeal(DEAL_TYPE_SELL_CANCELED,Magic,COMMENT)
//--- Check if the sell-stop properties are valid
      &&Valid_Order(ORDER_TYPE_SELL_STOP,mySellStop.Get().Open,mySellStop.Get().Stop,mySellStop.Get().Take))
     {
      //--- Iterate through the Lot-sizes if they're more than max-lot
      for(double i=Volume();i>=CSymbol.LotsMin()&&
          /* Check if current number of orders +2 more orders is less than
             account orders limit.*/
          (PositionsTotal()+Account.numOrders()+2)<Account.LimitOrders()
          ;i-=CSymbol.LotsMax())
        {
         //--- normalize Lot-size
         NormalizeLotsize(i);
         /* Open orders with a Lot-size not more than max-lot and set order expiration
         to the Symbol's session end time for the current day.
         */
         if(!Trade.BuyStop((i>CSymbol.LotsMax())?CSymbol.LotsMax():i,myBuyStop.Get().Open,
                           CSymbol.GetSymbolName(),myBuyStop.Get().Stop,myBuyStop.Get().Take,
                           ORDER_TIME_SPECIFIED,CTS.SessionEnd(),COMMENT)
            ||!Trade.SellStop((i>CSymbol.LotsMax())?CSymbol.LotsMax():i,mySellStop.Get().Open,
                              CSymbol.GetSymbolName(),mySellStop.Get().Stop,mySellStop.Get().Take,
                              ORDER_TIME_SPECIFIED,CTS.SessionEnd(),COMMENT))
           {
            //--- one or more orders failed to open.
            return false;
           }
        }
     }
   else
     {
      //--- Orders failed
      return false;
     }
//--- Return trade result.
   return true;
  }

//+------------------------------------------------------------------+
//|Will attempt to open a buy-stop order                             |
//+------------------------------------------------------------------+
bool CTradeManagement::OpenBuyStop(int SL,int TP,ulong Magic,string COMMENT=NULL)
  {
   SetBuyStop(SL,TP);
//--- Set the order type for Risk management calculation
   SetOrderType(ORDER_TYPE_BUY);
//--- Set open price for Risk management calculation
   OpenPrice = myBuyStop.Get().Open;
//--- Set close price for Risk management calculation
   ClosePrice = myBuyStop.Get().Stop;
//--- Set Trade magic number
   Trade.SetExpertMagicNumber(Magic);
//--- Check if there are any open trades or opened deals or canceled deals already
   if(!OpenOrder(ORDER_TYPE_BUY_STOP,Magic,COMMENT)&&!OpenedDeal(DEAL_TYPE_BUY,Magic,COMMENT)
//--- Check if the buy-stop properties are valid
      &&Valid_Order(ORDER_TYPE_BUY_STOP,myBuyStop.Get().Open,myBuyStop.Get().Stop,myBuyStop.Get().Take))
     {
      //--- Iterate through the Lot-sizes if they're more than max-lot
      for(double i=Volume();i>=CSymbol.LotsMin()&&
          /* Check if current number of orders +1 more orders is less than
                 account orders limit.*/
          (PositionsTotal()+Account.numOrders()+1)<Account.LimitOrders()
          ;i-=CSymbol.LotsMax())
        {
         //--- normalize Lot-size
         NormalizeLotsize(i);
         /* Open buy-stop order with a Lot-size not more than max-lot and set order expiration
         to the Symbol's session end time for the current day.
         */
         if(!Trade.BuyStop((i>CSymbol.LotsMax())?CSymbol.LotsMax():i,myBuyStop.Get().Open,
                           CSymbol.GetSymbolName(),myBuyStop.Get().Stop,myBuyStop.Get().Take,
                           ORDER_TIME_SPECIFIED,CTS.SessionEnd(),COMMENT))
           {
            //--- Order failed to open
            return false;
           }
        }
     }
   else
     {
      //--- Order failed
      return false;
     }
//--- Return trade result.
   return true;
  }

//+------------------------------------------------------------------+
//|Will attempt to open a sell-stop order                            |
//+------------------------------------------------------------------+
bool CTradeManagement::OpenSellStop(int SL,int TP,ulong Magic,string COMMENT=NULL)
  {
   SetSellStop(SL,TP);
//--- Set the order type for Risk management calculation
   SetOrderType(ORDER_TYPE_SELL);
//--- Set open price for Risk management calculation
   OpenPrice = mySellStop.Get().Open;
//--- Set close price for Risk management calculation
   ClosePrice = mySellStop.Get().Stop;
//--- Set Trade magic number
   Trade.SetExpertMagicNumber(Magic);
//--- Check if there are any open trades or opened deals or canceled deals already
   if(!OpenOrder(ORDER_TYPE_SELL_STOP,Magic,COMMENT)&&!OpenedDeal(DEAL_TYPE_SELL,Magic,COMMENT)
//--- Check if the sell-stop properties are valid
      &&Valid_Order(ORDER_TYPE_SELL_STOP,mySellStop.Get().Open,mySellStop.Get().Stop,mySellStop.Get().Take))
     {
      //--- Iterate through the Lot-sizes if they're more than max-lot
      for(double i=Volume();i>=CSymbol.LotsMin()&&
          /* Check if current number of orders +1 more orders is less than
                 account orders limit.*/
          (PositionsTotal()+Account.numOrders()+1)<Account.LimitOrders()
          ;i-=CSymbol.LotsMax())
        {
         //--- normalize Lot-size
         NormalizeLotsize(i);
         /* Open sell-stop order with a Lot-size not more than max-lot and set order expiration
         to the Symbol's session end time for the current day.
         */
         if(!Trade.SellStop((i>CSymbol.LotsMax())?CSymbol.LotsMax():i,mySellStop.Get().Open,
                            CSymbol.GetSymbolName(),mySellStop.Get().Stop,mySellStop.Get().Take,
                            ORDER_TIME_SPECIFIED,CTS.SessionEnd(),COMMENT))
           {
            //--- Order failed to open
            return false;
           }
        }
     }
   else
     {
      //--- Order failed
      return false;
     }
//--- Return trade result.
   return true;
  }

//+------------------------------------------------------------------+
//|Function will attempt to close all trades depending on the        |
//|position comment                                                  |
//+------------------------------------------------------------------+
void CTradeManagement::CloseTrades(string COMMENT_COMMON)
  {
//--- Iterate through all open positions
   for(int i=0; i<PositionsTotal(); i++)
     {
      ulong ticket = PositionGetTicket(i);
      //--- Check if Position ticket is above zero
      if(ticket>0)
        {
         //--- Check if the Position's Symbol,Comment is correct
         if(PositionGetString(POSITION_SYMBOL)==CSymbol.GetSymbolName()
            &&StringFind(PositionGetString(POSITION_COMMENT),COMMENT_COMMON)>=0)
           {
            //--- close trade.
            Trade.PositionClose(ticket);
           }
        }
     }
  }
//+------------------------------------------------------------------+
