//+------------------------------------------------------------------+
//|                                                      NewsTrading |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                            https://www.mql5.com/en/users/kaaiblo |
//+------------------------------------------------------------------+
#include "CommonVariables.mqh"

//-- Enumeration for DST schedule
enum DSTSchedule
  {
   AutoDst_Selection,//AUTO DST
   CustomDst_Selection//CUSTOM DST
  } MyDST;

DST_type MySchedule;//Variable for custom DST schedule
//+------------------------------------------------------------------+
//|TimeManagement class                                              |
//+------------------------------------------------------------------+
class CTimeManagement
  {

private:

   MqlDateTime       today;//private variable
   MqlDateTime       timeFormat;//private variable

public:

   //-- Checks if a date is within two other dates
   bool              DateIsInRange(datetime FirstTime,datetime SecondTime,datetime compareTime);
   //-- Check if two dates(Start&End) are within CompareStart & CompareEnd
   bool              DateIsInRange(datetime Start,datetime End,datetime CompareStart,datetime CompareEnd);
   bool              DateisToday(datetime TimeRepresented);//Checks if a date is within the current day
   int               SecondsS(int multiple=1);//Returns seconds
   int               MinutesS(int multiple=1);//Returns Minutes in seconds
   int               HoursS(int multiple=1);//Returns Hours in seconds
   int               DaysS(int multiple=1);//Returns Days in seconds
   int               WeeksS(int multiple=1);//Returns Weeks in seconds
   int               MonthsS(int multiple=1);//Returns Months in seconds
   int               YearsS(int multiple=1);//Returns Years in seconds
   int               ReturnYear(datetime time);//Returns the Year for a specific date
   int               ReturnMonth(datetime time);//Returns the Month for a specific date
   int               ReturnDay(datetime time);//Returns the Day for a specific date
   int               ReturnHour(datetime time);//Returns the Hour for a specific date
   int               ReturnMinute(datetime time);//Returns the Minute for a specific date
   int               ReturnSecond(datetime time);//Returns the Second for s specific date
   //-- Will return a datetime type of a date with an subtraction offset in seconds
   datetime          TimeMinusOffset(datetime standardtime,int timeoffset);
   //-- Will return a datetime type of a date with an addition offset in seconds
   datetime          TimePlusOffset(datetime standardtime,int timeoffset);
   //-- Will convert datetime to MqlDateTime
   MqlDateTime       Time(datetime Timetoformat);
   //-- Will convert MqlDateTime to datetime
   datetime          Time(MqlDateTime &Timetoformat);
   //-- Will return a datetime with changes to the hour,minute and second
   datetime          Time(datetime time,int Hour,int Minute,int Second);
   //-- Will return a datetime with changes to the hour and minute
   datetime          Time(datetime time,int Hour,int Minute);
   //-- Check current time is within a time range
   bool              TimeIsInRange(datetime BeginTime,datetime EndTime);
   //-- Check if current time is within preEvent time and Event time
   bool              TimePreEvent(datetime PreEvent,datetime Event);
   //-- Return MqlDateTime for current date time with custom hour and minute
   MqlDateTime       Today(int Hour,int Minute);
   //-- Return MqlDateTime for current date time with custom hour, minute and second
   MqlDateTime       Today(int Hour,int Minute,int Second);
   //-- Check current day of the week
   bool              isDayOfTheWeek(DayOfTheWeek Day);
   //-- Return enumeration Day of week for a certain date
   ENUM_DAY_OF_WEEK  DayOfWeek(datetime time);
  };

//+------------------------------------------------------------------+
//|Checks if a date is within two other dates                        |
//+------------------------------------------------------------------+
bool CTimeManagement::DateIsInRange(datetime FirstTime,datetime SecondTime,datetime compareTime)
  {
   return(FirstTime<=compareTime&&SecondTime>compareTime);
  }

//+------------------------------------------------------------------+
//|Check if two dates(Start&End) are within CompareStart & CompareEnd|
//+------------------------------------------------------------------+
bool CTimeManagement::DateIsInRange(datetime Start,datetime End,datetime CompareStart,datetime CompareEnd)
  {
   return(Start<=CompareStart&&CompareEnd<End);
  }

//+------------------------------------------------------------------+
//|Checks if a date is within the current day                        |
//+------------------------------------------------------------------+
bool CTimeManagement::DateisToday(datetime TimeRepresented)
  {
   TimeToStruct(TimeRepresented,timeFormat);
   TimeTradeServer(today);
   return(timeFormat.year==today.year&&timeFormat.mon==today.mon
          &&timeFormat.day==today.day);
  }

//+------------------------------------------------------------------+
//|Returns seconds                                                   |
//+------------------------------------------------------------------+
int CTimeManagement::SecondsS(int multiple=1)
  {
   return (1*multiple);
  }

//+------------------------------------------------------------------+
//|Returns Minutes in seconds                                        |
//+------------------------------------------------------------------+
int CTimeManagement::MinutesS(int multiple=1)
  {
   return (SecondsS(60)*multiple);
  }

//+------------------------------------------------------------------+
//|Returns Hours in seconds                                          |
//+------------------------------------------------------------------+
int CTimeManagement::HoursS(int multiple=1)
  {
   return (MinutesS(60)*multiple);
  }

//+------------------------------------------------------------------+
//|Returns Days in seconds                                           |
//+------------------------------------------------------------------+
int CTimeManagement::DaysS(int multiple=1)
  {
   return (HoursS(24)*multiple);
  }

//+------------------------------------------------------------------+
//|Returns Weeks in seconds                                          |
//+------------------------------------------------------------------+
int CTimeManagement::WeeksS(int multiple=1)
  {
   return (DaysS(7)*multiple);
  }

//+------------------------------------------------------------------+
//|Returns Months in seconds                                         |
//+------------------------------------------------------------------+
int CTimeManagement::MonthsS(int multiple=1)
  {
   return (WeeksS(4)*multiple);
  }

//+------------------------------------------------------------------+
//|Returns Years in seconds                                          |
//+------------------------------------------------------------------+
int CTimeManagement::YearsS(int multiple=1)
  {
   return (MonthsS(12)*multiple);
  }

//+------------------------------------------------------------------+
//|Returns the Year for a specific date                              |
//+------------------------------------------------------------------+
int CTimeManagement::ReturnYear(datetime time)
  {
   return Time(time).year;
  }

//+------------------------------------------------------------------+
//|Returns the Month for a specific date                             |
//+------------------------------------------------------------------+
int CTimeManagement::ReturnMonth(datetime time)
  {
   return Time(time).mon;
  }

//+------------------------------------------------------------------+
//|Returns the Day for a specific date                               |
//+------------------------------------------------------------------+
int CTimeManagement::ReturnDay(datetime time)
  {
   return Time(time).day;
  }

//+------------------------------------------------------------------+
//|Returns the Hour for a specific date                              |
//+------------------------------------------------------------------+
int CTimeManagement::ReturnHour(datetime time)
  {
   return Time(time).hour;
  }

//+------------------------------------------------------------------+
//|Returns the Minute for a specific date                            |
//+------------------------------------------------------------------+
int CTimeManagement::ReturnMinute(datetime time)
  {
   return Time(time).min;
  }

//+------------------------------------------------------------------+
//|Returns the Second for s specific date                            |
//+------------------------------------------------------------------+
int CTimeManagement::ReturnSecond(datetime time)
  {
   return Time(time).sec;
  }

//+------------------------------------------------------------------+
//|Will return a datetime type of a date with an subtraction offset  |
//|in seconds                                                        |
//+------------------------------------------------------------------+
datetime CTimeManagement::TimeMinusOffset(datetime standardtime,int timeoffset)
  {
   standardtime-=timeoffset;
   return standardtime;
  }

//+------------------------------------------------------------------+
//|Will return a datetime type of a date with an addition offset     |
//|in seconds                                                        |
//+------------------------------------------------------------------+
datetime CTimeManagement::TimePlusOffset(datetime standardtime,int timeoffset)
  {
   standardtime+=timeoffset;
   return standardtime;
  }

//+------------------------------------------------------------------+
//|Will convert datetime to MqlDateTime                              |
//+------------------------------------------------------------------+
MqlDateTime CTimeManagement::Time(datetime Timetoformat)
  {
   TimeToStruct(Timetoformat,timeFormat);
   return timeFormat;
  }

//+------------------------------------------------------------------+
//|Will convert MqlDateTime to datetime                              |
//+------------------------------------------------------------------+
datetime CTimeManagement::Time(MqlDateTime &Timetoformat)
  {
   return StructToTime(Timetoformat);
  }

//+------------------------------------------------------------------+
//|Will return a datetime with changes to the hour,minute and second |
//+------------------------------------------------------------------+
datetime CTimeManagement::Time(datetime time,int Hour,int Minute,int Second)
  {
   timeFormat=Time(time);
   timeFormat.hour=Hour;
   timeFormat.min=Minute;
   timeFormat.sec=Second;
   return StructToTime(timeFormat);
  }
//+------------------------------------------------------------------+
//|Will return a datetime with changes to the hour and minute        |
//+------------------------------------------------------------------+
datetime CTimeManagement::Time(datetime time,int Hour,int Minute)
  {
   timeFormat=Time(time);
   timeFormat.hour=Hour;
   timeFormat.min=Minute;
   return StructToTime(timeFormat);
  }

//+------------------------------------------------------------------+
//|Check current time is within a time range                         |
//+------------------------------------------------------------------+
bool CTimeManagement::TimeIsInRange(datetime BeginTime,datetime EndTime)
  {
   if(BeginTime<=TimeTradeServer()&&EndTime>=TimeTradeServer())
     {
      return true;
     }
   return false;
  }

//+------------------------------------------------------------------+
//|Check if current time is within preEvent time and Event time      |
//+------------------------------------------------------------------+
bool CTimeManagement::TimePreEvent(datetime PreEventTime,datetime EventTime)
  {
   if(PreEventTime<=TimeTradeServer()&&EventTime>TimeTradeServer())
     {
      return true;
     }
   return false;
  }

//+------------------------------------------------------------------+
//|Return MqlDateTime for current date time with custom hour and     |
//|minute                                                            |
//+------------------------------------------------------------------+
MqlDateTime CTimeManagement::Today(int Hour,int Minute)
  {
   TimeTradeServer(today);
   today.hour=Hour;
   today.min=Minute;
   return today;
  }

//+------------------------------------------------------------------+
//|Return MqlDateTime for current date time with custom hour, minute |
//|and second                                                        |
//+------------------------------------------------------------------+
MqlDateTime CTimeManagement::Today(int Hour,int Minute,int Second)
  {
   TimeTradeServer(today);
   today.hour=Hour;
   today.min=Minute;
   today.sec=Second;
   return today;
  }

//+------------------------------------------------------------------+
//|Check current day of the week                                     |
//+------------------------------------------------------------------+
bool CTimeManagement::isDayOfTheWeek(DayOfTheWeek Day)
  {
   switch(Day)
     {
      case  Monday://Monday
         if(DayOfWeek(TimeTradeServer())==MONDAY)
           {
            return true;
           }
         break;
      case Tuesday://Tuesday
         if(DayOfWeek(TimeTradeServer())==TUESDAY)
           {
            return true;
           }
         break;
      case Wednesday://Wednesday
         if(DayOfWeek(TimeTradeServer())==WEDNESDAY)
           {
            return true;
           }
         break;
      case Thursday://Thursday
         if(DayOfWeek(TimeTradeServer())==THURSDAY)
           {
            return true;
           }
         break;
      case Friday://Friday
         if(DayOfWeek(TimeTradeServer())==FRIDAY)
           {
            return true;
           }
         break;
      case AllDays://All days
         return true;
         break;
      default://Unknown
         break;
     }
   return false;
  }

//+------------------------------------------------------------------+
//|Return enumeration Day of week for a certain date                 |
//+------------------------------------------------------------------+
ENUM_DAY_OF_WEEK CTimeManagement::DayOfWeek(datetime time)
  {
   return (ENUM_DAY_OF_WEEK)Time(time).day_of_week;
  }
//+------------------------------------------------------------------+
