//+------------------------------------------------------------------+
//|                                                   SignalCMLP.mqh |
//|                   Copyright 2009-2017, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
#include <My\Cmlp_ad.mqh>
//+------------------------------------------------------------------+
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals based on MLP with Learning Rate Schedules.         |
//| Type=SignalAdvanced                                              |
//| Name=CMLP                                                        |
//| ShortName=CMLP                                                   |
//| Class=CSignalCMLP                                                |
//| Page=signal_cgan                                                 |
//| Parameter=TrainSet,int,260,Training Set Size                     |
//| Parameter=Epochs,int,65,Epochs                                   |
//| Parameter=LearningRate,double,0.0005,Learning Rate               |
//| Parameter=DecayEpochSteps,int,5,Decay Epoch Steps                |
//| Parameter=DecayRate,double,0.1,Decay Rate                        |
//| Parameter=PolynomialPower,int,2,Polynomial Power                 |
//| Parameter=MinLearningRate,double,0.0001,Minimum Learning Rate    |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalCMLP.                                               |
//| Purpose: Class of MLP with Learning Rate Schedules.              |
//|            Derives from class CExpertSignal.                     |
//+------------------------------------------------------------------+
enum ENUM_LEARNING
{  LEARNING_FIXED = 0,
   LEARNING_STEP_DECAY = 1,
   LEARNING_EXPONENTIAL_DECAY = 2,
   LEARNING_POLYNOMIAL_DECAY = 3,
   LEARNING_INVERSE_TIME_DECAY = 4,
   LEARNING_COSINE_ANNEALING = 5,
   LEARNING_CYCLICAL = 6,
   LEARNING_ONE_CYCLE = 7,
   LEARNING_ADAPTIVE = 8
};
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#define  __MLP_INPUTS 5
#define  __LAYERS 1
#define  __SIZES 8
#define  __MLP_OUTPUTS 1
int      __MLP_SETTINGS[2 + __LAYERS] // network settings

=
{  __MLP_INPUTS,  __SIZES,    __MLP_OUTPUTS
};
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalCMLP   : public CExpertSignal
{
protected:

   int                           m_train_set;                       //
   int                           m_epochs;                              // Epochs
   double                        m_learning_rate;                       // Epsilon

   ENUM_LEARNING                 m_learning_type;

   double                        m_train_deposit;
   double                        m_train_criteria;

   int                           m_decay_epoch_steps;
   double                        m_decay_rate;
   int                           m_polynomial_power;
   double                        m_min_learning_rate;                       // Epsilon
   ENUM_ADAPTIVE                 m_adaptive_type;

   double                        m_prior_learning_rate;

public:
                     CSignalCMLP(void);
                    ~CSignalCMLP(void);

   //--- methods of setting adjustable parameters
   void                          TrainSet(int value)
   {  m_train_set = value;
   }
   void                          Epochs(int value)
   {  m_epochs = value;
   }
   void                          LearningRate(double value)
   {  m_learning_rate = value;
   }
   void                          LearningType(ENUM_LEARNING value)
   {  m_learning_type = value;
   }
   void                          DecayEpochSteps(int value)
   {  m_decay_epoch_steps = value;
   }
   void                          DecayRate(double value)
   {  m_decay_rate = value;
   }
   void                          PolynomialPower(int value)
   {  m_polynomial_power = value;
   }
   void                          MinLearningRate(double value)
   {  m_min_learning_rate = value;
   }
   void                          AdaptiveType(ENUM_ADAPTIVE value)
   {  m_adaptive_type = value;
   }

   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   void                          GetOutput(double &Output);
   void                          SetOutput();

   Cmlp                          *MLP;
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CSignalCMLP::CSignalCMLP(void) :    m_train_set(260),
   m_epochs(30),
   m_learning_rate(0.0005),
   m_learning_type(LEARNING_STEP_DECAY),
   m_decay_epoch_steps(5),
   m_decay_rate(0.1),
   m_polynomial_power(2),
   m_min_learning_rate(0.0001),
   m_adaptive_type(ADAPTIVE_GRAD)

{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE + USE_SERIES_SPREAD + USE_SERIES_TIME;
   m_prior_learning_rate = m_learning_rate;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CSignalCMLP::~CSignalCMLP(void)
{  delete MLP;
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalCMLP::ValidationSettings(void)
{  if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_epochs < 1)
   {  printf(__FUNCSIG__ + " epochs size should at least be 1! ");
      return(false);
   }
   if(m_learning_rate < 0.0)
   {  printf(__FUNCSIG__ + " learning rate should be greater than 0.0! ");
      return(false);
   }
   MLP = new Cmlp(__MLP_SETTINGS, 0.005, 0.0);
   m_train_deposit = AccountInfoDouble(ACCOUNT_BALANCE);
   m_train_criteria = 0.0;
   m_prior_learning_rate = m_learning_rate;
//read best weights
//datetime _version;
//N.Get(m_symbol.Name()+"_"+EnumToString(m_period), m_train_criteria, _version);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalCMLP::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalCMLP::LongCondition(void)
{  int result = 0;
   double _mlp_output = 0.0;
   GetOutput(_mlp_output);
   m_symbol.Refresh();
   m_symbol.RefreshRates();
   if(_mlp_output > m_symbol.Bid())
   {  vector _scale;
      _scale.CopyRates(m_symbol.Name(), m_period, 8, 0, __MLP_INPUTS);
      result = int(round(100.0 * (fmin(_mlp_output, _scale.Max()) - _scale.Min()) / fmax(m_symbol.Point(), _scale.Max() - _scale.Min())));
   }
//printf(__FUNCSIG__ + " output is: %.5f, change is: %.5f, and result is: %i", _mlp_output, m_symbol.Bid()-_mlp_output, result);return(0);
   return(result);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalCMLP::ShortCondition(void)
{  int result = 0;
   double _mlp_output = 0.0;
   GetOutput(_mlp_output);
   m_symbol.Refresh();
   m_symbol.RefreshRates();
   if(_mlp_output < m_symbol.Bid())
   {  vector _scale;
      _scale.CopyRates(m_symbol.Name(), m_period, 8, 0, __MLP_INPUTS);
      result = int(round(100.0 * (fmin(_mlp_output, _scale.Max()) - _scale.Min()) / fmax(m_symbol.Point(), _scale.Max() - _scale.Min())));
   }
//printf(__FUNCSIG__ + " output is: %.5f, change is: %.5f, and result is: %i", _mlp_output, m_symbol.Bid()-_mlp_output, result);return(0);
   return(result);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSignalCMLP::GetOutput(double &Output)
{  Output = 0.0;
   double _learning_rate = m_learning_rate;
   for(int i = m_epochs; i >= 0; i--)
   {  if(m_learning_type == LEARNING_STEP_DECAY)
      {  int _epoch_index = int(MathFloor((m_epochs - i) / m_decay_epoch_steps));
         _learning_rate = m_learning_rate * pow(m_decay_rate, _epoch_index);
      }
      else if(m_learning_type == LEARNING_EXPONENTIAL_DECAY)
      {  _learning_rate = m_learning_rate * exp(-1.0 * m_decay_rate * (m_epochs - i + 1));
      }
      else if(m_learning_type == LEARNING_POLYNOMIAL_DECAY)
      {  _learning_rate = m_learning_rate * pow(1.0 - ((m_epochs - i) / m_epochs), m_polynomial_power);
      }
      else if(m_learning_type == LEARNING_INVERSE_TIME_DECAY)
      {  _learning_rate = m_prior_learning_rate / (1.0 + (m_decay_rate * (m_epochs - i)));
         m_prior_learning_rate = _learning_rate;
      }
      else if(m_learning_type == LEARNING_COSINE_ANNEALING)
      {  _learning_rate = m_min_learning_rate + (0.5 * (m_learning_rate - m_min_learning_rate) * (1.0 + MathCos(((m_epochs - i) * M_PI) / m_epochs)));
      }
      else if(m_learning_type == LEARNING_CYCLICAL)
      {  double _x = fabs(((2.0 * fmod(m_epochs - i, m_epochs)) / m_epochs) - 1.0);
         _learning_rate = m_min_learning_rate + ((m_learning_rate - m_min_learning_rate) * fmax(0.0, (1.0 - _x)));
      }
      else if(m_learning_type == LEARNING_ONE_CYCLE)
      {  double _cycle_position = (double)((m_epochs - i) % (2 * m_epochs)) / (2.0 * m_epochs);
         if (_cycle_position <= 0.5)
         {  _learning_rate = m_min_learning_rate + (2.0 * _cycle_position * (m_learning_rate - m_min_learning_rate));
         }
         else
         {  _learning_rate = m_learning_rate - (2.0 * (_cycle_position - 0.5) * (m_learning_rate - m_min_learning_rate));
         }
      }
      for(int ii = m_train_set; ii >= 0; ii--)
      {  vector _in, _out;
         if(_in.Init(__MLP_INPUTS) && _in.CopyRates(m_symbol.Name(), m_period, 8, ii + __MLP_OUTPUTS, __MLP_INPUTS) && _in.Size() == __MLP_INPUTS)
         {  MLP.Set(_in);
            MLP.Forward();
            if(ii > 0)// train
            {  if(_out.Init(__MLP_OUTPUTS) && _out.CopyRates(m_symbol.Name(), m_period, 8, ii, __MLP_OUTPUTS) && _out.Size() == __MLP_OUTPUTS)
               {  MLP.Get(_out);
                  if(m_learning_type == LEARNING_ADAPTIVE && m_adaptive_type != ADAPTIVE_NONE)
                  {  MLP.Backward(_learning_rate, m_adaptive_type, m_decay_rate);
                  }
                  else
                  {  MLP.Backward(_learning_rate, ADAPTIVE_NONE);
                  }
               }
            }
            else if(ii == 0 && i == 0)
            {  Output = MLP.output[0];
            }
         }
      }
   }
}
//+------------------------------------------------------------------+
//| Weights & Biases exporting function for network                  |
//+------------------------------------------------------------------+
void CSignalCMLP::SetOutput(void)
{  if(AccountInfoDouble(ACCOUNT_EQUITY) - m_train_deposit > m_train_criteria && m_train_criteria >= 0.0)
   {  MLP.Set(m_symbol.Name() + "_" + EnumToString(m_period), AccountInfoDouble(ACCOUNT_EQUITY) - m_train_deposit);
   }
}
//+------------------------------------------------------------------+
