//+------------------------------------------------------------------+
//|                                                      Network.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#define __NON_ZERO 1e-8
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum ENUM_ADAPTIVE
{  ADAPTIVE_NONE = -1,
   ADAPTIVE_GRAD = 0,
   ADAPTIVE_RMS = 1,
   ADAPTIVE_ME = 2,
   ADAPTIVE_DELTA = 3,
};
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class Cmlp
{
protected:
   matrix            weights[];
   vector            biases[];
   
   vector            adaptive_gradients[];
   vector            adaptive_deltas[];
   
   vector            target;

   vector            inputs;
   vector            hidden_outputs[];

   int               hidden_layers;
   
   double            Activation(double X);
   double            ActivationDerivative(double X);
   
   bool              validated;
   
   void              AdaptiveGradientUpdate(double LearningRate, int LayerIndex, vector &Gradients, vector &Outputs);
   void              AdaptiveRMSUpdate(double DecayRate, double LearningRate, int LayerIndex, vector &Gradients, vector &Outputs);
   void              AdaptiveMEUpdate(double DecayRate, double LearningRate, int LayerIndex, vector &Gradients, vector &Outputs);
   void              AdaptiveDeltaUpdate(double DecayRate, double LearningRate, int LayerIndex, vector &Gradients, vector &Outputs);
   
public:
   vector            output;

   void              Get(vector &Target)
   {                 target.Init(0);
      target.Copy(Target);
   }

   double            Get(ENUM_REGRESSION_METRIC R)
   {                 return(target.RegressionMetric(output, R));
   }

   void              Set(vector &Inputs)
   {                 inputs.Init(0);
      inputs.Copy(Inputs);
   }

   bool              Get(string File, double &Criteria, datetime &Version);
   bool              Set(string File, double Criteria);

   void              Forward();
   void              Backward(double LearningRate = 0.1, ENUM_ADAPTIVE AdaptiveType = ADAPTIVE_NONE, double DecayRate = 0.9);

   void              Cmlp(int &Settings[], double InitialWeight, double InitialBias)
   {  validated = false;
      int _size =    ArraySize(Settings);
      if(_size >= 2 && _size <= USHORT_MAX && Settings[ArrayMinimum(Settings)] > 0 && Settings[ArrayMaximum(Settings)] < USHORT_MAX)
      {  ArrayResize(weights, _size - 1);
         ArrayResize(biases, _size - 1);
         ArrayResize(adaptive_gradients, _size - 1);
         ArrayResize(adaptive_deltas, _size - 1);
         ArrayResize(hidden_outputs, _size - 2);
         hidden_layers = _size - 2;
         for(int i = 0; i < _size - 1; i++)
         {  weights[i].Init(Settings[i + 1], Settings[i]);
            weights[i].Fill(InitialWeight);
            biases[i].Init(Settings[i + 1]);
            biases[i].Fill(InitialBias);
            adaptive_gradients[i].Init(Settings[i + 1]);
            adaptive_gradients[i].Fill(0.0);
            adaptive_deltas[i].Init(Settings[i + 1]);
            adaptive_deltas[i].Fill(0.0);
            if(i < _size - 2)
            {  hidden_outputs[i].Init(Settings[i + 1]);
               hidden_outputs[i].Fill(0.0);
            }
         }
         output.Init(Settings[_size - 1]);
         target.Init(Settings[_size - 1]);
         validated = true;
      }
      else
      {  printf(__FUNCSIG__ + " invalid network settings. ");
         //~Cmlp(void);
      }
   };
   void              ~Cmlp(void) { };
};
//+------------------------------------------------------------------+
//| Activation activation function                                   |
//+------------------------------------------------------------------+
double Cmlp::Activation(double X)
{  return (1.0 / (1.0 + exp(-X)));
}
//+------------------------------------------------------------------+
//| Derivative of the Activation function                            |
//+------------------------------------------------------------------+
double Cmlp::ActivationDerivative(double X)
{  double _sigmoid = Activation(X);
   return (_sigmoid * (1.0 - _sigmoid));
}
//+------------------------------------------------------------------+
// Adaptive Gradient Update function
//+------------------------------------------------------------------+
void Cmlp::AdaptiveGradientUpdate(double LearningRate, int LayerIndex, vector &Gradients, vector &Outputs)
{  for (int i = 0; i < int(weights[LayerIndex].Rows()); i++)
   {  adaptive_gradients[LayerIndex][i] += (Gradients[i] * Gradients[i]);
      double _learning_rate = LearningRate / (MathSqrt(adaptive_gradients[LayerIndex][i]) + __NON_ZERO);
      for (int j = 0; j < int(weights[LayerIndex].Cols()); j++)
      {  weights[LayerIndex][i][j] -= (_learning_rate * Gradients[i] * Outputs[j]);
      }
      //
      biases[LayerIndex][i] -= _learning_rate * Gradients[i];
   }
}
//+------------------------------------------------------------------+
// Adaptive RMS Update function
//+------------------------------------------------------------------+
void Cmlp::AdaptiveRMSUpdate(double DecayRate, double LearningRate, int LayerIndex, vector &Gradients, vector &Outputs)
{  for (int i = 0; i < int(weights[LayerIndex].Rows()); i++)
   {  adaptive_gradients[LayerIndex][i] += (DecayRate * adaptive_gradients[LayerIndex][i]) + ((1.0 - DecayRate)*(Gradients[i] * Gradients[i]));
      double _learning_rate = LearningRate / (MathSqrt(adaptive_gradients[LayerIndex][i]) + __NON_ZERO);
      for (int j = 0; j < int(weights[LayerIndex].Cols()); j++)
      {  weights[LayerIndex][i][j] -= (_learning_rate * Gradients[i] * Outputs[j]);
      }
      biases[LayerIndex][i] -= _learning_rate * Gradients[i];
   }
}
//+------------------------------------------------------------------+
// Adaptive Mean Exponential(ME) Update function
//+------------------------------------------------------------------+
void Cmlp::AdaptiveMEUpdate(double DecayRate, double LearningRate, int LayerIndex, vector &Gradients, vector &Outputs)
{  for (int i = 0; i < int(weights[LayerIndex].Rows()); i++)
   {  adaptive_gradients[LayerIndex][i] = (DecayRate * adaptive_gradients[LayerIndex][i]) + ((1.0 - DecayRate)*(Gradients[i] * Gradients[i]));
      double _learning_rate = LearningRate / (fabs(adaptive_gradients[LayerIndex][i]) + __NON_ZERO);
      for (int j = 0; j < int(weights[LayerIndex].Cols()); j++)
      {  weights[LayerIndex][i][j] -= (_learning_rate * Gradients[i] * Outputs[j]);
      }
      //
      biases[LayerIndex][i] -= _learning_rate * Gradients[i];
   }
}
//+------------------------------------------------------------------+
// Adaptive Delta Update function
//+------------------------------------------------------------------+
void Cmlp::AdaptiveDeltaUpdate(double DecayRate, double LearningRate, int LayerIndex, vector &Gradients, vector &Outputs)
{  for (int i = 0; i < int(weights[LayerIndex].Rows()); i++)
   {  adaptive_gradients[LayerIndex][i] = (DecayRate * adaptive_gradients[LayerIndex][i]) + ((1.0 - DecayRate)*(Gradients[i] * Gradients[i]));
      double _delta = (MathSqrt(adaptive_deltas[LayerIndex][i] + __NON_ZERO) / MathSqrt(adaptive_gradients[LayerIndex][i] + __NON_ZERO)) * Gradients[i];
      adaptive_deltas[LayerIndex][i] = (DecayRate * adaptive_deltas[LayerIndex][i]) + ((1.0 - DecayRate) * _delta * _delta);
      for (int j = 0; j < int(weights[LayerIndex].Cols()); j++)
      {  weights[LayerIndex][i][j] -= (_delta * Outputs[j]);
      }
      // Bias update with AdaDelta
      biases[LayerIndex][i] -= _delta;
   }
}
//+------------------------------------------------------------------+
//| Forward pass through the neural network                          |
//+------------------------------------------------------------------+
void Cmlp::Forward()
{  if(!validated)
   {  printf(__FUNCSIG__ + " invalid network settings! ");
      return;
   }
// Calculate hidden layer values
   for(int h = 0; h < hidden_layers; h++)
   {  for (int i = 0; i < int(weights[h].Rows()); i++)
      {  double _sum = 0.0;
         if(h == 0)
         {  for (int j = 0; j < int(inputs.Size()); j++)
            {  _sum += inputs[j] * weights[h][i][j];
            }
         }
         else
         {  for (int j = 0; j < int(hidden_outputs[h - 1].Size()); j++)
            {  _sum += hidden_outputs[h - 1][j] * weights[h][i][j];
            }
         }
         //
         hidden_outputs[h][i] = Activation(_sum + biases[h][i]);
      }
   }
// Calculate output
   output.Fill(0.0);
   for (int i = 0; i < int(weights[hidden_layers].Rows()); i++)
   {  double _sum = 0.0;
      for (int j = 0; j < int(weights[hidden_layers].Cols()); j++)
      {  _sum += hidden_outputs[hidden_layers - 1][j] * weights[hidden_layers][i][j];
      }
      output[i] += Activation(_sum + biases[hidden_layers][i]);
   }
}
//+------------------------------------------------------------------+
//| Backward pass through the neural network to update weights       |
//| and biases using gradient descent                                |
//+------------------------------------------------------------------+
void Cmlp::Backward(double LearningRate = 0.1, ENUM_ADAPTIVE AdaptiveType = ADAPTIVE_NONE, double DecayRate = 0.9)
{  if(!validated)
   {  printf(__FUNCSIG__ + " invalid network settings! ");
      return;
   }
// Calculate output layer gradients
   vector _output_error = target - output;
   vector _output_gradients;
   _output_gradients.Init(output.Size());
   for (int i = 0; i < int(output.Size()); i++)
   {  _output_gradients[i] = _output_error[i] * ActivationDerivative(output[i]);
   }
// Update output layer weights and biases
   if(AdaptiveType == ADAPTIVE_NONE)
   {  for (int i = 0; i < int(output.Size()); i++)
      {  for (int j = 0; j < int(weights[hidden_layers].Cols()); j++)
         {  weights[hidden_layers][i][j] += LearningRate * _output_gradients[i] * hidden_outputs[hidden_layers - 1][j];
         }
         biases[hidden_layers][i] += LearningRate * _output_gradients[i];
      }
   }
// Adaptive updates
   else if(AdaptiveType != ADAPTIVE_NONE)
   {  if(AdaptiveType == ADAPTIVE_GRAD)
      {  AdaptiveGradientUpdate(LearningRate, hidden_layers, _output_gradients, hidden_outputs[hidden_layers - 1]);
      }
      else if(AdaptiveType == ADAPTIVE_RMS)
      {  AdaptiveRMSUpdate(DecayRate, LearningRate, hidden_layers, _output_gradients, hidden_outputs[hidden_layers - 1]);
      }
      else if(AdaptiveType == ADAPTIVE_ME)
      {  AdaptiveMEUpdate(DecayRate, LearningRate, hidden_layers, _output_gradients, hidden_outputs[hidden_layers - 1]);
      }
      else if(AdaptiveType == ADAPTIVE_DELTA)
      {  AdaptiveDeltaUpdate(DecayRate, LearningRate, hidden_layers, _output_gradients, hidden_outputs[hidden_layers - 1]);
      }
   }
// Calculate hidden layer gradients
   vector _hidden_gradients[];
   ArrayResize(_hidden_gradients, hidden_layers);
   for(int h = hidden_layers - 1; h >= 0;  h--)
   {  vector _hidden_target;
      _hidden_target.Init(hidden_outputs[h].Size());
      _hidden_target.Fill(0.0);
      _hidden_gradients[h].Init(hidden_outputs[h].Size());
      if(h == hidden_layers - 1)
      {  for(int j = 0; j < int(hidden_outputs[h].Size()); j++)
         {  double _sum = 0.0;
            for(int i = 0; i < int(target.Size()); i++)
            {  if(weights[h + 1][i][j] != 0.0)
               {  _sum += (target[i] / weights[h + 1][i][j]);
               }
            }
            _hidden_target[j] = ActivationDerivative(_sum - biases[h][j]);
         }
      }
      else if(h < hidden_layers - 1)
      {  for(int j = 0; j < int(hidden_outputs[h].Size()); j++)
         {  double _sum = 0.0;
            for(int i = 0; i < int(hidden_outputs[h + 1].Size()); i++)
            {  if(weights[h][i][j] != 0.0)
               {  _sum += (hidden_outputs[h + 1][i] / weights[h][i][j]);
               }
            }
            _hidden_target[j] = ActivationDerivative(_sum - biases[h][j]);
         }
      }
      vector _hidden_error = _hidden_target - hidden_outputs[h];
      for (int i = 0; i < int(_hidden_target.Size()); i++)
      {  _hidden_gradients[h][i] = _hidden_error[i] * ActivationDerivative(hidden_outputs[h][i]);
      }
   }
// Adaptive updates
   if(AdaptiveType != ADAPTIVE_NONE)
   {  for(int h = hidden_layers - 1; h >= 0;  h--)
      {  vector _outputs = inputs;
         if(h > 0)
         {  _outputs = hidden_outputs[h - 1];
         }
         if(AdaptiveType == ADAPTIVE_GRAD)
         {  AdaptiveGradientUpdate(LearningRate, h, _hidden_gradients[h], _outputs);
         }
         else if(AdaptiveType == ADAPTIVE_RMS)
         {  AdaptiveRMSUpdate(DecayRate, LearningRate, h, _hidden_gradients[h], _outputs);
         }
         else if(AdaptiveType == ADAPTIVE_ME)
         {  AdaptiveMEUpdate(DecayRate, LearningRate, h, _hidden_gradients[h], _outputs);
         }
         else if(AdaptiveType == ADAPTIVE_DELTA)
         {  AdaptiveDeltaUpdate(DecayRate, LearningRate, h, _hidden_gradients[h], _outputs);
         }
      }
   }
// Update hidden layer weights and biases
   else if(AdaptiveType == ADAPTIVE_NONE)
   {  for(int h = hidden_layers - 1; h >= 0;  h--)
      {  for (int i = 0; i < int(weights[h].Rows()); i++)
         {  for (int j = 0; j < int(weights[h].Cols()); j++)
            {  if(h == 0)
               {  weights[h][i][j] += LearningRate * _hidden_gradients[h][i] * inputs[j];
               }
               else if(h > 0)
               {  weights[h][i][j] += LearningRate * _hidden_gradients[h][i] * hidden_outputs[h - 1][j];
               }
            }
            biases[h][i] += LearningRate * _hidden_gradients[h][i];
         }
      }
   }
}
//+------------------------------------------------------------------+
//| Get Weights in import                                            |
//+------------------------------------------------------------------+
bool Cmlp::Get(string File, double &Criteria, datetime &Version)
{  ResetLastError();
   int _handle = FileOpen(MQLInfoString(MQL_PROGRAM_NAME) + "_" + File + ".bin", FILE_READ | FILE_BIN | FILE_COMMON);
   if(_handle != INVALID_HANDLE)
   {  Criteria = FileReadDouble(_handle);
      Version = FileReadInteger(_handle);
      PrintFormat(__FUNCSIG__ + " for the file: %s, we have the criteria: %.2f, logged on: %s ",
                  File, Criteria, TimeToString(Version));
      for(int i = 0; i < int(weights.Size()); i++)
      {  for(int j = 0; j < int(weights[i].Rows()); j++)
         {  for(int k = 0; k < int(weights[i].Cols()); k++)
            {  weights[i][j][k] = FileReadDouble(_handle);
               PrintFormat(__FUNCSIG__ +
                           " weight is: %.2f, for layer %i, at row: %i, and col: %i",
                           weights[i][j][k], i, j, k);
            }
         }
      }
      for(int i = 0; i < int(biases.Size()); i++)
      {  for(int j = 0; j < int(biases[i].Size()); j++)
         {  biases[i][j] = FileReadDouble(_handle);
            PrintFormat(__FUNCSIG__ +
                        " %.5f is the bias for layer %i, at row: %i",
                        biases[i][j], i, j);
         }
      }
      FileClose(_handle);
   }
   else
   {  printf(__FUNCSIG__ + " failed to create get handle for: " + File + ", err: " + IntegerToString(GetLastError()));
      FileClose(_handle);
      return(false);
   }
   FileClose(_handle);
   return(true);
}
//+------------------------------------------------------------------+
//| Set Weights for export                                           |
//+------------------------------------------------------------------+
bool Cmlp::Set(string File, double Criteria)
{  ResetLastError();
   int _handle = FileOpen(MQLInfoString(MQL_PROGRAM_NAME) + "_" + File + ".bin", FILE_WRITE | FILE_BIN | FILE_COMMON);
   if(_handle != INVALID_HANDLE)
   {  FileWriteDouble(_handle, Criteria);
      FileWriteInteger(_handle, int(TimeCurrent()));
      for(int i = 0; i < int(weights.Size()); i++)
      {  for(int j = 0; j < int(weights[i].Rows()); j++)
         {  for(int k = 0; k < int(weights[i].Cols()); k++)
            {  FileWriteDouble(_handle, weights[i][j][k]);
            }
         }
      }
      for(int i = 0; i < int(biases.Size()); i++)
      {  for(int j = 0; j < int(biases[i].Size()); j++)
         {  FileWriteDouble(_handle, biases[i][j]);
         }
      }
      FileClose(_handle);
   }
   else
   {  printf(__FUNCSIG__ + " failed to create set handle for: " + File + ", err: " + IntegerToString(GetLastError()));
      FileClose(_handle);
      return(false);
   }
   FileClose(_handle);
   return(true);
}
//+------------------------------------------------------------------+
