//+------------------------------------------------------------------+
//|                                                        Arima.mqh |
//|                        Copyright 2023, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Software Corp."
#property link      "https://www.mql5.com"
#include <Math\Stat\Math.mqh>
#include <Math\Stat\Normal.mqh>
#include <PowellsMethod.mqh>

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CArima:public PowellsMethod
{
  private:
     bool   m_const,m_istrained;
     uint   m_diff_order,m_ar_order,m_ma_order;
     uint   m_arlags[],m_malags[];
     double m_model[],m_sse;
     double m_differenced[],m_innovation[],m_leads[];
     
     void   difference(const uint difference_degree, double &data[], double &differenced[], double &leads[]);
     void   integrate(double &differenced[], double &leads[], double &integrated[]);
     virtual double  func(const double &p[]);
    
     
  public :
      CArima(void);
      CArima(const uint p,const uint d, const uint q,bool use_const_term=true);
     ~CArima(void);
     
      uint   GetMaxArLag(void)    { if(m_ar_order) return m_arlags[ArrayMaximum(m_arlags)]; else return 0;}
      uint   GetMinArLag(void)    { if(m_ar_order) return m_arlags[ArrayMinimum(m_arlags)]; else return 0;}
      uint   GetMaxMaLag(void)    { if(m_ma_order) return m_malags[ArrayMaximum(m_malags)]; else return 0;}
      uint   GetMinMaLag(void)    { if(m_ma_order) return m_malags[ArrayMinimum(m_malags)]; else return 0;}
      uint   GetArOrder(void)     { return m_ar_order;  }
      uint   GetMaOrder(void)     { return m_ma_order;  }
      uint   GetDiffOrder(void)   { return m_diff_order;}
      bool   IsTrained(void)      { return m_istrained; }
      double GetSSE(void)         { return m_sse;      }
      uint   GetArLagAt(const uint shift);
      uint   GetMaLagAt(const uint shift);
      
      
      bool   SetARLags(uint &ar_lags[]);
      bool   SetMALags(uint &ma_lags[]);
      bool   Fit(double &input_series[]);
      bool   Fit(double &input_series[],const uint p,const uint d, const uint q,bool use_const_term=true);
      string Summary(void);
      void   GetModelParameters(double &out_array[]) { ArrayCopy(out_array,m_model);      }
      void   GetModelInnovation(double &out_array[]) { ArrayCopy(out_array,m_innovation); } 
      
};

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CArima::CArima(void)
{
  m_ar_order=1;
  //--- 
  ArrayResize(m_arlags,m_ar_order);
  for(uint i=0;i<m_ar_order;i++)
         m_arlags[i]=i+1;
  //---       
  m_ma_order=m_diff_order=0;
  m_istrained=false;
  m_const=true;
  
  ArrayResize(m_model,m_ar_order+m_ma_order+m_const);
  ArrayInitialize(m_model,0);
}         
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CArima::CArima(const uint p,const uint d,const uint q,bool use_const_term=true)
{
 m_ar_order=m_ma_order=m_diff_order=0;
 
 if(d)
    m_diff_order=d;
 if(p)
    {
     m_ar_order=p;
     ArrayResize(m_arlags,p);
     for(uint i=0;i<m_ar_order;i++)
        m_arlags[i]=i+1; 
    } 
 if(q)
   {
    m_ma_order=q;
    ArrayResize(m_malags,q);
    for(uint i=0;i<m_ma_order;i++)
        m_malags[i]=i+1; 
   } 
 
 
 m_istrained=false;
 m_const=use_const_term; 
 
 ArrayResize(m_model,m_ar_order+m_ma_order+m_const);
 ArrayInitialize(m_model,0);
} 
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CArima::~CArima(void)
{
  ArrayFree(m_differenced);
  ArrayFree(m_leads);
  ArrayFree(m_innovation);
  ArrayFree(m_model);
  ArrayFree(m_arlags);   
  ArrayFree(m_malags); 
 
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
uint CArima::GetMaLagAt(const uint shift)
{
 if(m_ma_order)
   return(m_malags[shift]);
 else
   return 0;  
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
uint CArima::GetArLagAt(const uint shift)
{
 if(m_ar_order)
   return(m_arlags[shift]);
 else
   return 0;
}      
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CArima::SetARLags(uint &ar_lags[])
{
 if(ArraySize(ar_lags)!=(int)m_ar_order)
    return false;
 //---   
 if(!ArraySort(ar_lags))
    return false;
 //---
  int found =ArrayBsearch(ar_lags,0);
  if(ar_lags[found]<1)
   return false;
 //---   
 ArrayCopy(m_arlags,ar_lags);   
 //---
 return true;

}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CArima::SetMALags(uint &ma_lags[])
{
 if(ArraySize(ma_lags)!=(int)m_ma_order)
    return false;
 //---   
 if(!ArraySort(ma_lags))
    return false;
 //---
 int found =ArrayBsearch(ma_lags,0);
  if(ma_lags[found]<1)
   return false;
 //---
 ArrayCopy(m_malags,ma_lags);   
 //---
 return true;
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CArima::difference(const uint difference_degree, double &data[], double &differenced[], double &leads[])
{
   int n=ArraySize(data);
   
   if(difference_degree)
      {
       ArrayResize(leads,difference_degree);
       ArrayResize(differenced,ArraySize(data));
       ArrayCopy(differenced,data);
      }
       
   for (uint pass=0 ; pass<difference_degree ; pass++) 
      {
        --n ;
      leads[pass] = differenced[0] ;   // Preserve first point for undo
      for (int i=0 ; i<n ; i++)
         {
          differenced[i] = differenced[i+1] - differenced[i] ;
         } 
      }
       
   if(difference_degree>0)
      ArrayRemove(differenced,n,ArraySize(data)-n);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CArima::integrate(double &differenced[],double &leads[],double &integrated[])
{
 int i, pass, passes ;
   double temp, sum;
   
   passes=ArraySize(leads);
   
   int n = ArraySize(differenced);
   
   ArrayResize(differenced,n+passes);
   
 
 for ( pass=0 ; pass<passes ; pass++) {
         sum = leads[passes-pass-1] ;
      for (i=0 ; i<n ; i++) {
         temp = differenced[i] ;
         differenced[i] = sum ;
         sum += temp ;
         }
      differenced[n++] = sum ;
      }
           
 ArrayCopy(integrated,differenced);     
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CArima::func(const double &p[])
{
 //Print( __FUNCTION__);
 double error=0;
 double sum_error=0;
 double pred=0;
 uint start_shift=(m_ar_order)?GetMaxArLag():(m_ma_order)?GetMaxMaLag():0;
 
 int p_shift;
 
 for(uint i=start_shift;i<(uint)ArraySize(m_differenced);i++)
  {
   p_shift=0;
   pred=0;
   if(m_const)
      pred+=p[p_shift++];
   for(uint j=0;j<m_ar_order;j++)
      pred+=p[p_shift++]*m_differenced[i-m_arlags[j]];
   for(uint k=0;i>=GetMaxMaLag()&& k<m_ma_order;k++)
       pred+=p[p_shift++]*m_innovation[i-m_malags[k]];
   error=pred-m_differenced[i];
   m_innovation[i]=error;
   error*=error;
   sum_error+=error;
  }  
 return sum_error;
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CArima::Fit(double &input_series[])
{
   
 uint input_size=ArraySize(input_series);
 
 uint in = m_ar_order+ (m_ma_order*2);
 
 if(input_size<=0 || input_size<in)
    return false;
     
 if(m_diff_order)
     difference(m_diff_order,input_series,m_differenced,m_leads); 
 else
     ArrayCopy(m_differenced,input_series);    
     
 ArrayResize(m_innovation,ArraySize(m_differenced));          
 
 double parameters[];
 ArrayResize(parameters,(m_const)?m_ar_order+m_ma_order+1:m_ar_order+m_ma_order);
 ArrayInitialize(parameters,0.0);
 
 int iterations = Optimize(parameters);
 
 if(iterations>0)
    m_istrained=true;
 else
    return false;   
 
 m_sse=PowellsMethod::GetFret();
 
 ArrayCopy(m_model,parameters);
 
 return true;   
 
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CArima::Fit(double&input_series[],const uint p,const uint d,const uint q,bool use_const_term=true)
{
 m_ar_order=m_ma_order=m_diff_order=0;
 
 if(d)
    m_diff_order=d;
 if(p)
    {
     m_ar_order=p;
     ArrayResize(m_arlags,p);
     for(uint i=0;i<m_ar_order;i++)
        m_arlags[i]=i+1; 
    } 
 if(q)
   {
    m_ma_order=q;
    ArrayResize(m_malags,q);
    for(uint i=0;i<m_ma_order;i++)
        m_malags[i]=i+1; 
   } 
   
 m_istrained=false;
 m_const=use_const_term; 
 
 ZeroMemory(m_innovation);
 ZeroMemory(m_model);
 ZeroMemory(m_differenced);
    
 ArrayResize(m_model,m_ar_order+m_ma_order+m_const);
 ArrayInitialize(m_model,0);
    
 return Fit(input_series);
    
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
string CArima::Summary(void)
{
      string print = "Arima("+IntegerToString(m_ar_order)+","+IntegerToString(m_diff_order)+","+IntegerToString(m_ma_order)+")\n";
      print+= "SSE : "+string(m_sse);
      
      int k=0;
      
      if(m_const)
        print+="\nConstant: "+string(m_model[k++]);
      for(uint i=0;i<m_ar_order;i++)
        print+="\nAR coefficient at lag "+IntegerToString(m_arlags[i])+": "+string(m_model[k++]);
      for(uint j=0;j<m_ma_order;j++)
        print+="\nMA coefficient at lag "+IntegerToString(m_malags[j])+": "+string(m_model[k++]);   
     
   return print;       

}
