//+------------------------------------------------------------------+
//|                                                      Entropy.mqh |
//|                        Copyright 2023, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Software Corp."
#property link      "https://www.mql5.com"
#include <Graphics\Graphic.mqh>
#include <Math\Stat\Math.mqh>


enum ENUM_TRANSFORM
  {
   TRANSFORM_SQRT=0,//square root
   TRANSFORM_CUBRT, //cube root
   TRANSFORM_TANH,  //tan
   TRANSFORM_LOG,   //log
   TRANSFORM_LOGISTIC,//logistic
   TRANSFORM_EXTREME //extreme
  };


//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double relativeEntroy(int bins, double &indicator[])
  {
   int d_size=ArraySize(indicator);

   if(d_size<100)
     {
      Print("Error: Sample size is insufficient");
      return -1;
     }

   if(bins<2)
     {
      double _bins=(double(d_size)/1000.0)*50.0;
      bins=(int(_bins)%2>0)?int(_bins):int(_bins)+1;
     }
   int count[];

   ArrayResize(count,bins);

   double minval, maxval, multiple, probability, sum;

   minval=indicator[ArrayMinimum(indicator)];
   maxval=indicator[ArrayMaximum(indicator)];


   multiple =(bins-(1e-10))/(maxval-minval+(1e-60));

   ArrayInitialize(count,0);

   int index=0;

   for(int i=0; i<d_size; i++)
     {
      index=(int)(multiple*(indicator[i] - minval));
      if(index<0 || index>=bins)
        {
         Print("Critical error: Index out of bounds at line ",__LINE__);
         ChartSetInteger(0,CHART_SHOW,true);
         return -1;
        }
      count[index]++;
     }

   sum=0;
   for(int i=0; i<bins; i++)
     {
      if(count[i])
        {
         probability=double(count[i])/double(d_size);
         sum+=probability*MathLog(probability);
        }
     }

   return -(sum/MathLog(double(bins)));
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void DrawIndicatorDistribution(const int displaytime_seconds,const bool compare_transform,const ENUM_TRANSFORM applied_transform,const string indicatorname,double &indicator[])
  {
   ChartSetInteger(0,CHART_SHOW,false);

   long chart=0;
   string name="IndicatorAnalysis";

   int d_size=ArraySize(indicator);

   double _bins=(double(d_size)/1000.0)*50.0;
   int bins=(int(_bins)%2>0)?int(_bins):int(_bins)+1;

   double y[];
   double x[];
   double y2[];
   double x2[];
   double m_indicator[];
   double mean,variance,skewness,kurtosis,before_entropy,after_entropy;

   ArrayResize(y,bins);
   ArrayResize(x,bins);

   before_entropy=relativeEntroy(0,indicator);

   if(before_entropy<0)
     {
      ChartSetInteger(0,CHART_SHOW,true);
      return;
     }

   if(compare_transform)
      BoostEntropy(0,applied_transform,indicator,m_indicator);
   else
      MathMoments(indicator,mean,variance,skewness,kurtosis);

   ArraySort(indicator);

   double multiple;

   multiple = (bins-(1e-10))/(indicator[d_size-1]-indicator[0]+(1e-60));

   ArrayInitialize(y,0);

   int index=0,p_index=0;

   for(int i=0; i<d_size; i++)
     {
      index=(int)(multiple*(indicator[i] - indicator[0]));
      if(index<0 || index>=bins)
        {
         Print("Critical error: Index out of bounds at line ",__LINE__);
         ChartSetInteger(0,CHART_SHOW,true);
         return;
        }
      y[index]++;
     }

   for(int i=0; i<bins; i++)
     {
      x[i]=indicator[0]+(((indicator[d_size-1]-indicator[0])/double(bins))*(i+0.5));
     }




   if(compare_transform)
     {
      ArrayResize(y2,bins);

      after_entropy=relativeEntroy(0,m_indicator);

      if(before_entropy<0)
        {
         ChartSetInteger(0,CHART_SHOW,true);
         return;
        }

      ArraySort(m_indicator);

      ArrayInitialize(y2,0);

      ArrayResize(x2,bins);

      multiple = (bins-(1e-10))/(m_indicator[d_size-1]-m_indicator[0]+(1e-60));

      for(int i=0; i<d_size; i++)
        {
         index=(int)(multiple*(m_indicator[i] - m_indicator[0]));
         if(index<0 || index>=bins)
           {
            Print("Critical error: Index out of bounds at line ",__LINE__);
            ChartSetInteger(0,CHART_SHOW,true);
            return;
           }
         y2[index]++;
        }

      for(int i=0; i<bins; i++)
        {
         x2[i]=m_indicator[0]+(((m_indicator[d_size-1]-m_indicator[0])/double(bins))*(i+0.5));
        }

     }

   CGraphic graphic;
   if(ObjectFind(chart,name)<0)
      graphic.Create(chart,name,0,0,0,780,380);
   else
      graphic.Attach(chart,name);
   if(!compare_transform)
      graphic.BackgroundMain(StringFormat("Indicator Distribution: Mean=%G Variance=%G Skewness=%G Kurtosis=%G Entropy=%G",mean,variance,skewness,kurtosis,before_entropy));
   else
      graphic.BackgroundMain(StringFormat("Indicator Distribution: Entropy Before=%G Entropy After=%G",before_entropy,after_entropy));
   graphic.BackgroundMainSize(16);
//---
   graphic.CurveAdd(x,y,ColorToARGB(clrBlue),CURVE_HISTOGRAM,indicatorname).HistogramWidth(6);

   if(compare_transform)
      graphic.CurveAdd(x2,y2,ColorToARGB(clrRed),CURVE_LINES,"Transformed "+indicatorname).LinesWidth(3);
//---
   graphic.CurvePlotAll();
//---
   graphic.Update();
   Sleep(displaytime_seconds*1000);
   ChartSetInteger(0,CHART_SHOW,true);
   graphic.Destroy();
   ChartRedraw();
  }


//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void logistic_transform(double &data_in[],double &data_out[])
  {
   int d_size=ArraySize(data_in);

   if(d_size<1)
     {
      Print("Error: Sample size is insufficient");
      return;
     }

   ArrayResize(data_out,d_size);

   for(int i=0; i<d_size; i++)
     {
      data_out[i]=(1/(1+MathExp(-data_in[i])))-0.5;
     }

  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void extreme_transform(double &data_in[], double&data_out[])
  {
   int d_size=ArraySize(data_in);

   if(d_size<1)
     {
      Print("Error: Sample size is insufficient");
      return;
     }
   ArrayResize(data_out,d_size);
   double sorted[];
   ArrayResize(sorted,d_size);

   ArrayCopy(sorted,data_in);

   ArraySort(sorted);

   int index;

   for(int i=0; i<d_size; i++)
     {
      index=ArrayBsearch(sorted,data_in[i]);

      data_out[i]=(double(index+1)/double(d_size)) - 0.5;

     }

   return;

  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void BoostEntropy(const double percent,const ENUM_TRANSFORM function,double &data[], double &result[])
  {
   switch(function)
     {
      case TRANSFORM_TANH:
         MathTanh(data,result);
         return;
      case TRANSFORM_SQRT:
         MathSqrt(data,result);
         return;
      case TRANSFORM_CUBRT:
         MathPow(data,(1.0/3.0),result);
         return;
      case TRANSFORM_LOG:
         MathLog(data,result);
         return;
      case TRANSFORM_LOGISTIC:
         logistic_transform(data,result);
         return;
      case TRANSFORM_EXTREME:
         extreme_transform(data,result);
         return;
     }
  }               
//+------------------------------------------------------------------+
