//+------------------------------------------------------------------+
//| FILE:    Include/DeeFX/CExecutionSafety.mqh                      |
//| BRAND:   DeeFX Precision Labs                                    |
//| PRODUCT: LagShield Pro                                           |
//| VERSION: v1.2                                                    |
//| DATE:    2026-02-24                                              |
//| TYPE:    Include — Execution Safety Validator                    |
//| DESC:    Evaluates terminal ping + measured execution delay.      |
//|          Returns false if combined latency exceeds threshold.    |
//|          Execution delay = wall-clock bracket around CTrade op.  |
//+------------------------------------------------------------------+
// ChangeLog:
//   v1.0 — Initial release.
//   v1.1 — Guard macro renamed CEXECUTIONSAFETY_MQH -> CEXEC_SAFETY_MQH.
//           %I64u format specifiers replaced with (string) casts.
//   v1.2 — Init() signature changed void → bool (standard #14 compliance).
//           Init() now resets m_lastExecDelayMs + m_hasExecSample — prevents stale reinit state.
//+------------------------------------------------------------------+
#ifndef CEXEC_SAFETY_MQH
#define CEXEC_SAFETY_MQH

class CExecutionSafety
  {
private:
   int               m_safetyThresholdMs;   // Max combined ping + exec delay (ms)
   ulong             m_lastExecDelayMs;      // Most recent measured execution delay
   bool              m_hasExecSample;        // True once >= 1 measurement recorded

public:
                     CExecutionSafety()
     : m_safetyThresholdMs(150),
       m_lastExecDelayMs(0),
       m_hasExecSample(false)
     {}

   bool              Init(int safetyThresholdMs)
     {
      m_safetyThresholdMs = safetyThresholdMs;
      m_lastExecDelayMs   = 0;       // Reset — persists across OnDeinit/OnInit cycles
      m_hasExecSample     = false;
      return true;
     }

   //--- Record execution delay after any CTrade operation
   //    Usage: ulong t0 = GetTickCount64();
   //           trade.PositionClose(...);
   //           safety.RecordExecDelay(GetTickCount64() - t0);
   void              RecordExecDelay(ulong delayMs)
     {
      m_lastExecDelayMs = delayMs;
      m_hasExecSample   = true;
     }

   //--- Core safety check — returns false if ping + exec delay exceeds threshold
   bool              CheckExecutionSafety() const
     {
      long ping = TerminalInfoInteger(TERMINAL_PING_LAST);

      //--- Disconnected guard: TERMINAL_PING_LAST returns -1 with no server contact
      if(ping < 0)
        {
         PrintFormat("DeeFX | Safety FAIL | Ping: %d (disconnected)", ping);
         return false;
        }

      ulong execDelay = m_hasExecSample ? m_lastExecDelayMs : 0;
      ulong combined  = (ulong)ping + execDelay;
      bool  safe      = (combined <= (ulong)m_safetyThresholdMs);

      if(!safe)
         PrintFormat("DeeFX | Safety FAIL | Ping: %s ms | ExecDelay: %s ms | Combined: %s ms | Threshold: %d ms",
                     (string)ping, (string)execDelay, (string)combined, m_safetyThresholdMs);

      return safe;
     }

   //--- Accessors for dashboard display and external checks
   long              GetPingMs()      const { return TerminalInfoInteger(TERMINAL_PING_LAST); }
   ulong             GetExecDelayMs() const { return m_lastExecDelayMs; }
   bool              HasExecSample()  const { return m_hasExecSample; }
   int               GetThresholdMs() const { return m_safetyThresholdMs; }

   ulong             GetCombinedMs()  const
     {
      long ping = TerminalInfoInteger(TERMINAL_PING_LAST);
      if(ping < 0) return 9999;
      return (ulong)ping + m_lastExecDelayMs;
     }
  };

#endif // CEXEC_SAFETY_MQH
//+------------------------------------------------------------------+
