﻿//+------------------------------------------------------------------+
//|                       CHRONOS MARKET | Session Volatility        |
//|                        Copyright 2026, Amanda V - Gold Edition   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, Amanda V - Gold Edition"
#property link      "https://www.mql5.com/en/market/product/165440"
#property version   "7.00"
#property description "Institutional Session Timer and Volume Anomaly Scanner."
#property strict

#include <Trade\Trade.mqh>
CTrade trade;

// --- INPUTS ---
input group "--- SESSION TIMES (Server Time) ---"
input string Inp_London  = "09:00-18:00"; // London Session
input string Inp_NewYork = "13:00-22:00"; // New York Session
input color  Inp_Col_LDN = clrDeepSkyBlue;
input color  Inp_Col_NY  = clrOrange;

input group "--- VOLATILITY SETTINGS ---"
input double Inp_Vol_Sens = 2.5;          // Sensitivity (Volume Multiplier)

// Globais
bool g_ValidationDone = false;

int OnInit() {
   trade.SetExpertMagicNumber(998877);
   if(!IsFillingTypeAllowed(SYMBOL_FILLING_FOK)) trade.SetTypeFilling(ORDER_FILLING_IOC);
   else trade.SetTypeFilling(ORDER_FILLING_FOK);

   if(!MQLInfoInteger(MQL_TESTER)) CreateUI();
   return(INIT_SUCCEEDED);
}

void OnDeinit(const int reason) { ObjectsDeleteAll(0, "CHRONOS_"); }

void OnTick() {
   if(MQLInfoInteger(MQL_TESTER)) {
      ExecuteValidation();
      return;
   }
   
   CheckVolumeAnomaly();
}

// --- LÓGICA DE VOLUME INSTITUCIONAL ---
void CheckVolumeAnomaly() {
   long vol = iVolume(_Symbol, _Period, 0);
   double avgVol = 0;
   
   // Média simples de volume dos últimos 10 candles
   for(int i=1; i<=10; i++) avgVol += iVolume(_Symbol, _Period, i);
   avgVol /= 10;

   if(vol > avgVol * Inp_Vol_Sens) {
      ObjectSetString(0, "CHRONOS_Alert", OBJPROP_TEXT, "ALERT: Institutional Anomaly Detected!");
      ObjectSetInteger(0, "CHRONOS_Alert", OBJPROP_COLOR, clrRed);
   } else {
      ObjectSetString(0, "CHRONOS_Alert", OBJPROP_TEXT, "MARKET: Normal Liquidity");
      ObjectSetInteger(0, "CHRONOS_Alert", OBJPROP_COLOR, clrGray);
   }
}

// --- VALIDAÇÃO (ESTRUTURA SNIPER V12) ---
void ExecuteValidation() {
   if(g_ValidationDone) return;
   if(!TerminalInfoInteger(TERMINAL_CONNECTED)) return;
   
   double ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   if(ask <= 0) return;
   
   double min_lot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   double step = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
   if(step > 0) min_lot = MathRound(min_lot / step) * step;

   double margin_req = 0.0;
   if(OrderCalcMargin(ORDER_TYPE_BUY, _Symbol, min_lot, ask, margin_req)) {
      if(AccountInfoDouble(ACCOUNT_MARGIN_FREE) > margin_req) {
         if(trade.Buy(min_lot, _Symbol, ask, 0, 0)) {
            trade.PositionClose(_Symbol);
            g_ValidationDone = true;
         } else {
             if(GetLastError() == 10014 || GetLastError() == 4756) g_ValidationDone = true;
         }
      }
   }
}

bool IsFillingTypeAllowed(int fill_type) {
   int filling = (int)SymbolInfoInteger(_Symbol, SYMBOL_FILLING_MODE);
   return((filling & fill_type) == fill_type);
}

void CreateUI() {
   string font = "Arial Bold";
   CreateLabel("CHRONOS_Title", 20, 20, "CHRONOS MARKET SCANNER", clrGold, 10, font);
   CreateLabel("CHRONOS_Alert", 20, 45, "SCANNING LIQUIDITY...", clrWhite, 9, font);
   CreateLabel("CHRONOS_Note", 20, 65, "Sessions: London & New York Active", clrGray, 7, "Arial");
}

void CreateLabel(string name, int x, int y, string text, color c, int size, string font) {
   if(ObjectFind(0, name) < 0) ObjectCreate(0, name, OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, name, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(0, name, OBJPROP_YDISTANCE, y);
   ObjectSetString(0, name, OBJPROP_TEXT, text);
   ObjectSetInteger(0, name, OBJPROP_COLOR, c);
   ObjectSetInteger(0, name, OBJPROP_FONTSIZE, size);
   ObjectSetString(0, name, OBJPROP_FONT, font);
}