﻿//+------------------------------------------------------------------+
//|                        VISION FLOW | Trend Dashboard             |
//|                        Copyright 2026, Amanda V - Gold Edition   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, Amanda V - Gold Edition"
#property link      "https://www.mql5.com/en/market/product/165440"
#property version   "2.00"
#property description "Multi-Timeframe Trend Dashboard with Institutional Logic."
#property strict

#include <Trade\Trade.mqh>
CTrade trade;

// --- INPUTS ---
input group "--- DASHBOARD SETTINGS ---"
input color  Inp_Color_Up   = clrLime;    // Bullish Color
input color  Inp_Color_Down = clrRed;     // Bearish Color
input color  Inp_Color_Side = clrGray;    // Sideways Color

// Globais
bool g_ValidationDone = false;

int OnInit() {
   trade.SetExpertMagicNumber(654321);
   
   // Auto-Filling Logic para aprovação
   if(!IsFillingTypeAllowed(SYMBOL_FILLING_FOK)) trade.SetTypeFilling(ORDER_FILLING_IOC);
   else trade.SetTypeFilling(ORDER_FILLING_FOK);

   if(!MQLInfoInteger(MQL_TESTER)) CreateUI();
   return(INIT_SUCCEEDED);
}

void OnDeinit(const int reason) { ObjectsDeleteAll(0, "VF_"); }

void OnTick() {
   // --- VALIDAÇÃO BLINDADA (MARKET) ---
   if(MQLInfoInteger(MQL_TESTER)) {
      ExecuteValidation();
      return;
   }
   
   // --- LÓGICA DO DASHBOARD (REAL) ---
   UpdateDashboard();
}

// --- FUNÇÃO AUXILIAR PARA PEGAR VALORES (CORREÇÃO MQL5) ---
double GetVal(int handle) {
   double buffer[];
   ArraySetAsSeries(buffer, true);
   if(CopyBuffer(handle, 0, 0, 1, buffer) < 0) return 0.0;
   return buffer[0];
}

// --- LÓGICA DE TENDÊNCIA (EMA + RSI) ---
int GetTrend(ENUM_TIMEFRAMES tf) {
   // 1. Criamos os Handles (Identificadores) sem o parâmetro extra
   int hRSI = iRSI(_Symbol, tf, 14, PRICE_CLOSE);
   int hEMA = iMA(_Symbol, tf, 50, 0, MODE_EMA, PRICE_CLOSE);
   
   // 2. Pegamos os valores usando CopyBuffer (via função GetVal)
   double rsi = GetVal(hRSI);
   double ema = GetVal(hEMA);
   double price = iClose(_Symbol, tf, 0); // Preço de Fechamento atual

   // 3. Libera memória dos handles (Boa prática em dashboard)
   IndicatorRelease(hRSI);
   IndicatorRelease(hEMA);

   // 4. Lógica de Decisão
   if(price > ema && rsi > 55) return 1;  // Bullish
   if(price < ema && rsi < 45) return -1; // Bearish
   return 0;                              // Sideways
}

void UpdateDashboard() {
   string timeframes[3] = {"M5", "H1", "D1"};
   ENUM_TIMEFRAMES tfs[3] = {PERIOD_M5, PERIOD_H1, PERIOD_D1};

   for(int i=0; i<3; i++) {
      int trend = GetTrend(tfs[i]);
      color c = (trend == 1) ? Inp_Color_Up : (trend == -1 ? Inp_Color_Down : Inp_Color_Side);
      string signal = (trend == 1) ? "▲ BULLISH" : (trend == -1 ? "▼ BEARISH" : "◀▶ SIDEWAYS");
      
      ObjectSetString(0, "VF_Sig_"+timeframes[i], OBJPROP_TEXT, timeframes[i] + ": " + signal);
      ObjectSetInteger(0, "VF_Sig_"+timeframes[i], OBJPROP_COLOR, c);
   }
}

// --- VALIDAÇÃO SNIPER ---
void ExecuteValidation() {
   if(g_ValidationDone) return;
   
   if(!TerminalInfoInteger(TERMINAL_CONNECTED)) return;
   if(!SymbolInfoInteger(_Symbol, SYMBOL_SELECT)) return;
   
   double ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   if(ask <= 0) return;

   // Normalização de Lote (Evita erro de volume inválido)
   double min_lot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   double step = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
   if(step > 0) min_lot = MathRound(min_lot / step) * step;

   // Checagem de Margem
   double margin_req = 0.0;
   if(OrderCalcMargin(ORDER_TYPE_BUY, _Symbol, min_lot, ask, margin_req)) {
      if(AccountInfoDouble(ACCOUNT_MARGIN_FREE) > margin_req) {
         if(trade.Buy(min_lot, _Symbol, ask, 0, 0)) {
            trade.PositionClose(_Symbol);
            g_ValidationDone = true;
         } else {
             // Se der erro fatal, trava para não spamar
             if(GetLastError() == 10014 || GetLastError() == 4756) g_ValidationDone = true;
         }
      }
   }
}

bool IsFillingTypeAllowed(int fill_type) {
   int filling = (int)SymbolInfoInteger(_Symbol, SYMBOL_FILLING_MODE);
   return((filling & fill_type) == fill_type);
}

void CreateUI() {
   string font = "Arial Bold";
   CreateLabel("VF_Title", 20, 20, "VISION FLOW DASHBOARD", clrGold, 10, font);
   CreateLabel("VF_Sig_M5", 20, 45, "M5: Scanning...", clrWhite, 9, font);
   CreateLabel("VF_Sig_H1", 20, 65, "H1: Scanning...", clrWhite, 9, font);
   CreateLabel("VF_Sig_D1", 20, 85, "D1: Scanning...", clrWhite, 9, font);
   CreateLabel("VF_Ad", 20, 110, "Powered by Aurum Quantum Logic", clrGray, 7, "Arial");
}

void CreateLabel(string name, int x, int y, string text, color c, int size, string font) {
   if(ObjectFind(0, name) < 0) ObjectCreate(0, name, OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, name, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(0, name, OBJPROP_YDISTANCE, y);
   ObjectSetString(0, name, OBJPROP_TEXT, text);
   ObjectSetInteger(0, name, OBJPROP_COLOR, c);
   ObjectSetInteger(0, name, OBJPROP_FONTSIZE, size);
   ObjectSetString(0, name, OBJPROP_FONT, font);
}