//+------------------------------------------------------------------+
//|                                    Quantum Neural Calculator.mq5 |
//|                        Copyright 2026, Amanda V - Gold Edition   |
//|                        https://www.mql5.com/en/users/kayruyuta   |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, Amanda V - Gold Edition"
#property link      "https://www.mql5.com/en/market/product/165440"
#property version   "1.00"
#property description "Institutional Risk Calculator with ATR Volatility Analytics."
#property strict

#include <Trade\Trade.mqh>
CTrade trade;

// --- INPUTS ---
input group "--- QUANTUM SETTINGS ---"
input double Inp_RiskAmount = 50.0;     // Risk in USD ($)
input int    Inp_StopLoss   = 500;      // Stop Loss (Points)
input color  Inp_UI_Color   = clrGold;  // UI Theme Color

// Global Variables
int handleATR;
double atrBuffer[];

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   handleATR = iATR(_Symbol, _Period, 14);
   CreateUI();
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   ObjectsDeleteAll(0, "QNC_");
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   UpdateCalculator();
}

//+------------------------------------------------------------------+
//| Main Logic                                                       |
//+------------------------------------------------------------------+
void UpdateCalculator()
{
   double tickValue = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_VALUE);
   if(tickValue <= 0) return;

   // 1. Calculate Position Size
   double lot = Inp_RiskAmount / (Inp_StopLoss * tickValue);
   
   // 2. Normalize Lot Size
   double step = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
   double min_vol = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   
   if(step > 0) lot = MathFloor(lot / step) * step;
   if(lot < min_vol) lot = min_vol;

   // 3. Volatility Analytics (ATR)
   ArraySetAsSeries(atrBuffer, true);
   if(CopyBuffer(handleATR, 0, 0, 1, atrBuffer) > 0)
   {
      double range = iHigh(_Symbol, _Period, 1) - iLow(_Symbol, _Period, 1);
      string status = "STABLE";
      color statusColor = clrLime;
      
      // If current ATR is unusually high compared to recent range
      if(atrBuffer[0] > range * 0.6)
      {
         status = "HIGH VOLATILITY";
         statusColor = clrRed;
      }
      
      ObjectSetString(0, "QNC_Vol", OBJPROP_TEXT, "MARKET STATUS: " + status);
      ObjectSetInteger(0, "QNC_Vol", OBJPROP_COLOR, statusColor);
   }

   // 4. Update Dashboard
   ObjectSetString(0, "QNC_Lot", OBJPROP_TEXT, "LOT SIZE: " + DoubleToString(lot, 2));
   ObjectSetString(0, "QNC_Risk", OBJPROP_TEXT, "Risk: $" + DoubleToString(Inp_RiskAmount, 2) + " (SL: " + IntegerToString(Inp_StopLoss) + "pts)");
}

//+------------------------------------------------------------------+
//| UI Generation                                                    |
//+------------------------------------------------------------------+
void CreateUI()
{
   string font = "Arial Bold";
   CreateLabel("QNC_Title", 20, 20, "QUANTUM CALCULATOR (OPEN SOURCE)", Inp_UI_Color, 10, font);
   CreateLabel("QNC_Risk", 20, 40, "Initializing...", clrWhite, 8, "Arial");
   CreateLabel("QNC_Lot", 20, 60, "WAITING FOR TICK...", clrWhite, 10, font);
   CreateLabel("QNC_Vol", 20, 80, "Scanning...", clrGray, 8, "Arial");
}

void CreateLabel(string name, int x, int y, string text, color c, int size, string font)
{
   if(ObjectFind(0, name) < 0) ObjectCreate(0, name, OBJ_LABEL, 0, 0, 0);
   ObjectSetInteger(0, name, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(0, name, OBJPROP_YDISTANCE, y);
   ObjectSetInteger(0, name, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetString(0, name, OBJPROP_TEXT, text);
   ObjectSetInteger(0, name, OBJPROP_COLOR, c);
   ObjectSetInteger(0, name, OBJPROP_FONTSIZE, size);
   ObjectSetString(0, name, OBJPROP_FONT, font);
}