//+------------------------------------------------------------------+
//|                                        SessionTimeFilterDemo.mq5 |
//|                                        Copyright 2026, Algosphere |
//|                                      https://algosphere-quant.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, Algosphere"
#property link      "https://algosphere-quant.com"
#property version   "1.00"
#property description "Demo indicator for SessionTimeFilter Library"
#property description "Displays current trading session status on chart panel"
#property indicator_chart_window
#property indicator_plots 0

//+------------------------------------------------------------------+
//| Includes                                                         |
//+------------------------------------------------------------------+
#include "SessionTimeFilter.mqh"

//+------------------------------------------------------------------+
//| Input parameters                                                 |
//+------------------------------------------------------------------+
input group           "=== Display Settings ==="
input int             InpPanelX=10;                 // Panel X Position
input int             InpPanelY=30;                 // Panel Y Position
input color           InpBackgroundColor=C'32,32,32';  // Background Color
input color           InpTextColor=clrWhite;       // Text Color
input color           InpActiveColor=clrLime;      // Active Session Color
input color           InpInactiveColor=clrGray;    // Inactive Session Color

input group           "=== Session Settings ==="
input int             InpGMTOffset=0;              // Broker GMT Offset

//+------------------------------------------------------------------+
//| Defines                                                          |
//+------------------------------------------------------------------+
#define PREFIX        "STF_"
#define PANEL_WIDTH   195
#define PANEL_HEIGHT  175
#define ROW_SPACING   22

//+------------------------------------------------------------------+
//| Global variables                                                 |
//+------------------------------------------------------------------+
CSessionTimeFilter    g_session_filter;

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
  {
//--- Initialize session filter
   if(!g_session_filter.Init(InpGMTOffset))
     {
      Print("Error: Failed to initialize session filter");
      return(INIT_FAILED);
     }

//--- Create visual panel
   CreatePanel();
   UpdatePanel();

//--- Set timer for updates
   EventSetTimer(1);

   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//| Custom indicator deinitialization function                       |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//--- Kill timer
   EventKillTimer();

//--- Delete all panel objects
   ObjectsDeleteAll(0,PREFIX);
  }

//+------------------------------------------------------------------+
//| Timer function                                                   |
//+------------------------------------------------------------------+
void OnTimer()
  {
   UpdatePanel();
  }

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
  {
   return(rates_total);
  }

//+------------------------------------------------------------------+
//| Create the information panel                                     |
//+------------------------------------------------------------------+
void CreatePanel()
  {
   int x=InpPanelX;
   int y=InpPanelY;

//--- Create background rectangle
   CreateRectangleLabel(PREFIX+"Background",x,y,PANEL_WIDTH,PANEL_HEIGHT,
                        InpBackgroundColor,InpBackgroundColor);

//--- Create title label
   CreateTextLabel(PREFIX+"Title",x+10,y+8,"SESSION MONITOR",
                   InpTextColor,11,"Arial Bold");

//--- Create separator line
   CreateRectangleLabel(PREFIX+"Separator1",x+10,y+30,PANEL_WIDTH-20,1,
                        clrDimGray,clrDimGray);

//--- Create session status labels
   int row_y=y+40;

   CreateTextLabel(PREFIX+"LabelSydney",x+15,row_y,"Sydney:",InpTextColor,9,"Arial");
   CreateTextLabel(PREFIX+"ValueSydney",x+100,row_y,"---",InpInactiveColor,9,"Arial Bold");
   row_y+=ROW_SPACING;

   CreateTextLabel(PREFIX+"LabelTokyo",x+15,row_y,"Tokyo:",InpTextColor,9,"Arial");
   CreateTextLabel(PREFIX+"ValueTokyo",x+100,row_y,"---",InpInactiveColor,9,"Arial Bold");
   row_y+=ROW_SPACING;

   CreateTextLabel(PREFIX+"LabelLondon",x+15,row_y,"London:",InpTextColor,9,"Arial");
   CreateTextLabel(PREFIX+"ValueLondon",x+100,row_y,"---",InpInactiveColor,9,"Arial Bold");
   row_y+=ROW_SPACING;

   CreateTextLabel(PREFIX+"LabelNewYork",x+15,row_y,"New York:",InpTextColor,9,"Arial");
   CreateTextLabel(PREFIX+"ValueNewYork",x+100,row_y,"---",InpInactiveColor,9,"Arial Bold");
   row_y+=ROW_SPACING;

//--- Create second separator
   CreateRectangleLabel(PREFIX+"Separator2",x+10,row_y+5,PANEL_WIDTH-20,1,
                        clrDimGray,clrDimGray);
   row_y+=15;

//--- Create overlap indicator
   CreateTextLabel(PREFIX+"LabelOverlap",x+15,row_y,"LON-NY Overlap:",InpTextColor,9,"Arial");
   CreateTextLabel(PREFIX+"ValueOverlap",x+120,row_y,"---",InpInactiveColor,9,"Arial Bold");
   row_y+=ROW_SPACING;

//--- Create server time display
   CreateTextLabel(PREFIX+"LabelTime",x+15,row_y,"Server:",InpTextColor,9,"Arial");
   CreateTextLabel(PREFIX+"ValueTime",x+70,row_y,"00:00:00",InpTextColor,9,"Consolas");

//--- Redraw chart
   ChartRedraw();
  }

//+------------------------------------------------------------------+
//| Update the panel with current session status                     |
//+------------------------------------------------------------------+
void UpdatePanel()
  {
//--- Update Sydney session status
   bool is_sydney=g_session_filter.IsInSession(SESSION_SYDNEY);
   ObjectSetString(0,PREFIX+"ValueSydney",OBJPROP_TEXT,is_sydney ? "ACTIVE" : "Closed");
   ObjectSetInteger(0,PREFIX+"ValueSydney",OBJPROP_COLOR,is_sydney ? InpActiveColor : InpInactiveColor);

//--- Update Tokyo session status
   bool is_tokyo=g_session_filter.IsInSession(SESSION_TOKYO);
   ObjectSetString(0,PREFIX+"ValueTokyo",OBJPROP_TEXT,is_tokyo ? "ACTIVE" : "Closed");
   ObjectSetInteger(0,PREFIX+"ValueTokyo",OBJPROP_COLOR,is_tokyo ? InpActiveColor : InpInactiveColor);

//--- Update London session status
   bool is_london=g_session_filter.IsInSession(SESSION_LONDON);
   ObjectSetString(0,PREFIX+"ValueLondon",OBJPROP_TEXT,is_london ? "ACTIVE" : "Closed");
   ObjectSetInteger(0,PREFIX+"ValueLondon",OBJPROP_COLOR,is_london ? InpActiveColor : InpInactiveColor);

//--- Update New York session status
   bool is_newyork=g_session_filter.IsInSession(SESSION_NEWYORK);
   ObjectSetString(0,PREFIX+"ValueNewYork",OBJPROP_TEXT,is_newyork ? "ACTIVE" : "Closed");
   ObjectSetInteger(0,PREFIX+"ValueNewYork",OBJPROP_COLOR,is_newyork ? InpActiveColor : InpInactiveColor);

//--- Update London-NY overlap status
   bool is_overlap=g_session_filter.IsInSession(SESSION_OVERLAP);
   ObjectSetString(0,PREFIX+"ValueOverlap",OBJPROP_TEXT,is_overlap ? "YES" : "No");
   ObjectSetInteger(0,PREFIX+"ValueOverlap",OBJPROP_COLOR,is_overlap ? clrOrange : InpInactiveColor);

//--- Update server time
   MqlDateTime dt;
   TimeToStruct(TimeCurrent(),dt);
   string time_string=StringFormat("%02d:%02d:%02d",dt.hour,dt.min,dt.sec);
   ObjectSetString(0,PREFIX+"ValueTime",OBJPROP_TEXT,time_string);

//--- Redraw chart
   ChartRedraw();
  }

//+------------------------------------------------------------------+
//| Create a text label object                                       |
//+------------------------------------------------------------------+
void CreateTextLabel(const string name,
                     const int x,
                     const int y,
                     const string text,
                     const color clr,
                     const int font_size,
                     const string font_name)
  {
//--- Delete existing object if present
   if(ObjectFind(0,name)>=0)
      ObjectDelete(0,name);

//--- Create label object
   if(!ObjectCreate(0,name,OBJ_LABEL,0,0,0))
     {
      Print("Error creating label: ",name);
      return;
     }

//--- Set label properties
   ObjectSetInteger(0,name,OBJPROP_XDISTANCE,x);
   ObjectSetInteger(0,name,OBJPROP_YDISTANCE,y);
   ObjectSetInteger(0,name,OBJPROP_CORNER,CORNER_LEFT_UPPER);
   ObjectSetString(0,name,OBJPROP_TEXT,text);
   ObjectSetInteger(0,name,OBJPROP_COLOR,clr);
   ObjectSetInteger(0,name,OBJPROP_FONTSIZE,font_size);
   ObjectSetString(0,name,OBJPROP_FONT,font_name);
   ObjectSetInteger(0,name,OBJPROP_SELECTABLE,false);
   ObjectSetInteger(0,name,OBJPROP_HIDDEN,true);
  }

//+------------------------------------------------------------------+
//| Create a rectangle label object                                  |
//+------------------------------------------------------------------+
void CreateRectangleLabel(const string name,
                          const int x,
                          const int y,
                          const int width,
                          const int height,
                          const color bg_color,
                          const color border_color)
  {
//--- Delete existing object if present
   if(ObjectFind(0,name)>=0)
      ObjectDelete(0,name);

//--- Create rectangle object
   if(!ObjectCreate(0,name,OBJ_RECTANGLE_LABEL,0,0,0))
     {
      Print("Error creating rectangle: ",name);
      return;
     }

//--- Set rectangle properties
   ObjectSetInteger(0,name,OBJPROP_XDISTANCE,x);
   ObjectSetInteger(0,name,OBJPROP_YDISTANCE,y);
   ObjectSetInteger(0,name,OBJPROP_XSIZE,width);
   ObjectSetInteger(0,name,OBJPROP_YSIZE,height);
   ObjectSetInteger(0,name,OBJPROP_CORNER,CORNER_LEFT_UPPER);
   ObjectSetInteger(0,name,OBJPROP_BGCOLOR,bg_color);
   ObjectSetInteger(0,name,OBJPROP_BORDER_COLOR,border_color);
   ObjectSetInteger(0,name,OBJPROP_BORDER_TYPE,BORDER_FLAT);
   ObjectSetInteger(0,name,OBJPROP_SELECTABLE,false);
   ObjectSetInteger(0,name,OBJPROP_HIDDEN,true);
  }
//+------------------------------------------------------------------+
