//+------------------------------------------------------------------+
//|                                            SessionTimeFilter.mqh |
//|                                        Copyright 2026, Algosphere |
//|                                      https://algosphere-quant.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, Algosphere"
#property link      "https://algosphere-quant.com"
#property version   "1.00"
#property strict

//+------------------------------------------------------------------+
//| Defines                                                          |
//+------------------------------------------------------------------+
#define SESSION_MINUTES_PER_DAY    1440

//+------------------------------------------------------------------+
//| Enumeration of trading sessions                                  |
//+------------------------------------------------------------------+
enum ENUM_TRADING_SESSION
  {
   SESSION_SYDNEY=0,    // Sydney Session
   SESSION_TOKYO=1,     // Tokyo Session
   SESSION_LONDON=2,    // London Session
   SESSION_NEWYORK=3,   // New York Session
   SESSION_OVERLAP=4,   // London-NY Overlap
   SESSION_CUSTOM=5     // Custom Session
  };

//+------------------------------------------------------------------+
//| Structure for session time boundaries                            |
//+------------------------------------------------------------------+
struct SSessionTime
  {
   int               start_hour;      // Session start hour (0-23)
   int               start_minute;    // Session start minute (0-59)
   int               end_hour;        // Session end hour (0-23)
   int               end_minute;      // Session end minute (0-59)
   string            name;            // Session display name
  };

//+------------------------------------------------------------------+
//| Class for filtering trades by trading session                    |
//+------------------------------------------------------------------+
class CSessionTimeFilter
  {
private:
   SSessionTime      m_sessions[6];   // Array of session definitions
   int               m_gmt_offset;    // Broker GMT offset in hours
   bool              m_initialized;   // Initialization flag

   //--- Private methods
   void              InitDefaultSessions(void);
   int               TimeToMinutes(const int hour,const int minute);
   int               AdjustForGMT(const int minutes);

public:
   //--- Constructor and destructor
                     CSessionTimeFilter(void);
                    ~CSessionTimeFilter(void);

   //--- Initialization
   bool              Init(const int gmt_offset=0);

   //--- Session configuration
   void              SetCustomSession(const int start_hour,const int start_min,
                                      const int end_hour,const int end_min);

   //--- Session status methods
   bool              IsInSession(const ENUM_TRADING_SESSION session);
   bool              IsTimeInSession(const datetime time,const ENUM_TRADING_SESSION session);
   bool              IsInAnySessions(const bool sydney,const bool tokyo,
                                     const bool london,const bool newyork);

   //--- Information methods
   string            GetActiveSession(void);
   string            GetSessionName(const ENUM_TRADING_SESSION session);
   int               MinutesUntilSession(const ENUM_TRADING_SESSION session);
   int               MinutesRemainingInSession(const ENUM_TRADING_SESSION session);

   //--- Day of week methods
   bool              IsWeekday(void);
   bool              IsDayOfWeek(const int day);
  };

//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSessionTimeFilter::CSessionTimeFilter(void) : m_gmt_offset(0),
                                               m_initialized(false)
  {
   InitDefaultSessions();
  }

//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSessionTimeFilter::~CSessionTimeFilter(void)
  {
  }

//+------------------------------------------------------------------+
//| Initialize default session times (GMT)                           |
//+------------------------------------------------------------------+
void CSessionTimeFilter::InitDefaultSessions(void)
  {
//--- Sydney session: 21:00 - 06:00 GMT
   m_sessions[SESSION_SYDNEY].start_hour=21;
   m_sessions[SESSION_SYDNEY].start_minute=0;
   m_sessions[SESSION_SYDNEY].end_hour=6;
   m_sessions[SESSION_SYDNEY].end_minute=0;
   m_sessions[SESSION_SYDNEY].name="Sydney";

//--- Tokyo session: 00:00 - 09:00 GMT
   m_sessions[SESSION_TOKYO].start_hour=0;
   m_sessions[SESSION_TOKYO].start_minute=0;
   m_sessions[SESSION_TOKYO].end_hour=9;
   m_sessions[SESSION_TOKYO].end_minute=0;
   m_sessions[SESSION_TOKYO].name="Tokyo";

//--- London session: 07:00 - 16:00 GMT
   m_sessions[SESSION_LONDON].start_hour=7;
   m_sessions[SESSION_LONDON].start_minute=0;
   m_sessions[SESSION_LONDON].end_hour=16;
   m_sessions[SESSION_LONDON].end_minute=0;
   m_sessions[SESSION_LONDON].name="London";

//--- New York session: 12:00 - 21:00 GMT
   m_sessions[SESSION_NEWYORK].start_hour=12;
   m_sessions[SESSION_NEWYORK].start_minute=0;
   m_sessions[SESSION_NEWYORK].end_hour=21;
   m_sessions[SESSION_NEWYORK].end_minute=0;
   m_sessions[SESSION_NEWYORK].name="New York";

//--- London-NY overlap: 12:00 - 16:00 GMT
   m_sessions[SESSION_OVERLAP].start_hour=12;
   m_sessions[SESSION_OVERLAP].start_minute=0;
   m_sessions[SESSION_OVERLAP].end_hour=16;
   m_sessions[SESSION_OVERLAP].end_minute=0;
   m_sessions[SESSION_OVERLAP].name="LON-NY Overlap";

//--- Custom session: default same as London
   m_sessions[SESSION_CUSTOM].start_hour=7;
   m_sessions[SESSION_CUSTOM].start_minute=0;
   m_sessions[SESSION_CUSTOM].end_hour=16;
   m_sessions[SESSION_CUSTOM].end_minute=0;
   m_sessions[SESSION_CUSTOM].name="Custom";
  }

//+------------------------------------------------------------------+
//| Initialize the filter with GMT offset                            |
//+------------------------------------------------------------------+
bool CSessionTimeFilter::Init(const int gmt_offset=0)
  {
   m_gmt_offset=gmt_offset;
   m_initialized=true;
   return(true);
  }

//+------------------------------------------------------------------+
//| Convert hours and minutes to total minutes since midnight        |
//+------------------------------------------------------------------+
int CSessionTimeFilter::TimeToMinutes(const int hour,const int minute)
  {
   return(hour*60+minute);
  }

//+------------------------------------------------------------------+
//| Adjust minutes for GMT offset                                    |
//+------------------------------------------------------------------+
int CSessionTimeFilter::AdjustForGMT(const int minutes)
  {
   int adjusted=minutes+m_gmt_offset*60;
//--- Handle day wrap-around
   if(adjusted<0)
      adjusted+=SESSION_MINUTES_PER_DAY;
   if(adjusted>=SESSION_MINUTES_PER_DAY)
      adjusted-=SESSION_MINUTES_PER_DAY;
   return(adjusted);
  }

//+------------------------------------------------------------------+
//| Set custom session time boundaries                               |
//+------------------------------------------------------------------+
void CSessionTimeFilter::SetCustomSession(const int start_hour,const int start_min,
                                          const int end_hour,const int end_min)
  {
   m_sessions[SESSION_CUSTOM].start_hour=start_hour;
   m_sessions[SESSION_CUSTOM].start_minute=start_min;
   m_sessions[SESSION_CUSTOM].end_hour=end_hour;
   m_sessions[SESSION_CUSTOM].end_minute=end_min;
  }

//+------------------------------------------------------------------+
//| Check if current time is within the specified session            |
//+------------------------------------------------------------------+
bool CSessionTimeFilter::IsInSession(const ENUM_TRADING_SESSION session)
  {
//--- Ensure initialization
   if(!m_initialized)
      Init();

//--- Get current time
   MqlDateTime dt;
   TimeToStruct(TimeCurrent(),dt);

//--- Calculate minutes
   int current_minutes=TimeToMinutes(dt.hour,dt.min);
   int start_minutes=AdjustForGMT(TimeToMinutes(m_sessions[session].start_hour,
                                                 m_sessions[session].start_minute));
   int end_minutes=AdjustForGMT(TimeToMinutes(m_sessions[session].end_hour,
                                               m_sessions[session].end_minute));

//--- Handle sessions that cross midnight
   if(start_minutes>end_minutes)
      return(current_minutes>=start_minutes || current_minutes<end_minutes);

//--- Standard session check
   return(current_minutes>=start_minutes && current_minutes<end_minutes);
  }

//+------------------------------------------------------------------+
//| Check if specific time is within the specified session           |
//+------------------------------------------------------------------+
bool CSessionTimeFilter::IsTimeInSession(const datetime time,
                                         const ENUM_TRADING_SESSION session)
  {
//--- Ensure initialization
   if(!m_initialized)
      Init();

//--- Get time structure
   MqlDateTime dt;
   TimeToStruct(time,dt);

//--- Calculate minutes
   int current_minutes=TimeToMinutes(dt.hour,dt.min);
   int start_minutes=AdjustForGMT(TimeToMinutes(m_sessions[session].start_hour,
                                                 m_sessions[session].start_minute));
   int end_minutes=AdjustForGMT(TimeToMinutes(m_sessions[session].end_hour,
                                               m_sessions[session].end_minute));

//--- Handle sessions that cross midnight
   if(start_minutes>end_minutes)
      return(current_minutes>=start_minutes || current_minutes<end_minutes);

//--- Standard session check
   return(current_minutes>=start_minutes && current_minutes<end_minutes);
  }

//+------------------------------------------------------------------+
//| Check if currently in any of the specified sessions              |
//+------------------------------------------------------------------+
bool CSessionTimeFilter::IsInAnySessions(const bool sydney,const bool tokyo,
                                         const bool london,const bool newyork)
  {
   if(sydney && IsInSession(SESSION_SYDNEY))
      return(true);
   if(tokyo && IsInSession(SESSION_TOKYO))
      return(true);
   if(london && IsInSession(SESSION_LONDON))
      return(true);
   if(newyork && IsInSession(SESSION_NEWYORK))
      return(true);
   return(false);
  }

//+------------------------------------------------------------------+
//| Get string of currently active sessions                          |
//+------------------------------------------------------------------+
string CSessionTimeFilter::GetActiveSession(void)
  {
   string result="";

//--- Check each major session
   if(IsInSession(SESSION_SYDNEY))
      result+=(result=="" ? "" : ", ")+m_sessions[SESSION_SYDNEY].name;
   if(IsInSession(SESSION_TOKYO))
      result+=(result=="" ? "" : ", ")+m_sessions[SESSION_TOKYO].name;
   if(IsInSession(SESSION_LONDON))
      result+=(result=="" ? "" : ", ")+m_sessions[SESSION_LONDON].name;
   if(IsInSession(SESSION_NEWYORK))
      result+=(result=="" ? "" : ", ")+m_sessions[SESSION_NEWYORK].name;

//--- Return result or default message
   return(result=="" ? "No Major Session" : result);
  }

//+------------------------------------------------------------------+
//| Get the name of specified session                                |
//+------------------------------------------------------------------+
string CSessionTimeFilter::GetSessionName(const ENUM_TRADING_SESSION session)
  {
   return(m_sessions[session].name);
  }

//+------------------------------------------------------------------+
//| Calculate minutes until the specified session starts             |
//+------------------------------------------------------------------+
int CSessionTimeFilter::MinutesUntilSession(const ENUM_TRADING_SESSION session)
  {
//--- Return 0 if already in session
   if(IsInSession(session))
      return(0);

//--- Get current time
   MqlDateTime dt;
   TimeToStruct(TimeCurrent(),dt);

//--- Calculate difference
   int current_minutes=TimeToMinutes(dt.hour,dt.min);
   int start_minutes=AdjustForGMT(TimeToMinutes(m_sessions[session].start_hour,
                                                 m_sessions[session].start_minute));

   int diff=start_minutes-current_minutes;
   if(diff<0)
      diff+=SESSION_MINUTES_PER_DAY;

   return(diff);
  }

//+------------------------------------------------------------------+
//| Calculate minutes remaining in the current session               |
//+------------------------------------------------------------------+
int CSessionTimeFilter::MinutesRemainingInSession(const ENUM_TRADING_SESSION session)
  {
//--- Return 0 if not in session
   if(!IsInSession(session))
      return(0);

//--- Get current time
   MqlDateTime dt;
   TimeToStruct(TimeCurrent(),dt);

//--- Calculate difference
   int current_minutes=TimeToMinutes(dt.hour,dt.min);
   int end_minutes=AdjustForGMT(TimeToMinutes(m_sessions[session].end_hour,
                                               m_sessions[session].end_minute));

   int diff=end_minutes-current_minutes;
   if(diff<0)
      diff+=SESSION_MINUTES_PER_DAY;

   return(diff);
  }

//+------------------------------------------------------------------+
//| Check if current day is a weekday (Monday-Friday)                |
//+------------------------------------------------------------------+
bool CSessionTimeFilter::IsWeekday(void)
  {
   MqlDateTime dt;
   TimeToStruct(TimeCurrent(),dt);
   return(dt.day_of_week>=1 && dt.day_of_week<=5);
  }

//+------------------------------------------------------------------+
//| Check if current day matches specified day of week               |
//| Parameters: day - 0=Sunday, 1=Monday, ..., 6=Saturday            |
//+------------------------------------------------------------------+
bool CSessionTimeFilter::IsDayOfWeek(const int day)
  {
   MqlDateTime dt;
   TimeToStruct(TimeCurrent(),dt);
   return(dt.day_of_week==day);
  }
//+------------------------------------------------------------------+
