//+------------------------------------------------------------------+
//| Test_JsonLib_Professional.mq5                                    |
//+------------------------------------------------------------------+
#property copyright "ding9736"
#property link      "https://www.mql5.com/en/users/ding9736"
#property link      "https://github.com/ding9736/MQL5-JsonLib"
#define SCRIPT_VERSION "11.0"
#property version   SCRIPT_VERSION
#property script_show_inputs

#include "JsonLib.mqh"

const double BYTES_PER_KB = 1024.0;
const double BYTES_PER_MB = 1024.0 * 1024.0;

input group             "A. DOM API Core Functionality";
input string a_basicJson          = "{ \"symbol\": \"EURUSD\", \"bid\": 1.07251, \"ask\": 1.07254, \"active\": true, \"magic\": null }";

input group             "B. Advanced DOM Operations";
string b_complexJson              = "{\"store\":{\"book\":[{\"category\":\"reference\",\"author\":\"Nigel Rees\",\"title\":\"Sayings of the Century\",\"price\":8.95},{\"category\":\"fiction\",\"author\":\"Evelyn Waugh\",\"title\":\"Sword of Honour\",\"price\":12.99}],\"bicycle\":{\"color\":\"red\",\"price\":19.95}},\"expensive\":10}";
input string b_jsonPath           = "$.store.book[*].author";
input string b_testFilename       = "test_output_v28_1.json";
input string b_mergeBase          = "{\"user\":\"Alex\",\"settings\":{\"theme\":\"dark\",\"level\":5}}";
input string b_mergePatch         = "{\"settings\":{\"level\":10,\"notifications\":true},\"status\":\"online\"}";
input string b_rfc7396_patch      = "{\"status\":\"offline\",\"settings\":{\"theme\":null}}";

//
input group             "C. Data Binding (JsonMapper)";
input string c_userJson           = "{\"username\":\"zulu\",\"user_id\":7788,\"is_vip\":true,\"balance\":1234.56}";

input group             "D. Stream API";
input string d_streamJson         = "[{\"a\":1}, {\"b\":2}]";

input group             "E. Robustness & Performance";
input string e_unicodeJson        = "{\"slogan\": \"你好, 世界!\", \"currency\": \"€\"}";
input string e_malformedJson      = "{\"key\": value, \"error\":}";
input int    e_memTestLoops       = 5000;
input int    e_engineTestLoops    = 20000;
input int    e_perfElements       = 5000;

input group             "F. Graded File Performance Tests";
input bool   f_runFileTests       = true;
input double f_test_size_8KB      = 8.0;
input double f_test_size_16KB     = 16.0;
input double f_test_size_32KB     = 32.0;
input double f_test_size_64KB     = 64.0;
input double f_test_size_1MB      = 1.0;
input double f_test_size_5MB      = 5.0;
input double f_test_size_15MB     = 15.0;

//
class CUserProfile : public IJsonSerializable
{
public:
  string Name;
  long   ID;
  bool   IsVIP;
  double Balance;
  void FromJson(const JsonNode &node) override
  {
    Name=node.Get("username").AsString("Default");
    ID=node.Get("user_id").AsInt(0);
    IsVIP=node.Get("is_vip").AsBool(false);
    Balance = node.Get("balance").AsDouble(0.0);
  }
  JsonNode ToJson(JsonDocument &owner_doc) const override
  {
    JsonNode root = owner_doc.GetRoot();
    return root.Set("username", Name).Set("user_id", ID).Set("is_vip", IsVIP).Set("balance", Balance);
  }
};

//
class CSimpleStreamCounter : public IJsonStreamHandler
{
public:
  int object_count, array_count, key_count, value_count;
  void Reset()
  {
    object_count=0;
    array_count=0;
    key_count=0;
    value_count=0;
  }
  bool OnStartDocument() override
  {
    Reset();
    return true;
  }
  bool OnEndDocument() override
  {
    return true;
  }
  bool OnStartObject() override
  {
    object_count++;
    return true;
  }
  bool OnEndObject() override
  {
    return true;
  }
  bool OnStartArray() override
  {
    array_count++;
    return true;
  }
  bool OnEndArray() override
  {
    return true;
  }
  bool OnKey(const string &key) override
  {
    key_count++;
    return true;
  }
  bool OnString(const string &value) override
  {
    value_count++;
    return true;
  }
  bool OnNumber(const string &v, ENUM_JSON_TYPE t) override
  {
    value_count++;
    return true;
  }
  bool OnBool(bool value) override
  {
    value_count++;
    return true;
  }
  bool OnNull() override
  {
    value_count++;
    return true;
  }
};

struct TestFailureDetail
{
   string Checkpoint;
   string Expected;
   string Actual;
};

//
class TestScenarioResult : public CObject
{
public:
   string Name;
   bool Passed;
   string FatalError;
   TestFailureDetail Failures[];
   TestScenarioResult(string name) : Name(name), Passed(true), FatalError("") {}
};

//
class CTestFramework
{
private:
   TestScenarioResult* m_results[];
   int                 m_total_tests, m_passed_tests;
   TestScenarioResult* m_current_scenario;
   JsonParseOptions m_default_options;

   void   RecordFailure(const string cp, const string ex, const string ac)
   {
      if(CheckPointer(m_current_scenario)==POINTER_INVALID) return;
      m_current_scenario.Passed=false;
      int n=ArraySize(m_current_scenario.Failures);
      ArrayResize(m_current_scenario.Failures,n+1);
      m_current_scenario.Failures[n].Checkpoint=cp;
      m_current_scenario.Failures[n].Expected=ex;
      m_current_scenario.Failures[n].Actual=ac;
   }
   bool   AssertTrue(bool c, const string cp)
   {
      if(!c) RecordFailure(cp, "true", "false");
      return c;
   }
   bool   AssertEqualString(const string a, const string e, const string cp)
   {
      if(a!=e)
      {
         RecordFailure(cp, "'"+e+"'", "'"+a+"'");
         return false;
      }
      return true;
   }
   bool   AssertEqualInt(long a, long e, const string cp)
   {
      if(a!=e)
      {
         RecordFailure(cp, IntegerToString(e), IntegerToString(a));
         return false;
      }
      return true;
   }
   bool   AssertEqualDouble(double a, double e, const string cp)
   {
      if(MathAbs(a-e)>1e-9)
      {
         RecordFailure(cp, DoubleToString(e), DoubleToString(a));
         return false;
      }
      return true;
   }
   void   StartScenario(const string name)
   {
      ResetState();
      m_current_scenario = new TestScenarioResult(name);
      if(CheckPointer(m_current_scenario)==POINTER_INVALID) Print("CRITICAL: Failed to allocate TestScenarioResult!");
   }
   void   EndScenario()
   {
      if(CheckPointer(m_current_scenario)==POINTER_INVALID) return;
      int n=ArraySize(m_results);
      ArrayResize(m_results,n+1);
      m_results[n]=m_current_scenario;
      m_total_tests++;
      if(m_current_scenario.Passed) m_passed_tests++;
      m_current_scenario=NULL;
   }

   void   A1_DomBasicParsing();
   void   A2_DomCreationAndModification();
   void   A3_DomTypeChecking();
   void   B1_JsonPathQuery();
   void   B2_FileIO();
   void   B3_JsonMerge();
   void   B4_JsonMergePatch();
   void   C1_DataBindingMapper();
   void   D1_StreamApiParsing();
   void   E1_UnicodeCorrectness();
   void   E2_ErrorHandling();
   void   E3_MemoryPressure();
   void   E4_EngineSwitchingPerformance();
   void   E5_PayloadThroughput();
   void   F1_GradedFileSizePerformance();

   bool   GenerateTestJsonFile(const string &filename, const ulong target_bytes);
   ulong  GetFileSizeBytes(const string &filename);
   bool   ReadFileToString(const string &filename, string &content);
   string FormatTime(ulong time_us);

public:
   CTestFramework();
   ~CTestFramework();
   void RunAllTests();
   void PrintSummaryReport();
};

//
CTestFramework::CTestFramework()
{
   m_total_tests=0;
   m_passed_tests=0;
   m_current_scenario=NULL;
}

//
CTestFramework::~CTestFramework()
{
   for(int i=ArraySize(m_results)-1; i>=0; i--) if(CheckPointer(m_results[i])==POINTER_DYNAMIC) delete m_results[i];
}

//
void CTestFramework::A1_DomBasicParsing()
{
  StartScenario("A1. DOM: Basic Parsing");
  JsonError error;
  JsonDocument doc = JsonParse(a_basicJson, error, m_default_options);
  if(!AssertTrue(doc.IsValid(), "Document should be valid after parsing"))
  {
    m_current_scenario.FatalError = "Parsing failed: " + error.ToString();
    EndScenario();
    return;
  }
  JsonNode root = doc.GetRoot();
  AssertEqualString(root.Get("symbol").AsString(), "EURUSD", "Check string field 'symbol'");
  AssertEqualDouble(root.Get("bid").AsDouble(), 1.07251, "Check double field 'bid'");
  AssertTrue(root.Get("active").AsBool(), "Check bool field 'active'");
  AssertTrue(root.Get("magic").IsNull(), "Check null field 'magic'");
  EndScenario();
}

//
void CTestFramework::A2_DomCreationAndModification()
{
  StartScenario("A2. DOM: Creation & Modification");
  JsonDocument doc = JsonNewObject();
  JsonNode root = doc.GetRoot();
  root.SetObject("params").Set("ratio", 1.618);
  root.SetArray("tags").Add("fibonacci").Add("golden");
  if(!AssertTrue(doc.IsValid(), "Newly created document should be valid"))
  {
    EndScenario();
    return;
  }
  AssertEqualDouble(root.Get("params").Get("ratio").AsDouble(), 1.618, "Check created nested double");
  AssertEqualString(root.Get("tags")[1].AsString(), "golden", "Check created array element");
  root.Get("params").Set("ratio", 3.14159);
  root.Remove("tags");
  AssertEqualDouble(root.Get("params").Get("ratio").AsDouble(), 3.14159, "Check modified double");
  AssertTrue(!root.HasKey("tags"), "Check key 'tags' was removed");
  EndScenario();
}

//
void CTestFramework::A3_DomTypeChecking()
{
  StartScenario("A3. DOM: Type Checking");
  JsonError error;
  JsonDocument doc = JsonParse(a_basicJson, error, m_default_options);
  if(!AssertTrue(doc.IsValid(), "Document for type checking must be valid"))
  {
    EndScenario();
    return;
  }
  JsonNode root = doc.GetRoot();
  AssertTrue(root.Get("symbol").IsString(), "'symbol' should be IsString()");
  AssertTrue(root.Get("bid").IsNumber(), "'bid' should be IsNumber()");
  AssertTrue(root.Get("bid").IsDouble(), "'bid' should be IsDouble()");
  AssertTrue(!root.Get("bid").IsInt(), "'bid' should NOT be IsInt()");
  AssertTrue(root.Get("active").AsBool(), "'active' should be IsBool()");
  AssertTrue(root.Get("magic").IsNull(), "'magic' should be IsNull()");
  AssertTrue(!root.Get("non_existent").IsValid(), "Non-existent node should be !IsValid()");
  EndScenario();
}

//
void CTestFramework::B1_JsonPathQuery()
{
  StartScenario("B1. Advanced: JSONPath Query");
  JsonError error;
  JsonDocument doc = JsonParse(b_complexJson, error, m_default_options);
  if(!AssertTrue(doc.IsValid(), "Complex doc for JSONPath must be valid. Error: " + error.ToString()))
  {
    EndScenario();
    return;
  }
  JsonNode results[];
  int count = JsonQuerySelectNodes(doc.GetRoot(), b_jsonPath, results, error, MODE_ADVANCED);
  if(AssertEqualInt(count, 2, "JSONPath query should return 2 nodes"))
  {
    AssertEqualString(results[0].AsString(), "Nigel Rees", "Check first query result");
    AssertEqualString(results[1].AsString(), "Evelyn Waugh", "Check second query result");
  }
  EndScenario();
}

//
void CTestFramework::B2_FileIO()
{
  StartScenario("B2. Advanced: File I/O");
  JsonDocument doc_to_save = JsonNewObject();
  doc_to_save.GetRoot().Set("timestamp", (long)TimeCurrent());
  if(!AssertTrue(doc_to_save.SaveToFile(b_testFilename, true), "SaveToFile() should return true"))
  {
    EndScenario();
    return;
  }
  JsonError error;
  JsonDocument doc_from_file = JsonFromFile(b_testFilename, error, m_default_options);
  FileDelete(b_testFilename);
  if(!AssertTrue(doc_from_file.IsValid(), "Document loaded from file should be valid. Error: " + error.ToString()))
  {
    EndScenario();
    return;
  }
  AssertTrue(doc_from_file.GetRoot().HasKey("timestamp"), "File content validation");
  EndScenario();
}

//
void CTestFramework::B3_JsonMerge()
{
  StartScenario("B3. Advanced: Deep Merge");
  JsonError error;
  JsonDocument base = JsonParse(b_mergeBase, error, m_default_options);
  JsonDocument patch = JsonParse(b_mergePatch, error, m_default_options);
  if(!AssertTrue(base.IsValid() && patch.IsValid(), "Base and patch docs for merge must be valid"))
  {
    EndScenario();
    return;
  }
  JsonDocument merged = JsonQueryMerge(base, patch);
  if(!AssertTrue(merged.IsValid(), "Merged document must be valid"))
  {
    EndScenario();
    return;
  }
  JsonNode r = merged.GetRoot();
  AssertEqualString(r.Get("user").AsString(), "Alex", "Unchanged field 'user' should persist");
  AssertEqualString(r.Get("status").AsString(), "online", "New field 'status' should be added");
  AssertEqualInt(r.Get("settings").Get("level").AsInt(), 10, "Nested field 'level' should be overwritten");
  AssertTrue(r.Get("settings").Get("notifications").AsBool(), "New nested field 'notifications' should be added");
  EndScenario();
}

//
void CTestFramework::B4_JsonMergePatch()
{
  StartScenario("B4. Advanced: RFC 7396 Patch");
  JsonError error;
  JsonDocument doc = JsonParse(b_mergeBase, error, m_default_options);
  JsonDocument patch = JsonParse(b_rfc7396_patch, error, m_default_options);
  if(!AssertTrue(doc.IsValid() && patch.IsValid(), "Base and patch docs for patch must be valid"))
  {
    EndScenario();
    return;
  }
  doc.Patch(patch);
  JsonNode r = doc.GetRoot();
  AssertEqualString(r.Get("status").AsString(), "offline", "New field 'status' should be added by patch");
  AssertTrue(!r.Get("settings").HasKey("theme"), "Field 'theme' should be deleted by 'null' in patch");
  AssertEqualInt(r.Get("settings").Get("level").AsInt(), 5, "Unchanged nested field 'level' should persist");
  EndScenario();
}

//
void CTestFramework::C1_DataBindingMapper()
{
  StartScenario("C1. Data Binding: JsonMapper");
  JsonError error;
  JsonDocument doc = JsonParse(c_userJson, error, m_default_options);
  if(!AssertTrue(doc.IsValid(), "JSON for deserialization must be valid"))
  {
    EndScenario();
    return;
  }
  CUserProfile profile;
  if(AssertTrue(JsonMapper::Deserialize(doc.GetRoot(), profile), "Deserialization should succeed"))
  {
    AssertEqualString(profile.Name, "zulu", "Deserialized profile.Name");
    AssertEqualInt(profile.ID, 7788, "Deserialized profile.ID");
    AssertTrue(profile.IsVIP, "Deserialized profile.IsVIP");
    AssertEqualDouble(profile.Balance, 1234.56, "Deserialized profile.Balance");
  }
  JsonDocument new_doc = JsonNewObject();
  if(AssertTrue(JsonMapper::Serialize(profile, new_doc) && new_doc.IsValid(), "Serialization should succeed"))
  {
    AssertEqualString(new_doc.GetRoot().Get("username").AsString(), "zulu", "Serialized node 'username'");
    AssertEqualDouble(new_doc.GetRoot().Get("balance").AsDouble(), 1234.56, "Serialized node 'balance'");
  }
  EndScenario();
}

//
void CTestFramework::D1_StreamApiParsing()
{
  StartScenario("D1. API: Stream (SAX) Parsing");
  JsonError error;
  CSimpleStreamCounter counter;
  if(AssertTrue(JsonStreamParse(d_streamJson, GetPointer(counter), error, m_default_options), "Stream parsing should complete successfully. Error: "+error.ToString()))
  {
    AssertEqualInt(counter.array_count, 1, "Stream: Array count should be 1");
    AssertEqualInt(counter.object_count, 2, "Stream: Object count should be 2");
    AssertEqualInt(counter.key_count, 2, "Stream: Key count should be 2");
    AssertEqualInt(counter.value_count, 2, "Stream: Value count should be 2");
  }
  EndScenario();
}

//
void CTestFramework::E1_UnicodeCorrectness()
{
  StartScenario("E1. Robustness: Unicode");
  JsonError error;
  JsonDocument doc = JsonParse(e_unicodeJson, error, m_default_options);
  if(!AssertTrue(doc.IsValid(), "Unicode document should parse correctly"))
  {
    EndScenario();
    return;
  }
  AssertEqualString(doc.GetRoot().Get("slogan").AsString(), "你好, 世界!", "Check slogan field (Chinese)");
  string serialized = doc.ToString();
  AssertTrue(StringFind(serialized, "你好, 世界!") > -1, "Serialized string should contain Unicode");
  EndScenario();
}

//
void CTestFramework::E2_ErrorHandling()
{
  StartScenario("E2. Robustness: Error Handling");
  JsonError error;
  JsonParseOptions options;
  options.engine = ENGINE_STANDARD;
  JsonDocument doc = JsonParse(e_malformedJson, error, options);
  AssertTrue(!doc.IsValid(), "Parsing malformed JSON should result in an invalid document");
  AssertTrue(StringLen(error.message) > 0, "Error message should not be empty");
  AssertTrue(error.line > 0, "Error should report a valid line number");
  PrintFormat(" > Caught expected error: %s", error.ToString());
  EndScenario();
}

//
void CTestFramework::E3_MemoryPressure()
{
  StartScenario("E3. Performance: Memory Pressure");
  ulong start_mem = MQLInfoInteger(MQL_MEMORY_USED);
  for(int i = 0; i < e_memTestLoops; i++)
  {
    JsonDocument doc = JsonNewObject();
    doc.GetRoot().Set("index", (long)i).Set("data", "some_payload_data");
  }
  ulong end_mem = MQLInfoInteger(MQL_MEMORY_USED);
  string msg = StringFormat("Loop completed. Memory delta: %d KB. Check terminal journal for leaks.", (int)((long)(end_mem - start_mem) / 1024));
  AssertTrue(true, msg);
  EndScenario();
}

//
void CTestFramework::E4_EngineSwitchingPerformance()
{
  StartScenario("E4. Performance: Engine Switching");
  JsonError error;
  string micro_json = "{\"a\":1.23}";
  JsonParseOptions standard_opts;
  standard_opts.engine = ENGINE_STANDARD;
  JsonParseOptions rapid_opts;
  rapid_opts.engine = ENGINE_HIGH_SPEED;
  JsonParseOptions auto_opts;
  auto_opts.engine = ENGINE_AUTO;
  for(int i=0;i<100;i++) JsonParse(micro_json, error, m_default_options);
  ulong start = GetMicrosecondCount();
  for(int i=0; i<e_engineTestLoops; i++) JsonParse(micro_json, error, standard_opts);
  ulong standard_time = GetMicrosecondCount() - start;
  start = GetMicrosecondCount();
  for(int i=0; i<e_engineTestLoops; i++) JsonParse(micro_json, error, rapid_opts);
  ulong rapid_time = GetMicrosecondCount() - start;
  start = GetMicrosecondCount();
  for(int i=0; i<e_engineTestLoops; i++) JsonParse(micro_json, error, auto_opts);
  ulong auto_time = GetMicrosecondCount() - start;
  PrintFormat(" > Standard: %llu us | High-Speed: %llu us | Auto-Switch: %llu us", standard_time, rapid_time, auto_time);
  AssertTrue(rapid_time < standard_time, "High-Speed engine should be faster than Standard for small JSON");
  AssertTrue(auto_time <= rapid_time * 1.05, "Auto-switch should be nearly as fast as High-Speed (within 5% tolerance)");
  EndScenario();
}

//
void CTestFramework::E5_PayloadThroughput()
{
  StartScenario("E5. Performance: Payload Throughput");
  StringBuilder sb;
  sb.AppendChar('[');
  for(int i = 0; i < e_perfElements; i++)
  {
    sb.Append(StringFormat("{\"id\":%d,\"price\":%.5f}", i, 1.12345 + i * 0.00001));
    if(i < e_perfElements - 1) sb.AppendChar(',');
  }
  sb.AppendChar(']');
  string json = sb.ToString();
  long json_size_bytes = (long)StringLen(json);
  JsonError error;
  JsonParseOptions rapid_opts;
  rapid_opts.engine = ENGINE_HIGH_SPEED;
  ulong start = GetMicrosecondCount();
  JsonDocument doc = JsonParse(json, error, rapid_opts);
  ulong time_us = GetMicrosecondCount() - start;
  if(AssertTrue(doc.IsValid(), "Large payload must parse successfully"))
  {
    double time_s = (double)time_us / 1000000.0;
    double mbps = (time_s > 0) ? ((double)json_size_bytes / BYTES_PER_MB) / time_s : 0;
    PrintFormat(" > Parsed %d elements (%.1f KB) in %s. Throughput: %.2f MB/s", e_perfElements, (double)json_size_bytes / BYTES_PER_KB, FormatTime(time_us), mbps);
  }
  EndScenario();
}

//
ulong CTestFramework::GetFileSizeBytes(const string &filename)
{
   int h = FileOpen(filename, FILE_READ | FILE_BIN);
   if(h < 0) return 0;
   ulong size = (ulong)FileSize(h);
   FileClose(h);
   return size;
}

//
bool CTestFramework::ReadFileToString(const string &filename, string &content)
{
   content = "";
   int h = FileOpen(filename, FILE_READ | FILE_BIN);
   if (h < 0) return false;
   ulong file_size = FileSize(h);
   if (file_size == 0)
   {
      FileClose(h);
      return true;
   }
   uchar buffer[];
   if (FileReadArray(h, buffer) != (uint)file_size)
   {
      FileClose(h);
      return false;
   }
   FileClose(h);
   content = ::CharArrayToString(buffer, 0, -1, CP_UTF8);
   return true;
}

//
bool CTestFramework::GenerateTestJsonFile(const string &filename, const ulong target_bytes)
{
   int h = FileOpen(filename, FILE_WRITE | FILE_TXT | FILE_ANSI, "\t", CP_UTF8);
   if(h < 0)
   {
      PrintFormat("Error: Could not open file '%s' for writing. Code: %d", filename, GetLastError());
      return false;
   }
   string record_template = "{\"id\": %d, \"data\": \"some_fixed_payload_to_pad_the_file_size_and_test_throughput\", \"value\": %.4f}";
   string sample_record = StringFormat(record_template, 9999999, 9999.9999);
   ulong record_size = (ulong)(StringLen(sample_record) + 2);
   if(record_size <= 0) record_size = 120;
   ulong records_to_generate = (target_bytes > 2) ? ((target_bytes - 2) / record_size) : 1;
   if(records_to_generate==0 && target_bytes >0) records_to_generate=1;
   FileWriteString(h, "[\n");
   for(ulong i = 0; i < records_to_generate; i++)
   {
      string record_str = "  " + StringFormat(record_template, (int)i, i * 1.2345);
      FileWriteString(h, record_str);
      if(i < records_to_generate - 1) FileWriteString(h, ",\n");
      else FileWriteString(h, "\n");
   }
   FileWriteString(h, "]");
   FileClose(h);
   ulong actual_bytes = GetFileSizeBytes(filename);
   string target_unit, actual_unit;
   double target_val, actual_val;
   if (target_bytes < BYTES_PER_MB)
   {
      target_val = (double)target_bytes / BYTES_PER_KB;
      target_unit = "KB";
   }
   else
   {
      target_val = (double)target_bytes / BYTES_PER_MB;
      target_unit = "MB";
   }
   if (actual_bytes < BYTES_PER_MB)
   {
      actual_val = (double)actual_bytes / BYTES_PER_KB;
      actual_unit = "KB";
   }
   else
   {
      actual_val = (double)actual_bytes / BYTES_PER_MB;
      actual_unit = "MB";
   }
   PrintFormat(" > Generated '%s': Target: %.1f %s, Actual: %.1f %s", filename, target_val, target_unit, actual_val, actual_unit);
   return true;
}

//
string CTestFramework::FormatTime(ulong time_us)
{
   if (time_us < 5000)
   {
      return StringFormat("%llu us", time_us);
   }
   return StringFormat("%llu ms", time_us / 1000);
}

//
void CTestFramework::F1_GradedFileSizePerformance()
{
   if(!f_runFileTests)
   {
      Print("\n--- [F] Graded File Performance Tests --- \n > Skipped by input parameter.");
      return;
   }
   StartScenario("F1. Performance: Graded File Sizes");
   ulong file_sizes_bytes[7];
   file_sizes_bytes[0] = (ulong)(f_test_size_8KB  * BYTES_PER_KB);
   file_sizes_bytes[1] = (ulong)(f_test_size_16KB * BYTES_PER_KB);
   file_sizes_bytes[2] = (ulong)(f_test_size_32KB * BYTES_PER_KB);
   file_sizes_bytes[3] = (ulong)(f_test_size_64KB * BYTES_PER_KB);
   file_sizes_bytes[4] = (ulong)(f_test_size_1MB  * BYTES_PER_MB);
   file_sizes_bytes[5] = (ulong)(f_test_size_5MB  * BYTES_PER_MB);
   file_sizes_bytes[6] = (ulong)(f_test_size_15MB * BYTES_PER_MB);

   string file_names[] = { "perf_8KB.json", "perf_16KB.json", "perf_32KB.json", "perf_64KB.json", "perf_1MB.json", "perf_5MB.json", "perf_15MB.json" };
   JsonParseOptions std_opts, rapid_opts;
   std_opts.engine = ENGINE_STANDARD;
   rapid_opts.engine = ENGINE_HIGH_SPEED;
   for(int i=0; i<ArraySize(file_sizes_bytes); i++)
   {
      if(file_sizes_bytes[i] == 0) continue;
      double target_val = (file_sizes_bytes[i] < BYTES_PER_MB) ? (double)file_sizes_bytes[i]/BYTES_PER_KB : (double)file_sizes_bytes[i]/BYTES_PER_MB;
      string target_unit = (file_sizes_bytes[i] < BYTES_PER_MB) ? "KB" : "MB";
      PrintFormat("\n--- [F] Testing file: %s (Target: %.1f %s) ---", file_names[i], target_val, target_unit);
      if(!GenerateTestJsonFile(file_names[i], file_sizes_bytes[i]))
      {
         RecordFailure("File Generation", "Success", "Failed for " + file_names[i]);
         continue;
      }
      ulong actual_bytes = GetFileSizeBytes(file_names[i]);
      if(actual_bytes == 0)
      {
         RecordFailure("File Gen Check", ">0", "0 bytes for " + file_names[i]);
         FileDelete(file_names[i]);
         continue;
      }
      JsonError error;
      ulong start = GetMicrosecondCount();
      JsonDocument doc_stream = JsonFromFile(file_names[i], error, m_default_options);
      ulong time_stream = GetMicrosecondCount() - start;
      if (AssertTrue(doc_stream.IsValid(), file_names[i]+" stream parse. Error: "+error.ToString()))
      {
         double tp = (time_stream > 0) ? ((double)actual_bytes/BYTES_PER_MB) / ((double)time_stream/1000000.0) : 0;
         PrintFormat(" > JsonFromFile (Stream) : %s, Throughput: %.2f MB/s", FormatTime(time_stream), tp);
      }
      string content;
      if (ReadFileToString(file_names[i], content))
      {
         start = GetMicrosecondCount();
         JsonDocument doc_std = JsonParse(content, error, std_opts);
         ulong time_std = GetMicrosecondCount() - start;
         if (AssertTrue(doc_std.IsValid(), file_names[i]+" std-mem parse. Error: "+error.ToString()))
         {
            double tp = (time_std > 0) ? ((double)actual_bytes/BYTES_PER_MB) / ((double)time_std/1000000.0) : 0;
            PrintFormat(" > JsonParse (Std Mem)   : %s, Throughput: %.2f MB/s", FormatTime(time_std), tp);
         }
         start = GetMicrosecondCount();
         JsonDocument doc_rapid = JsonParse(content, error, rapid_opts);
         ulong time_rapid = GetMicrosecondCount() - start;
         if (AssertTrue(doc_rapid.IsValid(), file_names[i]+" rapid-mem parse. Error: "+error.ToString()))
         {
            double tp = (time_rapid > 0) ? ((double)actual_bytes/BYTES_PER_MB) / ((double)time_rapid/1000000.0) : 0;
            PrintFormat(" > JsonParse (Rapid Mem) : %s, Throughput: %.2f MB/s", FormatTime(time_rapid), tp);
         }
      }
      FileDelete(file_names[i]);
   }
   EndScenario();
}

//
void CTestFramework::RunAllTests()
{
   PrintFormat("====== MQL5 JSON Library Test Suite v%s (Titan Pro) ======", SCRIPT_VERSION);
   PrintFormat("Terminal: %s Build %d", TerminalInfoString(TERMINAL_COMPANY), (int)TerminalInfoInteger(TERMINAL_BUILD));
   Print("\n--- [A] DOM API Core Functionality ---");
   A1_DomBasicParsing();
   A2_DomCreationAndModification();
   A3_DomTypeChecking();
   Print("\n--- [B] Advanced DOM Operations ---");
   B1_JsonPathQuery();
   B2_FileIO();
   B3_JsonMerge();
   B4_JsonMergePatch();
   Print("\n--- [C] Data Binding (JsonMapper) ---");
   C1_DataBindingMapper();
   Print("\n--- [D] Stream API (SAX-style) ---");
   D1_StreamApiParsing();
   Print("\n--- [E] Robustness & Performance ---");
   E1_UnicodeCorrectness();
   E2_ErrorHandling();
   E3_MemoryPressure();
   E4_EngineSwitchingPerformance();
   E5_PayloadThroughput();
   F1_GradedFileSizePerformance();
}

//
void CTestFramework::PrintSummaryReport()
{
   string line_sep = "-----------------------------------------------------------------";
   Print("\n" + line_sep);
   Print("                          TEST SUMMARY REPORT                              ");
   Print(line_sep);
   for(int i = 0; i < ArraySize(m_results); i++)
   {
      TestScenarioResult *res = m_results[i];
      if(CheckPointer(res) == POINTER_INVALID) continue;
      string status = res.Passed ? "PASS" : "FAIL";
      string name_padded = res.Name;
      while(StringLen(name_padded) < 45) name_padded += ".";
      Print(name_padded + " [" + status + "]");
      if(!res.Passed)
      {
         if(StringLen(res.FatalError) > 0) PrintFormat("  [FATAL] %s", res.FatalError);
         for(int j = 0; j < ArraySize(res.Failures); j++)
         {
            TestFailureDetail f = res.Failures[j];
            PrintFormat("  - Checkpoint: %s", f.Checkpoint);
            PrintFormat("    - Expected: %s", f.Expected);
            PrintFormat("    - Actual  : %s", f.Actual);
         }
      }
   }
   Print(line_sep);
   PrintFormat("OVERALL RESULT: %d/%d SCENARIOS PASSED", m_passed_tests, m_total_tests);
   Print(line_sep);
   if(MQLInfoInteger(MQL_TESTER) || MQLInfoInteger(MQL_PROFILER))
   {
      Print("\nMemory Check: Performed by terminal at script completion.");
   }
}

//
void OnStart()
{
  CTestFramework framework;
  framework.RunAllTests();
  framework.PrintSummaryReport();
  ResetState();
  Print("JsonLib state has been reset at the end of script execution.");
}
//+------------------------------------------------------------------+
