//+------------------------------------------------------------------+
//| JsonLib.mqh                                                      |
//+------------------------------------------------------------------+
#property copyright "ding9736"
#property link      "https://www.mql5.com/en/users/ding9736"
#property link      "https://github.com/ding9736/MQL5-JsonLib"
#property version   "11.0"

#ifndef MQL5_JSON
#define MQL5_JSON

/**********************************************************************************************************************
*                                                                                                                     *
*                    MQL5-JsonLib: A Professional-Grade JSON Library for MQL5                                         *
*                                                                                                                     *
* =================================================================================================================== *
*                                                                                                                     *
*   Official In-Code Documentation                                                                                    *
*                                                                                                                     *
*   Version: 11.0                                                                                                     *
*   Author: ding9736                                                                                                  *
*   GitHub: https://github.com/ding9736/MQL5-JsonLib                                                                  *
*   License: MIT                                                                                                      *
*                                                                                                                     *
**********************************************************************************************************************/

/*
 * MQL5-JsonLib is a comprehensive, high-performance, and robust JSON solution designed specifically 
 * for the MQL5 language. It provides a full suite of professional-grade features, from simple DOM 
 * parsing and data binding to complex SAX-style stream processing and powerful JSONPath queries, 
 * aiming to equip MQL5 developers with a thoroughly modern tool for data exchange and manipulation.
 *
 * Whether you need to interface with web APIs for market data, manage complex EA/indicator 
 * configurations, or exchange information efficiently between different systems, JsonLib offers 
 * a stable, high-performance, and exceptionally easy-to-use interface.
 */


//--- Core Features --------------------------------------------------------------------------------------------------

/*
 *   - Dual-Engine Parsing: Intelligent switching between a high-speed engine for standard JSON
 *     and a feature-rich engine for non-standard formats (comments, trailing commas).
 *
 *   - Complete Document Object Model (DOM) API: Easily load, create, access, and modify JSON 
 *     documents with an intuitive, object-oriented interface.
 *
 *   - Efficient Stream-based (SAX) API: Process gigabyte-scale JSON files with minimal memory 
 *     footprint, ideal for handling large data streams.
 *
 *   - Powerful Data Queries:
 *      * JSONPath: Effortlessly query and extract data from complex structures.
 *      * JSON Pointer (RFC 6901): Provides lightweight and fast pinpoint element access.
 *
 *   - Seamless File I/O: Efficiently parse JSON directly from a file stream, or save documents
 *     in either pretty or compact format.
 *
 *   - Robust Data Binding (`JsonMapper`): Automatically map between MQL5 custom classes and 
 *     JSON nodes, significantly improving code structure and maintainability.
 *
 *   - Advanced Document Operations:
 *      * Deep Merge: Recursively merge two JSON objects.
 *      * RFC 7396 Patch: Supports the standardized JSON Patch protocol for partial updates.
 *
 *   - Robust Error Handling: Provides detailed error messages, including line, column, and context.
 */


//--- Performance Highlights -----------------------------------------------------------------------------------------

/*
 * Rigorously tested under MetaTrader 5 Build 5233.
 *
 * Table: Throughput for various file sizes
 * +----------------------+-------------------------+----------------------------+---------------------------+
 * | File Size (Actual)   | JsonFromFile (Stream)   | JsonParse (Standard Mem)   | JsonParse (Rapid Mem)     |
 * +----------------------+-------------------------+----------------------------+---------------------------+
 * | 7.6 KB               | 23.49 MB/s              | 30.93 MB/s                 | 34.37 MB/s                |
 * | 62.5 KB              | 28.42 MB/s              | 31.81 MB/s                 | 35.57 MB/s                |
 * | 1.0 MB               | 9.80 MB/s               | 9.45 MB/s                  | 9.91 MB/s                 |
 * | 15.4 MB              | 0.66 MB/s               | 0.67 MB/s                  | 0.68 MB/s                 |
 * +----------------------+-------------------------+----------------------------+---------------------------+
 * Conclusion: The high-speed engine excels for small-to-medium files. Stream parsing is optimal for large files.
 */


//--- Installation Guide ---------------------------------------------------------------------------------------------

/*
 * 1. Copy the `JsonLib.mqh` file and the entire `Core` folder into your MQL5 `Include` directory
 *    (typically located at `MQL5/Include/`).
 *
 * 2. In your EA, script, or indicator code, simply include the main header file:
 *    #include <JsonLib.mqh>
 */


//--- Quick Start: Master the Basics in 5 Minutes --------------------------------------------------------------------
/*
 * This example covers parsing, reading, modifying, and printing JSON data.
 */
//  Example: Quick Start
//--------------------------------------------------------------------------------------------------------------------
/*
#include <JsonLib.mqh>

void OnStart()
{
    // 1. Imagine this is market data from a web API
    string market_data_str = R"({
      "symbol": "EURUSD", "timestamp": 1672531200, "bid": 1.0705, "ask": 1.0708, 
      "tradeable": true, "levels": [1.0700, 1.0800]
    })";

    // 2. Parse the JSON string
    MQL5_Json::JsonError error;
    MQL5_Json::JsonDocument doc = MQL5_Json::JsonParse(market_data_str, error);

    // 3. Validate and access the data
    if(doc.IsValid())
    {
        MQL5_Json::JsonNode root = doc.GetRoot();
        
        // Safely read data using the [] operator and As...() methods
        string symbol = root["symbol"].AsString("N/A");
        double spread = root["ask"].AsDouble() - root["bid"].AsDouble();
        
        PrintFormat("Symbol: %s, Spread: %.5f", symbol, spread);

        // 4. Modify the data
        root.Set("tradeable", false); // Pause trading for this pair
        root.SetObject("metadata").Set("source", "My EA"); // Add a new child object

        // 5. Convert the modified document back to a formatted string
        Print("\n--- Modified JSON ---");
        Print(doc.ToString(true));
    }
    else
    {
        Print("JSON parsing failed: ", error.ToString());
    }
}
*/
//--------------------------------------------------------------------------------------------------------------------


/**********************************************************************************************************************
*                                                                                                                     *
*                                       --==[ Authoritative API Guide ]==--                                           *
*                                                                                                                     *
**********************************************************************************************************************/

//--- PART 1: Core DOM API Usage ---

//>> 1.1 Parsing JSON
//  Example: Parsing a string with non-standard format (comments)
//--------------------------------------------------------------------------------------------------------------------
/*
string text_with_comments = R"({"strategy_id":"MA_Cross_v2", "magic_number":654321,})"; // Contains a trailing comma
MQL5_Json::JsonError error;
MQL5_Json::JsonParseOptions options;
options.engine = MQL5_Json::ENGINE_STANDARD;
options.allow_trailing_commas = true;
MQL5_Json::JsonDocument doc = MQL5_Json::JsonParse(text_with_comments, error, options);
*/
//--------------------------------------------------------------------------------------------------------------------

//>> 1.2 Creating New JSON Documents
//  Example: Building a complete EA configuration
//--------------------------------------------------------------------------------------------------------------------
/*
MQL5_Json::JsonDocument doc = MQL5_Json::JsonNewObject();
MQL5_Json::JsonNode root = doc.GetRoot();
root.Set("ea_name", "Pro RSI Trader");
root.Set("version", 1.2);
MQL5_Json::JsonNode params = root.SetObject("parameters");
params.Set("rsi_period", 14);
params.Set("stop_loss_pips", 50);
MQL5_Json::JsonNode sessions = root.SetArray("trading_sessions");
sessions.Add("London");
sessions.Add("New_York");
Print(doc.ToString(true));
*/
//--------------------------------------------------------------------------------------------------------------------

//>> 1.3 Accessing and Reading Data
//  Example: Parsing a multi-level account info JSON
//--------------------------------------------------------------------------------------------------------------------
/*
string account_info_str = R"({"accountId": "USR-9876", "balance": 10250.75, "contact": {"email": "trader@example.com"}})";
MQL5_Json::JsonDocument doc = MQL5_Json::JsonParse(account_info_str);
MQL5_Json::JsonNode root = doc.GetRoot();
double balance = root["balance"].AsDouble(0.0);
string email = root["contact"]["email"].AsString("N/A");
PrintFormat("Account balance: %.2f, Contact: %s", balance, email);
*/
//--------------------------------------------------------------------------------------------------------------------

//>> 1.4 Modifying and Updating Data
//  Example: Dynamically adjusting a strategy configuration
//--------------------------------------------------------------------------------------------------------------------
/*
string config_str = "{\"risk_percent\": 1.0, \"symbols\": [\"EURUSD\"]}";
MQL5_Json::JsonDocument doc = MQL5_Json::JsonParse(config_str);
MQL5_Json::JsonNode root = doc.GetRoot();
root.Set("risk_percent", 1.5); // Update a value
root["symbols"].Add("AUDUSD"); // Add to an array
root.Set("comment", "Optimized"); // Add a new key
Print(doc.ToString(true));
*/
//--------------------------------------------------------------------------------------------------------------------

//--- PART 2: Advanced Queries and Operations ---

//>> 2.1 Powerful Queries with JSONPath
//  Example: Extracting headlines of high-impact news
//--------------------------------------------------------------------------------------------------------------------
/*
string news_api_response = R"({"articles": [{"headline":"Fed holds rates", "impact":"High"}, {"headline":"Earnings surprise", "impact":"Low"}]})";
MQL5_Json::JsonDocument doc = MQL5_Json::JsonParse(news_api_response);
JsonError error;
MQL5_Json::JsonNode high_impact_news[];
string path = "$.articles[?(@.impact == 'High')].headline";
int count = MQL5_Json::JsonQuery::SelectNodes(doc.GetRoot(), path, high_impact_news, error);
for (int i=0; i<count; i++) Print(high_impact_news[i].AsString());
*/
//--------------------------------------------------------------------------------------------------------------------

//>> 2.2 Pinpoint Access with JSON Pointer (RFC 6901)
//  Example: Accessing a deeply nested element directly
//--------------------------------------------------------------------------------------------------------------------
/*
string text = R"({"account":{"details":{"user_id":12345}},"orders":[{"id":"A1"}]})";
MQL5_Json::JsonNode root = MQL5_Json::JsonParse(text).GetRoot();
MQL5_Json::JsonNode user_id = root.QueryPointer("/account/details/user_id");
Print("User ID: ", user_id.AsInt());
*/
//--------------------------------------------------------------------------------------------------------------------

//--- PART 3: Advanced Programming Patterns ---

//>> 3.1 Data Binding with `JsonMapper`
//  Example: Mapping JSON to a custom MQL5 class
//--------------------------------------------------------------------------------------------------------------------
/*
class CTradeOrder : public MQL5_Json::IJsonSerializable { ... }; // Full class definition in main guide
string order_json = "{\"symbol\":\"EURUSD\", \"volume\":0.1}";
CTradeOrder my_order;
if (MQL5_Json::JsonMapper::Deserialize(MQL5_Json::JsonParse(order_json).GetRoot(), my_order)) {
    Print("Deserialized order for symbol: ", my_order.symbol);
}
*/
//--------------------------------------------------------------------------------------------------------------------

//>> 3.2 Stream Processing (SAX API) for Massive JSON Files
//  Example: Tallying trades from a huge log stream
//--------------------------------------------------------------------------------------------------------------------
/*
class CTradeLogHandler : public MQL5_Json::IJsonStreamHandler { ... }; // Full class definition in main guide
string huge_log_stream = "{\"trades\": [{\"profit\": 10}, {\"profit\": -5}, ...]}";
CTradeLogHandler handler;
JsonError error;
if (MQL5_Json::JsonStreamParse(huge_log_stream, GetPointer(handler), error)) {
    Print("Stream analysis complete.");
}
*/
//--------------------------------------------------------------------------------------------------------------------


//--- Error Handling -------------------------------------------------------------------------------------------------

// Example: Capturing and interpreting a typical JSON syntax error
//--------------------------------------------------------------------------------------------------------------------
/*
string bad_json = "{\"symbol\": \"EURUSD\" \"price\": 1.07}"; // Missing comma
MQL5_Json::JsonError error;
MQL5_Json::JsonDocument doc = MQL5_Json::JsonParse(bad_json, error);
if (!doc.IsValid())
{
    Print("--- JSON Parse Error Report ---");
    Print(error.ToString());
    // >> Output: Line 1, Col 21: Expected ':' after key in object near '"EURUSD" "price": 1.07'
}
*/
//--------------------------------------------------------------------------------------------------------------------


#include "Core/JsonLibMain.mqh"

#endif // MQL5_JSON

