//+------------------------------------------------------------------+
//| Core/JsonUtils.mqh                                               |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_UTILS
#define MQL5_JSON_UTILS

#include "JsonCore.mqh"

template <typename T>
bool IsInteger( const T s )
{
  //return((s != '.') && (s != 'e') && (s != 'E'));
  return((s != '.') && (s <= '9'));
}

bool IsIntegerFast( const string &num_str )
{
  const int Size = StringLen(num_str);  
  for (int i = 0; i < Size; i++)
    if (!IsInteger(num_str[i]))
      return(false);
  
  return(true);
}
class StringBuilder
{
private:
   ushort m_buffer[];
   int m_size, m_capacity;

   bool EnsureCapacity(int required_extra)
   {
      if(m_size + required_extra <= m_capacity) return true;
      int new_capacity = MathMax(m_size + required_extra, (int)(m_capacity * 1.8) + 64);
      if(ArrayResize(m_buffer, new_capacity) < 0) return false;
      m_capacity = new_capacity;
      return true;
   }
public:
   StringBuilder(int initial_capacity=256) : m_size(0)
   {
      m_capacity = initial_capacity;
      ArrayResize(m_buffer, m_capacity);
   }

   void Clear()
   {
      m_size=0;
   }
   int Length() const
   {
      return m_size;
   }
   string ToString() const
   {
      return (m_size > 0) ? ShortArrayToString(m_buffer, 0, m_size) : "";
   }

   void Append(const string s)
   {
      int len = StringLen(s);
      if(len > 0 && EnsureCapacity(len))
      {
         StringToShortArray(s, m_buffer, m_size);
         m_size += len;
      }
   }

   void AppendChar(ushort c)
   {
      if(EnsureCapacity(1)) m_buffer[m_size++] = c;
   }
};

long JsonHexToInteger(const string h)
{
   long result=0;
   for(int i=0;i<StringLen(h);i++)
   {
      result <<= 4;
      ushort c=StringGetCharacter(h,i);
      if(c>='0'&&c<='9') result += (c-'0');
      else if(c>='a'&&c<='f') result += (c-'a'+10);
      else if(c>='A'&&c<='F') result += (c-'A'+10);
   }
   return result;
}

bool JsonStringToUtf8Bytes(const string &s, uchar &out[], bool with_bom)
{
   ArrayFree(out);
   int len = StringLen(s);
   if(len == 0)
   {
      if(with_bom)
      {
         ArrayResize(out, 3);
         out[0] = 0xEF;
         out[1] = 0xBB;
         out[2] = 0xBF;
      }
      return true;
   }
   ushort shorts[];
   StringToShortArray(s, shorts);
   int bytes_needed = with_bom ? 3 : 0;
   for(int i=0; i<len; i++)
   {
      ushort c = shorts[i];
      if(c < 0x80) bytes_needed++;
      else if(c < 0x800) bytes_needed+=2;
      else bytes_needed+=3;
   }
   if(ArrayResize(out, bytes_needed) < 0) return false;
   int p = with_bom ? 3 : 0;
   if(with_bom)
   {
      out[0]=0xEF;
      out[1]=0xBB;
      out[2]=0xBF;
   }
   for(int i=0; i<len; i++)
   {
      ushort c = shorts[i];
      if(c < 0x80)
      {
         out[p++] = (uchar)c;
      }
      else if(c < 0x800)
      {
         out[p++] = (uchar)(0xC0 | (c >> 6));
         out[p++] = (uchar)(0x80 | (c & 0x3F));
      }
      else
      {
         out[p++] = (uchar)(0xE0 | (c >> 12));
         out[p++] = (uchar)(0x80 | ((c >> 6) & 0x3F));
         out[p++] = (uchar)(0x80 | (c & 0x3F));
      }
   }
   return true;
}

bool CharArrayToString(const uchar &arr[], string &out_str)
{
   int arr_size = ArraySize(arr);
   if(arr_size == 0)
   {
      out_str = "";
      return true;
   }
   int start_pos = 0;
   if(arr_size >= 3 && arr[0] == 0xEF && arr[1] == 0xBB && arr[2] == 0xBF)
   {
      start_pos = 3;
   }
   out_str = ::CharArrayToString(arr, start_pos, -1, CP_UTF8);
   return true;
}


#endif //MQL5_JSON_UTILS


