//+------------------------------------------------------------------+
//| Core/JsonTypes.mqh                                               |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_TYPES
#define MQL5_JSON_TYPES

#include "JsonCore.mqh"
#include "JsonSmartHashMap.mqh"
#include <Arrays/ArrayString.mqh>

class CJsonValue;
class CJsonDocument;

class CJsonNodePage : public CObject
{
public:
   CJsonValue m_nodes[];
   CJsonNodePage(const int page_size)
   {
      if(::ArrayResize(m_nodes, page_size) < 0) {}
   }
   ~CJsonNodePage() {}
};

class CJsonArena
{
private:
   string         m_strings[];
   int            m_string_count;
   int            m_string_capacity;
   CJsonNodePage* m_node_pages[];
   int            m_current_page_idx;
   int            m_current_node_idx;
   int            m_page_size;

   bool AddNewPage()
   {
      CJsonNodePage* new_page = new CJsonNodePage(m_page_size);
      if(CheckPointer(new_page) == POINTER_INVALID || ::ArraySize(new_page.m_nodes) == 0)
      {
         if(CheckPointer(new_page) != POINTER_INVALID) delete new_page;
         return false;
      }
      const int current_num_pages = ::ArraySize(m_node_pages);
      if(::ArrayResize(m_node_pages, current_num_pages + 1) < 0)
      {
         delete new_page;
         return false;
      }
      m_node_pages[current_num_pages] = new_page;
      m_current_page_idx = current_num_pages;
      m_current_node_idx = 0;
      return true;
   }

public:
   CJsonArena(const int initial_string_cap = 128, const int node_page_size = 256)
   {
      m_string_count = 0;
      m_string_capacity = initial_string_cap > 0 ? initial_string_cap : 16;
      if(ArrayResize(m_strings, m_string_capacity) < 0) m_string_capacity = 0;
      m_page_size = node_page_size > 0 ? node_page_size : 256;
      m_current_page_idx = -1;
      m_current_node_idx = m_page_size;
   }

   ~CJsonArena()
   {
      for(int i = 0; i < ::ArraySize(m_node_pages); i++)
      {
         if(CheckPointer(m_node_pages[i]) == POINTER_DYNAMIC) delete m_node_pages[i];
      }
      ArrayFree(m_node_pages);
      ArrayFree(m_strings);
   }

   string AllocateString(const string &str)
   {
      if(m_string_count >= m_string_capacity)
      {
         int new_capacity = (m_string_capacity > 0) ? m_string_capacity * 2 : 16;
         if(::ArrayResize(m_strings, new_capacity) < 0) return str;
         m_string_capacity = new_capacity;
      }
      m_strings[m_string_count] = str;
      return m_strings[m_string_count++];
   }

   CJsonValue* AllocateNode()
   {
      if(m_current_node_idx >= m_page_size)
      {
         if(!AddNewPage()) return NULL;
      }
      return GetPointer(m_node_pages[m_current_page_idx].m_nodes[m_current_node_idx++]);
   }
};

class CJsonDocument
{
public:
   CJsonValue* m_root;
   CJsonArena* m_arena_ptr;

   CJsonDocument(const int string_arena_size = 128, const int node_page_size = 256)
   {
      m_root = NULL;
      m_arena_ptr = new CJsonArena(string_arena_size, node_page_size);
      if(CheckPointer(m_arena_ptr) == POINTER_INVALID) m_arena_ptr = NULL;
   }

   ~CJsonDocument()
   {
      if(CheckPointer(m_arena_ptr) == POINTER_DYNAMIC) delete m_arena_ptr;
   }

   CJsonValue* CreateNode(const ENUM_JSON_TYPE t)
   {
      if(CheckPointer(m_arena_ptr) == POINTER_INVALID) return NULL;
      CJsonValue* node = m_arena_ptr.AllocateNode();
      if(CheckPointer(node) == POINTER_INVALID) return NULL;
      node._Init(t);
      node.m_doc = GetPointer(this);
      return node;
   }
};

class CJsonValue : public CObject
{
public:
   CJsonDocument* m_doc;
   ENUM_JSON_TYPE m_type;
   bool           m_bool;
   long           m_int;
   double         m_double;
   string         m_str;
   string         m_num_str;
   CJsonHashMap* m_obj_map;
   CJsonValue* m_arr[];

public:
   CJsonValue():m_doc(NULL), m_obj_map(NULL), m_type(JSON_INVALID) {}
   ~CJsonValue()
   {
      if(CheckPointer(m_obj_map) == POINTER_DYNAMIC) delete m_obj_map;
   }

   void _Init(const ENUM_JSON_TYPE type)
   {
      m_type = type;
      m_bool = false;
      m_int = 0;
      m_double = 0.0;
      m_str = "";
      m_num_str = "";
      if(ArraySize(m_arr) > 0) ArrayResize(m_arr, 0);
      if(CheckPointer(m_obj_map) == POINTER_DYNAMIC)
      {
         delete m_obj_map;
         m_obj_map = NULL;
      }
   }

   int Size() const
   {
      if(m_type==JSON_ARRAY) return ArraySize(m_arr);
      if(m_type==JSON_OBJECT && m_obj_map!=NULL) return m_obj_map.Size();
      return 0;
   }

   CJsonValue* At(int i) const
   {
      return (m_type==JSON_ARRAY && i>=0 && i<ArraySize(m_arr)) ? m_arr[i] : NULL;
   }
   void Add(CJsonValue* v)
   {
      if(m_type!=JSON_ARRAY || CheckPointer(v)==POINTER_INVALID) return;
      int s = ArraySize(m_arr);
      if(ArrayResize(m_arr, s + 1, 1e3) > 0) m_arr[s] = v;
   }
   bool Remove(int i)
   {
      if(m_type != JSON_ARRAY || i < 0 || i >= ArraySize(m_arr)) return false;
      int s = ArraySize(m_arr);
      if (i < s - 1) ArrayCopy(m_arr, m_arr, i, i + 1, s - 1 - i);
      ArrayResize(m_arr, s - 1);
      return true;
   }

   CJsonValue* Get(const string &k) const
   {
      return (m_type==JSON_OBJECT && m_obj_map!=NULL)?m_obj_map.Get(k):NULL;
   }
   bool HasKey(const string &k) const
   {
      return (m_type==JSON_OBJECT && m_obj_map!=NULL && m_obj_map.Get(k)!=NULL);
   }
   void GetKeys(string &k[]) const
   {
      ArrayFree(k);
      if(m_type==JSON_OBJECT && m_obj_map!=NULL)
      {
         int c;
         m_obj_map.GetKeys(k,c);
      }
   }
   bool Remove(const string &k)
   {
      return (m_type==JSON_OBJECT && m_obj_map!=NULL)?m_obj_map.Remove(k):false;
   }
   void Set(const string &key, CJsonValue* v)
   {
      if(m_type != JSON_OBJECT || CheckPointer(v)==POINTER_INVALID) return;
      if(m_obj_map == NULL)
      {
         m_obj_map = new CJsonHashMap();
         if(CheckPointer(m_obj_map) == POINTER_INVALID)
         {
            m_obj_map=NULL;
            return;
         }
      }
      if (m_doc == NULL || m_doc.m_arena_ptr == NULL) return;
      string arena_key = m_doc.m_arena_ptr.AllocateString(key);
      m_obj_map.Set(arena_key, v);
   }

   CJsonValue* Clone(CJsonDocument *target_doc) const;
};

CJsonValue* CJsonValue::Clone(CJsonDocument *target_doc) const
{
   if(CheckPointer(target_doc)==POINTER_INVALID) return NULL;
   CJsonValue *clone = target_doc.CreateNode(m_type);
   if(!clone) return NULL;
   clone.m_bool=m_bool;
   clone.m_int=m_int;
   clone.m_double=m_double;
   if(StringLen(m_str)>0) clone.m_str=target_doc.m_arena_ptr.AllocateString(m_str);
   if(StringLen(m_num_str)>0) clone.m_num_str=target_doc.m_arena_ptr.AllocateString(m_num_str);
   if(m_type == JSON_ARRAY)
   {
      int s=ArraySize(m_arr);
      if(s>0 && ArrayResize(clone.m_arr,s)>=0)
      {
         for(int i=0;i<s;i++) clone.m_arr[i]=m_arr[i].Clone(target_doc);
      }
   }
   else if(m_type==JSON_OBJECT && m_obj_map!=NULL && m_obj_map.Size()>0)
   {
      string k[];
      CJsonValue*v[];
      int c=0;
      m_obj_map.GetAllPairs(k,v,c);
      for(int i=0;i<c;i++)
      {
         if(CheckPointer(v[i]) != POINTER_INVALID) clone.Set(k[i],v[i].Clone(target_doc));
      }
   }
   return clone;
}

#endif // MQL5_JSON_TYPES
