//+------------------------------------------------------------------+
//| Core/JsonSmartHashMap.mqh                                        |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_HASHMAP
#define MQL5_JSON_HASHMAP
#property strict
#include <Arrays/ArrayString.mqh>
#include "JsonCore.mqh"

class CJsonValue;
uint XXH32_FromString(const string &key, uint seed)
{
   const uint P1=2654435761,P2=2246822519,P3=3266489917,P4=668265263,P5=374761393;
   uint h32;
   const int len = StringLen(key);
   int index = 0;
   if(len >= 4)
   {
      h32 = seed + P5;
      h32 += (uint)len * 2;
      for(; index <= len - 4; index += 4)
      {
         uint k1 = (uint)StringGetCharacter(key, index) |
                   ((uint)StringGetCharacter(key, index+1) << 16);
         k1 *= P2;
         k1 = ((k1 << 13) | (k1 >> 19));
         k1 *= P1;
         h32 += k1;
         h32 = ((h32 << 17) | (h32 >> 15));
         h32 *= P4;
         k1 =      (uint)StringGetCharacter(key, index+2) |
                   ((uint)StringGetCharacter(key, index+3) << 16);
         k1 *= P2;
         k1 = ((k1 << 13) | (k1 >> 19));
         k1 *= P1;
         h32 += k1;
         h32 = ((h32 << 17) | (h32 >> 15));
         h32 *= P4;
      }
   }
   else
   {
      h32 = seed + P5;
   }
   h32 += (uint)len * 2;
   while(index < len)
   {
      h32 += (uint)StringGetCharacter(key, index) * P3;
      h32 = ((h32 << 17) | (h32 >> 15)) * P4;
      index++;
   }
   h32 ^= h32 >> 15;
   h32 *= P2;
   h32 ^= h32 >> 13;
   h32 *= P3;
   h32 ^= h32 >> 16;
   return h32;
}


class CJsonHashMap
{
private:
   enum EntryState { EMPTY_VAL, ACTIVE, DELETED };
   struct Entry
   {
      string      key;
      CJsonValue* value;
      uint        hash;
      EntryState  state;
   };

   Entry   m_entries[];
   int     m_capacity;
   int     m_size;
   int     m_tombstones;
   uint    m_seed;

   static const double LOAD_FACTOR_THRESHOLD;

   int FindEntryIndex(const string &key, const uint hash) const
   {
      if(m_capacity == 0) return -1;
      uint index = hash % m_capacity;
      int tombstone_index = -1;
      for(int i = 0; i < m_capacity; i++)
      {
         Entry entry = m_entries[index];
         if(entry.state == EMPTY_VAL)
         {
            return (tombstone_index != -1) ? tombstone_index : (int)index;
         }
         if(entry.state == DELETED)
         {
            if(tombstone_index == -1) tombstone_index = (int)index;
         }
         else if(entry.hash == hash && entry.key == key)
         {
            return (int)index;
         }
         index = (index + 1) % m_capacity; // Linear probing
      }
      return (tombstone_index != -1) ? tombstone_index : -1;
   }

   void Resize(int new_capacity)
   {
      if (new_capacity == 0 || new_capacity <= m_size) return;
      Entry old_entries[];
      int old_capacity = m_capacity;
      if(old_capacity > 0)
      {
         if(ArrayResize(old_entries, old_capacity) < 0) return;
         for(int i = 0; i < old_capacity; i++) old_entries[i] = m_entries[i];
      }
      if(ArrayResize(m_entries, new_capacity) < 0) return;
      m_capacity = new_capacity;
      m_size = 0;
      m_tombstones = 0;
      for(int i = 0; i < m_capacity; i++) m_entries[i].state = EMPTY_VAL;
      if(old_capacity > 0)
      {
         for(int i = 0; i < old_capacity; i++)
         {
            if(old_entries[i].state == ACTIVE) Set(old_entries[i].key, old_entries[i].value);
         }
      }
   }

public:
   CJsonHashMap(int initial_capacity = 8)
   {
      m_size = 0;
      m_tombstones = 0;
      m_capacity = 0;
      ArrayFree(m_entries);
      #ifdef __MQL5__
      m_seed = (uint)GetTickCount64();
      #else 
      m_seed = (uint)GetTickCount();
      #endif
      if(initial_capacity > 0) Resize(initial_capacity);
   }

   int Size() const
   {
      return m_size;
   }

   void Set(const string &key, CJsonValue* value)
   {
      if((m_size + m_tombstones + 1) > m_capacity * LOAD_FACTOR_THRESHOLD)
      {
         Resize((m_capacity > 0) ? m_capacity * 2 : 8);
      }
      uint hash = XXH32_FromString(key, m_seed);
      int index = FindEntryIndex(key, hash);
      if(index == -1) return;
      bool is_new = (m_entries[index].state != ACTIVE);
      if(is_new)
      {
         if(m_entries[index].state == DELETED) m_tombstones--;
         m_size++;
         m_entries[index].key = key;
         m_entries[index].hash = hash;
         m_entries[index].state = ACTIVE;
      }
      m_entries[index].value = value;
   }

   CJsonValue* Get(const string &key) const
   {
      if(m_size == 0) return NULL;
      uint hash = XXH32_FromString(key, m_seed);
      uint index = hash % m_capacity;
      for(int i = 0; i < m_capacity; i++)
      {
         Entry entry = m_entries[index];
         if(entry.state == EMPTY_VAL) return NULL;
         if(entry.state == ACTIVE && entry.hash == hash && entry.key == key) return entry.value;
         index = (index + 1) % m_capacity;
      }
      return NULL;
   }

   bool Remove(const string &key)
   {
      if(m_size == 0) return false;
      uint hash = XXH32_FromString(key, m_seed);
      int index = FindEntryIndex(key, hash);
      if(index == -1 || m_entries[index].state != ACTIVE) return false;
      m_entries[index].state = DELETED;
      m_size--;
      m_tombstones++;
      return true;
   }

   void GetAllPairs(string &keys[], CJsonValue* &values[], int &count) const
   {
      count = 0;
      ArrayFree(keys);
      ArrayFree(values);
      if(m_size == 0) return;
      if(ArrayResize(keys, m_size) < 0 || ArrayResize(values, m_size) < 0) return;
      for(int i = 0; i < m_capacity; i++)
      {
         if(m_entries[i].state == ACTIVE)
         {
            keys[count] = m_entries[i].key;
            values[count] = m_entries[i].value;
            count++;
         }
      }
   }

   void GetKeys(string &out_keys[], int &out_count) const
   {
      out_count = 0;
      ArrayFree(out_keys);
      if (m_size == 0) return;
      if(ArrayResize(out_keys, m_size) < 0) return;
      for(int i = 0; i < m_capacity; i++)
      {
         if(m_entries[i].state == ACTIVE)
         {
            out_keys[out_count] = m_entries[i].key;
            out_count++;
         }
      }
   }
};

const double CJsonHashMap::LOAD_FACTOR_THRESHOLD = 0.75;

#endif // MQL5_JSON_HASHMAP
