//+------------------------------------------------------------------+
//| Core/JsonSerializer.mqh                                          |
//| (Version 12.2 - Final Production Build by AI on 2025-09-03)      |
//|                                                                  |
//| This version is the definitive, production-grade solution.       |
//| It combines the correct serialization logic from older versions  |
//| with a robust, native string implementation, completely          |
//| eliminating all previously identified bugs and risks:            |
//|                                                                  |
//| [FIX-CRITICAL] Eliminates the root cause of the "Extra data"     |
//|                error by replacing the faulty custom StringBuilder|
//|                with safe, native MQL5 string operations.         |
//| [FIX-CRITICAL] Corrects the fatal bug of serializing character   |
//|                codes instead of the characters themselves.       |
//| [FIX-MQL5]     Resolves compiler warnings by using integer codes |
//|                for non-standard escape sequences ('\b', '\f').   |
//| [ROBUST]       Implements JSON-standard compliant string escaping|
//|                for a stable and predictable output.              |
//+------------------------------------------------------------------+
#property strict
#ifndef MQL5_JSON_SERIALIZER
#define MQL5_JSON_SERIALIZER

#include "JsonCore.mqh"
#include "JsonTypes.mqh"

//+------------------------------------------------------------------+
//| CJsonSerializer - 负责将CJsonValue对象树转换为JSON字符串         |
//|                   (经过彻底重构，达到了生产级稳定性和正确性)      |
//+------------------------------------------------------------------+
class CJsonSerializer
{
private:
   string m_result_str;       // 用于构建最终JSON字符串
   bool   m_pretty;           // 是否启用美化格式（带缩进和换行）
   bool   m_escape_non_ascii; // 是否转义非ASCII字符

private:
   // --- 核心辅助函数 ---

   // 根据当前层级添加缩进（仅在m_pretty为true时生效）
   void Indent(int level);
   
   // [FIXED] 关键的字符串序列化函数，已完全重写
   void SerializeString(const string &s);
   
   // --- 主要递归函数 ---
   
   void SerializeValue(const CJsonValue *value, int level);
   void SerializeArray(const CJsonValue *value, int level);
   void SerializeObject(const CJsonValue *value, int level);

public:
   // [对外唯一接口]
   string Serialize(const CJsonValue *root, bool pretty, bool escape_non_ascii);
};


//+------------------------------------------------------------------+
//| [主入口] 开始序列化流程                                          |
//+------------------------------------------------------------------+
string CJsonSerializer::Serialize(const CJsonValue *root, bool p, bool e)
{
   // 每次调用都重置内部状态，确保线程安全和结果纯净
   m_result_str = ""; 
   m_pretty = p;
   m_escape_non_ascii = e;
   
   SerializeValue(root, 0); // 从根节点开始递归
   
   return m_result_str;
}

//+------------------------------------------------------------------+
//| [辅助] 添加缩进                                                  |
//+------------------------------------------------------------------+
void CJsonSerializer::Indent(int level)
{
   if(m_pretty)
   {
      m_result_str += "\n";
      for(int i = 0; i < level * 2; i++)
      {
         m_result_str += " ";
      }
   }
}

//+------------------------------------------------------------------+
//| [核心修复] 序列化字符串值，并正确处理转义字符                    |
//+------------------------------------------------------------------+
void CJsonSerializer::SerializeString(const string &s)
{
   m_result_str += "\"";
   
   int len = StringLen(s);
   for(int i = 0; i < len; i++)
   {
      ushort ch = StringGetCharacter(s, i);

      switch(ch)
      {
         case '"':  m_result_str += "\\\""; break;
         case '\\': m_result_str += "\\\\"; break;
         case '\n': m_result_str += "\\n";  break;
         case '\r': m_result_str += "\\r";  break;
         case '\t': m_result_str += "\\t";  break;

         // [MQL5 COMPILER FIX] MQL5 does not recognize '\b' or '\f'.
         // Use their integer ASCII codes directly for compatibility.
         case 8:    m_result_str += "\\b";  break; // Backspace
         case 12:   m_result_str += "\\f";  break; // Form Feed
         
         default:
            // 处理控制字符和需要转义的非ASCII字符
            if(ch < 32 || (ch > 127 && m_escape_non_ascii))
            {
               m_result_str += StringFormat("\\u%04x", ch);
            }
            else
            {
               // [CRITICAL LOGIC FIX]
               // This is the correct way to append a character.
               // Appending the character code as a string (e.g., ShortToString(97) -> "97") is a fatal bug.
               // StringFormat with %c is the standard, safe method.
               m_result_str += StringFormat("%c", ch);
            }
            break;
      }
   }
   
   m_result_str += "\"";
}

//+------------------------------------------------------------------+
//| [递归] 序列化JSON数组                                            |
//+------------------------------------------------------------------+
void CJsonSerializer::SerializeArray(const CJsonValue *value, int level)
{
   m_result_str += "[";
   
   int size = ArraySize(value.m_arr);
   if(size > 0)
   {
      // A robust loop with a prefix comma handles boundaries perfectly.
      for(int i = 0; i < size; i++)
      {
         if(i > 0)
         {
            m_result_str += ",";
         }
         Indent(level + 1);
         SerializeValue(value.m_arr[i], level + 1);
      }
      Indent(level);
   }
   
   m_result_str += "]";
}

//+------------------------------------------------------------------+
//| [递归] 序列化JSON对象                                            |
//+------------------------------------------------------------------+
void CJsonSerializer::SerializeObject(const CJsonValue *value, int level)
{
   m_result_str += "{";
   
   if(CheckPointer(value.m_obj_map) != POINTER_INVALID && value.m_obj_map.Size() > 0)
   {
      string keys[];
      CJsonValue *values[];
      int num_entries = 0;
      
      // Get all key-value pairs at once for better performance.
      value.m_obj_map.GetAllPairs(keys, values, num_entries);
      
      for(int i = 0; i < num_entries; i++)
      {
         if(i > 0)
         {
            m_result_str += ",";
         }
         Indent(level + 1);
         
         // Serialize the key (which is always a string in JSON)
         SerializeString(keys[i]);
         
         // Add the colon separator
         m_result_str += m_pretty ? ": " : ":";
            
         // Recursively serialize the value
         SerializeValue(values[i], level + 1);
      }
      Indent(level);
   }
   
   m_result_str += "}";
}


//+------------------------------------------------------------------+
//| [递归分发器] 根据值的类型选择相应的序列化方法                     |
//+------------------------------------------------------------------+
void CJsonSerializer::SerializeValue(const CJsonValue *v, int level)
{
   // Robust handling of invalid pointers or explicit NULL types.
   if(CheckPointer(v) == POINTER_INVALID || v.m_type == JSON_NULL)
   {
      m_result_str += "null";
      return;
   }
   
   switch(v.m_type)
   {
      case JSON_BOOL:
         m_result_str += (v.m_bool ? "true" : "false");
         break;
      
      case JSON_INT:
      case JSON_DOUBLE:
         // Using the pre-formatted number string is most efficient.
         m_result_str += v.m_num_str;
         break;
         
      case JSON_STRING:
         SerializeString(v.m_str);
         break;
         
      case JSON_ARRAY:
         SerializeArray(v, level);
         break;
         
      case JSON_OBJECT:
         SerializeObject(v, level);
         break;
         
      default:
         // Safely output "null" for any unknown or unexpected types.
         m_result_str += "null";
         break;
   }
}

#endif // MQL5_JSON_SERIALIZER