//+------------------------------------------------------------------+
//| Core/JsonPath.mqh                                                |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_PATH
#define MQL5_JSON_PATH

#include "JsonNode.mqh"
#include "JsonTypes.mqh"

template<typename T>
void FreePointerArray(T* &arr[])
{
   int size = ArraySize(arr);
   for(int i = 0; i < size; i++)
   {
      if(CheckPointer(arr[i]) == POINTER_DYNAMIC)
         delete arr[i];
   }
   ArrayFree(arr);
}

enum ENUM_JSONPATH_TOKEN_TYPE
{
   TOKEN_END, TOKEN_ROOT, TOKEN_CURRENT, TOKEN_DOT, TOKEN_RECURSIVE_DESCENT, TOKEN_WILDCARD,
   TOKEN_LBRACKET, TOKEN_RBRACKET, TOKEN_LPAREN, TOKEN_RPAREN, TOKEN_COMMA, TOKEN_COLON, TOKEN_QUESTION,
   TOKEN_IDENTIFIER, TOKEN_STRING_LITERAL, TOKEN_INT_LITERAL,
   TOKEN_EQ, TOKEN_NE, TOKEN_LT, TOKEN_LE, TOKEN_GT, TOKEN_GE, TOKEN_AND, TOKEN_OR
};

class CJsonPathToken : public CObject
{
public:
   ENUM_JSONPATH_TOKEN_TYPE type;
   string s_val;
   long   l_val;
   CJsonPathToken() {}
};

class CJsonPathLexer
{
private:
   string m_path;
   int    m_pos, m_len;
   ushort Peek(int offset=0) const
   {
      return (m_pos + offset < m_len) ? StringGetCharacter(m_path, m_pos + offset) : 0;
   }
   ushort Next()
   {
      return (m_pos < m_len) ? StringGetCharacter(m_path, m_pos++) : 0;
   }
   void AddToken(CJsonPathToken* &tokens[], CJsonPathToken* t)
   {
      int size = ArraySize(tokens);
      ArrayResize(tokens, size + 1);
      tokens[size] = t;
   }
public:
   bool Tokenize(const string &path, CJsonPathToken* &tokens[], JsonError &error)
   {
      m_path = path;
      m_len = StringLen(m_path);
      m_pos = 0;
      ArrayFree(tokens);
      while(m_pos < m_len)
      {
         ushort c = Peek();
         if(c==' '||c=='\r'||c=='\n'||c=='\t')
         {
            Next();
            continue;
         }
         CJsonPathToken *t = new CJsonPathToken();
         if(t == NULL) return false;
         if(c == '$')
         {
            t.type = TOKEN_ROOT;
            Next();
         }
         else if(c == '@')
         {
            t.type = TOKEN_CURRENT;
            Next();
         }
         else if(c == '.' && Peek(1) == '.')
         {
            t.type = TOKEN_RECURSIVE_DESCENT;
            Next();
            Next();
         }
         else if(c == '.')
         {
            t.type = TOKEN_DOT;
            Next();
         }
         else if(c == '*')
         {
            t.type = TOKEN_WILDCARD;
            Next();
         }
         else if(c == '[')
         {
            t.type = TOKEN_LBRACKET;
            Next();
         }
         else if(c == ']')
         {
            t.type = TOKEN_RBRACKET;
            Next();
         }
         else if(c == '(')
         {
            t.type = TOKEN_LPAREN;
            Next();
         }
         else if(c == ')')
         {
            t.type = TOKEN_RPAREN;
            Next();
         }
         else if(c == ',')
         {
            t.type = TOKEN_COMMA;
            Next();
         }
         else if(c == ':')
         {
            t.type = TOKEN_COLON;
            Next();
         }
         else if(c == '?')
         {
            t.type = TOKEN_QUESTION;
            Next();
         }
         else if(c == '\'' || c == '"')
         {
            t.type = TOKEN_STRING_LITERAL;
            ushort quote = Next();
            int start = m_pos;
            while(Peek() != quote)
            {
               if(m_pos >= m_len)
               {
                  error.message="Unterminated string literal";
                  delete t;
                  FreePointerArray(tokens);
                  return false;
               }
               if(Peek() == '\\') Next();
               Next();
            }
            t.s_val = StringSubstr(m_path, start, m_pos - start);
            Next();
         }
         else if((c >= '0' && c <= '9') || (c == '-' && Peek(1) >= '0' && Peek(1) <= '9'))
         {
            t.type = TOKEN_INT_LITERAL;
            int start = m_pos;
            Next();
            while(Peek() >= '0' && Peek() <= '9') Next();
            t.l_val = StringToInteger(StringSubstr(m_path, start, m_pos - start));
         }
         else if(c == '=' && Peek(1) == '=')
         {
            t.type = TOKEN_EQ;
            Next();
            Next();
         }
         else if(c == '!' && Peek(1) == '=')
         {
            t.type = TOKEN_NE;
            Next();
            Next();
         }
         else if(c == '<' && Peek(1) == '=')
         {
            t.type = TOKEN_LE;
            Next();
            Next();
         }
         else if(c == '>' && Peek(1) == '=')
         {
            t.type = TOKEN_GE;
            Next();
            Next();
         }
         else if(c == '>')
         {
            t.type = TOKEN_GT;
            Next();
         }
         else if(c == '<')
         {
            t.type = TOKEN_LT;
            Next();
         }
         else if(c == '&' && Peek(1) == '&')
         {
            t.type = TOKEN_AND;
            Next();
            Next();
         }
         else if(c == '|' && Peek(1) == '|')
         {
            t.type = TOKEN_OR;
            Next();
            Next();
         }
         else if(c=='_'||(c>='a'&&c<='z')||(c>='A'&&c<='Z'))
         {
            t.type=TOKEN_IDENTIFIER;
            int start=m_pos;
            while(m_pos < m_len)
            {
               ushort ic=Peek();
               if(ic=='_'||ic=='-'||(ic>='a'&&ic<='z')||(ic>='A'&&ic<='Z')||(ic>='0'&&ic<='9'))Next();
               else break;
            }
            t.s_val = StringSubstr(m_path, start, m_pos-start);
         }
         else
         {
            error.message="Invalid character";
            delete t;
            FreePointerArray(tokens);
            return false;
         }
         AddToken(tokens, t);
      }
      return true;
   }
};

class CJsonPathEvaluatorAdvanced
{
private:
   CJsonPathToken* m_tokens[];
   int m_token_pos;
   JsonNode m_root;

   CJsonPathToken* Peek(int offset=0)
   {
      int p = m_token_pos + offset;
      return (p >= 0 && p < ArraySize(m_tokens)) ? m_tokens[p] : NULL;
   }
   CJsonPathToken* Next()
   {
      return (m_token_pos < ArraySize(m_tokens)) ? m_tokens[m_token_pos++] : NULL;
   }

   void CollectUnique(const JsonNode &node, JsonNode* &list[])
   {
      if(!node.IsValid()) return;
      for(int i = 0; i < ArraySize(list); i++) if(list[i].m_value == node.m_value) return;
      int size = ArraySize(list);
      ArrayResize(list, size + 1);
      list[size] = new JsonNode(node);
   }

   void Scan(const JsonNode &node, JsonNode* &list[])
   {
      CollectUnique(node, list);
      if(node.IsObject())
      {
         string k[];
         node.GetKeys(k);
         for(int i=0;i<ArraySize(k);i++)Scan(node.Get(k[i]),list);
      }
      else if(node.IsArray())
      {
         for(int i=0;i<node.Size();i++)Scan(node.At(i),list);
      }
   }

   JsonNode ResolvePath(const JsonNode &ctx)
   {
      JsonNode current=ctx;
      while(Peek()!=NULL && (Peek().type==TOKEN_DOT || (Peek().type==TOKEN_LBRACKET && Peek(1)!=NULL && Peek(1).type==TOKEN_STRING_LITERAL) ))
      {
         if(Next().type == TOKEN_DOT)
         {
            CJsonPathToken *p=Next();
            if(p==NULL || p.type!=TOKEN_IDENTIFIER) return JsonNode();
            if(!current.IsObject()) return JsonNode();
            current=current.Get(p.s_val);
         }
         else
         {
            current=current.Get(Next().s_val);
            if(Next()==NULL||Next().type!=TOKEN_RBRACKET)return JsonNode();
         }
         if(!current.IsValid())return JsonNode();
      }
      return current;
   }

   bool EvaluateFilter(const JsonNode &ctx, bool &result)
   {
      return EvaluateOr(ctx, result);
   }
   bool EvaluateOr(const JsonNode &ctx, bool &or_res)
   {
      bool and_res;
      if(!EvaluateAnd(ctx, and_res)) return false;
      or_res=and_res;
      while(Peek()!=NULL&&Peek().type==TOKEN_OR)
      {
         Next();
         if(!EvaluateAnd(ctx, and_res))return false;
         or_res=or_res||and_res;
      }
      return true;
   }
   bool EvaluateAnd(const JsonNode &ctx, bool &and_res)
   {
      bool cmp_res;
      if(!EvaluateComparison(ctx, cmp_res)) return false;
      and_res=cmp_res;
      while(Peek()!=NULL&&Peek().type==TOKEN_AND)
      {
         Next();
         if(!EvaluateComparison(ctx, cmp_res))return false;
         and_res=and_res&&cmp_res;
      }
      return true;
   }
   bool EvaluateComparison(const JsonNode &ctx, bool &cmp_res)
   {
      JsonNode left=EvaluatePrimary(ctx);
      if(Peek()==NULL||(Peek().type<TOKEN_EQ||Peek().type>TOKEN_GE))
      {
         cmp_res=left.IsValid()&&!left.IsNull();
         return true;
      }
      CJsonPathToken *op=Next(), *right_token=Peek();
      if(left.IsNumber()&&right_token!=NULL&&right_token.type==TOKEN_INT_LITERAL)
      {
         double l=left.AsDouble();
         long r=Next().l_val;
         switch(op.type)
         {
         case TOKEN_EQ:
            cmp_res=(l==r);
            return true;
         case TOKEN_NE:
            cmp_res=(l!=r);
            return true;
         case TOKEN_LT:
            cmp_res=(l<r);
            return true;
         case TOKEN_LE:
            cmp_res=(l<=r);
            return true;
         case TOKEN_GT:
            cmp_res=(l>r);
            return true;
         case TOKEN_GE:
            cmp_res=(l>=r);
            return true;
         }
      }
      else if(left.IsString()&&right_token!=NULL&&right_token.type==TOKEN_STRING_LITERAL)
      {
         string l=left.AsString(),r=Next().s_val;
         switch(op.type)
         {
         case TOKEN_EQ:
            cmp_res=(l==r);
            return true;
         case TOKEN_NE:
            cmp_res=(l!=r);
            return true;
         }
      }
      cmp_res = false;
      return true;
   }
   JsonNode EvaluatePrimary(const JsonNode&ctx)
   {
      if(Peek()!=NULL&&Peek().type==TOKEN_CURRENT)
      {
         Next();
         return ResolvePath(ctx);
      }
      return JsonNode();
   }

   void HandleDot(JsonNode* &current_list[], JsonNode* &next_list[])
   {
      Next();
      CJsonPathToken*k=Next();
      if(k == NULL) return;
      for(int i=0;i<ArraySize(current_list);i++)
      {
         JsonNode*n=current_list[i];
         if(!n.IsObject())continue;
         if(k.type==TOKEN_WILDCARD)
         {
            string keys[];
            n.GetKeys(keys);
            for(int j=0;j<ArraySize(keys);j++)CollectUnique(n.Get(keys[j]),next_list);
         }
         else CollectUnique(n.Get(k.s_val),next_list);
      }
   }
   void HandleRecursiveDescent(JsonNode* &current_list[], JsonNode* &next_list[])
   {
      Next();
      CJsonPathToken*k=Next();
      if(k == NULL) return;
      JsonNode*scan_res[];
      for(int i=0;i<ArraySize(current_list);i++)Scan(*current_list[i],scan_res);
      for(int i=0;i<ArraySize(scan_res);i++)
      {
         JsonNode*sn=scan_res[i];
         if(!sn.IsObject())continue;
         if(k.type==TOKEN_WILDCARD)
         {
            string keys[];
            sn.GetKeys(keys);
            for(int j=0;j<ArraySize(keys);j++)CollectUnique(sn.Get(keys[j]),next_list);
         }
         else if(sn.HasKey(k.s_val))CollectUnique(sn.Get(k.s_val),next_list);
      }
      FreePointerArray(scan_res);
   }
   void HandleSubscript(JsonNode* &current_list[], JsonNode* &next_list[])
   {
      Next();
      if(Peek()==NULL) return;
      if(Peek().type==TOKEN_QUESTION)
      {
         Next();
         if(Next().type!=TOKEN_LPAREN)return;
         int base_pos=m_token_pos;
         for(int i=0;i<ArraySize(current_list);i++)
         {
            JsonNode*n=current_list[i];
            JsonNode*cands[];
            if(n.IsArray())
            {
               for(int j=0;j<n.Size();j++)
               {
                  int s=ArraySize(cands);
                  ArrayResize(cands,s+1);
                  cands[s]=new JsonNode(n.At(j));
               }
            }
            else
            {
               ArrayResize(cands,1);
               cands[0]=new JsonNode(*n);
            }
            for(int j=0;j<ArraySize(cands);j++)
            {
               m_token_pos=base_pos;
               bool res;
               if(EvaluateFilter(*cands[j],res)&&res)CollectUnique(*cands[j],next_list);
            }
            FreePointerArray(cands);
         }
         while(Peek()!=NULL&&Next().type!=TOKEN_RBRACKET);
         return;
      }
      if(Peek().type==TOKEN_WILDCARD)
      {
         Next();
         if(Peek() == NULL || Peek().type != TOKEN_RBRACKET) return;
         Next();
         for(int i=0;i<ArraySize(current_list);i++)
         {
            JsonNode*n=current_list[i];
            if(n.IsArray())for(int j=0;j<n.Size();j++)CollectUnique(n.At(j),next_list);
            else if(n.IsObject())
            {
               string k[];
               n.GetKeys(k);
               for(int j=0;j<ArraySize(k);j++)CollectUnique(n.Get(k[j]),next_list);
            }
         }
         return;
      }
      CJsonPathToken*indices[];
      bool is_slice=false;
      while(Peek()!=NULL&&Peek().type!=TOKEN_RBRACKET)
      {
         if(Peek().type==TOKEN_COLON)
         {
            is_slice=true;
            int sz=ArraySize(indices);
            ArrayResize(indices,sz+1);
            indices[sz]=NULL;
            Next();
            continue;
         }
         if(Peek().type==TOKEN_INT_LITERAL)
         {
            int sz=ArraySize(indices);
            ArrayResize(indices,sz+1);
            indices[sz]=Next();
         }
         if(Peek()!=NULL&&Peek().type==TOKEN_COMMA)Next();
         if(Peek()!=NULL&&Peek().type==TOKEN_COLON)is_slice=true;
      }
      Next();
      for(int i=0;i<ArraySize(current_list);i++)
      {
         JsonNode*n=current_list[i];
         if(!n.IsArray())continue;
         int size=n.Size();
         if(is_slice)
         {
            long start, end, step = 1;
            bool start_set = false, end_set = false;
            CJsonPathToken* t_start = (ArraySize(indices) > 0) ? indices[0] : NULL;
            CJsonPathToken* t_end   = (ArraySize(indices) > 1) ? indices[1] : NULL;
            CJsonPathToken* t_step  = (ArraySize(indices) > 2) ? indices[2] : NULL;
            if(t_step != NULL) step = t_step.l_val;
            if(step == 0) continue;
            if(t_start != NULL)
            {
               start = t_start.l_val;
               start_set = true;
            }
            if(t_end != NULL)
            {
               end = t_end.l_val;
               end_set = true;
            }
            if (step > 0)
            {
               if (!start_set) start = 0;
               if (!end_set) end = size;
            }
            else
            {
               if (!start_set) start = size - 1;
               if (!end_set) end = -1;
            }
            if(start < 0) start += size;
            if(end < 0) end += size;
            if (step > 0)
            {
               start = MathMax(0, MathMin(size, start));
               end = MathMax(0, MathMin(size, end));
            }
            else
            {
               start = MathMax(-1, MathMin(size - 1, start));
               end = MathMax(-1, MathMin(size - 1, end));
            }
            if(step > 0)
            {
               for(long j = start; j < end; j += step)
                  CollectUnique(n.At((int)j), next_list);
            }
            else
            {
               for(long j = start; j > end; j += step)
                  CollectUnique(n.At((int)j), next_list);
            }
         }
         else
         {
            for(int j=0;j<ArraySize(indices);j++)
            {
               long idx=indices[j].l_val;
               if(idx<0)idx+=size;
               if(idx>=0&&idx<size)CollectUnique(n.At((int)idx),next_list);
            }
         }
      }
      ArrayFree(indices);
   }

public:
   int Evaluate(JsonNode &out[], const JsonNode &root, const string &path, JsonError &error)
   {
      m_root = root;
      CJsonPathLexer lexer;
      if (!lexer.Tokenize(path, m_tokens, error)) return 0;
      if (ArraySize(m_tokens) == 0) return 0;
      JsonNode* current_list[];
      m_token_pos = 0;
      if (m_tokens[0].type == TOKEN_ROOT)
      {
         CollectUnique(m_root, current_list);
         Next();
      }
      else
      {
         error.message = "Path must start with '$'";
         FreePointerArray(m_tokens);
         return 0;
      }
      while(m_token_pos < ArraySize(m_tokens))
      {
         JsonNode* next_list[];
         CJsonPathToken *selector = Peek();
         if(selector==NULL)break;
         if(selector.type == TOKEN_DOT) HandleDot(current_list, next_list);
         else if(selector.type == TOKEN_LBRACKET) HandleSubscript(current_list, next_list);
         else if(selector.type == TOKEN_RECURSIVE_DESCENT) HandleRecursiveDescent(current_list, next_list);
         else
         {
            error.message="Unexpected token";
            FreePointerArray(m_tokens);
            FreePointerArray(current_list);
            FreePointerArray(next_list);
            return 0;
         }
         FreePointerArray(current_list);
         ArrayCopy(current_list, next_list, 0, 0, WHOLE_ARRAY);
         ArrayFree(next_list);
      }
      int total = ArraySize(current_list);
      if(total > 0 && ArrayResize(out, total) >= 0)
      {
         for(int i=0;i<total;i++) out[i] = *current_list[i];
      }
      else
      {
         total = 0;
      }
      FreePointerArray(m_tokens);
      FreePointerArray(current_list);
      return total;
   }
};

#endif //MQL5_JSON_PATH

