//+------------------------------------------------------------------+
//| Core/JsonNode.mqh                                                |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_NODE
#define MQL5_JSON_NODE

#include "JsonCore.mqh"
#include "JsonTypes.mqh"

class JsonNode
{
private:
   CJsonValue* CreateAndSetValue(long v);
   CJsonValue* CreateAndSetValue(double v);
   CJsonValue* CreateAndSetValue(bool v);
   CJsonValue* CreateAndSetValue(const string v);

public:
   CJsonValue* m_value;

   JsonNode():m_value(NULL) {}
   JsonNode(CJsonValue *v):m_value(v) {}
   JsonNode(const JsonNode &o):m_value(o.m_value) {}

   bool IsValid() const
   {
      return CheckPointer(m_value)!=POINTER_INVALID;
   }

   ENUM_JSON_TYPE GetType() const
   {
      return IsValid() ? m_value.m_type : JSON_INVALID;
   }

   int Size() const
   {
      return IsValid() ? m_value.Size() : 0;
   }

   bool IsNull()   const
   {
      return GetType()==JSON_NULL;
   }
   bool IsBool()   const
   {
      return GetType()==JSON_BOOL;
   }
   bool IsNumber() const
   {
      ENUM_JSON_TYPE t=GetType();
      return t==JSON_INT||t==JSON_DOUBLE;
   }
   bool IsInt()    const
   {
      return GetType()==JSON_INT;
   }
   bool IsDouble() const
   {
      return GetType()==JSON_DOUBLE;
   }
   bool IsString() const
   {
      return GetType()==JSON_STRING;
   }
   bool IsArray()  const
   {
      return GetType()==JSON_ARRAY;
   }
   bool IsObject() const
   {
      return GetType()==JSON_OBJECT;
   }

   bool HasKey(const string k) const
   {
      return IsValid() ? m_value.HasKey(k) : false;
   }

   void GetKeys(string &k[]) const
   {
      if(IsValid()) m_value.GetKeys(k);
      else ArrayFree(k);
   }

   string AsString(const string d="") const
   {
      if(!IsValid())return d;
      switch(m_value.m_type)
      {
      case JSON_STRING:
         return m_value.m_str;
      case JSON_INT:
         return IntegerToString(m_value.m_int);
      case JSON_DOUBLE:
         return DoubleToString(m_value.m_double,-1);
      case JSON_BOOL:
         return m_value.m_bool?"true":"false";
      default:
         return d;
      }
   }

   long AsInt(long d=0) const
   {
      if(!IsValid())return d;
      switch(m_value.m_type)
      {
      case JSON_INT:
         return m_value.m_int;
      case JSON_DOUBLE:
         return(long)m_value.m_double;
      case JSON_BOOL:
         return(long)m_value.m_bool;
      case JSON_STRING:
         return StringToInteger(m_value.m_str);
      default:
         return d;
      }
   }

   double AsDouble(double d=0.0) const
   {
      if(!IsValid())return d;
      switch(m_value.m_type)
      {
      case JSON_DOUBLE:
         return m_value.m_double;
      case JSON_INT:
         return(double)m_value.m_int;
      case JSON_BOOL:
         return(double)m_value.m_bool;
      case JSON_STRING:
         return StringToDouble(m_value.m_str);
      default:
         return d;
      }
   }

   bool AsBool(bool d=false) const
   {
      if(!IsValid())return d;
      switch(m_value.m_type)
      {
      case JSON_BOOL:
         return m_value.m_bool;
      case JSON_INT:
         return m_value.m_int!=0;
      case JSON_DOUBLE:
         return m_value.m_double!=0.0;
      case JSON_STRING:
         return StringCompare(m_value.m_str,"true",false)==0;
      default:
         return d;
      }
   }

   string NumberAsString(const string d="") const
   {
      return IsValid() && IsNumber() ? m_value.m_num_str : d;
   }

   JsonNode Get(const string k) const
   {
      return JsonNode(IsValid() ? m_value.Get(k) : NULL);
   }

   JsonNode At(int i) const
   {
      return JsonNode(IsValid() ? m_value.At(i) : NULL);
   }

   JsonNode QueryPointer(const string &p) const; // Implementation is still in this file's cpp part

   void ForEach(IJsonObjectVisitor*v)const
   {
      if(IsObject()&&v!=NULL)
      {
         string k[];
         GetKeys(k);
         for(int i=0;i<ArraySize(k);i++)v.Visit(k[i],Get(k[i]));
      }
   }

   void ForEach(IJsonArrayVisitor*v)const
   {
      if(IsArray()&&v!=NULL)
      {
         for(int i=0,s=Size();i<s;i++)v.Visit(i,At(i));
      }
   }

   JsonNode operator[](const string &k) const
   {
      return Get(k);
   }
   JsonNode operator[](int i) const
   {
      return At(i);
   }

   JsonNode SetObject(string k)
   {
      if(!IsValid()||m_value.m_doc==NULL)return JsonNode();
      if(m_value.m_type==JSON_NULL)m_value._Init(JSON_OBJECT);
      CJsonValue*n=m_value.m_doc.CreateNode(JSON_OBJECT);
      if(n != NULL) m_value.Set(k,n);
      return JsonNode(n);
   }

   JsonNode SetArray(string k)
   {
      if(!IsValid()||m_value.m_doc==NULL)return JsonNode();
      if(m_value.m_type==JSON_NULL)m_value._Init(JSON_OBJECT);
      CJsonValue*n=m_value.m_doc.CreateNode(JSON_ARRAY);
      if(n != NULL) m_value.Set(k,n);
      return JsonNode(n);
   }

   JsonNode Set(string k,const JsonNode&n)
   {
      if(IsValid() && n.IsValid() && m_value.m_doc != NULL)
      {
         CJsonValue *cloned_value = n.m_value.Clone(m_value.m_doc);
         if(cloned_value != NULL) m_value.Set(k, cloned_value);
      }
      return this;
   }

   JsonNode Set(string k,string v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue(v));
      return this;
   }
   JsonNode Set(string k,long v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue(v));
      return this;
   }
   JsonNode Set(string k,int v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Set(string k,uint v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Set(string k,short v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Set(string k,ushort v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Set(string k,ulong v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Set(string k,double v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue(v));
      return this;
   }
   JsonNode Set(string k,float v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue((double)v));
      return this;
   }
   JsonNode Set(string k,datetime v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Set(string k,bool v)
   {
      if(IsValid()) m_value.Set(k,CreateAndSetValue(v));
      return this;
   }

   JsonNode Remove(string k)
   {
      if(IsValid())m_value.Remove(k);
      return this;
   }

   JsonNode AddObject()
   {
      if(!IsValid()||m_value.m_doc==NULL)return JsonNode();
      if(m_value.m_type==JSON_NULL)m_value._Init(JSON_ARRAY);
      CJsonValue*n=m_value.m_doc.CreateNode(JSON_OBJECT);
      if(n != NULL) m_value.Add(n);
      return JsonNode(n);
   }

   JsonNode AddArray()
   {
      if(!IsValid()||m_value.m_doc==NULL)return JsonNode();
      if(m_value.m_type==JSON_NULL)m_value._Init(JSON_ARRAY);
      CJsonValue*n=m_value.m_doc.CreateNode(JSON_ARRAY);
      if(n != NULL) m_value.Add(n);
      return JsonNode(n);
   }

   JsonNode Add(const JsonNode&n)
   {
      if(IsValid() && n.IsValid() && m_value.m_doc != NULL)
      {
         CJsonValue *cloned_value = n.m_value.Clone(m_value.m_doc);
         if(cloned_value != NULL) m_value.Add(cloned_value);
      }
      return this;
   }

   JsonNode Add(string v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue(v));
      return this;
   }
   JsonNode Add(long v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue(v));
      return this;
   }
   JsonNode Add(int v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Add(uint v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Add(short v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Add(ushort v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Add(ulong v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Add(double v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue(v));
      return this;
   }
   JsonNode Add(float v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue((double)v));
      return this;
   }
   JsonNode Add(datetime v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue((long)v));
      return this;
   }
   JsonNode Add(bool v)
   {
      if(IsValid())m_value.Add(CreateAndSetValue(v));
      return this;
   }

   JsonNode Remove(int i)
   {
      if(IsValid())m_value.Remove(i);
      return this;
   }
};

JsonNode JsonNode::QueryPointer(const string &p) const
{
   if(!IsValid() || StringLen(p) == 0) return (p == "") ? this : JsonNode();
   if(StringGetCharacter(p, 0) != '/') return JsonNode();
   JsonNode current_node(m_value);
   int pos = 1;
   int len = StringLen(p);
   while(pos <= len && current_node.IsValid())
   {
      int next_slash = StringFind(p, "/", pos);
      if(next_slash < 0) next_slash = len;
      string token = StringSubstr(p, pos, next_slash - pos);
      StringReplace(token, "~1", "/");
      StringReplace(token, "~0", "~");
      if(current_node.IsObject())
      {
         current_node = current_node.Get(token);
      }
      else if(current_node.IsArray())
      {
         if(StringLen(token) > 1 && StringGetCharacter(token, 0) == '0') return JsonNode();
         long index = StringToInteger(token);
         if(IntegerToString(index) != token) return JsonNode();
         current_node = current_node.At((int)index);
      }
      else
      {
         return JsonNode();
      }
      if(next_slash == len) break;
      pos = next_slash + 1;
   }
   return current_node;
}

CJsonValue* JsonNode::CreateAndSetValue(long v)
{
   if(!IsValid()||m_value.m_doc==NULL)return NULL;
   CJsonValue*n=m_value.m_doc.CreateNode(JSON_INT);
   if(n)
   {
      n.m_int=v;
      n.m_num_str=IntegerToString(v);
   }
   return n;
}
CJsonValue* JsonNode::CreateAndSetValue(double v)
{
   if(!IsValid()||m_value.m_doc==NULL)return NULL;
   CJsonValue*n=m_value.m_doc.CreateNode(JSON_DOUBLE);
   if(n)
   {
      n.m_double=v;
      n.m_num_str=DoubleToString(v,-1);
   }
   return n;
}
CJsonValue* JsonNode::CreateAndSetValue(bool v)
{
   if(!IsValid()||m_value.m_doc==NULL)return NULL;
   CJsonValue*n=m_value.m_doc.CreateNode(JSON_BOOL);
   if(n)n.m_bool=v;
   return n;
}
CJsonValue* JsonNode::CreateAndSetValue(const string v)
{
   if(!IsValid()||m_value.m_doc==NULL)return NULL;
   CJsonValue*n=m_value.m_doc.CreateNode(JSON_STRING);
   if(n)n.m_str=v;
   return n;
}


#endif // MQL5_JSON_NODE
