//+------------------------------------------------------------------+
//| JsonLibMain.mqh                                                  |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_MAIN
#define MQL5_JSON_MAIN

#include "JsonCore.mqh"
#include "JsonDocument.mqh"
#include "JsonNode.mqh"
#include "JsonQuery.mqh"
#include "JsonMapper.mqh"
#include "JsonStreamParser.mqh"
#include "JsonStream.mqh"
#include "JsonRapidParser.mqh"
#include "JsonDomBuilder.mqh"
#include "JsonUtils.mqh"
#include "JsonAutoSwitchingParser.mqh"

JsonDocument JsonParse(const string &text, JsonError &error, const JsonParseOptions &options)
{
   error.Clear();
   CJsonDocument *doc_impl = new CJsonDocument();
   if(CheckPointer(doc_impl) == POINTER_INVALID)
   {
      error.message = "Memory allocation failed for JsonDocument";
      return JsonDocument(NULL);
   }
   CJsonValue* root = NULL;
   bool use_rapid_engine;
   if (options.engine == ENGINE_HIGH_SPEED)
   {
      use_rapid_engine = true;
   }
   else if (options.engine == ENGINE_STANDARD)
   {
      use_rapid_engine = false;
   }
   else
   {
      use_rapid_engine = CJsonAutoSwitchingParser::ShouldUseRapidEngine(options);
   }
   if(use_rapid_engine)
   {
      CJsonRapidParser rapid_parser;
      JsonToken tape[];
      if(!rapid_parser.ParseToTape(text, tape, error, options))
      {
         delete doc_impl;
         return JsonDocument(NULL);
      }
      CDomBuilder dom_builder;
      root = dom_builder.BuildFromTape(doc_impl, tape, text);
      if(CheckPointer(root) == POINTER_INVALID)
      {
         error.message = "DOM construction from tape failed: " + dom_builder.GetError();
      }
   }
   else
   {
      JsonStreamParser stream_parser;
      StringViewReader* string_reader = new StringViewReader(text);
      if(CheckPointer(string_reader) == POINTER_INVALID)
      {
         error.message = "Memory allocation failed for StringViewReader";
         delete doc_impl;
         return JsonDocument(NULL);
      }
      CDomBuilderHandler handler(doc_impl);
      if(!stream_parser.Parse(string_reader, GetPointer(handler), error, options, true))
      {
         delete doc_impl;
         return JsonDocument(NULL);
      }
      root = handler.GetRoot();
   }
   if(CheckPointer(root) == POINTER_INVALID)
   {
      if(error.message == "")
      {
         error.message = "Failed to obtain root node after parsing.";
      }
      delete doc_impl;
      return JsonDocument(NULL);
   }
   doc_impl.m_root = root;
   return JsonDocument(doc_impl);
}

JsonDocument JsonFromFile(const string &filename, JsonError &error, const JsonParseOptions &options)
{
   error.Clear();
   int h = FileOpen(filename, FILE_READ | FILE_BIN);
   if(h < 0)
   {
      error.message = "File open failed for '" + filename + "', err " + (string)GetLastError();
      return JsonDocument(NULL);
   }
   FileStreamReader* reader = new FileStreamReader(h);
   if(CheckPointer(reader) == POINTER_INVALID || !reader.IsValid())
   {
      error.message = "Failed to create FileStreamReader.";
      FileClose(h);
      if(CheckPointer(reader) != POINTER_INVALID) delete reader;
      return JsonDocument(NULL);
   }
   CJsonDocument *doc_impl = new CJsonDocument();
   if(CheckPointer(doc_impl) == POINTER_INVALID)
   {
      error.message = "Memory allocation failed for JsonDocument";
      delete reader;
      return JsonDocument(NULL);
   }
   JsonStreamParser stream_parser;
   CDomBuilderHandler handler(doc_impl);
   if(!stream_parser.Parse(reader, GetPointer(handler), error, options, true))
   {
      delete doc_impl;
      return JsonDocument(NULL);
   }
   CJsonValue* root = handler.GetRoot();
   if(CheckPointer(root) == POINTER_INVALID)
   {
      if(error.message == "")
      {
         error.message = "Failed to obtain root node after stream parsing.";
      }
      delete doc_impl;
      return JsonDocument(NULL);
   }
   doc_impl.m_root = root;
   return JsonDocument(doc_impl);
}

bool JsonStreamParse(const string &text,IJsonStreamHandler *handler,JsonError &error,const JsonParseOptions &options)
{
   StringViewReader* reader=new StringViewReader(text);
   if(CheckPointer(reader)==POINTER_INVALID)
   {
      error.message="Memory allocation failed";
      return false;
   }
   JsonStreamParser parser;
   return parser.Parse(reader,handler,error,options,true);
}

JsonDocument JsonNewObject()
{
   CJsonDocument *doc = new CJsonDocument();
   if(CheckPointer(doc)==POINTER_INVALID) return JsonDocument(NULL);
   doc.m_root = doc.CreateNode(JSON_OBJECT);
   if(CheckPointer(doc.m_root)==POINTER_INVALID)
   {
      delete doc;
      return JsonDocument(NULL);
   }
   return JsonDocument(doc);
}

JsonDocument JsonNewArray()
{
   CJsonDocument *doc = new CJsonDocument();
   if(CheckPointer(doc)==POINTER_INVALID) return JsonDocument(NULL);
   doc.m_root = doc.CreateNode(JSON_ARRAY);
   if(CheckPointer(doc.m_root)==POINTER_INVALID)
   {
      delete doc;
      return JsonDocument(NULL);
   }
   return JsonDocument(doc);
}

void ResetState()
{
   CJsonAutoSwitchingParser::ResetState();
}

#endif // MQL5_JSON_MAIN

