//+------------------------------------------------------------------+
//| Core/JsonDomBuilder.mqh                                          |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_DOM_BUILDER
#define MQL5_JSON_DOM_BUILDER

#include "JsonCore.mqh"
#include "JsonTypes.mqh"
#include "JsonUtils.mqh"

class CDomBuilder
{
private:
   CJsonDocument* m_doc;
   string         m_text;
   string         m_error;

   string GetStringFromSlice(int start, int length)
   {
      if (length == 0) return "";
      StringBuilder sb(length);
      int end = start + length;
      for (int i = start; i < end; i++)
      {
         ushort c = StringGetCharacter(m_text, i);
         if (c != '\\')
         {
            sb.AppendChar(c);
            continue;
         }
         i++;
         c = StringGetCharacter(m_text, i);
         switch(c)
         {
         case '"' :
            sb.AppendChar((ushort)'"');
            break;
         case '\\':
            sb.AppendChar((ushort)'\\');
            break;
         case '/' :
            sb.AppendChar((ushort)'/');
            break;
         case 'b' :
            sb.AppendChar((ushort)8);
            break;
         case 'f' :
            sb.AppendChar((ushort)12);
            break;
         case 'n' :
            sb.AppendChar((ushort)'\n');
            break;
         case 'r' :
            sb.AppendChar((ushort)'\r');
            break;
         case 't' :
            sb.AppendChar((ushort)'\t');
            break;
         case 'u' :
         {
            string hex_str = StringSubstr(m_text, i + 1, 4);
            sb.AppendChar((ushort)JsonHexToInteger(hex_str));
            i += 4;
            break;
         }
         }
      }
      return sb.ToString();
   }

   CJsonValue* BuildNode(int &tape_pos, const JsonToken &tape[])
   {
      if(tape_pos >= ArraySize(tape))
      {
         m_error = "Tape read out of bounds";
         return NULL;
      }
      JsonToken current_token = tape[tape_pos];
      CJsonValue* current_node = m_doc.CreateNode(current_token.type);
      if(!current_node)
      {
         m_error = "Node allocation failed";
         return NULL;
      }
      tape_pos++;
      switch(current_token.type)
      {
      case JSON_STRING:
         current_node.m_str = GetStringFromSlice(current_token.start, current_token.length);
         break;
      case JSON_DOUBLE:
      {
         string num_str = StringSubstr(m_text, current_token.start, current_token.length);
         if (IsIntegerFast(num_str))
         //if(StringFind(num_str, ".") < 0 && StringFind(num_str, "e") < 0 && StringFind(num_str, "E") < 0)
         {
            current_node.m_type = JSON_INT;
            current_node.m_int = StringToInteger(num_str);
         }
         else
         {
            current_node.m_double = StringToDouble(num_str);
         }
         current_node.m_num_str = m_doc.m_arena_ptr.AllocateString(num_str);
      }
      break;
      case JSON_BOOL:
         current_node.m_bool = (StringGetCharacter(m_text, current_token.start) == 't');
         break;
      case JSON_NULL:
         break;
      case JSON_OBJECT:
         for(int i=0; i < current_token.child_count; i++)
         {
            string key_str = GetStringFromSlice(tape[tape_pos].start, tape[tape_pos].length);
            tape_pos++;
            CJsonValue* child = BuildNode(tape_pos, tape);
            if(!child) return NULL;
            current_node.Set(key_str, child);
         }
         break;
      case JSON_ARRAY:
         for(int i=0; i < current_token.child_count; i++)
         {
            CJsonValue* child = BuildNode(tape_pos, tape);
            if(!child) return NULL;
            current_node.Add(child);
         }
         break;
      }
      return current_node;
   }
public:
   CJsonValue* BuildFromTape(CJsonDocument *doc, const JsonToken &tape[], const string &text)
   {
      m_error = "";
      m_doc = doc;
      m_text = text;
      if(ArraySize(tape) == 0) return doc.CreateNode(JSON_NULL);
      int tape_pos = 0;
      return BuildNode(tape_pos, tape);
   }
   string GetError() const
   {
      return m_error;
   }
};

class CDomBuilderHandler : public IJsonStreamHandler
{
private:
   CJsonDocument*m_doc;
   CJsonValue*m_stack[];
   int m_top;
   string m_current_key;

   CJsonValue *Peek()
   {
      return(m_top>=0)?m_stack[m_top]:NULL;
   }
   void Pop()
   {
      if(m_top>=0)m_top--;
   }
   void Push(CJsonValue *v)
   {
      AddValue(v);
      if(v.m_type==JSON_ARRAY||v.m_type==JSON_OBJECT)
      {
         m_top++;
         if(ArraySize(m_stack)<=m_top)ArrayResize(m_stack,m_top+1);
         m_stack[m_top]=v;
      }
   }
   bool AddValue(CJsonValue *v)
   {
      CJsonValue*p=Peek();
      if(!p)
      {
         if(ArraySize(m_stack)==0)ArrayResize(m_stack,1);
         m_stack[0]=v;
      }
      else if(p.m_type==JSON_ARRAY)
      {
         p.Add(v);
      }
      else if(p.m_type==JSON_OBJECT)
      {
         p.Set(m_current_key,v);
      }
      return true;
   }
public:
   CDomBuilderHandler(CJsonDocument*d):m_doc(d),m_top(-1) {}

   CJsonValue*GetRoot()
   {
      return(ArraySize(m_stack)>0)?m_stack[0]:NULL;
   }

   bool OnStartDocument()override
   {
      ArrayFree(m_stack);
      m_top=-1;
      return true;
   }
   bool OnEndDocument()override
   {
      return true;
   }
   bool OnStartObject()override
   {
      CJsonValue*o=m_doc.CreateNode(JSON_OBJECT);
      if(!o)return false;
      Push(o);
      return true;
   }
   bool OnEndObject()override
   {
      Pop();
      return true;
   }
   bool OnStartArray()override
   {
      CJsonValue*a=m_doc.CreateNode(JSON_ARRAY);
      if(!a)return false;
      Push(a);
      return true;
   }
   bool OnEndArray()override
   {
      Pop();
      return true;
   }
   bool OnKey(const string &k)override
   {
      m_current_key=k;
      return true;
   }
   bool OnString(const string &v)override
   {
      CJsonValue*s=m_doc.CreateNode(JSON_STRING);
      if(!s)return false;
      s.m_str=v;
      return AddValue(s);
   }
   bool OnBool(bool v)override
   {
      CJsonValue*b=m_doc.CreateNode(JSON_BOOL);
      if(!b)return false;
      b.m_bool=v;
      return AddValue(b);
   }
   bool OnNull()override
   {
      CJsonValue*n=m_doc.CreateNode(JSON_NULL);
      if(!n)return false;
      return AddValue(n);
   }
   bool OnNumber(const string &v,ENUM_JSON_TYPE t)override
   {
      if(StringFind(v,".")<0 && StringFind(v,"e")<0 && StringFind(v,"E")<0) t=JSON_INT;
      else t=JSON_DOUBLE;
      CJsonValue*n=m_doc.CreateNode(t);
      if(!n)return false;
      n.m_num_str = m_doc.m_arena_ptr.AllocateString(v);
      if(t==JSON_INT)n.m_int=StringToInteger(v);
      else n.m_double=StringToDouble(v);
      return AddValue(n);
   }
};

#endif // MQL5_JSON_DOM_BUILDER

