//+------------------------------------------------------------------+
//| Core/JsonDocument.mqh                                            |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_DOCUMENT
#define MQL5_JSON_DOCUMENT

#include <Object.mqh>

#include "JsonCore.mqh"
#include "JsonNode.mqh"
#include "JsonTypes.mqh"
#include "JsonSerializer.mqh"
#include "JsonUtils.mqh"

void JsonPatchHelper(JsonNode &target, const JsonNode &patch)
{
   if (!patch.IsObject() || !target.IsObject()) return;
   string keys[];
   patch.GetKeys(keys);
   for(int i = 0; i < ArraySize(keys); i++)
   {
      string key = keys[i];
      JsonNode patch_value = patch.Get(key);
      if (patch_value.IsNull())
      {
         target.Remove(key);
      }
      else
      {
         JsonNode target_value = target.Get(key);
         if (target_value.IsValid() && target_value.IsObject() && patch_value.IsObject())
         {
            JsonPatchHelper(target_value, patch_value);
         }
         else
         {
            target.Set(key, patch_value);
         }
      }
   }
}

class JsonDocument
{
private:
   CJsonDocument *m_impl;


   JsonDocument(const JsonDocument &other) {}
   void operator=(const JsonDocument &other) {}

public:
   JsonDocument() : m_impl(NULL) {}
   JsonDocument(CJsonDocument *impl) : m_impl(impl) {}
   JsonDocument(JsonDocument &other)
   {
      m_impl = other.m_impl;
      other.m_impl = NULL;
   }

   void operator=(JsonDocument &other)
   {
      if(GetPointer(this) == GetPointer(other)) return;
      if(CheckPointer(m_impl) == POINTER_DYNAMIC) delete m_impl;
      m_impl = other.m_impl;
      other.m_impl = NULL;
   }

   ~JsonDocument()
   {
      if(CheckPointer(m_impl) == POINTER_DYNAMIC)
         delete m_impl;
   }

   bool IsValid() const
   {
      return CheckPointer(m_impl) != POINTER_INVALID && CheckPointer(m_impl.m_root) != POINTER_INVALID;
   }

   JsonNode GetRoot() const
   {
      return JsonNode(IsValid() ? m_impl.m_root : NULL);
   }

   JsonDocument Clone() const
   {
      if(!IsValid()) return JsonDocument();
      CJsonDocument *doc_clone = new CJsonDocument();
      if(CheckPointer(doc_clone) == POINTER_INVALID) return JsonDocument();
      doc_clone.m_root = m_impl.m_root.Clone(doc_clone);
      if(CheckPointer(doc_clone.m_root) == POINTER_INVALID)
      {
         delete doc_clone;
         return JsonDocument();
      }
      return JsonDocument(doc_clone);
   }

   bool Patch(const JsonDocument &patch_doc)
   {
      if (!IsValid() || !patch_doc.IsValid()) return false;
      JsonNode target_node = GetRoot();
      JsonNode patch_node = patch_doc.GetRoot();
      if (!patch_node.IsObject() || !target_node.IsObject())
      {
         return false;
      }
      JsonPatchHelper(target_node, patch_node);
      return true;
   }

   string ToString(bool pretty=false, bool escape_non_ascii=false) const
   {
      if(!IsValid()) return "";
      CJsonSerializer s;
      return s.Serialize(m_impl.m_root, pretty, escape_non_ascii);
   }

   bool SaveToFile(const string &path, bool pretty=true, bool escape_non_ascii=false, bool with_bom=true) const
   {
      if(!IsValid()) return false;
      int h = FileOpen(path, FILE_WRITE | FILE_TXT | FILE_ANSI, "\t", CP_UTF8);
      if(h < 0)
      {
         Print("JsonDocument::SaveToFile failed. Could not open file '", path, "'. Error: ", GetLastError());
         return false;
      }
      string content = ToString(pretty, escape_non_ascii);
      FileWriteString(h, content);
      FileClose(h);
      return (GetLastError() == 0);
   }

   JsonNode operator[](const string &key) const
   {
      return GetRoot().Get(key);
   }
   JsonNode operator[](int index) const
   {
      return GetRoot().At(index);
   }
   CJsonDocument* _GetImpl() const
   {
      return m_impl;
   }
};

#endif // MQL5_JSON_DOCUMENT

