//+------------------------------------------------------------------+
//| Core/JsonAutoSwitchingParser.mqh                                 |
//+------------------------------------------------------------------+
#ifndef MQL5_JSON_AUTO_SWITCHING_PARSER
#define MQL5_JSON_AUTO_SWITCHING_PARSER

#include "JsonCore.mqh"

//+------------------------------------------------------------------+
//| CJsonAutoSwitchingParser                                         |
//|------------------------------------------------------------------+
//| Purpose:                                                         |
//| Manages the stateful decision for ENGINE_AUTO mode.              |
//| For performance, it "latches" onto the Standard Engine if ever   |
//| a non-standard feature is detected during a session. This avoids |
//| re-evaluating subsequent JSONs. However, this state can pollute  |
//| consecutive unit tests. The ResetState() method is provided to   |
//| clear this cached decision, ensuring test isolation.             |
//+------------------------------------------------------------------+
class CJsonAutoSwitchingParser
{
private:
   // Static flag to cache the decision. Once set, it stays true for
   // the lifetime of the program unless explicitly reset.
   static bool s_force_standard_engine;

public:
   // --- Static Public API ---

   // Resets the cached state. Crucial for unit testing environments.
   static void ResetState()
   {
      s_force_standard_engine = false;
   }

   // Determines if the high-speed engine should be used in ENGINE_AUTO mode.
   static bool ShouldUseRapidEngine(const JsonParseOptions &options)
   {
      // If the state is already "latched" to standard, don't re-evaluate.
      if(s_force_standard_engine)
      {
         return false; // Stick with the standard engine
      }
      // High-speed engine is compatible only with standard RFC 8259 JSON.
      // If any non-standard options are enabled, we must use the standard engine.
      bool is_rapid_compatible = !options.allow_comments
                                 && !options.allow_trailing_commas;
      if(!is_rapid_compatible)
      {
         // Non-standard feature detected. Latch the state to force
         // standard engine for all subsequent parses in this session.
         s_force_standard_engine = true;
         return false;
      }
      // It's compatible, use the high-speed engine.
      return true;
   }
};

// Initialize the static member variable
bool CJsonAutoSwitchingParser::s_force_standard_engine = false;

#endif // MQL5_JSON_AUTO_SWITCHING_PARSER


