//+------------------------------------------------------------------+
//|                                                 LotCalculate.mq5 |
//+------------------------------------------------------------------+
#property copyright "Wamek Script-2023"
#property link      "eawamek@gmail.com"
#property version   "1.00"
#property script_show_inputs



// Risk management enumeration
enum RiskMode { RiskByPercent, RiskByAmount };

// Input parameters 
input string Note = "Only selected input is used.";
input RiskMode riskMode = RiskByPercent;    // Select risk mode
input double RiskPercent = 1.0;             // Risk % of account balance
input double RiskAmount = 100.0;            // Risk amount in account currency
input double StopLossPips = 50.0;           // Stoploss in pips

//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
//---
   CalculateLotSize();
  }
//+------------------------------------------------------------------+



//+------------------------------------------------------------------+
//| Calculate optimal lot size based on risk parameters              |
//+------------------------------------------------------------------+
void CalculateLotSize()
  {
   double riskValue=0.0;
   int From4To5 = 1; // Default conversion factor
   
   // Validate stop loss
   if(StopLossPips <= 0)
     {
      Print("Error: Stop loss must be > 0");
      return;
     }

   // Get current symbol information
   double tickValue = SymbolInfoDouble(Symbol(), SYMBOL_TRADE_TICK_VALUE);
   double minLot    = SymbolInfoDouble(Symbol(), SYMBOL_VOLUME_MIN);
   double maxLot    = SymbolInfoDouble(Symbol(), SYMBOL_VOLUME_MAX);
   double lotStep   = SymbolInfoDouble(Symbol(), SYMBOL_VOLUME_STEP);
   int    digits    = (int)SymbolInfoInteger(Symbol(), SYMBOL_DIGITS);

   // Determine points per pip conversion
   int calc_mode = SymbolInfoInteger(Symbol(), SYMBOL_TRADE_CALC_MODE);
   
   if (calc_mode == SYMBOL_CALC_MODE_FOREX) // determine if instrument is forex
       From4To5 = (digits%2==1) ? 10 : 1;
   
   if (calc_mode == SYMBOL_CALC_MODE_CFDLEVERAGE) // determine if instrument is CFD leverage
       From4To5 = (digits%2==0) ? 10 : 1;
  
   // Validate market data
   if(tickValue <= 0 || minLot <= 0 || maxLot <= 0 || lotStep <= 0)
     {
      Print("Error: Invalid market data");
      Print("tickValue: ", tickValue, " minLot: ", minLot, " maxLot: ", maxLot, " lotStep: ", lotStep);
      return;
     }
    
   //--- Determine risk value
   if(riskMode == RiskByPercent)
     {
      if(RiskPercent <= 0)
        {
         Print("Risk Percent must be > 0.");
         return;
        }
      riskValue = (RiskPercent/100.0) * AccountInfoDouble(ACCOUNT_BALANCE);
     }
   else if(riskMode == RiskByAmount)
     {
      if(RiskAmount <= 0)
        {
         Print("Risk amount must be > 0.");
         return;
        }
      riskValue = RiskAmount;
     }    
     
   double stopLossPoints = StopLossPips * From4To5;

   // Calculate raw lot size
   double lotSize = riskValue / (stopLossPoints * tickValue);

   // Apply broker constraints
   lotSize = fmax(minLot, fmin(maxLot, lotSize));          // Clamp to min/max
   lotSize = round(lotSize / lotStep) * lotStep;           // Step normalization
   
   //--- Output results
   Alert(_Symbol+"\nCalculated Lot Size: ", lotSize,
         "\nRisk Method: ", EnumToString(riskMode),
         "\nRisk Value: ", riskValue,
         "\nTick Value: ", NormalizeDouble(tickValue,4),
          "\nStopLoss: ", stopLossPoints );
   return;
  }