//+------------------------------------------------------------------+
//|                                                    CTsLogger.mqh |
//|                                         Copyright 2025, Trefoter |
//|                             https://www.mql5.com/ru/users/useral |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, Trefoter"
#property link      "https://www.mql5.com/ru/users/useral"
#property strict

/* CTsLogger - a simple and flexible logging system specifically
designed for creating and debugging trading systems in MQL5.
The main advantage of CTsLogger is the ability to temporarily enable
debug mode for specific modules or code sections while maintaining
a lower global logging level.
This allows you to get detailed logging of specific code areas without
"drowning" in a flood of messages, and then disable it with a single command. */



//+------------------------------------------------------------------+
//| Enumerations for the logging system                              |
//+------------------------------------------------------------------+

// Log importance levels
enum ENUM_LOG_LEVEL
{
   LOG_LEVEL_ERROR,           // Errors only
   LOG_LEVEL_WARNING,         // Warnings and errors
   LOG_LEVEL_INFO,            // Informational messages
   LOG_LEVEL_DEBUG            // Debug information
};

//+------------------------------------------------------------------+
//| Logger class for trading system                                  |
//+------------------------------------------------------------------+
class CTsLogger
{
private:
   // Variables for storing logger settings
   ENUM_LOG_LEVEL  m_globalLogLevel;     // Global logging level
   string          m_logFileName;        // Log file name
   int             m_fileHandle;         // Log file handle
   bool            m_logToTerminal;      // Terminal output flag
   bool            m_initialized;        // Logger initialization flag
   string          m_debugModules[];     // Array of modules in debug mode
   bool            m_debugModeAll;       // Global debug mode flag
   bool            m_debugModulesPaused; // Debug mode pause flag
   
   // Private helper methods
   bool     OpenLogFile();                // Open log file
   void     CloseLogFile();               // Close log file
   void     WriteLog(string level, string moduleId, string message); // Write to log
   string   FormatLogMessage(string level, string moduleId, string message); // Format message
   bool     IsModuleInDebugMode(string moduleId); // Check debug mode for module
   int      FindModuleIndex(string moduleId);    // Find module in array
   string   GetParentModule(string moduleId);    // Get parent module
   bool     IsModuleOrParentInDebugMode(string moduleId); // Check module or parents
   bool     IsChildModule(string potentialChild, string parentModule); // Check child module
   
public:
   // Constructor and destructor
   CTsLogger();
   ~CTsLogger();
   
   // Initialization methods
   bool     Initialize(string logFileName, bool logToTerminal = true);
   void     SetGlobalLogLevel(ENUM_LOG_LEVEL level);
   
   // Logging methods
   void     Debug(string moduleId, string message);
   void     Info(string moduleId, string message);
   void     Warning(string moduleId, string message);
   void     Error(string moduleId, string message);
   
   // Debug mode management
   void     EnableDebugMode(string moduleId);
   void     DisableDebugMode(string moduleId);
   bool     IsInDebugMode(string moduleId);
   void     EnableDebugModeAll();
   void     DisableDebugModeAll();
   
   // Advanced debug mode management
   void     PauseDebugMode();              // Pause debug mode
   void     ResumeDebugMode();             // Resume debug mode
   bool     IsDebugModePaused();           // Check pause state
   void     ResetDebugModules();           // Complete reset of debug modules
   bool     HasChildDebugModules(string parentModule); // Check child modules
};

//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CTsLogger::CTsLogger()
{
   m_globalLogLevel = LOG_LEVEL_INFO;
   m_logFileName = "";
   m_fileHandle = INVALID_HANDLE;
   m_logToTerminal = true;
   m_initialized = false;
   ArrayResize(m_debugModules, 0);
   m_debugModeAll = false;
   m_debugModulesPaused = false;
}

//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CTsLogger::~CTsLogger()
{
   if(m_fileHandle != INVALID_HANDLE)
   {
      CloseLogFile();
   }
}

//+------------------------------------------------------------------+
//| Logger initialization                                            |
//+------------------------------------------------------------------+
bool CTsLogger::Initialize(string logFileName, bool logToTerminal = true)
{
   // If logger is already initialized, close the old file
   if(m_initialized && m_fileHandle != INVALID_HANDLE)
   {
      CloseLogFile();
   }
   
   // Set logging parameters
   m_logFileName = logFileName;
   m_logToTerminal = logToTerminal;
   
   // Open the log file if a file name is specified
   if(m_logFileName != "")
   {
      if(!OpenLogFile())
      {
         Print("ERROR: Failed to open log file (", GetLastError(), "): ", m_logFileName);
         // Continue operation even if file opening failed
         // (degradation to terminal logging)
      }
   }
   
   m_initialized = true;
   
   // Output initialization message
   string initMessage = "TsLogger initialized. Global level: " + 
                       EnumToString(m_globalLogLevel) + 
                       ", terminal output: " + (m_logToTerminal ? "Yes" : "No") +
                       ", file: " + (m_logFileName != "" ? m_logFileName : "none");
   
   Info("TsLogger", initMessage);
   
   return true;
}

//+------------------------------------------------------------------+
//| Open log file                                                    |
//+------------------------------------------------------------------+
bool CTsLogger::OpenLogFile()
{
   // Open file for writing with append capability
   m_fileHandle = FileOpen(m_logFileName, FILE_WRITE|FILE_READ|FILE_TXT|FILE_ANSI, '\t');
   
   if(m_fileHandle == INVALID_HANDLE)
   {
      Print("ERROR: Failed to open log file (", GetLastError(), "): ", m_logFileName);
      return false;
   }
   
   // Move pointer to end of file for data appending
   FileSeek(m_fileHandle, 0, SEEK_END);
   
   return true;
}

//+------------------------------------------------------------------+
//| Close log file                                                   |
//+------------------------------------------------------------------+
void CTsLogger::CloseLogFile()
{
   if(m_fileHandle != INVALID_HANDLE)
   {
      FileClose(m_fileHandle);
      m_fileHandle = INVALID_HANDLE;
   }
}

//+------------------------------------------------------------------+
//| Set global logging level                                         |
//+------------------------------------------------------------------+
void CTsLogger::SetGlobalLogLevel(ENUM_LOG_LEVEL level)
{
   m_globalLogLevel = level;
   
   if(m_initialized)
   {
      Info("TsLogger", "Global logging level set: " + EnumToString(level));
   }
}

//+------------------------------------------------------------------+
//| Format log message                                               |
//+------------------------------------------------------------------+
string CTsLogger::FormatLogMessage(string level, string moduleId, string message)
{
   string timestamp = TimeToString(TimeCurrent(), TIME_DATE|TIME_MINUTES|TIME_SECONDS);
   return timestamp + " [" + level + "] [" + moduleId + "] " + message;
}

//+------------------------------------------------------------------+
//| Write to log                                                     |
//+------------------------------------------------------------------+
void CTsLogger::WriteLog(string level, string moduleId, string message)
{
   string formattedMessage = FormatLogMessage(level, moduleId, message);
   
   // Output to terminal if allowed
   if(m_logToTerminal)
   {
      Print(formattedMessage);
   }
   
   // Write to file if open
   if(m_fileHandle != INVALID_HANDLE)
   {
      FileWrite(m_fileHandle, formattedMessage);
      // Don't close file after each write to improve performance
      FileFlush(m_fileHandle); // Flush buffer to disk
   }
}

//+------------------------------------------------------------------+
//| Get parent module from hierarchical identifier                   |
//+------------------------------------------------------------------+
string CTsLogger::GetParentModule(string moduleId)
{
   int dotPos = StringFind(moduleId, ".");
   if(dotPos > 0)
      return StringSubstr(moduleId, 0, dotPos);
   
   return moduleId; // Return ID itself if no hierarchy
}

//+------------------------------------------------------------------+
//| Check if a module is a child of another module                   |
//+------------------------------------------------------------------+
bool CTsLogger::IsChildModule(string potentialChild, string parentModule)
{
   // Check if potential child starts with parent + "."
   if(StringFind(potentialChild, parentModule + ".") == 0)
      return true;
   
   return false;
}

//+------------------------------------------------------------------+
//| Check if the module or one of its parents is in debug mode       |
//+------------------------------------------------------------------+
bool CTsLogger::IsModuleOrParentInDebugMode(string moduleId)
{
   int size = ArraySize(m_debugModules);
   
   // Check the module itself
   for(int i = 0; i < size; i++)
   {
      if(m_debugModules[i] == moduleId)
         return true;
   }
   
   // Check if any of the modules in the debug list is a parent
   // for the current module (e.g., if moduleId = "TradeModule.OrderExecution",
   // check if "TradeModule" has debug enabled)
   for(int i = 0; i < size; i++)
   {
      // If current module is lower in hierarchy than a module from the list
      if(IsChildModule(moduleId, m_debugModules[i]))
         return true;
   }
   
   return false;
}

//+------------------------------------------------------------------+
//| Check if module is in debug mode                                 |
//+------------------------------------------------------------------+
bool CTsLogger::IsModuleInDebugMode(string moduleId)
{
   // If global debug mode is enabled - always true
   if(m_debugModeAll)
      return true;
   
   // If debug mode is paused - return false
   if(m_debugModulesPaused)
      return false;
   
   // Check the module itself and its parent modules
   return IsModuleOrParentInDebugMode(moduleId);
}

//+------------------------------------------------------------------+
//| Search for module in debug modules array                         |
//+------------------------------------------------------------------+
int CTsLogger::FindModuleIndex(string moduleId)
{
   int size = ArraySize(m_debugModules);
   
   for(int i = 0; i < size; i++)
   {
      if(m_debugModules[i] == moduleId)
         return i;
   }
   
   return -1;
}

//+------------------------------------------------------------------+
//| Enable debug mode for specified module                           |
//+------------------------------------------------------------------+
void CTsLogger::EnableDebugMode(string moduleId)
{
   // Check if module is already in the list
   if(FindModuleIndex(moduleId) >= 0)
      return;
   
   // Add module to debug list
   int size = ArraySize(m_debugModules);
   ArrayResize(m_debugModules, size + 1);
   m_debugModules[size] = moduleId;
   
   // If debug mode was paused, unpause it
   if(m_debugModulesPaused)
      m_debugModulesPaused = false;
   
   Info("TsLogger", "Debug mode enabled for module: " + moduleId);
}

//+------------------------------------------------------------------+
//| Disable debug mode for specified module                          |
//+------------------------------------------------------------------+
void CTsLogger::DisableDebugMode(string moduleId)
{
   bool moduleFound = false;
   int size = ArraySize(m_debugModules);
   
   // Create temporary array to store remaining modules
   string tempModules[];
   int newSize = 0;
   
   // Copy to new array only those modules that are NOT
   // the specified module and not its child modules
   for(int i = 0; i < size; i++)
   {
      // If current module doesn't match the one being disabled and is not its child
      if(m_debugModules[i] != moduleId && !IsChildModule(m_debugModules[i], moduleId))
      {
         ArrayResize(tempModules, newSize + 1);
         tempModules[newSize++] = m_debugModules[i];
      }
      else
      {
         moduleFound = true;
      }
   }
   
   // If modules were found and removed
   if(moduleFound)
   {
      // Replace debug modules array with new one
      ArrayResize(m_debugModules, newSize);
      for(int i = 0; i < newSize; i++)
      {
         m_debugModules[i] = tempModules[i];
      }
      
      Info("TsLogger", "Debug mode disabled for module and its submodules: " + moduleId);
   }
}

//+------------------------------------------------------------------+
//| Check if debug mode is enabled for specified module              |
//+------------------------------------------------------------------+
bool CTsLogger::IsInDebugMode(string moduleId)
{
   return IsModuleInDebugMode(moduleId);
}

//+------------------------------------------------------------------+
//| Enable debug mode for all modules                                |
//+------------------------------------------------------------------+
void CTsLogger::EnableDebugModeAll()
{
   m_debugModeAll = true;
   // If debug mode was paused, unpause it
   if(m_debugModulesPaused)
      m_debugModulesPaused = false;
       
   Info("TsLogger", "Global debug mode enabled for all modules");
}

//+------------------------------------------------------------------+
//| Disable debug mode for all modules                               |
//+------------------------------------------------------------------+
void CTsLogger::DisableDebugModeAll()
{
   m_debugModeAll = false;
   m_debugModulesPaused = true; // Set pause flag
   Info("TsLogger", "Global debug mode disabled (modules preserved)");
}

//+------------------------------------------------------------------+
//| Pause debug mode for all modules                                 |
//+------------------------------------------------------------------+
void CTsLogger::PauseDebugMode()
{
   if(!m_debugModulesPaused && ArraySize(m_debugModules) > 0)
   {
      m_debugModulesPaused = true;
      Info("TsLogger", "Debug mode temporarily paused for all modules");
   }
}

//+------------------------------------------------------------------+
//| Resume debug mode for previously enabled modules                 |
//+------------------------------------------------------------------+
void CTsLogger::ResumeDebugMode()
{
   if(m_debugModulesPaused)
   {
      m_debugModulesPaused = false;
      Info("TsLogger", "Debug mode resumed for " + IntegerToString(ArraySize(m_debugModules)) + " modules");
   }
}

//+------------------------------------------------------------------+
//| Check if debug mode is paused                                    |
//+------------------------------------------------------------------+
bool CTsLogger::IsDebugModePaused()
{
   return m_debugModulesPaused;
}

//+------------------------------------------------------------------+
//| Complete reset of all debug modules                              |
//+------------------------------------------------------------------+
void CTsLogger::ResetDebugModules()
{
   int prevSize = ArraySize(m_debugModules);
   ArrayResize(m_debugModules, 0);
   m_debugModulesPaused = false;
   if(prevSize > 0)
      Info("TsLogger", "All debug modules completely reset (" + IntegerToString(prevSize) + " pcs.)");
}

//+------------------------------------------------------------------+
//| Check for child modules in debug mode                            |
//+------------------------------------------------------------------+
bool CTsLogger::HasChildDebugModules(string parentModule)
{
   int size = ArraySize(m_debugModules);
   
   for(int i = 0; i < size; i++)
   {
      if(IsChildModule(m_debugModules[i], parentModule))
         return true;
   }
   
   return false;
}

//+------------------------------------------------------------------+
//| Log message with DEBUG level                                     |
//+------------------------------------------------------------------+
void CTsLogger::Debug(string moduleId, string message)
{
   // Check if logger is initialized
   if(!m_initialized)
   {
      Print("WARNING: TsLogger not initialized. Message: [DEBUG] [", moduleId, "] ", message);
      return;
   }
   
   // Output debug messages if level is DEBUG or module is in debug mode
   if(m_globalLogLevel >= LOG_LEVEL_DEBUG || IsModuleInDebugMode(moduleId))
   {
      WriteLog("DEBUG", moduleId, message);
   }
}

//+------------------------------------------------------------------+
//| Log message with INFO level                                      |
//+------------------------------------------------------------------+
void CTsLogger::Info(string moduleId, string message)
{
   // Check if logger is initialized
   if(!m_initialized)
   {
      Print("WARNING: TsLogger not initialized. Message: [INFO] [", moduleId, "] ", message);
      return;
   }
   
   // Output informational messages if level is INFO or higher
   if(m_globalLogLevel >= LOG_LEVEL_INFO)
   {
      WriteLog("INFO", moduleId, message);
   }
}

//+------------------------------------------------------------------+
//| Log message with WARNING level                                   |
//+------------------------------------------------------------------+
void CTsLogger::Warning(string moduleId, string message)
{
   // Check if logger is initialized
   if(!m_initialized)
   {
      Print("WARNING: TsLogger not initialized. Message: [WARNING] [", moduleId, "] ", message);
      return;
   }
   
   // Output warnings if level is WARNING or higher
   if(m_globalLogLevel >= LOG_LEVEL_WARNING)
   {
      WriteLog("WARNING", moduleId, message);
   }
}

//+------------------------------------------------------------------+
//| Log message with ERROR level                                     |
//+------------------------------------------------------------------+
void CTsLogger::Error(string moduleId, string message)
{
   // Check if logger is initialized
   if(!m_initialized)
   {
      Print("ERROR: TsLogger not initialized. Message: [ERROR] [", moduleId, "] ", message);
      return;
   }
   
   // Always output errors regardless of global level
   WriteLog("ERROR", moduleId, message);
}

