//+------------------------------------------------------------------+
//|                                              CompareFunction.mqh |
//|                             2016-2017, metaquotes software corp. |
//|                                    2019-2021, dimitri pecheritsa |
//|                                         mql5.com/en/users/dmipec |
//|------------------------------------------------------------------|
//|   f| a universal overloaded compare method for all standard mql  |
//|types. returns 1 if x greater than y, -1 if x less than y, and 0  |
//|if x and y are equal. custom compare methods rely on two          |
//|interfaces: comparable and comparer. this array of functions      |
//|extends the idea behind the compare function file found in the mql|
//|standard librtary by metaquotes                                   |
//| 010| compare bool (x, y)                                         |
//| 020| compare char (x, y)                                         |
//| 030| compare uchar (x, y)                                        |
//| 040| compare short (x, y)                                        |
//| 050| compare ushort (x, y)                                       |
//| 060| compare color x, y)                                         |
//| 070| compare int (x, y)                                          |
//| 080| compare uint (x, y)                                         |
//| 090| compare datetime (x, y)                                     |
//| 100| compare long (x, y)                                         |
//| 110| compare ulong (x, y)                                        |
//| 120| compare float (x, y)                                        |
//| 130| compare double (x, y)                                       |
//| 140| compare string (x, y)                                       |
//| 150| compare custom (x, y)                                       |
//| 160| compare custom (x, y, comparer)                             |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| 010| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| bool                                                  |
//+------------------------------------------------------------------+
int Compare(const bool x,const bool y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 020| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| char                                                  |
//+------------------------------------------------------------------+
int Compare(const char x,const char y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 030| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| uchar                                                 |
//+------------------------------------------------------------------+
int Compare(const uchar x,const uchar y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 040| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| short                                                 |
//+------------------------------------------------------------------+
int Compare(const short x,const short y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 050| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| ushort                                                |
//+------------------------------------------------------------------+
int Compare(const ushort x,const ushort y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 060| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| color                                                 |
//+------------------------------------------------------------------+
int Compare(const color x,const color y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 070| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| int                                                   |
//+------------------------------------------------------------------+
int Compare(const int x,const int y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 080| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| uint                                                  |
//+------------------------------------------------------------------+
int Compare(const uint x,const uint y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 090| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| datetime                                              |
//+------------------------------------------------------------------+
int Compare(const datetime x,const datetime y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 100| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| long                                                  |
//+------------------------------------------------------------------+
int Compare(const long x,const long y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 110| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| ulong                                                 |
//+------------------------------------------------------------------+
int Compare(const ulong x,const ulong y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//| 120| compare x to y, by comparing the difference between x and y |
//|with some minimum value, where the user sets the number of        |
//|'digits' after the floating point for this minimum value          |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y  | float                                               |
//|  ->| digits| floating point accuracy, default 13                 |
//+------------------------------------------------------------------+
int Compare(const float x,const float y,int digits=13)
  {
   double min=pow(10,-fabs(digits));
   return (((fabs(x-y)<min))? 0 : ((x-y>=min))? 1 : -1);
  }
//+------------------------------------------------------------------+
//| 130| compare x to y, by comparing the difference between x and y |
//|with some minimum value, where the user sets the number of        |
//|'digits' after the floating point for this minimum value          |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y  | double                                              |
//|  ->| digits| floating point accuracy, default 13                 |
//+------------------------------------------------------------------+
int Compare(const double x,const double y,int digits=13)
  {
   double min=pow(10,-fabs(digits));
   return (((fabs(x-y)<min))? 0 : ((x-y>=min))? 1 : -1);
  }
//+------------------------------------------------------------------+
//| 140| compare x to y                                              |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| string                                                |
//+------------------------------------------------------------------+
int Compare(const string x,const string y)
  {
   return (x>y) ? 1 : (x<y) ? -1 : 0;
  }
//+------------------------------------------------------------------+
//|   i| generic interface for classes who want to support the       |
//|ability of being comparable to own various instances, via own     |
//|compare method that they must implement by this protocol          |
//+------------------------------------------------------------------+
template<typename T>
interface IComparable
  {
   int       Compare(T value);
  };
//+------------------------------------------------------------------+
//|   i| external generic comparer for instances of one class        |
//+------------------------------------------------------------------+
template<typename T>
interface IComparer
  {
   int       Compare(T x,T y);
  };
//+------------------------------------------------------------------+
//| 150| compare x to y by casting one of them to a comparable object|
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y| custom objects                                        |
//+------------------------------------------------------------------+
template<typename T>
int Compare(T x,T y)
  {
//--- try to convert to a comparable object
   IComparable<T>*comparable=dynamic_cast<IComparable<T>*>(x);
   return
      (comparable)?
//--- use the specified compare method
      comparable.Compare(y):
//--- unknown compare function
      int("nan");
  }
//+------------------------------------------------------------------+
//| 160| compare x to y using the external comparer. if comparer is  |
//|invalid returns nan, which means an error                         |
//|  <-| -1 less; 0 equal; 1 greater                                 |
//|  ->| x, y    | custom                                            |
//|  ->| comparer| an external compatible comparer                   |
//+------------------------------------------------------------------+
template<typename T>
int Compare(T x,T y,IComparer* comparer)
  {
//--- avoid invalid pointer
   if(CheckPointer(comparer)==0)
      return int("nan");
//--- use the proposed comparer
   return comparer.Compare(x,y);
  }
//+------------------------------------------------------------------+
