//+------------------------------------------------------------------+
//|                                                     TimeStep.mqh |
//|                                    Copyright 2018, Mukachi Corp. |
//|                                          https://www.mukachi.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2018, Mukachi Corp."
#property link      "https://www.mukachi.com"
#property version   "1.00"
//+------------------------------------------------------------------+
#include "Gates.mqh"
//+------------------------------------------------------------------+
//| Timestep (Cell)                                                  |
//+------------------------------------------------------------------+
class CTimeStep : public CObject
  {
private:
   int               m_gates;

   double            m_out;
   double            m_state;

   double            m_out_error;

   double            m_out_delta;
   double            m_state_delta;

   CA               *m_a;
   CI               *m_i;
   CF               *m_f;
   CO               *m_o;

   CGate            *m_gate[];

public:
                     CTimeStep(void);
                    ~CTimeStep(void);

   //--- Gates

   CA               *a() const { return(GetPointer(m_a)); }
   CI               *i() const { return(GetPointer(m_i)); }
   CF               *f() const { return(GetPointer(m_f)); }
   CO               *o() const { return(GetPointer(m_o)); }

   CGate            *gate(int idx) const { return(m_gate[idx]); }

   //--- CEC State (forward passing)

   double            out() const { return(m_out); }
   double            state() const { return(m_state); }

   //--- update state given previous state
   void              update_state(double prev_state);

   //--- CEC State (backward passing)
   double            out_delta() const { return(m_out_delta); }
   double            state_delta() const { return(m_state_delta); }
   //--- Upate Deltas
   void              out_delta(double out_delta);
   void              state_delta(double state_delta);
   //--- Total & Out Error
   double            out_error() const { return(m_out_error); }
   void              out_error(double out_error) { m_out_error=out_error; }
   //--- Update Gate Deltas
   void              update_delta(CTimeStep *prev,double state_delta,double out_delta);
  };
//+------------------------------------------------------------------+
CTimeStep::CTimeStep(void)
   : m_gates(4),m_out(0.0),m_state(0.0),
     m_out_error(0.0),m_state_delta(0.0)
  {
   ArrayResize(m_gate,m_gates);

   m_gate[0]=m_a=new CA();
   m_gate[1]=m_i=new CI();
   m_gate[2]=m_f=new CF();
   m_gate[3]=m_o=new CO();
  }
//+------------------------------------------------------------------+
CTimeStep::~CTimeStep(void)
  {
   for(int i=0;i<m_gates;i++)
     {
      if(CheckPointer(m_gate[i])==POINTER_DYNAMIC)
         delete(m_gate[i]);
     }
   ArrayFree(m_gate);
  }
//+------------------------------------------------------------------+
void CTimeStep::update_state(double prev_state)
  {
   m_state=a().out()*i().out()+f().out()*prev_state;
   m_out=tanh(m_state)*o().out();
  }
//+------------------------------------------------------------------+
void CTimeStep::update_delta(CTimeStep *prev,
                             double state_delta,
                             double out_delta)
  {
   m_state_delta=state_delta;

   m_a.delta(state_delta,i().out(),a().out());
   m_i.delta(state_delta,a().out(),i().out());

   if(prev==NULL) m_f.delta(state_delta,0,f().out());
   else m_f.delta(state_delta,prev.state(),f().out());

   m_o.delta(out_delta,state(),o().out());
  }
//+------------------------------------------------------------------+
