//+------------------------------------------------------------------+
//|                                                        Gates.mqh |
//|                                    Copyright 2018, Mukachi Corp. |
//|                                          https://www.mukachi.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2018, Mukachi Corp."
#property link      "https://www.mukachi.com"
#property version   "1.00"
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#include <Object.mqh>
//+------------------------------------------------------------------+
enum ENUM_ACTIVATION_TYPE
  {
   ACTIVATOR_TANH,
   ACTIVATOR_SIGMOID
  };
//+------------------------------------------------------------------+
enum ENUM_ACTIVATION_THRESHOLD
  {
   THRESHOLD_TANH=20,
   THRESHOLD_SIGMOID=10
  };
//+------------------------------------------------------------------+
//| Gate base                                                        |
//+------------------------------------------------------------------+
class CGate : public CObject
  {
protected:
   double            m_out;
   double            m_delta;

   double            dsigma(double out) { return(out*(1-out)); }
   void              activate(ENUM_ACTIVATION_TYPE activation,double net);

public:
   virtual          ~CGate(void){}
   //---
   virtual void      out(double net)=0;
   //---
   double            out() const { return(m_out); }
   double            delta() const { return(m_delta); }
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CGate::activate(ENUM_ACTIVATION_TYPE activation,double net)
  {
   switch(activation)
     {
      case ACTIVATOR_TANH:
        {
         if(net>THRESHOLD_TANH) m_out=1;
         else if(net<-THRESHOLD_TANH) m_out=-1;
         else
           {
            net=exp(net+net);
            m_out=(net-1.0)/(net+1.0);
           }
        }
      break;
      case ACTIVATOR_SIGMOID:
        {
         if(net>THRESHOLD_SIGMOID) m_out=1;
         else if(net<-THRESHOLD_SIGMOID) m_out=0;
         else m_out=1.0/(1.0+exp(-net));
        }
      break;
     }
  }
//+------------------------------------------------------------------+
//| Activation gate                                                  |
//+------------------------------------------------------------------+
class CA : public CGate
  {
public:
   void out(double net) override { activate(ACTIVATOR_TANH,net); }
   void delta(double state_delta,double i,double a)
     {
      m_delta=state_delta*i*(1-(a*a));
     }
  };
//+------------------------------------------------------------------+
//| Input Gate                                                       |
//+------------------------------------------------------------------+
class CI : public CGate
  {
public:
   void out(double net) override { activate(ACTIVATOR_SIGMOID,net); }
   void delta(double state_delta,double a,double i)
     {
      m_delta=state_delta*a*dsigma(i);
     }
  };
//+------------------------------------------------------------------+
//| Forget Gate                                                      |
//+------------------------------------------------------------------+
class CF : public CGate
  {
public:
   void out(double net) override { activate(ACTIVATOR_SIGMOID,net); }
   void delta(double state_delta,double prev_state,double f)
     {
      m_delta=state_delta*prev_state*dsigma(f);
     }
  };
//+------------------------------------------------------------------+
//| Output Gate                                                      |
//+------------------------------------------------------------------+
class CO : public CGate
  {
public:
   void out(double net) override { activate(ACTIVATOR_SIGMOID,net); }
   void delta(double out_delta,double state,double o)
     {
      m_delta=out_delta*tanh(state)*dsigma(o);
     }
  };
//+------------------------------------------------------------------+
