#include "Deal.mqh"

struct INTERVAL
{
private:
  void Init( void )
  {
    this.Profit = 0;
    this.ProfitPlus = 0;
    this.Total = 0;
    this.TotalPlus = 0;

    this.MaxDrawDown = 0;
    this.RecoveryFactor = 0;

    return;
  }

  void CalculateFinal( void )
  {
    this.ProfitMinus = this.Profit - this.ProfitPlus;

    this.ProfitFactor = this.ProfitMinus ? -this.ProfitPlus / this.ProfitMinus : DBL_MAX;
    this.Mean = this.Total ? this.Profit / this.Total : 0;

    this.TotalMinus = this.Total - this.TotalPlus;

    return;
  }

public:
  datetime OpenTime;
  datetime CloseTime;

  double Profit;
  double ProfitPlus;
  double ProfitMinus;

  int Total;
  int TotalPlus;
  int TotalMinus;

  double ProfitFactor;
  double Mean;

  double MaxDrawDown;
  double RecoveryFactor;

  datetime Begin;
  datetime End;

  void Calculate( const DEAL &Deals[], const int Left, const int Right )
  {
    this.Init();

    for (int i = Left; i < Right; i++)
      if (!Deals[i].IsNull())
      {
        if (Deals[i].Profit > 0)
        {
          this.ProfitPlus += Deals[i].Profit;

          this.TotalPlus++;
        }

        this.Profit += Deals[i].Profit;
        this.Total++;
      }

    this.CalculateFinal();

    this.OpenTime = Deals[Left - 1].OpenTime + 1;
    this.CloseTime = Deals[Right].OpenTime - 1;

    return;
  }

  void Calculate( const INTERVAL &Intervals[] )
  {
    this.Init();

    for (int i = ::ArraySize(Intervals) - 1; i >= 0; i--)
      this += Intervals[i];

    this.CalculateFinal();

    return;
  }

  void operator +=( const INTERVAL &Interval )
  {
    this.Profit += Interval.Profit;
    this.ProfitPlus += Interval.ProfitPlus;
    this.Total += Interval.Total;
    this.TotalPlus += Interval.TotalPlus;

    return;
  }

  bool IsTime( const datetime time ) const
  {
    return((this.OpenTime <= time) && (time <= this.CloseTime));
  }

  bool IsTime( const DEAL &Deal ) const
  {
    return(this.IsTime(Deal.OpenTime));
  }

  void CalculateDD( const DEAL &Deals[] )
  {
    const int Size = ::ArraySize(Deals);

    this.MaxDrawDown = 0;

    double TempProfit = 0;
    double MaxProfit = 0;
    double DrawDown;

    for (int i = 0; i < Size; i++)
      if (!Deals[i].IsNull() && this.IsTime(Deals[i]))
      {
        TempProfit += Deals[i].Profit;

        if (TempProfit > MaxProfit)
          MaxProfit = TempProfit;
        else if ((DrawDown = MaxProfit - TempProfit) > this.MaxDrawDown)
          this.MaxDrawDown = DrawDown;
      }

    this.RecoveryFactor = this.MaxDrawDown ? TempProfit / this.MaxDrawDown : 0;

    return;
  }

  string ToString( const double FullProfit = 0 ) const
  {
    return(::TimeToString(this.OpenTime, TIME_SECONDS) + " - " + ::TimeToString(this.CloseTime, TIME_SECONDS) +
           " : Profit = " + ::DoubleToString(this.Profit, 2) +
           (FullProfit ? " (" + ::DoubleToString(100 * this.Profit / FullProfit, 2) + "%)" : NULL) +
           ", Total = " + (string)this.Total +
           (this.Total ? " (" + ::DoubleToString(100.0 * this.TotalPlus / this.Total, 2) + "%)" : NULL) +
           ", PF = " + (this.ProfitFactor != DBL_MAX ? ::DoubleToString(this.ProfitFactor, 2) : "Max") +
           ", Mean = " + ::DoubleToString(this.Mean, 2) +
           (this.MaxDrawDown ? ", DD = " + ::DoubleToString(this.MaxDrawDown, 2) +
                               ", RF = " + ::DoubleToString(this.RecoveryFactor, 2) : NULL));
  }
};