// #define TICKS_ALPARI

#include "Rates.mqh"

class TICKS : public DATA<MqlTick>
{
private:
  static string GetTickFlag( uint tickflag )
  {
    string flag = " " + (string)tickflag;

  #define TICKFLAG_MACRO(A) flag += ((bool)(tickflag & TICK_FLAG_##A)) ? " TICK_FLAG_" + #A : ""; \
                            tickflag -= tickflag & TICK_FLAG_##A;
    TICKFLAG_MACRO(BID)
    TICKFLAG_MACRO(ASK)
    TICKFLAG_MACRO(LAST)
    TICKFLAG_MACRO(VOLUME)
    TICKFLAG_MACRO(BUY)
    TICKFLAG_MACRO(SELL)
  #undef TICKFLAG_MACRO

    if (tickflag)
      flag += " FLAG_UNKNOWN (" + (string)tickflag + ")";

    return(flag);
  }

  static long StringToTimeMsc( const string &Str )
  {
    static MqlDateTime DateTime = {0};

    DateTime.year = Str[0] * 1000 + Str[1] * 100 + Str[2] * 10 + Str[3] - '0' * 1111;
    DateTime.mon = Str[5] * 10 + Str[6] - '0' * 11;
    DateTime.day = Str[8] * 10 + Str[9] - '0' * 11;

    DateTime.hour = Str[11] * 10 + Str[12] - '0' * 11;
    DateTime.min = Str[14] * 10 + Str[15] - '0' * 11;
    DateTime.sec = Str[17] * 10 + Str[18] - '0' * 11;

    return(::StructToTime(DateTime) * (long)1000 + Str[20] * 100 + Str[21] * 10 + Str[22] - '0' * 111);
  }

  static double StringToDouble( const string &Str, uint &Pos )
  {
    static const double TenPow[] = {1, 10, 100, 1000, 10000, 100000, 1000000};

    const uint Size = ::StringLen(Str);

    ulong Res = 0;
    uint point = 0;

    while (Pos < Size)
    {
      const int Digit = Str[Pos];

      if (((Digit >= '0') && (Digit <= '9')))
        Res = Res * 10 + Digit - '0';
      else if (Digit == '.')
        point = Pos + 2;
      else
      {
        Pos++;

        break;
      }

      Pos++;
    }

    return(Res / TenPow[Pos - point]);
  }

  virtual bool StringToTick( const string &Str, MqlTick &Tick ) const
  {
    Tick.time_msc = TICKS::StringToTimeMsc(Str);
    Tick.time = (datetime)(Tick.time_msc / 1000);

    uint Pos = 24;

    Tick.bid = TICKS::StringToDouble(Str, Pos);
    Tick.ask = TICKS::StringToDouble(Str, Pos);

    return(true);
  }

  static long StringToTimeMsc( const uchar &Bytes[], const uint &Pos )
  {
    static MqlDateTime PrevDate = {0};
    static datetime PrevTime = 0;

    static MqlDateTime DateTime = {0};

    DateTime.year = Bytes[Pos + 0] * 1000 + Bytes[Pos + 1] * 100 + Bytes[Pos + 2] * 10 + Bytes[Pos + 3] - '0' * 1111;
    DateTime.mon = Bytes[Pos + 5] * 10 + Bytes[Pos + 6] - '0' * 11;
    DateTime.day = Bytes[Pos + 8] * 10 + Bytes[Pos + 9] - '0' * 11;

    if ((PrevDate.day != DateTime.day) || (PrevDate.mon != DateTime.mon) || (PrevDate.year != DateTime.year))
    {
      PrevTime = ::StructToTime(DateTime);

      PrevDate = DateTime;
    }

  #ifndef TICKS_ALPARI
    return((PrevTime +
            (Bytes[Pos + 11] * 10 + Bytes[Pos + 12] - '0' * 11) * 3600 +
            (Bytes[Pos + 14] * 10 + Bytes[Pos + 15] - '0' * 11) * 60 +
             Bytes[Pos + 17] * 10 + Bytes[Pos + 18] - '0' * 11) * (long)1000 + Bytes[Pos + 20] * 100 + Bytes[Pos + 21] * 10 + Bytes[Pos + 22] - '0' * 111);
  #else // TICKS_ALPARI
    return((PrevTime +
            (Bytes[Pos + 11] * 10 + Bytes[Pos + 12] - '0' * 11) * 3600 +
            (Bytes[Pos + 14] * 10 + Bytes[Pos + 15] - '0' * 11) * 60 +
             Bytes[Pos + 17] * 10 + Bytes[Pos + 18] - '0' * 11) * (long)1000); // http://ticks.alpari.org/
  #endif // TICKS_ALPARI
  }

  static double StringToDouble( const uchar &Bytes[], uint &Pos )
  {
    static const double TenPow[] = {1, 10, 100, 1000, 10000, 100000, 1000000};

    const uint Size = ::ArraySize(Bytes);

    ulong Res = 0;
    uint point = 0;

    while (Pos < Size)
    {
      const uchar Digit = Bytes[Pos];

      if (((Digit >= '0') && (Digit <= '9')))
        Res = Res * 10 + Digit - '0';
      else if (Digit == '.')
        point = Pos + 2;
      else
      {
        Pos++;

        break;
      }

      Pos++;
    }

    return(Res / TenPow[Pos - point]);
  }

  virtual bool StringToTick( const uchar &Bytes[], MqlTick &Tick, uint &Pos ) const
  {
    Tick.time_msc = TICKS::StringToTimeMsc(Bytes, Pos);
    Tick.time = (datetime)(Tick.time_msc / 1000);

  #ifndef TICKS_ALPARI
    Pos += 24;
  #else // TICKS_ALPARI
    // http://ticks.alpari.org/
    Pos += 20;

    TICKS::StringToDouble(Bytes, Pos);
  #endif // TICKS_ALPARI

    Tick.bid = TICKS::StringToDouble(Bytes, Pos);
    Tick.ask = TICKS::StringToDouble(Bytes, Pos);

    return(true);
  }

  static int GetDigits( double Price )
  {
    int Res = 0;

    while ((bool)(Price = ::NormalizeDouble(Price - (int)Price, 8)))
    {
      Price *= 10;

      Res++;
    }

    return(Res);
  }

  static int GetDigits( const MqlTick &Tick )
  {
    return(::MathMax(TICKS::GetDigits(Tick.bid), TICKS::GetDigits(Tick.ask)));
  }

public:
  #define TOSTRING(A) " " + #A + " = " + (string)Tick.A

  static string ToString( const MqlTick &Tick )
  {
    return(TOSTRING(time) + "." + ::IntegerToString(Tick.time_msc % 1000, 3, '0') +
           TOSTRING(bid) + TOSTRING(ask) + TOSTRING(last)+ TOSTRING(volume) + TICKS::GetTickFlag(Tick.flags));
  }

  #undef TOSTRING

  static int Correct( MqlTick &Ticks[] )
  {
    int Res = 0;
    const int Size = ::ArraySize(Ticks);

    if (Size)
    {
      Ticks[0].flags = TICK_FLAG_BID | TICK_FLAG_ASK;

      for (int i = 1; _CS(i < Size); i++)
      {
        Ticks[i].flags = ((Ticks[i].bid == Ticks[i - 1].bid) ? 0 : TICK_FLAG_BID) |
                         ((Ticks[i].ask == Ticks[i - 1].ask) ? 0 : TICK_FLAG_ASK) |
                         ((Ticks[i].last == Ticks[i - 1].last) ? 0 : TICK_FLAG_LAST);

        if (Ticks[i].time_msc < Ticks[i - 1].time_msc)
        {
//          _P(TICKS::ToString(Ticks[i - 1])); // #include <Debug.mqh>
//          _P(TICKS::ToString(Ticks[i]));

          Ticks[i].time_msc = Ticks[i - 1].time_msc;
          Ticks[i].time = Ticks[i - 1].time;

          Res++;
        }

      }
    }

    return(Res);
  }

  uint Add( const string &Str )
  {
    MqlTick Tick = {0};

    if (this.StringToTick(Str, Tick))
      this.Add(Tick); // https://www.mql5.com/ru/forum/1111/page2195#comment_7119365

    return(this.GetAmount());
  }

  uint Add( const uchar &Bytes[], const uint StartPos = 0 )
  {
    const uint Size = ::ArraySize(Bytes);
    bool FirstTick = true;

    MqlTick Tick = {0};

  #ifndef TICKS_ALPARI
    for (uint i = StartPos; i < Size; i++)
  #else // TICKS_ALPARI
    for (uint i = StartPos; i < Size;) // http://ticks.alpari.org/
  #endif // TICKS_ALPARI
    {
      TICKS::StringToTick(Bytes, Tick, i);

      if (FirstTick)
      {
      #ifndef TICKS_ALPARI
        this.SetReserve((int)((Size - StartPos) / (i - StartPos + 1)));
      #else // TICKS_ALPARI
        this.SetReserve((int)((Size - StartPos) / (i - StartPos))); // http://ticks.alpari.org/
      #endif // TICKS_ALPARI

        FirstTick = false;
      }

      this.Add(Tick);
    }

    return(this.GetAmount());
  }

  uint Add( const string &Str[] )
  {
    const int Size = this.SetReserve(::ArraySize(Str));

    for (int i = 0; _CS(i < Size); i++)
      this.Add(Str[i]);

    return(this.GetAmount());
  }

  uint Add( const MqlTick &Ticks[] )
  {
    ::ArrayCopy(this.Data, Ticks, ::ArraySize(this.Data));

    return(this.GetAmount());
  }

  int GetDigits( const uint Amount = 1000 ) const
  {
    int Res = 0;

    const uint Size = this.GetAmount();

    if ((!Amount) || (Amount >= Size))
      for (uint i = 0; _CS(i < Size); i++)
      {
        const int digits = TICKS::GetDigits(this.Data[i]);

        if (digits > Res)
          Res = digits;
      }
    else
      for (uint i = 0; _CS(i < Amount); i++)
      {
        const int digits = TICKS::GetDigits(this.Data[(int)(::MathRand() * (long)Size / (SHORT_MAX + 1))]);

        if (digits > Res)
          Res = digits;
      }

    return(Res);
  }

  int ToSymbol( const SYMBOL* const CustomSymbol ) const
  {
    return(CustomSymbol.CloneTicks(this.Data));
  }

  int Correct( void )
  {
    return(TICKS::Correct(this.Data));
  }

  uint ToRates( RATES* const Rates ) const
  {
    return(Rates.Add(this.Data));
  }
};