//+------------------------------------------------------------------+
//|                                                      C_Trade.mqh |
//|                                     Copyright 2017, Mark Wilson. |
//|                          https://www.mql5.com/en/users/balrog100 |
//+------------------------------------------------------------------+
#include <Object.mqh>
#include <GlobalChartFunctions.mqh>

#property copyright "Copyright 2017, Mark Wilson"
#property link      "https://www.mql5.com/en/users/balrog100"
#property version   "1.00"
#property strict
//#define D_TEST_PRINT_FUNCTION_START                      //Comment out if want to remove print functions.
//#define D_PRINT_TRADE_UPDATES                            //Comment out if want to remove print functions.
//#define D_TEST_PRINT_FUNCTION_END                        //Comment out if want to remove print functions.

//+------------------------------------------------------------------+
//| Enum's                                                           |
//+------------------------------------------------------------------+
enum E_BS
  {
   BUY=OP_BUY,
   SELL=OP_SELL
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum StopLossCheckMethod   //Used in C_TrailingStopLoss
  {
   PreviousClose,
   BidAsk,
   HighLow
  };
//+------------------------------------------------------------------+
//| Class C_Trade                                                  |
//+------------------------------------------------------------------+
class C_Trade : public CObject
  {
protected:
   //Protected Variables that are associated with most trades.
   int               m_intTicket;
   string            m_strSymbol;
   E_BS              m_eType;
   double            m_dblVolume;
   double            m_dblOpenPrice;
   datetime          m_dtOpenTime;
   double            m_dblStopLoss;
   double            m_dblTakeProfit;
   double            m_dblHiddenStopLoss;
   double            m_dblHiddenTakeProfit;
   string            m_strComment;
   int               m_intMagicNumber;
   //Protected variables that are specific to this class
   double            m_dblLiveSpread;
   bool              m_boolDrawLines;
public:
   //Constructor/Destructor
                     C_Trade();
                     C_Trade(const string m_strSymbol,const E_BS eType,const double dblVolume,const double dblOpenPrice,const double dblHiddenStopLoss,const double dblHiddenTakeProfit,const string strComment,const int intMagicNumber,const double dblLiveSpread,const bool boolDrawLines=False);
                    ~C_Trade();
   //Public Accessor Functions
   int GetTicket(){return this.m_intTicket;};
   string GetSymbol(){return this.m_strSymbol;};
   E_BS GetType(){return this.m_eType;};
   double GetVolume(){return this.m_dblVolume;};
   double GetOpenPrice(){return this.m_dblOpenPrice;};
   datetime GetOpenTime(){return this.m_dtOpenTime;};
   double GetStopLoss(){return this.m_dblStopLoss;};
   double GetTakeProfit(){return this.m_dblTakeProfit;};
   double GetHiddenStopLoss(){return this.m_dblHiddenStopLoss;};
   double GetHiddenTakeProfit(){return this.m_dblHiddenTakeProfit;};
   string GetComment(){return this.m_strComment;};
   int GetMagicNumber(){return this.m_intMagicNumber;};
   double GetLiveSpread(){return this.m_dblLiveSpread;};
   bool GetDrawLines(){return this.m_boolDrawLines;};
   //Public Functions
   void              Replicate(C_Trade &objTrade);
   virtual string    Description();
   virtual int       OrderSend(const double dblPrice,const int intSlippage,const color arrow_color=clrNONE);
   virtual bool      OrderModify(const double dblPrice,const double dblHiddenStopLoss,const double dblHiddenTakeProfit,const color arrow_color=clrNONE,const bool boolUpdateLiveTrade=True);
   virtual bool      OrderClose(const double dblPrice,const int intSlippage,const color arrow_color=clrNONE);
   virtual bool      ScanWithEveryTick(const int intSlippage,const color stoploss_color=clrNONE,const color takeprofit_color=clrNONE);
   void              StopLossRiskFor1LotInAccountCurrency(const double dblPrice,double &dblHiddenStopLossRisk,double &dblLiveStopLossRisk);
   bool              TradeIsLive();
private:
   //Private Functions
   void              ResetVariables();
   void              CalculateSuitableLiveStoplossAndTakeProfit(double dblPrice,double &dblStopLoss,double &dblTakeProfit);
   bool              DeleteHiddenLines();
   bool              AddHiddenLines();
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
C_Trade::C_Trade()
  {
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
C_Trade::C_Trade(const string strSymbol,const E_BS eType,const double dblVolume,const double dblOpenPrice,const double dblHiddenStopLoss,const double dblHiddenTakeProfit,const string strComment,const int intMagicNumber,const double dblLiveSpread,const bool boolDrawLines=False)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   this.ResetVariables();

   this.m_strSymbol=strSymbol;
   this.m_eType=eType;
   this.m_dblVolume=dblVolume;
   this.m_dblOpenPrice=dblOpenPrice;
   this.m_dblHiddenStopLoss=dblHiddenStopLoss;
   this.m_dblHiddenTakeProfit=dblHiddenTakeProfit;
   this.m_strComment=strComment;
   this.m_intMagicNumber=intMagicNumber;
   this.m_dblLiveSpread=dblLiveSpread;
   this.m_boolDrawLines= boolDrawLines;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
C_Trade::~C_Trade()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//For some reason MetaTrader hangs when I close the trade as the class is destroyed, I do seem to be able to delete the lines, 
//so I will just do that for now.
   this.DeleteHiddenLines();

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| Public Functions                                                 |
//+------------------------------------------------------------------+
void C_Trade::Replicate(C_Trade &objTrade)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   this.m_intTicket=objTrade.m_intTicket;
   this.m_strSymbol=objTrade.m_strSymbol;
   this.m_eType=objTrade.m_eType;
   this.m_dblVolume=objTrade.m_dblVolume;
   this.m_dblOpenPrice=objTrade.m_dblOpenPrice;
   this.m_dtOpenTime=objTrade.m_dtOpenTime;
   this.m_dblStopLoss=objTrade.m_dblStopLoss;
   this.m_dblTakeProfit=objTrade.m_dblTakeProfit;
   this.m_dblHiddenStopLoss=objTrade.m_dblHiddenStopLoss;
   this.m_dblHiddenTakeProfit=objTrade.m_dblHiddenTakeProfit;
   this.m_strComment=objTrade.m_strComment;
   this.m_intMagicNumber=objTrade.m_intMagicNumber;
   this.m_dblLiveSpread=objTrade.m_dblLiveSpread;
   this.m_boolDrawLines=objTrade.m_boolDrawLines;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
string C_Trade::Description()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   string strRet="   Tkt: "+IntegerToString(this.m_intTicket)+", SL: "+DoubleToString(this.m_dblStopLoss,5)+", HSL: "+DoubleToString(this.m_dblHiddenStopLoss,5)+", TP: "+DoubleToString(this.m_dblTakeProfit,5)+", HTP: "+DoubleToString(this.m_dblHiddenTakeProfit,5);

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return strRet;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int C_Trade::OrderSend(const double dblPrice,const int intSlippage,const color arrow_color=clrNONE)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Once the trade has been setup with the initiator, C_Trade(const string strSy ..., then this function is used to 
//create the actual trade.   It should be very similar to the standard OrderSend function.

//First, work out a stoploss and takeprofit level that is suitable for the live trade
   double dblStopLoss,dblTakeProfit;
   this.CalculateSuitableLiveStoplossAndTakeProfit(dblPrice,dblStopLoss,dblTakeProfit);

//Then attempt to initiate the trade.
   int intRet=OrderSend(this.m_strSymbol,this.m_eType,this.m_dblVolume,dblPrice,intSlippage,dblStopLoss,dblTakeProfit,this.m_strComment,this.m_intMagicNumber,0,arrow_color);

//If we have successfully opened the trade, then select it and populate the open price, ticket etc.
   if(intRet>0)
     {
      //Update the ticket.
      this.m_intTicket=intRet;

      if(OrderSelect(intRet,SELECT_BY_TICKET,MODE_TRADES))
        {
         this.m_dtOpenTime=OrderOpenTime();
         this.m_dblOpenPrice=OrderOpenPrice();
         this.m_dblStopLoss=OrderStopLoss();
         this.m_dblTakeProfit=OrderTakeProfit();

#ifdef D_PRINT_TRADE_UPDATES
         Print(__FUNCTION__,this.Description());
#endif
        }
      else
        {
         Print(__FUNCTION__," Error: Could not select trade ticket ",intRet);
        }

      //Add Lines
      if(this.m_boolDrawLines) this.AddHiddenLines();
     }

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return intRet;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade::OrderModify(const double dblPrice,const double dblHiddenStopLoss,const double dblHiddenTakeProfit,const color arrow_color=clrNONE,const bool boolUpdateLiveTrade=True)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   bool boolRet=True;

   if(boolUpdateLiveTrade)
     {
      //First, work out a stoploss and takeprofit level that is suitable for the live trade
      double dblStopLoss,dblTakeProfit;
      this.CalculateSuitableLiveStoplossAndTakeProfit(dblPrice,dblStopLoss,dblTakeProfit);

      //Attempt to modify the live trade
      boolRet=OrderModify(this.m_intTicket,dblPrice,dblStopLoss,dblTakeProfit,0,arrow_color);

      //If we update the live trade, then adjust the hidden ones and update the chart
      if(boolRet)
        {
         this.m_dblHiddenStopLoss=dblHiddenStopLoss;
         this.m_dblHiddenTakeProfit=dblHiddenTakeProfit;

#ifdef D_PRINT_TRADE_UPDATES
         Print(__FUNCTION__,this.Description());
#endif

         if(this.m_boolDrawLines) this.AddHiddenLines();
        }
     }
   else
     {
      //We are only updating the hidden stoploss and take profit (this should be more efficient, but more is left at risk if the connection goes down).
      this.m_dblHiddenStopLoss=dblHiddenStopLoss;
      this.m_dblHiddenTakeProfit=dblHiddenTakeProfit;

#ifdef D_PRINT_TRADE_UPDATES
      Print(__FUNCTION__,this.Description());
#endif

      if(this.m_boolDrawLines) this.AddHiddenLines();
     }

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return boolRet;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade::OrderClose(const double dblPrice,const int intSlippage,const color arrow_color=clrNONE)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Close the trade.
   bool boolRet=OrderClose(this.m_intTicket,this.m_dblVolume,dblPrice,intSlippage,arrow_color);

//If successful, then change the volume to zero
   if(boolRet)
     {
      this.m_dblVolume=0;

#ifdef D_PRINT_TRADE_UPDATES
      Print(__FUNCTION__,this.Description());
#endif

      if(this.m_boolDrawLines) this.DeleteHiddenLines();
     }

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return boolRet;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade::ScanWithEveryTick(const int intSlippage,const color stoploss_color=clrNONE,const color takeprofit_color=clrNONE)
  {

//This should be called on every tick to see if the spot has gone past the hidden stoploss or takeprofit.   If it has it attempts
//to close the trade and returns true if successful.

   bool boolRet=False;

//Report that we should close the object if we do not have a valid ticket number.
   if(this.m_intTicket<0)  return True;

   RefreshRates();

   if(this.m_eType==BUY)
     {  //Test buy trade to see if the bid has passed the hidden stoploss or takeprofit
      double dblBid=MarketInfo(this.m_strSymbol,MODE_BID);
      if(dblBid>this.m_dblHiddenTakeProfit && this.m_dblHiddenTakeProfit>0)
        {
         boolRet=this.OrderClose(dblBid,intSlippage,takeprofit_color);
         //If we cannot close the trade, but we can successfully select the trade in the history, assume that it has been shut down using 
         //another method and return True.
         if(!boolRet) boolRet=OrderSelect(this.m_intTicket,SELECT_BY_TICKET,MODE_HISTORY);
        }
      else if(dblBid<this.m_dblHiddenStopLoss)
        {
         boolRet=this.OrderClose(dblBid,intSlippage,stoploss_color);
         //If we cannot close the trade, but we can successfully select the trade in the history, assume that it has been shut down using 
         //another method and return True.
         if(!boolRet) boolRet=OrderSelect(this.m_intTicket,SELECT_BY_TICKET,MODE_HISTORY);
        }

     }
   else if(this.m_eType==SELL)
     {  //Test sell trade to see if the ask has passed the hidden stoploss or takeprofit
      double dblAsk=MarketInfo(this.m_strSymbol,MODE_ASK);
      if(dblAsk<this.m_dblHiddenTakeProfit)
        {
         boolRet=this.OrderClose(dblAsk,intSlippage,takeprofit_color);
         //If we cannot close the trade, but we can successfully select the trade in the history, assume that it has been shut down using 
         //another method and return True.
         if(!boolRet) boolRet=OrderSelect(this.m_intTicket,SELECT_BY_TICKET,MODE_HISTORY);
        }
      else if(dblAsk>this.m_dblHiddenStopLoss && this.m_dblHiddenStopLoss>0)
        {
         boolRet=this.OrderClose(dblAsk,intSlippage,stoploss_color);
         //If we cannot close the trade, but we can successfully select the trade in the history, assume that it has been shut down using 
         //another method and return True.
         if(!boolRet) boolRet=OrderSelect(this.m_intTicket,SELECT_BY_TICKET,MODE_HISTORY);
        }
     }

   return boolRet;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void C_Trade::StopLossRiskFor1LotInAccountCurrency(const double dblPrice,double &dblHiddenStopLossRisk,double &dblLiveStopLossRisk)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//Because the Live StopLoss Risk is dynamic and can change with the broker StopLevel/FreezeLevel, this routine uses
//the hidden stoploss and potential live stoploss should a trade be added now.

   double dblLiveStopLoss,dblLiveTakeProfit;

//Get the potential live stoploss that we would use if we created this trade now.
   this.CalculateSuitableLiveStoplossAndTakeProfit(dblPrice,dblLiveStopLoss,dblLiveTakeProfit);

//Get the tickvalue and ticksize for this trade.
   double dblMarketTickSize=MarketInfo(this.m_strSymbol,MODE_TICKSIZE);
   double dblMarketTickValue=MarketInfo(this.m_strSymbol,MODE_TICKVALUE);

   if(this.m_eType==BUY)
     {  //Trade is a buy
      dblHiddenStopLossRisk=MathMin(dblPrice-this.m_dblHiddenStopLoss,0)*dblMarketTickValue/dblMarketTickSize;
      dblLiveStopLossRisk=MathMin(dblPrice-dblLiveStopLoss,0)*dblMarketTickValue/dblMarketTickSize;
     }
   else
     {  //Assume a sell
      dblHiddenStopLossRisk=MathMin(this.m_dblHiddenStopLoss-dblPrice,0)*dblMarketTickValue/dblMarketTickSize;
      dblLiveStopLossRisk=MathMin(dblLiveStopLoss-dblPrice,0)*dblMarketTickValue/dblMarketTickSize;
     }

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade::TradeIsLive()
  {
//Assume that the trade is live if we have a ticket number >=0 and some volume
   bool boolRet=False;
   if(this.m_intTicket>=0 && this.m_dblVolume>0) boolRet=True;
   return boolRet;
  }
//+------------------------------------------------------------------+
//| Protected Functions                                              |
//+------------------------------------------------------------------+
void C_Trade::ResetVariables()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   this.m_intTicket=-1;
   this.m_intMagicNumber=0;
   this.m_strSymbol="";
   this.m_eType=BUY;
   this.m_dblVolume=0;
   this.m_dblOpenPrice=0;
   this.m_dtOpenTime=0;
   this.m_dblStopLoss=0;
   this.m_dblTakeProfit=0;
   this.m_dblHiddenStopLoss=0;
   this.m_dblHiddenTakeProfit=0;
   this.m_strComment="";
   this.m_dblLiveSpread=0;

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void C_Trade::CalculateSuitableLiveStoplossAndTakeProfit(double dblPrice,double &dblStopLoss,double &dblTakeProfit)
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

//We need to ensure that the stoploss and take profit have an extra live spread added onto them to avoid stoploss hunting, so
//by default we add/subtract the live spread.   However we also check the broker stoplevel and freezelevel to ensure that the live
//stoploss and takeprofit are larger than these distances.

//Initiate the stoploss and takeprofit to zero.
   dblStopLoss=0;
   dblTakeProfit=0;

//Get the value of 1 point.
   double dblPoint=MarketInfo(this.m_strSymbol,MODE_POINT);
   int intDigits=(int)MarketInfo(this.m_strSymbol,MODE_DIGITS);

//Find the StopLevel and FreezeLevel that are suitable for this symbol.
   double dblStopLevel=MarketInfo(this.m_strSymbol,MODE_STOPLEVEL);
   double dblFreezeLevel=MarketInfo(this.m_strSymbol,MODE_FREEZELEVEL);
   double dblMinDistance= MathMax(dblStopLevel,dblFreezeLevel)*dblPoint;

   if(this.m_eType==BUY)
     {
      if(this.m_dblHiddenStopLoss>0)
        {  //Have a positive hidden stoploss, so calculate a suitable live stoploss.
         double dblSLMin=dblPrice-dblMinDistance-this.m_dblLiveSpread;
         double dblSLMin2=this.m_dblHiddenStopLoss-this.m_dblLiveSpread;
         dblStopLoss=NormalizeDouble(MathMax(0,MathMin(dblSLMin,dblSLMin2)),intDigits);
        }
      if(this.m_dblHiddenTakeProfit>0)
        {
         double dblTPMax=dblPrice+dblMinDistance+this.m_dblLiveSpread;
         double dblTPMax2=this.m_dblHiddenTakeProfit+this.m_dblLiveSpread;
         dblTakeProfit=NormalizeDouble(MathMax(dblTPMax,dblTPMax2),intDigits);
        }
     }
   else if(this.m_eType==SELL)
     {
      if(this.m_dblHiddenStopLoss>0)
        {  //Have a positive hidden stoploss, so calculate a suitable live stoploss.
         double dblSLMax=dblPrice+dblMinDistance+this.m_dblLiveSpread;
         double dblSLMax2=this.m_dblHiddenStopLoss+this.m_dblLiveSpread;
         dblStopLoss=NormalizeDouble(MathMax(dblSLMax,dblSLMax2),intDigits);
        }
      if(this.m_dblHiddenTakeProfit>0)
        {
         double dblTPMin=dblPrice-dblMinDistance-this.m_dblLiveSpread;
         double dblTPMin2=this.m_dblHiddenTakeProfit-this.m_dblLiveSpread;
         dblTakeProfit=NormalizeDouble(MathMax(0,MathMin(dblTPMin,dblTPMin2)),intDigits);
        }
     }

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade::DeleteHiddenLines()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   string strNameSL="HSL_"+IntegerToString(this.m_intTicket);
   string strNameTP="HTP_"+IntegerToString(this.m_intTicket);

//Delete the lines if they already exist.
   DeleteLine(0,strNameSL,True);
   DeleteLine(0,strNameTP,True);

   ChartRedraw();

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return True;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool C_Trade::AddHiddenLines()
  {
#ifdef D_TEST_PRINT_FUNCTION_START
   Print(__FUNCTION__," Start");
#endif

   bool boolSL=True,boolTP=True;

//Get the colour of the lines
   long lngCol=clrNONE;
   ChartGetInteger(0,CHART_COLOR_STOP_LEVEL,0,lngCol);
   color colSTP=(color)lngCol;

//Get the names of the lines.
   string strNameSL="HSL_"+IntegerToString(this.m_intTicket);
   string strNameTP="HTP_"+IntegerToString(this.m_intTicket);

//Delete the lines if they already exist.
   this.DeleteHiddenLines();

//Create hidden stoploss line
   if(this.m_dblHiddenStopLoss>0) boolSL=CreateHorizontalLine(0,strNameSL,0,this.m_dblHiddenStopLoss,colSTP,STYLE_DASHDOT,1,True,True);

//Create hidden takeprofit line
   if(this.m_dblHiddenTakeProfit>0) boolTP=CreateHorizontalLine(0,strNameTP,0,this.m_dblHiddenTakeProfit,colSTP,STYLE_DASHDOT,1,True,True);

//Redraw the chart.
   ChartRedraw();

#ifdef D_TEST_PRINT_FUNCTION_END
   Print(__FUNCTION__," End");
#endif

   return (boolSL && boolTP);
  }
//+------------------------------------------------------------------+
