//+------------------------------------------------------------------+
//|                                    C_Trade_SteppedStopLoss_3.mq4 |
//|                                     Copyright 2017, Mark Wilson. |
//|                          https://www.mql5.com/en/users/balrog100 |
//+------------------------------------------------------------------+
#include <C_Trade_SteppedStopLoss.mqh>

#property copyright "Copyright 2017, Mark Wilson"
#property link      "https://www.mql5.com/en/users/balrog100"
#property version   "1.00"
#property strict


//+------------------------------------------------------------------+
//| Inputs                                                           |
//+------------------------------------------------------------------+
input E_BS  I_BuyOrSell=BUY;                                               //Buy or Sell
input int I_HiddenStopLossStepSizeInPoints=25;                             //Initial StopLoss StepSize in Points.
input int I_StopLossSteps=5;                                               //Number of Steps.
input double I_SpreadAjustInPoints=4;                                      //Fixed Offset from the Steps in Points.
input int I_StepBack=1;                                                    //Steps back in the array.
input int I_HiddenTakeProfitInPoints=150;                                  //TakeProfit (points)
input int I_LiveSpreadInPoints=40;                                         //Extra spread on top of hidden sl/tp for live trade.
input double I_Volume=0.1;                                                 //Volume/Lots
input int I_Slippage=5;                                                    //Slippage
input bool I_DrawLines=True;                                               //Draw Hidden SL/TP on chart

//+------------------------------------------------------------------+
//| Global Variables                                                 |
//+------------------------------------------------------------------+
C_Trade_SteppedStopLoss *g_objTrade;
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+


int OnInit()
  {

//This system is very similar to the C_Trade object, except it adds a stepped stoploss onto the trade.
//An array of stoploss values is built.   When the spot passes through a stoploss level i in the array (ie array[i])
//the actual hidden stoploss is moved to array[i-I_StepBack].   However a spread adjustment in points is also added/subtracted
//for buy/sell orders, so the hidden stoploss moves to array[i-I_StepBack]-I_SpreadAdjustInPoints*Point().

   Print(__FILE__+" "+__FUNCTION__,"   START-------------------------------------------------");

//Get the actual live spread.
   double dblLiveSpread=I_LiveSpreadInPoints*Point();
   double dblStepSize=I_HiddenStopLossStepSizeInPoints*Point();
   double dblSpreadAdjust=I_SpreadAjustInPoints*Point();

   double dblStopLossSteps[];
   ArrayResize(dblStopLossSteps,I_StopLossSteps);

//Get a suitable spot, sl and tp
   RefreshRates();
   double dblSpot,dblHiddenTakeProfit;
   if(I_BuyOrSell==BUY)
     {
      dblSpot=Ask;
      dblHiddenTakeProfit=dblSpot+I_HiddenTakeProfitInPoints*Point();
      for(int i=0;i<I_StopLossSteps;i++)
        {
         dblStopLossSteps[i]=dblSpot-dblStepSize+(dblStepSize*i);
        }
     }
   else
     {
      dblSpot=Bid;
      dblHiddenTakeProfit=dblSpot-I_HiddenTakeProfitInPoints*Point();
      for(int i=0;i<I_StopLossSteps;i++)
        {
         dblStopLossSteps[i]=dblSpot+dblStepSize-(dblStepSize*i);
        }
     }

//Create a new trade
   g_objTrade=NULL;
   g_objTrade= new C_Trade_SteppedStopLoss(Symbol(),I_BuyOrSell,I_Volume,dblSpot,dblStopLossSteps,I_StepBack,dblSpreadAdjust,dblHiddenTakeProfit,"",0,dblLiveSpread,I_DrawLines,clrOrange);
   if(!g_objTrade.OrderSend(dblSpot,I_Slippage))
     {
      //Trade Entry Error Processing could be added here.

      //Trade has not initiated properly, destroy the class
      delete g_objTrade;
      g_objTrade=NULL;
     }

//Print out the details
   string strPrint=g_objTrade.Description();
   Comment(strPrint);
   Print(__FILE__+" "+__FUNCTION__," ",strPrint);

   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {

//If the EA has been shut down, destroy g_objTrade.
   if(g_objTrade!=NULL)
     {
      //Annoying, I wanted to destroy the trade in the destructor, but it kept making MetaTrader hang.
      if(g_objTrade.TradeIsLive())
        {
         RefreshRates();
         double dblSpot=Bid;
         if(g_objTrade.GetType()==SELL) dblSpot=Ask;
         g_objTrade.OrderClose(dblSpot,I_Slippage,clrRed);

         //Trade Close Error Processing could be added here.
        }
      delete g_objTrade;
     }

   Print(__FILE__,"   END---------------------------------------------------");

  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//Run this with every tick to see if the trade has been closed.
   if(g_objTrade!=NULL)
     {
      if(g_objTrade.TradeIsLive())
        {
         if(g_objTrade.ScanWithEveryTick(I_Slippage))
           {  //Ensure trade is closed.
            delete g_objTrade;
            g_objTrade=NULL;
           }
        }
     }
  }
//+------------------------------------------------------------------+
