//+------------------------------------------------------------------+
//|                                                    LastOrder.mqh |
//|                                                   A.Lopatin 2017 |
//|                                              diver.stv@gmail.com |
//+------------------------------------------------------------------+
#property copyright "A.Lopatin 2017"
#property link      "diver.stv@gmail.com"
#property strict
#property version   "1.01"

//+------------------------------------------------------------------+
//|    get_last_order() returns ticket number of the last order.     |
//|    The function returns -1 value when it havent found last order.|
//|    Arguments: magic - Magic Number ID filtering orders. When 0   |
//|    it is not used.                                               |
//|    type - type order (buy, sell, sell limit, buy limit,          |
//|    sell stop, buy stop) for filtering orders.                    |
//|    mode - pool of orders: MODE-TRADES - trade orders,            |
//|    MODE_HISTORY - closed orders.                                 |
//+------------------------------------------------------------------+

int get_last_order( int magic, int type = -1, int mode = MODE_TRADES )
{
    int     orders_total  = 0, ticket  = -1;
    string  symbol        = Symbol();
    datetime opn_time     = 0, ord_time = 0;
    
    if( mode == MODE_HISTORY )
        orders_total  = OrdersHistoryTotal();
    else
        orders_total  = OrdersTotal(); 
    
    for( int i = 0; i < orders_total; i++ )
    {
        if( OrderSelect( i, SELECT_BY_POS, mode ) )
        {
            if( OrderSymbol() != symbol )
                continue;
            if( OrderType() == type || type == - 1 )
            {      
               if( OrderMagicNumber() == magic || magic == 0 )
               {
                   if( mode == MODE_TRADES )
                     ord_time = OrderOpenTime();
                   else
                     ord_time = OrderCloseTime();
                     
                   if( ord_time > opn_time )
                   {
                       opn_time = ord_time;
                       ticket = OrderTicket();
                   }
               }
            }
        }
        else
        {
         ticket = -1;
         break;
        }
    }//end for
    
    return(ticket);
}

//+------------------------------------------------------------------+
//|    get_first_order() returns ticket number of the first order.   |
//|    The function returns -1 value when it havent found first order|
//|    Arguments: magic - Magic Number ID filtering orders. When 0   |
//|    it is not used.                                               |
//|    type - type order (buy, sell, sell limit, buy limit,          |
//|    sell stop, buy stop) for filtering orders.                    |
//|    mode - pool of orders: MODE-TRADES - trade orders,            |
//|    MODE_HISTORY - closed orders.                                 |
//+------------------------------------------------------------------+
int get_first_order( int magic, int type = -1, int mode = MODE_TRADES )
{
    int     orders_total  = 0, ticket  = -1;
    string  symbol        = Symbol();
    datetime opn_time     = 2*TimeCurrent(), ord_time = 0;
    
    if( mode == MODE_HISTORY )
        orders_total  = OrdersHistoryTotal();
    else
        orders_total  = OrdersTotal(); 
    
    for( int i = 0; i < orders_total; i++ )
    {
        if( OrderSelect( i, SELECT_BY_POS, mode ) )
        {
            if( OrderSymbol() != symbol )
                continue;
            if( OrderType() == type || type == - 1 )
            {   
               if( OrderMagicNumber() == magic || magic == 0 )
               {
                    if( mode == MODE_TRADES )
                     ord_time = OrderOpenTime();
                   else
                     ord_time = OrderCloseTime();
                     
                   if( ord_time < opn_time )
                   {
                       opn_time = ord_time;
                       ticket = OrderTicket();
                   }
               }
            }
        }
        else
        {
         ticket = -1;
         break;
        }
    }//end for
    
    return(ticket);
}


//+------------------------------------------------------------------+
//|   order_lots() returns trade volume by ticket number.            |
//|   Arguments: ticket - ticket number of the order.                |
//|    mode - pool of orders: MODE-TRADES - trade orders,            |
//|    MODE_HISTORY - closed orders.                                 |
//+------------------------------------------------------------------+
double order_lots( int ticket, int mode = MODE_TRADES )
{
    double lots = 0;
    
    if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
        lots = OrderLots();
        
    return(lots);
}

//+----------------------------------------------------------------------+
//| order_comment() returns string comment of the order by ticket number |
//|   Arguments: ticket - ticket number of the order.                    |
//|    mode - pool of orders: MODE-TRADES - trade orders,                |
//|    MODE_HISTORY - closed orders.                                     |
//+----------------------------------------------------------------------+
string order_comment( int ticket, int mode = MODE_TRADES )
{
    string str = "";
    
    if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
        str = OrderComment();
        
    return(str);
}

//+----------------------------------------------------------------------+
//| order_type() returns type of the order by ticket number              |
//|   Arguments: ticket - ticket number of the order.                    |
//|    mode - pool of orders: MODE-TRADES - trade orders,                |
//|    MODE_HISTORY - closed orders.                                     |
//+----------------------------------------------------------------------+
int order_type( int ticket, int mode = MODE_TRADES )
{
    int type = -1;
    
    if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
        type = OrderType();
        
    return(type);
}

//+----------------------------------------------------------------------+
//| order_open_price() returns open price of the order by ticket number. |
//|   Arguments: ticket - ticket number of the order.                    |
//|    mode - pool of orders: MODE-TRADES - trade orders,                |
//|    MODE_HISTORY - closed orders.                                     |
//+----------------------------------------------------------------------+
double order_open_price( int ticket, int mode = MODE_TRADES )
{
    double price = -1;
    
    if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
        price = OrderOpenPrice();
        
    return(price);
}

//+-----------------------------------------------------------------------+
//| order_close_price() returns close price of the order by ticket number.|
//|   Arguments: ticket - ticket number of the order.                     |
//|    mode - pool of orders: MODE-TRADES - trade orders,                 |
//|    MODE_HISTORY - closed orders.                                      |
//+-----------------------------------------------------------------------+
double order_close_price( int ticket, int mode = MODE_TRADES )
{
    double price = -1;
    
    if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
        price = OrderClosePrice();
        
    return(price);
}

//+-----------------------------------------------------------------------+
//| order_open_time() returns open time of the order by ticket number.    |
//|   Arguments: ticket - ticket number of the order.                     |
//|    mode - pool of orders: MODE-TRADES - trade orders,                 |
//|    MODE_HISTORY - closed orders.                                      |
//+-----------------------------------------------------------------------+
datetime order_open_time( int ticket, int mode = MODE_TRADES )
{
    datetime time = 0;
    
    if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
        time = OrderOpenTime();
        
    return(time);
}

//+-----------------------------------------------------------------------+
//| order_close_time() returns close time of the order by ticket number.  |
//|   Arguments: ticket - ticket number of the order.                     |
//|    mode - pool of orders: MODE-TRADES - trade orders,                 |
//|    MODE_HISTORY - closed orders.                                      |
//+-----------------------------------------------------------------------+
datetime order_close_time( int ticket, int mode = MODE_TRADES )
{
    datetime time = 0;
    
    if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
        time = OrderCloseTime();
        
    return(time);
}

// --- Triggers of the closing the orders
#define CLOSE_BY_TP        1   // by takeprofit    
#define CLOSE_BY_SL        -1  // by stoploss
#define CLOSE_BY_MANUAL    0   // manual

//+--------------------------------------------------------------------------+
//| close_by() returns the trigger of closing the order:by takeprofit (1),   |
//| by stoploss (-1), manual(0).It is defined based on comment of the order. |
//| Arguments: ticket - ticket number of the order.                          |
//|            mode - pool of orders: MODE-TRADES - trade orders,            |
//|            MODE_HISTORY - closed orders.                                 |
//+--------------------------------------------------------------------------+
int close_by( int ticket, int mode = MODE_TRADES )
{
      int retn = CLOSE_BY_MANUAL;
      
      if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
      {
            
        if( -1 != StringFind( OrderComment(), "[sl]", 0 ) ) 
        {
           retn = CLOSE_BY_SL;
        }
        else if( -1 != StringFind( OrderComment(), "[tp]", 0 ) ) 
        {
          retn = CLOSE_BY_TP;
        }
      }
      
      return(retn);
}

//+-------------------------------------------------------------------------+
//| close_by2() returns the trigger of closing the order:by takeprofit (1), |
//| by stoploss (-1), manual(0).                                            |
//| It is defined based on the close price and open price of the order.     |
//| Arguments: ticket - ticket number of the order.                         |
//|            mode - pool of orders: MODE-TRADES - trade orders,           |
//|            MODE_HISTORY - closed orders.                                |
//+-------------------------------------------------------------------------+
int close_by2( int ticket, int mode = MODE_TRADES )
{
      int retn = CLOSE_BY_MANUAL;
      
      if( ticket > 0 && OrderSelect( ticket, SELECT_BY_TICKET, mode ) )
      {
        if( MathAbs(OrderClosePrice() - OrderStopLoss() ) <= 1*Point )
            retn = CLOSE_BY_SL;
        else if( MathAbs(OrderClosePrice() - OrderTakeProfit() ) <= 1*Point )
            retn = CLOSE_BY_TP;
      }
      
      return(retn);
}

//+-----------------------------------------------------------------------------+
//| orders_count() returns the count of the opened orders.                      |
//| Arguments: magic - Magic Number ID filtering orders. When 0 it is not used. |
//| type - orders type (buy, sell, sell limit, buy limit, sell stop, buy stop)  |
//| for filtering orders. When -1 is not used.                                  |
//| comment - comment string of the orders. When "" (empty) is not used.        |
//+-----------------------------------------------------------------------------+
int orders_count( int magic, int type = -1, string comment = "" )
{
    int orders_total = OrdersTotal(), count = 0;
    
    for( int i = 0; i < orders_total; i++ )
    {
        if( OrderSelect( i, SELECT_BY_POS, MODE_TRADES ) )
        {
            if( OrderSymbol() != Symbol() )
                continue;
            if( (OrderMagicNumber() == magic || magic == 0)
                 && (OrderType() == type || type == -1) )
            {
               if( comment == "" || OrderComment() == comment ) 
                  count++;
            } 
        }
    }
    
    return(count);
}

