library smtp_mail;

// real author: avoitenko
// https://login.mql5.com/en/users/avoitenko

uses
  System.SysUtils,
  System.Classes,
  IdExplicitTLSClientServerBase,
  IdSMTP,
  IdText,
  IdMessage,
  IdIOHandler,
  IdSSLOpenSSL,
  IdStack,
  IdReplySMTP,
  IdAttachmentFile,
  IdExceptionCore;

const
  MAIL_NO_CONNECTION=-2;
  MAIL_NO_INITIALIZATION=-1;
  MAIL_NO_ERROR=0;
  MAIL_TIMEOUT=1;
  MAIL_WRONG_LOGIN=2;
  MAIL_WRONG_HOST_NAME=3;
  MAIL_HOST_NOT_FOUND=4;

//+------------------------------------------------------------------+
//| MailConnect                                                      |
//+------------------------------------------------------------------+
function MailConnect(var mail: PInt64; const host:PWideChar; port:Integer;user,password:PWideChar; timeout:Integer=5000):Integer;stdcall;
var smtp:TIdSMTP;
begin

  result:=MAIL_NO_ERROR;

  //--- create smtp
  if(mail=nil)then
  begin
    smtp:=TIdSMTP.Create(nil);
    mail:=Pointer(smtp);
  end
  else
    smtp:=Pointer(mail);

//--- initialization
  Smtp.HeloName:='MetaTrader';
  Smtp.MailAgent:='MetaTrader Mail Agent';
  Smtp.Host:=StrPas(host);
  Smtp.Port:=port;

  Smtp.Username:=StrPas(user);
  Smtp.Password:=StrPas(password);
  Smtp.AuthType:=satDefault;
  Smtp.ConnectTimeout:=timeout;

//--- connection
  Smtp.IOHandler:=TIdSSLIOHandlerSocketOpenSSL.Create(SMTP);
  Smtp.IOHandler.Host:=host;
  Smtp.IOHandler.Port:=port;
  Smtp.IOHandler.Destination:=Smtp.Host + ':' + IntToStr(port);
  Smtp.UseTLS := utUseImplicitTLS;


  try
    smtp.Connect;
    smtp.Authenticate;
  except
//  on E: EIdException do
//  Showmessage(E.ClassName);
//  showmessage(E.Message);

    on E: EidSocketError do
      result:=MAIL_HOST_NOT_FOUND;//invalid host name

    on E:EIdHostRequired do
      result:=MAIL_WRONG_HOST_NAME;//need host name

    on E: EIdSMTPReplyError do
      result:=MAIL_WRONG_LOGIN;//invalid login/password

    on E: EIdConnectTimeout do
      result:=MAIL_TIMEOUT;//timeout

    on E: EIdSocketError do
      result:=E.LastError;//socket error

  end;
end;

//+------------------------------------------------------------------+
//| MailSendText                                                     |
//+------------------------------------------------------------------+
function MailSendText(var mail: PInt64; const mailto,from,subject,text,files:PWideChar): Integer; stdcall;
var I: Integer;
    msg: TIdMessage;
    smtp: TIdSMTP;
    txt: TIdText;
    list: TStringList;
    attach: TIdAttachmentFile;
begin

  result:=MAIL_NO_ERROR;

  //--- check pointer
  if(mail=nil) then
  begin
    result:=MAIL_NO_INITIALIZATION;
    Exit;
  end;

  smtp:=Pointer(mail);

//--- check connection
  if not smtp.Connected then
  begin
    result:=MAIL_NO_CONNECTION;
    Exit;
  end;

  msg:=TIdMessage.Create(smtp);
  try
//--- initialization
    msg.ContentType:='multipart/mixed';
    msg.CharSet:='UTF-8';
    msg.Subject:=StrPas(subject);
    msg.From.Address:=smtp.Username;
    msg.From.Text:=smtp.Username;
    msg.From.Name:=StrPas(from);
    msg.Recipients.EMailAddresses:=StrPas(mailto);

//--- text
    txt:=TIdText.Create(msg.MessageParts);
    txt.ContentType:='text/plain';
    txt.CharSet:='UTF-8';
    txt.Body.Text:=StrPas(text);

//--- attachments
    list:=TStringList.Create;
    try
      ExtractStrings([';'],[],files,list);

      for I:=0 to list.Count-1 do
      begin
        //--- string trim
        list[I]:=Trim(list[I]);
        if FileExists(list[I]) then
        begin
          attach:=TIdAttachmentFile.Create(msg.MessageParts, list[I]);
          attach.ContentType:= 'application/octet-stream;';
          attach.ContentDisposition:= 'attachment';
          attach.FileName:= ExtractFileName(list[I]);
        end;
      end;
    finally
      list.Free;
    end;

    try
      smtp.Send(msg);
    except

//  on E:EIdException do
//    ShowMessage(E.Message);

    on E:EIdSocketError do
        result:=E.LastError;
    end;

  finally
    msg.Free;
  end;

end;

//+------------------------------------------------------------------+
//| MailSend                                                         |
//+------------------------------------------------------------------+
function MailSendHtml(var mail: PInt64; const mailto,from,subject,html,files:PWideChar): Integer; stdcall;
var I: Integer;
    msg: TIdMessage;
    smtp: TIdSMTP;
    htmlpart: TIdText;
    list: TStringList;
    attach: TIdAttachmentFile;
begin

  result:=MAIL_NO_ERROR;

//--- check pointer
  if(mail=nil) then
  begin
    result:=MAIL_NO_INITIALIZATION;
    Exit;
  end;

  smtp:=Pointer(mail);

//--- check connection
  if not smtp.Connected then
  begin
    result:=MAIL_NO_CONNECTION;
    Exit;
  end;

  msg:=TIdMessage.Create(smtp);
  try
//--- initialization
    msg.ContentType:='multipart/mixed';
    msg.CharSet:='UTF-8';
    msg.Subject:=StrPas(subject);
    msg.From.Address:=smtp.Username;
    msg.From.Text:=smtp.Username;
    msg.From.Name:=StrPas(from);
    msg.Recipients.EMailAddresses:=StrPas(mailto);

//--- html part
    list:=TStringList.Create;
    try
      list.Add(html);
      htmlpart := TIdText.Create(msg.MessageParts, list);
      htmlpart.ContentType:='text/html';
      htmlpart.CharSet:='UTF-8';
    finally
      list.Free;
    end;

//--- attachments
    list:=TStringList.Create;
    try
      ExtractStrings([';'],[],files,list);

      for I:=0 to list.Count-1 do
      begin
        //--- string trim
        list[I]:=Trim(list[I]);

        if FileExists(list[I]) then
        begin
          attach:=TIdAttachmentFile.Create(msg.MessageParts, list[I]);
          attach.ContentType:= 'application/octet-stream;';
          attach.ContentDisposition:= 'attachment';
          attach.FileName:= ExtractFileName(list[I]);
        end;
      end;
    finally
      list.Free;
    end;

    try
      smtp.Send(msg);
    except

//  on E:EIdException do
//    ShowMessage(E.Message);

    on E:EIdSocketError do
        result:=E.LastError;
    end;

  finally
    msg.Free;
  end;

end;

//+------------------------------------------------------------------+
//| MailSendInlineTextFile                                           |
//+------------------------------------------------------------------+
function MailSendInlineTextFile(var mail: PInt64; const mailto,from,subject,files:PWideChar): Integer; stdcall;
var msg: TIdMessage;
    smtp: TIdSMTP;
    html: TIdText;
    list, list_files: TStringList;
    I: Integer;
begin

  result:=MAIL_NO_ERROR;
//--- check pointer
  if(mail=nil)then
  begin
    result:=MAIL_NO_INITIALIZATION;
    Exit;
  end;
  smtp:=Pointer(mail);

//--- check connection
  if not smtp.Connected then
  begin
    result:=MAIL_NO_CONNECTION;
    Exit;
  end;

  msg:=TIdMessage.Create(smtp);
  try
//--- initialization
    msg.ContentType:='multipart/mixed';
    msg.CharSet:='UTF-8';
    msg.Subject:=StrPas(subject);
    msg.From.Address:=smtp.Username;
    msg.From.Text:=smtp.Username;
    msg.From.Name:=StrPas(from);
    msg.Recipients.EMailAddresses:=StrPas(mailto);

//--- text
    with TIdText.Create(msg.MessageParts, nil) do
      ContentType:='text/plain';

//---
    list_files:=TStringList.Create;
    try
      ExtractStrings([';'],[],files,list_files);

      for  I:=0 to list_files.Count-1 do
      begin
        //--- string trim
        list_files[I]:=Trim(list_files[I]);

        if FileExists(list_files[I]) then
        begin
          list:=TStringList.Create;
          try
            list.LoadFromFile(list_files[I]);
            html := TIdText.Create(msg.MessageParts, list);
            html.CharSet:='UTF-8';
            html.ContentType:='text/plain';
          finally
            list.Free;
          end;
        end;
      end;
    finally
      list_files.Free;
    end;

//--- send
    try
      smtp.Send(msg);
    except

//  on E:EIdException do
//    ShowMessage(E.Message);

    on E:EIdSocketError do
        result:=E.LastError;
    end;

  finally
    msg.Free;
  end;

end;
//+------------------------------------------------------------------+
//| MailSendInlineHtmlFile                                           |
//+------------------------------------------------------------------+
function MailSendInlineHtmlFile(var mail: PInt64; const mailto,from,subject,files:PWideChar): Integer; stdcall;
var msg: TIdMessage;
    smtp: TIdSMTP;
    html: TIdText;
    list,list_files: TStringList;
    I :Integer;
begin

  result:=MAIL_NO_ERROR;

//--- check pointer
  if(mail=nil)then
  begin
    result:=MAIL_NO_INITIALIZATION;
    Exit;
  end;
  smtp:=Pointer(mail);

//--- check connection
  if not smtp.Connected then
  begin
    result:=MAIL_NO_CONNECTION;
    Exit;
  end;

  msg:=TIdMessage.Create(smtp);
  try
//--- initialization
    msg.ContentType:='multipart/mixed';
    msg.CharSet:='UTF-8';
    msg.Subject:=StrPas(subject);
    msg.From.Address:=smtp.Username;
    msg.From.Text:=smtp.Username;
    msg.From.Name:=StrPas(from);
    msg.Recipients.EMailAddresses:=StrPas(mailto);

//--- text
    with TIdText.Create(msg.MessageParts, nil) do
      ContentType:='text/plain';

//--- html part
    list_files:=TStringList.Create;
    try
      ExtractStrings([';'],[],files,list_files);

      for I:=0 to list_files.Count-1 do
      begin
        //--- string trim
        list_files[I]:=Trim(list_files[I]);

        if FileExists(list_files[I]) then
        begin
          list:=TStringList.Create;
          try
            list.LoadFromFile(list_files[I]);
            html := TIdText.Create(msg.MessageParts, list);
            html.CharSet:='UTF-8';
            html.ContentType:='text/html';
          finally
            list.Free;
          end;
        end;
      end;
    finally
      list_files.Free;
    end;

//--- send
    try
      smtp.Send(msg);
    except

//  on E:EIdException do
//    ShowMessage(E.Message);

    on E:EIdSocketError do
        result:=E.LastError;
    end;

  finally
    msg.Free;
  end;

end;

//+------------------------------------------------------------------+
//| MailClose                                                        |
//+------------------------------------------------------------------+
procedure MailClose(var mail:PInt64); stdcall;
var smtp:TIdSMTP;
begin
  if mail <> nil then
  begin
    smtp:=Pointer(mail);

    if smtp.IOHandler <> nil then
      smtp.IOHandler.Free;

    smtp.Free;
    mail:=nil;
  end;
end;

//+------------------------------------------------------------------+
//|   MailErrorDescription                                           |
//+------------------------------------------------------------------+
function MailErrorDescription(const errno:Integer):PWideChar; stdcall;
begin
  result:='';
  case errno of
    //--- my errors
    -0002: result:='No connection';
    -0001: result:='No initialization';
    00000: result:='No error';
    00001: result:='Timeout';
    00002: result:='Wrong login';
    00003: result:='Wrong host name';
    00004: result:='Host not found';

    //--- socket errors
    10004: result:='Interrupted function call';
    10013: result:='Permission denied';
    10014: result:='Bad address';
    10022: result:='Invalid argument';
    10024: result:='Too many open files';
    10035: result:='Resource temporarily unavailable';
    10036: result:='Operation now in progress';
    10037: result:='Operation already in progress';
    10038: result:='Socket operation on non-socket';
    10039: result:='Destination address required';
    10040: result:='Message too long';
    10041: result:='Protocol wrong type for socket';
    10042: result:='Bad protocol option';
    10043: result:='Protocol not supported';
    10044: result:='Socket type not supported';
    10045: result:='Operation not supported';
    10046: result:='Protocol family not supported';
    10047: result:='Address family not supported by protocol family';
    10048: result:='Address already in use';
    10049: result:='Cannot assign requested address';
    10050: result:='Network is down';
    10051: result:='Network is unreachable';
    10052: result:='Network dropped connection on reset';
    10053: result:='Software caused connection abort';
    10054: result:='Connection reset by peer';
    10055: result:='No buffer space available';
    10056: result:='Socket is already connected';
    10057: result:='Socket is not connected';
    10058: result:='Cannot send after socket shutdown';
    10060: result:='Connection timed out';
    10061: result:='Connection refused';
    10064: result:='Host is down';
    10065: result:='No route to host';
    10067: result:='Too many processes';
    10091: result:='Network subsystem is unavailable';
    10092: result:='WINSOCK.DLL version out of range';
    10093: result:='Successful WSAStartup not yet performed';
    10094: result:='Graceful shutdown in progress';
    11001: result:='Host not found';
    11002: result:='Non-authoritative host not found';
    11003: result:='This is a non-recoverable error';
    11004: result:='Valid name, no data record of requested type';
  end;
end;

exports
  MailConnect,
  MailSendText,
  MailSendHtml,
  MailSendInlineTextFile,
  MailSendInlineHtmlFile,
  MailErrorDescription,
  MailClose;
begin
end.
