//+------------------------------------------------------------------+
//|                                                 optimization.mqh |
//|            Copyright 2003-2022 Sergey Bochkanov (ALGLIB project) |
//|                             Copyright 2012-2023, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
//| Implementation of ALGLIB library in MetaQuotes Language 5        |
//|                                                                  |
//| The features of the library include:                             |
//| - Linear algebra (direct algorithms, EVD, SVD)                   |
//| - Solving systems of linear and non-linear equations             |
//| - Interpolation                                                  |
//| - Optimization                                                   |
//| - FFT (Fast Fourier Transform)                                   |
//| - Numerical integration                                          |
//| - Linear and nonlinear least-squares fitting                     |
//| - Ordinary differential equations                                |
//| - Computation of special functions                               |
//| - Descriptive statistics and hypothesis testing                  |
//| - Data analysis - classification, regression                     |
//| - Implementing linear algebra algorithms, interpolation, etc.    |
//|   in high-precision arithmetic (using MPFR)                      |
//|                                                                  |
//| This file is free software; you can redistribute it and/or       |
//| modify it under the terms of the GNU General Public License as   |
//| published by the Free Software Foundation (www.m_fsf.org); either  |
//| version 2 of the License, or (at your option) any later version. |
//|                                                                  |
//| This program is distributed in the hope that it will be useful,  |
//| but WITHOUT ANY WARRANTY; without even the implied warranty of   |
//| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the     |
//| GNU General Public License for more details.                     |
//+------------------------------------------------------------------+
#include "matrix.mqh"
#include "ap.mqh"
#include "alglibinternal.mqh"
#include "linalg.mqh"
//+------------------------------------------------------------------+
//| This structure is used to store OptGuard report, i.e. report on  |
//| the properties of the nonlinear function being optimized with    |
//| ALGLIB.                                                          |
//| After you tell your optimizer to activate OptGuardthis technology|
//| starts to silently monitor function values and gradients /       |
//| Jacobians being passed all around during your optimization       |
//| session. Depending on specific set of checks enabled OptGuard may|
//| perform additional function evaluations (say, about 3*N          |
//| evaluations if you want to check analytic gradient for errors).  |
//| Upon discovering that something strange happens (function values |
//| and/or gradient components change too sharply and/or             |
//| unexpectedly) OptGuard sets one of the "suspicion  flags"        |
//| (without interrupting optimization session).                     |
//| After optimization is done, you can examine OptGuard report.     |
//| Following report fields can be set:                              |
//|      * nonc0suspected                                            |
//|      * nonc1suspected                                            |
//|      * badgradsuspected                                          |
//| === WHAT CAN BE DETECTED WITH OptGuard INTEGRITY CHECKER =====   |
//| Following types of errors in your target function (constraints)  |
//| can be caught:                                                   |
//|   a) discontinuous functions ("non-C0" part of the report)       |
//|   b) functions with discontinuous derivative ("non-C1" part of   |
//|      the report)                                                 |
//|   c) errors in the analytic gradient provided by user            |
//| These types of errors result in optimizer stopping well before   |
//| reaching solution (most often - right after encountering         |
//| discontinuity).                                                  |
//| Type A errors are usually coding errors during  implementation   |
//| of the target function. Most "normal" problems involve continuous|
//| functions, and anyway you can't reliably optimize discontinuous  |
//| function.                                                        |
//| Type B errors are either coding errors or (in case code itself is|
//| correct) evidence of the fact that your problem is an "incorrect"|
//| one. Most optimizers (except for ones provided by MINNS          |
//| subpackage) do not support nonsmooth problems.                   |
//| Type C errors are coding errors which often prevent optimizer    |
//| from making even one step  or result in optimizing stopping too  |
//| early, as soon as actual descent direction becomes too different |
//| from one suggested by user-supplied gradient.                    |
//| === WHAT IS REPORTED =========================================== |
//| Following set of report fields deals with discontinuous target   |
//| functions, ones not belonging to C0 continuity class:            |
//|   * nonc0suspected - is a flag which is set upon discovering some|
//|     indication of the discontinuity. If this flag is false, the  |
//|     rest of "non-C0" fields should be ignored                    |
//|   * nonc0fidx - is an index of the function (0 for target        |
//|     function, 1 or higher for nonlinear constraints) which is    |
//|     suspected of being "non-C0"                                  |
//|   * nonc0lipshitzc - a Lipchitz constant for a function which was|
//|     suspected of being non-continuous.                           |
//|   * nonc0test0positive - set to indicate specific test which     |
//|     detected continuity violation (test #0)                      |
//| Following set of report fields deals with discontinuous gradient/|
//| Jacobian, i.e. with functions violating C1 continuity:           |
//|   * nonc1suspected - is a flag which is set upon discovering some|
//|     indication of the discontinuity. If this flag is false, the  |
//|     rest of "non-C1" fields should be ignored                    |
//|   * nonc1fidx - is an index of the function (0 for target        |
//|     function, 1 or higher for nonlinear constraints) which is    |
//|     suspected of being "non-C1"                                  |
//|   * nonc1lipshitzc - a Lipchitz constant for a function gradient |
//|     which was suspected of being non-smooth.                     |
//|   * nonc1test0positive - set to indicate specific test which     |
//|     detected continuity violation (test #0)                      |
//|   * nonc1test1positive - set to indicate specific test which     |
//|     detected continuity violation (test #1)                      |
//| Following set of report fields deals with errors in the gradient:|
//|   * badgradsuspected - is a flad which is set upon discovering an|
//|     error in the analytic gradient supplied by user              |
//|   * badgradfidx - index  of the function with bad gradient (0 for|
//|     target function, 1 or higher for nonlinear constraints)      |
//|   * badgradvidx - index of the variable                          |
//|   * badgradxbase - location where Jacobian is tested             |
//|   * following  matrices  store  user-supplied Jacobian and its   |
//|     numerical differentiation version (which is assumed to be    |
//|     free from the coding errors), both of them computed near the |
//|     initial point:                                               |
//|         * badgraduser, an array[K,N], analytic Jacobian supplied |
//|           by user                                                |
//|         * badgradnum,  an array[K,N], numeric  Jacobian computed |
//|           by ALGLIB                                              |
//| Here K is a total number of nonlinear functions (target  +       |
//| nonlinear constraints), N is a variable number.                  |
//| The element of badgraduser[] with index [badgradfidx,badgradvidx]|
//| is assumed to be wrong.                                          |
//| More detailed error log can be obtained from optimizer by        |
//| explicitly requesting reports for tests C0.0, C1.0, C1.1.        |
//+------------------------------------------------------------------+
struct COptGuardReport
  {
   bool              m_nonc0suspected;
   bool              m_nonc0test0positive;
   int               m_nonc0fidx;
   double            m_nonc0lipschitzc;
   bool              m_nonc1suspected;
   bool              m_nonc1test0positive;
   bool              m_nonc1test1positive;
   int               m_nonc1fidx;
   double            m_nonc1lipschitzc;
   bool              m_badgradsuspected;
   int               m_badgradfidx;
   int               m_badgradvidx;
   CRowDouble        m_badgradxbase;
   CMatrixDouble     m_badgraduser;
   CMatrixDouble     m_badgradnum;
   //--- constructor / destructor
                     COptGuardReport(void);
                    ~COptGuardReport(void) {}
   //---
   void              Copy(const COptGuardReport &obj);
   //--- overloading
   void              operator=(const COptGuardReport &obj) { Copy(obj); }
  };

//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
COptGuardReport::COptGuardReport(void)
  {
   m_nonc0suspected=false;
   m_nonc0test0positive=false;
   m_nonc0fidx=0;
   m_nonc0lipschitzc=0;
   m_nonc1suspected=false;
   m_nonc1test0positive=false;
   m_nonc1test1positive=false;
   m_nonc1fidx=0;
   m_nonc1lipschitzc=0;
   m_badgradsuspected=false;
   m_badgradfidx=0;
   m_badgradvidx=0;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void COptGuardReport::Copy(const COptGuardReport &obj)
  {
   m_nonc0suspected=obj.m_nonc0suspected;
   m_nonc0test0positive=obj.m_nonc0test0positive;
   m_nonc0fidx=obj.m_nonc0fidx;
   m_nonc0lipschitzc=obj.m_nonc0lipschitzc;
   m_nonc1suspected=obj.m_nonc1suspected;
   m_nonc1test0positive=obj.m_nonc1test0positive;
   m_nonc1test1positive=obj.m_nonc1test1positive;
   m_nonc1fidx=obj.m_nonc1fidx;
   m_nonc1lipschitzc=obj.m_nonc1lipschitzc;
   m_badgradsuspected=obj.m_badgradsuspected;
   m_badgradfidx=obj.m_badgradfidx;
   m_badgradvidx=obj.m_badgradvidx;
   m_badgradxbase=obj.m_badgradxbase;
   m_badgraduser=obj.m_badgraduser;
   m_badgradnum=obj.m_badgradnum;
  }
//+------------------------------------------------------------------+
//| This structure is used for detailed reporting about suspected C0 |
//| continuity violation.                                            |
//| === WHAT IS TESTED ============================================= |
//| C0 test  studies  function  values (not gradient!)  obtained     |
//| during line searches and monitors estimate of the Lipschitz      |
//| constant. Sudden spikes usually indicate that discontinuity was  |
//| detected.                                                        |
//| === WHAT IS REPORTED =========================================== |
//| Actually, report retrieval function returns TWO report           |
//| structures:                                                      |
//|   * one for most suspicious point found so far (one with highest |
//|     change in the function value), so called "strongest" report  |
//|   * another one for most detailed line search (more function     |
//|     evaluations = easier to understand what's going on) which    |
//|     triggered  test #0 criteria, so called "longest" report      |
//| In both cases following fields are returned:                     |
//|   * positive - is TRUE  when test flagged suspicious point;      |
//|     FALSE if test did not notice anything (in the latter cases   |
//|     fields below are empty).                                     |
//|   * fidx - is an index of the function (0 for target function, 1 |
//|     or higher for nonlinear constraints) which is suspected of   |
//|     being "non-C1"                                               |
//|   * x0[], d[] - arrays of length N which store initial point and |
//|     direction for line search (d[] can be normalized, but does   |
//|     not have to)                                                 |
//|   * stp[], f[] - arrays of length CNT which store step lengths   |
//|     and function values at these points; f[i] is evaluated in    |
//|     x0+stp[i]*d.                                                 |
//|   * stpidxa, stpidxb - we suspect that function violates C1      |
//|     continuity between steps #stpidxa and #stpidxb (usually we   |
//|     have  stpidxb=stpidxa+3, with most likely position of the    |
//|     violation between stpidxa+1 and stpidxa+2.                   |
//| You can plot function values stored in stp[] and f[] arrays and  |
//| study behavior of your function by your own eyes, just to be sure|
//| that test correctly reported C1 violation.                       |
//+------------------------------------------------------------------+
struct COptGuardNonC0Report
  {
   int               m_stpidxb;
   int               m_cnt;
   int               m_stpidxa;
   int               m_n;
   int               m_fidx;
   bool              m_positive;
   CRowDouble        m_d;
   CRowDouble        m_x0;
   CRowDouble        m_stp;
   CRowDouble        m_f;
   //--- constructor / destructor
                     COptGuardNonC0Report(void);
                    ~COptGuardNonC0Report(void) {}
   //---
   void              Copy(const COptGuardNonC0Report &obj);
   //--- overloading
   void              operator=(const COptGuardNonC0Report &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
COptGuardNonC0Report::COptGuardNonC0Report(void)
  {
   m_stpidxb=0;
   m_cnt=0;
   m_stpidxa=0;
   m_n=0;
   m_fidx=0;
   m_positive=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void COptGuardNonC0Report::Copy(const COptGuardNonC0Report &obj)
  {
   m_stpidxb=obj.m_stpidxb;
   m_cnt=obj.m_cnt;
   m_stpidxa=obj.m_stpidxa;
   m_n=obj.m_n;
   m_fidx=obj.m_fidx;
   m_positive=obj.m_positive;
   m_d=obj.m_d;
   m_x0=obj.m_x0;
   m_stp=obj.m_stp;
   m_f=obj.m_f;
  }
//+------------------------------------------------------------------+
//| This structure is used for detailed reporting about suspected C1 |
//| continuity violation as flagged by C1 test #0 (OptGuard  has     |
//| several tests for C1 continuity, this report is used by #0).     |
//| === WHAT IS TESTED ============================================= |
//| C1 test #0 studies function values (not gradient!) obtained      |
//| during line searches and monitors behavior of directional        |
//| derivative estimate. This test is less powerful than test #1, but|
//| it does not depend on gradient values and thus it is more robust |
//| against artifacts introduced by numerical differentiation.       |
//| === WHAT IS REPORTED =========================================== |
//| Actually, report retrieval function returns TWO report           |
//| structures:                                                      |
//|   * one for most suspicious point found so far (one with highest |
//|     change in the directional derivative), so called "strongest" |
//|     report                                                       |
//|   * another one for most detailed line search (more function     |
//|     evaluations = easier to understand what's going on) which    |
//|     triggered test #0 criteria, so called "longest" report       |
//| In both cases following fields are returned:                     |
//|   * positive - is TRUE  when test flagged suspicious point;      |
//|     FALSE if test did not notice anything (in the latter cases   |
//|     fields below are empty).                                     |
//|   * fidx - is an index of the function (0 for  target  function, |
//|     1 or higher for nonlinear constraints) which is suspected of |
//|     being "non-C1"                                               |
//|   * x0[], d[] - arrays of length N which store initial point and |
//|     direction for line search (d[] can be normalized, but does   |
//|     not have to)                                                 |
//|   * stp[], f[] - arrays of length CNT which store step lengths   |
//|     and function values at these points; f[i] is evaluated in    |
//|     x0+stp[i]*d.                                                 |
//|   * stpidxa, stpidxb - we  suspect  that  function  violates  C1 |
//|     continuity between steps #stpidxa and #stpidxb (usually we   |
//|     have stpidxb=stpidxa+3, with most likely position  of  the   |
//|     violation  between  stpidxa+1 and stpidxa+2.                 |
//| You can plot function values stored in stp[]  and  f[]  arrays   |
//| and study behavior of your function by your own eyes, just  to   |
//| be sure that test correctly reported C1 violation.               |
//+------------------------------------------------------------------+
struct COptGuardNonC1Test0Report
  {
   int               m_cnt;
   int               m_fidx;
   int               m_n;
   int               m_stpidxa;
   int               m_stpidxb;
   bool              m_positive;
   CRowDouble        m_d;
   CRowDouble        m_f;
   CRowDouble        m_stp;
   CRowDouble        m_x0;
   //--- constructor / destructor
                     COptGuardNonC1Test0Report(void);
                    ~COptGuardNonC1Test0Report(void) {}
   //---
   void              Copy(const COptGuardNonC1Test0Report &obj);
   //--- overloading
   void              operator=(const COptGuardNonC1Test0Report &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
COptGuardNonC1Test0Report::COptGuardNonC1Test0Report(void)
  {
   m_cnt=0;
   m_fidx=0;
   m_n=0;
   m_stpidxa=0;
   m_stpidxb=0;
   m_positive=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void COptGuardNonC1Test0Report::Copy(const COptGuardNonC1Test0Report &obj)
  {
   m_cnt=obj.m_cnt;
   m_fidx=obj.m_fidx;
   m_n=obj.m_n;
   m_stpidxa=obj.m_stpidxa;
   m_stpidxb=obj.m_stpidxb;
   m_positive=obj.m_positive;
   m_d=obj.m_d;
   m_f=obj.m_f;
   m_stp=obj.m_stp;
   m_x0=obj.m_x0;
  }
//+------------------------------------------------------------------+
//| This structure is used for detailed reporting about suspected C1 |
//| continuity violation as flagged by C1 test #1 (OptGuard  has     |
//| several tests for C1 continuity, this report is used by #1).     |
//| === WHAT IS TESTED ============================================= |
//| C1 test #1 studies individual  components  of  the  gradient  as |
//| recorded during line searches. Upon discovering discontinuity in |
//| the gradient  this test records specific component which was     |
//| suspected (or  one  with  highest indication of discontinuity if |
//| multiple components are suspected).                              |
//| When precise analytic gradient is provided this test is more     |
//| powerful than test #0  which  works  with  function  values  and |
//| ignores  user-provided gradient.  However,  test  #0  becomes    |
//| more powerful when numerical differentiation is employed (in such|
//| cases test #1 detects  higher  levels of numerical noise and     |
//| becomes too conservative).                                       |
//| This test also tells specific components of the gradient which   |
//| violate C1 continuity, which makes it more informative than #0,  |
//| which just tells that continuity is violated.                    |
//| === WHAT IS REPORTED =========================================== |
//| Actually, report retrieval function returns TWO report           |
//| structures:                                                      |
//|   * one for most suspicious point found so far (one with highest |
//|     change in the directional derivative), so called "strongest" |
//|     report                                                       |
//|   * another one for most detailed line search (more function     |
//|     evaluations = easier to understand what's going on) which    |
//|     triggered test #1 criteria, so called "longest" report       |
//| In both cases following fields are returned:                     |
//|   * positive - is TRUE  when test flagged suspicious point;      |
//|     FALSE if test did not notice anything (in the latter cases   |
//|     fields below are empty).                                     |
//|   * fidx - is an index of the function (0 for  target  function, |
//|     1 or higher for nonlinear constraints) which is suspected of |
//|     being "non-C1"                                               |
//|   * vidx - is an index of the variable in [0,N) with nonsmooth   |
//|     derivative                                                   |
//|   * x0[], d[] - arrays of length N which store initial point and |
//|     direction for line search (d[] can be normalized, but does   |
//|     not have to)                                                 |
//|   * stp[], g[] - arrays of length CNT which store step lengths   |
//|     and gradient values at these points; g[i] is evaluated in    |
//|     x0+stp[i]*d and contains vidx-th component of the gradient.  |
//|   * stpidxa, stpidxb - we  suspect  that  function  violates  C1 |
//|     continuity between steps #stpidxa and #stpidxb (usually we   |
//|     have stpidxb=stpidxa+3, with most likely  position  of  the  |
//|     violation  between  stpidxa+1 and stpidxa+2.                 |
//| You can plot function values stored in stp[]  and  g[]  arrays   |
//| and study behavior of your function by your own eyes, just to be |
//| sure that test correctly reported C1 violation.                  |
//+------------------------------------------------------------------+
struct COptGuardNonC1Test1Report
  {
   int               m_cnt;
   int               m_fidx;
   int               m_n;
   int               m_stpidxa;
   int               m_stpidxb;
   int               m_vidx;
   bool              m_positive;
   CRowDouble        m_d;
   CRowDouble        m_g;
   CRowDouble        m_stp;
   CRowDouble        m_x0;
   //--- constructor / destructor
                     COptGuardNonC1Test1Report(void);
                    ~COptGuardNonC1Test1Report(void) {}
   //---
   void              Copy(const COptGuardNonC1Test1Report &obj);
   //--- overloading
   void              operator=(const COptGuardNonC1Test1Report &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
COptGuardNonC1Test1Report::COptGuardNonC1Test1Report(void)
  {
   m_cnt=0;
   m_fidx=0;
   m_n=0;
   m_stpidxa=0;
   m_stpidxb=0;
   m_vidx=0;
   m_positive=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void COptGuardNonC1Test1Report::Copy(const COptGuardNonC1Test1Report &obj)
  {
   m_cnt=obj.m_cnt;
   m_fidx=obj.m_fidx;
   m_n=obj.m_n;
   m_stpidxa=obj.m_stpidxa;
   m_stpidxb=obj.m_stpidxb;
   m_vidx=obj.m_vidx;
   m_positive=obj.m_positive;
   m_d=obj.m_d;
   m_g=obj.m_g;
   m_stp=obj.m_stp;
   m_x0=obj.m_x0;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class COptGuardApi
  {
public:
   static void       OptGuardInitInternal(COptGuardReport &rep,int n,int k);
   static void       OptGuardExportReport(COptGuardReport &srcrep,int n,int k,bool badgradhasxj,COptGuardReport &dstrep);
   static void       SmoothnessMonitorExportC1Test0Report(COptGuardNonC1Test0Report &srcrep,CRowDouble &s,COptGuardNonC1Test0Report &dstrep);
   static void       SmoothnessMonitorExportC1Test1Report(COptGuardNonC1Test1Report &srcrep,CRowDouble &s,COptGuardNonC1Test1Report &dstrep);
   static bool       OptGuardAllClear(COptGuardReport &rep);

  };
//+------------------------------------------------------------------+
//| This subroutine initializes "internal" OptGuard report,  i.e. one|
//| intended for internal use by optimizers.                         |
//+------------------------------------------------------------------+
void COptGuardApi::OptGuardInitInternal(COptGuardReport &rep,
                                        int n,
                                        int k)
  {
   rep.m_nonc0suspected=false;
   rep.m_nonc0test0positive=false;
   rep.m_nonc0lipschitzc=0;
   rep.m_nonc0fidx=-1;
   rep.m_nonc1suspected=false;
   rep.m_nonc1test0positive=false;
   rep.m_nonc1test1positive=false;
   rep.m_nonc1lipschitzc=0;
   rep.m_nonc1fidx=-1;
   rep.m_badgradsuspected=false;
   rep.m_badgradfidx=-1;
   rep.m_badgradvidx=-1;
  }
//+------------------------------------------------------------------+
//| This subroutine exports report to user-readable representation   |
//| (all arrays are forced to have exactly same size as needed;      |
//| unused arrays are set to zero length).                           |
//+------------------------------------------------------------------+
void COptGuardApi::OptGuardExportReport(COptGuardReport &srcrep,
                                        int n,
                                        int k,
                                        bool badgradhasxj,
                                        COptGuardReport &dstrep)
  {
//--- copy
   dstrep.m_nonc0suspected=srcrep.m_nonc0suspected;
   dstrep.m_nonc0test0positive=srcrep.m_nonc0test0positive;
   if(srcrep.m_nonc0suspected)
     {
      dstrep.m_nonc0lipschitzc=srcrep.m_nonc0lipschitzc;
      dstrep.m_nonc0fidx=srcrep.m_nonc0fidx;
     }
   else
     {
      dstrep.m_nonc0lipschitzc=0;
      dstrep.m_nonc0fidx=-1;
     }
   dstrep.m_nonc1suspected=srcrep.m_nonc1suspected;
   dstrep.m_nonc1test0positive=srcrep.m_nonc1test0positive;
   dstrep.m_nonc1test1positive=srcrep.m_nonc1test1positive;
   if(srcrep.m_nonc1suspected)
     {
      dstrep.m_nonc1lipschitzc=srcrep.m_nonc1lipschitzc;
      dstrep.m_nonc1fidx=srcrep.m_nonc1fidx;
     }
   else
     {
      dstrep.m_nonc1lipschitzc=0;
      dstrep.m_nonc1fidx=-1;
     }
   dstrep.m_badgradsuspected=srcrep.m_badgradsuspected;
   if(srcrep.m_badgradsuspected)
     {
      dstrep.m_badgradfidx=srcrep.m_badgradfidx;
      dstrep.m_badgradvidx=srcrep.m_badgradvidx;
     }
   else
     {
      dstrep.m_badgradfidx=-1;
      dstrep.m_badgradvidx=-1;
     }
   if(badgradhasxj)
     {
      dstrep.m_badgradxbase=srcrep.m_badgradxbase;
      dstrep.m_badgraduser=srcrep.m_badgraduser;
      dstrep.m_badgradnum=srcrep.m_badgradnum;
     }
   else
     {
      dstrep.m_badgradxbase.Resize(0);
      dstrep.m_badgraduser.Resize(0,0);
      dstrep.m_badgradnum.Resize(0,0);
     }
  }
//+------------------------------------------------------------------+
//| This subroutine exports report to user-readable representation   |
//| (all arrays are forced to have exactly same size as needed;      |
//| unused arrays are set to zero length).                           |
//| NOTE: we assume that SrcRep contains scaled X0[] and  D[],  i.e. |
//| explicit variable scaling was applied. We need to rescale them   |
//| during export, that's why we need S[] parameter.                 |
//+------------------------------------------------------------------+
void COptGuardApi::SmoothnessMonitorExportC1Test0Report(COptGuardNonC1Test0Report &srcrep,
                                                        CRowDouble &s,
                                                        COptGuardNonC1Test0Report &dstrep)
  {
   dstrep.m_positive=srcrep.m_positive;
   if(srcrep.m_positive)
     {
      dstrep.m_stpidxa=srcrep.m_stpidxa;
      dstrep.m_stpidxb=srcrep.m_stpidxb;
      dstrep.m_fidx=srcrep.m_fidx;
      dstrep.m_cnt=srcrep.m_cnt;
      dstrep.m_n=srcrep.m_n;
      dstrep.m_x0=srcrep.m_x0*s+0;
      dstrep.m_d=srcrep.m_d*s+0;
      dstrep.m_stp=srcrep.m_stp;
      dstrep.m_f=srcrep.m_f;
     }
   else
     {
      dstrep.m_stpidxa=-1;
      dstrep.m_stpidxb=-1;
      dstrep.m_fidx=-1;
      dstrep.m_cnt=0;
      dstrep.m_n=0;
      dstrep.m_x0.Resize(0);
      dstrep.m_d.Resize(0);
      dstrep.m_stp.Resize(0);
      dstrep.m_f.Resize(0);
     }
  }
//+------------------------------------------------------------------+
//| This subroutine exports report to user-readable representation   |
//| (all arrays are forced to have exactly same size as needed;      |
//| unused arrays are set to zero length).                           |
//| NOTE: we assume that SrcRep contains scaled X0[], D[] and G[],   |
//|       i.e. explicit variable scaling was applied. We need to     |
//|       rescale them during export, that's why we need S[]         |
//|       parameter.                                                 |
//+------------------------------------------------------------------+
void COptGuardApi::SmoothnessMonitorExportC1Test1Report(COptGuardNonC1Test1Report &srcrep,
                                                        CRowDouble &s,
                                                        COptGuardNonC1Test1Report &dstrep)
  {
   dstrep.m_positive=srcrep.m_positive;
   if(srcrep.m_positive)
     {
      //--- check
      if(!CAp::Assert(srcrep.m_vidx>=0 && srcrep.m_vidx<srcrep.m_n,__FUNCTION__+": integrity check failed"))
         return;
      dstrep.m_stpidxa=srcrep.m_stpidxa;
      dstrep.m_stpidxb=srcrep.m_stpidxb;
      dstrep.m_fidx=srcrep.m_fidx;
      dstrep.m_vidx=srcrep.m_vidx;
      dstrep.m_cnt=srcrep.m_cnt;
      dstrep.m_n=srcrep.m_n;
      dstrep.m_x0=srcrep.m_x0*s+0;
      dstrep.m_d=srcrep.m_d*s+0;
      dstrep.m_stp=srcrep.m_stp;
      dstrep.m_g=srcrep.m_g/s[srcrep.m_vidx]+0;
      dstrep.m_g.Resize(srcrep.m_cnt);
     }
   else
     {
      dstrep.m_stpidxa=-1;
      dstrep.m_stpidxb=-1;
      dstrep.m_fidx=-1;
      dstrep.m_vidx=-1;
      dstrep.m_cnt=0;
      dstrep.m_n=0;
      dstrep.m_x0.Resize(0);
      dstrep.m_d.Resize(0);
      dstrep.m_stp.Resize(0);
      dstrep.m_g.Resize(0);
     }
  }
//+------------------------------------------------------------------+
//| Returns True when all flags are clear. Intended for easy  coding |
//| of  unit tests.                                                  |
//+------------------------------------------------------------------+
bool COptGuardApi::OptGuardAllClear(COptGuardReport &rep)
  {
   return (!(rep.m_badgradsuspected || rep.m_nonc0suspected || rep.m_nonc1suspected));
  }
//+------------------------------------------------------------------+
//| This structure is used to store temporary buffers for            |
//| L-BFGS-based preconditioner.                                     |
//+------------------------------------------------------------------+
struct CPrecBufLBFGS
  {
   CRowInt           m_bufb;
   CRowInt           m_idx;
   CRowDouble        m_alpha;
   CRowDouble        m_bufa;
   CRowDouble        m_norms;
   CRowDouble        m_rho;
   CMatrixDouble     m_yk;
   CRowDouble        norms;
   //--- constructor / destructor
                     CPrecBufLBFGS(void) {}
                    ~CPrecBufLBFGS(void) {}
   void              Copy(const CPrecBufLBFGS &obj);
   //--- overloading
   void              operator=(const CPrecBufLBFGS &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CPrecBufLBFGS::Copy(const CPrecBufLBFGS &obj)
  {
   m_bufb=obj.m_bufb;
   m_idx=obj.m_idx;
   m_alpha=obj.m_alpha;
   m_bufa=obj.m_bufa;
   m_norms=obj.m_norms;
   m_rho=obj.m_rho;
   m_yk=obj.m_yk;
  }
//+------------------------------------------------------------------+
//| This structure is used to store temporary buffers for LowRank    |
//| preconditioner.                                                  |
//+------------------------------------------------------------------+
struct CPrecBufLowRank
  {
   int               m_k;
   int               m_n;
   CRowDouble        m_bufc;
   CRowDouble        m_d;
   CRowDouble        m_tmp;
   CMatrixDouble     m_bufw;
   CMatrixDouble     m_bufz;
   CMatrixDouble     m_v;
                     CPrecBufLowRank(void) { m_k=0; m_n=0; }
                    ~CPrecBufLowRank(void) {}
   void              Copy(const CPrecBufLowRank &obj);
   //--- overloading
   void              operator=(const CPrecBufLowRank &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CPrecBufLowRank::Copy(const CPrecBufLowRank &obj)
  {
   m_k=obj.m_k;
   m_n=obj.m_n;
   m_bufc=obj.m_bufc;
   m_d=obj.m_d;
   m_tmp=obj.m_tmp;
   m_bufw=obj.m_bufw;
   m_bufz=obj.m_bufz;
   m_v=obj.m_v;
  }
//+------------------------------------------------------------------+
//| This structure is a smoothness monitor.                          |
//+------------------------------------------------------------------+
struct CSmoothnessMonitor
  {
   RCommState        m_probingrcomm;
   RCommState        m_rstateg0;
   COptGuardReport   m_rep;
   COptGuardNonC1Test1Report m_nonc1test1lngrep;
   COptGuardNonC1Test1Report m_nonc1test1strrep;
   COptGuardNonC1Test0Report m_nonc1test0lngrep;
   COptGuardNonC1Test0Report m_nonc1test0strrep;
   COptGuardNonC0Report m_nonc0lngrep;
   COptGuardNonC0Report m_nonc0strrep;
   int               m_enqueuedcnt;
   int               m_k;
   int               m_n;
   int               m_probingnstepsstored;
   int               m_probingnvalues;
   int               m_sortedcnt;
   double            m_nonc0currentrating;
   double            m_nonc0lngrating;
   double            m_nonc0strrating;
   double            m_nonc1currentrating;
   double            m_nonc1test0lngrating;
   double            m_nonc1test0strrating;
   double            m_nonc1test1lngrating;
   double            m_nonc1test1strrating;
   double            m_probingstepmax;
   double            m_probingstepscale;
   double            m_probingstp;
   bool              m_badgradhasxj;
   bool              m_checksmoothness;
   bool              m_linesearchspoiled;
   bool              m_linesearchstarted;
   bool              m_needfij;
   CRowInt           m_bufi;
   CRowInt           m_sortedidx;
   CRowInt           m_tmpidx;
   CRowDouble        m_bufr;
   CRowDouble        m_dcur;
   CRowDouble        m_deltax;
   CRowDouble        m_du;
   CRowDouble        m_enqueuedfunc;
   CRowDouble        m_enqueuedstp;
   CRowDouble        m_enqueuedx;
   CRowDouble        m_f;
   CRowDouble        m_f0;
   CRowDouble        m_fbase;
   CRowDouble        m_fc;
   CRowDouble        m_fi;
   CRowDouble        m_fm;
   CRowDouble        m_fp;
   CRowDouble        m_g;
   CRowDouble        m_jc;
   CRowDouble        m_jm;
   CRowDouble        m_jp;
   CRowDouble        m_probingf;
   CRowDouble        m_probingsteps;
   CRowDouble        m_s;
   CRowDouble        m_sortedstp;
   CRowDouble        m_stp;
   CRowDouble        m_x;
   CRowDouble        m_xbase;
   CRowDouble        m_xu;
   CMatrixDouble     m_enqueuedjac;
   CMatrixDouble     m_j;
   CMatrixDouble     m_j0;
   CMatrixDouble     m_jbasenum;
   CMatrixDouble     m_jbaseusr;
   CMatrixDouble     m_probingslopes;
   CMatrixDouble     m_probingvalues;
   //--- constructor / destructor
                     CSmoothnessMonitor(void);
                    ~CSmoothnessMonitor(void) {}
   void              Copy(const CSmoothnessMonitor &obj);
   //--- overloading
   void              operator=(const CSmoothnessMonitor &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSmoothnessMonitor::CSmoothnessMonitor(void)
  {
   m_enqueuedcnt=0;
   m_k=0;
   m_n=0;
   m_probingnstepsstored=0;
   m_probingnvalues=0;
   m_sortedcnt=0;
   m_nonc0currentrating=0;
   m_nonc0lngrating=0;
   m_nonc0strrating=0;
   m_nonc1currentrating=0;
   m_nonc1test0lngrating=0;
   m_nonc1test0strrating=0;
   m_nonc1test1lngrating=0;
   m_nonc1test1strrating=0;
   m_probingstepmax=0;
   m_probingstepscale=0;
   m_probingstp=0;
   m_badgradhasxj=false;
   m_checksmoothness=false;
   m_linesearchspoiled=false;
   m_linesearchstarted=false;
   m_needfij=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CSmoothnessMonitor::Copy(const CSmoothnessMonitor &obj)
  {
   m_probingrcomm=obj.m_probingrcomm;
   m_rstateg0=obj.m_rstateg0;
   m_rep=obj.m_rep;
   m_nonc1test1lngrep=obj.m_nonc1test1lngrep;
   m_nonc1test1strrep=obj.m_nonc1test1strrep;
   m_nonc1test0lngrep=obj.m_nonc1test0lngrep;
   m_nonc1test0strrep=obj.m_nonc1test0strrep;
   m_nonc0lngrep=obj.m_nonc0lngrep;
   m_nonc0strrep=obj.m_nonc0strrep;
   m_enqueuedcnt=obj.m_enqueuedcnt;
   m_k=obj.m_k;
   m_n=obj.m_n;
   m_probingnstepsstored=obj.m_probingnstepsstored;
   m_probingnvalues=obj.m_probingnvalues;
   m_sortedcnt=obj.m_sortedcnt;
   m_nonc0currentrating=obj.m_nonc0currentrating;
   m_nonc0lngrating=obj.m_nonc0lngrating;
   m_nonc0strrating=obj.m_nonc0strrating;
   m_nonc1currentrating=obj.m_nonc1currentrating;
   m_nonc1test0lngrating=obj.m_nonc1test0lngrating;
   m_nonc1test0strrating=obj.m_nonc1test0strrating;
   m_nonc1test1lngrating=obj.m_nonc1test1lngrating;
   m_nonc1test1strrating=obj.m_nonc1test1strrating;
   m_probingstepmax=obj.m_probingstepmax;
   m_probingstepscale=obj.m_probingstepscale;
   m_probingstp=obj.m_probingstp;
   m_badgradhasxj=obj.m_badgradhasxj;
   m_checksmoothness=obj.m_checksmoothness;
   m_linesearchspoiled=obj.m_linesearchspoiled;
   m_linesearchstarted=obj.m_linesearchstarted;
   m_needfij=obj.m_needfij;
   m_bufi=obj.m_bufi;
   m_sortedidx=obj.m_sortedidx;
   m_tmpidx=obj.m_tmpidx;
   m_bufr=obj.m_bufr;
   m_dcur=obj.m_dcur;
   m_deltax=obj.m_deltax;
   m_du=obj.m_du;
   m_enqueuedfunc=obj.m_enqueuedfunc;
   m_enqueuedstp=obj.m_enqueuedstp;
   m_enqueuedx=obj.m_enqueuedx;
   m_f=obj.m_f;
   m_f0=obj.m_f0;
   m_fbase=obj.m_fbase;
   m_fc=obj.m_fc;
   m_fi=obj.m_fi;
   m_fm=obj.m_fm;
   m_fp=obj.m_fp;
   m_g=obj.m_g;
   m_jc=obj.m_jc;
   m_jm=obj.m_jm;
   m_jp=obj.m_jp;
   m_probingf=obj.m_probingf;
   m_probingsteps=obj.m_probingsteps;
   m_s=obj.m_s;
   m_sortedstp=obj.m_sortedstp;
   m_stp=obj.m_stp;
   m_x=obj.m_x;
   m_xbase=obj.m_xbase;
   m_xu=obj.m_xu;
   m_enqueuedjac=obj.m_enqueuedjac;
   m_j=obj.m_j;
   m_j0=obj.m_j0;
   m_jbasenum=obj.m_jbasenum;
   m_jbaseusr=obj.m_jbaseusr;
   m_probingslopes=obj.m_probingslopes;
   m_probingvalues=obj.m_probingvalues;
  }
//+------------------------------------------------------------------+
//| Auxiliary functions for other classes                            |
//+------------------------------------------------------------------+
class COptServ
  {
public:
   //--- constants
   static const double m_ognoiselevelf;
   static const double m_ognoiselevelg;
   static const double m_ogminrating0;
   static const double m_ogminrating1;

   static void       CheckBcViolation(bool &HasBndL[],CRowDouble &bndl,bool &HasBndU[],CRowDouble &bndu,CRowDouble &x,int n,CRowDouble &s,bool nonunits,double &bcerr,int &bcidx);
   static void       CheckLcViolation(CMatrixDouble &cleic,CRowInt &lcsrcidx,int nec,int nic,CRowDouble &x,int n,double &lcerr,int &lcidx);
   static void       CheckNLcViolation(CRowDouble &fi,int ng,int nh,double &nlcerr,int &nlcidx);
   static void       UnScaleAndCheckNLcViolation(CRowDouble &fi,CRowDouble &fscales,int ng,int nh,double &nlcerr,int &nlcidx);
   static void       TrimPrepare(double f,double &threshold);
   static void       TrimFunction(double &f,double &g[],const int n,const double threshold);
   static void       TrimFunction(double &f,CRowDouble &g,const int n,const double threshold);
   static bool       EnforceBoundaryConstraints(CRowDouble &x,CRowDouble &bl,bool &havebl[],CRowDouble &bu,bool &havebu[],int nmain,int nslack);
   static void       ProjectGradientIntoBC(CRowDouble &x,CRowDouble &g,CRowDouble &bl,bool &havebl[],CRowDouble &bu,bool &havebu[],int nmain,int nslack);
   static void       CalculateStepBound(CRowDouble &x,CRowDouble &d,double alpha,CRowDouble &bndl,bool &havebndl[],CRowDouble &bndu,bool &havebndu[],int nmain,int nslack,int &variabletofreeze,double &valuetofreeze,double &maxsteplen);
   static int        PostProcessBoundedStep(CRowDouble &x,CRowDouble &xprev,CRowDouble &bndl,bool &havebndl[],CRowDouble &bndu,bool &havebndu[],int nmain,int nslack,int variabletofreeze,double valuetofreeze,double steptaken,double maxsteplen);
   static void       FilterDirection(CRowDouble &d,CRowDouble &x,CRowDouble &bndl,bool &havebndl[],CRowDouble &bndu,bool &havebndu[],CRowDouble &s,int nmain,int nslack,double droptol);
   static int        NumberOfChangedConstraints(CRowDouble &x,CRowDouble &xprev,CRowDouble &bndl,bool &havebndl[],CRowDouble &bndu,bool &havebndu[],int nmain,int nslack);
   static bool       FindFeasiblePoint(CRowDouble &x,CRowDouble &bndl,bool &havebndl[],CRowDouble &bndu,bool &havebndu[],int nmain,int nslack,CMatrixDouble &ce,int k,double epsi,int &qpits,int &gpaits);
   static bool       DerivativeCheck(double f0,double df0,double f1,double df1,double f,double df,double width);
   static void       EstimateParabolicModel(double absasum,double absasum2,double mx,double mb,double md,double d1,double d2,int &d1est,int &d2est);
   static void       InexactLBFGSPreconditioner(CRowDouble &s,int n,CRowDouble &d,CRowDouble &c,CMatrixDouble &w,int k,CPrecBufLBFGS &buf);
   static void       PrepareLowRankPreconditioner(CRowDouble &d,CRowDouble &c,CMatrixDouble &w,int n,int k,CPrecBufLowRank &buf);
   static void       ApplyLowRankPreconditioner(CRowDouble &s,CPrecBufLowRank &buf);
   static void       SmoothnessMonitorInit(CSmoothnessMonitor &monitor,CRowDouble &s,int n,int k,bool checksmoothness);
   static void       SmoothnessMonitorStartLineSearch(CSmoothnessMonitor &monitor,CRowDouble &x,CRowDouble &fi,CMatrixDouble &jac);
   static void       SmoothnessMonitorStartLineSearch1u(CSmoothnessMonitor &monitor,CRowDouble &s,CRowDouble &invs,CRowDouble &x,double f0,CRowDouble &j0);
   static void       SmoothnessMonitorEnqueuePoint(CSmoothnessMonitor &monitor,CRowDouble &d,double stp,CRowDouble &x,CRowDouble &fi,CMatrixDouble &jac);
   static void       SmoothnessMonitorEnqueuePoint1u(CSmoothnessMonitor &monitor,CRowDouble &s,CRowDouble &invs,CRowDouble &d,double stp,CRowDouble &x,double f0,CRowDouble &j0);
   static void       SmoothnessMonitorFinalizeLineSearch(CSmoothnessMonitor &monitor);
   static void       SmoothnessMonitorStartProbing(CSmoothnessMonitor &monitor,double stpmax,int nvalues,double stepscale);
   static bool       SmoothnessMonitorProbe(CSmoothnessMonitor &monitor);
   static void       SmoothnessMonitorTraceProbingResults(CSmoothnessMonitor &monitor);
   static void       SmoothnessMonitorTraceStatus(CSmoothnessMonitor &monitor,bool callersuggeststrace);
   static void       SmoothnessMonitorExportReport(CSmoothnessMonitor &monitor,COptGuardReport &rep);
   static bool       SmoothnessMonitorCheckGradientATX0(CSmoothnessMonitor &monitor,CRowDouble &unscaledx0,CRowDouble &s,CRowDouble &bndl,CRowDouble &bndu,bool hasboxconstraints,double teststep);

private:
   static double     FeasibilityError(CMatrixDouble &ce,CRowDouble &x,int nmain,int nslack,int k,CRowDouble &m_tmp0);
   static void       FeasibilityErrorGrad(CMatrixDouble &ce,CRowDouble &x,int nmain,int nslack,int k,double &err,CRowDouble &grad,CRowDouble &m_tmp0);
   static void       TestC0Continuity(double f0,double f1,double f2,double f3,double noise0,double noise1,double noise2,double noise3,double delta0,double delta1,double delta2,bool applyspecialcorrection,double &rating,double &lipschitz);
   static void       C1ContinuityTest0(CSmoothnessMonitor &monitor,int funcidx,int stpidx,int sortedcnt);
   static void       C1ContinuityTest1(CSmoothnessMonitor &monitor,int funcidx,int stpidx,int sortedcnt);
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
const double COptServ::m_ognoiselevelf=1.0E2*CMath::m_machineepsilon;
const double COptServ::m_ognoiselevelg=1.0E4*CMath::m_machineepsilon;
const double COptServ::m_ogminrating0=50.0;
const double COptServ::m_ogminrating1=50.0;
//+------------------------------------------------------------------+
//| This subroutine checks violation of the box constraints. On      |
//| output it sets bcerr to the maximum scaled violation, bcidx to   |
//| the index of the violating constraint.                           |
//|   if bcerr=0 (say, if no constraints are violated) then bcidx=-1.|
//|   If nonunits=false then s[] is not referenced at all (assumed   |
//|                     unit).                                       |
//+------------------------------------------------------------------+
void COptServ::CheckBcViolation(bool &HasBndL[],
                                CRowDouble &bndl,
                                bool &HasBndU[],
                                CRowDouble &bndu,
                                CRowDouble &x,
                                int n,
                                CRowDouble &s,
                                bool nonunits,
                                double &bcerr,
                                int &bcidx)
  {
//--- create variables
   int i=0;
   double vs=0;
   double ve=0;
//--- initialization
   bcerr=0;
   bcidx=-1;

   for(i=0; i<n; i++)
     {
      //--- Fetch scale
      if(nonunits)
         vs=1/s[i];
      else
         vs=1;
      //--- Check lower bound
      if(HasBndL[i] && x[i]<bndl[i])
        {
         ve=(bndl[i]-x[i])*vs;
         if(ve>bcerr)
           {
            bcerr=ve;
            bcidx=i;
           }
        }
      //--- Check upper bound
      if(HasBndU[i] && x[i]>bndu[i])
        {
         ve=(x[i]-bndu[i])*vs;
         if(ve>bcerr)
           {
            bcerr=ve;
            bcidx=i;
           }
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine checks violation of the general linear           |
//| constraints.                                                     |
//| Constraints are assumed to be un-normalized and stored in the    |
//| format "NEC equality ones followed by NIC inequality ones".      |
//| On output it sets lcerr to the maximum scaled violation, lcidx to|
//| the source index of the most violating constraint (row indexes of|
//| CLEIC are mapped to the indexes of the "original" constraints via|
//| LCSrcIdx[] array.                                                |
//|   if lcerr=0 (say, if no constraints are violated) then lcidx=-1.|
//|   If nonunits=false then s[] is not referenced at all (assumed   |
//|                     unit).                                       |
//+------------------------------------------------------------------+
void COptServ::CheckLcViolation(CMatrixDouble &cleic,
                                CRowInt &lcsrcidx,
                                int nec,
                                int nic,
                                CRowDouble &x,
                                int n,
                                double &lcerr,
                                int &lcidx)
  {
//--- create variables
   int i=0;
   int j=0;
   double cx=0;
   double cnrm=0;
   double v=0;

   lcerr=0;
   lcidx=-1;

   for(i=0; i<nec+nic; i++)
     {
      cx=-cleic.Get(i,n);
      cnrm=0;
      for(j=0; j<n; j++)
        {
         v=cleic.Get(i,j);
         cx+=v*x[j];
         cnrm+=v*v;
        }
      cnrm=MathSqrt(cnrm);
      cx=cx/CApServ::Coalesce(cnrm,1);
      if(i<nec)
        {
         cx=MathAbs(cx);
        }
      else
        {
         cx=MathMax(cx,0);
        }
      if(cx>lcerr)
        {
         lcerr=cx;
         lcidx=lcsrcidx[i];
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine checks violation of the nonlinear constraints.   |
//| Fi[0] is the target value (ignored), Fi[1:NG+NH] are values of   |
//| nonlinear constraints.                                           |
//| On output it sets nlcerr to the scaled violation, nlcidx to the  |
//| index of the most violating constraint in [0,NG+NH-1] range.     |
//|   if nlcerr=0 (say, if no constraints are violated) then         |
//|               nlcidx=-1.                                         |
//|   If nonunits=false then s[] is not referenced at all (assumed   |
//|               unit).                                             |
//+------------------------------------------------------------------+
void COptServ::CheckNLcViolation(CRowDouble &fi,
                                 int ng,
                                 int nh,
                                 double &nlcerr,
                                 int &nlcidx)
  {
//--- create variables
   int i=0;
   double v=0;

   nlcerr=0;
   nlcidx=-1;

   for(i=0; i<ng+nh; i++)
     {
      v=fi[i+1];
      if(i<ng)
         v=MathAbs(v);
      else
         v=MathMax(v,0);
      if(v>nlcerr)
        {
         nlcerr=v;
         nlcidx=i;
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine is same as CheckNLCViolation, but is works with  |
//| scaled constraints: it assumes that Fi[] were divided by         |
//| FScales[] vector  BEFORE passing them to this function.          |
//| The function checks scaled values, but reports unscaled errors.  |
//+------------------------------------------------------------------+
void COptServ::UnScaleAndCheckNLcViolation(CRowDouble &fi,
                                           CRowDouble &fscales,
                                           int ng,
                                           int nh,
                                           double &nlcerr,
                                           int &nlcidx)
  {
//--- create variables
   int i=0;
   double v=0;

   nlcerr=0;
   nlcidx=-1;
   for(i=0; i<ng+nh; i++)
     {
      //--- check
      if(!CAp::Assert(fscales[i+1]>0.0,__FUNCTION__+": integrity check failed"))
         return;
      v=fi[i+1]*fscales[i+1];
      if(i<ng)
         v=MathAbs(v);
      else
         v=MathMax(v,0);
      if(v>nlcerr)
        {
         nlcerr=v;
         nlcidx=i;
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine is used to prepare threshold value which will be |
//| used for trimming of the target function (see comments on        |
//| TrimFunction() for more information).                            |
//| This function accepts only one parameter: function value at the  |
//| starting point. It returns threshold which will be used for      |
//| trimming.                                                        |
//+------------------------------------------------------------------+
void COptServ::TrimPrepare(double f,double &threshold)
  {
   threshold=10*(MathAbs(f)+1);
  }
//+------------------------------------------------------------------+
//| This subroutine is used to "trim" target function, i.e. to do    |
//| following transformation:                                        |
//|                            { {F,G}          if F<Threshold       |
//|             {F_tr, G_tr} = {                                     |
//|                            { {Threshold, 0} if F>=Threshold      |
//| Such transformation allows us to  solve  problems  with          |
//| singularities by redefining function in such way that it becomes |
//| bounded from above.                                              |
//+------------------------------------------------------------------+
void COptServ::TrimFunction(double &f,double &g[],const int n,
                            const double threshold)
  {
//--- check
   if(f>=threshold)
     {
      f=threshold;
      ArrayFill(g,0,n,0);
     }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void COptServ::TrimFunction(double &f,CRowDouble &g,const int n,
                            const double threshold)
  {
//--- check
   if(f>=threshold)
     {
      f=threshold;
      if(g.Size()<=n)
         g=vector<double>::Zeros(n);
      else
         for(int i=0; i<n; i++)
            g.Set(i,0.0);
     }
  }
//+------------------------------------------------------------------+
//| This function enforces boundary constraints in the X.            |
//| This function correctly(although a bit inefficient) handles BL[i]|
//| which are -INF and BU[i] which are +INF.                       |
//| We have NMain + NSlack dimensional X, with first NMain components|
//| bounded by BL/BU, and next NSlack ones bounded by non-negativity |
//| constraints.                                                     |
//| INPUT PARAMETERS:                                                |
//|   X        -  array[NMain + NSlack], point                       |
//|   BL       -  array[NMain], lower bounds (may contain -INF, when|
//|               bound is not present)                              |
//|   HaveBL   -  array[NMain], if HaveBL[i] is False, then i-th     |
//|               bound is not present                               |
//|   BU       -  array[NMain], upper bounds (may contain +INF, when|
//|               bound is not present)                              |
//|   HaveBU   -  array[NMain], if HaveBU[i] is False, then i-th     |
//|               bound is not present                               |
//| OUTPUT PARAMETERS:                                               |
//|   X        -  X with all constraints being enforced              |
//| It returns True when constraints are consistent,                 |
//|            False - when constraints are inconsistent.            |
//+------------------------------------------------------------------+
bool COptServ::EnforceBoundaryConstraints(CRowDouble &x,
                                          CRowDouble &bl,
                                          bool &havebl[],
                                          CRowDouble &bu,
                                          bool &havebu[],
                                          int nmain,
                                          int nslack)
  {
   for(int i=0; i<nmain; i++)
     {
      if(havebl[i] && havebu[i] && bl[i]>bu[i])
         return(false);
      //---
      if(havebl[i] && x[i]<bl[i])
         x.Set(i,bl[i]);
      if(havebu[i] && x[i]>bu[i])
         x.Set(i,bu[i]);
     }

   for(int i=0; i<nslack; i++)
      if(x[nmain+i]<0.0)
         x.Set(nmain+i,0);
//--- return result
   return(true);
  }
//+------------------------------------------------------------------+
//| This function projects gradient into feasible area of boundary   |
//| constrained optimization  problem.  X  can  be  infeasible  with |
//| respect  to boundary constraints.  We  have  NMain + NSlack      |
//| dimensional  X,   with  first  NMain components bounded by BL/BU,|
//| and next NSlack ones bounded by non-negativity constraints.    |
//| INPUT PARAMETERS:                                                |
//|   X        -  array[NMain + NSlack], point                       |
//|   G        -  array[NMain + NSlack], gradient                    |
//|   BL       -  lower bounds(may contain -INF, when bound is not  |
//|               present)                                           |
//|   HaveBL   -  if HaveBL[i] is False, then i-th bound is not    |
//|               present                                            |
//|   BU       -  upper bounds(may contain +INF, when bound is not  |
//|               present)                                           |
//|   HaveBU   -  if HaveBU[i] is False, then i-th bound is not    |
//|               present                                            |
//| OUTPUT PARAMETERS:                                               |
//|   G        -  projection of G. Components of G which satisfy one |
//|               of the following                                   |
//|               (1)(X[I] <= BndL[I]) and (G[I] > 0), OR            |
//|               (2)(X[I] >= BndU[I]) and (G[I] < 0)                |
//|               are replaced by zeros.                             |
//| NOTE 1: this function assumes that constraints are feasible. It  |
//|         throws exception otherwise.                              |
//| NOTE 2: in fact, projection of ANTI - gradient is calculated,    |
//|         because  this function trims components of - G which     |
//|         points outside of the feasible area. However, working    |
//|         with - G is considered confusing, because all            |
//|         optimization source work with G.                         |
//+------------------------------------------------------------------+
void COptServ::ProjectGradientIntoBC(CRowDouble &x,
                                     CRowDouble &g,
                                     CRowDouble &bl,
                                     bool &havebl[],
                                     CRowDouble &bu,
                                     bool &havebu[],
                                     int nmain,
                                     int nslack)
  {
   for(int i=0; i<nmain; i++)
     {
      //--- check
      if(!CAp::Assert(!havebl[i] || !havebu[i] || bl[i]<=bu[i],__FUNCTION__+": internal error (infeasible constraints)"))
         return;
      if(havebl[i] && x[i]<=bl[i] && g[i]>0.0)
         g.Set(i,0);
      if(havebu[i] && x[i]>=bu[i] && g[i]<0.0)
         g.Set(i,0);
     }
   for(int i=0; i<nslack; i++)
      if(x[nmain+i]<=0.0 && g[nmain+i]>0.0)
         g.Set(nmain+i,0);
  }
//+------------------------------------------------------------------+
//| Given                                                            |
//|   a) initial point X0[NMain + NSlack] (feasible with respect     |
//|      to bound constraints)                                       |
//|   b) step vector alpha*D[NMain + NSlack]                         |
//|   c) boundary constraints BndL[NMain], BndU[NMain]               |
//|   d) implicit non-negativity constraints for slack variables   |
//|      this  function  calculates  bound  on  the step length      |
//|      subject to boundary constraints.                            |
//| It returns:                                                      |
//|   *  MaxStepLen - such step length that X0 + MaxStepLen*alpha*D  |
//|      is exactly at the boundary given by constraints             |
//|   *  VariableToFreeze - index of the constraint to be activated, |
//|      0 <= VariableToFreeze < NMain + NSlack                      |
//|   *  ValueToFreeze - value of the corresponding constraint.      |
//| Notes:                                                           |
//|   * it is possible that several constraints can be activated by  |
//|     the step at once. In such cases only one constraint is       |
//|     returned. It is caller responsibility to check other         |
//|     constraints. This function makes  sure that we activate at   |
//|     least one constraint, and everything else is the             |
//|     responsibility of the caller.                                |
//|   * steps smaller than MaxStepLen still can activate constraints |
//|     due to numerical errors. Thus purpose of this  function  is  |
//|     not to guard against accidental activation of the            |
//|     constraints - quite the reverse, its purpose is to activate  |
//|     at least constraint upon performing step which is too long.  |
//|   * in case there is no constraints to activate, we return       |
//|     negative VariableToFreeze and zero MaxStepLen and            |
//|     ValueToFreeze.                                               |
//|   * this function assumes that constraints are consistent; it    |
//|     throws exception otherwise.                                  |
//| INPUT PARAMETERS:                                                |
//|   X        -  array[NMain + NSlack], point. Must be feasible with|
//|               respect to bound constraints(exception will be     |
//|               thrown otherwise)                                  |
//|   D        -  array[NMain + NSlack], step direction              |
//|   alpha    -  scalar multiplier before D, alpha<>0               |
//|   BndL     -  lower bounds, array[NMain] (may contain -INF, when|
//|               bound is not present)                              |
//|   HaveBndL -  array[NMain], if HaveBndL[i] is False, then i-th   |
//|               bound is not present                               |
//|   BndU     -  array[NMain], upper bounds (may contain +INF, when|
//|               bound is not present)                              |
//|   HaveBndU -  array[NMain], if HaveBndU[i] is False, then i th   |
//|               bound is not present                               |
//|   NMain    -  number of main variables                           |
//|   NSlack   -  number of slack variables                          |
//| OUTPUT PARAMETERS:                                               |
//|   VariableToFreeze:                                              |
//|      * negative value     = step is unbounded, ValueToFreeze = 0,|
//|                             MaxStepLen = 0.                      |
//|      * Non-Negative value = at least one constraint, given by  |
//|                             this parameter, will  be  activated  |
//|                              upon performing maximum step.       |
//|   ValueToFreeze -  value of the variable which will be           |
//|                    constrained                                   |
//|   MaxStepLen    -  maximum length of the step. Can be zero when  |
//|                    step vector looks outside of the feasible area|
//+------------------------------------------------------------------+
void COptServ::CalculateStepBound(CRowDouble &x,CRowDouble &d,
                                  double alpha,CRowDouble &bndl,
                                  bool &havebndl[],CRowDouble &bndu,
                                  bool &havebndu[],int nmain,int nslack,
                                  int &variabletofreeze,
                                  double &valuetofreeze,
                                  double &maxsteplen)
  {
//--- create variables
   double prevmax=0;
   double initval=0;
//--- initialization
   variabletofreeze=0;
   valuetofreeze=0;
   maxsteplen=0;
//--- check
   if(!CAp::Assert(alpha!=0.0,__FUNCTION__+": zero alpha"))
      return;
   variabletofreeze=-1;
   initval=CMath::m_maxrealnumber;
   maxsteplen=initval;
   for(int i=0; i<nmain; i++)
     {
      if(havebndl[i] && (alpha*d[i])<0.0)
        {
         //--- check
         if(!CAp::Assert(x[i]>=bndl[i],__FUNCTION__+": infeasible X"))
            return;
         prevmax=maxsteplen;
         maxsteplen=CApServ::SafeMinPosRV(x[i]-bndl[i],-(alpha*d[i]),maxsteplen);
         if(maxsteplen<prevmax)
           {
            variabletofreeze=i;
            valuetofreeze=bndl[i];
           }
        }
      if(havebndu[i] && (alpha*d[i])>0.0)
        {
         //--- check
         if(!CAp::Assert(x[i]<=bndu[i],__FUNCTION__+": infeasible X"))
            return;
         prevmax=maxsteplen;
         maxsteplen=CApServ::SafeMinPosRV(bndu[i]-x[i],alpha*d[i],maxsteplen);
         if(maxsteplen<prevmax)
           {
            variabletofreeze=i;
            valuetofreeze=bndu[i];
           }
        }
     }
   for(int i=0; i<nslack; i++)
     {
      if((alpha*d[nmain+i])<0.0)
        {
         //--- check
         if(!CAp::Assert(x[nmain+i]>=0.0,__FUNCTION__+": infeasible X"))
            return;
         prevmax=maxsteplen;
         maxsteplen=CApServ::SafeMinPosRV(x[nmain+i],-(alpha*d[nmain+i]),maxsteplen);
         if(maxsteplen<prevmax)
           {
            variabletofreeze=nmain+i;
            valuetofreeze=0;
           }
        }
     }
   if(maxsteplen==initval)
     {
      valuetofreeze=0;
      maxsteplen=0;
     }
  }
//+------------------------------------------------------------------+
//| This function postprocesses bounded step by:                     |
//|   * analysing step length(whether it is equal to MaxStepLen) and |
//|     activating constraint given by VariableToFreeze if needed    |
//|   * checking for additional bound constraints to activate        |
//| This function uses final point of the step, quantities calculated|
//| by the CalculateStepBound() function.  As  result,  it  returns  |
//| point  which is exactly feasible with respect to boundary        |
//| constraints.                                                     |
//| NOTE 1: this function does NOT handle and check linear equality  |
//|         constraints                                              |
//| NOTE 2: when StepTaken = MaxStepLen we always activate at least  |
//|         one constraint                                           |
//| INPUT PARAMETERS:                                                |
//|   X        -  array[NMain + NSlack], final point to postprocess  |
//|   XPrev    -  array[NMain + NSlack], initial point               |
//|   BndL     -  lower bounds, array[NMain] (may contain -INF, when|
//|               bound is not present)                              |
//|   HaveBndL -  array[NMain], if HaveBndL[i] is False, then i-th   |
//|               bound is not present                               |
//|   BndU     -  array[NMain], upper bounds (may contain +INF, when|
//|               bound is not present)                              |
//|   HaveBndU - array[NMain], if HaveBndU[i] is False, then i-th    |
//|              bound is not present                                |
//|   NMain    -  number of main variables                           |
//|   NSlack   -  number of slack variables                          |
//|   VariableToFreeze - result of CalculateStepBound()              |
//|   ValueToFreeze -  result of CalculateStepBound()                |
//|   StepTaken-  actual step length (actual step is equal to the    |
//|               possibly non-unit step direction vector times this |
//|               parameter). StepTaken <= MaxStepLen.               |
//|   MaxStepLen- result of CalculateStepBound()                     |
//| OUTPUT PARAMETERS:                                               |
//|   X        -  point bounded with respect to constraints.         |
//|               components corresponding to active constraints are |
//|               exactly equal to the boundary values.              |
//| RESULT:                                                          |
//|   number of constraints activated in addition to previously      |
//|   active ones.                                                   |
//| Constraints which were DEACTIVATED are ignored(do not influence  |
//| function value).                                                 |
//+------------------------------------------------------------------+
int COptServ::PostProcessBoundedStep(CRowDouble &x,
                                     CRowDouble &xprev,
                                     CRowDouble &bndl,
                                     bool &havebndl[],
                                     CRowDouble &bndu,
                                     bool &havebndu[],
                                     int nmain,
                                     int nslack,
                                     int variabletofreeze,
                                     double valuetofreeze,
                                     double steptaken,
                                     double maxsteplen)
  {
//--- create variables
   int  result=0;
   int  i=0;
   bool wasactivated;
//--- check
   if(!CAp::Assert(variabletofreeze<0 || steptaken<=maxsteplen))
      return(result);
//--- Activate constraints
   if(variabletofreeze>=0 && steptaken==maxsteplen)
      x.Set(variabletofreeze,valuetofreeze);
   for(i=0; i<nmain; i++)
     {
      if(havebndl[i] && x[i]<bndl[i])
         x.Set(i,bndl[i]);
      if(havebndu[i] && x[i]>bndu[i])
         x.Set(i,bndu[i]);
     }
   for(i=0; i<nslack; i++)
      if(x[nmain+i]<=0.0)
         x.Set(nmain+i,0);
//--- Calculate number of constraints being activated
   for(i=0; i<nmain; i++)
     {
      wasactivated=(x[i]!=xprev[i] && ((havebndl[i] && x[i]==bndl[i]) || (havebndu[i] && x[i]==bndu[i])));
      wasactivated=wasactivated || variabletofreeze==i;
      if(wasactivated)
         result++;
     }
   for(i=0; i<nslack; i++)
     {
      wasactivated=(x[nmain+i]!=xprev[nmain+i] && x[nmain+i]==0.0);
      wasactivated=wasactivated || variabletofreeze==nmain+i;
      if(wasactivated)
         result++;
     }
   return(result);
  }
//+------------------------------------------------------------------+
//| The  purpose  of  this  function is to prevent algorithm from    |
//| "unsticking" from  the  active  bound  constraints  because  of  |
//| numerical noise in the gradient or Hessian.                      |
//| It is done by zeroing some components of the search direction D. |
//| D[i] is zeroed when both(a) and (b) are true:                    |
//|   a) corresponding X[i] is exactly at the boundary               |
//|   b) | D[i]*S[i] | <= DropTol*Sqrt(SUM(D[i] ^ 2 * S[I] ^ 2))     |
//| D can be step direction, antigradient, gradient, or anything     |
//| similar. Sign of D does not matter, nor matters step length.     |
//| NOTE 1: boundary constraints are expected to be consistent, as   |
//|         well as X is expected to be feasible. Exception will be  |
//|         thrown otherwise.                                        |
//| INPUT PARAMETERS:                                                |
//|   D        -  array[NMain + NSlack], direction                   |
//|   X        -  array[NMain + NSlack], current point               |
//|   BndL     -  lower bounds, array[NMain] (may contain -INF, when |
//|               bound is not present)                              |
//|   HaveBndL -  array[NMain], if HaveBndL[i] is False, then i-th   |
//|               bound is not present                               |
//|   BndU     -  array[NMain], upper bounds (may contain +INF, when |
//|               bound is not present)                              |
//|   HaveBndU -  array[NMain], if HaveBndU[i] is False, then i-th   |
//|               bound is not present                               |
//|   S        -  array[NMain + NSlack], scaling of the variables    |
//|   NMain    -  number of main variables                           |
//|   NSlack   -  number of slack variables                          |
//|   DropTol  -  drop tolerance, >= 0                               |
//| OUTPUT PARAMETERS:                                               |
//|   X        -  point bounded with respect to constraints.         |
//|               components corresponding to active constraints are |
//|               exactly equal to the boundary values.              |
//+------------------------------------------------------------------+
void COptServ::FilterDirection(CRowDouble &d,
                               CRowDouble &x,
                               CRowDouble &bndl,
                               bool &havebndl[],
                               CRowDouble &bndu,
                               bool &havebndu[],
                               CRowDouble &s,
                               int nmain,
                               int nslack,
                               double droptol)
  {
//--- create variables
   int    i=0;
   double scalednorm=0;
   bool   isactive=false;

   for(i=0; i<nmain+nslack; i++)
      scalednorm+=CMath::Sqr(d[i]*s[i]);
   scalednorm=MathSqrt(scalednorm);
   for(i=0; i<nmain; i++)
     {
      //--- check
      if(!CAp::Assert(!havebndl[i] || x[i]>=bndl[i],__FUNCTION__+": infeasible point"))
         return;
      if(!CAp::Assert(!havebndu[i] || x[i]<=bndu[i],__FUNCTION__+": infeasible point"))
         return;
      isactive=(havebndl[i] && x[i]==bndl[i]) || (havebndu[i] && x[i]==bndu[i]);
      if(isactive && MathAbs(d[i]*s[i])<=(droptol*scalednorm))
         d.Set(i,0.0);
     }
   for(i=0; i<nslack; i++)
     {
      //--- check
      if(!CAp::Assert(x[nmain+i]>=0.0,__FUNCTION__+": infeasible point"))
         return;
      if(x[nmain+i]==0.0 && MathAbs(d[nmain+i]*s[nmain+i])<=(droptol*scalednorm))
         d.Set(nmain+i,0.0);
     }
  }
//+------------------------------------------------------------------+
//| This function returns number of bound constraints whose State was|
//| changed (either activated or deactivated) when making step from  |
//| XPrev to X.                                                      |
//| Constraints are considered:                                      |
//|   * active - when we are exactly at the boundary                 |
//|   * inactive - when we are not at the boundary                   |
//| You should note that antigradient direction is NOT taken into    |
//| account when we make decions on the constraint status.           |
//| INPUT PARAMETERS:                                                |
//|   X        -  array[NMain + NSlack], final point. Must be        |
//|               feasible with respect to bound constraints.        |
//|   XPrev    -  array[NMain + NSlack], initial point. Must be      |
//|               feasible with respect to bound constraints.        |
//|   BndL     -  lower bounds, array[NMain] (may contain -INF, when |
//|               bound is not present)                              |
//|   HaveBndL -  array[NMain], if HaveBndL[i] is False, then i-th   |
//|               bound is not present                               |
//|   BndU     -  array[NMain], upper bounds (may contain +INF, when |
//|               bound is not present)                              |
//|   HaveBndU -  array[NMain], if HaveBndU[i] is False, then i-th   |
//|               bound is not present                               |
//|   NMain    -  number of main variables                           |
//|   NSlack   -  number of slack variables                          |
//| RESULT: number of constraints whose State was changed.           |
//+------------------------------------------------------------------+
int COptServ::NumberOfChangedConstraints(CRowDouble &x,
                                         CRowDouble &xprev,
                                         CRowDouble &bndl,
                                         bool &havebndl[],
                                         CRowDouble &bndu,
                                         bool &havebndu[],
                                         int nmain,
                                         int nslack)
  {
//--- create variables
   int  result=0;
   int  i=0;
   bool statuschanged;

   for(i=0; i<nmain; i++)
      if(x[i]!=xprev[i])
        {
         statuschanged=false;
         if(havebndl[i] && (x[i]==bndl[i] || xprev[i]==bndl[i]))
            statuschanged=true;
         if(havebndu[i] && (x[i]==bndu[i] || xprev[i]==bndu[i]))
            statuschanged=true;
         if(statuschanged)
            result++;
        }

   for(i=0; i<nslack; i++)
      if(x[nmain+i]!=xprev[nmain+i] && (x[nmain+i]==0.0 || xprev[nmain+i]==0.0))
         result=result+1;
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This function finds feasible point of(NMain + NSlack)-dimensional|
//| problem subject to NMain explicit boundary constraints (some     |
//| constraints can be omitted), NSlack implicit non-negativity      |
//| constraints, K linear equality constraints.                      |
//| INPUT PARAMETERS:                                                |
//|   X        -  array[NMain + NSlack], initial point.              |
//|   BndL     -  lower bounds, array[NMain] (may contain -INF, when |
//|               bound is not present)                              |
//|   HaveBndL -  array[NMain], if HaveBndL[i] is False, then i-th   |
//|               bound is not present                               |
//|   BndU     -  array[NMain], upper bounds (may contain +INF, when |
//|               bound is not present)                              |
//|   HaveBndU -  array[NMain], if HaveBndU[i] is False, then i-th   |
//|               bound is not present                               |
//|   NMain    -  number of main variables                           |
//|   NSlack   -  number of slack variables                          |
//|   CE       -  array[K, NMain + NSlack + 1], equality  constraints|
//|               CE*x = b. Rows contain constraints, first          |
//|               NMain + NSlack columns contain coefficients before |
//|               X[], last  column  contain right part.             |
//|   K        -  number of linear constraints                       |
//|   EpsI     -  infeasibility(error in the right part) allowed in  |
//|               the solution                                       |
//| OUTPUT PARAMETERS:                                               |
//|   X        -  feasible point or best infeasible point found      |
//|               beforec algorithm termination                      |
//|   QPIts    -  number of QP iterations(for debug purposes)        |
//|   GPAIts   -  number of GPA iterations(for debug purposes)       |
//| RESULT:                                                          |
//|   True in case X is feasible,                                    |
//|   False - if it is infeasible.                                   |
//+------------------------------------------------------------------+
bool COptServ::FindFeasiblePoint(CRowDouble &x,
                                 CRowDouble &bndl,
                                 bool &havebndl[],
                                 CRowDouble &bndu,
                                 bool &havebndu[],
                                 int nmain,
                                 int nslack,
                                 CMatrixDouble &CE,
                                 int k,
                                 double epsi,
                                 int &qpits,
                                 int &gpaits)
  {
//--- create variables
   bool   result;
   int    i=0;
   int    j=0;
   int    idx0=0;
   int    idx1=0;
   CRowDouble permx;
   CRowDouble xn;
   CRowDouble xa;
   CRowDouble newtonstep;
   CRowDouble g;
   CRowDouble pg;
   CRowDouble tau;
   CRowDouble s;
   double armijostep=0;
   double armijobeststep=0;
   double armijobestfeas=0;
   double v=0;
   double vv=0;
   double mx=0;
   double feaserr=0;
   double feaserr0=0;
   double feaserr1=0;
   double feasold=0;
   double feasnew=0;
   double pgnorm=0;
   double vn=0;
   double vd=0;
   double stp=0;
   int    vartofreeze=0;
   double valtofreeze=0;
   double maxsteplen=0;
   bool   werechangesinconstraints;
   bool   stage1isover;
   bool   converged;
   CRowDouble activeconstraints;
   CRowDouble tmpk;
   CRowDouble colnorms;
   int    nactive=0;
   int    nfree=0;
   CRowInt p1;
   CRowInt p2;
   CApBuff buf;
   int    itscount=0;
   int    itswithintolerance=0;
   int    maxitswithintolerance=0;
   int    badits=0;
   int    maxbadits=0;
   int    gparuns=0;
   int    maxarmijoruns=0;
   double infeasibilityincreasetolerance=0;
   CMatrixDouble permce;
   CMatrixDouble q;
   int    i_=0;
//--- initialization
   CMatrixDouble ce=CE;
   qpits=0;
   gpaits=0;
   maxitswithintolerance=3;
   maxbadits=3;
   maxarmijoruns=5;
   qpits=0;
   gpaits=0;
//--- Initial enforcement of the feasibility with respect to boundary constraints
//--- NOTE: after this block we assume that boundary constraints are consistent.
   if(!EnforceBoundaryConstraints(x,bndl,havebndl,bndu,havebndu,nmain,nslack))
      return(false);
//--- No linear constraints, we can exit right now
   if(k==0)
      return(true);
//--- Scale rows of CE in such way that max(CE[i,0..m_nmain+nslack-1])=1 for any i=0..m_k-1
   for(i=0; i<k; i++)
     {
      v=CAblasF::RMaxAbsR(nmain+nslack,ce,i);
      if(v!=0.0)
         ce.Row(i,ce[i]/v);
     }
//--- Allocate temporaries
   xn.Resize(nmain+nslack);
   xa.Resize(nmain+nslack);
   permx.Resize(nmain+nslack);
   g.Resize(nmain+nslack);
   pg.Resize(nmain+nslack);
   tmpk.Resize(k);
   permce.Resize(k,nmain+nslack);
   activeconstraints.Resize(nmain+nslack);
   newtonstep.Resize(nmain+nslack);
   s=vector<double>::Ones(nmain+nslack);
   colnorms=(ce*ce+0).Sum(0); //--- squerd summ by columns
//--- K>0, we have linear equality constraints combined with bound constraints.
//--- Try to find feasible point as minimizer of the quadratic function
//---     F(x) = 0.5*||CE*x-b||^2 = 0.5*x'*(CE'*CE)*x - (b'*CE)*x + 0.5*b'*b
//--- subject to boundary constraints given by BL, BU and non-negativity of
//--- the slack variables. BTW, we drop constant term because it does not
//--- actually influences on the solution.
//--- Below we will assume that K>0.
   itswithintolerance=0;
   badits=0;
   itscount=0;
   while(true)
     {
      //--- Dynamically adjust infeasibility error tolerance
      infeasibilityincreasetolerance=MathMax(CAblasF::RMaxAbsV(nmain+nslack,x),1)*(1000+nmain)*CMath::m_machineepsilon;
      //--- Stage 0: check for exact convergence
      converged=true;
      feaserr=FeasibilityError(ce,x,nmain,nslack,k,tmpk);
      for(i=0; i<k; i++)
        {
         //--- Calculate MX - maximum term in the left part
         //--- Terminate if error in the right part is not greater than 100*Eps*MX.
         //--- IMPORTANT: we must perform check for non-strict inequality, i.e. to use <= instead of <.
         //---            it will allow us to easily handle situations with zero rows of CE.
         //--- NOTE:      it is important to calculate feasibility error with dedicated
         //---            function. Once we had a situation when usage of "inline" code
         //---            resulted in different numerical values calculated at different
         //---            parts of program for exactly same X. However, this value is
         //---            essential for algorithm's ability to terminate before entering
         //---            infinite loop, so reproducibility of numerical results is very
         //---            important.
         mx=0;
         v=-ce.Get(i,nmain+nslack);
         for(j=0; j<nmain+nslack; j++)
           {
            mx=MathMax(mx,MathAbs(ce.Get(i,j)*x[j]));
            v+=ce.Get(i,j)*x[j];
           }
         converged=converged && MathAbs(v)<=(100.0*CMath::m_machineepsilon*mx);
        }
      feaserr0=feaserr;
      if(converged)
         return (feaserr<=epsi);
      //--- Stage 1: equality constrained quadratic programming
      //--- * treat active bound constraints as equality ones (constraint is considered
      //---   active when we are at the boundary, independently of the antigradient direction)
      //--- * calculate unrestricted Newton step to point XM (which may be infeasible)
      //---   calculate MaxStepLen = largest step in direction of XM which retains feasibility.
      //--- * perform bounded step from X to XN:
      //---   a) XN=XM                  (if XM is feasible)
      //---   b) XN=X-MaxStepLen*(XM-X) (otherwise)
      //--- * X := XN
      //--- * if XM (Newton step subject to currently active constraints) was feasible, goto Stage 2
      //--- * repeat Stage 1
      //--- NOTE 1: in order to solve constrained qudratic subproblem we will have to reorder
      //---         variables in such way that ones corresponding to inactive constraints will
      //---         be first, and active ones will be last in the list. CE and X are now
      //---                                                       [ xi ]
      //---         separated into two parts: CE = [CEi CEa], x = [    ], where CEi/Xi correspond
      //---                                                       [ xa ]
      //---         to INACTIVE constraints, and CEa/Xa correspond to the ACTIVE ones.
      //---         Now, instead of F=0.5*x'*(CE'*CE)*x - (b'*CE)*x + 0.5*b'*b, we have
      //---         F(xi) = 0.5*(CEi*xi,CEi*xi) + (CEa*xa-b,CEi*xi) + (0.5*CEa*xa-b,CEa*xa).
      //---         Here xa is considered constant, i.e. we optimize with respect to xi, leaving xa fixed.
      //---         We can solve it by performing SVD of CEi and calculating pseudoinverse of the
      //---         Hessian matrix. Of course, we do NOT calculate pseudoinverse explicitly - we
      //---         just use singular vectors to perform implicit multiplication by it.
      while(true)
        {
         //--- Calculate G - gradient subject to equality constraints,
         //--- multiply it by inverse of the Hessian diagonal to obtain initial
         //--- step vector.
         //--- Bound step subject to constraints which can be activated,
         //--- run Armijo search with increasing step size.
         //--- Search is terminated when feasibility error stops to decrease.
         //--- NOTE: it is important to test for "stops to decrease" instead
         //--- of "starts to increase" in order to correctly handle cases with
         //--- zero CE.
         armijobeststep=0.0;
         FeasibilityErrorGrad(ce,x,nmain,nslack,k,armijobestfeas,g,tmpk);
         for(i=0; i<nmain; i++)
           {
            if((havebndl[i] && x[i]==bndl[i]) || (havebndu[i] && x[i]==bndu[i]))
               g.Set(i,0.0);
           }
         for(i=0; i<nslack; i++)
            if(x[nmain+i]==0.0)
               g.Set(nmain+i,0.0);
         v=0.0;
         for(i=0; i<nmain+nslack; i++)
           {
            if(CMath::Sqr(colnorms[i])!=0.0)
               newtonstep.Set(i,-(g[i]/CMath::Sqr(colnorms[i])));
            else
               newtonstep.Set(i,0.0);
            v+=CMath::Sqr(newtonstep[i]);
           }
         if(v==0.0)
           {
            //--- Constrained gradient is zero, QP iterations are over
            break;
           }
         CalculateStepBound(x,newtonstep,1.0,bndl,havebndl,bndu,havebndu,nmain,nslack,vartofreeze,valtofreeze,maxsteplen);
         if(vartofreeze>=0 && maxsteplen==0.0)
           {
            //--- Can not perform step, QP iterations are over
            break;
           }
         if(vartofreeze>=0)
            armijostep=MathMin(1.0,maxsteplen);
         else
            armijostep=1;
         while(true)
           {
            xa=x.ToVector()+newtonstep*armijostep;
            EnforceBoundaryConstraints(xa,bndl,havebndl,bndu,havebndu,nmain,nslack);
            feaserr=FeasibilityError(ce,xa,nmain,nslack,k,tmpk);
            if(feaserr>=armijobestfeas)
               break;
            armijobestfeas=feaserr;
            armijobeststep=armijostep;
            armijostep=2.0*armijostep;
           }
         x+=newtonstep*armijobeststep+0;
         EnforceBoundaryConstraints(x,bndl,havebndl,bndu,havebndu,nmain,nslack);
         //--- Determine number of active and free constraints
         nactive=0;
         for(i=0; i<nmain; i++)
           {
            activeconstraints.Set(i,0);
            if(havebndl[i] && x[i]==bndl[i])
               activeconstraints.Set(i,1);
            if(havebndu[i] && x[i]==bndu[i])
               activeconstraints.Set(i,1);
            if(activeconstraints[i]>0.0)
               nactive++;
           }
         for(i=0; i<nslack; i++)
           {
            activeconstraints.Set(nmain+i,0);
            if(x[nmain+i]==0.0)
               activeconstraints.Set(nmain+i,1);
            if(activeconstraints[nmain+i]>0.0)
               nactive++;
           }
         nfree=nmain+nslack-nactive;
         if(nfree==0)
            break;
         qpits++;
         //--- Reorder variables: CE is reordered to PermCE, X is reordered to PermX
         CTSort::TagSortBuf(activeconstraints,nmain+nslack,p1,p2,buf);
         permce=ce;
         permx=x;
         for(j=0; j<nmain+nslack; j++)
           {
            if(p2[j]!=j)
              {
               idx0=p2[j];
               idx1=j;
               permce.SwapCols(idx0,idx1);
               permx.Swap(idx0,idx1);
              }
           }
         //--- Calculate (unprojected) gradient:
         //--- G(xi) = CEi'*(CEi*xi + CEa*xa - b)
         for(i=0; i<nfree; i++)
            g.Set(i,0);
         for(i=0; i<k; i++)
           {
            v=CAblasF::RDotVR(nmain+nslack,permx,permce,i);
            tmpk.Set(i,v-ce.Get(i,nmain+nslack));
           }
         for(i=0; i<k; i++)
           {
            v=tmpk[i];
            for(i_=0; i_<nfree; i_++)
               g.Add(i_,v*permce.Get(i,i_));
           }
         //--- Calculate Newton step using pseudoinverse PermCE:
         //---     F(xi)  = 0.5*xi'*H*xi + g'*xi    (Taylor decomposition)
         //---     XN     = -H^(-1)*g               (new point, solution of the QP subproblem)
         //---     H      = CEi'*CEi
         //---     H^(-1) can be calculated via QR or LQ decomposition (see below)
         //---     step   = -H^(-1)*g
         //--- NOTE: PermCE is destroyed after this block
         newtonstep=vector<double>::Zeros(nmain+nslack);
         if(k<=nfree)
           {
            //--- CEi    = L*Q
            //--- H      = Q'*L'*L*Q
            //--- inv(H) = Q'*inv(L)*inv(L')*Q
            //--- NOTE: we apply minor regularizing perturbation to diagonal of L,
            //---       which is equal to 10*K*Eps
            COrtFac::RMatrixLQ(permce,k,nfree,tau);
            COrtFac::RMatrixLQUnpackQ(permce,k,nfree,tau,k,q);
            vector<double> diag=permce.Diag();
            v=(MathAbs(diag)).Max();
            if(v!=0.0)
               permce.Diag(diag+10.0*k*CMath::m_machineepsilon*v);
            CAblas::RMatrixGemVect(k,nfree,1.0,q,0,0,0,g,0,0.0,tmpk,0);
            CAblas::RMatrixTrsVect(k,permce,0,0,false,false,1,tmpk,0);
            CAblas::RMatrixTrsVect(k,permce,0,0,false,false,0,tmpk,0);
            CAblas::RMatrixGemVect(nfree,k,-1.0,q,0,0,1,tmpk,0,0.0,newtonstep,0);
           }
         else
           {
            //--- CEi    = Q*R
            //--- H      = R'*R
            //--- inv(H) = inv(R)*inv(R')
            //--- NOTE: we apply minor regularizing perturbation to diagonal of R,
            //---       which is equal to 10*K*Eps
            COrtFac::RMatrixQR(permce,k,nfree,tau);
            vector<double> diag=permce.Diag();
            diag.Resize(nfree);
            v=(MathAbs(diag)).Max();
            if(v!=0.0)
               for(i=0; i<nfree; i++)
                 {
                  vv=10*nfree*CMath::m_machineepsilon*v;
                  if(diag[i]<0.0)
                     vv=-vv;
                  permce.Add(i,i,vv);
                 }
            for(i_=0; i_<nfree; i_++)
               newtonstep.Set(i_,-g[i_]);
            CAblas::RMatrixTrsVect(nfree,permce,0,0,true,false,1,newtonstep,0);
            CAblas::RMatrixTrsVect(nfree,permce,0,0,true,false,0,newtonstep,0);
           }
         //--- Post-reordering of Newton step
         for(j=nmain+nslack-1; j>=0; j--)
           {
            if(p2[j]!=j)
              {
               idx0=p2[j];
               idx1=j;
               newtonstep.Swap(idx0,idx1);
              }
           }
         //--- NewtonStep contains Newton step subject to active bound constraints.
         //--- Such step leads us to the minimizer of the equality constrained F,
         //--- but such minimizer may be infeasible because some constraints which
         //--- are inactive at the initial point can be violated at the solution.
         //--- Thus, we perform optimization in two stages:
         //--- a) perform bounded Newton step, i.e. step in the Newton direction
         //---    until activation of the first constraint
         //--- b) in case (MaxStepLen>0)and(MaxStepLen<1), perform additional iteration
         //---    of the Armijo line search in the rest of the Newton direction.
         CalculateStepBound(x,newtonstep,1.0,bndl,havebndl,bndu,havebndu,nmain,nslack,vartofreeze,valtofreeze,maxsteplen);
         if(vartofreeze>=0)
           {
            if(maxsteplen==0.0)
               //--- Activation of the constraints prevent us from performing step,
               //--- QP iterations are over
               break;
            v=MathMin(1.0,maxsteplen);
           }
         else
            v=1.0;
         xn=x.ToVector()+newtonstep*v;
         PostProcessBoundedStep(xn,x,bndl,havebndl,bndu,havebndu,nmain,nslack,vartofreeze,valtofreeze,v,maxsteplen);
         if(maxsteplen>0.0 && maxsteplen<1.0)
           {
            //--- Newton step was restricted by activation of the constraints,
            //--- perform Armijo iteration.
            //--- Initial estimate for best step is zero step. We try different
            //--- step sizes, from the 1-MaxStepLen (residual of the full Newton
            //--- step) to progressively smaller and smaller steps.
            armijobeststep=0.0;
            armijobestfeas=FeasibilityError(ce,xn,nmain,nslack,k,tmpk);
            armijostep=1-maxsteplen;
            for(j=0; j<maxarmijoruns; j++)
              {
               xa=xn.ToVector()+newtonstep*armijostep;
               EnforceBoundaryConstraints(xa,bndl,havebndl,bndu,havebndu,nmain,nslack);
               feaserr=FeasibilityError(ce,xa,nmain,nslack,k,tmpk);
               if(feaserr<armijobestfeas)
                 {
                  armijobestfeas=feaserr;
                  armijobeststep=armijostep;
                 }
               armijostep=0.5*armijostep;
              }
            xa=xn.ToVector()+newtonstep*armijobeststep;
            EnforceBoundaryConstraints(xa,bndl,havebndl,bndu,havebndu,nmain,nslack);
           }
         else
           {
            //--- Armijo iteration is not performed
            xa=xn;
           }
         stage1isover=(maxsteplen>=1.0 || maxsteplen==0.0);
         //--- Calculate feasibility errors for old and new X.
         //--- These quantinies are used for debugging purposes only.
         //--- However, we can leave them in release code because performance impact is insignificant.
         //--- Update X. Exit if needed.
         feasold=FeasibilityError(ce,x,nmain,nslack,k,tmpk);
         feasnew=FeasibilityError(ce,xa,nmain,nslack,k,tmpk);
         if(feasnew>=(feasold+infeasibilityincreasetolerance))
            break;
         x=xa;
         if(stage1isover)
            break;
        }
      //--- Stage 2: gradient projection algorithm (GPA)
      //--- * calculate feasibility error (with respect to linear equality constraints)
      //--- * calculate gradient G of F, project it into feasible area (G => PG)
      //--- * exit if norm(PG) is exactly zero or feasibility error is smaller than EpsC
      //--- * let XM be exact minimum of F along -PG (XM may be infeasible).
      //---   calculate MaxStepLen = largest step in direction of -PG which retains feasibility.
      //--- * perform bounded step from X to XN:
      //---   a) XN=XM              (if XM is feasible)
      //---   b) XN=X-MaxStepLen*PG (otherwise)
      //--- * X := XN
      //--- * stop after specified number of iterations or when no new constraints was activated
      //--- NOTES:
      //--- * grad(F) = (CE'*CE)*x - (b'*CE)^T
      //--- * CE[i] denotes I-th row of CE
      //--- * XM = X+stp*(-PG) where stp=(grad(F(X)),PG)/(CE*PG,CE*PG).
      //---   Here PG is a projected gradient, but in fact it can be arbitrary non-zero
      //---   direction vector - formula for minimum of F along PG still will be correct.
      werechangesinconstraints=false;
      for(gparuns=1; gparuns<=k; gparuns++)
        {
         //--- calculate feasibility error and G
         FeasibilityErrorGrad(ce,x,nmain,nslack,k,feaserr,g,tmpk);
         //--- project G, filter it (strip numerical noise)
         pg=g;
         ProjectGradientIntoBC(x,pg,bndl,havebndl,bndu,havebndu,nmain,nslack);
         FilterDirection(pg,x,bndl,havebndl,bndu,havebndu,s,nmain,nslack,1.0E-9);
         for(i=0; i<nmain+nslack; i++)
           {
            if(CMath::Sqr(colnorms[i])!=0.0)
               pg.Mul(i,1.0/CMath::Sqr(colnorms[i]));
            else
               pg.Set(i,0.0);
           }
         //--- Check GNorm and feasibility.
         //--- Exit when GNorm is exactly zero.
         pgnorm=MathSqrt(pg.Dot(pg));
         if(pgnorm==0.0)
           {
            result=(feaserr<=epsi);
            return(result);
           }
         //--- calculate planned step length
         vn=g.Dot(pg);
         CAblas::RMatrixGemVect(k,nmain+nslack,1.0,ce,0,0,0,pg,0,0.0,tmpk,0);
         vd=tmpk.Dot(tmpk);
         stp=vn/vd;
         //--- Calculate step bound.
         //--- Perform bounded step and post-process it
         CalculateStepBound(x,pg,-1.0,bndl,havebndl,bndu,havebndu,nmain,nslack,vartofreeze,valtofreeze,maxsteplen);
         if(vartofreeze>=0 && maxsteplen==0.0)
           {
            result=false;
            return(result);
           }
         if(vartofreeze>=0)
            v=MathMin(stp,maxsteplen);
         else
            v=stp;
         xn=x.ToVector()-pg*v;
         PostProcessBoundedStep(xn,x,bndl,havebndl,bndu,havebndu,nmain,nslack,vartofreeze,valtofreeze,v,maxsteplen);
         //--- update X
         //--- check stopping criteria
         werechangesinconstraints=(werechangesinconstraints || NumberOfChangedConstraints(xn,x,bndl,havebndl,bndu,havebndu,nmain,nslack)>0);
         x=xn;
         gpaits++;
         if(!werechangesinconstraints)
            break;
        }
      //--- Stage 3: decide to stop algorithm or not to stop
      //--- 1. we can stop when last GPA run did NOT changed constraints status.
      //---    It means that we've found final set of the active constraints even
      //---    before GPA made its run. And it means that Newton step moved us to
      //---    the minimum subject to the present constraints.
      //---    Depending on feasibility error, True or False is returned.
      feaserr=FeasibilityError(ce,x,nmain,nslack,k,tmpk);
      feaserr1=feaserr;
      if(feaserr1>=(feaserr0-infeasibilityincreasetolerance))
         badits++;
      else
         badits=0;
      if(feaserr<=epsi)
         itswithintolerance++;
      else
         itswithintolerance=0;
      if((!werechangesinconstraints || itswithintolerance>=maxitswithintolerance) || badits>=maxbadits)
        {
         result=(feaserr<=epsi);
         return(result);
        }
      itscount++;
     }
   return(true);
  }
//+------------------------------------------------------------------+
//| This function checks that input derivatives are right. First it  |
//| scales parameters DF0 and DF1 from segment [A; B] to [0; 1]. Then|
//| it builds Hermite spline and derivative of it in 0.5. Search     |
//| scale as Max(DF0, DF1, | F0 - F1 |). Right derivative has        |
//| to satisfy condition:                                            |
//|         | H - F | / S <= 0, 001, | H'-F' | / S <= 0, 001.        |
//| INPUT PARAMETERS:                                                |
//|   F0       -  function's value in X-TestStep point;              |
//|   DF0      -  derivative's value in X-TestStep point;            |
//|   F1       -  function's value in X+TestStep point;              |
//|   DF1      -  derivative's value in X+TestStep point;            |
//|   F        -  testing function's value;                          |
//|   DF       -  testing derivative's value;                        |
//|   Width    -  width of verification segment.                     |
//| RESULT:                                                          |
//|   If input derivatives is right then function returns true, else |
//|   function returns false.                                        |
//+------------------------------------------------------------------+
bool COptServ::DerivativeCheck(double f0,double df0,double f1,
                               double df1,double f,double df,
                               double width)
  {
//--- create variables
   double s=0;
   double h=0;
   double dh=0;
//--- Rescale input data to [0,1]
   df=width*df;
   df0=width*df0;
   df1=width*df1;
//--- Compute error scale, two sources are used:
//--- * magnitudes of derivatives and secants
//--- * magnitudes of input data times sqrt(machine_epsilon)
   s=0.0;
   s=MathMax(s,MathAbs(df0));
   s=MathMax(s,MathAbs(df1));
   s=MathMax(s,MathAbs(f1-f0));
   s=MathMax(s,MathSqrt(CMath::m_machineepsilon)*MathAbs(f0));
   s=MathMax(s,MathSqrt(CMath::m_machineepsilon)*MathAbs(f1));
//--- Compute H and dH/dX at the middle of interval
   h=0.5*(f0+f1)+0.125*(df0-df1);
   dh=1.5*(f1-f0)-0.250*(df0+df1);
//--- Check
   if(s!=0.0)
     {
      if((MathAbs(h-f)/s)>0.001 || (MathAbs(dh-df)/s)>0.001)
         return(false);
     }
   else
     {
      if((double)(h-f)!=0.0 || (double)(dh-df)!=0.0)
         return(false);
     }
//--- return result
   return(true);
  }
//+------------------------------------------------------------------+
//| Having quadratic target function                                 |
//|   f(x)=0.5*x'*A*x+b'*x+penaltyfactor*0.5*(C*x-b)'*(C*x-b)        |
//| and its parabolic model along direction D                        |
//|      F(x0 + alpha*D) = D2 * alpha ^ 2 + D1 * alpha               |
//| this function estimates numerical errors in the coefficients of  |
//| the model.                                                       |
//| It is important that this function does NOT calculate D1/D2 - it |
//| only estimates numerical errors introduced during evaluation and |
//| compares their magnitudes against magnitudes of numerical errors.|
//| As result, one of three outcomes is returned for each            |
//| coefficient:                                                     |
//|     *"true" coefficient is almost surely positive              |
//|     *"true" coefficient is almost surely negative              |
//|      * numerical errors in coefficient are so large that it can  |
//|        not be reliably distinguished from zero                   |
//| INPUT PARAMETERS:                                                |
//|   AbsASum     -  SUM( | A.Get(i,j) |)                               |
//|   AbsASum2    -  SUM(A.Get(i,j) ^ 2)                                |
//|   MB          -  max( | B |)                                     |
//|   MX          -  max( | X |)                                     |
//|   MD          -  max( | D |)                                     |
//|   D1          -  linear coefficient                              |
//|   D2          -  quadratic coefficient                           |
//| OUTPUT PARAMETERS:                                               |
//|   D1Est       -  estimate of D1 sign, accounting for possible    |
//|                  numerical errors:                               |
//|                  *>0   means "almost surely positive"(D1 > 0   |
//|                          and large)                              |
//|                  *<0   means "almost surely negative"(D1 < 0   |
//|                          and large)                              |
//|                  *=0   means "pessimistic estimate of numerical|
//|                          errors in D1 is larger than magnitude of|
//|                          D1 itself; it is impossible to reliably |
//|                          distinguish D1 from zero".              |
//|   D2Est       -  estimate of D2 sign,  accounting  for  possible |
//|                  numerical errors:                               |
//|                  *>0   means "almost surely positive"(D2 > 0   |
//|                          and large)                              |
//|                  *<0   means "almost surely negative"(D2 < 0   |
//|                          and large)                              |
//|                  *=0   means "pessimistic estimate of numerical|
//|                          errors in D2 is larger than magnitude of|
//|                          D2 itself; it is impossible to reliably |
//|                          distinguish D2 from zero".              |
//+------------------------------------------------------------------+
void COptServ::EstimateParabolicModel(double absasum,double absasum2,
                                      double mx,double mb,
                                      double md,double d1,
                                      double d2,int &d1est,
                                      int &d2est)
  {
//--- create variables
   double d1esterror=0;
   double d2esterror=0;
   double eps=0;
   double e1=0;
   double e2=0;
   d1est=0;
   d2est=0;
//--- Error estimates:
//--- * error in D1=d'*(A*x+b) is estimated as
//---   ED1 = eps*MAX_ABS(D)*(MAX_ABS(X)*ENORM(A)+MAX_ABS(B))
//--- * error in D2=0.5*d'*A*d is estimated as
//---   ED2 = eps*MAX_ABS(D)^2*ENORM(A)
//--- Here ENORM(A) is some pseudo-norm which reflects the way numerical
//--- error accumulates during addition. Two ways of accumulation are
//--- possible - worst case (errors always increase) and mean-case (errors
//--- may cancel each other). We calculate geometrical average of both:
//--- * ENORM_WORST(A) = SUM(|A[i,j]|)         error in N-term sum grows as O(N)
//--- * ENORM_MEAN(A)  = SQRT(SUM(A[i,j]^2))   error in N-term sum grows as O(sqrt(N))
//--- * ENORM(A)       = SQRT(ENORM_WORST(A),ENORM_MEAN(A))
   eps=4*CMath::m_machineepsilon;
   e1=eps*md*(mx*absasum+mb);
   e2=eps*md*(mx*MathSqrt(absasum2)+mb);
   d1esterror=MathSqrt(e1*e2);
   if(MathAbs(d1)<=d1esterror)
      d1est=0;
   else
      d1est=(int)MathSign(d1);
   e1=eps*md*md*absasum;
   e2=eps*md*md*MathSqrt(absasum2);
   d2esterror=MathSqrt(e1*e2);
   if(MathAbs(d2)<=d2esterror)
      d2est=0;
   else
      d2est=(int)MathSign(d2);
  }
//+------------------------------------------------------------------+
//| This function calculates inexact rank - K preconditioner for     |
//| Hessian matrix H = D + W'*C*W, where:                            |
//|   * H is a Hessian matrix, which is approximated by D / W / C    |
//|   * D is a diagonal matrix with positive entries                 |
//|   * W is a rank - K correction                                   |
//|   * C is a diagonal factor of rank - K correction                |
//| This preconditioner is inexact but fast - it requires O(N*K) time|
//| to be applied. Its main purpose - to be used in barrier / penalty|
//| / AUL methods, where ill - conditioning is created by combination|
//| of two factors:                                                  |
//|   * simple bounds on variables => ill - conditioned D            |
//|   * general barrier / penalty => correction W with large         |
//|     coefficient C(makes problem ill - conditioned) but W itself  |
//|     is well conditioned.                                         |
//| Preconditioner P is calculated by artificially constructing a set|
//| of BFGS updates which tries to reproduce behavior of H:          |
//|   * Sk = Wk(k - th row of W)                                     |
//|   * Yk = (D + Wk'*Ck*Wk)*Sk                                      |
//|   * Yk / Sk are reordered by ascending of C[k] * norm(Wk) ^ 2    |
//| Here we assume that rows of Wk are orthogonal or nearly          |
//| orthogonal, which allows us to have O(N * K + K^2) update instead|
//| of O(N * K ^ 2) one. Reordering of updates is essential for      |
//| having good performance on non-orthogonal problems (updates which|
//| do not add much of curvature are added first, and updates which  |
//| add very large eigenvalues are added last and override effect of |
//| the first updates).                                              |
//| On input this function takes direction S and components of H.    |
//| On output it returns inv(H) * S                                  |
//+------------------------------------------------------------------+
void COptServ::InexactLBFGSPreconditioner(CRowDouble &s,
                                          int n,
                                          CRowDouble &d,
                                          CRowDouble &c,
                                          CMatrixDouble &w,
                                          int k,
                                          CPrecBufLBFGS &buf)
  {
//--- create variables
   int    idx=0;
   int    i=0;
   int    j=0;
   double v=0;
   double v0=0;
   double v1=0;
   double vx=0;
   double vy=0;
   int    i_=0;
//--- allocate
   buf.m_norms.Resize(k);
   buf.m_alpha.Resize(k);
   buf.m_rho.Resize(k);
   buf.m_yk.Resize(k,n);
   buf.m_idx.Resize(k);
//--- Check inputs
   if(!CAp::Assert(d.Min()>0.0,"InexactLBFGSPreconditioner: D[]<=0"))
      return;
   if(!CAp::Assert(c.Min()>=0.0,"InexactLBFGSPreconditioner: C[]<0"))
      return;
//--- Reorder linear terms according to increase of second derivative.
//--- Fill Norms[] array.
   vector<double> temp;
   for(idx=0; idx<k; idx++)
     {
      temp=w[idx];
      temp.Resize(n);
      v=temp.Dot(temp);
      buf.m_norms.Set(idx,v*c[idx]);
      buf.m_idx.Set(idx,idx);
     }
   CTSort::TagSortFastI(buf.m_norms,buf.m_idx,buf.m_bufa,buf.m_bufb,k);
//--- Apply updates
   for(idx=0; idx<k; idx++)
     {
      //--- Select update to perform (ordered by ascending of second derivative)
      i=buf.m_idx[idx];
      //--- Calculate YK and Rho
      temp=w[i];
      temp.Resize(n);
      v0=temp.Dot(temp);
      v=v0*c[i];
      buf.m_yk.Row(i,(d+v)*temp);
      v=temp.Dot(buf.m_yk[i]+0);
      v1=CAblasF::RDotRR(n,buf.m_yk,i,buf.m_yk,i);
      if(v>0.0 && (v0*v1)>0.0 && (v/MathSqrt(v0*v1))>(n*10.0*CMath::m_machineepsilon))
         buf.m_rho.Set(i,1/v);
      else
         buf.m_rho.Set(i,0.0);
     }
   for(idx=k-1; idx>=0; idx--)
     {
      //--- Select update to perform (ordered by ascending of second derivative)
      i=buf.m_idx[idx];
      temp=w[i];
      temp.Resize(n);
      //--- Calculate Alpha[] according to L-BFGS algorithm
      //--- and update S[]
      v=s.Dot(temp);
      v*=buf.m_rho[i];
      buf.m_alpha.Set(i,v);
      s-=buf.m_yk[i]*v;
     }
   s/=d;
   for(idx=0; idx<k; idx++)
     {
      //--- Select update to perform (ordered by ascending of second derivative)
      i=buf.m_idx[idx];
      temp=w[i];
      temp.Resize(n);
      //--- Calculate Beta according to L-BFGS algorithm
      //--- and update S[]
      v=s.Dot(buf.m_yk[i]+0);
      v=buf.m_alpha[i]-buf.m_rho[i]*v;
      s+=temp*v;
     }
  }
//+------------------------------------------------------------------+
//| This function prepares exact low-rank preconditioner for Hessian |
//| matrix H = D + W'*C*W, where:                                    |
//|   * H is a Hessian matrix, which is approximated by D / W / C    |
//|   * D is a diagonal matrix with positive entries                 |
//|   * W is a rank - K correction                                   |
//|   * C is a diagonal factor of rank - K correction, positive      |
//|     semidefinite                                                 |
//| This preconditioner is exact but relatively slow -  it  requires |
//| O(N * K ^ 2) time to be prepared and O(N*K) time to be applied.  |
//| It is calculated with the help of Woodbury matrix identity.      |
//| It should be used as follows:                                    |
//|   * PrepareLowRankPreconditioner() call PREPARES data structure  |
//|   * subsequent calls to ApplyLowRankPreconditioner() APPLY       |
//|     preconditioner to user - specified search direction.         |
//+------------------------------------------------------------------+
void COptServ::PrepareLowRankPreconditioner(CRowDouble &d,
                                            CRowDouble &c,
                                            CMatrixDouble &w,
                                            int n,
                                            int k,
                                            CPrecBufLowRank &buf)
  {
//--- create variables
   int    i=0;
   int    j=0;
   double v=0;
   bool   b;
//--- Check inputs
   if(!CAp::Assert(n>0,__FUNCTION__+": N<=0"))
      return;
   if(!CAp::Assert(k>=0,__FUNCTION__+": N<=0"))
      return;
   if(!CAp::Assert(d.Min()>0.0,__FUNCTION__+": D[]<=0"))
      return;
   if(!CAp::Assert(c.Min()>=0.0,__FUNCTION__+": C[]<0"))
      return;
//--- Prepare buffer structure; skip zero entries of update.
   CApServ::RVectorSetLengthAtLeast(buf.m_d,n);
   CApServ::RMatrixSetLengthAtLeast(buf.m_v,k,n);
   CApServ::RVectorSetLengthAtLeast(buf.m_bufc,k);
   CApServ::RMatrixSetLengthAtLeast(buf.m_bufw,k+1,n);
   buf.m_n=n;
   buf.m_k=0;
   for(i=0; i<k; i++)
     {
      //--- Estimate magnitude of update row; skip zero rows (either W or C are zero)
      vector<double> temp=w[i];
      v=temp.Dot(temp)*c[i];
      if(v==0.0)
         continue;
      //--- check
      if(!CAp::Assert(v>0.0,__FUNCTION__+": internal error"))
         return;
      //--- Copy non-zero update to buffer
      buf.m_bufc.Set(buf.m_k,c[i]);
      buf.m_v.Row(buf.m_k,temp);
      buf.m_bufw.Row(buf.m_k,temp);
      buf.m_k++;
     }
//--- Reset K (for convenience)
   k=buf.m_k;
//--- Prepare diagonal factor; quick exit for K=0
   buf.m_d=d.Pow(-1.0)+0;
   if(k==0)
      return;
//--- Use Woodbury matrix identity
   buf.m_bufz=matrix<double>::Zeros(k,k);
   buf.m_bufc.Resize(k);
   buf.m_bufz.Diag(buf.m_bufc.Pow(-1.0)+0);
   buf.m_bufw.Row(k,d.Pow(-0.5)+0);
   for(i=0; i<k; i++)
      for(j=0; j<n; j++)
         buf.m_bufw.Mul(i,j,buf.m_bufw.Get(k,j));
   CAblas::RMatrixGemm(k,k,n,1.0,buf.m_bufw,0,0,0,buf.m_bufw,0,0,1,1.0,buf.m_bufz,0,0);
   b=CTrFac::SPDMatrixCholeskyRec(buf.m_bufz,0,k,true,buf.m_tmp);
//--- check
   if(!CAp::Assert(b,__FUNCTION__+": internal error (Cholesky failure)"))
      return;
   CAblas::RMatrixLeftTrsM(k,n,buf.m_bufz,0,0,true,false,1,buf.m_v,0,0);
   for(i=0; i<k; i++)
      for(j=0; j<n; j++)
         buf.m_v.Mul(i,j,buf.m_d[j]);
  }
//+------------------------------------------------------------------+
//| This function apply exact low - rank preconditioner prepared by  |
//| PrepareLowRankPreconditioner function (see its comments for more |
//| information).                                                    |
//+------------------------------------------------------------------+
void COptServ::ApplyLowRankPreconditioner(CRowDouble &s,
                                          CPrecBufLowRank &buf)
  {
//--- create variables
   int    i=0;
   int    j=0;
   double v=0;
   int    n=buf.m_n;
   int    k=buf.m_k;

   buf.m_tmp=vector<double>::Zeros(n);

   for(i=0; i<n; i++)
      buf.m_tmp.Set(i,buf.m_d[i]*s[i]);
   for(i=0; i<k; i++)
     {
      v=CAblasF::RDotVR(n,s,buf.m_v,i);
      for(j=0; j<n; j++)
         buf.m_tmp.Add(j,-buf.m_v.Get(i,j)*v);
     }
   for(i=0; i<n; i++)
      s.Set(i,buf.m_tmp[i]);
  }
//+------------------------------------------------------------------+
//| This subroutine initializes smoothness monitor at the beginning  |
//| of the optimization session. It requires variable scales to be   |
//| passed.                                                          |
//| It is possible to perform "dummy" initialization with N = K = 0. |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorInit(CSmoothnessMonitor &monitor,
                                     CRowDouble &s,
                                     int n,
                                     int k,
                                     bool checksmoothness)
  {
   monitor.m_n=n;
   monitor.m_k=k;
   monitor.m_checksmoothness=checksmoothness;
   monitor.m_linesearchspoiled=false;
   monitor.m_linesearchstarted=false;
   monitor.m_enqueuedcnt=0;
   monitor.m_sortedcnt=0;
   monitor.m_s=s;
   monitor.m_nonc0currentrating=0.0;
   monitor.m_nonc1currentrating=0.0;
   COptGuardApi::OptGuardInitInternal(monitor.m_rep,n,k);
   monitor.m_nonc0strrating=0.0;
   monitor.m_nonc0lngrating=-CMath::m_maxrealnumber;
   monitor.m_nonc0strrep.m_positive=false;
   monitor.m_nonc0lngrep.m_positive=false;
   monitor.m_nonc1test0strrating=0.0;
   monitor.m_nonc1test0lngrating=-CMath::m_maxrealnumber;
   monitor.m_nonc1test0strrep.m_positive=false;
   monitor.m_nonc1test0lngrep.m_positive=false;
   monitor.m_nonc1test1strrating=0.0;
   monitor.m_nonc1test1lngrating=-CMath::m_maxrealnumber;
   monitor.m_nonc1test1strrep.m_positive=false;
   monitor.m_nonc1test1lngrep.m_positive=false;
   monitor.m_badgradhasxj=false;
   monitor.m_rstateg0.ia.Resize(5);
   monitor.m_rstateg0.ra.Resize(4);
   monitor.m_rstateg0.stage=-1;
  }
//+------------------------------------------------------------------+
//| This subroutine starts line search                               |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorStartLineSearch(CSmoothnessMonitor &monitor,
                                                CRowDouble &x,
                                                CRowDouble &fi,
                                                CMatrixDouble &jac)
  {
//--- create variables
   int    i=0;
   int    j=0;
   double v=0;
   int    n=monitor.m_n;
   int    k=monitor.m_k;
//--- Skip if inactive or spoiled by NAN
   if(!monitor.m_checksmoothness)
      return;
   v=0;
   for(i=0; i<n; i++)
      v=0.5*v+x[i];
   for(i=0; i<k; i++)
      v=0.5*v+fi[i];
   for(i=0; i<k; i++)
      for(j=0; j<n; j++)
         v=0.5*v+jac.Get(i,j);
   if(!MathIsValidNumber(v))
     {
      monitor.m_linesearchspoiled=true;
      return;
     }
//--- Finalize previous line search
   if(monitor.m_enqueuedcnt>0)
      SmoothnessMonitorFinalizeLineSearch(monitor);
//--- Store initial point
   monitor.m_linesearchstarted=true;
   monitor.m_enqueuedcnt=1;
   monitor.m_enqueuedx=x;
   monitor.m_enqueuedfunc=fi;
   monitor.m_enqueuedjac=jac;
   monitor.m_enqueuedstp.Resize(monitor.m_enqueuedcnt);
   monitor.m_enqueuedx.Resize(monitor.m_enqueuedcnt*n);
   monitor.m_enqueuedfunc.Resize(monitor.m_enqueuedcnt*k);
   monitor.m_enqueuedjac.Resize(monitor.m_enqueuedcnt*k,n);
   monitor.m_enqueuedstp.Set(0,0.0);
//--- Initialize sorted representation
   monitor.m_sortedstp.Resize(1);
   monitor.m_sortedidx.Resize(1);
   monitor.m_sortedstp.Set(0,0.0);
   monitor.m_sortedidx.Set(0,0);
   monitor.m_sortedcnt=1;
  }
//+------------------------------------------------------------------+
//| This subroutine starts line search for a scalar function -       |
//| convenience wrapper for ....StartLineSearch() with unscaled      |
//| variables.                                                       |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorStartLineSearch1u(CSmoothnessMonitor &monitor,
                                                  CRowDouble &s,
                                                  CRowDouble &invs,
                                                  CRowDouble &x,
                                                  double f0,
                                                  CRowDouble &j0)
  {
//--- create variables
   int n=monitor.m_n;
   int k=monitor.m_k;
   if(!monitor.m_checksmoothness)
      return;
//--- check
   if(!CAp::Assert(k==1,__FUNCTION__+": K<>1"))
      return;

   monitor.m_xu=x*invs+0;
   monitor.m_xu.Resize(n);
   monitor.m_f0.Resize(1);
   monitor.m_j0.Resize(1,n);
   monitor.m_j0.Row(0,j0*s+0);
   monitor.m_f0.Set(0,f0);
   SmoothnessMonitorStartLineSearch(monitor,monitor.m_xu,monitor.m_f0,monitor.m_j0);
  }
//+------------------------------------------------------------------+
//| This subroutine enqueues one more trial point                    |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorEnqueuePoint(CSmoothnessMonitor &monitor,
                                             CRowDouble &d,
                                             double stp,
                                             CRowDouble &x,
                                             CRowDouble &fi,
                                             CMatrixDouble &jac)
  {
//--- create variables
   int    i=0;
   int    j=0;
   double v=0;
   int    enqueuedcnt=0;
   int    sortedcnt=0;
   bool   hasduplicates;
   int    funcidx=0;
   int    stpidx=0;
   double f0=0;
   double f1=0;
   double f2=0;
   double f3=0;
   double f4=0;
   double noise0=0;
   double noise1=0;
   double noise2=0;
   double noise3=0;
   double rating=0;
   double lipschitz=0;
   double nrm=0;
   double lengthrating=0;
   int    n=monitor.m_n;
   int    k=monitor.m_k;
//--- Skip if inactive or spoiled by NAN
   if(!monitor.m_checksmoothness || monitor.m_linesearchspoiled || !monitor.m_linesearchstarted)
      return;
   v=stp;
   for(i=0; i<n; i++)
      v=0.5*v+x[i];
   for(i=0; i<n; i++)
      v=0.5*v+d[i];
   for(i=0; i<k; i++)
      v=0.5*v+fi[i];
   for(i=0; i<k; i++)
      for(j=0; j<n; j++)
         v=0.5*v+jac.Get(i,j);
   if(!MathIsValidNumber(v))
     {
      monitor.m_linesearchspoiled=true;
      return;
     }
//--- Enqueue
   monitor.m_enqueuedcnt++;
   enqueuedcnt=monitor.m_enqueuedcnt;
   monitor.m_dcur=d;
   monitor.m_dcur.Resize(n);
   monitor.m_enqueuedstp.Resize(enqueuedcnt);
   monitor.m_enqueuedx.Resize(enqueuedcnt*n);
   monitor.m_enqueuedfunc.Resize(enqueuedcnt*k);
   monitor.m_enqueuedjac.Resize(enqueuedcnt*k,n);
   monitor.m_enqueuedstp.Set(enqueuedcnt-1,stp);
   for(j=0; j<n; j++)
      monitor.m_enqueuedx.Set((enqueuedcnt-1)*n+j,x[j]);
   for(i=0; i<k; i++)
      monitor.m_enqueuedfunc.Set((enqueuedcnt-1)*k+i,fi[i]);
   for(i=0; i<k; i++)
      for(j=0; j<n; j++)
         monitor.m_enqueuedjac.Set((enqueuedcnt-1)*k+i,j,jac.Get(i,j));
//--- Update sorted representation: insert to the end, reorder
   sortedcnt=monitor.m_sortedcnt;
   hasduplicates=false;
   for(i=0; i<sortedcnt; i++)
      hasduplicates=(hasduplicates || monitor.m_sortedstp[i]==stp);
   if(!hasduplicates)
     {
      monitor.m_sortedcnt++;
      sortedcnt=monitor.m_sortedcnt;
      monitor.m_sortedstp.Resize(sortedcnt);
      monitor.m_sortedidx.Resize(sortedcnt);
      monitor.m_sortedstp.Set(sortedcnt-1,stp);
      monitor.m_sortedidx.Set(sortedcnt-1,enqueuedcnt-1);
      for(i=sortedcnt-2; i>=0; i--)
        {
         if(monitor.m_sortedstp[i]<=monitor.m_sortedstp[i+1])
            break;
         monitor.m_sortedstp.Swap(i,i+1);
         monitor.m_sortedidx.Swap(i,i+1);
        }
     }
//--- Scan sorted representation, check for C0 and C1 continuity
//--- violations.
   monitor.m_f.Resize(sortedcnt);
   monitor.m_g.Resize(sortedcnt*n);
   for(funcidx=0; funcidx<k; funcidx++)
     {
      //--- Fetch current function and its gradient to the contiguous storage
      for(i=0; i<sortedcnt; i++)
        {
         monitor.m_f.Set(i,monitor.m_enqueuedfunc[monitor.m_sortedidx[i]*k+funcidx]);
         for(j=0; j<n; j++)
            monitor.m_g.Set(i*n+j,monitor.m_enqueuedjac.Get(monitor.m_sortedidx[i]*k+funcidx,j));
        }
      //--- Check C0 continuity.
      //--- The basis approach is that we find appropriate candidate point
      //--- (either a local minimum along the line - for target; or an interval
      //--- where function sign is changed - for constraints), calculate left
      //--- and right estimates of the Lipschitz constant (slopes between points
      //--- #0 and #1, #2 and #3), and then calculate slope between points #1 and
      //--- #2 and compare it with left/right estimates.
      //--- The actual approach is a bit more complex to account for different
      //--- sources of numerical noise and different false positive scenarios.
      if(funcidx==0)
        {
         for(stpidx=0; stpidx<sortedcnt-3; stpidx++)
           {
            f0=monitor.m_f[stpidx+0];
            f1=monitor.m_f[stpidx+1];
            f2=monitor.m_f[stpidx+2];
            f3=monitor.m_f[stpidx+3];
            noise0=m_ognoiselevelf*MathMax(MathAbs(f0),1.0);
            noise1=m_ognoiselevelf*MathMax(MathAbs(f1),1.0);
            noise2=m_ognoiselevelf*MathMax(MathAbs(f2),1.0);
            noise3=m_ognoiselevelf*MathMax(MathAbs(f3),1.0);
            if(!(f1<f0+(noise0+noise1) && f1<f2))
               continue;
            TestC0Continuity(f0,f1,f2,f3,noise0,noise1,noise2,noise3,monitor.m_sortedstp[stpidx+1]-monitor.m_sortedstp[stpidx+0],monitor.m_sortedstp[stpidx+2]-monitor.m_sortedstp[stpidx+1],monitor.m_sortedstp[stpidx+3]-monitor.m_sortedstp[stpidx+2],false,rating,lipschitz);
            if(rating>m_ogminrating0)
              {
               //--- Store to total report
               monitor.m_rep.m_nonc0suspected=true;
               monitor.m_rep.m_nonc0test0positive=true;
               if(rating>monitor.m_nonc0currentrating)
                 {
                  monitor.m_nonc0currentrating=rating;
                  monitor.m_rep.m_nonc0lipschitzc=lipschitz;
                  monitor.m_rep.m_nonc0fidx=funcidx;
                 }
               //--- Store to "strongest" report
               if(rating>monitor.m_nonc0strrating)
                 {
                  monitor.m_nonc0strrating=rating;
                  monitor.m_nonc0strrep.m_positive=true;
                  monitor.m_nonc0strrep.m_fidx=funcidx;
                  monitor.m_nonc0strrep.m_n=n;
                  monitor.m_nonc0strrep.m_cnt=sortedcnt;
                  monitor.m_nonc0strrep.m_stpidxa=stpidx+0;
                  monitor.m_nonc0strrep.m_stpidxb=stpidx+3;
                  monitor.m_nonc0strrep.m_x0.Resize(n);
                  monitor.m_nonc0strrep.m_d.Resize(n);
                  for(i=0; i<n; i++)
                    {
                     monitor.m_nonc0strrep.m_x0.Set(i,monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i]);
                     monitor.m_nonc0strrep.m_d.Set(i,monitor.m_dcur[i]);
                    }
                  monitor.m_nonc0strrep.m_stp=monitor.m_sortedstp;
                  monitor.m_nonc0strrep.m_f=monitor.m_f;
                  monitor.m_nonc0strrep.m_stp.Resize(sortedcnt);
                  monitor.m_nonc0strrep.m_f.Resize(sortedcnt);
                 }
               //--- Store to "longest" report
               nrm=0;
               for(i=0; i<n; i++)
                 {
                  nrm+=CMath::Sqr(monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i] -
                                    monitor.m_enqueuedx[monitor.m_sortedidx[sortedcnt-1]*n+i]);
                 }
               nrm=MathSqrt(nrm);
               nrm=MathMin(nrm,1.0);
               nrm=CApServ::Coalesce(nrm,CMath::m_machineepsilon);
               lengthrating=sortedcnt+MathLog(nrm)/MathLog(100);
               if(lengthrating>monitor.m_nonc0lngrating)
                 {
                  monitor.m_nonc0lngrating=lengthrating;
                  monitor.m_nonc0lngrep.m_positive=true;
                  monitor.m_nonc0lngrep.m_fidx=funcidx;
                  monitor.m_nonc0lngrep.m_n=n;
                  monitor.m_nonc0lngrep.m_cnt=sortedcnt;
                  monitor.m_nonc0lngrep.m_stpidxa=stpidx+0;
                  monitor.m_nonc0lngrep.m_stpidxb=stpidx+3;
                  monitor.m_nonc0lngrep.m_x0.Resize(n);
                  monitor.m_nonc0lngrep.m_d.Resize(n);
                  for(i=0; i<n; i++)
                    {
                     monitor.m_nonc0lngrep.m_x0.Set(i,monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i]);
                     monitor.m_nonc0lngrep.m_d.Set(i,monitor.m_dcur[i]);
                    }
                  monitor.m_nonc0lngrep.m_stp=monitor.m_sortedstp;
                  monitor.m_nonc0lngrep.m_f=monitor.m_f;
                  monitor.m_nonc0lngrep.m_stp.Resize(sortedcnt);
                  monitor.m_nonc0lngrep.m_f.Resize(sortedcnt);
                 }
              }
           }
        }
      //--- C1 continuity test #0
      for(stpidx=0; stpidx<sortedcnt-6; stpidx++)
        {
         //--- Fetch function values
         f2=monitor.m_f[stpidx+2];
         f3=monitor.m_f[stpidx+3];
         f4=monitor.m_f[stpidx+4];
         noise2=m_ognoiselevelf*MathMax(MathAbs(f2),1.0);
         noise3=m_ognoiselevelf*MathMax(MathAbs(f3),1.0);
         //--- Decide whether we want to test this interval or not; for target
         //--- function we test intervals around minimum, for constraints we
         //--- test intervals of sign change.
         if(funcidx==0)
           {
            //--- Skip if not minimum
            if(!(f3<f2+(noise2+noise3) && f3<f4))
               continue;
           }
         else
           {
            //--- Skip if sign does not change
            if(MathSign(f2*f4)>0)
               continue;
           }
         C1ContinuityTest0(monitor,funcidx,stpidx+0,sortedcnt);
         C1ContinuityTest0(monitor,funcidx,stpidx+1,sortedcnt);
        }
      //--- C1 continuity test #1
      for(stpidx=0; stpidx<sortedcnt-3; stpidx++)
        {
         //--- Fetch function values from the interval being tested
         f0=monitor.m_f[stpidx+0];
         f1=monitor.m_f[stpidx+1];
         f2=monitor.m_f[stpidx+2];
         f3=monitor.m_f[stpidx+3];
         noise0=m_ognoiselevelf*MathMax(MathAbs(f0),1.0);
         noise1=m_ognoiselevelf*MathMax(MathAbs(f1),1.0);
         noise2=m_ognoiselevelf*MathMax(MathAbs(f2),1.0);
         noise3=m_ognoiselevelf*MathMax(MathAbs(f3),1.0);
         //--- Decide whether we want to test this interval or not; for target
         //--- function we test intervals around minimum, for constraints we
         //--- test intervals of sign change.
         if(funcidx==0)
           {
            //--- Skip if not minimum
            if(!(f1<f0+(noise0+noise1) && f2<f3+noise2+noise3))
               continue;
           }
         else
           {
            //--- Skip if sign does not change
            if(MathSign(f0*f3)>0)
               continue;
           }
         C1ContinuityTest1(monitor,funcidx,stpidx,sortedcnt);
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine enqueues one more trial point for a task with    |
//| scalar function with unscaled variables - a convenience wrapper  |
//| for more general EnqueuePoint()                                  |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorEnqueuePoint1u(CSmoothnessMonitor &monitor,
                                               CRowDouble &s,
                                               CRowDouble &invs,
                                               CRowDouble &d,
                                               double stp,
                                               CRowDouble &x,
                                               double f0,
                                               CRowDouble &j0)
  {
//--- create variables
   int n=monitor.m_n;
   int k=monitor.m_k;
   if(!monitor.m_checksmoothness)
      return;
//--- check
   if(!CAp::Assert(k==1,__FUNCTION__+": K<>1"))
      return;

   monitor.m_xu=x*invs+0;
   monitor.m_du=d*invs+0;
   monitor.m_xu.Resize(n);
   monitor.m_du.Resize(n);
   monitor.m_f0.Resize(1);
   monitor.m_j0.Resize(1,n);
   monitor.m_j0.Row(0,j0*s+0);
   monitor.m_f0.Set(0,f0);
   SmoothnessMonitorEnqueuePoint(monitor,monitor.m_du,stp,monitor.m_xu,monitor.m_f0,monitor.m_j0);
  }
//+------------------------------------------------------------------+
//| This subroutine finalizes line search                            |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorFinalizeLineSearch(CSmoothnessMonitor &monitor)
  {
//--- As for now - nothing to be done.
  }
//+------------------------------------------------------------------+
//| This function starts aggressive probing for a range of step      |
//| lengths [0, StpMax].                                             |
//| This function stores NValues values per step, with the first one |
//| (index 0) value being "primary" one (target function / merit     |
//| function) and the rest being supplementary ones.                 |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorStartProbing(CSmoothnessMonitor &monitor,
                                             double stpmax,
                                             int nvalues,
                                             double stepscale)
  {
//--- check
   if(!CAp::Assert(MathIsValidNumber(stpmax) && stpmax>0.0,__FUNCTION__+": StpMax<=0"))
      return;
   if(!CAp::Assert(nvalues>=1,__FUNCTION__+": NValues<1"))
      return;
   if(!CAp::Assert(MathIsValidNumber(stepscale) && (double)(stepscale)>=0.0,__FUNCTION__+": StepScale<0"))
      return;

   monitor.m_probingnvalues=nvalues;
   monitor.m_probingnstepsstored=0;
   monitor.m_probingstepmax=stpmax;
   monitor.m_probingstepscale=stepscale;
   monitor.m_probingf.Resize(nvalues);
   monitor.m_probingrcomm.ia.Resize(2+1);
   monitor.m_probingrcomm.ra.Resize(3+1);
   monitor.m_probingrcomm.stage=-1;
  }
//+------------------------------------------------------------------+
//| This function performs aggressive probing.                       |
//| After each call it returns step to evaluate in Monitor.ProbingStp|
//| Load values being probed into Monitor.ProbingF and continue      |
//| iteration.                                                       |
//| Monitor.ProbingF[0] is a special value which is used to guide    |
//| probing process towards discontinuities and nonsmooth points.    |
//+------------------------------------------------------------------+
bool COptServ::SmoothnessMonitorProbe(CSmoothnessMonitor &monitor)
  {
//--- create variables
   int    i=0;
   int    j=0;
   int    idx=0;
   double vlargest=0;
   double v=0;
   double v0=0;
   double v1=0;
//--- Reverse communication preparations
//--- I know it looks ugly, but it works the same way
//--- anywhere from C++ to Python.
//--- This code initializes locals by:
//--- * random values determined during code
//---   generation - on first subroutine call
//--- * values from previous call - on subsequent calls
   if(monitor.m_probingrcomm.stage>=0)
     {
      i=monitor.m_probingrcomm.ia[0];
      j=monitor.m_probingrcomm.ia[1];
      idx=monitor.m_probingrcomm.ia[2];
      vlargest=monitor.m_probingrcomm.ra[0];
      v=monitor.m_probingrcomm.ra[1];
      v0=monitor.m_probingrcomm.ra[2];
      v1=monitor.m_probingrcomm.ra[3];
     }
   else
     {
      i=359;
      j=-58;
      idx=-919;
      vlargest=-909;
      v=81;
      v0=255;
      v1=74;
     }
   if(monitor.m_probingrcomm.stage==0)
     {
      monitor.m_probingvalues.Row(monitor.m_probingnstepsstored,monitor.m_probingf);
      monitor.m_probingslopes.Row(monitor.m_probingnstepsstored,vector<double>::Zeros(monitor.m_probingnvalues));
      monitor.m_probingnstepsstored++;
      //--- Resort
      for(j=monitor.m_probingnstepsstored-1; j>=1; j--)
        {
         if(monitor.m_probingsteps[j-1]<=monitor.m_probingsteps[j])
            break;
         monitor.m_probingsteps.Swap(j-1,j);
         monitor.m_probingvalues.SwapRows(j-1,j);
        }
      i++;
     }
   else
      //--- Routine body
      i=0;
   if(i>40)
      return(false);
//--- Increase storage size
   monitor.m_probingsteps.Resize(monitor.m_probingnstepsstored+1);
   monitor.m_probingvalues.Resize(monitor.m_probingnstepsstored+1,monitor.m_probingnvalues);
   monitor.m_probingslopes.Resize(monitor.m_probingnstepsstored+1,monitor.m_probingnvalues);
//--- Determine probing step length, save step to the end of the storage
   if(i<=10)
     {
      //--- First 11 steps are performed over equidistant grid
      monitor.m_probingstp=(double)i/10.0*monitor.m_probingstepmax;
     }
   else
     {
      //--- Subsequent steps target either points with maximum change in F[0]
      //--- (search for discontinuity) or maximum change in slope of F[0] (search
      //--- for nonsmoothness)
      //--- check
      if(!CAp::Assert(monitor.m_probingnstepsstored>=3,__FUNCTION__+": critical integrity check failed"))
         return(false);
      if(i%2==0)
        {
         //--- Target interval with maximum change in F[0]
         idx=-1;
         vlargest=0;
         for(j=0; j<=monitor.m_probingnstepsstored-2; j++)
           {
            v=MathAbs(monitor.m_probingvalues.Get(j+1,0)-monitor.m_probingvalues.Get(j,0));
            if(idx<0 || v>vlargest)
              {
               idx=j;
               vlargest=v;
              }
           }
         monitor.m_probingstp=0.5*(monitor.m_probingsteps[idx]+monitor.m_probingsteps[idx+1]);
        }
      else
        {
         //--- Target interval [J,J+2] with maximum change in slope of F[0], select
         //--- subinterval [J,J+1] or [J+1,J+2] with maximum length.
         idx=-1;
         vlargest=0;
         for(j=0; j<=monitor.m_probingnstepsstored-3; j++)
           {
            v0=(monitor.m_probingvalues.Get(j+1,0)-monitor.m_probingvalues.Get(j+0,0))/(monitor.m_probingsteps[j+1]-monitor.m_probingsteps[j+0]+CMath::m_machineepsilon);
            v1=(monitor.m_probingvalues.Get(j+2,0)-monitor.m_probingvalues.Get(j+1,0))/(monitor.m_probingsteps[j+2]-monitor.m_probingsteps[j+1]+CMath::m_machineepsilon);
            v=MathAbs(v0-v1);
            if(idx<0 || v>vlargest)
              {
               idx=j;
               vlargest=v;
              }
           }
         if((double)(monitor.m_probingsteps[idx+2]-monitor.m_probingsteps[idx+1])>(double)(monitor.m_probingsteps[idx+1]-monitor.m_probingsteps[idx+0]))
           {
            monitor.m_probingstp=0.5*(monitor.m_probingsteps[idx+2]+monitor.m_probingsteps[idx+1]);
           }
         else
           {
            monitor.m_probingstp=0.5*(monitor.m_probingsteps[idx+1]+monitor.m_probingsteps[idx+0]);
           }
        }
     }
   monitor.m_probingsteps.Set(monitor.m_probingnstepsstored,monitor.m_probingstp);
//--- Retrieve user values
   monitor.m_probingrcomm.stage=0;
//--- Saving State
   monitor.m_probingrcomm.ia.Set(0,i);
   monitor.m_probingrcomm.ia.Set(1,j);
   monitor.m_probingrcomm.ia.Set(2,idx);
   monitor.m_probingrcomm.ra.Set(0,vlargest);
   monitor.m_probingrcomm.ra.Set(1,v);
   monitor.m_probingrcomm.ra.Set(2,v0);
   monitor.m_probingrcomm.ra.Set(3,v1);
   return(true);
  }
//+------------------------------------------------------------------+
//| This function prints probing results to trace log.               |
//| Tracing is performed using fixed width for all columns, so you   |
//| may print a header before printing trace - and reasonably expect |
//| that its width will match that of the trace. This function       |
//| promises that it wont change  trace output format without        |
//| introducing breaking changes into its signature.                 |
//| NOTE: this function ALWAYS tries to print results; it is caller's|
//| responsibility to decide whether he needs tracing or not.        |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorTraceProbingResults(CSmoothnessMonitor &monitor)
  {
//--- create variables
   int    i=0;
   int    j=0;
   double steplen=0;
//--- Compute slopes
   for(i=0; i<=monitor.m_probingnstepsstored-2; i++)
     {
      for(j=0; j<=monitor.m_probingnvalues-1; j++)
        {
         steplen=(monitor.m_probingsteps[i+1]-monitor.m_probingsteps[i]+100.0*CMath::m_machineepsilon)*(monitor.m_probingstepscale+CMath::m_machineepsilon);
         monitor.m_probingslopes.Set(i,j,(monitor.m_probingvalues.Get(i+1,j)-monitor.m_probingvalues.Get(i,j))/steplen);
        }
     }
   if(monitor.m_probingnstepsstored>=1)
      for(j=0; j<monitor.m_probingnvalues; j++)
         monitor.m_probingslopes.Set(monitor.m_probingnstepsstored-1,j,monitor.m_probingslopes.Get(MathMax(monitor.m_probingnstepsstored-2,0),j));
//--- Print to trace log
   CAp::Trace("*** ----------");
   for(j=0; j<monitor.m_probingnvalues; j++)
      CAp::Trace("-------------------------");
   CAp::Trace("\n");
   for(i=0; i<monitor.m_probingnstepsstored; i++)
     {
      CAp::Trace(StringFormat("*** | %0.4f |",monitor.m_probingsteps[i]));
      for(j=0; j<monitor.m_probingnvalues; j++)
         CAp::Trace(StringFormat(" %11.3E %10.2E |",monitor.m_probingvalues.Get(i,j) - monitor.m_probingvalues.Get(0,j),monitor.m_probingslopes.Get(i,j)));
      CAp::Trace("\n");
     }
   CAp::Trace("*** ----------");
   for(j=0; j<monitor.m_probingnvalues; j++)
      CAp::Trace("-------------------------");
   CAp::Trace("\n");
  }
//+------------------------------------------------------------------+
//| This subroutine tells monitor to output trace Info.              |
//| If CallerSuggestsTrace = True, monitor  ALWAYS  prints  trace,   |
//| even if no suspicions were raised during optimization. If        |
//| CallerSuggestsTrace = False, the monitor will print trace only   |
//| if:                                                              |
//|   * trace was requested by trace tag 'OPTGUARD' AND suspicious   |
//|     points were found during optimization                        |
//|   * trace was requested by trace tag 'OPTGUARD.ALWAYS' - always  |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorTraceStatus(CSmoothnessMonitor &monitor,
                                            bool callersuggeststrace)
  {
//--- create variables
   bool   needreport;
   bool   needxdreport;
   bool   suspicionsraised;
   int    i=0;
   double slope=0;
//--- Do we need trace report?
   suspicionsraised=(monitor.m_rep.m_nonc0suspected || monitor.m_rep.m_nonc1suspected || monitor.m_rep.m_badgradsuspected);
   needreport=false;
   needreport=needreport || callersuggeststrace;
   needreport=needreport || CAp::IsTraceEnabled("OPTGUARD.ALWAYS");
   needreport=needreport || (CAp::IsTraceEnabled("OPTGUARD") && suspicionsraised);
   if(!needreport)
      return;
   needxdreport=needreport && CAp::IsTraceEnabled("OPTIMIZERS.X");

   CAp::Trace("\n");
   CAp::Trace("////////////////////////////////////////////////////////////////////////////////////////////////////\n");
   CAp::Trace("//--- OPTGUARD INTEGRITY CHECKER REPORT                                                              //\n");
   CAp::Trace("////////////////////////////////////////////////////////////////////////////////////////////////////\n");
   if(!suspicionsraised)
     {
      CAp::Trace("> no discontinuity/nonsmoothness/bad-gradient suspicions were raised during optimization\n");
      return;
     }
   if(monitor.m_rep.m_nonc0suspected)
      CAp::Trace("> [WARNING] suspected discontinuity (aka C0-discontinuity)\n");
   if(monitor.m_rep.m_nonc1suspected)
      CAp::Trace("> [WARNING] suspected nonsmoothness (aka C1-discontinuity)\n");
   CAp::Trace("> printing out test reports...\n");
   if(monitor.m_rep.m_nonc0suspected && monitor.m_rep.m_nonc0test0positive)
     {
      CAp::Trace("> printing out discontinuity test #0 report:\n");
      CAp::Trace("*** -------------------------------------------------------\n");
      CAp::Trace("*** | Test #0 for discontinuity was triggered  (this test |\n");
      CAp::Trace("*** | analyzes changes in function values). See below for |\n");
      CAp::Trace("*** | detailed Info:                                      |\n");
      CAp::Trace(StringFormat("*** | * function index:       %10.m_d",monitor.m_nonc0lngrep.m_fidx));
      if(monitor.m_nonc0lngrep.m_fidx==0)
         CAp::Trace(" (target)         |\n");
      else
         CAp::Trace(" (constraint)     |\n");
      CAp::Trace(StringFormat("*** | * F() Lipschitz const:  %10.2E                  |\n",monitor.m_rep.m_nonc0lipschitzc));
      CAp::Trace("*** | Printing out log of suspicious line search XK+Stp*D |\n");
      CAp::Trace("*** | Look for abrupt changes in slope.                   |\n");
      if(!needxdreport)
        {
         CAp::Trace("*** | NOTE: XK and D are  not printed  by default. If you |\n");
         CAp::Trace("*** |       need them,add trace tag OPTIMIZERS.X         |\n");
        }
      CAp::Trace("*** -------------------------------------------------------\n");
      CAp::Trace("*** |  step along D   |     delta F     |      slope      |\n");
      CAp::Trace("*** ------------------------------------------------------|\n");
      for(i=0; i<=monitor.m_nonc0lngrep.m_cnt-1; i++)
        {
         slope=monitor.m_nonc0lngrep.m_f[MathMin(i+1,monitor.m_nonc0lngrep.m_cnt-1)]-monitor.m_nonc0lngrep.m_f[i];
         slope=slope/(1.0e-15+monitor.m_nonc0lngrep.m_stp[MathMin(i+1,monitor.m_nonc0lngrep.m_cnt-1)]-monitor.m_nonc0lngrep.m_stp[i]);
         CAp::Trace(StringFormat("*** |  %13.5E  |  %13.5E  |   %13.5E   |",monitor.m_nonc0lngrep.m_stp[i],monitor.m_nonc0lngrep.m_f[i] - monitor.m_nonc0lngrep.m_f[0],slope));
         if(i>=monitor.m_nonc0lngrep.m_stpidxa && i<=monitor.m_nonc0lngrep.m_stpidxb)
           {
            CAp::Trace(" <---");
           }
         CAp::Trace("\n");
        }
      CAp::Trace("*** ------------------------------------------------------|\n");
      if(needxdreport)
        {
         CAp::Trace("*** > printing raw variables\n");
         CAp::Trace("*** XK = ");
         CApServ::TraceVectorUnscaledUnshiftedAutopRec(monitor.m_nonc0lngrep.m_x0,monitor.m_n,monitor.m_s,true,monitor.m_s,false);
         CAp::Trace("\n");
         CAp::Trace("*** D  = ");
         CApServ::TraceVectorUnscaledUnshiftedAutopRec(monitor.m_nonc0lngrep.m_d,monitor.m_n,monitor.m_s,true,monitor.m_s,false);
         CAp::Trace("\n");
         CAp::Trace("*** > printing scaled variables (values are divided by user-specified scales)\n");
         CAp::Trace("*** XK = ");
         CApServ::TraceVectorAutopRec(monitor.m_nonc0lngrep.m_x0,0,monitor.m_n);
         CAp::Trace("\n");
         CAp::Trace("*** D  = ");
         CApServ::TraceVectorAutopRec(monitor.m_nonc0lngrep.m_d,0,monitor.m_n);
         CAp::Trace("\n");
        }
     }
   if(monitor.m_rep.m_nonc1suspected && monitor.m_rep.m_nonc1test0positive)
     {
      CAp::Trace("> printing out nonsmoothness test #0 report:\n");
      CAp::Trace("*** -------------------------------------------------------\n");
      CAp::Trace("*** | Test #0 for nonsmoothness was triggered  (this test |\n");
      CAp::Trace("*** | analyzes changes in  function  values  and  ignores |\n");
      CAp::Trace("*** | gradient Info). See below for detailed Info:        |\n");
      CAp::Trace(StringFormat("*** | * function index:         %10.m_d",monitor.m_nonc1test0lngrep.m_fidx));
      if(monitor.m_nonc1test0lngrep.m_fidx==0)
        {
         CAp::Trace(" (target)       |\n");
        }
      else
        {
         CAp::Trace(" (constraint)   |\n");
        }
      CAp::Trace(StringFormat("*** | * dF/dX Lipschitz const:  %10.2E                |\n",monitor.m_rep.m_nonc1lipschitzc));
      CAp::Trace("*** | Printing out log of suspicious line search XK+Stp*D |\n");
      CAp::Trace("*** | Look for abrupt changes in slope.                   |\n");
      if(!needxdreport)
        {
         CAp::Trace("*** | NOTE: XK and D are  not printed  by default. If you |\n");
         CAp::Trace("*** |       need them,add trace tag OPTIMIZERS.X         |\n");
        }
      CAp::Trace("*** -------------------------------------------------------\n");
      CAp::Trace("*** |  step along D   |     delta F     |      slope      |\n");
      CAp::Trace("*** ------------------------------------------------------|\n");
      for(i=0; i<=monitor.m_nonc1test0lngrep.m_cnt-1; i++)
        {
         slope=monitor.m_nonc1test0lngrep.m_f[MathMin(i+1,monitor.m_nonc1test0lngrep.m_cnt-1)]-monitor.m_nonc1test0lngrep.m_f[i];
         slope=slope/(1.0e-15+monitor.m_nonc1test0lngrep.m_stp[MathMin(i+1,monitor.m_nonc1test0lngrep.m_cnt-1)]-monitor.m_nonc1test0lngrep.m_stp[i]);
         CAp::Trace(StringFormat("*** |  %13.5E  |  %13.5E  |   %13.5E   |",monitor.m_nonc1test0lngrep.m_stp[i],monitor.m_nonc1test0lngrep.m_f[i] - monitor.m_nonc1test0lngrep.m_f[0],slope));
         if(i>=monitor.m_nonc1test0lngrep.m_stpidxa && i<=monitor.m_nonc1test0lngrep.m_stpidxb)
           {
            CAp::Trace(" <---");
           }
         CAp::Trace("\n");
        }
      CAp::Trace("*** ------------------------------------------------------|\n");
      if(needxdreport)
        {
         CAp::Trace("*** > printing raw variables\n");
         CAp::Trace("*** XK = ");
         CApServ::TraceVectorUnscaledUnshiftedAutopRec(monitor.m_nonc1test0lngrep.m_x0,monitor.m_n,monitor.m_s,true,monitor.m_s,false);
         CAp::Trace("\n");
         CAp::Trace("*** D  = ");
         CApServ::TraceVectorUnscaledUnshiftedAutopRec(monitor.m_nonc1test0lngrep.m_d,monitor.m_n,monitor.m_s,true,monitor.m_s,false);
         CAp::Trace("\n");
         CAp::Trace("*** > printing scaled variables (values are divided by user-specified scales)\n");
         CAp::Trace("*** XK = ");
         CApServ::TraceVectorAutopRec(monitor.m_nonc1test0lngrep.m_x0,0,monitor.m_n);
         CAp::Trace("\n");
         CAp::Trace("*** D  = ");
         CApServ::TraceVectorAutopRec(monitor.m_nonc1test0lngrep.m_d,0,monitor.m_n);
         CAp::Trace("\n");
        }
     }
   if(monitor.m_rep.m_nonc1suspected && monitor.m_rep.m_nonc1test1positive)
     {
      CAp::Trace("> printing out nonsmoothness test #1 report:\n");
      CAp::Trace("*** -------------------------------------------------------\n");
      CAp::Trace("*** | Test #1 for nonsmoothness was triggered  (this test |\n");
      CAp::Trace("*** | analyzes changes in gradient components). See below |\n");
      CAp::Trace("*** | for detailed Info:                                  |\n");
      CAp::Trace(StringFormat("*** | * function index:         %10.m_d",monitor.m_nonc1test1lngrep.m_fidx));
      if(monitor.m_nonc1test1lngrep.m_fidx==0)
        {
         CAp::Trace(" (target)       |\n");
        }
      else
        {
         CAp::Trace(" (constraint)   |\n");
        }
      CAp::Trace(StringFormat("*** | * variable index I:       %10.m_d                |\n",monitor.m_nonc1test1lngrep.m_vidx));
      CAp::Trace(StringFormat("*** | * dF/dX Lipschitz const:  %10.2E                |\n",monitor.m_rep.m_nonc1lipschitzc));
      CAp::Trace("*** | Printing out log of suspicious line search XK+Stp*D |\n");
      CAp::Trace("*** | Look for abrupt changes in slope.                   |\n");
      if(!needxdreport)
        {
         CAp::Trace("*** | NOTE: XK and D are  not printed  by default. If you |\n");
         CAp::Trace("*** |       need them,add trace tag OPTIMIZERS.X         |\n");
        }
      CAp::Trace("*** -------------------------------------------------------\n");
      CAp::Trace("*** |  step along D   |     delta Gi    |      slope      |\n");
      CAp::Trace("*** ------------------------------------------------------|\n");
      for(i=0; i<monitor.m_nonc1test1lngrep.m_cnt; i++)
        {
         slope=monitor.m_nonc1test1lngrep.m_g[MathMin(i+1,monitor.m_nonc1test1lngrep.m_cnt-1)]-monitor.m_nonc1test1lngrep.m_g[i];
         slope=slope/(1.0e-15+monitor.m_nonc1test1lngrep.m_stp[MathMin(i+1,monitor.m_nonc1test1lngrep.m_cnt-1)]-monitor.m_nonc1test1lngrep.m_stp[i]);
         CAp::Trace(StringFormat("*** |  %13.5E  |  %13.5E  |   %13.5E   |",monitor.m_nonc1test1lngrep.m_stp[i],monitor.m_nonc1test1lngrep.m_g[i] - monitor.m_nonc1test1lngrep.m_g[0],slope));
         if(i>=monitor.m_nonc1test1lngrep.m_stpidxa && i<=monitor.m_nonc1test1lngrep.m_stpidxb)
           {
            CAp::Trace(" <---");
           }
         CAp::Trace("\n");
        }
      CAp::Trace("*** ------------------------------------------------------|\n");
      if(needxdreport)
        {
         CAp::Trace("*** > printing raw variables\n");
         CAp::Trace("*** XK = ");
         CApServ::TraceVectorUnscaledUnshiftedAutopRec(monitor.m_nonc1test1lngrep.m_x0,monitor.m_n,monitor.m_s,true,monitor.m_s,false);
         CAp::Trace("\n");
         CAp::Trace("*** D  = ");
         CApServ::TraceVectorUnscaledUnshiftedAutopRec(monitor.m_nonc1test1lngrep.m_d,monitor.m_n,monitor.m_s,true,monitor.m_s,false);
         CAp::Trace("\n");
         CAp::Trace("*** > printing scaled variables (values are divided by user-specified scales)\n");
         CAp::Trace("*** XK = ");
         CApServ::TraceVectorAutopRec(monitor.m_nonc1test1lngrep.m_x0,0,monitor.m_n);
         CAp::Trace("\n");
         CAp::Trace("*** D  = ");
         CApServ::TraceVectorAutopRec(monitor.m_nonc1test1lngrep.m_d,0,monitor.m_n);
         CAp::Trace("\n");
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine exports report to user - readable representation |
//| (all arrays are forced to have exactly same size as needed;      |
//| unused arrays are set to zero length).                           |
//+------------------------------------------------------------------+
void COptServ::SmoothnessMonitorExportReport(CSmoothnessMonitor &monitor,
                                             COptGuardReport &rep)
  {
//--- Finalize last line search, just to be sure
   if(monitor.m_enqueuedcnt>0)
      SmoothnessMonitorFinalizeLineSearch(monitor);
//--- Export report
   COptGuardApi::OptGuardExportReport(monitor.m_rep,monitor.m_n,monitor.m_k,monitor.m_badgradhasxj,rep);
  }
//+------------------------------------------------------------------+
//| Check numerical gradient at point X0 (unscaled variables!), with |
//| optional box constraints [BndL, BndU](if HasBoxConstraints=True) |
//| and  with  scale vector S[].                                     |
//| Step S[i] * TestStep is performed along I-th variable.           |
//| NeedFiJ rcomm protocol is used to request derivative information.|
//| Box constraints BndL / BndU are expected to be feasible. It is   |
//| possible  to have BndL = BndU.                                   |
//+------------------------------------------------------------------+
bool COptServ::SmoothnessMonitorCheckGradientATX0(CSmoothnessMonitor &monitor,
                                                  CRowDouble &unscaledx0,
                                                  CRowDouble &s,
                                                  CRowDouble &bndl,
                                                  CRowDouble &bndu,
                                                  bool hasboxconstraints,
                                                  double teststep)
  {
//--- create variables
   int    n=0;
   int    k=0;
   int    i=0;
   int    j=0;
   int    varidx=0;
   double v=0;
   double vp=0;
   double vm=0;
   double vc=0;
   int    label=-1;
//--- Reverse communication preparations
//--- I know it looks ugly, but it works the same way
//--- anywhere from C++ to Python.
//--- This code initializes locals by:
//--- * random values determined during code
//---   generation - on first subroutine call
//--- * values from previous call - on subsequent calls
   if(monitor.m_rstateg0.stage>=0)
     {
      n=monitor.m_rstateg0.ia[0];
      k=monitor.m_rstateg0.ia[1];
      i=monitor.m_rstateg0.ia[2];
      j=monitor.m_rstateg0.ia[3];
      varidx=monitor.m_rstateg0.ia[4];
      v=monitor.m_rstateg0.ra[0];
      vp=monitor.m_rstateg0.ra[1];
      vm=monitor.m_rstateg0.ra[2];
      vc=monitor.m_rstateg0.ra[3];
     }
   else
     {
      n=-788;
      k=809;
      i=205;
      j=-838;
      varidx=939;
      v=-526;
      vp=763;
      vm=-541;
      vc=-698;
     }
//--- select
   switch(monitor.m_rstateg0.stage)
     {
      case 0:
         label=0;
         break;
      case 1:
         label=1;
         break;
      case 2:
         label=2;
         break;
      case 3:
         label=3;
         break;
      //--- Routine body
      default:
         n=monitor.m_n;
         k=monitor.m_k;
         monitor.m_needfij=false;
         //--- Quick exit
         if(n<=0 || k<=0 || !MathIsValidNumber(teststep) || teststep==0.0)
            return(false);
         teststep=MathAbs(teststep);
         //--- Allocate storage
         CApServ::RVectorSetLengthAtLeast(monitor.m_x,n);
         CApServ::RVectorSetLengthAtLeast(monitor.m_fi,k);
         CApServ::RMatrixSetLengthAtLeast(monitor.m_j,k,n);
         CApServ::RVectorSetLengthAtLeast(monitor.m_xbase,n);
         CApServ::RVectorSetLengthAtLeast(monitor.m_fbase,k);
         CApServ::RVectorSetLengthAtLeast(monitor.m_fm,k);
         CApServ::RVectorSetLengthAtLeast(monitor.m_fc,k);
         CApServ::RVectorSetLengthAtLeast(monitor.m_fp,k);
         CApServ::RVectorSetLengthAtLeast(monitor.m_jm,k);
         CApServ::RVectorSetLengthAtLeast(monitor.m_jc,k);
         CApServ::RVectorSetLengthAtLeast(monitor.m_jp,k);
         CApServ::RMatrixSetLengthAtLeast(monitor.m_jbaseusr,k,n);
         CApServ::RMatrixSetLengthAtLeast(monitor.m_jbasenum,k,n);
         CApServ::RVectorSetLengthAtLeast(monitor.m_rep.m_badgradxbase,n);
         CApServ::RMatrixSetLengthAtLeast(monitor.m_rep.m_badgraduser,k,n);
         CApServ::RMatrixSetLengthAtLeast(monitor.m_rep.m_badgradnum,k,n);
         //--- Set XBase/Jacobian presence flag
         monitor.m_badgradhasxj=true;
         //--- Determine reference point, compute function vector and user-supplied Jacobian
         for(i=0; i<n; i++)
           {
            v=unscaledx0[i];
            if((hasboxconstraints && MathIsValidNumber(bndl[i])) && v<bndl[i])
               v=bndl[i];
            if((hasboxconstraints && MathIsValidNumber(bndu[i])) && v>bndu[i])
               v=bndu[i];
            monitor.m_xbase.Set(i,v);
            monitor.m_rep.m_badgradxbase.Set(i,v);
            monitor.m_x.Set(i,v);
           }
         monitor.m_needfij=true;
         monitor.m_rstateg0.stage=0;
         label=-1;
         break;
     }
//--- main loop
   while(label>=0)
      switch(label)
        {
         case 0:
            monitor.m_needfij=false;
            monitor.m_fbase=monitor.m_fi;
            monitor.m_jbaseusr=monitor.m_j;
            monitor.m_rep.m_badgraduser=monitor.m_j;
            //--- Check Jacobian column by column
            varidx=0;
         case 4:
            if(varidx>n-1)
              {
               label=6;
               break;
              }
            //--- Determine test location.
            v=monitor.m_xbase[varidx];
            vm=v-s[varidx]*teststep;
            vp=v+s[varidx]*teststep;
            if((hasboxconstraints && MathIsValidNumber(bndl[varidx])) && vm<bndl[varidx])
               vm=bndl[varidx];
            if((hasboxconstraints && MathIsValidNumber(bndu[varidx])) && vp>bndu[varidx])
               vp=bndu[varidx];
            vc=vm+(vp-vm)/2;
            //--- Quickly skip fixed variables
            if(vm==vp || vc==vm || vc==vp)
              {
               monitor.m_rep.m_badgradnum.Col(varidx,vector<double>::Zeros(k));
               label=5;
               break;
              }
            //--- Compute F/J at three trial points
            monitor.m_x=monitor.m_xbase;
            monitor.m_x.Set(varidx,vm);
            monitor.m_needfij=true;
            monitor.m_rstateg0.stage=1;
            label=-1;
            break;
         case 1:
            monitor.m_needfij=false;
            monitor.m_fm=monitor.m_fi;
            monitor.m_jm=monitor.m_j.Col(varidx)+0;
            monitor.m_x=monitor.m_xbase;
            monitor.m_x.Set(varidx,vc);
            monitor.m_needfij=true;
            monitor.m_rstateg0.stage=2;
            label=-1;
            break;
         case 2:
            monitor.m_needfij=false;
            monitor.m_fc=monitor.m_fi;
            monitor.m_jc=monitor.m_j.Col(varidx)+0;
            monitor.m_x=monitor.m_xbase;
            monitor.m_x.Set(varidx,vp);
            monitor.m_needfij=true;
            monitor.m_rstateg0.stage=3;
            label=-1;
            break;
         case 3:
            monitor.m_needfij=false;
            monitor.m_fp=monitor.m_fi;
            monitor.m_jp=monitor.m_j.Col(varidx)+0;
            //--- Check derivative
            for(i=0; i<k; i++)
              {
               monitor.m_rep.m_badgradnum.Set(i,varidx,(monitor.m_fp[i]-monitor.m_fm[i])/(vp-vm));
               if(!DerivativeCheck(monitor.m_fm[i],monitor.m_jm[i]*s[varidx],monitor.m_fp[i],monitor.m_jp[i]*s[varidx],monitor.m_fc[i],monitor.m_jc[i]*s[varidx],(vp-vm)/s[varidx]))
                 {
                  monitor.m_rep.m_badgradsuspected=true;
                  monitor.m_rep.m_badgradfidx=i;
                  monitor.m_rep.m_badgradvidx=varidx;
                 }
              }
         case 5:
            varidx=varidx+1;
            label=4;
            break;
         case 6:
            return(false);
        }
//--- Saving State
   monitor.m_rstateg0.ia.Set(0,n);
   monitor.m_rstateg0.ia.Set(1,k);
   monitor.m_rstateg0.ia.Set(2,i);
   monitor.m_rstateg0.ia.Set(3,j);
   monitor.m_rstateg0.ia.Set(4,varidx);
   monitor.m_rstateg0.ra.Set(0,v);
   monitor.m_rstateg0.ra.Set(1,vp);
   monitor.m_rstateg0.ra.Set(2,vm);
   monitor.m_rstateg0.ra.Set(3,vc);
   return(true);
  }
//+------------------------------------------------------------------+
//| This function calculates feasibility error(square root of sum of |
//| squared errors) for a Kx(NMain + NSlack) system of linear        |
//| equalities.                                                      |
//| INPUT PARAMETERS:                                                |
//|   CE       -  set of K equality constraints,                     |
//|               array[K, NMain + NSlack + 1]                       |
//|   X        -  candidate point, array [NMain + NSlack]            |
//|   NMain    -  number of primary variables                        |
//|   NSlack   -  number of slack variables                          |
//|   K        -  number of constraints                              |
//|   Tmp0     -  possible preallocated buffer, automatically resized|
//| RESULT: Sqrt(SUM(Err ^ 2))                                       |
//+------------------------------------------------------------------+
double COptServ::FeasibilityError(CMatrixDouble &ce,
                                  CRowDouble &x,
                                  int nmain,
                                  int nslack,
                                  int k,
                                  CRowDouble &tmp0)
  {
   double result=0;

   tmp0=ce.Col(nmain+nslack)*(-1.0);
   CAblas::RMatrixGemVect(k,nmain+nslack,1.0,ce,0,0,0,x,0,1.0,tmp0,0);
   result=tmp0.Dot(tmp0);
   result=MathSqrt(result);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This function calculates feasibility error(square root of sum of |
//| squared errors) for a Kx(NMain + NSlack) system  of  linear      |
//| equalities  and  error gradient(with respect to x)               |
//| INPUT PARAMETERS:                                                |
//|   CE       -  set of K equality constraints,                     |
//|               array[K, NMain + NSlack + 1]                       |
//|   X        -  candidate point, array [NMain + NSlack]            |
//|   NMain    -  number of primary variables                        |
//|   NSlack   -  number of slack variables                          |
//|   K        -  number of constraints                              |
//|   Grad     -  preallocated array[NMain + NSlack]                 |
//|   Tmp0     -  possible preallocated buffer, automatically resized|
//| RESULT:                                                          |
//|   Err      -  Sqrt(SUM(Err ^ 2))                                 |
//|   Grad     -  error gradient with respect to X,                  |
//|               array[NMain + NSlack]                              |
//+------------------------------------------------------------------+
void COptServ::FeasibilityErrorGrad(CMatrixDouble &ce,
                                    CRowDouble &x,
                                    int nmain,
                                    int nslack,
                                    int k,
                                    double &err,
                                    CRowDouble &grad,
                                    CRowDouble &tmp0)
  {
   err=0;
//--- check
   if(!CAp::Assert(grad.Size()>=nmain+nslack,__FUNCTION__+": integrity check failed"))
      return;

   tmp0.Resize(k);
   CAblas::RMatrixGemVect(k,nmain+nslack,1.0,ce,0,0,0,x,0,0.0,tmp0,0);
   tmp0-=ce.Col(nmain+nslack)+0;
   err=tmp0.Dot(tmp0);
   err=MathSqrt(err);
   CAblas::RMatrixGemVect(nmain+nslack,k,1.0,ce,0,0,1,tmp0,0,0.0,grad,0);
  }
//+------------------------------------------------------------------+
//| This subroutine checks C0 continuity and returns continuity      |
//| rating (normalized value, with values above 50 - 500 being good  |
//| indication of the discontinuity) and Lipschitz constant.         |
//| An interval between F1 and F2 is  tested  for(dis) continuity.   |
//| Per - point noise estimates are provided. Delta[i] is a step from|
//| F[i] to F[i + 1].                                                |
//| ApplySpecialCorrection parameter should be set to True if you use|
//| this function to estimate continuity of the model around minimum;|
//| it adds special correction which helps to detect "max(0,1/x)" -  |
//| like discontinuities. Without this correction algorithm will     |
//| still work, but will be a bit  less powerful. Do not use this    |
//| correction for situations when you want to estimate continuity   |
//| around some non - extremal point - it may result in spurious     |
//| discontinuities being reported.                                  |
//+------------------------------------------------------------------+
void COptServ::TestC0Continuity(double f0,double f1,double f2,
                                double f3,double noise0,double noise1,
                                double noise2,double noise3,double delta0,
                                double delta1,double delta2,
                                bool applyspecialcorrection,
                                double &rating,double &lipschitz)
  {
//--- create variables
   double lipschitz01=0;
   double lipschitz12=0;
   double lipschitz23=0;
   rating=0;
   lipschitz=0;
//--- Compute Lipschitz constant for the interval [0,1],
//--- add noise correction in order to get increased estimate (makes
//--- comparison below more conservative).
   lipschitz01=(MathAbs(f1-f0)+(noise0+noise1))/delta0;
//--- Compute Lipschitz constant for the interval [StpIdx+1,StpIdx+2],
//--- SUBTRACT noise correction in order to get decreased estimate (makes
//--- comparison below more conservative).
   lipschitz12=MathMax(MathAbs(f2-f1)-(noise1+noise2),0.0)/delta1;
//--- Compute Lipschitz constant for the interval [StpIdx+2,StpIdx+3]
//--- using special algorithm:
//--- a) if F3<F2-Noise23, Lipschitz constant is assumed to be zero
//--- b) otherwise, we compute Lipschitz constant as usual,
//---    with noise correction term being added
//--- We need case (a) because some kinds of discontinuities
//--- (like one introduced by max(1/x,0)) manifest themselves
//--- in a quite special way.
   if(applyspecialcorrection && f3<f2-(noise2+noise3))
      lipschitz23=0;
   else
      lipschitz23=(MathAbs(f3-f2)+(noise2+noise3))/delta2;
//--- Compute rating (ratio of two Lipschitz constants)
//--- check
   if(!CAp::Assert(MathMax(lipschitz01,lipschitz23)>0,__FUNCTION__+": integrity check failed"))
      return;
   rating=lipschitz12/MathMax(lipschitz01,lipschitz23);
   lipschitz=lipschitz12;
  }
//+------------------------------------------------------------------+
//| This subroutine checks C1 continuity using test #0(function      |
//| values from the line search log are studied, gradient is not     |
//| used).                                                           |
//| An interval between F[StpIdx + 0] and F[StpIdx + 5] is tested for|
//| continuity. An normalized error metric (Lipschitz constant growth|
//| for the  derivative) for the interval in question is calculated. |
//| Values above  50  are  a  good indication of the discontinuity.  |
//| A six - point algorithm is used for testing, so we expect that   |
//| Monitor.F and Monitor.Stp have enough points for this test.      |
//+------------------------------------------------------------------+
void COptServ::C1ContinuityTest0(CSmoothnessMonitor &monitor,
                                 int funcidx,
                                 int stpidx,
                                 int sortedcnt)
  {
//--- create variables
   double f0=0;
   double f1=0;
   double f2=0;
   double f3=0;
   double f4=0;
   double f5=0;
   double noise0=0;
   double noise1=0;
   double noise2=0;
   double noise3=0;
   double noise4=0;
   double noise5=0;
   double delta0=0;
   double delta1=0;
   double delta2=0;
   double delta3=0;
   double delta4=0;
   double d0=0;
   double d1=0;
   double d2=0;
   double d3=0;
   double newnoise0=0;
   double newnoise1=0;
   double newnoise2=0;
   double newnoise3=0;
   double newdelta0=0;
   double newdelta1=0;
   double newdelta2=0;
   double rating=0;
   double lipschitz=0;
   double lengthrating=0;
   int    i=0;
   int    n=monitor.m_n;
   double nrm=0;
//--- check
   if(!CAp::Assert(stpidx+5<sortedcnt,__FUNCTION__+": integrity check failed"))
      return;
   if(!CAp::Assert(monitor.m_sortedstp[0]==0.0,__FUNCTION__+": integrity check failed"))
      return;
   if(!CAp::Assert(monitor.m_sortedstp[sortedcnt-1]>0.0,__FUNCTION__+": integrity check failed"))
      return;
//--- Fetch F, noise, Delta's
   f0=monitor.m_f[stpidx+0];
   f1=monitor.m_f[stpidx+1];
   f2=monitor.m_f[stpidx+2];
   f3=monitor.m_f[stpidx+3];
   f4=monitor.m_f[stpidx+4];
   f5=monitor.m_f[stpidx+5];
   noise0=m_ognoiselevelf*MathMax(MathAbs(f0),1.0);
   noise1=m_ognoiselevelf*MathMax(MathAbs(f1),1.0);
   noise2=m_ognoiselevelf*MathMax(MathAbs(f2),1.0);
   noise3=m_ognoiselevelf*MathMax(MathAbs(f3),1.0);
   noise4=m_ognoiselevelf*MathMax(MathAbs(f4),1.0);
   noise5=m_ognoiselevelf*MathMax(MathAbs(f5),1.0);
   delta0=monitor.m_sortedstp[stpidx+1]-monitor.m_sortedstp[stpidx+0];
   delta1=monitor.m_sortedstp[stpidx+2]-monitor.m_sortedstp[stpidx+1];
   delta2=monitor.m_sortedstp[stpidx+3]-monitor.m_sortedstp[stpidx+2];
   delta3=monitor.m_sortedstp[stpidx+4]-monitor.m_sortedstp[stpidx+3];
   delta4=monitor.m_sortedstp[stpidx+5]-monitor.m_sortedstp[stpidx+4];
//--- Differentiate functions, get derivative values and noise
//--- estimates at points (0+1)/2, (1+2)/2, (3+4)/2, (3+4)/2,
//--- (4+5)/2. Compute new step values NewDelta[i] and new
//--- noise estimates.
   d0=(f1-f0)/delta0;
   d1=(f2-f1)/delta1;
   d2=(f4-f3)/delta3;
   d3=(f5-f4)/delta4;
   newnoise0=(noise0+noise1)/delta0;
   newnoise1=(noise1+noise2)/delta1;
   newnoise2=(noise3+noise4)/delta3;
   newnoise3=(noise4+noise5)/delta4;
   newdelta0=0.5*(delta0+delta1);
   newdelta1=0.5*delta1+delta2+0.5*delta3;
   newdelta2=0.5*(delta3+delta4);
//--- Test with C0 continuity tester. "Special correction" is
//--- turned off for this test.
   TestC0Continuity(d0,d1,d2,d3,newnoise0,newnoise1,newnoise2,newnoise3,newdelta0,newdelta1,newdelta2,false,rating,lipschitz);
//--- Store results
   if(rating>m_ogminrating1)
     {
      //--- Store to total report
      monitor.m_rep.m_nonc1test0positive=true;
      if(rating>monitor.m_nonc1currentrating)
        {
         monitor.m_nonc1currentrating=rating;
         monitor.m_rep.m_nonc1suspected=true;
         monitor.m_rep.m_nonc1lipschitzc=lipschitz;
         monitor.m_rep.m_nonc1fidx=funcidx;
        }
      //--- Store to "strongest" report
      if(rating>monitor.m_nonc1test0strrating)
        {
         monitor.m_nonc1test0strrating=rating;
         monitor.m_nonc1test0strrep.m_positive=true;
         monitor.m_nonc1test0strrep.m_fidx=funcidx;
         monitor.m_nonc1test0strrep.m_n=n;
         monitor.m_nonc1test0strrep.m_cnt=sortedcnt;
         monitor.m_nonc1test0strrep.m_stpidxa=stpidx+1;
         monitor.m_nonc1test0strrep.m_stpidxb=stpidx+4;
         monitor.m_nonc1test0strrep.m_d=monitor.m_dcur;
         monitor.m_nonc1test0strrep.m_x0.Resize(n);
         monitor.m_nonc1test0strrep.m_d.Resize(n);
         for(i=0; i<n; i++)
            monitor.m_nonc1test0strrep.m_x0.Set(i,monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i]);
         monitor.m_nonc1test0strrep.m_stp=monitor.m_sortedstp;
         monitor.m_nonc1test0strrep.m_f=monitor.m_f;
         monitor.m_nonc1test0strrep.m_stp.Resize(sortedcnt);
         monitor.m_nonc1test0strrep.m_f.Resize(sortedcnt);
        }
      //--- Store to "longest" report
      nrm=0;
      for(i=0; i<n; i++)
         nrm+=CMath::Sqr(monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i]-monitor.m_enqueuedx[monitor.m_sortedidx[sortedcnt-1]*n+i]);
      nrm=MathSqrt(nrm);
      nrm=MathMin(nrm,1.0);
      nrm=CApServ::Coalesce(nrm,CMath::m_machineepsilon);
      lengthrating=sortedcnt+MathLog(nrm)/MathLog(100);
      if(lengthrating>monitor.m_nonc1test0lngrating)
        {
         monitor.m_nonc1test0lngrating=lengthrating;
         monitor.m_nonc1test0lngrep.m_positive=true;
         monitor.m_nonc1test0lngrep.m_fidx=funcidx;
         monitor.m_nonc1test0lngrep.m_n=n;
         monitor.m_nonc1test0lngrep.m_cnt=sortedcnt;
         monitor.m_nonc1test0lngrep.m_stpidxa=stpidx+1;
         monitor.m_nonc1test0lngrep.m_stpidxb=stpidx+4;
         monitor.m_nonc1test0lngrep.m_d=monitor.m_dcur;
         monitor.m_nonc1test0lngrep.m_stp=monitor.m_sortedstp;
         monitor.m_nonc1test0lngrep.m_f=monitor.m_f;
         monitor.m_nonc1test0lngrep.m_x0.Resize(n);
         monitor.m_nonc1test0lngrep.m_d.Resize(n);
         monitor.m_nonc1test0lngrep.m_stp.Resize(sortedcnt);
         monitor.m_nonc1test0lngrep.m_f.Resize(sortedcnt);
         for(i=0; i<n; i++)
            monitor.m_nonc1test0lngrep.m_x0.Set(i,monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i]);
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine checks C1 continuity using test #1(individual    |
//| gradient components from the line search log are studied for     |
//| continuity).                                                     |
//| An interval between F[StpIdx + 0] and F[StpIdx + 3]is  tested for|
//| continuity. An normalized error metric(Lipschitz constant growth |
//| for the  derivative) for the interval in question is calculated. |
//| Values above  50  are  a  good indication of the discontinuity.  |
//+------------------------------------------------------------------+
void COptServ::C1ContinuityTest1(CSmoothnessMonitor &monitor,
                                 int funcidx,
                                 int stpidx,
                                 int sortedcnt)
  {
//--- create variables
   int    n=monitor.m_n;
   int    i=0;
   int    varidx=0;
   double f0=0;
   double f1=0;
   double f2=0;
   double f3=0;
   double noise0=0;
   double noise1=0;
   double noise2=0;
   double noise3=0;
   double nrm=0;
   double rating=0;
   double lengthrating=0;
   double lipschitz=0;
//--- check
   if(!CAp::Assert(stpidx+3<sortedcnt,__FUNCTION__+": integrity check failed"))
      return;
   if(!CAp::Assert(monitor.m_sortedstp[0]==0.0,__FUNCTION__+": integrity check failed"))
      return;
   if(!CAp::Assert(monitor.m_sortedstp[sortedcnt-1]>0.0,__FUNCTION__+": integrity check failed"))
      return;
//--- Study each component of the gradient in the interval in question
   for(varidx=0; varidx<n; varidx++)
     {
      f0=monitor.m_g[(stpidx+0)*n+varidx];
      f1=monitor.m_g[(stpidx+1)*n+varidx];
      f2=monitor.m_g[(stpidx+2)*n+varidx];
      f3=monitor.m_g[(stpidx+3)*n+varidx];
      noise0=m_ognoiselevelg*MathMax(MathAbs(f0),1.0);
      noise1=m_ognoiselevelg*MathMax(MathAbs(f1),1.0);
      noise2=m_ognoiselevelg*MathMax(MathAbs(f2),1.0);
      noise3=m_ognoiselevelg*MathMax(MathAbs(f3),1.0);
      TestC0Continuity(f0,f1,f2,f3,noise0,noise1,noise2,noise3,monitor.m_sortedstp[stpidx+1]-monitor.m_sortedstp[stpidx+0],monitor.m_sortedstp[stpidx+2]-monitor.m_sortedstp[stpidx+1],monitor.m_sortedstp[stpidx+3]-monitor.m_sortedstp[stpidx+2],false,rating,lipschitz);
      //--- Store results
      if(rating>m_ogminrating1)
        {
         //--- Store to total report
         monitor.m_rep.m_nonc1test1positive=true;
         if(rating>monitor.m_nonc1currentrating)
           {
            monitor.m_nonc1currentrating=rating;
            monitor.m_rep.m_nonc1suspected=true;
            monitor.m_rep.m_nonc1lipschitzc=lipschitz;
            monitor.m_rep.m_nonc1fidx=funcidx;
           }
         //--- Store to "strongest" report
         if(rating>monitor.m_nonc1test1strrating)
           {
            monitor.m_nonc1test1strrating=rating;
            monitor.m_nonc1test1strrep.m_positive=true;
            monitor.m_nonc1test1strrep.m_fidx=funcidx;
            monitor.m_nonc1test1strrep.m_vidx=varidx;
            monitor.m_nonc1test1strrep.m_n=n;
            monitor.m_nonc1test1strrep.m_cnt=sortedcnt;
            monitor.m_nonc1test1strrep.m_stpidxa=stpidx+0;
            monitor.m_nonc1test1strrep.m_stpidxb=stpidx+3;
            monitor.m_nonc1test1strrep.m_d=monitor.m_dcur;
            monitor.m_nonc1test1strrep.m_stp=monitor.m_sortedstp;
            monitor.m_nonc1test1strrep.m_x0.Resize(n);
            monitor.m_nonc1test1strrep.m_d.Resize(n);
            monitor.m_nonc1test1strrep.m_stp.Resize(sortedcnt);
            monitor.m_nonc1test1strrep.m_g.Resize(sortedcnt);
            for(i=0; i<n; i++)
               monitor.m_nonc1test1strrep.m_x0.Set(i,monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i]);
            for(i=0; i<sortedcnt ; i++)
               monitor.m_nonc1test1strrep.m_g.Set(i,monitor.m_g[i*n+varidx]);
           }
         //--- Store to "longest" report
         nrm=0;
         for(i=0; i<n; i++)
            nrm+=CMath::Sqr(monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i]-monitor.m_enqueuedx[monitor.m_sortedidx[sortedcnt-1]*n+i]);
         nrm=MathSqrt(nrm);
         nrm=MathMin(nrm,1.0);
         nrm=CApServ::Coalesce(nrm,CMath::m_machineepsilon);
         lengthrating=sortedcnt+MathLog(nrm)/MathLog(100);
         if(lengthrating>monitor.m_nonc1test1lngrating)
           {
            monitor.m_nonc1test1lngrating=lengthrating;
            monitor.m_nonc1test1lngrep.m_positive=true;
            monitor.m_nonc1test1lngrep.m_fidx=funcidx;
            monitor.m_nonc1test1lngrep.m_vidx=varidx;
            monitor.m_nonc1test1lngrep.m_n=n;
            monitor.m_nonc1test1lngrep.m_cnt=sortedcnt;
            monitor.m_nonc1test1lngrep.m_stpidxa=stpidx+0;
            monitor.m_nonc1test1lngrep.m_stpidxb=stpidx+3;
            monitor.m_nonc1test1lngrep.m_d=monitor.m_dcur;
            monitor.m_nonc1test1lngrep.m_stp=monitor.m_sortedstp;
            monitor.m_nonc1test1lngrep.m_x0.Resize(n);
            monitor.m_nonc1test1lngrep.m_d.Resize(n);
            monitor.m_nonc1test1lngrep.m_stp.Resize(sortedcnt);
            monitor.m_nonc1test1lngrep.m_g.Resize(sortedcnt);
            for(i=0; i<n; i++)
               monitor.m_nonc1test1lngrep.m_x0.Set(i,monitor.m_enqueuedx[monitor.m_sortedidx[0]*n+i]);
            for(i=0; i<sortedcnt; i++)
               monitor.m_nonc1test1lngrep.m_g.Set(i,monitor.m_g[i*n+varidx]);
           }
        }
     }
  }
//+------------------------------------------------------------------+
//| This object stores State of the nonlinear CG optimizer.          |
//| You should use ALGLIB functions to work with this object.        |
//+------------------------------------------------------------------+
class CMinCGState
  {
public:
   //--- variables
   int               m_cgtype;
   int               m_debugrestartscount;
   int               m_k;
   int               m_maxits;
   int               m_mcinfo;
   int               m_mcstage;
   int               m_n;
   int               m_nfev;
   int               m_prectype;
   int               m_repiterationscount;
   int               m_repnfev;
   int               m_repterminationtype;
   int               m_rstimer;
   int               m_smoothnessguardlevel;
   int               m_vcnt;
   double            m_betady;
   double            m_betahs;
   double            m_curstpmax;
   double            m_diffstep;
   double            m_epsf;
   double            m_epsg;
   double            m_epsx;
   double            m_f;
   double            m_fbase;
   double            m_fm1;
   double            m_fm2;
   double            m_fold;
   double            m_fp1;
   double            m_fp2;
   double            m_lastgoodstep;
   double            m_lastscaledstep;
   double            m_stp;
   double            m_stpmax;
   double            m_suggestedstep;
   double            m_teststep;
   double            m_trimthreshold;
   bool              m_algpowerup;
   bool              m_drep;
   bool              m_innerresetneeded;
   bool              m_lsend;
   bool              m_lsstart;
   bool              m_needf;
   bool              m_needfg;
   bool              m_terminationneeded;
   bool              m_userterminationneeded;
   bool              m_xrep;
   bool              m_xupdated;
   //--- objects
   RCommState        m_rstate;
   CSmoothnessMonitor m_smonitor;
   CLinMinState      m_lstate;
   //--- arrays
   CRowDouble        m_d;
   CRowDouble        m_diagh;
   CRowDouble        m_diaghl2;
   CRowDouble        m_dk;
   CRowDouble        m_dn;
   CRowDouble        m_g;
   CRowDouble        m_invs;
   CRowDouble        m_lastscaleused;
   CRowDouble        m_s;
   CRowDouble        m_work0;
   CRowDouble        m_work1;
   CRowDouble        m_x;
   CRowDouble        m_xbase;
   CRowDouble        m_xk;
   CRowDouble        m_xn;
   CRowDouble        m_yk;
   //--- matrix
   CMatrixDouble     m_vcorr;
   //--- constructor, destructor
                     CMinCGState(void);
                    ~CMinCGState(void) {}
   //--- copy
   void              Copy(const CMinCGState &obj);
   //--- overloading
   void              operator=(const CMinCGState &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CMinCGState::CMinCGState(void)
  {
   m_cgtype=0;
   m_debugrestartscount=0;
   m_k=0;
   m_maxits=0;
   m_mcinfo=0;
   m_mcstage=0;
   m_n=0;
   m_nfev=0;
   m_prectype=0;
   m_repiterationscount=0;
   m_repnfev=0;
   m_repterminationtype=0;
   m_rstimer=0;
   m_smoothnessguardlevel=0;
   m_vcnt=0;
   m_betady=0;
   m_betahs=0;
   m_curstpmax=0;
   m_diffstep=0;
   m_epsf=0;
   m_epsg=0;
   m_epsx=0;
   m_f=0;
   m_fbase=0;
   m_fm1=0;
   m_fm2=0;
   m_fold=0;
   m_fp1=0;
   m_fp2=0;
   m_lastgoodstep=0;
   m_lastscaledstep=0;
   m_stp=0;
   m_stpmax=0;
   m_suggestedstep=0;
   m_teststep=0;
   m_trimthreshold=0;
   m_algpowerup=false;
   m_drep=false;
   m_innerresetneeded=false;
   m_lsend=false;
   m_lsstart=false;
   m_needf=false;
   m_needfg=false;
   m_terminationneeded=false;
   m_userterminationneeded=false;
   m_xrep=false;
   m_xupdated=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CMinCGState::Copy(const CMinCGState &obj)
  {
   m_cgtype=obj.m_cgtype;
   m_debugrestartscount=obj.m_debugrestartscount;
   m_k=obj.m_k;
   m_maxits=obj.m_maxits;
   m_mcinfo=obj.m_mcinfo;
   m_mcstage=obj.m_mcstage;
   m_n=obj.m_n;
   m_nfev=obj.m_nfev;
   m_prectype=obj.m_prectype;
   m_repiterationscount=obj.m_repiterationscount;
   m_repnfev=obj.m_repnfev;
   m_repterminationtype=obj.m_repterminationtype;
   m_rstimer=obj.m_rstimer;
   m_smoothnessguardlevel=obj.m_smoothnessguardlevel;
   m_vcnt=obj.m_vcnt;
   m_betady=obj.m_betady;
   m_betahs=obj.m_betahs;
   m_curstpmax=obj.m_curstpmax;
   m_diffstep=obj.m_diffstep;
   m_epsf=obj.m_epsf;
   m_epsg=obj.m_epsg;
   m_epsx=obj.m_epsx;
   m_f=obj.m_f;
   m_fbase=obj.m_fbase;
   m_fm1=obj.m_fm1;
   m_fm2=obj.m_fm2;
   m_fold=obj.m_fold;
   m_fp1=obj.m_fp1;
   m_fp2=obj.m_fp2;
   m_lastgoodstep=obj.m_lastgoodstep;
   m_lastscaledstep=obj.m_lastscaledstep;
   m_stp=obj.m_stp;
   m_stpmax=obj.m_stpmax;
   m_suggestedstep=obj.m_suggestedstep;
   m_teststep=obj.m_teststep;
   m_trimthreshold=obj.m_trimthreshold;
   m_algpowerup=obj.m_algpowerup;
   m_drep=obj.m_drep;
   m_innerresetneeded=obj.m_innerresetneeded;
   m_lsend=obj.m_lsend;
   m_lsstart=obj.m_lsstart;
   m_needf=obj.m_needf;
   m_needfg=obj.m_needfg;
   m_terminationneeded=obj.m_terminationneeded;
   m_userterminationneeded=obj.m_userterminationneeded;
   m_xrep=obj.m_xrep;
   m_xupdated=obj.m_xupdated;
   m_rstate=obj.m_rstate;
   m_smonitor=obj.m_smonitor;
   m_d=obj.m_d;
   m_diagh=obj.m_diagh;
   m_diaghl2=obj.m_diaghl2;
   m_dk=obj.m_dk;
   m_dn=obj.m_dn;
   m_g=obj.m_g;
   m_invs=obj.m_invs;
   m_lastscaleused=obj.m_lastscaleused;
   m_s=obj.m_s;
   m_work0=obj.m_work0;
   m_work1=obj.m_work1;
   m_x=obj.m_x;
   m_xbase=obj.m_xbase;
   m_xk=obj.m_xk;
   m_xn=obj.m_xn;
   m_yk=obj.m_yk;
   m_vcorr=obj.m_vcorr;
   m_lstate=obj.m_lstate;
  }
//+------------------------------------------------------------------+
//| This object stores State of the nonlinear CG optimizer.          |
//| You should use ALGLIB functions to work with this object.        |
//+------------------------------------------------------------------+
class CMinCGStateShell
  {
private:
   CMinCGState       m_innerobj;

public:
   //--- constructors, destructor
                     CMinCGStateShell(void) {}
                     CMinCGStateShell(CMinCGState &obj) { m_innerobj.Copy(obj); }
                    ~CMinCGStateShell(void) {}
   //--- methods
   bool              GetNeedF(void);
   void              SetNeedF(const bool b);
   bool              GetNeedFG(void);
   void              SetNeedFG(const bool b);
   bool              GetXUpdated(void);
   void              SetXUpdated(const bool b);
   double            GetF(void);
   void              SetF(const double d);
   CMinCGState      *GetInnerObj(void);
  };
//+------------------------------------------------------------------+
//| Returns the value of the variable needf                          |
//+------------------------------------------------------------------+
bool CMinCGStateShell::GetNeedF(void)
  {
   return(m_innerobj.m_needf);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable needf                         |
//+------------------------------------------------------------------+
void CMinCGStateShell::SetNeedF(const bool b)
  {
   m_innerobj.m_needf=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable needfg                         |
//+------------------------------------------------------------------+
bool CMinCGStateShell::GetNeedFG(void)
  {
   return(m_innerobj.m_needfg);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable needfg                        |
//+------------------------------------------------------------------+
void CMinCGStateShell::SetNeedFG(const bool b)
  {
   m_innerobj.m_needfg=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable xupdated                       |
//+------------------------------------------------------------------+
bool CMinCGStateShell::GetXUpdated(void)
  {
   return(m_innerobj.m_xupdated);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable xupdated                      |
//+------------------------------------------------------------------+
void CMinCGStateShell::SetXUpdated(const bool b)
  {
   m_innerobj.m_xupdated=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable f                              |
//+------------------------------------------------------------------+
double CMinCGStateShell::GetF(void)
  {
   return(m_innerobj.m_f);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable f                             |
//+------------------------------------------------------------------+
void CMinCGStateShell::SetF(const double d)
  {
   m_innerobj.m_f=d;
  }
//+------------------------------------------------------------------+
//| Return object of class                                           |
//+------------------------------------------------------------------+
CMinCGState *CMinCGStateShell::GetInnerObj(void)
  {
   return(GetPointer(m_innerobj));
  }
//+------------------------------------------------------------------+
//| Auxiliary class for CMinCG                                       |
//+------------------------------------------------------------------+
class CMinCGReport
  {
public:
   int               m_iterationscount;
   int               m_nfev;
   int               m_terminationtype;
   //--- constructor, destructor
                     CMinCGReport(void) { ZeroMemory(this); }
                    ~CMinCGReport(void) {}
   //--- copy
   void              Copy(const CMinCGReport &obj);
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CMinCGReport::Copy(const CMinCGReport &obj)
  {
//--- copy variables
   m_iterationscount=obj.m_iterationscount;
   m_nfev=obj.m_nfev;
   m_terminationtype=obj.m_terminationtype;
  }
//+------------------------------------------------------------------+
//| This class is a shell for class CMinCGReport                     |
//+------------------------------------------------------------------+
class CMinCGReportShell
  {
private:
   CMinCGReport      m_innerobj;

public:
   //--- constructors, destructor
                     CMinCGReportShell(void) {}
                     CMinCGReportShell(CMinCGReport &obj) { m_innerobj.Copy(obj); }
                    ~CMinCGReportShell(void) {}
   //--- methods
   int               GetIterationsCount(void);
   void              SetIterationsCount(const int i);
   int               GetNFev(void);
   void              SetNFev(const int i);
   int               GetTerminationType(void);
   void              SetTerminationType(const int i);
   CMinCGReport     *GetInnerObj(void);
  };
//+------------------------------------------------------------------+
//| Returns the value of the variable iterationscount                |
//+------------------------------------------------------------------+
int CMinCGReportShell::GetIterationsCount(void)
  {
   return(m_innerobj.m_iterationscount);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable iterationscount               |
//+------------------------------------------------------------------+
void CMinCGReportShell::SetIterationsCount(const int i)
  {
   m_innerobj.m_iterationscount=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable nfev                           |
//+------------------------------------------------------------------+
int CMinCGReportShell::GetNFev(void)
  {
   return(m_innerobj.m_nfev);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable nfev                          |
//+------------------------------------------------------------------+
void CMinCGReportShell::SetNFev(const int i)
  {
   m_innerobj.m_nfev=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable terminationtype                |
//+------------------------------------------------------------------+
int CMinCGReportShell::GetTerminationType(void)
  {
   return(m_innerobj.m_terminationtype);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable terminationtype               |
//+------------------------------------------------------------------+
void CMinCGReportShell::SetTerminationType(const int i)
  {
   m_innerobj.m_terminationtype=i;
  }
//+------------------------------------------------------------------+
//| Return object of class                                           |
//+------------------------------------------------------------------+
CMinCGReport *CMinCGReportShell::GetInnerObj(void)
  {
   return(GetPointer(m_innerobj));
  }
//+------------------------------------------------------------------+
//| Conjugate gradient optimizer                                     |
//+------------------------------------------------------------------+
class CMinCG
  {
public:
   //--- class constants
   static const int  m_rscountdownlen;
   static const double m_gtol;

   //--- public methods
   static void       MinCGCreate(const int n,double &x[],CMinCGState &State);
   static void       MinCGCreate(const int n,CRowDouble &x,CMinCGState &State);
   static void       MinCGCreateF(const int n,double &x[],const double diffstep,CMinCGState &State);
   static void       MinCGCreateF(const int n,CRowDouble &x,const double diffstep,CMinCGState &State);
   static void       MinCGSetCond(CMinCGState &State,const double epsg,const double epsf,double epsx,const int m_maxits);
   static void       MinCGSetScale(CMinCGState &State,double &s[]);
   static void       MinCGSetScale(CMinCGState &State,CRowDouble &s);
   static void       MinCGSetXRep(CMinCGState &State,const bool needxrep);
   static void       MinCGSetDRep(CMinCGState &State,const bool needdrep);
   static void       MinCGSetCGType(CMinCGState &State,int cgtype);
   static void       MinCGSetStpMax(CMinCGState &State,const double stpmax);
   static void       MinCGSuggestStep(CMinCGState &State,const double stp);
   static double     MinCGLastGoodStep(CMinCGState &State);
   static void       MinCGSetPrecDefault(CMinCGState &State);
   static void       MinCGSetPrecDiag(CMinCGState &State,double &d[]);
   static void       MinCGSetPrecDiag(CMinCGState &State,CRowDouble &d);
   static void       MinCGSetPrecScale(CMinCGState &State);
   static void       MinCGOptGuardGradient(CMinCGState &State,double teststep);
   static void       MinCGOptGuardSmoothness(CMinCGState &State,int level);
   static void       MinCGOptGuardResults(CMinCGState &State,COptGuardReport &rep);
   static void       MinCGOptGuardNonC1Test0Results(CMinCGState &State,COptGuardNonC1Test0Report &strrep,COptGuardNonC1Test0Report &lngrep);
   static void       MinCGOptGuardNonC1Test1Results(CMinCGState &State,COptGuardNonC1Test1Report &strrep,COptGuardNonC1Test1Report &lngrep);
   static void       MinCGResults(CMinCGState &State,double &x[],CMinCGReport &rep);
   static void       MinCGResults(CMinCGState &State,CRowDouble &x,CMinCGReport &rep);
   static void       MinCGResultsBuf(CMinCGState &State,double &x[],CMinCGReport &rep);
   static void       MinCGResultsBuf(CMinCGState &State,CRowDouble &x,CMinCGReport &rep);
   static void       MinCGRestartFrom(CMinCGState &State,double &x[]);
   static void       MinCGRestartFrom(CMinCGState &State,CRowDouble &x);
   static void       MinCGRequestTermination(CMinCGState &State);
   static void       MinCGSetPrecDiagFast(CMinCGState &State,double &d[]);
   static void       MinCGSetPrecDiagFast(CMinCGState &State,CRowDouble &d);
   static void       MinCGSetPrecLowRankFast(CMinCGState &State,double &d1[],double &c[],CMatrixDouble &v,const int vcnt);
   static void       MinCGSetPrecLowRankFast(CMinCGState &State,CRowDouble &d1,CRowDouble &c,CMatrixDouble &v,const int vcnt);
   static void       MinCGSetPrecVarPart(CMinCGState &State,double &d2[]);
   static void       MinCGSetPrecVarPart(CMinCGState &State,CRowDouble &d2);
   static bool       MinCGIteration(CMinCGState &State);

private:
   static void       ClearRequestFields(CMinCGState &State);
   static void       PreconditionedMultiply(CMinCGState &State,CRowDouble &x,CRowDouble &work0,CRowDouble &work1);
   static double     PreconditionedMultiply2(CMinCGState &State,CRowDouble &x,CRowDouble &y,CRowDouble &work0,CRowDouble &work1);
   static void       MinCGInitInternal(const int n,const double diffstep,CMinCGState &State);
  };
//+------------------------------------------------------------------+
//| Initialize constants                                             |
//+------------------------------------------------------------------+
const int    CMinCG::m_rscountdownlen=10;
const double CMinCG::m_gtol=0.3;
//+------------------------------------------------------------------+
//|         NONLINEAR CONJUGATE GRADIENT METHOD                      |
//| DESCRIPTION:                                                     |
//| The subroutine minimizes function F(x) of N arguments by using   |
//| one of the nonlinear conjugate gradient methods.                 |
//| These CG methods are globally convergent (even on non-convex     |
//| functions) as long as grad(f) is Lipschitz continuous in a some  |
//| neighborhood of the L = { x : f(x)<=f(x0) }.                     |
//| REQUIREMENTS:                                                    |
//| Algorithm will request following information during its          |
//| operation:                                                       |
//| * function value F and its gradient G (simultaneously) at given  |
//|   point X                                                        |
//| USAGE:                                                           |
//| 1. User initializes algorithm State with MinCGCreate() call      |
//| 2. User tunes m_solver parameters with MinCGSetCond(),             |
//|    MinCGSetStpMax() and other functions                          |
//| 3. User calls MinCGOptimize() function which takes algorithm     |
//|    State and pointer (delegate, etc.) to callback function which |
//|    calculates F/G.                                               |
//| 4. User calls MinCGResults() to get solution                     |
//| 5. Optionally, user may call MinCGRestartFrom() to solve another |
//|    problem with same N but another starting point and/or another |
//|    function. MinCGRestartFrom() allows to reuse already          |
//|    initialized structure.                                        |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>0:                          |
//|                 * if given, only leading N elements of X are used|
//|                 * if not given, automatically determined from    |
//|                   size of X                                      |
//|     X       -   starting point, array[0..m_n-1].                   |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm State           |
//+------------------------------------------------------------------+
void CMinCG::MinCGCreate(const int n,double &x[],CMinCGState &State)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N too small!"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(x)>=n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- function call
   MinCGInitInternal(n,0.0,State);
//--- function call
   MinCGRestartFrom(State,x);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGCreate(const int n,CRowDouble &x,CMinCGState &State)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N too small!"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(x)>=n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- function call
   MinCGInitInternal(n,0.0,State);
//--- function call
   MinCGRestartFrom(State,x);
  }
//+------------------------------------------------------------------+
//| The subroutine is finite difference variant of MinCGCreate().    |
//| It uses finite differences in order to differentiate target      |
//| function.                                                        |
//| Description below contains information which is specific to this |
//| function only. We recommend to read comments on MinCGCreate() in |
//| order to get more information about creation of CG optimizer.    |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>0:                          |
//|                 * if given, only leading N elements of X are     |
//|                   used                                           |
//|                 * if not given, automatically determined from    |
//|                   size of X                                      |
//|     X       -   starting point, array[0..m_n-1].                   |
//|     DiffStep-   differentiation step, >0                         |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm State           |
//| NOTES:                                                           |
//| 1. algorithm uses 4-point central formula for differentiation.   |
//| 2. differentiation step along I-th axis is equal to              |
//|    DiffStep*S[I] where S[] is scaling vector which can be set by |
//|    MinCGSetScale() call.                                         |
//| 3. we recommend you to use moderate values of differentiation    |
//|    step. Too large step will result in too large truncation      |
//|    errors, while too small step will result in too large         |
//|    numerical errors. 1.0E-6 can be good value to start with.     |
//| 4. Numerical differentiation is very inefficient - one gradient  |
//|    calculation needs 4*N function evaluations. This function will|
//|    work for any N - either small (1...10), moderate (10...100) or|
//|    large  (100...). However, performance penalty will be too     |
//|    severe for any N's except for small ones.                     |
//|    We should also say that code which relies on numerical        |
//|    differentiation is less robust and precise. L-BFGS  needs     |
//|    exact gradient values. Imprecise gradient may slow down       |
//|    convergence, especially on highly nonlinear problems.         |
//|    Thus we recommend to use this function for fast prototyping   |
//|    on small- dimensional problems only, and to implement         |
//|    analytical gradient as soon as possible.                      |
//+------------------------------------------------------------------+
void CMinCG::MinCGCreateF(const int n,double &x[],const double diffstep,
                          CMinCGState &State)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N too small!"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(x)>=n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(diffstep),__FUNCTION__+": DiffStep is infinite or NaN!"))
      return;
//--- check
   if(!CAp::Assert(diffstep>0.0,__FUNCTION__+": DiffStep is non-positive!"))
      return;
//--- function call
   MinCGInitInternal(n,diffstep,State);
//--- function call
   MinCGRestartFrom(State,x);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGCreateF(const int n,CRowDouble &x,const double diffstep,
                          CMinCGState &State)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N too small!"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(x)>=n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(diffstep),__FUNCTION__+": DiffStep is infinite or NaN!"))
      return;
//--- check
   if(!CAp::Assert(diffstep>0.0,__FUNCTION__+": DiffStep is non-positive!"))
      return;
//--- function call
   MinCGInitInternal(n,diffstep,State);
//--- function call
   MinCGRestartFrom(State,x);
  }
//+------------------------------------------------------------------+
//| This function sets stopping conditions for CG optimization       |
//| algorithm.                                                       |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     EpsG    -   >=0                                              |
//|                 The subroutine finishes its work if the condition|
//|                 |v|<EpsG is satisfied, where:                    |
//|                 * |.| means Euclidian norm                       |
//|                 * v - scaled gradient vector, v[i]=g[i]*s[i]     |
//|                 * g - gradient                                   |
//|                 * s - scaling coefficients set by MinCGSetScale()|
//|     EpsF    -   >=0                                              |
//|                 The subroutine finishes its work if on k+1-th    |
//|                 iteration the condition |F(k+1)-F(k)| <=         |
//|                 <= EpsF*max{|F(k)|,|F(k+1)|,1} is satisfied.     |
//|     EpsX    -   >=0                                              |
//|                 The subroutine finishes its work if on k+1-th    |
//|                 iteration the condition |v|<=EpsX is fulfilled,  |
//|                 where:                                           |
//|                 * |.| means Euclidian norm                       |
//|                 * v - scaled step vector, v[i]=dx[i]/s[i]        |
//|                 * dx - ste pvector, dx=X(k+1)-X(k)               |
//|                 * s - scaling coefficients set by MinCGSetScale()|
//|     MaxIts  -   maximum number of iterations. If MaxIts=0, the   |
//|                 number of iterations is unlimited.               |
//| Passing EpsG=0, EpsF=0, EpsX=0 and MaxIts=0 (simultaneously) will|
//| lead to automatic stopping criterion selection (small EpsX).     |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetCond(CMinCGState &State,const double epsg,
                          const double epsf,double epsx,const int m_maxits)
  {
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsg),__FUNCTION__+": EpsG is not finite number!"))
      return;
//--- check
   if(!CAp::Assert(epsg>=0.0,__FUNCTION__+": negative EpsG!"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsf),__FUNCTION__+": EpsF is not finite number!"))
      return;
//--- check
   if(!CAp::Assert(epsf>=0.0,__FUNCTION__+": negative EpsF!"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsx),__FUNCTION__+": EpsX is not finite number!"))
      return;
//--- check
   if(!CAp::Assert(epsx>=0.0,__FUNCTION__+": negative EpsX!"))
      return;
//--- check
   if(!CAp::Assert(m_maxits>=0,__FUNCTION__+": negative MaxIts!"))
      return;
//--- check
   if(epsg==0.0 && epsf==0.0 && epsx==0.0 && m_maxits==0)
      epsx=1.0E-6;
//--- change values
   State.m_epsg=epsg;
   State.m_epsf=epsf;
   State.m_epsx=epsx;
   State.m_maxits=m_maxits;
  }
//+------------------------------------------------------------------+
//| This function sets scaling coefficients for CG optimizer.        |
//| ALGLIB optimizers use scaling matrices to test stopping          |
//| conditions (step size and gradient are scaled before comparison  |
//| with tolerances). Scale of the I-th variable is a translation    |
//| invariant measure of:                                            |
//| a) "how large" the variable is                                   |
//| b) how large the step should be to make significant changes in   |
//|    the function                                                  |
//| Scaling is also used by finite difference variant of CG          |
//| optimizer - step along I-th axis is equal to DiffStep*S[I].      |
//| In most optimizers (and in the CG too) scaling is NOT a form of  |
//| preconditioning. It just affects stopping conditions. You should |
//| set preconditioner by separate call to one of the                |
//| MinCGSetPrec...() functions.                                     |
//| There is special preconditioning mode, however, which uses       |
//| scaling coefficients to form diagonal preconditioning matrix.    |
//| You can turn this mode on, if you want. But you should understand|
//| that scaling is not the same thing as preconditioning - these are|
//| two different, although related forms of tuning m_solver.          |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure stores algorithm State                 |
//|     S       -   array[N], non-zero scaling coefficients          |
//|                 S[i] may be negative, sign doesn't matter.       |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetScale(CMinCGState &State,double &s[])
  {
//--- check
   if(!CAp::Assert(CAp::Len(s)>=State.m_n,__FUNCTION__+": Length(S)<N"))
      return;

   for(int i=0; i<State.m_n; i++)
     {
      //--- check
      if(!CAp::Assert(CMath::IsFinite(s[i]),__FUNCTION__+": S contains infinite or NAN elements"))
         return;
      //--- check
      if(!CAp::Assert(s[i]!=0.0,__FUNCTION__+": S contains zero elements"))
         return;
      State.m_s.Set(i,MathAbs(s[i]));
     }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetScale(CMinCGState &State,CRowDouble &s)
  {
//--- check
   if(!CAp::Assert(CAp::Len(s)>=State.m_n,__FUNCTION__+": Length(S)<N"))
      return;

   for(int i=0; i<State.m_n; i++)
     {
      //--- check
      if(!CAp::Assert(CMath::IsFinite(s[i]),__FUNCTION__+": S contains infinite or NAN elements"))
         return;
      //--- check
      if(!CAp::Assert(s[i]!=0.0,__FUNCTION__+": S contains zero elements"))
         return;
     }

   State.m_s=s.Abs()+0;
  }
//+------------------------------------------------------------------+
//| This function turns on/off reporting.                            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     NeedXRep-   whether iteration reports are needed or not      |
//| If NeedXRep is True, algorithm will call rep() callback function |
//| if it is provided to MinCGOptimize().                            |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetXRep(CMinCGState &State,const bool needxrep)
  {
   State.m_xrep=needxrep;
  }
//+------------------------------------------------------------------+
//| This function turns on/off line search reports.                  |
//| These reports are described in more details in developer-only    |
//| comments on CMinCGState &object.                                   |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     NeedDRep-   whether line search reports are needed or not    |
//| This function is intended for private use only. Turning it on    |
//| artificially may cause program failure.                          |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetDRep(CMinCGState &State,const bool needdrep)
  {
   State.m_drep=needdrep;
  }
//+------------------------------------------------------------------+
//| This function sets CG algorithm.                                 |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     CGType  -   algorithm type:                                  |
//|                 * -1    automatic selection of the best          |
//|                         algorithm                                |
//|                 * 0     DY (Dai and Yuan) algorithm              |
//|                 * 1     Hybrid DY-HS algorithm                   |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetCGType(CMinCGState &State,int cgtype)
  {
//--- check
   if(!CAp::Assert(cgtype>=-1 && cgtype<=1,__FUNCTION__+": incorrect CGType!"))
      return;
//--- check
   if(cgtype==-1)
      cgtype=1;
//--- change value
   State.m_cgtype=cgtype;
  }
//+------------------------------------------------------------------+
//| This function sets maximum step length                           |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     StpMax  -   maximum step length, >=0. Set StpMax to 0.0, if  |
//|                 you don't want to limit step length.             |
//| Use this subroutine when you optimize target function which      |
//| contains exp() or other fast growing functions, and optimization |
//| algorithm makes too large steps which leads to overflow. This    |
//| function allows us to reject steps that are too large (and       |
//| therefore expose us to the possible overflow) without actually   |
//| calculating function value at the x+stp*d.                       |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetStpMax(CMinCGState &State,const double stpmax)
  {
//--- check
   if(!CAp::Assert(CMath::IsFinite(stpmax),__FUNCTION__+": StpMax is not finite!"))
      return;
//--- check
   if(!CAp::Assert(stpmax>=0.0,__FUNCTION__+": StpMax<0!"))
      return;
//--- change value
   State.m_stpmax=stpmax;
  }
//+------------------------------------------------------------------+
//| This function allows to suggest initial step length to the CG    |
//| algorithm.                                                       |
//| Suggested step length is used as starting point for the line     |
//| search. It can be useful when you have badly scaled problem, i.e.|
//| when ||grad|| (which is used as initial estimate for the first   |
//| step) is many orders of magnitude different from the desired     |
//| step.                                                            |
//| Line search may fail on such problems without good estimate of   |
//| initial step length. Imagine, for example, problem with          |
//| ||grad||=10^50 and desired step equal to 0.1 Line search         |
//| function will use 10^50 as initial step, then it will decrease   |
//| step length by 2 (up to 20 attempts) and will get 10^44, which is|
//| still too large.                                                 |
//| This function allows us to tell than line search should be       |
//| started from some moderate step length, like 1.0, so algorithm   |
//| will be able to detect desired step length in a several searches.|
//| Default behavior (when no step is suggested) is to use           |
//| preconditioner, if it is available, to generate initial estimate |
//| of step length.                                                  |
//| This function influences only first iteration of algorithm. It   |
//| should be called between MinCGCreate/MinCGRestartFrom() call and |
//| MinCGOptimize call. Suggested step is ignored if you have        |
//| preconditioner.                                                  |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure used to store algorithm State.         |
//|     Stp     -   initial estimate of the step length.             |
//|                 Can be zero (no estimate).                       |
//+------------------------------------------------------------------+
void CMinCG::MinCGSuggestStep(CMinCGState &State,const double stp)
  {
//--- check
   if(!CAp::Assert(CMath::IsFinite(stp),__FUNCTION__+": Stp is infinite or NAN"))
      return;
//--- check
   if(!CAp::Assert(stp>=0.0,__FUNCTION__+": Stp<0"))
      return;
//--- change value
   State.m_suggestedstep=stp;
  }
//+------------------------------------------------------------------+
//| This developer-only function allows to retrieve unscaled (!)     |
//| length of last good step (i.e. step which resulted in sufficient |
//| decrease of target function).                                    |
//| It can be used in for solution of sequential optimization        |
//| subproblems, where MinCGSuggestStep() is called with length of   |
//| previous step as parameter.                                      |
//| INPUT PARAMETERS:                                                |
//|   State    -  structure used to store algorithm State.           |
//| RESULT:                                                          |
//|   length of last good step being accepted                        |
//| NOTE: result of this function is undefined if you called it      |
//| before                                                           |
//+------------------------------------------------------------------+
double CMinCG::MinCGLastGoodStep(CMinCGState &State)
  {
   return(State.m_lastgoodstep);
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: preconditioning is turned    |
//| off.                                                             |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//| NOTE: you can change preconditioner "on the fly", during         |
//| algorithm iterations.                                            |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecDefault(CMinCGState &State)
  {
//--- change values
   State.m_prectype=0;
   State.m_innerresetneeded=true;
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: diagonal of approximate      |
//| Hessian is used.                                                 |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     D       -   diagonal of the approximate Hessian,             |
//|                 array[0..m_n-1], (if larger, only leading N        |
//|                 elements are used).                              |
//| NOTE: you can change preconditioner "on the fly", during         |
//| algorithm iterations.                                            |
//| NOTE 2: D[i] should be positive. Exception will be thrown        |
//| otherwise.                                                       |
//| NOTE 3: you should pass diagonal of approximate Hessian - NOT    |
//| ITS INVERSE.                                                     |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecDiag(CMinCGState &State,double &d[])
  {
//--- check
   if(!CAp::Assert(CAp::Len(d)>=State.m_n,__FUNCTION__+": D is too short"))
      return;

   for(int i=0; i<State.m_n; i++)
     {
      //--- check
      if(!CAp::Assert(CMath::IsFinite(d[i]),__FUNCTION__+": D contains infinite or NAN elements"))
         return;
      //--- check
      if(!CAp::Assert(d[i]>0.0,__FUNCTION__+": D contains non-positive elements"))
         return;
     }
//--- function call
   MinCGSetPrecDiagFast(State,d);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecDiag(CMinCGState &State,CRowDouble &d)
  {
//--- check
   if(!CAp::Assert(CAp::Len(d)>=State.m_n,__FUNCTION__+": D is too short"))
      return;

   for(int i=0; i<State.m_n; i++)
     {
      //--- check
      if(!CAp::Assert(CMath::IsFinite(d[i]),__FUNCTION__+": D contains infinite or NAN elements"))
         return;
      //--- check
      if(!CAp::Assert(d[i]>0.0,__FUNCTION__+": D contains non-positive elements"))
         return;
     }
//--- function call
   MinCGSetPrecDiagFast(State,d);
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: scale-based diagonal         |
//| preconditioning.                                                 |
//| This preconditioning mode can be useful when you don't have      |
//| approximate diagonal of Hessian, but you know that your variables|
//| are badly scaled (for example, one variable is in [1,10], and    |
//| another in [1000,100000]), and most part of the ill-conditioning |
//| comes from different scales of vars.                             |
//| In this case simple scale-based  preconditioner,                 |
//| with H.Set(i, 1/(s[i]^2), can greatly improve convergence.         |
//| IMPRTANT: you should set scale of your variables with            |
//| MinCGSetScale() call (before or after MinCGSetPrecScale() call). |
//| Without knowledge of the scale of your variables scale-based     |
//| preconditioner will be just unit matrix.                         |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//| NOTE: you can change preconditioner "on the fly", during         |
//| algorithm iterations.                                            |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecScale(CMinCGState &State)
  {
//--- change values
   State.m_prectype=3;
   State.m_innerresetneeded=true;
  }
//+------------------------------------------------------------------+
//| This  function  activates/deactivates verification  of  the      |
//| user-supplied analytic gradient.                                 |
//| Upon  activation  of  this  option  OptGuard  integrity  checker |
//| performs numerical differentiation of your target function  at   |
//| the  initial  point (note: future versions may also perform check|
//| at the final point) and compares numerical gradient with analytic|
//| one provided by you.                                             |
//| If difference is too large, an error flag is set and optimization|
//| session continues. After optimization session is over, you can   |
//| retrieve the report which  stores both  gradients  and  specific |
//| components  highlighted  as suspicious by the OptGuard.          |
//| The primary OptGuard report can be retrieved with                |
//| MinCGOptGuardResults().                                          |
//| IMPORTANT: gradient check is a high-overhead option which  will  |
//|            cost you about 3*N additional function evaluations. In|
//|            many cases it may cost as much as the rest of the     |
//|            optimization session.                                 |
//| YOU SHOULD NOT USE IT IN THE PRODUCTION CODE UNLESS YOU WANT TO  |
//| CHECK DERIVATIVES PROVIDED BY SOME THIRD PARTY.                  |
//| NOTE: unlike previous incarnation of the gradient checking code, |
//|       OptGuard does NOT interrupt optimization even if it        |
//|       discovers bad gradient.                                    |
//| INPUT PARAMETERS:                                                |
//|   State    -  structure used to store algorithm State            |
//|   TestStep -  verification step used for numerical               |
//|               differentiation:                                   |
//|               * TestStep=0 turns verification off                |
//|               * TestStep>0 activates verification                |
//|               You should carefully choose TestStep. Value  which |
//|               is too large (so large that  function  behavior  is|
//|               non-cubic at this scale) will lead to false alarms.|
//|               Too short step will result in rounding  errors     |
//|               dominating numerical derivative.                   |
//|               You may use different step for different parameters|
//|               by means of setting scale with MinCGSetScale().    |
//| === EXPLANATION ================================================ |
//| In order to verify gradient algorithm performs following steps:  |
//|   * two trial steps are made to X[i]-TestStep*S[i] and           |
//|     X[i]+TestStep*S[i], where X[i] is i-th component of the      |
//|     initial point and S[i] is a  scale of i-th parameter         |
//|   * F(X) is evaluated at these trial points                      |
//|   * we perform one more evaluation in the middle point of the    |
//|     interval                                                     |
//|   * we  build  cubic  model using function values and derivatives|
//|     at trial points and we compare its prediction with actual    |
//|     value in  the  middle point                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGOptGuardGradient(CMinCGState &State,
                                   double teststep)
  {
//--- check
   if(!CAp::Assert(MathIsValidNumber(teststep),__FUNCTION__+": TestStep contains NaN or INF"))
      return;
   if(!CAp::Assert(teststep>=0.0,__FUNCTION__+": invalid argument TestStep(TestStep<0)"))
      return;

   State.m_teststep=teststep;
  }
//+------------------------------------------------------------------+
//| This function activates/deactivates nonsmoothness monitoring     |
//| option of the OptGuard integrity  checker. Smoothness  monitor   |
//| silently observes solution process and tries to detect ill-posed |
//| problems, i.e. ones with:                                        |
//|   a) discontinuous target function(non - C0)                     |
//|   b) nonsmooth     target function(non - C1)                     |
//| Smoothness monitoring does NOT interrupt optimization  even if it|
//| suspects that your problem is nonsmooth. It just sets            |
//| corresponding flags in the OptGuard report which can be retrieved|
//| after optimization is over.                                      |
//| Smoothness monitoring is a moderate overhead option which often  |
//| adds less than 1 % to the optimizer running time. Thus, you can  |
//| use it even for large scale problems.                            |
//| NOTE: OptGuard does NOT guarantee that  it  will  always  detect |
//|       C0 / C1 continuity violations.                             |
//| First, minor errors are hard to catch-say, a 0.0001 difference in|
//| the model values at two sides of the gap may be due to           |
//| discontinuity of the model - or simply because the model has     |
//| changed.                                                         |
//| Second, C1 - violations  are  especially  difficult  to  detect  |
//| in a noninvasive way. The optimizer usually performs very  short |
//| steps near the nonsmoothness, and differentiation  usually       |
//| introduces a lot of numerical noise. It is hard to tell whether  |
//| some tiny discontinuity in the slope is due to real nonsmoothness|
//| or just  due to numerical noise alone.                           |
//| Our top priority was to avoid false positives, so in some rare   |
//| cases minor errors may went unnoticed(however, in most cases they|
//| can be spotted with restart from different initial point).       |
//| INPUT PARAMETERS:                                                |
//|   State    -  algorithm State                                    |
//|   Level    -  monitoring level:                                  |
//|                  * 0 - monitoring is disabled                    |
//|                  * 1 - noninvasive low - overhead monitoring;    |
//|                        function values and / or gradients are    |
//|                        recorded, but OptGuard does not try to    |
//|                        perform additional evaluations  in  order |
//|                        to get more information about suspicious  |
//|                        locations.                                |
//| === EXPLANATION ================================================ |
//| One major source of headache during optimization is the          |
//| possibility of the coding errors in the target function /        |
//| constraints (or their gradients). Such errors most often manifest|
//| themselves  as  discontinuity  or nonsmoothness of the target /  |
//| constraints.                                                     |
//| Another frequent situation is when you try to optimize something |
//| involving lots of min() and max() operations, i.e. nonsmooth     |
//| target. Although not a coding error, it is nonsmoothness anyway -|
//| and smooth  optimizers  usually stop right after encountering    |
//| nonsmoothness, well before reaching solution.                    |
//| OptGuard integrity checker helps you to catch such situations:   |
//| it monitors function values / gradients being passed  to  the    |
//| optimizer  and  tries  to errors. Upon discovering suspicious    |
//| pair of points it  raises  appropriate flag (and allows you to   |
//| continue optimization). When optimization is done, you can study |
//| OptGuard result.                                                 |
//+------------------------------------------------------------------+
void CMinCG::MinCGOptGuardSmoothness(CMinCGState &State,int level)
  {
//--- check
   if(!CAp::Assert(level==0 || level==1,__FUNCTION__+": unexpected value of level parameter"))
      return;

   State.m_smoothnessguardlevel=level;
  }
//+------------------------------------------------------------------+
//| Results of OptGuard integrity check, should be called  after     |
//| optimization session is over.                                    |
//| === PRIMARY REPORT ============================================= |
//| OptGuard performs several checks which are intended to catch     |
//| common errors in the implementation of nonlinear function /      |
//| gradient:                                                        |
//|      * incorrect analytic gradient                               |
//|      * discontinuous(non - C0) target functions (constraints)    |
//|      * nonsmooth(non - C1) target functions (constraints)        |
//| Each of these checks is activated with appropriate function:     |
//|      * MinCGOptGuardGradient() for gradient verification         |
//|      * MinCGOptGuardSmoothness() for C0 / C1 checks              |
//| Following flags are set when these errors are suspected:         |
//|      * rep.badgradsuspected, and additionally:                   |
//|      * rep.badgradvidx for specific variable (gradient element)  |
//|                        suspected                                 |
//|      * rep.badgradxbase, a point where gradient is tested        |
//|      * rep.badgraduser, user - provided gradient (stored as 2D   |
//|                         matrix with single row in order to make  |
//|                         report  structure  compatible  with  more|
//|                         complex optimizers like MinNLC or MinLM) |
//|      * rep.badgradnum,  reference gradient obtained via numerical|
//|                         differentiation(stored as  2D matrix with|
//|                         single row in order to make report       |
//|                         structure compatible with more complex   |
//|                         optimizers  like  MinNLC or MinLM)       |
//|      * rep.nonc0suspected                                        |
//|      * rep.nonc1suspected                                        |
//| === ADDITIONAL REPORTS / LOGS ================================== |
//| Several different tests are performed to catch C0 / C1 errors,   |
//| you can find out specific test signaled error by looking to:     |
//|      * rep.nonc0test0positive, for non - C0 test #0              |
//|      * rep.nonc1test0positive, for non - C1 test #0              |
//|      * rep.nonc1test1positive, for non - C1 test #1              |
//| Additional information (including line search logs) can be       |
//| obtained by means of:                                            |
//|      * MinCGOptGuardNonC1Test0Results()                          |
//|      * MinCGOptGuardNonC1Test1Results()                          |
//| which return detailed error reports, specific points where       |
//| discontinuities were found, and so on.                           |
//| ================================================================ |
//| INPUT PARAMETERS:                                                |
//|   State    -  algorithm State                                    |
//| OUTPUT PARAMETERS:                                               |
//|   Rep      -  generic OptGuard report; more  detailed  reports   |
//|               can  be retrieved with other functions.            |
//| NOTE: false negatives(nonsmooth problems are not identified as   |
//|       nonsmooth ones) are possible although unlikely.            |
//| The reason  is  that  you  need  to  make several evaluations    |
//| around nonsmoothness  in order to accumulate enough information  |
//| about function curvature. Say, if you start right from the       |
//| nonsmooth point, optimizer simply won't get enough data to       |
//| understand what is going wrong before it terminates due to abrupt|
//| changes in the  derivative. It is also  possible  that  "unlucky"|
//| step  will  move  us  to  the termination too quickly.           |
//| Our current approach is to have less than 0.1 %  false  negatives|
//| in our test examples (measured with multiple restarts from random|
//| points), and to have exactly 0 % false positives.                |
//+------------------------------------------------------------------+
void CMinCG::MinCGOptGuardResults(CMinCGState &State,COptGuardReport &rep)
  {
   COptServ::SmoothnessMonitorExportReport(State.m_smonitor,rep);
  }
//+------------------------------------------------------------------+
//| Detailed results of the OptGuard integrity check for             |
//| nonsmoothness test #0                                            |
//| Nonsmoothness (non-C1) test #0 studies  function  values (not    |
//| gradient!) obtained during line searches and monitors  behavior  |
//| of  the  directional derivative estimate.                        |
//| This test is less powerful than test #1, but it does  not  depend|
//| on  the gradient values and thus it is more robust against       |
//| artifacts introduced by numerical differentiation.               |
//| Two reports are returned:                                        |
//|  *a "strongest" one, corresponding to line search which   had  |
//|     highest value of the nonsmoothness indicator                 |
//|  *a "longest" one, corresponding to line search which had more |
//|     function evaluations, and thus is more detailed              |
//| In both cases following fields are returned:                     |
//|   * positive - is TRUE  when test flagged suspicious point;      |
//|                   FALSE if test did not notice anything (in the  |
//|                   latter cases fields below are empty).          |
//|   * x0[], d[] - arrays of length N which store initial point and |
//|                 direction for line search(d[] can be normalized, |
//|                 but does not have to)                            |
//|   * stp[], f[]- arrays of length CNT which store step lengths    |
//|                 and function values at these points; f[i] is     |
//|                 evaluated in x0 + stp[i]*d.                      |
//|   * stpidxa, stpidxb - we  suspect  that  function  violates  C1 |
//|                 continuity between steps #stpidxa and #stpidxb   |
//|                 (usually we have  stpidxb = stpidxa + 3, with    |
//|                 most likely position of the  violation  between  |
//|                 stpidxa + 1 and stpidxa + 2.                     |
//| ================================================================ |
//| = SHORTLY SPEAKING: build a 2D plot of(stp, f) and look at it -  |
//| =                   you will see where C1 continuity is violated.|
//| ================================================================ |
//| INPUT PARAMETERS:                                                |
//|   State    -  algorithm State                                    |
//| OUTPUT PARAMETERS:                                               |
//|   StrRep  - C1 test #0 "strong" report                         |
//|   LngRep  - C1 test #0 "long" report                           |
//+------------------------------------------------------------------+
void CMinCG::MinCGOptGuardNonC1Test0Results(CMinCGState &State,
                                            COptGuardNonC1Test0Report &strrep,
                                            COptGuardNonC1Test0Report &lngrep)
  {
   COptGuardApi::SmoothnessMonitorExportC1Test0Report(State.m_smonitor.m_nonc1test0strrep,State.m_lastscaleused,strrep);
   COptGuardApi::SmoothnessMonitorExportC1Test0Report(State.m_smonitor.m_nonc1test0lngrep,State.m_lastscaleused,lngrep);
  }
//+------------------------------------------------------------------+
//| Detailed results of the OptGuard integrity check for             |
//| nonsmoothness test #1                                            |
//| Nonsmoothness (non-C1) test #0 studies  function  values (not    |
//| gradient!) obtained during line searches and monitors  behavior  |
//| of  the  directional derivative estimate.                        |
//| This test is less powerful than test #1, but it does  not  depend|
//| on  the gradient values and thus it is more robust against       |
//| artifacts introduced by numerical differentiation.               |
//| Two reports are returned:                                        |
//|  *a "strongest" one, corresponding to line search which   had  |
//|     highest value of the nonsmoothness indicator                 |
//|  *a "longest" one, corresponding to line search which had more |
//|     function evaluations, and thus is more detailed              |
//| In both cases following fields are returned:                     |
//|   * positive - is TRUE  when test flagged suspicious point;      |
//|                   FALSE if test did not notice anything (in the  |
//|                   latter cases fields below are empty).          |
//|   * x0[], d[] - arrays of length N which store initial point and |
//|                 direction for line search(d[] can be normalized, |
//|                 but does not have to)                            |
//|   * stp[], f[]- arrays of length CNT which store step lengths    |
//|                 and function values at these points; f[i] is     |
//|                 evaluated in x0 + stp[i]*d.                      |
//|   * stpidxa, stpidxb - we  suspect  that  function  violates  C1 |
//|                 continuity between steps #stpidxa and #stpidxb   |
//|                 (usually we have  stpidxb = stpidxa + 3, with    |
//|                 most likely position of the  violation  between  |
//|                 stpidxa + 1 and stpidxa + 2.                     |
//| ================================================================ |
//| = SHORTLY SPEAKING: build a 2D plot of(stp, f) and look at it -  |
//| =                   you will see where C1 continuity is violated.|
//| ================================================================ |
//| INPUT PARAMETERS:                                                |
//|   State    -  algorithm State                                    |
//| OUTPUT PARAMETERS:                                               |
//|   StrRep  - C1 test #1 "strong" report                         |
//|   LngRep  - C1 test #1 "long" report                           |
//+------------------------------------------------------------------+
void CMinCG::MinCGOptGuardNonC1Test1Results(CMinCGState &State,
                                            COptGuardNonC1Test1Report &strrep,
                                            COptGuardNonC1Test1Report &lngrep)
  {
   COptGuardApi::SmoothnessMonitorExportC1Test1Report(State.m_smonitor.m_nonc1test1strrep,State.m_lastscaleused,strrep);
   COptGuardApi::SmoothnessMonitorExportC1Test1Report(State.m_smonitor.m_nonc1test1lngrep,State.m_lastscaleused,lngrep);
  }
//+------------------------------------------------------------------+
//| Conjugate gradient results                                       |
//| INPUT PARAMETERS:                                                |
//|     State   -   algorithm State                                  |
//| OUTPUT PARAMETERS:                                               |
//|     X       -   array[0..m_n-1], solution                        |
//|     Rep     -   optimization report:                             |
//|                 * Rep.TerminationType completetion code:         |
//|                     *  1    relative function improvement is no  |
//|                             more than EpsF.                      |
//|                     *  2    relative step is no more than EpsX.  |
//|                     *  4    gradient norm is no more than EpsG   |
//|                     *  5    MaxIts steps was taken               |
//|                     *  7    stopping conditions are too          |
//|                             stringent, further improvement is    |
//|                             impossible, we return best X found   |
//|                             so far                               |
//|                     *  8    terminated by user                   |
//|                 * Rep.IterationsCount contains iterations count  |
//|                 * NFEV countains number of function calculations |
//+------------------------------------------------------------------+
void CMinCG::MinCGResults(CMinCGState &State,double &x[],CMinCGReport &rep)
  {
//--- reset memory
   ArrayResize(x,0);
//--- function call
   MinCGResultsBuf(State,x,rep);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGResults(CMinCGState &State,CRowDouble &x,CMinCGReport &rep)
  {
//--- reset memory
   x.Resize(0);
//--- function call
   MinCGResultsBuf(State,x,rep);
  }
//+------------------------------------------------------------------+
//| Conjugate gradient results                                       |
//| Buffered implementation of MinCGResults(), which uses            |
//| pre-allocated buffer to store X[]. If buffer size is too small,  |
//| it resizes buffer.It is intended to be used in the inner cycles  |
//| of performance critical algorithms where array reallocation      |
//| penalty is too large to be ignored.                              |
//+------------------------------------------------------------------+
void CMinCG::MinCGResultsBuf(CMinCGState &State,double &x[],CMinCGReport &rep)
  {
//--- create a variable
   int i_=0;
//--- copy
   State.m_xn.ToArray(x);
//--- change values
   rep.m_iterationscount=State.m_repiterationscount;
   rep.m_nfev=State.m_repnfev;
   rep.m_terminationtype=State.m_repterminationtype;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGResultsBuf(CMinCGState &State,CRowDouble &x,CMinCGReport &rep)
  {
//--- copy
   x=State.m_xn;
//--- change values
   rep.m_iterationscount=State.m_repiterationscount;
   rep.m_nfev=State.m_repnfev;
   rep.m_terminationtype=State.m_repterminationtype;
  }
//+------------------------------------------------------------------+
//| This  subroutine  restarts  CG  algorithm from new point. All    |
//| optimization parameters are left unchanged.                      |
//| This function allows to solve multiple optimization problems     |
//| (which must have same number of dimensions) without object       |
//| reallocation penalty.                                            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure used to store algorithm State.         |
//|     X       -   new starting point.                              |
//+------------------------------------------------------------------+
void CMinCG::MinCGRestartFrom(CMinCGState &State,double &x[])
  {
//--- check
   if(!CAp::Assert(CAp::Len(x)>=State.m_n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,State.m_n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- copy
   State.m_xbase=x;
   State.m_xbase.Resize(State.m_n);
//--- function call
   MinCGSuggestStep(State,0.0);
//--- allocation
   State.m_rstate.ia.Resize(2);
   State.m_rstate.ra.Resize(3);
//--- change value
   State.m_rstate.stage=-1;
//--- function call
   ClearRequestFields(State);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGRestartFrom(CMinCGState &State,CRowDouble &x)
  {
//--- check
   if(!CAp::Assert(CAp::Len(x)>=State.m_n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,State.m_n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- copy
   State.m_xbase=x;
   State.m_xbase.Resize(State.m_n);
//--- function call
   MinCGSuggestStep(State,0.0);
//--- allocation
   State.m_rstate.ia.Resize(2);
   State.m_rstate.ra.Resize(3);
//--- change value
   State.m_rstate.stage=-1;
//--- function call
   ClearRequestFields(State);
  }
//+------------------------------------------------------------------+
//| This subroutine submits request for termination of running       |
//| optimizer. It should be called from user-supplied callback when  |
//| user decides that it  is time to "smoothly" terminate            |
//| optimization process. As result, optimizer stops at point which  |
//| was "current accepted" when termination request was submitted and|
//| returns error code 8 (successful termination).                   |
//| INPUT PARAMETERS:                                                |
//|   State    -  optimizer structure                                |
//| NOTE: after request for  termination  optimizer  may   perform   |
//|       several additional calls to user-supplied callbacks. It    |
//|       does NOT guarantee to stop immediately - it just guarantees|
//|       that these additional calls will be discarded later.       |
//| NOTE: calling this function on optimizer which is NOT running    |
//|       will have no effect.                                       |
//| NOTE: multiple calls to this function are possible. First call is|
//|       counted, subsequent calls are silently ignored.            |
//+------------------------------------------------------------------+
void CMinCG::MinCGRequestTermination(CMinCGState &State)
  {
   State.m_userterminationneeded=true;
  }
//+------------------------------------------------------------------+
//| Faster version of MinCGSetPrecDiag(), for time-critical parts of |
//| code, without safety checks.                                     |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecDiagFast(CMinCGState &State,double &d[])
  {
//--- change values
   State.m_prectype=2;
   State.m_vcnt=0;
   State.m_innerresetneeded=true;
//--- copy
   State.m_diagh=d;
   State.m_diaghl2=vector<double>::Zeros(State.m_n);
   State.m_diagh.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecDiagFast(CMinCGState &State,CRowDouble &d)
  {
//--- change values
   State.m_prectype=2;
   State.m_vcnt=0;
   State.m_innerresetneeded=true;
//--- copy
   State.m_diagh=d;
   State.m_diaghl2=vector<double>::Zeros(State.m_n);
   State.m_diagh.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//| This function sets low-rank preconditioner for Hessian matrix    |
//| H=D+V'*C*V, where:                                               |
//| * H is a Hessian matrix, which is approximated by D/V/C          |
//| * D=D1+D2 is a diagonal matrix, which includes two positive      |
//|   definite terms:                                                |
//|   * constant term D1 (is not updated or infrequently updated)    |
//|   * variable term D2 (can be cheaply updated from iteration to   |
//|     iteration)                                                   |
//| * V is a low-rank correction                                     |
//| * C is a diagonal factor of low-rank correction                  |
//| Preconditioner P is calculated using approximate Woodburry       |
//| formula:                                                         |
//|     P  = D^(-1) - D^(-1)*V'*(C^(-1)+V*D1^(-1)*V')^(-1)*V*D^(-1)  |
//|        = D^(-1) - D^(-1)*VC'*VC*D^(-1),                          |
//| where                                                            |
//|     VC = sqrt(B)*V                                               |
//|     B  = (C^(-1)+V*D1^(-1)*V')^(-1)                              |
//| Note that B is calculated using constant term (D1) only, which   |
//| allows us to update D2 without recalculation of B or VC. Such    |
//| preconditioner is exact when D2 is zero. When D2 is non-zero, it |
//| is only approximation, but very good and cheap one.              |
//| This function accepts D1, V, C.                                  |
//| D2 is set to zero by default.                                    |
//| Cost of this update is O(N*VCnt*VCnt), but D2 can be updated in  |
//| just O(N) by MinCGSetPrecVarPart.                                |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecLowRankFast(CMinCGState &State,double &d1[],
                                     double &c[],CMatrixDouble &v,
                                     const int vcnt)
  {
   CRowDouble D=d1;
   CRowDouble C=c;
   MinCGSetPrecLowRankFast(State,D,C,v,vcnt);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecLowRankFast(CMinCGState &State,CRowDouble &d1,
                                     CRowDouble &c,CMatrixDouble &v,
                                     const int vcnt)
  {
//--- create variables
   int    i=0;
   int    i_=0;
   int    j=0;
   int    k=0;
   int    n=0;
   double t=0;
//--- create matrix
   CMatrixDouble b;
//--- check
   if(vcnt==0)
     {
      //--- function call
      MinCGSetPrecDiagFast(State,d1);
      return;
     }
//--- initialization
   n=State.m_n;
   b=matrix<double>::Zeros(vcnt,vcnt);
//--- function call
   CApServ::RMatrixSetLengthAtLeast(State.m_vcorr,vcnt,n);
   State.m_prectype=2;
   State.m_vcnt=vcnt;
   State.m_innerresetneeded=true;
//--- copy
   State.m_diagh=d1;
   State.m_diaghl2=vector<double>::Zeros(n);
   State.m_diagh.Resize(n);
//--- calculation
   for(i=0; i<vcnt; i++)
     {
      vector<double> temp=v[i]/d1.ToVector();
      for(j=i; j<vcnt; j++)
        {
         t=temp.Dot(v[j]+0);
         b.Set(i,j,t);
        }
      b.Set(i,i,b.Get(i,i)+1.0/c[i]);
     }
//--- check
   if(!CTrFac::SPDMatrixCholeskyRec(b,0,vcnt,true,State.m_work0))
     {
      State.m_vcnt=0;
      return;
     }
//--- calculation
   State.m_vcorr=v;
   for(i=0; i<vcnt; i++)
     {
      //--- change values
      for(j=0; j<i; j++)
        {
         t=b.Get(j,i);
         State.m_vcorr.Row(i,State.m_vcorr[i]-State.m_vcorr[j]*t);
        }
      t=1.0/b.Get(i,i);
      //--- change values
      State.m_vcorr.Row(i,State.m_vcorr[i]*t);
     }
  }
//+------------------------------------------------------------------+
//| This function updates variable part (diagonal matrix D2)         |
//| of low-rank preconditioner.                                      |
//| This update is very cheap and takes just O(N) time.              |
//| It has no effect with default preconditioner.                    |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecVarPart(CMinCGState &State,double &d2[])
  {
//--- copy
   State.m_diaghl2=d2;
   State.m_diaghl2.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinCG::MinCGSetPrecVarPart(CMinCGState &State,CRowDouble &d2)
  {
//--- copy
   State.m_diaghl2=d2;
   State.m_diaghl2.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//| Clears request fileds (to be sure that we don't forgot to clear  |
//| something)                                                       |
//+------------------------------------------------------------------+
void CMinCG::ClearRequestFields(CMinCGState &State)
  {
//--- change values
   State.m_needf=false;
   State.m_needfg=false;
   State.m_xupdated=false;
   State.m_lsstart=false;
   State.m_lsend=false;
   State.m_algpowerup=false;
  }
//+------------------------------------------------------------------+
//| This function calculates preconditioned product H^(-1)*x and     |
//| stores result back into X. Work0[] and Work1[] are used as       |
//| temporaries (size must be at least N; this function doesn't      |
//| allocate arrays).                                                |
//+------------------------------------------------------------------+
void CMinCG::PreconditionedMultiply(CMinCGState &State,CRowDouble &x,
                                    CRowDouble &work0,CRowDouble &work1)
  {
//--- create variables
   int    i=0;
   int    n=0;
   int    vcnt=0;
   double v=0;
   int    i_=0;
//--- initialization
   n=State.m_n;
   vcnt=State.m_vcnt;
//--- check
   if(State.m_prectype==0)
      return;
//--- check
   if(State.m_prectype==3)
     {
      x*=State.m_s.Pow(2)+0;
      //--- exit the function
      return;
     }
//--- check
   if(!CAp::Assert(State.m_prectype==2,__FUNCTION__+": internal error (unexpected PrecType)"))
      return;
//--- handle part common for VCnt=0 and VCnt<>0
   x/=State.m_diagh.ToVector()+State.m_diaghl2.ToVector();
//--- if VCnt>0
   if(vcnt>0)
     {
      //--- calculation work0
      for(i=0; i<vcnt; i++)
         work0.Set(i,CAblasF::RDotVR(n,x,State.m_vcorr,i));
      //--- calculation work1
      State.m_work1=vector<double>::Zeros(n);
      for(i=0; i<vcnt; i++)
        {
         v=work0[i];
         for(i_=0; i_<=n-1; i_++)
            State.m_work1.Add(i_,v*State.m_vcorr.Get(i,i_));
        }
      //--- change x
      x-=State.m_work1.ToVector()/(State.m_diagh+State.m_diaghl2);
     }
  }
//+------------------------------------------------------------------+
//| This function calculates preconditioned product x'*H^(-1)*y.     |
//| Work0[] and Work1[] are used as temporaries (size must be at     |
//| least N; this function doesn't allocate arrays).                 |
//+------------------------------------------------------------------+
double CMinCG::PreconditionedMultiply2(CMinCGState &State,CRowDouble &x,
                                       CRowDouble &y,CRowDouble &work0,
                                       CRowDouble &work1)
  {
//--- create variables
   double result=0;
   int    i=0;
   int    n=0;
   int    vcnt=0;
   double v0=0;
   double v1=0;
   int    i_=0;
//--- initialization
   n=State.m_n;
   vcnt=State.m_vcnt;
//--- no preconditioning
   if(State.m_prectype==0)
     {
      result=x.Dot(y);
      //--- return result
      return(result);
     }
//--- check
   if(State.m_prectype==3)
     {
      result=x.Dot(y.ToVector()*State.m_s.Pow(2.0));
      //--- return result
      return(result);
     }
//--- check
   if(!CAp::Assert(State.m_prectype==2,__FUNCTION__+": internal error (unexpected PrecType)"))
      return(EMPTY_VALUE);
//--- low rank preconditioning
   vector<double> temp=State.m_diagh+State.m_diaghl2;
   result=x.Dot(y.ToVector()/temp);
//--- check
   if(vcnt>0)
     {
      //--- prepare arrays
      work0=x.ToVector()/temp;
      work1=y.ToVector()/temp;
      for(i=0; i<=vcnt-1; i++)
        {
         //--- calculation
         v0=CAblasF::RDotVR(n,work0,State.m_vcorr,i);
         v1=CAblasF::RDotVR(n,work1,State.m_vcorr,i);
         //--- get result
         result-=v0*v1;
        }
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Internal initialization subroutine                               |
//+------------------------------------------------------------------+
void CMinCG::MinCGInitInternal(const int n,const double diffstep,
                               CMinCGState &State)
  {
//--- initialization
   State.m_teststep=0;
   State.m_smoothnessguardlevel=0;
   COptServ::SmoothnessMonitorInit(State.m_smonitor,State.m_s,0,0,false);
   State.m_n=n;
   State.m_diffstep=diffstep;
   State.m_lastgoodstep=0;
//--- function call
   MinCGSetCond(State,0,0,0,0);
//--- function call
   MinCGSetXRep(State,false);
//--- function call
   MinCGSetDRep(State,false);
//--- function call
   MinCGSetStpMax(State,0);
//--- function call
   MinCGSetCGType(State,-1);
//--- function call
   MinCGSetPrecDefault(State);
//--- allocation
   State.m_xk.Resize(n);
   State.m_dk.Resize(n);
   State.m_xn.Resize(n);
   State.m_dn.Resize(n);
   State.m_x.Resize(n);
   State.m_d.Resize(n);
   State.m_g.Resize(n);
   State.m_work0.Resize(n);
   State.m_work1.Resize(n);
   State.m_yk.Resize(n);
   State.m_xbase.Resize(n);
   State.m_s=vector<double>::Ones(n);
   State.m_invs=vector<double>::Ones(n);
   State.m_lastscaleused=vector<double>::Ones(n);
  }
//+------------------------------------------------------------------+
//| NOTES:                                                           |
//| 1. This function has two different implementations: one which    |
//|    uses exact (analytical) user-supplied  gradient, and one which|
//|    uses function value only and numerically differentiates       |
//|    function in order to obtain gradient.                         |
//|    Depending on the specific function used to create optimizer   |
//|    object (either MinCGCreate() for analytical gradient or       |
//|    MinCGCreateF() for numerical differentiation) you should      |
//|    choose appropriate variant of MinCGOptimize() - one which     |
//|    accepts function AND gradient or one which accepts function   |
//|    ONLY.                                                         |
//|    Be careful to choose variant of MinCGOptimize() which         |
//|    corresponds to your optimization scheme! Table below lists    |
//|    different combinations of callback (function/gradient) passed |
//|    to MinCGOptimize() and specific function used to create       |
//|    optimizer.                                                    |
//|                   |         USER PASSED TO MinCGOptimize()       |
//|    CREATED WITH   |  function only   |  function and gradient    |
//|    ------------------------------------------------------------  |
//|    MinCGCreateF() |     work                FAIL                 |
//|    MinCGCreate()  |     FAIL                work                 |
//|    Here "FAIL" denotes inappropriate combinations of optimizer   |
//|    creation function and MinCGOptimize() version. Attemps to use |
//|    such combination (for example, to create optimizer with       |
//|    MinCGCreateF() and to pass gradient information to            |
//|    MinCGOptimize()) will lead to exception being thrown. Either  |
//|    you did not pass gradient when it WAS needed or you passed    |
//|    gradient when it was NOT needed.                              |
//+------------------------------------------------------------------+
bool CMinCG::MinCGIteration(CMinCGState &State)
  {
//--- create variables
   int    n=0;
   int    i=0;
   double betak=0;
   double v=0;
   double vv=0;
   int    i_=0;
   int    label=-1;
//--- This code initializes locals by:
//--- * random values determined during code
//---   generation - on first subroutine call
//--- * values from previous call - on subsequent calls
//
   if(State.m_rstate.stage>=0)
     {
      n=State.m_rstate.ia[0];
      i=State.m_rstate.ia[1];
      betak=State.m_rstate.ra[0];
      v=State.m_rstate.ra[1];
      vv=State.m_rstate.ra[2];
     }
   else
     {
      n=359;
      i=-58;
      betak=-919;
      v=-909;
      vv=81;
     }
   switch(State.m_rstate.stage)
     {
      case 0:
         label=0;
         break;
      case 1:
         label=1;
         break;
      case 2:
         label=2;
         break;
      case 3:
         label=3;
         break;
      case 4:
         label=4;
         break;
      case 5:
         label=5;
         break;
      case 6:
         label=6;
         break;
      case 7:
         label=7;
         break;
      case 8:
         label=8;
         break;
      case 9:
         label=9;
         break;
      case 10:
         label=10;
         break;
      case 11:
         label=11;
         break;
      case 12:
         label=12;
         break;
      case 13:
         label=13;
         break;
      case 14:
         label=14;
         break;
      case 15:
         label=15;
         break;
      case 16:
         label=16;
         break;
      case 17:
         label=17;
         break;
      default:
         //--- Routine body
         //--- Prepare
         n=State.m_n;
         State.m_terminationneeded=false;
         State.m_userterminationneeded=false;
         State.m_repterminationtype=0;
         State.m_repiterationscount=0;
         State.m_repnfev=0;
         State.m_debugrestartscount=0;
         COptServ::SmoothnessMonitorInit(State.m_smonitor,State.m_s,n,1,State.m_smoothnessguardlevel>0);
         State.m_lastscaleused=State.m_s;
         State.m_invs=State.m_s.Pow(-1.0)+0;
         //---  Check, that transferred derivative value is right
         ClearRequestFields(State);
         if(!(State.m_diffstep==0.0 && State.m_teststep>0.0))
            label=18;
         else
            label=20;
         break;
     }
//--- Main loop
   while(label>=0)
      switch(label)
        {
         case 20:
            if(!COptServ::SmoothnessMonitorCheckGradientATX0(State.m_smonitor,State.m_xbase,State.m_s,State.m_s,State.m_s,false,State.m_teststep))
              {
               label=21;
               break;
              }
            State.m_x=State.m_smonitor.m_x;
            State.m_needfg=true;
            State.m_rstate.stage=0;
            label=-1;
            break;
         case 0:
            State.m_needfg=false;
            State.m_smonitor.m_fi.Set(0,State.m_f);
            State.m_smonitor.m_j.Row(0,State.m_g);
            label=20;
            break;
         case 21:
         case 18:
            //--- Preparations continue:
            //--- * set XK
            //--- * calculate F/G
            //--- * set DK to -G
            //--- * powerup algo (it may change preconditioner)
            //--- * apply preconditioner to DK
            //--- * report update of X
            //--- * check stopping conditions for G
            State.m_x=State.m_xbase;
            State.m_xk=State.m_x;
            ClearRequestFields(State);
            if(State.m_diffstep!=0.0)
              {
               label=22;
               break;
              }
            State.m_needfg=true;
            State.m_rstate.stage=1;
            label=-1;
            break;
         case 1:
            State.m_needfg=false;
            label=23;
            break;
         case 22:
            State.m_needf=true;
            State.m_rstate.stage=2;
            label=-1;
            break;
         case 2:
            State.m_fbase=State.m_f;
            i=0;
         case 24:
            if(i>n-1)
              {
               label=26;
               break;
              }
            v=State.m_x[i];
            State.m_x.Set(i,v-State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=3;
            label=-1;
            break;
         case 3:
            State.m_fm2=State.m_f;
            State.m_x.Set(i,v-0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=4;
            label=-1;
            break;
         case 4:
            State.m_fm1=State.m_f;
            State.m_x.Set(i,v+0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=5;
            label=-1;
            break;
         case 5:
            State.m_fp1=State.m_f;
            State.m_x.Set(i,v+State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=6;
            label=-1;
            break;
         case 6:
            State.m_fp2=State.m_f;
            State.m_x.Set(i,v);
            State.m_g.Set(i,(8*(State.m_fp1-State.m_fm1)-(State.m_fp2-State.m_fm2))/(6*State.m_diffstep*State.m_s[i]));
            i++;
            label=24;
            break;
         case 26:
            State.m_f=State.m_fbase;
            State.m_needf=false;
         case 23:
            if(!State.m_drep)
              {
               label=27;
               break;
              }
            //--- Report algorithm powerup (if needed)
            ClearRequestFields(State);
            State.m_algpowerup=true;
            State.m_rstate.stage=7;
            label=-1;
            break;
         case 7:
            State.m_algpowerup=false;
         case 27:
            COptServ::TrimPrepare(State.m_f,State.m_trimthreshold);
            State.m_dk=State.m_g.ToVector()*(-1.0);
            PreconditionedMultiply(State,State.m_dk,State.m_work0,State.m_work1);
            if(!State.m_xrep)
              {
               label=29;
               break;
              }
            ClearRequestFields(State);
            State.m_xupdated=true;
            State.m_rstate.stage=8;
            label=-1;
            break;
         case 8:
            State.m_xupdated=false;
         case 29:
            if(State.m_terminationneeded || State.m_userterminationneeded)
              {
               //--- Combined termination point for "internal" termination by TerminationNeeded flag
               //--- and for "user" termination by MinCGRequestTermination() (UserTerminationNeeded flag).
               //--- In this location rules for both of methods are same, thus only one exit point is needed.
               State.m_xn=State.m_xk;
               State.m_repterminationtype=8;
               return(false);
              }
            v=MathPow(State.m_g*State.m_s+0,2.0).Sum();
            if(MathSqrt(v)<=State.m_epsg)
              {
               State.m_xn=State.m_xk;
               State.m_repterminationtype=4;
               return(false);
              }
            State.m_repnfev=1;
            State.m_k=0;
            State.m_fold=State.m_f;
            //--- Choose initial step.
            //--- Apply preconditioner, if we have something other than default.
            if(State.m_prectype==2 || State.m_prectype==3)
              {
               //--- because we use preconditioner, step length must be equal
               //--- to the norm of DK
               v=State.m_dk.Dot(State.m_dk);
               State.m_lastgoodstep=MathSqrt(v);
              }
            else
              {
               //--- No preconditioner is used, we try to use suggested step
               if(State.m_suggestedstep>0.0)
                  State.m_lastgoodstep=State.m_suggestedstep;
               else
                  State.m_lastgoodstep=1.0;
              }
            //--- Main cycle
            State.m_rstimer=m_rscountdownlen;
         case 31:
            //--- * clear reset flag
            //--- * clear termination flag
            //--- * store G[k] for later calculation of Y[k]
            //--- * prepare starting point and direction and step length for line search
            State.m_innerresetneeded=false;
            State.m_terminationneeded=false;
            State.m_yk=State.m_g*(-1.0)+0;
            State.m_d=State.m_dk;
            State.m_x=State.m_xk;
            State.m_mcstage=0;
            State.m_stp=1.0;
            CLinMin::LinMinNormalized(State.m_d,State.m_stp,n);
            if(State.m_lastgoodstep!=0.0)
               State.m_stp=State.m_lastgoodstep;
            State.m_curstpmax=State.m_stpmax;
            //--- Report beginning of line search (if needed)
            //--- Terminate algorithm, if user request was detected
            if(!State.m_drep)
              {
               label=33;
               break;
              }
            ClearRequestFields(State);
            State.m_lsstart=true;
            State.m_rstate.stage=9;
            label=-1;
            break;
         case 9:
            State.m_lsstart=false;
         case 33:
            if(State.m_terminationneeded)
              {
               State.m_xn=State.m_x;
               State.m_repterminationtype=8;
               return(false);
              }
            //--- Minimization along D
            COptServ::SmoothnessMonitorStartLineSearch1u(State.m_smonitor,State.m_s,State.m_invs,State.m_x,State.m_f,State.m_g);
            CLinMin::MCSrch(n,State.m_x,State.m_f,State.m_g,State.m_d,State.m_stp,State.m_curstpmax,m_gtol,State.m_mcinfo,State.m_nfev,State.m_work0,State.m_lstate,State.m_mcstage);
         case 35:
            if(State.m_mcstage==0)
              {
               label=36;
               break;
              }
            //--- Calculate function/gradient using either
            //--- analytical gradient supplied by user
            //--- or finite difference approximation.
            //--- "Trim" function in order to handle near-singularity points.
            ClearRequestFields(State);
            if(State.m_diffstep!=0.0)
              {
               label=37;
               break;
              }
            State.m_needfg=true;
            State.m_rstate.stage=10;
            label=-1;
            break;
         case 10:
            State.m_needfg=false;
            label=38;
            break;
         case 37:
            State.m_needf=true;
            State.m_rstate.stage=11;
            label=-1;
            break;
         case 11:
            State.m_fbase=State.m_f;
            i=0;
         case 39:
            if(i>n-1)
              {
               label=41;
               break;
              }
            v=State.m_x[i];
            State.m_x.Set(i,v-State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=12;
            label=-1;
            break;
         case 12:
            State.m_fm2=State.m_f;
            State.m_x.Set(i,v-0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=13;
            label=-1;
            break;
         case 13:
            State.m_fm1=State.m_f;
            State.m_x.Set(i,v+0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=14;
            label=-1;
            break;
         case 14:
            State.m_fp1=State.m_f;
            State.m_x.Set(i,v+State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=15;
            label=-1;
            break;
         case 15:
            State.m_fp2=State.m_f;
            State.m_x.Set(i,v);
            State.m_g.Set(i,(8*(State.m_fp1-State.m_fm1)-(State.m_fp2-State.m_fm2))/(6*State.m_diffstep*State.m_s[i]));
            i++;
            label=39;
            break;
         case 41:
            State.m_f=State.m_fbase;
            State.m_needf=false;
         case 38:
            COptServ::SmoothnessMonitorEnqueuePoint1u(State.m_smonitor,State.m_s,State.m_invs,State.m_d,State.m_stp,State.m_x,State.m_f,State.m_g);
            COptServ::TrimFunction(State.m_f,State.m_g,n,State.m_trimthreshold);
            //--- Call MCSRCH again
            CLinMin::MCSrch(n,State.m_x,State.m_f,State.m_g,State.m_d,State.m_stp,State.m_curstpmax,m_gtol,State.m_mcinfo,State.m_nfev,State.m_work0,State.m_lstate,State.m_mcstage);
            label=35;
            break;
         case 36:
            COptServ::SmoothnessMonitorFinalizeLineSearch(State.m_smonitor);
            //---*terminate algorithm if "user" request for detected
            //--- * report end of line search
            //--- * store current point to XN
            //--- * report iteration
            //---*terminate algorithm if "internal" request was detected
            if(State.m_userterminationneeded)
              {
               State.m_xn=State.m_xk;
               State.m_repterminationtype=8;
               return(false);
              }
            if(!State.m_drep)
              {
               label=42;
               break;
              }
            //--- Report end of line search (if needed)
            ClearRequestFields(State);
            State.m_lsend=true;
            State.m_rstate.stage=16;
            label=-1;
            break;
         case 16:
            State.m_lsend=false;
         case 42:
            State.m_xn=State.m_x;
            if(!State.m_xrep)
              {
               label=44;
               break;
              }
            ClearRequestFields(State);
            State.m_xupdated=true;
            State.m_rstate.stage=17;
            label=-1;
            break;
         case 17:
            State.m_xupdated=false;
         case 44:
            if(State.m_terminationneeded)
              {
               State.m_xn=State.m_x;
               State.m_repterminationtype=8;
               return(false);
              }
            //--- Line search is finished.
            //--- * calculate BetaK
            //--- * calculate DN
            //--- * update timers
            //--- * calculate step length:
            //---   * LastScaledStep is ALWAYS calculated because it is used in the stopping criteria
            //---   * LastGoodStep is updated only when MCINFO is equal to 1 (Wolfe conditions hold).
            //---     See below for more explanation.
            if(State.m_mcinfo==1 && !State.m_innerresetneeded)
              {
               //--- Standard Wolfe conditions hold
               //--- Calculate Y[K] and D[K]'*Y[K]
               State.m_yk+=State.m_g;
               vv=State.m_yk.Dot(State.m_dk);
               //--- Calculate BetaK according to DY formula
               v=PreconditionedMultiply2(State,State.m_g,State.m_g,State.m_work0,State.m_work1);
               State.m_betady=v/vv;
               //--- Calculate BetaK according to HS formula
               v=PreconditionedMultiply2(State,State.m_g,State.m_yk,State.m_work0,State.m_work1);
               State.m_betahs=v/vv;
               //--- Choose BetaK
               if(State.m_cgtype==0)
                  betak=State.m_betady;
               if(State.m_cgtype==1)
                  betak=MathMax(0,MathMin(State.m_betady,State.m_betahs));
              }
            else
              {
               //--- Something is wrong (may be function is too wild or too flat)
               //--- or we just have to restart algo.
               //--- We'll set BetaK=0, which will restart CG algorithm.
               //--- We can stop later (during normal checks) if stopping conditions are met.
               //
               betak=0;
               State.m_debugrestartscount++;
              }
            if(State.m_repiterationscount>0 && State.m_repiterationscount%(3+n)==0)
              {
               //--- clear Beta every N iterations
               betak=0;
              }
            if(State.m_mcinfo==1 || State.m_mcinfo==5)
               State.m_rstimer=m_rscountdownlen;
            else
               State.m_rstimer--;
            State.m_dn=State.m_g*(-1.0)+0;
            PreconditionedMultiply(State,State.m_dn,State.m_work0,State.m_work1);
            State.m_dn+=State.m_dk.ToVector()*betak;
            State.m_lastscaledstep=MathPow(State.m_d.ToVector()/State.m_s.ToVector(),2.0).Sum();
            State.m_lastscaledstep=State.m_stp*MathSqrt(State.m_lastscaledstep);
            if(State.m_mcinfo==1)
              {
               //--- Step is good (Wolfe conditions hold), update LastGoodStep.
               //--- This check for MCINFO=1 is essential because sometimes in the
               //--- constrained optimization setting we may take very short steps
               //--- (like 1E-15) because we were very close to boundary of the
               //--- feasible area. Such short step does not mean that we've converged
               //--- to the solution - it was so short because we were close to the
               //--- boundary and there was a limit on step length.
               //--- So having such short step is quite normal situation. However, we
               //--- should NOT start next iteration from step whose initial length is
               //--- estimated as 1E-15 because it may lead to the failure of the
               //--- linear minimizer (step is too short, function does not changes,
               //--- line search stagnates).
               State.m_lastgoodstep=CAblasF::RDotV2(n,State.m_d);
               State.m_lastgoodstep=State.m_stp*MathSqrt(State.m_lastgoodstep);
              }
            //--- Update information.
            //--- Check stopping conditions.
            v=MathPow(State.m_g.ToVector()*State.m_s.ToVector(),2.0).Sum();
            if(!MathIsValidNumber(v) || !MathIsValidNumber(State.m_f))
              {
               //--- Abnormal termination - infinities in function/gradient
               State.m_repterminationtype=-8;
               return(false);
              }
            State.m_repnfev+=State.m_nfev;
            State.m_repiterationscount++;
            if(State.m_repiterationscount>=State.m_maxits && State.m_maxits>0)
              {
               //--- Too many iterations
               State.m_repterminationtype=5;
               return(false);
              }
            if(MathSqrt(v)<=State.m_epsg)
              {
               //--- Gradient is small enough
               State.m_repterminationtype=4;
               return(false);
              }
            if(!State.m_innerresetneeded)
              {
               //--- These conditions are checked only when no inner reset was requested by user
               if((double)(State.m_fold-State.m_f)<=(double)(State.m_epsf*MathMax(MathAbs(State.m_fold),MathMax(MathAbs(State.m_f),1.0))))
                 {
                  //--- F(k+1)-F(k) is small enough
                  State.m_repterminationtype=1;
                  return(false);
                 }
               if(State.m_lastscaledstep<=State.m_epsx)
                 {
                  //--- X(k+1)-X(k) is small enough
                  State.m_repterminationtype=2;
                  return(false);
                 }
              }
            if(State.m_rstimer<=0)
              {
               //--- Too many subsequent restarts
               State.m_repterminationtype=7;
               return(false);
              }
            //--- Shift Xk/Dk, update other information
            State.m_xk=State.m_xn;
            State.m_dk=State.m_dn;
            State.m_fold=State.m_f;
            State.m_k++;
            label=31;
            break;
         case 32:
            return(false);
        }
//--- Saving State
   State.m_rstate.ia.Set(0,n);
   State.m_rstate.ia.Set(1,i);
   State.m_rstate.ra.Set(0,betak);
   State.m_rstate.ra.Set(1,v);
   State.m_rstate.ra.Set(2,vv);
//--- return result
   return(true);
  }
//+------------------------------------------------------------------+
//| This structure is a SNNLS(Specialized Non-Negative Least Squares)|
//| m_solver.                                                          |
//| It solves problems of the form | A*x - b | ^ 2 => min subject to |
//| non-negativity constraints on SOME components of x, with         |
//| structured A(first NS columns are just unit matrix, next ND      |
//| columns store dense part).                                       |
//| This m_solver is suited for solution of many sequential NNLS       |
//| subproblems - it keeps track of previously allocated memory and  |
//| reuses it as much as possible.                                   |
//+------------------------------------------------------------------+
struct CSNNLSSolver
  {
   int               m_debugmaxinnerits;
   int               m_nd;
   int               m_nr;
   int               m_ns;
   double            m_debugflops;
   bool              m_nnc[];
   CRowInt           m_rdtmprowmap;
   CRowDouble        m_b;
   CRowDouble        m_cb;
   CRowDouble        m_cborg;
   CRowDouble        m_crb;
   CRowDouble        m_cx;
   CRowDouble        m_d;
   CRowDouble        m_diagaa;
   CRowDouble        m_dx;
   CRowDouble        m_g;
   CRowDouble        m_r;
   CRowDouble        m_regdiag;
   CRowDouble        m_tmp0;
   CRowDouble        m_tmp1;
   CRowDouble        m_tmp2;
   CRowDouble        m_tmpcholesky;
   CRowDouble        m_trdd;
   CRowDouble        m_xn;
   CRowDouble        m_xp;
   CMatrixDouble     m_densea;
   CMatrixDouble     m_tmpca;
   CMatrixDouble     m_tmplq;
   CMatrixDouble     m_trda;
                     CSNNLSSolver(void);
                    ~CSNNLSSolver(void) {}
   void              Copy(const CSNNLSSolver &obj);
   //--- overloading
   void              operator=(const CSNNLSSolver &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSNNLSSolver::CSNNLSSolver(void)
  {
   m_debugmaxinnerits=0;
   m_nd=0;
   m_nr=0;
   m_ns=0;
   m_debugflops=0;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CSNNLSSolver::Copy(const CSNNLSSolver &obj)
  {
   m_debugmaxinnerits=obj.m_debugmaxinnerits;
   m_nd=obj.m_nd;
   m_nr=obj.m_nr;
   m_ns=obj.m_ns;
   m_debugflops=obj.m_debugflops;
   ArrayCopy(m_nnc,obj.m_nnc);
   m_rdtmprowmap=obj.m_rdtmprowmap;
   m_b=obj.m_b;
   m_cb=obj.m_cb;
   m_cborg=obj.m_cborg;
   m_crb=obj.m_crb;
   m_cx=obj.m_cx;
   m_d=obj.m_d;
   m_diagaa=obj.m_diagaa;
   m_dx=obj.m_dx;
   m_g=obj.m_g;
   m_r=obj.m_r;
   m_regdiag=obj.m_regdiag;
   m_tmp0=obj.m_tmp0;
   m_tmp1=obj.m_tmp1;
   m_tmp2=obj.m_tmp2;
   m_tmpcholesky=obj.m_tmpcholesky;
   m_trdd=obj.m_trdd;
   m_xn=obj.m_xn;
   m_xp=obj.m_xp;
   m_densea=obj.m_densea;
   m_tmpca=obj.m_tmpca;
   m_tmplq=obj.m_tmplq;
   m_trda=obj.m_trda;
  }
//+------------------------------------------------------------------+
//| Specialized Non-Negative Least Squares                           |
//+------------------------------------------------------------------+
class CSNNLS
  {
public:
   static void       SNNLSInit(int nsmax,int ndmax,int nrmax,CSNNLSSolver &s);
   static void       SNNLSSetProblem(CSNNLSSolver &s,CMatrixDouble &a,CRowDouble &b,int ns,int nd,int nr);
   static void       SNNLSDropNNC(CSNNLSSolver &s,int idx);
   static void       SNNLSSolve(CSNNLSSolver &s,CRowDouble &x);

private:
   static void       FuncGradU(CSNNLSSolver &s,CRowDouble &x,CRowDouble &r,CRowDouble &g,double &f);
   static void       Func(CSNNLSSolver &s,CRowDouble &x,double &f);
   static void       TRDPrepare(CSNNLSSolver &s,CRowDouble &x,CRowDouble &diag,double lambdav,CRowDouble &trdd,CMatrixDouble &trda,CRowDouble &m_tmp0,CRowDouble &m_tmp1,CRowDouble &tmp2,CMatrixDouble &tmplq);
   static void       TRDSolve(CRowDouble &trdd,CMatrixDouble &trda,int ns,int nd,CRowDouble &d);
   static void       TRDFixVariable(CRowDouble &trdd,CMatrixDouble &trda,int ns,int nd,int idx,CRowDouble &tmp);
  };
//+------------------------------------------------------------------+
//| This subroutine is used to initialize SNNLS m_solver.              |
//| By default, empty NNLS problem is produced, but we allocated     |
//| enough space to store problems with NSMax + NDMax columns and    |
//| NRMax rows. It is good place to provide algorithm with initial   |
//| estimate of the space requirements, although you may             |
//| underestimate problem size or even pass zero estimates - in this |
//| case buffer variables will be resized automatically when you set |
//| NNLS problem.                                                    |
//| Previously allocated buffer variables are reused as much as      |
//| possible. This function does not clear structure completely, it  |
//| tries to preserve as much dynamically allocated memory as        |
//| possible.                                                        |
//+------------------------------------------------------------------+
void CSNNLS::SNNLSInit(int nsmax,
                       int ndmax,
                       int nrmax,
                       CSNNLSSolver &s)
  {
   s.m_ns=0;
   s.m_nd=0;
   s.m_nr=0;
   s.m_debugflops=0.0;
   s.m_debugmaxinnerits=0;
   s.m_densea.Resize(nrmax,ndmax);
   s.m_tmpca.Resize(nrmax,ndmax);
   s.m_b.Resize(nrmax);
   ArrayResize(s.m_nnc,nsmax+ndmax);
  }
//+------------------------------------------------------------------+
//| This subroutine is used to set NNLS problem:                     |
//|         ([ 1     |      ]   [   ]   [   ]) ^ 2                   |
//|         ([   1   |      ]   [   ]   [   ])                       |
//|      min([     1 |  Ad  ] * [ x ] - [ b ])    s.m_t. x >= 0        |
//|         ([       |      ]   [   ]   [   ])                       |
//|         ([       |      ]   [   ]   [   ])                       |
//| where:                                                           |
//|   * identity matrix has NS*NS size(NS <= NR, NS can be zero)     |
//|   * dense matrix Ad has NR*ND size                               |
//|   * b is NR * 1 vector                                           |
//|   * x is(NS + ND) * 1 vector                                     |
//|   * all elements of x are Non - Negative(this constraint can     |
//|     be removed later by calling SNNLSDropNNC() function)         |
//| Previously allocated buffer variables are reused as much as      |
//| possible. After you set problem, you can solve it with           |
//| SNNLSSolve().                                                    |
//| INPUT PARAMETERS:                                                |
//|   S        -  SNNLS m_solver, must be initialized with SNNLSInit() |
//|               call                                               |
//|   A        -  array[NR, ND], dense part of the system            |
//|   B        -  array[NR], right part                              |
//|   NS       -  size of the sparse part of the system, 0<=NS<=NR   |
//|   ND       -  size of the dense part of the system, ND >= 0      |
//|   NR       -  rows count, NR > 0                                 |
//| NOTE 1: You can have NS + ND = 0, m_solver will correctly accept   |
//|         such combination and return empty array as problem       |
//|         solution.                                                |
//+------------------------------------------------------------------+
void CSNNLS::SNNLSSetProblem(CSNNLSSolver &s,
                             CMatrixDouble &a,
                             CRowDouble &b,
                             int ns,
                             int nd,
                             int nr)
  {
//--- check
   if(!CAp::Assert(nd>=0,__FUNCTION__+": ND<0"))
      return;
   if(!CAp::Assert(ns>=0,__FUNCTION__+": NS<0"))
      return;
   if(!CAp::Assert(nr>0,__FUNCTION__+": NR<=0"))
      return;
   if(!CAp::Assert(ns<=nr,__FUNCTION__+": NS>NR"))
      return;
   if(!CAp::Assert(a.Rows()>=nr || nd==0,__FUNCTION__+": rows(A)<NR"))
      return;
   if(!CAp::Assert(a.Cols()>=nd,__FUNCTION__+": cols(A)<ND"))
      return;
   if(!CAp::Assert(b.Size()>=nr,__FUNCTION__+": length(B)<NR"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteMatrix(a,nr,nd),__FUNCTION__+": A contains INF/NAN"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(b,nr),__FUNCTION__+": B contains INF/NAN"))
      return;
//--- Copy problem
   s.m_ns=ns;
   s.m_nd=nd;
   s.m_nr=nr;
   if(nd>0)
     {
      s.m_densea=a;
      s.m_densea.Resize(nr,nd);
     }
   s.m_b=b;
   s.m_b.Resize(nr);
   ArrayResize(s.m_nnc,ns+nd);
   ArrayInitialize(s.m_nnc,true);
  }
//+------------------------------------------------------------------+
//| This subroutine drops non-negativity constraint from the  problem|
//| set by SNNLSSetProblem() call. This function must be called AFTER|
//| problem is set, because each SetProblem() call resets constraints|
//| to their default State (all constraints are present).            |
//| INPUT PARAMETERS:                                                |
//|   S        -  SNNLS m_solver, must be initialized with SNNLSInit() |
//|               call, problem must be set with SNNLSSetProblem()   |
//|               call.                                              |
//|   Idx      -  constraint index, 0 <= IDX < NS + ND               |
//+------------------------------------------------------------------+
void CSNNLS::SNNLSDropNNC(CSNNLSSolver &s,int idx)
  {
//--- check
   if(!CAp::Assert(idx>=0,__FUNCTION__+": Idx<0"))
      return;
   if(!CAp::Assert(idx<s.m_ns+s.m_nd,__FUNCTION__+": Idx>=NS+ND"))
      return;

   s.m_nnc[idx]=false;
  }
//+------------------------------------------------------------------+
//| This subroutine is used to solve NNLS problem.                   |
//| INPUT PARAMETERS:                                                |
//|   S        -  SNNLS m_solver, must be initialized with SNNLSInit() |
//|               call and problem must be set up with               |
//|               SNNLSSetProblem() call.                            |
//|   X        -  possibly preallocated buffer, automatically resized|
//|               if needed                                          |
//| OUTPUT PARAMETERS:                                               |
//|   X        -  array[NS + ND], solution                           |
//| NOTE:                                                            |
//|   1. You can have NS + ND = 0, m_solver will correctly accept such |
//|      combination and return empty array as problem solution.     |
//|   2. Internal field S.DebugFLOPS contains rough estimate of FLOPs|
//|      used to solve problem. It can be used for debugging         |
//|      purposes. This field is real - valued.                      |
//+------------------------------------------------------------------+
void CSNNLS::SNNLSSolve(CSNNLSSolver &s,CRowDouble &x)
  {
//--- create variables
   int    i=0;
   int    ns=s.m_ns;
   int    nd=s.m_nd;
   int    nr=s.m_nr;
   bool   wasactivation=false;
   double lambdav=0;
   double v0=0;
   double v1=0;
   double v=0;
   int    outerits=0;
   int    innerits=0;
   int    maxouterits=0;
   double xtol=0;
   double kicklength=0;
   bool   kickneeded=false;
   double f0=0;
   double f1=0;
   double dnrm=0;
   int    actidx=0;
   double stp=0;
   double stpmax=0;
//--- Prepare
   s.m_debugflops=0.0;
//--- Handle special cases:
//--- * NS+ND=0
//--- * ND=0
   if(ns+nd==0)
      return;
   if(nd==0)
     {
      x=vector<double>::Zeros(ns);
      for(i=0; i<ns; i++)
        {
         if(s.m_nnc[i])
            x.Set(i,MathMax(s.m_b[i],0.0));
         else
            x.Set(i,s.m_b[i]);
        }
      return;
     }
//--- Main cycle of BLEIC-SNNLS algorithm.
//--- Below we assume that ND>0.
   x=vector<double>::Zeros(ns+nd);
   s.m_xn.Resize(ns+nd);
   s.m_xp.Resize(ns+nd);
   s.m_g.Resize(ns+nd);
   s.m_d.Resize(ns+nd);
   s.m_r.Resize(nr);
   s.m_diagaa.Resize(nd);
   s.m_regdiag=vector<double>::Ones(ns+nd);
   s.m_dx.Resize(ns+nd);
   lambdav=1.0E6*CMath::m_machineepsilon;
   maxouterits=10;
   outerits=0;
   innerits=0;
   xtol=1.0E3*CMath::m_machineepsilon;
   kicklength=MathSqrt(CMath::m_minrealnumber);
   while(true)
     {
      //--- Initial check for correctness of X
      for(i=0; i<ns+nd; i++)
        {
         //--- check
         if(!CAp::Assert(!s.m_nnc[i] || x[i]>=0.0,__FUNCTION__+": integrity check failed"))
            return;
        }
      //--- Calculate gradient G and constrained descent direction D
      FuncGradU(s,x,s.m_r,s.m_g,f0);
      for(i=0; i<ns+nd; i++)
        {
         if(s.m_nnc[i] && x[i]==0.0 && s.m_g[i]>0.0)
            s.m_d.Set(i,0.0);
         else
            s.m_d.Set(i,-s.m_g[i]);
        }
      //--- Decide whether we need "kick" stage: special stage
      //--- that moves us away from boundary constraints which are
      //--- not strictly active (i.e. such constraints that x[i]=0.0 and d[i]>0).
      //--- If we need kick stage, we make a kick - and restart iteration.
      //--- If not, after this block we can rely on the fact that
      //--- for all x[i]=0.0 we have d[i]=0.0
      //--- NOTE: we do not increase outer iterations counter here
      kickneeded=false;
      for(i=0; i<ns+nd; i++)
        {
         if(s.m_nnc[i] && x[i]==0.0 && s.m_d[i]>0.0)
            kickneeded=true;
        }
      if(kickneeded)
        {
         //--- Perform kick.
         //--- Restart.
         //--- Do not increase iterations counter.
         for(i=0; i<ns+nd; i++)
           {
            if(x[i]==0.0 && s.m_d[i]>0.0)
               x.Add(i,kicklength);
           }
         continue;
        }
      //--- Newton phase
      //--- Reduce problem to constrained triangular form and perform Newton
      //--- steps with quick activation of constrants  (triangular  form  is
      //--- updated in order to handle changed constraints).
      s.m_xp=x;
      TRDPrepare(s,x,s.m_regdiag,lambdav,s.m_trdd,s.m_trda,s.m_tmp0,s.m_tmp1,s.m_tmp2,s.m_tmplq);
      while(true)
        {
         //--- Skip if debug limit on inner iterations count is turned on.
         if(s.m_debugmaxinnerits>0 && innerits>=s.m_debugmaxinnerits)
            break;
         //--- Prepare step vector.
         FuncGradU(s,x,s.m_r,s.m_g,f0);
         for(i=0; i<ns+nd; i++)
           {
            if(s.m_nnc[i] && x[i]==0.0)
               s.m_d.Set(i,0.0);
            else
               s.m_d.Set(i,-s.m_g[i]);
           }
         TRDSolve(s.m_trdd,s.m_trda,ns,nd,s.m_d);
         //--- Perform unconstrained trial step and compare function values.
         for(i=0; i<ns+nd; i++)
           {
            s.m_xn.Set(i,x[i]+s.m_d[i]);
           }
         Func(s,s.m_xn,f1);
         if(f1>=f0)
            break;
         //--- Calculate length of D, maximum step and component which is
         //--- activated by this step. Break if D is exactly zero.
         dnrm=CAblasF::RDotV2(ns+nd,s.m_d);
         dnrm=MathSqrt(dnrm);
         actidx=-1;
         stpmax=1.0E50;
         for(i=0; i<ns+nd; i++)
           {
            if(s.m_nnc[i] && s.m_d[i]<0.0)
              {
               v=stpmax;
               stpmax=CApServ::SafeMinPosRV(x[i],-s.m_d[i],stpmax);
               if(stpmax<v)
                  actidx=i;
              }
           }
         if(dnrm==0.0)
            break;
         //--- Perform constrained step and update X
         //--- and triangular model.
         stp=MathMin(1.0,stpmax);
         for(i=0; i<ns+nd; i++)
           {
            v=x[i]+stp*s.m_d[i];
            if(s.m_nnc[i])
               v=MathMax(v,0.0);
            s.m_xn.Set(i,v);
           }
         if(stp==stpmax && actidx>=0)
            s.m_xn.Set(actidx,0.0);
         wasactivation=false;
         for(i=0; i<ns+nd; i++)
           {
            if(s.m_xn[i]==0.0 && x[i]!=0.0)
              {
               wasactivation=true;
               TRDFixVariable(s.m_trdd,s.m_trda,ns,nd,i,s.m_tmpcholesky);
              }
           }
         for(i=0; i<ns+nd; i++)
            x.Set(i,s.m_xn[i]);
         //--- Increment iterations counter.
         //--- Terminate if no constraint was activated.
         innerits++;
         if(!wasactivation)
            break;
        }
      //--- Update outer iterations counter.
      //--- Break if necessary:
      //--- * maximum number of outer iterations performed
      //--- * relative change in X is small enough
      outerits++;
      if(outerits>=maxouterits)
         break;
      v=0;
      for(i=0; i<ns+nd; i++)
        {
         v0=MathAbs(s.m_xp[i]);
         v1=MathAbs(x[i]);
         if(v0!=0.0 || v1!=0.0)
            v=MathMax(v,MathAbs(x[i]-s.m_xp[i])/MathMax(v0,v1));
        }
      if(v<=xtol)
         break;
     }
  }
//+------------------------------------------------------------------+
//| This function calculates:                                        |
//|   * residual vector R = A * x - b                                |
//|   * unconstrained gradient vector G                              |
//|   * function value F = 0.5 * | R | ^ 2                           |
//| R and G must have at least N elements.                           |
//+------------------------------------------------------------------+
void CSNNLS::FuncGradU(CSNNLSSolver &s,
                       CRowDouble &x,
                       CRowDouble &r,
                       CRowDouble &g,
                       double &f)
  {
//--- create variables
   int    i=0;
   int    nr=s.m_nr;
   int    nd=s.m_nd;
   int    ns=s.m_ns;
   double v=0;
   int    i_=0;
   int    i1_=0;
//--- initialization
   f=0;

   for(i=0; i<nr; i++)
     {
      i1_=ns;
      v=0.0;
      for(i_=0; i_<nd; i_++)
         v+=s.m_densea.Get(i,i_)*x[i_+i1_];
      if(i<ns)
         v+= x[i];
      v-=s.m_b[i];
      r.Set(i,v);
      f+=0.5*v*v;
     }
   for(i=0; i<ns; i++)
      g.Set(i,r[i]);
   for(i=ns; i<ns+nd; i++)
      g.Set(i,0.0);
   for(i=0; i<nr; i++)
     {
      v=r[i];
      i1_=-ns;
      for(i_=ns; i_<ns+nd; i_++)
         g.Add(i_,v*s.m_densea.Get(i,i_+i1_));
     }
  }
//+------------------------------------------------------------------+
//| This function calculates function value F = 0.5 * | R | ^ 2 at X.|
//+------------------------------------------------------------------+
void CSNNLS::Func(CSNNLSSolver &s,CRowDouble &x,double &f)
  {
//--- create variables
   int    i=0;
   int    nr=s.m_nr;
   int    nd=s.m_nd;
   int    ns=s.m_ns;
   double v=0;
   int    i_=0;
   int    i1_=0;
//--- initialization
   f=0;

   for(i=0; i<nr; i++)
     {
      i1_=ns;
      v=0.0;
      for(i_=0; i_<nd; i_++)
         v+=s.m_densea.Get(i,i_)*x[i_+i1_];
      if(i<ns)
         v+= x[i];
      v-=s.m_b[i];
      f+=0.5*v*v;
     }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSNNLS::TRDPrepare(CSNNLSSolver &s,CRowDouble &x,
                        CRowDouble &diag,double lambdav,
                        CRowDouble &trdd,CMatrixDouble &trda,
                        CRowDouble &tmp0,CRowDouble &tmp1,
                        CRowDouble &tmp2,CMatrixDouble &tmplq)
  {
//--- create variables
   int    i=0;
   int    j=0;
   int    nr=s.m_nr;
   int    nd=s.m_nd;
   int    ns=s.m_ns;
   double v=0;
   double cs=0;
   double sn=0;
   double r=0;
//--- Triangular reduction
   trdd.Resize(ns);
   trda.Resize(ns+nd,nd);
   tmplq.Resize(nd,nr+nd);
   for(i=0; i<ns; i++)
     {
      //--- Apply rotation to I-th row and corresponding row of
      //--- regularizer. Here V is diagonal element of I-th row,
      //--- which is set to 1.0 or 0.0 depending on variable
      //--- status (constrained or not).
      v=1.0;
      if(s.m_nnc[i] && x[i]==0.0)
         v=0.0;
      CRotations::GenerateRotation(v,lambdav,cs,sn,r);
      trdd.Set(i,cs*v+sn*lambdav);
      for(j=0; j<nd; j++)
        {
         v=s.m_densea.Get(i,j);
         trda.Set(i,j,cs*v);
         tmplq.Set(j,i,-(sn*v));
        }
     }
   for(j=0; j<nd; j++)
      for(i=ns; i<nr; i++)
         tmplq.Set(j,i,s.m_densea.Get(i,j));

   for(j=0; j<nd; j++)
     {
      if(s.m_nnc[ns+j] && x[ns+j]==0.0)
        {
         //--- Variable is constrained, entire row is set to zero.
         for(i=0; i<nr; i++)
            tmplq.Set(j,i,0.0);
         for(i=0; i<ns; i++)
            trda.Set(i,j,0.0);
        }
     }
   for(i=0; i<nd; i++)
     {
      for(j=0; j<nd; j++)
         tmplq.Set(j,nr+i,0.0);
      tmplq.Set(i,nr+i,lambdav*diag[i]);
     }
   tmp0=vector<double>::Zeros(nr+nd+1);
   tmp1=vector<double>::Zeros(nr+nd+1);
   tmp2=vector<double>::Zeros(nr+nd+1);
   COrtFac::RMatrixLQBaseCase(tmplq,nd,nr+nd,tmp0,tmp1,tmp2);
   for(i=0; i<nd; i++)
     {
      if(tmplq.Get(i,i)<0.0)
         for(j=i; j<nd; j++)
            tmplq.Mul(j,i,-1.0);
     }
   for(i=0; i<nd; i++)
     {
      for(j=0; j<=i; j++)
         trda.Set(ns+j,i,tmplq.Get(i,j));
     }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSNNLS::TRDSolve(CRowDouble &trdd,CMatrixDouble &trda,
                      int ns,int nd,CRowDouble &d)
  {
//--- create variables
   int    i=0;
   int    j=0;
   double v=0;
//--- Solve U'*y=d first.
//--- This section includes two parts:
//--- * solve diagonal part of U'
//--- * solve dense part of U'
   for(i=0; i<ns; i++)
     {
      d.Mul(i,1.0/trdd[i]);
      v=d[i];
      for(j=0; j<nd; j++)
         d.Add(ns+j,- v*trda.Get(i,j));
     }
   for(i=0; i<nd; i++)
     {
      d.Mul(ns+i,1.0/trda.Get(ns+i,i));
      v=d[ns+i];
      for(j=i+1; j<nd; j++)
         d.Add(ns+j,- v*trda.Get(ns+i,j));
     }
//--- Solve U*x=y then.
//--- This section includes two parts:
//--- * solve trailing triangular part of U
//--- * solve combination of diagonal and dense parts of U
   for(i=nd-1; i>=0; i--)
     {
      v=0.0;
      for(j=i+1; j<nd; j++)
         v+= trda.Get(ns+i,j)*d[ns+j];
      d.Set(ns+i,(d[ns+i]-v)/trda.Get(ns+i,i));
     }
   for(i=ns-1; i>=0; i--)
     {
      v=0.0;
      for(j=0; j<nd; j++)
         v+= trda.Get(i,j)*d[ns+j];
      d.Set(i,(d[i]-v)/trdd[i]);
     }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSNNLS::TRDFixVariable(CRowDouble &trdd,CMatrixDouble &trda,
                            int ns,int nd,int idx,CRowDouble &tmp)
  {
//--- create variables
   int    i=0;
   int    j=0;
   int    k=0;
   double cs=0;
   double sn=0;
   double r=0;
   double v=0;
   double vv=0;
//--- check
   if(!CAp::Assert(ns>=0,__FUNCTION__+": integrity error"))
      return;
   if(!CAp::Assert(nd>=0,__FUNCTION__+": integrity error"))
      return;
   if(!CAp::Assert(ns+nd>0,__FUNCTION__+": integrity error"))
      return;
   if(!CAp::Assert(idx>=0,__FUNCTION__+": integrity error"))
      return;
   if(!CAp::Assert(idx<ns+nd,__FUNCTION__+": integrity error"))
      return;
   tmp.Resize(nd);
//--- Depending on variable index, two situations are possible
   if(idx<ns)
     {
      //--- We fix variable in the diagonal part of the model. It means
      //--- that prior to fixing we have:
      //---     (     |     )
      //---     (  D  |     )
      //---     (     |     )
      //---     (-----|  A  )
      //---     (     |0    )
      //---     (     |00   )
      //---     (     |000  )
      //---     (     |0000 )
      //---     (     |00000)
      //--- then we replace idx-th column by zeros:
      //---     (D 0  |     )
      //---     (  0  |     )
      //---     (  0 D|     )
      //---     (-----|  A  )
      //---     (     |     )
      //---     (     |     )
      //---     (     |     )
      //--- and append row with unit element to bottom, in order to
      //--- regularize problem
      //---     (D 0  |     )
      //---     (  0  |     )
      //---     (  0 D|     )
      //---     (-----|  A  )
      //---     (     |     )
      //---     (     |     )
      //---     (     |     )
      //---     (00100|00000) <- appended
      //--- and then we nullify this row by applying rotations:
      //---     (D 0  |     )
      //---     (  0  |     ) <- first rotation is applied here
      //---     (  0 D|     )
      //---     (-----|  A  ) <- subsequent rotations are applied to this row and rows below
      //---     (     |     )
      //---     (     |     )
      //---     (     |     )
      //---     (  0  |  0  ) <- as result, row becomes zero
      //--- and triangular structure is preserved
      if(nd==0)
        {
         //--- Quick exit for empty dense part
         trdd.Set(idx,1.0);
         return;
        }
      for(j=0; j<nd; j++)
        {
         //--- Apply first rotation
         tmp.Set(j,trda.Get(idx,j));
         trda.Set(idx,j,0.0);
        }
      trdd.Set(idx,1.0);
      for(i=0; i<nd; i++)
        {
         if(tmp[i]!=0.0)
           {
            //--- Apply subsequent rotations with bottom triangular part of A
            CRotations::GenerateRotation(trda.Get(ns+i,i),tmp[i],cs,sn,r);
            for(j=i; j<nd; j++)
              {
               v=trda.Get(ns+i,j);
               vv=tmp[j];
               trda.Set(ns+i,j,v*cs+vv*sn);
               tmp.Set(j,vv*cs-v*sn);
              }
           }
        }
     }
   else
     {
      //--- We fix variable in the dense part of the model. It means
      //--- that prior to fixing we have:
      //---     (     |     )
      //---     (  D  |     )
      //---     (     |     )
      //---     (-----|  A  )
      //---     (     |0    )
      //---     (     |00   )
      //---     (     |000  )
      //---     (     |0000 )
      //---     (     |00000)
      //--- then we replace idx-th column by zeros:
      //---     (     |  0  )
      //---     (  D  |  0  )
      //---     (     |  0  )
      //---     (-----|A 0 A)
      //---     (     |  0  )
      //---     (     |  0  )
      //---     (     |  0  )
      //--- and append row with unit element to bottom, in order to
      //--- regularize problem
      //---     (     |  0  )
      //---     (  D  |  0  )
      //---     (     |  0  )
      //---     (-----|A 0 A)
      //---     (     |  0  )
      //---     (     |  0  )
      //---     (     |  0  )
      //---     (00000|00100) <- appended
      //--- and then we nullify this row by applying rotations:
      //---     (D 0  |     )
      //---     (  0  |     )
      //---     (  0 D|     )
      //---     (-----|  A  )
      //---     (     |     )
      //---     (     |     ) <- first rotation is applied here
      //---     (     |     ) <- subsequent rotations are applied to rows below
      //---     (  0  |  0  ) <- as result, row becomes zero
      //--- and triangular structure is preserved.
      k=idx-ns;
      for(i=0; i<ns+nd; i++)
         trda.Set(i,k,0.0);
      for(j=k+1; j<nd; j++)
        {
         //--- Apply first rotation
         tmp.Set(j,trda.Get(idx,j));
         trda.Set(idx,j,0.0);
        }
      trda.Set(idx,k,1.0);
      for(i=k+1; i<nd; i++)
        {
         if(tmp[i]!=0.0)
           {
            //--- Apply subsequent rotations with bottom triangular part of A
            CRotations::GenerateRotation(trda.Get(ns+i,i),tmp[i],cs,sn,r);
            for(j=i; j<nd; j++)
              {
               v=trda.Get(ns+i,j);
               vv=tmp[j];
               trda.Set(ns+i,j,v*cs+vv*sn);
               tmp.Set(j,vv*cs-v*sn);
              }
           }
        }
     }
  }
//+------------------------------------------------------------------+
//| This structure describes set of linear constraints (boundary and |
//| general ones) which can be active and inactive. It also has      |
//| functionality to work with current point and current  gradient   |
//| (determine  active  constraints, move current point, project     |
//| gradient into  constrained  subspace,  perform constrained       |
//| preconditioning and so on.                                       |
//| This structure is intended to be used by constrained optimizers  |
//| for management of all constraint - related functionality.        |
//| External code may access following internal fields of the        |
//| structure:                                                       |
//|   XC       -  stores current point, array[N]. can be accessed    |
//|               only in optimization mode                          |
//|   CStatus  -  active set, array[N + NEC + NIC]:                  |
//|               * CStatus[I]>0 I-Th constraint is in the active set|
//|               * CStatus[I]=0 I-Th constraint is at the boundary, |
//|                              but inactive                        |
//|               * CStatus[I]<0 I-Th constraint is far from the     |
//|                              boundary (and inactive)             |
//|               * elements from 0 to N-1 correspond to boundary    |
//|                 constraints                                      |
//|               * elements from N to N + NEC + NIC - 1 correspond  |
//|                 to linear constraints                            |
//|               * elements from N to N + NEC - 1 are always + 1    |
//|   FeasInitPt - SASStartOptimization() sets this flag to True if, |
//|               after enforcement of box constraints, initial point|
//|               was feasible w.m_r.m_t. general linear constraints.    |
//|               This field can be used by unit test to validate    |
//|               some details of initial point calculation algorithm|
//|   SparseBatch - indices of box constraints which are NOT included|
//|               into dense batch(were activated at the beginning of|
//|               the orthogonalization); SparseBatchSize elements   |
//|               are stored.                                        |
//|   SparseBatchSize - size of sparse batcj                         |
//|   PDenseBatch,                                                   |
//|   IDenseBatch,                                                   |
//|   SDenseBatch - after call to SASRebuildBasis() these matrices   |
//|               store active constraints(except for ones included  |
//|               into sparse batch), reorthogonalized with respect  |
//|               to some inner product:                             |
//|                  a) for "P" batch - one given by preconditioner  |
//|                     matrix (inverse Hessian)                     |
//|                  b) for "S" one - one given by square of the     |
//|                     scale matrix                                 |
//|                  c) for "I" one - traditional dot product        |
//|                     array[DenseBatchSize, N + 1]                 |
//| All three matrices are linearly equivalent to each other.        |
//| IMPORTANT: you have to call SASRebuildBasis() before accessing   |
//|            these arrays in order to make sure that they are up to|
//|            date.                                                 |
//| DenseBatchSize - dense batch size(PBasis / SBasis / IBasis)      |
//+------------------------------------------------------------------+
struct CSActiveSet
  {
   int               m_algostate;
   int               m_basisage;
   int               m_densebatchsize;
   int               m_n;
   int               m_nec;
   int               m_nic;
   int               m_sparsebatchsize;
   bool              m_basisisready;
   bool              m_constraintschanged;
   bool              m_feasinitpt;
   bool              m_HasBndL[];
   bool              m_HasBndU[];
   bool              m_hasxc;
   bool              m_mtnew[];
   bool              m_rctmpisequality[];
   CSNNLSSolver      m_solver;
   CRowInt           m_cstatus;
   CRowInt           m_mtas;
   CRowInt           m_rctmpconstraintidx;
   CRowInt           m_sparsebatch;
   CRowDouble        m_bndl;
   CRowDouble        m_bndu;
   CRowDouble        m_cdtmp;
   CRowDouble        m_corrtmp;
   CRowDouble        m_h;
   CRowDouble        m_mtx;
   CRowDouble        m_rctmpg;
   CRowDouble        m_rctmplambdas;
   CRowDouble        m_rctmprightpart;
   CRowDouble        m_rctmps;
   CRowDouble        m_s;
   CRowDouble        m_scntmp;
   CRowDouble        m_tmp0;
   CRowDouble        m_tmpci;
   CRowDouble        m_tmpcp;
   CRowDouble        m_tmpcs;
   CRowDouble        m_tmpfeas;
   CRowDouble        m_tmpnormestimates;
   CRowDouble        m_tmpprodp;
   CRowDouble        m_tmpprods;
   CRowDouble        m_tmpreciph;
   CRowDouble        m_unitdiagonal;
   CRowDouble        m_xc;
   CMatrixDouble     m_cleic;
   CMatrixDouble     m_idensebatch;
   CMatrixDouble     m_pdensebatch;
   CMatrixDouble     m_rctmpdense0;
   CMatrixDouble     m_rctmpdense1;
   CMatrixDouble     m_sdensebatch;
   CMatrixDouble     m_tmpbasis;
   CMatrixDouble     m_tmpm0;
   //--- constructor / destructor
                     CSActiveSet(void);
                    ~CSActiveSet(void) {}
   void              Copy(const CSActiveSet &obj);
   //--- overloading
   void              operator=(const CSActiveSet &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
CSActiveSet::CSActiveSet(void)
  {
   m_algostate=0;
   m_basisage=0;
   m_densebatchsize=0;
   m_n=0;
   m_nec=0;
   m_nic=0;
   m_sparsebatchsize=0;
   m_basisisready=false;
   m_constraintschanged=false;
   m_feasinitpt=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CSActiveSet::Copy(const CSActiveSet &obj)
  {
   m_algostate=obj.m_algostate;
   m_basisage=obj.m_basisage;
   m_densebatchsize=obj.m_densebatchsize;
   m_n=obj.m_n;
   m_nec=obj.m_nec;
   m_nic=obj.m_nic;
   m_sparsebatchsize=obj.m_sparsebatchsize;
   m_basisisready=obj.m_basisisready;
   m_constraintschanged=obj.m_constraintschanged;
   m_feasinitpt=obj.m_feasinitpt;
   ArrayCopy(m_HasBndL,obj.m_HasBndL);
   ArrayCopy(m_HasBndU,obj.m_HasBndU);
   m_hasxc=obj.m_hasxc;
   ArrayCopy(m_mtnew,obj.m_mtnew);
   ArrayCopy(m_rctmpisequality,obj.m_rctmpisequality);
   m_solver=obj.m_solver;
   m_cstatus=obj.m_cstatus;
   m_mtas=obj.m_mtas;
   m_rctmpconstraintidx=obj.m_rctmpconstraintidx;
   m_sparsebatch=obj.m_sparsebatch;
   m_bndl=obj.m_bndl;
   m_bndu=obj.m_bndu;
   m_cdtmp=obj.m_cdtmp;
   m_corrtmp=obj.m_corrtmp;
   m_h=obj.m_h;
   m_mtx=obj.m_mtx;
   m_rctmpg=obj.m_rctmpg;
   m_rctmplambdas=obj.m_rctmplambdas;
   m_rctmprightpart=obj.m_rctmprightpart;
   m_rctmps=obj.m_rctmps;
   m_s=obj.m_s;
   m_scntmp=obj.m_scntmp;
   m_tmp0=obj.m_tmp0;
   m_tmpci=obj.m_tmpci;
   m_tmpcp=obj.m_tmpcp;
   m_tmpcs=obj.m_tmpcs;
   m_tmpfeas=obj.m_tmpfeas;
   m_tmpnormestimates=obj.m_tmpnormestimates;
   m_tmpprodp=obj.m_tmpprodp;
   m_tmpprods=obj.m_tmpprods;
   m_tmpreciph=obj.m_tmpreciph;
   m_unitdiagonal=obj.m_unitdiagonal;
   m_xc=obj.m_xc;
   m_cleic=obj.m_cleic;
   m_idensebatch=obj.m_idensebatch;
   m_pdensebatch=obj.m_pdensebatch;
   m_rctmpdense0=obj.m_rctmpdense0;
   m_rctmpdense1=obj.m_rctmpdense1;
   m_sdensebatch=obj.m_sdensebatch;
   m_tmpbasis=obj.m_tmpbasis;
   m_tmpm0=obj.m_tmpm0;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSActiveSets
  {
public:
   //--- constants
   static const int m_maxbasisage ;
   static const double m_maxbasisdecay;
   static const double m_minnormseparation;

   static void       SASInit(int n,CSActiveSet &s);
   static void       SASSetScale(CSActiveSet &State,CRowDouble &s);
   static void       SASSetPrecDiag(CSActiveSet &State,CRowDouble &d);
   static void       SASSetBC(CSActiveSet &State,CRowDouble &bndl,CRowDouble &bndu);
   static void       SASSetLC(CSActiveSet &State,CMatrixDouble &c,CRowInt &ct,int k);
   static void       SASSetLCX(CSActiveSet &State,CMatrixDouble &cleic,int nec,int nic);
   static bool       SASStartOptimization(CSActiveSet &State,CRowDouble &x);
   static void       SASExploreDirection(CSActiveSet &State,CRowDouble &d,double &stpmax,int &cidx,double &vval);
   static int        SASMoveTo(CSActiveSet &State,CRowDouble &xn,bool needact,int cidx,double cval);
   static void       SASImmediateActivation(CSActiveSet &State,int cidx,double cval);
   static void       SASConstrainedDescent(CSActiveSet &State,CRowDouble &g,CRowDouble &d);
   static void       SASConstrainedDescentPrec(CSActiveSet &State,CRowDouble &g,CRowDouble &d);
   static void       SASConstrainedDirection(CSActiveSet &State,CRowDouble &d);
   static void       SASConstrainedDirectionPrec(CSActiveSet &State,CRowDouble &d);
   static void       SASCorrection(CSActiveSet &State,CRowDouble &x,double &penalty);
   static double     SASActiveLCPenalty1(CSActiveSet &State,CRowDouble &x);
   static double     SASScaledConstrainedNorm(CSActiveSet &State,CRowDouble &d);
   static void       SASStopOptimization(CSActiveSet &State);
   static void       SASReactivateConstraints(CSActiveSet &State,CRowDouble &gc);
   static void       SASReactivateConstraintsPrec(CSActiveSet &State,CRowDouble &gc);
   static void       SASRebuildBasis(CSActiveSet &State);
   static void       SASAppendToBasis(CSActiveSet &State,bool &newentries[]);

private:
   static void       ConstrainedDescent(CSActiveSet &State,CRowDouble &g,CRowDouble &h,CMatrixDouble &ha,bool normalize,CRowDouble &d);
   static void       ReactivateConstraints(CSActiveSet &State,CRowDouble &gc,CRowDouble &h);
  };
//+------------------------------------------------------------------+
//| Constants                                                        |
//+------------------------------------------------------------------+
const int CSActiveSets::m_maxbasisage=5;
const double CSActiveSets::m_maxbasisdecay=0.01;
const double CSActiveSets::m_minnormseparation=0.25;
//+------------------------------------------------------------------+
//| This subroutine is used to initialize active set. By default,    |
//| empty N-variable model with no constraints is generated.         |
//| Previously allocated buffer variables are reused as much as      |
//| possible.                                                        |
//| Two use cases for this object are described below.               |
//| CASE 1 - STEEPEST DESCENT :                                      |
//|   SASInit()                                                      |
//|   repeat:                                                        |
//|      SASReactivateConstraints()                                  |
//|      SASDescentDirection()                                       |
//|      SASExploreDirection()                                       |
//|      SASMoveTo()                                                 |
//|   until convergence                                              |
//| CASE 1 - PRECONDITIONED STEEPEST DESCENT:                        |
//|   SASInit()                                                      |
//|   repeat:                                                        |
//|      SASReactivateConstraintsPrec()                              |
//|      SASDescentDirectionPrec()                                   |
//|      SASExploreDirection()                                       |
//|      SASMoveTo()                                                 |
//|   until convergence                                              |
//+------------------------------------------------------------------+
void CSActiveSets::SASInit(int n,CSActiveSet &s)
  {
   s.m_n=n;
   s.m_algostate=0;
//--- Constraints
   s.m_constraintschanged=true;
   s.m_nec=0;
   s.m_nic=0;
   s.m_bndl=vector<double>::Full(n,AL_NEGINF);
   s.m_bndu=vector<double>::Full(n,AL_POSINF);
   ArrayResize(s.m_HasBndL,n);
   ArrayResize(s.m_HasBndU,n);
   ArrayInitialize(s.m_HasBndL,false);
   ArrayInitialize(s.m_HasBndU,false);
//--- current point, scale
   s.m_hasxc=false;
   s.m_xc=vector<double>::Zeros(n);
   s.m_s=vector<double>::Ones(n);
   s.m_h=s.m_s;
//--- Other
   s.m_unitdiagonal=s.m_s;
  }
//+------------------------------------------------------------------+
//| This function sets scaling coefficients for SAS object.          |
//| ALGLIB optimizers use scaling matrices to test stopping          |
//| conditions (step size and gradient are scaled before comparison  |
//| with tolerances). Scale of the I-Th variable is a translation    |
//| invariant measure of:                                            |
//|   a) "how large" the variable is                                 |
//|   b) how large the step should be to make significant changes in |
//|      the function                                                |
//| During orthogonalization phase, scale is used to calculate drop  |
//| tolerances (whether vector is significantly non-zero or not).  |
//| INPUT PARAMETERS:                                                |
//|   State    -  structure stores algorithm State                   |
//|   S        -  array[N], non-zero scaling coefficients S[i] may |
//|               be negative, sign doesn't matter.                  |
//+------------------------------------------------------------------+
void CSActiveSets::SASSetScale(CSActiveSet &State,CRowDouble &s)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==0,__FUNCTION__+": you may change scale only in modification mode"))
      return;
   if(!CAp::Assert(s.Size()>=State.m_n,__FUNCTION__+": Length(S)<N"))
      return;

   for(int i=0; i<State.m_n; i++)
     {
      if(!CAp::Assert(MathIsValidNumber(s[i]),__FUNCTION__+": S contains infinite or NAN elements"))
         return;
      if(!CAp::Assert(s[i]!=0.0,__FUNCTION__+": S contains zero elements"))
         return;
     }

   State.m_s=s.Abs()+0;
   State.m_s.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: diagonal of approximate      |
//| Hessian is used.                                                 |
//| INPUT PARAMETERS:                                                |
//|   State    -  structure which stores algorithm State             |
//|   D        -  diagonal of the approximate Hessian, array[0..m_n-1],|
//|               (if larger, only leading N elements are used).     |
//| NOTE 1: D[i] should be positive. Exception will be thrown        |
//|         otherwise.                                               |
//| NOTE 2: you should pass diagonal of approximate Hessian - NOT    |
//|         ITS INVERSE.                                             |
//+------------------------------------------------------------------+
void CSActiveSets::SASSetPrecDiag(CSActiveSet &State,CRowDouble &d)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==0,__FUNCTION__+": you may change preconditioner only in modification mode"))
      return;
   if(!CAp::Assert(d.Size()>=State.m_n,__FUNCTION__+": D is too short"))
      return;

   for(int i=0; i<State.m_n; i++)
     {
      if(!CAp::Assert(MathIsValidNumber(d[i]),__FUNCTION__+": D contains infinite or NAN elements"))
         return;
      if(!CAp::Assert(d[i]>0.0,__FUNCTION__+": D contains non-positive elements"))
         return;
     }

   State.m_h=d;
   State.m_h.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//| This function sets / changes boundary constraints.               |
//| INPUT PARAMETERS:                                                |
//|   State    -  structure stores algorithm State                   |
//|   BndL     -  lower bounds, array[N]. If some(all) variables are |
//|               unbounded, you may specify very small number or    |
//|               -INF.                                              |
//|   BndU     -  upper bounds, array[N]. If some(all) variables are |
//|               unbounded, you may specify very large number or    |
//|               +INF.                                              |
//| NOTE 1: it is possible to specify BndL.Set(i, BndU[i]. In this case|
//|         I-th variable will be "frozen" at X[i]=BndL[i]=BndU[i].  |
//+------------------------------------------------------------------+
void CSActiveSets::SASSetBC(CSActiveSet &State,CRowDouble &bndl,
                            CRowDouble &bndu)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==0,__FUNCTION__+": you may change constraints only in modification mode"))
      return;

   int n=State.m_n;
   if(!CAp::Assert(bndl.Size()>=n,__FUNCTION__+": Length(BndL)<N"))
      return;
   if(!CAp::Assert(bndu.Size()>=n,__FUNCTION__+": Length(BndU)<N"))
      return;

   for(int i=0; i<n; i++)
     {
      if(!CAp::Assert(MathIsValidNumber(bndl[i]) || AL_NEGINF==bndl[i],__FUNCTION__+": BndL contains NAN or +INF"))
         return;
      if(!CAp::Assert(MathIsValidNumber(bndu[i]) || AL_POSINF==bndu[i],__FUNCTION__+": BndL contains NAN or -INF"))
         return;
      State.m_bndl.Set(i,bndl[i]);
      State.m_HasBndL[i]=MathIsValidNumber(bndl[i]);
      State.m_bndu.Set(i,bndu[i]);
      State.m_HasBndU[i]=MathIsValidNumber(bndu[i]);
     }

   State.m_constraintschanged=true;
  }
//+------------------------------------------------------------------+
//| This function sets linear constraints for SAS object.            |
//| Linear constraints are inactive by default(after initial         |
//| creation).                                                       |
//| INPUT PARAMETERS:                                                |
//|   State    -  SAS structure                                      |
//|   C        -  linear constraints, array[K, N + 1]. Each row of C |
//|               represents one constraint, either equality or      |
//|               inequality (see below):                            |
//|               * first N elements correspond to coefficients,     |
//|               * last element corresponds to the right part.      |
//|               All elements of C(including right part) must be    |
//|               finite.                                            |
//|   CT       -  type of constraints, array[K]:                     |
//|               * if CT[i] > 0, then I-Th constraint is            |
//|                  C[i, *] * x >= C[i, n + 1]                      |
//|               * if CT.Set(i, 0, then I-Th constraint is            |
//|                  C[i, *] * x = C[i, n + 1]                       |
//|               * if CT[i] < 0, then I-Th constraint is            |
//|                  C[i, *] * x <= C[i, n + 1]                      |
//|   K        -  number of equality / inequality constraints, K >= 0|
//| NOTE 1: linear(non - bound) constraints are satisfied only       |
//|         approximately:                                           |
//|         * there always exists some minor violation(about Epsilon |
//|           in magnitude) due to rounding errors                   |
//|         * numerical differentiation, if used, may lead to        |
//|           function evaluations outside of the feasible area,     |
//|           because algorithm does NOT change numerical            |
//|           differentiation formula according to linear            |
//|           constraints. If you want constraints to be satisfied   |
//|           exactly, try to reformulate your problem in such manner|
//|           that all constraints will become boundary ones (this   |
//|           kind of constraints is always satisfied exactly, both  |
//|           in the final solution and in all intermediate points). |
//+------------------------------------------------------------------+
void CSActiveSets::SASSetLC(CSActiveSet &State,CMatrixDouble &c,
                            CRowInt &ct,
                            int k)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==0,__FUNCTION__+": you may change constraints only in modification mode"))
      return;

   int n=State.m_n;
//--- First, check for errors in the inputs
   if(!CAp::Assert(k>=0,__FUNCTION__+": K<0"))
      return;
   if(!CAp::Assert(c.Cols()>=n+1 || k==0,__FUNCTION__+": Cols(C)<N+1"))
      return;
   if(!CAp::Assert(c.Rows()>=k,__FUNCTION__+": Rows(C)<K"))
      return;
   if(!CAp::Assert(ct.Size()>=k,__FUNCTION__+": Length(CT)<K"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteMatrix(c,k,n+1),__FUNCTION__+": C contains infinite or NaN values!"))
      return;
//--- Handle zero K
   if(k==0)
     {
      State.m_nec=0;
      State.m_nic=0;
      State.m_constraintschanged=true;
      return;
     }
//--- Equality constraints are stored first, in the upper
//--- NEC rows of State.CLEIC matrix. Inequality constraints
//--- are stored in the next NIC rows.
//--- NOTE: we convert inequality constraints to the form
//--- A*x<=b before copying them.
   State.m_cleic.Resize(k,n+1);
   State.m_nec=0;
   State.m_nic=0;
   for(int i=0; i<k; i++)
     {
      if(ct[i]==0)
        {
         State.m_cleic.Row(State.m_nec,c[i]+0);
         State.m_nec++;
        }
     }
   for(int i=0; i<k; i++)
     {
      if(ct[i]!=0)
        {
         if(ct[i]>0)
            State.m_cleic.Row(State.m_nec+State.m_nic,c[i]*(-1.0));
         else
            State.m_cleic.Row(State.m_nec+State.m_nic,c[i]+0);
         State.m_nic++;
        }
     }
//--- Mark State as changed
   State.m_constraintschanged=true;
  }
//+------------------------------------------------------------------+
//| Another variation of SASSetLC(), which accepts linear constraints|
//| using another representation.                                    |
//| Linear constraints are inactive by default(after initial         |
//| creation).                                                       |
//| INPUT PARAMETERS:                                                |
//|   State    -  SAS structure                                      |
//|   CLEIC    -  linear constraints, array[NEC + NIC, N + 1]. Each  |
//|               row of C represents one constraint:                |
//|               * first N elements correspond to coefficients,     |
//|               * last element corresponds to the right part. First|
//|                 NEC rows store equality constraints, next NIC -  |
//|                 are inequality ones. All elements of C(including |
//|                 right part) must be finite.                      |
//|   NEC      -  number of equality constraints, NEC >= 0           |
//|   NIC      -  number of inequality constraints, NIC >= 0         |
//| NOTE 1: linear(non - bound) constraints are satisfied only       |
//|         approximately:                                           |
//|         * there always exists some minor violation(about Epsilon |
//|           in magnitude) due to rounding errors                   |
//|         * numerical differentiation, if used, may lead to        |
//|           function evaluations outside of the feasible area,     |
//|           because algorithm does NOT change numerical            |
//|           differentiation formula according to linear constraints|
//|           If you want constraints to be satisfied exactly, try   |
//|           to reformulate your problem in such manner that all    |
//|           constraints will become boundary ones (this kind of    |
//|           constraints is always satisfied exactly, both in the   |
//|           final solution and in all intermediate points).        |
//+------------------------------------------------------------------+
void CSActiveSets::SASSetLCX(CSActiveSet &State,CMatrixDouble &cleic,
                             int nec,int nic)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==0,__FUNCTION__+": you may change constraints only in modification mode"))
      return;

   int n=State.m_n;
//--- First, check for errors in the inputs
   if(!CAp::Assert(nec>=0,__FUNCTION__+": NEC<0"))
      return;
   if(!CAp::Assert(nic>=0,__FUNCTION__+": NIC<0"))
      return;
   if(!CAp::Assert(cleic.Cols()>=n+1 || nec+nic==0,__FUNCTION__+": Cols(CLEIC)<N+1"))
      return;
   if(!CAp::Assert(cleic.Rows()>=nec+nic,__FUNCTION__+": Rows(CLEIC)<NEC+NIC"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteMatrix(cleic,nec+nic,n+1),__FUNCTION__+": CLEIC contains infinite or NaN values!"))
      return;
//--- Store constraints
   State.m_cleic=cleic;
   State.m_cleic.Resize(nec+nic,n+1);
   State.m_nec=nec;
   State.m_nic=nic;
//--- Mark State as changed
   State.m_constraintschanged=true;
  }
//+------------------------------------------------------------------+
//| This subroutine turns on optimization mode:                      |
//|   1. feasibility in X is enforced(in case X=S.XC and constraints |
//|      have not changed, algorithm just uses X without any         |
//|      modifications at all)                                       |
//|   2. constraints are marked as "candidate" or "inactive"         |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   X        -  initial point(candidate), array[N]. It is expected |
//|               that X contains only finite values(we do not check |
//|               it).                                               |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  State is changed                                   |
//|   X        -  initial point can be changed to enforce feasibility|
//| RESULT:                                                          |
//|   True in case feasible point was found(mode was changed         |
//|        to "optimization")                                        |
//|   False in case no feasible point was found(mode was not changed)|
//+------------------------------------------------------------------+
bool CSActiveSets::SASStartOptimization(CSActiveSet &State,
                                        CRowDouble &x)
  {
//--- create variables
   bool   result=false;
   int    n=0;
   int    nec=0;
   int    nic=0;
   int    i=0;
   int    j=0;
   double v=0;
   double v0=0;
   double v1=0;
   double vv=0;
   double vc=0;
   double vx=0;
   int    i_=0;
//--- check
   if(!CAp::Assert(State.m_algostate==0,__FUNCTION__+": already in optimization mode"))
      return(false);

   n=State.m_n;
   nec=State.m_nec;
   nic=State.m_nic;
//--- Enforce feasibility and calculate set of "candidate"/"active" constraints.
//--- Always active equality constraints are marked as "active", all other constraints
//--- are marked as "candidate".
   State.m_cstatus.Resize(n+nec+nic);
   for(i=0; i<n; i++)
     {
      if(State.m_HasBndL[i] && State.m_HasBndU[i])
         if(State.m_bndl[i]>State.m_bndu[i])
            return(result);
     }
   State.m_xc=x;
   if(State.m_nec+State.m_nic>0)
     {
      //--- General linear constraints are present.
      //--- Try to use fast code for feasible initial point with modest
      //--- memory requirements.
      State.m_tmp0=x;
      State.m_cstatus.Fill(-1,0,n);
      State.m_feasinitpt=true;
      for(i=0; i<n; i++)
        {
         if((State.m_HasBndL[i] && State.m_HasBndU[i]) && State.m_bndl[i]==State.m_bndu[i])
           {
            State.m_tmp0.Set(i,State.m_bndl[i]);
            State.m_cstatus.Set(i,1);
            continue;
           }
         if(State.m_HasBndL[i] && State.m_tmp0[i]<=State.m_bndl[i])
           {
            State.m_cstatus.Set(i,0);
            State.m_tmp0.Set(i,State.m_bndl[i]);
           }
         if(State.m_HasBndU[i] && State.m_tmp0[i]>=State.m_bndu[i])
           {
            State.m_cstatus.Set(i,0);
            State.m_tmp0.Set(i,State.m_bndu[i]);
           }
        }
      for(i=0; i<State.m_nec+State.m_nic; i++)
        {
         v=-State.m_cleic.Get(i,n);
         v0=0;
         v1=0;
         for(j=0; j<n; j++)
           {
            vx=State.m_tmp0[j]/State.m_s[j];
            vc=State.m_cleic.Get(i,j)*State.m_s[j];
            v+= vx*vc;
            v0+=CMath::Sqr(vx);
            v1+=CMath::Sqr(vc);
           }
         vv=MathSqrt(v0)*MathSqrt(v1)*1000*CMath::m_machineepsilon;
         if(i<State.m_nec)
           {
            State.m_cstatus.Set(n+i,1);
            State.m_feasinitpt=State.m_feasinitpt && MathAbs(v)<vv;
           }
         else
           {
            State.m_feasinitpt=State.m_feasinitpt && v<vv;
            if(v<(-vv))
               State.m_cstatus.Set(n+i,-1);
            else
               State.m_cstatus.Set(n+i,0);
           }
        }
      if(State.m_feasinitpt)
        {
         State.m_xc=State.m_tmp0;
        }
      //--- Fast code failed? Use general code with ~(N+NIC)^2 memory requirements
      if(!State.m_feasinitpt)
        {
         State.m_tmp0.Resize(n);
         State.m_tmpfeas.Resize(n+State.m_nic);
         State.m_tmpm0.Resize(State.m_nec+State.m_nic,n+State.m_nic+1);
         for(i=0; i<State.m_nec+State.m_nic; i++)
           {
            for(i_=0; i_<n; i_++)
               State.m_tmpm0.Set(i,i_,State.m_cleic.Get(i,i_));
            for(j=n; j<n+State.m_nic; j++)
               State.m_tmpm0.Set(i,j,0);
            if(i>=State.m_nec)
               State.m_tmpm0.Set(i,n+i-State.m_nec,1.0);
            State.m_tmpm0.Set(i,n+State.m_nic,State.m_cleic.Get(i,n));
           }
         for(i_=0; i_<n; i_++)
            State.m_tmpfeas.Set(i_,State.m_xc[i_]);
         for(i=0; i<State.m_nic; i++)
           {
            v=CAblasF::RDotVR(n,State.m_xc,State.m_cleic,i+State.m_nec);
            State.m_tmpfeas.Set(i+n,MathMax(State.m_cleic.Get(i+State.m_nec,n)-v,0.0));
           }
         if(!COptServ::FindFeasiblePoint(State.m_tmpfeas,State.m_bndl,State.m_HasBndL,State.m_bndu,State.m_HasBndU,n,State.m_nic,State.m_tmpm0,State.m_nec+State.m_nic,1.0E-6,i,j))
            return(result);
         for(i_=0; i_<n; i_++)
            State.m_xc.Set(i_,State.m_tmpfeas[i_]);
         for(i=0; i<n; i++)
           {
            if((State.m_HasBndL[i] && State.m_HasBndU[i]) && State.m_bndl[i]==State.m_bndu[i])
              {
               State.m_cstatus.Set(i,1);
               continue;
              }
            if((State.m_HasBndL[i] && State.m_xc[i]==State.m_bndl[i]) || (State.m_HasBndU[i] && State.m_xc[i]==State.m_bndu[i]))
              {
               State.m_cstatus.Set(i,0);
               continue;
              }
            State.m_cstatus.Set(i,-1);
           }
         for(i=0; i<State.m_nec; i++)
            State.m_cstatus.Set(n+i,1);
         for(i=0; i<State.m_nic; i++)
           {
            if(State.m_tmpfeas[n+i]==0.0)
               State.m_cstatus.Set(n+State.m_nec+i,0);
            else
               State.m_cstatus.Set(n+State.m_nec+i,-1);
           }
        }
     }
   else
     {
      //--- Only box constraints are present, quick code can be used
      for(i=0; i<n; i++)
        {
         State.m_cstatus.Set(i,-1);
         if(State.m_HasBndL[i] && State.m_HasBndU[i] && State.m_bndl[i]==State.m_bndu[i])
           {
            State.m_cstatus.Set(i,1);
            State.m_xc.Set(i,State.m_bndl[i]);
            continue;
           }
         if(State.m_HasBndL[i] && State.m_xc[i]<=State.m_bndl[i])
           {
            State.m_xc.Set(i,State.m_bndl[i]);
            State.m_cstatus.Set(i,0);
            continue;
           }
         if(State.m_HasBndU[i] && State.m_xc[i]>=State.m_bndu[i])
           {
            State.m_xc.Set(i,State.m_bndu[i]);
            State.m_cstatus.Set(i,0);
            continue;
           }
        }
      State.m_feasinitpt=true;
     }
//--- Change State, allocate temporaries
   result=true;
   State.m_algostate=1;
   State.m_basisisready=false;
   State.m_hasxc=true;
   return(result);
  }
//+------------------------------------------------------------------+
//| This function explores search direction and calculates bound for |
//| step as well as information for activation of constraints.       |
//| INPUT PARAMETERS:                                                |
//|   State    -  SAS structure which stores current point and all   |
//|               other active set related information               |
//|   D        -  descent direction to explore                       |
//| OUTPUT PARAMETERS:                                               |
//|   StpMax   -  upper limit on step length imposed by yet inactive |
//|               constraints. Can be zero in case some constraints  |
//|               can be activated by zero step. Equal to some large |
//|               value in case step is unlimited.                   |
//|   CIdx     -  -1 for unlimited step, in [0, N + NEC + NIC) in    |
//|               case of limited step.                              |
//|   VVal     -  value which is assigned to X[CIdx] during          |
//|               activation. For CIdx<0 or CIdx >= N some dummy     |
//|               value is assigned to this parameter.               |
//+------------------------------------------------------------------+
void CSActiveSets::SASExploreDirection(CSActiveSet &State,CRowDouble &d,
                                       double &stpmax,int &cidx,
                                       double &vval)
  {
//--- create variables
   int    n=0;
   int    nec=0;
   int    nic=0;
   int    i=0;
   double prevmax=0;
   double vc=0;
   double vd=0;
   int    i_=0;

   stpmax=0;
   cidx=0;
   vval=0;
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return;

   n=State.m_n;
   nec=State.m_nec;
   nic=State.m_nic;
   cidx=-1;
   vval=0;
   stpmax=1.0E50;
   for(i=0; i<n; i++)
     {
      if(State.m_cstatus[i]<=0)
        {
         //--- check
         if(!CAp::Assert(!State.m_HasBndL[i] || State.m_xc[i]>=State.m_bndl[i],__FUNCTION__+": internal error - infeasible X"))
            return;
         if(!CAp::Assert(!State.m_HasBndU[i] || State.m_xc[i]<=State.m_bndu[i],__FUNCTION__+": internal error - infeasible X"))
            return;
         if(State.m_HasBndL[i] && d[i]<0.0)
           {
            prevmax=stpmax;
            stpmax=CApServ::SafeMinPosRV(State.m_xc[i]-State.m_bndl[i],-d[i],stpmax);
            if(stpmax<prevmax)
              {
               cidx=i;
               vval=State.m_bndl[i];
              }
           }
         if(State.m_HasBndU[i] && d[i]>0.0)
           {
            prevmax=stpmax;
            stpmax=CApServ::SafeMinPosRV(State.m_bndu[i]-State.m_xc[i],d[i],stpmax);
            if(stpmax<prevmax)
              {
               cidx=i;
               vval=State.m_bndu[i];
              }
           }
        }
     }
   for(i=nec; i<nec+nic; i++)
     {
      if(State.m_cstatus[n+i]<=0)
        {
         vc=0.0;
         for(i_=0; i_<n; i_++)
           {
            vc+=State.m_cleic.Get(i,i_)*State.m_xc[i_];
           }
         vc=vc-State.m_cleic.Get(i,n);
         vd=0.0;
         for(i_=0; i_<n; i_++)
            vd+=State.m_cleic.Get(i,i_)*d[i_];
         if(vd<=0.0)
            continue;
         if(vc<0.0)
           {
            //--- XC is strictly feasible with respect to I-th constraint,
            //--- we can perform non-zero step because there is non-zero distance
            //--- between XC and bound.
            prevmax=stpmax;
            stpmax=CApServ::SafeMinPosRV(-vc,vd,stpmax);
            if(stpmax<prevmax)
               cidx=n+i;
           }
         else
           {
            //--- XC is at the boundary (or slightly beyond it), and step vector
            //--- points beyond the boundary.
            //--- The only thing we can do is to perform zero step and activate
            //--- I-th constraint.
            stpmax=0;
            cidx=n+i;
           }
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine moves current point to XN, which can be:         |
//|   a) point in the direction previously explored  with            |
//|      SASExploreDirection() function (in this case NeedAct/CIdx/  |
//|      CVal are used)                                              |
//|   b) point in arbitrary direction, not necessarily previously    |
//|      checked  with SASExploreDirection() function.               |
//| Step may activate one constraint. It is assumed than XN  is      |
//| approximately feasible(small error as  large  as several  ulps   |
//| is possible). Strict feasibility  with  respect  to  bound       |
//| constraints is enforced during activation, feasibility with      |
//| respect to general linear constraints is  not enforced.          |
//| This function activates boundary constraints, such that both is  |
//| True:                                                            |
//|   1) XC[I] is not at the boundary                                |
//|   2) XN[I] is at the boundary or beyond it                       |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   XN       -  new point.                                         |
//|   NeedAct  -  True in case one constraint needs activation       |
//|   CIdx     -  index of constraint, in [0, N + NEC + NIC). Ignored|
//|               if NeedAct is false. This value is calculated by   |
//|               SASExploreDirection().                             |
//|   CVal     -  for CIdx in [0, N) this field stores value which is|
//|               assigned to XC[CIdx] during activation. CVal is    |
//|               ignored in other cases. This value is calculated by|
//|               SASExploreDirection().                             |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  current point and list of active constraints are   |
//|               changed.                                           |
//| RESULT:                                                          |
//|   > 0, in case at least one inactive non - candidate constraint  |
//|        was activated                                             |
//|  =0, in case only "candidate" constraints were activated       |
//|   < 0, in case no constraints were activated by the step         |
//| NOTE: in general case State.XC<>XN because activation of         |
//|       constraints may slightly change current point(to enforce   |
//|       feasibility).                                              |
//+------------------------------------------------------------------+
int CSActiveSets::SASMoveTo(CSActiveSet &State,CRowDouble &xn,
                            bool needact,int cidx,double cval)
  {
//--- create variables
   int  result=0;
   int  n=0;
   int  nec=0;
   int  nic=0;
   int  i=0;
   bool wasactivation;
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return(result);

   n=State.m_n;
   nec=State.m_nec;
   nic=State.m_nic;
//--- Save previous State, update current point
   State.m_mtx.Resize(n);
   State.m_mtas.Resize(n+nec+nic);
   for(i=0; i<n; i++)
     {
      State.m_mtx.Set(i,State.m_xc[i]);
      State.m_xc.Set(i,xn[i]);
     }
   for(i=0; i<n+nec+nic ; i++)
      State.m_mtas.Set(i,State.m_cstatus[i]);
//--- Activate constraints
   ArrayResize(State.m_mtnew,n+nec+nic);
   ArrayInitialize(State.m_mtnew,false);
   wasactivation=false;
   if(needact)
     {
      //--- Activation
      //--- check
      if(!CAp::Assert(cidx>=0 && cidx<n+nec+nic,__FUNCTION__+": incorrect CIdx"))
         return(result);
      if(cidx<n)
        {
         //--- CIdx in [0,N-1] means that bound constraint was activated.
         //--- We activate it explicitly to avoid situation when roundoff-error
         //--- prevents us from moving EXACTLY to x=CVal.
         State.m_xc.Set(cidx,cval);
        }
      State.m_cstatus.Set(cidx,1);
      State.m_mtnew[cidx]=true;
      wasactivation=true;
     }
   for(i=0; i<n; i++)
     {
      //--- Post-check (some constraints may be activated because of numerical errors)
      if((State.m_HasBndL[i] && State.m_xc[i]<=State.m_bndl[i]) && State.m_xc[i]!=State.m_mtx[i])
        {
         State.m_xc.Set(i,State.m_bndl[i]);
         State.m_cstatus.Set(i,1);
         State.m_mtnew[i]=true;
         wasactivation=true;
        }
      if((State.m_HasBndU[i] && State.m_xc[i]>=State.m_bndu[i]) && State.m_xc[i]!=State.m_mtx[i])
        {
         State.m_xc.Set(i,State.m_bndu[i]);
         State.m_cstatus.Set(i,1);
         State.m_mtnew[i]=true;
         wasactivation=true;
        }
     }
//--- Determine return status:
//--- * -1 in case no constraints were activated
//---* 0 in case only "candidate" constraints were activated
//---*+1 in case at least one "non-candidate" constraint was activated
   if(wasactivation)
     {
      //--- Step activated one/several constraints, but sometimes it is spurious
      //--- activation - RecalculateConstraints() tells us that constraint is
      //--- inactive (negative Largrange multiplier), but step activates it
      //--- because of numerical noise.
      //--- This block of code checks whether step activated truly new constraints
      //--- (ones which were not in the active set at the solution):
      //--- * for non-boundary constraint it is enough to check that previous value
      //---   of CStatus[i] is negative (=far from boundary), and new one is
      //---   positive (=we are at the boundary, constraint is activated).
      //--- * for boundary constraints previous criterion won't work. Each variable
      //---   has two constraints, and simply checking their status is not enough -
      //---   we have to correctly identify cases when we leave one boundary
      //---   (PrevActiveSet[i]=0) and move to another boundary (CStatus[i]>0).
      //---   Such cases can be identified if we compare previous X with new X.
      //--- In case only "candidate" constraints were activated, result variable
      //--- is set to 0. In case at least one new constraint was activated, result
      //--- is set to 1.
      result=0;
      for(i=0; i<n; i++)
        {
         if(State.m_cstatus[i]>0 && State.m_xc[i]!=State.m_mtx[i])
            result=1;
        }
      for(i=n; i<n+State.m_nec+State.m_nic; i++)
        {
         if(State.m_mtas[i]<0 && State.m_cstatus[i]>0)
            result=1;
        }
     }
   else
     {
      //--- No activation, return -1
      result=-1;
     }
//--- Update basis
   SASAppendToBasis(State,State.m_mtnew);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This subroutine performs immediate activation of one constraint: |
//|  *"immediate" means that we do not have to move to activate it |
//|   * in case boundary constraint is activated, we enforce current |
//|     point to be exactly at the boundary                          |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   CIdx     -  index of constraint, in [0, N + NEC + NIC). This   |
//|               value is calculated by SASExploreDirection().      |
//|   CVal     -  for CIdx in [0, N) this field stores value which is|
//|               assigned to XC[CIdx] during activation. CVal is    |
//|               ignored in other cases. This value is calculated   |
//|               by SASExploreDirection().                          |
//+------------------------------------------------------------------+
void CSActiveSets::SASImmediateActivation(CSActiveSet &State,int cidx,
                                          double cval)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return;

   if(cidx<State.m_n)
      State.m_xc.Set(cidx,cval);
   State.m_cstatus.Set(cidx,1);
   CApServ::BVectorSetLengthAtLeast(State.m_mtnew,State.m_n+State.m_nec+State.m_nic);
   ArrayInitialize(State.m_mtnew,false);
   State.m_mtnew[cidx]=true;
   SASAppendToBasis(State,State.m_mtnew);
  }
//+------------------------------------------------------------------+
//| This subroutine calculates descent direction subject to current  |
//| active set.                                                      |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   G        -  array[N], gradient                                 |
//|   D        -  possibly prealocated buffer; automatically resized |
//|               if needed.                                         |
//| OUTPUT PARAMETERS:                                               |
//|   D        -  descent direction projected onto current active set|
//|               Components of D which correspond to active boundary|
//|               constraints are forced to be exactly zero. In case |
//|               D is non-zero, it is normalized to have unit norm. |
//| NOTE: in case active set has N  active  constraints (or  more),  |
//|       descent direction is forced to be exactly zero.            |
//+------------------------------------------------------------------+
void CSActiveSets::SASConstrainedDescent(CSActiveSet &State,
                                         CRowDouble &g,
                                         CRowDouble &d)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return;
//--- function call
   SASRebuildBasis(State);
   ConstrainedDescent(State,g,State.m_unitdiagonal,State.m_idensebatch,true,d);
  }
//+------------------------------------------------------------------+
//| This subroutine calculates preconditioned descent direction      |
//| subject to current active set.                                   |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   G        -  array[N], gradient                                 |
//|   D        -  possibly prealocated buffer; automatically resized |
//|               if needed.                                         |
//| OUTPUT PARAMETERS:                                               |
//|   D        -  descent direction projected onto current active set|
//|               Components of D which correspond to active boundary|
//|               constraints are forced to be exactly zero. In case |
//|               D is non-zero, it is normalized to have unit norm. |
//| NOTE: in case active set has N active constraints (or more),     |
//|       descent direction is forced to be exactly zero.            |
//+------------------------------------------------------------------+
void CSActiveSets::SASConstrainedDescentPrec(CSActiveSet &State,
                                             CRowDouble &g,
                                             CRowDouble &d)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return;
//--- function call
   SASRebuildBasis(State);
   ConstrainedDescent(State,g,State.m_h,State.m_pdensebatch,true,d);
  }
//+------------------------------------------------------------------+
//| This subroutine calculates projection of direction vector to     |
//| current active set.                                              |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   D        -  array[N], direction                                |
//| OUTPUT PARAMETERS:                                               |
//|   D        -  direction projected onto current active set.       |
//|               Components of D which correspond to active boundary|
//|               constraints are forced to be exactly zero.         |
//| NOTE: in case active set has N active constraints (or more),     |
//|       descent direction is forced to be exactly zero.            |
//+------------------------------------------------------------------+
void CSActiveSets::SASConstrainedDirection(CSActiveSet &State,
                                           CRowDouble &d)
  {
   int i=0;
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return;
//--- function call
   SASRebuildBasis(State);
   ConstrainedDescent(State,d,State.m_unitdiagonal,State.m_idensebatch,false,State.m_cdtmp);
//--- copy
   d=State.m_cdtmp;
   d*=(-1.0);
   d.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//| This subroutine calculates product of direction vector and       |
//| preconditioner multiplied subject to current active set.         |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   D        -  array[N], direction                                |
//| OUTPUT PARAMETERS:                                               |
//|   D        -  preconditioned direction projected onto current    |
//|               active set. Components of D which correspond to    |
//|               active boundary constraints are forced to be       |
//|               exactly zero.                                      |
//| NOTE: in case active set has N active constraints (or  more),    |
//|       descent direction is forced to be exactly zero.            |
//+------------------------------------------------------------------+
void CSActiveSets::SASConstrainedDirectionPrec(CSActiveSet &State,
                                               CRowDouble &d)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return;
//--- function call
   SASRebuildBasis(State);
   ConstrainedDescent(State,d,State.m_h,State.m_pdensebatch,false,State.m_cdtmp);
//--- copy
   d=State.m_cdtmp;
   d*=(-1.0);
   d.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//| This subroutine performs correction of some (possibly infeasible)|
//| point with respect to a) current active set, b) all boundary     |
//| constraints, both active and inactive:                           |
//|   0) we calculate L1 penalty term for violation of active linear |
//|      constraints (one which is returned by SASActiveLCPenalty1() |
//|      function).                                                  |
//|   1) first, it performs projection(orthogonal with respect       |
//|      to scale matrix S) of X into current active set: X -> X1.   |
//|   2) next, we perform projection with respect to ALL boundary    |
//|      constraints which are violated at X1: X1 -> X2.             |
//|   3) X is replaced by X2.                                        |
//| The idea is that this function can preserve and enforce          |
//| feasibility during optimization, and additional penalty parameter|
//| can be used to prevent algo from leaving feasible set because of |
//| rounding errors.                                                 |
//| INPUT PARAMETERS:                                                |
//|   S     -  active set object                                     |
//|   X     -  array[N], candidate point                             |
//| OUTPUT PARAMETERS:                                               |
//|   X    - "improved" candidate point:                           |
//|            a) feasible with respect to all boundary constraints  |
//|            b) feasibility with respect to active set is retained |
//|               at good level.                                     |
//|   Penalty - penalty term, which can be added to function value   |
//|            if user wants to penalize violation of constraints    |
//|            (recommended).                                        |
//| NOTE: this function is not intended to find exact projection     |
//|       (i.e. best approximation) of X into feasible set. It just  |
//|       improves situation a bit.                                  |
//| Regular use of this function will help you to retain feasibility |
//| - if you already have something to start with and constrain your |
//| steps is such way that the only source of infeasibility are      |
//| roundoff errors.                                                 |
//+------------------------------------------------------------------+
void CSActiveSets::SASCorrection(CSActiveSet &State,CRowDouble &x,
                                 double &penalty)
  {
//--- create variables
   int    i=0;
   int    j=0;
   int    n=0;
   double v=0;
   int    i_=0;
   penalty=0;
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return;
//--- function call
   SASRebuildBasis(State);
   n=State.m_n;
//--- Calculate penalty term.
   penalty=SASActiveLCPenalty1(State,x);
//--- Perform projection 1.
//--- This projecton is given by:
//---     x_proj = x - S*S*As'*(As*x-b)
//--- where x is original x before projection, S is a scale matrix,
//--- As is a matrix of equality constraints (active set) which were
//--- orthogonalized with respect to inner product given by S (i.e. we
//--- have As*S*S'*As'=I), b is a right part of the orthogonalized
//--- constraints.
//--- NOTE: you can verify that x_proj is strictly feasible w.m_r.m_t.
//---       active set by multiplying it by As - you will get
//---       As*x_proj = As*x - As*x + b = b.
//---       This formula for projection can be obtained by solving
//---       following minimization problem.
//---           min ||inv(S)*(x_proj-x)||^2 s.m_t. As*x_proj=b
//--- NOTE: we apply sparse batch by examining CStatus[]; it is guaranteed
//---       to contain sparse batch, but avoids roundoff errors associated
//---       with the fact that some box constraints were moved to sparse
//---       storage
   State.m_corrtmp=x;
   State.m_corrtmp.Resize(n);
   for(i=0; i<State.m_densebatchsize; i++)
     {
      v=-State.m_sdensebatch.Get(i,n);
      for(j=0; j<n; j++)
         v+= State.m_sdensebatch.Get(i,j)*State.m_corrtmp[j];
      for(j=0; j<n; j++)
         State.m_corrtmp.Add(j,- v*State.m_sdensebatch.Get(i,j)*CMath::Sqr(State.m_s[j]));
     }
   for(i=0; i<n; i++)
     {
      if(State.m_cstatus[i]>0)
         State.m_corrtmp.Set(i,State.m_xc[i]);
     }
//--- Perform projection 2
   for(i=0; i<n; i++)
     {
      x.Set(i,State.m_corrtmp[i]);
      if(State.m_HasBndL[i] && x[i]<State.m_bndl[i])
         x.Set(i,State.m_bndl[i]);
      if(State.m_HasBndU[i] && x[i]>State.m_bndu[i])
         x.Set(i,State.m_bndu[i]);
     }
  }
//+------------------------------------------------------------------+
//| This subroutine returns L1 penalty for violation of active       |
//| general linear constraints(violation of boundary or inactive     |
//| linear constraints is not added to penalty).                     |
//| Penalty term is equal to:                                        |
//|      Penalty = SUM(Abs((C_i * x - R_i) / Alpha_i))               |
//| Here:                                                            |
//|   * summation is performed for I = 0...NEC + NIC-1,              |
//|     CStatus[N+I] > 0 (only for rows of CLEIC which are in active |
//|     set)                                                         |
//|   * C_i is I-Th row of CLEIC                                     |
//|   * R_i is corresponding right part                              |
//|   * S is a scale matrix                                          |
//|   * Alpha_i = || S * C_i || - is a scaling coefficient which     |
//|     "normalizes" I-Th summation term according to its scale.     |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   X        -  array[N], candidate point                          |
//+------------------------------------------------------------------+
double CSActiveSets::SASActiveLCPenalty1(CSActiveSet &State,
                                         CRowDouble &x)
  {
//--- create variables
   double result=0;
   int    i=0;
   int    j=0;
   int    n=0;
   int    nec=0;
   int    nic=0;
   double v=0;
   double alpha=0;
   double p=0;
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return(result);
//--- function call
   SASRebuildBasis(State);
   n=State.m_n;
   nec=State.m_nec;
   nic=State.m_nic;
//--- Calculate penalty term.
   result=0;
   for(i=0; i<nec+nic; i++)
     {
      if(State.m_cstatus[n+i]>0)
        {
         alpha=0;
         p=-State.m_cleic.Get(i,n);
         for(j=0; j<n; j++)
           {
            v=State.m_cleic.Get(i,j);
            p=p+v*x[j];
            alpha+=CMath::Sqr(v*State.m_s[j]);
           }
         alpha=MathSqrt(alpha);
         if(alpha!=0.0)
            result+=MathAbs(p/alpha);
        }
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This subroutine calculates scaled norm of vector after projection|
//| onto subspace of active constraints. Most often this function is |
//| used to test stopping conditions.                                |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//|   D        -  vector whose norm is calculated                    |
//| RESULT:                                                          |
//|   Vector norm (after projection and scaling)                     |
//| NOTE: projection is performed first, scaling is performed after  |
//|       projection                                                 |
//| NOTE: if we have N active constraints, zero value(exact zero)    |
//|       is returned                                                |
//+------------------------------------------------------------------+
double CSActiveSets::SASScaledConstrainedNorm(CSActiveSet &State,
                                              CRowDouble &d)
  {
//--- create variables
   double result=0;
   int    i=0;
   int    n=0;
   double v=0;
   int    i_=0;
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": is not in optimization mode"))
      return(result);

   n=State.m_n;
//--- Prepare basis (if needed)
   SASRebuildBasis(State);
//--- Calculate descent direction
   if(State.m_sparsebatchsize+State.m_densebatchsize>=n)
     {
      //--- Quick exit if number of active constraints is N or larger
      result=0.0;
      return(result);
     }
   State.m_scntmp=d;
   State.m_scntmp.Resize(n);
   for(i=0; i<=State.m_densebatchsize-1; i++)
     {
      v=0.0;
      for(i_=0; i_<n; i_++)
         v+=State.m_idensebatch.Get(i,i_)*State.m_scntmp[i_];
      for(i_=0; i_<n; i_++)
         State.m_scntmp.Add(i_,- v*State.m_idensebatch.Get(i,i_));
     }
   for(i=0; i<n; i++)
     {
      if(State.m_cstatus[i]>0)
         State.m_scntmp.Set(i,0);
     }
   v=0.0;
   for(i=0; i<n; i++)
      v+= CMath::Sqr(State.m_s[i]*State.m_scntmp[i]);
   result=MathSqrt(v);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This subroutine turns off optimization mode.                     |
//| INPUT PARAMETERS:                                                |
//|   S        -  active set object                                  |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  State is changed                                   |
//| NOTE: this function can be called many times for optimizer which |
//|       was already stopped.                                       |
//+------------------------------------------------------------------+
void CSActiveSets::SASStopOptimization(CSActiveSet &State)
  {
   State.m_algostate=0;
  }
//+------------------------------------------------------------------+
//| This function recalculates constraints - activates and           |
//| deactivates them according to gradient value at current point.   |
//| Algorithm assumes that we want to make steepest descent step from|
//| current point; constraints are activated and deactivated in such |
//| way that we won't violate any constraint by steepest descent step|
//| After call to this function active set is ready to  try steepest |
//| descent step (SASDescentDirection-SASExploreDirection-SASMoveTo).|
//| Only already "active" and "candidate" elements of ActiveSet are  |
//| examined; constraints which are not active are not examined.     |
//| INPUT PARAMETERS:                                                |
//|   State    -  active set object                                  |
//|   GC       -  array[N], gradient at XC                           |
//| OUTPUT PARAMETERS:                                               |
//|   State    -  active set object, with new set of constraint      |
//+------------------------------------------------------------------+
void CSActiveSets::SASReactivateConstraints(CSActiveSet &State,
                                            CRowDouble &gc)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": must be in optimization mode"))
      return;
//--- function call
   ReactivateConstraints(State,gc,State.m_unitdiagonal);
  }
//+------------------------------------------------------------------+
//| This function recalculates constraints - activates and           |
//| deactivates them according to gradient value at current point.   |
//| Algorithm assumes that we want to make Quasi - Newton step from  |
//| current point with diagonal Quasi - Newton matrix H. Constraints |
//| are activated and deactivated in such way that we won't violate  |
//| any constraint by step.                                          |
//| After call to this function active set is ready to try           |
//| preconditioned steepest descent step (SASDescentDirection -      |
//| SASExploreDirection - SASMoveTo).                                |
//| Only already "active" and "candidate" elements of ActiveSet are  |
//| examined; constraints which are not active are not examined.     |
//| INPUT PARAMETERS:                                                |
//|   State    -  active set object                                  |
//|   GC       -  array[N], gradient at XC                           |
//| OUTPUT PARAMETERS:                                               |
//|   State    -  active set object, with new set of constraint      |
//+------------------------------------------------------------------+
void CSActiveSets::SASReactivateConstraintsPrec(CSActiveSet &State,
                                                CRowDouble &gc)
  {
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": must be in optimization mode"))
      return;
//--- function call
   ReactivateConstraints(State,gc,State.m_h);
  }
//+------------------------------------------------------------------+
//| This function builds three orthonormal basises for current active|
//| set:                                                             |
//|   * P - orthogonal one, which is orthogonalized with inner       |
//|         product (x, y) = x'*P*y, where P=inv(H) is current       |
//|         preconditioner                                           |
//|   * S - orthogonal one, which is orthogonalized with inner       |
//|         product (x, y) = x'*S'*S * y, where S is diagonal        |
//|         scaling matrix                                           |
//|   * I - orthogonal one, which is orthogonalized with standard    |
//|         dot product                                              |
//| NOTE: all sets of orthogonal vectors are guaranteed  to  have    |
//|       same size. P - orthogonal basis is built first,            |
//|       I / S - orthogonal basises are forced to have same number  |
//|       of vectors as P - orthogonal one(padded  by  zero vectors  |
//|       if needed).                                                |
//| NOTE: this function tracks changes in active set; first call     |
//|       will result in reorthogonalization                         |
//| INPUT PARAMETERS:                                                |
//|   State    -  active set object                                  |
//|   H        -  diagonal preconditioner, H[i] > 0                  |
//| OUTPUT PARAMETERS:                                               |
//|   State    -  active set object with new basis                   |
//+------------------------------------------------------------------+
void CSActiveSets::SASRebuildBasis(CSActiveSet &State)
  {
//--- create variables
   int    n=0;
   int    nec=0;
   int    nic=0;
   int    i=0;
   int    j=0;
   bool   hasactivelin=false;
   int    candidatescnt=0;
   double v=0;
   double vv=0;
   double vmax=0;
   int    kmax=0;
   int    i_=0;

   if(State.m_basisisready)
      return;

   n=State.m_n;
   nec=State.m_nec;
   nic=State.m_nic;
   State.m_tmp0.Resize(n);
   State.m_tmpprodp.Resize(n);
   State.m_tmpprods.Resize(n);
   State.m_tmpcp.Resize(n+1);
   State.m_tmpcs.Resize(n+1);
   State.m_tmpci.Resize(n+1);
   State.m_tmpbasis.Resize(nec+nic,n+1);
   State.m_pdensebatch.Resize(nec+nic,n+1);
   State.m_idensebatch.Resize(nec+nic,n+1);
   State.m_sdensebatch.Resize(nec+nic,n+1);
   State.m_sparsebatch.Resize(n);
   State.m_sparsebatchsize=0;
   State.m_densebatchsize=0;
   State.m_basisage=0;
   State.m_basisisready=true;
//--- Determine number of active boundary and non-boundary
//--- constraints, move them to TmpBasis. Quick exit if no
//--- non-boundary constraints were detected.
   hasactivelin=false;
   for(i=0; i<nec+nic; i++)
     {
      if(State.m_cstatus[n+i]>0)
         hasactivelin=true;
     }
   for(j=0; j<n; j++)
     {
      if(State.m_cstatus[j]>0)
        {
         State.m_sparsebatch.Set(State.m_sparsebatchsize,j);
         State.m_sparsebatchsize++;
        }
     }
   if(!hasactivelin)
      return;
//--- Prepare precomputed values
   State.m_tmpreciph=State.m_h.Pow(-1.0)+0;
   State.m_tmpreciph.Resize(n);
//--- Prepare initial candidate set:
//--- * select active constraints
//--- * normalize (inner product is given by preconditioner)
//--- * orthogonalize with respect to active box constraints
//--- * copy normalized/orthogonalized candidates to PBasis/SBasis/IBasis
   candidatescnt=0;
   for(i=0; i<nec+nic; i++)
     {
      if(State.m_cstatus[n+i]>0)
        {
         State.m_tmpbasis.Row(candidatescnt,State.m_cleic[i]+0);
         candidatescnt++;
        }
     }
   for(i=0; i<candidatescnt; i++)
     {
      v=0.0;
      for(j=0; j<n; j++)
         v+=CMath::Sqr(State.m_tmpbasis.Get(i,j))*State.m_tmpreciph[j];
      if(v>0.0)
        {
         v=1/MathSqrt(v);
         State.m_tmpbasis.Row(i,State.m_tmpbasis[i]*v);
        }
     }
   for(j=0; j<n; j++)
     {
      if(State.m_cstatus[j]>0)
        {
         for(i=0; i<candidatescnt; i++)
           {
            State.m_tmpbasis.Add(i,n,-State.m_tmpbasis.Get(i,j)*State.m_xc[j]);
            State.m_tmpbasis.Set(i,j,0.0);
           }
        }
     }
   for(i=0; i<candidatescnt; i++)
      State.m_pdensebatch.Row(i,State.m_tmpbasis[i]+0);
   State.m_sdensebatch=State.m_pdensebatch;
   State.m_idensebatch=State.m_pdensebatch;
//--- Perform orthogonalization of general linear constraints with respect
//--- to each other (constraints in P/S/IBasis are already normalized w.m_r.m_t.
//--- box constraints). During this process we select strictly active constraints
//--- from the candidate set, and drop ones which were detected as redundant
//--- during orthogonalization.
//--- Orthogonalization is performed with the help of Gram-Schmidt process.
//--- Due to accumulation of round-off errors it is beneficial to perform
//--- pivoting, i.e. to select candidate vector with largest norm at each
//--- step.
//--- First (basic) version of the algorithm is:
//---     0. split all constraints into two sets: basis ones (initially empty)
//---        and candidate ones (all constraints)
//---     1. fill PBasis with H-normalized candidate constraints, fill
//---        corresponding entries of S/IBasis with corresponding
//---        (non-normalized) constraints
//---     2. select row of PBasis with largest norm, move it (and its S/IBasis
//---        counterparts) to the beginning of the candidate set, H-normalize
//---        this row (rows of S/IBasis are normalized using corresponding norms).
//---        Stop if largest row is nearly (or exactly) zero.
//---     3. orthogonalize remaining rows of P/S/IBasis with respect to
//---        one chosen at step (2). It can be done efficiently using
//---        combination of DGEMV/DGER BLAS calls.
//---     4. increase basis size by one, decrease candidate set size by one,
//---        goto (2)
//--- However, naive implementation of the algorithm above spends significant
//--- amount of time in step (2) - selection of row with largest H-norm. Step
//--- (3) can be efficiently implemented with optimized BLAS, but we have no
//--- optimized BLAS kernels for step(2). And because step (3) changes row norms,
//--- step (2) have to be re-calculated every time, which is quite slow.
//--- We can save significant amount of calculations by noticing that:
//--- * step (3) DECREASES row norms, but never increases it
//--- * we can maintain upper bounds for row H-norms is a separate array,
//---   use them for initial evaluation of best candidates, and update them
//---   after we find some promising row (all bounds are invalidated after
//---   step 3, but their old values still carry some information)
//--- * it is beneficial re-evaluate bounds only for rows which are
//---   significantly (at least few percents) larger than best one found so far
//--- * because rows are initially normalized, initial values for upper bounds
//---   can be set to 1.0
   if(!CAp::Assert(State.m_densebatchsize==0,__FUNCTION__+": integrity check failed"))
      return;
   if(!CAp::Assert(m_minnormseparation>0.0,__FUNCTION__+": integrity check failed"))
      return;
   State.m_tmpnormestimates=vector<double>::Ones(candidatescnt);
   while(State.m_sparsebatchsize+State.m_densebatchsize<n)
     {
      //--- No candidates? We are done!
      if(candidatescnt==0)
         break;
      //--- Find largest vector
      vmax=0;
      kmax=-1;
      for(i=State.m_densebatchsize; i<State.m_densebatchsize+candidatescnt; i++)
        {
         //--- Use upper bound for row norm for initial evaluation.
         //--- Skip rows whose upper bound is less than best candidate
         //--- found so far.
         //--- NOTE: in fact, we may skip rows whose upper bound is
         //---       marginally higher than that of best candidate.
         //---       No need to perform costly re-evaluation in order
         //---       to get just few percents of improvement.
         if(State.m_tmpnormestimates[i]<(vmax*(1+m_minnormseparation)))
            continue;
         //--- OK, upper bound is large enough... lets perform full
         //--- re-evaluation and update of the estimate.
         v=0.0;
         for(j=0; j<n; j++)
           {
            vv=State.m_pdensebatch.Get(i,j);
            v+= vv*vv*State.m_tmpreciph[j];
           }
         v=MathSqrt(v);
         State.m_tmpnormestimates.Set(i,v);
         //--- Now compare with best candidate so far
         if(v>vmax)
           {
            vmax=v;
            kmax=i;
           }
        }
      if(vmax<(1.0E4*CMath::m_machineepsilon) || kmax<0)
        {
         //--- All candidates are either zero or too small (after orthogonalization)
         candidatescnt=0;
         break;
        }
      //--- Candidate is selected for inclusion into basis set.
      //--- Move candidate row to the beginning of candidate array (which is
      //--- right past the end of the approved basis). Normalize (for P-basis
      //--- we perform preconditioner-based normalization, for S-basis - scale
      //--- based, for I-basis - identity based).
      State.m_pdensebatch.SwapRows(State.m_densebatchsize,kmax);
      State.m_sdensebatch.SwapRows(State.m_densebatchsize,kmax);
      State.m_idensebatch.SwapRows(State.m_densebatchsize,kmax);
      State.m_tmpnormestimates.Swap(State.m_densebatchsize,kmax);
      v=1/vmax;
      State.m_pdensebatch.Row(State.m_densebatchsize,State.m_pdensebatch[State.m_densebatchsize]*v);
      v=0;
      for(j=0; j<n; j++)
        {
         vv=State.m_sdensebatch.Get(State.m_densebatchsize,j)*State.m_s[j];
         v+= vv*vv;
        }
      //--- check
      if(!CAp::Assert(v>0.0,__FUNCTION__+": integrity check failed,SNorm=0"))
         return;
      v=1/MathSqrt(v);
      State.m_sdensebatch.Row(State.m_densebatchsize,State.m_sdensebatch[State.m_densebatchsize]*v);
      v=0;
      for(j=0; j<n; j++)
        {
         vv=State.m_idensebatch.Get(State.m_densebatchsize,j);
         v+= vv*vv;
        }
      //--- check
      if(!CAp::Assert(v>0.0,__FUNCTION__+": integrity check failed,INorm=0"))
         return;
      v=1/MathSqrt(v);
      State.m_idensebatch.Row(State.m_densebatchsize,State.m_idensebatch[State.m_densebatchsize]*v);
      //--- Reorthogonalize other candidates with respect to candidate #0:
      //--- * calculate projections en masse with GEMV()
      //--- * subtract projections with GER()
      State.m_tmp0.Resize(candidatescnt-1);
      for(j=0; j<n; j++)
        {
         State.m_tmpprodp.Set(j,State.m_pdensebatch.Get(State.m_densebatchsize,j)/State.m_h[j]);
         State.m_tmpprods.Set(j,State.m_sdensebatch.Get(State.m_densebatchsize,j)*CMath::Sqr(State.m_s[j]));
        }
      State.m_tmpcp=State.m_pdensebatch[State.m_densebatchsize]+0;
      State.m_tmpcs=State.m_sdensebatch[State.m_densebatchsize]+0;
      State.m_tmpci=State.m_idensebatch[State.m_densebatchsize]+0;
      CAblas::RMatrixGemVect(candidatescnt-1,n,1.0,State.m_pdensebatch,State.m_densebatchsize+1,0,0,State.m_tmpprodp,0,0.0,State.m_tmp0,0);
      CAblas::RMatrixGer(candidatescnt-1,n+1,State.m_pdensebatch,State.m_densebatchsize+1,0,-1.0,State.m_tmp0,0,State.m_tmpcp,0);
      CAblas::RMatrixGemVect(candidatescnt-1,n,1.0,State.m_sdensebatch,State.m_densebatchsize+1,0,0,State.m_tmpprods,0,0.0,State.m_tmp0,0);
      CAblas::RMatrixGer(candidatescnt-1,n+1,State.m_sdensebatch,State.m_densebatchsize+1,0,-1.0,State.m_tmp0,0,State.m_tmpcs,0);
      CAblas::RMatrixGemVect(candidatescnt-1,n,1.0,State.m_idensebatch,State.m_densebatchsize+1,0,0,State.m_tmpci,0,0.0,State.m_tmp0,0);
      CAblas::RMatrixGer(candidatescnt-1,n+1,State.m_idensebatch,State.m_densebatchsize+1,0,-1.0,State.m_tmp0,0,State.m_tmpci,0);
      //--- Increase basis, decrease candidates count
      State.m_densebatchsize++;
      candidatescnt--;
     }
  }
//+------------------------------------------------------------------+
//| This function appends new constraints(if possible; sometimes it  |
//| isn't!) to three orthonormal basises for current active set:     |
//|   * P - orthogonal one, which is orthogonalized with inner       |
//|         product (x, y) = x'*P*y, where P=inv(H) is current       |
//|         preconditioner                                           |
//|   * S - orthogonal one, which is orthogonalized with inner       |
//|         product (x, y) = x'*S'*S*y, where S is diagonal scaling  |
//|         matrix                                                   |
//|   * I - orthogonal one, which is orthogonalized with standard    |
//|         dot product                                              |
//| NOTE: all sets of orthogonal vectors are guaranteed to have same |
//|       size. P - orthogonal basis is built first, I/S - orthogonal|
//|       basises are forced to have same number of vectors as       |
//|       P - orthogonal one(padded by zero vectors if needed).      |
//| NOTE: this function may fail to update basis without full        |
//|       recalculation; in such case it will set BasisIsReady to    |
//|       False and silently return; if it succeeds, it will increase|
//|       BasisSize.                                                 |
//| INPUT PARAMETERS:                                                |
//|   State       -  active set object                               |
//|   NewEntries  -  array[N + NEC + NIC], indexes of constraints    |
//|                  being added are marked as True; it is           |
//|                  responsibility of the caller to specify only    |
//|                  those constraints which were previously         |
//|                  inactive; when some constraint is already in the|
//|                  active set, algorithm behavior is undefined.    |
//| OUTPUT PARAMETERS:                                               |
//|   State       -  active set object with new basis                |
//+------------------------------------------------------------------+
void CSActiveSets::SASAppendToBasis(CSActiveSet &State,
                                    bool &newentries[])
  {
//--- create variables
   int    n=0;
   int    nec=0;
   int    nic=0;
   int    i=0;
   int    j=0;
   int    t=0;
   int    nact=0;
   double v=0;
   double vp=0;
   double vs=0;
   double vi=0;
   double initnormp=0;
   double projnormp=0;
   double projnorms=0;
   double projnormi=0;
   int    i_=0;
//--- check
   if(!State.m_basisisready)
      return;

   n=State.m_n;
   nec=State.m_nec;
   nic=State.m_nic;
//--- Count number of constraints to activate;
//--- perform integrity check.
   nact=0;
   for(i=0; i<n; i++)
      if(newentries[i])
         nact++;
   for(i=n; i<n+nec; i++)
     {
      //--- check
      if(!CAp::Assert(!newentries[i],__FUNCTION__+": integrity check failed (appendtobasis.0)"))
         return;
     }
   for(i=n+nec; i<n+nec+nic; i++)
     {
      if(newentries[i])
         nact++;
     }
   if(nact+State.m_basisage>m_maxbasisage)
     {
      State.m_basisisready=false;
      return;
     }
//--- Resize basis matrices if needed
   State.m_pdensebatch.Resize(State.m_densebatchsize+nact,n+1);
   State.m_sdensebatch.Resize(State.m_densebatchsize+nact,n+1);
   State.m_idensebatch.Resize(State.m_densebatchsize+nact,n+1);
//--- Try adding recommended entries to basis.
//--- If reorthogonalization removes too much of candidate constraint,
//--- we will invalidate basis and try to rebuild it from scratch.
   State.m_tmp0.Resize(n+1);
   State.m_tmpcp.Resize(n+1);
   State.m_tmpcs.Resize(n+1);
   State.m_tmpci.Resize(n+1);
   State.m_tmpprodp.Resize(n);
   State.m_tmpprods.Resize(n);
   for(t=0; t<n+nec+nic; t++)
     {
      if(newentries[t])
        {
         //--- Basis is full? Quick skip!
         if(State.m_sparsebatchsize+State.m_densebatchsize>=n)
           {
            //--- check
            if(!CAp::Assert(State.m_sparsebatchsize+State.m_densebatchsize==n,__FUNCTION__+": integrity check failed (SASAppendToBasis)"))
               return;
            break;
           }
         //--- Copy constraint to temporary storage.
         if(t<n)
           {
            //--- Copy box constraint
            State.m_tmp0=vector<double>::Zeros(n+1);
            State.m_tmp0.Set(t,1.0);
            State.m_tmp0.Set(n,State.m_xc[t]);
           }
         else
           {
            //--- Copy general linear constraint
            State.m_tmp0=State.m_cleic[t-n]+0;
           }
         //--- Calculate initial norm (preconditioner is used for norm calculation).
         initnormp=0.0;
         for(j=0; j<n; j++)
           {
            v=State.m_tmp0[j];
            initnormp=initnormp+v*v/State.m_h[j];
           }
         initnormp=MathSqrt(initnormp);
         if(initnormp==0.0)
           {
            //--- Well, it is not expected. Let's just rebuild basis
            //--- from scratch and forget about this strange situation...
            State.m_basisisready=false;
            return;
           }
         //--- Orthogonalize Tmp0 w.m_r.m_t. sparse batch (box constraints stored in sparse storage).
         //--- Copy to TmpCP/TmpCS/TmpCI (P for preconditioner-based inner product
         //--- used for orthogonalization, S for scale-based orthogonalization,
         //--- I for "traditional" inner product used for Gram-Schmidt orthogonalization).
         for(i=0; i<State.m_sparsebatchsize; i++)
           {
            j=State.m_sparsebatch[i];
            State.m_tmp0.Add(n,- State.m_tmp0[j]*State.m_xc[j]);
            State.m_tmp0.Set(j,0.0);
           }
         State.m_tmpcp=State.m_tmp0;
         State.m_tmpcs=State.m_tmp0;
         State.m_tmpci=State.m_tmp0;
         //--- Orthogonalize TmpCP/S/I with respect to active linear constraints from dense batch.
         //--- Corresponding norm (preconditioner, scale, identity) is used in each case.
         State.m_tmpprodp=State.m_h.Pow(-1.0)+0;
         State.m_tmpprods=State.m_s.Pow(2)+0;
         for(i=0; i<State.m_densebatchsize; i++)
           {
            vp=0;
            vs=0;
            vi=0;
            for(j=0; j<n; j++)
              {
               vp+=State.m_pdensebatch.Get(i,j)*State.m_tmpcp[j]*State.m_tmpprodp[j];
               vs+=State.m_sdensebatch.Get(i,j)*State.m_tmpcs[j]*State.m_tmpprods[j];
               vi+=State.m_idensebatch.Get(i,j)*State.m_tmpci[j];
              }
            State.m_tmpcp-=State.m_pdensebatch[i]*vp;
            State.m_tmpcs-=State.m_sdensebatch[i]*vs;
            State.m_tmpci-=State.m_idensebatch[i]*vi;
           }
         projnormp=0.0;
         projnorms=0.0;
         projnormi=0.0;
         for(j=0; j<n; j++)
           {
            projnormp+=CMath::Sqr(State.m_tmpcp[j])/State.m_h[j];
            projnorms+=CMath::Sqr(State.m_tmpcs[j])*CMath::Sqr(State.m_s[j]);
            projnormi+=CMath::Sqr(State.m_tmpci[j]);
           }
         projnormp=MathSqrt(projnormp);
         projnorms=MathSqrt(projnorms);
         projnormi=MathSqrt(projnormi);
         if(projnormp<=(m_maxbasisdecay*initnormp))
           {
            State.m_basisisready=false;
            return;
            //--- Nearly zero row, skip
           }
         //--- check
         if(!CAp::Assert(projnormp>0.0,__FUNCTION__+": integrity check failed,ProjNormP=0"))
            return;
         if(!CAp::Assert(projnorms>0.0,__FUNCTION__+": integrity check failed,ProjNormS=0"))
            return;
         if(!CAp::Assert(projnormi>0.0,__FUNCTION__+": integrity check failed,ProjNormI=0"))
            return;
         v=1/projnormp;
         State.m_pdensebatch.Row(State.m_densebatchsize,State.m_tmpcp*v+0);
         v=1/projnorms;
         State.m_sdensebatch.Row(State.m_densebatchsize,State.m_tmpcs*v+0);
         v=1/projnormi;
         State.m_idensebatch.Row(State.m_densebatchsize,State.m_tmpci*v+0);
         //--- Increase set size
         State.m_densebatchsize++;
         State.m_basisage++;
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine calculates preconditioned descent direction      |
//| subject to current active set.                                   |
//| INPUT PARAMETERS:                                                |
//|   State    -  active set object                                  |
//|   G        -  array[N], gradient                                 |
//|   H        -  array[N], Hessian matrix                           |
//|   HA       -  active constraints orthogonalized in such way that |
//|               HA*inv(H)*HA'= I.                                  |
//|   Normalize-  whether we need normalized descent or not          |
//|   D        -  possibly preallocated buffer; automatically resized|
//| OUTPUT PARAMETERS:                                               |
//|   D        -  descent direction projected onto current active set|
//|               Components of D which correspond to active boundary|
//|               constraints are forced to be exactly zero. In case |
//|               D is non-zero and Normalize is True, it is         |
//|               normalized to have unit norm.                      |
//| NOTE: if we have N active constraints, D is explicitly set to    |
//|       zero.                                                      |
//+------------------------------------------------------------------+
void CSActiveSets::ConstrainedDescent(CSActiveSet &State,
                                      CRowDouble &g,
                                      CRowDouble &h,
                                      CMatrixDouble &ha,
                                      bool normalize,
                                      CRowDouble &d)
  {
//--- create variables
   int    i=0;
   int    n=0;
   double v=0;
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": internal error in ConstrainedDescent() - not in optimization mode"))
      return;
   if(!CAp::Assert(State.m_basisisready,__FUNCTION__+": internal error in ConstrainedDescent() - no basis"))
      return;

   n=State.m_n;
//--- Calculate preconditioned constrained descent direction:
//---     d := -inv(H)*( g - HA'*(HA*inv(H)*g) )
//--- Formula above always gives direction which is orthogonal to rows of HA.
//--- You can verify it by multiplication of both sides by HA[i] (I-th row),
//--- taking into account that HA*inv(H)*HA'= I (by definition of HA - it is
//--- orthogonal basis with inner product given by inv(H)).
   d=g;
   d.Resize(n);
   for(i=0; i<State.m_densebatchsize; i++)
     {
      vector<double> row=ha[i];
      row.Resize(n);
      v=d.Dot(row/h.ToVector());
      d-=row*v;
     }
   for(i=0; i<n; i++)
     {
      if(State.m_cstatus[i]>0)
         d.Set(i,0);
     }
   d/=h.ToVector()*(-1);
   v=MathSqrt(CAblasF::RDotV2(n,d));
   if(State.m_sparsebatchsize+State.m_densebatchsize>=n)
     {
      v=0;
      d=vector<double>::Zeros(n);
     }
   if(normalize && v>0.0)
      d/=v;
  }
//+------------------------------------------------------------------+
//| This function recalculates constraints - activates  and          |
//| deactivates them according to gradient value at current point.   |
//| Algorithm  assumes  that  we  want  to make Quasi - Newton step  |
//| from current point with diagonal Quasi - Newton matrix H.        |
//| Constraints are activated and deactivated in such way that we    |
//| won't violate any constraint by step.                            |
//| Only already "active" and "candidate" elements of ActiveSet are  |
//| examined; constraints which are not active are not examined.     |
//| INPUT PARAMETERS:                                                |
//|   State    -  active set object                                  |
//|   GC       -  array[N], gradient at XC                           |
//|   H        -  array[N], Hessian matrix                           |
//| OUTPUT PARAMETERS:                                               |
//|   State    -  active set object, with new set of constraint      |
//+------------------------------------------------------------------+
void CSActiveSets::ReactivateConstraints(CSActiveSet &State,
                                         CRowDouble &gc,
                                         CRowDouble &h)
  {
//--- return result
   int    n=0;
   int    nec=0;
   int    nic=0;
   int    i=0;
   int    j=0;
   int    idx0=0;
   int    idx1=0;
   double v=0;
   int    nactivebnd=0;
   int    nactivelin=0;
   int    nactiveconstraints=0;
   double rowscale=0;
   int    i_=0;
//--- check
   if(!CAp::Assert(State.m_algostate==1,__FUNCTION__+": must be in optimization mode"))
      return;
//--- Prepare
   n=State.m_n;
   nec=State.m_nec;
   nic=State.m_nic;
   State.m_basisisready=false;
//--- Handle important special case - no linear constraints,
//--- only boundary constraints are present
   if(nec+nic==0)
     {
      for(i=0; i<n; i++)
        {
         if(State.m_HasBndL[i] && State.m_HasBndU[i] && State.m_bndl[i]==State.m_bndu[i])
           {
            State.m_cstatus.Set(i,1);
            continue;
           }
         if(State.m_HasBndL[i] && State.m_xc[i]==State.m_bndl[i] && gc[i]>=0.0)
           {
            State.m_cstatus.Set(i,1);
            continue;
           }
         if(State.m_HasBndU[i] && State.m_xc[i]==State.m_bndu[i] && gc[i]<=0.0)
           {
            State.m_cstatus.Set(i,1);
            continue;
           }
         State.m_cstatus.Set(i,-1);
        }
      return;
     }
//--- General case.
//--- Calculate descent direction
   State.m_rctmpg=gc*(-1.0)+0;
//--- Allocate temporaries.
   State.m_rctmpg.Resize(n);
   State.m_rctmprightpart.Resize(n);
   State.m_rctmps.Resize(n);
   State.m_rctmpdense0.Resize(n,nec+nic);
   State.m_rctmpdense1.Resize(n,nec+nic);
   State.m_rctmpconstraintidx.Resize(n+nec+nic);
   CApServ::BVectorSetLengthAtLeast(State.m_rctmpisequality,n+nec+nic);
//--- Determine candidates to the active set.
//--- After this block constraints become either "inactive" (CStatus[i]<0)
//--- or "candidates" (CStatus[i]=0). Previously active constraints always
//--- become "candidates".
   State.m_cstatus.Fill(-1,0,n);
   for(i=n; i<=n+nec+nic-1; i++)
     {
      if(State.m_cstatus[i]>0)
         State.m_cstatus.Set(i,0);
      else
         State.m_cstatus.Set(i,-1);
     }
   nactiveconstraints=0;
   nactivebnd=0;
   nactivelin=0;
   for(i=0; i<n; i++)
     {
      //--- Activate boundary constraints:
      //--- * copy constraint index to RCTmpConstraintIdx
      //---*set corresponding element of CStatus[] to "candidate"
      //--- * fill RCTmpS by either +1 (lower bound) or -1 (upper bound)
      //--- * set RCTmpIsEquality to False (BndL<BndU) or True (BndL=BndU)
      //--- * increase counters
      if(State.m_HasBndL[i] && State.m_HasBndU[i] && State.m_bndl[i]==State.m_bndu[i])
        {
         //--- Equality constraint is activated
         State.m_rctmpconstraintidx.Set(nactiveconstraints,i);
         State.m_cstatus.Set(i,0);
         State.m_rctmps.Set(i,1.0);
         State.m_rctmpisequality[nactiveconstraints]=true;
         nactiveconstraints++;
         nactivebnd++;
         continue;
        }
      if(State.m_HasBndL[i] && State.m_xc[i]==State.m_bndl[i])
        {
         //--- Lower bound is activated
         State.m_rctmpconstraintidx.Set(nactiveconstraints,i);
         State.m_cstatus.Set(i,0);
         State.m_rctmps.Set(i,-1.0);
         State.m_rctmpisequality[nactiveconstraints]=false;
         nactiveconstraints++;
         nactivebnd++;
         continue;
        }
      if(State.m_HasBndU[i] && State.m_xc[i]==State.m_bndu[i])
        {
         //--- Upper bound is activated
         State.m_cstatus.Set(i,0);
         State.m_rctmpconstraintidx.Set(nactiveconstraints,i);
         State.m_rctmps.Set(i,1.0);
         State.m_rctmpisequality[nactiveconstraints]=false;
         nactiveconstraints++;
         nactivebnd++;
         continue;
        }
     }
   for(i=0; i<nec+nic; i++)
     {
      if(i>=nec && State.m_cstatus[n+i]<0)
        {
         //--- Inequality constraints are skipped if both (a) constraint was
         //--- not active, and (b) we are too far away from the boundary.
         rowscale=0.0;
         v=-State.m_cleic.Get(i,n);
         for(j=0; j<n; j++)
           {
            v+= State.m_cleic.Get(i,j)*State.m_xc[j];
            rowscale=MathMax(rowscale,MathAbs(State.m_cleic.Get(i,j)*State.m_s[j]));
           }
         if(v<=(-1.0E5*CMath::m_machineepsilon*rowscale))
           {
            //--- NOTE: it is important to check for non-strict inequality
            //---       because we have to correctly handle zero constraint
            //---       0*x<=0
            continue;
           }
        }
      for(i_=0; i_<n; i_++)
         State.m_rctmpdense0.Set(i_,nactivelin,State.m_cleic.Get(i,i_));
      State.m_cstatus.Set(n+i,0);
      State.m_rctmpconstraintidx.Set(nactiveconstraints,n+i);
      State.m_rctmpisequality[nactiveconstraints]=i<nec;
      nactiveconstraints++;
      nactivelin++;
     }
//--- Skip if no "candidate" constraints was found
   if(nactiveconstraints==0)
     {
      for(i=0; i<n; i++)
        {
         if(State.m_HasBndL[i] && State.m_HasBndU[i] && State.m_bndl[i]==State.m_bndu[i])
           {
            State.m_cstatus.Set(i,1);
            continue;
           }
         if(State.m_HasBndL[i] && State.m_xc[i]==State.m_bndl[i] && gc[i]>=0.0)
           {
            State.m_cstatus.Set(i,1);
            continue;
           }
         if(State.m_HasBndU[i] && State.m_xc[i]==State.m_bndu[i] && gc[i]<=0.0)
           {
            State.m_cstatus.Set(i,1);
            continue;
           }
        }
      return;
     }
//--- General case.
//--- APPROACH TO CONSTRAINTS ACTIVATION/DEACTIVATION
//--- We have NActiveConstraints "candidates": NActiveBnd boundary candidates,
//--- NActiveLin linear candidates. Indexes of boundary constraints are stored
//--- in RCTmpConstraintIdx[0:NActiveBnd-1], indexes of linear ones are stored
//--- in RCTmpConstraintIdx[NActiveBnd:NActiveBnd+NActiveLin-1]. Some of the
//--- constraints are equality ones, some are inequality - as specified by
//--- RCTmpIsEquality[i].
//--- Now we have to determine active subset of "candidates" set. In order to
//--- do so we solve following constrained minimization problem:
//---         (                         )^2
//---     min ( SUM(lambda[i]*A[i]) + G )
//---         (                         )
//--- Here:
//--- * G is a gradient (column vector)
//--- * A[i] is a column vector, linear (left) part of I-th constraint.
//---   I=0..NActiveConstraints-1, first NActiveBnd elements of A are just
//---   subset of identity matrix (boundary constraints), next NActiveLin
//---   elements are subset of rows of the matrix of general linear constraints.
//--- * lambda[i] is a Lagrange multiplier corresponding to I-th constraint
//--- NOTE: for preconditioned setting A is replaced by A*H^(-0.5), G is
//---       replaced by G*H^(-0.5). We apply this scaling at the last stage,
//---       before passing data to NNLS m_solver.
//--- Minimization is performed subject to non-negativity constraints on
//--- lambda[i] corresponding to inequality constraints. Inequality constraints
//--- which correspond to non-zero lambda are activated, equality constraints
//--- are always considered active.
//--- Informally speaking, we "decompose" descent direction -G and represent
//--- it as sum of constraint vectors and "residual" part (which is equal to
//--- the actual descent direction subject to constraints).
//--- SOLUTION OF THE NNLS PROBLEM
//--- We solve this optimization problem with Non-Negative Least Squares m_solver,
//--- which can efficiently solve least squares problems of the form
//---         ( [ I | AU ]     )^2
//---     min ( [   |    ]*x-b )   s.m_t. non-negativity constraints on some x[i]
//---         ( [ 0 | AL ]     )
//--- In order to use this m_solver we have to rearrange rows of A[] and G in
//--- such way that first NActiveBnd columns of A store identity matrix (before
//--- sorting non-zero elements are randomly distributed in the first NActiveBnd
//--- columns of A, during sorting we move them to first NActiveBnd rows).
//--- Then we create instance of NNLS m_solver (we reuse instance left from the
//--- previous run of the optimization problem) and solve NNLS problem.
   idx0=0;
   idx1=nactivebnd;
   for(i=0; i<n; i++)
     {
      if(State.m_cstatus[i]>=0)
        {
         v=1/MathSqrt(h[i]);
         for(j=0; j<=nactivelin-1; j++)
            State.m_rctmpdense1.Set(idx0,j,State.m_rctmpdense0.Get(i,j)/State.m_rctmps[i]*v);
         State.m_rctmprightpart.Set(idx0,State.m_rctmpg[i]/State.m_rctmps[i]*v);
         idx0++;
        }
      else
        {
         v=1/MathSqrt(h[i]);
         for(j=0; j<nactivelin; j++)
            State.m_rctmpdense1.Set(idx1,j,State.m_rctmpdense0.Get(i,j)*v);
         State.m_rctmprightpart.Set(idx1,State.m_rctmpg[i]*v);
         idx1++;
        }
     }
   CSNNLS::SNNLSInit(n,MathMin(nec+nic,n),n,State.m_solver);
   CSNNLS::SNNLSSetProblem(State.m_solver,State.m_rctmpdense1,State.m_rctmprightpart,nactivebnd,nactiveconstraints-nactivebnd,n);
   for(i=0; i<=nactiveconstraints-1; i++)
     {
      if(State.m_rctmpisequality[i])
         CSNNLS::SNNLSDropNNC(State.m_solver,i);
     }
   CSNNLS::SNNLSSolve(State.m_solver,State.m_rctmplambdas);
//--- After solution of the problem we activate equality constraints (always active)
//--- and inequality constraints with non-zero Lagrange multipliers. Then we reorthogonalize
//--- active constraints.
   State.m_cstatus.Fill(-1,0,n+nec+nic);
   for(i=0; i<nactiveconstraints; i++)
     {
      if(State.m_rctmpisequality[i] || State.m_rctmplambdas[i]>0.0)
         State.m_cstatus.Set(State.m_rctmpconstraintidx[i],1);
      else
         State.m_cstatus.Set(State.m_rctmpconstraintidx[i],0);
     }
   SASRebuildBasis(State);
  }
//+------------------------------------------------------------------+
//| This object stores nonlinear optimizer State.                    |
//| You should use functions provided by MinBLEIC subpackage to work |
//| with this object                                                 |
//+------------------------------------------------------------------+
class CMinBLEICState
  {
public:
   //--- variables
   int               m_bufsize;
   int               m_cidx;
   int               m_maxits;
   int               m_mcstage;
   int               m_nec;
   int               m_nfev;
   int               m_nic;
   int               m_nmain;
   int               m_nonmonotoniccnt;
   int               m_nslack;
   int               m_prectype;
   int               m_repdebugfeasgpaits;
   int               m_repdebugfeasqpits;
   int               m_repinneriterationscount;
   int               m_repnfev;
   int               m_repouteriterationscount;
   int               m_repterminationtype;
   int               m_repvaridx;
   int               m_smoothnessguardlevel;
   double            m_activationstep;
   double            m_curstpmax;
   double            m_cval;
   double            m_diffstep;
   double            m_epsf;
   double            m_epsg;
   double            m_epsx;
   double            m_f;
   double            m_fbase;
   double            m_fc;
   double            m_fm1;
   double            m_fm2;
   double            m_fn;
   double            m_fp1;
   double            m_fp2;
   double            m_fp;
   double            m_gm1;
   double            m_gp1;
   double            m_lastgoodstep;
   double            m_lastscaledgoodstep;
   double            m_maxscaledgrad;
   double            m_repdebugdx;
   double            m_repdebugeqerr;
   double            m_repdebugff;
   double            m_repdebugfs;
   double            m_stp;
   double            m_stpmax;
   double            m_teststep;
   double            m_trimthreshold;
   double            m_xm1;
   double            m_xp1;
   bool              m_boundedstep;
   bool              m_drep;
   bool              m_lsstart;
   bool              m_needf;
   bool              m_needfg;
   bool              m_steepestdescentstep;
   bool              m_userterminationneeded;
   bool              m_xrep;
   bool              m_xupdated;
   //--- objects
   RCommState        m_rstate;
   CSmoothnessMonitor m_smonitor;
   CSNNLSSolver      m_solver;
   CSActiveSet       m_sas;
   CLinMinState      m_lstate;
   //--- arrays
   bool              m_HasBndL[];
   bool              m_HasBndU[];
   CRowDouble        m_bndl;
   CRowDouble        m_bndu;
   CRowDouble        m_bufrho;
   CRowDouble        m_buftheta;
   CRowDouble        m_cgc;
   CRowDouble        m_cgn;
   CRowDouble        m_d;
   CRowDouble        m_diagh;
   CRowDouble        m_g;
   CRowDouble        m_invs;
   CRowDouble        m_lastscaleused;
   CRowDouble        m_s;
   CRowDouble        m_tmp0;
   CRowDouble        m_tmpprec;
   CRowDouble        m_ugc;
   CRowDouble        m_ugn;
   CRowDouble        m_work;
   CRowDouble        m_x;
   CRowDouble        m_xn;
   CRowDouble        m_xp;
   CRowDouble        m_xstart;
   //--- matrix
   CMatrixDouble     m_bufsk;
   CMatrixDouble     m_bufyk;
   CMatrixDouble     m_cleic;
   //--- constructor, destructor
                     CMinBLEICState(void);
                    ~CMinBLEICState(void) {}
   //--- copy
   void              Copy(const CMinBLEICState &obj);
   //--- overloading
   void              operator=(const CMinBLEICState &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CMinBLEICState::CMinBLEICState(void)
  {
   m_bufsize=0;
   m_cidx=0;
   m_maxits=0;
   m_mcstage=0;
   m_nec=0;
   m_nfev=0;
   m_nic=0;
   m_nmain=0;
   m_nonmonotoniccnt=0;
   m_nslack=0;
   m_prectype=0;
   m_repdebugfeasgpaits=0;
   m_repdebugfeasqpits=0;
   m_repinneriterationscount=0;
   m_repnfev=0;
   m_repouteriterationscount=0;
   m_repterminationtype=0;
   m_repvaridx=0;
   m_smoothnessguardlevel=0;
   m_activationstep=0;
   m_curstpmax=0;
   m_cval=0;
   m_diffstep=0;
   m_epsf=0;
   m_epsg=0;
   m_epsx=0;
   m_f=0;
   m_fbase=0;
   m_fc=0;
   m_fm1=0;
   m_fm2=0;
   m_fn=0;
   m_fp1=0;
   m_fp2=0;
   m_fp=0;
   m_gm1=0;
   m_gp1=0;
   m_lastgoodstep=0;
   m_lastscaledgoodstep=0;
   m_maxscaledgrad=0;
   m_repdebugdx=0;
   m_repdebugeqerr=0;
   m_repdebugff=0;
   m_repdebugfs=0;
   m_stp=0;
   m_stpmax=0;
   m_teststep=0;
   m_trimthreshold=0;
   m_xm1=0;
   m_xp1=0;
   m_boundedstep=false;
   m_drep=false;
   m_lsstart=false;
   m_needf=false;
   m_needfg=false;
   m_steepestdescentstep=false;
   m_userterminationneeded=false;
   m_xrep=false;
   m_xupdated=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CMinBLEICState::Copy(const CMinBLEICState &obj)
  {
   m_bufsize=obj.m_bufsize;
   m_cidx=obj.m_cidx;
   m_maxits=obj.m_maxits;
   m_mcstage=obj.m_mcstage;
   m_nec=obj.m_nec;
   m_nfev=obj.m_nfev;
   m_nic=obj.m_nic;
   m_nmain=obj.m_nmain;
   m_nonmonotoniccnt=obj.m_nonmonotoniccnt;
   m_nslack=obj.m_nslack;
   m_prectype=obj.m_prectype;
   m_repdebugfeasgpaits=obj.m_repdebugfeasgpaits;
   m_repdebugfeasqpits=obj.m_repdebugfeasqpits;
   m_repinneriterationscount=obj.m_repinneriterationscount;
   m_repnfev=obj.m_repnfev;
   m_repouteriterationscount=obj.m_repouteriterationscount;
   m_repterminationtype=obj.m_repterminationtype;
   m_repvaridx=obj.m_repvaridx;
   m_smoothnessguardlevel=obj.m_smoothnessguardlevel;
   m_activationstep=obj.m_activationstep;
   m_curstpmax=obj.m_curstpmax;
   m_cval=obj.m_cval;
   m_diffstep=obj.m_diffstep;
   m_epsf=obj.m_epsf;
   m_epsg=obj.m_epsg;
   m_epsx=obj.m_epsx;
   m_f=obj.m_f;
   m_fbase=obj.m_fbase;
   m_fc=obj.m_fc;
   m_fm1=obj.m_fm1;
   m_fm2=obj.m_fm2;
   m_fn=obj.m_fn;
   m_fp1=obj.m_fp1;
   m_fp2=obj.m_fp2;
   m_fp=obj.m_fp;
   m_gm1=obj.m_gm1;
   m_gp1=obj.m_gp1;
   m_lastgoodstep=obj.m_lastgoodstep;
   m_lastscaledgoodstep=obj.m_lastscaledgoodstep;
   m_maxscaledgrad=obj.m_maxscaledgrad;
   m_repdebugdx=obj.m_repdebugdx;
   m_repdebugeqerr=obj.m_repdebugeqerr;
   m_repdebugff=obj.m_repdebugff;
   m_repdebugfs=obj.m_repdebugfs;
   m_stp=obj.m_stp;
   m_stpmax=obj.m_stpmax;
   m_teststep=obj.m_teststep;
   m_trimthreshold=obj.m_trimthreshold;
   m_xm1=obj.m_xm1;
   m_xp1=obj.m_xp1;
   m_boundedstep=obj.m_boundedstep;
   m_drep=obj.m_drep;
   ArrayCopy(m_HasBndL,obj.m_HasBndL);
   ArrayCopy(m_HasBndU,obj.m_HasBndU);
   m_lsstart=obj.m_lsstart;
   m_needf=obj.m_needf;
   m_needfg=obj.m_needfg;
   m_steepestdescentstep=obj.m_steepestdescentstep;
   m_userterminationneeded=obj.m_userterminationneeded;
   m_xrep=obj.m_xrep;
   m_xupdated=obj.m_xupdated;
   m_rstate=obj.m_rstate;
   m_smonitor=obj.m_smonitor;
   m_solver=obj.m_solver;
   m_sas=obj.m_sas;
   m_bndl=obj.m_bndl;
   m_bndu=obj.m_bndu;
   m_bufrho=obj.m_bufrho;
   m_buftheta=obj.m_buftheta;
   m_cgc=obj.m_cgc;
   m_cgn=obj.m_cgn;
   m_d=obj.m_d;
   m_diagh=obj.m_diagh;
   m_g=obj.m_g;
   m_invs=obj.m_invs;
   m_lastscaleused=obj.m_lastscaleused;
   m_s=obj.m_s;
   m_tmp0=obj.m_tmp0;
   m_tmpprec=obj.m_tmpprec;
   m_ugc=obj.m_ugc;
   m_ugn=obj.m_ugn;
   m_work=obj.m_work;
   m_x=obj.m_x;
   m_xn=obj.m_xn;
   m_xp=obj.m_xp;
   m_xstart=obj.m_xstart;
   m_bufsk=obj.m_bufsk;
   m_bufyk=obj.m_bufyk;
   m_cleic=obj.m_cleic;
   m_lstate=obj.m_lstate;
  }
//+------------------------------------------------------------------+
//| This object stores nonlinear optimizer State.                    |
//| You should use functions provided by MinBLEIC subpackage to work |
//| with this object                                                 |
//+------------------------------------------------------------------+
class CMinBLEICStateShell
  {
private:
   CMinBLEICState    m_innerobj;

public:
   //--- constructors, destructor
                     CMinBLEICStateShell(void) {}
                     CMinBLEICStateShell(CMinBLEICState &obj) { m_innerobj.Copy(obj); }
                    ~CMinBLEICStateShell(void) {}
   //--- methods
   bool              GetNeedF(void);
   void              SetNeedF(const bool b);
   bool              GetNeedFG(void);
   void              SetNeedFG(const bool b);
   bool              GetXUpdated(void);
   void              SetXUpdated(const bool b);
   double            GetF(void);
   void              SetF(const double d);
   CMinBLEICState   *GetInnerObj(void);
  };
//+------------------------------------------------------------------+
//| Returns the value of the variable needf                          |
//+------------------------------------------------------------------+
bool CMinBLEICStateShell::GetNeedF(void)
  {
   return(m_innerobj.m_needf);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable needf                         |
//+------------------------------------------------------------------+
void CMinBLEICStateShell::SetNeedF(const bool b)
  {
   m_innerobj.m_needf=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable needfg                         |
//+------------------------------------------------------------------+
bool CMinBLEICStateShell::GetNeedFG(void)
  {
   return(m_innerobj.m_needfg);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable needfg                        |
//+------------------------------------------------------------------+
void CMinBLEICStateShell::SetNeedFG(const bool b)
  {
   m_innerobj.m_needfg=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable xupdated                       |
//+------------------------------------------------------------------+
bool CMinBLEICStateShell::GetXUpdated(void)
  {
   return(m_innerobj.m_xupdated);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable xupdated                      |
//+------------------------------------------------------------------+
void CMinBLEICStateShell::SetXUpdated(const bool b)
  {
   m_innerobj.m_xupdated=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable f                              |
//+------------------------------------------------------------------+
double CMinBLEICStateShell::GetF(void)
  {
   return(m_innerobj.m_f);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable f                             |
//+------------------------------------------------------------------+
void CMinBLEICStateShell::SetF(const double d)
  {
   m_innerobj.m_f=d;
  }
//+------------------------------------------------------------------+
//| Return object of class                                           |
//+------------------------------------------------------------------+
CMinBLEICState *CMinBLEICStateShell::GetInnerObj(void)
  {
   return(GetPointer(m_innerobj));
  }
//+------------------------------------------------------------------+
//| This structure stores optimization report:                       |
//| * InnerIterationsCount      number of inner iterations           |
//| * OuterIterationsCount      number of outer iterations           |
//| * NFEV                      number of gradient evaluations       |
//| * TerminationType           termination type (see below)         |
//| TERMINATION CODES                                                |
//| TerminationType field contains completion code,which can be:     |
//|   -10   unsupported combination of algorithm Settings:           |
//|         1) StpMax is set to non-zero value,                      |
//|         AND 2) non-default preconditioner is used.               |
//|         You can't use both features at the same moment,          |
//|         so you have to choose one of them (and to turn           |
//|         off another one).                                        |
//|   -3    inconsistent constraints. Feasible point is              |
//|         either nonexistent or too hard to find. Try to           |
//|         restart optimizer with better initial                    |
//|         approximation                                            |
//|    4    conditions on constraints are fulfilled                  |
//|         with error less than or equal to EpsC                    |
//|    5    MaxIts steps was taken                                   |
//|    7    stopping conditions are too stringent,                   |
//|         further improvement is impossible,                       |
//|         X contains best point found so far.                      |
//| ADDITIONAL FIELDS                                                |
//| There are additional fields which can be used for debugging:     |
//| * DebugEqErr                error in the equality constraints    |
//|                             (2-norm)                             |
//| * DebugFS                   f,calculated at projection of initial|
//|                             point to the feasible set            |
//| * DebugFF                   f,calculated at the final point      |
//| * DebugDX                   |X_start-X_final|                    |
//+------------------------------------------------------------------+
class CMinBLEICReport
  {
public:
   //--- variables
   int               m_debugfeasgpaits;
   int               m_debugfeasqpits;
   int               m_inneriterationscount;
   int               m_iterationscount;
   int               m_nfev;
   int               m_outeriterationscount;
   int               m_terminationtype;
   int               m_varidx;
   double            m_debugdx;
   double            m_debugeqerr;
   double            m_debugff;
   double            m_debugfs;
   //--- constructor, destructor
                     CMinBLEICReport(void) { ZeroMemory(this); }
                    ~CMinBLEICReport(void) {}
   //--- copy
   void              Copy(const CMinBLEICReport &obj);
   //--- overloading
   void              operator=(const CMinBLEICReport &obj) { Copy(obj); }

  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CMinBLEICReport::Copy(const CMinBLEICReport &obj)
  {
//--- copy variables
   m_debugfeasgpaits=obj.m_debugfeasgpaits;
   m_debugfeasqpits=obj.m_debugfeasqpits;
   m_inneriterationscount=obj.m_inneriterationscount;
   m_iterationscount=obj.m_iterationscount;
   m_nfev=obj.m_nfev;
   m_outeriterationscount=obj.m_outeriterationscount;
   m_terminationtype=obj.m_terminationtype;
   m_varidx=obj.m_varidx;
   m_debugdx=obj.m_debugdx;
   m_debugeqerr=obj.m_debugeqerr;
   m_debugff=obj.m_debugff;
   m_debugfs=obj.m_debugfs;
  }
//+------------------------------------------------------------------+
//| This structure stores optimization report:                       |
//| * InnerIterationsCount      number of inner iterations           |
//| * OuterIterationsCount      number of outer iterations           |
//| * NFEV                      number of gradient evaluations       |
//| * TerminationType           termination type (see below)         |
//| TERMINATION CODES                                                |
//| TerminationType field contains completion code,which can be:     |
//|   -10   unsupported combination of algorithm Settings:           |
//|         1) StpMax is set to non-zero value,                      |
//|         AND 2) non-default preconditioner is used.               |
//|         You can't use both features at the same moment,          |
//|         so you have to choose one of them (and to turn           |
//|         off another one).                                        |
//|   -3    inconsistent constraints. Feasible point is              |
//|         either nonexistent or too hard to find. Try to           |
//|         restart optimizer with better initial                    |
//|         approximation                                            |
//|    4    conditions on constraints are fulfilled                  |
//|         with error less than or equal to EpsC                    |
//|    5    MaxIts steps was taken                                   |
//|    7    stopping conditions are too stringent,                   |
//|         further improvement is impossible,                       |
//|         X contains best point found so far.                      |
//| ADDITIONAL FIELDS                                                |
//| There are additional fields which can be used for debugging:     |
//| * DebugEqErr                error in the equality constraints    |
//|                             (2-norm)                             |
//| * DebugFS                   f,calculated at projection of initial|
//|                             point to the feasible set            |
//| * DebugFF                   f,calculated at the final point      |
//| * DebugDX                   |X_start-X_final|                    |
//+------------------------------------------------------------------+
class CMinBLEICReportShell
  {
private:
   CMinBLEICReport   m_innerobj;

public:
   //--- constructors, destructor
                     CMinBLEICReportShell(void) {}
                     CMinBLEICReportShell(CMinBLEICReport &obj) { m_innerobj.Copy(obj); }
                    ~CMinBLEICReportShell(void) {}
   //--- methods
   int               GetInnerIterationsCount(void);
   void              SetInnerIterationsCount(const int i);
   int               GetOuterIterationsCount(void);
   void              SetOuterIterationsCount(const int i);
   int               GetNFev(void);
   void              SetNFev(const int i);
   int               GetTerminationType(void);
   void              SetTerminationType(const int i);
   double            GetDebugEqErr(void);
   void              SetDebugEqErr(const double d);
   double            GetDebugFS(void);
   void              SetDebugFS(const double d);
   double            GetDebugFF(void);
   void              SetDebugFF(const double d);
   double            GetDebugDX(void);
   void              SetDebugDX(const double d);
   CMinBLEICReport  *GetInnerObj(void);
  };
//+------------------------------------------------------------------+
//| Returns the value of the variable inneriterationscount           |
//+------------------------------------------------------------------+
int CMinBLEICReportShell::GetInnerIterationsCount(void)
  {
   return(m_innerobj.m_inneriterationscount);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable inneriterationscount          |
//+------------------------------------------------------------------+
void CMinBLEICReportShell::SetInnerIterationsCount(const int i)
  {
   m_innerobj.m_inneriterationscount=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable outeriterationscount           |
//+------------------------------------------------------------------+
int CMinBLEICReportShell::GetOuterIterationsCount(void)
  {
   return(m_innerobj.m_outeriterationscount);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable outeriterationscount          |
//+------------------------------------------------------------------+
void CMinBLEICReportShell::SetOuterIterationsCount(const int i)
  {
   m_innerobj.m_outeriterationscount=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable nfev                           |
//+------------------------------------------------------------------+
int CMinBLEICReportShell::GetNFev(void)
  {
   return(m_innerobj.m_nfev);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable nfev                          |
//+------------------------------------------------------------------+
void CMinBLEICReportShell::SetNFev(const int i)
  {
   m_innerobj.m_nfev=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable terminationtype                |
//+------------------------------------------------------------------+
int CMinBLEICReportShell::GetTerminationType(void)
  {
   return(m_innerobj.m_terminationtype);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable terminationtype               |
//+------------------------------------------------------------------+
void CMinBLEICReportShell::SetTerminationType(const int i)
  {
   m_innerobj.m_terminationtype=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable debugeqerr                     |
//+------------------------------------------------------------------+
double CMinBLEICReportShell::GetDebugEqErr(void)
  {
   return(m_innerobj.m_debugeqerr);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable debugeqerr                    |
//+------------------------------------------------------------------+
void CMinBLEICReportShell::SetDebugEqErr(const double d)
  {
   m_innerobj.m_debugeqerr=d;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable debugfs                        |
//+------------------------------------------------------------------+
double CMinBLEICReportShell::GetDebugFS(void)
  {
   return(m_innerobj.m_debugfs);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable debugfs                       |
//+------------------------------------------------------------------+
void CMinBLEICReportShell::SetDebugFS(const double d)
  {
   m_innerobj.m_debugfs=d;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable debugff                        |
//+------------------------------------------------------------------+
double CMinBLEICReportShell::GetDebugFF(void)
  {
   return(m_innerobj.m_debugff);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable debugff                       |
//+------------------------------------------------------------------+
void CMinBLEICReportShell::SetDebugFF(const double d)
  {
   m_innerobj.m_debugff=d;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable debugdx                        |
//+------------------------------------------------------------------+
double CMinBLEICReportShell::GetDebugDX(void)
  {
   return(m_innerobj.m_debugdx);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable debugdx                       |
//+------------------------------------------------------------------+
void CMinBLEICReportShell::SetDebugDX(const double d)
  {
   m_innerobj.m_debugdx=d;
  }
//+------------------------------------------------------------------+
//| Return object of class                                           |
//+------------------------------------------------------------------+
CMinBLEICReport *CMinBLEICReportShell::GetInnerObj(void)
  {
   return(GetPointer(m_innerobj));
  }
//+------------------------------------------------------------------+
//| Bound constrained optimization with additional linear equality   |
//| and inequality constraints                                       |
//+------------------------------------------------------------------+
class CMinBLEIC
  {
public:
   //--- class constants
   static const double m_maxnonmonotoniclen;
   static const double m_gtol;
   static const double m_initialdecay;
   static const double m_mindecay;
   static const double m_decaycorrection;
   static const double m_penaltyfactor;

   //--- public methods
   static void       MinBLEICCreate(const int n,double &x[],CMinBLEICState &State);
   static void       MinBLEICCreate(const int n,CRowDouble &x,CMinBLEICState &State);
   static void       MinBLEICCreateF(const int n,double &x[],const double diffstep,CMinBLEICState &State);
   static void       MinBLEICCreateF(const int n,CRowDouble &x,const double diffstep,CMinBLEICState &State);
   static void       MinBLEICSetBC(CMinBLEICState &State,double &bndl[],double &bndu[]);
   static void       MinBLEICSetBC(CMinBLEICState &State,CRowDouble &bndl,CRowDouble &bndu);
   static void       MinBLEICSetLC(CMinBLEICState &State,CMatrixDouble &c,int &ct[],const int k);
   static void       MinBLEICSetLC(CMinBLEICState &State,CMatrixDouble &c,CRowInt &ct,const int k);
   static void       MinBLEICSetInnerCond(CMinBLEICState &State,const double epsg,const double epsf,const double epsx);
   static void       MinBLEICSetOuterCond(CMinBLEICState &State,const double epsx,const double epsi);
   static void       MinBLEICSetCond(CMinBLEICState &State,double epsg,double epsf,double epsx,int m_maxits);
   static void       MinBLEICSetScale(CMinBLEICState &State,double &s[]);
   static void       MinBLEICSetScale(CMinBLEICState &State,CRowDouble &s);
   static void       MinBLEICSetPrecDefault(CMinBLEICState &State);
   static void       MinBLEICSetPrecDiag(CMinBLEICState &State,double &d[]);
   static void       MinBLEICSetPrecDiag(CMinBLEICState &State,CRowDouble &d);
   static void       MinBLEICSetPrecScale(CMinBLEICState &State);
   static void       MinBLEICSetMaxIts(CMinBLEICState &State,const int m_maxits);
   static void       MinBLEICSetXRep(CMinBLEICState &State,const bool needxrep);
   static void       MinBLEICSetDRep(CMinBLEICState &State,bool needdrep);
   static void       MinBLEICSetStpMax(CMinBLEICState &State,const double stpmax);
   static void       MinBLEICOptGuardGradient(CMinBLEICState &State,double &teststep);
   static void       MinBLEICOptGuardSmoothness(CMinBLEICState &State,int level);
   static void       MinBLEICOptGuardResults(CMinBLEICState &State,COptGuardReport &rep);
   static void       MinBLEICOptGuardNonC1Test0Results(CMinBLEICState &State,COptGuardNonC1Test0Report &strrep,COptGuardNonC1Test0Report &lngrep);
   static void       MinBLEICOptGuardNonC1Test1Results(CMinBLEICState &State,COptGuardNonC1Test1Report &strrep,COptGuardNonC1Test1Report &lngrep);
   static void       MinBLEICResults(CMinBLEICState &State,double &x[],CMinBLEICReport &rep);
   static void       MinBLEICResults(CMinBLEICState &State,CRowDouble &x,CMinBLEICReport &rep);
   static void       MinBLEICResultsBuf(CMinBLEICState &State,double &x[],CMinBLEICReport &rep);
   static void       MinBLEICResultsBuf(CMinBLEICState &State,CRowDouble &x,CMinBLEICReport &rep);
   static void       MinBLEICRestartFrom(CMinBLEICState &State,double &x[]);
   static void       MinBLEICRestartFrom(CMinBLEICState &State,CRowDouble &x);
   static void       MinBLEICRequestTermination(CMinBLEICState &State);
   static void       MinBLEICEmergencyTermination(CMinBLEICState &State);
   static bool       MinBLEICIteration(CMinBLEICState &State);

private:
   static void       ClearRequestFields(CMinBLEICState &State);
   static void       MinBLEICInitInternal(const int n,CRowDouble &x,const double diffstep,CMinBLEICState &State);
   static void       UpdateEstimateOfGoodStep(double &estimate,double newstep);
  };
//+------------------------------------------------------------------+
//| Initialize constants                                             |
//+------------------------------------------------------------------+
const double CMinBLEIC::m_maxnonmonotoniclen=1.0E-7;
const double CMinBLEIC::m_gtol=0.4;
const double CMinBLEIC::m_initialdecay=0.5;
const double CMinBLEIC::m_mindecay=0.1;
const double CMinBLEIC::m_decaycorrection=0.8;
const double CMinBLEIC::m_penaltyfactor=100;
//+------------------------------------------------------------------+
//|                      BOUND CONSTRAINED OPTIMIZATION              |
//|        WITH ADDITIONAL LINEAR EQUALITY AND INEQUALITY CONSTRAINTS|
//| DESCRIPTION:                                                     |
//| The subroutine minimizes function F(x) of N arguments subject to |
//| any combination of:                                              |
//| * bound constraints                                              |
//| * linear inequality constraints                                  |
//| * linear equality constraints                                    |
//| REQUIREMENTS:                                                    |
//| * user must provide function value and gradient                  |
//| * starting point X0 must be feasible or                          |
//|   not too far away from the feasible set                         |
//| * grad(f) must be Lipschitz continuous on a level set:           |
//|   L = { x : f(x)<=f(x0) }                                        |
//| * function must be defined everywhere on the feasible set F      |
//| USAGE:                                                           |
//| Constrained optimization if far more complex than the            |
//| unconstrained one. Here we give very brief outline of the BLEIC  |
//| optimizer. We strongly recommend you to read examples in the     |
//| ALGLIB Reference Manual and to read ALGLIB User Guide on         |
//| optimization, which is available at                              |
//| http://www.alglib.net/optimization/                              |
//| 1. User initializes algorithm State with MinBLEICCreate() call   |
//| 2. USer adds boundary and/or linear constraints by calling       |
//|    MinBLEICSetBC() and MinBLEICSetLC() functions.                |
//| 3. User sets stopping conditions for underlying unconstrained    |
//|    m_solver with MinBLEICSetInnerCond() call.                      |
//|    This function controls accuracy of underlying optimization    |
//|    algorithm.                                                    |
//| 4. User sets stopping conditions for outer iteration by calling  |
//|    MinBLEICSetOuterCond() function.                              |
//|    This function controls handling of boundary and inequality    |
//|    constraints.                                                  |
//| 5. Additionally, user may set limit on number of internal        |
//|    iterations by MinBLEICSetMaxIts() call.                       |
//|    This function allows to prevent algorithm from looping        |
//|    forever.                                                      |
//| 6. User calls MinBLEICOptimize() function which takes algorithm  |
//|    State and pointer (delegate, etc.) to callback function       |
//|    which calculates F/G.                                         |
//| 7. User calls MinBLEICResults() to get solution                  |
//| 8. Optionally user may call MinBLEICRestartFrom() to solve       |
//|    another problem with same N but another starting point.       |
//|    MinBLEICRestartFrom() allows to reuse already initialized     |
//|    structure.                                                    |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>0:                          |
//|                 * if given, only leading N elements of X are     |
//|                   used                                           |
//|                 * if not given, automatically determined from    |
//|                   size ofX                                       |
//|     X       -   starting point, array[N]:                        |
//|                 * it is better to set X to a feasible point      |
//|                 * but X can be infeasible, in which case         |
//|                   algorithm will try to find feasible point      |
//|                   first, using X as initial approximation.       |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure stores algorithm State                 |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICCreate(const int n,double &x[],CMinBLEICState &State)
  {
   CRowDouble X=x;
   MinBLEICCreate(n,X,State);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICCreate(const int n,CRowDouble &x,CMinBLEICState &State)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(x)>=n,__FUNCTION__+": Length(X)<N"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- function call
   MinBLEICInitInternal(n,x,0.0,State);
  }
//+------------------------------------------------------------------+
//| The subroutine is finite difference variant of MinBLEICCreate(). |
//| It uses finite differences in order to differentiate target      |
//| function.                                                        |
//| Description below contains information which is specific to this |
//| function only. We recommend to read comments on MinBLEICCreate() |
//| in order to get more information about creation of BLEIC         |
//| optimizer.                                                       |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>0:                          |
//|                 * if given, only leading N elements of X are used|
//|                 * if not given, automatically determined from    |
//|                   size of X                                      |
//|     X       -   starting point, array[0..m_n-1].                   |
//|     DiffStep-   differentiation step, >0                         |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm State           |
//| NOTES:                                                           |
//| 1. algorithm uses 4-point central formula for differentiation.   |
//| 2. differentiation step along I-th axis is equal to DiffStep*S[I]|
//|    where S[] is scaling vector which can be set by               |
//|    MinBLEICSetScale() call.                                      |
//| 3. we recommend you to use moderate values of differentiation    |
//|    step. Too large step will result in too large truncation      |
//|    errors, while too small step will result in too large         |
//|    numerical errors. 1.0E-6 can be good value to start with.     |
//| 4. Numerical differentiation is very inefficient - one  gradient |
//|    calculation needs 4*N function evaluations. This function will|
//|    work for any N - either small (1...10), moderate (10...100) or|
//|    large (100...). However, performance penalty will be too      |
//|    severe for any N's except for small ones.                     |
//|    We should also say that code which relies on numerical        |
//|    differentiation is less robust and precise. CG needs exact    |
//|    gradient values. Imprecise gradient may slow down convergence,|
//|    especially on highly nonlinear problems.                      |
//|    Thus we recommend to use this function for fast prototyping on|
//|    small - dimensional problems only, and to implement analytical|
//|    gradient as soon as possible.                                 |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICCreateF(const int n,double &x[],
                                const double diffstep,
                                CMinBLEICState &State)
  {
   CRowDouble X=x;
   MinBLEICCreateF(n,X,diffstep,State);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICCreateF(const int n,CRowDouble &x,
                                const double diffstep,
                                CMinBLEICState &State)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1"))
      return;
//--- check
   if(!CAp::Assert(x.Size()>=n,__FUNCTION__+": Length(X)<N"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(diffstep),__FUNCTION__+": DiffStep is infinite or NaN!"))
      return;
//--- check
   if(!CAp::Assert(diffstep>0.0,__FUNCTION__+": DiffStep is non-positive!"))
      return;
//--- function call
   MinBLEICInitInternal(n,x,diffstep,State);
  }
//+------------------------------------------------------------------+
//| This function sets boundary constraints for BLEIC optimizer.     |
//| Boundary constraints are inactive by default (after initial      |
//| creation). They are preserved after algorithm restart with       |
//| MinBLEICRestartFrom().                                           |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure stores algorithm State                 |
//|     BndL    -   lower bounds, array[N].                          |
//|                 If some (all) variables are unbounded, you may   |
//|                 specify very small number or -INF.               |
//|     BndU    -   upper bounds, array[N].                          |
//|                 If some (all) variables are unbounded, you may   |
//|                 specify very large number or +INF.               |
//| NOTE 1: it is possible to specify BndL[i]=BndU[i]. In this case  |
//| I-th variable will be "frozen" at X[i]=BndL[i]=BndU[i].          |
//| NOTE 2: this m_solver has following useful properties:             |
//| * bound constraints are always satisfied exactly                 |
//| * function is evaluated only INSIDE area specified by bound      |
//|   constraints, even when numerical differentiation is used       |
//|   (algorithm adjusts nodes according to boundary constraints)    |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetBC(CMinBLEICState &State,double &bndl[],
                              double &bndu[])
  {
   CRowDouble BndL=bndl;
   CRowDouble BndU=bndu;
   MinBLEICSetBC(State,BndL,BndU);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetBC(CMinBLEICState &State,CRowDouble &bndl,
                              CRowDouble &bndu)
  {
   int n=State.m_nmain;
//--- check
   if(!CAp::Assert(CAp::Len(bndl)>=n,__FUNCTION__+": Length(BndL)<N"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(bndu)>=n,__FUNCTION__+": Length(BndU)<N"))
      return;

   for(int i=0; i<n; i++)
     {
      //--- check
      if(!CAp::Assert(CMath::IsFinite(bndl[i]) || AL_NEGINF==bndl[i],__FUNCTION__+": BndL contains NAN or +INF"))
         return;
      //--- check
      if(!CAp::Assert(CMath::IsFinite(bndu[i]) || AL_POSINF==bndu[i],__FUNCTION__+": BndU contains NAN or -INF"))
         return;
      //--- change values
      State.m_bndl.Set(i,bndl[i]);
      State.m_bndu.Set(i,bndu[i]);
      State.m_HasBndL[i]=MathIsValidNumber(bndl[i]);
      State.m_HasBndU[i]=MathIsValidNumber(bndu[i]);
     }
   CSActiveSets::SASSetBC(State.m_sas,bndl,bndu);
  }
//+------------------------------------------------------------------+
//| This function sets linear constraints for BLEIC optimizer.       |
//| Linear constraints are inactive by default (after initial        |
//| creation). They are preserved after algorithm restart with       |
//| MinBLEICRestartFrom().                                           |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure previously allocated with              |
//|                MinBLEICCreate call.                              |
//|     C       -   linear constraints, array[K,N+1].                |
//|                 Each row of C represents one constraint, either  |
//|                 equality or inequality (see below):              |
//|                 * first N elements correspond to coefficients,   |
//|                 * last element corresponds to the right part.    |
//|                 All elements of C (including right part) must be |
//|                 finite.                                          |
//|     CT      -   type of constraints, array[K]:                   |
//|                 * if CT[i]>0, then I-th constraint is            |
//|                   C[i,*]*x >= C[i,n+1]                           |
//|                 * if CT[i]=0, then I-th constraint is            |
//|                   C[i,*]*x  = C[i,n+1]                           |
//|                 * if CT[i]<0, then I-th constraint is            |
//|                   C[i,*]*x <= C[i,n+1]                           |
//|     K       -   number of equality/inequality constraints, K>=0: |
//|                 * if given, only leading K elements of C/CT are  |
//|                   used                                           |
//|                 * if not given, automatically determined from    |
//|                   sizes of C/CT                                  |
//| NOTE 1: linear (non-bound) constraints are satisfied only        |
//| approximately:                                                   |
//| * there always exists some minor violation (about Epsilon in     |
//|   magnitude) due to rounding errors                              |
//| * numerical differentiation, if used, may lead to function       |
//|   evaluations outside of the feasible area, because algorithm    |
//|   does NOT change numerical differentiation formula according to |
//|   linear constraints.                                            |
//| If you want constraints to be satisfied exactly, try to          |
//| reformulate your problem in such manner that all constraints will|
//| become boundary ones (this kind of constraints is always         |
//| satisfied exactly, both in the final solution and in all         |
//| intermediate points).                                            |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetLC(CMinBLEICState &State,CMatrixDouble &c,
                              int &ct[],const int k)
  {
   CRowInt CT=ct;
   MinBLEICSetLC(State,c,CT,k);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetLC(CMinBLEICState &State,CMatrixDouble &c,
                              CRowInt &ct,const int k)
  {
//--- create variables
   int nmain=State.m_nmain;
   int i=0;
//--- First,check for errors in the inputs
   if(!CAp::Assert(k>=0,__FUNCTION__+": K<0"))
      return;
//--- check
   if(!CAp::Assert((int)CAp::Cols(c)>=nmain+1 || k==0,__FUNCTION__+": Cols(C)<N+1"))
      return;
//--- check
   if(!CAp::Assert((int)CAp::Rows(c)>=k,__FUNCTION__+": Rows(C)<K"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(ct)>=k,__FUNCTION__+": Length(CT)<K"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteMatrix(c,k,nmain+1),__FUNCTION__+": C contains infinite or NaN values!"))
      return;
//-- Handle zero K
   if(k==0)
     {
      State.m_nec=0;
      State.m_nic=0;
      CSActiveSets::SASSetLC(State.m_sas,c,ct,0);
      return;
     }
//-- Equality constraints are stored first, in the upper
//-- NEC rows of State.CLEIC matrix. Inequality constraints
//-- are stored in the next NIC rows.
//-- NOTE: we convert inequality constraints to the form
//-- A*x<=b before copying them.
   State.m_cleic.Resize(k,nmain+1);
   State.m_nec=0;
   State.m_nic=0;
   for(i=0; i<k; i++)
     {
      if(ct[i]==0)
        {
         State.m_cleic.Row(State.m_nec,c[i]+0);
         State.m_nec++;
        }
     }
   for(i=0; i<k; i++)
     {
      if(ct[i]!=0)
        {
         if(ct[i]>0)
            State.m_cleic.Row(State.m_nec+State.m_nic,c[i]*(-1.0));
         else
            State.m_cleic.Row(State.m_nec+State.m_nic,c[i]+0);
         State.m_nic++;
        }
     }
//-- Normalize rows of State.CLEIC: each row must have unit norm.
//-- Norm is calculated using first N elements (i.e. right part is
//-- not counted when we calculate norm).
   for(i=0; i<k; i++)
     {
      double v=CAblasF::RDotRR(nmain,State.m_cleic,i,State.m_cleic,i);
      if(v==0.0)
         continue;
      v=1/MathSqrt(v);
      State.m_cleic.Row(i,State.m_cleic[i]*v);
     }
   CSActiveSets::SASSetLC(State.m_sas,c,ct,k);
  }
//+------------------------------------------------------------------+
//| This function sets stopping conditions for the underlying        |
//| nonlinear CG optimizer. It controls overall accuracy of solution.|
//| These conditions should be strict enough in order for algorithm  |
//| to converge.                                                     |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     EpsG    -   >=0                                              |
//|                 The subroutine finishes its work if the condition|
//|                 |v|<EpsG is satisfied, where:                    |
//|                 * |.| means Euclidian norm                       |
//|                 * v - scaled gradient vector, v[i]=g[i]*s[i]     |
//|                 * g - gradient                                   |
//|                 * s - scaling coefficients set by                |
//|                       MinBLEICSetScale()                         |
//|     EpsF    -   >=0                                              |
//|                 The subroutine finishes its work if on k+1-th    |
//|                 iteration the condition |F(k+1)-F(k)| <=         |
//|                 <= EpsF*max{|F(k)|,|F(k+1)|,1} is satisfied.     |
//|     EpsX    -   >=0                                              |
//|                 The subroutine finishes its work if on k+1-th    |
//|                 iteration the condition |v|<=EpsX is fulfilled,  |
//|                 where:                                           |
//|                 * |.| means Euclidian norm                       |
//|                 * v - scaled step vector, v[i]=dx[i]/s[i]        |
//|                 * dx - ste pvector, dx=X(k+1)-X(k)               |
//|                 * s - scaling coefficients set by                |
//|                       MinBLEICSetScale()                         |
//| Passing EpsG=0, EpsF=0 and EpsX=0 (simultaneously) will lead to  |
//| automatic stopping criterion selection.                          |
//| These conditions are used to terminate inner iterations. However,|
//| you need to tune termination conditions for outer iterations too.|
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetInnerCond(CMinBLEICState &State,const double epsg,
                                     const double epsf,const double epsx)
  {
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsg),__FUNCTION__+": EpsG is not finite number"))
      return;
//--- check
   if(!CAp::Assert(epsg>=0.0,__FUNCTION__+": negative EpsG"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsf),__FUNCTION__+": EpsF is not finite number"))
      return;
//--- check
   if(!CAp::Assert(epsf>=0.0,__FUNCTION__+": negative EpsF"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsx),__FUNCTION__+": EpsX is not finite number"))
      return;
//--- check
   if(!CAp::Assert(epsx>=0.0,__FUNCTION__+": negative EpsX"))
      return;
//--- change values
   State.m_epsg=epsg;
   State.m_epsf=epsf;
   State.m_epsx=epsx;
  }
//+------------------------------------------------------------------+
//| This function sets stopping conditions for outer iteration of    |
//| BLEIC algo.                                                      |
//| These conditions control accuracy of constraint handling and     |
//| amount of infeasibility allowed in the solution.                 |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     EpsX    -   >0, stopping condition on outer iteration step   |
//|                     length                                       |
//|     EpsI    -   >0, stopping condition on infeasibility          |
//| Both EpsX and EpsI must be non-zero.                             |
//| MEANING OF EpsX                                                  |
//| EpsX is a stopping condition for outer iterations. Algorithm will|
//| stop when solution of the current modified subproblem will be    |
//| within EpsX (using 2-norm) of the previous solution.             |
//| MEANING OF EpsI                                                  |
//| EpsI controls feasibility properties - algorithm won't stop until|
//| all inequality constraints will be satisfied with error (distance|
//| from current point to the feasible area) at most EpsI.           |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetOuterCond(CMinBLEICState &State,const double epsx,
                                     const double epsi)
  {
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsx),__FUNCTION__+": EpsX is not finite number"))
      return;
//--- check
   if(!CAp::Assert(epsx>=0.0,__FUNCTION__+": non-positive EpsX"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsi),__FUNCTION__+": EpsI is not finite number"))
      return;
//--- check
   if(!CAp::Assert((double)(epsi)>0.0,__FUNCTION__+": non-positive EpsI"))
      return;
//--- change values
   State.m_epsx=epsx;
  }
//+------------------------------------------------------------------+
//| This function sets stopping conditions for the optimizer.        |
//| INPUT PARAMETERS:                                                |
//|   State    -  structure which stores algorithm State             |
//|   EpsG     -  >=0, The  subroutine  finishes  its  work if  the  |
//|                    condition |v|<EpsG is satisfied, where:       |
//|                     * |.| means Euclidian norm                   |
//|                     * v - scaled gradient vector, v[i]=g[i]*s[i] |
//|                     * g - gradient                               |
//|                     * s - scaling coefficients set by            |
//|                           MinBLEICSetScale()                     |
//|   EpsF     -  >=0, The subroutine finishes its work if on k+1-th |
//|                    iteration the condition                       |
//|                       |F(k+1)-F(k)|<=EpsF*max{|F(k)|,|F(k+1)|,1} |
//|                    is satisfied.                                 |
//|   EpsX     -  >=0, The subroutine finishes its work if on k+1-th |
//|                    iteration the condition |v|<=EpsX is fulfilled|
//|                    where:                                        |
//|                     * |.| means Euclidian norm                   |
//|                     * v - scaled step vector, v[i]=dx[i]/s[i]    |
//|                     * dx - step vector, dx=X(k+1)-X(k)           |
//|                     * s - scaling coefficients set by            |
//|                           MinBLEICSetScale()                     |
//|   MaxIts   -  maximum number of iterations. If MaxIts=0, the     |
//|               number  of iterations is unlimited.                |
//| Passing EpsG=0, EpsF=0 and EpsX=0 and MaxIts=0 (simultaneously)  |
//| will lead to automatic stopping criterion selection.             |
//| NOTE: when SetCond() called with non-zero MaxIts, BLEIC solver   |
//|       may perform slightly more than MaxIts iterations. I.e.,    |
//|      MaxIts  sets  non-strict limit on iterations count.         |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetCond(CMinBLEICState &State,
                                double epsg,
                                double epsf,
                                double epsx,
                                int m_maxits)
  {
//--- check
   if(!CAp::Assert(MathIsValidNumber(epsg),__FUNCTION__+": EpsG is not finite number"))
      return;
   if(!CAp::Assert(epsg>=0.0,__FUNCTION__+": negative EpsG"))
      return;
   if(!CAp::Assert(MathIsValidNumber(epsf),__FUNCTION__+": EpsF is not finite number"))
      return;
   if(!CAp::Assert(epsf>=0.0,__FUNCTION__+": negative EpsF"))
      return;
   if(!CAp::Assert(MathIsValidNumber(epsx),__FUNCTION__+": EpsX is not finite number"))
      return;
   if(!CAp::Assert(epsx>=0.0,__FUNCTION__+": negative EpsX"))
      return;
   if(!CAp::Assert(m_maxits>=0,__FUNCTION__+": negative MaxIts!"))
      return;
   if(epsg==0.0 && epsf==0.0 && epsx==0.0 && m_maxits==0)
      epsx=1.0E-6;
   State.m_epsg=epsg;
   State.m_epsf=epsf;
   State.m_epsx=epsx;
   State.m_maxits=m_maxits;
  }
//+------------------------------------------------------------------+
//| This function sets scaling coefficients for BLEIC optimizer.     |
//| ALGLIB optimizers use scaling matrices to test stopping          |
//| conditions (step size and gradient are scaled before comparison  |
//| with tolerances). Scale of the I-th variable is a translation    |
//| invariant measure of:                                            |
//| a) "how large" the variable is                                   |
//| b) how large the step should be to make significant changes in   |
//| the function                                                     |
//| Scaling is also used by finite difference variant of the         |
//| optimizer - step along I-th axis is equal to DiffStep*S[I].      |
//| In most optimizers (and in the BLEIC too) scaling is NOT a form  |
//| of preconditioning. It just affects stopping conditions. You     |
//| should set preconditioner by separate call to one of the         |
//| MinBLEICSetPrec...() functions.                                  |
//| There is a special preconditioning mode, however, which uses     |
//| scaling coefficients to form diagonal preconditioning matrix.    |
//| You can turn this mode on, if you want. But you should understand|
//| that scaling is not the same thing as preconditioning - these are|
//| two different, although related forms of tuning m_solver.          |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure stores algorithm State                 |
//|     S       -   array[N], non-zero scaling coefficients          |
//|                 S[i] may be negative, sign doesn't matter.       |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetScale(CMinBLEICState &State,double &s[])
  {
   CRowDouble Scale=s;
   MinBLEICSetScale(State,Scale);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetScale(CMinBLEICState &State,CRowDouble &s)
  {
//--- check
   if(!CAp::Assert(s.Size()>=State.m_nmain,__FUNCTION__+": Length(S)<N"))
      return;
   for(int i=0; i<State.m_nmain; i++)
     {
      //--- check
      if(!CAp::Assert(MathIsValidNumber(s[i]),__FUNCTION__+": S contains infinite or NAN elements"))
         return;
      if(!CAp::Assert(s[i]!=0.0,__FUNCTION__+": S contains zero elements"))
         return;
      State.m_s.Set(i,MathAbs(s[i]));
     }
   CSActiveSets::SASSetScale(State.m_sas,s);
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: preconditioning is turned    |
//| off.                                                             |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetPrecDefault(CMinBLEICState &State)
  {
   State.m_prectype=0;
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: diagonal of approximate      |
//| Hessian is used.                                                 |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     D       -   diagonal of the approximate Hessian,             |
//|                 array[0..m_n-1], (if larger, only leading N        |
//|                 elements are used).                              |
//| NOTE 1: D[i] should be positive. Exception will be thrown        |
//|         otherwise.                                               |
//| NOTE 2: you should pass diagonal of approximate Hessian - NOT    |
//|         ITS INVERSE.                                             |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetPrecDiag(CMinBLEICState &State,double &d[])
  {
   CRowDouble D=d;
   CMinBLEIC::MinBLEICSetPrecDiag(State,D);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetPrecDiag(CMinBLEICState &State,CRowDouble &d)
  {
//--- check
   if(!CAp::Assert(CAp::Len(d)>=State.m_nmain,__FUNCTION__+": D is too short"))
      return;
   for(int i=0; i<=State.m_nmain-1; i++)
     {
      //--- check
      if(!CAp::Assert(CMath::IsFinite(d[i]),__FUNCTION__+": D contains infinite or NAN elements"))
         return;
      //--- check
      if(!CAp::Assert(d[i]>0.0,__FUNCTION__+": D contains non-positive elements"))
         return;
     }
//--- function call
   CApServ::RVectorSetLengthAtLeast(State.m_diagh,State.m_nmain);
   State.m_prectype=2;
//--- copy
   State.m_diagh=d;
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: scale-based diagonal         |
//| preconditioning.                                                 |
//| This preconditioning mode can be useful when you don't have      |
//| approximate diagonal of Hessian, but you know that your variables|
//| are badly scaled (for example, one variable is in [1,10], and    |
//| another in [1000,100000]), and most part of the ill-conditioning |
//| comes from different scales of vars.                             |
//| In this case simple scale-based preconditioner, with H.Set(i,      |
//| = 1/(s[i]^2), can greatly improve convergence.                   |
//| IMPRTANT: you should set scale of your variables with            |
//| MinBLEICSetScale() call (before or after MinBLEICSetPrecScale()  |
//| call). Without knowledge of the scale of your variables          |
//| scale-based preconditioner will be just unit matrix.             |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetPrecScale(CMinBLEICState &State)
  {
   State.m_prectype=3;
  }
//+------------------------------------------------------------------+
//| This function allows to stop algorithm after specified number of |
//| inner iterations.                                                |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     MaxIts  -   maximum number of inner iterations.              |
//|                 If MaxIts=0, the number of iterations is         |
//|                 unlimited.                                       |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetMaxIts(CMinBLEICState &State,
                                  const int m_maxits)
  {
//--- check
   if(!CAp::Assert(m_maxits>=0,__FUNCTION__+": negative MaxIts!"))
      return;
//--- change value
   State.m_maxits=m_maxits;
  }
//+------------------------------------------------------------------+
//| This function turns on/off reporting.                            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     NeedXRep-   whether iteration reports are needed or not      |
//| If NeedXRep is True, algorithm will call rep() callback function |
//| if it is provided to MinBLEICOptimize().                         |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetXRep(CMinBLEICState &State,
                                const bool needxrep)
  {
   State.m_xrep=needxrep;
  }
//+------------------------------------------------------------------+
//| This function turns on/off line search reports.                  |
//| These reports are described in more details in developer-only    |
//| comments on MinBLEICState object.                                |
//| INPUT PARAMETERS:                                                |
//|   State    -  structure which stores algorithm State             |
//|   NeedDRep -  whether line search reports are needed or not      |
//| This function is intended for private use only. Turning it on    |
//| artificially may cause program failure.                          |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetDRep(CMinBLEICState &State,
                                bool needdrep)
  {
   State.m_drep=needdrep;
  }
//+------------------------------------------------------------------+
//| This function sets maximum step length                           |
//| IMPORTANT: this feature is hard to combine with preconditioning. |
//| You can't set upper limit on step length, when you solve         |
//| optimization problem with linear (non-boundary) constraints AND  |
//| preconditioner turned on.                                        |
//| When non-boundary constraints are present, you have to either a) |
//| use preconditioner, or b) use upper limit on step length. YOU    |
//| CAN'T USE BOTH! In this case algorithm will terminate with       |
//| appropriate error code.                                          |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     StpMax  -   maximum step length, >=0. Set StpMax to 0.0, if  |
//|                 you don't want to limit step length.             |
//| Use this subroutine when you optimize target function which      |
//| contains exp() or other fast growing functions, and optimization |
//| algorithm makes too large steps which lead to overflow. This     |
//| function allows us to reject steps that are too large (and       |
//| therefore expose us to the possible overflow) without actually   |
//| calculating function value at the x+stp*d.                       |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICSetStpMax(CMinBLEICState &State,
                                  const double stpmax)
  {
//--- check
   if(!CAp::Assert(CMath::IsFinite(stpmax),__FUNCTION__+": StpMax is not finite!"))
      return;
//--- check
   if(!CAp::Assert(stpmax>=0.0,__FUNCTION__+": StpMax<0!"))
      return;
//--- change value
   State.m_stpmax=stpmax;
  }
//+------------------------------------------------------------------+
//| This function activates/deactivates verification of the user-    |
//| supplied analytic gradient.                                      |
//| Upon activation of this option OptGuard integrity checker        |
//| performs numerical differentiation of your target function at the|
//| initial point (note: future versions may also perform check at   |
//| the final point) and compares numerical gradient with analytic   |
//| one provided by you.                                             |
//| If difference is too large, an error flag is set and optimization|
//| session continues. After optimization session is over, you can   |
//| retrieve the report which stores both gradients and specific     |
//| components highlighted as suspicious by the OptGuard.            |
//| The primary OptGuard report can be retrieved with                |
//| MinBLEICOptGuardResults().                                       |
//| IMPORTANT: gradient check is a high-overhead option which will   |
//|            cost you about 3*N additional function evaluations.   |
//|            In many cases it may cost as much as the rest of the  |
//|            optimization session.                                 |
//| YOU SHOULD NOT USE IT IN THE PRODUCTION CODE UNLESS YOU WANT TO  |
//|         CHECK DERIVATIVES PROVIDED BY SOME THIRD PARTY.          |
//| NOTE: unlike previous incarnation of the gradient checking code, |
//|       OptGuard does NOT interrupt optimization even if it        |
//|       discovers bad gradient.                                    |
//| INPUT PARAMETERS:                                                |
//|      State    -  structure used to store algorithm State         |
//|      TestStep -  verification step used for numerical            |
//|                  differentiation:                                |
//|                     * TestStep=0 turns verification off          |
//|                     * TestStep>0 activates verification          |
//|                  You should carefully choose TestStep. Value     |
//|                  which is too large (so large that function      |
//|                  behavior is non-cubic at this scale) will lead  |
//|                  to false alarms. Too short step will result in  |
//|                  rounding errors dominating numerical derivative.|
//| You may use different step for different parameters by means of  |
//| setting scale with MinBLEICSetScale().                           |
//| === EXPLANATION ================================================ |
//| In order to verify gradient algorithm performs following steps:  |
//|   * two trial steps are made to X[i]-TestStep*S[i] and           |
//|     X[i]+TestStep*S[i], where X[i] is i-th component of the      |
//|     initial point and S[i] is a scale of i-th parameter          |
//|   * F(X) is evaluated at these trial points                      |
//|   * we perform one more evaluation in the middle point of the    |
//|     interval                                                     |
//|   * we build cubic model using function values and derivatives at|
//|     trial points and we compare its prediction with actual value |
//|     in the middle point                                          |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICOptGuardGradient(CMinBLEICState &State,
                                         double &teststep)
  {
//--- check
   if(!CAp::Assert(MathIsValidNumber(teststep),__FUNCTION__+": TestStep contains NaN or INF"))
      return;
   if(!CAp::Assert(teststep>=0.0,__FUNCTION__+": invalid argument TestStep(TestStep<0)"))
      return;
   State.m_teststep=teststep;
  }
//+------------------------------------------------------------------+
//| This function activates/deactivates nonsmoothness monitoring     |
//| option of the OptGuard integrity checker. Smoothness monitor     |
//| silently observes solution process and tries to detect ill-posed |
//| problems, i.e. ones with:                                        |
//|   a) discontinuous target function (non-C0)                      |
//|   b) nonsmooth  target function (non-C1)                         |
//| Smoothness monitoring does NOT interrupt optimization even if it |
//| suspects that your problem is nonsmooth. It just sets            |
//| corresponding flags in the OptGuard report which can be retrieved|
//| after optimization is over.                                      |
//| Smoothness monitoring is a moderate overhead option which often  |
//| adds less than 1% to the optimizer running time. Thus, you can   |
//| use it even for large scale problems.                            |
//| NOTE: OptGuard does NOT guarantee that it will always detect     |
//|       C0/C1 continuity violations.                               |
//| First, minor errors are hard to catch - say, a 0.0001 difference |
//| in the model values at two sides of the gap may be due           |
//| to discontinuity of the model - or simply because the model has  |
//| changed.                                                         |
//| Second, C1-violations are especially difficult to detect in a    |
//| noninvasive way. The optimizer usually performs very short steps |
//| near the nonsmoothness, and differentiation usually introduces a |
//| lot of numerical noise. It is hard to tell whether some tiny     |
//| discontinuity in the slope is due to real nonsmoothness or just  |
//| due to numerical noise alone.                                    |
//| Our top priority was to avoid false positives, so in some rare   |
//| cases minor errors may went unnoticed (however, in most cases    |
//| they can be spotted with restart from different initial point).  |
//| INPUT PARAMETERS:                                                |
//|   State    -  algorithm State                                    |
//|   Level    -  monitoring level:                                  |
//|               * 0 - monitoring is disabled                       |
//|               * 1 - noninvasive low-overhead monitoring; function|
//|                     values and/or gradients are recorded, but    |
//|                     OptGuard does not try to perform additional  |
//|                     evaluations in order to get more information |
//|                     about suspicious locations.                  |
//| === EXPLANATION ================================================ |
//| One major source of headache during optimization is the          |
//| possibility of the coding errors in the target function /        |
//| constraints (or their gradients). Such errors most often manifest|
//| themselves as discontinuity or nonsmoothness of the target /     |
//| constraints.                                                     |
//| Another frequent situation is when you try to optimize something |
//| involving lots of min() and max() operations, i.e. nonsmooth     |
//| target. Although not a coding error, it is nonsmoothness anyway -|
//| and smooth optimizers usually stop right after encountering      |
//| nonsmoothness, well before reaching solution.                    |
//| OptGuard integrity checker helps you to catch such situations: it|
//| monitors function values/gradients being passed to the optimizer |
//| and tries to errors. Upon discovering suspicious pair of points  |
//| it raises appropriate flag (and allows you to continue           |
//| optimization). When optimization is done, you can study OptGuard |
//| result.                                                          |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICOptGuardSmoothness(CMinBLEICState &State,
                                           int level)
  {
//--- check
   if(!CAp::Assert(level==0 || level==1,__FUNCTION__+": unexpected value of level parameter"))
      return;
   State.m_smoothnessguardlevel=level;
  }
//+------------------------------------------------------------------+
//| Results of OptGuard integrity check, should be called after      |
//| optimization session is over.                                    |
//| === PRIMARY REPORT ============================================= |
//| OptGuard performs several checks which are intended to catch     |
//| common errors in the implementation of nonlinear function /      |
//| gradient:                                                        |
//|   * incorrect analytic gradient                                  |
//|   * discontinuous (non-C0) target functions (constraints)        |
//|   * nonsmooth  (non-C1) target functions (constraints)           |
//| Each of these checks is activated with appropriate function:     |
//|   * MinBLEICOptGuardGradient() for gradient verification         |
//|   * MinBLEICOptGuardSmoothness() for C0/C1 checks                |
//| Following flags are set when these errors are suspected:         |
//|   * rep.badgradsuspected, and additionally:                      |
//|      * rep.badgradvidx for specific variable (gradient element)  |
//|        suspected                                                 |
//|      * rep.badgradxbase, a point where gradient is tested        |
//|      * rep.badgraduser, user-provided gradient (stored as 2D     |
//|        matrix with single row in order to make report structure  |
//|        compatible with more complex optimizers like MinNLC or    |
//|        MinLM)                                                    |
//|      * rep.badgradnum, reference gradient obtained via numerical |
//|        differentiation (stored as 2D matrix with single row in   |
//|        order to make report structure compatible with more       |
//|        complex optimizers like MinNLC or MinLM)                  |
//|   * rep.nonc0suspected                                           |
//|   * rep.nonc1suspected                                           |
//| === ADDITIONAL REPORTS/LOGS ==================================== |
//| Several different tests are performed to catch C0/C1 errors, you |
//| can find out specific test signaled error by looking to:         |
//|   * rep.nonc0test0positive, for non-C0 test #0                   |
//|   * rep.nonc1test0positive, for non-C1 test #0                   |
//|   * rep.nonc1test1positive, for non-C1 test #1                   |
//| Additional information (including line search logs) can be       |
//| obtained by means of:                                            |
//|   * MinBLEICOptGuardNonC1Test0Results()                          |
//|   * MinBLEICOptGuardNonC1Test1Results()                          |
//| which return detailed error reports, specific points where       |
//| discontinuities were found, and so on.                           |
//| ================================================================ |
//| INPUT PARAMETERS:                                                |
//|   State    -  algorithm State                                    |
//| OUTPUT PARAMETERS:                                               |
//|   Rep      -  generic OptGuard report; more detailed reports     |
//|               can be retrieved with other functions.             |
//| NOTE: false negatives (nonsmooth problems are not identified as  |
//|       nonsmooth ones) are possible although unlikely.            |
//| The reason is that you need to make several evaluations around   |
//| nonsmoothness in order to accumulate enough information about    |
//| function curvature. Say, if you start right from the nonsmooth   |
//| point, optimizer simply won't get enough data to understand what |
//| is going wrong before it terminates due to abrupt changes in the |
//| derivative. It is also possible that "unlucky" step will move us |
//| to the termination too quickly.                                  |
//| Our current approach is to have less than 0.1% false negatives in|
//| our test examples (measured with multiple restarts from random   |
//| points), and to have exactly 0% false positives.                 |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICOptGuardResults(CMinBLEICState &State,
                                        COptGuardReport &rep)
  {
   COptServ::SmoothnessMonitorExportReport(State.m_smonitor,rep);
  }
//+------------------------------------------------------------------+
//| Detailed results of the OptGuard integrity check for             |
//| nonsmoothness test #0                                            |
//| Nonsmoothness (non-C1) test #0 studies function values (not      |
//| gradient!) obtained during line searches and monitors behavior   |
//| of the directional derivative estimate.                          |
//| This test is less powerful than test #1, but it does not depend  |
//| on the gradient values and thus it is more robust against        |
//| artifacts introduced by numerical differentiation.               |
//| Two reports are returned:                                        |
//|  *a "strongest" one, corresponding to line search which had    |
//|       highest value of the nonsmoothness indicator               |
//|  *a "longest" one, corresponding to line search which had more |
//|       function evaluations, and thus is more detailed            |
//| In both cases following fields are returned:                     |
//|   * positive  - is TRUE when test flagged suspicious point;      |
//|                    FALSE if test did not notice anything         |
//|                    (in the latter cases fields below are empty). |
//|   * x0[], d[] - arrays of length N which store initial point and |
//|                 direction for line search (d[] can be normalized,|
//|                 but does not have to)                            |
//|   * stp[], f[]- arrays of length CNT which store step lengths and|
//|                 function values at these points; f[i] is         |
//|                 evaluated in x0+stp[i]*d.                        |
//|   * stpidxa, stpidxb - we suspect that function violates C1      |
//|                 continuity between steps #stpidxa and #stpidxb   |
//|                 (usually we have stpidxb=stpidxa+3, with most    |
//|                 likely position of the violation between         |
//|                 stpidxa+1 and stpidxa+2.                         |
//| ================================================================ |
//| = SHORTLY SPEAKING: build a 2D plot of (stp,f) and look at it -  |
//| =                   you will see where C1 continuity is violated.|
//| ================================================================ |
//| INPUT PARAMETERS:                                                |
//|   State    -  algorithm State                                    |
//| OUTPUT PARAMETERS:                                               |
//|   StrRep  - C1 test #0 "strong" report                         |
//|   LngRep  - C1 test #0 "long" report                           |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICOptGuardNonC1Test0Results(CMinBLEICState &State,
                                                  COptGuardNonC1Test0Report &strrep,
                                                  COptGuardNonC1Test0Report &lngrep)
  {
   COptGuardApi::SmoothnessMonitorExportC1Test0Report(State.m_smonitor.m_nonc1test0strrep,State.m_lastscaleused,strrep);
   COptGuardApi::SmoothnessMonitorExportC1Test0Report(State.m_smonitor.m_nonc1test0lngrep,State.m_lastscaleused,lngrep);
  }
//+------------------------------------------------------------------+
//| Detailed results of the OptGuard integrity check for             |
//| nonsmoothness test #1                                            |
//| Nonsmoothness (non-C1) test #1 studies individual components of  |
//| the gradient computed during line search.                        |
//| When precise analytic gradient is provided this test is more     |
//| powerful than test #0 which works with function values and       |
//| ignores user-provided gradient. However, test #0 becomes more    |
//| powerful when numerical differentiation is employed (in such     |
//| cases test #1 detects higher levels of numerical noise and       |
//| becomes too conservative).                                       |
//| This test also tells specific components of the gradient which   |
//| violate C1 continuity, which makes it more informative than #0,  |
//| which just tells that continuity is violated.                    |
//| Two reports are returned:                                        |
//|  *a "strongest" one, corresponding to line search which had    |
//|     highest value of the nonsmoothness indicator                 |
//|  *a "longest" one, corresponding to line search which had more |
//|     function evaluations, and thus is more detailed              |
//| In both cases following fields are returned:                     |
//|   * positive  -  is TRUE when test flagged suspicious point;     |
//|                     FALSE if test did not notice anything        |
//|                     (in the latter cases fields below are empty).|
//|   * vidx      -  is an index of the variable in [0,N) with       |
//|                  nonsmooth derivative                            |
//|   * x0[], d[] -  arrays of length N which store initial point and|
//|                  direction for line search (d[] can be normalized|
//|                  but does not have to)                           |
//|   * stp[], g[]-  arrays of length CNT which store step lengths   |
//|                  and gradient values at these points; g[i] is    |
//|                  evaluated in x0+stp[i]*d and contains vidx-th   |
//|                  component of the gradient.                      |
//|   * stpidxa, stpidxb - we suspect that function violates C1      |
//|                  continuity between steps #stpidxa and #stpidxb  |
//|                  (usually we have stpidxb=stpidxa+3, with most   |
//|                  likely position of the violation between        |
//|                  stpidxa+1 and stpidxa+2.                        |
//| ================================================================ |
//| = SHORTLY SPEAKING: build a 2D plot of (stp,f) and look at it -  |
//| =                   you will see where C1 continuity is violated.|
//| ================================================================ |
//| INPUT PARAMETERS:                                                |
//|   State    -  algorithm State                                    |
//| OUTPUT PARAMETERS:                                               |
//|   StrRep  - C1 test #1 "strong" report                         |
//|   LngRep  - C1 test #1 "long" report                           |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICOptGuardNonC1Test1Results(CMinBLEICState &State,
                                                  COptGuardNonC1Test1Report &strrep,
                                                  COptGuardNonC1Test1Report &lngrep)
  {
   COptGuardApi::SmoothnessMonitorExportC1Test1Report(State.m_smonitor.m_nonc1test1strrep,State.m_lastscaleused,strrep);
   COptGuardApi::SmoothnessMonitorExportC1Test1Report(State.m_smonitor.m_nonc1test1lngrep,State.m_lastscaleused,lngrep);
  }
//+------------------------------------------------------------------+
//| BLEIC results                                                    |
//| INPUT PARAMETERS:                                                |
//|     State   -   algorithm State                                  |
//| OUTPUT PARAMETERS:                                               |
//|     X       -   array[0..m_n-1], solution                          |
//|     Rep     -   optimization report. You should check Rep.       |
//|                 TerminationType in order to distinguish          |
//|                 successful termination from unsuccessful one.    |
//|                 More information about fields of this structure  |
//|                 can be found in the comments on MinBLEICReport   |
//|                 datatype.                                        |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICResults(CMinBLEICState &State,double &x[],
                                CMinBLEICReport &rep)
  {
//--- reset memory
   ArrayResize(x,0);
//--- function call
   MinBLEICResultsBuf(State,x,rep);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICResults(CMinBLEICState &State,CRowDouble &x,
                                CMinBLEICReport &rep)
  {
//--- reset memory
   x.Resize(0);
//--- function call
   MinBLEICResultsBuf(State,x,rep);
  }
//+------------------------------------------------------------------+
//| BLEIC results                                                    |
//| Buffered implementation of MinBLEICResults() which uses          |
//| pre-allocated buffer to store X[]. If buffer size is too small,  |
//| it resizes buffer. It is intended to be used in the inner cycles |
//| of performance critical algorithms where array reallocation      |
//| penalty is too large to be ignored.                              |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICResultsBuf(CMinBLEICState &State,double &x[],
                                   CMinBLEICReport &rep)
  {
   CRowDouble X;
   MinBLEICResultsBuf(State,X,rep);
   X.ToArray(x);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICResultsBuf(CMinBLEICState &State,CRowDouble &x,
                                   CMinBLEICReport &rep)
  {
//--- change values
   rep.m_iterationscount=State.m_repinneriterationscount;
   rep.m_inneriterationscount=State.m_repinneriterationscount;
   rep.m_outeriterationscount=State.m_repouteriterationscount;
   rep.m_nfev=State.m_repnfev;
   rep.m_varidx=State.m_repvaridx;
   rep.m_terminationtype=State.m_repterminationtype;
//--- check
   if(State.m_repterminationtype>0)
      x=State.m_sas.m_xc;
   else
      x=vector<double>::Full(State.m_nmain,AL_NaN);
//--- change values
   rep.m_debugeqerr=State.m_repdebugeqerr;
   rep.m_debugfs=State.m_repdebugfs;
   rep.m_debugff=State.m_repdebugff;
   rep.m_debugdx=State.m_repdebugdx;
   rep.m_debugfeasqpits=State.m_repdebugfeasqpits;
   rep.m_debugfeasgpaits=State.m_repdebugfeasgpaits;
  }
//+------------------------------------------------------------------+
//| This subroutine restarts algorithm from new point.               |
//| All optimization parameters (including constraints) are left     |
//| unchanged.                                                       |
//| This function allows to solve multiple optimization problems     |
//| (which must have same number of dimensions) without object       |
//| reallocation penalty.                                            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure previously allocated with              |
//|                 MinBLEICCreate call.                             |
//|     X       -   new starting point.                              |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICRestartFrom(CMinBLEICState &State,double &x[])
  {
   CRowDouble X=x;
   MinBLEICRestartFrom(State,X);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICRestartFrom(CMinBLEICState &State,CRowDouble &x)
  {
   int n=State.m_nmain;
//--- First,check for errors in the inputs
   if(!CAp::Assert(CAp::Len(x)>=n,__FUNCTION__+": Length(X)<N"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- Set XC
   State.m_xstart=x;
   State.m_xstart.Resize(n);
//--- prepare RComm facilities
   State.m_rstate.ia.Resize(7);
   ArrayResize(State.m_rstate.ba,1);
   State.m_rstate.ra.Resize(6);
   State.m_rstate.stage=-1;
//--- function call
   ClearRequestFields(State);
   CSActiveSets::SASStopOptimization(State.m_sas);
  }
//+------------------------------------------------------------------+
//| This subroutine submits request for termination of running       |
//| optimizer. It should be called from user-supplied callback when  |
//| user decides that it is time to "smoothly" terminate optimization|
//| process. As result, optimizer stops at point which was "current  |
//| accepted" when termination  request  was submitted and returns   |
//| error code 8 (successful termination).                           |
//| INPUT PARAMETERS:                                                |
//|   State    -  optimizer structure                                |
//| NOTE: after  request  for  termination  optimizer  may   perform |
//|       several additional calls to user-supplied callbacks. It    |
//|       does NOT guarantee to stop immediately - it just guarantees|
//|       that these additional calls will be discarded later.       |
//| NOTE: calling this function on optimizer which is NOT running    |
//|       will have no effect.                                       |
//| NOTE: multiple calls to this function are possible. First call is|
//|       counted, subsequent calls are silently ignored.            |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICRequestTermination(CMinBLEICState &State)
  {
   State.m_userterminationneeded=true;
  }
//+------------------------------------------------------------------+
//| This subroutine finalizes internal structures after emergency    |
//| termination from State.LSStart report (see comments on           |
//| MinBLEICState for more information).                             |
//| INPUT PARAMETERS:                                                |
//|   State    -  structure after exit from LSStart report           |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICEmergencyTermination(CMinBLEICState &State)
  {
   CSActiveSets::SASStopOptimization(State.m_sas);
  }
//+------------------------------------------------------------------+
//| Clears request fileds (to be sure that we don't forget to clear  |
//| something)                                                       |
//+------------------------------------------------------------------+
void CMinBLEIC::ClearRequestFields(CMinBLEICState &State)
  {
//--- change values
   State.m_needf=false;
   State.m_needfg=false;
   State.m_xupdated=false;
   State.m_lsstart=false;
  }
//+------------------------------------------------------------------+
//| Internal initialization subroutine                               |
//+------------------------------------------------------------------+
void CMinBLEIC::MinBLEICInitInternal(const int n,CRowDouble &x,
                                     const double diffstep,
                                     CMinBLEICState &State)
  {
//--- create matrix
   CMatrixDouble c;
//--- create array
   CRowInt ct;
//--- initialization
   State.m_teststep=0;
   State.m_smoothnessguardlevel=0;
   COptServ::SmoothnessMonitorInit(State.m_smonitor,State.m_s,0,0,false);
   State.m_nmain=n;
   State.m_diffstep=diffstep;
   CSActiveSets::SASInit(n,State.m_sas);
//--- allocation
   ArrayResize(State.m_HasBndL,n);
   ArrayResize(State.m_HasBndU,n);
   State.m_bndl=vector<double>::Full(n,AL_NEGINF);
   State.m_bndu=vector<double>::Full(n,AL_POSINF);
   State.m_xstart.Resize(n);
   State.m_cgc.Resize(n);
   State.m_ugc.Resize(n);
   State.m_xn.Resize(n);
   State.m_cgn.Resize(n);
   State.m_ugn.Resize(n);
   State.m_xp.Resize(n);
   State.m_d.Resize(n);
   State.m_s=vector<double>::Ones(n);
   State.m_invs=vector<double>::Ones(n);
   State.m_lastscaleused=vector<double>::Ones(n);
   State.m_x.Resize(n);
   State.m_g.Resize(n);
   State.m_work.Resize(n);
   ArrayInitialize(State.m_HasBndL,false);
   ArrayInitialize(State.m_HasBndU,false);
//--- function call
   MinBLEICSetLC(State,c,ct,0);
//--- function call
   MinBLEICSetCond(State,0.0,0.0,0.0,0);
//--- function call
   MinBLEICSetXRep(State,false);
//--- function call
   MinBLEICSetDRep(State,false);
//--- function call
   MinBLEICSetStpMax(State,0.0);
//--- function call
   MinBLEICSetPrecDefault(State);
//--- function call
   MinBLEICRestartFrom(State,x);
  }
//+------------------------------------------------------------------+
//| NOTES:                                                           |
//| 1. This function has two different implementations: one which    |
//|    uses exact (analytical) user-supplied gradient, and one which |
//|    uses function value only and numerically differentiates       |
//|    function in order to obtain gradient.                         |
//|    Depending on the specific function used to create optimizer   |
//|    object (either MinBLEICCreate() for analytical gradient or    |
//|    MinBLEICCreateF() for numerical differentiation) you should   |
//|    choose appropriate variant of MinBLEICOptimize() - one which  |
//|    accepts function AND gradient or one which accepts function   |
//|    ONLY.                                                         |
//|    Be careful to choose variant of MinBLEICOptimize() which      |
//|    corresponds to your optimization scheme! Table below lists    |
//|    different combinations of callback (function/gradient) passed |
//|    to MinBLEICOptimize() and specific function used to create    |
//|    optimizer.                                                    |
//|                      |         USER PASSED TO MinBLEICOptimize() |
//|    CREATED WITH      |  function only   |  function and gradient |
//|    ------------------------------------------------------------  |
//|    MinBLEICCreateF() |     work                FAIL              |
//|    MinBLEICCreate()  |     FAIL                work              |
//|    Here "FAIL" denotes inappropriate combinations of optimizer   |
//|    creation function and MinBLEICOptimize() version. Attemps to  |
//|    use such combination (for example, to create optimizer with   |
//|    MinBLEICCreateF() and to pass gradient information to         |
//|    MinCGOptimize()) will lead to exception being thrown. Either  |
//|    you did not pass gradient when it WAS needed or you passed    |
//|    gradient when it was NOT needed.                              |
//+------------------------------------------------------------------+
bool CMinBLEIC::MinBLEICIteration(CMinBLEICState &State)
  {
//--- create variables
   int    n=0;
   int    m=0;
   int    i=0;
   int    j=0;
   double v=0;
   double vv=0;
   double v0=0;
   bool   b=false;
   int    mcinfo=0;
   int    actstatus=0;
   int    itidx=0;
   double penalty=0;
   double ginit=0;
   double gdecay=0;
   int    i_=0;
   int    label=-1;
//--- This code initializes locals by:
//--- * random values determined during code
//---   generation - on first subroutine call
//--- * values from previous call - on subsequent calls
   if(State.m_rstate.stage>=0)
     {
      n=State.m_rstate.ia[0];
      m=State.m_rstate.ia[1];
      i=State.m_rstate.ia[2];
      j=State.m_rstate.ia[3];
      mcinfo=State.m_rstate.ia[4];
      actstatus=State.m_rstate.ia[5];
      itidx=State.m_rstate.ia[6];
      b=State.m_rstate.ba[0];
      v=State.m_rstate.ra[0];
      vv=State.m_rstate.ra[1];
      v0=State.m_rstate.ra[2];
      penalty=State.m_rstate.ra[3];
      ginit=State.m_rstate.ra[4];
      gdecay=State.m_rstate.ra[5];
     }
   else
     {
      n=359;
      m=-58;
      i=-919;
      j=-909;
      mcinfo=81;
      actstatus=255;
      itidx=74;
      b=false;
      v=809;
      vv=205;
      v0=-838;
      penalty=939;
      ginit=-526;
      gdecay=763;
     }
   switch(State.m_rstate.stage)
     {
      case 0:
         label=0;
         break;
      case 1:
         label=1;
         break;
      case 2:
         label=2;
         break;
      case 3:
         label=3;
         break;
      case 4:
         label=4;
         break;
      case 5:
         label=5;
         break;
      case 6:
         label=6;
         break;
      case 7:
         label=7;
         break;
      case 8:
         label=8;
         break;
      case 9:
         label=9;
         break;
      case 10:
         label=10;
         break;
      case 11:
         label=11;
         break;
      case 12:
         label=12;
         break;
      case 13:
         label=13;
         break;
      case 14:
         label=14;
         break;
      case 15:
         label=15;
         break;
      case 16:
         label=16;
         break;
      case 17:
         label=17;
         break;
      case 18:
         label=18;
         break;
      case 19:
         label=19;
         break;
      case 20:
         label=20;
         break;
      case 21:
         label=21;
         break;
      default:
         //--- Routine body
         //--- Algorithm parameters:
         //--- * M          number of L-BFGS corrections.
         //---              This coefficient remains fixed during iterations.
         //--- * GDecay     desired decrease of constrained gradient during L-BFGS iterations.
         //---              This coefficient is decreased after each L-BFGS round until
         //---              it reaches minimum decay.
         m=MathMin(5,State.m_nmain);
         gdecay=m_initialdecay;
         //--- Init
         n=State.m_nmain;
         State.m_steepestdescentstep=false;
         State.m_userterminationneeded=false;
         State.m_repterminationtype=0;
         State.m_repinneriterationscount=0;
         State.m_repouteriterationscount=0;
         State.m_repnfev=0;
         State.m_repvaridx=-1;
         State.m_repdebugeqerr=0.0;
         State.m_repdebugfs=AL_NaN;
         State.m_repdebugff=AL_NaN;
         State.m_repdebugdx=AL_NaN;
         if(State.m_stpmax!=0.0 && State.m_prectype!=0)
           {
            State.m_repterminationtype=-10;
            return(false);
           }
         CApServ::RMatrixSetLengthAtLeast(State.m_bufyk,m+1,n);
         CApServ::RMatrixSetLengthAtLeast(State.m_bufsk,m+1,n);
         CApServ::RVectorSetLengthAtLeast(State.m_bufrho,m);
         CApServ::RVectorSetLengthAtLeast(State.m_buftheta,m);
         CApServ::RVectorSetLengthAtLeast(State.m_tmp0,n);
         COptServ::SmoothnessMonitorInit(State.m_smonitor,State.m_s,n,1,State.m_smoothnessguardlevel>0);
         State.m_lastscaleused=State.m_s;
         State.m_invs=vector<double>::Ones(n)/State.m_s.ToVector();
         //---  Check analytic derivative
         ClearRequestFields(State);
         if(!(State.m_diffstep==0.0 && State.m_teststep>0.0))
           {
            label=22;
            break;
           }
         label=24;
         break;
     }
//--- main loop
   while(label>=0)
      switch(label)
        {
         case 24:
            if(!COptServ::SmoothnessMonitorCheckGradientATX0(State.m_smonitor,State.m_xstart,State.m_s,State.m_bndl,State.m_bndu,true,State.m_teststep))
              {
               label=25;
               break;
              }
            State.m_x=State.m_smonitor.m_x;
            State.m_needfg=true;
            State.m_rstate.stage=0;
            label=-1;
            break;
         case 0:
            State.m_needfg=false;
            State.m_smonitor.m_fi.Set(0,State.m_f);
            State.m_smonitor.m_j.Row(0,State.m_g);
            label=24;
            break;
         case 25:
         case 22:
            //--- Fill TmpPrec with current preconditioner
            switch(State.m_prectype)
              {
               case 2:
                  State.m_tmpprec=State.m_diagh;
                  break;
               case 3:
                  State.m_tmpprec=State.m_s.Pow(-2.0)+0;
                  break;
               default:
                  State.m_tmpprec=vector<double>::Ones(n);
              }
            CSActiveSets::SASSetPrecDiag(State.m_sas,State.m_tmpprec);
            //--- Start optimization
            if(!CSActiveSets::SASStartOptimization(State.m_sas,State.m_xstart))
              {
               State.m_repterminationtype=-3;
               return(false);
              }
            //--- Main cycle of BLEIC-PG algorithm
            State.m_repterminationtype=0;
            State.m_lastgoodstep=0;
            State.m_lastscaledgoodstep=0;
            State.m_maxscaledgrad=0;
            State.m_nonmonotoniccnt=(int)MathRound(1.5*(n+State.m_nic))+5;
            State.m_x=State.m_sas.m_xc;
            ClearRequestFields(State);
            if(State.m_diffstep!=0.0)
              {
               label=26;
               break;
              }
            State.m_needfg=true;
            State.m_rstate.stage=1;
            label=-1;
            break;
         case 1:
            State.m_needfg=false;
            label=27;
            break;
         case 26:
            State.m_needf=true;
            State.m_rstate.stage=2;
            label=-1;
            break;
         case 2:
            State.m_needf=false;
         case 27:
            State.m_fc=State.m_f;
            COptServ::TrimPrepare(State.m_f,State.m_trimthreshold);
            State.m_repnfev++;
            if(!State.m_xrep)
              {
               label=28;
               break;
              }
            //--- Report current point
            State.m_x=State.m_sas.m_xc;
            State.m_f=State.m_fc;
            State.m_xupdated=true;
            State.m_rstate.stage=3;
            label=-1;
            break;
         case 3:
            State.m_xupdated=false;
         case 28:
            if(State.m_userterminationneeded)
              {
               //--- User requested termination
               CSActiveSets::SASStopOptimization(State.m_sas);
               State.m_repterminationtype=8;
               return(false);
              }
         case 30:
            //--- Preparations
            //--- (a) calculate unconstrained gradient
            //--- (b) determine initial active set
            //--- (c) update MaxScaledGrad
            //--- (d) check F/G for NAN/INF, abnormally terminate algorithm if needed
            State.m_x=State.m_sas.m_xc;
            ClearRequestFields(State);
            if(State.m_diffstep!=0.0)
              {
               label=32;
               break;
              }
            //--- Analytic gradient
            State.m_needfg=true;
            State.m_rstate.stage=4;
            label=-1;
            break;
         case 4:
            State.m_needfg=false;
            label=33;
            break;
         case 32:
            //--- Numerical differentiation
            State.m_needf=true;
            State.m_rstate.stage=5;
            label=-1;
            break;
         case 5:
            State.m_fbase=State.m_f;
            i=0;
         case 34:
            if(i>n-1)
              {
               label=36;
               break;
              }
            v=State.m_x[i];
            b=false;
            if(State.m_HasBndL[i])
               b=b || (v-State.m_diffstep*State.m_s[i])<State.m_bndl[i];
            if(State.m_HasBndU[i])
               b=b || (v+State.m_diffstep*State.m_s[i])>State.m_bndu[i];
            if(b)
              {
               label=37;
               break;
              }
            State.m_x.Set(i,v-State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=6;
            label=-1;
            break;
         case 6:
            State.m_fm2=State.m_f;
            State.m_x.Set(i,v-0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=7;
            label=-1;
            break;
         case 7:
            State.m_fm1=State.m_f;
            State.m_x.Set(i,v+0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=8;
            label=-1;
            break;
         case 8:
            State.m_fp1=State.m_f;
            State.m_x.Set(i,v+State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=9;
            label=-1;
            break;
         case 9:
            State.m_fp2=State.m_f;
            State.m_g.Set(i,(8*(State.m_fp1-State.m_fm1)-(State.m_fp2-State.m_fm2))/(6*State.m_diffstep*State.m_s[i]));
            label=38;
            break;
         case 37:
            State.m_xm1=v-State.m_diffstep*State.m_s[i];
            State.m_xp1=v+State.m_diffstep*State.m_s[i];
            if(State.m_HasBndL[i] && State.m_xm1<State.m_bndl[i])
               State.m_xm1=State.m_bndl[i];
            if(State.m_HasBndU[i] && State.m_xp1>State.m_bndu[i])
               State.m_xp1=State.m_bndu[i];
            State.m_x.Set(i,State.m_xm1);
            State.m_rstate.stage=10;
            label=-1;
            break;
         case 10:
            State.m_fm1=State.m_f;
            State.m_x.Set(i,State.m_xp1);
            State.m_rstate.stage=11;
            label=-1;
            break;
         case 11:
            State.m_fp1=State.m_f;
            if(State.m_xm1!=State.m_xp1)
               State.m_g.Set(i,(State.m_fp1-State.m_fm1)/(State.m_xp1-State.m_xm1));
            else
               State.m_g.Set(i,0);
         case 38:
            State.m_x.Set(i,v);
            i++;
            label=34;
            break;
         case 36:
            State.m_f=State.m_fbase;
            State.m_needf=false;
         case 33:
            State.m_fc=State.m_f;
            State.m_ugc=State.m_g;
            State.m_cgc=State.m_g;
            CSActiveSets::SASReactivateConstraintsPrec(State.m_sas,State.m_ugc);
            CSActiveSets::SASConstrainedDirection(State.m_sas,State.m_cgc);
            ginit=MathPow(State.m_cgc*State.m_s+0,2).Sum();
            ginit=MathSqrt(ginit);
            State.m_maxscaledgrad=MathMax(State.m_maxscaledgrad,ginit);
            if(!MathIsValidNumber(ginit) || !MathIsValidNumber(State.m_fc))
              {
               //--- Abnormal termination - infinities in function/gradient
               CSActiveSets::SASStopOptimization(State.m_sas);
               State.m_repterminationtype=-8;
               return(false);
              }
            if(State.m_userterminationneeded)
              {
               //--- User requested termination
               CSActiveSets::SASStopOptimization(State.m_sas);
               State.m_repterminationtype=8;
               return(false);
              }
            //--- LBFGS stage:
            //--- * during LBFGS iterations we activate new constraints, but never
            //---   deactivate already active ones.
            //--- * we perform at most N iterations of LBFGS before re-evaluating
            //---   active set and restarting LBFGS.
            //--- * first iteration of LBFGS is a special - it is performed with
            //---   minimum set of active constraints, algorithm termination can
            //---   be performed only at this State.m_ We call this iteration
            //---  "steepest descent step".
            //--- About termination:
            //--- * LBFGS iterations can be terminated because of two reasons:
            //---  *"termination" - non-zero termination code in RepTerminationType,
            //---     which means that optimization is done
            //---  *"restart" - zero RepTerminationType, which means that we
            //---     have to re-evaluate active set and resume LBFGS stage.
            //---*one more option is "refresh" - to continue LBFGS iterations,
            //---   but with all BFGS updates (Sk/Yk pairs) being dropped;
            //---   it happens after changes in active set
            State.m_bufsize=0;
            State.m_steepestdescentstep=true;
            itidx=-1;
         case 39:
            if(itidx>=n-1)
              {
               label=40;
               break;
              }
            //--- Increment iterations counter
            //--- NOTE: we have strong reasons to use such complex scheme
            //---       instead of just for() loop - this counter may be
            //---       decreased at some occasions to perform "restart"
            //---       of an iteration.
            itidx++;
            //--- At the beginning of each iteration:
            //--- * SAS.XC stores current point
            //--- * FC stores current function value
            //--- * UGC stores current unconstrained gradient
            //--- * CGC stores current constrained gradient
            //--- * D stores constrained step direction (calculated at this block)
            //--- Check gradient-based stopping criteria
            //--- This stopping condition is tested only for step which is the
            //--- first step of LBFGS (subsequent steps may accumulate active
            //--- constraints thus they should NOT be used for stopping - gradient
            //--- may be small when constrained, but these constraints may be
            //--- deactivated by the subsequent steps)
            if(State.m_steepestdescentstep  && 
               CSActiveSets::SASScaledConstrainedNorm(State.m_sas,State.m_ugc)<=State.m_epsg)
              {
               //--- Gradient is small enough.
               //--- Optimization is terminated
               State.m_repterminationtype=4;
               label=40;
               break;
              }
            //--- 1. Calculate search direction D according to L-BFGS algorithm
            //---    using constrained preconditioner to perform inner multiplication.
            //--- 2. Evaluate scaled length of direction D; restart LBFGS if D is zero
            //---    (it may be possible that we found minimum, but it is also possible
            //---    that some constraints need deactivation)
            //--- 3. If D is non-zero, try to use previous scaled step length as initial estimate for new step.
            State.m_work=State.m_cgc;
            for(i=State.m_bufsize-1; i>=0; i--)
              {
               v=CAblasF::RDotVR(n,State.m_work,State.m_bufsk,i);
               State.m_buftheta.Set(i,v);
               vv=v*State.m_bufrho[i];
               for(i_=0; i_<n; i_++)
                  State.m_work.Add(i_,- State.m_bufyk.Get(i,i_)*vv);
              }
            CSActiveSets::SASConstrainedDirectionPrec(State.m_sas,State.m_work);
            for(i=0; i<State.m_bufsize; i++)
              {
               v=CAblasF::RDotVR(n,State.m_work,State.m_bufyk,i);
               vv=State.m_bufrho[i]*(-v+State.m_buftheta[i]);
               for(i_=0; i_<n; i_++)
                  State.m_work.Add(i_,State.m_bufsk.Get(i,i_)*vv);
              }
            CSActiveSets::SASConstrainedDirection(State.m_sas,State.m_work);
            State.m_d=State.m_work.ToVector()*(-1.0);
            v=MathPow(State.m_d/State.m_s+0,2.0).Sum();
            v=MathSqrt(v);
            if(v==0.0)
              {
               //--- Search direction is zero.
               //--- If we perform "steepest descent step", algorithm is terminated.
               //--- Otherwise we just restart LBFGS.
               if(State.m_steepestdescentstep)
                  State.m_repterminationtype=4;
               label=40;
               break;
              }
            //--- check
            if(!CAp::Assert(v>0.0,__FUNCTION__+": internal error"))
               return(false);
            if(State.m_lastscaledgoodstep>0.0 && v>0.0)
               State.m_stp=State.m_lastscaledgoodstep/v;
            else
               State.m_stp=1.0/v;
            //--- Calculate bound on step length.
            //--- Step direction is stored
            CSActiveSets::SASExploreDirection(State.m_sas,State.m_d,State.m_curstpmax,State.m_cidx,State.m_cval);
            State.m_activationstep=State.m_curstpmax;
            if(State.m_cidx>=0 && State.m_activationstep==0.0)
              {
               //--- We are exactly at the boundary, immediate activation
               //--- of constraint is required. LBFGS stage is continued
               //--- with "refreshed" model.
               //--- ! IMPORTANT: we do not clear SteepestDescent flag here,
               //--- !            it is very important for correct stopping
               //--- !            of algorithm.
               //--- ! IMPORTANT: we decrease iteration counter in order to
               //---              preserve computational budget for iterations.
               CSActiveSets::SASImmediateActivation(State.m_sas,State.m_cidx,State.m_cval);
               State.m_bufsize=0;
               itidx=itidx-1;
               label=39;
               break;
              }
            if(State.m_stpmax>0.0)
              {
               v=CAblasF::RDotV2(n,State.m_d);
               v=MathSqrt(v);
               if(v>0.0)
                  State.m_curstpmax=MathMin(State.m_curstpmax,State.m_stpmax/v);
              }
            //--- Report beginning of line search (if requested by caller).
            //--- See description of the MinBLEICState for more information
            //--- about fields accessible to caller.
            //--- Caller may do following:
            //--- * change State.Stp and load better initial estimate of
            //---   the step length.
            //--- Caller may not terminate algorithm.
            if(!State.m_drep)
              {
               label=41;
               break;
              }
            ClearRequestFields(State);
            State.m_lsstart=true;
            State.m_boundedstep=State.m_cidx>=0;
            State.m_x=State.m_sas.m_xc;
            State.m_rstate.stage=12;
            label=-1;
            break;
         case 12:
            State.m_lsstart=false;
         case 41:
            //--- Minimize F(x+alpha*d)
            State.m_xn=State.m_sas.m_xc;
            State.m_cgn=State.m_cgc;
            State.m_ugn=State.m_ugc;
            State.m_fn=State.m_fc;
            State.m_mcstage=0;
            COptServ::SmoothnessMonitorStartLineSearch1u(State.m_smonitor,State.m_s,State.m_invs,State.m_xn,State.m_fn,State.m_ugn);
            CLinMin::MCSrch(n,State.m_xn,State.m_fn,State.m_ugn,State.m_d,State.m_stp,State.m_curstpmax,m_gtol,mcinfo,State.m_nfev,State.m_work,State.m_lstate,State.m_mcstage);
         case 43:
            if(State.m_mcstage==0)
              {
               label=44;
               break;
              }
            //--- Perform correction (constraints are enforced)
            //--- Copy XN to X
            CSActiveSets::SASCorrection(State.m_sas,State.m_xn,penalty);
            State.m_x=State.m_xn;
            //--- Gradient, either user-provided or numerical differentiation
            ClearRequestFields(State);
            if(State.m_diffstep!=0.0)
              {
               label=45;
               break;
              }
            //--- Analytic gradient
            State.m_needfg=true;
            State.m_rstate.stage=13;
            label=-1;
            break;
         case 13:
            State.m_needfg=false;
            State.m_repnfev++;
            label=46;
            break;
         case 45:
            //--- Numerical differentiation
            State.m_needf=true;
            State.m_rstate.stage=14;
            label=-1;
            break;
         case 14:
            State.m_fbase=State.m_f;
            i=0;
         case 47:
            if(i>n-1)
              {
               label=49;
               break;
              }
            v=State.m_x[i];
            b=false;
            if(State.m_HasBndL[i])
               b=b || (v-State.m_diffstep*State.m_s[i])<State.m_bndl[i];
            if(State.m_HasBndU[i])
               b=b || (v+State.m_diffstep*State.m_s[i])>State.m_bndu[i];
            if(b)
              {
               label=50;
               break;
              }
            State.m_x.Set(i,v-State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=15;
            label=-1;
            break;
         case 15:
            State.m_fm2=State.m_f;
            State.m_x.Set(i,v-0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=16;
            label=-1;
            break;
         case 16:
            State.m_fm1=State.m_f;
            State.m_x.Set(i,v+0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=17;
            label=-1;
            break;
         case 17:
            State.m_fp1=State.m_f;
            State.m_x.Set(i,v+State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=18;
            label=-1;
            break;
         case 18:
            State.m_fp2=State.m_f;
            State.m_g.Set(i,(8*(State.m_fp1-State.m_fm1)-(State.m_fp2-State.m_fm2))/(6*State.m_diffstep*State.m_s[i]));
            State.m_repnfev+=4;
            label=51;
            break;
         case 50:
            State.m_xm1=v-State.m_diffstep*State.m_s[i];
            State.m_xp1=v+State.m_diffstep*State.m_s[i];
            if(State.m_HasBndL[i] && State.m_xm1<State.m_bndl[i])
               State.m_xm1=State.m_bndl[i];
            if(State.m_HasBndU[i] && State.m_xp1>State.m_bndu[i])
               State.m_xp1=State.m_bndu[i];
            State.m_x.Set(i,State.m_xm1);
            State.m_rstate.stage=19;
            label=-1;
            break;
         case 19:
            State.m_fm1=State.m_f;
            State.m_x.Set(i,State.m_xp1);
            State.m_rstate.stage=20;
            label=-1;
            break;
         case 20:
            State.m_fp1=State.m_f;
            if(State.m_xm1!=State.m_xp1)
               State.m_g.Set(i,(State.m_fp1-State.m_fm1)/(State.m_xp1-State.m_xm1));
            else
               State.m_g.Set(i,0);
            State.m_repnfev+=2;
         case 51:
            State.m_x.Set(i,v);
            i++;
            label=47;
            break;
         case 49:
            State.m_f=State.m_fbase;
            State.m_needf=false;
         case 46:
            //--- Back to MCSRCH
            //--- NOTE: penalty term from correction is added to FN in order
            //---       to penalize increase in infeasibility.
            COptServ::SmoothnessMonitorEnqueuePoint1u(State.m_smonitor,State.m_s,State.m_invs,State.m_d,State.m_stp,State.m_x,State.m_f,State.m_g);
            State.m_fn=State.m_f+m_penaltyfactor*State.m_maxscaledgrad*penalty;
            State.m_cgn=State.m_g;
            State.m_ugn=State.m_g;
            CSActiveSets::SASConstrainedDirection(State.m_sas,State.m_cgn);
            COptServ::TrimFunction(State.m_fn,State.m_cgn,n,State.m_trimthreshold);
            CLinMin::MCSrch(n,State.m_xn,State.m_fn,State.m_ugn,State.m_d,State.m_stp,State.m_curstpmax,m_gtol,mcinfo,State.m_nfev,State.m_work,State.m_lstate,State.m_mcstage);
            label=43;
            break;
         case 44:
            State.m_bufsk.Row(State.m_bufsize,State.m_xn-State.m_sas.m_xc+0);
            State.m_bufyk.Row(State.m_bufsize,State.m_cgn-State.m_cgc+0);
            COptServ::SmoothnessMonitorFinalizeLineSearch(State.m_smonitor);
            //--- Check for presence of NAN/INF in function/gradient
            v=State.m_fn;
            for(i=0; i<n; i++)
               v=0.1*v+State.m_ugn[i];
            if(!MathIsValidNumber(v))
              {
               //--- Abnormal termination - infinities in function/gradient
               State.m_repterminationtype=-8;
               label=40;
               break;
              }
            //--- Handle possible failure of the line search or request for termination
            if(mcinfo!=1 && mcinfo!=5)
              {
               //--- We can not find step which decreases function value. We have
               //--- two possibilities:
               //--- (a) numerical properties of the function do not allow us to
               //---     find good step.
               //--- (b) we are close to activation of some constraint, and it is
               //---     so close that step which activates it leads to change in
               //---     target function which is smaller than numerical noise.
               //--- Optimization algorithm must be able to handle case (b), because
               //--- inability to handle it will cause failure when algorithm
               //--- started very close to boundary of the feasible area.
               //--- In order to correctly handle such cases we allow limited amount
               //--- of small steps which increase function value.
               v=MathPow(State.m_d/State.m_s*State.m_curstpmax,2.0).Sum();
               v=MathSqrt(v);
               b=false;
               if(State.m_cidx>=0 && v<=m_maxnonmonotoniclen && State.m_nonmonotoniccnt>0)
                 {
                  //--- We try to enforce non-monotonic step:
                  //--- * Stp    := CurStpMax
                  //--- * MCINFO := 5
                  //--- * XN     := XC+CurStpMax*D
                  //--- * non-monotonic counter is decreased
                  //--- NOTE: UGN/CGN are not updated because step is so short that we assume that
                  //---       GN is approximately equal to GC.
                  //--- NOTE: prior to enforcing such step we check that it does not increase infeasibility
                  //---       of constraints beyond tolerable level
                  v=State.m_curstpmax;
                  State.m_tmp0=State.m_sas.m_xc.ToVector()+State.m_d*v;
                  State.m_stp=State.m_curstpmax;
                  mcinfo=5;
                  State.m_xn=State.m_tmp0;
                  State.m_nonmonotoniccnt--;
                  b=true;
                 }
               if(!b)
                 {
                  //--- Numerical properties of the function do not allow
                  //--- us to solve problem. Here we have two possibilities:
                  //---*if it is "steepest descent" step, we can terminate
                  //---   algorithm because we are close to minimum
                  //---*if it is NOT "steepest descent" step, we should restart
                  //---   LBFGS iterations.
                  if(State.m_steepestdescentstep)
                    {
                     //--- Algorithm is terminated
                     State.m_repterminationtype=7;
                     label=40;
                     break;
                    }
                  else
                    {
                     //--- Re-evaluate active set and restart LBFGS
                     label=40;
                     break;
                    }
                 }
              }
            if(State.m_userterminationneeded)
              {
               label=40;
               break;
              }
            //--- Current point is updated:
            //--- * move XC/FC/GC to XP/FP/GP
            //--- * change current point remembered by SAS structure
            //--- * move XN/FN/GN to XC/FC/GC
            //--- * report current point and update iterations counter
            //--- * if MCINFO=1, push new pair SK/YK to LBFGS buffer
            State.m_fp=State.m_fc;
            State.m_xp=State.m_sas.m_xc;
            State.m_fc=State.m_fn;
            State.m_cgc=State.m_cgn;
            State.m_ugc=State.m_ugn;
            actstatus=CSActiveSets::SASMoveTo(State.m_sas,State.m_xn,State.m_cidx>=0 && State.m_stp>=State.m_activationstep,State.m_cidx,State.m_cval);
            if(!State.m_xrep)
              {
               label=52;
               break;
              }
            State.m_x=State.m_sas.m_xc;
            ClearRequestFields(State);
            State.m_xupdated=true;
            State.m_rstate.stage=21;
            label=-1;
            break;
         case 21:
            State.m_xupdated=false;
         case 52:
            State.m_repinneriterationscount++;
            if(mcinfo==1)
              {
               //--- Accept new LBFGS update given by Sk,Yk
               if(State.m_bufsize==m)
                 {
                  //--- Buffer is full, shift contents by one row
                  for(i=0; i<State.m_bufsize; i++)
                    {
                     State.m_bufsk.Row(i,State.m_bufsk,i+1);
                     State.m_bufyk.Row(i,State.m_bufyk,i+1);
                    }
                  for(i=0; i<State.m_bufsize-1; i++)
                    {
                     State.m_bufrho.Set(i,State.m_bufrho[i+1]);
                     State.m_buftheta.Set(i,State.m_buftheta[i+1]);
                    }
                 }
               else
                 {
                  //--- Buffer is not full, increase buffer size by 1
                  State.m_bufsize++;
                 }
               v=CAblasF::RDotRR(n,State.m_bufyk,State.m_bufsize-1,State.m_bufsk,State.m_bufsize-1);
               vv=CAblasF::RDotRR(n,State.m_bufyk,State.m_bufsize-1,State.m_bufyk,State.m_bufsize-1);
               if(v==0.0 || vv==0.0)
                 {
                  //--- Strange internal error in LBFGS - either YK=0
                  //--- (which should not have been) or (SK,YK)=0 (again,
                  //--- unexpected). It should not take place because
                  //--- MCINFO=1, which signals "good" step. But just
                  //--- to be sure we have special branch of code which
                  //--- restarts LBFGS
                  label=40;
                  break;
                 }
               State.m_bufrho.Set(State.m_bufsize-1,1/v);
               //--- check
               if(!CAp::Assert(State.m_bufsize<=m,__FUNCTION__+": internal error"))
                  return(false);
               //--- Update length of the good step
               vector<double> temp=State.m_sas.m_xc-State.m_xp;
               v=MathPow(temp/State.m_s.ToVector(),2.0).Sum();
               vv=temp.Dot(temp);
               State.m_lastgoodstep=MathSqrt(vv);
               UpdateEstimateOfGoodStep(State.m_lastscaledgoodstep,MathSqrt(v));
              }
            //--- Check stopping criteria
            //--- Step size and function-based stopping criteria are tested only
            //--- for step which satisfies Wolfe conditions and is the first step of
            //--- LBFGS (subsequent steps may accumulate active constraints thus
            //--- they should NOT be used for stopping; step size or function change
            //--- may be small when constrained, but these constraints may be
            //--- deactivated by the subsequent steps).
            //--- MaxIts-based stopping condition is checked for all kinds of steps.
            if(mcinfo==1 && State.m_steepestdescentstep)
              {
               //--- Step is small enough
               v=MathPow((State.m_sas.m_xc-State.m_xp)/State.m_s.ToVector(),2.0).Sum();
               v=MathSqrt(v);
               if(v<=State.m_epsx)
                 {
                  State.m_repterminationtype=2;
                  label=40;
                  break;
                 }
               //--- Function change is small enough
               if(MathAbs(State.m_fp-State.m_fc)<=(State.m_epsf*MathMax(MathAbs(State.m_fc),MathMax(MathAbs(State.m_fp),1.0))))
                 {
                  State.m_repterminationtype=1;
                  label=40;
                  break;
                 }
              }
            if(State.m_maxits>0 && State.m_repinneriterationscount>=State.m_maxits)
              {
               State.m_repterminationtype=5;
               label=40;
               break;
              }
            //--- Clear "steepest descent" flag.
            State.m_steepestdescentstep=false;
            //--- Smooth reset (LBFGS memory model is refreshed) or hard restart:
            //--- * LBFGS model is refreshed, if line search was performed with activation of constraints
            //--- * algorithm is restarted if scaled gradient decreased below GDecay
            if(actstatus>=0)
              {
               State.m_bufsize=0;
               label=39;
               break;
              }
            v=MathPow(State.m_cgc*State.m_s+0,2.0).Sum();
            if(MathSqrt(v)<(gdecay*ginit))
              {
               label=40;
               break;
              }
            label=39;
            break;
         case 40:
            if(State.m_userterminationneeded)
              {
               //--- User requested termination
               State.m_repterminationtype=8;
               label=31;
               break;
              }
            if(State.m_repterminationtype!=0)
              {
               //--- Algorithm terminated
               label=31;
               break;
              }
            //--- Decrease decay coefficient. Subsequent L-BFGS stages will
            //--- have more stringent stopping criteria.
            gdecay=MathMax(gdecay*m_decaycorrection,m_mindecay);
            label=30;
            break;
         case 31:
            CSActiveSets::SASStopOptimization(State.m_sas);
            State.m_repouteriterationscount=1;
            return(false);
        }
//-- Saving State
   State.m_rstate.ba[0]=b;
   State.m_rstate.ia.Set(0,n);
   State.m_rstate.ia.Set(1,m);
   State.m_rstate.ia.Set(2,i);
   State.m_rstate.ia.Set(3,j);
   State.m_rstate.ia.Set(4,mcinfo);
   State.m_rstate.ia.Set(5,actstatus);
   State.m_rstate.ia.Set(6,itidx);
   State.m_rstate.ra.Set(0,v);
   State.m_rstate.ra.Set(1,vv);
   State.m_rstate.ra.Set(2,v0);
   State.m_rstate.ra.Set(3,penalty);
   State.m_rstate.ra.Set(4,ginit);
   State.m_rstate.ra.Set(5,gdecay);
//--- return result
   return(true);
  }
//+------------------------------------------------------------------+
//| This subroutine updates estimate of the good step length given:  |
//|   1) previous estimate                                           |
//|   2) new length of the good step                                 |
//| It makes sure that estimate does not change too rapidly - ratio  |
//| of new and old estimates will be at least 0.01, at most 100.0    |
//| In case previous estimate of good step is zero (no estimate),    |
//| new estimate is used unconditionally.                            |
//+------------------------------------------------------------------+
void CMinBLEIC::UpdateEstimateOfGoodStep(double &estimate,
                                         double newstep)
  {
//--- check
   if(estimate==0.0)
     {
      estimate=newstep;
      return;
     }
   if(newstep<(estimate*0.01))
     {
      estimate=estimate*0.01;
      return;
     }
   if(newstep>(estimate*100))
     {
      estimate=estimate*100;
      return;
     }
   estimate=newstep;
  }
//+------------------------------------------------------------------+
//| Auxiliary class for CMinLBFGS                                    |
//+------------------------------------------------------------------+
class CMinLBFGSState
  {
public:
   //--- variables
   int               m_k;
   int               m_m;
   int               m_maxits;
   int               m_mcstage;
   int               m_n;
   int               m_nfev;
   int               m_p;
   int               m_preck;
   int               m_prectype;
   int               m_q;
   int               m_repiterationscount;
   int               m_repnfev;
   int               m_repterminationtype;
   int               m_smoothnessguardlevel;
   double            m_diffstep;
   double            m_epsf;
   double            m_epsg;
   double            m_epsx;
   double            m_f;
   double            m_fbase;
   double            m_fm1;
   double            m_fm2;
   double            m_fold;
   double            m_fp1;
   double            m_fp2;
   double            m_gammak;
   double            m_stp;
   double            m_stpmax;
   double            m_teststep;
   double            m_trimthreshold;
   bool              m_needf;
   bool              m_needfg;
   bool              m_userterminationneeded;
   bool              m_xrep;
   bool              m_xupdated;
   RCommState        m_rstate;
   CSmoothnessMonitor m_smonitor;
   CRowDouble        m_autobuf;
   CRowDouble        m_d;
   CRowDouble        m_diagh;
   CRowDouble        m_g;
   CRowDouble        m_invs;
   CRowDouble        m_lastscaleused;
   CRowDouble        m_precc;
   CRowDouble        m_precd;
   CRowDouble        m_rho;
   CRowDouble        m_s;
   CRowDouble        m_theta;
   CRowDouble        m_work;
   CRowDouble        m_x;
   CRowDouble        m_xbase;
   CRowDouble        m_xp;
   CPrecBufLowRank   m_lowrankbuf;
   CPrecBufLBFGS     m_precbuf;
   CMatrixDouble     m_denseh;
   CMatrixDouble     m_precw;
   CMatrixDouble     m_sk;
   CMatrixDouble     m_yk;
   CLinMinState      m_lstate;
   //--- constructor, destructor
                     CMinLBFGSState(void);
                    ~CMinLBFGSState(void) {}
   //--- copy
   void              Copy(CMinLBFGSState &obj);
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CMinLBFGSState::CMinLBFGSState(void)
  {
   m_k=0;
   m_m=0;
   m_maxits=0;
   m_mcstage=0;
   m_n=0;
   m_nfev=0;
   m_p=0;
   m_preck=0;
   m_prectype=0;
   m_q=0;
   m_repiterationscount=0;
   m_repnfev=0;
   m_repterminationtype=0;
   m_smoothnessguardlevel=0;
   m_diffstep=0;
   m_epsf=0;
   m_epsg=0;
   m_epsx=0;
   m_f=0;
   m_fbase=0;
   m_fm1=0;
   m_fm2=0;
   m_fold=0;
   m_fp1=0;
   m_fp2=0;
   m_gammak=0;
   m_stp=0;
   m_stpmax=0;
   m_teststep=0;
   m_trimthreshold=0;
   m_needf=false;
   m_needfg=false;
   m_userterminationneeded=false;
   m_xrep=false;
   m_xupdated=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CMinLBFGSState::Copy(CMinLBFGSState &obj)
  {
//--- copy variables
   m_k=obj.m_k;
   m_m=obj.m_m;
   m_maxits=obj.m_maxits;
   m_mcstage=obj.m_mcstage;
   m_n=obj.m_n;
   m_nfev=obj.m_nfev;
   m_p=obj.m_p;
   m_preck=obj.m_preck;
   m_prectype=obj.m_prectype;
   m_q=obj.m_q;
   m_repiterationscount=obj.m_repiterationscount;
   m_repnfev=obj.m_repnfev;
   m_repterminationtype=obj.m_repterminationtype;
   m_smoothnessguardlevel=obj.m_smoothnessguardlevel;
   m_diffstep=obj.m_diffstep;
   m_epsf=obj.m_epsf;
   m_epsg=obj.m_epsg;
   m_epsx=obj.m_epsx;
   m_f=obj.m_f;
   m_fbase=obj.m_fbase;
   m_fm1=obj.m_fm1;
   m_fm2=obj.m_fm2;
   m_fold=obj.m_fold;
   m_fp1=obj.m_fp1;
   m_fp2=obj.m_fp2;
   m_gammak=obj.m_gammak;
   m_stp=obj.m_stp;
   m_stpmax=obj.m_stpmax;
   m_teststep=obj.m_teststep;
   m_trimthreshold=obj.m_trimthreshold;
   m_needf=obj.m_needf;
   m_needfg=obj.m_needfg;
   m_userterminationneeded=obj.m_userterminationneeded;
   m_xrep=obj.m_xrep;
   m_xupdated=obj.m_xupdated;
   m_rstate=obj.m_rstate;
   m_smonitor=obj.m_smonitor;
   m_autobuf=obj.m_autobuf;
   m_d=obj.m_d;
   m_diagh=obj.m_diagh;
   m_g=obj.m_g;
   m_invs=obj.m_invs;
   m_lastscaleused=obj.m_lastscaleused;
   m_precc=obj.m_precc;
   m_precd=obj.m_precd;
   m_rho=obj.m_rho;
   m_s=obj.m_s;
   m_theta=obj.m_theta;
   m_work=obj.m_work;
   m_x=obj.m_x;
   m_xbase=obj.m_xbase;
   m_xp=obj.m_xp;
   m_lowrankbuf=obj.m_lowrankbuf;
   m_precbuf=obj.m_precbuf;
   m_denseh=obj.m_denseh;
   m_precw=obj.m_precw;
   m_sk=obj.m_sk;
   m_yk=obj.m_yk;
   m_lstate=obj.m_lstate;
  }
//+------------------------------------------------------------------+
//| This class is a shell for class CMinLBFGSState                   |
//+------------------------------------------------------------------+
class CMinLBFGSStateShell
  {
private:
   CMinLBFGSState    m_innerobj;

public:
   //--- constructors, destructor
                     CMinLBFGSStateShell(void) {}
                     CMinLBFGSStateShell(CMinLBFGSState &obj) { m_innerobj.Copy(obj); }
                    ~CMinLBFGSStateShell(void) {}
   //--- methods
   bool              GetNeedF(void);
   void              SetNeedF(const bool b);
   bool              GetNeedFG(void);
   void              SetNeedFG(const bool b);
   bool              GetXUpdated(void);
   void              SetXUpdated(const bool b);
   double            GetF(void);
   void              SetF(const double d);
   CMinLBFGSState   *GetInnerObj(void);
  };
//+------------------------------------------------------------------+
//| Returns the value of the variable needf                          |
//+------------------------------------------------------------------+
bool CMinLBFGSStateShell::GetNeedF(void)
  {
   return(m_innerobj.m_needf);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable needf                         |
//+------------------------------------------------------------------+
void CMinLBFGSStateShell::SetNeedF(const bool b)
  {
   m_innerobj.m_needf=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable needfg                         |
//+------------------------------------------------------------------+
bool CMinLBFGSStateShell::GetNeedFG(void)
  {
   return(m_innerobj.m_needfg);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable needfg                        |
//+------------------------------------------------------------------+
void CMinLBFGSStateShell::SetNeedFG(const bool b)
  {
   m_innerobj.m_needfg=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable xupdated                       |
//+------------------------------------------------------------------+
bool CMinLBFGSStateShell::GetXUpdated(void)
  {
   return(m_innerobj.m_xupdated);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable xupdated                      |
//+------------------------------------------------------------------+
void CMinLBFGSStateShell::SetXUpdated(const bool b)
  {
   m_innerobj.m_xupdated=b;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable f                              |
//+------------------------------------------------------------------+
double CMinLBFGSStateShell::GetF(void)
  {
   return(m_innerobj.m_f);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable f                             |
//+------------------------------------------------------------------+
void CMinLBFGSStateShell::SetF(const double d)
  {
   m_innerobj.m_f=d;
  }
//+------------------------------------------------------------------+
//| Return object of class                                           |
//+------------------------------------------------------------------+
CMinLBFGSState *CMinLBFGSStateShell::GetInnerObj(void)
  {
   return(GetPointer(m_innerobj));
  }
//+------------------------------------------------------------------+
//| Auxiliary class for CMinLFBFGS                                   |
//+------------------------------------------------------------------+
class CMinLBFGSReport
  {
public:
   //--- variables
   int               m_iterationscount;
   int               m_nfev;
   int               m_terminationtype;
   //--- constructor, destructor
                     CMinLBFGSReport(void) { ZeroMemory(this); }
                    ~CMinLBFGSReport(void) {}
   //--- copy
   void              Copy(CMinLBFGSReport &obj);
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CMinLBFGSReport::Copy(CMinLBFGSReport &obj)
  {
//--- copy variables
   m_iterationscount=obj.m_iterationscount;
   m_nfev=obj.m_nfev;
   m_terminationtype=obj.m_terminationtype;
  }
//+------------------------------------------------------------------+
//| This class is a shell for class CMinLBFGSReport                  |
//+------------------------------------------------------------------+
class CMinLBFGSReportShell
  {
private:
   CMinLBFGSReport   m_innerobj;

public:
   //--- constructors, destructor
                     CMinLBFGSReportShell(void) {}
                     CMinLBFGSReportShell(CMinLBFGSReport &obj) { m_innerobj.Copy(obj); }
                    ~CMinLBFGSReportShell(void) {}
   //--- methods
   int               GetIterationsCount(void);
   void              SetIterationsCount(const int i);
   int               GetNFev(void);
   void              SetNFev(const int i);
   int               GetTerminationType(void);
   void              SetTerminationType(const int i);
   CMinLBFGSReport  *GetInnerObj(void);
  };
//+------------------------------------------------------------------+
//| Returns the value of the variable iterationscount                |
//+------------------------------------------------------------------+
int CMinLBFGSReportShell::GetIterationsCount(void)
  {
   return(m_innerobj.m_iterationscount);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable iterationscount               |
//+------------------------------------------------------------------+
void CMinLBFGSReportShell::SetIterationsCount(const int i)
  {
   m_innerobj.m_iterationscount=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable nfev                           |
//+------------------------------------------------------------------+
int CMinLBFGSReportShell::GetNFev(void)
  {
   return(m_innerobj.m_nfev);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable nfev                          |
//+------------------------------------------------------------------+
void CMinLBFGSReportShell::SetNFev(const int i)
  {
   m_innerobj.m_nfev=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable terminationtype                |
//+------------------------------------------------------------------+
int CMinLBFGSReportShell::GetTerminationType(void)
  {
   return(m_innerobj.m_terminationtype);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable terminationtype               |
//+------------------------------------------------------------------+
void CMinLBFGSReportShell::SetTerminationType(const int i)
  {
   m_innerobj.m_terminationtype=i;
  }
//+------------------------------------------------------------------+
//| Return object of class                                           |
//+------------------------------------------------------------------+
CMinLBFGSReport *CMinLBFGSReportShell::GetInnerObj(void)
  {
   return(GetPointer(m_innerobj));
  }
//+------------------------------------------------------------------+
//| Limited memory BFGS method for large scale optimization          |
//+------------------------------------------------------------------+
class CMinLBFGS
  {
public:
   //--- constant
   static const double m_gtol;
   //--- public methods
   static void       MinLBFGSCreate(const int n,const int m,double &x[],CMinLBFGSState &State);
   static void       MinLBFGSCreate(const int n,const int m,CRowDouble &x,CMinLBFGSState &State);
   static void       MinLBFGSCreateF(const int n,const int m,double &x[],const double diffstep,CMinLBFGSState &State);
   static void       MinLBFGSSetCond(CMinLBFGSState &State,const double epsg,const double epsf,double epsx,const int m_maxits);
   static void       MinLBFGSSetXRep(CMinLBFGSState &State,const bool needxrep);
   static void       MinLBFGSSetStpMax(CMinLBFGSState &State,const double stpmax);
   static void       MinLBFGSSetScale(CMinLBFGSState &State,double &s[]);
   static void       MinLBFGSCreateX(const int n,const int m,double &x[],int flags,const double diffstep,CMinLBFGSState &State);
   static void       MinLBFGSCreateX(const int n,const int m,CRowDouble &x,int flags,const double diffstep,CMinLBFGSState &State);
   static void       MinLBFGSSetPrecDefault(CMinLBFGSState &State);
   static void       MinLBFGSSetPrecCholesky(CMinLBFGSState &State,CMatrixDouble &p,const bool IsUpper);
   static void       MinLBFGSSetPrecDiag(CMinLBFGSState &State,double &d[]);
   static void       MinLBFGSSetPrecScale(CMinLBFGSState &State);
   static void       MinLBFGSSetPrecRankKLBFGSFast(CMinLBFGSState &State,CRowDouble &d,CRowDouble &c,CMatrixDouble &w,int cnt);
   static void       MinLBFGSSetPrecLowRankExact(CMinLBFGSState &State,CRowDouble &d,CRowDouble &c,CMatrixDouble &w,int cnt);
   static void       MinLBFGSResults(CMinLBFGSState &State,double &x[],CMinLBFGSReport &rep);
   static void       MinLBFGSResults(CMinLBFGSState &State,CRowDouble &x,CMinLBFGSReport &rep);
   static void       MinLBFGSResultsBuf(CMinLBFGSState &State,double &x[],CMinLBFGSReport &rep);
   static void       MinLBFGSResultsBuf(CMinLBFGSState &State,CRowDouble &x,CMinLBFGSReport &rep);
   static void       MinLBFGSRestartFrom(CMinLBFGSState &State,double &x[]);
   static void       MinLBFGSRestartFrom(CMinLBFGSState &State,CRowDouble &x);
   static void       MinLBFGSRequestTermination(CMinLBFGSState &State);

   static bool       MinLBFGSIteration(CMinLBFGSState &State);

private:
   static void       ClearRequestFields(CMinLBFGSState &State);
  };
//+------------------------------------------------------------------+
//| Initialize constants                                             |
//+------------------------------------------------------------------+
const double CMinLBFGS::m_gtol=0.4;
//+------------------------------------------------------------------+
//|         LIMITED MEMORY BFGS METHOD FOR LARGE SCALE OPTIMIZATION  |
//| DESCRIPTION:                                                     |
//| The subroutine minimizes function F(x) of N arguments by using a |
//| quasi - Newton method (LBFGS scheme) which is optimized to use a |
//| minimum  amount of memory.                                       |
//| The subroutine generates the approximation of an inverse Hessian |
//| matrix by using information about the last M steps of the        |
//| algorithm (instead of N). It lessens a required amount of memory |
//| from a value of order N^2 to a value of order 2*N*M.             |
//| REQUIREMENTS:                                                    |
//| Algorithm will request following information during its          |
//| operation:                                                       |
//| * function value F and its gradient G (simultaneously) at given  |
//| point X                                                          |
//| USAGE:                                                           |
//| 1. User initializes algorithm State with MinLBFGSCreate() call   |
//| 2. User tunes m_solver parameters with MinLBFGSSetCond()           |
//|    MinLBFGSSetStpMax() and other functions                       |
//| 3. User calls MinLBFGSOptimize() function which takes algorithm  |
//|    State and pointer (delegate, etc.) to callback function which |
//|    calculates F/G.                                               |
//| 4. User calls MinLBFGSResults() to get solution                  |
//| 5. Optionally user may call MinLBFGSRestartFrom() to solve       |
//|    another problem with same N/M but another starting point      |
//|    and/or another function. MinLBFGSRestartFrom() allows to reuse|
//|    already initialized structure.                                |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension. N>0                           |
//|     M       -   number of corrections in the BFGS scheme of      |
//|                 Hessian approximation update. Recommended value: |
//|                 3<=M<=7. The smaller value causes worse          |
//|                 convergence, the bigger will not cause a         |
//|                 considerably better convergence, but will cause  |
//|                 a fall in  the performance. M<=N.                |
//|     X       -   initial solution approximation, array[0..m_n-1].   |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm State           |
//| NOTES:                                                           |
//| 1. you may tune stopping conditions with MinLBFGSSetCond()       |
//|    function                                                      |
//| 2. if target function contains exp() or other fast growing       |
//|    functions, and optimization algorithm makes too large steps   |
//|    which leads to overflow, use MinLBFGSSetStpMax() function to  |
//|    bound algorithm's steps. However, L-BFGS rarely needs such a  |
//|    tuning.                                                       |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSCreate(const int n,const int m,double &x[],
                               CMinLBFGSState &State)
  {
   CRowDouble X=x;
   MinLBFGSCreate(n,m,X,State);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSCreate(const int n,const int m,CRowDouble &x,
                               CMinLBFGSState &State)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1!"))
      return;
//--- check
   if(!CAp::Assert(m>=1,__FUNCTION__+": M<1"))
      return;
//--- check
   if(!CAp::Assert(m<=n,__FUNCTION__+": M>N"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(x)>=n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- function call
   MinLBFGSCreateX(n,m,x,0,0.0,State);
  }
//+------------------------------------------------------------------+
//| The subroutine is finite difference variant of MinLBFGSCreate(). |
//| It uses finite differences in order to differentiate target      |
//| function.                                                        |
//| Description below contains information which is specific to this |
//| function only. We recommend to read comments on MinLBFGSCreate() |
//| in order to get more information about creation of LBFGS         |
//| optimizer.                                                       |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>0:                          |
//|                 * if given, only leading N elements of X are used|
//|                 * if not given, automatically determined from    |
//|                   size of X                                      |
//|     M       -   number of corrections in the BFGS scheme of      |
//|                 Hessian approximation update. Recommended value: |
//|                 3<=M<=7. The smaller value causes worse          |
//|                 convergence, the bigger will not cause a         |
//|                 considerably better convergence, but will cause a|
//|                 fall in  the performance. M<=N.                  |
//|     X       -   starting point, array[0..m_n-1].                   |
//|     DiffStep-   differentiation step, >0                         |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm State           |
//| NOTES:                                                           |
//| 1. algorithm uses 4-point central formula for differentiation.   |
//| 2. differentiation step along I-th axis is equal to DiffStep*S[I]|
//|    where S[] is scaling vector which can be set by               |
//|    MinLBFGSSetScale() call.                                      |
//| 3. we recommend you to use moderate values of differentiation    |
//|    step. Too large step will result in too large truncation      |
//|    errors, while too small step will result in too large         |
//|    numerical errors. 1.0E-6 can be good value to start with.     |
//| 4. Numerical differentiation is very inefficient - one gradient  |
//|    calculation needs 4*N function evaluations. This function will|
//|    work for any N - either small (1...10), moderate (10...100) or|
//|    large (100...). However, performance penalty will be too      |
//|    severe for any N's except for small ones.                     |
//|    We should also say that code which relies on numerical        |
//|    differentiation is less robust and precise. LBFGS needs exact |
//|    gradient values. Imprecise gradient may slow down convergence,|
//|    especially on highly nonlinear problems.                      |
//|    Thus we recommend to use this function for fast prototyping on|
//|    small- dimensional problems only, and to implement analytical |
//|    gradient as soon as possible.                                 |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSCreateF(const int n,const int m,double &x[],
                                const double diffstep,CMinLBFGSState &State)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N too small!"))
      return;
//--- check
   if(!CAp::Assert(m>=1,__FUNCTION__+": M<1"))
      return;
//--- check
   if(!CAp::Assert(m<=n,__FUNCTION__+": M>N"))
      return;
//--- check
   if(!CAp::Assert(CAp::Len(x)>=n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(diffstep),__FUNCTION__+": DiffStep is infinite or NaN!"))
      return;
//--- check
   if(!CAp::Assert(diffstep>0.0,__FUNCTION__+": DiffStep is non-positive!"))
      return;
//--- function call
   MinLBFGSCreateX(n,m,x,0,diffstep,State);
  }
//+------------------------------------------------------------------+
//| This function sets stopping conditions for L-BFGS optimization   |
//| algorithm.                                                       |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     EpsG    -   >=0                                              |
//|                 The subroutine finishes its work if the condition|
//|                 |v|<EpsG is satisfied, where:                    |
//|                 * |.| means Euclidian norm                       |
//|                 * v - scaled gradient vector, v[i]=g[i]*s[i]     |
//|                 * g - gradient                                   |
//|                 * s - scaling coefficients set by                |
//|                       MinLBFGSSetScale()                         |
//|     EpsF    -   >=0                                              |
//|                 The subroutine finishes its work if on k+1-th    |
//|                 iteration the condition |F(k+1)-F(k)| <=         |
//|                 <= EpsF*max{|F(k)|,|F(k+1)|,1} is satisfied.     |
//|     EpsX    -   >=0                                              |
//|                 The subroutine finishes its work if on k+1-th    |
//|                 iteration the condition |v|<=EpsX is fulfilled,  |
//|                 where:                                           |
//|                 * |.| means Euclidian norm                       |
//|                 * v - scaled step vector, v[i]=dx[i]/s[i]        |
//|                 * dx - ste pvector, dx=X(k+1)-X(k)               |
//|                 * s - scaling coefficients set by                |
//|                   MinLBFGSSetScale()                             |
//|     MaxIts  -   maximum number of iterations. If MaxIts=0, the   |
//|                 number of iterations is unlimited.               |
//| Passing EpsG=0, EpsF=0, EpsX=0 and MaxIts=0 (simultaneously) will|
//| lead to automatic stopping criterion selection (small EpsX).     |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetCond(CMinLBFGSState &State,const double epsg,
                                const double epsf,double epsx,
                                const int m_maxits)
  {
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsg),__FUNCTION__+": EpsG is not finite number!"))
      return;
//--- check
   if(!CAp::Assert(epsg>=0.0,__FUNCTION__+": negative EpsG!"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsf),__FUNCTION__+": EpsF is not finite number!"))
      return;
//--- check
   if(!CAp::Assert(epsf>=0.0,__FUNCTION__+": negative EpsF!"))
      return;
//--- check
   if(!CAp::Assert(CMath::IsFinite(epsx),__FUNCTION__+": EpsX is not finite number!"))
      return;
//--- check
   if(!CAp::Assert(epsx>=0.0,__FUNCTION__+": negative EpsX!"))
      return;
//--- check
   if(!CAp::Assert(m_maxits>=0,__FUNCTION__+": negative MaxIts!"))
      return;
//--- check
   if(((epsg==0.0 && epsf==0.0) && epsx==0.0) && m_maxits==0)
      epsx=1.0E-6;
//--- change values
   State.m_epsg=epsg;
   State.m_epsf=epsf;
   State.m_epsx=epsx;
   State.m_maxits=m_maxits;
  }
//+------------------------------------------------------------------+
//| This function turns on/off reporting.                            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     NeedXRep-   whether iteration reports are needed or not      |
//| If NeedXRep is True, algorithm will call rep() callback function |
//| if it is provided to MinLBFGSOptimize().                         |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetXRep(CMinLBFGSState &State,const bool needxrep)
  {
   State.m_xrep=needxrep;
  }
//+------------------------------------------------------------------+
//| This function sets maximum step length                           |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     StpMax  -   maximum step length, >=0. Set StpMax to 0.0      |
//|                 (default), if you don't want to limit step       |
//|                 length.                                          |
//| Use this subroutine when you optimize target function which      |
//| contains exp() or other fast growing functions, and optimization |
//| algorithm makes too large steps which leads to overflow. This    |
//| function allows us to reject steps that are too large (and       |
//| therefore expose us to the possible overflow) without actually   |
//| calculating function value at the x+stp*d.                       |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetStpMax(CMinLBFGSState &State,
                                  const double stpmax)
  {
//--- check
   if(!CAp::Assert(CMath::IsFinite(stpmax),__FUNCTION__+": StpMax is not finite!"))
      return;
//--- check
   if(!CAp::Assert(stpmax>=0.0,__FUNCTION__+": StpMax<0!"))
      return;
//--- change value
   State.m_stpmax=stpmax;
  }
//+------------------------------------------------------------------+
//| This function sets scaling coefficients for LBFGS optimizer.     |
//| ALGLIB optimizers use scaling matrices to test stopping          |
//| conditions (step size and gradient are scaled before comparison  |
//| with tolerances). Scale of the I-th variable is a translation    |
//| invariant measure of:                                            |
//| a) "how large" the variable is                                   |
//| b) how large the step should be to make significant changes in   |
//|    the function                                                  |
//| Scaling is also used by finite difference variant of the         |
//| optimizer - step along I-th axis is equal to DiffStep*S[I].      |
//| In most optimizers (and in the LBFGS too) scaling is NOT a form  |
//| of preconditioning. It just affects stopping conditions. You     |
//| should set preconditioner by separate call to one of the         |
//| MinLBFGSSetPrec...() functions.                                  |
//| There is special preconditioning mode, however, which uses       |
//| scaling coefficients to form diagonal preconditioning matrix.    |
//| You can turn this mode on, if you want. But  you should          |
//| understand that scaling is not the same thing as                 |
//| preconditioning - these are two different, although related      |
//| forms of tuning m_solver.                                          |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure stores algorithm State                 |
//|     S       -   array[N], non-zero scaling coefficients          |
//|                 S[i] may be negative, sign doesn't matter.       |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetScale(CMinLBFGSState &State,double &s[])
  {
//--- check
   if(!CAp::Assert(CAp::Len(s)>=State.m_n,__FUNCTION__+": Length(S)<N"))
      return;

   for(int i=0; i<State.m_n; i++)
     {
      //--- check
      if(!CAp::Assert(CMath::IsFinite(s[i]),__FUNCTION__+": S contains infinite or NAN elements"))
         return;
      //--- check
      if(!CAp::Assert(s[i]!=0.0,__FUNCTION__+": S contains zero elements"))
         return;
      State.m_s.Set(i,MathAbs(s[i]));
     }
  }
//+------------------------------------------------------------------+
//| Extended subroutine for internal use only.                       |
//| Accepts additional parameters:                                   |
//|     Flags - additional Settings:                                 |
//|             * Flags = 0     means no additional Settings         |
//|            *Flags=1     "do not allocate memory". used when  |
//|                             solving a many subsequent tasks with |
//|                             same N/M values. First call MUST be  |
//|                             without this flag bit set, subsequent|
//|                             calls of MinLBFGS with same          |
//|                             MinLBFGSState structure can set Flags|
//|                             to 1.                                |
//|     DiffStep - numerical differentiation step                    |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSCreateX(const int n,const int m,double &x[],
                                int flags,const double diffstep,
                                CMinLBFGSState &State)
  {
   CRowDouble X=x;
   MinLBFGSCreateX(n,m,X,flags,diffstep,State);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSCreateX(const int n,const int m,CRowDouble &x,
                                int flags,const double diffstep,
                                CMinLBFGSState &State)
  {
//--- create variable
   bool allocatemem;
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N too small!"))
      return;
//--- check
   if(!CAp::Assert(m>=1,__FUNCTION__+": M too small!"))
      return;
//--- check
   if(!CAp::Assert(m<=n,__FUNCTION__+": M too large!"))
      return;
//--- Initialize
   State.m_diffstep=diffstep;
   State.m_n=n;
   State.m_m=m;
   allocatemem=flags%2==0;
   flags=flags/2;
//--- check
   if(allocatemem)
     {
      //--- allocation
      State.m_rho.Resize(m);
      State.m_theta.Resize(m);
      State.m_yk.Resize(m,n);
      State.m_sk.Resize(m,n);
      State.m_d.Resize(n);
      State.m_xp.Resize(n);
      State.m_x.Resize(n);
      State.m_xbase.Resize(n);
      State.m_g.Resize(n);
      State.m_work.Resize(n);
     }
//--- function call
   MinLBFGSSetCond(State,0,0,0,0);
//--- function call
   MinLBFGSSetXRep(State,false);
//--- function call
   MinLBFGSSetStpMax(State,0);
//--- function call
   MinLBFGSRestartFrom(State,x);
//--- change values
   State.m_s=vector<double>::Ones(n);
   State.m_invs=vector<double>::Ones(n);
   State.m_lastscaleused=vector<double>::Ones(n);
   State.m_prectype=0;
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: default preconditioner       |
//| (simple scaling, same for all elements of X) is used.            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//| NOTE: you can change preconditioner "on the fly", during         |
//| algorithm iterations.                                            |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetPrecDefault(CMinLBFGSState &State)
  {
   State.m_prectype=0;
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: Cholesky factorization of    |
//| approximate Hessian is used.                                     |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     P       -   triangular preconditioner, Cholesky factorization|
//|                 of the approximate Hessian. array[0..m_n-1,0..m_n-1],|
//|                 (if larger, only leading N elements are used).   |
//|     IsUpper -   whether upper or lower triangle of P is given    |
//|                 (other triangle is not referenced)               |
//| After call to this function preconditioner is changed to P (P is |
//| copied into the internal buffer).                                |
//| NOTE: you can change preconditioner "on the fly", during         |
//| algorithm iterations.                                            |
//| NOTE 2: P should be nonsingular. Exception will be thrown        |
//| otherwise.                                                       |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetPrecCholesky(CMinLBFGSState &State,
                                        CMatrixDouble &p,
                                        const bool IsUpper)
  {
//--- check
   if(!CAp::Assert(CApServ::IsFiniteRTrMatrix(p,State.m_n,IsUpper),__FUNCTION__+": P contains infinite or NAN values!"))
      return;
//--- initialization
   double mx=0;
   for(int i=0; i<State.m_n; i++)
      mx=MathMax(mx,MathAbs(p[i][i]));
//--- check
   if(!CAp::Assert(mx>0.0,__FUNCTION__+": P is strictly singular!"))
      return;
//--- check
   if((int)CAp::Rows(State.m_denseh)<State.m_n || (int)CAp::Cols(State.m_denseh)<State.m_n)
      State.m_denseh.Resize(State.m_n,State.m_n);
//--- initialization
   State.m_prectype=1;
//--- check
   if(IsUpper)
      CAblas::RMatrixCopy(State.m_n,State.m_n,p,0,0,State.m_denseh,0,0);
   else
      CAblas::RMatrixTranspose(State.m_n,State.m_n,p,0,0,State.m_denseh,0,0);
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: diagonal of approximate      |
//| Hessian is used.                                                 |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//|     D       -   diagonal of the approximate Hessian,             |
//|                 array[0..m_n-1], (if larger, only leading N        |
//|                 elements are used).                              |
//| NOTE: you can change preconditioner "on the fly", during         |
//| algorithm iterations.                                            |
//| NOTE 2: D[i] should be positive. Exception will be thrown        |
//| otherwise.                                                       |
//| NOTE 3: you should pass diagonal of approximate Hessian - NOT    |
//| ITS INVERSE.                                                     |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetPrecDiag(CMinLBFGSState &State,double &d[])
  {
//--- check
   if(!CAp::Assert(CAp::Len(d)>=State.m_n,__FUNCTION__+": D is too short"))
      return;
   for(int i=0; i<State.m_n; i++)
     {
      //--- check
      if(!CAp::Assert(CMath::IsFinite(d[i]),__FUNCTION__+": D contains infinite or NAN elements"))
         return;
      //--- check
      if(!CAp::Assert(d[i]>0.0,__FUNCTION__+": D contains non-positive elements"))
         return;
     }
//--- change values
   State.m_prectype=2;
   State.m_diagh=d;
   State.m_diagh.Resize(State.m_n);
  }
//+------------------------------------------------------------------+
//| Modification of the preconditioner: scale-based diagonal         |
//| preconditioning.                                                 |
//| This preconditioning mode can be useful when you don't have      |
//| approximate diagonal of Hessian, but you know that your variables|
//| are badly scaled (for example, one variable is in [1,10], and    |
//| another in [1000,100000]), and most part of the ill-conditioning |
//| comes from different scales of vars.                             |
//| In this case simple scale-based preconditioner, with H.Set(i,      |
//| = 1/(s[i]^2), can greatly improve convergence.                   |
//| IMPRTANT: you should set scale of your variables with            |
//| MinLBFGSSetScale() call (before or after MinLBFGSSetPrecScale()  |
//| call). Without knowledge of the scale of your variables          |
//| scale-based preconditioner will be just unit matrix.             |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm State           |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetPrecScale(CMinLBFGSState &State)
  {
   State.m_prectype=3;
  }
//+------------------------------------------------------------------+
//| This function sets low-rank preconditioner for Hessian matrix    |
//|                           H=D+W'*C*W,                            |
//| where:                                                           |
//|   * H is a Hessian matrix, which is approximated by D/W/C        |
//|   * D is a NxN diagonal positive definite matrix                 |
//|   * W is a KxN low-rank correction                               |
//|   * C is a KxK positive definite diagonal factor of low-rank     |
//|            correction                                            |
//| This preconditioner is inexact but fast - it requires O(N*K) time|
//| to be applied. Preconditioner P is calculated by artificially    |
//| constructing a set of BFGS updates which tries to reproduce      |
//| behavior of H:                                                   |
//|   * Sk = Wk (k-th row of W)                                      |
//|   * Yk = (D+Wk'*Ck*Wk)*Sk                                        |
//|   * Yk/Sk are reordered by ascending of C[k]*norm(Wk)^2          |
//| Here we assume that rows of Wk are orthogonal or nearly          |
//| orthogonal, which allows us to have O(N*K+K^2) update instead    |
//| of O(N*K^2) one. Reordering of updates is essential for having   |
//| good performance on non-orthogonal problems (updates which do not|
//| add much of curvature are added first, and updates which add very|
//| large eigenvalues are added last and override effect of the first|
//| updates).                                                        |
//| In practice, this preconditioner is perfect when ortogonal       |
//| correction is applied; on non-orthogonal problems sometimes it   |
//| allows to achieve 5x speedup (when compared to non-preconditioned|
//| solver).                                                         |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetPrecRankKLBFGSFast(CMinLBFGSState &State,
                                              CRowDouble &d,CRowDouble &c,CMatrixDouble &w,int cnt)
  {
   int n=State.m_n;
   State.m_prectype=4;
   State.m_preck=cnt;
//--- copy
   State.m_precd=d;
   State.m_precc=c;
   State.m_precw=w;
   State.m_precc.Resize(cnt);
   State.m_precd.Resize(n);
   State.m_precw.Resize(cnt,n);
  }
//+------------------------------------------------------------------+
//| This function sets exact low-rank preconditioner for Hessian     |
//| matrix H=D+W'*C*W, where:                                        |
//|   * H is a Hessian matrix, which is approximated by D/W/C        |
//|   * D is a NxN diagonal positive definite matrix                 |
//|   * W is a KxN low-rank correction                               |
//|   * C is a KxK semidefinite diagonal factor of low-rank          |
//|            correction                                            |
//| This preconditioner is exact but slow - it requires O(N*K^2) time|
//| to be built and O(N*K) time to be applied. Woodbury matrix       |
//| identity is  used  to build inverse matrix.                      |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSSetPrecLowRankExact(CMinLBFGSState &State,
                                            CRowDouble &d,
                                            CRowDouble &c,
                                            CMatrixDouble &w,
                                            int cnt)
  {
   State.m_prectype=5;
   COptServ::PrepareLowRankPreconditioner(d,c,w,State.m_n,cnt,State.m_lowrankbuf);
  }
//+------------------------------------------------------------------+
//| L-BFGS algorithm results                                         |
//| INPUT PARAMETERS:                                                |
//|     State   -   algorithm State                                  |
//| OUTPUT PARAMETERS:                                               |
//|     X       -   array[0..m_n-1], solution                          |
//|     Rep     -   optimization report:                             |
//|                 * Rep.TerminationType completetion code:         |
//|                     * -2    rounding errors prevent further      |
//|                             improvement. X contains best point   |
//|                             found.                               |
//|                     * -1    incorrect parameters were specified  |
//|                     *  1    relative function improvement is no  |
//|                             more than EpsF.                      |
//|                     *  2    relative step is no more than EpsX.  |
//|                     *  4    gradient norm is no more than EpsG   |
//|                     *  5    MaxIts steps was taken               |
//|                     *  7    stopping conditions are too          |
//|                             stringent, further improvement is    |
//|                             impossible                           |
//|                 * Rep.IterationsCount contains iterations count  |
//|                 * NFEV countains number of function calculations |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSResults(CMinLBFGSState &State,double &x[],
                                CMinLBFGSReport &rep)
  {
//--- reset memory
   ArrayResize(x,0);
//--- function call
   MinLBFGSResultsBuf(State,x,rep);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSResults(CMinLBFGSState &State,CRowDouble &x,
                                CMinLBFGSReport &rep)
  {
//--- reset memory
   x.Resize(0);
//--- function call
   MinLBFGSResultsBuf(State,x,rep);
  }
//+------------------------------------------------------------------+
//| L-BFGS algorithm results                                         |
//| Buffered implementation of MinLBFGSResults which uses            |
//| pre-allocated buffer to store X[]. If buffer size is too small,  |
//| it resizes buffer. It is intended to be used in the inner cycles |
//| of performance critical algorithms where array reallocation      |
//| penalty is too large to be ignored.                              |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSResultsBuf(CMinLBFGSState &State,double &x[],
                                   CMinLBFGSReport &rep)
  {
   CRowDouble X;
   MinLBFGSResultsBuf(State,X,rep);
   X.ToArray(x);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSResultsBuf(CMinLBFGSState &State,CRowDouble &x,
                                   CMinLBFGSReport &rep)
  {
//--- copy
   x=State.m_x;
//--- change values
   rep.m_iterationscount=State.m_repiterationscount;
   rep.m_nfev=State.m_repnfev;
   rep.m_terminationtype=State.m_repterminationtype;
  }
//+------------------------------------------------------------------+
//| This  subroutine restarts LBFGS algorithm from new point. All    |
//| optimization parameters are left unchanged.                      |
//| This function allows to solve multiple optimization problems     |
//| (which must have same number of dimensions) without object       |
//| reallocation penalty.                                            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure used to store algorithm State          |
//|     X       -   new starting point.                              |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSRestartFrom(CMinLBFGSState &State,double &x[])
  {
   CRowDouble X=x;
   MinLBFGSRestartFrom(State,X);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSRestartFrom(CMinLBFGSState &State,CRowDouble &x)
  {
//--- check
   if(!CAp::Assert(CAp::Len(x)>=State.m_n,__FUNCTION__+": Length(X)<N!"))
      return;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,State.m_n),__FUNCTION__+": X contains infinite or NaN values!"))
      return;
//--- copy
   State.m_xbase=x;
//--- allocation
   State.m_rstate.ia.Resize(6);
   State.m_rstate.ra.Resize(2);
   State.m_rstate.stage=-1;
//--- function call
   ClearRequestFields(State);
  }
//+------------------------------------------------------------------+
//| This subroutine submits request for termination of running       |
//| optimizer. It should be called from user-supplied callback when  |
//| user decides that it is time to "smoothly" terminate optimization|
//| process. As result, optimizer stops at point which was "current  |
//| accepted" when termination request was submitted and returns     |
//| error code 8 (successful termination).                           |
//| INPUT PARAMETERS:                                                |
//|   State    -  optimizer structure                                |
//| NOTE: after request for termination optimizer may perform several|
//|       additional calls to user-supplied callbacks. It does  NOT  |
//|       guarantee to stop immediately - it just guarantees that    |
//|       these additional calls will be discarded later.            |
//| NOTE: calling this function on optimizer which is NOT running    |
//|       will have no effect.                                       |
//| NOTE: multiple calls to this function are possible. First call is|
//|       counted, subsequent calls are silently ignored.            |
//+------------------------------------------------------------------+
void CMinLBFGS::MinLBFGSRequestTermination(CMinLBFGSState &State)
  {
   State.m_userterminationneeded=true;
  }
//+------------------------------------------------------------------+
//| Clears request fileds (to be sure that we don't forgot to clear  |
//| something)                                                       |
//+------------------------------------------------------------------+
void CMinLBFGS::ClearRequestFields(CMinLBFGSState &State)
  {
//--- change values
   State.m_needf=false;
   State.m_needfg=false;
   State.m_xupdated=false;
  }
//+------------------------------------------------------------------+
//| NOTES:                                                           |
//| 1. This function has two different implementations: one which    |
//|    uses exact (analytical) user-supplied gradient, and one which |
//|    uses function value only and numerically differentiates       |
//|    function in order to obtain gradient.                         |
//|    Depending on the specific function used to create optimizer   |
//|    object (either MinLBFGSCreate() for analytical gradient or    |
//|    MinLBFGSCreateF() for numerical differentiation) you should   |
//|    choose appropriate variant of MinLBFGSOptimize() - one which  |
//|    accepts function AND gradient or one which accepts function   |
//|    ONLY.                                                         |
//|    Be careful to choose variant of MinLBFGSOptimize() which      |
//|    corresponds to your optimization scheme! Table below lists    |
//|    different combinations of callback (function/gradient) passed |
//|    to MinLBFGSOptimize() and specific function used to create    |
//|    optimizer.                                                    |
//|                      |         USER PASSED TO MinLBFGSOptimize() |
//|    CREATED WITH      |  function only   |  function and gradient |
//|    ------------------------------------------------------------  |
//|    MinLBFGSCreateF() |     work                FAIL              |
//|    MinLBFGSCreate()  |     FAIL                work              |
//|    Here "FAIL" denotes inappropriate combinations of optimizer   |
//|    creation function and MinLBFGSOptimize() version. Attemps to  |
//|    use such combination (for example, to create optimizer with   |
//|    MinLBFGSCreateF() and to pass gradient information to         |
//|    MinCGOptimize()) will lead to exception being thrown. Either  |
//|    you did not pass gradient when it WAS needed or you passed    |
//|    gradient when it was NOT needed.                              |
//+------------------------------------------------------------------+
bool CMinLBFGS::MinLBFGSIteration(CMinLBFGSState &State)
  {
//--- create variables
   bool   result=false;
   int    n=0;
   int    m=0;
   int    i=0;
   int    j=0;
   int    ic=0;
   int    mcinfo=0;
   double v=0;
   double vv=0;
   int    i_=0;
   int    label=-1;
//--- Reverse communication preparations
//--- This code initializes locals by:
//--- * random values determined during code
//---   generation - on first subroutine call
//--- * values from previous call - on subsequent calls
   if(State.m_rstate.stage>=0)
     {
      n=State.m_rstate.ia[0];
      m=State.m_rstate.ia[1];
      i=State.m_rstate.ia[2];
      j=State.m_rstate.ia[3];
      ic=State.m_rstate.ia[4];
      mcinfo=State.m_rstate.ia[5];
      v=State.m_rstate.ra[0];
      vv=State.m_rstate.ra[1];
     }
   else
     {
      n=359;
      m=-58;
      i=-919;
      j=-909;
      ic=81;
      mcinfo=255;
      v=74;
      vv=-788;
     }

   switch(State.m_rstate.stage)
     {
      case 0:
         label=0;
         break;
      case 1:
         label=1;
         break;
      case 2:
         label=2;
         break;
      case 3:
         label=3;
         break;
      case 4:
         label=4;
         break;
      case 5:
         label=5;
         break;
      case 6:
         label=6;
         break;
      case 7:
         label=7;
         break;
      case 8:
         label=8;
         break;
      case 9:
         label=9;
         break;
      case 10:
         label=10;
         break;
      case 11:
         label=11;
         break;
      case 12:
         label=12;
         break;
      case 13:
         label=13;
         break;
      case 14:
         label=14;
         break;
      default:
         //--- Routine body
         //--- Unload frequently used variables from State structure
         //--- (just for typing convinience)
         n=State.m_n;
         m=State.m_m;
         //--- Init
         State.m_userterminationneeded=false;
         State.m_repterminationtype=0;
         State.m_repiterationscount=0;
         State.m_repnfev=0;
         COptServ::SmoothnessMonitorInit(State.m_smonitor,State.m_s,n,1,State.m_smoothnessguardlevel>0);
         State.m_invs.Resize(n);
         State.m_lastscaleused=State.m_s;
         State.m_invs=State.m_s.Pow(-1)+0;
         //---  Check, that transferred derivative value is right
         State.m_stp=0;
         ClearRequestFields(State);
         if(!(State.m_diffstep==0.0 && State.m_teststep>0.0))
           {
            label=15;
            break;
           }
         label=17;
         break;
     }
//--- main loop
   while(label>=0)
     {
      switch(label)
        {
         case 17:
            if(!COptServ::SmoothnessMonitorCheckGradientATX0(State.m_smonitor,State.m_xbase,State.m_s,State.m_s,State.m_s,false,State.m_teststep))
              {
               label=18;
               break;
              }
            State.m_x=State.m_smonitor.m_x;
            State.m_needfg=true;
            State.m_rstate.stage=0;
            label=-1;
            break;
         case 0:
            State.m_needfg=false;
            State.m_smonitor.m_fi.Set(0,State.m_f);
            State.m_smonitor.m_j.Row(0,State.m_g);
            label=17;
            break;
         case 18:
         case 15:
            //--- Calculate F/G at the initial point
            State.m_x=State.m_xbase;
            State.m_stp=0;
            ClearRequestFields(State);
            if(State.m_diffstep!=0.0)
              {
               label=19;
               break;
              }
            State.m_needfg=true;
            State.m_rstate.stage=1;
            label=-1;
            break;
         case 1:
            State.m_needfg=false;
            label=20;
            break;
         case 19:
            State.m_needf=true;
            State.m_rstate.stage=2;
            label=-1;
            break;
         case 2:
            State.m_fbase=State.m_f;
            i=0;
         case 21:
            if(i>n-1)
              {
               label=23;
               break;
              }
            v=State.m_x[i];
            State.m_x.Set(i,v-State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=3;
            label=-1;
            break;
         case 3:
            State.m_fm2=State.m_f;
            State.m_x.Set(i,v-0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=4;
            label=-1;
            break;
         case 4:
            State.m_fm1=State.m_f;
            State.m_x.Set(i,v+0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=5;
            label=-1;
            break;
         case 5:
            State.m_fp1=State.m_f;
            State.m_x.Set(i,v+State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=6;
            label=-1;
            break;
         case 6:
            State.m_fp2=State.m_f;
            State.m_x.Set(i,v);
            State.m_g.Set(i,(8*(State.m_fp1-State.m_fm1)-(State.m_fp2-State.m_fm2))/(6*State.m_diffstep*State.m_s[i]));
            i++;
            label=21;
            break;
         case 23:
            State.m_f=State.m_fbase;
            State.m_needf=false;
         case 20:
            COptServ::TrimPrepare(State.m_f,State.m_trimthreshold);
            if(!State.m_xrep)
              {
               label=24;
               break;
              }
            ClearRequestFields(State);
            State.m_xupdated=true;
            State.m_rstate.stage=7;
            label=-1;
            break;
         case 7:
            State.m_xupdated=false;
         case 24:
            if(State.m_userterminationneeded)
              {
               //--- User requested termination
               State.m_repterminationtype=8;
               result=false;
               return(result);
              }
            State.m_repnfev=1;
            State.m_fold=State.m_f;
            v=MathPow(State.m_g*State.m_s+0,2.0).Sum();
            if(MathSqrt(v)<=State.m_epsg)
              {
               State.m_repterminationtype=4;
               result=false;
               return(result);
              }
            //--- Choose initial step and direction.
            //--- Apply preconditioner, if we have something other than default.
            State.m_d=State.m_g.ToVector()*(-1.0);
            switch(State.m_prectype)
              {
               case 0:
                  //--- Default preconditioner is used, but we can't use it before iterations will start
                  v=State.m_g.Dot(State.m_g);
                  v=MathSqrt(v);
                  if(State.m_stpmax==0.0)
                     State.m_stp=MathMin(1.0/v,1);
                  else
                     State.m_stp=MathMin(1.0/v,State.m_stpmax);
                  break;
               case 1:
                  //--- Cholesky preconditioner is used
                  CFbls::FblsCholeskySolve(State.m_denseh,1.0,n,true,State.m_d,State.m_autobuf);
                  State.m_stp=1;
                  break;
               case 2:
                  //--- diagonal approximation is used
                  State.m_d/=State.m_diagh;
                  State.m_stp=1;
                  break;
               case 3:
                  //--- scale-based preconditioner is used
                  State.m_d*=State.m_s.Pow(2.0)+0;
                  State.m_stp=1;
                  break;
               case 4:
                  //--- rank-k BFGS-based preconditioner is used
                  COptServ::InexactLBFGSPreconditioner(State.m_d,n,State.m_precd,State.m_precc,State.m_precw,State.m_preck,State.m_precbuf);
                  State.m_stp=1;
                  break;
               case 5:
                  //--- exact low-rank preconditioner is used
                  COptServ::ApplyLowRankPreconditioner(State.m_d,State.m_lowrankbuf);
                  State.m_stp=1;
                  break;
              }
            //--- Main cycle
            State.m_k=0;
         case 26:
            //--- Main cycle: prepare to 1-D line search
            State.m_p=State.m_k%m;
            State.m_q=MathMin(State.m_k,m-1);
            //--- Store X[k], G[k]
            State.m_xp=State.m_x;
            State.m_sk.Row(State.m_p,State.m_x*(-1)+0);
            State.m_yk.Row(State.m_p,State.m_g*(-1)+0);
            //--- Minimize F(x+alpha*d)
            //--- Calculate S[k], Y[k]
            State.m_mcstage=0;
            if(State.m_k!=0)
               State.m_stp=1.0;
            CLinMin::LinMinNormalized(State.m_d,State.m_stp,n);
            COptServ::SmoothnessMonitorStartLineSearch1u(State.m_smonitor,State.m_s,State.m_invs,State.m_x,State.m_f,State.m_g);
            CLinMin::MCSrch(n,State.m_x,State.m_f,State.m_g,State.m_d,State.m_stp,State.m_stpmax,m_gtol,mcinfo,State.m_nfev,State.m_work,State.m_lstate,State.m_mcstage);
         case 28:
            if(State.m_mcstage==0)
              {
               label=29;
               break;
              }
            ClearRequestFields(State);
            if(State.m_diffstep!=0.0)
              {
               label=30;
               break;
              }
            State.m_needfg=true;
            State.m_rstate.stage=8;
            label=-1;
            break;
         case 8:
            State.m_needfg=false;
            label=31;
            break;
         case 30:
            State.m_needf=true;
            State.m_rstate.stage=9;
            label=-1;
            break;
         case 9:
            State.m_fbase=State.m_f;
            i=0;
         case 32:
            if(i>n-1)
              {
               label=34;
               break;
              }
            v=State.m_x[i];
            State.m_x.Set(i,v-State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=10;
            label=-1;
            break;
         case 10:
            State.m_fm2=State.m_f;
            State.m_x.Set(i,v-0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=11;
            label=-1;
            break;
         case 11:
            State.m_fm1=State.m_f;
            State.m_x.Set(i,v+0.5*State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=12;
            label=-1;
            break;
         case 12:
            State.m_fp1=State.m_f;
            State.m_x.Set(i,v+State.m_diffstep*State.m_s[i]);
            State.m_rstate.stage=13;
            label=-1;
            break;
         case 13:
            State.m_fp2=State.m_f;
            State.m_x.Set(i,v);
            State.m_g.Set(i,(8*(State.m_fp1-State.m_fm1)-(State.m_fp2-State.m_fm2))/(6*State.m_diffstep*State.m_s[i]));
            i++;
            label=32;
            break;
         case 34:
            State.m_f=State.m_fbase;
            State.m_needf=false;
         case 31:
            COptServ::SmoothnessMonitorEnqueuePoint1u(State.m_smonitor,State.m_s,State.m_invs,State.m_d,State.m_stp,State.m_x,State.m_f,State.m_g);
            COptServ::TrimFunction(State.m_f,State.m_g,n,State.m_trimthreshold);
            CLinMin::MCSrch(n,State.m_x,State.m_f,State.m_g,State.m_d,State.m_stp,State.m_stpmax,m_gtol,mcinfo,State.m_nfev,State.m_work,State.m_lstate,State.m_mcstage);
            label=28;
            break;
         case 29:
            COptServ::SmoothnessMonitorFinalizeLineSearch(State.m_smonitor);
            if(State.m_userterminationneeded)
              {
               //--- User requested termination.
               //--- Restore previous point and return.
               State.m_x=State.m_xp;
               State.m_repterminationtype=8;
               result=false;
               return(result);
              }
            if(!State.m_xrep)
              {
               label=35;
               break;
              }
            //--- report
            ClearRequestFields(State);
            State.m_xupdated=true;
            State.m_rstate.stage=14;
            label=-1;
            break;
         case 14:
            State.m_xupdated=false;
         case 35:
            State.m_repnfev=State.m_repnfev+State.m_nfev;
            State.m_repiterationscount=State.m_repiterationscount+1;
            State.m_sk.Row(State.m_p,State.m_x.ToVector()+State.m_sk[State.m_p]);
            State.m_yk.Row(State.m_p,State.m_g.ToVector()+State.m_yk[State.m_p]);
            //--- Stopping conditions
            v=MathPow(State.m_g*State.m_s+0,2.0).Sum();
            if(!MathIsValidNumber(v) || !MathIsValidNumber(State.m_f))
              {
               //--- Abnormal termination - infinities in function/gradient
               State.m_repterminationtype=-8;
               result=false;
               return(result);
              }
            if(State.m_repiterationscount>=State.m_maxits && State.m_maxits>0)
              {
               //--- Too many iterations
               State.m_repterminationtype=5;
               result=false;
               return(result);
              }
            if(MathSqrt(v)<=State.m_epsg)
              {
               //--- Gradient is small enough
               State.m_repterminationtype=4;
               result=false;
               return(result);
              }
            if((State.m_fold-State.m_f)<=(State.m_epsf*MathMax(MathAbs(State.m_fold),MathMax(MathAbs(State.m_f),1.0))))
              {
               //--- F(k+1)-F(k) is small enough
               State.m_repterminationtype=1;
               result=false;
               return(result);
              }
            v=MathPow(State.m_sk[State.m_p]/State.m_s.ToVector(),2.0).Sum();
            if(MathSqrt(v)<=State.m_epsx)
              {
               //--- X(k+1)-X(k) is small enough
               State.m_repterminationtype=2;
               result=false;
               return(result);
              }
            //--- If Wolfe conditions are satisfied, we can update
            //--- limited memory model.
            //--- However, if conditions are not satisfied (NFEV limit is met,
            //--- function is too wild, ...), we'll skip L-BFGS update
            if(mcinfo!=1)
              {
               //--- Skip update.
               //--- In such cases we'll initialize search direction by
               //--- antigradient vector, because it  leads to more
               //--- transparent code with less number of special cases
               State.m_fold=State.m_f;
               State.m_d=State.m_g.ToVector()*(-1.0);
              }
            else
              {
               //--- Calculate Rho[k], GammaK
               v=0.0;
               for(i_=0; i_<n; i_++)
                  v+=State.m_yk.Get(State.m_p,i_)*State.m_sk.Get(State.m_p,i_);
               vv=0.0;
               for(i_=0; i_<n; i_++)
                  vv+=State.m_yk.Get(State.m_p,i_)*State.m_yk.Get(State.m_p,i_);
               if(v==0.0 || vv==0.0)
                 {
                  //--- Rounding errors make further iterations impossible.
                  State.m_repterminationtype=-2;
                  result=false;
                  return(result);
                 }
               State.m_rho.Set(State.m_p,1/v);
               State.m_gammak=v/vv;
               //---  Calculate d(k+1) = -H(k+1)*g(k+1)
               //---  for I:=K downto K-Q do
               //---      V = s(i)^T * work(iteration:I)
               //---      theta(i) = V
               //---      work(iteration:I+1) = work(iteration:I) - V*Rho(i)*y(i)
               //---  work(last iteration) = H0*work(last iteration) - preconditioner
               //---  for I:=K-Q to K do
               //---      V = y(i)^T*work(iteration:I)
               //---      work(iteration:I+1) = work(iteration:I) +(-V+theta(i))*Rho(i)*s(i)
               //---  NOW WORK CONTAINS d(k+1)
               State.m_work=State.m_g;
               for(i=State.m_k; i>=State.m_k-State.m_q; i--)
                 {
                  ic=i%m;
                  v=State.m_work.DotR(State.m_sk,ic);
                  State.m_theta.Set(ic,v);
                  vv=v*State.m_rho[ic];
                  State.m_work-=State.m_yk[ic]*vv;
                 }
               switch(State.m_prectype)
                 {
                  case 0:
                     //--- Simple preconditioner is used
                     v=State.m_gammak;
                     State.m_work*=v;
                     break;
                  case 1:
                     //--- Cholesky preconditioner is used
                     CFbls::FblsCholeskySolve(State.m_denseh,1,n,true,State.m_work,State.m_autobuf);
                     break;
                  case 2:
                     //--- diagonal approximation is used
                     State.m_work/=State.m_diagh;
                     break;
                  case 3:
                     //--- scale-based preconditioner is used
                     State.m_work*=State.m_s.Pow(2.0)+0;
                     break;
                  case 4:
                     //--- Rank-K BFGS-based preconditioner is used
                     COptServ::InexactLBFGSPreconditioner(State.m_work,n,State.m_precd,State.m_precc,State.m_precw,State.m_preck,State.m_precbuf);
                     break;
                  case 5:
                     //--- Exact low-rank preconditioner is used
                     COptServ::ApplyLowRankPreconditioner(State.m_work,State.m_lowrankbuf);
                     break;
                 }
               for(i=State.m_k-State.m_q; i<=State.m_k; i++)
                 {
                  ic=i%m;
                  v=State.m_work.DotR(State.m_yk,ic);
                  vv=State.m_rho[ic]*(-v+State.m_theta[ic]);
                  State.m_work+=State.m_sk[ic]*vv;
                 }
               State.m_d=State.m_work.ToVector()*(-1.0);
               //--- Next step
               State.m_fold=State.m_f;
               State.m_k++;
              }
            label=26;
            break;
         case 27:
            result=false;
            return(result);
        }
     }
//--- Saving State
   result=true;
   State.m_rstate.ia.Set(0,n);
   State.m_rstate.ia.Set(1,m);
   State.m_rstate.ia.Set(2,i);
   State.m_rstate.ia.Set(3,j);
   State.m_rstate.ia.Set(4,ic);
   State.m_rstate.ia.Set(5,mcinfo);
   State.m_rstate.ra.Set(0,v);
   State.m_rstate.ra.Set(1,vv);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Variables of IPM  method(primal and dual, slacks)                |
//+------------------------------------------------------------------+
struct CVIPMVars
  {
   int               m_m;
   int               m_n;
   CRowDouble        m_g;
   CRowDouble        m_p;
   CRowDouble        m_q;
   CRowDouble        m_s;
   CRowDouble        m_t;
   CRowDouble        m_v;
   CRowDouble        m_w;
   CRowDouble        m_x;
   CRowDouble        m_y;
   CRowDouble        m_z;
   //--- constructor / destructor
                     CVIPMVars(void) { m_m=0; m_n=0; }
                    ~CVIPMVars(void) {}
   //---
   void              Copy(const CVIPMVars &obj);
   //--- overloading
   void              operator=(const CVIPMVars &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CVIPMVars::Copy(const CVIPMVars &obj)
  {
   m_m=obj.m_m;
   m_n=obj.m_n;
   m_g=obj.m_g;
   m_p=obj.m_p;
   m_q=obj.m_q;
   m_s=obj.m_s;
   m_t=obj.m_t;
   m_v=obj.m_v;
   m_w=obj.m_w;
   m_x=obj.m_x;
   m_y=obj.m_y;
   m_z=obj.m_z;
  }
//+------------------------------------------------------------------+
//| Reduced(M + N)*(M + N) KKT system stored in sparse format        |
//+------------------------------------------------------------------+
struct CVIPMReducedSparseSystem
  {
   int               m_ntotal;
   bool              m_isdiagonal[];
   CSparseMatrix     m_rawsystem;
   CSpCholAnalysis   m_analysis;
   CRowInt           m_coldegrees;
   CRowInt           m_priorities;
   CRowInt           m_rowdegrees;
   CRowDouble        m_effectivediag;
   //--- constructor / destructor
                     CVIPMReducedSparseSystem(void) { m_ntotal=0; }
                    ~CVIPMReducedSparseSystem(void) {}
   void              Copy(const CVIPMReducedSparseSystem &obj);
   //--- overloading
   void              operator=(const CVIPMReducedSparseSystem &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CVIPMReducedSparseSystem::Copy(const CVIPMReducedSparseSystem &obj)
  {
   m_ntotal=obj.m_ntotal;
   ArrayCopy(m_isdiagonal,obj.m_isdiagonal);
   m_rawsystem=obj.m_rawsystem;
   m_analysis=obj.m_analysis;
   m_coldegrees=obj.m_coldegrees;
   m_priorities=obj.m_priorities;
   m_rowdegrees=obj.m_rowdegrees;
   m_effectivediag=obj.m_effectivediag;
  }
//+------------------------------------------------------------------+
//| Right - hand side for KKT system:                                |
//|   * Rho corresponds to Ax - w = b                                |
//|   * Nu corresponds to x - g = l                                  |
//|   * Tau corresponds to x + t = u                                 |
//|   * Alpha corresponds to w + p = r                               |
//|   * Sigma corresponds to A'y+z-s-Hx=c                            |
//|   * Beta corresponds to y + q - v = 0                            |
//|   * GammaZ, GammaS, GammaQ, GammaW correspond to complementarity |
//|     conditions                                                   |
//+------------------------------------------------------------------+
struct CVIPMRightHandSide
  {
   CRowDouble        m_alpha;
   CRowDouble        m_beta;
   CRowDouble        m_gammaq;
   CRowDouble        m_gammas;
   CRowDouble        m_gammaw;
   CRowDouble        m_gammaz;
   CRowDouble        m_nu;
   CRowDouble        m_rho;
   CRowDouble        m_sigma;
   CRowDouble        m_tau;
   //--- constructor / destructor
                     CVIPMRightHandSide(void) {}
                    ~CVIPMRightHandSide(void) {}
   //---
   void              Copy(const CVIPMRightHandSide &obj);
   //--- overloading
   void              operator=(const CVIPMRightHandSide &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CVIPMRightHandSide::Copy(const CVIPMRightHandSide &obj)
  {
   m_alpha=obj.m_alpha;
   m_beta=obj.m_beta;
   m_gammaq=obj.m_gammaq;
   m_gammas=obj.m_gammas;
   m_gammaw=obj.m_gammaw;
   m_gammaz=obj.m_gammaz;
   m_nu=obj.m_nu;
   m_rho=obj.m_rho;
   m_sigma=obj.m_sigma;
   m_tau=obj.m_tau;
  }
//+------------------------------------------------------------------+
//| VIPM State                                                       |
//+------------------------------------------------------------------+
struct CVIPMState
  {
   int               m_cntgz;
   int               m_cntpq;
   int               m_cntts;
   int               m_cntwv;
   int               m_factorizationtype;
   int               m_hkind;
   int               m_mdense;
   int               m_msparse;
   int               m_n;
   int               m_nmain;
   int               m_repiterationscount;
   int               m_repncholesky;
   double            m_epsd;
   double            m_epsgap;
   double            m_epsp;
   double            m_targetscale;
   bool              m_aflips[];
   bool              m_dodetailedtrace;
   bool              m_dotrace;
   bool              m_factorizationpoweredup;
   bool              m_factorizationpresent;
   bool              m_HasBndL[];
   bool              m_HasBndU[];
   bool              m_hasgz[];
   bool              m_haspq[];
   bool              m_hasr[];
   bool              m_hasts[];
   bool              m_haswv[];
   bool              m_isdiagonalh;
   bool              m_isfrozen[];
   bool              m_islinear;
   bool              m_slacksforequalityconstraints;
   CSparseMatrix     m_combinedaslack;
   CSparseMatrix     m_sparseafull;
   CSparseMatrix     m_sparseamain;
   CSparseMatrix     m_sparseh;
   CSparseMatrix     m_tmpsparse0;
   CVIPMVars         m_best;
   CVIPMVars         m_current;
   CVIPMVars         m_deltaaff;
   CVIPMVars         m_deltacorr;
   CVIPMVars         m_trial;
   CVIPMVars         m_zerovars;
   CVIPMRightHandSide m_rhs;
   CVIPMReducedSparseSystem m_reducedsparsesystem;
   CRowInt           m_tmpi;
   CRowDouble        m_ascales;
   CRowDouble        m_b;
   CRowDouble        m_bndl;
   CRowDouble        m_bndu;
   CRowDouble        m_c;
   CRowDouble        m_deltaxy;
   CRowDouble        m_diagddr;
   CRowDouble        m_diagde;
   CRowDouble        m_diagder;
   CRowDouble        m_diagdq;
   CRowDouble        m_diagdqi;
   CRowDouble        m_diagdqiri;
   CRowDouble        m_diagds;
   CRowDouble        m_diagdsi;
   CRowDouble        m_diagdsiri;
   CRowDouble        m_diagdw;
   CRowDouble        m_diagdwi;
   CRowDouble        m_diagdwir;
   CRowDouble        m_diagdz;
   CRowDouble        m_diagdzi;
   CRowDouble        m_diagdziri;
   CRowDouble        m_diagr;
   CRowDouble        m_dummyr;
   CRowDouble        m_factinvregdzrz;
   CRowDouble        m_factregdhrh;
   CRowDouble        m_factregewave;
   CRowDouble        m_facttmpdiag;
   CRowDouble        m_invscl;
   CRowDouble        m_tmp0;
   CRowDouble        m_tmp1;
   CRowDouble        m_r;
   CRowDouble        m_rawbndl;
   CRowDouble        m_rawbndu;
   CRowDouble        m_rhsalphacap;
   CRowDouble        m_rhsbetacap;
   CRowDouble        m_rhsnucap;
   CRowDouble        m_rhstaucap;
   CRowDouble        m_scl;
   CRowDouble        m_tmp2;
   CRowDouble        m_tmpaty;
   CRowDouble        m_tmpax;
   CRowDouble        m_tmphx;
   CRowDouble        m_tmplaggrad;
   CRowDouble        m_tmpy;
   CRowDouble        m_xorigin;
   CMatrixDouble     m_denseafull;
   CMatrixDouble     m_denseamain;
   CMatrixDouble     m_denseh;
   CMatrixDouble     m_factdensehaug;
   CMatrixDouble     m_tmpr2;
   //--- constructor / destructor
                     CVIPMState(void);
                    ~CVIPMState(void) {}
   //---
   void              Copy(const CVIPMState &obj);
   //--- overloading
   void              operator=(const CVIPMState &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CVIPMState::CVIPMState(void)
  {
   m_cntgz=0;
   m_cntpq=0;
   m_cntts=0;
   m_cntwv=0;
   m_factorizationtype=0;
   m_hkind=0;
   m_mdense=0;
   m_msparse=0;
   m_n=0;
   m_nmain=0;
   m_repiterationscount=0;
   m_repncholesky=0;
   m_epsd=0;
   m_epsgap=0;
   m_epsp=0;
   m_targetscale=0;
   m_dodetailedtrace=false;
   m_dotrace=false;
   m_factorizationpoweredup=false;
   m_factorizationpresent=false;
   m_isdiagonalh=false;
   m_islinear=false;
   m_slacksforequalityconstraints=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CVIPMState::Copy(const CVIPMState &obj)
  {
   m_cntgz=obj.m_cntgz;
   m_cntpq=obj.m_cntpq;
   m_cntts=obj.m_cntts;
   m_cntwv=obj.m_cntwv;
   m_factorizationtype=obj.m_factorizationtype;
   m_hkind=obj.m_hkind;
   m_mdense=obj.m_mdense;
   m_msparse=obj.m_msparse;
   m_n=obj.m_n;
   m_nmain=obj.m_nmain;
   m_repiterationscount=obj.m_repiterationscount;
   m_repncholesky=obj.m_repncholesky;
   m_epsd=obj.m_epsd;
   m_epsgap=obj.m_epsgap;
   m_epsp=obj.m_epsp;
   m_targetscale=obj.m_targetscale;
   ArrayCopy(m_aflips,obj.m_aflips);
   m_dodetailedtrace=obj.m_dodetailedtrace;
   m_dotrace=obj.m_dotrace;
   m_factorizationpoweredup=obj.m_factorizationpoweredup;
   m_factorizationpresent=obj.m_factorizationpresent;
   ArrayCopy(m_HasBndL,obj.m_HasBndL);
   ArrayCopy(m_HasBndU,obj.m_HasBndU);
   ArrayCopy(m_hasgz,obj.m_hasgz);
   ArrayCopy(m_haspq,obj.m_haspq);
   ArrayCopy(m_hasr,obj.m_hasr);
   ArrayCopy(m_hasts,obj.m_hasts);
   ArrayCopy(m_haswv,obj.m_haswv);
   m_isdiagonalh=obj.m_isdiagonalh;
   ArrayCopy(m_isfrozen,obj.m_isfrozen);
   m_islinear=obj.m_islinear;
   m_slacksforequalityconstraints=obj.m_slacksforequalityconstraints;
   m_combinedaslack=obj.m_combinedaslack;
   m_sparseafull=obj.m_sparseafull;
   m_sparseamain=obj.m_sparseamain;
   m_sparseh=obj.m_sparseh;
   m_tmpsparse0=obj.m_tmpsparse0;
   m_best=obj.m_best;
   m_current=obj.m_current;
   m_deltaaff=obj.m_deltaaff;
   m_deltacorr=obj.m_deltacorr;
   m_trial=obj.m_trial;
   m_zerovars=obj.m_zerovars;
   m_rhs=obj.m_rhs;
   m_reducedsparsesystem=obj.m_reducedsparsesystem;
   m_tmpi=obj.m_tmpi;
   m_ascales=obj.m_ascales;
   m_b=obj.m_b;
   m_bndl=obj.m_bndl;
   m_bndu=obj.m_bndu;
   m_c=obj.m_c;
   m_deltaxy=obj.m_deltaxy;
   m_diagddr=obj.m_diagddr;
   m_diagde=obj.m_diagde;
   m_diagder=obj.m_diagder;
   m_diagdq=obj.m_diagdq;
   m_diagdqi=obj.m_diagdqi;
   m_diagdqiri=obj.m_diagdqiri;
   m_diagds=obj.m_diagds;
   m_diagdsi=obj.m_diagdsi;
   m_diagdsiri=obj.m_diagdsiri;
   m_diagdw=obj.m_diagdw;
   m_diagdwi=obj.m_diagdwi;
   m_diagdwir=obj.m_diagdwir;
   m_diagdz=obj.m_diagdz;
   m_diagdzi=obj.m_diagdzi;
   m_diagdziri=obj.m_diagdziri;
   m_diagr=obj.m_diagr;
   m_dummyr=obj.m_dummyr;
   m_factinvregdzrz=obj.m_factinvregdzrz;
   m_factregdhrh=obj.m_factregdhrh;
   m_factregewave=obj.m_factregewave;
   m_facttmpdiag=obj.m_facttmpdiag;
   m_invscl=obj.m_invscl;
   m_tmp0=obj.m_tmp0;
   m_tmp1=obj.m_tmp1;
   m_r=obj.m_r;
   m_rawbndl=obj.m_rawbndl;
   m_rawbndu=obj.m_rawbndu;
   m_rhsalphacap=obj.m_rhsalphacap;
   m_rhsbetacap=obj.m_rhsbetacap;
   m_rhsnucap=obj.m_rhsnucap;
   m_rhstaucap=obj.m_rhstaucap;
   m_scl=obj.m_scl;
   m_tmp2=obj.m_tmp2;
   m_tmpaty=obj.m_tmpaty;
   m_tmpax=obj.m_tmpax;
   m_tmphx=obj.m_tmphx;
   m_tmplaggrad=obj.m_tmplaggrad;
   m_tmpy=obj.m_tmpy;
   m_xorigin=obj.m_xorigin;
   m_denseafull=obj.m_denseafull;
   m_denseamain=obj.m_denseamain;
   m_denseh=obj.m_denseh;
   m_factdensehaug=obj.m_factdensehaug;
   m_tmpr2=obj.m_tmpr2;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CVIPMSolver
  {
public:
   //--- constants
   static const double m_muquasidense;
   static const int  m_maxipmits;
   static const double m_initslackval;
   static const double m_steplengthdecay;
   static const double m_stagnationdelta;
   static const double m_primalinfeasible1;
   static const double m_dualinfeasible1;
   static const double m_bigy;
   static const double m_ygrowth;
   static const int  m_itersfortoostringentcond;
   static const int  m_minitersbeforedroppingbounds;
   static const int  m_minitersbeforeinfeasible;
   static const int  m_minitersbeforestagnation;
   static const int  m_minitersbeforeeworststagnation;
   static const int  m_primalstagnationlen;
   static const int  m_dualstagnationlen;
   static const double m_bigconstrxtol;
   static const double m_bigconstrmag;
   static const double m_minitersbeforesafeguards;
   static const double m_badsteplength;

   static void       VIPMInitDense(CVIPMState &State,CRowDouble &s,CRowDouble &xorigin,int n);
   static void       VIPMInitDenseWithSlacks(CVIPMState &State,CRowDouble &s,CRowDouble &xorigin,int nmain,int n);
   static void       VIPMInitSparse(CVIPMState &State,CRowDouble &s,CRowDouble &xorigin,int n);
   static void       VIPMSetQuadraticLinear(CVIPMState &State,CMatrixDouble &denseh,CSparseMatrix &sparseh,int hkind,bool IsUpper,CRowDouble &c);
   static void       VIPMSetConstraints(CVIPMState &State,CRowDouble &bndl,CRowDouble &bndu,CSparseMatrix &sparsea,int msparse,CMatrixDouble &densea,int mdense,CRowDouble &cl,CRowDouble &cu);
   static void       VIPMSetCond(CVIPMState &State,double epsp,double epsd,double epsgap);
   static void       VIPMOptimize(CVIPMState &State,bool dropbigbounds,CRowDouble &xs,CRowDouble &lagbc,CRowDouble &laglc,int &terminationtype);

private:
   static void       VARSInitByZero(CVIPMVars &vstate,int n,int m);
   static void       VARSInitFrom(CVIPMVars &vstate,CVIPMVars &vsrc);
   static void       VarsAddStep(CVIPMVars &vstate,CVIPMVars &vdir,double stpp,double stpd);
   static double     VarsComputeComplementarityGap(CVIPMVars &vstate);
   static double     VARSComputeMu(CVIPMState &State,CVIPMVars &vstate);
   static void       ReducedSystemInit(CVIPMReducedSparseSystem &s,CVIPMState &solver);
   static bool       ReducedSystemFactorizeWithAddEnd(CVIPMReducedSparseSystem &s,CRowDouble &d,double modeps,double badchol,double &sumsq,double &errsq);
   static void       ReducedSystemSolve(CVIPMReducedSparseSystem &s,CRowDouble &b);
   static void       VIPMInit(CVIPMState &State,CRowDouble &s,CRowDouble &xorigin,int n,int nmain,int ftype);
   static double     VIPMTarget(CVIPMState &State,CRowDouble &x);
   static void       MultiplyGEAX(CVIPMState &State,double alpha,CRowDouble &x,int offsx,double beta,CRowDouble &y,int offsax);
   static void       MultiplyGEATX(CVIPMState &State,double alpha,CRowDouble &x,int offsx,double beta,CRowDouble &y,int offsy);
   static void       MultiplyHX(CVIPMState &State,CRowDouble &x,CRowDouble &hx);
   static void       VIPMMultiply(CVIPMState &State,CRowDouble &x,CRowDouble &y,CRowDouble &hx,CRowDouble &ax,CRowDouble &aty);
   static void       VIPMPowerUp(CVIPMState &State,double regfree);
   static bool       VIPMFactorize(CVIPMState &State,double alpha0,CRowDouble &d,double beta0,CRowDouble &e,double alpha11,double beta11,double modeps,double dampeps);
   static void       SolveReducedKKTSystem(CVIPMState &State,CRowDouble &deltaxy);
   static bool       VIPMPrecomputeNewtonFactorization(CVIPMState &State,CVIPMVars &v0,double regeps,double modeps,double dampeps,double dampfree);
   static void       SolveKKTSystem(CVIPMState &State,CVIPMRightHandSide &rhs,CVIPMVars &sol);
   static bool       VIPMComputeStepDirection(CVIPMState &State,CVIPMVars &v0,double muestimate,CVIPMVars &vdestimate,CVIPMVars &vdresult,double reg,bool isdampepslarge);
   static void       VIPMComputeStepLength(CVIPMState &State,CVIPMVars &v0,CVIPMVars &vs,double stepdecay,double &alphap,double &alphad);
   static void       VIPMPerformStep(CVIPMState &State,double alphap,double alphad);
   static void       ComputeErrors(CVIPMState &State,double &errp2,double &errd2,double &errpinf,double &errdinf,double &egap);
   static void       RunIntegrityChecks(CVIPMState &State,CVIPMVars &v0,CVIPMVars &vd,double alphap,double alphad);
   static void       TraceProgress(CVIPMState &State,double mu,double muaff,double sigma,double alphap,double alphad);
   static void       RHSCompute(CVIPMState &State,CVIPMVars &v0,double muestimate,CVIPMVars &direstimate,CVIPMRightHandSide &rhs,double reg);
   static void       RHSSubtract(CVIPMState &State,CVIPMRightHandSide &rhs,CVIPMVars &v0,CVIPMVars &vdcandidate,double reg);
   static double     RHSPrimal2(CVIPMRightHandSide &rhs);
   static double     RHSDual2(CVIPMRightHandSide &rhs);
   static double     RHSPrimalInf(CVIPMRightHandSide &rhs);
   static double     RHSDualInf(CVIPMRightHandSide &rhs);
   static double     RHSCompl2(CVIPMRightHandSide &rhs);
   static double     MinNZ(CRowDouble &x,int n);
   static double     MinProdNZ(CRowDouble &x,CRowDouble &y,int n);
   static double     MaxProdNZ(CRowDouble &x,CRowDouble &y,int n);
  };
//+------------------------------------------------------------------+
//| Constants                                                        |
//+------------------------------------------------------------------+
const double CVIPMSolver::m_muquasidense=2.0;
const int CVIPMSolver::m_maxipmits=200;
const double CVIPMSolver::m_initslackval=100.0;
const double CVIPMSolver::m_steplengthdecay=0.95;
const double CVIPMSolver::m_stagnationdelta=0.99999;
const double CVIPMSolver::m_primalinfeasible1=1.0E-3;
const double CVIPMSolver::m_dualinfeasible1=1.0E-3;
const double CVIPMSolver::m_bigy=1.0E8;
const double CVIPMSolver::m_ygrowth=1.0E6;
const int CVIPMSolver::m_itersfortoostringentcond=25;
const int CVIPMSolver::m_minitersbeforedroppingbounds=3;
const int CVIPMSolver::m_minitersbeforeinfeasible=3;
const int CVIPMSolver::m_minitersbeforestagnation=5;
const int CVIPMSolver::m_minitersbeforeeworststagnation=50;
const int CVIPMSolver::m_primalstagnationlen=5;
const int CVIPMSolver::m_dualstagnationlen=7;
const double CVIPMSolver::m_bigconstrxtol=1.0E-5;
const double CVIPMSolver::m_bigconstrmag=1.0E3;
const double CVIPMSolver::m_minitersbeforesafeguards=5;
const double CVIPMSolver::m_badsteplength=1.0E-3;
//+------------------------------------------------------------------+
//| Initializes QP - IPM State and prepares it to receive quadratic/ |
//| linear terms and constraints.                                    |
//| The solver is configured to work internally with dense NxN       |
//| factorization, no matter what exactly is passed - dense or sparse|
//| matrices.                                                        |
//| INPUT PARAMETERS:                                                |
//|   State    -  solver State to be configured; previously allocated|
//|               memory is reused as much as possible               |
//|   S        -  scale vector, array[N]:                            |
//|               * I-th element contains scale of I-th variable,    |
//|               * S[I] > 0                                         |
//|   XOrigin  -  origin term, array[N]. Can be zero. The solver     |
//|               solves problem of the form                         |
//|   >                                                              |
//|   >  min(0.5*(x-x_origin)'*A*(x-x_origin)+b'*(x-x_origin))       |
//|   >                                                              |
//| The terms A and b(as well as constraints) will be specified later|
//| with separate calls.                                             |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMInitDense(CVIPMState &State,
                                CRowDouble &s,
                                CRowDouble &xorigin,
                                int n)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(s,n),__FUNCTION__+": S contains infinite or NaN elements"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(xorigin,n),__FUNCTION__+": XOrigin contains infinite or NaN elements"))
      return;
   VIPMInit(State,s,xorigin,n,n,0);
  }
//+------------------------------------------------------------------+
//| Initializes QP-IPM State and prepares it to receive quadratic/   |
//| linear terms and constraints.                                    |
//| The solver is configured to work internally with dense NxN     |
//| problem divided into two distinct parts-"main" and slack one:  |
//|   * dense quadratic term  is  a  NMain*NMain  matrix(NMain <= N),|
//|     quadratic coefficients are zero for variables outside of     |
//|     [0, NMain) range)                                            |
//|   * linear term is general vector of length N                    |
//|   * linear constraints have special structure for variable with  |
//|     indexes in [NMain, N) range: at most one element per column  |
//|     can be nonzero.                                              |
//| This mode is intended for problems arising during SL1QP nonlinear|
//| programming.                                                     |
//| INPUT PARAMETERS:                                                |
//|   State    -  m_solver  State  to  be configured; previously     |
//|               allocated memory is reused as much as possible     |
//|   S        -  scale vector, array[N]:                            |
//|               * I-th element contains scale of I-th variable,    |
//|               * S[I] > 0                                         |
//|   XOrigin  -  origin term, array[N]. Can be zero. The solver   |
//|               solves problem of the form                         |
//|   >                                                              |
//|   > min(0.5 * (x - x_origin)'*A*(x-x_origin)+b' * (x - x_origin))|
//|   >                                                              |
//| The terms A and b(as well as constraints) will be specified later|
//| with separate calls.                                             |
//|   NMain   - number of "main" variables, 1 <= NMain <= N        |
//|   N        -  total number of variables including slack ones     |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMInitDenseWithSlacks(CVIPMState &State,
                                          CRowDouble &s,
                                          CRowDouble &xorigin,
                                          int nmain,
                                          int n)
  {
//--- check
   if(!CAp::Assert(nmain>=1,__FUNCTION__+": NMain<1"))
      return;
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1"))
      return;
   if(!CAp::Assert(nmain<=n,__FUNCTION__+": NMain>N"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(s,n),__FUNCTION__+": S contains infinite or NaN elements"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(xorigin,n),__FUNCTION__+": XOrigin contains infinite or NaN elements"))
      return;
   VIPMInit(State,s,xorigin,n,nmain,0);
  }
//+------------------------------------------------------------------+
//| Initializes QP-IPM State and prepares it to receive quadratic/   |
//| linear terms and constraints.                                    |
//| The solver is configured to work internally with sparse          |
//| (N + M)x(N + M) factorization no matter what exactly is passed - |
//| dense or sparse matrices. Dense quadratic term will be sparsified|
//| prior to storage.                                                |
//| INPUT PARAMETERS:                                                |
//|   State    -  solver State to be configured; previously allocated|
//|               memory is reused as much as possible               |
//|   S        -  scale vector, array[N]:                            |
//|               * I-th element contains scale of I-th variable,    |
//|               * S[I] > 0                                         |
//|   XOrigin  -  origin term, array[N]. Can be zero. The solver     |
//|               solves problem of the form                         |
//|   >                                                              |
//|   >   min(0.5*(x - x_origin)'*A*(x-x_origin)+b'*(x - x_origin))  |
//|   >                                                              |
//| The terms A and b(as well as constraints) will be specified later|
//| with separate calls.                                             |
//|   N        -  total number of variables, N >= 1                  |
//| This optimization mode assumes that no slack variables is present|
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMInitSparse(CVIPMState &State,
                                 CRowDouble &s,
                                 CRowDouble &xorigin,
                                 int n)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(s,n),__FUNCTION__+": S contains infinite or NaN elements"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(xorigin,n),__FUNCTION__+": XOrigin contains infinite or NaN elements"))
      return;
   VIPMInit(State,s,xorigin,n,n,1);
  }
//+------------------------------------------------------------------+
//| Sets linear / quadratic terms for QP - IPM m_solver              |
//| If you initialized m_solver with VIMPInitDenseWithSlacks(), NMain|
//| below is a number of non-slack variables. In other cases,        |
//| NMain = N.                                                       |
//| INPUT PARAMETERS:                                                |
//|   State    -  instance initialized with one of the initialization|
//|               functions                                          |
//|   DenseH   -  if HKind = 0: array[NMain, NMain], dense quadratic |
//|               term (either upper or lower triangle)              |
//|   SparseH  -  if HKind = 1: array[NMain, NMain], sparse quadratic|
//|               term (either upper or lower triangle)              |
//|   HKind    -  0 or 1, quadratic term format                      |
//|   IsUpper  -  whether dense / sparse H contains lower or upper   |
//|               triangle of the quadratic term                     |
//|   C        -  array[N], linear term                              |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMSetQuadraticLinear(CVIPMState &State,
                                         CMatrixDouble &denseh,
                                         CSparseMatrix &sparseh,
                                         int hkind,
                                         bool IsUpper,
                                         CRowDouble &c)
  {
//--- create variables
   int    nmain=State.m_nmain;
   int    n=State.m_n;
   int    i=0;
   int    j=0;
   int    k=0;
   int    j0=0;
   int    j1=0;
   double v=0;
   double vv=0;
   int    nnz=0;
   int    offs=0;
//--- check
   if(!CAp::Assert(hkind==0 || hkind==1,__FUNCTION__+": incorrect HKind"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(c,n),__FUNCTION__+": C contains infinite or NaN elements"))
      return;
   if(!CAp::Assert(State.m_factorizationtype==0 || State.m_factorizationtype==1,__FUNCTION__+": unexpected factorization type"))
      return;
//--- Set problem Info, reset factorization flag
   State.m_islinear=false;
   State.m_factorizationpresent=false;
   State.m_factorizationpoweredup=false;
//--- Linear term
   State.m_c=c;
   State.m_c.Resize(n);
//--- Quadratic term and normalization
//--- NOTE: we perform integrity check for inifinities/NANs by
//---       computing sum of all matrix elements and checking its
//---       value for being finite. It is a bit faster than checking
//---       each element individually.
   State.m_hkind=-1;
   State.m_targetscale=1.0;
   switch(State.m_factorizationtype)
     {
      case 0:
         //--- Quadratic term is stored in dense format: either copy dense
         //--- term of densify sparse one
         State.m_hkind=0;
         switch(hkind)
           {
            case 0:
               //--- Copy dense quadratic term
               if(IsUpper)
                  State.m_denseh=denseh.Transpose()+0;
               else
                  State.m_denseh=denseh;
               State.m_denseh.Resize(nmain,nmain);
               break;
            case 1:
               //--- Extract sparse quadratic term
               if(!CAp::Assert(sparseh.m_MatrixType==1,__FUNCTION__+": unexpected sparse matrix format"))
                  return;
               if(!CAp::Assert(sparseh.m_M==nmain,__FUNCTION__+": unexpected sparse matrix size"))
                  return;
               if(!CAp::Assert(sparseh.m_N==nmain,__FUNCTION__+": unexpected sparse matrix size"))
                  return;
               State.m_denseh.Resize(nmain,nmain);
               for(i=0; i<nmain; i++)
                  for(j=0; j<=i; j++)
                     State.m_denseh.Set(i,j,0);
               for(i=0; i<nmain; i++)
                 {
                  //--- diagonal element
                  if(sparseh.m_DIdx[i]!=sparseh.m_UIdx[i])
                     State.m_denseh.Set(i,i,sparseh.m_Vals[sparseh.m_DIdx[i]]);
                  //--- Off-diagonal elements
                  if(IsUpper)
                    {
                     //--- superdiagonal elements are moved to subdiagonal part
                     j0=sparseh.m_UIdx[i];
                     j1=sparseh.m_RIdx[i+1];
                     for(j=j0; j<j1; j++)
                        State.m_denseh.Set(sparseh.m_Idx[j],i,sparseh.m_Vals[j]);
                    }
                  else
                    {
                     //--- subdiagonal elements are moved to subdiagonal part
                     j0=sparseh.m_RIdx[i];
                     j1=sparseh.m_DIdx[i];
                     for(j=j0; j<j1; j++)
                        State.m_denseh.Set(i,sparseh.m_Idx[j],sparseh.m_Vals[j]);
                    }
                 }
               break;
           }
         vv=0;
         for(i=0; i<nmain; i++)
           {
            for(j=0; j<=i; j++)
               vv+=State.m_denseh.Get(i,j);
           }
         //--- check
         if(!CAp::Assert(MathIsValidNumber(vv),__FUNCTION__+": DenseH contains infinite or NaN values!"))
            return;
         CLPQPServ::ScaleDenseQPInplace(State.m_denseh,false,nmain,State.m_c,n,State.m_scl);
         State.m_targetscale=CLPQPServ::NormalizeDenseQPInplace(State.m_denseh,false,nmain,State.m_c,n);
         State.m_isdiagonalh=false;
         break;
      case 1:
         //--- check
         if(!CAp::Assert(nmain==n,__FUNCTION__+": critical integrity check failed,NMain!=N"))
            return;
         //--- Quadratic term is stored in sparse format: either sparsify dense
         //--- term or copy the sparse one
         State.m_hkind=1;
         State.m_sparseh.m_MatrixType=1;
         State.m_sparseh.m_M=n;
         State.m_sparseh.m_N=n;
         switch(hkind)
           {
            case 0:
               //--- Sparsify dense term
               nnz=0;
               for(i=0; i<n; i++)
                 {
                  nnz++;
                  if(IsUpper)
                    {
                     j0=i+1;
                     j1=n-1;
                    }
                  else
                    {
                     j0=0;
                     j1=i-1;
                    }
                  for(j=j0; j<=j1; j++)
                     if(denseh.Get(i,j)!=0)
                        nnz++;
                 }
               State.m_sparseh.m_RIdx.Resize(n+1);
               State.m_sparseh.m_Idx.Resize(nnz);
               State.m_sparseh.m_Vals.Resize(nnz);
               State.m_sparseh.m_RIdx.Set(0,0);
               offs=0;
               vv=0;
               for(i=0; i<n; i++)
                 {
                  //--- Off-diagonal elements are copied only when nonzero
                  if(!IsUpper)
                    {
                     for(j=0; j<=i-1; j++)
                       {
                        if(denseh.Get(i,j)!=0)
                          {
                           v=denseh.Get(i,j);
                           State.m_sparseh.m_Idx.Set(offs,j);
                           State.m_sparseh.m_Vals.Set(offs,v);
                           vv+=v;
                           offs++;
                          }
                       }
                    }
                  //--- Diagonal element is always copied
                  v=denseh.Get(i,i);
                  State.m_sparseh.m_Idx.Set(offs,i);
                  State.m_sparseh.m_Vals.Set(offs,v);
                  vv+=v;
                  offs++;
                  //--- Off-diagonal elements are copied only when nonzero
                  if(IsUpper)
                    {
                     for(j=i+1; j<n; j++)
                       {
                        v=denseh.Get(i,j);
                        if(v!=0)
                          {
                           State.m_sparseh.m_Idx.Set(offs,j);
                           State.m_sparseh.m_Vals.Set(offs,v);
                           vv+=v;
                           offs++;
                          }
                       }
                    }
                  //--- Finalize row
                  State.m_sparseh.m_RIdx.Set(i+1,offs);
                 }
               //--- check
               if(!CAp::Assert(MathIsValidNumber(vv),__FUNCTION__+": DenseH contains infinite or NaN values!"))
                  return;
               if(!CAp::Assert(offs==nnz,__FUNCTION__+": integrity check failed"))
                  return;
               CSparse::SparseCreateCRSInplace(State.m_sparseh);
               break;
            case 1:
               //--- Copy sparse quadratic term, but make sure that we have diagonal elements
               //--- present (we add diagonal if it is not present)
               if(!CAp::Assert(sparseh.m_MatrixType==1,__FUNCTION__+": unexpected sparse matrix format"))
                  return;
               if(!CAp::Assert(sparseh.m_M==n,__FUNCTION__+": unexpected sparse matrix size"))
                  return;
               if(!CAp::Assert(sparseh.m_N==n,__FUNCTION__+": unexpected sparse matrix size"))
                  return;
               State.m_sparseh.m_RIdx.Resize(n+1);
               State.m_sparseh.m_Idx.Resize(sparseh.m_RIdx[n]+n);
               State.m_sparseh.m_Vals.Resize(sparseh.m_RIdx[n]+n);
               State.m_sparseh.m_RIdx.Set(0,0);
               offs=0;
               vv=0;
               for(i=0; i<n; i++)
                 {
                  //--- Copy subdiagonal elements (if needed)
                  if(!IsUpper)
                    {
                     j0=sparseh.m_RIdx[i];
                     j1=sparseh.m_DIdx[i]-1;
                     for(k=j0; k<=j1; k++)
                       {
                        v=sparseh.m_Vals[k];
                        State.m_sparseh.m_Idx.Set(offs,sparseh.m_Idx[k]);
                        State.m_sparseh.m_Vals.Set(offs,v);
                        vv+=v;
                        offs++;
                       }
                    }
                  //--- Diagonal element is always copied
                  v=0;
                  if(sparseh.m_UIdx[i]!=sparseh.m_DIdx[i])
                     v=sparseh.m_Vals[sparseh.m_DIdx[i]];
                  State.m_sparseh.m_Idx.Set(offs,i);
                  State.m_sparseh.m_Vals.Set(offs,v);
                  vv+=v;
                  offs++;
                  //--- Copy superdiagonal elements (if needed)
                  if(IsUpper)
                    {
                     j0=sparseh.m_UIdx[i];
                     j1=sparseh.m_RIdx[i+1]-1;
                     for(k=j0; k<=j1; k++)
                       {
                        v=sparseh.m_Vals[k];
                        State.m_sparseh.m_Idx.Set(offs,sparseh.m_Idx[k]);
                        State.m_sparseh.m_Vals.Set(offs,v);
                        vv+=v;
                        offs++;
                       }
                    }
                  //--- Finalize row
                  State.m_sparseh.m_RIdx.Set(i+1,offs);
                 }
               //--- check
               if(!CAp::Assert(MathIsValidNumber(vv),__FUNCTION__+": SparseH contains infinite or NaN values!"))
                  return;
               if(!CAp::Assert(offs<=State.m_sparseh.m_Vals.Size() && offs<=State.m_sparseh.m_Idx.Size(),__FUNCTION__+": integrity check failed"))
                  return;
               CSparse::SparseCreateCRSInplace(State.m_sparseh);
               if(IsUpper)
                 {
                  CSparse::SparseCopyTransposeCRSBuf(State.m_sparseh,State.m_tmpsparse0);
                  CSparse::SparseCopyBuf(State.m_tmpsparse0,State.m_sparseh);
                 }
               break;
           }
         CLPQPServ::ScaleSparseQPInplace(State.m_scl,n,State.m_sparseh,State.m_c);
         State.m_targetscale=CLPQPServ::NormalizeSparseQPInplace(State.m_sparseh,false,State.m_c,n);
         State.m_isdiagonalh=State.m_sparseh.m_RIdx[n]==n;
         break;
     }
   CAp::Assert(State.m_hkind>=0,__FUNCTION__+": integrity check failed");
  }
//+------------------------------------------------------------------+
//| Sets constraints for QP - IPM m_solver                           |
//| INPUT PARAMETERS:                                                |
//|   State    -  instance initialized with one of the initialization|
//|               functions                                          |
//|   BndL, BndU - lower and upper bound. BndL[] can be - INF,       |
//|                                       BndU[] can be + INF.       |
//|   SparseA  -  sparse constraint matrix, CRS format               |
//|   MSparse  -  number of sparse constraints                       |
//|   DenseA   -  array[MDense, N], dense part of the constraints    |
//|   MDense   -  number of dense constraints                        |
//|   CL, CU   -  lower and upper bounds for constraints, first      |
//| MSparse are bounds for sparse part, following MDense ones are    |
//| bounds for dense part, MSparse + MDense in total.                |
//|            - INF <= CL[I] <= CU[I] <= +INF.                      |
//| This function throws exception if constraints have inconsistent  |
//| bounds, i.e. either BndL[I] > BndU[I] or CL[I] > CU[I]. In all   |
//| other cases it succeeds.                                         |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMSetConstraints(CVIPMState &State,
                                     CRowDouble &bndl,
                                     CRowDouble &bndu,
                                     CSparseMatrix &sparsea,
                                     int msparse,
                                     CMatrixDouble &densea,
                                     int mdense,
                                     CRowDouble &cl,
                                     CRowDouble &cu)
  {
//--- create variables
   int    n=State.m_n;
   int    nmain=State.m_nmain;
   int    nslack=n-nmain;
   int    m=0;
   int    i=0;
   int    j=0;
   int    j0=0;
   int    j1=0;
   int    k=0;
   int    offsmain=0;
   int    offscombined=0;
   double vs=0;
   double v=0;
//--- check
   if(!CAp::Assert(mdense>=0,__FUNCTION__+": MDense<0"))
      return;
   if(!CAp::Assert(msparse>=0,__FUNCTION__+": MSparse<0"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteMatrix(densea,mdense,n),__FUNCTION__+": DenseA contains infinite or NaN values!"))
      return;
   if(!CAp::Assert(msparse==0 || sparsea.m_MatrixType==1,__FUNCTION__+": non-CRS constraint matrix!"))
      return;
   if(!CAp::Assert(msparse==0 || (sparsea.m_M==msparse && sparsea.m_N==n),__FUNCTION__+": constraint matrix has incorrect size"))
      return;
   if(!CAp::Assert(cl.Size()>=mdense+msparse,__FUNCTION__+": CL is too short!"))
      return;
   if(!CAp::Assert(cu.Size()>=mdense+msparse,__FUNCTION__+": CU is too short!"))
      return;
//--- Reset factorization flag
   State.m_factorizationpresent=false;
   State.m_factorizationpoweredup=false;
//--- Box constraints
   State.m_bndl.Resize(n);
   State.m_bndu.Resize(n);
   State.m_rawbndl.Resize(n);
   State.m_rawbndu.Resize(n);
   ArrayResize(State.m_HasBndL,n);
   ArrayResize(State.m_HasBndU,n);
   for(i=0; i<n; i++)
     {
      State.m_HasBndL[i]=MathIsValidNumber(bndl[i]);
      State.m_HasBndU[i]=MathIsValidNumber(bndu[i]);
      //--- check
      if(!CAp::Assert(!(State.m_HasBndL[i] && State.m_HasBndU[i] && bndl[i]>bndu[i]),__FUNCTION__+": inconsistent range for box constraints"))
         return;
      State.m_bndl.Set(i,bndl[i]);
      State.m_bndu.Set(i,bndu[i]);
      State.m_rawbndl.Set(i,bndl[i]);
      State.m_rawbndu.Set(i,bndu[i]);
     }
   CLPQPServ::ScaleShiftBCInplace(State.m_scl,State.m_xorigin,State.m_bndl,State.m_bndu,n);
//--- Linear constraints (full matrices)
   m=mdense+msparse;
   State.m_b.Resize(m);
   State.m_r.Resize(m);
   State.m_ascales.Resize(m);
   ArrayResize(State.m_aflips,m);
   ArrayResize(State.m_hasr,m);
   if(msparse>0)
      CSparse::SparseCopyToCRSBuf(sparsea,State.m_sparseafull);
   if(mdense>0)
      State.m_denseafull=densea;
   State.m_denseafull.Resize(mdense,n);
   for(i=0; i<m; i++)
     {
      //--- check
      if(!CAp::Assert(MathIsValidNumber(cl[i]) || AL_NEGINF==cl[i],__FUNCTION__+": CL is not finite number or -INF"))
         return;
      if(!CAp::Assert(MathIsValidNumber(cu[i]) || AL_POSINF==cu[i],__FUNCTION__+": CU is not finite number or +INF"))
         return;
      //--- Store range
      if(MathIsValidNumber(cl[i]) || MathIsValidNumber(cu[i]))
        {
         //--- Non-degenerate constraint, at least one of bounds is present
         if(MathIsValidNumber(cl[i]))
           {
            if(!CAp::Assert(!MathIsValidNumber(cu[i]) || cu[i]>=cl[i],__FUNCTION__+": inconsistent range (right-hand side) for linear constraint"))
               return;
            if(MathIsValidNumber(cu[i]))
              {
               //--- We have both CL and CU, i.e. CL <= A*x <= CU.
               //--- It can be either equality constraint (no slacks) or range constraint
               //--- (two pairs of slacks variables).
               //--- Try to arrange things in such a way that |CU|>=|CL| (it can be done
               //--- by multiplication by -1 and boundaries swap).
               //--- Having |CU|>=|CL| will allow us to drop huge irrelevant bound CU,
               //--- if we find it irrelevant during computations. Due to limitations
               //--- of our slack variable substitution, it can be done only for CU.
               if(MathAbs(cu[i])>=MathAbs(cl[i]))
                 {
                  State.m_b.Set(i,cl[i]);
                  State.m_r.Set(i,cu[i]-cl[i]);
                  State.m_hasr[i]=true;
                  State.m_aflips[i]=false;
                  vs=1;
                 }
               else
                 {
                  State.m_b.Set(i,-cu[i]);
                  State.m_r.Set(i,cu[i]-cl[i]);
                  State.m_hasr[i]=true;
                  State.m_aflips[i]=true;
                  vs=-1;
                 }
              }
            else
              {
               //--- Only lower bound: CL <= A*x.
               //--- One pair of slack variables added.
               State.m_b.Set(i,cl[i]);
               State.m_r.Set(i,AL_POSINF);
               State.m_hasr[i]=false;
               State.m_aflips[i]=false;
               vs=1;
              }
           }
         else
           {
            //--- Only upper bound: A*x <= CU
            //--- One pair of slack variables added.
            State.m_b.Set(i,-cu[i]);
            State.m_r.Set(i,AL_POSINF);
            State.m_hasr[i]=false;
            State.m_aflips[i]=true;
            vs=-1;
           }
        }
      else
        {
         //--- Degenerate constraint -inf <= Ax <= +inf.
         //--- Generate dummy formulation.
         State.m_b.Set(i,-1);
         State.m_r.Set(i,2);
         State.m_hasr[i]=true;
         State.m_aflips[i]=false;
         vs=0;
        }
      //--- Store matrix row and its scaling coefficient
      if(i<msparse)
        {
         j0=State.m_sparseafull.m_RIdx[i];
         j1=State.m_sparseafull.m_RIdx[i+1];
         for(j=j0; j<j1; j++)
            State.m_sparseafull.m_Vals.Mul(j,vs);
        }
      else
         State.m_denseafull.Row(i-msparse,State.m_denseafull[i-msparse]*vs);
      State.m_ascales.Set(i,vs);
     }
   CLPQPServ::ScaleShiftMixedBRLCInplace(State.m_scl,State.m_xorigin,n,State.m_sparseafull,msparse,State.m_denseafull,mdense,State.m_b,State.m_r);
   CLPQPServ::NormalizeMixedBRLCInplace(State.m_sparseafull,msparse,State.m_denseafull,mdense,State.m_b,State.m_r,n,true,State.m_tmp0,true);
   State.m_ascales*=State.m_tmp0;
   State.m_mdense=mdense;
   State.m_msparse=msparse;
//--- Separate main and slack parts of the constraint matrices
   State.m_tmpi.Resize(nslack);
   State.m_tmpi.Fill(0);
   State.m_combinedaslack.m_M=mdense+msparse;
   State.m_combinedaslack.m_N=nslack;
   State.m_combinedaslack.m_RIdx.Resize(mdense+msparse+1);
   State.m_combinedaslack.m_Idx.Resize(nslack);
   State.m_combinedaslack.m_Vals.Resize(nslack);
   State.m_combinedaslack.m_RIdx.Set(0,0);
   State.m_sparseamain.m_M=msparse;
   State.m_sparseamain.m_N=nmain;
   if(msparse>0)
     {
      State.m_sparseamain.m_RIdx.Resize(msparse+1);
      State.m_sparseamain.m_Idx.Resize(sparsea.m_RIdx[msparse]);
      State.m_sparseamain.m_Vals.Resize(sparsea.m_RIdx[msparse]);
      State.m_sparseamain.m_RIdx.Set(0,0);
      for(i=0; i<msparse; i++)
        {
         offsmain=State.m_sparseamain.m_RIdx[i];
         offscombined=State.m_combinedaslack.m_RIdx[i];
         j0=State.m_sparseafull.m_RIdx[i];
         j1=State.m_sparseafull.m_RIdx[i+1];
         for(j=j0; j<j1; j++)
           {
            v=State.m_sparseafull.m_Vals[j];
            k=State.m_sparseafull.m_Idx[j];
            if(k<nmain)
              {
               State.m_sparseamain.m_Idx.Set(offsmain,k);
               State.m_sparseamain.m_Vals.Set(offsmain,v);
               offsmain++;
              }
            else
              {
               //--- check
               if(!CAp::Assert(State.m_tmpi[k-nmain]==0,__FUNCTION__+": slack column contains more than one nonzero element"))
                  return;
               State.m_combinedaslack.m_Idx.Set(offscombined,k-nmain);
               State.m_combinedaslack.m_Vals.Set(offscombined,v);
               State.m_tmpi.Add(k-nmain,1);
               offscombined++;
              }
           }
         State.m_sparseamain.m_RIdx.Set(i+1,offsmain);
         State.m_combinedaslack.m_RIdx.Set(i+1,offscombined);
        }
     }
   CSparse::SparseCreateCRSInplace(State.m_sparseamain);
   if(mdense>0)
     {
      State.m_denseamain=State.m_denseafull;
      State.m_denseamain.Resize(mdense,nmain);
      for(i=0; i<mdense; i++)
        {
         offscombined=State.m_combinedaslack.m_RIdx[msparse+i];
         for(k=nmain; k<n; k++)
           {
            if(State.m_denseafull.Get(i,k)!=0)
              {
               //--- check
               if(!CAp::Assert(State.m_tmpi[k-nmain]==0,__FUNCTION__+": slack column contains more than one nonzero element"))
                  return;
               State.m_combinedaslack.m_Idx.Set(offscombined,k-nmain);
               State.m_combinedaslack.m_Vals.Set(offscombined,State.m_denseafull.Get(i,k));
               State.m_tmpi.Add(k-nmain,1);
               offscombined++;
              }
           }
         State.m_combinedaslack.m_RIdx.Set(msparse+i+1,offscombined);
        }
     }
   CSparse::SparseCreateCRSInplace(State.m_combinedaslack);
  }
//+------------------------------------------------------------------+
//| Sets stopping criteria for QP - IPM m_solver.                    |
//| You can set all epsilon-values to one small value, about 1.0E-6. |
//| INPUT PARAMETERS:                                                |
//|   State       -  instance initialized with one of the            |
//|                  initialization functions                        |
//|   EpsP        -  maximum primal error allowed in the  solution,  |
//|                  EpsP >= 0. Zero will be  automatically replaced |
//|                  by recommended default value,  which is  equal  |
//|                  to 10 * Sqrt(Epsilon) in the current version    |
//|   EpsD        -  maximum  dual  error allowed in the  solution,  |
//|                  EpsP >= 0. Zero will be  automatically replaced |
//|                  by recommended default value,  which is  equal  |
//|                  to 10 * Sqrt(Epsilon) in the current version    |
//|   EpsGap      -  maximum  duality gap allowed in the  solution,  |
//|                  EpsP >= 0. Zero will be  automatically replaced |
//|                  by recommended default value,  which is  equal  |
//|                  to 10 * Sqrt(Epsilon) in the current version    |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMSetCond(CVIPMState &State,
                              double epsp,
                              double epsd,
                              double epsgap)
  {
//--- check
   if(!CAp::Assert(MathIsValidNumber(epsp) && epsp>=0.0,__FUNCTION__+": EpsP is infinite or negative"))
      return;
   if(!CAp::Assert(MathIsValidNumber(epsd) && epsd>=0.0,__FUNCTION__+": EpsD is infinite or negative"))
      return;
   if(!CAp::Assert(MathIsValidNumber(epsgap) && epsgap>=0.0,__FUNCTION__+": EpsP is infinite or negative"))
      return;

   double sml=MathSqrt(CMath::m_machineepsilon);
   State.m_epsp=CApServ::Coalesce(epsp,sml);
   State.m_epsd=CApServ::Coalesce(epsd,sml);
   State.m_epsgap=CApServ::Coalesce(epsgap,sml);
  }
//+------------------------------------------------------------------+
//| Solve QP problem.                                                |
//| INPUT PARAMETERS:                                                |
//|   State    -  solver instance                                    |
//|   DropBigBounds - If True, algorithm may drop box and linear     |
//|               constraints with huge bound values that destabilize|
//|               algorithm.                                         |
//| OUTPUT PARAMETERS:                                               |
//|   XS       -  array[N], solution                                 |
//|   LagBC    -  array[N], Lagrange multipliers for box constraints |
//|   LagLC    -  array[M], Lagrange multipliers for linear          |
//|               constraints                                        |
//|   TerminationType - completion code, positive values for success,|
//|               negative for failures(XS constrains best point     |
//|               found so far):                                     |
//|               * -2  the task is either unbounded or infeasible;  |
//|                     the IPM solver has difficulty distinguishing |
//|                     between these two.                           |
//|               * +1  stopping criteria are met                    |
//|               * +7  stopping criteria are too stringent          |
//| RESULT:                                                          |
//|   This function ALWAYS returns something meaningful in XS, LagBC,|
//|   LagLC - either solution or the best point so far, even for     |
//|   negative TerminationType.                                      |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMOptimize(CVIPMState &State,
                               bool dropbigbounds,
                               CRowDouble &xs,
                               CRowDouble &lagbc,
                               CRowDouble &laglc,
                               int &terminationtype)
  {
//--- create variables
   int    n=State.m_n;
   int    m=State.m_mdense+State.m_msparse;
   int    i=0;
   int    iteridx=0;
   double mu=0;
   double muaff=0;
   double sigma=0;
   double alphaaffp=0;
   double alphaaffd=0;
   double alphap=0;
   double alphad=0;
   int    primalstagnationcnt=0;
   int    dualstagnationcnt=0;
   double regeps=0;
   double dampeps=0;
   double safedampeps=0;
   double modeps=0;
   double maxdampeps=0;
   double regfree=0;
   double dampfree=0;
   int    droppedbounds=0;
   double primalxscale=0;
   double errp2=0;
   double errd2=0;
   double errpinf=0;
   double errdinf=0;
   double preverrp2=0;
   double preverrd2=0;
   double errgap=0;
   double eprimal=0;
   double edual=0;
   double egap=0;
   double mumin=0;
   double mustop=0;
   double y0nrm=0;
   double bady=0;
   double mxprimal=0;
   double mxdeltaprimal=0;
   int    bestiteridx=0;
   double besterr=0;
   double bestegap=0;
   double besteprimal=0;
   double bestedual=0;
   bool   loadbest=false;

   terminationtype=0;
   State.m_dotrace=CAp::IsTraceEnabled("IPM");
   State.m_dodetailedtrace=State.m_dotrace && CAp::IsTraceEnabled("IPM.DETAILED");
//--- Prepare outputs
   xs=vector<double>::Zeros(n);
   lagbc=vector<double>::Zeros(n);
   laglc=vector<double>::Zeros(m);
//--- Some integrity checks:
//--- * we need PrimalStagnationLen<DualStagnationLen in order to be able to correctly
//---   detect infeasible instances (stagnated dual error is present in both infeasible
//---   and unbounded instances, so we should check for primal stagnation a few iters
//---   before checking for dual stagnation)
   if(!CAp::Assert(m_primalstagnationlen<m_dualstagnationlen,__FUNCTION__+": critical integrity failure - incorrect configuration parameters"))
      return;
//--- Trace output (if needed)
   if(State.m_dotrace)
     {
      CAp::Trace("\n\n");
      CAp::Trace("////////////////////////////////////////////////////////////////////////////////////////////////////\n");
      CAp::Trace("//--- IPM SOLVER STARTED                                                                             //\n");
      CAp::Trace("////////////////////////////////////////////////////////////////////////////////////////////////////\n");
     }
//--- Prepare regularization coefficients:
//--- * RegEps - one that is applied to initial (5N+5M)x(5N+5M) KKT system. This one has to be
//---   small because it perturbs solution returned by the algorithm. Essential in order to
//---   avoid stalling at extremely large points.
//--- * ModEps - small modification applied to LDLT decomposition in order to preserve sign
//---   of diagonal elements
//--- * DampEps - damping coefficient for damped Newton step. Comes along with SafeDampEps
//---   (threshold value when some safeguards are turned off in order to preserve convergence
//---   speed) and MaxDampEps - threshold value when we consider problem overregularized and stop.
//--- * DampFree - additional damping coefficient for free variables
   regfree=MathPow(CMath::m_machineepsilon,0.75);
   dampfree=0;
   regeps=100*CMath::m_machineepsilon;
   modeps=(100+MathSqrt(n))*CMath::m_machineepsilon;
   dampeps=(100+MathSqrt(n))*CMath::m_machineepsilon;
   safedampeps=MathSqrt(CMath::m_machineepsilon);
   maxdampeps=MathSqrt(MathSqrt(CMath::m_machineepsilon));
//--- Set up initial State
   State.m_repiterationscount=0;
   State.m_repncholesky=0;
   mustop=(100+MathSqrt(n))*CMath::m_machineepsilon;
   mumin=0.01*mustop;
   VIPMPowerUp(State,regfree);
   VARSInitFrom(State.m_best,State.m_current);
   VARSInitByZero(State.m_zerovars,n,m);
   VARSInitByZero(State.m_deltaaff,n,m);
   VARSInitByZero(State.m_deltacorr,n,m);
   bestiteridx=-1;
   besterr=CMath::m_maxrealnumber;
   bestegap=CMath::m_maxrealnumber;
   besteprimal=CMath::m_maxrealnumber;
   bestedual=CMath::m_maxrealnumber;
   TraceProgress(State,0.0,0.0,0.0,0.0,0.0);
   y0nrm=0;
   y0nrm=MathMax(y0nrm,CAblasF::RMaxAbsV(m,State.m_current.m_y));
   y0nrm=MathMax(y0nrm,CAblasF::RMaxAbsV(m,State.m_current.m_v));
   y0nrm=MathMax(y0nrm,CAblasF::RMaxAbsV(m,State.m_current.m_q));
   y0nrm=MathMax(y0nrm,CAblasF::RMaxAbsV(n,State.m_current.m_z));
   y0nrm=MathMax(y0nrm,CAblasF::RMaxAbsV(n,State.m_current.m_s));
//--- Start iteration
   loadbest=true;
   primalstagnationcnt=0;
   dualstagnationcnt=0;
   terminationtype=7;
   errp2=CMath::m_maxrealnumber;
   errd2=CMath::m_maxrealnumber;
   for(iteridx=0; iteridx<m_maxipmits; iteridx++)
     {
      //--- Trace beginning
      if(State.m_dotrace)
         CAp::Trace(StringFormat("=== PREDICTOR-CORRECTOR STEP %2d ====================================================================\n",iteridx));
      //--- Check regularization status, terminate if overregularized
      if(dampeps>=maxdampeps)
        {
         if(State.m_dotrace)
           {
            CAp::Trace("> tried to increase regularization parameter,but it is too large\n");
            CAp::Trace("> it is likely that stopping conditions are too stringent,stopping at the best point found so far\n");
           }
         terminationtype=7;
         break;
        }
      //--- Precompute factorization
      //--- NOTE: we use "m_solver" regularization coefficient at this moment
      if(!VIPMPrecomputeNewtonFactorization(State,State.m_current,regeps,modeps,dampeps,dampfree))
        {
         //--- KKT factorization failed.
         //--- Increase regularization parameter and skip this iteration.
         dampeps*=10;
         if(State.m_dotrace)
           {
            CAp::Trace("> LDLT factorization failed due to rounding errors\n");
            CAp::Trace(StringFormat("> increasing damping coefficient to %.2E,skipping iteration\n",dampeps));
           }
         continue;
        }
      //--- Compute Mu
      mu=VARSComputeMu(State,State.m_current);
      //--- Compute affine scaling step for Mehrotra's predictor-corrector algorithm
      if(!VIPMComputeStepDirection(State,State.m_current,0.0,State.m_zerovars,State.m_deltaaff,regeps,dampeps>=safedampeps))
        {
         //--- Affine scaling step failed due to numerical errors.
         //--- Increase regularization parameter and skip this iteration.
         dampeps*=10;
         if(State.m_dotrace)
           {
            CAp::Trace("> affine scaling step failed to decrease residual due to rounding errors\n");
            CAp::Trace(StringFormat("> increasing damping coefficient to %.2E,skipping iteration\n",dampeps));
           }
         continue;
        }
      VIPMComputeStepLength(State,State.m_current,State.m_deltaaff,m_steplengthdecay,alphaaffp,alphaaffd);
      //--- Compute MuAff and centering parameter
      VARSInitFrom(State.m_trial,State.m_current);
      VarsAddStep(State.m_trial,State.m_deltaaff,alphaaffp,alphaaffd);
      muaff=VARSComputeMu(State,State.m_trial);
      sigma=MathMin(MathPow((muaff+mumin)/(mu+mumin),3),1.0);
      //--- check
      if(!CAp::Assert(MathIsValidNumber(sigma) && sigma<=1.0,__FUNCTION__+": critical integrity check failed for Sigma (infinite or greater than 1)"))
         return;
      //--- Compute corrector step
      if(!VIPMComputeStepDirection(State,State.m_current,sigma*mu+mumin,State.m_deltaaff,State.m_deltacorr,regeps,dampeps>=safedampeps))
        {
         //--- Affine scaling step failed due to numerical errors.
         //--- Increase regularization parameter and skip this iteration.
         dampeps*=10;
         if(State.m_dotrace)
           {
            CAp::Trace("> corrector step failed to decrease residual due to rounding errors\n");
            CAp::Trace(StringFormat("> increasing damping coefficient to %.2E,skipping iteration\n",dampeps));
           }
         continue;
        }
      VIPMComputeStepLength(State,State.m_current,State.m_deltacorr,m_steplengthdecay,alphap,alphad);
      if((double)(iteridx)>=m_minitersbeforesafeguards && (alphap<=m_badsteplength || alphad<=m_badsteplength))
        {
         //--- Affine scaling step failed due to numerical errors.
         //--- Increase regularization parameter and skip this iteration.
         dampeps*=10;
         if(State.m_dotrace)
           {
            CAp::Trace("> step length is too short,suspecting rounding errors\n");
            CAp::Trace(StringFormat("> increasing damping coefficient to %.2E,skipping iteration\n",dampeps));
           }
         continue;
        }
      //--- Perform a step
      RunIntegrityChecks(State,State.m_current,State.m_deltacorr,alphap,alphad);
      VIPMPerformStep(State,alphap,alphad);
      TraceProgress(State,mu,muaff,sigma,alphap,alphad);
      //--- Check for excessive bounds (one that are so large that they are both irrelevant
      //--- and destabilizing due to their magnitude)
      if(dropbigbounds && iteridx>=m_minitersbeforedroppingbounds)
        {
         //--- check
         if(!CAp::Assert((10.0*m_bigconstrmag)<=(1.0/m_bigconstrxtol),__FUNCTION__+": integrity check failed (incorrect BigConstr Settings)"))
            return;
         droppedbounds=0;
         //--- Determine variable and step scales.
         //--- Both quantities are bounded from below by 1.0
         mxprimal=1.0;
         mxprimal=MathMax(mxprimal,CAblasF::RMaxAbsV(n,State.m_current.m_x));
         mxprimal=MathMax(mxprimal,CAblasF::RMaxAbsV(n,State.m_current.m_g));
         mxprimal=MathMax(mxprimal,CAblasF::RMaxAbsV(n,State.m_current.m_t));
         mxprimal=MathMax(mxprimal,CAblasF::RMaxAbsV(m,State.m_current.m_w));
         mxprimal=MathMax(mxprimal,CAblasF::RMaxAbsV(m,State.m_current.m_p));
         mxdeltaprimal=1.0;
         mxdeltaprimal=MathMax(mxdeltaprimal,alphap*CAblasF::RMaxAbsV(n,State.m_deltacorr.m_x));
         mxdeltaprimal=MathMax(mxdeltaprimal,alphap*CAblasF::RMaxAbsV(n,State.m_deltacorr.m_g));
         mxdeltaprimal=MathMax(mxdeltaprimal,alphap*CAblasF::RMaxAbsV(n,State.m_deltacorr.m_t));
         mxdeltaprimal=MathMax(mxdeltaprimal,alphap*CAblasF::RMaxAbsV(m,State.m_deltacorr.m_w));
         mxdeltaprimal=MathMax(mxdeltaprimal,alphap*CAblasF::RMaxAbsV(m,State.m_deltacorr.m_p));
         //--- If changes in primal variables are small enough, try dropping too large bounds
         if(mxdeltaprimal<(mxprimal*m_bigconstrxtol))
           {
            //--- Drop irrelevant box constraints
            primalxscale=1.0;
            primalxscale=MathMax(primalxscale,CAblasF::RMaxAbsV(n,State.m_current.m_x));
            for(i=0; i<n; i++)
              {
               if((State.m_HasBndL[i] && State.m_hasgz[i]) && MathAbs(State.m_bndl[i])>(m_bigconstrmag*primalxscale))
                 {
                  State.m_hasgz[i]=false;
                  State.m_current.m_g.Set(i,0);
                  State.m_current.m_z.Set(i,0);
                  State.m_cntgz--;
                  droppedbounds++;
                 }
               if((State.m_HasBndU[i] && State.m_hasts[i]) && MathAbs(State.m_bndu[i])>(m_bigconstrmag*primalxscale))
                 {
                  State.m_hasts[i]=false;
                  State.m_current.m_t.Set(i,0);
                  State.m_current.m_s.Set(i,0);
                  State.m_cntts--;
                  droppedbounds++;
                 }
              }
            //--- Drop irrelevant linear constraints. Due to specifics of the m_solver
            //--- we can drop only right part part of b<=Ax<=b+r.
            //--- We can't drop b<=A from b<=A<=b+r because it impossible with our choice of
            //--- slack variables. Usually we do not need to do so because we reorder constraints
            //--- during initialization in such a way that |b+r|>|b| and because typical
            //--- applications do not have excessively large lower AND upper bound (user may
            //--- specify large value for 'absent' bound, but usually he does not mark both bounds as absent).
            MultiplyGEAX(State,1.0,State.m_current.m_x,0,0.0,State.m_tmpax,0);
            primalxscale=1.0;
            primalxscale=MathMax(primalxscale,CAblasF::RMaxAbsV(n,State.m_current.m_x));
            primalxscale=MathMax(primalxscale,CAblasF::RMaxAbsV(m,State.m_tmpax));
            for(i=0; i<m; i++)
              {
               if(State.m_hasr[i] && State.m_haspq[i] && MathAbs(State.m_b[i]+State.m_r[i])>(m_bigconstrmag*primalxscale) && MathAbs(State.m_b[i])<(m_bigconstrmag*primalxscale))
                 {
                  //--- check
                  if(!CAp::Assert(State.m_haswv[i] && State.m_haspq[i],__FUNCTION__+": unexpected integrity check failure (4y64)"))
                     return;
                  State.m_haspq[i]=false;
                  State.m_current.m_p.Set(i,0);
                  State.m_current.m_q.Set(i,0);
                  State.m_cntpq--;
                  droppedbounds++;
                 }
              }
            //--- Trace output
            if(droppedbounds>0)
               if(State.m_dotrace)
                  CAp::Trace(StringFormat("[NOTICE] detected %d irrelevant constraints with huge bounds,X converged to values well below them,dropping...\n",droppedbounds));
           }
        }
      //--- Check stopping criteria
      //--- * primal and dual stagnation are checked only when following criteria are met:
      //---   1) Mu is smaller than 1 (we already converged close enough)
      //---   2) we performed more than MinItersBeforeStagnation iterations
      preverrp2=errp2;
      preverrd2=errd2;
      ComputeErrors(State,errp2,errd2,errpinf,errdinf,errgap);
      mu=VARSComputeMu(State,State.m_current);
      egap=errgap;
      eprimal=errpinf;
      edual=errdinf;
      if(MathMax(egap,MathMax(eprimal,edual))<besterr)
        {
         //--- Save best point found so far
         VARSInitFrom(State.m_best,State.m_current);
         bestiteridx=iteridx;
         besterr=MathMax(egap,MathMax(eprimal,edual));
         bestegap=egap;
         besteprimal=eprimal;
         bestedual=edual;
        }
      if(bestiteridx>0 && iteridx>bestiteridx+m_minitersbeforeeworststagnation)
        {
         if(State.m_dotrace)
            CAp::Trace(StringFormat("> worst of primal/dual/gap errors stagnated for %d its,stopping at the best point found so far\n",m_minitersbeforeeworststagnation));
         break;
        }
      if(egap<=State.m_epsgap && errp2>=(m_stagnationdelta*preverrp2) && errpinf>=m_primalinfeasible1 && iteridx>=m_minitersbeforestagnation)
        {
         primalstagnationcnt++;
         if(primalstagnationcnt>=m_primalstagnationlen)
           {
            if(State.m_dotrace)
               CAp::Trace(StringFormat("> primal error stagnated for %d its,stopping at the best point found so far\n",m_primalstagnationlen));
            break;
           }
        }
      else
         primalstagnationcnt=0;
      if(egap<=State.m_epsgap && errd2>=(m_stagnationdelta*preverrd2) && errdinf>=m_dualinfeasible1 && iteridx>=m_minitersbeforestagnation)
        {
         dualstagnationcnt++;
         if(dualstagnationcnt>=m_dualstagnationlen)
           {
            if(State.m_dotrace)
               CAp::Trace(StringFormat("> dual error stagnated for %d its,stopping at the best point found so far\n",m_dualstagnationlen));
            break;
           }
        }
      else
         dualstagnationcnt=0;
      if(mu<=mustop && iteridx>=m_itersfortoostringentcond)
        {
         if(State.m_dotrace)
            CAp::Trace("> stopping conditions are too stringent,stopping at the best point found so far\n");
         terminationtype=7;
         break;
        }
      if(egap<=State.m_epsgap && eprimal<=State.m_epsp && edual<=State.m_epsd)
        {
         if(State.m_dotrace)
            CAp::Trace("> stopping criteria are met\n");
         terminationtype=1;
         loadbest=false;
         break;
        }
      bady=m_bigy;
      bady=MathMax(bady,m_ygrowth*y0nrm);
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(n,State.m_current.m_x));
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(n,State.m_current.m_g));
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(n,State.m_current.m_t));
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(m,State.m_current.m_w));
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(m,State.m_current.m_p));
      if(CAblasF::RMaxAbsV(m,State.m_current.m_y)>=bady && iteridx>=m_minitersbeforeinfeasible)
        {
         if(State.m_dotrace)
            CAp::Trace(StringFormat("> |Y| increased beyond %.1E,stopping at the best point found so far\n",bady));
         break;
        }
     }
//--- Load best point, perform some checks
   if(loadbest)
     {
      //--- Load best point
      //--- NOTE: TouchReal() is used to avoid spurious compiler warnings about 'set but unused'
      if(State.m_dotrace)
         CAp::Trace(StringFormat("> the best point so far is one from iteration %d\n",bestiteridx));
      VARSInitFrom(State.m_current,State.m_best);
      //--- If no error flags were set yet, check solution quality
      bady=m_bigy;
      bady=MathMax(bady,m_ygrowth*y0nrm);
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(n,State.m_current.m_x));
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(n,State.m_current.m_g));
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(n,State.m_current.m_t));
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(m,State.m_current.m_w));
      bady=MathMax(bady,m_ygrowth*CAblasF::RMaxAbsV(m,State.m_current.m_p));
      if(terminationtype>0 && CAblasF::RMaxAbsV(m,State.m_current.m_y)>=bady)
        {
         terminationtype=-2;
         if(State.m_dotrace)
            CAp::Trace(StringFormat("> |Y| increased beyond %.1E,declaring infeasibility/unboundedness\n",bady));
        }
      if(terminationtype>0 && besteprimal>=m_primalinfeasible1)
        {
         terminationtype=-2;
         if(State.m_dotrace)
            CAp::Trace("> primal error at the best point is too high,declaring infeasibility/unboundedness\n");
        }
      if(terminationtype>0 && bestedual>=m_dualinfeasible1)
        {
         terminationtype=-2;
         if(State.m_dotrace)
            CAp::Trace("> dual error at the best point is too high,declaring infeasibility/unboundedness\n");
        }
     }
//--- Output
   MultiplyHX(State,State.m_current.m_x,State.m_tmp0);
   CAblasF::RAddV(n,1.0,State.m_c,State.m_tmp0);
   MultiplyGEATX(State,-1.0,State.m_current.m_y,0,1.0,State.m_tmp0,0);
   for(i=0; i<n; i++)
     {
      if(State.m_isfrozen[i])
        {
         //--- I-th variable is frozen, use its frozen value.
         //--- By the definition, I-th Lagrangian multiplier is an I-th component of Lagrangian gradient
         xs.Set(i,State.m_current.m_x[i]);
         lagbc.Set(i,-State.m_tmp0[i]);
        }
      else
        {
         xs.Set(i,State.m_current.m_x[i]);
         lagbc.Set(i,0.0);
         if(State.m_hasgz[i])
            lagbc.Add(i,- State.m_current.m_z[i]);
         if(State.m_hasts[i])
            lagbc.Add(i,State.m_current.m_s[i]);
        }
     }
   laglc=State.m_current.m_y*(-1.0)+0;
//--- Unscale point and Lagrange multipliers
   CLPQPServ::UnscaleUnshiftPointBC(State.m_scl,State.m_xorigin,State.m_rawbndl,State.m_rawbndu,State.m_bndl,State.m_bndu,State.m_HasBndL,State.m_HasBndU,xs,n);
   lagbc*=State.m_scl.Pow(-1.0)*State.m_targetscale;
   for(i=0; i<m; i++)
      laglc.Mul(i,State.m_targetscale/CApServ::Coalesce(State.m_ascales[i],1.0));
  }
//+------------------------------------------------------------------+
//| Allocates place for variables of IPM and fills by zeros.         |
//+------------------------------------------------------------------+
void CVIPMSolver::VARSInitByZero(CVIPMVars &vstate,
                                 int n,
                                 int m)
  {
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1"))
      return;
   if(!CAp::Assert(m>=0,__FUNCTION__+": M<0"))
      return;
   vstate.m_n=n;
   vstate.m_m=m;
   vstate.m_x=vector<double>::Zeros(n);
   vstate.m_g=vector<double>::Zeros(n);
   vstate.m_t=vector<double>::Zeros(n);
   vstate.m_z=vector<double>::Zeros(n);
   vstate.m_s=vector<double>::Zeros(n);
   vstate.m_y=vector<double>::Zeros(m);
   vstate.m_w=vector<double>::Zeros(m);
   vstate.m_p=vector<double>::Zeros(m);
   vstate.m_v=vector<double>::Zeros(m);
   vstate.m_q=vector<double>::Zeros(m);
  }
//+------------------------------------------------------------------+
//| Allocates place for variables of IPM and fills them by values of |
//| the source                                                       |
//+------------------------------------------------------------------+
void CVIPMSolver::VARSInitFrom(CVIPMVars &vstate,
                               CVIPMVars &vsrc)
  {
//--- check
   if(!CAp::Assert(vsrc.m_n>=1,__FUNCTION__+": N<1"))
      return;
   if(!CAp::Assert(vsrc.m_m>=0,__FUNCTION__+": M<0"))
      return;
//--- copy
   vstate=vsrc;
  }
//+------------------------------------------------------------------+
//| Adds to variables direction vector times step length. Different  |
//| lengths are used for primal and dual steps.                      |
//+------------------------------------------------------------------+
void CVIPMSolver::VarsAddStep(CVIPMVars &vstate,
                              CVIPMVars &vdir,
                              double stpp,
                              double stpd)
  {
//--- create variables
   int n=vstate.m_n;
   int m=vstate.m_m;
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1"))
      return;
   if(!CAp::Assert(m>=0,__FUNCTION__+": M<0"))
      return;
   if(!CAp::Assert(n==vdir.m_n,__FUNCTION__+": sizes mismatch"))
      return;
   if(!CAp::Assert(m==vdir.m_m,__FUNCTION__+": sizes mismatch"))
      return;

   vstate.m_x+=vdir.m_x*stpp+0;
   vstate.m_g+=vdir.m_g*stpp+0;
   vstate.m_t+=vdir.m_t*stpp+0;
   vstate.m_z+=vdir.m_z*stpd+0;
   vstate.m_s+=vdir.m_s*stpd+0;
   vstate.m_w+=vdir.m_w*stpp+0;
   vstate.m_p+=vdir.m_p*stpp+0;
   vstate.m_y+=vdir.m_y*stpd+0;
   vstate.m_v+=vdir.m_v*stpd+0;
   vstate.m_q+=vdir.m_q*stpd+0;
  }
//+------------------------------------------------------------------+
//| Computes complementarity gap                                     |
//+------------------------------------------------------------------+
double CVIPMSolver::VarsComputeComplementarityGap(CVIPMVars &vstate)
  {
//--- create variables
   double result=0;
   int    n=vstate.m_n;
   int    m=vstate.m_m;

   result=CAblasF::RDotV(n,vstate.m_z,vstate.m_g)+CAblasF::RDotV(n,vstate.m_s,vstate.m_t);
   result+=CAblasF::RDotV(m,vstate.m_v,vstate.m_w)+CAblasF::RDotV(m,vstate.m_p,vstate.m_q);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes empirical value of the barrier parameter Mu             |
//+------------------------------------------------------------------+
double CVIPMSolver::VARSComputeMu(CVIPMState &State,CVIPMVars &vstate)
  {
//--- create variable
   double result=0;

   result=CAblasF::RDotV(vstate.m_n,vstate.m_z,vstate.m_g)+CAblasF::RDotV(vstate.m_n,vstate.m_s,vstate.m_t);
   result+=CAblasF::RDotV(vstate.m_m,vstate.m_v,vstate.m_w)+CAblasF::RDotV(vstate.m_m,vstate.m_p,vstate.m_q);
   result/=CApServ::Coalesce(State.m_cntgz+State.m_cntts+State.m_cntwv+State.m_cntpq,1);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Initializes reduced sparse system.                               |
//| Works only for sparse IPM.                                       |
//+------------------------------------------------------------------+
void CVIPMSolver::ReducedSystemInit(CVIPMReducedSparseSystem &s,
                                    CVIPMState &solver)
  {
//--- create variables
   int ntotal=0;
   int nnzmax=0;
   int factldlt=0;
   int permpriorityamd=0;
   int offs=0;
   int rowoffs=0;
   int i=0;
   int j=0;
   int k=0;
   int k0=0;
   int k1=0;
   int sumdeg=0;
   int colthreshold=0;
   int rowthreshold=0;
   int eligiblecols=0;
   int eligiblerows=0;
//--- check
   if(!CAp::Assert(solver.m_factorizationtype==1,__FUNCTION__+": unexpected factorization type"))
      return;
   if(!CAp::Assert(solver.m_hkind==1,__FUNCTION__+": unexpected HKind"))
      return;
   ntotal=solver.m_n+solver.m_mdense+solver.m_msparse;
   s.m_ntotal=ntotal;
   s.m_effectivediag.Resize(ntotal);
//--- Determine maximum amount of memory required to store sparse matrices
   nnzmax=solver.m_sparseh.m_RIdx[solver.m_n];
   if(solver.m_msparse>0)
      nnzmax+=solver.m_sparseafull.m_RIdx[solver.m_msparse];
   if(solver.m_mdense>0)
      nnzmax+=solver.m_n*solver.m_mdense;
   nnzmax+=ntotal;
//--- Prepare strictly lower triangle of template KKT matrix (KKT system without D and E
//--- terms being added to diagonals)
   s.m_rawsystem.m_M=ntotal;
   s.m_rawsystem.m_N=ntotal;
   s.m_rawsystem.m_Idx.Resize(nnzmax);
   s.m_rawsystem.m_Vals.Resize(nnzmax);
   s.m_rawsystem.m_RIdx.Resize(ntotal+1);
   s.m_rawsystem.m_RIdx.Set(0,0);
//if(solver.m_dodetailedtrace)
//---   CSparse::SparseTrace(s.m_rawsystem);
   offs=0;
   rowoffs=0;
   sumdeg=0;
   CAblasF::ISetAllocV(solver.m_n,0,s.m_coldegrees);
   CAblasF::ISetAllocV(solver.m_msparse+solver.m_mdense,0,s.m_rowdegrees);
   CAblasF::BSetAllocV(solver.m_n,true,s.m_isdiagonal);
   for(i=0; i<solver.m_n; i++)
     {
      //--- check
      if(!CAp::Assert(solver.m_sparseh.m_DIdx[i]+1==solver.m_sparseh.m_UIdx[i],__FUNCTION__+": critical integrity check failed for diagonal of H"))
         return;
      if(!solver.m_isfrozen[i])
        {
         //--- Entire row is not frozen, but some of its entries can be.
         //--- Output non-frozen offdiagonal entries.
         k0=solver.m_sparseh.m_RIdx[i];
         k1=solver.m_sparseh.m_DIdx[i];
         for(k=k0; k<k1; k++)
           {
            j=solver.m_sparseh.m_Idx[k];
            if(!solver.m_isfrozen[j])
              {
               s.m_rawsystem.m_Idx.Set(offs,j);
               s.m_rawsystem.m_Vals.Set(offs,-solver.m_sparseh.m_Vals[k]);
               s.m_isdiagonal[i]=false;
               s.m_isdiagonal[j]=false;
               offs++;
              }
           }
         //--- Output diagonal entry (it is always not frozen)
         s.m_rawsystem.m_Idx.Set(offs,i);
         s.m_rawsystem.m_Vals.Set(offs,-solver.m_sparseh.m_Vals[solver.m_sparseh.m_DIdx[i]]);
         offs++;
        }
      else
        {
         //--- Entire row is frozen, output just -1
         s.m_rawsystem.m_Idx.Set(offs,i);
         s.m_rawsystem.m_Vals.Set(offs,-1.0);
         offs++;
        }
      rowoffs++;
      s.m_rawsystem.m_RIdx.Set(rowoffs,offs);
      //if(solver.m_dodetailedtrace)
      //---   CSparse::SparseTrace(s.m_rawsystem);
     }
   for(i=0; i<solver.m_msparse; i++)
     {
      k0=solver.m_sparseafull.m_RIdx[i];
      k1=solver.m_sparseafull.m_RIdx[i+1];
      for(k=k0; k<k1; k++)
        {
         j=solver.m_sparseafull.m_Idx[k];
         if(!solver.m_isfrozen[j])
           {
            s.m_rawsystem.m_Idx.Set(offs,j);
            s.m_rawsystem.m_Vals.Set(offs,solver.m_sparseafull.m_Vals[k]);
            s.m_rowdegrees.Add(i,1);
            s.m_coldegrees.Add(j,1);
            sumdeg++;
            offs++;
           }
        }
      s.m_rawsystem.m_Idx.Set(offs,rowoffs);
      s.m_rawsystem.m_Vals.Set(offs,0.0);
      offs++;
      rowoffs++;
      s.m_rawsystem.m_RIdx.Set(rowoffs,offs);
      //if(solver.m_dodetailedtrace)
      //---   CSparse::SparseTrace(s.m_rawsystem);
     }
   for(i=0; i<solver.m_mdense; i++)
     {
      for(k=0; k<solver.m_n; k++)
        {
         if(solver.m_denseafull.Get(i,k)!=0.0 && !solver.m_isfrozen[k])
           {
            s.m_rawsystem.m_Idx.Set(offs,k);
            s.m_rawsystem.m_Vals.Set(offs,solver.m_denseafull.Get(i,k));
            s.m_rowdegrees.Add(solver.m_msparse+i,1);
            s.m_coldegrees.Add(k,1);
            sumdeg++;
            offs++;
           }
        }
      s.m_rawsystem.m_Idx.Set(offs,rowoffs);
      s.m_rawsystem.m_Vals.Set(offs,0.0);
      offs++;
      rowoffs++;
      s.m_rawsystem.m_RIdx.Set(rowoffs,offs);
      //if(solver.m_dodetailedtrace)
      //---   CSparse::SparseTrace(s.m_rawsystem);
     }
//--- check
   if(!CAp::Assert(rowoffs==ntotal,__FUNCTION__+": critical integrity check failed"))
      return;
   if(!CAp::Assert(offs<=nnzmax,__FUNCTION__+": critical integrity check failed"))
      return;
   CSparse::SparseCreateCRSInplace(s.m_rawsystem);
//if(solver.m_dodetailedtrace)
//---   CSparse::SparseTrace(s.m_rawsystem);
//--- Prepare reordering
   colthreshold=(int)MathRound(m_muquasidense*sumdeg/solver.m_n)+2;
   rowthreshold=(int)MathRound(m_muquasidense*sumdeg/(solver.m_msparse+solver.m_mdense+1))+2;
   eligiblecols=0;
   eligiblerows=0;
   CAblasF::ISetAllocV(ntotal,0,s.m_priorities);
   for(i=0; i<solver.m_n; i++)
     {
      if(s.m_isdiagonal[i] && s.m_coldegrees[i]<=colthreshold)
         eligiblecols++;
     }
   for(i=0; i<(solver.m_mdense+solver.m_msparse); i++)
     {
      if(s.m_rowdegrees[i]<=rowthreshold)
         eligiblerows++;
     }
   if(solver.m_dotrace)
      CAp::Trace("> initializing KKT system; no priority ordering being applied\n");
//--- Perform factorization analysis using sparsity pattern (but not numerical values)
   factldlt=1;
   permpriorityamd=3;
   if(!CSpChol::SpSymmAnalyze(s.m_rawsystem,s.m_priorities,factldlt,permpriorityamd,s.m_analysis))
      CAp::Assert(false,__FUNCTION__+": critical integrity check failed,symbolically degenerate KKT system encountered");
  }
//+------------------------------------------------------------------+
//| Computes factorization of A + D, where A is  internally  stored  |
//| KKT matrix and D is user-supplied diagonal term.The factorization|
//| is stored internally and should never be accessed directly.      |
//| ModEps and BadChol are user supplied tolerances for modified     |
//| Cholesky / LDLT.                                                 |
//| Returns True on success, False on LDLT failure.                  |
//| On success outputs diagonal reproduction error ErrSq, and sum of |
//| squared diagonal elements SumSq                                  |
//+------------------------------------------------------------------+
bool CVIPMSolver::ReducedSystemFactorizeWithAddEnd(CVIPMReducedSparseSystem &s,
                                                   CRowDouble &d,
                                                   double modeps,
                                                   double badchol,
                                                   double &sumsq,
                                                   double &errsq)
  {
//--- create variables
   bool result=true;
   int  ntotal=s.m_ntotal;
   sumsq=0;
   errsq=0;

   for(int i=0; i<ntotal; i++)
      s.m_effectivediag.Set(i,s.m_rawsystem.m_Vals[s.m_rawsystem.m_DIdx[i]]+d[i]);
   CSpChol::SpSymmReloadDiagonal(s.m_analysis,s.m_effectivediag);
   CSpChol::SpSymmSetModificationStrategy(s.m_analysis,1,modeps,badchol,0.0,0.0);
   if(CSpChol::SpSymmFactorize(s.m_analysis))
      CSpChol::SpSymmDiagErr(s.m_analysis,sumsq,errsq);
   else
     {
      sumsq=0;
      errsq=0;
      result=false;
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Solve reduced KKT system, replacing right part by its solution.  |
//+------------------------------------------------------------------+
void CVIPMSolver::ReducedSystemSolve(CVIPMReducedSparseSystem &s,
                                     CRowDouble &b)
  {
   CSpChol::SpSymmSolve(s.m_analysis,b);
  }
//+------------------------------------------------------------------+
//| Initializes QP-IPM State and prepares it to receive quadratic /  |
//| linear terms and constraints.                                    |
//| The solver is configured to work internally with factorization |
//| FType                                                            |
//| INPUT PARAMETERS:                                                |
//|   State    -  solver  State  to  be configured; previously       |
//|               allocated memory is reused as much as possible     |
//|   S        -  scale vector, array[N]:                            |
//|               * I-th element contains scale of I-th variable,    |
//|               * S[I] > 0                                         |
//|   XOrigin  -  origin term, array[N]. Can be zero. The solver     |
//|               solves problem of the form                         |
//|   >                                                              |
//|   >  min(0.5 * (x-x_origin)'*A*(x-x_origin)+b' * (x-x_origin))   |
//|   >                                                              |
//| The terms A and b (as well as constraints) will be specified     |
//| later with separate calls.                                       |
//|   FType    -  factorization type:                                |
//|               * 0 for dense NxN factorization (normal equations) |
//|               * 1 for sparse(N + M)                              |
//|                   x(N + M) factorization                         |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMInit(CVIPMState &State,
                           CRowDouble &s,
                           CRowDouble &xorigin,
                           int n,
                           int nmain,
                           int ftype)
  {
//--- create variables
   int nslack=n-nmain;
   int i=0;
   int j=0;
//--- check
   if(!CAp::Assert(n>=1,__FUNCTION__+": N<1"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(s,n),__FUNCTION__+": S contains infinite or NaN elements"))
      return;
   if(!CAp::Assert(CApServ::IsFiniteVector(xorigin,n),__FUNCTION__+": XOrigin contains infinite or NaN elements"))
      return;
   if(!CAp::Assert(ftype==0 || ftype==1,__FUNCTION__+": unexpected FType"))
      return;
   if(!CAp::Assert(nmain>=1,__FUNCTION__+": NMain<1"))
      return;
   if(!CAp::Assert(nmain<=n,__FUNCTION__+": NMain>N"))
      return;
//--- Problem metrics, Settings and type
   State.m_n=n;
   State.m_nmain=nmain;
   State.m_islinear=true;
   State.m_factorizationtype=ftype;
   State.m_factorizationpresent=false;
   State.m_factorizationpoweredup=false;
   State.m_slacksforequalityconstraints=true;
   VIPMSetCond(State,0.0,0.0,0.0);
//--- Reports
   State.m_repiterationscount=0;
   State.m_repncholesky=0;
//--- Trace
   State.m_dotrace=false;
   State.m_dodetailedtrace=false;
//--- Scale and origin
   State.m_scl.Resize(n);
   State.m_invscl.Resize(n);
   State.m_xorigin.Resize(n);
   for(i=0; i<n; i++)
     {
      //--- check
      if(!CAp::Assert(s[i]>0.0,__FUNCTION__+": S[i] is non-positive"))
         return;
      State.m_scl.Set(i,s[i]);
      State.m_invscl.Set(i,1/s[i]);
      State.m_xorigin.Set(i,xorigin[i]);
     }
   State.m_targetscale=1.0;
//--- Linear and quadratic terms - default value
   State.m_c=vector<double>::Zeros(n);
   State.m_hkind=-1;
   switch(ftype)
     {
      case 0:
         //--- Dense quadratic term
         State.m_denseh.Resize(nmain,nmain);
         for(i=0; i<nmain; i++)
            for(j=0; j<=i; j++)
               State.m_denseh.Set(i,j,0);
         State.m_hkind=0;
         State.m_isdiagonalh=false;
         break;
      case 1:
         //--- Sparse quadratic term
         State.m_sparseh.m_MatrixType=1;
         State.m_sparseh.m_M=n;
         State.m_sparseh.m_N=n;
         State.m_sparseh.m_NInitialized=n;
         State.m_sparseh.m_Idx.Resize(n);
         State.m_sparseh.m_Vals=vector<double>::Zeros(n);
         State.m_sparseh.m_RIdx.Resize(n+1);
         for(i=0; i<n; i++)
           {
            State.m_sparseh.m_Idx.Set(i,i);
            State.m_sparseh.m_RIdx.Set(i,i);
           }
         State.m_sparseh.m_RIdx.Set(n,n);
         CSparse::SparseCreateCRSInplace(State.m_sparseh);
         State.m_hkind=1;
         State.m_isdiagonalh=true;
         break;
     }
//--- check
   if(!CAp::Assert(State.m_hkind>=0,__FUNCTION__+": integrity check failed"))
      return;
//--- Box constraints - default values
   State.m_bndl=vector<double>::Full(n,AL_NEGINF);
   State.m_bndu=vector<double>::Full(n,AL_POSINF);
   CApServ::BVectorSetLengthAtLeast(State.m_HasBndL,n);
   CApServ::BVectorSetLengthAtLeast(State.m_HasBndU,n);
   ArrayInitialize(State.m_HasBndL,false);
   ArrayInitialize(State.m_HasBndU,false);
//--- Linear constraints - empty
   State.m_mdense=0;
   State.m_msparse=0;
   State.m_combinedaslack.m_M=0;
   State.m_combinedaslack.m_N=nslack;
   State.m_sparseamain.m_M=0;
   State.m_sparseamain.m_N=nmain;
   CSparse::SparseCreateCRSInplace(State.m_sparseamain);
   CSparse::SparseCreateCRSInplace(State.m_combinedaslack);
  }
//+------------------------------------------------------------------+
//| Computes target function 0.5 * x'*H*x+c'*x                       |
//+------------------------------------------------------------------+
double CVIPMSolver::VIPMTarget(CVIPMState &State,
                               CRowDouble &x)
  {
//--- create variables
   double result=0;
   int    n=State.m_n;
   int    nmain=State.m_nmain;
   int    i=0;
   int    j=0;
   int    k=0;
   int    j0=0;
   int    j1=0;
   double v=0;
//--- check
   if(!CAp::Assert(State.m_hkind==0 || State.m_hkind==1,__FUNCTION__+": unexpected HKind"))
      return(result);
   switch(State.m_hkind)
     {
      //--- Dense
      case 0:
         for(i=0; i<nmain; i++)
           {
            for(j=0; j<=i-1; j++)
               result+=x[i]*State.m_denseh.Get(i,j)*x[j];
            result +=+0.5*CMath::Sqr(x[i])*State.m_denseh.Get(i,i);
           }
         for(i=0; i<n; i++)
            result+=State.m_c[i]*x[i];
         break;
      //--- Sparse
      case 1:
         for(i=0; i<n; i++)
           {
            result+=State.m_c[i]*x[i];
            j0=State.m_sparseh.m_RIdx[i];
            j1=State.m_sparseh.m_DIdx[i]-1;
            for(k=j0; k<=j1; k++)
              {
               v=State.m_sparseh.m_Vals[k];
               j=State.m_sparseh.m_Idx[k];
               result+=v*x[i]*x[j];
              }
            //--- check
            if(!CAp::Assert(State.m_sparseh.m_UIdx[i]!=State.m_sparseh.m_DIdx[i],__FUNCTION__+": sparse diagonal not found"))
               return(0);
            v=State.m_sparseh.m_Vals[State.m_sparseh.m_DIdx[i]];
            result+=0.5*v*CMath::Sqr(x[i]);
           }
         break;
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes Y := alpha * A * x + beta * Y                           |
//| where   A is constraint matrix,                                  |
//|         X is user - specified source,                            |
//|         Y is target.                                             |
//| Beta can be zero(in this case original contents of Y is ignored).|
//| If Beta is nonzero, we expect that Y contains preallocated array.|
//+------------------------------------------------------------------+
void CVIPMSolver::MultiplyGEAX(CVIPMState &State,
                               double alpha,
                               CRowDouble &x,
                               int offsx,
                               double beta,
                               CRowDouble &y,
                               int offsax)
  {
//--- create variables
   int n=State.m_n;
   int m=State.m_mdense+State.m_msparse;
   int mdense=State.m_mdense;
   int msparse=State.m_msparse;

   if(beta==0.0)
      y.Resize(offsax+m);
   else
     {
      //--- check
      if(!CAp::Assert(y.Size()>=offsax+m,__FUNCTION__+": Y is too short"))
         return;
     }
   if(msparse>0)
      CSparse::SparseGemV(State.m_sparseafull,alpha,0,x,offsx,beta,y,offsax);
   if(mdense>0)
      CAblas::RMatrixGemVect(mdense,n,alpha,State.m_denseafull,0,0,0,x,offsx,beta,y,offsax+msparse);
  }
//+------------------------------------------------------------------+
//| Computes Y := alpha * A'*x + beta*Y                              |
//| where   A is constraint matrix,                                  |
//|         X is user - specified source,                            |
//|         Y is target.                                             |
//| Beta can be zero, in this case we automatically reallocate target|
//| if it is too short (but do NOT reallocate it if its size is large|
//| enough). If Beta is nonzero, we expect that Y contains           |
//| preallocated array.                                              |
//+------------------------------------------------------------------+
void CVIPMSolver::MultiplyGEATX(CVIPMState &State,
                                double alpha,
                                CRowDouble &x,
                                int offsx,
                                double beta,
                                CRowDouble &y,
                                int offsy)
  {
//--- create variables
   int n=State.m_n;
   int mdense=State.m_mdense;
   int msparse=State.m_msparse;

   if(beta==0.0)
     {
      y.Resize(offsy+n);
      CAblasF::RSetVX(n,0.0,y,offsy);
     }
   else
     {
      //--- check
      if(!CAp::Assert(y.Size()>=offsy+n,__FUNCTION__+": Y is too short"))
         return;
      CAblasF::RMulVX(n,beta,y,offsy);
     }
   if(msparse>0)
      CSparse::SparseGemV(State.m_sparseafull,alpha,1,x,offsx,1.0,y,offsy);
   if(mdense>0)
      CAblas::RMatrixGemVect(n,mdense,alpha,State.m_denseafull,0,0,1,x,offsx+msparse,1.0,y,offsy);
  }
//+------------------------------------------------------------------+
//| Computes H*x, does not support advanced functionality of GEAX /  |
//| GEATX                                                            |
//+------------------------------------------------------------------+
void CVIPMSolver::MultiplyHX(CVIPMState &State,
                             CRowDouble &x,
                             CRowDouble &hx)
  {
//--- create variables
   int n=State.m_n;
   int nmain=State.m_nmain;
   int i=0;

   hx.Resize(n);
//--- check
   if(!CAp::Assert(State.m_hkind==0 || State.m_hkind==1,__FUNCTION__+": unexpected HKind"))
      return;
   switch(State.m_hkind)
     {
      case 0:
         CAblas::RMatrixSymVect(nmain,1.0,State.m_denseh,0,0,false,x,0,0.0,hx,0);
         for(i=nmain; i<n; i++)
            hx.Set(i,0);
         for(i=0; i<n; i++)
            hx.Add(i,x[i]*State.m_diagr[i]);
         break;
      case 1:
         //--- check
         if(!CAp::Assert(State.m_sparseh.m_N==n && State.m_sparseh.m_M==n,__FUNCTION__+": sparse H has incorrect size"))
            return;
         if(State.m_isdiagonalh)
           {
            //--- H is known to be diagonal, much faster code can be used
            CAblasF::RCopyV(n,State.m_diagr,hx);
            CAblasF::RAddV(n,1.0,State.m_sparseh.m_Vals,hx);
            CAblasF::RMergeMulV(n,x,hx);
           }
         else
           {
            //--- H is a general sparse matrix, use generic sparse matrix-vector multiply
            CSparse::SparseSMV(State.m_sparseh,false,x,hx);
            for(i=0; i<n; i++)
               hx.Add(i,x[i]*State.m_diagr[i]);
           }
         break;
     }
  }
//+------------------------------------------------------------------+
//| Computes products H*x, A*x, A^T*y                                |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMMultiply(CVIPMState &State,
                               CRowDouble &x,
                               CRowDouble &y,
                               CRowDouble &hx,
                               CRowDouble &ax,
                               CRowDouble &aty)
  {
   MultiplyGEAX(State,1.0,x,0,0.0,ax,0);
   MultiplyGEATX(State,1.0,y,0,0.0,aty,0);
   MultiplyHX(State,x,hx);
  }
//+------------------------------------------------------------------+
//| This function "powers up" factorization, i.e. prepares some      |
//| important temporaries. It should be called once prior to the     |
//| first call to VIPMInitialPoint() or VIPMFactorize().             |
//| Parameters:                                                      |
//|   RegFree  -  regularization for free variables; good value      |
//|               sqrt(MachineEpsilon)                               |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMPowerUp(CVIPMState &State,
                              double regfree)
  {
//--- create variables
   int    n=State.m_n;
   int    m=State.m_mdense+State.m_msparse;
   int    i=0;
   double v=0;
   double vrhs=0;
   double priorcoeff=0;
   double initprimslack=0;
   double initdualslack=0;
   double maxinitialnoncentrality=0;
   double maxinitialimbalance=0;
   double mu0=0;
   double mumin=0;
   bool   success=false;
//--- check
   if(!CAp::Assert(State.m_factorizationtype==0 || State.m_factorizationtype==1,__FUNCTION__+": unexpected factorization type"))
      return;
//--- check
   if(!CAp::Assert(!State.m_factorizationpoweredup,__FUNCTION__+": repeated call"))
      return;
   maxinitialnoncentrality=1.0E-6;
   maxinitialimbalance=1.0E-6;
//--- Set up information about presence of slack variables.
//--- Decide which components of X should be frozen.
//--- Compute diagonal regularization matrix R.
   CAblasF::BCopyAllocV(n,State.m_HasBndL,State.m_hasgz);
   CAblasF::BCopyAllocV(n,State.m_HasBndU,State.m_hasts);
   CAblasF::BSetAllocV(n,false,State.m_isfrozen);
   CAblasF::RSetAllocV(n,0.0,State.m_diagr);
   VARSInitByZero(State.m_current,n,m);
   for(i=0; i<n; i++)
     {
      if(State.m_HasBndL[i] && State.m_HasBndU[i] && State.m_bndl[i]==State.m_bndu[i])
        {
         State.m_isfrozen[i]=true;
         State.m_hasgz[i]=false;
         State.m_hasts[i]=false;
         State.m_current.m_x.Set(i,State.m_bndl[i]);
        }
      if(!State.m_HasBndL[i] && !State.m_HasBndU[i])
         State.m_diagr.Set(i,regfree);
     }
   CAblasF::BAllocV(m,State.m_haspq);
   CAblasF::BAllocV(m,State.m_haswv);
   for(i=0; i<m; i++)
     {
      State.m_haswv[i]=(State.m_slacksforequalityconstraints || !State.m_hasr[i] || State.m_r[i]>0.0);
      State.m_haspq[i]=(State.m_hasr[i] && State.m_haswv[i]);
     }
   State.m_cntgz=0;
   State.m_cntts=0;
   State.m_cntwv=0;
   State.m_cntpq=0;
   for(i=0; i<n; i++)
     {
      if(State.m_hasgz[i])
         State.m_cntgz++;
      if(State.m_hasts[i])
         State.m_cntts++;
     }
   for(i=0; i<m; i++)
     {
      if(State.m_haswv[i])
         State.m_cntwv++;
      if(State.m_haspq[i])
         State.m_cntpq++;
     }
//--- Special initialization for sparse version
   if(State.m_factorizationtype==1)
      ReducedSystemInit(State.m_reducedsparsesystem,State);
   State.m_factorizationpoweredup=true;
//--- Set up initial values of primal and dual variables X and Y by solving
//--- modified KKT system which tries to enforce linear constraints (ignoring
//--- box constraints for a while) subject to minimization of additional prior
//--- term which moves solution towards some interior point.
//--- Here we expect that State.Current.X contains zeros in non-fixed variables
//--- and their fixed values for fixed ones.
   priorcoeff=1.0;
   success=VIPMFactorize(State,0.0,State.m_diagddr,0.0,State.m_diagder,priorcoeff,priorcoeff,CMath::m_machineepsilon,CMath::m_machineepsilon);
//--- check
   if(!CAp::Assert(success,__FUNCTION__+": impossible failure of LDLT factorization"))
      return;
   MultiplyHX(State,State.m_current.m_x,State.m_tmp0);
   MultiplyGEAX(State,1.0,State.m_current.m_x,0,0.0,State.m_tmp1,0);
   CAblasF::RAllocV(n+m,State.m_deltaxy);
   for(i=0; i<n; i++)
      State.m_deltaxy.Set(i,State.m_c[i]+State.m_tmp0[i]);
   for(i=0; i<m; i++)
     {
      //--- We need to specify target right-hand sides for constraints.
      //--- Ether zero, b or b+r is used (depending on presence of r and
      //--- magnitudes of b and b+r, and subject to current State of frozen
      //--- variables).
      vrhs=State.m_b[i]-State.m_tmp1[i];
      if(State.m_hasr[i])
        {
         //--- Range constraint b<=Ax<=b+r
         if(vrhs>=0.0)
           {
            //--- 0<=b<=b+r, select target at lower bound
            v=vrhs;
           }
         else
           {
            //--- b<=0, b+r can have any sign.
            //--- Select zero target if possible, if not - one with smallest absolute value.
            v=MathMin(vrhs+State.m_r[i],0.0);
           }
        }
      else
        {
         //--- Single-sided constraint Ax>=b.
         //--- Select zero target if possible, if not - one with smallest absolute value.
         v=MathMax(vrhs,0.0);
        }
      State.m_deltaxy.Set(n+i,v);
     }
   SolveReducedKKTSystem(State,State.m_deltaxy);
   for(i=0; i<n; i++)
     {
      if(!State.m_isfrozen[i])
         State.m_current.m_x.Set(i,State.m_deltaxy[i]);
     }
   for(i=0; i<m; i++)
      State.m_current.m_y.Set(i,State.m_deltaxy[n+i]);
//--- Set up slacks according to our own heuristic
   initprimslack=MathMax(m_initslackval,CAblasF::RMaxAbsV(n,State.m_current.m_x));
   initdualslack=MathMax(m_initslackval,CAblasF::RMaxAbsV(m,State.m_current.m_y));
   MultiplyGEAX(State,1.0,State.m_current.m_x,0,0.0,State.m_tmpax,0);
   mu0=1.0;
   for(i=0; i<n; i++)
     {
      if(State.m_hasgz[i])
        {
         State.m_current.m_g.Set(i,MathMax(MathAbs(State.m_current.m_x[i]-State.m_bndl[i]),initprimslack));
         State.m_current.m_z.Set(i,MathMax(State.m_current.m_g[i]*maxinitialimbalance,initdualslack));
         mu0=MathMax(mu0,State.m_current.m_g[i]*State.m_current.m_z[i]);
        }
      if(State.m_hasts[i])
        {
         State.m_current.m_t.Set(i,MathMax(MathAbs(State.m_current.m_x[i]-State.m_bndu[i]),initprimslack));
         State.m_current.m_s.Set(i,MathMax(State.m_current.m_t[i]*maxinitialimbalance,initdualslack));
         mu0=MathMax(mu0,State.m_current.m_t[i]*State.m_current.m_s[i]);
        }
     }
   for(i=0; i<m; i++)
     {
      if(State.m_haswv[i])
        {
         State.m_current.m_w.Set(i,MathMax(MathAbs(State.m_tmpax[i]-State.m_b[i]),initprimslack));
         State.m_current.m_v.Set(i,MathMax(State.m_current.m_w[i]*maxinitialimbalance,MathMax(MathAbs(State.m_current.m_y[i]),m_initslackval)));
         mu0=MathMax(mu0,State.m_current.m_w[i]*State.m_current.m_v[i]);
        }
      if(State.m_haspq[i])
        {
         State.m_current.m_p.Set(i,MathMax(MathAbs(State.m_r[i]-State.m_current.m_w[i]),initprimslack));
         State.m_current.m_q.Set(i,MathMax(State.m_current.m_p[i]*maxinitialimbalance,MathMax(MathAbs(State.m_current.m_y[i]),m_initslackval)));
         mu0=MathMax(mu0,State.m_current.m_p[i]*State.m_current.m_q[i]);
        }
     }
//--- Additional shift to ensure that initial point is not too non-centered
   mumin=mu0*maxinitialnoncentrality;
   for(i=0; i<n; i++)
     {
      if(State.m_hasgz[i] && (State.m_current.m_g[i]*State.m_current.m_z[i])<mumin)
        {
         v=MathSqrt(mumin/(State.m_current.m_g[i]*State.m_current.m_z[i]));
         State.m_current.m_g.Mul(i,v);
         State.m_current.m_z.Mul(i,v);
        }
      if(State.m_hasts[i] && (double)(State.m_current.m_t[i]*State.m_current.m_s[i])<mumin)
        {
         v=MathSqrt(mumin/(State.m_current.m_t[i]*State.m_current.m_s[i]));
         State.m_current.m_t.Mul(i,v);
         State.m_current.m_s.Mul(i,v);
        }
     }
   for(i=0; i<m; i++)
     {
      if(State.m_haswv[i] && (State.m_current.m_w[i]*State.m_current.m_v[i])<mumin)
        {
         v=MathSqrt(mumin/(State.m_current.m_w[i]*State.m_current.m_v[i]));
         State.m_current.m_w.Mul(i,v);
         State.m_current.m_v.Mul(i,v);
        }
      if(State.m_haspq[i] && (State.m_current.m_p[i]*State.m_current.m_q[i])<mumin)
        {
         v=MathSqrt(mumin/(State.m_current.m_p[i]*State.m_current.m_q[i]));
         State.m_current.m_p.Mul(i,v);
         State.m_current.m_q.Mul(i,v);
        }
     }
//--- Almost done
   if(State.m_dotrace)
      CAp::Trace("> initial point was generated\n");
  }
//+------------------------------------------------------------------+
//| This function performs factorization of modified KKT system      |
//|   (                             |                    )           |
//|   (-(H + alpha0 * D + alpha1*I) |       A^T          )           |
//|   (                             |                    )           |
//|   (---------------------------- | -------------------)           |
//|   (                             |                    )           |
//|   (                A            | beta0 * E + beta1*I)           |
//|   (                             |                    )           |
//| where:                                                           |
//|      * H is an NxN quadratic term                                |
//|      * A is an MxN matrix of linear constraint                   |
//|      * alpha0, alpha1, beta0, beta1 are nonnegative scalars      |
//|      * D and E are diagonal matrices with nonnegative entries    |
//|        (which are ignored if alpha0 and beta0 are zero - arrays  |
//|        are not referenced at all)                                |
//|      * I is an NxN or MxM identity matrix                        |
//| Additionally, regularizing term                                  |
//|      (           |           )                                   |
//|      ( -reg * I  |           )                                   |
//|      (           |           )                                   |
//|      (---------- | ----------)                                   |
//|      (           |           )                                   |
//|      (           |  +reg*I   )                                   |
//|      (           |           )                                   |
//| is added to the entire KKT system prior to factorization in order|
//| to improve its numerical stability.                              |
//| Returns True on success, False on falure of factorization (it is |
//| recommended to increase regularization parameter and try one more|
//| time).                                                           |
//+------------------------------------------------------------------+
bool CVIPMSolver::VIPMFactorize(CVIPMState &State,
                                double alpha0,
                                CRowDouble &d,
                                double beta0,
                                CRowDouble &e,
                                double alpha11,
                                double beta11,
                                double modeps,
                                double dampeps)
  {
//--- create variables
   int    n=State.m_n;
   int    nmain=State.m_nmain;
   int    nslack=n-nmain;
   int    m=State.m_mdense+State.m_msparse;
   int    mdense=State.m_mdense;
   int    msparse=State.m_msparse;
   int    i=0;
   int    j=0;
   int    k=0;
   int    k0=0;
   int    k1=0;
   int    ka=0;
   int    kb=0;
   int    ja=0;
   int    jb=0;
   double va=0;
   double vb=0;
   double v=0;
   double vv=0;
   double badchol=1.0E50;
   double sumsq=0;
   double errsq=0;
//--- check
   if(!CAp::Assert(MathIsValidNumber(alpha0) && alpha0>=0.0,__FUNCTION__+": Alpha0 is infinite or negative"))
      return(false);
   if(!CAp::Assert(MathIsValidNumber(alpha11) && alpha11>=0.0,__FUNCTION__+": Alpha1 is infinite or negative"))
      return(false);
   if(!CAp::Assert(MathIsValidNumber(beta0) && beta0>=0.0,__FUNCTION__+": Beta0 is infinite or negative"))
      return(false);
   if(!CAp::Assert(MathIsValidNumber(beta11) && beta11>=0.0,__FUNCTION__+": Beta1 is infinite or negative"))
      return(false);
   if(!CAp::Assert(State.m_factorizationtype==0 || State.m_factorizationtype==1,__FUNCTION__+": unexpected factorization type"))
      return(false);
   if(!CAp::Assert(State.m_factorizationpoweredup,__FUNCTION__+": critical integrity check failed (no powerup stage)"))
      return(false);

   State.m_factorizationpresent=false;
//--- Dense NxN normal equations approach
   if(State.m_factorizationtype==0)
     {
      //--- A problem formulation with possible slacks.
      //--- === A FORMULATION WITHOUT FROZEN VARIABLES ===
      //--- We have to solve following system:
      //---     [ -(H+Dh+Rh)         Ah'  ] [ Xh ]   [ Bh ]
      //---     [          -(Dz+Rz)  Az'  ] [ Xz ] = [ Bz ]
      //---     [   Ah     Az         E   ] [ Y  ]   [ By ]
      //--- with Xh being NMain-dimensional vector, Xz being NSlack-dimensional vector, constraint
      //--- matrix A being divided into non-slack and slack parts Ah and Az (and Ah, in turn, being
      //--- divided into sparse and dense parts), Rh and Rz being diagonal regularization matrix,
      //--- Y being M-dimensional vector.
      //--- NOTE: due to definition of slack variables following holds: for any diagonal matrix W
      //---       a product Az*W*Az' is a diagonal matrix.
      //--- From the second line we get
      //---     Xz = inv(Dz+Rz)*Az'*y - inv(Dz+Rz)*Bz
      //---        = inv(Dz+Rz)*Az'*y - BzWave
      //--- Using this value for Zx, third line gives us
      //---     Y  = inv(E+Az*inv(Dz+Rz)*Az')*(By+Az*BzWave-Ah*Xh)
      //---        = inv(EWave)*(ByWave-Ah*Xh)
      //---        with EWave = E+Az*inv(Dz+Rz)*Az' and ByWave = By+Az*BzWave
      //--- Finally, first line gives us
      //---     Xh = -inv(H+Dh+Rh+Ah'*inv(EWave)*Ah)*(Bh-Ah'*inv(EWave)*ByWave)
      //---        = -inv(HWave)*BhWave
      //---        with HWave = H+Dh+Rh+Ah'*inv(EWave)*Ah and BhWave = Bh-Ah'*inv(EWave)*ByWave
      //--- In order to prepare factorization we need to compute:
      //--- (a) diagonal matrices Dh, Rh, Dz and Rz (and precomputed inverse of Dz+Rz)
      //--- (b) EWave
      //--- (c) HWave
      //--- === SPECIAL HANDLING OF FROZEN VARIABLES ===
      //--- Frozen variables result in zero steps, i.e. zero components of Xh and Xz.
      //--- It could be implemented by explicit modification of KKT system (zeroing out
      //--- columns/rows of KKT matrix, rows of right part, putting 1's to diagonal).
      //--- However, it is possible to do without actually modifying quadratic term and
      //--- constraints:
      //--- * freezing elements of Xz can be implemented by zeroing out corresponding
      //---   columns of inv(Dz+Rz) because Az always appears in computations along with diagonal Dz+Rz.
      //--- * freezing elements of Xh is a bit more complex - it needs:
      //---   * zeroing out columns/rows of HWave and setting up unit diagonal prior to solving for Xh
      //---   * explicitly zeroing out computed elements of Xh prior to computing Y and Xz
      State.m_factregdhrh.Resize(nmain);
      State.m_factinvregdzrz.Resize(nslack);
      for(i=0; i<n; i++)
        {
         v=0;
         if(alpha0>0)
            v+=alpha0*d[i];
         if(alpha11>0)
            v+=alpha11;
         v+=State.m_diagr[i]+dampeps;
         //--- check
         if(!CAp::Assert(v>0,__FUNCTION__+": integrity check failed,degenerate diagonal matrix"))
            return(false);
         if(i>=nmain)
           {
            if(!State.m_isfrozen[i])
               State.m_factinvregdzrz.Set(i-nmain,1/v);
            else
               State.m_factinvregdzrz.Set(i-nmain,0.0);
           }
         else
            State.m_factregdhrh.Set(i,v);
        }
      //--- Now we are ready to compute EWave
      State.m_factregewave.Resize(m);
      for(i=0; i<m; i++)
        {
         //--- Compute diagonal element of E
         v=0;
         if(beta0>0)
            v+=beta0*e[i];
         if(beta11>0)
            v+=beta11;
         v+=dampeps;
         //--- check
         if(!CAp::Assert(v>0,__FUNCTION__+": integrity check failed,degenerate diagonal matrix"))
            return(false);
         //--- Compute diagonal modification Az*inv(Dz)*Az'
         k0=State.m_combinedaslack.m_RIdx[i];
         k1=State.m_combinedaslack.m_RIdx[i+1];
         for(k=k0; k<k1; k++)
           {
            vv=State.m_combinedaslack.m_Vals[k];
            v+=vv*vv*State.m_factinvregdzrz[State.m_combinedaslack.m_Idx[k]];
           }
         //--- Save EWave
         State.m_factregewave.Set(i,v);
        }
      //--- Now we are ready to compute HWave:
      //--- * store H
      //--- * add Dh
      //--- * add Ah'*inv(EWave)*Ah
      //--- check
      if(!CAp::Assert(State.m_hkind==0,__FUNCTION__+": unexpected HKind"))
         return(false);
      State.m_factdensehaug=State.m_denseh;
      State.m_factdensehaug.Resize(nmain,nmain);
      for(i=0; i<nmain; i++)
         State.m_factdensehaug.Add(i,i,State.m_factregdhrh[i]);
      if(msparse>0)
        {
         //--- Handle sparse part of Ah in Ah'*inv(EWave)*Ah
         for(i=0; i<msparse; i++)
           {
            v=1.0/State.m_factregewave[i];
            k0=State.m_sparseamain.m_RIdx[i];
            k1=State.m_sparseamain.m_RIdx[i+1];
            for(ka=k0; ka<k1; ka++)
              {
               ja=State.m_sparseamain.m_Idx[ka];
               va=State.m_sparseamain.m_Vals[ka];
               for(kb=k0; kb<=ka; kb++)
                 {
                  jb=State.m_sparseamain.m_Idx[kb];
                  vb=State.m_sparseamain.m_Vals[kb];
                  State.m_factdensehaug.Add(ja,jb,v*va*vb);
                 }
              }
           }
        }
      if(mdense>0)
        {
         //--- Handle dense part of Ah in Ah'*inv(EWave)*Ah
         State.m_tmpr2=State.m_denseamain;
         State.m_tmpr2.Resize(mdense,nmain);
         for(i=0; i<mdense; i++)
           {
            v=1.0/MathSqrt(State.m_factregewave[msparse+i]);
            for(j=0; j<nmain; j++)
               State.m_tmpr2.Mul(i,j,v);
           }
         CAblas::RMatrixSyrk(nmain,mdense,1.0,State.m_tmpr2,0,0,2,1.0,State.m_factdensehaug,0,0,false);
        }
      //--- Zero out rows/cols of HWave corresponding to frozen variables, set up unit diagonal
      State.m_tmp0=vector<double>::Ones(nmain);
      for(i=0; i<nmain; i++)
        {
         if(State.m_isfrozen[i])
           {
            State.m_tmp0.Set(i,0.0);
            //--- Entire row is nullified except for diagonal element
            CAblasF::RSetR(i+1,0.0,State.m_factdensehaug,i);
            State.m_factdensehaug.Set(i,i,1.0);
           }
         else
           {
            //--- Only some components are nullified
            CAblasF::RMergeMulVR(i+1,State.m_tmp0,State.m_factdensehaug,i);
           }
        }
      //--- Compute Cholesky factorization of HWave
      if(!CTrFac::SPDMatrixCholesky(State.m_factdensehaug,nmain,false))
         return(false);
      v=(State.m_factdensehaug.Diag(0)+0).Sum();
      if(!MathIsValidNumber(v) || v>badchol)
         return(false);
      State.m_factorizationpresent=true;
     }
//--- Sparse (M+N)x(M+N) factorization
   if(State.m_factorizationtype==1)
     {
      //--- Generate reduced KKT matrix
      State.m_facttmpdiag.Resize(n+m);
      for(i=0; i<n; i++)
        {
         vv=0;
         if(alpha0>0)
            vv+=alpha0*d[i];
         if(alpha11>0)
            vv+=alpha11;
         vv+=State.m_diagr[i]+dampeps;
         State.m_facttmpdiag.Set(i,-vv);
         //--- check
         if(!CAp::Assert(vv>0,__FUNCTION__+": integrity check failed,degenerate diagonal matrix"))
            return(false);
        }
      for(i=0; i<msparse+mdense; i++)
        {
         vv=0;
         if(beta0>0)
            vv+=beta0*e[i];
         if(beta11>0)
            vv+=beta11;
         vv+=dampeps;
         State.m_facttmpdiag.Set(n+i,vv);
         //--- check
         if(!CAp::Assert(vv>0,__FUNCTION__+": integrity check failed,degenerate diagonal matrix"))
            return(false);
        }
      //--- Perform factorization
      //--- Perform additional integrity check: LDLT should reproduce diagonal of initial KKT system with good precision
      if(!ReducedSystemFactorizeWithAddEnd(State.m_reducedsparsesystem,State.m_facttmpdiag,modeps,badchol,sumsq,errsq))
        {
         return(false);
        }
      if(MathSqrt(errsq/(1+sumsq))>MathSqrt(CMath::m_machineepsilon))
        {
         if(State.m_dotrace)
            CAp::Trace(StringFormat("LDLT-diag-err= %.3E (diagonal reproduction error)\n",MathSqrt(errsq / (1 + sumsq))));
         return(false);
        }
      State.m_factorizationpresent=true;
      //--- Trace
      if(State.m_dotrace)
        {
         CSpChol::SpSymmExtract(State.m_reducedsparsesystem.m_analysis,State.m_tmpsparse0,State.m_tmp0,State.m_tmpi);
         CAp::Trace("--- sparse KKT factorization report ----------------------------------------------------------------\n");
         CAp::Trace("> diagonal terms D and E\n");
         if(alpha0!=0.0)
           {
            v=MathAbs(d[0]);
            vv=MathAbs(d[0]);
            for(i=1; i<n; i++)
              {
               v=MathMin(v,MathAbs(d[i]));
               vv=MathMax(vv,MathAbs(d[i]));
              }
            CAp::Trace(StringFormat("diagD        = %.3E (min) ... %.3E (max)\n",v,vv));
           }
         if(m>0 && beta0!=0.0)
           {
            v=MathAbs(e[0]);
            vv=MathAbs(e[0]);
            for(i=1; i<m; i++)
              {
               v=MathMin(v,MathAbs(e[i]));
               vv=MathMax(vv,MathAbs(e[i]));
              }
            CAp::Trace(StringFormat("diagE        = %.3E (min) ... %.3E (max)\n",v,vv));
           }
         CAp::Trace("> LDLT factorization of entire KKT matrix\n");
         v=MathAbs(State.m_tmp0[0]);
         vv=MathAbs(State.m_tmp0[0]);
         for(i=1; i<State.m_tmpsparse0.m_N; i++)
           {
            v=MathMax(v,MathAbs(State.m_tmp0[i]));
            vv=MathMin(vv,MathAbs(State.m_tmp0[i]));
           }
         CAp::Trace(StringFormat("|D|          = %.3E (min) ... %.3E (max)\n",vv,v));
         v=0.0;
         for(i=0; i<State.m_tmpsparse0.m_N; i++)
           {
            k0=State.m_tmpsparse0.m_RIdx[i];
            k1=State.m_tmpsparse0.m_DIdx[i];
            for(k=k0; k<=k1; k++)
               v=MathMax(v,MathAbs(State.m_tmpsparse0.m_Vals[k]));
           }
         CAp::Trace(StringFormat("max(|L|)     = %.3E\n",v));
         CAp::Trace(StringFormat("diag-err     = %.3E (diagonal reproduction error)\n",MathSqrt(errsq / (1 + sumsq))));
        }
     }
//--- Done, integrity control
//--- check
   if(!CAp::Assert(State.m_factorizationpresent,__FUNCTION__+": integrity check failed"))
      return(false);
   State.m_repncholesky++;
//--- return result
   return(true);
  }
//+------------------------------------------------------------------+
//| A low - level function which solves KKT system whose regularized |
//| (!) factorization was prepared by VIPMFactorize(). No iterative  |
//| refinement is performed.                                         |
//| On input, right - hand - side is stored in DeltaXY; on output,   |
//| solution replaces DeltaXY.                                       |
//+------------------------------------------------------------------+
void CVIPMSolver::SolveReducedKKTSystem(CVIPMState &State,
                                        CRowDouble &deltaxy)
  {
//--- create variables
   int n=State.m_n;
   int nmain=State.m_nmain;
   int nslack=n-nmain;
   int m=State.m_mdense+State.m_msparse;
   int mdense=State.m_mdense;
   int msparse=State.m_msparse;
   int i=0;
//--- check
   if(!CAp::Assert(State.m_factorizationpresent,__FUNCTION__+": integrity check failed - factorization is not present"))
      return;
   if(!CAp::Assert(State.m_factorizationtype==0 || State.m_factorizationtype==1,__FUNCTION__+": unexpected factorization type"))
      return;
//--- Dense solving
   switch(State.m_factorizationtype)
     {
      case 0:
         //--- Compute
         //---     BzWave = inv(Dz+Rz)*Bz
         //---     ByWave = By+Az*BzWave
         //---     BhWave = Bh-Ah'*inv(EWave)*ByWave
         for(i=0; i<nslack; i++)
            deltaxy.Mul(nmain+i,State.m_factinvregdzrz[i]);
         CSparse::SparseGemV(State.m_combinedaslack,1.0,0,deltaxy,nmain,1.0,deltaxy,n);
         State.m_tmp1.Resize(m);
         for(i=0; i<m; i++)
            State.m_tmp1.Set(i,deltaxy[n+i]/State.m_factregewave[i]);
         CSparse::SparseGemV(State.m_sparseamain,-1.0,1,State.m_tmp1,0,1.0,deltaxy,0);
         CAblas::RMatrixGemVect(nmain,mdense,-1.0,State.m_denseamain,0,0,1,State.m_tmp1,msparse,1.0,deltaxy,0);
         //--- Compute Xh = -inv(HWave)*BhWave.
         //--- Zero out components corresponding to frozen variables.
         for(i=0; i<nmain; i++)
            deltaxy.Mul(i,-1);
         CAblas::RMatrixTrsVect(nmain,State.m_factdensehaug,0,0,false,false,0,deltaxy,0);
         CAblas::RMatrixTrsVect(nmain,State.m_factdensehaug,0,0,false,false,1,deltaxy,0);
         for(i=0; i<n; i++)
           {
            if(State.m_isfrozen[i])
               deltaxy.Set(i,0);
           }
         //--- Compute Y = inv(EWave)*(ByWave-Ah*Xh)
         CSparse::SparseGemV(State.m_sparseamain,-1.0,0,deltaxy,0,1.0,deltaxy,n);
         CAblas::RMatrixGemVect(mdense,nmain,-1.0,State.m_denseamain,0,0,0,deltaxy,0,1.0,deltaxy,n+msparse);
         for(i=0; i<m; i++)
            deltaxy.Mul(n+i,1.0/State.m_factregewave[i]);
         //--- Compute Xz = -(BzWave - inv(Dz+Rz)*Az'*y)
         State.m_tmp0.Resize(nslack);
         for(i=0; i<nslack; i++)
            State.m_tmp0.Set(i,0);
         CSparse::SparseGemV(State.m_combinedaslack,1.0,1,deltaxy,n,1.0,State.m_tmp0,0);
         for(i=0; i<nslack; i++)
            deltaxy.Set(nmain+i,-(deltaxy[nmain+i]-State.m_factinvregdzrz[i]*State.m_tmp0[i]));
         //--- Done
         break;
      //--- Sparse solving
      case 1:
         ReducedSystemSolve(State.m_reducedsparsesystem,deltaxy);
         for(i=0; i<n; i++)
           {
            if(State.m_isfrozen[i])
               deltaxy.Set(i,0);
           }
         break;
      default:
         CAp::Assert(false,__FUNCTION__+": integrity check failed - unexpected factorization");
         break;
     }
  }
//+------------------------------------------------------------------+
//| Generates precomputed temporary vectors and KKT factorization at |
//| the beginning of the current iteration.                          |
//| This function uses representation of KKT system inspired by      |
//| Vanderbei slack variable approach, but with additional           |
//| regularization being applied all along computations.             |
//| On successful factorization returns True; on failure returns     |
//| False - it is recommended to increase regularization parameter   |
//| and try one more time.                                           |
//| --- DESCRIPTION ------------------------------------------------ |
//| Initial KKT system proposed by Vanderbei has following structure:|
//|   (1) -DS*deltaT-I*deltaS = -mu/T+s+DELTAT*DELTAS/T = -GammaS    |
//|   (2) -DZ*deltaG-I*deltaZ = -mu/G+z+DELTAG*DELTAZ/G = -GammaZ    |
//|   (3) -DQ*deltaP-I*deltaQ = -mu/P+q+DELTAP*DELTAQ/P = -GammaQ    |
//|   (4) -DW*deltaV-I*deltaW = -mu/V+w+DELTAV*DELTAW/V = -GammaW    |
//|   (5) - I*deltaY-I*deltaQ+I*deltaV = y-q+v =  Beta               |
//|   (6) -H*deltaX+A'*deltaY+I*deltaZ-I*deltaS=c-A'*y-z+s+H*x=Sigma |
//|   (7)   A * deltaX - I * deltaW = b - A * x + w =  Rho           |
//|   (8)   I * deltaX - I * deltaG = l - x + g     =  Nu            |
//|   (9)  -I * deltaX - I * deltaT = -u + x + t    = -Tau           |
//|   (10) -I * deltaW - I * deltaP = -r + w + p    = -Alpha         |
//| where:                                                           |
//|   DS = diag(S / T)                                               |
//|   DZ = diag(Z / G)                                               |
//|   DQ = diag(Q / P)                                               |
//|   DW = diag(W / V)                                               |
//| This linear system is actually symmetric indefinite one, that can|
//| be regularized by modifying equations (5), (6), (7), (8), (9),   |
//| (10):                                                            |
//|   (5) -I*deltaY-I*deltaQ+I*deltaV-REG*deltaW = y+q-v+REG*w = Beta|
//|   (6) -(H+REG)*deltaX+A'*deltaY+I*deltaZ-I*deltaS =              |
//|         c-A'*y-z+s+(H+REG)*x = Sigma                             |
//|   (7)  A*deltaX-I*deltaW+REG*deltaY = b-A*x+w-REG*y = Rho        |
//|   (8)  I*deltaX-I*deltaG+REG*deltaZ = l-x+g-REG*z   = Nu         |
//|   (9) -I*deltaX-I*deltaT+REG*deltaS = -u+x+t-REG*s  = -Tau       |
//|   (10)-I*deltaW-I*deltaP+REG*deltaQ = -r+w+p-REG*q  = -Alpha     |
//| NOTE: regularizing equations (5)-(10) seems to be beneficial     |
//|       because their coefficients are well-normalized, usually    |
//|       having unit scale. Contrary to that, equations (1)-(4) are |
//|       wildly nonnormalized, and regularization ruins algorithm   |
//|       convergence.                                               |
//| From(1), (2), (3) and (4) we obtain                              |
//|      deltaT = (GammaS - I * deltaS) / DS                         |
//|      deltaG = (GammaZ - I * deltaZ) / DZ                         |
//|      deltaP = (GammaQ - I * deltaQ) / DQ                         |
//|      deltaV = (GammaW - I * deltaW) / DW                         |
//| and substitute them to equations to obtain                       |
//|   (5) -I*deltaY-I*deltaQ-(inv(DW)+REG)*deltaW =                  |
//|                      Beta - inv(DW) * GammaW  =  BetaCap         |
//|   (8)  I*deltaX+(inv(DZ)+REG)*deltaZ = Nu+inv(DZ)*GammaZ = NuCap |
//|   (9) I*deltaX+(inv(DS)+REG)*deltaS=-(Tau-inv(DS)*GammaS)=-TauCap|
//|   (10) -I * deltaW + (inv(DQ) + REG) * deltaQ =                  |
//|                   -(Alpha - inv(DQ) * GammaQ) = -AlphaCap        |
//|   (6)   A'*deltaY + I*deltaZ - I*deltaS - (H+REG)*deltaX =       |
//|               c-A' * y - z + s + (H + REG) * x    =  Sigma       |
//|   (7) REG*deltaY+A*deltaX-I*deltaW = b-A*x+w-REG*y = Rho         |
//| then, we obtain(here IRI stands for Invert - Regularize - Invert)|
//|         DQIRI  = inv(inv(DQ) + REG)                              |
//|         DZIRI  = inv(inv(DZ) + REG)                              |
//|         DSIRI  = inv(inv(DS) + REG)                              |
//|         deltaQ = (I * deltaW - AlphaCap) * DQIRI                 |
//|         deltaZ = (NuCap - I * deltaX) * DZIRI                    |
//|         deltaS = (I * deltaX - TauCap) * DSIRI                   |
//|         DWIR   = inv(DW) + REG                                   |
//| and after substitution                                           |
//|   (5) -I*deltaY-(DQIRI+DWIR)*deltaW = BetaCap-DQIRI*AlphaCap     |
//|   (6)  A'*deltaY - (H+REG+DSIRI+DZIRI)*deltaX =                  |
//|                     Sigma-DSIRI*TauCap-DZIRI*NuCap               |
//|   (7)  REG * deltaY + A * deltaX - I * deltaW        = Rho       |
//| finally, we obtain                                               |
//|         DE     = inv(DQIRI + DWIR)                               |
//|         DER    = DE + REG                                        |
//|         DDR    = DSIRI + DZIRI + REG                             |
//| deltaW = -(BetaCap - DQIRI * AlphaCap + I * deltaY) * DE         |
//| and after substitution                                           |
//|   (6)  - (H + DDR) * deltaX +  A'*deltaY =                       |
//|                        Sigma-DSIRI*TauCap-DZIRI*NuCap            |
//|   (7) A*deltaX + DER*deltaY = Rho-DE*(BetaCap-DQIRI*AlphaCap)    |
//+------------------------------------------------------------------+
bool CVIPMSolver::VIPMPrecomputeNewtonFactorization(CVIPMState &State,
                                                    CVIPMVars &v0,
                                                    double regeps,
                                                    double modeps,
                                                    double dampeps,
                                                    double dampfree)
  {
//--- create variables
   bool result=false;
   int  n=State.m_n;
   int  m=State.m_mdense+State.m_msparse;
   int  i=0;

   State.m_diagdz=vector<double>::Zeros(n);
   State.m_diagdzi=vector<double>::Zeros(n);
   State.m_diagdziri=vector<double>::Zeros(n);
   State.m_diagds=vector<double>::Zeros(n);
   State.m_diagdsi=vector<double>::Zeros(n);
   State.m_diagdsiri=vector<double>::Zeros(n);
   State.m_diagdw=vector<double>::Zeros(m);
   State.m_diagdwi=vector<double>::Zeros(m);
   State.m_diagdwir=vector<double>::Zeros(m);
   State.m_diagdq=vector<double>::Zeros(m);
   State.m_diagdqi=vector<double>::Zeros(m);
   State.m_diagdqiri=vector<double>::Zeros(m);
   State.m_diagddr.Resize(n);
   State.m_diagde.Resize(m);
   State.m_diagder.Resize(m);
//--- Handle temporary matrices arising due to box constraints
   for(i=0; i<n; i++)
     {
      //--- Lower bound: G*inv(Z) and Z*inv(G)
      if(State.m_hasgz[i])
        {
         //--- check
         if(!CAp::Assert(v0.m_g[i]>0.0 && v0.m_z[i]>0.0,__FUNCTION__+": integrity failure - G[i]<=0 or Z[i]<=0"))
            return(false);
         State.m_diagdz.Set(i,v0.m_z[i]/v0.m_g[i]);
         State.m_diagdzi.Set(i,1.0/State.m_diagdz[i]);
         State.m_diagdziri.Set(i,1.0/(State.m_diagdzi[i]+regeps));
        }
      else
        {
         //--- check
         if(!CAp::Assert(v0.m_g[i]==0.0 && v0.m_z[i]==0.0,__FUNCTION__+": integrity failure - G[i]<>0 or Z[i]<>0 for absent lower bound"))
            return(false);
        }
      //--- Upper bound: T*inv(S) and S*inv(T)
      if(State.m_hasts[i])
        {
         //--- check
         if(!CAp::Assert(v0.m_t[i]>0.0 && v0.m_s[i]>0.0,__FUNCTION__+": integrity failure - T[i]<=0 or S[i]<=0"))
            return(false);
         State.m_diagds.Set(i,v0.m_s[i]/v0.m_t[i]);
         State.m_diagdsi.Set(i,1/State.m_diagds[i]);
         State.m_diagdsiri.Set(i,1/(State.m_diagdsi[i]+regeps));
        }
      else
        {
         //--- check
         if(!CAp::Assert(v0.m_t[i]==0.0 && v0.m_s[i]==0.0,__FUNCTION__+": integrity failure - T[i]<>0 or S[i]<>0 for absent upper bound"))
            return(false);
        }
      //--- Diagonal term D
      State.m_diagddr.Set(i,State.m_diagdziri[i]+State.m_diagdsiri[i]+regeps);
      if(!State.m_hasgz[i] && !State.m_hasts[i])
         State.m_diagddr.Add(i,dampfree);
     }
//--- Handle temporary matrices arising due to linear constraints: with lower bound B[]
//--- or with lower and upper bounds.
   for(i=0; i<m; i++)
     {
      //--- Lower bound
      if(State.m_haswv[i])
        {
         //--- check
         if(!CAp::Assert(v0.m_v[i]>0.0 && v0.m_w[i]>0.0,__FUNCTION__+": integrity failure - V[i]<=0 or W[i]<=0"))
            return(false);
         State.m_diagdw.Set(i,v0.m_w[i]/v0.m_v[i]);
         State.m_diagdwi.Set(i,1/State.m_diagdw[i]);
         State.m_diagdwir.Set(i,State.m_diagdwi[i]+regeps);
        }
      else
        {
         //--- check
         if(!CAp::Assert(v0.m_v[i]==0.0 && v0.m_w[i]==0.0,__FUNCTION__+": integrity failure - V[i]<>0 or W[i]<>0 for linear equality constraint"))
            return(false);
        }
      //--- Upper bound
      if(State.m_haspq[i])
        {
         //--- check
         if(!CAp::Assert(v0.m_p[i]>0.0 && v0.m_q[i]>0.0,__FUNCTION__+": integrity failure - P[i]<=0 or Q[i]<=0"))
            return(false);
         State.m_diagdq.Set(i,v0.m_q[i]/v0.m_p[i]);
         State.m_diagdqi.Set(i,1/State.m_diagdq[i]);
         State.m_diagdqiri.Set(i,1/(State.m_diagdqi[i]+regeps));
        }
      else
        {
         //--- check
         if(!CAp::Assert(v0.m_p[i]==0.0 && v0.m_q[i]==0.0,__FUNCTION__+": integrity failure - P[i]<>0 or Q[i]<>0 for absent linear constraint"))
            return(false);
        }
      //--- Diagonal term E
      if(State.m_haswv[i] || State.m_haspq[i])
         State.m_diagde.Set(i,1/(State.m_diagdwir[i]+State.m_diagdqiri[i]));
      else
         State.m_diagde.Set(i,0.0);
      State.m_diagder.Set(i,State.m_diagde[i]+regeps);
     }
//--- Perform factorization
   result=VIPMFactorize(State,1.0,State.m_diagddr,1.0,State.m_diagder,0.0,0.0,modeps,dampeps);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Solves KKT system stored in VIPMState with user - passed RHS.    |
//| Sol must be preallocated VIPMVars object whose initial values are|
//| ignored.                                                         |
//+------------------------------------------------------------------+
void CVIPMSolver::SolveKKTSystem(CVIPMState &State,
                                 CVIPMRightHandSide &rhs,
                                 CVIPMVars &sol)
  {
//--- create variables
   int n=State.m_n;
   int m=State.m_mdense+State.m_msparse;
   int i=0;
//--- Compute elimination temporaries
//--- RhsAlphaCap  = RhsAlpha - InvDQ*GammaQ
//--- RhsNuCap     = RhsNu    + InvDZ*GammaZ
//--- RhsTauCap    = RhsTau   - InvDS*GammaS
//--- RhsBetaCap   = RhsBeta  - InvDW*GammaW
   State.m_rhsnucap.Resize(n);
   State.m_rhstaucap.Resize(n);
   State.m_rhsbetacap.Resize(m);
   State.m_rhsalphacap.Resize(m);
   CAblasF::RCopyNegMulAddV(m,State.m_diagdqi,rhs.m_gammaq,rhs.m_alpha,State.m_rhsalphacap);
   CAblasF::RCopyMulAddV(n,State.m_diagdzi,rhs.m_gammaz,rhs.m_nu,State.m_rhsnucap);
   CAblasF::RCopyNegMulAddV(n,State.m_diagdsi,rhs.m_gammas,rhs.m_tau,State.m_rhstaucap);
   CAblasF::RCopyNegMulAddV(m,State.m_diagdwi,rhs.m_gammaw,rhs.m_beta,State.m_rhsbetacap);
//--- Solve reduced KKT system
   State.m_deltaxy.Resize(n+m);
   for(i=0; i<n; i++)
      State.m_deltaxy.Set(i,rhs.m_sigma[i]-State.m_diagdziri[i]*State.m_rhsnucap[i]-State.m_diagdsiri[i]*State.m_rhstaucap[i]);
   for(i=0; i<m; i++)
      State.m_deltaxy.Set(n+i,rhs.m_rho[i]-State.m_diagde[i]*(State.m_rhsbetacap[i]-State.m_diagdqiri[i]*State.m_rhsalphacap[i]));
   SolveReducedKKTSystem(State,State.m_deltaxy);
//--- Perform backsubstitution
   for(i=0; i<n; i++)
     {
      sol.m_x.Set(i,State.m_deltaxy[i]);
      sol.m_s.Set(i,State.m_diagdsiri[i]*(sol.m_x[i]-State.m_rhstaucap[i]));
      sol.m_z.Set(i,State.m_diagdziri[i]*(State.m_rhsnucap[i]-sol.m_x[i]));
      sol.m_g.Set(i,State.m_diagdzi[i]*(rhs.m_gammaz[i]-sol.m_z[i]));
      sol.m_t.Set(i,State.m_diagdsi[i]*(rhs.m_gammas[i]-sol.m_s[i]));
     }
   for(i=0; i<m; i++)
     {
      sol.m_y.Set(i,State.m_deltaxy[n+i]);
      sol.m_w.Set(i,-(State.m_diagde[i]*(State.m_rhsbetacap[i]-State.m_diagdqiri[i]*State.m_rhsalphacap[i]+sol.m_y[i])));
      sol.m_q.Set(i,State.m_diagdqiri[i]*(sol.m_w[i]-State.m_rhsalphacap[i]));
      sol.m_v.Set(i,State.m_diagdwi[i]*(rhs.m_gammaw[i]-sol.m_w[i]));
      sol.m_p.Set(i,State.m_diagdqi[i]*(rhs.m_gammaq[i]-sol.m_q[i]));
     }
  }
//+------------------------------------------------------------------+
//| Compute VIPM step by solving KKT system.                         |
//| VDResult must be preallocated VIPMVars object  whose  initial    |
//| values are ignored.                                              |
//| Returns False on failure to compute step direction with          |
//| reasonable accuracy (it is advised to terminate iterations       |
//| immediately).                                                    |
//+------------------------------------------------------------------+
bool CVIPMSolver::VIPMComputeStepDirection(CVIPMState &State,
                                           CVIPMVars &v0,
                                           double muestimate,
                                           CVIPMVars &vdestimate,
                                           CVIPMVars &vdresult,
                                           double reg,
                                           bool isdampepslarge)
  {
//--- create variables
   int    n=State.m_n;
   int    m=State.m_mdense+State.m_msparse;
   double vrhsprim2=0;
   double vrhsdual2=0;
   double vrhscmpl2=0;
   double vresprim2=0;
   double vresdual2=0;
   double vrescmpl2=0;
   double vrhspriminf=0;
   double vrhsdualinf=0;
   double vrespriminf=0;
   double vresdualinf=0;
   double badres=1.01;
   double verybadres=1.0E3;
   double residualgrowth=0;
   bool   primaldestabilized=false;
   bool   dualdestabilized=false;
//--- Initial m_solver report
   if(State.m_dotrace)
      CAp::Trace("--- detailed KKT solver report ---------------------------------------------------------------------\n");
//--- Solve KKT system with right-hand sides coming from primal, dual
//--- and complementary slackness conditions. Analyze solution,
//--- terminate immediately if primal/dual residuals are way too high.
   RHSCompute(State,v0,muestimate,vdestimate,State.m_rhs,reg);
   vrhsprim2=RHSPrimal2(State.m_rhs);
   vrhsdual2=RHSDual2(State.m_rhs);
   vrhscmpl2=RHSCompl2(State.m_rhs);
   vrhspriminf=RHSPrimalInf(State.m_rhs);
   vrhsdualinf=RHSDualInf(State.m_rhs);
   if(State.m_dotrace)
     {
      CAp::Trace("> primal/dual/complementarity right-hand-side\n");
      CAp::Trace(StringFormat("rhs-prim     = %.3E (2-norm)\n",MathSqrt(vrhsprim2)));
      CAp::Trace(StringFormat("rhs-dual     = %.3E (2-norm)\n",MathSqrt(vrhsdual2)));
      CAp::Trace(StringFormat("rhs-cmpl     = %.3E (2-norm)\n",MathSqrt(vrhscmpl2)));
     }
   SolveKKTSystem(State,State.m_rhs,vdresult);
   RHSSubtract(State,State.m_rhs,v0,vdresult,reg);
   vresprim2=RHSPrimal2(State.m_rhs);
   vresdual2=RHSDual2(State.m_rhs);
   vrescmpl2=RHSCompl2(State.m_rhs);
   vrespriminf=RHSPrimalInf(State.m_rhs);
   vresdualinf=RHSDualInf(State.m_rhs);
   if(State.m_dotrace)
     {
      CAp::Trace("> primal/dual/complementarity residuals compared with RHS\n");
      CAp::Trace(StringFormat("res/rhs prim = %.3E\n",MathSqrt(vresprim2 / CApServ::Coalesce(vrhsprim2,1))));
      CAp::Trace(StringFormat("res/rhs dual = %.3E\n",MathSqrt(vresdual2 / CApServ::Coalesce(vrhsdual2,1))));
      CAp::Trace(StringFormat("res/rhs cmpl = %.3E\n",MathSqrt(vrescmpl2 / CApServ::Coalesce(vrhscmpl2,1))));
      CAp::Trace(StringFormat("res/rhs all  = %.3E\n",MathSqrt((vresprim2 + vresdual2 + vrescmpl2) / CApServ::Coalesce(vrhsprim2 + vrhsdual2 + vrhscmpl2,1))));
     }
   primaldestabilized=(vrhspriminf<=State.m_epsp && vrespriminf>=MathMax(verybadres*vrhspriminf,State.m_epsp));
   dualdestabilized=(vrhsdualinf<=State.m_epsd && vresdualinf>=MathMax(verybadres*vrhsdualinf,State.m_epsd));
   residualgrowth=MathSqrt((vresprim2+vresdual2+vrescmpl2)/CApServ::Coalesce(vrhsprim2+vrhsdual2+vrhscmpl2,1));
   if((primaldestabilized || dualdestabilized) && residualgrowth>(0.01*MathSqrt(CMath::m_machineepsilon)) && !isdampepslarge)
     {
      if(State.m_dotrace)
         CAp::Trace("> primal/dual residual growth is too high,signaling presence of numerical errors\n");
      return(false);
     }
   if(residualgrowth>badres)
     {
      if(State.m_dotrace)
         CAp::Trace("> total residual is too high,signaling presence of numerical errors\n");
      return(false);
     }
//--- return result
   return(true);
  }
//+------------------------------------------------------------------+
//| This function estimates primal and dual step lengths (subject to |
//| step decay parameter, which should be in [0, 1] range).          |
//| Current version returns same step lengths for primal and dual    |
//| steps.                                                           |
//| INPUT PARAMETERS:                                                |
//|   State    -  solver State                                       |
//|   V0       -  current point (we ignore one stored in             |
//|               State.Current)                                     |
//|   VS       -  step direction                                     |
//|   StepDecay-  decay parameter, the step is multiplied by this    |
//|               coefficient. 1.0 corresponds to full step length   |
//|               being returned. Values in (0, 1] range.            |
//| OUTPUT PARAMETERS:                                               |
//|   AlphaP   -  primal step(after applying decay coefficient)      |
//|   AlphaD   -  dual   step(after applying decay coefficient)      |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMComputeStepLength(CVIPMState &State,
                                        CVIPMVars &v0,
                                        CVIPMVars &vs,
                                        double stepdecay,
                                        double &alphap,
                                        double &alphad)
  {
//--- create variables
   int    n=State.m_n;
   int    m=State.m_mdense+State.m_msparse;
   int    i=0;
   double alpha=0;
   alphap=0;
   alphad=0;
//--- check
   if(!CAp::Assert(n==v0.m_n && m==v0.m_m,__FUNCTION__+": sizes mismatch"))
      return;
   alphap=1;
   alphad=1;
   for(i=0; i<n; i++)
     {
      //--- Primal
      if(vs.m_g[i]<0.0)
         alphap=CApServ::SafeMinPosRV(v0.m_g[i],-vs.m_g[i],alphap);
      if(vs.m_t[i]<0.0)
         alphap=CApServ::SafeMinPosRV(v0.m_t[i],-vs.m_t[i],alphap);
      //--- Dual
      if(vs.m_z[i]<0.0)
         alphad=CApServ::SafeMinPosRV(v0.m_z[i],-vs.m_z[i],alphad);
      if(vs.m_s[i]<0.0)
         alphad=CApServ::SafeMinPosRV(v0.m_s[i],-vs.m_s[i],alphad);
     }
   for(i=0; i<m; i++)
     {
      //--- Primal
      if(vs.m_w[i]<0.0)
         alphap=CApServ::SafeMinPosRV(v0.m_w[i],-vs.m_w[i],alphap);
      if(vs.m_p[i]<0.0)
         alphap=CApServ::SafeMinPosRV(v0.m_p[i],-vs.m_p[i],alphap);
      //--- Dual
      if(vs.m_v[i]<0.0)
         alphad=CApServ::SafeMinPosRV(v0.m_v[i],-vs.m_v[i],alphad);
      if(vs.m_q[i]<0.0)
         alphad=CApServ::SafeMinPosRV(v0.m_q[i],-vs.m_q[i],alphad);
     }
//--- Because we may solve QP problem, step length has to be same for primal and dual variables
   alpha=MathMin(alphap,alphad);
//--- Apply decay
   alphap=stepdecay*alpha;
   alphad=stepdecay*alpha;
  }
//+------------------------------------------------------------------+
//| This function performs IPM step, updates  iteration  counts  and |
//| performs following additional checks:                            |
//|   * it monitors status of box/linear constraints  and  smoothly  |
//|     drops ones with too large bounds (a variable or linear sum is|
//|     well below constraint bound for several iterations)          |
//| INPUT PARAMETERS:                                                |
//|   State    -  m_solver State                                     |
//|   AlphaP   -  primal step to perform                             |
//|   AlphaD   -  dual step to perform                               |
//| OUTPUT PARAMETERS:                                               |
//+------------------------------------------------------------------+
void CVIPMSolver::VIPMPerformStep(CVIPMState &State,
                                  double alphap,
                                  double alphad)
  {
//--- Perform step
   VarsAddStep(State.m_current,State.m_deltacorr,alphap,alphad);
//--- Update iterations count
   State.m_repiterationscount++;
  }
//+------------------------------------------------------------------+
//| Compute primal / dual errors and complementarity gap             |
//+------------------------------------------------------------------+
void CVIPMSolver::ComputeErrors(CVIPMState &State,
                                double &errp2,
                                double &errd2,
                                double &errpinf,
                                double &errdinf,
                                double &egap)
  {
//--- create variables
   int    n=State.m_n;
   int    m=State.m_mdense+State.m_msparse;
   int    i=0;
   int    cntp2=0;
   int    cntd2=0;
   double v=0;

   errp2=0;
   errd2=0;
   errpinf=0;
   errdinf=0;
   egap=0;
//--- Compute primal and dual infeasibilities
   VIPMMultiply(State,State.m_current.m_x,State.m_current.m_y,State.m_tmphx,State.m_tmpax,State.m_tmpaty);
   cntp2=0;
   errp2=0;
   errpinf=0;
   for(i=0; i<m; i++)
     {
      v=State.m_tmpax[i]-State.m_current.m_w[i]-State.m_b[i];
      errp2=errp2+v*v;
      errpinf=MathMax(errpinf,MathAbs(v));
      cntp2++;
      if(State.m_haspq[i])
        {
         v=State.m_current.m_w[i]+State.m_current.m_p[i]-State.m_r[i];
         errp2=errp2+v*v;
         errpinf=MathMax(errpinf,MathAbs(v));
         cntp2++;
        }
     }
   for(i=0; i<n; i++)
     {
      if(State.m_hasgz[i])
        {
         v=State.m_current.m_x[i]-State.m_current.m_g[i]-State.m_bndl[i];
         errp2+=v*v;
         errpinf=MathMax(errpinf,MathAbs(v));
         cntp2++;
        }
      if(State.m_hasts[i])
        {
         v=State.m_current.m_x[i]+State.m_current.m_t[i]-State.m_bndu[i];
         errp2+=v*v;
         errpinf=MathMax(errpinf,MathAbs(v));
         cntp2++;
        }
     }
   errp2=MathSqrt(errp2/CApServ::Coalesce(cntp2,1));
   cntd2=0;
   errd2=0;
   errdinf=0;
   for(i=0; i<n; i++)
     {
      if(!State.m_isfrozen[i])
        {
         v=State.m_tmphx[i]+State.m_c[i]-State.m_tmpaty[i];
         if(State.m_hasgz[i])
            v-=State.m_current.m_z[i];
         if(State.m_hasts[i])
            v+=State.m_current.m_s[i];
         errd2+=v*v;
         errdinf=MathMax(errdinf,MathAbs(v));
         cntd2++;
        }
     }
   for(i=0; i<m; i++)
     {
      v=0;
      if(State.m_haswv[i])
         v=State.m_current.m_y[i]-State.m_current.m_v[i];
      if(State.m_haspq[i])
         v+=State.m_current.m_q[i];
      errd2+=v*v;
      errdinf=MathMax(errdinf,MathAbs(v));
      if(State.m_haswv[i] || State.m_haspq[i])
         cntd2++;
     }
   errd2=MathSqrt(errd2/CApServ::Coalesce(cntd2,1));
   egap=VarsComputeComplementarityGap(State.m_current)/(1.0+MathAbs(VIPMTarget(State,State.m_current.m_x)));
  }
//+------------------------------------------------------------------+
//| Performs integrity checks for current point and step             |
//+------------------------------------------------------------------+
void CVIPMSolver::RunIntegrityChecks(CVIPMState &State,
                                     CVIPMVars &v0,
                                     CVIPMVars &vd,
                                     double alphap,
                                     double alphad)
  {
//--- create variables
   int n=State.m_n;
   int m=State.m_mdense+State.m_msparse;
   int i=0;
//--- check
   if(!CAp::Assert(MathIsValidNumber(alphap) && alphap>=0.0,__FUNCTION__+": bad AlphaP"))
      return;
   if(!CAp::Assert(MathIsValidNumber(alphad) && alphad>=0.0,__FUNCTION__+": bad AlphaD"))
      return;
   for(i=0; i<n; i++)
     {
      if(State.m_hasgz[i])
        {
         //--- check
         if(!CAp::Assert(!State.m_isfrozen[i],__FUNCTION__+": integrity failure - X[I] is frozen"))
            return;
         if(!CAp::Assert(v0.m_g[i]>0.0 && v0.m_z[i]>0.0,__FUNCTION__+": integrity failure - G[i]<=0 or Z[i]<=0"))
            return;
        }
      else
        {
         //--- check
         if(!CAp::Assert(v0.m_g[i]==0.0 && v0.m_z[i]==0.0,__FUNCTION__+": integrity failure - G[i]<>0 or Z[i]<>0 for absent lower bound"))
            return;
         if(!CAp::Assert(vd.m_g[i]==0.0 && vd.m_z[i]==0.0,__FUNCTION__+": integrity failure - G[i]<>0 or Z[i]<>0 for absent lower bound"))
            return;
        }
      if(State.m_hasts[i])
        {
         if(!CAp::Assert(!State.m_isfrozen[i],__FUNCTION__+": integrity failure - X[I] is frozen"))
            return;
         if(!CAp::Assert(v0.m_t[i]>0.0 && v0.m_s[i]>0.0,__FUNCTION__+": integrity failure - T[i]<=0 or S[i]<=0"))
            return;
        }
      else
        {
         if(!CAp::Assert(v0.m_t[i]==0.0 && v0.m_s[i]==0.0,__FUNCTION__+": integrity failure - T[i]<>0 or S[i]<>0 for absent upper bound"))
            return;
         if(!CAp::Assert(vd.m_t[i]==0.0 && vd.m_s[i]==0.0,__FUNCTION__+": integrity failure - T[i]<>0 or S[i]<>0 for absent upper bound"))
            return;
        }
     }
   for(i=0; i<m; i++)
     {
      if(!CAp::Assert(State.m_haswv[i] || !State.m_haspq[i],__FUNCTION__+": inconsistent HasWV/HasPQ"))
         return;
      if(State.m_haswv[i])
        {
         if(!CAp::Assert(v0.m_v[i]>0.0 && v0.m_w[i]>0.0,__FUNCTION__+": integrity failure - V[i]<=0 or W[i]<=0"))
            return;
        }
      else
        {
         if(!CAp::Assert(v0.m_v[i]==0.0 && v0.m_w[i]==0.0,__FUNCTION__+": integrity failure - V[i]<>0 or W[i]<>0 for linear equality constraint"))
            return;
         if(!CAp::Assert(vd.m_v[i]==0.0 && vd.m_w[i]==0.0,__FUNCTION__+": integrity failure - V[i]<>0 or W[i]<>0 for linear equality constraint"))
            return;
        }
      if(State.m_haspq[i])
        {
         if(!CAp::Assert(v0.m_p[i]>0.0 && v0.m_q[i]>0.0,__FUNCTION__+": integrity failure - P[i]<=0 or Q[i]<=0"))
            return;
        }
      else
        {
         if(!CAp::Assert(v0.m_p[i]==0.0 && v0.m_q[i]==0.0,__FUNCTION__+": integrity failure - P[i]<>0 or Q[i]<>0 for absent range of linear constraint"))
            return;
         if(!CAp::Assert(vd.m_p[i]==0.0 && vd.m_q[i]==0.0,__FUNCTION__+": integrity failure - P[i]<>0 or Q[i]<>0 for absent range of linear constraint"))
            return;
        }
     }
  }
//+------------------------------------------------------------------+
//| Evaluate progress so far, outputs trace data, if requested to do |
//| so.                                                              |
//+------------------------------------------------------------------+
void CVIPMSolver::TraceProgress(CVIPMState &State,
                                double mu,
                                double muaff,
                                double sigma,
                                double alphap,
                                double alphad)
  {
//--- create variables
   int    n=State.m_n;
   int    m=State.m_mdense+State.m_msparse;
   int    i=0;
   double v=0;
   double errp2=0;
   double errd2=0;
   double errpinf=0;
   double errdinf=0;
   double errgap=0;

   if(!State.m_dotrace)
      return;
//--- Print high-level information
   ComputeErrors(State,errp2,errd2,errpinf,errdinf,errgap);
   CAp::Trace("--- step report ------------------------------------------------------------------------------------\n");
   CAp::Trace("> step information\n");
   CAp::Trace(StringFormat("mu_init = %.3E    (at the beginning)\n",mu));
   CAp::Trace(StringFormat("mu_aff  = %.3E    (by affine scaling step)\n",muaff));
   CAp::Trace(StringFormat("sigma   = %.3E    (centering parameter)\n",sigma));
   CAp::Trace(StringFormat("alphaP  = %.3E    (primal step)\n",alphap));
   CAp::Trace(StringFormat("alphaD  = %.3E    (dual   step)\n",alphad));
   CAp::Trace(StringFormat("mu_cur  = %.3E    (after the step)\n",VARSComputeMu(State,State.m_current)));
   CAp::Trace("> errors\n");
   CAp::Trace(StringFormat("errP    = %.3E    (primal infeasibility,inf-norm)\n",errpinf));
   CAp::Trace(StringFormat("errD    = %.3E    (dual infeasibility, inf-norm)\n",errdinf));
   CAp::Trace(StringFormat("errGap  = %.3E    (complementarity gap)\n",errgap));
   CAp::Trace("> current point information (inf-norm)\n");
   CAp::Trace(StringFormat("|X|=%8.1E,|G|=%8.1E,|T|=%8.1E,|W|=%8.1E,|P|=%8.1E\n",CAblasF::RMaxAbsV(n,State.m_current.m_x),CAblasF::RMaxAbsV(n,State.m_current.m_g),CAblasF::RMaxAbsV(n,State.m_current.m_t),CAblasF::RMaxAbsV(m,State.m_current.m_w),CAblasF::RMaxAbsV(m,State.m_current.m_p)));
   CAp::Trace(StringFormat("|Y|=%8.1E,|Z|=%8.1E,|S|=%8.1E,|V|=%8.1E,|Q|=%8.1E\n",CAblasF::RMaxAbsV(m,State.m_current.m_y),CAblasF::RMaxAbsV(n,State.m_current.m_z),CAblasF::RMaxAbsV(n,State.m_current.m_s),CAblasF::RMaxAbsV(m,State.m_current.m_v),CAblasF::RMaxAbsV(m,State.m_current.m_q)));
//--- Print variable stats, if required
   if(State.m_dotrace)
     {
      CAp::Trace("--- variable statistics ----------------------------------------------------------------------------\n");
      CAp::Trace("> smallest values for nonnegative vars\n");
      CAp::Trace(StringFormat("primal:       minG=%8.1E  minT=%8.1E  minW=%8.1E  minP=%8.1E\n",MinNZ(State.m_current.m_g,n),MinNZ(State.m_current.m_t,n),MinNZ(State.m_current.m_w,m),MinNZ(State.m_current.m_p,m)));
      CAp::Trace(StringFormat("dual:         minZ=%8.1E  minS=%8.1E  minV=%8.1E  minQ=%8.1E\n",MinNZ(State.m_current.m_z,n),MinNZ(State.m_current.m_s,n),MinNZ(State.m_current.m_v,m),MinNZ(State.m_current.m_q,m)));
      CAp::Trace("> min and max complementary slackness\n");
      CAp::Trace(StringFormat("min:            GZ=%8.1E    TS=%8.1E    WV=%8.1E    PQ=%8.1E\n",MinProdNZ(State.m_current.m_g,State.m_current.m_z,n),MinProdNZ(State.m_current.m_t,State.m_current.m_s,n),MinProdNZ(State.m_current.m_w,State.m_current.m_v,m),MinProdNZ(State.m_current.m_p,State.m_current.m_q,m)));
      CAp::Trace(StringFormat("max:            GZ=%8.1E    TS=%8.1E    WV=%8.1E    PQ=%8.1E\n",MaxProdNZ(State.m_current.m_g,State.m_current.m_z,n),MaxProdNZ(State.m_current.m_t,State.m_current.m_s,n),MaxProdNZ(State.m_current.m_w,State.m_current.m_v,m),MaxProdNZ(State.m_current.m_p,State.m_current.m_q,m)));
     }
//--- Detailed output (all variables values, not suited for high-dimensional problems)
   if(State.m_dodetailedtrace)
     {
      VIPMMultiply(State,State.m_current.m_x,State.m_current.m_y,State.m_tmphx,State.m_tmpax,State.m_tmpaty);
      State.m_tmplaggrad=vector<double>::Zeros(n);
      for(i=0; i<n; i++)
        {
         if(!State.m_isfrozen[i])
           {
            v=State.m_tmphx[i]+State.m_c[i]-State.m_tmpaty[i];
            if(State.m_hasgz[i])
               v-=State.m_current.m_z[i];
            if(State.m_hasts[i])
               v+=State.m_current.m_s[i];
            State.m_tmplaggrad.Set(i,v);
           }
        }
      CAp::Trace("--- printing raw data (prior to applying variable scales and Shifting by XOrigin) ------------------\n");
      CAp::Trace("X (raw)         = ");
      CApServ::TraceVectorUnscaledUnshiftedAutopRec(State.m_current.m_x,n,State.m_scl,true,State.m_xorigin,true);
      CAp::Trace("\n");
      CAp::Trace("--- printing scaled data (after applying variable scales and Shifting by XOrigin) ------------------\n");
      CAp::Trace("> reporting X,Lagrangian gradient\n");
      CAp::Trace("Xnew            = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_x,0,n);
      CAp::Trace("\n");
      CAp::Trace("Lag-grad        = ");
      CApServ::TraceVectorAutopRec(State.m_tmplaggrad,0,n);
      CAp::Trace("\n");
      CAp::Trace("--- printing new point -----------------------------------------------------------------------------\n");
      CAp::Trace("> primal slacks and dual multipliers for box constraints\n");
      CAp::Trace("G (L prim slck) = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_g,0,n);
      CAp::Trace("\n");
      CAp::Trace("Z (L dual mult) = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_z,0,n);
      CAp::Trace("\n");
      CAp::Trace("T (U prim slck) = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_t,0,n);
      CAp::Trace("\n");
      CAp::Trace("S (U dual mult) = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_s,0,n);
      CAp::Trace("\n");
      CAp::Trace("> primal slacks and dual multipliers for linear constraints,B/R stand for B<=Ax<=B+R\n");
      CAp::Trace("Y (lag mult)    = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_y,0,m);
      CAp::Trace("\n");
      CAp::Trace("W (B prim slck) = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_w,0,m);
      CAp::Trace("\n");
      CAp::Trace("V (B dual mult) = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_v,0,m);
      CAp::Trace("\n");
      CAp::Trace("P (R prim slck) = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_p,0,m);
      CAp::Trace("\n");
      CAp::Trace("Q (R dual mult) = ");
      CApServ::TraceVectorAutopRec(State.m_current.m_q,0,m);
      CAp::Trace("\n");
     }
   CAp::Trace("\n");
  }
//+------------------------------------------------------------------+
//| Compute right - hand side for KKT system.                        |
//| INPUT PARAMETERS:                                                |
//|   State    -  IPM State                                          |
//|   V0       -  current point(used to compute RHS)                 |
//|   MuEstimate - estimate of Mu(can be zero)                       |
//|   DirEstimate - estimate of delta's (can be zero)                |
//| OUTPUT PARAMETERS:                                               |
//|   Rhs      -  RHS                                                |
//+------------------------------------------------------------------+
void CVIPMSolver::RHSCompute(CVIPMState &State,
                             CVIPMVars &v0,
                             double muestimate,
                             CVIPMVars &direstimate,
                             CVIPMRightHandSide &rhs,
                             double reg)
  {
//--- create variables
   int n=State.m_n;
   int m=State.m_mdense+State.m_msparse;
   int i=0;
//--- Allocate
   rhs.m_sigma.Resize(n);
   rhs.m_nu.Resize(n);
   rhs.m_tau.Resize(n);
   rhs.m_gammaz.Resize(n);
   rhs.m_gammas.Resize(n);
   rhs.m_gammaw.Resize(m);
   rhs.m_gammaq.Resize(m);
   rhs.m_beta=vector<double>::Zeros(m);
   rhs.m_rho=vector<double>::Zeros(m);
   rhs.m_alpha=vector<double>::Zeros(m);
//--- Compute products H*x, A*x, A^T*y
//--- We compute these products in one location for the sake of simplicity.
   VIPMMultiply(State,v0.m_x,v0.m_y,State.m_tmphx,State.m_tmpax,State.m_tmpaty);
//--- Compute right-hand side:
//--- Rho      = b - A*x + w
//--- Nu       = l - x + g
//--- Tau      = u - x - t
//--- Alpha    = r - w - p
//--- Sigma    = c - A^T*y - z + s + (H+REG)*x
//--- Beta     = y + q - v
   for(i=0; i<m; i++)
     {
      rhs.m_rho.Set(i,State.m_b[i]-State.m_tmpax[i]-reg*v0.m_y[i]);
      if(State.m_haswv[i])
        {
         //--- Inequality/range constraint
         rhs.m_rho.Add(i,v0.m_w[i]);
        }
      else
        {
         //--- Equality constraint without slack variables, W[i]=0
         if(!CAp::Assert(v0.m_w[i]==0.0,__FUNCTION__+": W[i]<>0 for linear equality constraint"))
            return;
        }
     }
   for(i=0; i<n; i++)
     {
      if(State.m_hasgz[i])
        {
         //--- Lower bound is present
         rhs.m_nu.Set(i,State.m_bndl[i]-v0.m_x[i]+v0.m_g[i]-reg*v0.m_z[i]);
        }
      else
        {
         //--- Lower bound is absent, g.Set(i, 0
         if(!CAp::Assert(v0.m_g[i]==0.0,__FUNCTION__+": G[i]<>0 for absent constraint"))
            return;
         rhs.m_nu.Set(i,0);
        }
     }
   for(i=0; i<n; i++)
     {
      if(State.m_hasts[i])
        {
         //--- Upper bound is present
         rhs.m_tau.Set(i,State.m_bndu[i]-v0.m_x[i]-v0.m_t[i]+reg*v0.m_s[i]);
        }
      else
        {
         //--- Upper bound is absent, t.Set(i, 0
         if(!CAp::Assert(v0.m_t[i]==0.0,__FUNCTION__+": T[i]<>0 for absent constraint"))
            return;
         rhs.m_tau.Set(i,0);
        }
     }
   for(i=0; i<m; i++)
     {
      if(State.m_haspq[i])
         rhs.m_alpha.Set(i,State.m_r[i]-v0.m_w[i]-v0.m_p[i]+reg*v0.m_q[i]);
     }
   for(i=0; i<n; i++)
     {
      if(!State.m_isfrozen[i])
        {
         rhs.m_sigma.Set(i,State.m_c[i]-State.m_tmpaty[i]+State.m_tmphx[i]+reg*v0.m_x[i]);
         if(State.m_hasgz[i])
            rhs.m_sigma.Add(i,-v0.m_z[i]);
         if(State.m_hasts[i])
            rhs.m_sigma.Add(i,v0.m_s[i]);
        }
      else
         rhs.m_sigma.Set(i,0);
     }
   for(i=0; i<m; i++)
     {
      if(State.m_haswv[i])
         rhs.m_beta.Add(i,v0.m_y[i]-v0.m_v[i]+reg*v0.m_w[i]);
      if(State.m_haspq[i])
         rhs.m_beta.Add(i,v0.m_q[i]);
     }
//--- Compute right-hand side:
//--- GammaZ   = mu*inv(G)*e - z - inv(G)*DELTAG*deltaZ
//--- GammaW   = mu*inv(V)*e - w - inv(V)*DELTAV*deltaW
//--- GammaS   = mu*inv(T)*e - s - inv(T)*DELTAT*deltaS
//--- GammaQ   = mu*inv(P)*e - q - inv(P)*DELTAP*deltaQ
   for(i=0; i<n; i++)
     {
      if(State.m_hasgz[i])
        {
         if(!CAp::Assert(v0.m_g[i]>0.0,__FUNCTION__+": G[i]<=0"))
            return;
         rhs.m_gammaz.Set(i,muestimate/v0.m_g[i]-v0.m_z[i]-direstimate.m_g[i]*direstimate.m_z[i]/v0.m_g[i]);
        }
      else
        {
         if(!CAp::Assert(v0.m_g[i]==0.0,__FUNCTION__+": G[i]<>0 for absent constraint"))
            return;
         if(!CAp::Assert(v0.m_z[i]==0.0,__FUNCTION__+": Z[i]<>0 for absent constraint"))
            return;
         rhs.m_gammaz.Set(i,0);
        }
     }
   for(i=0; i<m; i++)
     {
      if(State.m_haswv[i])
        {
         //--- Inequality/range constraint
         if(!CAp::Assert(v0.m_v[i]>0.0,__FUNCTION__+": V[i]<=0"))
            return;
         rhs.m_gammaw.Set(i,muestimate/v0.m_v[i]-v0.m_w[i]-direstimate.m_v[i]*direstimate.m_w[i]/v0.m_v[i]);
        }
      else
        {
         //--- Equality constraint
         if(!CAp::Assert(v0.m_v[i]==0.0,__FUNCTION__+": V[i]<>0 for equality constraint"))
            return;
         if(!CAp::Assert(v0.m_w[i]==0.0,__FUNCTION__+": W[i]<>0 for equality constraint"))
            return;
         rhs.m_gammaw.Set(i,0);
        }
     }
   for(i=0; i<n; i++)
     {
      if(State.m_hasts[i])
        {
         //--- Upper bound is present
         if(!CAp::Assert(v0.m_t[i]>0.0,__FUNCTION__+": T[i]<=0"))
            return;
         rhs.m_gammas.Set(i,muestimate/v0.m_t[i]-v0.m_s[i]-direstimate.m_t[i]*direstimate.m_s[i]/v0.m_t[i]);
        }
      else
        {
         //--- Upper bound is absent
         if(!CAp::Assert(v0.m_t[i]==0.0,__FUNCTION__+": T[i]<>0 for absent constraint"))
            return;
         if(!CAp::Assert(v0.m_s[i]==0.0,__FUNCTION__+": S[i]<>0 for absent constraint"))
            return;
         rhs.m_gammas.Set(i,0);
        }
     }
   for(i=0; i<m; i++)
     {
      if(State.m_haspq[i])
        {
         if(!CAp::Assert(v0.m_p[i]>0.0,__FUNCTION__+": P[i]<=0"))
            return;
         rhs.m_gammaq.Set(i,muestimate/v0.m_p[i]-v0.m_q[i]-direstimate.m_p[i]*direstimate.m_q[i]/v0.m_p[i]);
        }
      else
        {
         if(!CAp::Assert(v0.m_p[i]==0.0,__FUNCTION__+": P[i]<>0 for absent range"))
            return;
         if(!CAp::Assert(v0.m_q[i]==0.0,__FUNCTION__+": Q[i]<>0 for absent range"))
            return;
         rhs.m_gammaq.Set(i,0);
        }
     }
  }
//+------------------------------------------------------------------+
//| Subtracts KKT*cand from already computed RHS.                    |
//| A pair of RHSCompute/RHSSubtract calls results in residual being |
//| loaded into the RHS structure.                                   |
//| INPUT PARAMETERS:                                                |
//|   State       -  IPM State                                       |
//|   V0          -  current point(used to compute RHS)              |
//|   MuEstimate  -  estimate of Mu(can be zero)                     |
//|   DirEstimate -  estimate of delta's (can be zero)               |
//|   ResidualFrom-  whether we want to compute RHS or residual      |
//|                  computed using VDCandidate                      |
//|   VDCandidate -  solution candidate                              |
//| OUTPUT PARAMETERS:                                               |
//|   Rhs         -  either RHS or residual RHS - KKT*Cand           |
//+------------------------------------------------------------------+
void CVIPMSolver::RHSSubtract(CVIPMState &State,
                              CVIPMRightHandSide &rhs,
                              CVIPMVars &v0,
                              CVIPMVars &vdcandidate,
                              double reg)
  {
//--- create variables
   int n=State.m_n;
   int m=State.m_mdense+State.m_msparse;
   int i=0;

   VIPMMultiply(State,vdcandidate.m_x,vdcandidate.m_y,State.m_tmphx,State.m_tmpax,State.m_tmpaty);
//--- Residual for Rho, Nu, Tau, Alpha, Sigma, Beta
   for(i=0; i<m; i++)
     {
      if(State.m_haswv[i])
        {
         rhs.m_rho.Add(i,-(State.m_tmpax[i]-vdcandidate.m_w[i]+reg*vdcandidate.m_y[i]));
         rhs.m_beta.Add(i,-(-vdcandidate.m_y[i]+vdcandidate.m_v[i]-reg*vdcandidate.m_w[i]));
         rhs.m_gammaw.Add(i,-(v0.m_w[i]/v0.m_v[i]*vdcandidate.m_v[i]+vdcandidate.m_w[i]));
        }
      else
         rhs.m_rho.Add(i,-(State.m_tmpax[i]+reg*vdcandidate.m_y[i]));
      //---
      if(State.m_haspq[i])
        {
         rhs.m_alpha.Add(i,-(vdcandidate.m_w[i]+vdcandidate.m_p[i]-reg*vdcandidate.m_q[i]));
         rhs.m_beta.Add(i,vdcandidate.m_q[i]);
         rhs.m_gammaq.Add(i,-(v0.m_q[i]/v0.m_p[i]*vdcandidate.m_p[i]+vdcandidate.m_q[i]));
        }
     }
   for(i=0; i<n; i++)
     {
      //--- Residual for GammaZ, GammaW, GammaS, GammaQ
      if(State.m_hasgz[i])
        {
         rhs.m_nu.Add(i,-(vdcandidate.m_x[i]-vdcandidate.m_g[i]+reg*vdcandidate.m_z[i]));
         rhs.m_gammaz.Add(i,-(v0.m_z[i]/v0.m_g[i]*vdcandidate.m_g[i]+vdcandidate.m_z[i]));
        }
      if(State.m_hasts[i])
        {
         rhs.m_tau.Add(i,-(vdcandidate.m_x[i]+vdcandidate.m_t[i]-reg*vdcandidate.m_s[i]));
         rhs.m_gammas.Add(i,-(v0.m_s[i]/v0.m_t[i]*vdcandidate.m_t[i]+vdcandidate.m_s[i]));
        }
      //---
      if(!State.m_isfrozen[i])
        {
         rhs.m_sigma.Add(i,-(State.m_tmpaty[i]-State.m_tmphx[i]-reg*vdcandidate.m_x[i]));
         if(State.m_hasgz[i])
            rhs.m_sigma.Add(i,-vdcandidate.m_z[i]);
         if(State.m_hasts[i])
            rhs.m_sigma.Add(i,+ vdcandidate.m_s[i]);
        }
     }
  }
//+------------------------------------------------------------------+
//| Computes sum of squared primal terms of RHS                      |
//| INPUT PARAMETERS:                                                |
//|   Rhs      -  RHS structure                                      |
//|   N, M     -  problem metrics                                    |
//| RESULT:                                                          |
//|   sum(sqr()) computed over primal terms(Rho, Nu, Tau, Alpha)     |
//+------------------------------------------------------------------+
double CVIPMSolver::RHSPrimal2(CVIPMRightHandSide &rhs)
  {
   double result=rhs.m_rho.Dot(rhs.m_rho);
   result+=rhs.m_nu.Dot(rhs.m_nu);
   result+=rhs.m_tau.Dot(rhs.m_tau);
   result+=rhs.m_alpha.Dot(rhs.m_alpha);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes sum of squared dual terms of RHS                        |
//| INPUT PARAMETERS:                                                |
//|   Rhs      -  RHS structure                                      |
//|   N, M     -  problem metrics                                    |
//| RESULT:                                                          |
//|   sum(sqr()) computed over dual terms(Sigma, Beta)               |
//+------------------------------------------------------------------+
double CVIPMSolver::RHSDual2(CVIPMRightHandSide &rhs)
  {
   double result=rhs.m_sigma.Dot(rhs.m_sigma);
   result+=rhs.m_beta.Dot(rhs.m_beta);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes inf - norm of primal terms of RHS                       |
//| INPUT PARAMETERS:                                                |
//|   Rhs      -  RHS structure                                      |
//|   N, M     -  problem metrics                                    |
//| RESULT:                                                          |
//|   max(abs()) computed over primal terms(Rho, Nu, Tau, Alpha)     |
//+------------------------------------------------------------------+
double CVIPMSolver::RHSPrimalInf(CVIPMRightHandSide &rhs)
  {
   double result=MathMax((rhs.m_rho.Abs()+0).Max(),(rhs.m_nu.Abs()+0).Max());
   result=MathMax(result,(rhs.m_tau.Abs()+0).Max());
   result=MathMax(result,(rhs.m_alpha.Abs()+0).Max());
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes inf - norm of dual terms of RHS                         |
//| INPUT PARAMETERS:                                                |
//|   Rhs      -  RHS structure                                      |
//|   N, M     -  problem metrics                                    |
//| RESULT:                                                          |
//|   max(abs()) computed over dual terms(Sigma, Beta)               |
//+------------------------------------------------------------------+
double CVIPMSolver::RHSDualInf(CVIPMRightHandSide &rhs)
  {
   double result=MathMax((rhs.m_sigma.Abs()+0).Max(),(rhs.m_beta.Abs()+0).Max());
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes maximum over complementarity slackness terms of RHS     |
//| INPUT PARAMETERS:                                                |
//|   Rhs      -  RHS structure                                      |
//|   N, M     -  problem metrics                                    |
//| RESULT:                                                          |
//|   max(abs()) computed over complementarity terms                 |
//|              (GammaZ, GammaS, GammaW, GammaQ)                    |
//+------------------------------------------------------------------+
double CVIPMSolver::RHSCompl2(CVIPMRightHandSide &rhs)
  {
   double result=rhs.m_gammaz.Dot(rhs.m_gammaz);
   result+=rhs.m_gammas.Dot(rhs.m_gammas);
   result+=rhs.m_gammaw.Dot(rhs.m_gammaw);
   result+=rhs.m_gammaq.Dot(rhs.m_gammaq);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes minimum nonzero value of the vector. Returns 0 if all   |
//| components are nonpositive.                                      |
//| INPUT PARAMETERS:                                                |
//|   X        -  vector                                             |
//|   N        -  length                                             |
//+------------------------------------------------------------------+
double CVIPMSolver::MinNZ(CRowDouble &x,
                          int n)
  {
//--- create variables
   double result=0;
   bool   nz=false;

   for(int i=0; i<n; i++)
     {
      if(x[i]>0.0)
        {
         if(!nz)
           {
            result=x[i];
            nz=true;
           }
         else
            result=MathMin(result,x[i]);
        }
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes minimum product of nonzero components.                  |
//| Returns 0 if all components are nonpositive.                     |
//| INPUT PARAMETERS:                                                |
//|   X        -  vector                                             |
//|   Y        -  vector                                             |
//|   N        -  length                                             |
//+------------------------------------------------------------------+
double CVIPMSolver::MinProdNZ(CRowDouble &x,
                              CRowDouble &y,
                              int n)
  {
//--- create variables
   double result=0;
   bool   nz=false;

   for(int i=0; i<n; i++)
     {
      if(x[i]>0.0 && y[i]>0.0)
        {
         if(!nz)
           {
            result=x[i]*y[i];
            nz=true;
           }
         else
            result=MathMin(result,x[i]*y[i]);
        }
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Computes maximum product of nonzero components.                  |
//| Returns 0 if all components are nonpositive.                     |
//| INPUT PARAMETERS:                                                |
//|   X        -  vector                                             |
//|   Y        -  vector                                             |
//|   N        -  length                                             |
//+------------------------------------------------------------------+
double CVIPMSolver::MaxProdNZ(CRowDouble &x,
                              CRowDouble &y,
                              int n)
  {
//--- create variables
   double result=0;
   bool   nz=false;

   for(int i=0; i<n; i++)
     {
      if(x[i]>0.0 && y[i]>0.0)
        {
         if(!nz)
           {
            result=x[i]*y[i];
            nz=true;
           }
         else
            result=MathMax(result,x[i]*y[i]);
        }
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This structure describes convex quadratic model of the form:     |
//|f(x)=0.5*(Alpha*x'*A*x+Tau*x'*D*x)+0.5*Theta*(Q*x-r)'*(Q*x-r)+b'*x|
//| where:                                                           |
//|   * Alpha>=0, Tau>=0, Theta>=0, Alpha+Tau>0.                     |
//|   * A is NxN matrix, Q is NxK matrix (N>=1, K>=0), b is Nx1      |
//|     vector, D is NxN diagonal matrix.                            |
//|  *"main" quadratic term Alpha*A+Lambda*D is symmetric positive |
//|     definite                                                     |
//| Structure may contain optional equality constraints of the form  |
//| x[i]=x0[i], in this case functions provided by this unit         |
//| calculate Newton step subject to these equality constraints.     |
//+------------------------------------------------------------------+
struct CConvexQuadraticModel
  {
   int               m_ecakind;
   int               m_k;
   int               m_n;
   int               m_nfree;
   double            m_alpha;
   double            m_ec;
   double            m_tau;
   double            m_theta;
   double            m_tk0;
   double            m_tq0;
   bool              m_activeset[];
   bool              m_isactivesetchanged;
   bool              m_islineartermchanged;
   bool              m_ismaintermchanged;
   bool              m_issecondarytermchanged;
   CRowDouble        m_b;
   CRowDouble        m_d;
   CRowDouble        m_eb;
   CRowDouble        m_ecadiag;
   CRowDouble        m_r;
   CRowDouble        m_tb;
   CRowDouble        m_tk1;
   CRowDouble        m_tmp0;
   CRowDouble        m_tmp1;
   CRowDouble        m_tmpg;
   CRowDouble        m_tq1;
   CRowDouble        m_tq2diag;
   CRowDouble        m_txc;
   CRowDouble        m_xc;
   CMatrixDouble     m_a;
   CMatrixDouble     m_ecadense;
   CMatrixDouble     m_eccm;
   CMatrixDouble     m_eq;
   CMatrixDouble     m_q;
   CMatrixDouble     m_tk2;
   CMatrixDouble     m_tmp2;
   CMatrixDouble     m_tq2dense;
   //--- constructor / destructor
                     CConvexQuadraticModel(void);
                    ~CConvexQuadraticModel(void) {}
   //---
   void              Copy(const CConvexQuadraticModel &obj);
   //--- overloading
   void              operator=(const CConvexQuadraticModel &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CConvexQuadraticModel::CConvexQuadraticModel(void)
  {
   m_ecakind=0;
   m_k=0;
   m_n=0;
   m_nfree=0;
   m_alpha=0;
   m_ec=0;
   m_tau=0;
   m_theta=0;
   m_tk0=0;
   m_tq0=0;
   m_isactivesetchanged=false;
   m_islineartermchanged=false;
   m_ismaintermchanged=false;
   m_issecondarytermchanged=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CConvexQuadraticModel::Copy(const CConvexQuadraticModel &obj)
  {
   m_ecakind=obj.m_ecakind;
   m_k=obj.m_k;
   m_n=obj.m_n;
   m_nfree=obj.m_nfree;
   m_alpha=obj.m_alpha;
   m_ec=obj.m_ec;
   m_tau=obj.m_tau;
   m_theta=obj.m_theta;
   m_tk0=obj.m_tk0;
   m_tq0=obj.m_tq0;
   ArrayCopy(m_activeset,obj.m_activeset);
   m_isactivesetchanged=obj.m_isactivesetchanged;
   m_islineartermchanged=obj.m_islineartermchanged;
   m_ismaintermchanged=obj.m_ismaintermchanged;
   m_issecondarytermchanged=obj.m_issecondarytermchanged;
   m_b=obj.m_b;
   m_d=obj.m_d;
   m_eb=obj.m_eb;
   m_ecadiag=obj.m_ecadiag;
   m_r=obj.m_r;
   m_tb=obj.m_tb;
   m_tk1=obj.m_tk1;
   m_tmp0=obj.m_tmp0;
   m_tmp1=obj.m_tmp1;
   m_tmpg=obj.m_tmpg;
   m_tq1=obj.m_tq1;
   m_tq2diag=obj.m_tq2diag;
   m_txc=obj.m_txc;
   m_xc=obj.m_xc;
   m_a=obj.m_a;
   m_ecadense=obj.m_ecadense;
   m_eccm=obj.m_eccm;
   m_eq=obj.m_eq;
   m_q=obj.m_q;
   m_tk2=obj.m_tk2;
   m_tmp2=obj.m_tmp2;
   m_tq2dense=obj.m_tq2dense;
  }
//+------------------------------------------------------------------+
//| Convex Quadratic Models                                          |
//+------------------------------------------------------------------+
class CCQModels
  {
public:
   static const int  m_newtonrefinementits;

   static void       CQMInit(int n,CConvexQuadraticModel &s);
   static void       CQMSetA(CConvexQuadraticModel &s,CMatrixDouble &a,bool IsUpper,double alpha);
   static void       CQMGetA(CConvexQuadraticModel &s,CMatrixDouble &a);
   static void       CQMRewriteDenseDiagonal(CConvexQuadraticModel &s,CRowDouble &z);
   static void       CQMSetD(CConvexQuadraticModel &s,CRowDouble &d,double tau);
   static void       CQMDropA(CConvexQuadraticModel &s);
   static void       CQMSetB(CConvexQuadraticModel &s,CRowDouble &b);
   static void       CQMSetQ(CConvexQuadraticModel &s,CMatrixDouble &q,CRowDouble &r,int k,double theta);
   static void       CQMSetActiveSet(CConvexQuadraticModel &s,CRowDouble &x,bool &activeset[]);
   static double     CQMEval(CConvexQuadraticModel &s,CRowDouble &x);
   static void       CQMEvalX(CConvexQuadraticModel &s,CRowDouble &x,double &r,double &noise);
   static void       CQMGradUnconstrained(CConvexQuadraticModel &s,CRowDouble &x,CRowDouble &g);
   static double     CQMXTADX2(CConvexQuadraticModel &s,CRowDouble &x,CRowDouble &tmp);
   static void       CQMADX(CConvexQuadraticModel &s,CRowDouble &x,CRowDouble &y);
   static bool       CQMConstrainedOptimum(CConvexQuadraticModel &s,CRowDouble &x);
   static void       CQMScaleVector(CConvexQuadraticModel &s,CRowDouble &x);
   static void       CQMGetDiagA(CConvexQuadraticModel &s,CRowDouble &x);
   static double     CQMDebugConstrainedEvalT(CConvexQuadraticModel &s,CRowDouble &x);
   static double     CQMDebugConstrainedEvalE(CConvexQuadraticModel &s,CRowDouble &x);

private:
   static bool       CQMRebuild(CConvexQuadraticModel &s);
   static void       CQMSolveEA(CConvexQuadraticModel &s,CRowDouble &x,CRowDouble &tmp);
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
const int CCQModels::m_newtonrefinementits=3;
//+------------------------------------------------------------------+
//| This subroutine is used to initialize CQM. By default, empty NxN |
//| model is generated, with Alpha=Lambda=Theta=0.0 and zero b.      |
//| Previously allocated buffer variables are reused as much as      |
//| possible.                                                        |
//+------------------------------------------------------------------+
void CCQModels::CQMInit(int n,CConvexQuadraticModel &s)
  {
//--- initialization
   s.m_n=n;
   s.m_k=0;
   s.m_nfree=n;
   s.m_ecakind=-1;
   s.m_alpha=0.0;
   s.m_tau=0.0;
   s.m_theta=0.0;
   s.m_ismaintermchanged=true;
   s.m_issecondarytermchanged=true;
   s.m_islineartermchanged=true;
   s.m_isactivesetchanged=true;
   ArrayResize(s.m_activeset,n);
   s.m_xc=vector<double>::Zeros(n);
   s.m_eb=vector<double>::Zeros(n);
   s.m_tq1=vector<double>::Zeros(n);
   s.m_txc=vector<double>::Zeros(n);
   s.m_tb=vector<double>::Zeros(n);
   s.m_b=vector<double>::Zeros(n);
   s.m_tk1=vector<double>::Zeros(n);
   ArrayInitialize(s.m_activeset,false);
  }
//+------------------------------------------------------------------+
//| This subroutine changes main quadratic term of the model.        |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//|   A        -  NxN matrix, only upper or lower triangle is        |
//|               referenced                                         |
//|   IsUpper  -  True, when matrix is stored in upper triangle      |
//|   Alpha    -  multiplier; when Alpha=0, A is not referenced at   |
//|               all                                                |
//+------------------------------------------------------------------+
void CCQModels::CQMSetA(CConvexQuadraticModel &s,
                        CMatrixDouble &a,
                        bool IsUpper,
                        double alpha)
  {
//--- create variables
   int    i=0;
   int    j=0;
   double v=0;
//--- check
   if(!CAp::Assert(MathIsValidNumber(alpha) && alpha>=0.0,__FUNCTION__+": Alpha<0 or is not finite number"))
      return;
   if(!CAp::Assert(alpha==0.0 || CApServ::IsFiniteRTrMatrix(a,s.m_n,IsUpper),__FUNCTION__+": A is not finite NxN matrix"))
      return;
   s.m_alpha=alpha;
   if(alpha>0.0)
     {
      s.m_a.Resize(s.m_n,s.m_n);
      s.m_ecadense.Resize(s.m_n,s.m_n);
      s.m_tq2dense.Resize(s.m_n,s.m_n);
      for(i=0; i<s.m_n; i++)
        {
         for(j=i; j<s.m_n; j++)
           {
            if(IsUpper)
               v=a.Get(i,j);
            else
               v=a.Get(j,i);
            s.m_a.Set(i,j,v);
            s.m_a.Set(j,i,v);
           }
        }
     }
   s.m_ismaintermchanged=true;
  }
//+------------------------------------------------------------------+
//| This subroutine changes main quadratic term of the model.        |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//|   A        -  possibly preallocated buffer                       |
//| OUTPUT PARAMETERS:                                               |
//|   A        -  NxN matrix, full matrix is returned. Zero matrix is|
//|               returned if model is empty.                        |
//+------------------------------------------------------------------+
void CCQModels::CQMGetA(CConvexQuadraticModel &s,
                        CMatrixDouble &a)
  {
//--- create variables
   double v=0;
//--- copy
   if(s.m_alpha>0.0)
     {
      v=s.m_alpha;
      a=s.m_a*v+0;
     }
   else
     {
      a=matrix<double>::Zeros(s.m_n,s.m_n);
     }
  }
//+------------------------------------------------------------------+
//| This subroutine rewrites diagonal of the main quadratic term of  |
//| the model (dense A) by vector Z/Alpha (current value of the Alpha|
//| coefficient is used).                                            |
//| IMPORTANT: in case model has no dense quadratic term, this       |
//|            function allocates N*N dense matrix of zeros, and     |
//|            fills its diagonal by non-zero values.                |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//|   Z        -  new diagonal, array[N]                             |
//+------------------------------------------------------------------+
void CCQModels::CQMRewriteDenseDiagonal(CConvexQuadraticModel &s,
                                        CRowDouble &z)
  {
   if(s.m_alpha==0.0)
     {
      s.m_a=matrix<double>::Zeros(s.m_n,s.m_n);
      s.m_ecadense.Resize(s.m_n,s.m_n);
      s.m_tq2dense.Resize(s.m_n,s.m_n);
      s.m_alpha=1.0;
     }
   vector<double> diag=z/s.m_alpha;
   diag.Resize(s.m_n);
   s.m_a.Diag(diag);
   s.m_ismaintermchanged=true;
  }
//+------------------------------------------------------------------+
//| This subroutine changes diagonal quadratic term of the model.    |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//|   D        -  array[N], semidefinite diagonal matrix             |
//|   Tau      -  multiplier; when Tau=0, D is not referenced at all |
//+------------------------------------------------------------------+
void CCQModels::CQMSetD(CConvexQuadraticModel &s,CRowDouble &d,double tau)
  {
//--- check
   if(!CAp::Assert(MathIsValidNumber(tau) && tau>=0.0,__FUNCTION__+": Tau<0 or is not finite number"))
      return;
   if(!CAp::Assert(tau==0.0 || CApServ::IsFiniteVector(d,s.m_n),__FUNCTION__+": D is not finite Nx1 vector"))
      return;
   s.m_tau=tau;
   if(tau>0.0)
     {
      s.m_d=d;
      s.m_d.Resize(s.m_n);
      //--- check
      if(!CAp::Assert(s.m_d.Min()>=0.0,__FUNCTION__+": D[i]<0"))
         return;
      s.m_ecadiag=vector<double>::Zeros(s.m_n);
      s.m_tq2diag=vector<double>::Zeros(s.m_n);
     }
   s.m_ismaintermchanged=true;
  }
//+------------------------------------------------------------------+
//| This subroutine drops main quadratic term A from the model. It is|
//| same as call to CQMSetA() with zero A, but gives better          |
//| performance because algorithm knows that matrix is zero and can  |
//| optimize subsequent calculations.                                |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//+------------------------------------------------------------------+
void CCQModels::CQMDropA(CConvexQuadraticModel &s)
  {
   s.m_alpha=0.0;
   s.m_ismaintermchanged=true;
  }
//+------------------------------------------------------------------+
//| This subroutine changes linear term of the model                 |
//+------------------------------------------------------------------+
void CCQModels::CQMSetB(CConvexQuadraticModel &s,CRowDouble &b)
  {
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(b,s.m_n),__FUNCTION__+": B is not finite vector"))
      return;

   s.m_b=b;
   s.m_b.Resize(s.m_n);
   s.m_islineartermchanged=true;
  }
//+------------------------------------------------------------------+
//| This subroutine changes linear term of the model                 |
//+------------------------------------------------------------------+
void CCQModels::CQMSetQ(CConvexQuadraticModel &s,
                        CMatrixDouble &q,
                        CRowDouble &r,
                        int k,
                        double theta)
  {
//--- check
   if(!CAp::Assert(k>=0,__FUNCTION__+": K<0"))
      return;
   if(!CAp::Assert(k==0 || theta==0.0 || CApServ::IsFiniteMatrix(q,k,s.m_n),__FUNCTION__+": Q is not finite matrix"))
      return;
   if(!CAp::Assert(k==0 || theta==0.0 || CApServ::IsFiniteVector(r,k),__FUNCTION__+": R is not finite vector"))
      return;
   if(!CAp::Assert(MathIsValidNumber(theta) && theta>=0.0,__FUNCTION__+": Theta<0 or is not finite number"))
      return;
//--- degenerate case: K=0 or Theta=0
   if(k==0 || theta==0.0)
     {
      s.m_k=0;
      s.m_theta=0;
      s.m_issecondarytermchanged=true;
      return;
     }
//--- General case: both Theta>0 and K>0
   s.m_k=k;
   s.m_theta=theta;
   s.m_q=q;
   s.m_r=r;
   s.m_q.Resize(k,s.m_n);
   s.m_r.Resize(k);
   CApServ::RMatrixSetLengthAtLeast(s.m_eq,k,s.m_n);
   CApServ::RMatrixSetLengthAtLeast(s.m_eccm,k,k);
   CApServ::RMatrixSetLengthAtLeast(s.m_tk2,k,s.m_n);
   s.m_issecondarytermchanged=true;
  }
//+------------------------------------------------------------------+
//| This subroutine changes active set                               |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//|   X        -  array[N], constraint values                        |
//|   ActiveSet-  array[N], active set. If ActiveSet[I] = True,      |
//|               then I-th variables is constrained to X[I].        |
//+------------------------------------------------------------------+
void CCQModels::CQMSetActiveSet(CConvexQuadraticModel &s,CRowDouble &x,
                                bool &activeset[])
  {
//--- check
   if(!CAp::Assert(x.Size()>=s.m_n,__FUNCTION__+": Length(X)<N"))
      return;
   if(!CAp::Assert(ArraySize(activeset)>=s.m_n,__FUNCTION__+": Length(ActiveSet)<N"))
      return;

   for(int i=0; i<s.m_n; i++)
     {
      s.m_isactivesetchanged=s.m_isactivesetchanged || (s.m_activeset[i] && !activeset[i]);
      s.m_isactivesetchanged=s.m_isactivesetchanged || (activeset[i] && !s.m_activeset[i]);
      s.m_activeset[i]=activeset[i];
      if(activeset[i])
        {
         //--- check
         if(!CAp::Assert(MathIsValidNumber(x[i]),__FUNCTION__+": X[] contains infinite constraints"))
            return;
         s.m_isactivesetchanged=s.m_isactivesetchanged || (s.m_xc[i]!=x[i]);
         s.m_xc.Set(i,x[i]);
        }
     }
  }
//+------------------------------------------------------------------+
//| This subroutine evaluates model at X. Active constraints are     |
//| ignored.                                                         |
//+------------------------------------------------------------------+
double CCQModels::CQMEval(CConvexQuadraticModel &s,CRowDouble &X)
  {
//--- create variables
   double result=0;
   int    n=s.m_n;
   double v=0;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(X,n),__FUNCTION__+": X is not finite vector"))
      return(result);
//--- main quadratic term
   CRowDouble x=X;
   x.Resize(n);
   if(s.m_alpha>0.0)
      result=x.Dot(x.MatMul(s.m_a)*(s.m_alpha*0.5));
   if(s.m_tau>0.0)
      result+=s.m_tau*s.m_d.Dot(x*x/2.0);
//--- secondary quadratic term
   if(s.m_theta>0.0)
     {
      for(int i=0; i<s.m_k; i++)
        {
         v=x.Dot(s.m_q[i]+0);
         result+=0.5*s.m_theta*CMath::Sqr(v-s.m_r[i]);
        }
     }
//--- linear term
   result+=x.Dot(s.m_b);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This subroutine evaluates model at X. Active constraints are     |
//| ignored.                                                         |
//| It returns:                                                      |
//|   R        -  model value                                        |
//|   Noise    -  estimate of the numerical noise in data            |
//+------------------------------------------------------------------+
void CCQModels::CQMEvalX(CConvexQuadraticModel &s,
                         CRowDouble &x,
                         double &r,
                         double &noise)
  {
//--- create variables
   int    n=s.m_n;
   int    i=0;
   int    j=0;
   double v=0;
   double v2=0;
   double mxq=0;
   double eps=2*CMath::m_machineepsilon;
//--- initialization
   r=0;
   noise=0;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X is not finite vector"))
      return;
//--- Main quadratic term.
//--- Noise from the main quadratic term is equal to the
//--- maximum summand in the term.
   if(s.m_alpha>0.0)
     {
      for(i=0; i<n; i++)
         for(j=0; j<n; j++)
           {
            v=s.m_alpha*0.5*x[i]*s.m_a.Get(i,j)*x[j];
            r+=v;
            noise=MathMax(noise,eps*MathAbs(v));
           }
     }
   if(s.m_tau>0.0)
     {
      for(i=0; i<n; i++)
        {
         v=0.5*CMath::Sqr(x[i])*s.m_tau*s.m_d[i];
         r+=v;
         noise=MathMax(noise,eps*MathAbs(v));
        }
     }
//--- secondary quadratic term
//--- Noise from the secondary quadratic term is estimated as follows:
//--- * noise in qi*x-r[i] is estimated as
//---   Eps*MXQ = Eps*max(|r[i]|, |q[i,j]*x[j]|)
//--- * noise in (qi*x-r[i])^2 is estimated as
//---   NOISE = (|qi*x-r[i]|+Eps*MXQ)^2-(|qi*x-r[i]|)^2
//---         = Eps*MXQ*(2*|qi*x-r[i]|+Eps*MXQ)
   if(s.m_theta>0.0)
     {
      for(i=0; i<s.m_k; i++)
        {
         v=0.0;
         mxq=MathAbs(s.m_r[i]);
         for(j=0; j<n; j++)
           {
            v2=s.m_q.Get(i,j)*x[j];
            v+=v2;
            mxq=MathMax(mxq,MathAbs(v2));
           }
         r+=0.5*s.m_theta*CMath::Sqr(v-s.m_r[i]);
         noise=MathMax(noise,eps*mxq*(2*MathAbs(v-s.m_r[i])+eps*mxq));
        }
     }
//--- linear term
   for(i=0; i<s.m_n; i++)
     {
      r+=x[i]
          *s.m_b[i];
      noise=MathMax(noise,eps*MathAbs(x[i]*s.m_b[i]));
     }
//--- Final update of the noise
   noise=n*noise;
  }
//+------------------------------------------------------------------+
//| This  subroutine  evaluates  gradient of the model; active       |
//| constraints are ignored.                                         |
//| INPUT PARAMETERS:                                                |
//|   S        -  convex model                                       |
//|   X        -  point, array[N]                                    |
//|   G        -  possibly preallocated buffer; resized, if too small|
//+------------------------------------------------------------------+
void CCQModels::CQMGradUnconstrained(CConvexQuadraticModel &s,
                                     CRowDouble &x,
                                     CRowDouble &g)
  {
//--- create variables
   int    n=s.m_n;
   int    i=0;
   double v=0;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X is not finite vector"))
      return;
   g=vector<double>::Zeros(n);
//--- main quadratic term
   if(s.m_alpha>0.0)
     {
      for(i=0; i<n; i++)
        {
         v=x.Dot(s.m_a[i]*s.m_alpha);
         g.Add(i,v);
        }
     }
   if(s.m_tau>0.0)
      g+=x*s.m_d*s.m_tau;
//--- secondary quadratic term
   if(s.m_theta>0.0)
     {
      for(i=0; i<s.m_k; i++)
        {
         v=x.Dot(s.m_q[i]+0);
         v=s.m_theta*(v-s.m_r[i]);
         g+=s.m_q[i]*v;
        }
     }
//--- linear term
   g+=s.m_b;
  }
//+------------------------------------------------------------------+
//| This subroutine evaluates x'*(0.5*alpha*A+tau*D)*x               |
//| NOTE: Tmp[] must be preallocated array whose length is at least N|
//+------------------------------------------------------------------+
double CCQModels::CQMXTADX2(CConvexQuadraticModel &s,
                            CRowDouble &x,
                            CRowDouble &tmp)
  {
//--- create variables
   double result=0;
   int    n=s.m_n;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X is not finite vector"))
      return(0.0);
   if(!CAp::Assert(tmp.Size()>=n,__FUNCTION__+": Length(Tmp)<N"))
      return(0.0);
//--- main quadratic term
   if(s.m_alpha>0.0)
      result+=s.m_alpha*0.5*CAblas::RMatrixSyvMVect(n,s.m_a,0,0,true,x,0,tmp);
   if(s.m_tau>0.0)
     {
      result+=s.m_d.Dot(x*x/2.0)*s.m_tau;
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This subroutine evaluates(0.5 * alpha * A + tau*D)*x             |
//| Y is automatically resized if needed                             |
//+------------------------------------------------------------------+
void CCQModels::CQMADX(CConvexQuadraticModel &s,
                       CRowDouble &x,
                       CRowDouble &y)
  {
   int n=s.m_n;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X is not finite vector"))
      return;
   y=vector<double>::Zeros(n);
//--- main quadratic term
   if(s.m_alpha>0.0)
      CAblas::RMatrixSymVect(n,s.m_alpha,s.m_a,0,0,true,x,0,1.0,y,0);
   if(s.m_tau>0.0)
      y+=x*s.m_d*s.m_tau;
  }
//+------------------------------------------------------------------+
//| This subroutine finds optimum of the model. It returns False  on |
//| failure (indefinite / semidefinite matrix). Optimum  is  found   |
//| subject  to  active constraints.                                 |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//|   X        -  possibly preallocated buffer; automatically        |
//|               resized, if too small enough.                      |
//+------------------------------------------------------------------+
bool CCQModels::CQMConstrainedOptimum(CConvexQuadraticModel &s,
                                      CRowDouble &x)
  {
//--- create variables
   int    n=0;
   int    nfree=0;
   int    k=0;
   int    i=0;
   double v=0;
   int    cidx0=0;
   int    itidx=0;
   int    i_=0;
//--- Rebuild internal structures
   if(!CQMRebuild(s))
      return(false);

   n=s.m_n;
   k=s.m_k;
   nfree=s.m_nfree;
//--- Calculate initial point for the iterative refinement:
//--- * free components are set to zero
//--- * constrained components are set to their constrained values
   x=vector<double>::Zeros(n);
   for(i=0; i<n; i++)
     {
      if(s.m_activeset[i])
         x.Set(i,s.m_xc[i]);
     }
//--- Iterative refinement.
//--- In an ideal world without numerical errors it would be enough
//--- to make just one Newton step from initial point:
//---   x_new = -H^(-1)*grad(x=0)
//--- However, roundoff errors can significantly deteriorate quality
//--- of the solution. So we have to recalculate gradient and to
//--- perform Newton steps several times.
//--- Below we perform fixed number of Newton iterations.
   for(itidx=0; itidx<m_newtonrefinementits; itidx++)
     {
      //--- Calculate gradient at the current point.
      //--- Move free components of the gradient in the beginning.
      CQMGradUnconstrained(s,x,s.m_tmpg);
      cidx0=0;
      for(i=0; i<n; i++)
        {
         if(!s.m_activeset[i])
           {
            s.m_tmpg.Set(cidx0,s.m_tmpg[i]);
            cidx0++;
           }
        }
      //--- Free components of the extrema are calculated in the first NFree elements of TXC.
      //--- First, we have to calculate original Newton step, without rank-K perturbations
      for(i_=0; i_<nfree; i_++)
         s.m_txc.Set(i_,-s.m_tmpg[i_]);
      CQMSolveEA(s,s.m_txc,s.m_tmp0);
      //--- Then, we account for rank-K correction.
      //--- Woodbury matrix identity is used.
      if(s.m_k>0 && s.m_theta>0.0)
        {
         s.m_tmp0=vector<double>::Zeros(MathMax(nfree,k));
         s.m_tmp1=vector<double>::Zeros(MathMax(nfree,k));
         for(i_=0; i_<nfree; i_++)
            s.m_tmp1.Set(i_,-s.m_tmpg[i_]);
         CQMSolveEA(s,s.m_tmp1,s.m_tmp0);
         for(i=0; i<k; i++)
           {
            v=0.0;
            for(i_=0; i_<nfree; i_++)
               v+=s.m_eq.Get(i,i_)*s.m_tmp1[i_];
            s.m_tmp0.Set(i,v);
           }
         CFbls::FblsCholeskySolve(s.m_eccm,1.0,k,true,s.m_tmp0,s.m_tmp1);
         for(i=0; i<nfree; i++)
            s.m_tmp1.Set(i,0.0);
         for(i=0; i<k; i++)
           {
            v=s.m_tmp0[i];
            for(i_=0; i_<nfree; i_++)
               s.m_tmp1.Add(i_,v*s.m_eq.Get(i,i_));
           }
         CQMSolveEA(s,s.m_tmp1,s.m_tmp0);
         for(i_=0; i_<nfree; i_++)
            s.m_txc.Add(i_,-s.m_tmp1[i_]);
        }
      //--- Unpack components from TXC into X. We pass through all
      //--- free components of X and add our step.
      cidx0=0;
      for(i=0; i<n; i++)
        {
         if(!s.m_activeset[i])
           {
            x.Add(i,s.m_txc[cidx0]);
            cidx0++;
           }
        }
     }
//--- return result
   return(true);
  }
//+------------------------------------------------------------------+
//| This function scales vector  by  multiplying it by inverse of the|
//| diagonal of the Hessian matrix. It should be used to  accelerate |
//| steepest descent phase of the QP m_solver.                         |
//| Although it is called "scale-grad", it can be called for any     |
//| vector, whether it is gradient, anti-gradient, or just some      |
//| vector.                                                          |
//| This function does NOT takes into account current set of         |
//| constraints, it just performs matrix-vector multiplication       |
//| without taking into account constraints.                         |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//|   X        -  vector to scale                                    |
//| OUTPUT PARAMETERS:                                               |
//|   X        -  scaled vector                                      |
//| NOTE: when called for non-SPD matrices, it silently skips        |
//|       components of X which correspond to zero or negative       |
//|       diagonal elements.                                         |
//| NOTE: this function uses diagonals of A and D; it ignores        |
//|       Q-rank-K term of the quadratic model.                      |
//+------------------------------------------------------------------+
void CCQModels::CQMScaleVector(CConvexQuadraticModel &s,CRowDouble &x)
  {
//--- create variables
   int    n=s.m_n;
   double v=0;

   for(int i=0; i<n; i++)
     {
      v=0.0;
      if(s.m_alpha>0.0)
         v+=s.m_a.Get(i,i);
      if(s.m_tau>0.0)
         v+=s.m_d[i];
      if(v>0.0)
         x.Mul(i,1.0/v);
     }
  }
//+------------------------------------------------------------------+
//| This function returns diagonal of the A - term.                  |
//| INPUT PARAMETERS:                                                |
//|   S        -  model                                              |
//| OUTPUT PARAMETERS:                                               |
//|   D        -  diagonal of the A( or zero)                        |
//+------------------------------------------------------------------+
void CCQModels::CQMGetDiagA(CConvexQuadraticModel &s,CRowDouble &x)
  {
   int n=s.m_n;

   if(s.m_alpha>0)
      x=s.m_a.Diag(0)+0;
   else
      x=vector<double>::Zeros(n);
  }
//+------------------------------------------------------------------+
//| This subroutine calls CQMRebuild() and evaluates model at X      |
//| subject to active constraints.                                   |
//| It is intended for  debug  purposes only, because it evaluates   |
//| model by means of temporaries, which were calculated  by         |
//| CQMRebuild(). The only purpose of this function  is  to  check   |
//| correctness of CQMRebuild() by comparing results of this function|
//| with ones obtained by CQMEval(), which is used as reference point|
//| The idea is that significant deviation in results of these  two  |
//| functions  is  evidence  of  some  error  in  the CQMRebuild().  |
//| NOTE: suffix T denotes that temporaries marked by T-prefix are   |
//|       used. There is one more variant of this function, which    |
//|       uses "effective" model built by CQMRebuild().              |
//| NOTE2: in case CQMRebuild() fails(due to model non-convexity),   |
//|        this function returns NAN.                                |
//+------------------------------------------------------------------+
double CCQModels::CQMDebugConstrainedEvalT(CConvexQuadraticModel &s,
                                           CRowDouble &x)
  {
//--- create variables
   double result=0.0;
   int    n=s.m_n;
   int    nfree=0;
   int    i=0;
   int    j=0;
   double v=0;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X is not finite vector"))
      return(result);
   if(!CQMRebuild(s))
      return(AL_NaN);

   nfree=s.m_nfree;
//--- Reorder variables
   j=0;
   for(i=0; i<n; i++)
     {
      if(!s.m_activeset[i])
        {
         //--- check
         if(!CAp::Assert(j<nfree,__FUNCTION__+": internal error"))
            return(AL_NaN);
         s.m_txc.Set(j,x[i]);
         j++;
        }
     }
//--- TQ2, TQ1, TQ0
   if(s.m_alpha>0.0)
     {
      //--- Dense TQ2
      for(i=0; i<nfree; i++)
         for(j=0; j<nfree; j++)
            result+=0.5*s.m_txc[i]*s.m_tq2dense.Get(i,j)*s.m_txc[j];
     }
   else
     {
      //--- Diagonal TQ2
      for(i=0; i<nfree; i++)
         result+=0.5*s.m_tq2diag[i]*CMath::Sqr(s.m_txc[i]);
     }
   result+=s.m_tq1.Dot(s.m_txc);
   result=result+s.m_tq0;
//--- TK2, TK1, TK0
   if(s.m_k>0 && s.m_theta>0.0)
     {
      for(i=0; i<s.m_k; i++)
        {
         v=CAblasF::RDotVR(nfree,s.m_txc,s.m_tk2,i);
         result=result+0.5*CMath::Sqr(v);
        }
      result+=s.m_tk1.Dot(s.m_txc);
      result=result+s.m_tk0;
     }
//--- TB (Bf and Bc parts)
   result+=s.m_tb.Dot(s.m_txc);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This subroutine calls CQMRebuild() and evaluates model at X      |
//| subject to active constraints.                                   |
//| It is intended for debug purposes only, because it evaluates     |
//| model by means of "effective" matrices built by CQMRebuild(). The|
//| only purpose of this function is to check correctness of         |
//| CQMRebuild() by comparing results of this function with ones     |
//| obtained by CQMEval(), which is used as reference point. The idea|
//| is that significant deviation in results of these two functions  |
//| is evidence of some error in the CQMRebuild().                   |
//| NOTE: suffix E denotes that effective matrices. There is one more|
//|       variant of this function, which uses temporary matrices    |
//|       built by CQMRebuild().                                     |
//| NOTE2: in case CQMRebuild() fails(due to model non-convexity),   |
//|        this function returns NAN.                                |
//+------------------------------------------------------------------+
double CCQModels::CQMDebugConstrainedEvalE(CConvexQuadraticModel &s,
                                           CRowDouble &x)
  {
//--- create variables
   double result=0;
   int    n=s.m_n;
   int    nfree=0;
   int    i=0;
   int    j=0;
   double v=0;
//--- check
   if(!CAp::Assert(CApServ::IsFiniteVector(x,n),__FUNCTION__+": X is not finite vector"))
      return(AL_NaN);
   if(!CQMRebuild(s))
      return(AL_NaN);

   nfree=s.m_nfree;
//--- Reorder variables
   j=0;
   for(i=0; i<n; i++)
     {
      if(!s.m_activeset[i])
        {
         //--- check
         if(!CAp::Assert(j<nfree,__FUNCTION__+": internal error"))
            return(AL_NaN);
         s.m_txc.Set(j,x[i]);
         j++;
        }
     }
//--- ECA
//--- check
   if(!CAp::Assert(s.m_ecakind==0 || s.m_ecakind==1 || (s.m_ecakind==-1 && nfree==0),__FUNCTION__+": unexpected ECAKind"))
      return(AL_NaN);
   if(s.m_ecakind==0)
     {
      //--- Dense ECA
      for(i=0; i<nfree; i++)
        {
         v=0.0;
         for(j=i; j<nfree; j++)
            v+= s.m_ecadense.Get(i,j)*s.m_txc[j];
         result+=0.5*CMath::Sqr(v);
        }
     }
   if(s.m_ecakind==1)
     {
      //--- Diagonal ECA
      for(i=0; i<nfree; i++)
         result+=0.5*CMath::Sqr(s.m_ecadiag[i]*s.m_txc[i]);
     }
//--- EQ
   for(i=0; i<s.m_k; i++)
     {
      v=0.0;
      for(j=0; j<nfree; j++)
         v+=s.m_eq.Get(i,j)*s.m_txc[j];
      result+=0.5*CMath::Sqr(v);
     }
//--- EB
   for(i=0; i<nfree; i++)
      result+=s.m_eb[i]*s.m_txc[i];
//--- EC
   result+=s.m_ec;
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Internal function, rebuilds "effective" model subject to         |
//| constraints. Returns False on failure (non-SPD main quadratic    |
//| term)                                                            |
//+------------------------------------------------------------------+
bool CCQModels::CQMRebuild(CConvexQuadraticModel &s)
  {
//--- create variables
   bool   result=true;
   int    n=s.m_n;
   int    k=s.m_k;
   int    nfree=0;
   int    i=0;
   int    j=0;
   int    ridx0=0;
   int    ridx1=0;
   int    cidx0=0;
   int    cidx1=0;
   double v=0;
   int    i_=0;
//--- Non-SPD model, quick exit
   if(s.m_alpha==0.0 && s.m_tau==0.0)
      return(false);
//--- Determine number of free variables.
//--- Fill TXC - array whose last N-NFree elements store constraints.
   if(s.m_isactivesetchanged)
     {
      s.m_nfree=0;
      for(i=0; i<n; i++)
        {
         if(!s.m_activeset[i])
            s.m_nfree++;
        }
      j=s.m_nfree;
      for(i=0; i<n; i++)
         if(s.m_activeset[i])
           {
            s.m_txc.Set(j,s.m_xc[i]);
            j++;
           }
     }
   nfree=s.m_nfree;
//--- Re-evaluate TQ2/TQ1/TQ0, if needed
   if(s.m_isactivesetchanged || s.m_ismaintermchanged)
     {
      //--- Handle cases Alpha>0 and Alpha=0 separately:
      //--- * in the first case we have dense matrix
      //--- * in the second one we have diagonal matrix, which can be
      //---   handled more efficiently
      if(s.m_alpha>0.0)
        {
         //--- Alpha>0, dense QP
         //--- Split variables into two groups - free (F) and constrained (C). Reorder
         //--- variables in such way that free vars come first, constrained are last:
         //--- x = [xf, xc].
         //--- Main quadratic term x'*(alpha*A+tau*D)*x now splits into quadratic part,
         //--- linear part and constant part:
         //---                   ( alpha*Aff+tau*Df  alpha*Afc        ) ( xf )
         //---   0.5*( xf' xc' )*(                                    )*(    ) =
         //---                   ( alpha*Acf         alpha*Acc+tau*Dc ) ( xc )
         //---   = 0.5*xf'*(alpha*Aff+tau*Df)*xf + (alpha*Afc*xc)'*xf + 0.5*xc'(alpha*Acc+tau*Dc)*xc
         //--- We store these parts into temporary variables:
         //--- * alpha*Aff+tau*Df, alpha*Afc, alpha*Acc+tau*Dc are stored into upper
         //---   triangle of TQ2
         //--- * alpha*Afc*xc is stored into TQ1
         //--- * 0.5*xc'(alpha*Acc+tau*Dc)*xc is stored into TQ0
         //--- Below comes first part of the work - generation of TQ2:
         //--- * we pass through rows of A and copy I-th row into upper block (Aff/Afc) or
         //---   lower one (Acf/Acc) of TQ2, depending on presence of X[i] in the active set.
         //---   RIdx0 variable contains current position for insertion into upper block,
         //---   RIdx1 contains current position for insertion into lower one.
         //--- * within each row, we copy J-th element into left half (Aff/Acf) or right
         //---   one (Afc/Acc), depending on presence of X[j] in the active set. CIdx0
         //---   contains current position for insertion into left block, CIdx1 contains
         //---   position for insertion into right one.
         //--- * during copying, we multiply elements by alpha and add diagonal matrix D.
         ridx0=0;
         ridx1=s.m_nfree;
         for(i=0; i<n; i++)
           {
            cidx0=0;
            cidx1=s.m_nfree;
            for(j=0; j<n; j++)
              {
               if(!s.m_activeset[i] && !s.m_activeset[j])
                 {
                  //--- Element belongs to Aff
                  v=s.m_alpha*s.m_a.Get(i,j);
                  if(i==j && s.m_tau>0.0)
                     v+=s.m_tau*s.m_d[i];
                  s.m_tq2dense.Set(ridx0,cidx0,v);
                 }
               if(!s.m_activeset[i] && s.m_activeset[j])
                 {
                  //--- Element belongs to Afc
                  s.m_tq2dense.Set(ridx0,cidx1,s.m_alpha*s.m_a.Get(i,j));
                 }
               if(s.m_activeset[i] && !s.m_activeset[j])
                 {
                  //--- Element belongs to Acf
                  s.m_tq2dense.Set(ridx1,cidx0,s.m_alpha*s.m_a.Get(i,j));
                 }
               if(s.m_activeset[i] && s.m_activeset[j])
                 {
                  //--- Element belongs to Acc
                  v=s.m_alpha*s.m_a.Get(i,j);
                  if(i==j && s.m_tau>0.0)
                     v+=s.m_tau*s.m_d[i];
                  s.m_tq2dense.Set(ridx1,cidx1,v);
                 }
               if(s.m_activeset[j])
                  cidx1++;
               else
                  cidx0++;
              }
            if(s.m_activeset[i])
               ridx1++;
            else
               ridx0++;
           }
         //--- Now we have TQ2, and we can evaluate TQ1.
         //--- In the special case when we have Alpha=0, NFree=0 or NFree=N,
         //--- TQ1 is filled by zeros.
         s.m_tq1=vector<double>::Zeros(n);
         if(s.m_nfree>0 && s.m_nfree<n)
            CAblas::RMatrixMVect(s.m_nfree,n-s.m_nfree,s.m_tq2dense,0,s.m_nfree,0,s.m_txc,s.m_nfree,s.m_tq1,0);
         //--- And finally, we evaluate TQ0.
         v=0.0;
         for(i=s.m_nfree; i<n; i++)
            for(j=s.m_nfree; j<n; j++)
               v+= 0.5*s.m_txc[i]*s.m_tq2dense.Get(i,j)*s.m_txc[j];
         s.m_tq0=v;
        }
      else
        {
         //--- Alpha=0, diagonal QP
         //--- Split variables into two groups - free (F) and constrained (C). Reorder
         //--- variables in such way that free vars come first, constrained are last:
         //--- x = [xf, xc].
         //--- Main quadratic term x'*(tau*D)*x now splits into quadratic and constant
         //--- parts:
         //---                   ( tau*Df        ) ( xf )
         //---   0.5*( xf' xc' )*(               )*(    ) =
         //---                   (        tau*Dc ) ( xc )
         //---   = 0.5*xf'*(tau*Df)*xf + 0.5*xc'(tau*Dc)*xc
         //--- We store these parts into temporary variables:
         //--- * tau*Df is stored in TQ2Diag
         //--- * 0.5*xc'(tau*Dc)*xc is stored into TQ0
         s.m_tq0=0.0;
         ridx0=0;
         for(i=0; i<n; i++)
           {
            if(!s.m_activeset[i])
              {
               s.m_tq2diag.Set(ridx0,s.m_tau*s.m_d[i]);
               ridx0++;
              }
            else
               s.m_tq0+=0.5*s.m_tau*s.m_d[i]*CMath::Sqr(s.m_xc[i]);
           }
         s.m_tq1=vector<double>::Zeros(n);
        }
     }
//--- Re-evaluate TK2/TK1/TK0, if needed
   if(s.m_isactivesetchanged || s.m_issecondarytermchanged)
     {
      //
      //--- Split variables into two groups - free (F) and constrained (C). Reorder
      //--- variables in such way that free vars come first, constrained are last:
      //--- x = [xf, xc].
      //
      //--- Secondary term theta*(Q*x-r)'*(Q*x-r) now splits into quadratic part,
      //--- linear part and constant part:
      //---             (          ( xf )     )'  (          ( xf )     )
      //---   0.5*theta*( (Qf Qc)'*(    ) - r ) * ( (Qf Qc)'*(    ) - r ) =
      //---             (          ( xc )     )   (          ( xc )     )
      //
      //---   = 0.5*theta*xf'*(Qf'*Qf)*xf + theta*((Qc*xc-r)'*Qf)*xf +
      //---     + theta*(-r'*(Qc*xc-r)-0.5*r'*r+0.5*xc'*Qc'*Qc*xc)
      //
      //--- We store these parts into temporary variables:
      //--- * sqrt(theta)*Qf is stored into TK2
      //--- * theta*((Qc*xc-r)'*Qf) is stored into TK1
      //--- * theta*(-r'*(Qc*xc-r)-0.5*r'*r+0.5*xc'*Qc'*Qc*xc) is stored into TK0
      //
      //--- We use several other temporaries to store intermediate results:
      //--- * Tmp0 - to store Qc*xc-r
      //--- * Tmp1 - to store Qc*xc
      //
      //--- Generation of TK2/TK1/TK0 is performed as follows:
      //--- * we fill TK2/TK1/TK0 (to handle K=0 or Theta=0)
      //--- * other steps are performed only for K>0 and Theta>0
      //--- * we pass through columns of Q and copy I-th column into left block (Qf) or
      //---   right one (Qc) of TK2, depending on presence of X[i] in the active set.
      //---   CIdx0 variable contains current position for insertion into upper block,
      //---   CIdx1 contains current position for insertion into lower one.
      //--- * we calculate Qc*xc-r and store it into Tmp0
      //--- * we calculate TK0 and TK1
      //--- * we multiply leading part of TK2 which stores Qf by sqrt(theta)
      //---   it is important to perform this step AFTER calculation of TK0 and TK1,
      //---   because we need original (non-modified) Qf to calculate TK0 and TK1.
      //
      s.m_tk2=matrix<double>::Zeros(k,n);
      s.m_tk1=vector<double>::Zeros(n);
      s.m_tk0=0.0;
      if(s.m_k>0 && s.m_theta>0.0)
        {
         //
         //--- Split Q into Qf and Qc
         //--- Calculate Qc*xc-r, store in Tmp0
         //
         s.m_tmp0.Resize(k);
         s.m_tmp1=vector<double>::Zeros(k);
         cidx0=0;
         cidx1=nfree;
         for(j=0; j<n; j++)
           {
            if(s.m_activeset[j])
              {
               s.m_tk2.Col(cidx1,s.m_q.Col(j)+0);
               s.m_tmp1+=s.m_q.Col(j)*s.m_txc[cidx1];
               cidx1++;
              }
            else
              {
               s.m_tk2.Col(cidx0,s.m_q.Col(j)+0);
               cidx0++;
              }
           }
         s.m_tmp0=s.m_tmp1-s.m_r+0;
         //--- Calculate TK0
         v=0.0;
         for(i=0; i<k; i++)
            v+= s.m_theta*(0.5*CMath::Sqr(s.m_tmp1[i])-s.m_r[i]*s.m_tmp0[i]-0.5*CMath::Sqr(s.m_r[i]));
         s.m_tk0=v;
         //--- Calculate TK1
         if(nfree>0)
           {
            for(i=0; i<k; i++)
              {
               v=s.m_theta*s.m_tmp0[i];
               for(i_=0; i_<nfree; i_++)
                  s.m_tk1.Add(i_,v*s.m_tk2.Get(i,i_));
              }
           }
         //--- Calculate TK2
         if(nfree>0)
           {
            v=MathSqrt(s.m_theta);
            for(i=0; i<k; i++)
              {
               for(i_=0; i_<nfree; i_++)
                  s.m_tk2.Mul(i,i_,v);
              }
           }
        }
     }
//--- Re-evaluate TB
   if(s.m_isactivesetchanged || s.m_islineartermchanged)
     {
      ridx0=0;
      ridx1=nfree;
      for(i=0; i<n; i++)
        {
         if(s.m_activeset[i])
           {
            s.m_tb.Set(ridx1,s.m_b[i]);
            ridx1++;
           }
         else
           {
            s.m_tb.Set(ridx0,s.m_b[i]);
            ridx0++;
           }
        }
     }
//--- Compose ECA: either dense ECA or diagonal ECA
   if((s.m_isactivesetchanged || s.m_ismaintermchanged) && nfree>0)
     {
      if(s.m_alpha>0.0)
        {
         //--- Dense ECA
         s.m_ecakind=0;
         for(i=0; i<nfree; i++)
           {
            for(j=i; j<nfree; j++)
               s.m_ecadense.Set(i,j,s.m_tq2dense.Get(i,j));
           }
         if(!CTrFac::SPDMatrixCholeskyRec(s.m_ecadense,0,nfree,true,s.m_tmp0))
           {
            result=false;
            return(result);
           }
        }
      else
        {
         //--- Diagonal ECA
         s.m_ecakind=1;
         for(i=0; i<nfree; i++)
           {
            if(s.m_tq2diag[i]<0.0)
              {
               result=false;
               return(result);
              }
            s.m_ecadiag.Set(i,MathSqrt(s.m_tq2diag[i]));
           }
        }
     }
//--- Compose EQ
   if(s.m_isactivesetchanged || s.m_issecondarytermchanged)
     {
      for(i=0; i<k; i++)
         for(j=0; j<nfree; j++)
            s.m_eq.Set(i,j,s.m_tk2.Get(i,j));
     }
//--- Calculate ECCM
   if((s.m_isactivesetchanged || s.m_ismaintermchanged || s.m_issecondarytermchanged) && s.m_k>0 && s.m_theta>0.0 && nfree>0)
     {
      //--- Calculate ECCM-Cholesky factor of the "effective" capacitance
      //--- matrix CM = I + EQ*inv(EffectiveA)*EQ'.
      //--- We calculate CM as follows:
      //---   CM = I + EQ*inv(EffectiveA)*EQ'
      //---      = I + EQ*ECA^(-1)*ECA^(-T)*EQ'
      //---      = I + (EQ*ECA^(-1))*(EQ*ECA^(-1))'
      //--- Then we perform Cholesky decomposition of CM.
      s.m_tmp2.Resize(k,n);
      CAblas::RMatrixCopy(k,nfree,s.m_eq,0,0,s.m_tmp2,0,0);
      //--- check
      if(!CAp::Assert(s.m_ecakind==0 || s.m_ecakind==1,__FUNCTION__+": unexpected ECAKind"))
         return(false);
      if(s.m_ecakind==0)
         CAblas::RMatrixRightTrsM(k,nfree,s.m_ecadense,0,0,true,false,0,s.m_tmp2,0,0);
      if(s.m_ecakind==1)
        {
         for(i=0; i<k; i++)
            for(j=0; j<nfree; j++)
               s.m_tmp2.Mul(i,j,1.0/s.m_ecadiag[j]);
        }
      s.m_eccm=matrix<double>::Identity(k,k);
      CAblas::RMatrixSyrk(k,nfree,1.0,s.m_tmp2,0,0,0,1.0,s.m_eccm,0,0,true);
      if(!CTrFac::SPDMatrixCholeskyRec(s.m_eccm,0,k,true,s.m_tmp0))
        {
         result=false;
         return(result);
        }
     }
//--- Compose EB and EC
//--- NOTE: because these quantities are cheap to compute, we do not
//--- use caching here.
   for(i=0; i<nfree; i++)
      s.m_eb.Set(i,s.m_tq1[i]+s.m_tk1[i]+s.m_tb[i]);
   s.m_ec=s.m_tq0+s.m_tk0;
   for(i=nfree; i<n; i++)
      s.m_ec+=s.m_tb[i]*s.m_txc[i];
//--- Change cache status - everything is cached
   s.m_ismaintermchanged=false;
   s.m_issecondarytermchanged=false;
   s.m_islineartermchanged=false;
   s.m_isactivesetchanged=false;
   return(result);
  }
//+------------------------------------------------------------------+
//| Internal function, solves system Effective_A*x = b.              |
//| It should be called after successful completion of CQMRebuild(). |
//| INPUT PARAMETERS:                                                |
//|   S        -  quadratic model, after call to CQMRebuild()        |
//|   X        -  right part B, array[S.NFree]                       |
//|   Tmp      -  temporary array, automatically reallocated if      |
//|               needed                                             |
//| OUTPUT PARAMETERS:                                               |
//|   X        -  solution, array[S.NFree]                           |
//| NOTE: when called with zero S.NFree, returns silently            |
//| NOTE: this function assumes that EA is non - degenerate          |
//+------------------------------------------------------------------+
void CCQModels::CQMSolveEA(CConvexQuadraticModel &s,CRowDouble &x,
                           CRowDouble &tmp)
  {
//--- check
   if(!CAp::Assert(s.m_ecakind==0 || s.m_ecakind==1 || (s.m_ecakind==-1 && s.m_nfree==0),__FUNCTION__+": unexpected ECAKind"))
      return;

   if(s.m_ecakind==0)
     {
      //--- Dense ECA, use FBLSCholeskySolve() dense m_solver.
      CFbls::FblsCholeskySolve(s.m_ecadense,1.0,s.m_nfree,true,x,tmp);
     }

   if(s.m_ecakind==1)
     {
      //--- Diagonal ECA
      for(int i=0; i<s.m_nfree; i++)
         x.Mul(i,1.0/CMath::Sqr(s.m_ecadiag[i]));
     }
  }
//+------------------------------------------------------------------+
//| This object stores Settings for QQP m_solver.                      |
//| It must be initialized with QQPLoadDefaults().                   |
//| After initialization you may change Settings.                    |
//+------------------------------------------------------------------+
struct CQQPSettings
  {
   int               m_cgmaxits;
   int               m_cgminits;
   int               m_cnmaxupdates;
   int               m_maxouterits;
   int               m_sparsesolver;
   double            m_epsf;
   double            m_epsg;
   double            m_epsx;
   bool              m_cgphase;
   bool              m_cnphase;
   //--- constructor / destructor
                     CQQPSettings(void) { ZeroMemory(this); }
                    ~CQQPSettings(void) {}
   void              Copy(const CQQPSettings &obj);
   //--- overloading
   void              operator=(const CQQPSettings &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CQQPSettings::Copy(const CQQPSettings &obj)
  {
   m_cgmaxits=obj.m_cgmaxits;
   m_cgminits=obj.m_cgminits;
   m_cnmaxupdates=obj.m_cnmaxupdates;
   m_maxouterits=obj.m_maxouterits;
   m_sparsesolver=obj.m_sparsesolver;
   m_epsf=obj.m_epsf;
   m_epsg=obj.m_epsg;
   m_epsx=obj.m_epsx;
   m_cgphase=obj.m_cgphase;
   m_cnphase=obj.m_cnphase;
  }
//+------------------------------------------------------------------+
//| This object stores temporaries used by QuickQP m_solver.           |
//+------------------------------------------------------------------+
struct CQQPBuffers
  {
   int               m_akind;
   int               m_cnmodelage;
   int               m_n;
   int               m_nfree;
   int               m_repinneriterationscount;
   int               m_repncholesky;
   int               m_repncupdates;
   int               m_repouteriterationscount;
   double            m_absamax;
   double            m_absasum;
   double            m_absasum2;
   bool              m_activated[];
   bool              m_havebndl[];
   bool              m_havebndu[];
   bool              m_sparseupper;
   bool              m_tmpcnb[];
   CSparseMatrix     m_sparsea;
   CSparseMatrix     m_sparsecca;
   CSparseBuffers    m_sbuf;
   CSActiveSet       m_sas;
   CRowInt           m_tmpcni;
   CRowInt           m_yidx;
   CRowDouble        m_b;
   CRowDouble        m_bndl;
   CRowDouble        m_bndu;
   CRowDouble        m_cgc;
   CRowDouble        m_cgp;
   CRowDouble        m_dc;
   CRowDouble        m_dp;
   CRowDouble        m_gc;
   CRowDouble        m_regdiag;
   CRowDouble        m_regx0;
   CRowDouble        m_stpbuf;
   CRowDouble        m_tmp0;
   CRowDouble        m_tmp1;
   CRowDouble        m_tmpcn;
   CRowDouble        m_xf;
   CRowDouble        m_xp;
   CRowDouble        m_xs;
   CMatrixDouble     m_densea;
   CMatrixDouble     m_densez;
   //--- constructor / destructor
                     CQQPBuffers(void);
                    ~CQQPBuffers(void) {}
   //---
   void              Copy(const CQQPBuffers &obj);
   //--- overloading
   void              operator=(const CQQPBuffers &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CQQPBuffers::CQQPBuffers(void)
  {
   m_akind=0;
   m_cnmodelage=0;
   m_n=0;
   m_nfree=0;
   m_repinneriterationscount=0;
   m_repncholesky=0;
   m_repncupdates=0;
   m_repouteriterationscount=0;
   m_absamax=0;
   m_absasum=0;
   m_absasum2=0;
   m_sparseupper=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CQQPBuffers::Copy(const CQQPBuffers &obj)
  {
   m_akind=obj.m_akind;
   m_cnmodelage=obj.m_cnmodelage;
   m_n=obj.m_n;
   m_nfree=obj.m_nfree;
   m_repinneriterationscount=obj.m_repinneriterationscount;
   m_repncholesky=obj.m_repncholesky;
   m_repncupdates=obj.m_repncupdates;
   m_repouteriterationscount=obj.m_repouteriterationscount;
   m_absamax=obj.m_absamax;
   m_absasum=obj.m_absasum;
   m_absasum2=obj.m_absasum2;
   ArrayCopy(m_activated,obj.m_activated);
   ArrayCopy(m_havebndl,obj.m_havebndl);
   ArrayCopy(m_havebndu,obj.m_havebndu);
   m_sparseupper=obj.m_sparseupper;
   ArrayCopy(m_tmpcnb,obj.m_tmpcnb);
   m_sparsea=obj.m_sparsea;
   m_sparsecca=obj.m_sparsecca;
   m_sbuf=obj.m_sbuf;
   m_sas=obj.m_sas;
   m_tmpcni=obj.m_tmpcni;
   m_yidx=obj.m_yidx;
   m_b=obj.m_b;
   m_bndl=obj.m_bndl;
   m_bndu=obj.m_bndu;
   m_cgc=obj.m_cgc;
   m_cgp=obj.m_cgp;
   m_dc=obj.m_dc;
   m_dp=obj.m_dp;
   m_gc=obj.m_gc;
   m_regdiag=obj.m_regdiag;
   m_regx0=obj.m_regx0;
   m_stpbuf=obj.m_stpbuf;
   m_tmp0=obj.m_tmp0;
   m_tmp1=obj.m_tmp1;
   m_tmpcn=obj.m_tmpcn;
   m_xf=obj.m_xf;
   m_xp=obj.m_xp;
   m_xs=obj.m_xs;
   m_densea=obj.m_densea;
   m_densez=obj.m_densez;
  }
//+------------------------------------------------------------------+
//| QQP m_solver                                                       |
//+------------------------------------------------------------------+
class CQQPSolver
  {
public:
   static const int  m_quickqprestartcg;
   static const double m_regz;

   static void       QQPLoadDefaults(int n,CQQPSettings &s);
   static void       QQPCopySettings(CQQPSettings &src,CQQPSettings &dst);
   static void       QQPPreAllocateGrowDense(CQQPBuffers &sstate,int nexpected,int ngrowto);
   static void       QQPOptimize(CConvexQuadraticModel &cqmac,CSparseMatrix &sparseac,CMatrixDouble &denseac,int akind,bool IsUpper,CRowDouble &bc,CRowDouble &bndlc,CRowDouble &bnduc,CRowDouble &sc,CRowDouble &xoriginc,int nc,CQQPSettings &Settings,CQQPBuffers &sstate,CRowDouble &xs,int &terminationtype);

private:
   static double     ProjectedTargetFunction(CQQPBuffers &sstate,CRowDouble &x,CRowDouble &d,double stp,CRowDouble &m_tmp0,CRowDouble &m_tmp1);
   static void       TargetGradient(CQQPBuffers &sstate,CRowDouble &x,CRowDouble &g);
   static void       QuadraticModel(CQQPBuffers &sstate,CRowDouble &x,CRowDouble &d,CRowDouble &g,double &d1,int &d1est,double &d2,int &d2est,CRowDouble &m_tmp0);
   static void       FindBestStepAndMove(CQQPBuffers &sstate,CSActiveSet &sas,CRowDouble &d,double stp,bool needact,int cidx,double cval,CRowDouble &addsteps,int addstepscnt,bool &activated[],CRowDouble &m_tmp0,CRowDouble &m_tmp1);
   static bool       CNewtonBuild(CQQPBuffers &sstate,int sparsesolver,int &ncholesky);
   static bool       CNewtonUpdate(CQQPBuffers &sstate,CQQPSettings &Settings,int &ncupdates);
   static bool       CNewtonStep(CQQPBuffers &sstate,CQQPSettings &Settings,CRowDouble &gc);
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
const int CQQPSolver::m_quickqprestartcg=50;
const double CQQPSolver::m_regz=1.0E-9;
//+------------------------------------------------------------------+
//| This function initializes QQPSettings structure with default     |
//| Settings.                                                        |
//| Newly created structure MUST be initialized by default           |
//| Settings - or by copy of the already initialized structure.      |
//+------------------------------------------------------------------+
void CQQPSolver::QQPLoadDefaults(int n,CQQPSettings &s)
  {
   s.m_epsg=0.0;
   s.m_epsf=0.0;
   s.m_epsx=1.0E-6;
   s.m_maxouterits=0;
   s.m_cgphase=true;
   s.m_cnphase=true;
   s.m_cgminits=5;
   s.m_cgmaxits=MathMax(s.m_cgminits,(int)MathRound(1+0.33*n));
   s.m_sparsesolver=0;
   s.m_cnmaxupdates=(int)MathRound(1+0.1*n);
  }
//+------------------------------------------------------------------+
//| This function initializes QQPSettings  structure  with  copy  of |
//| another, already initialized structure.                          |
//+------------------------------------------------------------------+
void CQQPSolver::QQPCopySettings(CQQPSettings &src,CQQPSettings &dst)
  {
   dst.m_epsg=src.m_epsg;
   dst.m_epsf=src.m_epsf;
   dst.m_epsx=src.m_epsx;
   dst.m_maxouterits=src.m_maxouterits;
   dst.m_cgphase=src.m_cgphase;
   dst.m_cnphase=src.m_cnphase;
   dst.m_cgminits=src.m_cgminits;
   dst.m_cgmaxits=src.m_cgmaxits;
   dst.m_sparsesolver=src.m_sparsesolver;
   dst.m_cnmaxupdates=src.m_cnmaxupdates;
  }
//+------------------------------------------------------------------+
//| This function performs preallocation of internal 2D matrices. If |
//| matrix size is less than expected, we grow to some larger value  |
//| (specified by user).                                             |
//| It can be useful in cases when we solve many subsequent QP       |
//| problems with increasing sizes - helps to avoid multiple         |
//| allocations.                                                     |
//| INPUT PARAMETERS:                                                |
//|   SState      -  object which stores temporaries:                |
//|                  * uninitialized object is automatically         |
//|                    initialized                                   |
//|                  * previously allocated memory is reused as much |
//|                    as possible                                   |
//|   NExpected   -  if internal buffers have size enough for        |
//|                  NExpected, no preallocation happens. If size is |
//|                  less than NExpected, buffers are preallocated up|
//|                  to NGrowTo*NGrowTo                              |
//|   NGrowTo     -  new size                                        |
//| OUTPUT PARAMETERS:                                               |
//|   SState      -  temporary buffers, some of them are preallocated|
//+------------------------------------------------------------------+
void CQQPSolver::QQPPreAllocateGrowDense(CQQPBuffers &sstate,
                                         int nexpected,
                                         int ngrowto)
  {
   if(sstate.m_densea.Rows()<nexpected || sstate.m_densea.Cols()<nexpected)
      sstate.m_densea.Resize(ngrowto,ngrowto);
   if(sstate.m_densez.Rows()<nexpected || sstate.m_densez.Cols()<nexpected)
      sstate.m_densez.Resize(ngrowto,ngrowto);
  }
//+------------------------------------------------------------------+
//| This function runs QQP m_solver; it returns after optimization     |
//| process was completed. Following QP problem is solved:           |
//|   min(0.5 * (x - x_origin)'*A*(x-x_origin)+b' * (x - x_origin))  |
//| subject to boundary constraints.                                 |
//| IMPORTANT: UNLIKE MANY OTHER SOLVERS, THIS FUNCTION DOES NOT     |
//|            REQUIRE YOU TO INITIALIZE STATE OBJECT. IT CAN BE     |
//|            AUTOMATICALLY INITIALIZED DURING SOLUTION PROCESS.    |
//| INPUT PARAMETERS:                                                |
//|   AC       -  for dense problems given by CQM model(AKind = 0)   |
//|   A        -  term of CQM object contains system matrix. Other   |
//|               terms are unspecified and should not be referenced.|
//|   SparseAC -  for sparse problems(AKind = 1)                     |
//|   DenseAC  -  for traditional dense matrices(AKind = 2)          |
//|   AKind    -  matrix term to use:                                |
//|               * 0 for dense CQM(CQMAC)                           |
//|               * 1 for sparse matrix(SparseAC)                    |
//|               * 2 for dense matrix(DenseAC)                      |
//|   IsUpper  -  which triangle of SparseAC / DenseAC stores matrix-|
//|               upper or lower one (for dense matrices this        |
//|               parameter is not actual).                          |
//|   BC       -  linear term, array[NC]                             |
//|   BndLC    -  lower bound, array[NC]                             |
//|   BndUC    -  upper bound, array[NC]                             |
//|   SC       -  scale vector, array[NC]:                           |
//|               * I-Th element contains scale of I-Th variable,    |
//|               * SC[I] > 0                                        |
//|   XOriginC -  origin term, array[NC]. Can be zero.               |
//|   NC       -  number of variables in the  original  formulation  |
//|               (no slack variables).                              |
//|   CLEICC   -  linear equality / inequality constraints. Present  |
//|               version of this function does NOT provide publicly |
//|               available support for linear constraints. This     |
//|               feature will be introduced in the future versions  |
//|               of the function.                                   |
//|   NEC, NIC -  number of equality / inequality constraints. MUST  |
//|               BE ZERO IN THE CURRENT VERSION!!!                  |
//|   Settings -  QQPSettings object initialized by one of the       |
//|               initialization functions.                          |
//|   SState   -  object which stores temporaries:                   |
//|               * uninitialized object is automatically initialized|
//|               * previously allocated memory is reused as much as |
//|                 possible                                         |
//|   XS       -  initial point, array[NC]                           |
//| OUTPUT PARAMETERS:                                               |
//|   XS       -  last point                                         |
//|   TerminationType - termination type:                            |
//|                     *                                            |
//|                     *                                            |
//|                     *                                            |
//+------------------------------------------------------------------+
void CQQPSolver::QQPOptimize(CConvexQuadraticModel &cqmac,
                             CSparseMatrix &sparseac,
                             CMatrixDouble &denseac,
                             int akind,
                             bool IsUpper,
                             CRowDouble &bc,
                             CRowDouble &bndlc,
                             CRowDouble &bnduc,
                             CRowDouble &sc,
                             CRowDouble &xoriginc,
                             int nc,
                             CQQPSettings &Settings,
                             CQQPBuffers &sstate,
                             CRowDouble &xs,
                             int &terminationtype)
  {
//--- create variables
   int    n=0;
   int    i=0;
   int    j=0;
   int    k=0;
   double v=0;
   double vv=0;
   double d2=0;
   double d1=0;
   int    d1est=0;
   int    d2est=0;
   bool   needact;
   double reststp=0;
   double fullstp=0;
   double stpmax=0;
   double stp=0;
   int    stpcnt=0;
   int    cidx=0;
   double cval=0;
   int    cgcnt=0;
   int    cgmax=0;
   int    newtcnt=0;
   int    sparsesolver=0;
   double beta=0;
   bool   b;
   double fprev=0;
   double fcur=0;
   bool   problemsolved;
   bool   isconstrained;
   double f0=0;
   double f1=0;
   int    i_=0;

   terminationtype=0;
//--- Primary checks
   if(!CAp::Assert(akind==0 || akind==1 || akind==2,__FUNCTION__+": incorrect AKind"))
      return;

   sstate.m_n=nc;
   n=sstate.m_n;
   terminationtype=0;
   sstate.m_repinneriterationscount=0;
   sstate.m_repouteriterationscount=0;
   sstate.m_repncholesky=0;
   sstate.m_repncupdates=0;
//--- Several checks
//--- * matrix size
//--- * scale vector
//--- * consistency of bound constraints
//--- * consistency of Settings
   if(akind==1)
     {
      if(!CAp::Assert(CSparse::SparseGetNRows(sparseac)==n,__FUNCTION__+": rows(SparseAC)<>N"))
         return;
      if(!CAp::Assert(CSparse::SparseGetNCols(sparseac)==n,__FUNCTION__+": cols(SparseAC)<>N"))
         return;
     }
   if(!CAp::Assert(CApServ::IsFiniteVector(sc,n) && sc.Min()>0.0,__FUNCTION__+": incorrect scale"))
      return;
   for(i=0; i<n; i++)
     {
      if(MathIsValidNumber(bndlc[i]) && MathIsValidNumber(bnduc[i]))
         if(bndlc[i]>bnduc[i])
           {
            terminationtype=-3;
            return;
           }
     }
   if(!CAp::Assert(Settings.m_cgphase || Settings.m_cnphase,__FUNCTION__+": both phases (CG and Newton) are inactive"))
      return;
//--- Allocate data structures
   CApServ::BVectorSetLengthAtLeast(sstate.m_havebndl,n);
   CApServ::BVectorSetLengthAtLeast(sstate.m_havebndu,n);
   sstate.m_bndl.Resize(n);
   sstate.m_bndu.Resize(n);
   sstate.m_xs.Resize(n);
   sstate.m_xf.Resize(n);
   sstate.m_xp.Resize(n);
   sstate.m_gc.Resize(n);
   sstate.m_cgc.Resize(n);
   sstate.m_cgp.Resize(n);
   sstate.m_dc.Resize(n);
   sstate.m_dp.Resize(n);
   sstate.m_tmp0.Resize(n);
   sstate.m_tmp1.Resize(n);
   sstate.m_stpbuf.Resize(15);
   CSActiveSets::SASInit(n,sstate.m_sas);
//--- Scale/shift problem coefficients:
//---     min { 0.5*(x-x0)'*A*(x-x0) + b'*(x-x0) }
//--- becomes (after transformation "x = S*y+x0")
//---     min { 0.5*y'*(S*A*S)*y + (S*b)'*y
//--- Modified A_mod=S*A*S and b_mod=S*(b+A*x0) are
//--- stored into SState.DenseA and SState.B.
   sstate.m_b=sc*bc+0;
   sstate.m_b.Resize(n);
   sstate.m_akind=-99;
   if(akind==0)
     {
      //--- Dense QP problem - just copy and scale.
      sstate.m_densea.Resize(n,n);
      CCQModels::CQMGetA(cqmac,sstate.m_densea);
      sstate.m_akind=0;
      sstate.m_absamax=0;
      sstate.m_absasum=0;
      sstate.m_absasum2=0;
      for(i=0; i<n; i++)
         for(j=0; j<n; j++)
           {
            v=sc[i]*sstate.m_densea.Get(i,j)*sc[j];
            vv=MathAbs(v);
            sstate.m_densea.Set(i,j,v);
            sstate.m_absamax=MathMax(sstate.m_absamax,vv);
            sstate.m_absasum=sstate.m_absasum+vv;
            sstate.m_absasum2=sstate.m_absasum2+vv*vv;
           }
     }
   if(akind==1)
     {
      //--- Sparse QP problem - a bit tricky. Depending on format of the
      //--- input we use different strategies for copying matrix:
      //--- * SKS matrices are copied to SKS format
      //--- * anything else is copied to CRS format
      CSparse::SparseCopyToSKSBuf(sparseac,sstate.m_sparsea);
      if(IsUpper)
         CSparse::SparseTransposeSKS(sstate.m_sparsea);
      sstate.m_akind=1;
      sstate.m_sparseupper=false;
      sstate.m_absamax=0;
      sstate.m_absasum=0;
      sstate.m_absasum2=0;
      for(i=0; i<n; i++)
        {
         k=sstate.m_sparsea.m_RIdx[i];
         for(j=i-sstate.m_sparsea.m_DIdx[i]; j<=i; j++)
           {
            v=sc[i]*sstate.m_sparsea.m_Vals[k]*sc[j];
            vv=MathAbs(v);
            sstate.m_sparsea.m_Vals.Set(k,v);
            if(i==j)
              {
               //--- Diagonal terms are counted only once
               sstate.m_absamax=MathMax(sstate.m_absamax,vv);
               sstate.m_absasum=sstate.m_absasum+vv;
               sstate.m_absasum2=sstate.m_absasum2+vv*vv;
              }
            else
              {
               //--- Offdiagonal terms are counted twice
               sstate.m_absamax=MathMax(sstate.m_absamax,vv);
               sstate.m_absasum=sstate.m_absasum+2*vv;
               sstate.m_absasum2=sstate.m_absasum2+2*vv*vv;
              }
            k++;
           }
        }
     }
   if(akind==2)
     {
      //--- Dense QP problem - just copy and scale.
      sstate.m_densea.Resize(n,n);
      sstate.m_akind=0;
      sstate.m_absamax=0;
      sstate.m_absasum=0;
      sstate.m_absasum2=0;
      if(IsUpper)
        {
         for(i=0; i<n; i++)
           {
            for(j=i; j<n; j++)
              {
               v=sc[i]*denseac.Get(i,j)*sc[j];
               vv=MathAbs(v);
               sstate.m_densea.Set(i,j,v);
               sstate.m_densea.Set(j,i,v);
               if((double)(i)==v)
                  k=1;
               else
                  k=2;
               sstate.m_absamax=MathMax(sstate.m_absamax,vv);
               sstate.m_absasum+=vv*k;
               sstate.m_absasum2+=vv*vv*k;
              }
           }
        }
      else
        {
         for(i=0; i<n; i++)
           {
            for(j=0; j<=i; j++)
              {
               v=sc[i]*denseac.Get(i,j)*sc[j];
               vv=MathAbs(v);
               sstate.m_densea.Set(i,j,v);
               sstate.m_densea.Set(j,i,v);
               if((double)(i)==v)
                  k=1;
               else
                  k=2;
               sstate.m_absamax=MathMax(sstate.m_absamax,vv);
               sstate.m_absasum=sstate.m_absasum+vv*k;
               sstate.m_absasum2=sstate.m_absasum2+vv*vv*k;
              }
           }
        }
     }
//--- check
   if(!CAp::Assert(sstate.m_akind>=0,__FUNCTION__+": integrity check failed"))
      return;
//--- Load box constraints into State structure.
//--- We apply transformation to variables: y=(x-x_origin)/s,
//--- each of the constraints is appropriately shifted/scaled.
   for(i=0; i<n; i++)
     {
      sstate.m_havebndl[i]=MathIsValidNumber(bndlc[i]);
      if(sstate.m_havebndl[i])
         sstate.m_bndl.Set(i,(bndlc[i]-xoriginc[i])/sc[i]);
      else
        {
         //--- check
         if(!CAp::Assert(AL_NEGINF==bndlc[i],__FUNCTION__+": incorrect lower bound"))
            return;
         sstate.m_bndl.Set(i,AL_NEGINF);
        }
      sstate.m_havebndu[i]=MathIsValidNumber(bnduc[i]);
      if(sstate.m_havebndu[i])
         sstate.m_bndu.Set(i,(bnduc[i]-xoriginc[i])/sc[i]);
      else
        {
         //--- check
         if(!CAp::Assert(AL_POSINF==bnduc[i],__FUNCTION__+": incorrect upper bound"))
            return;
         sstate.m_bndu.Set(i,AL_POSINF);
        }
     }
//--- Process initial point:
//--- * set it to XS-XOriginC
//--- * make sure that boundary constraints are preserved by transformation
   for(i=0; i<n; i++)
     {
      sstate.m_xs.Set(i,(xs[i]-xoriginc[i])/sc[i]);
      if(sstate.m_havebndl[i] && sstate.m_xs[i]<sstate.m_bndl[i])
         sstate.m_xs.Set(i,sstate.m_bndl[i]);
      if(sstate.m_havebndu[i] && sstate.m_xs[i]>sstate.m_bndu[i])
         sstate.m_xs.Set(i,sstate.m_bndu[i]);
      if(sstate.m_havebndl[i] && xs[i]==bndlc[i])
         sstate.m_xs.Set(i,sstate.m_bndl[i]);
      if(sstate.m_havebndu[i] && xs[i]==bnduc[i])
         sstate.m_xs.Set(i,sstate.m_bndu[i]);
     }
//--- Select sparse direct m_solver
   if(akind==1)
     {
      sparsesolver=Settings.m_sparsesolver;
      if(sparsesolver==0)
         sparsesolver=1;
      if(CSparse::SparseIsSKS(sstate.m_sparsea))
         sparsesolver=2;
      sparsesolver=2;
      //--- check
      if(!CAp::Assert(sparsesolver==1 || sparsesolver==2,__FUNCTION__+": incorrect SparseSolver"))
         return;
     }
   else
      sparsesolver=0;
//--- For unconstrained problems - try to use fast approach which requires
//--- just one unregularized Cholesky decomposition for solution. If it fails,
//--- switch to general QQP code.
   problemsolved=false;
   isconstrained=false;
   for(i=0; i<n; i++)
      isconstrained=(isconstrained || sstate.m_havebndl[i] || sstate.m_havebndu[i]);
   if(!isconstrained && Settings.m_cnphase && akind==0)
     {
      sstate.m_densez=sstate.m_densea;
      sstate.m_densez.Resize(n,n);
      sstate.m_tmpcn.Resize(n);
      sstate.m_repncholesky++;
      if(CTrFac::SPDMatrixCholeskyRec(sstate.m_densez,0,n,true,sstate.m_tmpcn))
        {
         sstate.m_xf=sstate.m_xs;
         sstate.m_dc=vector<double>::Zeros(n);
         f0=ProjectedTargetFunction(sstate,sstate.m_xf,sstate.m_dc,0.0,sstate.m_tmpcn,sstate.m_tmp1);
         for(k=0; k<=3; k++)
           {
            CAblas::RMatrixMVect(n,n,sstate.m_densea,0,0,0,sstate.m_xf,0,sstate.m_gc,0);
            sstate.m_gc+=sstate.m_b;
            sstate.m_dc=sstate.m_gc;
            sstate.m_dc*=(-1.0);
            CFbls::FblsCholeskySolve(sstate.m_densez,1.0,n,true,sstate.m_dc,sstate.m_tmpcn);
            f1=ProjectedTargetFunction(sstate,sstate.m_xf,sstate.m_dc,1.0,sstate.m_tmpcn,sstate.m_tmp1);
            if(f1>=f0)
               break;
            sstate.m_xf+=sstate.m_dc;
            f0=f1;
           }
         terminationtype=2;
         problemsolved=true;
        }
     }
//--- Attempt to solve problem with fast approach failed, use generic QQP
   if(!problemsolved)
     {
      //--- Prepare "active set" structure
      CSActiveSets::SASSetBC(sstate.m_sas,sstate.m_bndl,sstate.m_bndu);
      if(!CSActiveSets::SASStartOptimization(sstate.m_sas,sstate.m_xs))
        {
         terminationtype=-3;
         return;
        }
      //--- Main loop.
      //--- Following variables are used:
      //--- * GC stores current gradient (unconstrained)
      //--- * CGC stores current gradient (constrained)
      //--- * DC stores current search direction
      //--- * CGP stores constrained gradient at previous point
      //---   (zero on initial entry)
      //--- * DP stores previous search direction
      //---   (zero on initial entry)
      cgmax=Settings.m_cgminits;
      sstate.m_repinneriterationscount=0;
      sstate.m_repouteriterationscount=0;
      while(true)
        {
         if(Settings.m_maxouterits>0 && sstate.m_repouteriterationscount>=Settings.m_maxouterits)
           {
            terminationtype=5;
            break;
           }
         if(sstate.m_repouteriterationscount>0)
           {
            //--- Check EpsF- and EpsX-based stopping criteria.
            //--- Because problem was already scaled, we do not scale step before checking its length.
            //--- NOTE: these checks are performed only after at least one outer iteration was made.
            if(Settings.m_epsf>0.0)
              {
               //--- NOTE 1: here we rely on the fact that ProjectedTargetFunction() ignore D when Stp=0
               //--- NOTE 2: code below handles situation when update increases function value instead
               //---         of decreasing it.
               fprev=ProjectedTargetFunction(sstate,sstate.m_xp,sstate.m_dc,0.0,sstate.m_tmp0,sstate.m_tmp1);
               fcur=ProjectedTargetFunction(sstate,sstate.m_sas.m_xc,sstate.m_dc,0.0,sstate.m_tmp0,sstate.m_tmp1);
               if((fprev-fcur)<=(Settings.m_epsf*MathMax(MathAbs(fprev),MathMax(MathAbs(fcur),1.0))))
                 {
                  terminationtype=1;
                  break;
                 }
              }
            if(Settings.m_epsx>0.0)
              {
               v=0.0;
               for(i=0; i<n; i++)
                 {
                  v+= CMath::Sqr(sstate.m_xp[i]-sstate.m_sas.m_xc[i]);
                 }
               if(MathSqrt(v)<=Settings.m_epsx)
                 {
                  terminationtype=2;
                  break;
                 }
              }
           }
         sstate.m_repouteriterationscount++;
         sstate.m_xp=sstate.m_sas.m_xc;
         if(!Settings.m_cgphase)
            cgmax=0;
         sstate.m_cgp=vector<double>::Zeros(n);
         sstate.m_dp=vector<double>::Zeros(n);
         for(cgcnt=0; cgcnt<=cgmax-1; cgcnt++)
           {
            //--- Calculate unconstrained gradient GC for "extended" QP problem
            //--- Determine active set, current constrained gradient CGC.
            //--- Check gradient-based stopping condition.
            //
            //--- NOTE: because problem was scaled, we do not have to apply scaling
            //---       to gradient before checking stopping condition.
            TargetGradient(sstate,sstate.m_sas.m_xc,sstate.m_gc);
            CSActiveSets::SASReactivateConstraints(sstate.m_sas,sstate.m_gc);
            sstate.m_cgc=sstate.m_gc;
            CSActiveSets::SASConstrainedDirection(sstate.m_sas,sstate.m_cgc);
            v=CAblasF::RDotV2(n,sstate.m_cgc);
            if(MathSqrt(v)<=Settings.m_epsg)
              {
               terminationtype=4;
               break;
              }
            //--- Prepare search direction DC and explore it.
            //--- We try to use CGP/DP to prepare conjugate gradient step,
            //--- but we resort to steepest descent step (Beta=0) in case
            //--- we are at I-th boundary, but DP[I]<>0.
            //--- Such approach allows us to ALWAYS have feasible DC, with
            //--- guaranteed compatibility with both feasible area and current
            //--- active set.
            //--- Automatic CG reset performed every time DP is incompatible
            //--- with current active set and/or feasible area. We also
            //--- perform reset every QuickQPRestartCG iterations.
            sstate.m_dc=sstate.m_cgc;
            sstate.m_dc*=(-1.0);
            v=0.0;
            vv=0.0;
            b=false;
            for(i=0; i<n; i++)
              {
               v+= sstate.m_cgc[i]*sstate.m_cgc[i];
               vv+=sstate.m_cgp[i]*sstate.m_cgp[i];
               b=b || (sstate.m_havebndl[i] && sstate.m_sas.m_xc[i]==sstate.m_bndl[i] && sstate.m_dp[i]!=0.0);
               b=b || (sstate.m_havebndu[i] && sstate.m_sas.m_xc[i]==sstate.m_bndu[i] && sstate.m_dp[i]!=0.0);
              }
            b=b || vv==0.0;
            b=b || cgcnt%m_quickqprestartcg==0;
            if(!b)
               beta=v/vv;
            else
               beta=0.0;
            sstate.m_dc+=sstate.m_dp*beta+0;
            CSActiveSets::SASConstrainedDirection(sstate.m_sas,sstate.m_dc);
            CSActiveSets::SASExploreDirection(sstate.m_sas,sstate.m_dc,stpmax,cidx,cval);
            //--- Build quadratic model of F along descent direction:
            //---     F(xc+alpha*D) = D2*alpha^2 + D1*alpha
            //--- Terminate algorithm if needed.
            //--- NOTE: we do not maintain constant term D0
            QuadraticModel(sstate,sstate.m_sas.m_xc,sstate.m_dc,sstate.m_gc,d1,d1est,d2,d2est,sstate.m_tmp0);
            if(d1==0.0 && d2==0.0)
              {
               //--- D1 and D2 are exactly zero, success.
               //--- After this if-then we assume that D is non-zero.
               terminationtype=4;
               break;
              }
            if(d1est>=0)
              {
               //--- Numerical noise is too large, it means that we are close
               //--- to minimum - and that further improvement is impossible.
               //--- After this if-then we assume that D1 is definitely negative
               //--- (even under presence of numerical errors).
               terminationtype=7;
               break;
              }
            if(d2est<=0 && cidx<0)
              {
               //--- Function is unbounded from below:
               //--- * D1<0 (verified by previous block)
               //--- * D2Est<=0, which means that either D2<0 - or it can not
               //---   be reliably distinguished from zero.
               //--- * step is unconstrained
               //--- If these conditions are true, we abnormally terminate QP
               //--- algorithm with return code -4
               terminationtype=-4;
               break;
              }
            //--- Perform step along DC.
            //--- In this block of code we maintain two step length:
            //--- * RestStp -  restricted step, maximum step length along DC which does
            //---              not violate constraints
            //--- * FullStp -  step length along DC which minimizes quadratic function
            //---              without taking constraints into account. If problem is
            //---              unbounded from below without constraints, FullStp is
            //---              forced to be RestStp.
            //--- So, if function is convex (D2>0):
            //--- * FullStp = -D1/(2*D2)
            //--- * RestStp = restricted FullStp
            //--- * 0<=RestStp<=FullStp
            //--- If function is non-convex, but bounded from below under constraints:
            //--- * RestStp = step length subject to constraints
            //--- * FullStp = RestStp
            //--- After RestStp and FullStp are initialized, we generate several trial
            //--- steps which are different multiples of RestStp and FullStp.
            if(d2est>0)
              {
               //--- check
               if(!CAp::Assert(d1<0.0,__FUNCTION__+": internal error"))
                  return;
               fullstp=-(d1/(2*d2));
               needact=(fullstp>=stpmax);
               if(needact)
                 {
                  //--- check
                  if(!CAp::Assert(sstate.m_stpbuf.Size()>=3,__FUNCTION__+": StpBuf overflow"))
                     return;
                  reststp=stpmax;
                  stp=reststp;
                  sstate.m_stpbuf.Set(0,reststp*4);
                  sstate.m_stpbuf.Set(1,fullstp);
                  sstate.m_stpbuf.Set(2,fullstp/4);
                  stpcnt=3;
                 }
               else
                 {
                  reststp=fullstp;
                  stp=fullstp;
                  stpcnt=0;
                 }
              }
            else
              {
               //--- check
               if(!CAp::Assert(cidx>=0,__FUNCTION__+": internal error"))
                  return;
               if(!CAp::Assert(sstate.m_stpbuf.Size()>=2,__FUNCTION__+": StpBuf overflow"))
                  return;
               reststp=stpmax;
               fullstp=stpmax;
               stp=reststp;
               needact=true;
               sstate.m_stpbuf.Set(0,4*reststp);
               stpcnt=1;
              }
            FindBestStepAndMove(sstate,sstate.m_sas,sstate.m_dc,stp,needact,cidx,cval,sstate.m_stpbuf,stpcnt,sstate.m_activated,sstate.m_tmp0,sstate.m_tmp1);
            //--- Update CG information.
            sstate.m_dp=sstate.m_dc;
            sstate.m_cgp=sstate.m_cgc;
            //--- Update iterations counter
            sstate.m_repinneriterationscount++;
           }
         if(terminationtype!=0)
            break;
         cgmax=Settings.m_cgmaxits;
         //--- Generate YIdx - reordering of variables for constrained Newton phase.
         //--- Free variables come first, fixed are last ones.
         newtcnt=0;
         while(true)
           {
            //--- Skip iteration if constrained Newton is turned off.
            if(!Settings.m_cnphase)
               break;
            //--- At the first iteration   - build Cholesky decomposition of Hessian.
            //--- At subsequent iterations - refine Hessian by adding new constraints.
            //--- Loop is terminated in following cases:
            //--- * Hessian is not positive definite subject to current constraints
            //---   (termination during initial decomposition)
            //--- * there were no new constraints being activated
            //---   (termination during update)
            //--- * all constraints were activated during last step
            //---   (termination during update)
            //--- * CNMaxUpdates were performed on matrix
            //---   (termination during update)
            if(newtcnt==0)
              {
               //--- Perform initial Newton step. If Cholesky decomposition fails,
               //--- increase number of CG iterations to CGMaxIts - it should help
               //--- us to find set of constraints which will make matrix positive
               //--- definite.
               b=CNewtonBuild(sstate,sparsesolver,sstate.m_repncholesky);
               if(b)
                  cgmax=Settings.m_cgminits;
              }
            else
               b=CNewtonUpdate(sstate,Settings,sstate.m_repncupdates);
            if(!b)
               break;
            newtcnt++;
            //--- Calculate gradient GC.
            TargetGradient(sstate,sstate.m_sas.m_xc,sstate.m_gc);
            //--- Bound-constrained Newton step
            sstate.m_dc=sstate.m_gc;
            if(!CNewtonStep(sstate,Settings,sstate.m_dc))
               break;
            QuadraticModel(sstate,sstate.m_sas.m_xc,sstate.m_dc,sstate.m_gc,d1,d1est,d2,d2est,sstate.m_tmp0);
            if(d1est>=0)
              {
               //--- We are close to minimum, derivative is nearly zero, break Newton iteration
               break;
              }
            if(d2est>0)
              {
               //--- Positive definite matrix, we can perform Newton step
               //--- check
               if(!CAp::Assert(d1<0.0,__FUNCTION__+": internal error"))
                  return;
               fullstp=-(d1/(2*d2));
               CSActiveSets::SASExploreDirection(sstate.m_sas,sstate.m_dc,stpmax,cidx,cval);
               needact=(fullstp>=stpmax);
               if(needact)
                 {
                  //--- check
                  if(!CAp::Assert(sstate.m_stpbuf.Size()>=3,__FUNCTION__+": StpBuf overflow"))
                     return;
                  reststp=stpmax;
                  stp=reststp;
                  sstate.m_stpbuf.Set(0,reststp*4);
                  sstate.m_stpbuf.Set(1,fullstp);
                  sstate.m_stpbuf.Set(2,fullstp/4);
                  stpcnt=3;
                 }
               else
                 {
                  reststp=fullstp;
                  stp=fullstp;
                  stpcnt=0;
                 }
               FindBestStepAndMove(sstate,sstate.m_sas,sstate.m_dc,stp,needact,cidx,cval,sstate.m_stpbuf,stpcnt,sstate.m_activated,sstate.m_tmp0,sstate.m_tmp1);
              }
            else
              {
               //--- Matrix is semi-definite or indefinite, but regularized
               //--- Cholesky succeeded and gave us descent direction in DC.
               //--- We will investigate it and try to perform descent step:
               //--- * first, we explore direction:
               //---   * if it is unbounded, we stop algorithm with
               //---     appropriate termination code -4.
               //---   * if StpMax=0, we break Newton phase and return to
               //---     CG phase - constraint geometry is complicated near
               //---     current point, so it is better to use simpler algo.
               //--- * second, we check that bounded step decreases function;
               //---   if not, we again skip to CG phase
               //--- * finally, we use FindBestStep...() function to choose
               //---   between bounded step and projection of full-length step
               //---   (latter may give additional decrease in
               CSActiveSets::SASExploreDirection(sstate.m_sas,sstate.m_dc,stpmax,cidx,cval);
               if(cidx<0)
                 {
                  //--- Function is unbounded from below:
                  //--- * D1<0 (verified by previous block)
                  //--- * D2Est<=0, which means that either D2<0 - or it can not
                  //---   be reliably distinguished from zero.
                  //--- * step is unconstrained
                  //--- If these conditions are true, we abnormally terminate QP
                  //--- algorithm with return code -4
                  terminationtype=-4;
                  break;
                 }
               if(stpmax==0.0)
                 {
                  //--- Resort to CG phase.
                  //--- Increase number of CG iterations.
                  cgmax=Settings.m_cgmaxits;
                  break;
                 }
               //--- check
               if(!CAp::Assert(stpmax>0.0,__FUNCTION__+": internal error"))
                  return;
               f0=ProjectedTargetFunction(sstate,sstate.m_sas.m_xc,sstate.m_dc,0.0,sstate.m_tmp0,sstate.m_tmp1);
               f1=ProjectedTargetFunction(sstate,sstate.m_sas.m_xc,sstate.m_dc,stpmax,sstate.m_tmp0,sstate.m_tmp1);
               if(f1>=f0)
                 {
                  //--- Descent direction does not actually decrease function value.
                  //--- Resort to CG phase
                  //--- Increase number of CG iterations.
                  cgmax=Settings.m_cgmaxits;
                  break;
                 }
               //--- check
               if(!CAp::Assert(sstate.m_stpbuf.Size()>=3,__FUNCTION__+": StpBuf overflow"))
                  return;
               reststp=stpmax;
               stp=reststp;
               sstate.m_stpbuf.Set(0,reststp*4);
               sstate.m_stpbuf.Set(1,1.00);
               sstate.m_stpbuf.Set(2,0.25);
               stpcnt=3;
               FindBestStepAndMove(sstate,sstate.m_sas,sstate.m_dc,stp,true,cidx,cval,sstate.m_stpbuf,stpcnt,sstate.m_activated,sstate.m_tmp0,sstate.m_tmp1);
              }
           }
         if(terminationtype!=0)
            break;
        }
      CSActiveSets::SASStopOptimization(sstate.m_sas);
      sstate.m_xf=sstate.m_sas.m_xc;
     }
//--- Stop optimization and unpack results.
//--- Add XOriginC to XS and make sure that boundary constraints are
//--- both (a) satisfied, (b) preserved. Former means that "shifted"
//--- point is feasible, while latter means that point which was exactly
//--- at the boundary before shift will be exactly at the boundary
//--- after shift.
   for(i=0; i<n; i++)
     {
      xs.Set(i,sc[i]*sstate.m_xf[i]+xoriginc[i]);
      if(sstate.m_havebndl[i] && xs[i]<bndlc[i])
         xs.Set(i,bndlc[i]);
      if(sstate.m_havebndu[i] && xs[i]>bnduc[i])
         xs.Set(i,bnduc[i]);
      if(sstate.m_havebndl[i] && sstate.m_xf[i]==sstate.m_bndl[i])
         xs.Set(i,bndlc[i]);
      if(sstate.m_havebndu[i] && sstate.m_xf[i]==sstate.m_bndu[i])
         xs.Set(i,bnduc[i]);
     }
  }
//+------------------------------------------------------------------+
//| Target function at point PROJ(X + Stp*D), where PROJ(.) is a     |
//| projection into feasible set.                                    |
//| NOTE: if Stp = 0, D is not referenced at all. Thus, there is no  |
//|       need to fill it by some meaningful values for Stp = 0.     |
//| This subroutine uses temporary buffers  Tmp0 / 1,  which  are    |
//| automatically resized if needed.                                 |
//+------------------------------------------------------------------+
double CQQPSolver::ProjectedTargetFunction(CQQPBuffers &sstate,
                                           CRowDouble &x,
                                           CRowDouble &d,
                                           double stp,
                                           CRowDouble &m_tmp0,
                                           CRowDouble &m_tmp1)
  {
//--- create variables
   double result=0;
   int    n=sstate.m_n;
   double v=0;

   m_tmp0.Resize(n);
   m_tmp1.Resize(n);
//--- Calculate projected point
   for(int i=0; i<n; i++)
     {
      if(stp!=0.0)
         v=x[i]+stp*d[i];
      else
         v=x[i];
      if(sstate.m_havebndl[i] && v<sstate.m_bndl[i])
         v=sstate.m_bndl[i];
      if(sstate.m_havebndu[i] && v>sstate.m_bndu[i])
         v=sstate.m_bndu[i];
      m_tmp0.Set(i,v);
     }
//--- Function value at the Tmp0:
//--- f(x) = 0.5*x'*A*x + b'*x
   result=sstate.m_b.Dot(m_tmp0);
   if(sstate.m_akind==0)
     {
      //--- Dense matrix A
      result+=0.5*CAblas::RMatrixSyvMVect(n,sstate.m_densea,0,0,true,m_tmp0,0,m_tmp1);
     }
   else
     {
      //--- sparse matrix A
      //--- check
      if(!CAp::Assert(sstate.m_akind==1,__FUNCTION__+": unexpected AKind in ProjectedTargetFunction"))
         return(0.0);
      result+=0.5*CSparse::SparseVSMV(sstate.m_sparsea,sstate.m_sparseupper,m_tmp0);
     }
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| Gradient of the target function:                                 |
//|         f(x) = 0.5 * x'*A*x + b'*x                               |
//| which is equal to grad = A * x + b                               |
//| Here:                                                            |
//|   * x is array[N]                                                |
//|   * A is array[N, N]                                             |
//|   * b is array[N]                                                |
//| INPUT PARAMETERS:                                                |
//|   SState   -  structure which stores function terms(not modified)|
//|   X        -  location                                           |
//|   G        -  possibly preallocated buffer                       |
//| OUTPUT PARAMETERS:                                               |
//|   G        -  array[N], gradient                                 |
//+------------------------------------------------------------------+
void CQQPSolver::TargetGradient(CQQPBuffers &sstate,
                                CRowDouble &x,
                                CRowDouble &g)
  {
   int n=sstate.m_n;
   g.Resize(n);

   if(sstate.m_akind==0)
     {
      //--- Dense matrix A
      CAblas::RMatrixSymVect(n,1.0,sstate.m_densea,0,0,true,x,0,0.0,g,0);
     }
   else
     {
      //--- Sparse matrix A
      //--- check
      if(!CAp::Assert(sstate.m_akind==1,__FUNCTION__+": unexpected AKind in TargetGradient"))
         return;
      CSparse::SparseSMV(sstate.m_sparsea,sstate.m_sparseupper,x,g);
     }

   g+=sstate.m_b;
  }
//+------------------------------------------------------------------+
//| First and second derivatives of the "extended" target function   |
//| along specified direction. Target function is called "extended"  |
//| because of additional slack variables and has form:              |
//|   f(x)=0.5*x'*A*x+b'*x+penaltyfactor*0.5*(C*x-b)'*(C*x-b)        |
//| with gradient grad = A * x + b + penaltyfactor * C'*(C*x-b)      |
//| Quadratic model has form                                         |
//|         F(x0 + alpha*D) = D2 * alpha ^ 2 + D1 * alpha            |
//| INPUT PARAMETERS:                                                |
//|   SState   -  structure which is used to obtain quadratic term   |
//|               of the model                                       |
//|   X        -  current point, array[N]                            |
//|   D        -  direction across which derivatives are calculated, |
//|               array[N]                                           |
//|   G        -  gradient at current point (pre-calculated by       |
//|               caller), array[N]                                  |
//| OUTPUT PARAMETERS:                                               |
//|   D1       -  linear coefficient                                 |
//|   D1Est    -  estimate of D1 sign, accounting for possible       |
//|               numerical errors:                                  |
//|               *>0    means "almost surely positive"            |
//|               *<0    means "almost surely negative"            |
//|               *=0    means "pessimistic estimate of numerical  |
//|                        errors in D1 is larger than magnitude of  |
//|                        D1 itself; it is impossible to reliably   |
//|                        distinguish D1 from zero".                |
//|   D2       -  quadratic coefficient                              |
//|   D2Est    -  estimate of D2 sign, accounting for possible       |
//|               numerical errors:                                  |
//|               *>0    means "almost surely positive"            |
//|               *<0    means "almost surely negative"            |
//|               *=0    means "pessimistic estimate of numerical  |
//|                        errors in D2 is larger than magnitude of  |
//|                        D2 itself; it is impossible to reliably   |
//|                        distinguish D2 from zero".                |
//+------------------------------------------------------------------+
void CQQPSolver::QuadraticModel(CQQPBuffers &sstate,CRowDouble &x,
                                CRowDouble &d,CRowDouble &g,
                                double &d1,int &d1est,double &d2,
                                int &d2est,CRowDouble &m_tmp0)
  {
//--- create variables
   int    n=sstate.m_n;
   double v=0;
   double mx=(x.Abs()+0).Max();
   double md=(d.Abs()+0).Max();
   double mb=(sstate.m_b.Abs()+0).Max();

   d1=0;
   d1est=0;
   d2=0;
   d2est=0;
//--- Maximums
//--- D2
   if(sstate.m_akind==0)
     {
      //--- Dense matrix A
      d2=0.5*CAblas::RMatrixSyvMVect(n,sstate.m_densea,0,0,true,d,0,m_tmp0);
     }
   else
     {
      //--- Sparse matrix A
      //--- check
      if(!CAp::Assert(sstate.m_akind==1,__FUNCTION__+": unexpected AKind in TargetGradient"))
         return;
      d2=0.5*CSparse::SparseVSMV(sstate.m_sparsea,sstate.m_sparseupper,d);
     }
   v=d.Dot(g);
   d1=v;
//--- Error estimates
   COptServ::EstimateParabolicModel(sstate.m_absasum,sstate.m_absasum2,mx,mb,md,d1,d2,d1est,d2est);
  }
//+------------------------------------------------------------------+
//| This function accepts quadratic model of the form                |
//| f(x) = 0.5*x'*A*x+b'*x+penaltyfactor*0.5*(C*x-b)'*(C*x-b)        |
//| and list of possible steps along direction D. It chooses best    |
//| step (one which achieves minimum value of the target function)   |
//| and moves current point (given by SAS object) to the new location|
//| Step is bounded subject to boundary constraints.                 |
//| Candidate steps are divided into two groups:                     |
//|  *"default" step, which is always performed when no candidate  |
//|     steps LONGER THAN THE DEFAULT ONE is given. This candidate   |
//|     MUST reduce target function value; it is responsibility of   |
//|     caller to provide default candidate which reduces target     |
//|     function.                                                    |
//|  *"additional candidates", which may be shorter or longer than |
//|     the default step. Candidates which are shorter that the      |
//|     default step are ignored; candidates which are longer than   |
//|     the "default" step are tested.                               |
//| The idea is that we ALWAYS try "default" step, and it is         |
//| responsibility of the caller to provide us with something which  |
//| is worth trying. This step may activate some constraint - that's |
//| why we stopped at "default" step size. However, we may also try  |
//| longer steps which may activate  additional constraints and      |
//| further reduce function value.                                   |
//| INPUT PARAMETERS:                                                |
//|   SState   -  structure which stores model                       |
//|   SAS      -  active set structure which stores current point in |
//|               SAS.XC                                             |
//|   D        -  direction for step                                 |
//|   Stp     - step length for "default" candidate                |
//|   NeedAct  -  whether default candidate activates some constraint|
//|               if NeedAct is True, constraint given by CIdc/CVal  |
//|               is GUARANTEED to be activated in the final point.  |
//|   CIdx     -  if NeedAct is True, stores index of the constraint |
//|               to activate                                        |
//|   CVal     -  if NeedAct is True, stores constrained value;      |
//|               SAS.XC[CIdx] is forced to be equal to CVal.        |
//|   AddSteps -  array[AddStepsCnt] of additional steps:            |
//|               * AddSteps[] <= Stp are ignored                    |
//|               * AddSteps[] > Stp are tried                       |
//|   Activated - possibly preallocated buffer; previously allocated |
//|               memory will be reused.                             |
//|   Tmp0 / 1 -  possibly preallocated buffers; previously allocated|
//|               memory will be reused.                             |
//| OUTPUT PARAMETERS:                                               |
//|   SAS      -  SAS.XC is set to new point; if there was a         |
//|               constraint specified by NeedAct/CIdx/CVal, it will |
//|               be activated (other constraints may be activated   |
//|               too, but this one is guaranteed to be active in the|
//|               final point).                                      |
//|   Activated - elements of this array are set to True, if I-Th    |
//|               constraint as inactive at previous point, but      |
//|               become active in the new one.                      |
//| Situations when we deactivate xi >= 0 and activate xi <= 1 are   |
//| considered as activation of previously inactive constraint       |
//+------------------------------------------------------------------+
void CQQPSolver::FindBestStepAndMove(CQQPBuffers &sstate,
                                     CSActiveSet &sas,
                                     CRowDouble &d,
                                     double stp,
                                     bool needact,
                                     int cidx,
                                     double cval,
                                     CRowDouble &addsteps,
                                     int addstepscnt,
                                     bool &activated[],
                                     CRowDouble &m_tmp0,
                                     CRowDouble &m_tmp1)
  {
//--- create variables
   int    n=sstate.m_n;
   int    i=0;
   int    k=0;
   double v=0;
   double stpbest=0;
   double fbest=0;
   double fcand=0;

   m_tmp0.Resize(n);
   CApServ::BVectorSetLengthAtLeast(activated,n);
//--- Calculate initial step, store to Tmp0
//--- NOTE: Tmp0 is guaranteed to be feasible w.m_r.m_t. boundary constraints
   for(i=0; i<n; i++)
     {
      v=sas.m_xc[i]+stp*d[i];
      if(sstate.m_havebndl[i] && v<sstate.m_bndl[i])
         v=sstate.m_bndl[i];
      if(sstate.m_havebndu[i] && v>sstate.m_bndu[i])
         v=sstate.m_bndu[i];
      m_tmp0.Set(i,v);
     }
   if(needact)
      m_tmp0.Set(cidx,cval);
//--- Try additional steps, if AddStepsCnt>0
   if(addstepscnt>0)
     {
      //--- Find best step
      stpbest=stp;
      fbest=ProjectedTargetFunction(sstate,sas.m_xc,d,stpbest,m_tmp0,m_tmp1);
      for(k=0; k<=addstepscnt-1; k++)
        {
         if(addsteps[k]>stp)
           {
            fcand=ProjectedTargetFunction(sstate,sas.m_xc,d,addsteps[k],m_tmp0,m_tmp1);
            if(fcand<fbest)
              {
               fbest=fcand;
               stpbest=addsteps[k];
              }
           }
        }
      //--- Prepare best step
      //--- NOTE: because only AddSteps[]>Stp were checked,
      //---       this step will activate constraint CIdx.
      for(i=0; i<n; i++)
        {
         v=sas.m_xc[i]+stpbest*d[i];
         if(sstate.m_havebndl[i] && v<sstate.m_bndl[i])
            v=sstate.m_bndl[i];
         if(sstate.m_havebndu[i] && v>sstate.m_bndu[i])
            v=sstate.m_bndu[i];
         m_tmp0.Set(i,v);
        }
      if(needact)
         m_tmp0.Set(cidx,cval);
     }
//--- Fill Activated array by information about activated constraints.
//--- Perform step
   for(i=0; i<n; i++)
     {
      activated[i]=false;
      v=m_tmp0[i];
      if(v==sas.m_xc[i])
         continue;
      if(sstate.m_havebndl[i] && v==sstate.m_bndl[i])
         activated[i]=true;
      if(sstate.m_havebndu[i] && v==sstate.m_bndu[i])
         activated[i]=true;
     }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
   CSActiveSets::SASMoveTo(sas,m_tmp0,needact,cidx,cval);
  }
//+------------------------------------------------------------------+
//| This function prepares data for constrained Newton step for      |
//| penalized quadratic model of the form                            |
//|   f(x) = 0.5*x'*A*x+b'*x+penaltyfactor*0.5*(C*x-b)'*(C*x-b)      |
//| where A can be dense or sparse, and model is considered subject  |
//| to equality constraints specified by SState.SAS.XC object.       |
//| Constraint is considered active if XC[i] is exactly BndL[i] or   |
//| BndU[i], i.e. we ignore internal list of constraints monitored   |
//| by SAS object. Our own set of constraints includes all           |
//| constraints stored by SAS, but also may include some constraints |
//| which are inactive in SAS.                                       |
//| "Preparation" means that Cholesky decomposition of the effective |
//| system matrix is performed, and we can perform constrained Newton|
//| step.                                                            |
//| This function works as black box. It uses fields of SState which |
//| are marked as "Variables for constrained Newton phase", and only |
//| this function and its friends know about these variables.        |
//| Everyone else should use:                                        |
//|   * CNewtonBuild()  to prepare initial Cholesky decomposition for|
//|                     step                                         |
//|   * CNewtonStep()   to perform constrained Newton step           |
//|   * CNewtonUpdate() to update Cholesky matrix after point was    |
//|                     moved and constraints were updated. In some  |
//|                     cases it is possible to efficiently real -   |
//|                     calculate Cholesky decomposition if you know |
//|                     which constraints were activated. If         |
//|                     efficient real - calculation is impossible,  |
//|                     this function returns False.                 |
//| INPUT PARAMETERS:                                                |
//|   SState      -  structure which stores model and temporaries    |
//|                  for CN phase; in particular, SAS.XC stores      |
//|                  current point.                                  |
//|   SparseSolver - which sparse m_solver to use for sparse model;    |
//|                  ignored for dense QP. Can be:                   |
//|                  * 2 -   SKS - based Cholesky                    |
//|   NCholesky   -  counter which is incremented after Cholesky     |
//|                  (successful or failed one)                      |
//| OUTPUT PARAMETERS:                                               |
//|   NCholesky   -  possibly updated counter                        |
//| RESULT:                                                          |
//|   True, if Cholesky decomposition was successfully performed.    |
//|   False, if a) matrix was semi - definite or indefinite, or      |
//|             b) particular combination of matrix type(sparse)     |
//|                and constraints (general linear) is not supported.|
//| NOTE: this function may routinely return False, for indefinite   |
//|       matrices or for sparse problems with general linear        |
//|       constraints. You should be able to handle such situations. |
//+------------------------------------------------------------------+
bool CQQPSolver::CNewtonBuild(CQQPBuffers &sstate,int sparsesolver,
                              int &ncholesky)
  {
//--- create variables
   int    n=sstate.m_n;
   int    i=0;
   int    j=0;
   int    k=0;
   double v=0;
   bool   b;
   int    ridx0=0;
   int    ridx1=0;
   int    nfree=0;
   int    i_=0;
//--- 1. Set CNModelAge to zero
//--- 2. Generate YIdx - reordering of variables such that free variables
//---    come first and are ordered by ascending, fixed are last ones and
//---    have no particular ordering.
//--- This step is same for dense and sparse problems.
   sstate.m_cnmodelage=0;
   sstate.m_yidx.Resize(n);
   ridx0=0;
   ridx1=n-1;
   sstate.m_yidx.Fill(-1,0,n);
   for(i=0; i<n; i++)
     {
      //--- check
      if(!CAp::Assert(!sstate.m_havebndl[i] || sstate.m_sas.m_xc[i]>=sstate.m_bndl[i],__FUNCTION__+": internal error"))
         return(false);
      if(!CAp::Assert(!sstate.m_havebndu[i] || sstate.m_sas.m_xc[i]<=sstate.m_bndu[i],__FUNCTION__+": internal error"))
         return(false);
      b=false;
      b=b || (sstate.m_havebndl[i] && sstate.m_sas.m_xc[i]==sstate.m_bndl[i]);
      b=b || (sstate.m_havebndu[i] && sstate.m_sas.m_xc[i]==sstate.m_bndu[i]);
      if(b)
        {
         sstate.m_yidx.Set(ridx1,i);
         ridx1--;
        }
      else
        {
         sstate.m_yidx.Set(ridx0,i);
         ridx0++;
        }
     }
//--- check
   if(!CAp::Assert(ridx0==ridx1+1,__FUNCTION__+": internal error"))
      return(false);
   nfree=ridx0;
   sstate.m_nfree=nfree;
   if(nfree==0)
      return(false);
//--- Constrained Newton matrix: dense version
   if(sstate.m_akind==0)
     {
      sstate.m_densez=sstate.m_densea;
      sstate.m_densez.Resize(n,n);
      sstate.m_tmpcn.Resize(n);
      for(i=1; i<nfree; i++)
        {
         //--- check
         if(!CAp::Assert(sstate.m_yidx[i]>sstate.m_yidx[i-1],__FUNCTION__+": integrity check failed"))
            return(false);
        }
      for(i=0; i<nfree; i++)
        {
         k=sstate.m_yidx[i];
         for(j=i; j<nfree; j++)
            sstate.m_densez.Set(i,j,sstate.m_densez.Get(k,sstate.m_yidx[j]));
        }
      sstate.m_regdiag.Resize(n);
      for(i=0; i<nfree; i++)
        {
         v=0.0;
         for(j=0; j<i; j++)
            v+= MathAbs(sstate.m_densez.Get(j,i));
         for(j=i; j<nfree; j++)
            v+= MathAbs(sstate.m_densez.Get(i,j));
         if(v==0.0)
            v=1.0;
         sstate.m_regdiag.Set(i,m_regz*v);
        }
      for(i=0; i<nfree; i++)
         sstate.m_densez.Add(i,i,sstate.m_regdiag[i]);
      ncholesky++;
      if(!CTrFac::SPDMatrixCholeskyRec(sstate.m_densez,0,nfree,true,sstate.m_tmpcn))
         return(false);
      for(i=nfree-1; i>=0; i--)
        {
         for(i_=i; i_<nfree; i_++)
            sstate.m_tmpcn.Set(i_,sstate.m_densez.Get(i,i_));
         k=sstate.m_yidx[i];
         for(j=k; j<n; j++)
            sstate.m_densez.Set(k,j,0);
         for(j=i; j<nfree; j++)
            sstate.m_densez.Set(k,sstate.m_yidx[j],sstate.m_tmpcn[j]);
        }
      for(i=nfree; i<n; i++)
        {
         k=sstate.m_yidx[i];
         sstate.m_densez.Set(k,k,1.0);
         for(j=k+1; j<n; j++)
            sstate.m_densez.Set(k,j,0);
        }
      //--- return result
      return(true);
     }
//--- Constrained Newton matrix: sparse version
   if(sstate.m_akind==1)
     {
      //--- check
      if(!CAp::Assert(sparsesolver==2,__FUNCTION__+": internal error"))
         return(false);
      //--- Copy sparse A to Z and fill rows/columns corresponding to active
      //--- constraints by zeros. Diagonal elements corresponding to active
      //--- constraints are filled by unit values.
      CSparse::SparseCopyToSKSBuf(sstate.m_sparsea,sstate.m_sparsecca);
      sstate.m_tmpcn=vector<double>::Zeros(n);
      for(i=nfree; i<n; i++)
         sstate.m_tmpcn.Set(sstate.m_yidx[i],1);
      for(i=0; i<n; i++)
        {
         k=sstate.m_sparsecca.m_RIdx[i];
         for(j=i-sstate.m_sparsecca.m_DIdx[i]; j<=i; j++)
           {
            if(sstate.m_tmpcn[i]!=0.0 || sstate.m_tmpcn[j]!=0.0)
              {
               //--- I-th or J-th variable is in active set (constrained)
               if(i==j)
                  sstate.m_sparsecca.m_Vals.Set(k,1.0);
               else
                  sstate.m_sparsecca.m_Vals.Set(k,0.0);
              }
            k++;
           }
        }
      //--- Perform sparse Cholesky
      ncholesky++;
      if(!CTrFac::SparseCholeskySkyLine(sstate.m_sparsecca,n,sstate.m_sparseupper))
         return(false);
      //--- return result
      return(true);
     }
//--- Unexpected :)
   CAp::Assert(false,__FUNCTION__+": internal error");
   return(false);
  }
//+------------------------------------------------------------------+
//| This function updates equality-constrained Cholesky matrix after |
//| activation of the new equality constraints. Matrix being updated |
//| is quadratic term of the function below                          |
//|   f(x) = 0.5*x'*A*x+b'*x+penaltyfactor*0.5*(C*x-b)'*(C*x-b)      |
//| where A can be dense or CSparse::                                |
//| This function uses YIdx[] array(set by CNewtonBuild() function)  |
//| to distinguish between active and inactive constraints.          |
//| This function works as black box. It uses fields of SState which |
//| are marked as "Variables for constrained Newton phase", and only |
//| this function and its friends know about these variables.        |
//| Everyone else should use:                                        |
//|   * CNewtonBuild()  to prepare initial Cholesky decomposition for|
//|                     step                                         |
//|   * CNewtonStep()   to perform constrained Newton step           |
//|   * CNewtonUpdate() to update Cholesky matrix after point was    |
//|                     moved and constraints were updated. In some  |
//|                     cases it is possible to efficiently real -   |
//|                     calculate Cholesky decomposition if you know |
//|                     which constraints were activated. If         |
//|                     efficient real - calculation is impossible,  |
//|                     this function returns False.                 |
//| INPUT PARAMETERS:                                                |
//|   SState      -  structure which stores model and temporaries for|
//|                  CN phase; in particular, SAS.XC stores current  |
//|                  point.                                          |
//|   Settings    -  QQPSettings object which was initialized by     |
//|                  appropriate construction function.              |
//|   NCUpdates   -  counter which is incremented after each update  |
//|                  (one update means one variable being fixed)     |
//| OUTPUT PARAMETERS:                                               |
//|   NCUpdates   -  possibly updated counter                        |
//| RESULT:                                                          |
//|   True, if Cholesky decomposition was successfully performed.    |
//|   False, if a) model age was too high, or b) particular          |
//|                combination of matrix type(sparse) and constraints|
//|                (general linear) is not supported                 |
//| NOTE: this function may routinely return False. You should be    |
//|       able to handle such situations.                            |
//+------------------------------------------------------------------+
bool CQQPSolver::CNewtonUpdate(CQQPBuffers &sstate,CQQPSettings &Settings,
                               int &ncupdates)
  {
//--- return result
   int  n=sstate.m_n;
   int  nfree=sstate.m_nfree;
   int  ntofix=0;
   bool b;
   int  ridx0=0;
   int  ridx1=0;
   int  i=0;
   int  k=0;
//--- Cholesky updates for sparse problems are not supported
   if(sstate.m_akind==1)
      return(false);
//--- Determine variables to fix and move them to YIdx[NFree-NToFix:NFree-1]
//--- Exit if CNModelAge increased too much.
   sstate.m_tmpcni.Resize(n);
   ridx0=0;
   ridx1=nfree-1;
   for(i=0; i<nfree; i++)
      sstate.m_tmpcni.Set(i,-1);
   for(k=0; k<nfree; k++)
     {
      i=sstate.m_yidx[k];
      //--- check
      if(!CAp::Assert(!sstate.m_havebndl[i] || sstate.m_sas.m_xc[i]>=sstate.m_bndl[i],__FUNCTION__+": internal error"))
         return(false);
      if(!CAp::Assert(!sstate.m_havebndu[i] || sstate.m_sas.m_xc[i]<=sstate.m_bndu[i],__FUNCTION__+": internal error"))
         return(false);
      b=false;
      b=b || (sstate.m_havebndl[i] && sstate.m_sas.m_xc[i]==sstate.m_bndl[i]);
      b=b || (sstate.m_havebndu[i] && sstate.m_sas.m_xc[i]==sstate.m_bndu[i]);
      if(b)
        {
         sstate.m_tmpcni.Set(ridx1,i);
         ridx1--;
        }
      else
        {
         sstate.m_tmpcni.Set(ridx0,i);
         ridx0++;
        }
     }
//--- check
   if(!CAp::Assert(ridx0==ridx1+1,__FUNCTION__+": internal error"))
      return(false);
   ntofix=nfree-ridx0;
   if(ntofix==0 || ntofix==nfree)
      return(false);
   if(sstate.m_cnmodelage+ntofix>Settings.m_cnmaxupdates)
      return(false);
   for(i=0; i<nfree; i++)
     {
      sstate.m_yidx.Set(i,sstate.m_tmpcni[i]);
     }
//--- Constrained Newton matrix: dense version.
   if(sstate.m_akind==0)
     {
      //--- Update Cholesky matrix with SPDMatrixCholeskyUpdateFixBuf()
      CApServ::BVectorSetLengthAtLeast(sstate.m_tmpcnb,n);
      ArrayInitialize(sstate.m_tmpcnb,false);
      for(i=nfree-ntofix; i<nfree; i++)
         sstate.m_tmpcnb[sstate.m_yidx[i]]=true;
      CTrFac::SPDMatrixCholeskyUpdateFixBuf(sstate.m_densez,n,true,sstate.m_tmpcnb,sstate.m_tmpcn);
      //--- Update information stored in State and exit
      sstate.m_nfree=nfree-ntofix;
      sstate.m_cnmodelage=sstate.m_cnmodelage+ntofix;
      ncupdates=ncupdates+ntofix;
      //--- return result
      return(true);
     }
//--- Unexpected :)
   CAp::Assert(false,__FUNCTION__+": internal error");
   return(false);
  }
//+------------------------------------------------------------------+
//| This function prepares equality - constrained Newton step using  |
//| previously calculated constrained Cholesky matrix of the problem |
//|   f(x) = 0.5*x'*A*x+b'*x+penaltyfactor*0.5*(C*x-b)'*(C*x-b)      |
//| where A can be dense or CSparse::                                |
//| As input, this function accepts gradient at the current location.|
//| As output, it returns step vector (replaces gradient).           |
//| This function works as black box. It uses fields of SState which |
//| are marked as "Variables for constrained Newton phase", and only |
//| this function and its friends know about these variables.        |
//| Everyone else should use:                                        |
//|   * CNewtonBuild()  to prepare initial Cholesky decomposition for|
//|                     step                                         |
//|   * CNewtonStep()   to perform constrained Newton step           |
//|   * CNewtonUpdate() to update Cholesky matrix after point was    |
//|                     moved and constraints were updated. In some  |
//|                     cases it is possible to efficiently real -   |
//|                     calculate Cholesky decomposition if you know |
//|                     which constraints were activated. If         |
//|                     efficient real - calculation is impossible,  |
//|                     this function returns False.                 |
//| INPUT PARAMETERS:                                                |
//|   SState      -  structure which stores model and temporaries for|
//|                  CN phase; in particular, SAS.XC stores current  |
//|                  point.                                          |
//|   Settings    -  QQPSettings object which was initialized by     |
//|                  appropriate construction function.              |
//|   GC          -  array[N], gradient of the target function       |
//| OUTPUT PARAMETERS:                                               |
//|   GC          -  array[N], step vector(on success)               |
//| RESULT:                                                          |
//|   True,    if step was successfully calculated.                  |
//|   False,   if step calculation failed:                           |
//|               a) gradient was exactly zero,                      |
//|               b) gradient norm was smaller than EpsG (stopping   |
//|                  condition)                                      |
//|               c) all variables were equality - constrained       |
//| NOTE: this function may routinely return False. You should be    |
//|       able to handle such situations.                            |
//+------------------------------------------------------------------+
bool CQQPSolver::CNewtonStep(CQQPBuffers &sstate,CQQPSettings &Settings,
                             CRowDouble &gc)
  {
//--- create variables
   int    n=sstate.m_n;
   int    nfree=sstate.m_nfree;
   double v=0;

   for(int i=nfree; i<n; i++)
      gc.Set(sstate.m_yidx[i],0.0);
   v=gc.Dot(gc);
   if(MathSqrt(v)<=Settings.m_epsg)
      return(false);

   gc*=(-1.0);
   if(sstate.m_akind==0)
     {
      //--- Dense Newton step.
      //--- Use straightforward Cholesky m_solver.
      CFbls::FblsCholeskySolve(sstate.m_densez,1.0,n,true,gc,sstate.m_tmpcn);
      return(true);
     }
   if(sstate.m_akind==1)
     {
      //--- Sparse Newton step.
      //--- We have T*T' = L*L' = U'*U (depending on specific triangle stored in SparseCCA).
      if(sstate.m_sparseupper)
        {
         CSparse::SparseTRSV(sstate.m_sparsecca,sstate.m_sparseupper,false,1,gc);
         CSparse::SparseTRSV(sstate.m_sparsecca,sstate.m_sparseupper,false,0,gc);
        }
      else
        {
         CSparse::SparseTRSV(sstate.m_sparsecca,sstate.m_sparseupper,false,0,gc);
         CSparse::SparseTRSV(sstate.m_sparsecca,sstate.m_sparseupper,false,1,gc);
        }
      return(true);
     }
//--- return result
   CAp::Assert(false,__FUNCTION__+": internal error");
   return(false);
  }
//+------------------------------------------------------------------+
//| This object stores Settings for DENSE - AUL m_solver.              |
//| It must be initialized with QPDENSEAULLoadDefaults().            |
//| After initialization you may change Settings.                    |
//+------------------------------------------------------------------+
struct CQPDenseAULSettings
  {
   int               m_outerits;
   double            m_epsx;
   double            m_rho;
   //--- constructor / destructor
                     CQPDenseAULSettings(void) { ZeroMemory(this); }
                    ~CQPDenseAULSettings(void) {}
   //---
   void              Copy(const CQPDenseAULSettings &obj);
   //--- overloading
   void              operator=(const CQPDenseAULSettings &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CQPDenseAULSettings::Copy(const CQPDenseAULSettings &obj)
  {
   m_outerits=obj.m_outerits;
   m_epsx=obj.m_epsx;
   m_rho=obj.m_rho;
  }
//+------------------------------------------------------------------+
//| This object stores temporaries used by Dense - AUL m_solver.       |
//+------------------------------------------------------------------+
struct CQPDenseAULBuffers
  {
   int               m_repinneriterationscount;
   int               m_repncholesky;
   int               m_repnmv;
   int               m_repnwrk0;
   int               m_repnwrk1;
   int               m_repnwrkchanges;
   int               m_repnwrkf;
   int               m_repouteriterationscount;
   bool              m_sclsfthasbndl[];
   bool              m_sclsfthasbndu[];
   CSparseMatrix     m_dummysparse;
   CRowInt           m_cidx;
   CRowInt           m_nicnact;
   CRowDouble        m_cscales;
   CRowDouble        m_d;
   CRowDouble        m_deltax;
   CRowDouble        m_exb;
   CRowDouble        m_exbndl;
   CRowDouble        m_exbndu;
   CRowDouble        m_exscale;
   CRowDouble        m_exxc;
   CRowDouble        m_exxorigin;
   CRowDouble        m_modelg;
   CRowDouble        m_nicerr;
   CRowDouble        m_nulc;
   CRowDouble        m_nulcest;
   CRowDouble        m_qrrightpart;
   CRowDouble        m_qrsv0;
   CRowDouble        m_qrsvx1;
   CRowDouble        m_qrtau;
   CRowDouble        m_sclsftb;
   CRowDouble        m_sclsftbndl;
   CRowDouble        m_sclsftbndu;
   CRowDouble        m_sclsftxc;
   CRowDouble        m_tmp0;
   CRowDouble        m_tmpg;
   CQQPSettings      m_qqpsettingsuser;
   CQQPBuffers       m_qqpbuf;
   CMatrixDouble     m_exa;
   CMatrixDouble     m_qrkkt;
   CMatrixDouble     m_sclsfta;
   CMatrixDouble     m_sclsftcleic;
   CMatrixDouble     m_tmp2;
   CConvexQuadraticModel m_dummycqm;
   //--- constructor / destructor
                     CQPDenseAULBuffers(void);
                    ~CQPDenseAULBuffers(void) {}
   //---
   void              Copy(const CQPDenseAULBuffers &obj);
   //--- overloading
   void              operator=(const CQPDenseAULBuffers &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CQPDenseAULBuffers::CQPDenseAULBuffers(void)
  {
   m_repinneriterationscount=0;
   m_repncholesky=0;
   m_repnmv=0;
   m_repnwrk0=0;
   m_repnwrk1=0;
   m_repnwrkchanges=0;
   m_repnwrkf=0;
   m_repouteriterationscount=0;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CQPDenseAULBuffers::Copy(const CQPDenseAULBuffers &obj)
  {
   m_repinneriterationscount=obj.m_repinneriterationscount;
   m_repncholesky=obj.m_repncholesky;
   m_repnmv=obj.m_repnmv;
   m_repnwrk0=obj.m_repnwrk0;
   m_repnwrk1=obj.m_repnwrk1;
   m_repnwrkchanges=obj.m_repnwrkchanges;
   m_repnwrkf=obj.m_repnwrkf;
   m_repouteriterationscount=obj.m_repouteriterationscount;
   ArrayCopy(m_sclsfthasbndl,obj.m_sclsfthasbndl);
   ArrayCopy(m_sclsfthasbndu,obj.m_sclsfthasbndu);
   m_dummysparse=obj.m_dummysparse;
   m_cidx=obj.m_cidx;
   m_nicnact=obj.m_nicnact;
   m_cscales=obj.m_cscales;
   m_d=obj.m_d;
   m_deltax=obj.m_deltax;
   m_exb=obj.m_exb;
   m_exbndl=obj.m_exbndl;
   m_exbndu=obj.m_exbndu;
   m_exscale=obj.m_exscale;
   m_exxc=obj.m_exxc;
   m_exxorigin=obj.m_exxorigin;
   m_modelg=obj.m_modelg;
   m_nicerr=obj.m_nicerr;
   m_nulc=obj.m_nulc;
   m_nulcest=obj.m_nulcest;
   m_qrrightpart=obj.m_qrrightpart;
   m_qrsv0=obj.m_qrsv0;
   m_qrsvx1=obj.m_qrsvx1;
   m_qrtau=obj.m_qrtau;
   m_sclsftb=obj.m_sclsftb;
   m_sclsftbndl=obj.m_sclsftbndl;
   m_sclsftbndu=obj.m_sclsftbndu;
   m_sclsftxc=obj.m_sclsftxc;
   m_tmp0=obj.m_tmp0;
   m_tmpg=obj.m_tmpg;
   m_qqpsettingsuser=obj.m_qqpsettingsuser;
   m_qqpbuf=obj.m_qqpbuf;
   m_exa=obj.m_exa;
   m_qrkkt=obj.m_qrkkt;
   m_sclsfta=obj.m_sclsfta;
   m_sclsftcleic=obj.m_sclsftcleic;
   m_tmp2=obj.m_tmp2;
   m_dummycqm=obj.m_dummycqm;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CQPDenseAULSolver
  {
public:
   //--- constants
   static const double m_evictionlevel;
   static const double m_expansionratio;

   static void       QPDenseAULLoadDefaults(int nmain,CQPDenseAULSettings &s);
   static void       QPDenseAULOptimize(CConvexQuadraticModel &a,CSparseMatrix &sparsea,int akind,bool sparseaupper,CRowDouble &b,CRowDouble &bndl,CRowDouble &bndu,CRowDouble &s,CRowDouble &xorigin,int nn,CMatrixDouble &cleic,int dnec,int dnic,CSparseMatrix &scleic,int snec,int snic,bool renormlc,CQPDenseAULSettings &Settings,CQPDenseAULBuffers &State,CRowDouble &xs,CRowDouble &lagbc,CRowDouble &laglc,int &terminationtype);

private:
   static void       GenerateExModel(CMatrixDouble &sclsfta,CRowDouble &sclsftb,int nmain,CRowDouble &sclsftbndl,bool &sclsfthasbndl[],CRowDouble &sclsftbndu,bool &sclsfthasbndu[],CMatrixDouble &sclsftcleic,int sclsftnec,int sclsftnic,CRowDouble &nulc,double rho,CMatrixDouble &exa,CRowDouble &exb,CRowDouble &exbndl,CRowDouble &exbndu,CMatrixDouble &tmp2);
   static void       GenerateExInitialPoint(CRowDouble &sclsftxc,int nmain,int nslack,CRowDouble &exxc);
   static void       UpdateLagrangeMultipliers(CMatrixDouble &sclsfta,CRowDouble &sclsftb,int nmain,CRowDouble &sclsftbndl,bool &sclsfthasbndl[],CRowDouble &sclsftbndu,bool &sclsfthasbndu[],CMatrixDouble &sclsftcleic,int sclsftnec,int sclsftnic,CRowDouble &exxc,CRowDouble &nulcest,CQPDenseAULBuffers &buffers);
   static void       ScaleShiftOriginalPproblem(CConvexQuadraticModel &a,CSparseMatrix &sparsea,int akind,bool sparseaupper,CRowDouble &b,CRowDouble &bndl,CRowDouble &bndu,CRowDouble &s,CRowDouble &xorigin,int nmain,CMatrixDouble &cleic,int dnec,int dnic,CSparseMatrix &scleic,int snec,int snic,bool renormlc,CQPDenseAULBuffers &State,CRowDouble &xs);

   static double     NormalizeQuadraticTerm(CMatrixDouble &a,CRowDouble &b,int n,CMatrixDouble &cleic,int nec,int nic,bool usecleic,CMatrixDouble &tmp2);
   static void       SelectInitialWorkingSet(CMatrixDouble &a,int nmain,CMatrixDouble &cleic,int nec,int nic,CRowDouble &m_tmp0,CMatrixDouble &tmp2,int &nicwork,bool &allowwseviction);
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
const double CQPDenseAULSolver::m_evictionlevel=-0.01;
const double CQPDenseAULSolver::m_expansionratio=0.20;
//+------------------------------------------------------------------+
//| This function initializes QPDENSEAULSettings structure with      |
//| default Settings.                                                |
//| Newly created structure MUST be initialized by default Settings -|
//| or by copy of the already initialized structure.                 |
//+------------------------------------------------------------------+
void CQPDenseAULSolver::QPDenseAULLoadDefaults(int nmain,
                                               CQPDenseAULSettings &s)
  {
   s.m_epsx=1.0E-6;
   s.m_outerits=5;
   s.m_rho=100.0;
  }
//+------------------------------------------------------------------+
//| This function runs Dense-AUL m_solver; it returns after            |
//| optimization process was completed. Following QP problem is      |
//| solved:                                                          |
//|   min(0.5 * (x - x_origin)'*A*(x-x_origin)+b' * (x - x_origin))  |
//| subject to combination of box and general linear dense / sparse  |
//| constraints.                                                     |
//| INPUT PARAMETERS:                                                |
//|   DenseA      -  for dense problems(AKind = 0), A-term of CQM    |
//|                  object contains system matrix. Other terms are  |
//|                  unspecified and should not be referenced.       |
//|   SparseA     -  for sparse problems(AKind = 1), CRS format      |
//|   AKind       -  sparse matrix format:                           |
//|                  * 0 for dense matrix                            |
//|                  * 1 for sparse matrix                           |
//|   SparseUpper -  which triangle of SparseAC stores matrix - upper|
//|                  or lower one (for dense matrices this parameter |
//|                  is not actual).                                 |
//|   B           -  linear term, array[N]                           |
//|   BndL        -  lower bound, array[N]                           |
//|   BndU        -  upper bound, array[N]                           |
//|   S           -  scale vector, array[NC]:                        |
//|                  * I-Th element contains scale of I-Th variable, |
//|                  * SC[I] > 0                                     |
//|   XOrigin     -  origin term, array[NC]. Can be zero.            |
//|   N           -  number of variables in the  original formulation|
//|                  (no slack variables).                           |
//|   CLEIC       -  dense linear equality / inequality constraints. |
//|                  Equality constraints come first.                |
//|   NEC, NIC    -  number of dense equality/inequality constraints.|
//|   SCLEIC      -  sparse linear equality / inequality constraints.|
//|                  Equality constraints come first.                |
//|   SNEC, SNIC  -  number of sparse equality/inequality constraints|
//|   RenormLC    -  whether constraints should be renormalized      |
//|                  (recommended) or used "as is".                  |
//|   Settings    -  QPDENSEAULSettings object initialized by one of |
//|                  the initialization functions.                   |
//|   State       -  object which stores temporaries                 |
//|   XS          -  initial point, array[NC]                        |
//| OUTPUT PARAMETERS:                                               |
//|   XS          -  last point                                      |
//|   TerminationType - termination type:                            |
//|                   *                                              |
//|                   *                                              |
//|                   *                                              |
//+------------------------------------------------------------------+
void CQPDenseAULSolver::QPDenseAULOptimize(CConvexQuadraticModel &a,
                                           CSparseMatrix &sparsea,
                                           int akind,bool sparseaupper,
                                           CRowDouble &b,
                                           CRowDouble &bndl,
                                           CRowDouble &bndu,
                                           CRowDouble &s,
                                           CRowDouble &xorigin,
                                           int nn,CMatrixDouble &cleic,
                                           int dnec,int dnic,
                                           CSparseMatrix &scleic,
                                           int snec,int snic,bool renormlc,
                                           CQPDenseAULSettings &Settings,
                                           CQPDenseAULBuffers &State,
                                           CRowDouble &xs,CRowDouble &lagbc,
                                           CRowDouble &laglc,
                                           int &terminationtype)
  {
//--- create variables
   int    i=0;
   int    j=0;
   int    k=0;
   double v=0;
   double vv=0;
   double rho=Settings.m_rho;
   double epsx=Settings.m_epsx;
   int    outeridx=0;
   int    nmain=nn;
   int    nslack=dnic+snic;
   int    ntotal=nmain+nslack;
   int    nectotal=dnec+snec;
   int    nictotal=dnic+snic;
   int    ktotal=dnec+dnic+snec+snic;
   double maxrho=1.0E12;
   double feaserr=0;
   double feaserrprev=0;
   double requestedfeasdecrease=0.33;
   int    goodcounter=0;
   int    stagnationcounter=0;
   int    nicwork=0;
   int    kwork=0;
   int    nwork=0;
   bool   allowwseviction;
   bool   workingsetextended;
   double targetscale=0;
   int    i_=0;

   terminationtype=0;
   if(epsx<=0.0)
      epsx=1.0E-9;
//--- Integrity checks
   if(snec+snic>0)
     {
      if(!CAp::Assert(scleic.m_MatrixType==1,__FUNCTION__+": unexpected sparse matrix format"))
         return;
      if(!CAp::Assert(scleic.m_M==snec+snic,__FUNCTION__+": unexpected sparse matrix size"))
         return;
      if(!CAp::Assert(scleic.m_N==nmain+1,__FUNCTION__+": unexpected sparse matrix size"))
         return;
     }
//--- Prepare
   State.m_repinneriterationscount=0;
   State.m_repouteriterationscount=0;
   State.m_repncholesky=0;
   State.m_repnmv=0;
   State.m_repnwrkchanges=0;
   State.m_repnwrk0=0;
   State.m_repnwrk1=0;
   State.m_repnwrkf=0;
   terminationtype=0;
   State.m_cidx.Resize(ktotal);
   State.m_nulc.Resize(ktotal);
   State.m_nulcest.Resize(ktotal);
   State.m_exb.Resize(ntotal);
   State.m_exxc.Resize(ntotal);
   State.m_exxorigin.Resize(ntotal);
   State.m_exbndl.Resize(ntotal);
   State.m_exbndu.Resize(ntotal);
   State.m_exscale.Resize(ntotal);
   State.m_tmp0.Resize(ntotal);
   State.m_nicerr.Resize(nictotal);
   State.m_nicnact.Resize(nictotal);
//--- Allocate Lagrange multipliers, fill by default values (zeros)
   lagbc=vector<double>::Zeros(nmain);
   laglc=vector<double>::Zeros(ktotal);
//--- Prepare scaled/shifted model in dense format - input parameters
//--- are converted and stored in State.SclSftA/B/HasBndL/HasBndU/BndL/BndU/CLEIC/XC/CScales
   ScaleShiftOriginalPproblem(a,sparsea,akind,sparseaupper,b,bndl,bndu,s,xorigin,nmain,cleic,dnec,dnic,scleic,snec,snic,renormlc,State,xs);
//--- Normalize model in such way that norm(A)~1 (very roughly)
//--- We have two lower bounds for sigma_max(A):
//--- * first estimate is provided by Frobenius norm, it is equal to ANorm/NMain
//--- * second estimate is provided by max(CAC)
//--- We select largest one of these estimates, because using just one
//--- of them is prone to different failure modes. Then, we divide A and B
//--- by this estimate.
   targetscale=NormalizeQuadraticTerm(State.m_sclsfta,State.m_sclsftb,nmain,State.m_sclsftcleic,nectotal,nictotal,renormlc,State.m_tmp2);
//--- Select working set of inequality constraints.
//--- Although it is possible to process all inequality constraints
//--- at once, in one large batch, some QP problems have NIC>>N constraints,
//--- but only minor fraction of them is inactive in the solution.
//--- Because algorithm running time is O((N+NEC+NIC)^3), we can
//--- save a lot of time if we process only those inequality constraints
//--- which need activation. Generally, NEC<N, and only O(N) inequality
//--- constraints are active in the solution.
//--- We can do so by solving problem first without general inequality
//--- constraints at all (box and general equality constraints are added),
//--- and by iteratively adding more and more inequality constraints in
//--- order to get feasible solution. Such set of inequality constraints
//--- is called "working set".
//--- NOTE: such approach works reliably only for convex QP problems; non-convex
//---       QP problem can be unbounded when some constraints are dropped.
//--- NOTE: we can also remove some constraints from working set, but eviction
//---       can be performed only limited amount of times (at most once); if
//---       constraint is added to working set second time, it is never removed.
//--- NOTE: we do not perform constraint eviction on non-convex problems
   SelectInitialWorkingSet(State.m_sclsfta,nmain,State.m_sclsftcleic,nectotal,nictotal,State.m_tmp0,State.m_tmp2,nicwork,allowwseviction);
   kwork=nectotal+nicwork;
   nwork=nmain+nicwork;
   State.m_repnwrk0=nicwork;
   for(i=0; i<nicwork; i++)
      State.m_nicnact.Set(i,1);
   for(i=nicwork; i<=nictotal-1; i++)
      State.m_nicnact.Set(i,0);
//--- Perform outer iteration
   for(i=0; i<ktotal; i++)
     {
      State.m_cidx.Set(i,i);
      State.m_nulc.Set(i,0);
     }
   for(i=0; i<ntotal; i++)
     {
      State.m_exscale.Set(i,1.0);
      State.m_exxorigin.Set(i,0.0);
     }
   GenerateExInitialPoint(State.m_sclsftxc,nmain,nslack,State.m_exxc);
   goodcounter=0;
   stagnationcounter=0;
   feaserr=CMath::m_maxrealnumber;
   for(outeridx=0; outeridx<Settings.m_outerits; outeridx++)
     {
      //--- Repeat loop until working set stabilizes.
      do
        {
         //--- Preallocate space for ExA and for QQP m_solver; we do not allocate
         //--- array[NTotal,NTotal] from the start because NTotal can be much
         //--- larger than NMain for problems with large amount of inequality
         //--- constraints, and we usually need NWork=O(NMain).
         //--- NOTE: for the sake of simplicity, 1-dimensional arrays were
         //---       preallocated to the maximum size required (NTotal).
         if(State.m_exa.Rows()<nwork || State.m_exa.Cols()<nwork)
           {
            i=nwork+nwork/3+1;
            State.m_exa.Resize(i,i);
           }
         CQQPSolver::QQPPreAllocateGrowDense(State.m_qqpbuf,nwork,i);
         //--- Generate penalized quadratic model
         GenerateExModel(State.m_sclsfta,State.m_sclsftb,nmain,State.m_sclsftbndl,State.m_sclsfthasbndl,State.m_sclsftbndu,State.m_sclsfthasbndu,State.m_sclsftcleic,nectotal,nicwork,State.m_nulc,rho,State.m_exa,State.m_exb,State.m_exbndl,State.m_exbndu,State.m_tmp2);
         //--- Solve extended QP problem subject to current working set of general
         //--- inequality constraints.
         CQQPSolver::QQPLoadDefaults(nwork,State.m_qqpsettingsuser);
         State.m_qqpsettingsuser.m_maxouterits=50;
         State.m_qqpsettingsuser.m_epsg=0.0;
         State.m_qqpsettingsuser.m_epsf=0.0;
         State.m_qqpsettingsuser.m_epsx=0.01*epsx;
         State.m_qqpsettingsuser.m_cnphase=true;
         CQQPSolver::QQPOptimize(State.m_dummycqm,State.m_dummysparse,State.m_exa,2,true,State.m_exb,State.m_exbndl,State.m_exbndu,State.m_exscale,State.m_exxorigin,nwork,State.m_qqpsettingsuser,State.m_qqpbuf,State.m_exxc,k);
         State.m_repncholesky+=State.m_qqpbuf.m_repncholesky;
         //--- Evaluate violation of constraints
         for(i=0; i<nictotal; i++)
           {
            v=0.0;
            for(i_=0; i_<nmain; i_++)
               v+=State.m_sclsftcleic.Get(nectotal+i,i_)*State.m_exxc[i_];
            v-=State.m_sclsftcleic.Get(nectotal+i,nmain);
            State.m_nicerr.Set(i,v);
           }
         //--- Working set expansion:
         //--- * select limited amount of most violated constraints
         //--- * perform permutation of non-work constraints such that
         //---   candidate constraint is first the list (update XC and NuLC)
         //--- * increase working set size by 1
         //--- * increase activation count for new constraint by 1 (this count
         //---   is used later by working set eviction phase)
         //--- * repeat
         //--- NOTE: we use selection sort algorithm because its O(NAdded*NWork) cost
         //---       is still comparable to the cost of constraints evaluation
         workingsetextended=false;
         i=0;
         while((double)(i)<(1+m_expansionratio*nmain) && nicwork<nictotal)
           {
            //--- Select most violated constraint
            k=nicwork;
            for(j=nicwork; j<nictotal; j++)
              {
               if(State.m_nicerr[j]>State.m_nicerr[k])
                  k=j;
              }
            //--- If violation is positive, add it
            if(State.m_nicerr[k]>0.0)
              {
               State.m_sclsftcleic.SwapRows(nectotal+nicwork,nectotal+k);
               State.m_nicerr.Swap(nicwork,k);
               State.m_nicnact.Swap(nicwork,k);
               State.m_cidx.Swap(nectotal+nicwork,nectotal+k);
               State.m_cscales.Swap(nectotal+nicwork,nectotal+k);
               State.m_exxc.Set(nmain+nicwork,0.0);
               State.m_nulc.Set(nectotal+nicwork,0.0);
               State.m_nicnact.Add(nicwork,1);
               nicwork++;
               nwork++;
               kwork++;
               i++;
               workingsetextended=true;
              }
            else
               break;
           }
         //--- Working set eviction:
         //--- * select constraints which are (1) far away from the
         //---   boundary, AND (2) has less than two activation attempts
         //---   (if constraint is regularly activated/deactivated, we keep
         //---   it in the working set no matter what)
         //--- * remove such constraints from the working set one by one
         if(allowwseviction)
           {
            for(k=nicwork-1; k>=0; k--)
              {
               if(State.m_nicerr[k]<m_evictionlevel && State.m_nicnact[k]<=1)
                 {
                  State.m_sclsftcleic.SwapRows(nectotal+nicwork-1,nectotal+k);
                  State.m_cidx.Swap(nectotal+nicwork-1,nectotal+k);
                  State.m_cscales.Swap(nectotal+nicwork-1,nectotal+k);
                  State.m_nicerr.Swap(nicwork-1,k);
                  State.m_nicnact.Swap(nicwork-1,k);
                  State.m_exxc.Swap(nmain+nicwork-1,nmain+k);
                  State.m_nulc.Swap(nectotal+nicwork-1,nectotal+k);
                  nicwork--;
                  nwork--;
                  kwork--;
                 }
              }
           }
         //--- Report working set statistics
         if(State.m_repnwrk1==0)
            State.m_repnwrk1=nicwork;
         State.m_repnwrkf=nicwork;
         if(workingsetextended)
            State.m_repnwrkchanges++;
        }
      while(workingsetextended);
      //--- Estimate Lagrange multipliers using alternative algorithm
      for(i_=0; i_<kwork; i_++)
         State.m_nulcest.Set(i_,State.m_nulc[i_]);
      UpdateLagrangeMultipliers(State.m_sclsfta,State.m_sclsftb,nmain,State.m_sclsftbndl,State.m_sclsfthasbndl,State.m_sclsftbndu,State.m_sclsfthasbndu,State.m_sclsftcleic,nectotal,nicwork,State.m_exxc,State.m_nulcest,State);
      //--- Update XC and Lagrange multipliers
      feaserrprev=feaserr;
      feaserr=0;
      for(i=0; i<kwork; i++)
        {
         //--- Calculate I-th feasibility error in V using formula for distance
         //--- between point and line (here we calculate actual distance between
         //--- XN and hyperplane Ci'*XN=Bi, which is different from error Ci'*XN-Bi).
         v=0;
         vv=0;
         for(j=0; j<nmain; j++)
           {
            v+= State.m_sclsftcleic.Get(i,j)*State.m_exxc[j];
            vv+=CMath::Sqr(State.m_sclsftcleic.Get(i,j));
           }
         if(i>=nectotal)
           {
            v+= State.m_exxc[nmain+(i-nectotal)];
            vv++;
           }
         v-=State.m_sclsftcleic.Get(i,nmain);
         vv=CApServ::Coalesce(vv,1);
         v=v/MathSqrt(vv);
         //--- Calculate magnitude of Lagrangian update (and Lagrangian parameters themselves)
         feaserr+=CMath::Sqr(v);
         State.m_nulc.Set(i,State.m_nulcest[i]);
        }
      feaserr=MathSqrt(feaserr);
      if(feaserr<epsx)
         goodcounter++;
      else
         goodcounter=0;
      if(feaserr>(feaserrprev*requestedfeasdecrease))
         stagnationcounter++;
      else
         stagnationcounter=0;
      if(goodcounter>=2)
         break;
      if(stagnationcounter>=2)
         rho=MathMin(rho*10.0,maxrho);
      else
         rho=MathMin(rho*1.41,maxrho);
     }
//--- Convert Lagrange multipliers from internal format to one expected
//--- by caller:
//--- * reorder multipliers for linear constraints
//--- * compute residual from gradient+linearconstraints
//--- * compute multipliers for box constraints from residual
//--- * rescale everything
   for(i=0; i<nectotal+nicwork; i++)
      if(State.m_cscales[i]!=0)
         laglc.Set(State.m_cidx[i],-(State.m_nulc[i]*targetscale/State.m_cscales[i]));
      else
         laglc.Set(State.m_cidx[i],AL_NaN);
   State.m_tmpg.Resize(nmain);
   for(i=0; i<nmain; i++)
     {
      v=State.m_sclsftb[i];
      for(j=0; j<nmain; j++)
         v+= State.m_sclsfta.Get(i,j)*State.m_exxc[j];
      State.m_tmpg.Set(i,v);
     }
   CAblas::RMatrixGemVect(nmain,nectotal+nicwork,-1.0,State.m_sclsftcleic,0,0,1,State.m_nulc,0,1.0,State.m_tmpg,0);
   for(i=0; i<nmain; i++)
     {
      if((State.m_sclsfthasbndl[i] && State.m_exxc[i]==State.m_sclsftbndl[i]) || (State.m_sclsfthasbndu[i] && State.m_exxc[i]==State.m_sclsftbndu[i]))
         lagbc.Set(i,-State.m_tmpg[i]);
     }
   for(i=0; i<nmain; i++)
      lagbc.Mul(i,targetscale/s[i]);
//--- Unpack results.
//--- Add XOrigin to XC and make sure that boundary constraints are
//--- satisfied.
   for(i=0; i<nmain; i++)
     {
      //--- Unscale/unshift
      xs.Set(i,s[i]*State.m_exxc[i]+xorigin[i]);
      //--- Make sure that point is feasible w.m_r.m_t. box constraints.
      //--- Enforce box constraints which were active in the scaled/shifted solution.
      if(State.m_sclsfthasbndl[i])
        {
         if(xs[i]<bndl[i])
            xs.Set(i,bndl[i]);
         if(State.m_exxc[i]==State.m_sclsftbndl[i])
            xs.Set(i,bndl[i]);
        }
      if(State.m_sclsfthasbndu[i])
        {
         if(xs[i]>bndu[i])
            xs.Set(i,bndu[i]);
         if(State.m_exxc[i]==State.m_sclsftbndu[i])
            xs.Set(i,bndu[i]);
        }
     }
   terminationtype=2;
  }
//+------------------------------------------------------------------+
//| This function generates box - constrained QP problem, which is   |
//| penalized and augmented formulation of original linearly         |
//| constrained problem                                              |
//+------------------------------------------------------------------+
void CQPDenseAULSolver::GenerateExModel(CMatrixDouble &sclsfta,
                                        CRowDouble &sclsftb,
                                        int nmain,
                                        CRowDouble &sclsftbndl,
                                        bool &sclsfthasbndl[],
                                        CRowDouble &sclsftbndu,
                                        bool &sclsfthasbndu[],
                                        CMatrixDouble &sclsftcleic,
                                        int sclsftnec,
                                        int sclsftnic,
                                        CRowDouble &nulc,
                                        double rho,
                                        CMatrixDouble &exa,
                                        CRowDouble &exb,
                                        CRowDouble &exbndl,
                                        CRowDouble &exbndu,
                                        CMatrixDouble &tmp2)
  {
//--- create variables
   int    nslack=sclsftnic;
   int    ntotal=nmain+nslack;
   int    i=0;
   int    j=0;
   double v=0;
   int    i_=0;
//--- Integrity check for properly preallocated storage
   if(!CAp::Assert(exa.Rows()>=ntotal && exa.Cols()>=ntotal,__FUNCTION__+" - integrity check failed"))
      return;
   if(!CAp::Assert(exb.Size()>=ntotal && exbndl.Size()>=ntotal && exbndu.Size()>=ntotal,__FUNCTION__+" - integrity check failed"))
      return;
//--- Primary quadratic term
   for(i=0; i<ntotal; i++)
      for(j=i; j<ntotal; j++)
         exa.Set(i,j,0);
   for(i=0; i<nmain; i++)
      for(j=i; j<nmain; j++)
         exa.Set(i,j,sclsfta.Get(i,j));
//--- Primary linear term
   for(i=0; i<ntotal; i++)
      exb.Set(i,0);
   for(i=0; i<nmain; i++)
      exb.Set(i,sclsftb[i]);
//--- Box constraints - move primary, add slack
   for(i=0; i<nmain; i++)
     {
      if(sclsfthasbndl[i])
         exbndl.Set(i,sclsftbndl[i]);
      else
         exbndl.Set(i,AL_NEGINF);
      if(sclsfthasbndu[i])
         exbndu.Set(i,sclsftbndu[i]);
      else
         exbndu.Set(i,AL_POSINF);
     }
   for(i=nmain; i<ntotal; i++)
     {
      exbndl.Set(i,0);
      exbndu.Set(i,AL_POSINF);
     }
//--- Handle equality constraints:
//--- * modify quadratic term
//--- * modify linear term
//--- * add Lagrangian term
   tmp2.Resize(sclsftnec+sclsftnic,ntotal);
   for(i=0; i<sclsftnec+sclsftnic; i++)
     {
      //--- Given constraint row ci and right hand side ri,
      //--- I-th quadratic constraint adds penalty term
      //---     0.5*Rho*(ci'*x-ri)^2 =
      //---     = 0.5*Rho*(ci'*x-ri)^T*(ci'*x-ri) =
      //---     = 0.5*Rho*(x'*ci-ri')*(ci'*x-ri) =
      //---     = 0.5*Rho*(x'*ci*ci'*x - ri'*ci'*x - x'*ci*ri + ri'*ri )
      //---     = 0.5*Rho*(x'*(ci*ci')*x - 2*ri*(ci'*x) + ri^2 )
      //--- Thus, quadratic term is updated by
      //---     0.5*Rho*(ci*ci')
      //--- (with actual update to ExA being performed without 0.5
      //--- multiplier because entire matrix is post-multipliead by 0.5)
      //--- and linear term receives update
      //---     -Rho*ri*ci
      //--- Similaryly, lagrangian term is -NUi*(ci'*x-ri),
      //--- so linear term is updated by
      //---     -NUi*ci
      //--- Because our model does not take into account constant term,
      //--- we calculate just quadratic and linear terms.
      for(i_=0; i_<nmain; i_++)
         tmp2.Set(i,i_,sclsftcleic.Get(i,i_));
      for(j=nmain; j<ntotal; j++)
         tmp2.Set(i,j,0);
      if(i>=sclsftnec)
         tmp2.Set(i,nmain+i-sclsftnec,1.0);
      v=-(rho*sclsftcleic.Get(i,nmain));
      for(i_=0; i_<ntotal; i_++)
         exb.Add(i_,v*tmp2.Get(i,i_));
      v=-nulc[i];
      for(i_=0; i_<ntotal; i_++)
         exb.Add(i_,v*tmp2.Get(i,i_));
     }
   CAblas::RMatrixSyrk(ntotal,sclsftnec+sclsftnic,rho,tmp2,0,0,2,1.0,exa,0,0,true);
  }
//+------------------------------------------------------------------+
//| This function generates initial point for  "extended"  box -     |
//| constrained QP problem.                                          |
//+------------------------------------------------------------------+
void CQPDenseAULSolver::GenerateExInitialPoint(CRowDouble &sclsftxc,
                                               int nmain,
                                               int nslack,
                                               CRowDouble &exxc)
  {
   int ntotal=nmain+nslack;

   exxc=sclsftxc;
   exxc.Resize(ntotal);
   for(int i=nmain; i<ntotal; i++)
      exxc.Set(i,0);
  }
//+------------------------------------------------------------------+
//| This function estimates Lagrange multipliers for scaled - shifted|
//| QP problem (here "scaled-shifted" means that we performed        |
//| variable scaling and subtracted origin) given by quadratic       |
//| term A, linear term B, box constraints and linear constraint     |
//| matrix.                                                          |
//| It is assumed that all linear constraints are equality  ones,    |
//| with first NEC ones being constraints without slack  variables,  |
//| and next NIC ones having slack variables. The only inequality    |
//| constraints we have are box ones, with first NMain ones being    |
//| "general" box constraints, and next NIC ones being non-negativity|
//| constraints(not specified explicitly).                           |
//| We also make use of the current point XC, which is used to       |
//| determine active box constraints.                                |
//| Actual QP problem size is NMain + NIC, but some parameters have  |
//| lower dimensionality.                                            |
//| Parameters sizes are:                                            |
//|   * A is assumed to be array[NMain, NMain]                       |
//|   * B is assumed to be array[NMain]                              |
//|   * BndL, BndU are array[NMain]                                  |
//|   * CLEIC is array[NEC + NIC, NMain + 1] (last item in a row     |
//|     containts right part)                                        |
//|   * ExXC is array[NMain + NIC], holds current point              |
//|   * NuLCEst is array[NEC + NIC], holds initial values of Lagrange|
//|     coeffs                                                       |
//| On exit NuLCEst is updated with new estimate of Lagrange         |
//| multipliers.                                                     |
//+------------------------------------------------------------------+
void CQPDenseAULSolver::UpdateLagrangeMultipliers(CMatrixDouble &sclsfta,
                                                  CRowDouble &sclsftb,
                                                  int nmain,
                                                  CRowDouble &sclsftbndl,
                                                  bool &sclsfthasbndl[],
                                                  CRowDouble &sclsftbndu,
                                                  bool &sclsfthasbndu[],
                                                  CMatrixDouble &sclsftcleic,
                                                  int sclsftnec,
                                                  int sclsftnic,
                                                  CRowDouble &exxc,
                                                  CRowDouble &nulcest,
                                                  CQPDenseAULBuffers &buffers)
  {
//--- create variables
   int    nslack=sclsftnic;
   int    ntotal=nmain+nslack;
   int    ktotal=sclsftnec+sclsftnic;
   int    nqrrows=0;
   int    nqrcols=0;
   int    i=0;
   int    j=0;
   double lambdareg=0;
   double mxdiag=0;
   double v=0;
   bool   isactive;
   int    i_=0;
//--- Given current point ExXC, we can determine active and inactive
//--- constraints. After we drop inactive inequality constraints, we
//--- have equality-only constrained QP problem, with mix of general
//--- linear equality constraints and "simple" constraints Xi=Ci.
//--- Problem min(0.5*x'*A*x + b'*x) s.m_t. C*x=d (general linear
//--- constraints) can be solved by explicitly writing out Lagrange
//--- equations:
//---     [ A  C' ] [ X ]   [ -b]
//---     [       ] [   ] = [   ]
//---     [ C     ] [ L ]   [ d ]
//--- or
//---         [ X ]
//---     A1* [   ] = b1
//---         [ L ]
//--- where X stands for solution itself, and L stands for Lagrange
//--- multipliers. It can be easily solved with direct linear m_solver.
//--- However, such formulation does not account for "simple" equality
//--- constraints on variables. It is possible to include "simple"
//--- constraints into "general" ones (i.e. append (0 ... 0 -1 0 ... 0)'
//--- to the constraint matrix), but it will increase problem
//--- size.
//--- Another approach is to use initial values of X and L (X0 and L0)
//--- as starting point, and to solve for "offset" from (X0, L0):
//---        [ X0+X1 ]
//---     A1*[       ] = b1
//---        [ L0+L1 ]
//--- or
//---        [ X1 ]           [ X0 ]
//---     A1*[    ] = b1 - A1*[    ]
//---        [ L1 ]           [ L0 ]
//--- In such formulation components of X1 which correspond to active
//--- constraints on variables are "frozen" at value 0 (because we have
//--- equality constraint, offset from constrained value have to be zero).
//--- Thus, we can rewrite corresponding columns of A1 with zeros - and
//--- use this space to store (0 ... 0 -1 0 ... 0)', which is used to
//--- account for Lagrange multipliers for "simple" constraints.
   nqrcols=ntotal+ktotal;
   nqrrows=nqrcols;
   buffers.m_qrsv0=exxc;
   buffers.m_qrsv0.Resize(nqrcols);
   buffers.m_qrsvx1.Resize(nqrcols);
   for(i=0; i<ktotal; i++)
      buffers.m_qrsv0.Set(ntotal+i,nulcest[i]);
   lambdareg=1.0E-8;
   while(true)
     {
      //--- Initialize matrix A1 and right part b1 with zeros
      buffers.m_qrkkt=matrix<double>::Zeros(nqrcols+nqrcols,nqrcols+1);
      buffers.m_qrrightpart=vector<double>::Zeros(nqrcols+nqrcols);
      //--- Append quadratic term (note: we implicitly add NSlack zeros to
      //--- A and b).
      mxdiag=0;
      for(i=0; i<nmain; i++)
        {
         for(j=0; j<nmain; j++)
            buffers.m_qrkkt.Set(i,j,sclsfta.Get(i,j));
         buffers.m_qrrightpart.Set(i,-sclsftb[i]);
         mxdiag=MathMax(mxdiag,MathAbs(sclsfta.Get(i,i)));
        }
      mxdiag=CApServ::Coalesce(mxdiag,1);
      //--- Append general linear constraints
      for(i=0; i<ktotal; i++)
        {
         for(j=0; j<nmain; j++)
           {
            buffers.m_qrkkt.Set(ntotal+i,j,-sclsftcleic.Get(i,j));
            buffers.m_qrkkt.Set(j,ntotal+i,-sclsftcleic.Get(i,j));
           }
         if(i>=sclsftnec)
           {
            buffers.m_qrkkt.Set(ntotal+i,nmain+(i-sclsftnec),-1);
            buffers.m_qrkkt.Set(nmain+(i-sclsftnec),ntotal+i,-1);
           }
         buffers.m_qrrightpart.Set(ntotal+i,-sclsftcleic.Get(i,nmain));
        }
      //--- Append regularizer to the bottom of the matrix
      //--- (it will be factored in during QR decomposition)
      if(lambdareg>0.0)
        {
         nqrrows=nqrcols+nqrcols;
         for(i=0; i<nqrcols; i++)
            buffers.m_qrkkt.Set(nqrcols+i,i,lambdareg*mxdiag);
        }
      //--- Subtract reference point (X0,L0) from the system
      for(i=0; i<nqrcols; i++)
        {
         v=0.0;
         for(i_=0; i_<nqrcols; i_++)
            v+=buffers.m_qrkkt.Get(i,i_)*buffers.m_qrsv0[i_];
         buffers.m_qrrightpart.Add(i,-v);
        }
      //--- Handle active "simple" equality constraints
      for(i=0; i<ntotal; i++)
        {
         isactive=false;
         if(i<nmain && ((sclsfthasbndl[i] && exxc[i]==sclsftbndl[i]) || (sclsfthasbndu[i] && exxc[i]==sclsftbndu[i])))
            isactive=true;
         if(i>=nmain && exxc[i]==0.0)
            isactive=true;
         if(!isactive)
            continue;
         for(j=0; j<nqrrows; j++)
            buffers.m_qrkkt.Set(j,i,0);
         buffers.m_qrkkt.Set(i,i,-1);
        }
      //--- Solve via QR decomposition:
      //--- * append right part to the system matrix
      //--- * perform QR decomposition of the extended matrix (right part is implicitly
      //---   multiplied by Q during decomposition; believe me, it works!)
      //--- * check condition number, increase regularization value if necessary and retry
      //--- * solve triangular system, break iteration
      for(i=0; i<nqrrows; i++)
         buffers.m_qrkkt.Set(i,nqrcols,buffers.m_qrrightpart[i]);
      COrtFac::RMatrixQR(buffers.m_qrkkt,nqrrows,nqrcols+1,buffers.m_qrtau);
      if(CRCond::RMatrixTrRCond1(buffers.m_qrkkt,nqrcols,true,false)<=(1000.0*CMath::m_machineepsilon))
        {
         lambdareg=CApServ::Coalesce(10*lambdareg,1.0E-13);
         continue;
        }
      for(i=nqrcols-1; i>=0; i--)
        {
         v=buffers.m_qrkkt.Get(i,nqrcols);
         for(j=i+1; j<nqrcols; j++)
            v-=buffers.m_qrkkt.Get(i,j)*buffers.m_qrsvx1[j];
         buffers.m_qrsvx1.Set(i,v/buffers.m_qrkkt.Get(i,i));
        }
      break;
     }
//--- Update Lagrange coefficients
   for(i=0; i<ktotal; i++)
      nulcest.Set(i,buffers.m_qrsv0[ntotal+i]+buffers.m_qrsvx1[ntotal+i]);
  }
//+------------------------------------------------------------------+
//| This function generates scaled (by S) and shifted (by XC)        |
//| reformulation of the original problem.                           |
//| INPUT PARAMETERS:                                                |
//|   DenseA      -  for dense problems(AKind = 0), A - term of CQM  |
//|                  object contains system matrix. Other terms are  |
//|                  unspecified and should not be referenced.       |
//|   SparseA     -  for sparse problems(AKind = 1), CRS format      |
//|   AKind       -  sparse matrix format:                           |
//|                  * 0 for dense matrix                            |
//|                  * 1 for sparse matrix                           |
//|   SparseUpper -  which triangle of SparseAC stores matrix - upper|
//|                  or lower one (for dense matrices this parameter |
//|                  is not actual).                                 |
//|   B           -  linear term, array[N]                           |
//|   BndL        -  lower bound, array[N]                           |
//|   BndU        -  upper bound, array[N]                           |
//|   S           -  scale vector, array[NC]:                        |
//|                  * I-Th element contains scale of I-Th variable, |
//|                  * SC[I] > 0                                     |
//|   XOrigin     -  origin term, array[NC]. Can be zero.            |
//|   N           -  number of variables in the original formulation |
//|                  (no slack variables).                           |
//|   CLEIC       -  dense linear equality/inequality constraints.   |
//|                  Equality constraints come first.                |
//|   NEC, NIC    -  number of dense equality/inequality constraints.|
//|   SCLEIC      -  sparse linear equality / inequality constraints.|
//|                  Equality constraints come first.                |
//|   SNEC, SNIC  -  number of sparse equality/inequality constraints|
//|   RenormLC    -  whether constraints should be renormalized      |
//|                  (recommended) or used "as is".                  |
//|   Settings    -  QPDENSEAULSettings object initialized by one of |
//|                  the initialization functions.                   |
//|   State       -  object which stores temporaries                 |
//|   XS          -  initial point, array[NC]                        |
//| On output, following fields of the State structure are modified: |
//|   * SclSftA   -  array[NMain, NMain], quadratic term, both       |
//|                  triangles                                       |
//|   * SclSftB   -  array[NMain], linear term                       |
//|   * SclSftXC  -  array[NMain], initial point                     |
//|   * SclSftHasBndL,                                               |
//|     SclSftHasBndU,                                               |
//|     SclSftBndL,                                                  |
//|     SclSftBndU - array[NMain], lower / upper bounds              |
//|   * SclSftCLEIC- array[KTotal, NMain + 1], general linear        |
//|                  constraints                                     |
//| NOTE: State.Tmp2 is used to store temporary array[NMain, NMain]  |
//+------------------------------------------------------------------+
void CQPDenseAULSolver::ScaleShiftOriginalPproblem(CConvexQuadraticModel &a,
                                                   CSparseMatrix &sparsea,
                                                   int akind,
                                                   bool sparseaupper,
                                                   CRowDouble &b,
                                                   CRowDouble &bndl,
                                                   CRowDouble &bndu,
                                                   CRowDouble &s,
                                                   CRowDouble &xorigin,
                                                   int nmain,
                                                   CMatrixDouble &cleic,
                                                   int dnec,
                                                   int dnic,
                                                   CSparseMatrix &scleic,
                                                   int snec,
                                                   int snic,
                                                   bool renormlc,
                                                   CQPDenseAULBuffers &State,
                                                   CRowDouble &xs)
  {
//--- create variables
   int    i=0;
   int    j=0;
   int    k=0;
   int    j0=0;
   int    j1=0;
   double v=0;
   double vv=0;
   int    ktotal=0;
//--- check
   if(!CAp::Assert(akind==0 || akind==1,__FUNCTION__+": unexpected AKind"))
      return;

   ktotal=dnec+dnic+snec+snic;
   CApServ::BVectorSetLengthAtLeast(State.m_sclsfthasbndl,nmain);
   CApServ::BVectorSetLengthAtLeast(State.m_sclsfthasbndu,nmain);
   State.m_sclsfta=matrix<double>::Zeros(nmain,nmain);
   State.m_sclsftb.Resize(nmain);
   State.m_sclsftxc.Resize(nmain);
   State.m_sclsftbndl.Resize(nmain);
   State.m_sclsftbndu.Resize(nmain);
   State.m_sclsftcleic.Resize(ktotal,nmain+1);
   State.m_cscales.Resize(ktotal);
   if(akind==0)
     {
      //--- Extract dense A and scale
      CCQModels::CQMGetA(a,State.m_tmp2);
      for(i=0; i<nmain; i++)
        {
         for(j=i; j<nmain; j++)
           {
            v=State.m_tmp2.Get(i,j)*s[i]*s[j];
            State.m_sclsfta.Set(i,j,v);
            State.m_sclsfta.Set(j,i,v);
           }
        }
     }
   if(akind==1)
     {
      //--- Extract sparse A and scale
      //--- check
      if(!CAp::Assert(sparsea.m_MatrixType==1,__FUNCTION__+": unexpected sparse matrix format"))
         return;
      if(!CAp::Assert(sparsea.m_M==nmain,__FUNCTION__+": unexpected sparse matrix size"))
         return;
      if(!CAp::Assert(sparsea.m_N==nmain,__FUNCTION__+": unexpected sparse matrix size"))
         return;
      if(sparseaupper)
        {
         for(i=0; i<nmain; i++)
           {
            if(sparsea.m_DIdx[i]!=sparsea.m_UIdx[i])
               State.m_sclsfta.Set(i,i,sparsea.m_Vals[sparsea.m_DIdx[i]]*s[i]*s[i]);
            j0=sparsea.m_UIdx[i];
            j1=sparsea.m_RIdx[i+1]-1;
            for(j=j0; j<=j1; j++)
              {
               k=sparsea.m_Idx[j];
               v=sparsea.m_Vals[j]*s[i]*s[k];
               State.m_sclsfta.Set(i,k,v);
               State.m_sclsfta.Set(k,i,v);
              }
           }
        }
      else
        {
         for(i=0; i<nmain; i++)
           {
            if(sparsea.m_DIdx[i]!=sparsea.m_UIdx[i])
               State.m_sclsfta.Set(i,i,sparsea.m_Vals[sparsea.m_DIdx[i]]*s[i]*s[i]);
            j0=sparsea.m_RIdx[i];
            j1=sparsea.m_DIdx[i]-1;
            for(j=j0; j<=j1; j++)
              {
               k=sparsea.m_Idx[j];
               v=sparsea.m_Vals[j]*s[i]*s[k];
               State.m_sclsfta.Set(i,k,v);
               State.m_sclsfta.Set(k,i,v);
              }
           }
        }
     }
   for(i=0; i<nmain; i++)
     {
      State.m_sclsfthasbndl[i]=MathIsValidNumber(bndl[i]);
      State.m_sclsfthasbndu[i]=MathIsValidNumber(bndu[i]);
      State.m_sclsftb.Set(i,b[i]*s[i]);
      State.m_sclsftxc.Set(i,(xs[i]-xorigin[i])/s[i]);
      State.m_sclsftbndl.Set(i,bndl[i]);
      State.m_sclsftbndu.Set(i,bndu[i]);
     }
   CLPQPServ::ScaleShiftBCInplace(s,xorigin,State.m_sclsftbndl,State.m_sclsftbndu,nmain);
   for(i=0; i<=ktotal-1; i++)
      for(j=0; j<=nmain; j++)
         State.m_sclsftcleic.Set(i,j,0);
   for(i=0; i<dnec; i++)
     {
      for(j=0; j<nmain; j++)
        {
         v=cleic.Get(i,j)*s[j];
         State.m_sclsftcleic.Set(i,j,v);
        }
      State.m_sclsftcleic.Set(i,nmain,cleic.Get(i,nmain));
     }
   for(i=0; i<dnic; i++)
     {
      for(j=0; j<nmain; j++)
        {
         v=cleic.Get(dnec+i,j)*s[j];
         State.m_sclsftcleic.Set(dnec+snec+i,j,v);
        }
      State.m_sclsftcleic.Set(dnec+snec+i,nmain,cleic.Get(dnec+i,nmain));
     }
   for(i=0; i<snec; i++)
     {
      //--- Because constraints are sparse, everything is a bit tricky -
      //--- it is possible that N-th element of the row is zero and not
      //--- stored; it is also possible that entire row is empty.
      j0=scleic.m_RIdx[i];
      j1=scleic.m_RIdx[i+1]-1;
      if(j1>=j0 && scleic.m_Idx[j1]==nmain)
        {
         State.m_sclsftcleic.Set(dnec+i,nmain,scleic.m_Vals[j1]);
         j1--;
        }
      for(j=j0; j<=j1; j++)
        {
         k=scleic.m_Idx[j];
         v=scleic.m_Vals[j]*s[k];
         State.m_sclsftcleic.Set(dnec+i,k,v);
        }
     }
   for(i=0; i<snic; i++)
     {
      //--- Because constraints are sparse, everything is a bit tricky -
      //--- it is possible that N-th element of the row is zero and not
      //--- stored; it is also possible that entire row is empty.
      j0=scleic.m_RIdx[snec+i];
      j1=scleic.m_RIdx[snec+i+1]-1;
      if(j1>=j0 && scleic.m_Idx[j1]==nmain)
        {
         State.m_sclsftcleic.Set(dnec+snec+dnic+i,nmain,scleic.m_Vals[j1]);
         j1--;
        }
      for(j=j0; j<=j1; j++)
        {
         k=scleic.m_Idx[j];
         v=scleic.m_Vals[j]*s[k];
         State.m_sclsftcleic.Set(dnec+snec+dnic+i,k,v);
        }
     }
   if(renormlc && ktotal>0)
     {
      //--- Normalize linear constraints in such way that they have unit norm
      //--- (after variable scaling)
      for(i=0; i<ktotal; i++)
        {
         vv=0.0;
         for(j=0; j<nmain; j++)
           {
            v=State.m_sclsftcleic.Get(i,j);
            vv+=v*v;
           }
         vv=MathSqrt(vv);
         State.m_cscales.Set(i,vv);
         if(vv>0.0)
           {
            vv=1/vv;
            for(j=0; j<=nmain; j++)
               State.m_sclsftcleic.Mul(i,j,vv);
           }
        }
     }
   else
     {
      //--- Load unit scales
      for(i=0; i<ktotal; i++)
         State.m_cscales.Set(i,1.0);
     }
   for(i=0; i<ktotal; i++)
     {
      //--- Apply XOrigin
      v=0.0;
      for(j=0; j<nmain; j++)
         v+= State.m_sclsftcleic.Get(i,j)*(xorigin[j]/s[j]);
      State.m_sclsftcleic.Add(i,nmain,-v);
     }
  }
//+------------------------------------------------------------------+
//| Normalize model in such way that norm(A)~1(very roughly)         |
//| We have two lower bounds for sigma_max(A):                       |
//|   * first estimate is provided by Frobenius norm, it is equal    |
//|     to ANorm / NMain                                             |
//|   * second estimate is provided by max(CAC)                      |
//| We select largest one of these estimates, because using just one |
//| of them is prone to different failure modes. Then, we divide A   |
//| and B by this estimate.                                          |
//| INPUT PARAMETERS:                                                |
//|   A        -  array[N, N], quadratic term, full triangle is given|
//|   B        -  array[N], linear term                              |
//|   N        -  problem size                                       |
//|   CLEIC    -  array[NEC + NIC, N + 1], linear equality/inequality|
//|               constraints                                        |
//|   NEC      -  number of equality constraints                     |
//|   NIC      -  number of inequality constraints                   |
//|   UseCLEIC -  additional normalization of A in such way that     |
//|               CLEIC * A * CLEIC'~1:                              |
//|               * if False, CLEIC is ignored                       |
//|               * if True, CLEIC rows MUST have unit norm (we      |
//|                 check it)                                        |
//|   Tmp2     -  additional buffer, possibly preallocated           |
//| OUTPUT PARAMETERS:                                               |
//|   A, B     -  appropriately rescaled by 1 / SCL                  |
//| RESULT: multiplier SCL                                           |
//+------------------------------------------------------------------+
double CQPDenseAULSolver::NormalizeQuadraticTerm(CMatrixDouble &a,
                                                 CRowDouble &b,
                                                 int n,
                                                 CMatrixDouble &cleic,
                                                 int nec,
                                                 int nic,
                                                 bool usecleic,
                                                 CMatrixDouble &tmp2)
  {
//--- create variables
   double result=0;
   int    i=0;
   int    j=0;
   double anorm=0;
   double maxcac=0;
   double v=0;
   double vv=0;
   int    nmain=n;
   int    ktotal=nec+nic;

   for(i=0; i<nmain; i++)
      for(j=0; j<nmain; j++)
         anorm=anorm+CMath::Sqr(a.Get(i,j));
   anorm=MathSqrt(anorm);
   if(usecleic && ktotal>0)
     {
      //--- Calculate max(|diag(C*A*C')|), where C is constraint matrix
      tmp2.Resize(ktotal,nmain);
      CAblas::RMatrixGemm(ktotal,nmain,nmain,1.0,cleic,0,0,0,a,0,0,0,0.0,tmp2,0,0);
      maxcac=0.0;
      for(i=0; i<ktotal; i++)
        {
         v=0;
         vv=0;
         for(j=0; j<nmain; j++)
           {
            v+= tmp2.Get(i,j)*cleic.Get(i,j);
            vv+=CMath::Sqr(cleic.Get(i,j));
           }
         //--- check
         if(!CAp::Assert(MathAbs(vv-1)<1.0E-9 || vv==0.0,__FUNCTION__+": integrity check failed"))
            return(result);
         maxcac=MathMax(maxcac,MathAbs(v));
        }
     }
   else
      maxcac=0;
   result=CApServ::Coalesce(MathMax(maxcac,anorm/nmain),1);
   v=1/result;
   for(i=0; i<nmain; i++)
      for(j=0; j<nmain; j++)
         a.Mul(i,j,v);
   for(i=0; i<nmain; i++)
      b.Mul(i,v);
//--- return result
   return(result);
  }
//+------------------------------------------------------------------+
//| This function selects initial working set of general inequality  |
//| constraints for QP problem:                                      |
//|   * for non - convex QP problems - NICWork = NIC is returned     |
//|   * otherwise                    - NICWork = 0 is returned (we   |
//|                                    have to determine working set |
//|                                    iteratively)                  |
//| INPUT PARAMETERS:                                                |
//|   A           -  array[NMain], quadratic term, full matrix is    |
//|                  stored                                          |
//|   NMain      - number of variables in the "original" QP problem|
//|   CLEIC       -  array[NEC + NIC, NMain + 1], constraint matrix  |
//|   NEC         -  number of equality constraints                  |
//|   NIC         -  number of inequality constraints                |
//| OUTPUT PARAMETERS:                                               |
//|   NICWork     -  recommended size of working set; in current     |
//|                  version either all (NICWork = NIC) or none      |
//|                  (NICWork = 0) constraints are included.         |
//|   AllowWSEviction - whether problem properties allow eviction    |
//|                  of constraints from working set or not.         |
//|                  Non-convex problems do not allow eviction,      |
//|                  convex ones do.                                 |
//+------------------------------------------------------------------+
void CQPDenseAULSolver::SelectInitialWorkingSet(CMatrixDouble &a,
                                                int nmain,
                                                CMatrixDouble &cleic,
                                                int nec,
                                                int nic,
                                                CRowDouble &m_tmp0,
                                                CMatrixDouble &tmp2,
                                                int &nicwork,
                                                bool &allowwseviction)
  {
   nicwork=0;
   allowwseviction=false;
   tmp2=a;
   tmp2.Resize(nmain,nmain);
   m_tmp0.Resize(nmain);

   if(!CTrFac::SPDMatrixCholeskyRec(tmp2,0,nmain,true,m_tmp0))
     {
      //--- Matrix is indefinite.
      //--- We have to select full working set, otherwise algorithm may fail
      //--- because problem with reduced working set can be unbounded from below.
      nicwork=nic;
      allowwseviction=false;
     }
   else
     {
      //--- Positive definite matrix.
      //--- We can select zero initial working set and expand it later.
      nicwork=0;
      allowwseviction=true;
     }
  }
//+------------------------------------------------------------------+
//| This object stores Settings for QPBLEIC m_solver.                  |
//| It must be initialized with QPBLEICLoadDefaults().               |
//| After initialization you may change Settings.                    |
//+------------------------------------------------------------------+
struct CQPBLEICSettings
  {
   int               m_maxits;
   double            m_epsf;
   double            m_epsg;
   double            m_epsx;
   //--- constructor / destructor
                     CQPBLEICSettings(void) { ZeroMemory(this); }
                    ~CQPBLEICSettings(void) {}
   //--- copy
   void              Copy(const CQPBLEICSettings &obj);
   //--- overloading
   void              operator=(const CQPBLEICSettings &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CQPBLEICSettings::Copy(const CQPBLEICSettings &obj)
  {
   m_maxits=obj.m_maxits;
   m_epsf=obj.m_epsf;
   m_epsg=obj.m_epsg;
   m_epsx=obj.m_epsx;
  }
//+------------------------------------------------------------------+
//| This object stores temporaries used by QuickQP m_solver.           |
//+------------------------------------------------------------------+
struct CQPBLEICbuffers
  {
   int               m_repinneriterationscount;
   int               m_repouteriterationscount;
   CRowInt           m_tmpi;
   CRowDouble        m_tmp0;
   CRowDouble        m_tmp1;
   CMinBLEICState    m_solver;
   CMinBLEICReport   m_solverrep;
   //--- constructor / destructor
                     CQPBLEICbuffers(void);
                    ~CQPBLEICbuffers(void) {}
   //--- copy
   void              Copy(const CQPBLEICbuffers &obj);
   //--- overloading
   void              operator=(const CQPBLEICbuffers &obj) { Copy(obj); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CQPBLEICbuffers::CQPBLEICbuffers(void)
  {
   m_repinneriterationscount=0;
   m_repouteriterationscount=0;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CQPBLEICbuffers::Copy(const CQPBLEICbuffers &obj)
  {
   m_repinneriterationscount=obj.m_repinneriterationscount;
   m_repouteriterationscount=obj.m_repouteriterationscount;
   m_tmpi=obj.m_tmpi;
   m_tmp0=obj.m_tmp0;
   m_tmp1=obj.m_tmp1;
   m_solver=obj.m_solver;
   m_solverrep=obj.m_solverrep;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CQPBLEICSolver
  {
public:
   static void       QPBLEICLoadDefaults(int nmain,CQPBLEICSettings &s);
   static void       QPBLEICCopySettings(CQPBLEICSettings &src,CQPBLEICSettings &dst);
   static void       QPBLEICOptimize(CConvexQuadraticModel &a,CSparseMatrix &sparsea,int akind,bool sparseaupper,double absasum,double absasum2,CRowDouble &b,CRowDouble &bndl,CRowDouble &bndu,CRowDouble &s,CRowDouble &xorigin,int n,CMatrixDouble &cleic,int nec,int nic,CQPBLEICSettings &Settings,CQPBLEICbuffers &sstate,bool &firstcall,CRowDouble &xs,int &terminationtype);
  };
//+------------------------------------------------------------------+
//| This function initializes QPBLEICSettings structure with default |
//| Settings.                                                        |
//| Newly created structure MUST be initialized by default Settings -|
//| or by copy of the already initialized structure.                 |
//+------------------------------------------------------------------+
void CQPBLEICSolver::QPBLEICLoadDefaults(int nmain,
                                         CQPBLEICSettings &s)
  {
   s.m_epsg=0.0;
   s.m_epsf=0.0;
   s.m_epsx=1.0E-6;
   s.m_maxits=0;
  }
//+------------------------------------------------------------------+
//| This function initializes QPBLEICSettings  structure  with  copy |
//| of another, already initialized structure.                       |
//+------------------------------------------------------------------+
void CQPBLEICSolver::QPBLEICCopySettings(CQPBLEICSettings &src,
                                         CQPBLEICSettings &dst)
  {
   dst.m_epsg=src.m_epsg;
   dst.m_epsf=src.m_epsf;
   dst.m_epsx=src.m_epsx;
   dst.m_maxits=src.m_maxits;
  }
//+------------------------------------------------------------------+
//| This function runs QPBLEIC m_solver; it returns after            |
//| optimization process was completed. Following QP problem is      |
//| solved:                                                          |
//|   min(0.5 * (x - x_origin)'*A*(x-x_origin)+b' * (x - x_origin))  |
//| subject to boundary constraints.                                 |
//| INPUT PARAMETERS:                                                |
//|   AC       -  for dense problems(AKind = 0), A - term of CQM     |
//|               object contains system matrix. Other terms are     |
//|               unspecified and should not be referenced.          |
//|   SparseAC -  for sparse problems(AKind = 1)                     |
//|   AKind    -  sparse matrix format:                              |
//|               * 0 for dense matrix                               |
//|               * 1 for sparse matrix                              |
//|   SparseUpper - which triangle of SparseAC stores matrix - upper |
//|               or lower one (for dense matrices this parameter is |
//|               not actual).                                       |
//|   AbsASum  -  SUM( | A.Get(i,j) |)                                  |
//|   AbsASum2 -  SUM(A.Get(i,j) ^ 2)                                   |
//|   BC       -  linear term, array[NC]                             |
//|   BndLC    -  lower bound, array[NC]                             |
//|   BndUC    -  upper bound, array[NC]                             |
//|   SC       -  scale vector, array[NC]:                           |
//|               * I-th element contains scale of I-th variable,    |
//|               * SC[I] > 0                                        |
//|   XOriginC -  origin term, array[NC]. Can be zero.               |
//|   NC       -  number of variables in the  original  formulation  |
//|               (no slack variables).                              |
//|   CLEICC   -  linear equality / inequality constraints. Present  |
//|               version of this function does NOT provide  publicly|
//|               available support for linear constraints. This     |
//|               feature will be introduced in the future versions  |
//|               of the function.                                   |
//|   NEC, NIC -  number of equality/inequality constraints. MUST BE |
//|               ZERO IN THE CURRENT VERSION!!!                     |
//|   Settings -  QPBLEICSettings object initialized by one of the   |
//|               initialization functions.                          |
//|   SState   -  object which stores temporaries:                   |
//|               * if uninitialized object was passed, FirstCall    |
//|                 parameter MUST be set to True; object will be    |
//|                 automatically initialized by the function, and   |
//|                 FirstCall will be set to False.                  |
//|               * if FirstCall = False, it is assumed that this    |
//|                 parameter was already initialized by previous    |
//|                 call to this function with same problem          |
//|                 dimensions(variable count N).                    |
//|   FirstCall-  whether it is first call of this function for this |
//|               specific instance of SState, with this number of   |
//|               variables N specified.                             |
//|   XS       -  initial point, array[NC]                           |
//| OUTPUT PARAMETERS:                                               |
//|   XS       -  last point                                         |
//|   FirstCall-  uncondtionally set to False                        |
//|   TerminationType - termination type:                            |
//|                        *                                         |
//|                        *                                         |
//|                        *                                         |
//+------------------------------------------------------------------+
void CQPBLEICSolver::QPBLEICOptimize(CConvexQuadraticModel &a,
                                     CSparseMatrix &sparsea,
                                     int akind,bool sparseaupper,
                                     double absasum,double absasum2,
                                     CRowDouble &b,CRowDouble &bndl,
                                     CRowDouble &bndu,CRowDouble &s,
                                     CRowDouble &xorigin,int n,
                                     CMatrixDouble &cleic,int nec,
                                     int nic,CQPBLEICSettings &Settings,
                                     CQPBLEICbuffers &sstate,
                                     bool &firstcall,CRowDouble &xs,
                                     int &terminationtype)
  {
//--- create variables
   int    i=0;
   double d2=0;
   double d1=0;
   double d0=0;
   double v=0;
   double v0=0;
   double v1=0;
   double md=0;
   double mx=0;
   double mb=0;
   int    d1est=0;
   int    d2est=0;
   int    i_=0;
   terminationtype=0;
//--- check
   if(!CAp::Assert(akind==0 || akind==1,__FUNCTION__+": unexpected AKind"))
      return;

   sstate.m_repinneriterationscount=0;
   sstate.m_repouteriterationscount=0;
   terminationtype=0;
//--- Prepare m_solver object, if needed
   if(firstcall)
     {
      CMinBLEIC::MinBLEICCreate(n,xs,sstate.m_solver);
      firstcall=false;
     }
//--- Prepare max(|B|)
   mb=0.0;
   for(i=0; i<n; i++)
      mb=MathMax(mb,MathAbs(b[i]));
//--- Temporaries
   sstate.m_tmpi.Resize(nec+nic);
   sstate.m_tmp0.Resize(n);
   sstate.m_tmp1.Resize(n);
   sstate.m_tmpi.Fill(0,0,nec);
   sstate.m_tmpi.Fill(-1,nec,nic);
   CMinBLEIC::MinBLEICSetLC(sstate.m_solver,cleic,sstate.m_tmpi,nec+nic);
   CMinBLEIC::MinBLEICSetBC(sstate.m_solver,bndl,bndu);
   CMinBLEIC::MinBLEICSetDRep(sstate.m_solver,true);
   CMinBLEIC::MinBLEICSetCond(sstate.m_solver,CMath::m_minrealnumber,0.0,0.0,Settings.m_maxits);
   CMinBLEIC::MinBLEICSetScale(sstate.m_solver,s);
   CMinBLEIC::MinBLEICSetPrecScale(sstate.m_solver);
   CMinBLEIC::MinBLEICRestartFrom(sstate.m_solver,xs);
   while(CMinBLEIC::MinBLEICIteration(sstate.m_solver))
     {
      //--- Line search started
      if(sstate.m_solver.m_lsstart)
        {
         //--- Iteration counters:
         //--- * inner iterations count is increased on every line search
         //--- * outer iterations count is increased only at steepest descent line search
         sstate.m_repinneriterationscount++;
         if(sstate.m_solver.m_steepestdescentstep)
            sstate.m_repouteriterationscount++;
         //--- Build quadratic model of F along descent direction:
         //---     F(x+alpha*d) = D2*alpha^2 + D1*alpha + D0
         //--- Calculate estimates of linear and quadratic term
         //--- (term magnitude is compared with magnitude of numerical errors)
         d0=sstate.m_solver.m_f;
         d1=sstate.m_solver.m_d.Dot(sstate.m_solver.m_g);
         d2=0;
         switch(akind)
           {
            case 0:
               d2=CCQModels::CQMXTADX2(a,sstate.m_solver.m_d,sstate.m_tmp0);
               break;
            case 1:
               CSparse::SparseSMV(sparsea,sparseaupper,sstate.m_solver.m_d,sstate.m_tmp0);
               d2=sstate.m_solver.m_d.Dot(sstate.m_tmp0);
               d2/=2;
               break;
           }
         mx=0.0;
         md=0.0;
         for(i=0; i<n; i++)
           {
            mx=MathMax(mx,MathAbs(sstate.m_solver.m_x[i]));
            md=MathMax(md,MathAbs(sstate.m_solver.m_d[i]));
           }
         COptServ::EstimateParabolicModel(absasum,absasum2,mx,mb,md,d1,d2,d1est,d2est);
         //--- Tests for "normal" convergence.
         //--- This line search may be started from steepest descent
         //--- stage (stage 2) or from L-BFGS stage (stage 3) of the
         //--- BLEIC algorithm. Depending on stage type, different
         //--- checks are performed.
         //--- Say, L-BFGS stage is an equality-constrained refinement
         //--- stage of BLEIC. This stage refines current iterate
         //--- under "frozen" equality constraints. We can terminate
         //--- iterations at this stage only when we encounter
         //--- unconstrained direction of negative curvature. In all
         //--- other cases (say, when constrained gradient is zero)
         //--- we should not terminate algorithm because everything may
         //--- change after de-activating presently active constraints.
         //--- Tests for convergence are performed only at "steepest descent" stage
         //--- of the BLEIC algorithm, and only when function is non-concave
         //--- (D2 is positive or approximately zero) along direction D.
         //--- NOTE: we do not test iteration count (MaxIts) here, because
         //---       this stopping condition is tested by BLEIC itself.
         if(sstate.m_solver.m_steepestdescentstep && d2est>=0)
           {
            if(d1est>=0)
              {
               //--- "Emergency" stopping condition: D is non-descent direction.
               //--- Sometimes it is possible because of numerical noise in the
               //--- target function.
               terminationtype=4;
               xs=sstate.m_solver.m_x;
               break;
              }
            if(d2est>0)
              {
               //--- Stopping condition #4 - gradient norm is small:
               //--- 1. rescale State.Solver.D and State.Solver.G according to
               //---    current scaling, store results to Tmp0 and Tmp1.
               //--- 2. Normalize Tmp0 (scaled direction vector).
               //--- 3. compute directional derivative (in scaled variables),
               //---    which is equal to DOTPRODUCT(Tmp0,Tmp1).
               sstate.m_tmp0=sstate.m_solver.m_d/s+0;
               sstate.m_tmp1=sstate.m_solver.m_g*s+0;
               v=sstate.m_tmp0.Dot(sstate.m_tmp0);
               //--- check
               if(!CAp::Assert(v>0.0,__FUNCTION__+": inernal errror (scaled direction is zero)"))
                  return;
               v=1/MathSqrt(v);
               sstate.m_tmp0*=v;
               v=sstate.m_tmp0.Dot(sstate.m_tmp1);
               if(MathAbs(v)<=Settings.m_epsg)
                 {
                  terminationtype=4;
                  xs=sstate.m_solver.m_x;
                  break;
                 }
               //--- Stopping condition #1 - relative function improvement is small:
               //--- 1. calculate steepest descent step:   V = -D1/(2*D2)
               //--- 2. calculate function change:         V1= D2*V^2 + D1*V
               //--- 3. stop if function change is small enough
               v=-(d1/(2*d2));
               v1=d2*v*v+d1*v;
               if(MathAbs(v1)<=Settings.m_epsf*MathMax(d0,1.0))
                 {
                  terminationtype=1;
                  xs=sstate.m_solver.m_x;
                  break;
                 }
               //--- Stopping condition #2 - scaled step is small:
               //--- 1. calculate step multiplier V0 (step itself is D*V0)
               //--- 2. calculate scaled step length V
               //--- 3. stop if step is small enough
               v0=-(d1/(2*d2));
               v=0;
               for(i=0; i<n; i++)
                  v+=CMath::Sqr(v0*sstate.m_solver.m_d[i]/s[i]);
               if(MathSqrt(v)<=Settings.m_epsx)
                 {
                  terminationtype=2;
                  xs=sstate.m_solver.m_x;
                  break;
                 }
              }
           }
         //--- Test for unconstrained direction of negative curvature
         if((d2est<0 || (d2est==0 && d1est<0)) && !sstate.m_solver.m_boundedstep)
           {
            //--- Function is unbounded from below:
            //--- * function will decrease along D, i.e. either:
            //---   * D2<0
            //---   * D2=0 and D1<0
            //--- * step is unconstrained
            //--- If these conditions are true, we abnormally terminate QP
            //--- algorithm with return code -4 (we can do so at any stage
            //--- of BLEIC - whether it is L-BFGS or steepest descent one).
            terminationtype=-4;
            xs=sstate.m_solver.m_x;
            break;
           }
         //--- Suggest new step (only if D1 is negative far away from zero,
         //--- D2 is positive far away from zero).
         if(d1est<0 && d2est>0)
            sstate.m_solver.m_stp=CApServ::SafeMinPosRV(-d1,2*d2,sstate.m_solver.m_curstpmax);
        }
      //--- Gradient evaluation
      if(sstate.m_solver.m_needfg)
        {
         sstate.m_tmp0=sstate.m_solver.m_x-xorigin+0;
         switch(akind)
           {
            case 0:
               CCQModels::CQMADX(a,sstate.m_tmp0,sstate.m_tmp1);
               break;
            case 1:
               CSparse::SparseSMV(sparsea,sparseaupper,sstate.m_tmp0,sstate.m_tmp1);
               break;
           }
         v0=sstate.m_tmp0.Dot(sstate.m_tmp1);
         v1=sstate.m_tmp0.Dot(b);
         sstate.m_solver.m_f=0.5*v0+v1;
         sstate.m_solver.m_g=sstate.m_tmp1+b+0;
        }
     }
   if(terminationtype==0)
     {
      //--- BLEIC optimizer was terminated by one of its inner stopping
      //--- conditions. Usually it is iteration counter (if such
      //--- stopping condition was specified by user).
      CMinBLEIC::MinBLEICResultsBuf(sstate.m_solver,xs,sstate.m_solverrep);
      terminationtype=sstate.m_solverrep.m_terminationtype;
     }
   else
     {
      //--- BLEIC optimizer was terminated in "emergency" mode by QP
      //--- m_solver.
      //--- NOTE: such termination is "emergency" only when viewed from
      //---       BLEIC's position. QP m_solver sees such termination as
      //---       routine one, triggered by QP's stopping criteria.
      CMinBLEIC::MinBLEICEmergencyTermination(sstate.m_solver);
     }
  }
//+------------------------------------------------------------------+
//| This object stores nonlinear optimizer State.                    |
//| You should use functions provided by MinQP subpackage to work    |
//| with this object                                                 |
//+------------------------------------------------------------------+
class CMinQPState
  {
public:
   //--- variables
   int               m_akind;
   int               m_algokind;
   int               m_mdense;
   int               m_msparse;
   int               m_n;
   int               m_repinneriterationscount;
   int               m_repncholesky;
   int               m_repnmv;
   int               m_repouteriterationscount;
   int               m_repterminationtype;
   int               m_stype;
   double            m_absamax;
   double            m_absasum2;
   double            m_absasum;
   double            m_veps;
   bool              m_dbgskipconstraintnormalization;
   bool              m_havex;
   bool              m_qpbleicfirstcall;
   bool              m_sparseaupper;
   //--- objects
   CVIPMState        m_vsolver;
   CQQPSettings      m_qqpsettingsuser;
   CQQPBuffers       m_qqpbuf;
   CQPDenseAULSettings m_qpdenseaulsettingsuser;
   CQPDenseAULBuffers m_qpdenseaulbuf;
   CQPBLEICbuffers   m_qpbleicbuf;
   CQPBLEICSettings  m_qpbleicsettingsuser;
   CConvexQuadraticModel m_a;
   //--- arrays
   bool              m_havebndl[];
   bool              m_havebndu[];
   CRowInt           m_elagidx;
   CRowDouble        m_b;
   CRowDouble        m_bndl;
   CRowDouble        m_bndu;
   CRowDouble        m_cl;
   CRowDouble        m_cu;
   CRowDouble        m_effectives;
   CRowDouble        m_elaglc;
   CRowDouble        m_elagmlt;
   CRowDouble        m_replagbc;
   CRowDouble        m_replaglc;
   CRowDouble        m_s;
   CRowDouble        m_startx;
   CRowDouble        m_tmp0;
   CRowDouble        m_wrkbndl;
   CRowDouble        m_wrkbndu;
   CRowDouble        m_wrkcl;
   CRowDouble        m_wrkcu;
   CRowDouble        m_xorigin;
   CRowDouble        m_xs;
   //--- matrix
   CSparseMatrix     m_dummysparse;
   CSparseMatrix     m_sparsea;
   CSparseMatrix     m_sparsec;
   CSparseMatrix     m_wrksparsec;
   CMatrixDouble     m_densec;
   CMatrixDouble     m_dummyr2;
   CMatrixDouble     m_ecleic;
   CMatrixDouble     m_tmpr2;
   CMatrixDouble     m_wrkdensec;
   //--- constructor, destructor
                     CMinQPState(void);
                    ~CMinQPState(void) {}
   //--- copy
   void              Copy(const CMinQPState &obj);
   //--- overloading
   void              operator=(const CMinQPState &obj) { Copy(obj); }

  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CMinQPState::CMinQPState(void)
  {
   m_akind=0;
   m_algokind=0;
   m_mdense=0;
   m_msparse=0;
   m_n=0;
   m_repinneriterationscount=0;
   m_repncholesky=0;
   m_repnmv=0;
   m_repouteriterationscount=0;
   m_repterminationtype=0;
   m_stype=0;
   m_absamax=0;
   m_absasum2=0;
   m_absasum=0;
   m_veps=0;
   m_dbgskipconstraintnormalization=false;
   m_havex=false;
   m_qpbleicfirstcall=false;
   m_sparseaupper=false;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CMinQPState::Copy(const CMinQPState &obj)
  {
   m_akind=obj.m_akind;
   m_algokind=obj.m_algokind;
   m_mdense=obj.m_mdense;
   m_msparse=obj.m_msparse;
   m_n=obj.m_n;
   m_repinneriterationscount=obj.m_repinneriterationscount;
   m_repncholesky=obj.m_repncholesky;
   m_repnmv=obj.m_repnmv;
   m_repouteriterationscount=obj.m_repouteriterationscount;
   m_repterminationtype=obj.m_repterminationtype;
   m_stype=obj.m_stype;
   m_absamax=obj.m_absamax;
   m_absasum2=obj.m_absasum2;
   m_absasum=obj.m_absasum;
   m_veps=obj.m_veps;
   m_dbgskipconstraintnormalization=obj.m_dbgskipconstraintnormalization;
   ArrayCopy(m_havebndl,obj.m_havebndl);
   ArrayCopy(m_havebndu,obj.m_havebndu);
   m_havex=obj.m_havex;
   m_qpbleicfirstcall=obj.m_qpbleicfirstcall;
   m_sparseaupper=obj.m_sparseaupper;
   m_vsolver=obj.m_vsolver;
   m_dummysparse=obj.m_dummysparse;
   m_sparsea=obj.m_sparsea;
   m_sparsec=obj.m_sparsec;
   m_wrksparsec=obj.m_wrksparsec;
   m_elagidx=obj.m_elagidx;
   m_b=obj.m_b;
   m_bndl=obj.m_bndl;
   m_bndu=obj.m_bndu;
   m_cl=obj.m_cl;
   m_cu=obj.m_cu;
   m_effectives=obj.m_effectives;
   m_elaglc=obj.m_elaglc;
   m_elagmlt=obj.m_elagmlt;
   m_replagbc=obj.m_replagbc;
   m_replaglc=obj.m_replaglc;
   m_s=obj.m_s;
   m_startx=obj.m_startx;
   m_tmp0=obj.m_tmp0;
   m_wrkbndl=obj.m_wrkbndl;
   m_wrkbndu=obj.m_wrkbndu;
   m_wrkcl=obj.m_wrkcl;
   m_wrkcu=obj.m_wrkcu;
   m_xorigin=obj.m_xorigin;
   m_xs=obj.m_xs;
   m_qqpsettingsuser=obj.m_qqpsettingsuser;
   m_qqpbuf=obj.m_qqpbuf;
   m_qpdenseaulsettingsuser=obj.m_qpdenseaulsettingsuser;
   m_qpdenseaulbuf=obj.m_qpdenseaulbuf;
   m_qpbleicbuf=obj.m_qpbleicbuf;
   m_qpbleicsettingsuser=obj.m_qpbleicsettingsuser;
   m_densec=obj.m_densec;
   m_dummyr2=obj.m_dummyr2;
   m_ecleic=obj.m_ecleic;
   m_tmpr2=obj.m_tmpr2;
   m_wrkdensec=obj.m_wrkdensec;
   m_a=obj.m_a;
  }
//+------------------------------------------------------------------+
//| This object stores nonlinear optimizer State.                    |
//| You should use functions provided by MinQP subpackage to work    |
//| with this object                                                 |
//+------------------------------------------------------------------+
class CMinQPStateShell
  {
private:
   CMinQPState       m_innerobj;

public:
   //--- constructors, destructor
                     CMinQPStateShell(void) {}
                     CMinQPStateShell(CMinQPState &obj) { m_innerobj.Copy(obj); }
                    ~CMinQPStateShell(void) {}
   //--- method
   CMinQPState      *GetInnerObj(void) { return(GetPointer(m_innerobj)); }
  };
//+------------------------------------------------------------------+
//| This structure stores optimization report:                       |
//| * InnerIterationsCount      number of inner iterations           |
//| * OuterIterationsCount      number of outer iterations           |
//| * NCholesky                 number of Cholesky decomposition     |
//| * NMV                       number of matrix-vector products     |
//|                             (only products calculated as part of |
//|                             iterative process are counted)       |
//| * TerminationType           completion code (see below)          |
//| Completion codes:                                                |
//| * -5    inappropriate m_solver was used:                           |
//|         * Cholesky m_solver for semidefinite or indefinite problems|
//|         * Cholesky m_solver for problems with non-boundary         |
//|           constraints                                            |
//| * -3    inconsistent constraints (or, maybe, feasible point is   |
//|         too hard to find). If you are sure that constraints are  |
//|         feasible, try to restart optimizer with better initial   |
//|         approximation.                                           |
//| *  4    successful completion                                    |
//| *  5    MaxIts steps was taken                                   |
//| *  7    stopping conditions are too stringent,                   |
//|         further improvement is impossible,                       |
//|         X contains best point found so far.                      |
//+------------------------------------------------------------------+
class CMinQPReport
  {
public:
   //--- variables
   int               m_inneriterationscount;
   int               m_outeriterationscount;
   int               m_nmv;
   int               m_ncholesky;
   int               m_terminationtype;
   //--- arrays
   CRowDouble        m_lagbc;
   CRowDouble        m_laglc;
   //--- constructor, destructor
                     CMinQPReport(void);
                    ~CMinQPReport(void) {}
   //--- copy
   void              Copy(const CMinQPReport &obj);
   //--- overloading
   void              operator=(const CMinQPReport &obj) { Copy(obj); }

  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CMinQPReport::CMinQPReport(void)
  {
   m_inneriterationscount=0;
   m_outeriterationscount=0;
   m_nmv=0;
   m_ncholesky=0;
   m_terminationtype=0;
  }
//+------------------------------------------------------------------+
//| Copy                                                             |
//+------------------------------------------------------------------+
void CMinQPReport::Copy(const CMinQPReport &obj)
  {
//--- copy variables
   m_inneriterationscount=obj.m_inneriterationscount;
   m_outeriterationscount=obj.m_outeriterationscount;
   m_nmv=obj.m_nmv;
   m_ncholesky=obj.m_ncholesky;
   m_terminationtype=obj.m_terminationtype;
//--- copy arrays
   m_lagbc=obj.m_lagbc;
   m_laglc=obj.m_laglc;
  }
//+------------------------------------------------------------------+
//| This structure stores optimization report:                       |
//| * InnerIterationsCount      number of inner iterations           |
//| * OuterIterationsCount      number of outer iterations           |
//| * NCholesky                 number of Cholesky decomposition     |
//| * NMV                       number of matrix-vector products     |
//|                             (only products calculated as part of |
//|                             iterative process are counted)       |
//| * TerminationType           completion code (see below)          |
//| Completion codes:                                                |
//| * -5    inappropriate m_solver was used:                           |
//|         * Cholesky m_solver for semidefinite or indefinite problems|
//|         * Cholesky m_solver for problems with non-boundary         |
//|           constraints                                            |
//| * -3    inconsistent constraints (or, maybe, feasible point is   |
//|         too hard to find). If you are sure that constraints are  |
//|         feasible, try to restart optimizer with better initial   |
//|         approximation.                                           |
//| *  4    successful completion                                    |
//| *  5    MaxIts steps was taken                                   |
//| *  7    stopping conditions are too stringent,                   |
//|         further improvement is impossible,                       |
//|         X contains best point found so far.                      |
//+------------------------------------------------------------------+
class CMinQPReportShell
  {
private:
   CMinQPReport      m_innerobj;

public:
   //--- constructors, destructor
                     CMinQPReportShell(void) {}
                     CMinQPReportShell(CMinQPReport &obj) { m_innerobj.Copy(obj); }
                    ~CMinQPReportShell(void) {}
   //--- methods
   int               GetInnerIterationsCount(void);
   void              SetInnerIterationsCount(const int i);
   int               GetOuterIterationsCount(void);
   void              SetOuterIterationsCount(const int i);
   int               GetNMV(void);
   void              SetNMV(const int i);
   int               GetNCholesky(void);
   void              SetNCholesky(const int i);
   int               GetTerminationType(void);
   void              SetTerminationType(const int i);
   CMinQPReport     *GetInnerObj(void);
  };
//+------------------------------------------------------------------+
//| Returns the value of the variable inneriterationscount           |
//+------------------------------------------------------------------+
int CMinQPReportShell::GetInnerIterationsCount(void)
  {
   return(m_innerobj.m_inneriterationscount);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable inneriterationscount          |
//+------------------------------------------------------------------+
void CMinQPReportShell::SetInnerIterationsCount(const int i)
  {
   m_innerobj.m_inneriterationscount=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable outeriterationscount           |
//+------------------------------------------------------------------+
int CMinQPReportShell::GetOuterIterationsCount(void)
  {
   return(m_innerobj.m_outeriterationscount);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable outeriterationscount          |
//+------------------------------------------------------------------+
void CMinQPReportShell::SetOuterIterationsCount(const int i)
  {
   m_innerobj.m_outeriterationscount=i;
  }
//+------------------------------------------------------------------+
//| Returns the value of the variable nmv                            |
//+------------------------------------------------------------------+
int CMinQPReportShell::GetNMV(void)
  {
   return(m_innerobj.m_nmv);
  }
//+------------------------------------------------------------------+
//| Changing the value of the variable nmv                           |
//+------------------------------------------------------------------+
void CMinQPReportShel