//+------------------------------------------------------------------+
//|                                                    TimeShift.mq5 |
//|                                    Copyright (c) 2020, Marketeer |
//|                           https://www.mql5.com/ru/articles/8226/ |
//+------------------------------------------------------------------+
#property copyright "Copyright © 2020, Marketeer"
#property link      "https://www.mql5.com/en/users/marketeer"
#property version   "1.0"
#property description "Non-trading expert generating a custom symbol with specified time shift.\n"
#property description "Reshape quotes into alternative bar patterns ahead of time."


// I N C L U D E S

#include <Symbol.mqh>


// I N P U T S

input int Shift = 600; // Shift (seconds)
input bool Reset = false;
input bool EmulateTicks = true;
input ENUM_TIMEFRAMES DefaultTimeframe = PERIOD_CURRENT;
input datetime Start = D'2020.07.01';

ulong startMsc;
const ulong dayLong = PeriodSeconds(PERIOD_D1) * 1000;


// G L O B A L S

string symbolName;
bool firstRun;
bool stopAll;
bool justCreated;
ulong lastTick;

MqlRates rates[];


// A P P L I C A T I O N

bool reset()
{
  int size;
  do
  {
    ResetLastError();
    int deleted = CustomRatesDelete(symbolName, 0, LONG_MAX);
    int err = GetLastError();
    if(err != ERR_SUCCESS)
    {
      Alert("CustomRatesDelete failed, ", err);
      return false;
    }
    else
    {
      Print("Rates deleted: ", deleted);
    }
    
    ResetLastError();
    deleted = CustomTicksDelete(symbolName, 0, LONG_MAX);
    if(deleted == -1)
    {
      Print("CustomTicksDelete failed ", GetLastError());
      return false;
    }
    else
    {
      Print("Ticks deleted: ", deleted);
    }
  
    // wait for changes to take effect
    Sleep(1000);

    MqlTick array[];
    size = CopyTicks(symbolName, array, COPY_TICKS_ALL, 0, 10);
    Print("Remaining ticks: ", size);
  } while(size > 0 && !IsStopped());
  
  return size > -1;
}


int fillArray(ulong &_start)
{
  MqlTick array[];
  const ulong _stop = _start + dayLong - 1;
  int size = CopyTicksRange(_Symbol, array, COPY_TICKS_ALL, _start, _stop);
  if(size == -1)
  {
    Print("CopyTicks failed: ", GetLastError());
    stopAll = true;
  }
  else
  {
    if(size > 0)
    {
      lastTick = array[size - 1].time_msc;

      Comment("Time: ", array[0].time, "'", array[0].time_msc % 1000, " - ", array[size - 1].time, "'", array[size - 1].time_msc % 1000);
      ChartRedraw();
      for(int i = 0; i < size; i++)
      {
        array[i].time += Shift;
        array[i].time_msc += Shift * 1000;
      }
      if(CustomTicksReplace(symbolName, _start, _stop, array) == -1)
      {
        Print("Tick replace error:", GetLastError());
        stopAll = true;
      }
    }

    _start += dayLong;
  }
  return size;
}

void add()
{
  MqlTick array[];
  int size = CopyTicksRange(_Symbol, array, COPY_TICKS_ALL, lastTick + 1, LONG_MAX);
  if(size > 0)
  {
    lastTick = array[size - 1].time_msc;
    for(int i = 0; i < size; i++)
    {
      array[i].time += Shift;
      array[i].time_msc += Shift * 1000;
    }
    if(CustomTicksAdd(symbolName, array) == -1)
    {
      Print("Tick error: ", GetLastError());
      ArrayPrint(array);
      stopAll = true;
    }
  }
}


// E V E N T   H A N D L E R S

int OnInit(void)
{
  stopAll = false;
  justCreated = false;
  lastTick = 0;
  
  if(SymbolInfoInteger(_Symbol, SYMBOL_CUSTOM))
  {
    Alert("" + _Symbol + " is a custom symbol. Only built-in symbol can be used as a host.");
    return INIT_FAILED;
  }
  
  symbolName = _Symbol + "_shift_" + (string)Shift;

  if(!SymbolSelect(symbolName, true))
  {
    const SYMBOL Symb(symbolName);
    Symb.CloneProperties(_Symbol);
    justCreated = true;
    
    if(!SymbolSelect(symbolName, true))
    {
      Alert("Can't select symbol:", symbolName, " err:", GetLastError());
      return INIT_FAILED;
    }
  }
  
  EventSetTimer(1);
  firstRun = true;
  
  return INIT_SUCCEEDED;
}

void OnTimer()
{
  OnTick();
}

void OnTick(void)
{
  if(stopAll) return;
  
  if(firstRun)
  {
    if(!TerminalInfoInteger(TERMINAL_CONNECTED))
    {
      Print("Waiting for connection...");
      return;
    }
    if(!SymbolIsSynchronized(_Symbol))
    {
      Print("Unsynchronized, skipping ticks...");
      return;
    }
    EventKillTimer();

    if(Reset)
    {
      stopAll = !reset();
      if(stopAll) return;
    }
    
    const datetime existing = iTime(symbolName, PERIOD_D1, 0);
    Print("Filling ", symbolName);
    Print("Last day: ", (string)existing, ", requested start: ", (string)Start);
    startMsc = (ulong)MathMax(Start, existing) * 1000;
    int daycount = 0;
    while(fillArray(startMsc) >= 0 && (++daycount > 0) && startMsc <= (ulong)TimeCurrent() * 1000 && !stopAll);
    printf("Filled %d days in %s", daycount, symbolName);
    Comment("");
    
    if(justCreated)
    {
      long id = ChartOpen(symbolName, DefaultTimeframe);
      if(id == 0)
      {
        Alert("Can't open new chart for ", symbolName, ", code: ", GetLastError());
      }
      else
      {
        Sleep(1000);
        ChartSetSymbolPeriod(id, symbolName, DefaultTimeframe);
        ChartSetInteger(id, CHART_MODE, CHART_CANDLES);
      }
      justCreated = false;
    }
    
    firstRun = false;
    return;
  }
  
  if(EmulateTicks)
  {
    add();
  }
}

void OnDeinit(const int reason)
{
}
