//+------------------------------------------------------------------+
//|                                              EqualVolumeBars.mq5 |
//|                                 Copyright © 2008-2020, komposter |
//|                                 Copyright © 2018-2020, Marketeer |
//|                           https://www.mql5.com/ru/articles/8226/ |
//+------------------------------------------------------------------+
#property copyright "Copyright © 2018-2020, Marketeer"
#property link "https://www.mql5.com/en/users/marketeer"
#property description "Non-trading EA generating equivolume and/or range bars as a custom symbol.\n"
#property description "This is a port of MT4 offline chart generator to MT5.\n"
#property description "Original MT4 EqualVolumeBars - https://www.mql5.com/en/code/7737\n"
#property version "6.0"

#include <Symbol.mqh>


#define TICKS_ARRAY 10000

// RangeBars are available for real ticks only (FromM1 = false)
enum mode
{
  EqualTickVolumes = 0,
  EqualRealVolumes = 1,
  RangeBars = 2
};

input mode WorkMode = EqualTickVolumes;
input int TicksInBar = 100;
input bool FromM1 = true;
input datetime StartDate = 0;
input string CustomPrefix = "";

datetime time, now_time;
double now_close, now_open, now_low, now_high;
long now_volume, now_real;

int pre_time, last_fpos = 0;
double pre_close;

string prefix = "";
string symbolName;
bool justCreated;

bool Reset()
{
  int size;
  do
  {
    ResetLastError();
    int deleted = CustomRatesDelete(symbolName, 0, LONG_MAX);
    int err = GetLastError();
    if(err != ERR_SUCCESS)
    {
      Alert("CustomRatesDelete failed, ", err);
      return false;
    }
    else
    {
      Print("Rates deleted: ", deleted);
    }
  
    ResetLastError();
    deleted = CustomTicksDelete(symbolName, 0, LONG_MAX);
    if(deleted == -1)
    {
      Print("CustomTicksDelete failed ", GetLastError());
      return false;
    }
    else
    {
      Print("Ticks deleted: ", deleted);
    }
    
    // wait for changes to take effect in the core threads
    Sleep(1000);

    MqlTick _array[];
    size = CopyTicks(symbolName, _array, COPY_TICKS_ALL, 0, 10);
    Print("Remaining ticks: ", size);
  } while (size > 0 && !IsStopped());
  // NB. this can not work everytime as expected
  // if getting ERR_CUSTOM_TICKS_WRONG_ORDER or similar error - the last resort
  // is to wipe out the custom symbol manually from GUI, and then restart this EA

  return size > -1; // success
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int OnInit()
{
  int cnt_bars = 0;
  justCreated = false;
  
  if(WorkMode == RangeBars && FromM1 == true)
  {
    Print("Range bars can be created by ticks only, FromM1 should be false");
    return INIT_PARAMETERS_INCORRECT;
  }

  if(CustomPrefix != "")
  {
    prefix = CustomPrefix;
  }
  else
  {
    if(WorkMode == EqualTickVolumes)
      prefix = "_Eqv" + (string)TicksInBar;
    else if(WorkMode == EqualRealVolumes)
      prefix = "_Qrv" + (string)TicksInBar;
    else
      prefix = "_Rng" + (string)TicksInBar;
  }

  // create custom symbol
  symbolName = prefix + _Symbol;
  if(!SymbolSelect(symbolName, true))
  {
    Print("Creating \"", symbolName, "\"");
    const SYMBOL Symb(symbolName);
    Symb.CloneProperties(_Symbol);
    justCreated = true;

    if(!SymbolSelect(symbolName, true))
    {
      Alert("Can't select symbol:", symbolName, " err:", GetLastError());
      return INIT_FAILED;
    }
  }
  else
  {
    Print("Resetting \"", prefix, _Symbol, "\"");
    if(!Reset()) return INIT_FAILED;
  }

  //+------------------------------------------------------------------+
  //| Process history
  //+------------------------------------------------------------------+
  ulong cursor = StartDate * 1000;

  double bid;
  now_time = 60;
  now_close = 0;
  now_open = 0;
  now_low = 0;
  now_high = 0;
  now_volume = 0;
  now_real = 0;
  
  if(!FromM1)
  {
    Print("Processing tick history...");
    TicksBuffer tb;
    
    while(tb.fill(cursor, true) && !IsStopped())
    {
      MqlTick t;
      while(tb.read(t))
      {
        time = t.time;
        bid = t.last != 0 ? t.last : t.bid;
        if(bid == 0) continue; // skip bad ticks
        
        // TODO: check bid against bar high-low range, skip prices outside the bar

        // New or first bar
        if(IsNewBar() || now_volume < 1)
        {
          while(IsNewBar())
          {
            if(WorkMode == RangeBars)
            {
              FixRange();
            }
            WriteToFile(now_time, now_open, now_low, now_high, now_close,
              WorkMode == EqualTickVolumes ? TicksInBar : now_volume,
              WorkMode == EqualRealVolumes ? TicksInBar : now_real);
            cnt_bars++;
            
            if((cnt_bars % 1000) == 0)
            {
              Comment(time, " -> ", now_time);
            }
            if(WorkMode == EqualTickVolumes) now_volume -= TicksInBar;
            if(WorkMode == EqualRealVolumes) now_real -= TicksInBar;
            if(WorkMode == RangeBars) break;
            now_time += 60;
          }

          // Normalize down to a minute
          time = time / 60;
          time *= 60;

          // Eliminate 2 bars with the same name
          if(time <= now_time) time = now_time + 60;

          now_time = time;
          now_open = bid;
          now_low = bid;
          now_high = bid;
          now_close = bid;
          now_volume = 1;
          now_real += (long)t.volume;
        }
        else
        {
          if(bid < now_low) now_low = bid;
          if(bid > now_high) now_high = bid;
          now_close = bid;
          now_volume++;
          now_real += (long)t.volume;
        }
      }
    }
    Comment("");
  }
  // FromM1
  else
  {
    Print("Processing M1 bars history...");
    const int n = iBarShift(_Symbol, PERIOD_M1, StartDate);
    for(int i = n - 1; i >= 0 && !IsStopped(); i--)
    {
      now_time = iTime(_Symbol, PERIOD_M1, i);

      // New or first bar
      if(IsNewBar() || now_volume < 1)
      {
        time = iTime(_Symbol, PERIOD_M1, i);

        while(IsNewBar())
        {
          WriteToFile(now_time, now_open, now_low, now_high, now_close,
            WorkMode == EqualTickVolumes ? TicksInBar : now_volume,
            WorkMode == EqualRealVolumes ? TicksInBar : now_real);
          cnt_bars++;
          if((cnt_bars % 1000) == 0)
          {
            Comment(time, " -> ", now_time, " ", i, "/", n);
          }
          if(WorkMode == EqualTickVolumes) now_volume -= TicksInBar;
          if(WorkMode == EqualRealVolumes) now_real -= TicksInBar;
          now_time += 60;
        }

        // Eliminate 2 bars with the same name
        if(time <= now_time) time = now_time + 60;

        now_time = time;
        now_open = iOpen(_Symbol, PERIOD_M1, i);
        now_low = iLow(_Symbol, PERIOD_M1, i);
        now_high = iHigh(_Symbol, PERIOD_M1, i);
        now_close = iClose(_Symbol, PERIOD_M1, i);
        if(WorkMode != EqualTickVolumes) now_volume = 0;
        now_volume += iVolume(_Symbol, PERIOD_M1, i);
        if(WorkMode != EqualRealVolumes) now_real = 0;
        now_real += iRealVolume(_Symbol, PERIOD_M1, i);

        while(i == 0 && IsNewBar()) // now_volume can overflow and this is not handled by the while loop above if it's the last bar
        {
          WriteToFile(now_time, now_open, now_low, now_high, now_close,
            WorkMode == EqualTickVolumes ? TicksInBar : now_volume,
            WorkMode == EqualRealVolumes ? TicksInBar : now_real);
          cnt_bars++;
          if(WorkMode == EqualTickVolumes) now_volume -= TicksInBar;
          if(WorkMode == EqualRealVolumes) now_real -= TicksInBar;
          now_time += 60;
        }
      }
      else
      {
        if(iLow(_Symbol, PERIOD_M1, i) < now_low) now_low = iLow(_Symbol, PERIOD_M1, i);
        if(iHigh(_Symbol, PERIOD_M1, i) > now_high) now_high = iHigh(_Symbol, PERIOD_M1, i);
        now_close = iClose(_Symbol, PERIOD_M1, i);
        now_volume += iVolume(_Symbol, PERIOD_M1, i);
        now_real += iRealVolume(_Symbol, PERIOD_M1, i);
      }
    }
    Comment("");
  }
  
  if(IsStopped())
  {
    Print("Interrupted. The custom symbol data is inconsistent - please, delete");
    return INIT_FAILED;
  }

  Print("Bar 0: ", now_time, " ", now_volume, " ", now_real);
  // write bar 0 to chart
  WriteToFile(now_time, now_open, now_low, now_high, now_close, now_volume, now_real);

  // show stats
  Print(cnt_bars, " bars written");
  Print("Open \"", prefix, _Symbol, "\" chart to view results");

  OpenCustomChart();

  RefreshWindow(now_time);

  return INIT_SUCCEEDED;
}

class TicksBuffer
{
  private:
    MqlTick array[];
    int tick;
  
  public:
    bool fill(ulong &cursor, const bool history = false)
    {
      int size = history ? CopyTicks(_Symbol, array, COPY_TICKS_ALL, cursor, TICKS_ARRAY) : CopyTicksRange(_Symbol, array, COPY_TICKS_ALL, cursor);
      if(size == -1)
      {
        Print("CopyTicks failed: ", GetLastError());
        return false;
      }
      else if(size == 0)
      {
        if(history) Print("End of CopyTicks at ", (datetime)(cursor / 1000));
        return false;
      }
      
      if((ulong)array[0].time_msc < cursor)
      {
        Print("Tick rewind bug, ", (datetime)(cursor / 1000));
        return false;
      }
      
      cursor = array[size - 1].time_msc + 1;
      tick = 0;
    
      return true;
    }
    
    bool read(MqlTick &t)
    {
      if(tick < ArraySize(array))
      {
        t = array[tick++];
        return true;
      }
      return false;
    }
};

void OpenCustomChart()
{
  if(justCreated)
  {
    long id = ChartOpen(symbolName, PERIOD_M1);
    if(id == 0)
    {
      Alert("Can't open new chart for ", symbolName, ", code: ", GetLastError());
    }
    else
    {
      Sleep(1000);
      ChartSetSymbolPeriod(id, symbolName, PERIOD_M1);
      ChartSetInteger(id, CHART_MODE, CHART_CANDLES);
    }
    justCreated = false;
  }
}

void OnDeinit(const int reason)
{
  Comment("");
}

void OnTick()
{
  static ulong cursor = 0;
  MqlTick t;
  
  if(cursor == 0)
  {
    if(SymbolInfoTick(_Symbol, t))
    {
      HandleTick(t);
      cursor = t.time_msc + 1;
    }
  }
  else
  {
    TicksBuffer tb;
    while(tb.fill(cursor))
    {
      while(tb.read(t))
      {
        HandleTick(t);
      }
    }
  }

  RefreshWindow(now_time);
}

//+------------------------------------------------------------------+
//| Process incoming ticks one by one
//+------------------------------------------------------------------+
void HandleTick(const MqlTick &t)
{
  now_volume++;
  now_real += (long)t.volume;

  if(!IsNewBar()) // bar continues
  {
    if(t.bid < now_low) now_low = t.bid;
    if(t.bid > now_high) now_high = t.bid;
    now_close = t.bid;
    
    // write bar 0 to chart (-1 for volume stands for upcoming refresh)
    WriteToFile(now_time, now_open, now_low, now_high, now_close, now_volume - 1, now_real);
  }
  else // new bar tick
  {
    Comment("New bar ", now_time);
    
    if(WorkMode == RangeBars)
    {
      FixRange();
    }
    // write bar 1
    WriteToFile(now_time, now_open, now_low, now_high, now_close,
      WorkMode == EqualTickVolumes ? TicksInBar : now_volume,
      WorkMode == EqualRealVolumes ? TicksInBar : now_real);

    // normalize down to a minute
    time = iTime(_Symbol, PERIOD_M1, 0) / 60;
    time *= 60;

    // Eliminate 2 bars with the same name
    while(time <= now_time) time = now_time + 60;

    now_time = time;
    now_open = t.bid;
    now_low = t.bid;
    now_high = t.bid;
    now_close = t.bid;
    if(WorkMode == EqualTickVolumes) now_volume = 1;
    if(WorkMode == EqualRealVolumes) now_real -= TicksInBar;

    // write bar 0 (-1 for volume stands for upcoming refresh)
    WriteToFile(now_time, now_open, now_low, now_high, now_close, now_volume - 1, now_real);
  }
}

void RefreshWindow(const datetime t)
{
  MqlTick ta[1];
  SymbolInfoTick(_Symbol, ta[0]);
  ta[0].time = t;
  ta[0].time_msc = ta[0].time * 1000;
  if(CustomTicksAdd(symbolName, ta) == -1) // NB! this call may increment number of ticks per bar
  {
    Print("CustomTicksAdd failed:", GetLastError(), " ", (long) ta[0].time);
    ArrayPrint(ta);
  }
}

void WriteToFile(datetime t, double o, double l, double h, double c, long v, long m = 0)
{
  MqlRates r[1];

  r[0].time = t;
  r[0].open = o;
  r[0].low = l;
  r[0].high = h;
  r[0].close = c;
  r[0].tick_volume = v;
  r[0].spread = 0;
  r[0].real_volume = m;

  int code = CustomRatesUpdate(symbolName, r);
  if(code < 1)
  {
    Print("CustomRatesUpdate failed: ", GetLastError());
  }
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool IsNewBar()
{
  if(WorkMode == EqualTickVolumes)
  {
    if(now_volume > TicksInBar) return true;
  }
  else if(WorkMode == EqualRealVolumes)
  {
    if(now_real > TicksInBar) return true;
  }
  else if(WorkMode == RangeBars)
  {
    if((now_high - now_low) / _Point > TicksInBar) return true;
  }

  return false;
}

void FixRange()
{
  const int excess = (int)((now_high + (_Point / 2)) / _Point) - (int)((now_low + (_Point / 2)) / _Point) - TicksInBar;
  if(excess > 0)
  {
    if(now_close > now_open)
    {
      now_high -= excess * _Point;
      if(now_high < now_close) now_close = now_high;
    }
    else if(now_close < now_open)
    {
      now_low += excess * _Point;
      if(now_low > now_close) now_close = now_low;
    }
  }
}