//-----------------------------------------------------------------------------------
//                                                                      RNDXor128.mqh
//                                                                      2011, victorg
//                                                                http://www.mql5.com
//-----------------------------------------------------------------------------------
#property copyright "2011, victorg"
#property link      "http://www.mql5.com"

#include <Object.mqh>

//-----------------------------------------------------------------------------------
// Generation of pseudo-random sequences. The algorithm Xorshift RNG (George Marsaglia)
// with (2**128)-1 period of generated sequence is used.
//        uint rand_xor128()
//          {
//          static uint x=123456789,y=362436069,z=521288629,w=88675123;
//          uint t=(x^(x<<11));x=y;y=z;z=w;
//          return(w=(w^(w>>19))^(t^(t>>8)));
//          }
// :
//  Rand()      - even distribution within the range [0,UINT_MAX=4294967295].
//  Rand_01()   - even distribution withing the range [0,1].
//  Rand_Norm() - normal distribution with zero mean and one dispersion.
//  Rand_Exp()  - exponential distribution with parameter 1.0.
//  Rand_Laplace() - Laplace distribution with parameter 1.0
//  Reset()     - resetting all basic values to their initial state.
//  SRand()     - setting new initial values of the generator.
//-----------------------------------------------------------------------------------
#define xor32  xx=xx^(xx<<13);xx=xx^(xx>>17);xx=xx^(xx<<5)
#define xor128 t=(x^(x<<11));x=y;y=z;z=w;w=(w^(w>>19))^(t^(t>>8))
#define inidat x=123456789;y=362436069;z=521288629;w=88675123;xx=2463534242

class RNDXor128:public CObject
  {
protected:
  uint      x,y,z,w,xx,t;
  uint      UINT_half;
public:
            RNDXor128()       {UINT_half=UINT_MAX>>1;inidat;};
//-----------------------------------------------------------------------------------
// Returns one integral number number from a pseudo-random sequence.
// INPUT:  no.
// OUTPUT: integral number in the double format.
// REMARK: Even law of distribution within the range [0,UINT_MAX=4294967295]
//-----------------------------------------------------------------------------------
  double    Rand()            {xor128;return((double)w);};
//-----------------------------------------------------------------------------------
// Fills the array with a pseudo-random sequence of integral numbers.
// INPUT:  a - array for placing the sequence.
//         n - number of generated elements of the sequence.
// OUTPUT: 0  no errors.
//        -1  error: n value is less than 1.
//        -2  error: array size is less than n.
// REMARK: Even law of distribution within the range [0,UINT_MAX=4294967295]
//-----------------------------------------------------------------------------------
  int       Rand(double& a[],int n)
                              {int i;if(n<1)return(-1);
                               if(ArraySize(a)<n)return(-2);
                               for(i=0;i<n;i++){xor128;a[i]=(double)w;}
                               return(0);};
//-----------------------------------------------------------------------------------
// Returns one value from the pseudo-random sequence.
// INPUT:  no.
// OUTPUT: number in the double format.
// REMARK: Even law of distribution withing the range [0,1].
//-----------------------------------------------------------------------------------
  double    Rand_01()         {xor128;return((double)w/UINT_MAX);};
//-----------------------------------------------------------------------------------
// Fills the array with values of a pseudo-random sequence.
// INPUT:  a - array for placing the sequence.
//         n - number of generated elements of the sequence.
// OUTPUT: 0  no errors.
//        -1  error: n value is less than 1.
//        -2  error: array size is less than n.
// REMARK: Even law of distribution withing the range [0,1].
//-----------------------------------------------------------------------------------
  int       Rand_01(double& a[],int n)
                              {int i;if(n<1)return(-1);
                               if(ArraySize(a)<n)return(-2);
                               for(i=0;i<n;i++){xor128;a[i]=(double)w/UINT_MAX;}
                               return(0);};
//-----------------------------------------------------------------------------------
// Returns one value from the pseudo-random sequence.
// INPUT:  no.
// OUTPUT: number in the double format.
// REMARK: Normal law of distribution with zero mean and one dispersion.
//         To get a sequence with normal law of distribution, 
//         the Box-Muller transformation is used.
//-----------------------------------------------------------------------------------
  double    Rand_Norm()       {double v1,v2,s,sln;static double ra;static uint b=0;
                               if(b==w){b=0;return(ra);}
                               do{
                                 xor128;v1=(double)w/UINT_half-1.0;
                                 xor128;v2=(double)w/UINT_half-1.0;
                                 s=v1*v1+v2*v2;
                                 }
                               while(s>=1.0||s==0.0);
                               sln=MathLog(s);sln=MathSqrt((-sln-sln)/s);
                               ra=v2*sln;b=w;
                               return(v1*sln);};
//-----------------------------------------------------------------------------------
// Fills the array with values of a pseudo-random sequence.
// (Box-Muller transform).
// INPUT:  a - array for placing the sequence.
//         n - number of generated elements of the sequence.
// OUTPUT: 0  no errors.
//        -1  error: n value is less than 1.
//        -2  error: array size is less than n.
// REMARK: Normal law of distribution with zero mean and one dispersion.
//         To get a sequence with normal law of distribution, 
//         the Box-Muller transformation is used.
//-----------------------------------------------------------------------------------
  int       Rand_Norm(double& a[],int n)
                              {int i;if(n<1)return(-1);
                               if(ArraySize(a)<n)return(-2);
                               for(i=0;i<n;i++)a[i]=Rand_Norm();
                               return(0);};
//-----------------------------------------------------------------------------------
// Returns one value from the pseudo-random sequence.
// INPUT:  no.
// OUTPUT: number in the double format.
// REMARK: Exponential law of distribution with parameter 1.0.
//-----------------------------------------------------------------------------------
  double    Rand_Exp()        {xor128;if(w==0)return(DBL_MAX);
                               return(-MathLog((double)w/UINT_MAX));};
//-----------------------------------------------------------------------------------
// Fills the array with values of a pseudo-random sequence.
// INPUT:  a - array for placing the sequence.
//         n - number of generated elements of the sequence.
// OUTPUT: 0  no errors.
//        -1  error: n value is less than 1.
//        -2  error: array size is less than n.
// REMARK: Exponential law of distribution with parameter 1.0.
//-----------------------------------------------------------------------------------
  int       Rand_Exp(double& a[],int n)
                              {int i;if(n<1)return(-1);
                               if(ArraySize(a)<n)return(-2);
                               for(i=0;i<n;i++)a[i]=Rand_Exp();
                               return(0);};
//-----------------------------------------------------------------------------------
// Returns one value from the pseudo-random sequence. with
// Laplace distribution (double exponential) with parameter 1.0.  
// INPUT:  no.
// OUTPUT: number in the double format.
// REMARK: Laplace distribution (double exponential) with parameter 1.0.  
//-----------------------------------------------------------------------------------
  double    Rand_Laplace()    {double a;xor128;
                              a=(double)w/UINT_half;
                              if(w>UINT_half)
                                {a=2.0-a;
                                if(a==0.0)return(-DBL_MAX);
                                return(MathLog(a));}
                              else
                                {if(a==0.0)return(DBL_MAX);
                                return(-MathLog(a));}};
//-----------------------------------------------------------------------------------
// Fills the array with values of a pseudo-random sequence.
// INPUT:  a - array for placing the sequence.
//         n - number of generated elements of the sequence.
// OUTPUT: 0  no errors.
//        -1  error: n value is less than 1.
//        -2  error: array size is less than n.
// REMARK: Laplace distribution (double exponential) with parameter 1.0.
//-----------------------------------------------------------------------------------
  int       Rand_Laplace(double& a[],int n)
                              {int i;if(n<1)return(-1);
                               if(ArraySize(a)<n)return(-2);
                               for(i=0;i<n;i++)a[i]=Rand_Laplace();
                               return(0);};
//-----------------------------------------------------------------------------------
// Resetting the generator if a pseudo-random sequence to initial state.
// INPUT:  no.
// OUTPUT: no.
// REMARK: For the generator of random number to work, four static
//         variables of the uint type are required. After calling the generator, they are used for saving
//         values that are used for generating of the next
//         random number. When creating an instance of the RNDXor128 class, those variables
//         are assigned with initial values. Call of the Reset() functions restores
//         those initial values resetting the generator to initial state.  
//-----------------------------------------------------------------------------------
  void      Reset()           {inidat;};
//-----------------------------------------------------------------------------------
// Generates and set the initial values for the generator of random numbers.
// INPUT:  seed - a random number from the range [0,UINT_MAX=4294967295].
// OUTPUT: no.
// REMARK: Using a value of input parameter is generates and sets
//         all four initial values for the generator of random numbers. When seed=0
//         the function randomly changes the initial value.
//-----------------------------------------------------------------------------------
  void      SRand(uint seed)  {int i;if(seed!=0)xx=seed;
                               for(i=0;i<16;i++){xor32;}
                               xor32;x=xx;xor32;y=xx;
                               xor32;z=xx;xor32;w=xx;
                               for(i=0;i<16;i++){xor128;}};
//-----------------------------------------------------------------------------------
// Sets initial values for the generator of random numbers.
// INPUT:  xs - value of the variable x from the range [0,UINT_MAX=4294967295].
//         ys - value of the variable y from the range [0,UINT_MAX=4294967295].
//         zs - value of the variable z from the range [0,UINT_MAX=4294967295].
//         ws - value of the variable w from the range [0,UINT_MAX=4294967295].
// OUTPUT: 0  no errors.
//        -1  error: all input parameters are equal to zero.
// REMARK: All four input parameters cannot have zero
//         values.
//-----------------------------------------------------------------------------------
  int       SRand(uint xs,uint ys,uint zs,uint ws)
                              {int i;if(xs==0&&ys==0&&zs==0&&ws==0)return(-1);
                               x=xs;y=ys;z=zs;w=ws;
                               for(i=0;i<16;i++){xor128;}
                               return(0);};
  };
//-----------------------------------------------------------------------------------
