//+------------------------------------------------------------------+
//|                                                       		Trace |
//|                             Copyright 2006-2011, www.FXmaster.de |
//|                                                  www.FXmaster.de |
//+------------------------------------------------------------------+
#property copyright "Copyright 2006-2011, www.FXmaster.de"
#property link      "www.FXmaster.de"

#include "Tree.mqh"

class CTraceView; // provisional declaration
//------------------------------------------------------------------	class CTraceCtrl
class CTraceCtrl
{
public:
	CTreeCtrl *m_stack; // object of graph
	CTreeCtrl *m_prop; // object of the list of properties
	CTreeCtrl *m_file; // grouping by files
	CTreeCtrl *m_class; // grouping by classes
	CTraceView* m_traceview; // pointer to view of the class
	CNode* m_cur; // pointer to current node

public:
	CTraceCtrl() { Create(); Reset(); } // created tracer
	~CTraceCtrl() { delete m_stack; delete m_prop; delete m_file; delete m_class; } // deleted tracer
	void Create(); // created tracer
	void In(string apath, string afile, int aline, string atext, string adesc, int aid); // entering a specified node 
	void Out(int aid); // exit before a specified node
	bool StepBack(); // exit from a node one step higher (moving to the parent)
	void Reset() { m_cur=m_stack.m_root; m_stack.Reset(); m_file.Reset(); m_class.Reset(); } // resetting all nodes
	void Clear() { m_cur=m_stack.m_root; m_stack.Clear(); m_file.Clear(); m_class.Clear(); } // resetting all nodes

public:
	void AddWatch(string name, string val); // checking the debugging mode for a node
	void Break(); // break for a node
};
//------------------------------------------------------------------	CTraceCtrl
void CTraceCtrl::Create()
{
	m_stack=new CTreeCtrl; m_stack.m_root.m_text="TRACE"; m_stack.m_mode=1; // created object of graph
	m_prop=new CTreeCtrl; m_prop.m_root.m_text="INFO"; // created object of graph
	m_file=new CTreeCtrl; m_file.m_root.m_text="FILE"; // created object of graph
	m_class=new CTreeCtrl; m_class.m_root.m_text="CLASS"; //   
	m_cur=m_stack.m_root; // set pointer to the root
	m_stack.m_bBreak=true;
}
//------------------------------------------------------------------	In
void CTraceCtrl::In(string apath, string afile, int aline, string atext, string adesc, int aid)
{	
	if (NIL(m_stack) || NIL(m_cur)) return; // validation of pointer
	if (aid>=0) Out(aid); // if a required node is set, go to it
	// check names of already existing nodes 
	int i, size=ArraySize(m_cur.m_next); // get the list of branches of the node
	bool b=false, ident;
	for (i=0; i<size; i++) // search for the required name in the branches of the current node
	{
		if (NIL(m_cur.m_next[i])) continue; // validity of pointer
		if (m_cur.m_next[i].m_text==atext) { b=true; break; } // stop search if it is found
	}
	ident=(m_cur.m_file==afile && m_cur.m_line==aline && m_cur.m_path==apath);
	
	if (b) 
	{
		m_cur=m_cur.m_next[i]; // move to the next node
		m_cur.m_desc=adesc; // set attribute
	}
	else if (!ident)// if the node is not found, create the a new one
	{	
		ArrayResize(m_cur.m_next, size+1); // extended the array
		m_cur.m_next[size]=new CNode;  // created the node
		if (NIL(m_cur.m_next[size])) return; // checked
		m_stack.m_maxid++; // increased the global counter
		m_cur.m_next[size].m_prev=m_cur; // moved the pointer to the parent
		m_cur=m_cur.m_next[size]; // moved the current pointer to the new node
		m_cur.m_id=m_stack.m_maxid; m_cur.m_text=atext; 
		
		// set attributes
		string afunc=atext, aclass="__base__"; int k=StringFind(atext, "::");
		if (k>0) { aclass=StringSubstr(atext, 0, k); afunc=StringSubstr(atext, k+2); }
		m_cur.m_path=apath; m_cur.m_file=afile; m_cur.m_line=aline;
		m_cur.m_func=afunc; m_cur.m_class=aclass; m_cur.m_desc=adesc;
		
		// group in trees
		m_stack.SortBy(TG_TEXT, true);
		m_file.Clear();  m_stack.GroupBy(TG_FILE, m_file); m_file.SortBy(TG_TEXT, true);
		m_class.Clear(); m_stack.GroupBy(TG_CLASS, m_class); m_class.SortBy(TG_TEXT, true);
	}
	m_cur.m_uses++; 
	m_cur.m_tick0=GetTickCount(); m_cur.m_last=TimeLocal(); // set the number of calls and the time of entering
}
//------------------------------------------------------------------	Out
void CTraceCtrl::Out(int aid)
{
	if (aid<0) { StepBack(); return; }
	CNode *node=m_stack.FindNode(aid); if (NIL(node)) return; // found node with the required id
	while (m_cur!=node) if (!StepBack()) break; // exit from nodes, until we reach the required one
	if (m_cur!=node) m_cur=node; // if it is in an adjacent branch, jump
	if (NIL(m_cur)) m_cur=node; // it the current one is not specified, set it
}
//------------------------------------------------------------------	StepBack
bool CTraceCtrl::StepBack()
{
	if (NIL(m_stack.m_root)) { Print("-err Out root"); return(false); } // validity of pointer
	if (NIL(m_cur)) { m_cur=m_stack.m_root; return(false); } // validity of pointer
	if (m_cur.m_tick0>0) m_cur.m_tick+=GetTickCount()-m_cur.m_tick0; m_cur.m_tick0=0; // count time
	if (m_cur==m_stack.m_root) return(false); // if we are already on top
	if (NIL(m_cur.m_prev)) { Print("-err Out "+m_cur.m_text+"("+(string)m_cur.m_id+")"); return(false); } // validity of pointer
	m_cur=m_cur.m_prev; // moved to the parent node
	return(true);
}
//------------------------------------------------------------------	AddWatch
void CTraceCtrl::AddWatch(string name, string val) // adding Watch variable to the current node
{
	if (!NIL(m_trace) && !NIL(m_trace.m_cur)) m_trace.m_cur.AddWatch(name, val);
}
//------------------------------------------------------------------	Break
void CTraceCtrl::Break() // checking the debugging mode for a node
{
	if (NIL(m_traceview) || NIL(m_stack)) return; // checking validity
	if (!m_stack.m_bBreak) return; // checking if debugging is allowed
	
	m_stack.BreakBy(TG_ALL, NULL, false); // removed the m_break flags from all nodes
	m_cur.m_break=true; // activated only for the current one
	m_traceview.m_viewstack.m_sid=m_cur.m_id; // moved selection to it
	m_stack.ExpandBy(TG_UP, m_cur, true, m_cur); // expand the parent nodes if they are closed
	m_traceview.Draw(); // drew everything
	string namedbg=m_traceview.m_viewstack.m_name+string(m_cur.m_id)+".dbg"; // got name of the BREAK button
	string name=m_traceview.m_viewstack.m_name+".break"; // got name of the BREAK button
	bool state=ObjectGetInteger(m_traceview.m_chart, namedbg, OBJPROP_STATE);
	while(!state) // execute the loop until the button is pressed
	{
		Sleep(500); // made pause
		state=ObjectGetInteger(m_traceview.m_chart, namedbg, OBJPROP_STATE);  // check its state
		if (!m_traceview.IsOpenView()) break; // if the window is closed, exit
		m_stack.m_bBreak=ObjectGetInteger(m_traceview.m_chart, name, OBJPROP_STATE);  // check its state
		state=state || !m_stack.m_bBreak; // consider the state of the button debugging in the system
		m_traceview.Draw(); // drew possible changes
	}
	m_cur.m_break=false; // removed the flag
	m_traceview.Draw(); // drew update
}

//------------------------------------------------------------------	CIn
class CIn
{
public:
	void In(string apath, string afile, int aline, string afunc, string adesc)
	{
		if (NIL(m_trace)) return; // if there is no graph, exit
		if (NIL(m_trace.m_stack)) return;
		if (NIL(m_trace.m_stack.m_root)) return;
		if (NIL(m_trace.m_cur))  m_trace.m_cur=m_trace.m_stack.m_root;
		m_trace.In(apath, afile, aline, afunc, adesc, -1); // enter the next one
		
		if (m_trace.m_cur.m_brkuse>0 && m_trace.m_cur.m_uses>=m_trace.m_cur.m_brkuse) // if the number of calls is reached
			m_trace.Break(); //enter debugging
	}
	void ~CIn() { if (!NIL(m_trace)) m_trace.Out(-1); } // exit higher
};
// definition of macros
#define _IN(desc)		CIn _in; _in.In(__PATH__, __FILE__, __LINE__, __FUNCTION__, string(desc))
#define _IN1(desc)	CIn _in1; _in1.In(__PATH__, __FILE__, __LINE__, __FUNCTION__, string(desc))
#define _IN2(desc)	CIn _in2; _in2.In(__PATH__, __FILE__, __LINE__, __FUNCTION__, string(desc))
#define _IN3(desc)	CIn _in3; _in3.In(__PATH__, __FILE__, __LINE__, __FUNCTION__, string(desc))
#define _IN4(desc)	CIn _in4; _in4.In(__PATH__, __FILE__, __LINE__, __FUNCTION__, string(desc))
#define _IN5(desc)	CIn _in5; _in5.In(__PATH__, __FILE__, __LINE__, __FUNCTION__, string(desc))

#define _WATCH(w, v) 		if (!NIL(m_trace)) if (!NIL(m_trace.m_cur)) m_trace.m_cur.AddWatch(w, string(v));

#define _BRKUSES(u)	if (!NIL(m_trace)) if (!NIL(m_trace.m_cur)) if (m_trace.m_cur.m_brkuse==-2) m_trace.m_cur.m_brkuse=long(u);

#include "View\TraceView.mqh" // connection the description of the class

extern CTraceCtrl* m_trace; // the only instance of the tracer
