﻿//+------------------------------------------------------------------+
//|  Canvas Graphing PART 2 - Statistical Regression (Cyberpunk).mq5 |
//|                           Copyright 2026, Allan Munene Mutiiria. |
//|                                   https://t.me/Forex_Algo_Trader |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, Allan Munene Mutiiria."
#property link "https://t.me/Forex_Algo_Trader"
#property version "1.00"
#property strict
#property description "Dual Theme Regression: Standard & Cyberpunk Modes"

#include <Math\Alglib\alglib.mqh>
#include <Canvas\Canvas.mqh>

//+------------------------------------------------------------------+
//| Enumerations                                                     |
//+------------------------------------------------------------------+
enum ResizeDirection
  {
   NO_RESIZE,                // No resize
   RESIZE_BOTTOM_EDGE,       // Resize bottom edge
   RESIZE_RIGHT_EDGE,        // Resize right edge
   RESIZE_CORNER             // Resize corner
  };

//+------------------------------------------------------------------+
//| Inputs                                                           |
//+------------------------------------------------------------------+
input group "=== REGRESSION SETTINGS ==="
input int                maxHistoryBars = 200;          // Maximum History Bars
input ENUM_TIMEFRAMES    chartTimeframe = PERIOD_CURRENT; // Chart Timeframe
input string             primarySymbol = "AUDUSDm";     // Primary Symbol (X-axis)
input string             secondarySymbol = "EURUSDm";   // Secondary Symbol (Y-axis)

input group "=== CANVAS DISPLAY SETTINGS ==="
input int                initialCanvasX = 20;           // Initial Canvas X Position
input int                initialCanvasY = 30;           // Initial Canvas Y Position
input int                initialCanvasWidth = 600;      // Initial Canvas Width
input int                initialCanvasHeight = 400;     // Initial Canvas Height
input int                plotPadding = 10;              // Plot Area Internal Padding (px)

input group "=== THEME COLOR (SINGLE CONTROL!) ==="
input color              themeColor = clrDodgerBlue;    // Master Theme Color
input bool               showBorderFrame = true;        // Show Border Frame

input group "=== REGRESSION LINE SETTINGS ==="
input color              regressionLineColor = clrBlue; // Regression Line Color
input int                regressionLineWidth = 2;       // Regression Line Width
input color              dataPointsColor = clrRed;      // Data Points Color
input int                dataPointSize = 3;             // Data Point Size

input group "=== BACKGROUND SETTINGS ==="
input bool               enableBackgroundFill = true;   // Enable Background Fill
input color              backgroundTopColor = clrWhite; // Background Top Color
input double             backgroundOpacityLevel = 0.95; // Background Opacity (0-1)

input group "=== TEXT AND LABELS ==="
input int                titleFontSize = 14;            // Title Font Size
input color              titleTextColor = clrBlack;     // Title Text Color
input int                labelFontSize = 11;            // Label Font Size
input color              labelTextColor = clrBlack;     // Label Text Color
input int                axisLabelFontSize = 12;        // Axis Labels Font Size
input bool               showStatistics = true;         // Show Statistics & Legend

input group "=== STATS & LEGEND PANEL SETTINGS ==="
input int                statsPanelX = 70;              // Stats Panel X Position
input int                statsPanelY = 10;              // Stats Panel Y Offset (from header)
input int                statsPanelWidth = 140;         // Stats Panel Width
input int                statsPanelHeight = 65;         // Stats Panel Height
input int                panelFontSize = 13;            // Stats & Legend Font Size
input int                legendHeight = 35;             // Legend Panel Height

input group "=== INTERACTION SETTINGS ==="
input bool               enableDragging = true;         // Enable Canvas Dragging
input bool               enableResizing = true;         // Enable Canvas Resizing
input int                resizeGripSize = 8;            // Resize Grip Size (pixels)

input group "=== CYBERPUNK MODE SETTINGS ==="
input double             glowIntensity = 0.8;           // Glow Intensity (0-1)

//+------------------------------------------------------------------+
//| Global Variables                                                 |
//+------------------------------------------------------------------+
CCanvas mainCanvas;                                     // Declare main canvas
string canvasObjectName = "RegressionCanvas_Main";      // Set canvas object name

int currentPositionX = initialCanvasX;                  // Initialize current X position
int currentPositionY = initialCanvasY;                  // Initialize current Y position
int currentWidthPixels = initialCanvasWidth;            // Initialize current width
int currentHeightPixels = initialCanvasHeight;          // Initialize current height

bool isDraggingCanvas = false;                          // Initialize dragging flag
bool isResizingCanvas = false;                          // Initialize resizing flag
int dragStartX = 0, dragStartY = 0;                     // Initialize drag start coordinates
int canvasStartX = 0, canvasStartY = 0;                 // Initialize canvas start coordinates

int resizeStartX = 0, resizeStartY = 0;                 // Initialize resize start coordinates
int resizeInitialWidth = 0, resizeInitialHeight = 0;    // Initialize resize initial dimensions
ResizeDirection activeResizeMode = NO_RESIZE;           // Initialize active resize mode
ResizeDirection hoverResizeMode = NO_RESIZE;            // Initialize hover resize mode

bool isHoveringCanvas = false;                          // Initialize canvas hover flag
bool isHoveringHeader = false;                          // Initialize header hover flag
bool isHoveringResizeZone = false;                      // Initialize resize hover flag
bool isHoveringThemeButton = false;                     // Initialize theme button hover flag
int lastMouseX = 0, lastMouseY = 0;                     // Initialize last mouse coordinates
int previousMouseButtonState = 0;                       // Initialize previous mouse state

const int MIN_CANVAS_WIDTH = 300;                       // Set minimum canvas width
const int MIN_CANVAS_HEIGHT = 200;                      // Set minimum canvas height
const int HEADER_BAR_HEIGHT = 35;                       // Set header bar height

double regressionSlope = 0.0;                           // Initialize regression slope
double regressionIntercept = 0.0;                       // Initialize regression intercept
double correlationCoefficient = 0.0;                    // Initialize correlation coefficient
double rSquared = 0.0;                                  // Initialize R-squared

double primaryClosePrices[];                            // Declare primary close prices array
double secondaryClosePrices[];                          // Declare secondary close prices array
bool dataLoadedSuccessfully = false;                    // Initialize data loaded flag

bool isCyberpunkThemeEnabled = false;                   // Initialize cyberpunk theme flag
int themeToggleButtonX = 0;                             // Initialize theme button X
int themeToggleButtonY = 0;                             // Initialize theme button Y
const int THEME_TOGGLE_BUTTON_SIZE = 25;                // Set theme toggle button size

const color CYBER_PRIMARY_COLOR = C'0,191,255';         // Set cyber primary color
const color CYBER_SECONDARY_COLOR = C'255,0,255';       // Set cyber secondary color
const color CYBER_DATA_POINTS_COLOR = C'255,50,150';    // Set cyber data points color
const color CYBER_REGRESSION_COLOR = C'0,150,255';      // Set cyber regression color

int currentAnimationFrame = 0;                          // Initialize animation frame

//+------------------------------------------------------------------+
//| Lighten color                                                    |
//+------------------------------------------------------------------+
color LightenColor(color baseColor, double factor)
  {
   //--- Extract red component
   uchar r = (uchar)((baseColor >> 16) & 0xFF);
   //--- Extract green component
   uchar g = (uchar)((baseColor >> 8) & 0xFF);
   //--- Extract blue component
   uchar b = (uchar)(baseColor & 0xFF);
   
   //--- Lighten red
   r = (uchar)MathMin(255, r + (255 - r) * factor);
   //--- Lighten green
   g = (uchar)MathMin(255, g + (255 - g) * factor);
   //--- Lighten blue
   b = (uchar)MathMin(255, b + (255 - b) * factor);
   
   //--- Return lightened color
   return (r << 16) | (g << 8) | b;
  }

//+------------------------------------------------------------------+
//| Darken color                                                     |
//+------------------------------------------------------------------+
color DarkenColor(color baseColor, double factor)
  {
   //--- Extract red component
   uchar r = (uchar)((baseColor >> 16) & 0xFF);
   //--- Extract green component
   uchar g = (uchar)((baseColor >> 8) & 0xFF);
   //--- Extract blue component
   uchar b = (uchar)(baseColor & 0xFF);
   
   //--- Darken red
   r = (uchar)(r * (1.0 - factor));
   //--- Darken green
   g = (uchar)(g * (1.0 - factor));
   //--- Darken blue
   b = (uchar)(b * (1.0 - factor));
   
   //--- Return darkened color
   return (r << 16) | (g << 8) | b;
  }

//+------------------------------------------------------------------+
//| Blend colors                                                     |
//+------------------------------------------------------------------+
color BlendColors(color color1, color color2, double ratio)
  {
   //--- Extract color1 red
   uchar r1 = (uchar)((color1 >> 16) & 0xFF);
   //--- Extract color1 green
   uchar g1 = (uchar)((color1 >> 8) & 0xFF);
   //--- Extract color1 blue
   uchar b1 = (uchar)(color1 & 0xFF);
   
   //--- Extract color2 red
   uchar r2 = (uchar)((color2 >> 16) & 0xFF);
   //--- Extract color2 green
   uchar g2 = (uchar)((color2 >> 8) & 0xFF);
   //--- Extract color2 blue
   uchar b2 = (uchar)(color2 & 0xFF);
   
   //--- Blend red
   uchar r = (uchar)(r1 * (1.0 - ratio) + r2 * ratio);
   //--- Blend green
   uchar g = (uchar)(g1 * (1.0 - ratio) + g2 * ratio);
   //--- Blend blue
   uchar b = (uchar)(b1 * (1.0 - ratio) + b2 * ratio);
   
   //--- Return blended color
   return (r << 16) | (g << 8) | b;
  }

//+------------------------------------------------------------------+
//| Initialize expert                                                |
//+------------------------------------------------------------------+
int OnInit()
  {
   //--- Set current X from input
   currentPositionX = initialCanvasX;
   //--- Set current Y from input
   currentPositionY = initialCanvasY;
   //--- Set current width from input
   currentWidthPixels = initialCanvasWidth;
   //--- Set current height from input
   currentHeightPixels = initialCanvasHeight;

   //--- Create canvas or fail
   if (!CreateCanvas())
     {
      //--- Print error
      Print("ERROR: Failed to create regression canvas");
      //--- Return failure
      return(INIT_FAILED);
     }

   //--- Load prices or fail
   if (!loadSymbolClosePrices())
     {
      //--- Print error
      Print("ERROR: Failed to load price data for symbols");
      //--- Return failure
      return(INIT_FAILED);
     }

   //--- Compute regression or fail
   if (!computeLinearRegression())
     {
      //--- Print error
      Print("ERROR: Failed to calculate regression parameters");
      //--- Return failure
      return(INIT_FAILED);
     }

   //--- Render visualization
   renderVisualization();

   //--- Enable mouse events
   ChartSetInteger(0, CHART_EVENT_MOUSE_MOVE, true);
   //--- Redraw chart
   ChartRedraw();
   
   //--- Set timer for animations
   EventSetMillisecondTimer(50);

   //--- Return success
   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//| Deinitialize expert                                              |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   //--- Kill timer
   EventKillTimer();
   //--- Destroy canvas
   mainCanvas.Destroy();
   //--- Redraw chart
   ChartRedraw();
  }

//+------------------------------------------------------------------+
//| Handle timer                                                     |
//+------------------------------------------------------------------+
void OnTimer()
  {
   //--- Check cyberpunk enabled
   if (isCyberpunkThemeEnabled)
     {
      //--- Increment frame
      currentAnimationFrame++;
      //--- Reset frame
      if (currentAnimationFrame > 360) currentAnimationFrame = 0;
      //--- Render
      renderVisualization();
      //--- Redraw chart
      ChartRedraw();
     }
  }

//+------------------------------------------------------------------+
//| Handle tick                                                      |
//+------------------------------------------------------------------+
void OnTick()
  {
   //--- Store last bar time
   static datetime lastBarTimestamp = 0;
   //--- Get current bar time
   datetime currentBarTimestamp = iTime(_Symbol, chartTimeframe, 0);

   //--- Check new bar
   if (currentBarTimestamp > lastBarTimestamp)
     {
      //--- Reload prices
      if (loadSymbolClosePrices())
        {
         //--- Recalculate regression
         if (computeLinearRegression())
           {
            //--- Update visualization
            renderVisualization();
            //--- Redraw chart
            ChartRedraw();
           }
        }
      //--- Update last time
      lastBarTimestamp = currentBarTimestamp;
     }
  }

//+------------------------------------------------------------------+
//| Create canvas                                                    |
//+------------------------------------------------------------------+
bool CreateCanvas()
  {
   //--- Create bitmap label
   if (!mainCanvas.CreateBitmapLabel(0, 0, canvasObjectName, 
       currentPositionX, currentPositionY, currentWidthPixels, currentHeightPixels, 
       COLOR_FORMAT_ARGB_NORMALIZE))
     {
      //--- Return failure
      return false;
     }
   //--- Return success
   return true;
  }

//+------------------------------------------------------------------+
//| Load symbol close prices                                         |
//+------------------------------------------------------------------+
bool loadSymbolClosePrices()
  {
   //--- Select primary symbol
   if (!SymbolSelect(primarySymbol, true))
     {
      //--- Print error
      Print("ERROR: Primary symbol not found: ", primarySymbol);
      //--- Return failure
      return false;
     }

   //--- Select secondary symbol
   if (!SymbolSelect(secondarySymbol, true))
     {
      //--- Print error
      Print("ERROR: Secondary symbol not found: ", secondarySymbol);
      //--- Return failure
      return false;
     }

   //--- Copy primary closes
   int copiedPrimary = CopyClose(primarySymbol, chartTimeframe, 1, maxHistoryBars, primaryClosePrices);
   //--- Check copy success
   if (copiedPrimary <= 0)
     {
      //--- Print error
      Print("ERROR: Failed to copy data for ", primarySymbol, ". Error: ", GetLastError());
      //--- Return failure
      return false;
     }

   //--- Copy secondary closes
   int copiedSecondary = CopyClose(secondarySymbol, chartTimeframe, 1, maxHistoryBars, secondaryClosePrices);
   //--- Check copy success
   if (copiedSecondary <= 0)
     {
      //--- Print error
      Print("ERROR: Failed to copy data for ", secondarySymbol, ". Error: ", GetLastError());
      //--- Return failure
      return false;
     }

   //--- Get min bars
   int actualBars = MathMin(copiedPrimary, copiedSecondary);
   //--- Resize primary array
   ArrayResize(primaryClosePrices, actualBars);
   //--- Resize secondary array
   ArrayResize(secondaryClosePrices, actualBars);

   //--- Set loaded flag
   dataLoadedSuccessfully = true;
   //--- Print success
   Print("SUCCESS: Loaded ", actualBars, " bars for both symbols");
   //--- Return success
   return true;
  }

//+------------------------------------------------------------------+
//| Compute linear regression                                        |
//+------------------------------------------------------------------+
bool computeLinearRegression()
  {
   //--- Get data size
   int dataSize = ArraySize(primaryClosePrices);
   //--- Check valid size
   if (dataSize <= 0 || ArraySize(secondaryClosePrices) != dataSize)
     {
      //--- Return failure
      return false;
     }

   //--- Declare temp arrays
   double tempPrimary[], tempSecondary[];
   //--- Set primary as series
   ArraySetAsSeries(tempPrimary, true);
   //--- Set secondary as series
   ArraySetAsSeries(tempSecondary, true);
   //--- Copy primary
   ArrayCopy(tempPrimary, primaryClosePrices);
   //--- Copy secondary
   ArrayCopy(tempSecondary, secondaryClosePrices);

   //--- Create regression matrix
   CMatrixDouble regressionMatrix(dataSize, 2);

   //--- Loop over data
   for (int i = 0; i < dataSize; i++)
     {
      //--- Set X value
      regressionMatrix.Set(i, 0, tempPrimary[i]);
      //--- Set Y value
      regressionMatrix.Set(i, 1, tempSecondary[i]);
     }

   //--- Declare linear regression
   CLinReg linearRegression;
   //--- Declare linear model
   CLinearModel linearModel;
   //--- Declare report
   CLRReport regressionReport;
   //--- Declare return code
   int returnCode;

   //--- Build regression
   linearRegression.LRBuild(regressionMatrix, dataSize, 1, returnCode, linearModel, regressionReport);

   //--- Check success
   if (returnCode != 1)
     {
      //--- Print error
      Print("ERROR: Linear regression calculation failed with code: ", returnCode);
      //--- Return failure
      return false;
     }

   //--- Declare vars count
   int numberOfVars;
   //--- Declare coefficients
   double coefficientsArray[];
   //--- Unpack model
   linearRegression.LRUnpack(linearModel, coefficientsArray, numberOfVars);

   //--- Set slope
   regressionSlope = coefficientsArray[0];
   //--- Set intercept
   regressionIntercept = coefficientsArray[1];

   //--- Compute statistics
   computeStatistics();

   //--- Print equation
   PrintFormat("Regression Equation: Y = %.6f + %.6f * X", regressionIntercept, regressionSlope);
   //--- Print stats
   PrintFormat("Correlation: %.4f | R-Squared: %.4f", correlationCoefficient, rSquared);

   //--- Return success
   return true;
  }

//+------------------------------------------------------------------+
//| Compute statistics                                               |
//+------------------------------------------------------------------+
void computeStatistics()
  {
   //--- Get size
   int n = ArraySize(primaryClosePrices);
   //--- Return if empty
   if (n <= 0) return;

   //--- Initialize sums
   double sumX = 0, sumY = 0, sumXY = 0, sumX2 = 0, sumY2 = 0;

   //--- Loop over data
   for (int i = 0; i < n; i++)
     {
      //--- Get X
      double x = primaryClosePrices[i];
      //--- Get Y
      double y = secondaryClosePrices[i];

      //--- Accumulate X
      sumX += x;
      //--- Accumulate Y
      sumY += y;
      //--- Accumulate XY
      sumXY += x * y;
      //--- Accumulate X2
      sumX2 += x * x;
      //--- Accumulate Y2
      sumY2 += y * y;
     }

   //--- Compute mean X
   double meanX = sumX / n;
   //--- Compute mean Y
   double meanY = sumY / n;

   //--- Compute numerator
   double numerator = n * sumXY - sumX * sumY;
   //--- Compute denominator X
   double denominatorX = MathSqrt(n * sumX2 - sumX * sumX);
   //--- Compute denominator Y
   double denominatorY = MathSqrt(n * sumY2 - sumY * sumY);

   //--- Check denominators
   if (denominatorX != 0 && denominatorY != 0)
     {
      //--- Compute correlation
      correlationCoefficient = numerator / (denominatorX * denominatorY);
      //--- Compute R-squared
      rSquared = correlationCoefficient * correlationCoefficient;
     }
   else
     {
      //--- Set correlation to 0
      correlationCoefficient = 0;
      //--- Set R-squared to 0
      rSquared = 0;
     }
  }

//+------------------------------------------------------------------+
//| Render visualization                                             |
//+------------------------------------------------------------------+
void renderVisualization()
  {
   //--- Check cyberpunk mode
   if (isCyberpunkThemeEnabled)
     {
      //--- Render cyberpunk
      renderCyberpunkTheme();
     }
   else
     {
      //--- Render standard
      renderStandardTheme();
     }
  }

//+------------------------------------------------------------------+
//| Render standard theme                                            |
//+------------------------------------------------------------------+
void renderStandardTheme()
  {
   //--- Erase canvas
   mainCanvas.Erase(0);

   //--- Check background fill
   if (enableBackgroundFill)
     {
      //--- Draw gradient
      drawGradientBackground();
     }

   //--- Draw border
   drawCanvasBorder();
   //--- Draw header
   drawHeaderBar();
   //--- Draw toggle button
   drawThemeToggleButton();
   //--- Draw plot
   drawRegressionPlot();

   //--- Check show stats
   if (showStatistics)
     {
      //--- Draw stats panel
      drawStatisticsPanel();
      //--- Draw legend
      drawLegend();
     }

   //--- Check resize hover
   if (isHoveringResizeZone && enableResizing)
     {
      //--- Draw indicator
      drawResizeIndicator();
     }

   //--- Update canvas
   mainCanvas.Update();
  }

//+------------------------------------------------------------------+
//| Render cyberpunk theme                                           |
//+------------------------------------------------------------------+
void renderCyberpunkTheme()
  {
   //--- Erase canvas
   mainCanvas.Erase(0);

   //--- Draw cyber background
   drawCyberpunkBackground();
   //--- Draw holographic border
   drawHolographicBorder();
   //--- Draw futuristic header
   drawFuturisticHeader();
   //--- Draw toggle button
   drawThemeToggleButton();
   //--- Draw cyber plot
   drawRegressionPlotCyberpunk();

   //--- Check show stats
   if (showStatistics)
     {
      //--- Draw glass stats
      drawGlassMorphismStatsPanel();
      //--- Draw glass legend
      drawGlassMorphismLegend();
     }

   //--- Check resize hover
   if (isHoveringResizeZone && enableResizing)
     {
      //--- Draw neon indicator
      drawNeonResizeIndicator();
     }

   //--- Update canvas
   mainCanvas.Update();
  }

//+------------------------------------------------------------------+
//| Draw theme toggle button                                         |
//+------------------------------------------------------------------+
void drawThemeToggleButton()
  {
   //--- Compute button X
   themeToggleButtonX = currentWidthPixels - THEME_TOGGLE_BUTTON_SIZE - 8;
   //--- Compute button Y
   themeToggleButtonY = (HEADER_BAR_HEIGHT - THEME_TOGGLE_BUTTON_SIZE) / 2;
   
   //--- Declare button color
   color buttonColor;
   //--- Check cyber mode
   if (isCyberpunkThemeEnabled)
     {
      //--- Set cyber color
      buttonColor = isHoveringThemeButton ? LightenColor(CYBER_PRIMARY_COLOR, 0.3) : CYBER_PRIMARY_COLOR;
     }
   else
     {
      //--- Set standard color
      buttonColor = isHoveringThemeButton ? LightenColor(themeColor, 0.3) : themeColor;
     }
   
   //--- Get button ARGB
   uint argbButton = ColorToARGB(buttonColor, 200);
   //--- Get border ARGB
   uint argbBorder = ColorToARGB(isCyberpunkThemeEnabled ? CYBER_SECONDARY_COLOR : themeColor, 255);
   
   //--- Fill button
   mainCanvas.FillRectangle(themeToggleButtonX, themeToggleButtonY, 
                                     themeToggleButtonX + THEME_TOGGLE_BUTTON_SIZE, 
                                     themeToggleButtonY + THEME_TOGGLE_BUTTON_SIZE, argbButton);
   
   //--- Draw border
   mainCanvas.Rectangle(themeToggleButtonX, themeToggleButtonY, 
                                 themeToggleButtonX + THEME_TOGGLE_BUTTON_SIZE, 
                                 themeToggleButtonY + THEME_TOGGLE_BUTTON_SIZE, argbBorder);
   
   //--- Set icon font
   mainCanvas.FontSet("Arial Black", 16);
   //--- Get text ARGB
   uint argbText = ColorToARGB(clrWhite, 255);
   //--- Set icon text
   string iconText = isCyberpunkThemeEnabled ? "◆" : "●";
   //--- Draw icon
   mainCanvas.TextOut(themeToggleButtonX + THEME_TOGGLE_BUTTON_SIZE/2, themeToggleButtonY + 3, 
                                iconText, argbText, TA_CENTER);
  }

//+------------------------------------------------------------------+
//| Draw gradient background                                         |
//+------------------------------------------------------------------+
void drawGradientBackground()
  {
   //--- Compute bottom color
   color bottomColor = LightenColor(themeColor, 0.85);
   
   //--- Loop over rows
   for (int y = HEADER_BAR_HEIGHT; y < currentHeightPixels; y++)
     {
      //--- Compute factor
      double gradientFactor = (double)(y - HEADER_BAR_HEIGHT) / (currentHeightPixels - HEADER_BAR_HEIGHT);
      //--- Interpolate color
      color currentRowColor = InterpolateColors(backgroundTopColor, bottomColor, gradientFactor);
      //--- Compute alpha
      uchar alphaChannel = (uchar)(255 * backgroundOpacityLevel);
      //--- Get ARGB
      uint argbColor = ColorToARGB(currentRowColor, alphaChannel);

      //--- Loop over columns
      for (int x = 0; x < currentWidthPixels; x++)
        {
         //--- Set pixel
         mainCanvas.PixelSet(x, y, argbColor);
        }
     }
  }

//+------------------------------------------------------------------+
//| Draw canvas border                                               |
//+------------------------------------------------------------------+
void drawCanvasBorder()
  {
   //--- Return if no border
   if (!showBorderFrame) return;

   //--- Get border color
   color borderColor = isHoveringResizeZone ? DarkenColor(themeColor, 0.2) : themeColor;
   //--- Get ARGB border
   uint argbBorder = ColorToARGB(borderColor, 255);

   //--- Draw outer border
   mainCanvas.Rectangle(0, 0, currentWidthPixels - 1, currentHeightPixels - 1, argbBorder);
   //--- Draw inner border
   mainCanvas.Rectangle(1, 1, currentWidthPixels - 2, currentHeightPixels - 2, argbBorder);
  }

//+------------------------------------------------------------------+
//| Draw header bar                                                  |
//+------------------------------------------------------------------+
void drawHeaderBar()
  {
   //--- Declare header color
   color headerColor;
   //--- Check dragging
   if (isDraggingCanvas)
     {
      //--- Set darker
      headerColor = DarkenColor(themeColor, 0.1);
     }
   //--- Check hovering
   else if (isHoveringHeader)
     {
      //--- Set medium light
      headerColor = LightenColor(themeColor, 0.4);
     }
   else
     {
      //--- Set very light
      headerColor = LightenColor(themeColor, 0.7);
     }
   //--- Get ARGB header
   uint argbHeader = ColorToARGB(headerColor, 255);

   //--- Fill header
   mainCanvas.FillRectangle(0, 0, currentWidthPixels - 1, HEADER_BAR_HEIGHT, argbHeader);

   //--- Check show border
   if (showBorderFrame)
     {
      //--- Get ARGB border
      uint argbBorder = ColorToARGB(themeColor, 255);
      //--- Draw outer
      mainCanvas.Rectangle(0, 0, currentWidthPixels - 1, HEADER_BAR_HEIGHT, argbBorder);
      //--- Draw inner
      mainCanvas.Rectangle(1, 1, currentWidthPixels - 2, HEADER_BAR_HEIGHT - 1, argbBorder);
     }

   //--- Set title font
   mainCanvas.FontSet("Arial Bold", titleFontSize);
   //--- Get ARGB text
   uint argbText = ColorToARGB(titleTextColor, 255);

   //--- Format title
   string titleText = StringFormat("%s vs %s - Linear Regression", secondarySymbol, primarySymbol);
   //--- Draw title
   mainCanvas.TextOut(currentWidthPixels / 2, (HEADER_BAR_HEIGHT - titleFontSize) / 2, 
                                titleText, argbText, TA_CENTER);
  }

//+------------------------------------------------------------------+
//| Calculate optimal ticks                                          |
//+------------------------------------------------------------------+
int calculateOptimalTicks(double minValue, double maxValue, int pixelRange, double &tickValues[])
  {
   //--- Compute range
   double range = maxValue - minValue;
   //--- Check invalid
   if (range == 0 || pixelRange <= 0)
     {
      //--- Resize to 1
      ArrayResize(tickValues, 1);
      //--- Set single tick
      tickValues[0] = minValue;
      //--- Return 1
      return 1;
     }
   
   //--- Compute target count
   int targetTickCount = (int)(pixelRange / 50.0);
   //--- Min 3
   if (targetTickCount < 3) targetTickCount = 3;
   //--- Max 20
   if (targetTickCount > 20) targetTickCount = 20;
   
   //--- Compute rough step
   double roughStep = range / (double)(targetTickCount - 1);
   //--- Compute magnitude
   double magnitude = MathPow(10.0, MathFloor(MathLog10(roughStep)));
   //--- Normalize
   double normalized = roughStep / magnitude;
   
   //--- Declare nice normalized
   double niceNormalized;
   //--- Set 1.0
   if (normalized <= 1.0) niceNormalized = 1.0;
   //--- Set 1.0
   else if (normalized <= 1.5) niceNormalized = 1.0;
   //--- Set 2.0
   else if (normalized <= 2.0) niceNormalized = 2.0;
   //--- Set 2.0
   else if (normalized <= 2.5) niceNormalized = 2.0;
   //--- Set 2.5
   else if (normalized <= 3.0) niceNormalized = 2.5;
   //--- Set 4.0
   else if (normalized <= 4.0) niceNormalized = 4.0;
   //--- Set 5.0
   else if (normalized <= 5.0) niceNormalized = 5.0;
   //--- Set 5.0
   else if (normalized <= 7.5) niceNormalized = 5.0;
   //--- Set 10.0
   else niceNormalized = 10.0;
   
   //--- Compute step
   double step = niceNormalized * magnitude;
   
   //--- Compute tick min
   double tickMin = MathFloor(minValue / step) * step;
   //--- Compute tick max
   double tickMax = MathCeil(maxValue / step) * step;
   
   //--- Compute num ticks
   int numTicks = (int)MathRound((tickMax - tickMin) / step) + 1;
   
   //--- Check too many
   if (numTicks > 25)
     {
      //--- Double step
      step *= 2.0;
      //--- Recalc min
      tickMin = MathFloor(minValue / step) * step;
      //--- Recalc max
      tickMax = MathCeil(maxValue / step) * step;
      //--- Recalc num
      numTicks = (int)MathRound((tickMax - tickMin) / step) + 1;
     }
   
   //--- Check too few
   if (numTicks < 3)
     {
      //--- Halve step
      step /= 2.0;
      //--- Recalc min
      tickMin = MathFloor(minValue / step) * step;
      //--- Recalc max
      tickMax = MathCeil(maxValue / step) * step;
      //--- Recalc num
      numTicks = (int)MathRound((tickMax - tickMin) / step) + 1;
     }
   
   //--- Resize array
   ArrayResize(tickValues, numTicks);
   //--- Loop to set ticks
   for (int i = 0; i < numTicks; i++)
     {
      //--- Set tick value
      tickValues[i] = tickMin + i * step;
     }
   
   //--- Return count
   return numTicks;
  }

//+------------------------------------------------------------------+
//| Format tick label                                                |
//+------------------------------------------------------------------+
string formatTickLabel(double value, double range)
  {
   //--- Format no decimals
   if (range > 100) return DoubleToString(value, 0);
   //--- Format 1 decimal
   else if (range > 10) return DoubleToString(value, 1);
   //--- Format 2 decimals
   else if (range > 1) return DoubleToString(value, 2);
   //--- Format 3 decimals
   else if (range > 0.1) return DoubleToString(value, 3);
   //--- Format 4 decimals
   else return DoubleToString(value, 4);
  }

//+------------------------------------------------------------------+
//| Draw regression plot                                             |
//+------------------------------------------------------------------+
void drawRegressionPlot()
  {
   //--- Return if no data
   if (!dataLoadedSuccessfully) return;

   //--- Set plot left
   int plotAreaLeft = 60;
   //--- Set plot right
   int plotAreaRight = currentWidthPixels - 40;
   //--- Set plot top
   int plotAreaTop = HEADER_BAR_HEIGHT + 10;
   //--- Set plot bottom
   int plotAreaBottom = currentHeightPixels - 50;

   //--- Set draw left
   int drawAreaLeft = plotAreaLeft + plotPadding;
   //--- Set draw right
   int drawAreaRight = plotAreaRight - plotPadding;
   //--- Set draw top
   int drawAreaTop = plotAreaTop + plotPadding;
   //--- Set draw bottom
   int drawAreaBottom = plotAreaBottom - plotPadding;

   //--- Compute plot width
   int plotWidth = drawAreaRight - drawAreaLeft;
   //--- Compute plot height
   int plotHeight = drawAreaBottom - drawAreaTop;

   //--- Return if invalid
   if (plotWidth <= 0 || plotHeight <= 0) return;

   //--- Init min X
   double minX = primaryClosePrices[0];
   //--- Init max X
   double maxX = primaryClosePrices[0];
   //--- Init min Y
   double minY = secondaryClosePrices[0];
   //--- Init max Y
   double maxY = secondaryClosePrices[0];

   //--- Get data points
   int dataPoints = ArraySize(primaryClosePrices);
   //--- Loop over points
   for (int i = 1; i < dataPoints; i++)
     {
      //--- Update min X
      if (primaryClosePrices[i] < minX) minX = primaryClosePrices[i];
      //--- Update max X
      if (primaryClosePrices[i] > maxX) maxX = primaryClosePrices[i];
      //--- Update min Y
      if (secondaryClosePrices[i] < minY) minY = secondaryClosePrices[i];
      //--- Update max Y
      if (secondaryClosePrices[i] > maxY) maxY = secondaryClosePrices[i];
     }

   //--- Compute range X
   double rangeX = maxX - minX;
   //--- Compute range Y
   double rangeY = maxY - minY;

   //--- Set min range X
   if (rangeX == 0) rangeX = 1;
   //--- Set min range Y
   if (rangeY == 0) rangeY = 1;

   //--- Get axis ARGB
   uint argbAxisColor = ColorToARGB(clrBlack, 255);
   
   //--- Loop for thick Y-axis
   for (int thick = 0; thick < 2; thick++)
     {
      //--- Draw Y-axis line
      mainCanvas.Line(plotAreaLeft - thick, plotAreaTop, plotAreaLeft - thick, plotAreaBottom, argbAxisColor);
     }
   
   //--- Loop for thick X-axis
   for (int thick = 0; thick < 2; thick++)
     {
      //--- Draw X-axis line
      mainCanvas.Line(plotAreaLeft, plotAreaBottom + thick, plotAreaRight, plotAreaBottom + thick, argbAxisColor);
     }

   //--- Set tick font
   mainCanvas.FontSet("Arial", axisLabelFontSize);
   //--- Get tick label ARGB
   uint argbTickLabel = ColorToARGB(clrBlack, 255);
   
   //--- Declare Y ticks
   double yTickValues[];
   //--- Compute Y ticks
   int numYTicks = calculateOptimalTicks(minY, maxY, plotHeight, yTickValues);
   
   //--- Loop over Y ticks
   for (int i = 0; i < numYTicks; i++)
     {
      //--- Get Y value
      double yValue = yTickValues[i];
      //--- Skip out of range
      if (yValue < minY || yValue > maxY) continue;
      
      //--- Compute Y pos
      int yPos = drawAreaBottom - (int)((yValue - minY) / rangeY * plotHeight);
      
      //--- Draw tick
      mainCanvas.Line(plotAreaLeft - 5, yPos, plotAreaLeft, yPos, argbAxisColor);
      
      //--- Format label
      string yLabel = formatTickLabel(yValue, rangeY);
      //--- Draw label
      mainCanvas.TextOut(plotAreaLeft - 8, yPos - axisLabelFontSize/2, yLabel, argbTickLabel, TA_RIGHT);
     }

   //--- Declare X ticks
   double xTickValues[];
   //--- Compute X ticks
   int numXTicks = calculateOptimalTicks(minX, maxX, plotWidth, xTickValues);
   
   //--- Loop over X ticks
   for (int i = 0; i < numXTicks; i++)
     {
      //--- Get X value
      double xValue = xTickValues[i];
      //--- Skip out of range
      if (xValue < minX || xValue > maxX) continue;
      
      //--- Compute X pos
      int xPos = drawAreaLeft + (int)((xValue - minX) / rangeX * plotWidth);
      
      //--- Draw tick
      mainCanvas.Line(xPos, plotAreaBottom, xPos, plotAreaBottom + 5, argbAxisColor);
      
      //--- Format label
      string xLabel = formatTickLabel(xValue, rangeX);
      //--- Draw label
      mainCanvas.TextOut(xPos, plotAreaBottom + 7, xLabel, argbTickLabel, TA_CENTER);
     }

   //--- Get points ARGB
   uint argbPoints = ColorToARGB(dataPointsColor, 255);

   //--- Loop over points
   for (int i = 0; i < dataPoints; i++)
     {
      //--- Compute screen X
      int screenX = drawAreaLeft + (int)((primaryClosePrices[i] - minX) / rangeX * plotWidth);
      //--- Compute screen Y
      int screenY = drawAreaBottom - (int)((secondaryClosePrices[i] - minY) / rangeY * plotHeight);

      //--- Draw point
      drawCirclePoint(screenX, screenY, dataPointSize, argbPoints);
     }

   //--- Compute start Y
   double lineStartY = regressionIntercept + regressionSlope * minX;
   //--- Compute end Y
   double lineEndY = regressionIntercept + regressionSlope * maxX;

   //--- Set start screen X
   int lineStartScreenX = drawAreaLeft;
   //--- Compute start screen Y
   int lineStartScreenY = drawAreaBottom - (int)((lineStartY - minY) / rangeY * plotHeight);
   //--- Set end screen X
   int lineEndScreenX = drawAreaRight;
   //--- Compute end screen Y
   int lineEndScreenY = drawAreaBottom - (int)((lineEndY - minY) / rangeY * plotHeight);

   //--- Get line ARGB
   uint argbLine = ColorToARGB(regressionLineColor, 255);

   //--- Loop for width
   for (int w = 0; w < regressionLineWidth; w++)
     {
      //--- Draw line
      mainCanvas.LineAA(lineStartScreenX, lineStartScreenY + w - regressionLineWidth/2, 
                                  lineEndScreenX, lineEndScreenY + w - regressionLineWidth/2, argbLine);
     }

   //--- Set axis label font
   mainCanvas.FontSet("Arial Bold", labelFontSize);
   //--- Get axis label ARGB
   uint argbAxisLabel = ColorToARGB(clrBlack, 255);

   //--- Set X label
   string xAxisLabel = primarySymbol + " (X-axis)";
   //--- Draw X label
   mainCanvas.TextOut(currentWidthPixels / 2, currentHeightPixels - 20, xAxisLabel, argbAxisLabel, TA_CENTER);

   //--- Set Y label
   string yAxisLabel = secondarySymbol + " (Y-axis)";
   //--- Set vertical angle
   mainCanvas.FontAngleSet(900);
   //--- Draw Y label
   mainCanvas.TextOut(12, currentHeightPixels / 2, yAxisLabel, argbAxisLabel, TA_CENTER);
   //--- Reset angle
   mainCanvas.FontAngleSet(0);
  }

//+------------------------------------------------------------------+
//| Draw circle point                                                |
//+------------------------------------------------------------------+
void drawCirclePoint(int centerX, int centerY, int radius, uint argbColor)
  {
   //--- Extract source alpha
   uchar srcAlpha = (uchar)((argbColor >> 24) & 0xFF);
   //--- Extract source red
   uchar srcRed = (uchar)((argbColor >> 16) & 0xFF);
   //--- Extract source green
   uchar srcGreen = (uchar)((argbColor >> 8) & 0xFF);
   //--- Extract source blue
   uchar srcBlue = (uchar)(argbColor & 0xFF);
   
   //--- Adjust radius
   double radiusDouble = (double)radius + 0.5;
   //--- Compute extent
   int extent = radius + 2;
   
   //--- Loop over dy
   for (int dy = -extent; dy <= extent; dy++)
     {
      //--- Loop over dx
      for (int dx = -extent; dx <= extent; dx++)
        {
         //--- Compute distance
         double distance = MathSqrt((double)(dx * dx + dy * dy));
         
         //--- Check within radius
         if (distance <= radiusDouble)
           {
            //--- Set full coverage
            double coverage = 1.0;
            //--- Check edge
            if (distance > radiusDouble - 1.0)
              {
               //--- Compute coverage
               coverage = radiusDouble - distance;
               //--- Clamp min
               if (coverage < 0) coverage = 0;
               //--- Clamp max
               if (coverage > 1.0) coverage = 1.0;
              }
            
            //--- Compute final alpha
            uchar finalAlpha = (uchar)(srcAlpha * coverage);
            //--- Skip if transparent
            if (finalAlpha == 0) continue;
            
            //--- Compose color
            uint pixelColor = ((uint)finalAlpha << 24) | ((uint)srcRed << 16) | 
                             ((uint)srcGreen << 8) | (uint)srcBlue;
            
            //--- Compute pixel X
            int px = centerX + dx;
            //--- Compute pixel Y
            int py = centerY + dy;
            //--- Check bounds
            if (px >= 0 && px < currentWidthPixels && py >= 0 && py < currentHeightPixels)
              {
               //--- Blend pixel
               blendPixelSet(mainCanvas, px, py, pixelColor);
              }
           }
        }
     }
  }

//+------------------------------------------------------------------+
//| Draw statistics panel                                            |
//+------------------------------------------------------------------+
void drawStatisticsPanel()
  {
   //--- Set panel X
   int panelX = statsPanelX;
   //--- Set panel Y
   int panelY = HEADER_BAR_HEIGHT + statsPanelY;
   //--- Set panel width
   int panelWidth = statsPanelWidth;
   //--- Set panel height
   int panelHeight = statsPanelHeight;

   //--- Compute bg color
   color panelBgColor = LightenColor(themeColor, 0.9);
   //--- Set alpha
   uchar bgAlpha = 153;
   //--- Get panel bg ARGB
   uint argbPanelBg = ColorToARGB(panelBgColor, bgAlpha);
   //--- Get border ARGB
   uint argbBorder = ColorToARGB(themeColor, 255);
   //--- Get text ARGB
   uint argbText = ColorToARGB(clrBlack, 255);

   //--- Loop over rows
   for (int y = panelY; y <= panelY + panelHeight; y++)
     {
      //--- Loop over columns
      for (int x = panelX; x <= panelX + panelWidth; x++)
        {
         //--- Blend bg pixel
         blendPixelSet(mainCanvas, x, y, argbPanelBg);
        }
     }

   //--- Draw top border
   for (int x = panelX; x <= panelX + panelWidth; x++)
     {
      //--- Blend border pixel
      blendPixelSet(mainCanvas, x, panelY, argbBorder);
     }
   //--- Draw right border
   for (int y = panelY; y <= panelY + panelHeight; y++)
     {
      //--- Blend border pixel
      blendPixelSet(mainCanvas, panelX + panelWidth, y, argbBorder);
     }
   //--- Draw left border
   for (int y = panelY; y <= panelY + panelHeight; y++)
     {
      //--- Blend border pixel
      blendPixelSet(mainCanvas, panelX, y, argbBorder);
     }

   //--- Set stats font
   mainCanvas.FontSet("Arial", panelFontSize);

   //--- Set text Y
   int textY = panelY + 8;
   //--- Set line spacing
   int lineSpacing = panelFontSize;

   //--- Format equation
   string equationText = StringFormat("Y = %.3f + %.3f * X", regressionIntercept, regressionSlope);
   //--- Draw equation
   mainCanvas.TextOut(panelX + 8, textY, equationText, argbText, TA_LEFT);
   //--- Update Y
   textY += lineSpacing;

   //--- Format correlation
   string correlationText = StringFormat("Correlation: %.4f", correlationCoefficient);
   //--- Draw correlation
   mainCanvas.TextOut(panelX + 8, textY, correlationText, argbText, TA_LEFT);
   //--- Update Y
   textY += lineSpacing;

   //--- Format R-squared
   string rSquaredText = StringFormat("R-Squared: %.4f", rSquared);
   //--- Draw R-squared
   mainCanvas.TextOut(panelX + 8, textY, rSquaredText, argbText, TA_LEFT);
   //--- Update Y
   textY += lineSpacing;

   //--- Format points
   string dataPointsText = StringFormat("Points: %d", ArraySize(primaryClosePrices));
   //--- Draw points
   mainCanvas.TextOut(panelX + 8, textY, dataPointsText, argbText, TA_LEFT);
  }

//+------------------------------------------------------------------+
//| Draw legend                                                      |
//+------------------------------------------------------------------+
void drawLegend()
  {
   //--- Set legend X
   int legendX = statsPanelX;
   //--- Set legend Y
   int legendY = HEADER_BAR_HEIGHT + statsPanelY + statsPanelHeight;
   //--- Set legend width
   int legendWidth = statsPanelWidth;
   //--- Set legend height
   int legendHeightThis = legendHeight;

   //--- Compute bg color
   color legendBgColor = LightenColor(themeColor, 0.9);
   //--- Set alpha
   uchar bgAlpha = 153;
   //--- Get legend bg ARGB
   uint argbLegendBg = ColorToARGB(legendBgColor, bgAlpha);
   //--- Get border ARGB
   uint argbBorder = ColorToARGB(themeColor, 255);
   //--- Get text ARGB
   uint argbText = ColorToARGB(clrBlack, 255);

   //--- Loop over rows
   for (int y = legendY; y <= legendY + legendHeightThis; y++)
     {
      //--- Loop over columns
      for (int x = legendX; x <= legendX + legendWidth; x++)
        {
         //--- Blend bg pixel
         blendPixelSet(mainCanvas, x, y, argbLegendBg);
        }
     }

   //--- Draw top border
   for (int x = legendX; x <= legendX + legendWidth; x++)
     {
      //--- Blend border pixel
      blendPixelSet(mainCanvas, x, legendY, argbBorder);
     }
   //--- Draw right border
   for (int y = legendY; y <= legendY + legendHeightThis; y++)
     {
      //--- Blend border pixel
      blendPixelSet(mainCanvas, legendX + legendWidth, y, argbBorder);
     }
   //--- Draw bottom border
   for (int x = legendX; x <= legendX + legendWidth; x++)
     {
      //--- Blend border pixel
      blendPixelSet(mainCanvas, x, legendY + legendHeightThis, argbBorder);
     }
   //--- Draw left border
   for (int y = legendY; y <= legendY + legendHeightThis; y++)
     {
      //--- Blend border pixel
      blendPixelSet(mainCanvas, legendX, y, argbBorder);
     }

   //--- Set legend font
   mainCanvas.FontSet("Arial", panelFontSize);

   //--- Set item Y
   int itemY = legendY + 10;
   //--- Set line spacing
   int lineSpacing = panelFontSize;

   //--- Get red dot ARGB
   uint argbRedDot = ColorToARGB(dataPointsColor, 255);
   //--- Draw data point
   drawCirclePoint(legendX + 12, itemY, dataPointSize, argbRedDot);
   //--- Draw data label
   mainCanvas.TextOut(legendX + 22, itemY - 4, "Data Points", argbText, TA_LEFT);
   //--- Update Y
   itemY += lineSpacing;

   //--- Get blue line ARGB
   uint argbBlueLine = ColorToARGB(regressionLineColor, 255);
   //--- Loop to draw line
   for (int i = 0; i < 15; i++)
     {
      //--- Blend line pixel
      blendPixelSet(mainCanvas, legendX + 7 + i, itemY, argbBlueLine);
      //--- Blend below pixel
      blendPixelSet(mainCanvas, legendX + 7 + i, itemY + 1, argbBlueLine);
     }
   //--- Draw line label
   mainCanvas.TextOut(legendX + 27, itemY - 4, "Regression Line", argbText, TA_LEFT);
  }

//+------------------------------------------------------------------+
//| Draw resize indicator                                            |
//+------------------------------------------------------------------+
void drawResizeIndicator()
  {
   //--- Get indicator ARGB
   uint argbIndicator = ColorToARGB(themeColor, 255);

   //--- Check corner
   if (hoverResizeMode == RESIZE_CORNER || activeResizeMode == RESIZE_CORNER)
     {
      //--- Compute corner X
      int cornerX = currentWidthPixels - resizeGripSize;
      //--- Compute corner Y
      int cornerY = currentHeightPixels - resizeGripSize;

      //--- Fill corner
      mainCanvas.FillRectangle(cornerX, cornerY, currentWidthPixels - 1, currentHeightPixels - 1, argbIndicator);

      //--- Loop for lines
      for (int i = 0; i < 3; i++)
        {
         //--- Compute offset
         int offset = i * 3;
         //--- Draw diagonal
         mainCanvas.Line(cornerX + offset, currentHeightPixels - 1, 
                                  currentWidthPixels - 1, cornerY + offset, argbIndicator);
        }
     }

   //--- Check right
   if (hoverResizeMode == RESIZE_RIGHT_EDGE || activeResizeMode == RESIZE_RIGHT_EDGE)
     {
      //--- Compute indicator Y
      int indicatorY = currentHeightPixels / 2 - 15;
      //--- Fill right
      mainCanvas.FillRectangle(currentWidthPixels - 3, indicatorY, 
                                        currentWidthPixels - 1, indicatorY + 30, argbIndicator);
     }

   //--- Check bottom
   if (hoverResizeMode == RESIZE_BOTTOM_EDGE || activeResizeMode == RESIZE_BOTTOM_EDGE)
     {
      //--- Compute indicator X
      int indicatorX = currentWidthPixels / 2 - 15;
      //--- Fill bottom
      mainCanvas.FillRectangle(indicatorX, currentHeightPixels - 3, 
                                        indicatorX + 30, currentHeightPixels - 1, argbIndicator);
     }
  }

//+------------------------------------------------------------------+
//| Draw cyberpunk background                                        |
//+------------------------------------------------------------------+
void drawCyberpunkBackground()
  {
   //--- Set top color
   color topColor = C'10,10,25';
   //--- Set bottom color
   color bottomColor = C'20,5,30';
   
   //--- Compute alpha
   uchar alphaChannel = (uchar)(255 * backgroundOpacityLevel);
   
   //--- Loop over rows
   for (int y = HEADER_BAR_HEIGHT; y < currentHeightPixels; y++)
     {
      //--- Compute factor
      double factor = (double)(y - HEADER_BAR_HEIGHT) / (currentHeightPixels - HEADER_BAR_HEIGHT);
      //--- Blend row color
      color rowColor = BlendColors(topColor, bottomColor, factor);
      //--- Get ARGB
      uint argbColor = ColorToARGB(rowColor, alphaChannel);

      //--- Loop over columns
      for (int x = 0; x < currentWidthPixels; x++)
        {
         //--- Check for star
         if (MathRand() % 800 == 0)
           {
            //--- Compute brightness
            uchar brightness = (uchar)(100 + MathRand() % 155);
            //--- Get star color
            uint starColor = ColorToARGB(C'255,255,255', brightness);
            //--- Blend star
            blendPixelSet(mainCanvas, x, y, starColor);
           }
         else
           {
            //--- Blend normal
            blendPixelSet(mainCanvas, x, y, argbColor);
           }
        }
     }
  }

//+------------------------------------------------------------------+
//| Draw holographic border                                          |
//+------------------------------------------------------------------+
void drawHolographicBorder()
  {
   //--- Compute phase
   double phase = (currentAnimationFrame % 360) * M_PI / 180.0;
   //--- Compute glow factor
   double glowFactor = (MathSin(phase) + 1.0) * 0.5;
   
   //--- Blend border color
   color borderColor = BlendColors(CYBER_PRIMARY_COLOR, CYBER_SECONDARY_COLOR, glowFactor);
   
   //--- Loop for glow layers
   for (int i = 0; i < 3; i++)
     {
      //--- Compute alpha
      uchar alpha = (uchar)(80 * glowIntensity * (3 - i) / 3.0);
      //--- Get glow color
      uint glowColor = ColorToARGB(borderColor, alpha);
      
      //--- Draw top glow
      for (int x = i; x < currentWidthPixels - i; x++)
        {
         //--- Blend pixel
         blendPixelSet(mainCanvas, x, i, glowColor);
        }
      //--- Draw bottom glow
      for (int x = i; x < currentWidthPixels - i; x++)
        {
         //--- Blend pixel
         blendPixelSet(mainCanvas, x, currentHeightPixels - 1 - i, glowColor);
        }
      //--- Draw left glow
      for (int y = i; y < currentHeightPixels - i; y++)
        {
         //--- Blend pixel
         blendPixelSet(mainCanvas, i, y, glowColor);
        }
      //--- Draw right glow
      for (int y = i; y < currentHeightPixels - i; y++)
        {
         //--- Blend pixel
         blendPixelSet(mainCanvas, currentWidthPixels - 1 - i, y, glowColor);
        }
     }
   
   //--- Get border ARGB
   uint argbBorder = ColorToARGB(borderColor, 255);
   //--- Draw outer border
   mainCanvas.Rectangle(0, 0, currentWidthPixels - 1, currentHeightPixels - 1, argbBorder);
   //--- Draw inner border
   mainCanvas.Rectangle(1, 1, currentWidthPixels - 2, currentHeightPixels - 2, argbBorder);
   
   //--- Draw corner accents
   drawCornerAccents(borderColor);
  }

//+------------------------------------------------------------------+
//| Draw corner accents                                              |
//+------------------------------------------------------------------+
void drawCornerAccents(color accentColor)
  {
   //--- Get accent ARGB
   uint argbAccent = ColorToARGB(accentColor, 255);
   //--- Set accent size
   int accentSize = 15;
   
   //--- Loop for top-left
   for (int i = 0; i < 3; i++)
     {
      //--- Draw horizontal
      mainCanvas.Line(5, 5 + i, accentSize, 5 + i, argbAccent);
      //--- Draw vertical
      mainCanvas.Line(5 + i, 5, 5 + i, accentSize, argbAccent);
     }
   
   //--- Loop for top-right
   for (int i = 0; i < 3; i++)
     {
      //--- Draw horizontal
      mainCanvas.Line(currentWidthPixels - accentSize - 1, 5 + i, currentWidthPixels - 6, 5 + i, argbAccent);
      //--- Draw vertical
      mainCanvas.Line(currentWidthPixels - 6 - i, 5, currentWidthPixels - 6 - i, accentSize, argbAccent);
     }
   
   //--- Loop for bottom-left
   for (int i = 0; i < 3; i++)
     {
      //--- Draw horizontal
      mainCanvas.Line(5, currentHeightPixels - 6 - i, accentSize, currentHeightPixels - 6 - i, argbAccent);
      //--- Draw vertical
      mainCanvas.Line(5 + i, currentHeightPixels - accentSize - 1, 5 + i, currentHeightPixels - 6, argbAccent);
     }
   
   //--- Loop for bottom-right
   for (int i = 0; i < 3; i++)
     {
      //--- Draw horizontal
      mainCanvas.Line(currentWidthPixels - accentSize - 1, currentHeightPixels - 6 - i, currentWidthPixels - 6, currentHeightPixels - 6 - i, argbAccent);
      //--- Draw vertical
      mainCanvas.Line(currentWidthPixels - 6 - i, currentHeightPixels - accentSize - 1, currentWidthPixels - 6 - i, currentHeightPixels - 6, argbAccent);
     }
  }

//+------------------------------------------------------------------+
//| Draw futuristic header                                           |
//+------------------------------------------------------------------+
void drawFuturisticHeader()
  {
   //--- Compute phase
   double phase = (currentAnimationFrame % 360) * M_PI / 180.0;
   //--- Compute anim factor
   double animFactor = (MathSin(phase) + 1.0) * 0.5;
   
   //--- Loop over header rows
   for (int y = 0; y < HEADER_BAR_HEIGHT; y++)
     {
      //--- Compute grad factor
      double gradFactor = (double)y / HEADER_BAR_HEIGHT;
      
      //--- Declare header color
      color headerColor;
      //--- Check dragging
      if (isDraggingCanvas)
        {
         //--- Blend dragged
         headerColor = BlendColors(DarkenColor(CYBER_PRIMARY_COLOR, 0.7), DarkenColor(CYBER_SECONDARY_COLOR, 0.7), gradFactor);
        }
      //--- Check hovering
      else if (isHoveringHeader)
        {
         //--- Blend hovered
         headerColor = BlendColors(DarkenColor(CYBER_PRIMARY_COLOR, 0.5), DarkenColor(CYBER_SECONDARY_COLOR, 0.5), gradFactor);
        }
      else
        {
         //--- Blend normal
         headerColor = BlendColors(DarkenColor(CYBER_PRIMARY_COLOR, 0.8), DarkenColor(CYBER_SECONDARY_COLOR, 0.8), gradFactor);
        }
      
      //--- Get header ARGB
      uint argbHeader = ColorToARGB(headerColor, 255);
      
      //--- Loop over columns
      for (int x = 0; x < currentWidthPixels; x++)
        {
         //--- Set pixel
         mainCanvas.PixelSet(x, y, argbHeader);
        }
     }
   
   //--- Blend border
   color borderColor = BlendColors(CYBER_PRIMARY_COLOR, CYBER_SECONDARY_COLOR, animFactor);
   //--- Get border ARGB
   uint argbBorder = ColorToARGB(borderColor, 255);
   //--- Draw outer
   mainCanvas.Rectangle(0, 0, currentWidthPixels - 1, HEADER_BAR_HEIGHT, argbBorder);
   //--- Draw inner
   mainCanvas.Rectangle(1, 1, currentWidthPixels - 2, HEADER_BAR_HEIGHT - 1, argbBorder);
   
   //--- Set title font
   mainCanvas.FontSet("Arial Black", titleFontSize);
   
   //--- Format title
   string titleText = StringFormat("◆ %s vs %s - Linear Regression ◆", secondarySymbol, primarySymbol);
   
   //--- Get glow ARGB
   uint glowColor = ColorToARGB(CYBER_PRIMARY_COLOR, (uchar)(120 * glowIntensity));
   //--- Loop for glow Y
   for (int offsetY = -1; offsetY <= 1; offsetY++)
     {
      //--- Loop for glow X
      for (int offsetX = -1; offsetX <= 1; offsetX++)
        {
         //--- Skip center
         if (offsetX == 0 && offsetY == 0) continue;
         //--- Draw glow text
         mainCanvas.TextOut(currentWidthPixels / 2 + offsetX, (HEADER_BAR_HEIGHT - titleFontSize) / 2 + offsetY, 
                                     titleText, glowColor, TA_CENTER);
        }
     }
   
   //--- Set text color
   uint textColor = ColorToARGB(C'255,105,180', 255);
   //--- Draw title
   mainCanvas.TextOut(currentWidthPixels / 2, (HEADER_BAR_HEIGHT - titleFontSize) / 2, 
                                titleText, textColor, TA_CENTER);
  }

//+------------------------------------------------------------------+
//| Draw regression plot cyberpunk                                   |
//+------------------------------------------------------------------+
void drawRegressionPlotCyberpunk()
  {
   //--- Return if no data
   if (!dataLoadedSuccessfully) return;

   //--- Set plot left
   int plotAreaLeft = 60;
   //--- Set plot right
   int plotAreaRight = currentWidthPixels - 40;
   //--- Set plot top
   int plotAreaTop = HEADER_BAR_HEIGHT + 10;
   //--- Set plot bottom
   int plotAreaBottom = currentHeightPixels - 50;

   //--- Set draw left
   int drawAreaLeft = plotAreaLeft + plotPadding;
   //--- Set draw right
   int drawAreaRight = plotAreaRight - plotPadding;
   //--- Set draw top
   int drawAreaTop = plotAreaTop + plotPadding;
   //--- Set draw bottom
   int drawAreaBottom = plotAreaBottom - plotPadding;

   //--- Compute plot width
   int plotWidth = drawAreaRight - drawAreaLeft;
   //--- Compute plot height
   int plotHeight = drawAreaBottom - drawAreaTop;

   //--- Return if invalid
   if (plotWidth <= 0 || plotHeight <= 0) return;

   //--- Init min X
   double minX = primaryClosePrices[0];
   //--- Init max X
   double maxX = primaryClosePrices[0];
   //--- Init min Y
   double minY = secondaryClosePrices[0];
   //--- Init max Y
   double maxY = secondaryClosePrices[0];

   //--- Get data points
   int dataPoints = ArraySize(primaryClosePrices);
   //--- Loop over points
   for (int i = 1; i < dataPoints; i++)
     {
      //--- Update min X
      if (primaryClosePrices[i] < minX) minX = primaryClosePrices[i];
      //--- Update max X
      if (primaryClosePrices[i] > maxX) maxX = primaryClosePrices[i];
      //--- Update min Y
      if (secondaryClosePrices[i] < minY) minY = secondaryClosePrices[i];
      //--- Update max Y
      if (secondaryClosePrices[i] > maxY) maxY = secondaryClosePrices[i];
     }

   //--- Compute range X
   double rangeX = maxX - minX;
   //--- Compute range Y
   double rangeY = maxY - minY;

   //--- Set min range X
   if (rangeX == 0) rangeX = 1;
   //--- Set min range Y
   if (rangeY == 0) rangeY = 1;

   //--- Draw grid
   drawCyberpunkGrid(drawAreaLeft, drawAreaTop, plotWidth, plotHeight);

   //--- Set axis color
   color axisColor = CYBER_PRIMARY_COLOR;
   //--- Get axis ARGB
   uint argbAxis = ColorToARGB(axisColor, 255);
   
   //--- Get glow ARGB
   uint glowColor = ColorToARGB(axisColor, (uchar)(60 * glowIntensity));
   //--- Loop for glow
   for (int g = 1; g <= 2; g++)
     {
      //--- Draw Y glow
      for (int y = plotAreaTop; y <= plotAreaBottom; y++)
        {
         //--- Blend left glow
         blendPixelSet(mainCanvas, plotAreaLeft - g - 1, y, glowColor);
         //--- Blend right glow
         blendPixelSet(mainCanvas, plotAreaLeft + g + 1, y, glowColor);
        }
      //--- Draw X glow
      for (int x = plotAreaLeft; x <= plotAreaRight; x++)
        {
         //--- Blend bottom glow
         blendPixelSet(mainCanvas, x, plotAreaBottom + g + 1, glowColor);
         //--- Blend top glow
         blendPixelSet(mainCanvas, x, plotAreaBottom - g - 1, glowColor);
        }
     }
   
   //--- Loop for thick Y-axis
   for (int thick = 0; thick < 2; thick++)
     {
      //--- Draw Y-axis line
      mainCanvas.Line(plotAreaLeft - thick, plotAreaTop, plotAreaLeft - thick, plotAreaBottom, argbAxis);
     }
   
   //--- Loop for thick X-axis
   for (int thick = 0; thick < 2; thick++)
     {
      //--- Draw X-axis line
      mainCanvas.Line(plotAreaLeft, plotAreaBottom + thick, plotAreaRight, plotAreaBottom + thick, argbAxis);
     }

   //--- Set tick font
   mainCanvas.FontSet("Consolas", axisLabelFontSize);
   //--- Get tick label ARGB
   uint argbTickLabel = ColorToARGB(CYBER_PRIMARY_COLOR, 255);
   
   //--- Declare Y ticks
   double yTickValues[];
   //--- Compute Y ticks
   int numYTicks = calculateOptimalTicks(minY, maxY, plotHeight, yTickValues);
   
   //--- Loop over Y ticks
   for (int i = 0; i < numYTicks; i++)
     {
      //--- Get Y value
      double yValue = yTickValues[i];
      //--- Skip out of range
      if (yValue < minY || yValue > maxY) continue;
      
      //--- Compute Y pos
      int yPos = drawAreaBottom - (int)((yValue - minY) / rangeY * plotHeight);
      
      //--- Draw tick
      mainCanvas.Line(plotAreaLeft - 5, yPos, plotAreaLeft, yPos, argbAxis);
      
      //--- Format label
      string yLabel = formatTickLabel(yValue, rangeY);
      //--- Draw label
      mainCanvas.TextOut(plotAreaLeft - 8, yPos - axisLabelFontSize/2, yLabel, argbTickLabel, TA_RIGHT);
     }

   //--- Declare X ticks
   double xTickValues[];
   //--- Compute X ticks
   int numXTicks = calculateOptimalTicks(minX, maxX, plotWidth, xTickValues);
   
   //--- Loop over X ticks
   for (int i = 0; i < numXTicks; i++)
     {
      //--- Get X value
      double xValue = xTickValues[i];
      //--- Skip out of range
      if (xValue < minX || xValue > maxX) continue;
      
      //--- Compute X pos
      int xPos = drawAreaLeft + (int)((xValue - minX) / rangeX * plotWidth);
      
      //--- Draw tick
      mainCanvas.Line(xPos, plotAreaBottom, xPos, plotAreaBottom + 5, argbAxis);
      
      //--- Format label
      string xLabel = formatTickLabel(xValue, rangeX);
      //--- Draw label
      mainCanvas.TextOut(xPos, plotAreaBottom + 7, xLabel, argbTickLabel, TA_CENTER);
     }

   //--- Compute start Y
   double lineStartY = regressionIntercept + regressionSlope * minX;
   //--- Compute end Y
   double lineEndY = regressionIntercept + regressionSlope * maxX;

   //--- Set start screen X
   int lineStartScreenX = drawAreaLeft;
   //--- Compute start screen Y
   int lineStartScreenY = drawAreaBottom - (int)((lineStartY - minY) / rangeY * plotHeight);
   //--- Set end screen X
   int lineEndScreenX = drawAreaRight;
   //--- Compute end screen Y
   int lineEndScreenY = drawAreaBottom - (int)((lineEndY - minY) / rangeY * plotHeight);

   //--- Get line ARGB
   uint argbLine = ColorToARGB(CYBER_REGRESSION_COLOR, 255);
   //--- Loop for width
   for (int w = 0; w < regressionLineWidth; w++)
     {
      //--- Draw line
      mainCanvas.LineAA(lineStartScreenX, lineStartScreenY + w - regressionLineWidth/2, 
                                  lineEndScreenX, lineEndScreenY + w - regressionLineWidth/2, argbLine);
     }

   //--- Compute pulse phase
   double pulsePhase = (currentAnimationFrame % 60) * M_PI / 30.0;
   //--- Compute pulse factor
   double pulseFactor = (MathSin(pulsePhase) + 1.0) * 0.3 + 0.7;
   //--- Compute point size
   int effectivePointSize = (int)(dataPointSize * pulseFactor);
   
   //--- Loop over points
   for (int i = 0; i < dataPoints; i++)
     {
      //--- Compute screen X
      int screenX = drawAreaLeft + (int)((primaryClosePrices[i] - minX) / rangeX * plotWidth);
      //--- Compute screen Y
      int screenY = drawAreaBottom - (int)((secondaryClosePrices[i] - minY) / rangeY * plotHeight);

      //--- Draw neon point
      drawNeonPoint(screenX, screenY, effectivePointSize, CYBER_DATA_POINTS_COLOR);
     }

   //--- Set axis label font
   mainCanvas.FontSet("Arial Bold", labelFontSize);
   //--- Get axis label ARGB
   uint argbAxisLabel = ColorToARGB(CYBER_SECONDARY_COLOR, 255);

   //--- Set X label
   string xAxisLabel = "► " + primarySymbol + " (X-AXIS)";
   //--- Draw X label
   mainCanvas.TextOut(currentWidthPixels / 2, currentHeightPixels - 20, xAxisLabel, argbAxisLabel, TA_CENTER);

   //--- Set Y label
   string yAxisLabel = "► " + secondarySymbol + " (Y-AXIS)";
   //--- Set vertical angle
   mainCanvas.FontAngleSet(900);
   //--- Draw Y label
   mainCanvas.TextOut(12, currentHeightPixels / 2, yAxisLabel, argbAxisLabel, TA_CENTER);
   //--- Reset angle
   mainCanvas.FontAngleSet(0);
  }

//+------------------------------------------------------------------+
//| Draw cyberpunk grid                                              |
//+------------------------------------------------------------------+
void drawCyberpunkGrid(int startX, int startY, int width, int height)
  {
   //--- Set grid spacing
   int gridSpacing = 40;
   //--- Compute phase
   double phase = (currentAnimationFrame % 120) * M_PI / 60.0;
   //--- Compute alpha
   uchar gridAlpha = (uchar)(30 + 15 * MathSin(phase));
   
   //--- Set grid color
   color gridColor = CYBER_PRIMARY_COLOR;
   //--- Get grid ARGB
   uint argbGrid = ColorToARGB(gridColor, gridAlpha);
   
   //--- Loop vertical lines
   for (int x = startX; x < startX + width; x += gridSpacing)
     {
      //--- Loop over height
      for (int y = startY; y < startY + height; y++)
        {
         //--- Blend grid pixel
         blendPixelSet(mainCanvas, x, y, argbGrid);
        }
     }
   
   //--- Loop horizontal lines
   for (int y = startY; y < startY + height; y += gridSpacing)
     {
      //--- Loop over width
      for (int x = startX; x < startX + width; x++)
        {
         //--- Blend grid pixel
         blendPixelSet(mainCanvas, x, y, argbGrid);
        }
     }
  }

//+------------------------------------------------------------------+
//| Draw neon point                                                  |
//+------------------------------------------------------------------+
void drawNeonPoint(int centerX, int centerY, int radius, color pointColor)
  {
   //--- Loop for glow
   for (int glowRadius = radius + 4; glowRadius > radius; glowRadius--)
     {
      //--- Compute alpha
      uchar glowAlpha = (uchar)(80 * glowIntensity * (radius + 4 - glowRadius) / 4.0);
      //--- Get glow color
      uint glowColor = ColorToARGB(pointColor, glowAlpha);
      //--- Draw glow circle
      drawCirclePoint(centerX, centerY, glowRadius, glowColor);
     }
   
   //--- Get point ARGB
   uint argbPoint = ColorToARGB(pointColor, 255);
   //--- Draw point circle
   drawCirclePoint(centerX, centerY, radius, argbPoint);
   
   //--- Get center ARGB
   uint argbCenter = ColorToARGB(C'255,255,255', 200);
   //--- Draw center circle
   drawCirclePoint(centerX, centerY, radius - 1, argbCenter);
  }

//+------------------------------------------------------------------+
//| Draw glass morphism stats panel                                  |
//+------------------------------------------------------------------+
void drawGlassMorphismStatsPanel()
  {
   //--- Set panel X
   int panelX = statsPanelX;
   //--- Set panel Y
   int panelY = HEADER_BAR_HEIGHT + statsPanelY;
   //--- Set panel width
   int panelWidth = statsPanelWidth;
   //--- Set panel height
   int panelHeight = statsPanelHeight;

   //--- Compute bg color
   color panelBgColor = DarkenColor(CYBER_PRIMARY_COLOR, 0.85);
   //--- Set alpha
   uchar bgAlpha = 120;
   //--- Get panel bg ARGB
   uint argbPanelBg = ColorToARGB(panelBgColor, bgAlpha);
   
   //--- Loop over rows
   for (int y = panelY; y <= panelY + panelHeight; y++)
     {
      //--- Loop over columns
      for (int x = panelX; x <= panelX + panelWidth; x++)
        {
         //--- Blend bg pixel
         blendPixelSet(mainCanvas, x, y, argbPanelBg);
        }
     }
   
   //--- Blend border
   color borderColor = BlendColors(CYBER_PRIMARY_COLOR, CYBER_SECONDARY_COLOR, 0.5);
   //--- Get border ARGB
   uint argbBorder = ColorToARGB(borderColor, 255);
   
   //--- Get glow ARGB
   uint glowColor = ColorToARGB(borderColor, (uchar)(60 * glowIntensity));
   //--- Loop for glow
   for (int g = 1; g <= 2; g++)
     {
      //--- Draw top glow
      for (int x = panelX - g; x <= panelX + panelWidth + g; x++)
        {
         //--- Blend pixel
         blendPixelSet(mainCanvas, x, panelY - g, glowColor);
        }
      //--- Draw bottom glow
      for (int x = panelX - g; x <= panelX + panelWidth + g; x++)
        {
         //--- Blend pixel
         blendPixelSet(mainCanvas, x, panelY + panelHeight + g, glowColor);
        }
      //--- Draw left glow
      for (int y = panelY - g; y <= panelY + panelHeight + g; y++)
        {
         //--- Blend pixel
         blendPixelSet(mainCanvas, panelX - g, y, glowColor);
        }
      //--- Draw right glow
      for (int y = panelY - g; y <= panelY + panelHeight + g; y++)
        {
         //--- Blend pixel
         blendPixelSet(mainCanvas, panelX + panelWidth + g, y, glowColor);
        }
     }
   
   //--- Draw top border
   for (int x = panelX; x <= panelX + panelWidth; x++)
     {
      //--- Blend border pixel
      blendPixelSet(mainCanvas, x, panelY, argbBorder);
      //--- Blend bottom border
      blendPixelSet(mainCanvas, x, panelY + panelHeight, argbBorder);
     }
   //--- Draw left and right borders
   for (int y = panelY; y <= panelY + panelHeight; y++)
     {
      //--- Blend left
      blendPixelSet(mainCanvas, panelX, y, argbBorder);
      //--- Blend right
      blendPixelSet(mainCanvas, panelX + panelWidth, y, argbBorder);
     }

   //--- Set stats font
   mainCanvas.FontSet("Consolas", panelFontSize);
   //--- Get text ARGB
   uint argbText = ColorToARGB(CYBER_PRIMARY_COLOR, 255);

   //--- Set text Y
   int textY = panelY + 8;
   //--- Set line spacing
   int lineSpacing = panelFontSize;

   //--- Format equation
   string equationText = StringFormat("Y = %.3f + %.3f * X", regressionIntercept, regressionSlope);
   //--- Draw equation
   mainCanvas.TextOut(panelX + 8, textY, equationText, argbText, TA_LEFT);
   //--- Update Y
   textY += lineSpacing;

   //--- Format correlation
   string correlationText = StringFormat("Correlation: %.4f", correlationCoefficient);
   //--- Draw correlation
   mainCanvas.TextOut(panelX + 8, textY, correlationText, argbText, TA_LEFT);
   //--- Update Y
   textY += lineSpacing;

   //--- Format R-squared
   string rSquaredText = StringFormat("R-Squared: %.4f", rSquared);
   //--- Draw R-squared
   mainCanvas.TextOut(panelX + 8, textY, rSquaredText, argbText, TA_LEFT);
   //--- Update Y
   textY += lineSpacing;

   //--- Format points
   string dataPointsText = StringFormat("Points: %d", ArraySize(primaryClosePrices));
   //--- Draw points
   mainCanvas.TextOut(panelX + 8, textY, dataPointsText, argbText, TA_LEFT);
  }

//+------------------------------------------------------------------+
//| Draw glass morphism legend                                       |
//+------------------------------------------------------------------+
void drawGlassMorphismLegend()
  {
   //--- Set legend X
   int legendX = statsPanelX;
   //--- Set legend Y
   int legendY = HEADER_BAR_HEIGHT + statsPanelY + statsPanelHeight;
   //--- Set legend width
   int legendWidth = statsPanelWidth;
   //--- Set legend height
   int legendHeightThis = legendHeight;

   //--- Compute bg color
   color legendBgColor = DarkenColor(CYBER_SECONDARY_COLOR, 0.85);
   //--- Set alpha
   uchar bgAlpha = 120;
   //--- Get legend bg ARGB
   uint argbLegendBg = ColorToARGB(legendBgColor, bgAlpha);
   
   //--- Loop over rows
   for (int y = legendY; y <= legendY + legendHeightThis; y++)
     {
      //--- Loop over columns
      for (int x = legendX; x <= legendX + legendWidth; x++)
        {
         //--- Blend bg pixel
         blendPixelSet(mainCanvas, x, y, argbLegendBg);
        }
     }
   
   //--- Blend border
   color borderColor = BlendColors(CYBER_SECONDARY_COLOR, CYBER_PRIMARY_COLOR, 0.5);
   //--- Get border ARGB
   uint argbBorder = ColorToARGB(borderColor, 255);
   
   //--- Get glow ARGB
   uint glowColor = ColorToARGB(borderColor, (uchar)(60 * glowIntensity));
   //--- Loop for glow
   for (int g = 1; g <= 2; g++)
     {
      //--- Draw top and bottom glow
      for (int x = legendX - g; x <= legendX + legendWidth + g; x++)
        {
         //--- Blend top
         blendPixelSet(mainCanvas, x, legendY - g, glowColor);
         //--- Blend bottom
         blendPixelSet(mainCanvas, x, legendY + legendHeightThis + g, glowColor);
        }
      //--- Draw left and right glow
      for (int y = legendY - g; y <= legendY + legendHeightThis + g; y++)
        {
         //--- Blend left
         blendPixelSet(mainCanvas, legendX - g, y, glowColor);
         //--- Blend right
         blendPixelSet(mainCanvas, legendX + legendWidth + g, y, glowColor);
        }
     }
   
   //--- Draw top and bottom borders
   for (int x = legendX; x <= legendX + legendWidth; x++)
     {
      //--- Blend top
      blendPixelSet(mainCanvas, x, legendY, argbBorder);
      //--- Blend bottom
      blendPixelSet(mainCanvas, x, legendY + legendHeightThis, argbBorder);
     }
   //--- Draw left and right borders
   for (int y = legendY; y <= legendY + legendHeightThis; y++)
     {
      //--- Blend left
      blendPixelSet(mainCanvas, legendX, y, argbBorder);
      //--- Blend right
      blendPixelSet(mainCanvas, legendX + legendWidth, y, argbBorder);
     }

   //--- Set legend font
   mainCanvas.FontSet("Consolas", panelFontSize);
   //--- Get text ARGB
   uint argbText = ColorToARGB(CYBER_SECONDARY_COLOR, 255);

   //--- Set item Y
   int itemY = legendY + 10;
   //--- Set line spacing
   int lineSpacing = panelFontSize;

   //--- Draw neon point
   drawNeonPoint(legendX + 12, itemY, dataPointSize, CYBER_DATA_POINTS_COLOR);
   //--- Draw data label
   mainCanvas.TextOut(legendX + 22, itemY - 4, "Data Points", argbText, TA_LEFT);
   //--- Update Y
   itemY += lineSpacing;

   //--- Get line color
   uint argbLineColor = ColorToARGB(CYBER_REGRESSION_COLOR, 255);
   //--- Loop to draw line
   for (int i = 0; i < 15; i++)
     {
      //--- Blend line pixel
      blendPixelSet(mainCanvas, legendX + 7 + i, itemY, argbLineColor);
      //--- Blend below pixel
      blendPixelSet(mainCanvas, legendX + 7 + i, itemY + 1, argbLineColor);
     }
   //--- Draw line label
   mainCanvas.TextOut(legendX + 27, itemY - 4, "Regression Line", argbText, TA_LEFT);
  }

//+------------------------------------------------------------------+
//| Draw neon resize indicator                                       |
//+------------------------------------------------------------------+
void drawNeonResizeIndicator()
  {
   //--- Blend indicator color
   color indicatorColor = BlendColors(CYBER_PRIMARY_COLOR, CYBER_SECONDARY_COLOR, 0.5);
   //--- Get indicator ARGB
   uint argbIndicator = ColorToARGB(indicatorColor, 200);
   
   //--- Get glow ARGB
   uint glowColor = ColorToARGB(indicatorColor, (uchar)(100 * glowIntensity));
   
   //--- Check corner
   if (hoverResizeMode == RESIZE_CORNER || activeResizeMode == RESIZE_CORNER)
     {
      //--- Compute corner X
      int cornerX = currentWidthPixels - resizeGripSize;
      //--- Compute corner Y
      int cornerY = currentHeightPixels - resizeGripSize;
      
      //--- Loop for glow
      for (int g = 1; g <= 3; g++)
        {
         //--- Loop Y glow
         for (int y = cornerY - g; y < currentHeightPixels + g; y++)
           {
            //--- Loop X glow
            for (int x = cornerX - g; x < currentWidthPixels + g; x++)
              {
               //--- Blend glow
               blendPixelSet(mainCanvas, x, y, glowColor);
              }
           }
        }
      
      //--- Fill corner
      mainCanvas.FillRectangle(cornerX, cornerY, currentWidthPixels - 1, currentHeightPixels - 1, argbIndicator);
      
      //--- Set line color
      uint lineColor = ColorToARGB(C'255,255,255', 255);
      //--- Loop for lines
      for (int i = 0; i < 4; i++)
        {
         //--- Compute offset
         int offset = i * 3;
         //--- Draw diagonal
         mainCanvas.Line(cornerX + offset, currentHeightPixels - 1, 
                                  currentWidthPixels - 1, cornerY + offset, lineColor);
        }
     }
   
   //--- Check right
   if (hoverResizeMode == RESIZE_RIGHT_EDGE || activeResizeMode == RESIZE_RIGHT_EDGE)
     {
      //--- Compute indicator Y
      int indicatorY = currentHeightPixels / 2 - 15;
      //--- Loop for glow
      for (int g = 1; g <= 3; g++)
        {
         //--- Fill glow
         mainCanvas.FillRectangle(currentWidthPixels - 3 - g, indicatorY - g, 
                                          currentWidthPixels - 1 + g, indicatorY + 30 + g, glowColor);
        }
      //--- Fill indicator
      mainCanvas.FillRectangle(currentWidthPixels - 3, indicatorY, 
                                       currentWidthPixels - 1, indicatorY + 30, argbIndicator);
     }
   
   //--- Check bottom
   if (hoverResizeMode == RESIZE_BOTTOM_EDGE || activeResizeMode == RESIZE_BOTTOM_EDGE)
     {
      //--- Compute indicator X
      int indicatorX = currentWidthPixels / 2 - 15;
      //--- Loop for glow
      for (int g = 1; g <= 3; g++)
        {
         //--- Fill glow
         mainCanvas.FillRectangle(indicatorX - g, currentHeightPixels - 3 - g, 
                                          indicatorX + 30 + g, currentHeightPixels - 1 + g, glowColor);
        }
      //--- Fill indicator
      mainCanvas.FillRectangle(indicatorX, currentHeightPixels - 3, 
                                       indicatorX + 30, currentHeightPixels - 1, argbIndicator);
     }
  }

//+------------------------------------------------------------------+
//| Check mouse over header                                          |
//+------------------------------------------------------------------+
bool isMouseOverHeaderBar(int mouseX, int mouseY)
  {
   //--- Return if over header
   return (mouseX >= currentPositionX && mouseX <= currentPositionX + currentWidthPixels &&
           mouseY >= currentPositionY && mouseY <= currentPositionY + HEADER_BAR_HEIGHT);
  }

//+------------------------------------------------------------------+
//| Check mouse over theme button                                    |
//+------------------------------------------------------------------+
bool isMouseOverThemeButton(int mouseX, int mouseY)
  {
   //--- Compute relative X
   int relX = mouseX - currentPositionX;
   //--- Compute relative Y
   int relY = mouseY - currentPositionY;
   
   //--- Return if over button
   return (relX >= themeToggleButtonX && relX <= themeToggleButtonX + THEME_TOGGLE_BUTTON_SIZE &&
           relY >= themeToggleButtonY && relY <= themeToggleButtonY + THEME_TOGGLE_BUTTON_SIZE);
  }

//+------------------------------------------------------------------+
//| Check mouse in resize zone                                       |
//+------------------------------------------------------------------+
bool isMouseInResizeZone(int mouseX, int mouseY, ResizeDirection &resizeMode)
  {
   //--- Return false if disabled
   if (!enableResizing) return false;

   //--- Compute relative X
   int relativeX = mouseX - currentPositionX;
   //--- Compute relative Y
   int relativeY = mouseY - currentPositionY;

   //--- Check right edge
   bool nearRightEdge = (relativeX >= currentWidthPixels - resizeGripSize && 
                         relativeX <= currentWidthPixels &&
                         relativeY >= HEADER_BAR_HEIGHT && 
                         relativeY <= currentHeightPixels);

   //--- Check bottom edge
   bool nearBottomEdge = (relativeY >= currentHeightPixels - resizeGripSize && 
                          relativeY <= currentHeightPixels &&
                          relativeX >= 0 && 
                          relativeX <= currentWidthPixels);

   //--- Check corner
   bool nearCorner = (relativeX >= currentWidthPixels - resizeGripSize && 
                      relativeX <= currentWidthPixels &&
                      relativeY >= currentHeightPixels - resizeGripSize && 
                      relativeY <= currentHeightPixels);

   //--- Set corner
   if (nearCorner)
     {
      //--- Set mode
      resizeMode = RESIZE_CORNER;
      //--- Return true
      return true;
     }
   //--- Set right
   else if (nearRightEdge)
     {
      //--- Set mode
      resizeMode = RESIZE_RIGHT_EDGE;
      //--- Return true
      return true;
     }
   //--- Set bottom
   else if (nearBottomEdge)
     {
      //--- Set mode
      resizeMode = RESIZE_BOTTOM_EDGE;
      //--- Return true
      return true;
     }

   //--- Set no resize
   resizeMode = NO_RESIZE;
   //--- Return false
   return false;
  }

//+------------------------------------------------------------------+
//| Handle canvas resize                                             |
//+------------------------------------------------------------------+
void handleCanvasResize(int mouseX, int mouseY)
  {
   //--- Compute delta X
   int deltaX = mouseX - resizeStartX;
   //--- Compute delta Y
   int deltaY = mouseY - resizeStartY;

   //--- Init new width
   int newWidth = currentWidthPixels;
   //--- Init new height
   int newHeight = currentHeightPixels;

   //--- Check right or corner
   if (activeResizeMode == RESIZE_RIGHT_EDGE || activeResizeMode == RESIZE_CORNER)
     {
      //--- Compute new width
      newWidth = MathMax(MIN_CANVAS_WIDTH, resizeInitialWidth + deltaX);
     }

   //--- Check bottom or corner
   if (activeResizeMode == RESIZE_BOTTOM_EDGE || activeResizeMode == RESIZE_CORNER)
     {
      //--- Compute new height
      newHeight = MathMax(MIN_CANVAS_HEIGHT, resizeInitialHeight + deltaY);
     }

   //--- Get chart width
   int chartWidth = (int)ChartGetInteger(0, CHART_WIDTH_IN_PIXELS);
   //--- Get chart height
   int chartHeight = (int)ChartGetInteger(0, CHART_HEIGHT_IN_PIXELS);

   //--- Clamp width
   newWidth = MathMin(newWidth, chartWidth - currentPositionX - 10);
   //--- Clamp height
   newHeight = MathMin(newHeight, chartHeight - currentPositionY - 10);

   //--- Check changed
   if (newWidth != currentWidthPixels || newHeight != currentHeightPixels)
     {
      //--- Update width
      currentWidthPixels = newWidth;
      //--- Update height
      currentHeightPixels = newHeight;

      //--- Resize canvas
      mainCanvas.Resize(currentWidthPixels, currentHeightPixels);
      //--- Set X size
      ObjectSetInteger(0, canvasObjectName, OBJPROP_XSIZE, currentWidthPixels);
      //--- Set Y size
      ObjectSetInteger(0, canvasObjectName, OBJPROP_YSIZE, currentHeightPixels);

      //--- Render again
      renderVisualization();
      //--- Redraw chart
      ChartRedraw();
     }
  }

//+------------------------------------------------------------------+
//| Handle canvas drag                                               |
//+------------------------------------------------------------------+
void handleCanvasDrag(int mouseX, int mouseY)
  {
   //--- Compute delta X
   int deltaX = mouseX - dragStartX;
   //--- Compute delta Y
   int deltaY = mouseY - dragStartY;

   //--- Compute new X
   int newX = canvasStartX + deltaX;
   //--- Compute new Y
   int newY = canvasStartY + deltaY;

   //--- Get chart width
   int chartWidth = (int)ChartGetInteger(0, CHART_WIDTH_IN_PIXELS);
   //--- Get chart height
   int chartHeight = (int)ChartGetInteger(0, CHART_HEIGHT_IN_PIXELS);

   //--- Clamp X
   newX = MathMax(0, MathMin(chartWidth - currentWidthPixels, newX));
   //--- Clamp Y
   newY = MathMax(0, MathMin(chartHeight - currentHeightPixels, newY));

   //--- Update X
   currentPositionX = newX;
   //--- Update Y
   currentPositionY = newY;

   //--- Set X distance
   ObjectSetInteger(0, canvasObjectName, OBJPROP_XDISTANCE, currentPositionX);
   //--- Set Y distance
   ObjectSetInteger(0, canvasObjectName, OBJPROP_YDISTANCE, currentPositionY);

   //--- Redraw chart
   ChartRedraw();
  }

//+------------------------------------------------------------------+
//| Interpolate colors                                               |
//+------------------------------------------------------------------+
color InterpolateColors(color startColor, color endColor, double factor)
  {
   //--- Extract start red
   uchar r1 = (uchar)((startColor >> 16) & 0xFF);
   //--- Extract start green
   uchar g1 = (uchar)((startColor >> 8) & 0xFF);
   //--- Extract start blue
   uchar b1 = (uchar)(startColor & 0xFF);

   //--- Extract end red
   uchar r2 = (uchar)((endColor >> 16) & 0xFF);
   //--- Extract end green
   uchar g2 = (uchar)((endColor >> 8) & 0xFF);
   //--- Extract end blue
   uchar b2 = (uchar)(endColor & 0xFF);

   //--- Interpolate red
   uchar r = (uchar)(r1 + factor * (r2 - r1));
   //--- Interpolate green
   uchar g = (uchar)(g1 + factor * (g2 - g1));
   //--- Interpolate blue
   uchar b = (uchar)(b1 + factor * (b2 - b1));

   //--- Return interpolated color
   return (r << 16) | (g << 8) | b;
  }

//+------------------------------------------------------------------+
//| Blend pixel set                                                  |
//+------------------------------------------------------------------+
void blendPixelSet(CCanvas &canvas, int x, int y, uint src)
  {
   //--- Return if out of bounds
   if (x < 0 || x >= canvas.Width() || y < 0 || y >= canvas.Height()) return;
   
   //--- Get destination pixel
   uint dst = canvas.PixelGet(x, y);
   
   //--- Compute source alpha
   double sa = ((src >> 24) & 0xFF) / 255.0;
   //--- Compute source red
   double sr = ((src >> 16) & 0xFF) / 255.0;
   //--- Compute source green
   double sg = ((src >> 8) & 0xFF) / 255.0;
   //--- Compute source blue
   double sb = (src & 0xFF) / 255.0;
   
   //--- Compute dest alpha
   double da = ((dst >> 24) & 0xFF) / 255.0;
   //--- Compute dest red
   double dr = ((dst >> 16) & 0xFF) / 255.0;
   //--- Compute dest green
   double dg = ((dst >> 8) & 0xFF) / 255.0;
   //--- Compute dest blue
   double db = (dst & 0xFF) / 255.0;
   
   //--- Compute out alpha
   double out_a = sa + da * (1 - sa);
   //--- Check transparent
   if (out_a == 0)
     {
      //--- Set transparent
      canvas.PixelSet(x, y, 0);
      //--- Return
      return;
     }
   
   //--- Compute out red
   double out_r = (sr * sa + dr * da * (1 - sa)) / out_a;
   //--- Compute out green
   double out_g = (sg * sa + dg * da * (1 - sa)) / out_a;
   //--- Compute out blue
   double out_b = (sb * sa + db * da * (1 - sa)) / out_a;
   
   //--- Compute final alpha
   uchar oa = (uchar)(out_a * 255 + 0.5);
   //--- Compute final red
   uchar or_ = (uchar)(out_r * 255 + 0.5);
   //--- Compute final green
   uchar og = (uchar)(out_g * 255 + 0.5);
   //--- Compute final blue
   uchar ob = (uchar)(out_b * 255 + 0.5);
   
   //--- Compose color
   uint out_col = ((uint)oa << 24) | ((uint)or_ << 16) | ((uint)og << 8) | (uint)ob;
   //--- Set blended pixel
   canvas.PixelSet(x, y, out_col);
  }

//+------------------------------------------------------------------+
//| Handle chart event                                               |
//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
  {
   //--- Check mouse move
   if (id == CHARTEVENT_MOUSE_MOVE)
     {
      //--- Set mouse X
      int mouseX = (int)lparam;
      //--- Set mouse Y
      int mouseY = (int)dparam;
      //--- Set mouse state
      int mouseState = (int)sparam;

      //--- Store prev canvas hover
      bool previousHoverState = isHoveringCanvas;
      //--- Store prev header hover
      bool previousHeaderHoverState = isHoveringHeader;
      //--- Store prev resize hover
      bool previousResizeHoverState = isHoveringResizeZone;
      //--- Store prev theme hover
      bool previousThemeButtonHover = isHoveringThemeButton;

      //--- Update canvas hover
      isHoveringCanvas = (mouseX >= currentPositionX && mouseX <= currentPositionX + currentWidthPixels &&
                         mouseY >= currentPositionY && mouseY <= currentPositionY + currentHeightPixels);

      //--- Update header hover
      isHoveringHeader = isMouseOverHeaderBar(mouseX, mouseY);
      //--- Update theme hover
      isHoveringThemeButton = isMouseOverThemeButton(mouseX, mouseY);
      //--- Update resize hover
      isHoveringResizeZone = isMouseInResizeZone(mouseX, mouseY, hoverResizeMode);

      //--- Check if redraw needed
      bool needRedraw = (previousHoverState != isHoveringCanvas || 
                        previousHeaderHoverState != isHoveringHeader ||
                        previousResizeHoverState != isHoveringResizeZone ||
                        previousThemeButtonHover != isHoveringThemeButton);

      //--- Check button press
      if (mouseState == 1 && previousMouseButtonState == 0)
        {
         //--- Check theme button
         if (isHoveringThemeButton)
           {
            //--- Toggle theme
            isCyberpunkThemeEnabled = !isCyberpunkThemeEnabled;
            //--- Render
            renderVisualization();
            //--- Redraw chart
            ChartRedraw();
           }
         //--- Check drag start
         else if (enableDragging && isHoveringHeader && !isHoveringResizeZone)
           {
            //--- Set dragging
            isDraggingCanvas = true;
            //--- Set start X
            dragStartX = mouseX;
            //--- Set start Y
            dragStartY = mouseY;
            //--- Set canvas X
            canvasStartX = currentPositionX;
            //--- Set canvas Y
            canvasStartY = currentPositionY;
            //--- Disable scroll
            ChartSetInteger(0, CHART_MOUSE_SCROLL, false);
            //--- Set redraw
            needRedraw = true;
           }
         //--- Check resize start
         else if (isHoveringResizeZone)
           {
            //--- Set resizing
            isResizingCanvas = true;
            //--- Set active mode
            activeResizeMode = hoverResizeMode;
            //--- Set start X
            resizeStartX = mouseX;
            //--- Set start Y
            resizeStartY = mouseY;
            //--- Set initial width
            resizeInitialWidth = currentWidthPixels;
            //--- Set initial height
            resizeInitialHeight = currentHeightPixels;
            //--- Disable scroll
            ChartSetInteger(0, CHART_MOUSE_SCROLL, false);
            //--- Set redraw
            needRedraw = true;
           }
        } 
      //--- Check drag
      else if (mouseState == 1 && previousMouseButtonState == 1)
        {
         //--- Handle drag
         if (isDraggingCanvas)
           {
            //--- Handle drag
            handleCanvasDrag(mouseX, mouseY);
           }
         //--- Handle resize
         else if (isResizingCanvas)
           {
            //--- Handle resize
            handleCanvasResize(mouseX, mouseY);
           }
        } 
      //--- Check button release
      else if (mouseState == 0 && previousMouseButtonState == 1)
        {
         //--- Check active
         if (isDraggingCanvas || isResizingCanvas)
           {
            //--- Reset dragging
            isDraggingCanvas = false;
            //--- Reset resizing
            isResizingCanvas = false;
            //--- Reset mode
            activeResizeMode = NO_RESIZE;
            //--- Enable scroll
            ChartSetInteger(0, CHART_MOUSE_SCROLL, true);
            //--- Set redraw
            needRedraw = true;
           }
        }

      //--- Check redraw
      if (needRedraw)
        {
         //--- Render
         renderVisualization();
         //--- Redraw chart
         ChartRedraw();
        }

      //--- Update last X
      lastMouseX = mouseX;
      //--- Update last Y
      lastMouseY = mouseY;
      //--- Update prev state
      previousMouseButtonState = mouseState;
     }
  }
//+------------------------------------------------------------------+