﻿//+------------------------------------------------------------------+
//|                           Rounded Rectangle & Triangle PART2.mq5 |
//|                           Copyright 2026, Allan Munene Mutiiria. |
//|                                   https://t.me/Forex_Algo_Trader |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, Allan Munene Mutiiria."
#property link      "https://t.me/Forex_Algo_Trader"
#property version   "1.00"
#property strict

#include <Canvas\Canvas.mqh>

//+------------------------------------------------------------------+
//| Enumerations                                                     |
//+------------------------------------------------------------------+
enum BUBBLE_ORIENTATION {
   ORIENT_UP    = 0, // Pointer faces up
   ORIENT_DOWN  = 1, // Pointer faces down
   ORIENT_LEFT  = 2, // Pointer faces left
   ORIENT_RIGHT = 3  // Pointer faces right
};

//+------------------------------------------------------------------+
//| Inputs                                                           |
//+------------------------------------------------------------------+
input group "Position & Layout"
input int    shapesPositionX                  = 20;   // All shapes X position
input int    shapesPositionY                  = 20;   // All shapes Y position
input int    shapesGapPixels                  = 15;   // Gap between shapes (pixels)

input group "Rectangle"
input int    rectangleWidthPixels             = 250;  // Rectangle width
input int    rectangleHeightPixels            = 100;  // Rectangle height
input int    rectangleCornerRadiusPixels      = 5;    // Rectangle corner radius
input bool   rectangleShowBorder              = true; // Show rectangle border
input int    rectangleBorderThicknessPixels   = 1;    // Rectangle border thickness
input color  rectangleBorderColor             = clrBlue; // Rectangle border color
input int    rectangleBorderOpacityPercent    = 80;     // Rectangle border opacity (0-100%)
input color  rectangleBackgroundColor         = clrBlue; // Rectangle background color
input int    rectangleBackgroundOpacityPercent= 30;     // Rectangle background opacity (0-100%)

input group "Triangle"
input int    triangleBaseWidthPixels          = 250;    // Triangle base width (pixels)
input double triangleHeightAsPercentOfWidth   = 86.6;   // Height as % of width (86.6=equilateral, <86.6=flat, >86.6=tall)
input int    triangleCornerRadiusPixels       = 12;     // Triangle corner radius
input bool   triangleShowBorder               = true;  // Show triangle border
input int    triangleBorderThicknessPixels    = 1;     // Triangle border thickness
input color  triangleBorderColor              = clrRed; // Triangle border color
input int    triangleBorderOpacityPercent     = 80;     // Triangle border opacity (0-100%)
input color  triangleBackgroundColor          = clrRed; // Triangle background color
input int    triangleBackgroundOpacityPercent = 30;     // Triangle background opacity (0-100%)

input group "Bubble Shape"
input int              bubbleBodyWidthPixels            = 250;          // Bubble body width
input int              bubbleBodyHeightPixels           = 100;          // Bubble body height
input int              bubbleBodyCornerRadiusPixels     = 5;            // Bubble body corner radius
input int              bubblePointerBaseWidthPixels     = 60;           // Bubble pointer base width
input int              bubblePointerHeightPixels        = 40;           // Bubble pointer height
input int              bubblePointerApexRadiusPixels    = 12;           // Bubble pointer apex radius
input int              bubblePointerBaseOffsetPixels    = 0;            // Bubble pointer offset from center (0=centered, +/-=shift)
input BUBBLE_ORIENTATION bubblePointerOrientation       = ORIENT_UP;    // Bubble pointer orientation
input bool             bubbleShowBorder                 = true;        // Show bubble border
input int              bubbleBorderThicknessPixels      = 2;            // Bubble border thickness
input color            bubbleBorderColor                = clrGreen;     // Bubble border color
input int              bubbleBorderOpacityPercent       = 80;           // Bubble border opacity (0-100%)
input color            bubbleBackgroundColor            = clrGreen;     // Bubble background color
input int              bubbleBackgroundOpacityPercent   = 30;           // Bubble background opacity (0-100%)

input group "General"
input int    supersamplingLevel               = 4;     // Supersampling level (1=off, 2=2x, 4=4x)
input double borderExtensionMultiplier        = 0.23;    // Border extension multiplier (fraction of thickness)

//+------------------------------------------------------------------+
//| Global Variables                                                 |
//+------------------------------------------------------------------+
CCanvas rectangleCanvas, rectangleHighResCanvas;                //--- Declare rectangle canvas objects
CCanvas triangleCanvas,  triangleHighResCanvas;                 //--- Declare triangle canvas objects
CCanvas bubbleCanvas,    bubbleHighResCanvas;                   //--- Declare bubble canvas objects

string  rectangleCanvasName = "RoundedRectCanvas";              //--- Set rectangle canvas name
string  triangleCanvasName  = "RoundedTriCanvas";               //--- Set triangle canvas name
string  bubbleCanvasName    = "BubbleCanvas";                   //--- Set bubble canvas name

int     supersamplingFactor;                                    //--- Store supersampling factor
int     computedTriangleHeightPixels;                           //--- Store computed triangle height

double triangleSharpVerticesX[3], triangleSharpVerticesY[3];    //--- Store triangle sharp vertices X and Y
double triangleArcCentersX[3], triangleArcCentersY[3];          //--- Store triangle arc centers X and Y
double triangleTangentPointsX[3][2], triangleTangentPointsY[3][2]; //--- Store triangle tangent points X and Y
double triangleArcStartAngles[3], triangleArcEndAngles[3];      //--- Store triangle arc start and end angles
int    triangleHighResWidth, triangleHighResHeight;             //--- Store triangle high-res width and height

double bubbleBodyLeft, bubbleBodyTop, bubbleBodyRight, bubbleBodyBottom; //--- Store bubble body coordinates
double bubblePointerVerticesX[3], bubblePointerVerticesY[3];    //--- Store bubble pointer vertices X and Y
double bubblePointerArcCentersX[3], bubblePointerArcCentersY[3]; //--- Store bubble pointer arc centers X and Y
double bubblePointerTangentPointsX[3][2], bubblePointerTangentPointsY[3][2]; //--- Store bubble pointer tangent points X and Y
double bubblePointerArcStartAngles[3], bubblePointerArcEndAngles[3]; //--- Store bubble pointer arc start and end angles
int    bubblePointerApexIndex;                                  //--- Store bubble pointer apex index
double bubblePointerBaseStart, bubblePointerBaseEnd;            //--- Store bubble pointer base start and end
bool   bubbleIsHorizontalOrientation;                           //--- Store if bubble orientation is horizontal

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit() {
   supersamplingFactor = supersamplingLevel;                    //--- Assign supersampling factor from input
   if(supersamplingFactor < 1) {                                //--- Check if supersampling factor is less than 1
      Print("Warning: supersamplingLevel must be at least 1. Setting to 1."); //--- Print warning message
      supersamplingFactor = 1;                                  //--- Set supersampling factor to minimum value
   }

   computedTriangleHeightPixels = (int)MathRound((double)triangleBaseWidthPixels * triangleHeightAsPercentOfWidth / 100.0); //--- Calculate triangle height based on width and percentage
   if(computedTriangleHeightPixels < 10) {                      //--- Check if computed height is too small
      Print("Warning: Computed triangle height too small (" + string(computedTriangleHeightPixels) + "px). Minimum set to 10."); //--- Print warning message
      computedTriangleHeightPixels = 10;                        //--- Set minimum height value
   }

   int rectangleCanvasWidth = rectangleWidthPixels  + 40;       //--- Compute rectangle canvas width with padding
   int rectangleCanvasHeight = rectangleHeightPixels + 40;      //--- Compute rectangle canvas height with padding
   int rectanglePositionY = shapesPositionY;                    //--- Set rectangle Y position

   if(!rectangleCanvas.CreateBitmapLabel(0, 0, rectangleCanvasName, shapesPositionX, rectanglePositionY,
                                    rectangleCanvasWidth, rectangleCanvasHeight, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create rectangle canvas bitmap label
      Print("Error creating rectangle canvas: ", GetLastError()); //--- Print error message if creation fails
      return(INIT_FAILED);                                      //--- Return initialization failure
   }

   if(!rectangleHighResCanvas.Create(rectangleCanvasName + "_hires",
                        rectangleCanvasWidth * supersamplingFactor,
                        rectangleCanvasHeight * supersamplingFactor, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create high-res rectangle canvas
      Print("Error creating rectangle hi-res canvas: ", GetLastError()); //--- Print error message if creation fails
      return(INIT_FAILED);                                      //--- Return initialization failure
   }

   int triangleCanvasWidth  = triangleBaseWidthPixels   + 40;   //--- Compute triangle canvas width with padding
   int triangleCanvasHeight  = computedTriangleHeightPixels + 40; //--- Compute triangle canvas height with padding
   int trianglePositionY = rectanglePositionY + rectangleCanvasHeight + shapesGapPixels; //--- Set triangle Y position below rectangle

   triangleHighResWidth = triangleCanvasWidth  * supersamplingFactor; //--- Compute high-res triangle width
   triangleHighResHeight  = triangleCanvasHeight  * supersamplingFactor; //--- Compute high-res triangle height

   if(!triangleCanvas.CreateBitmapLabel(0, 0, triangleCanvasName, shapesPositionX, trianglePositionY,
                                   triangleCanvasWidth, triangleCanvasHeight, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create triangle canvas bitmap label
      Print("Error creating triangle canvas: ", GetLastError()); //--- Print error message if creation fails
      return(INIT_FAILED);                                      //--- Return initialization failure
   }

   if(!triangleHighResCanvas.Create(triangleCanvasName + "_hires",
                       triangleHighResWidth, triangleHighResHeight, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create high-res triangle canvas
      Print("Error creating triangle hi-res canvas: ", GetLastError()); //--- Print error message if creation fails
      return(INIT_FAILED);                                      //--- Return initialization failure
   }

   bubbleIsHorizontalOrientation = (bubblePointerOrientation == ORIENT_LEFT || bubblePointerOrientation == ORIENT_RIGHT); //--- Determine if orientation is horizontal

   int bubbleCanvasWidth, bubbleCanvasHeight;                   //--- Declare bubble canvas dimensions
   if(bubbleIsHorizontalOrientation) {                          //--- Check for horizontal orientation
      bubbleCanvasWidth = bubbleBodyWidthPixels + bubblePointerHeightPixels + 40; //--- Compute width for horizontal
      bubbleCanvasHeight = bubbleBodyHeightPixels + 40;         //--- Compute height for horizontal
   } else {                                                     //--- Handle vertical orientation
      bubbleCanvasWidth = bubbleBodyWidthPixels + 40;           //--- Compute width for vertical
      bubbleCanvasHeight = bubbleBodyHeightPixels + bubblePointerHeightPixels + 40; //--- Compute height for vertical
   }

   int bubblePositionY = trianglePositionY + triangleCanvasHeight + shapesGapPixels; //--- Set bubble Y position below triangle

   if(!bubbleCanvas.CreateBitmapLabel(0, 0, bubbleCanvasName, shapesPositionX, bubblePositionY,
                                    bubbleCanvasWidth, bubbleCanvasHeight, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create bubble canvas bitmap label
      Print("Error creating bubble canvas: ", GetLastError());  //--- Print error message if creation fails
      return(INIT_FAILED);                                      //--- Return initialization failure
   }

   if(!bubbleHighResCanvas.Create(bubbleCanvasName + "_hires",
                        bubbleCanvasWidth * supersamplingFactor,
                        bubbleCanvasHeight * supersamplingFactor, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create high-res bubble canvas
      Print("Error creating bubble hi-res canvas: ", GetLastError()); //--- Print error message if creation fails
      return(INIT_FAILED);                                      //--- Return initialization failure
   }

   rectangleCanvas.Erase(ColorToARGB(clrNONE, 0));              //--- Clear rectangle canvas
   rectangleHighResCanvas.Erase(ColorToARGB(clrNONE, 0));       //--- Clear high-res rectangle canvas
   triangleCanvas.Erase(ColorToARGB(clrNONE, 0));               //--- Clear triangle canvas
   triangleHighResCanvas.Erase(ColorToARGB(clrNONE, 0));        //--- Clear high-res triangle canvas
   bubbleCanvas.Erase(ColorToARGB(clrNONE, 0));                 //--- Clear bubble canvas
   bubbleHighResCanvas.Erase(ColorToARGB(clrNONE, 0));          //--- Clear high-res bubble canvas

   PrecomputeTriangleGeometry();                                //--- Precompute triangle geometry
   PrecomputeBubbleGeometry();                                  //--- Precompute bubble geometry
   
   DrawRoundedRectangle();                                      //--- Draw rounded rectangle
   DrawRoundedTriangle();                                       //--- Draw rounded triangle
   DrawBubble();                                                //--- Draw bubble

   rectangleCanvas.Update();                                    //--- Update rectangle canvas display
   triangleCanvas.Update();                                     //--- Update triangle canvas display
   bubbleCanvas.Update();                                       //--- Update bubble canvas display

   return(INIT_SUCCEEDED);                                      //--- Return initialization success
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason) {
   rectangleHighResCanvas.Destroy();                            //--- Destroy high-res rectangle canvas
   rectangleCanvas.Destroy();                                   //--- Destroy rectangle canvas
   ObjectDelete(0, rectangleCanvasName);                        //--- Delete rectangle canvas object

   triangleHighResCanvas.Destroy();                             //--- Destroy high-res triangle canvas
   triangleCanvas.Destroy();                                    //--- Destroy triangle canvas
   ObjectDelete(0, triangleCanvasName);                         //--- Delete triangle canvas object

   bubbleHighResCanvas.Destroy();                               //--- Destroy high-res bubble canvas
   bubbleCanvas.Destroy();                                      //--- Destroy bubble canvas
   ObjectDelete(0, bubbleCanvasName);                           //--- Delete bubble canvas object

   ChartRedraw();                                               //--- Redraw chart
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick() {
}

//+------------------------------------------------------------------+
//| Chart event function                                             |
//+------------------------------------------------------------------+
void OnChartEvent(const int id,
                  const long &lparam,
                  const double &dparam,
                  const string &sparam) {
   if(id == CHARTEVENT_CHART_CHANGE) {                          //--- Check for chart change event
      rectangleCanvas.Erase(ColorToARGB(clrNONE, 0));           //--- Clear rectangle canvas
      rectangleHighResCanvas.Erase(ColorToARGB(clrNONE, 0));    //--- Clear high-res rectangle canvas
      DrawRoundedRectangle();                                   //--- Redraw rounded rectangle
      rectangleCanvas.Update();                                 //--- Update rectangle canvas display

      triangleCanvas.Erase(ColorToARGB(clrNONE, 0));            //--- Clear triangle canvas
      triangleHighResCanvas.Erase(ColorToARGB(clrNONE, 0));     //--- Clear high-res triangle canvas
      DrawRoundedTriangle();                                    //--- Redraw rounded triangle
      triangleCanvas.Update();                                  //--- Update triangle canvas display

      bubbleCanvas.Erase(ColorToARGB(clrNONE, 0));              //--- Clear bubble canvas
      bubbleHighResCanvas.Erase(ColorToARGB(clrNONE, 0));       //--- Clear high-res bubble canvas
      DrawBubble();                                             //--- Redraw bubble
      bubbleCanvas.Update();                                    //--- Update bubble canvas display
   }
}

//+------------------------------------------------------------------+
//| Shared Utilities                                                 |
//+------------------------------------------------------------------+
uint ColorToARGBWithOpacity(color clr, int opacityPercent) {
   uchar redComponent = (uchar)((clr >> 0)  & 0xFF);            //--- Extract red component
   uchar greenComponent = (uchar)((clr >> 8)  & 0xFF);          //--- Extract green component
   uchar blueComponent = (uchar)((clr >> 16) & 0xFF);           //--- Extract blue component
   uchar alphaComponent = (uchar)((opacityPercent * 255) / 100); //--- Calculate alpha from opacity
   return ((uint)alphaComponent << 24) | ((uint)redComponent << 16) | ((uint)greenComponent << 8) | (uint)blueComponent; //--- Combine into ARGB value
}

void BicubicDownsample(CCanvas &targetCanvas, CCanvas &highResCanvas) {
   int targetWidth  = targetCanvas.Width();                     //--- Get target canvas width
   int targetHeight = targetCanvas.Height();                    //--- Get target canvas height

   for(int pixelY = 0; pixelY < targetHeight; pixelY++) {       //--- Loop over target height pixels
      for(int pixelX = 0; pixelX < targetWidth; pixelX++) {     //--- Loop over target width pixels
         double sourceX = pixelX * supersamplingFactor;          //--- Calculate source X position
         double sourceY = pixelY * supersamplingFactor;          //--- Calculate source Y position

         double sumAlpha = 0, sumRed = 0, sumGreen = 0, sumBlue = 0; //--- Initialize sum variables
         double weightSum = 0;                                   //--- Initialize weight sum

         for(int deltaY = 0; deltaY < supersamplingFactor; deltaY++) { //--- Loop over delta Y for supersampling
            for(int deltaX = 0; deltaX < supersamplingFactor; deltaX++) { //--- Loop over delta X for supersampling
               int sourcePixelX = (int)(sourceX + deltaX);       //--- Compute source pixel X
               int sourcePixelY = (int)(sourceY + deltaY);       //--- Compute source pixel Y

               if(sourcePixelX >= 0 && sourcePixelX < highResCanvas.Width() && sourcePixelY >= 0 && sourcePixelY < highResCanvas.Height()) { //--- Check if within high-res bounds
                  uint pixelValue = highResCanvas.PixelGet(sourcePixelX, sourcePixelY); //--- Get pixel value from high-res

                  uchar alpha = (uchar)((pixelValue >> 24) & 0xFF); //--- Extract alpha component
                  uchar red = (uchar)((pixelValue >> 16) & 0xFF); //--- Extract red component
                  uchar green = (uchar)((pixelValue >> 8)  & 0xFF); //--- Extract green component
                  uchar blue = (uchar)(pixelValue         & 0xFF); //--- Extract blue component

                  double weight = 1.0;                           //--- Set weight to 1.0
                  sumAlpha += alpha * weight;                    //--- Accumulate weighted alpha
                  sumRed += red * weight;                        //--- Accumulate weighted red
                  sumGreen += green * weight;                    //--- Accumulate weighted green
                  sumBlue += blue * weight;                      //--- Accumulate weighted blue
                  weightSum += weight;                           //--- Accumulate total weight
               }
            }
         }

         if(weightSum > 0) {                                       //--- Check if weight sum is positive
            uchar finalAlpha = (uchar)(sumAlpha / weightSum);      //--- Compute final alpha
            uchar finalRed = (uchar)(sumRed / weightSum);          //--- Compute final red
            uchar finalGreen = (uchar)(sumGreen / weightSum);      //--- Compute final green
            uchar finalBlue = (uchar)(sumBlue / weightSum);        //--- Compute final blue

            uint finalColor = ((uint)finalAlpha << 24) | ((uint)finalRed << 16) |
                              ((uint)finalGreen << 8)  | (uint)finalBlue; //--- Combine into final color
            targetCanvas.PixelSet(pixelX, pixelY, finalColor);     //--- Set pixel on target canvas
         }
      }
   }
}

double NormalizeAngle(double angle) {
   double twoPi = 2.0 * M_PI;                                   //--- Define two pi constant
   angle = MathMod(angle, twoPi);                               //--- Modulo angle by two pi
   if(angle < 0) angle += twoPi;                                //--- Adjust if angle is negative
   return angle;                                                //--- Return normalized angle
}

bool IsAngleBetween(double angle, double startAngle, double endAngle) {
   angle = NormalizeAngle(angle);                               //--- Normalize angle
   startAngle = NormalizeAngle(startAngle);                     //--- Normalize start angle
   endAngle = NormalizeAngle(endAngle);                         //--- Normalize end angle
   
   double span = NormalizeAngle(endAngle - startAngle);         //--- Compute span
   double relativeAngle = NormalizeAngle(angle - startAngle);   //--- Compute relative angle
   
   return relativeAngle <= span;                                //--- Return if within span
}

void FillQuadrilateral(CCanvas &canvas, double &verticesX[], double &verticesY[], uint fillColor) {
   double minY = verticesY[0], maxY = verticesY[0];             //--- Initialize min and max Y
   for(int i = 1; i < 4; i++) {                                 //--- Loop over vertices
      if(verticesY[i] < minY) minY = verticesY[i];              //--- Update min Y
      if(verticesY[i] > maxY) maxY = verticesY[i];              //--- Update max Y
   }

   int yStart = (int)MathCeil(minY);                            //--- Compute start Y
   int yEnd   = (int)MathFloor(maxY);                           //--- Compute end Y

   for(int y = yStart; y <= yEnd; y++) {                        //--- Loop over scanlines
      double scanlineY = (double)y + 0.5;                       //--- Set scanline Y position
      double xIntersections[8];                                 //--- Declare intersections array
      int intersectionCount = 0;                                //--- Initialize intersection count

      for(int i = 0; i < 4; i++) {                              //--- Loop over edges
         int nextIndex = (i + 1) % 4;                           //--- Get next index
         double x0 = verticesX[i],    y0 = verticesY[i];        //--- Get start coordinates
         double x1 = verticesX[nextIndex], y1 = verticesY[nextIndex]; //--- Get end coordinates

         double edgeMinY = (y0 < y1) ? y0 : y1;                 //--- Compute edge min Y
         double edgeMaxY = (y0 > y1) ? y0 : y1;                 //--- Compute edge max Y

         if(scanlineY < edgeMinY || scanlineY > edgeMaxY) continue; //--- Skip if outside edge Y range
         if(MathAbs(y1 - y0) < 1e-12) continue;                 //--- Skip if horizontal edge

         double interpolationFactor = (scanlineY - y0) / (y1 - y0); //--- Compute interpolation factor
         if(interpolationFactor < 0.0 || interpolationFactor > 1.0) continue; //--- Skip if outside segment

         xIntersections[intersectionCount++] = x0 + interpolationFactor * (x1 - x0); //--- Add intersection X
      }

      for(int a = 0; a < intersectionCount - 1; a++)            //--- Sort intersections (bubble sort)
         for(int b = a + 1; b < intersectionCount; b++)          //--- Inner loop for sorting
            if(xIntersections[a] > xIntersections[b]) {          //--- Check if swap needed
               double temp = xIntersections[a];                  //--- Temporary store
               xIntersections[a] = xIntersections[b];            //--- Swap values
               xIntersections[b] = temp;                         //--- Complete swap
            }

      for(int pairIndex = 0; pairIndex + 1 < intersectionCount; pairIndex += 2) { //--- Loop over pairs
         int xLeft  = (int)MathCeil(xIntersections[pairIndex]);  //--- Compute left X
         int xRight = (int)MathFloor(xIntersections[pairIndex + 1]); //--- Compute right X
         for(int x = xLeft; x <= xRight; x++)                   //--- Loop over horizontal span
            canvas.PixelSet(x, y, fillColor);                   //--- Set pixel with fill color
      }
   }
}

//+------------------------------------------------------------------+
//| Rounded Rectangle                                                |
//+------------------------------------------------------------------+
void DrawRoundedRectangle() {
   int positionX      = 10 * supersamplingFactor;               //--- Set X position scaled
   int positionY      = 10 * supersamplingFactor;               //--- Set Y position scaled
   int scaledWidth  = rectangleWidthPixels  * supersamplingFactor; //--- Scale width
   int scaledHeight = rectangleHeightPixels * supersamplingFactor; //--- Scale height
   int scaledRadius = rectangleCornerRadiusPixels * supersamplingFactor; //--- Scale radius

   uint backgroundColorARGB     = ColorToARGBWithOpacity(rectangleBackgroundColor,     rectangleBackgroundOpacityPercent); //--- Get background ARGB
   uint borderColorARGB = ColorToARGBWithOpacity(rectangleBorderColor, rectangleBorderOpacityPercent); //--- Get border ARGB

   FillRoundedRectangleHiRes(positionX, positionY, scaledWidth, scaledHeight, scaledRadius, backgroundColorARGB); //--- Fill high-res rectangle

   if(rectangleShowBorder && rectangleBorderThicknessPixels > 0) //--- Check if border should be shown
      DrawRoundedRectangleBorderHiRes(positionX, positionY, scaledWidth, scaledHeight, scaledRadius, borderColorARGB); //--- Draw border on high-res

   BicubicDownsample(rectangleCanvas, rectangleHighResCanvas);  //--- Downsample to display canvas

   rectangleCanvas.FontSet("Arial", 13, FW_NORMAL);             //--- Set font for text
   string displayText = "Rounded Rectangle";                    //--- Set display text
   int textWidth, textHeight;                                   //--- Declare text dimensions
   rectangleCanvas.TextSize(displayText, textWidth, textHeight); //--- Get text size
   int textPositionX = 10 + (rectangleWidthPixels  - textWidth)  / 2; //--- Compute text X position
   int textPositionY = 10 + (rectangleHeightPixels - textHeight) / 2; //--- Compute text Y position
   rectangleCanvas.TextOut(textPositionX, textPositionY, displayText, (uint)0xFF000000, TA_LEFT); //--- Draw text on canvas
}

void FillRoundedRectangleHiRes(int positionX, int positionY, int width, int height, int radius, uint fillColor) {
   rectangleHighResCanvas.FillRectangle(positionX + radius, positionY,          positionX + width - radius, positionY + height,          fillColor); //--- Fill central rectangle
   rectangleHighResCanvas.FillRectangle(positionX,          positionY + radius, positionX + radius,         positionY + height - radius, fillColor); //--- Fill left strip
   rectangleHighResCanvas.FillRectangle(positionX + width - radius, positionY + radius, positionX + width, positionY + height - radius, fillColor); //--- Fill right strip

   FillCircleQuadrant(positionX + radius,         positionY + radius,          radius, fillColor, 2); //--- Fill top-left quadrant
   FillCircleQuadrant(positionX + width - radius, positionY + radius,          radius, fillColor, 1); //--- Fill top-right quadrant
   FillCircleQuadrant(positionX + radius,         positionY + height - radius, radius, fillColor, 3); //--- Fill bottom-left quadrant
   FillCircleQuadrant(positionX + width - radius, positionY + height - radius, radius, fillColor, 4); //--- Fill bottom-right quadrant
}

void FillCircleQuadrant(int centerX, int centerY, int radius, uint fillColor, int quadrant) {
   double radiusDouble = (double)radius;                        //--- Convert radius to double

   for(int deltaY = -radius - 1; deltaY <= radius + 1; deltaY++) { //--- Loop over delta Y
      for(int deltaX = -radius - 1; deltaX <= radius + 1; deltaX++) { //--- Loop over delta X
         bool inQuadrant = false;                               //--- Initialize quadrant flag
         if(quadrant == 1 && deltaX >= 0 && deltaY <= 0) inQuadrant = true; //--- Check top-right
         else if(quadrant == 2 && deltaX <= 0 && deltaY <= 0) inQuadrant = true; //--- Check top-left
         else if(quadrant == 3 && deltaX <= 0 && deltaY >= 0) inQuadrant = true; //--- Check bottom-left
         else if(quadrant == 4 && deltaX >= 0 && deltaY >= 0) inQuadrant = true; //--- Check bottom-right

         if(!inQuadrant) continue;                              //--- Skip if not in quadrant

         double distance = MathSqrt(deltaX * deltaX + deltaY * deltaY); //--- Compute distance
         if(distance <= radiusDouble)                           //--- Check if within radius
            rectangleHighResCanvas.PixelSet(centerX + deltaX, centerY + deltaY, fillColor); //--- Set pixel
      }
   }
}

void DrawRoundedRectangleBorderHiRes(int positionX, int positionY, int width, int height, int radius, uint borderColorARGB) {
   int scaledThickness = rectangleBorderThicknessPixels * supersamplingFactor; //--- Scale border thickness

   DrawRectStraightEdge(positionX + radius, positionY, positionX + width - radius, positionY, scaledThickness, borderColorARGB); //--- Draw top edge
   DrawRectStraightEdge(positionX + width - radius, positionY + height - 1, positionX + radius, positionY + height - 1, scaledThickness, borderColorARGB); //--- Draw bottom edge
   DrawRectStraightEdge(positionX, positionY + height - radius, positionX, positionY + radius, scaledThickness, borderColorARGB); //--- Draw left edge
   DrawRectStraightEdge(positionX + width - 1, positionY + radius, positionX + width - 1, positionY + height - radius, scaledThickness, borderColorARGB); //--- Draw right edge

   DrawRectCornerArcPrecise(positionX + radius, positionY + radius, radius, scaledThickness, borderColorARGB, 
                            M_PI, M_PI * 1.5);                  //--- Draw top-left arc
   DrawRectCornerArcPrecise(positionX + width - radius, positionY + radius, radius, scaledThickness, borderColorARGB,
                            M_PI * 1.5, M_PI * 2.0);            //--- Draw top-right arc
   DrawRectCornerArcPrecise(positionX + radius, positionY + height - radius, radius, scaledThickness, borderColorARGB,
                            M_PI * 0.5, M_PI);                  //--- Draw bottom-left arc
   DrawRectCornerArcPrecise(positionX + width - radius, positionY + height - radius, radius, scaledThickness, borderColorARGB,
                            0.0, M_PI * 0.5);                   //--- Draw bottom-right arc
}

void DrawRectStraightEdge(double startX, double startY, double endX, double endY, int thickness, uint borderColor) {
   double deltaX = endX - startX;                               //--- Compute delta X
   double deltaY = endY - startY;                               //--- Compute delta Y
   double edgeLength = MathSqrt(deltaX*deltaX + deltaY*deltaY); //--- Compute edge length
   if(edgeLength < 1e-6) return;                                //--- Return if length too small

   double perpendicularX = -deltaY / edgeLength;                //--- Compute perpendicular X
   double perpendicularY = deltaX / edgeLength;                 //--- Compute perpendicular Y

   double edgeDirectionX = deltaX / edgeLength;                 //--- Compute edge direction X
   double edgeDirectionY = deltaY / edgeLength;                 //--- Compute edge direction Y

   double halfThickness = (double)thickness / 2.0;              //--- Compute half thickness
   
   double extensionLength = borderExtensionMultiplier * (double)thickness; //--- Set extension length

   double extendedStartX = startX - edgeDirectionX * extensionLength; //--- Extend start X
   double extendedStartY = startY - edgeDirectionY * extensionLength; //--- Extend start Y
   double extendedEndX = endX + edgeDirectionX * extensionLength; //--- Extend end X
   double extendedEndY = endY + edgeDirectionY * extensionLength; //--- Extend end Y

   double verticesX[4], verticesY[4];                           //--- Declare vertices arrays
   verticesX[0] = extendedStartX - perpendicularX * halfThickness;  verticesY[0] = extendedStartY - perpendicularY * halfThickness; //--- Set vertex 0
   verticesX[1] = extendedStartX + perpendicularX * halfThickness;  verticesY[1] = extendedStartY + perpendicularY * halfThickness; //--- Set vertex 1
   verticesX[2] = extendedEndX + perpendicularX * halfThickness;  verticesY[2] = extendedEndY + perpendicularY * halfThickness; //--- Set vertex 2
   verticesX[3] = extendedEndX - perpendicularX * halfThickness;  verticesY[3] = extendedEndY - perpendicularY * halfThickness; //--- Set vertex 3

   FillQuadrilateral(rectangleHighResCanvas, verticesX, verticesY, borderColor); //--- Fill quadrilateral for edge
}

void DrawRectCornerArcPrecise(int centerX, int centerY, int radius, int thickness, uint borderColor,
                              double startAngle, double endAngle) {
   int halfThickness = thickness / 2;                           //--- Compute half thickness
   double outerRadius = (double)radius + halfThickness;         //--- Compute outer radius
   double innerRadius = (double)radius - halfThickness;         //--- Compute inner radius
   if(innerRadius < 0) innerRadius = 0;                         //--- Set inner radius to zero if negative

   int pixelRange = (int)(outerRadius + 2);                     //--- Compute pixel range

   for(int deltaY = -pixelRange; deltaY <= pixelRange; deltaY++) { //--- Loop over delta Y
      for(int deltaX = -pixelRange; deltaX <= pixelRange; deltaX++) { //--- Loop over delta X
         double distance = MathSqrt(deltaX * deltaX + deltaY * deltaY); //--- Compute distance
         if(distance < innerRadius || distance > outerRadius) continue; //--- Skip if outside radii

         double angle = MathArctan2((double)deltaY, (double)deltaX); //--- Compute angle
         
         if(IsAngleBetween(angle, startAngle, endAngle))        //--- Check if angle within range
            rectangleHighResCanvas.PixelSet(centerX + deltaX, centerY + deltaY, borderColor); //--- Set pixel
      }
   }
}

//+------------------------------------------------------------------+
//| Rounded Triangle                                                 |
//+------------------------------------------------------------------+
void PrecomputeTriangleGeometry() {
   int scalingFactor = supersamplingFactor;                     //--- Set scaling factor

   double basePositionX = 10.0 * scalingFactor;                 //--- Set base X position scaled
   double basePositionY = 10.0 * scalingFactor;                 //--- Set base Y position scaled
   double baseWidth = (double)triangleBaseWidthPixels   * scalingFactor; //--- Scale base width
   double baseHeight = (double)computedTriangleHeightPixels * scalingFactor; //--- Scale base height

   triangleSharpVerticesX[0] = basePositionX + baseWidth / 2.0; triangleSharpVerticesY[0] = basePositionY; //--- Set top vertex
   triangleSharpVerticesX[1] = basePositionX;               triangleSharpVerticesY[1] = basePositionY + baseHeight; //--- Set bottom-left vertex
   triangleSharpVerticesX[2] = basePositionX + baseWidth;          triangleSharpVerticesY[2] = basePositionY + baseHeight; //--- Set bottom-right vertex

   double scaledRadius = (double)triangleCornerRadiusPixels * scalingFactor; //--- Scale radius

   for(int cornerIndex = 0; cornerIndex < 3; cornerIndex++) {   //--- Loop over corners
      int previousIndex = (cornerIndex + 2) % 3;                //--- Get previous index
      int nextIndex = (cornerIndex + 1) % 3;                    //--- Get next index

      double edgeA_X = triangleSharpVerticesX[cornerIndex] - triangleSharpVerticesX[previousIndex],  edgeA_Y = triangleSharpVerticesY[cornerIndex] - triangleSharpVerticesY[previousIndex]; //--- Compute edge A vector
      double edgeA_Length = MathSqrt(edgeA_X*edgeA_X + edgeA_Y*edgeA_Y); //--- Compute edge A length
      edgeA_X /= edgeA_Length;  edgeA_Y /= edgeA_Length;          //--- Normalize edge A

      double edgeB_X = triangleSharpVerticesX[nextIndex] - triangleSharpVerticesX[cornerIndex],  edgeB_Y = triangleSharpVerticesY[nextIndex] - triangleSharpVerticesY[cornerIndex]; //--- Compute edge B vector
      double edgeB_Length = MathSqrt(edgeB_X*edgeB_X + edgeB_Y*edgeB_Y); //--- Compute edge B length
      edgeB_X /= edgeB_Length;  edgeB_Y /= edgeB_Length;          //--- Normalize edge B

      double normalA_X =  edgeA_Y,  normalA_Y = -edgeA_X;       //--- Compute normal A
      double normalB_X =  edgeB_Y,  normalB_Y = -edgeB_X;       //--- Compute normal B

      double bisectorX = normalA_X + normalB_X,  bisectorY = normalA_Y + normalB_Y; //--- Compute bisector
      double bisectorLength = MathSqrt(bisectorX*bisectorX + bisectorY*bisectorY); //--- Compute bisector length
      if(bisectorLength < 1e-12) { bisectorX = normalA_X; bisectorY = normalA_Y; bisectorLength = MathSqrt(bisectorX*bisectorX + bisectorY*bisectorY); } //--- Handle small bisector
      bisectorX /= bisectorLength;  bisectorY /= bisectorLength; //--- Normalize bisector

      double cosInteriorAngle = (-edgeA_X)*edgeB_X + (-edgeA_Y)*edgeB_Y; //--- Compute cosine of interior angle
      if(cosInteriorAngle >  1.0) cosInteriorAngle =  1.0;      //--- Clamp cosine upper
      if(cosInteriorAngle < -1.0) cosInteriorAngle = -1.0;      //--- Clamp cosine lower
      double halfAngle = MathArccos(cosInteriorAngle) / 2.0;    //--- Compute half angle
      double sinHalfAngle   = MathSin(halfAngle);               //--- Compute sine of half angle
      if(sinHalfAngle < 1e-12) sinHalfAngle = 1e-12;            //--- Set minimum sine value

      double distanceToCenter = scaledRadius / sinHalfAngle;    //--- Compute distance to arc center
      triangleArcCentersX[cornerIndex] = triangleSharpVerticesX[cornerIndex] + bisectorX * distanceToCenter; //--- Set arc center X
      triangleArcCentersY[cornerIndex] = triangleSharpVerticesY[cornerIndex] + bisectorY * distanceToCenter; //--- Set arc center Y

      double deltaX_A = triangleSharpVerticesX[cornerIndex] - triangleSharpVerticesX[previousIndex],  deltaY_A = triangleSharpVerticesY[cornerIndex] - triangleSharpVerticesY[previousIndex]; //--- Compute delta A
      double lengthSquared_A = deltaX_A*deltaX_A + deltaY_A*deltaY_A; //--- Compute length squared A
      double interpolationFactor_A = ((triangleArcCentersX[cornerIndex] - triangleSharpVerticesX[previousIndex])*deltaX_A + (triangleArcCentersY[cornerIndex] - triangleSharpVerticesY[previousIndex])*deltaY_A) / lengthSquared_A; //--- Compute factor A
      triangleTangentPointsX[cornerIndex][1] = triangleSharpVerticesX[previousIndex] + interpolationFactor_A * deltaX_A; //--- Set tangent point X arriving
      triangleTangentPointsY[cornerIndex][1] = triangleSharpVerticesY[previousIndex] + interpolationFactor_A * deltaY_A; //--- Set tangent point Y arriving

      double deltaX_B = triangleSharpVerticesX[nextIndex] - triangleSharpVerticesX[cornerIndex],  deltaY_B = triangleSharpVerticesY[nextIndex] - triangleSharpVerticesY[cornerIndex]; //--- Compute delta B
      double lengthSquared_B = deltaX_B*deltaX_B + deltaY_B*deltaY_B; //--- Compute length squared B
      double interpolationFactor_B = ((triangleArcCentersX[cornerIndex] - triangleSharpVerticesX[cornerIndex])*deltaX_B + (triangleArcCentersY[cornerIndex] - triangleSharpVerticesY[cornerIndex])*deltaY_B) / lengthSquared_B; //--- Compute factor B
      triangleTangentPointsX[cornerIndex][0] = triangleSharpVerticesX[cornerIndex] + interpolationFactor_B * deltaX_B; //--- Set tangent point X leaving
      triangleTangentPointsY[cornerIndex][0] = triangleSharpVerticesY[cornerIndex] + interpolationFactor_B * deltaY_B; //--- Set tangent point Y leaving

      triangleArcStartAngles[cornerIndex] = MathArctan2(triangleTangentPointsY[cornerIndex][1] - triangleArcCentersY[cornerIndex], triangleTangentPointsX[cornerIndex][1] - triangleArcCentersX[cornerIndex]); //--- Set start angle
      triangleArcEndAngles[cornerIndex]   = MathArctan2(triangleTangentPointsY[cornerIndex][0] - triangleArcCentersY[cornerIndex], triangleTangentPointsX[cornerIndex][0] - triangleArcCentersX[cornerIndex]); //--- Set end angle
   }
}

bool TriangleAngleInArcSweep(int cornerIndex, double angle) {
   double twoPi  = 2.0 * M_PI;                                  //--- Define two pi constant
   double startAngleMod  = MathMod(triangleArcStartAngles[cornerIndex] + twoPi, twoPi); //--- Modulo start angle
   double endAngleMod  = MathMod(triangleArcEndAngles[cornerIndex]   + twoPi, twoPi); //--- Modulo end angle
   angle      = MathMod(angle             + twoPi, twoPi);      //--- Modulo angle

   double ccwSpan = MathMod(endAngleMod - startAngleMod + twoPi, twoPi); //--- Compute CCW span

   if(ccwSpan <= M_PI) {                                        //--- Check if short way is CCW
      double relativeAngle    = MathMod(angle - startAngleMod + twoPi, twoPi); //--- Compute relative angle
      return(relativeAngle <= ccwSpan + 1e-6);                  //--- Return if within CCW span
   } else {                                                     //--- Else short way is CW
      double cwSpan = twoPi - ccwSpan;                          //--- Compute CW span
      double relativeAngle    = MathMod(angle - endAngleMod + twoPi, twoPi); //--- Compute relative angle
      return(relativeAngle <= cwSpan + 1e-6);                   //--- Return if within CW span
   }
}

void DrawRoundedTriangle() {
   uint backgroundColorARGB     = ColorToARGBWithOpacity(triangleBackgroundColor,     triangleBackgroundOpacityPercent); //--- Get background ARGB
   uint borderColorARGB = ColorToARGBWithOpacity(triangleBorderColor, triangleBorderOpacityPercent); //--- Get border ARGB

   FillRoundedTriangleHiRes(backgroundColorARGB);               //--- Fill high-res triangle

   if(triangleShowBorder && triangleBorderThicknessPixels > 0)  //--- Check if border should be shown
      DrawRoundedTriangleBorderHiRes(borderColorARGB);          //--- Draw border on high-res

   BicubicDownsample(triangleCanvas, triangleHighResCanvas);    //--- Downsample to display canvas

   triangleCanvas.FontSet("Arial", 13, FW_NORMAL);              //--- Set font for text
   string displayText = "Rounded Triangle";                     //--- Set display text
   int textWidth, textHeight;                                   //--- Declare text dimensions
   triangleCanvas.TextSize(displayText, textWidth, textHeight); //--- Get text size
   int textPositionX = 10 + (triangleBaseWidthPixels   - textWidth)  / 2; //--- Compute text X position
   int textPositionY = 10 + (computedTriangleHeightPixels - textHeight) / 2; //--- Compute text Y position
   triangleCanvas.TextOut(textPositionX, textPositionY, displayText, (uint)0xFF000000, TA_LEFT); //--- Draw text on canvas
}

void FillRoundedTriangleHiRes(uint fillColor) {
   double minY = triangleSharpVerticesY[0], maxY = triangleSharpVerticesY[0]; //--- Initialize min and max Y
   for(int i = 1; i < 3; i++) {                                 //--- Loop over vertices
      if(triangleSharpVerticesY[i] < minY) minY = triangleSharpVerticesY[i]; //--- Update min Y
      if(triangleSharpVerticesY[i] > maxY) maxY = triangleSharpVerticesY[i]; //--- Update max Y
   }

   int yStart = (int)MathCeil(minY);                            //--- Compute start Y
   int yEnd   = (int)MathFloor(maxY);                           //--- Compute end Y

   for(int y = yStart; y <= yEnd; y++) {                        //--- Loop over scanlines
      double scanlineY = (double)y + 0.5;                       //--- Set scanline Y position

      double xIntersections[12];                                //--- Declare intersections array
      int    intersectionCount = 0;                             //--- Initialize intersection count

      for(int edgeIndex = 0; edgeIndex < 3; edgeIndex++) {      //--- Loop over straight edges
         int nextIndex = (edgeIndex + 1) % 3;                   //--- Get next index
         double startX = triangleTangentPointsX[edgeIndex][0],    startY = triangleTangentPointsY[edgeIndex][0]; //--- Get start tangent
         double endX = triangleTangentPointsX[nextIndex][1], endY = triangleTangentPointsY[nextIndex][1]; //--- Get end tangent

         double edgeMinY = (startY < endY) ? startY : endY;     //--- Compute edge min Y
         double edgeMaxY = (startY > endY) ? startY : endY;     //--- Compute edge max Y

         if(scanlineY < edgeMinY || scanlineY > edgeMaxY) continue; //--- Skip if outside edge Y
         if(MathAbs(endY - startY) < 1e-12)      continue;      //--- Skip if horizontal

         double interpolationFactor = (scanlineY - startY) / (endY - startY); //--- Compute factor
         if(interpolationFactor < 0.0 || interpolationFactor > 1.0) continue; //--- Skip if outside segment

         xIntersections[intersectionCount++] = startX + interpolationFactor * (endX - startX); //--- Add intersection X
      }

      for(int cornerIndex = 0; cornerIndex < 3; cornerIndex++) { //--- Loop over corner arcs
         double centerX = triangleArcCentersX[cornerIndex],  centerY = triangleArcCentersY[cornerIndex]; //--- Get arc center
         double radius  = (double)triangleCornerRadiusPixels * supersamplingFactor; //--- Get scaled radius
         double deltaY = scanlineY - centerY;                   //--- Compute delta Y

         if(MathAbs(deltaY) > radius) continue;                 //--- Skip if outside radius

         double deltaX = MathSqrt(radius*radius - deltaY*deltaY); //--- Compute delta X

         double candidates[2];                                  //--- Declare candidates array
         candidates[0] = centerX - deltaX;                      //--- Set left candidate
         candidates[1] = centerX + deltaX;                      //--- Set right candidate

         for(int candidateIndex = 0; candidateIndex < 2; candidateIndex++) { //--- Loop over candidates
            double angle = MathArctan2(scanlineY - centerY, candidates[candidateIndex] - centerX); //--- Compute angle
            if(TriangleAngleInArcSweep(cornerIndex, angle))     //--- Check if in arc sweep
               xIntersections[intersectionCount++] = candidates[candidateIndex]; //--- Add intersection
         }
      }

      for(int a = 0; a < intersectionCount - 1; a++)            //--- Sort intersections (bubble sort)
         for(int b = a + 1; b < intersectionCount; b++)          //--- Inner loop for sorting
            if(xIntersections[a] > xIntersections[b]) {          //--- Check if swap needed
               double temp = xIntersections[a];                  //--- Temporary store
               xIntersections[a]   = xIntersections[b];          //--- Swap values
               xIntersections[b]   = temp;                       //--- Complete swap
            }

      for(int pairIndex = 0; pairIndex + 1 < intersectionCount; pairIndex += 2) { //--- Loop over pairs
         int xLeft  = (int)MathCeil(xIntersections[pairIndex]);  //--- Compute left X
         int xRight = (int)MathFloor(xIntersections[pairIndex + 1]); //--- Compute right X
         for(int x = xLeft; x <= xRight; x++)                   //--- Loop over horizontal span
            triangleHighResCanvas.PixelSet(x, y, fillColor);    //--- Set pixel with fill color
      }
   }
}

void DrawRoundedTriangleBorderHiRes(uint borderColor) {
   int scaledThickness = triangleBorderThicknessPixels * supersamplingFactor; //--- Scale border thickness

   for(int edgeIndex = 0; edgeIndex < 3; edgeIndex++) {         //--- Loop over edges
      int nextIndex = (edgeIndex + 1) % 3;                      //--- Get next index
      double startX = triangleTangentPointsX[edgeIndex][0],    startY = triangleTangentPointsY[edgeIndex][0]; //--- Get start tangent
      double endX = triangleTangentPointsX[nextIndex][1], endY = triangleTangentPointsY[nextIndex][1]; //--- Get end tangent

      DrawTriStraightEdge(startX, startY, endX, endY, scaledThickness, borderColor); //--- Draw straight edge
   }

   for(int cornerIndex = 0; cornerIndex < 3; cornerIndex++)     //--- Loop over corners
      DrawTriCornerArcPrecise(cornerIndex, scaledThickness, borderColor); //--- Draw corner arc
}

void DrawTriStraightEdge(double startX, double startY, double endX, double endY, int thickness, uint borderColor) {
   double deltaX = endX - startX;                               //--- Compute delta X
   double deltaY = endY - startY;                               //--- Compute delta Y
   double edgeLength = MathSqrt(deltaX*deltaX + deltaY*deltaY); //--- Compute edge length
   if(edgeLength < 1e-6) return;                                //--- Return if length too small

   double perpendicularX = -deltaY / edgeLength;                //--- Compute perpendicular X
   double perpendicularY = deltaX / edgeLength;                 //--- Compute perpendicular Y

   double edgeDirectionX = deltaX / edgeLength;                 //--- Compute edge direction X
   double edgeDirectionY = deltaY / edgeLength;                 //--- Compute edge direction Y

   double halfThickness = (double)thickness / 2.0;              //--- Compute half thickness
   
   double extensionLength = borderExtensionMultiplier * (double)thickness; //--- Set extension length

   double extendedStartX = startX - edgeDirectionX * extensionLength; //--- Extend start X
   double extendedStartY = startY - edgeDirectionY * extensionLength; //--- Extend start Y
   double extendedEndX = endX + edgeDirectionX * extensionLength; //--- Extend end X
   double extendedEndY = endY + edgeDirectionY * extensionLength; //--- Extend end Y

   double verticesX[4], verticesY[4];                           //--- Declare vertices arrays
   verticesX[0] = extendedStartX - perpendicularX * halfThickness;  verticesY[0] = extendedStartY - perpendicularY * halfThickness; //--- Set vertex 0
   verticesX[1] = extendedStartX + perpendicularX * halfThickness;  verticesY[1] = extendedStartY + perpendicularY * halfThickness; //--- Set vertex 1
   verticesX[2] = extendedEndX + perpendicularX * halfThickness;  verticesY[2] = extendedEndY + perpendicularY * halfThickness; //--- Set vertex 2
   verticesX[3] = extendedEndX - perpendicularX * halfThickness;  verticesY[3] = extendedEndY - perpendicularY * halfThickness; //--- Set vertex 3

   FillQuadrilateral(triangleHighResCanvas, verticesX, verticesY, borderColor); //--- Fill quadrilateral for edge
}

void DrawTriCornerArcPrecise(int cornerIndex, int thickness, uint borderColor) {
   double centerX = triangleArcCentersX[cornerIndex],  centerY = triangleArcCentersY[cornerIndex]; //--- Get arc center
   double radius  = (double)triangleCornerRadiusPixels * supersamplingFactor; //--- Get scaled radius

   int    halfThickness = thickness / 2;                        //--- Compute half thickness
   double outerRadius = radius + halfThickness;                 //--- Compute outer radius
   double innerRadius = radius - halfThickness;                 //--- Compute inner radius
   if(innerRadius < 0) innerRadius = 0;                         //--- Set inner radius to zero if negative

   int pixelRange = (int)(outerRadius + 2);                     //--- Compute pixel range

   for(int deltaY = -pixelRange; deltaY <= pixelRange; deltaY++) { //--- Loop over delta Y
      for(int deltaX = -pixelRange; deltaX <= pixelRange; deltaX++) { //--- Loop over delta X
         double distance = MathSqrt((double)(deltaX*deltaX + deltaY*deltaY)); //--- Compute distance
         if(distance < innerRadius || distance > outerRadius) continue; //--- Skip if outside radii

         double angle = MathArctan2((double)deltaY, (double)deltaX); //--- Compute angle
         
         if(TriangleAngleInArcSweep(cornerIndex, angle)) {      //--- Check if in arc sweep
            int pixelX = (int)MathRound(centerX + deltaX);      //--- Round to pixel X
            int pixelY = (int)MathRound(centerY + deltaY);      //--- Round to pixel Y
            if(pixelX >= 0 && pixelX < triangleHighResWidth && pixelY >= 0 && pixelY < triangleHighResHeight) //--- Check if within bounds
               triangleHighResCanvas.PixelSet(pixelX, pixelY, borderColor); //--- Set pixel
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Bubble Shape                                                     |
//+------------------------------------------------------------------+
void PrecomputeBubbleGeometry() {
   int scalingFactor = supersamplingFactor;                     //--- Set scaling factor
   double baseOffset = 10.0 * scalingFactor;                    //--- Set base offset scaled
   
   double centeringAdjustment;                                  //--- Declare centering adjustment

   if(bubblePointerOrientation == ORIENT_UP) {                  //--- Check for up orientation
      bubbleBodyLeft = baseOffset;                              //--- Set body left
      bubbleBodyTop = baseOffset + bubblePointerHeightPixels * scalingFactor; //--- Set body top
      bubbleBodyRight = bubbleBodyLeft + bubbleBodyWidthPixels * scalingFactor; //--- Set body right
      bubbleBodyBottom = bubbleBodyTop + bubbleBodyHeightPixels * scalingFactor; //--- Set body bottom

      centeringAdjustment = (bubbleBodyWidthPixels * scalingFactor - bubblePointerBaseWidthPixels * scalingFactor) / 2.0; //--- Compute centering
      double actualOffset = centeringAdjustment + (bubblePointerBaseOffsetPixels * scalingFactor); //--- Apply offset
      
      double pointerCenterX = bubbleBodyLeft + actualOffset + (bubblePointerBaseWidthPixels * scalingFactor) / 2.0; //--- Compute pointer center X
      
      bubblePointerVerticesX[0] = pointerCenterX;               //--- Set apex X
      bubblePointerVerticesY[0] = baseOffset;                   //--- Set apex Y
      
      bubblePointerVerticesX[1] = bubbleBodyLeft + actualOffset; //--- Set left X
      bubblePointerVerticesY[1] = bubbleBodyTop;                //--- Set left Y
      
      bubblePointerVerticesX[2] = bubblePointerVerticesX[1] + bubblePointerBaseWidthPixels * scalingFactor; //--- Set right X
      bubblePointerVerticesY[2] = bubbleBodyTop;                //--- Set right Y
      
      bubblePointerApexIndex = 0;                               //--- Set apex index
      bubblePointerBaseStart = bubblePointerVerticesX[1];       //--- Set base start
      bubblePointerBaseEnd = bubblePointerVerticesX[2];         //--- Set base end
      
   } else if(bubblePointerOrientation == ORIENT_DOWN) {        //--- Check for down orientation
      bubbleBodyLeft = baseOffset;                              //--- Set body left
      bubbleBodyTop = baseOffset;                               //--- Set body top
      bubbleBodyRight = bubbleBodyLeft + bubbleBodyWidthPixels * scalingFactor; //--- Set body right
      bubbleBodyBottom = bubbleBodyTop + bubbleBodyHeightPixels * scalingFactor; //--- Set body bottom

      centeringAdjustment = (bubbleBodyWidthPixels * scalingFactor - bubblePointerBaseWidthPixels * scalingFactor) / 2.0; //--- Compute centering
      double actualOffset = centeringAdjustment + (bubblePointerBaseOffsetPixels * scalingFactor); //--- Apply offset
      
      double pointerCenterX = bubbleBodyLeft + actualOffset + (bubblePointerBaseWidthPixels * scalingFactor) / 2.0; //--- Compute pointer center X
      
      bubblePointerVerticesX[0] = pointerCenterX;               //--- Set apex X
      bubblePointerVerticesY[0] = bubbleBodyBottom + bubblePointerHeightPixels * scalingFactor; //--- Set apex Y
      
      bubblePointerVerticesX[1] = bubbleBodyLeft + actualOffset + bubblePointerBaseWidthPixels * scalingFactor; //--- Set right X
      bubblePointerVerticesY[1] = bubbleBodyBottom;             //--- Set right Y
      
      bubblePointerVerticesX[2] = bubbleBodyLeft + actualOffset; //--- Set left X
      bubblePointerVerticesY[2] = bubbleBodyBottom;             //--- Set left Y
      
      bubblePointerApexIndex = 0;                               //--- Set apex index
      bubblePointerBaseStart = bubblePointerVerticesX[2];       //--- Set base start
      bubblePointerBaseEnd = bubblePointerVerticesX[1];         //--- Set base end
      
   } else if(bubblePointerOrientation == ORIENT_LEFT) {        //--- Check for left orientation
      bubbleBodyLeft = baseOffset + bubblePointerHeightPixels * scalingFactor; //--- Set body left
      bubbleBodyTop = baseOffset;                               //--- Set body top
      bubbleBodyRight = bubbleBodyLeft + bubbleBodyWidthPixels * scalingFactor; //--- Set body right
      bubbleBodyBottom = bubbleBodyTop + bubbleBodyHeightPixels * scalingFactor; //--- Set body bottom

      centeringAdjustment = (bubbleBodyHeightPixels * scalingFactor - bubblePointerBaseWidthPixels * scalingFactor) / 2.0; //--- Compute centering
      double actualOffset = centeringAdjustment + (bubblePointerBaseOffsetPixels * scalingFactor); //--- Apply offset
      
      double pointerCenterY = bubbleBodyTop + actualOffset + (bubblePointerBaseWidthPixels * scalingFactor) / 2.0; //--- Compute pointer center Y
      
      bubblePointerVerticesX[0] = baseOffset;                   //--- Set apex X
      bubblePointerVerticesY[0] = pointerCenterY;               //--- Set apex Y
      
      bubblePointerVerticesX[1] = bubbleBodyLeft;               //--- Set bottom X
      bubblePointerVerticesY[1] = bubbleBodyTop + actualOffset + bubblePointerBaseWidthPixels * scalingFactor; //--- Set bottom Y
      
      bubblePointerVerticesX[2] = bubbleBodyLeft;               //--- Set top X
      bubblePointerVerticesY[2] = bubbleBodyTop + actualOffset; //--- Set top Y
      
      bubblePointerApexIndex = 0;                               //--- Set apex index
      bubblePointerBaseStart = bubblePointerVerticesY[2];       //--- Set base start
      bubblePointerBaseEnd = bubblePointerVerticesY[1];         //--- Set base end
      
   } else {                                                     //--- Handle right orientation
      bubbleBodyLeft = baseOffset;                              //--- Set body left
      bubbleBodyTop = baseOffset;                               //--- Set body top
      bubbleBodyRight = bubbleBodyLeft + bubbleBodyWidthPixels * scalingFactor; //--- Set body right
      bubbleBodyBottom = bubbleBodyTop + bubbleBodyHeightPixels * scalingFactor; //--- Set body bottom

      centeringAdjustment = (bubbleBodyHeightPixels * scalingFactor - bubblePointerBaseWidthPixels * scalingFactor) / 2.0; //--- Compute centering
      double actualOffset = centeringAdjustment + (bubblePointerBaseOffsetPixels * scalingFactor); //--- Apply offset
      
      double pointerCenterY = bubbleBodyTop + actualOffset + (bubblePointerBaseWidthPixels * scalingFactor) / 2.0; //--- Compute pointer center Y
      
      bubblePointerVerticesX[0] = bubbleBodyRight + bubblePointerHeightPixels * scalingFactor; //--- Set apex X
      bubblePointerVerticesY[0] = pointerCenterY;               //--- Set apex Y
      
      bubblePointerVerticesX[1] = bubbleBodyRight;              //--- Set top X
      bubblePointerVerticesY[1] = bubbleBodyTop + actualOffset; //--- Set top Y
      
      bubblePointerVerticesX[2] = bubbleBodyRight;              //--- Set bottom X
      bubblePointerVerticesY[2] = bubbleBodyTop + actualOffset + bubblePointerBaseWidthPixels * scalingFactor; //--- Set bottom Y
      
      bubblePointerApexIndex = 0;                               //--- Set apex index
      bubblePointerBaseStart = bubblePointerVerticesY[1];       //--- Set base start
      bubblePointerBaseEnd = bubblePointerVerticesY[2];         //--- Set base end
   }

   ComputeBubbleTriangleRoundedCorners();                       //--- Compute rounded corners for bubble pointer
}

void ComputeBubbleTriangleRoundedCorners() {
   double scaledRadius = (double)bubblePointerApexRadiusPixels * supersamplingFactor; //--- Scale apex radius

   int cornerIndex = bubblePointerApexIndex;                    //--- Set corner index to apex
   
   int previousIndex = (cornerIndex + 2) % 3;                   //--- Get previous index
   int nextIndex = (cornerIndex + 1) % 3;                       //--- Get next index

   double edgeA_X = bubblePointerVerticesX[cornerIndex] - bubblePointerVerticesX[previousIndex]; //--- Compute edge A X
   double edgeA_Y = bubblePointerVerticesY[cornerIndex] - bubblePointerVerticesY[previousIndex]; //--- Compute edge A Y
   double edgeA_Length = MathSqrt(edgeA_X*edgeA_X + edgeA_Y*edgeA_Y); //--- Compute edge A length
   edgeA_X /= edgeA_Length;  edgeA_Y /= edgeA_Length;           //--- Normalize edge A

   double edgeB_X = bubblePointerVerticesX[nextIndex] - bubblePointerVerticesX[cornerIndex]; //--- Compute edge B X
   double edgeB_Y = bubblePointerVerticesY[nextIndex] - bubblePointerVerticesY[cornerIndex]; //--- Compute edge B Y
   double edgeB_Length = MathSqrt(edgeB_X*edgeB_X + edgeB_Y*edgeB_Y); //--- Compute edge B length
   edgeB_X /= edgeB_Length;  edgeB_Y /= edgeB_Length;           //--- Normalize edge B

   double normalA_X =  edgeA_Y,  normalA_Y = -edgeA_X;          //--- Compute normal A
   double normalB_X =  edgeB_Y,  normalB_Y = -edgeB_X;          //--- Compute normal B

   double bisectorX = normalA_X + normalB_X,  bisectorY = normalA_Y + normalB_Y; //--- Compute bisector
   double bisectorLength = MathSqrt(bisectorX*bisectorX + bisectorY*bisectorY); //--- Compute bisector length
   if(bisectorLength < 1e-12) { bisectorX = normalA_X; bisectorY = normalA_Y; bisectorLength = MathSqrt(bisectorX*bisectorX + bisectorY*bisectorY); } //--- Handle small bisector
   bisectorX /= bisectorLength;  bisectorY /= bisectorLength;   //--- Normalize bisector

   double cosInteriorAngle = (-edgeA_X)*edgeB_X + (-edgeA_Y)*edgeB_Y; //--- Compute cosine of interior angle
   if(cosInteriorAngle >  1.0) cosInteriorAngle =  1.0;         //--- Clamp cosine upper
   if(cosInteriorAngle < -1.0) cosInteriorAngle = -1.0;         //--- Clamp cosine lower
   double halfAngle = MathArccos(cosInteriorAngle) / 2.0;       //--- Compute half angle
   double sinHalfAngle = MathSin(halfAngle);                    //--- Compute sine of half angle
   if(sinHalfAngle < 1e-12) sinHalfAngle = 1e-12;               //--- Set minimum sine value

   double distanceToCenter = scaledRadius / sinHalfAngle;       //--- Compute distance to arc center
   bubblePointerArcCentersX[cornerIndex] = bubblePointerVerticesX[cornerIndex] + bisectorX * distanceToCenter; //--- Set arc center X
   bubblePointerArcCentersY[cornerIndex] = bubblePointerVerticesY[cornerIndex] + bisectorY * distanceToCenter; //--- Set arc center Y

   double deltaX_A = bubblePointerVerticesX[cornerIndex] - bubblePointerVerticesX[previousIndex]; //--- Compute delta A X
   double deltaY_A = bubblePointerVerticesY[cornerIndex] - bubblePointerVerticesY[previousIndex]; //--- Compute delta A Y
   double lengthSquared_A = deltaX_A*deltaX_A + deltaY_A*deltaY_A; //--- Compute length squared A
   double interpolationFactor_A = ((bubblePointerArcCentersX[cornerIndex] - bubblePointerVerticesX[previousIndex])*deltaX_A + 
                                   (bubblePointerArcCentersY[cornerIndex] - bubblePointerVerticesY[previousIndex])*deltaY_A) / lengthSquared_A; //--- Compute factor A
   bubblePointerTangentPointsX[cornerIndex][1] = bubblePointerVerticesX[previousIndex] + interpolationFactor_A * deltaX_A; //--- Set tangent point X arriving
   bubblePointerTangentPointsY[cornerIndex][1] = bubblePointerVerticesY[previousIndex] + interpolationFactor_A * deltaY_A; //--- Set tangent point Y arriving

   double deltaX_B = bubblePointerVerticesX[nextIndex] - bubblePointerVerticesX[cornerIndex]; //--- Compute delta B X
   double deltaY_B = bubblePointerVerticesY[nextIndex] - bubblePointerVerticesY[cornerIndex]; //--- Compute delta B Y
   double lengthSquared_B = deltaX_B*deltaX_B + deltaY_B*deltaY_B; //--- Compute length squared B
   double interpolationFactor_B = ((bubblePointerArcCentersX[cornerIndex] - bubblePointerVerticesX[cornerIndex])*deltaX_B + 
                                   (bubblePointerArcCentersY[cornerIndex] - bubblePointerVerticesY[cornerIndex])*deltaY_B) / lengthSquared_B; //--- Compute factor B
   bubblePointerTangentPointsX[cornerIndex][0] = bubblePointerVerticesX[cornerIndex] + interpolationFactor_B * deltaX_B; //--- Set tangent point X leaving
   bubblePointerTangentPointsY[cornerIndex][0] = bubblePointerVerticesY[cornerIndex] + interpolationFactor_B * deltaY_B; //--- Set tangent point Y leaving

   bubblePointerArcStartAngles[cornerIndex] = MathArctan2(bubblePointerTangentPointsY[cornerIndex][1] - bubblePointerArcCentersY[cornerIndex], 
                                                          bubblePointerTangentPointsX[cornerIndex][1] - bubblePointerArcCentersX[cornerIndex]); //--- Set start angle
   bubblePointerArcEndAngles[cornerIndex] = MathArctan2(bubblePointerTangentPointsY[cornerIndex][0] - bubblePointerArcCentersY[cornerIndex], 
                                                        bubblePointerTangentPointsX[cornerIndex][0] - bubblePointerArcCentersX[cornerIndex]); //--- Set end angle
   
   for(int i = 0; i < 3; i++) {                                 //--- Loop over corners
      if(i == bubblePointerApexIndex) continue;                 //--- Skip apex corner
      
      bubblePointerTangentPointsX[i][0] = bubblePointerVerticesX[i]; //--- Set tangent X leaving to vertex
      bubblePointerTangentPointsY[i][0] = bubblePointerVerticesY[i]; //--- Set tangent Y leaving to vertex
      bubblePointerTangentPointsX[i][1] = bubblePointerVerticesX[i]; //--- Set tangent X arriving to vertex
      bubblePointerTangentPointsY[i][1] = bubblePointerVerticesY[i]; //--- Set tangent Y arriving to vertex
   }
}

bool BubbleAngleInArcSweep(int cornerIndex, double angle) {
   double twoPi = 2.0 * M_PI;                                   //--- Define two pi constant
   double startAngleMod = MathMod(bubblePointerArcStartAngles[cornerIndex] + twoPi, twoPi); //--- Modulo start angle
   double endAngleMod = MathMod(bubblePointerArcEndAngles[cornerIndex] + twoPi, twoPi); //--- Modulo end angle
   angle = MathMod(angle + twoPi, twoPi);                       //--- Modulo angle

   double ccwSpan = MathMod(endAngleMod - startAngleMod + twoPi, twoPi); //--- Compute CCW span

   if(ccwSpan <= M_PI) {                                        //--- Check if short way is CCW
      double relativeAngle = MathMod(angle - startAngleMod + twoPi, twoPi); //--- Compute relative angle
      return(relativeAngle <= ccwSpan + 1e-6);                  //--- Return if within CCW span
   } else {                                                     //--- Else short way is CW
      double cwSpan = twoPi - ccwSpan;                          //--- Compute CW span
      double relativeAngle = MathMod(angle - endAngleMod + twoPi, twoPi); //--- Compute relative angle
      return(relativeAngle <= cwSpan + 1e-6);                   //--- Return if within CW span
   }
}

void DrawBubble() {
   uint backgroundColorARGB = ColorToARGBWithOpacity(bubbleBackgroundColor, bubbleBackgroundOpacityPercent); //--- Get background ARGB
   uint borderColorARGB = ColorToARGBWithOpacity(bubbleBorderColor, bubbleBorderOpacityPercent); //--- Get border ARGB

   FillBubble(backgroundColorARGB);                             //--- Fill bubble

   if(bubbleShowBorder && bubbleBorderThicknessPixels > 0)      //--- Check if border should be shown
      DrawBubbleBorder(borderColorARGB);                        //--- Draw bubble border

   BicubicDownsample(bubbleCanvas, bubbleHighResCanvas);        //--- Downsample to display canvas

   bubbleCanvas.FontSet("Arial", 13, FW_NORMAL);                //--- Set font for text
   string displayText = "Bubble";                               //--- Set display text
   int textWidth, textHeight;                                   //--- Declare text dimensions
   bubbleCanvas.TextSize(displayText, textWidth, textHeight);   //--- Get text size
   
   int bodyDisplayLeft = (int)(bubbleBodyLeft / supersamplingFactor); //--- Compute display left
   int bodyDisplayTop = (int)(bubbleBodyTop / supersamplingFactor); //--- Compute display top
   int bodyDisplayWidth = (int)((bubbleBodyRight - bubbleBodyLeft) / supersamplingFactor); //--- Compute display width
   int bodyDisplayHeight = (int)((bubbleBodyBottom - bubbleBodyTop) / supersamplingFactor); //--- Compute display height
   
   int textPositionX = bodyDisplayLeft + (bodyDisplayWidth - textWidth) / 2; //--- Compute text X position
   int textPositionY = bodyDisplayTop + (bodyDisplayHeight - textHeight) / 2; //--- Compute text Y position
   bubbleCanvas.TextOut(textPositionX, textPositionY, displayText, (uint)0xFF000000, TA_LEFT); //--- Draw text on canvas
}

void FillBubble(uint fillColor) {
   FillBubbleRoundedRectangle(bubbleBodyLeft, bubbleBodyTop, bubbleBodyRight - bubbleBodyLeft, bubbleBodyBottom - bubbleBodyTop, 
                              bubbleBodyCornerRadiusPixels * supersamplingFactor, fillColor); //--- Fill bubble body rectangle

   FillBubbleRoundedTriangle(fillColor);                        //--- Fill bubble pointer triangle
}

void FillBubbleRoundedRectangle(double left, double top, double width, double height, int radius, uint fillColor) {
   bubbleHighResCanvas.FillRectangle((int)(left + radius), (int)top, (int)(left + width - radius), (int)(top + height), fillColor); //--- Fill central rectangle
   bubbleHighResCanvas.FillRectangle((int)left, (int)(top + radius), (int)(left + radius), (int)(top + height - radius), fillColor); //--- Fill left strip
   bubbleHighResCanvas.FillRectangle((int)(left + width - radius), (int)(top + radius), (int)(left + width), (int)(top + height - radius), fillColor); //--- Fill right strip

   FillBubbleCircleQuadrant((int)(left + radius), (int)(top + radius), radius, fillColor, 2); //--- Fill top-left quadrant
   FillBubbleCircleQuadrant((int)(left + width - radius), (int)(top + radius), radius, fillColor, 1); //--- Fill top-right quadrant
   FillBubbleCircleQuadrant((int)(left + radius), (int)(top + height - radius), radius, fillColor, 3); //--- Fill bottom-left quadrant
   FillBubbleCircleQuadrant((int)(left + width - radius), (int)(top + height - radius), radius, fillColor, 4); //--- Fill bottom-right quadrant
}

void FillBubbleCircleQuadrant(int centerX, int centerY, int radius, uint fillColor, int quadrant) {
   double radiusDouble = (double)radius;                        //--- Convert radius to double

   for(int deltaY = -radius - 1; deltaY <= radius + 1; deltaY++) { //--- Loop over delta Y
      for(int deltaX = -radius - 1; deltaX <= radius + 1; deltaX++) { //--- Loop over delta X
         bool inQuadrant = false;                               //--- Initialize quadrant flag
         if(quadrant == 1 && deltaX >= 0 && deltaY <= 0) inQuadrant = true; //--- Check top-right
         else if(quadrant == 2 && deltaX <= 0 && deltaY <= 0) inQuadrant = true; //--- Check top-left
         else if(quadrant == 3 && deltaX <= 0 && deltaY >= 0) inQuadrant = true; //--- Check bottom-left
         else if(quadrant == 4 && deltaX >= 0 && deltaY >= 0) inQuadrant = true; //--- Check bottom-right

         if(!inQuadrant) continue;                              //--- Skip if not in quadrant

         double distance = MathSqrt(deltaX * deltaX + deltaY * deltaY); //--- Compute distance
         if(distance <= radiusDouble)                           //--- Check if within radius
            bubbleHighResCanvas.PixelSet(centerX + deltaX, centerY + deltaY, fillColor); //--- Set pixel
      }
   }
}

void FillBubbleRoundedTriangle(uint fillColor) {
   if(bubbleIsHorizontalOrientation) {                          //--- Check for horizontal orientation
      double minX = bubblePointerVerticesX[0], maxX = bubblePointerVerticesX[0]; //--- Initialize min and max X
      for(int i = 1; i < 3; i++) {                              //--- Loop over vertices
         if(bubblePointerVerticesX[i] < minX) minX = bubblePointerVerticesX[i]; //--- Update min X
         if(bubblePointerVerticesX[i] > maxX) maxX = bubblePointerVerticesX[i]; //--- Update max X
      }

      int xStart = (int)MathCeil(minX);                         //--- Compute start X
      int xEnd   = (int)MathFloor(maxX);                        //--- Compute end X

      for(int x = xStart; x <= xEnd; x++) {                     //--- Loop over scanlines
         double scanlineX = (double)x + 0.5;                    //--- Set scanline X position
         double yIntersections[12];                             //--- Declare intersections array
         int intersectionCount = 0;                             //--- Initialize intersection count

         for(int edgeIndex = 0; edgeIndex < 3; edgeIndex++) {   //--- Loop over edges
            int nextIndex = (edgeIndex + 1) % 3;                //--- Get next index

            if(edgeIndex != bubblePointerApexIndex && nextIndex != bubblePointerApexIndex) continue; //--- Skip non-apex edges

            double startX, startY, endX, endY;                  //--- Declare edge coordinates

            if(edgeIndex == bubblePointerApexIndex) {           //--- Check if from apex
               startX = bubblePointerTangentPointsX[bubblePointerApexIndex][0]; //--- Set start X from tangent
               startY = bubblePointerTangentPointsY[bubblePointerApexIndex][0]; //--- Set start Y from tangent
               endX   = bubblePointerVerticesX[nextIndex];     //--- Set end X to next vertex
               endY   = bubblePointerVerticesY[nextIndex];     //--- Set end Y to next vertex
            } else {                                            //--- Handle to apex
               startX = bubblePointerVerticesX[edgeIndex];     //--- Set start X from vertex
               startY = bubblePointerVerticesY[edgeIndex];     //--- Set start Y from vertex
               endX   = bubblePointerTangentPointsX[bubblePointerApexIndex][1]; //--- Set end X to tangent
               endY   = bubblePointerTangentPointsY[bubblePointerApexIndex][1]; //--- Set end Y to tangent
            }

            double edgeMinX = (startX < endX) ? startX : endX;  //--- Compute edge min X
            double edgeMaxX = (startX > endX) ? startX : endX;  //--- Compute edge max X

            if(scanlineX < edgeMinX || scanlineX > edgeMaxX) continue; //--- Skip if outside edge X
            if(MathAbs(endX - startX) < 1e-12) continue;        //--- Skip if vertical edge

            double interpolationFactor = (scanlineX - startX) / (endX - startX); //--- Compute factor
            if(interpolationFactor < 0.0 || interpolationFactor > 1.0) continue; //--- Skip if outside segment

            yIntersections[intersectionCount++] = startY + interpolationFactor * (endY - startY); //--- Add intersection Y
         }

         {                                                      //--- Intersect apex arc block
            int cornerIndex = bubblePointerApexIndex;           //--- Set corner index
            double centerX  = bubblePointerArcCentersX[cornerIndex]; //--- Get center X
            double centerY  = bubblePointerArcCentersY[cornerIndex]; //--- Get center Y
            double radius   = (double)bubblePointerApexRadiusPixels * supersamplingFactor; //--- Get scaled radius
            double deltaX   = scanlineX - centerX;              //--- Compute delta X

            if(MathAbs(deltaX) <= radius) {                     //--- Check if within radius
               double deltaY = MathSqrt(radius * radius - deltaX * deltaX); //--- Compute delta Y

               double candidates[2];                            //--- Declare candidates array
               candidates[0] = centerY - deltaY;                //--- Set top candidate
               candidates[1] = centerY + deltaY;                //--- Set bottom candidate

               for(int candidateIndex = 0; candidateIndex < 2; candidateIndex++) { //--- Loop over candidates
                  double angle = MathArctan2(candidates[candidateIndex] - centerY, scanlineX - centerX); //--- Compute angle
                  if(BubbleAngleInArcSweep(cornerIndex, angle)) //--- Check if in arc sweep
                     yIntersections[intersectionCount++] = candidates[candidateIndex]; //--- Add intersection
               }
            }
         }

         for(int a = 0; a < intersectionCount - 1; a++)         //--- Sort intersections (bubble sort)
            for(int b = a + 1; b < intersectionCount; b++)       //--- Inner loop for sorting
               if(yIntersections[a] > yIntersections[b]) {       //--- Check if swap needed
                  double temp = yIntersections[a];               //--- Temporary store
                  yIntersections[a] = yIntersections[b];         //--- Swap values
                  yIntersections[b] = temp;                      //--- Complete swap
               }

         for(int pairIndex = 0; pairIndex + 1 < intersectionCount; pairIndex += 2) { //--- Loop over pairs
            int yTop    = (int)MathCeil(yIntersections[pairIndex]); //--- Compute top Y
            int yBottom = (int)MathFloor(yIntersections[pairIndex + 1]); //--- Compute bottom Y
            for(int y = yTop; y <= yBottom; y++)                //--- Loop over vertical span
               bubbleHighResCanvas.PixelSet(x, y, fillColor);   //--- Set pixel with fill color
         }
      }

   } else {                                                     //--- Handle vertical orientation
      double minY = bubblePointerVerticesY[0], maxY = bubblePointerVerticesY[0]; //--- Initialize min and max Y
      for(int i = 1; i < 3; i++) {                              //--- Loop over vertices
         if(bubblePointerVerticesY[i] < minY) minY = bubblePointerVerticesY[i]; //--- Update min Y
         if(bubblePointerVerticesY[i] > maxY) maxY = bubblePointerVerticesY[i]; //--- Update max Y
      }

      int yStart = (int)MathCeil(minY);                         //--- Compute start Y
      int yEnd = (int)MathFloor(maxY);                          //--- Compute end Y

      for(int y = yStart; y <= yEnd; y++) {                     //--- Loop over scanlines
         double scanlineY = (double)y + 0.5;                    //--- Set scanline Y position
         double xIntersections[12];                             //--- Declare intersections array
         int intersectionCount = 0;                             //--- Initialize intersection count

         for(int edgeIndex = 0; edgeIndex < 3; edgeIndex++) {   //--- Loop over edges
            int nextIndex = (edgeIndex + 1) % 3;                //--- Get next index
            
            if(edgeIndex != bubblePointerApexIndex && nextIndex != bubblePointerApexIndex) continue; //--- Skip non-apex edges
            
            double startX, startY, endX, endY;                  //--- Declare edge coordinates
            
            if(edgeIndex == bubblePointerApexIndex) {           //--- Check if from apex
               startX = bubblePointerTangentPointsX[bubblePointerApexIndex][0]; //--- Set start X from tangent
               startY = bubblePointerTangentPointsY[bubblePointerApexIndex][0]; //--- Set start Y from tangent
               endX = bubblePointerVerticesX[nextIndex];       //--- Set end X to next vertex
               endY = bubblePointerVerticesY[nextIndex];       //--- Set end Y to next vertex
            } else {                                            //--- Handle to apex
               startX = bubblePointerVerticesX[edgeIndex];     //--- Set start X from vertex
               startY = bubblePointerVerticesY[edgeIndex];     //--- Set start Y from vertex
               endX = bubblePointerTangentPointsX[bubblePointerApexIndex][1]; //--- Set end X to tangent
               endY = bubblePointerTangentPointsY[bubblePointerApexIndex][1]; //--- Set end Y to tangent
            }

            double edgeMinY = (startY < endY) ? startY : endY;  //--- Compute edge min Y
            double edgeMaxY = (startY > endY) ? startY : endY;  //--- Compute edge max Y

            if(scanlineY < edgeMinY || scanlineY > edgeMaxY) continue; //--- Skip if outside edge Y
            if(MathAbs(endY - startY) < 1e-12) continue;        //--- Skip if horizontal

            double interpolationFactor = (scanlineY - startY) / (endY - startY); //--- Compute factor
            if(interpolationFactor < 0.0 || interpolationFactor > 1.0) continue; //--- Skip if outside segment

            xIntersections[intersectionCount++] = startX + interpolationFactor * (endX - startX); //--- Add intersection X
         }

         int cornerIndex = bubblePointerApexIndex;              //--- Set corner index
         double centerX = bubblePointerArcCentersX[cornerIndex]; //--- Get center X
         double centerY = bubblePointerArcCentersY[cornerIndex]; //--- Get center Y
         double radius = (double)bubblePointerApexRadiusPixels * supersamplingFactor; //--- Get scaled radius
         double deltaY = scanlineY - centerY;                   //--- Compute delta Y

         if(MathAbs(deltaY) <= radius) {                        //--- Check if within radius
            double deltaX = MathSqrt(radius*radius - deltaY*deltaY); //--- Compute delta X

            double candidates[2];                               //--- Declare candidates array
            candidates[0] = centerX - deltaX;                   //--- Set left candidate
            candidates[1] = centerX + deltaX;                   //--- Set right candidate

            for(int candidateIndex = 0; candidateIndex < 2; candidateIndex++) { //--- Loop over candidates
               double angle = MathArctan2(scanlineY - centerY, candidates[candidateIndex] - centerX); //--- Compute angle
               if(BubbleAngleInArcSweep(cornerIndex, angle))    //--- Check if in arc sweep
                  xIntersections[intersectionCount++] = candidates[candidateIndex]; //--- Add intersection
            }
         }

         for(int a = 0; a < intersectionCount - 1; a++)         //--- Sort intersections (bubble sort)
            for(int b = a + 1; b < intersectionCount; b++)       //--- Inner loop for sorting
               if(xIntersections[a] > xIntersections[b]) {       //--- Check if swap needed
                  double temp = xIntersections[a];               //--- Temporary store
                  xIntersections[a] = xIntersections[b];         //--- Swap values
                  xIntersections[b] = temp;                      //--- Complete swap
               }

         for(int pairIndex = 0; pairIndex + 1 < intersectionCount; pairIndex += 2) { //--- Loop over pairs
            int xLeft = (int)MathCeil(xIntersections[pairIndex]); //--- Compute left X
            int xRight = (int)MathFloor(xIntersections[pairIndex + 1]); //--- Compute right X
            for(int x = xLeft; x <= xRight; x++)                //--- Loop over horizontal span
               bubbleHighResCanvas.PixelSet(x, y, fillColor);   //--- Set pixel with fill color
         }
      }
   }
}

void DrawBubbleBorder(uint borderColorARGB) {
   int scaledThickness = bubbleBorderThicknessPixels * supersamplingFactor; //--- Scale border thickness
   int scaledBodyRadius = bubbleBodyCornerRadiusPixels * supersamplingFactor; //--- Scale body radius

   if(bubblePointerOrientation == ORIENT_UP || bubblePointerOrientation == ORIENT_DOWN) { //--- Check for up or down orientation
      if(bubblePointerOrientation == ORIENT_DOWN) {             //--- Check for down orientation
         DrawBubbleHorizontalEdge(bubbleBodyLeft + scaledBodyRadius, bubbleBodyTop, bubbleBodyRight - scaledBodyRadius, bubbleBodyTop, scaledThickness, borderColorARGB); //--- Draw top edge
      } else {                                                  //--- Handle up orientation
         if(bubblePointerBaseStart > bubbleBodyLeft + scaledBodyRadius) //--- Check if base start exceeds left radius
            DrawBubbleHorizontalEdge(bubbleBodyLeft + scaledBodyRadius, bubbleBodyTop, bubblePointerBaseStart, bubbleBodyTop, scaledThickness, borderColorARGB); //--- Draw left segment of top edge
         if(bubblePointerBaseEnd < bubbleBodyRight - scaledBodyRadius) //--- Check if base end below right radius
            DrawBubbleHorizontalEdge(bubblePointerBaseEnd, bubbleBodyTop, bubbleBodyRight - scaledBodyRadius, bubbleBodyTop, scaledThickness, borderColorARGB); //--- Draw right segment of top edge
      }

      if(bubblePointerOrientation == ORIENT_UP) {               //--- Check for up orientation
         DrawBubbleHorizontalEdge(bubbleBodyRight - scaledBodyRadius, bubbleBodyBottom, bubbleBodyLeft + scaledBodyRadius, bubbleBodyBottom, scaledThickness, borderColorARGB); //--- Draw bottom edge
      } else {                                                  //--- Handle down orientation
         if(bubblePointerBaseStart > bubbleBodyLeft + scaledBodyRadius) //--- Check if base start exceeds left radius
            DrawBubbleHorizontalEdge(bubblePointerBaseStart, bubbleBodyBottom, bubbleBodyLeft + scaledBodyRadius, bubbleBodyBottom, scaledThickness, borderColorARGB); //--- Draw left segment of bottom edge
         if(bubblePointerBaseEnd < bubbleBodyRight - scaledBodyRadius) //--- Check if base end below right radius
            DrawBubbleHorizontalEdge(bubbleBodyRight - scaledBodyRadius, bubbleBodyBottom, bubblePointerBaseEnd, bubbleBodyBottom, scaledThickness, borderColorARGB); //--- Draw right segment of bottom edge
      }

      DrawBubbleVerticalEdge(bubbleBodyLeft, bubbleBodyBottom - scaledBodyRadius, bubbleBodyLeft, bubbleBodyTop + scaledBodyRadius, scaledThickness, borderColorARGB); //--- Draw left edge
      DrawBubbleVerticalEdge(bubbleBodyRight, bubbleBodyTop + scaledBodyRadius, bubbleBodyRight, bubbleBodyBottom - scaledBodyRadius, scaledThickness, borderColorARGB); //--- Draw right edge

   } else {                                                     //--- Handle left or right orientation
      DrawBubbleHorizontalEdge(bubbleBodyLeft + scaledBodyRadius, bubbleBodyTop, bubbleBodyRight - scaledBodyRadius, bubbleBodyTop, scaledThickness, borderColorARGB); //--- Draw top edge
      DrawBubbleHorizontalEdge(bubbleBodyRight - scaledBodyRadius, bubbleBodyBottom, bubbleBodyLeft + scaledBodyRadius, bubbleBodyBottom, scaledThickness, borderColorARGB); //--- Draw bottom edge

      if(bubblePointerOrientation == ORIENT_RIGHT) {            //--- Check for right orientation
         DrawBubbleVerticalEdge(bubbleBodyLeft, bubbleBodyBottom - scaledBodyRadius, bubbleBodyLeft, bubbleBodyTop + scaledBodyRadius, scaledThickness, borderColorARGB); //--- Draw left edge
      } else {                                                  //--- Handle left orientation
         if(bubblePointerBaseStart > bubbleBodyTop + scaledBodyRadius) //--- Check if base start exceeds top radius
            DrawBubbleVerticalEdge(bubbleBodyLeft, bubblePointerBaseStart, bubbleBodyLeft, bubbleBodyTop + scaledBodyRadius, scaledThickness, borderColorARGB); //--- Draw top segment of left edge
         if(bubblePointerBaseEnd < bubbleBodyBottom - scaledBodyRadius) //--- Check if base end below bottom radius
            DrawBubbleVerticalEdge(bubbleBodyLeft, bubbleBodyBottom - scaledBodyRadius, bubbleBodyLeft, bubblePointerBaseEnd, scaledThickness, borderColorARGB); //--- Draw bottom segment of left edge
      }

      if(bubblePointerOrientation == ORIENT_LEFT) {             //--- Check for left orientation
         DrawBubbleVerticalEdge(bubbleBodyRight, bubbleBodyTop + scaledBodyRadius, bubbleBodyRight, bubbleBodyBottom - scaledBodyRadius, scaledThickness, borderColorARGB); //--- Draw right edge
      } else {                                                  //--- Handle right orientation
         if(bubblePointerBaseStart > bubbleBodyTop + scaledBodyRadius) //--- Check if base start exceeds top radius
            DrawBubbleVerticalEdge(bubbleBodyRight, bubbleBodyTop + scaledBodyRadius, bubbleBodyRight, bubblePointerBaseStart, scaledThickness, borderColorARGB); //--- Draw top segment of right edge
         if(bubblePointerBaseEnd < bubbleBodyBottom - scaledBodyRadius) //--- Check if base end below bottom radius
            DrawBubbleVerticalEdge(bubbleBodyRight, bubblePointerBaseEnd, bubbleBodyRight, bubbleBodyBottom - scaledBodyRadius, scaledThickness, borderColorARGB); //--- Draw bottom segment of right edge
      }
   }

   DrawBubbleCornerArc((int)(bubbleBodyLeft + scaledBodyRadius), (int)(bubbleBodyTop + scaledBodyRadius), scaledBodyRadius, scaledThickness, borderColorARGB, M_PI, M_PI * 1.5); //--- Draw top-left arc
   DrawBubbleCornerArc((int)(bubbleBodyRight - scaledBodyRadius), (int)(bubbleBodyTop + scaledBodyRadius), scaledBodyRadius, scaledThickness, borderColorARGB, M_PI * 1.5, M_PI * 2.0); //--- Draw top-right arc
   DrawBubbleCornerArc((int)(bubbleBodyLeft + scaledBodyRadius), (int)(bubbleBodyBottom - scaledBodyRadius), scaledBodyRadius, scaledThickness, borderColorARGB, M_PI * 0.5, M_PI); //--- Draw bottom-left arc
   DrawBubbleCornerArc((int)(bubbleBodyRight - scaledBodyRadius), (int)(bubbleBodyBottom - scaledBodyRadius), scaledBodyRadius, scaledThickness, borderColorARGB, 0.0, M_PI * 0.5); //--- Draw bottom-right arc

   for(int edgeIndex = 0; edgeIndex < 3; edgeIndex++) {         //--- Loop over pointer edges
      int nextIndex = (edgeIndex + 1) % 3;                      //--- Get next index
      
      if(edgeIndex != bubblePointerApexIndex && nextIndex != bubblePointerApexIndex) continue; //--- Skip non-apex edges
      
      double startX, startY, endX, endY;                        //--- Declare edge coordinates
      
      if(edgeIndex == bubblePointerApexIndex) {                 //--- Check if from apex
         startX = bubblePointerTangentPointsX[bubblePointerApexIndex][0]; //--- Set start X from tangent
         startY = bubblePointerTangentPointsY[bubblePointerApexIndex][0]; //--- Set start Y from tangent
         endX = bubblePointerVerticesX[nextIndex];              //--- Set end X to next vertex
         endY = bubblePointerVerticesY[nextIndex];              //--- Set end Y to next vertex
      } else {                                                  //--- Handle to apex
         startX = bubblePointerVerticesX[edgeIndex];            //--- Set start X from vertex
         startY = bubblePointerVerticesY[edgeIndex];            //--- Set start Y from vertex
         endX = bubblePointerTangentPointsX[bubblePointerApexIndex][1]; //--- Set end X to tangent
         endY = bubblePointerTangentPointsY[bubblePointerApexIndex][1]; //--- Set end Y to tangent
      }

      DrawBubbleStraightEdge(startX, startY, endX, endY, scaledThickness, borderColorARGB); //--- Draw straight edge
   }

   DrawBubbleTriangleCornerArc(bubblePointerApexIndex, scaledThickness, borderColorARGB); //--- Draw apex arc
}

void DrawBubbleHorizontalEdge(double startX, double startY, double endX, double endY, int thickness, uint edgeColor) {
   DrawBubbleStraightEdge(startX, startY, endX, endY, thickness, edgeColor); //--- Draw horizontal edge using straight edge
}

void DrawBubbleVerticalEdge(double startX, double startY, double endX, double endY, int thickness, uint edgeColor) {
   DrawBubbleStraightEdge(startX, startY, endX, endY, thickness, edgeColor); //--- Draw vertical edge using straight edge
}

void DrawBubbleStraightEdge(double startX, double startY, double endX, double endY, int thickness, uint edgeColor) {
   double deltaX = endX - startX;                               //--- Compute delta X
   double deltaY = endY - startY;                               //--- Compute delta Y
   double edgeLength = MathSqrt(deltaX*deltaX + deltaY*deltaY); //--- Compute edge length
   if(edgeLength < 1e-6) return;                                //--- Return if length too small

   double perpendicularX = -deltaY / edgeLength;                //--- Compute perpendicular X
   double perpendicularY = deltaX / edgeLength;                 //--- Compute perpendicular Y

   double edgeDirectionX = deltaX / edgeLength;                 //--- Compute edge direction X
   double edgeDirectionY = deltaY / edgeLength;                 //--- Compute edge direction Y

   double halfThickness = (double)thickness / 2.0;              //--- Compute half thickness
   double extensionLength = borderExtensionMultiplier * (double)thickness; //--- Set extension length
   
   double extendedStartX = startX - edgeDirectionX * extensionLength; //--- Extend start X
   double extendedStartY = startY - edgeDirectionY * extensionLength; //--- Extend start Y
   double extendedEndX = endX + edgeDirectionX * extensionLength; //--- Extend end X
   double extendedEndY = endY + edgeDirectionY * extensionLength; //--- Extend end Y

   double verticesX[4], verticesY[4];                           //--- Declare vertices arrays
   verticesX[0] = extendedStartX - perpendicularX * halfThickness;  verticesY[0] = extendedStartY - perpendicularY * halfThickness; //--- Set vertex 0
   verticesX[1] = extendedStartX + perpendicularX * halfThickness;  verticesY[1] = extendedStartY + perpendicularY * halfThickness; //--- Set vertex 1
   verticesX[2] = extendedEndX + perpendicularX * halfThickness;  verticesY[2] = extendedEndY + perpendicularY * halfThickness; //--- Set vertex 2
   verticesX[3] = extendedEndX - perpendicularX * halfThickness;  verticesY[3] = extendedEndY - perpendicularY * halfThickness; //--- Set vertex 3

   FillQuadrilateral(bubbleHighResCanvas, verticesX, verticesY, edgeColor); //--- Fill quadrilateral for edge
}

void DrawBubbleCornerArc(int centerX, int centerY, int radius, int thickness, uint edgeColor,
                         double startAngle, double endAngle) {
   int halfThickness = thickness / 2;                           //--- Compute half thickness
   double outerRadius = (double)radius + halfThickness;         //--- Compute outer radius
   double innerRadius = (double)radius - halfThickness;         //--- Compute inner radius
   if(innerRadius < 0) innerRadius = 0;                         //--- Set inner radius to zero if negative

   int pixelRange = (int)(outerRadius + 2);                     //--- Compute pixel range

   for(int deltaY = -pixelRange; deltaY <= pixelRange; deltaY++) { //--- Loop over delta Y
      for(int deltaX = -pixelRange; deltaX <= pixelRange; deltaX++) { //--- Loop over delta X
         double distance = MathSqrt(deltaX * deltaX + deltaY * deltaY); //--- Compute distance
         if(distance < innerRadius || distance > outerRadius) continue; //--- Skip if outside radii

         double angle = MathArctan2((double)deltaY, (double)deltaX); //--- Compute angle
         
         if(IsAngleBetween(angle, startAngle, endAngle))        //--- Check if angle within range
            bubbleHighResCanvas.PixelSet(centerX + deltaX, centerY + deltaY, edgeColor); //--- Set pixel
      }
   }
}

void DrawBubbleTriangleCornerArc(int cornerIndex, int thickness, uint edgeColor) {
   double centerX = bubblePointerArcCentersX[cornerIndex];      //--- Get center X
   double centerY = bubblePointerArcCentersY[cornerIndex];      //--- Get center Y
   double radius = (double)bubblePointerApexRadiusPixels * supersamplingFactor; //--- Get scaled radius

   int halfThickness = thickness / 2;                           //--- Compute half thickness
   double outerRadius = radius + halfThickness;                 //--- Compute outer radius
   double innerRadius = radius - halfThickness;                 //--- Compute inner radius
   if(innerRadius < 0) innerRadius = 0;                         //--- Set inner radius to zero if negative

   int pixelRange = (int)(outerRadius + 2);                     //--- Compute pixel range

   for(int deltaY = -pixelRange; deltaY <= pixelRange; deltaY++) { //--- Loop over delta Y
      for(int deltaX = -pixelRange; deltaX <= pixelRange; deltaX++) { //--- Loop over delta X
         double distance = MathSqrt((double)(deltaX*deltaX + deltaY*deltaY)); //--- Compute distance
         if(distance < innerRadius || distance > outerRadius) continue; //--- Skip if outside radii

         double angle = MathArctan2((double)deltaY, (double)deltaX); //--- Compute angle
         
         if(BubbleAngleInArcSweep(cornerIndex, angle)) {        //--- Check if in arc sweep
            int pixelX = (int)MathRound(centerX + deltaX);      //--- Round to pixel X
            int pixelY = (int)MathRound(centerY + deltaY);      //--- Round to pixel Y
            if(pixelX >= 0 && pixelX < bubbleHighResCanvas.Width() && pixelY >= 0 && pixelY < bubbleHighResCanvas.Height()) //--- Check if within bounds
               bubbleHighResCanvas.PixelSet(pixelX, pixelY, edgeColor); //--- Set pixel
         }
      }
   }
}
//+------------------------------------------------------------------+