﻿//+------------------------------------------------------------------+
//|                                       Canvas Dashboard PART4.mq5 |
//|                           Copyright 2026, Allan Munene Mutiiria. |
//|                                   https://t.me/Forex_Algo_Trader |
//+------------------------------------------------------------------+
#property copyright "Copyright 2026, Allan Munene Mutiiria."
#property link "https://t.me/Forex_Algo_Trader"
#property version "1.00"
#property strict

#include <Canvas/Canvas.mqh>

//+------------------------------------------------------------------+
//| Resources                                                        |
//+------------------------------------------------------------------+
#resource "1. Transparent MT5 bmp image.bmp" // Define background image resource

//+------------------------------------------------------------------+
//| Enumerations                                                     |
//+------------------------------------------------------------------+
enum ENUM_BACKGROUND_MODE {             // Define background mode enumeration
   NoColor          = 0,                // No color mode
   SingleColor      = 1,                // Single color mode
   GradientTwoColors = 2                // Gradient with two colors mode
};

enum ENUM_RESIZE_MODE {                 // Define resize mode enumeration
   NONE,                                // No resize mode
   BOTTOM,                              // Bottom resize mode
   RIGHT,                               // Right resize mode
   BOTTOM_RIGHT                         // Bottom-right resize mode
};

//+------------------------------------------------------------------+
//| Canvas objects                                                   |
//+------------------------------------------------------------------+
CCanvas canvasGraph;                    //--- Declare graph canvas object
CCanvas canvasStats;                    //--- Declare stats canvas object
CCanvas canvasStatsHighRes;             //--- Declare stats high-res canvas object
CCanvas canvasHeader;                   //--- Declare header canvas object
CCanvas canvasText;                     //--- Declare text canvas object
CCanvas canvasTextHighRes;              //--- Declare text high-res canvas object

//+------------------------------------------------------------------+
//| Canvas names                                                     |
//+------------------------------------------------------------------+
string canvasGraphName = "GraphCanvas"; //--- Set graph canvas name
string canvasStatsName = "StatsCanvas"; //--- Set stats canvas name
string canvasStatsHighResName = "StatsCanvasHighRes"; //--- Set stats high-res name
string canvasHeaderName = "HeaderCanvas"; //--- Set header canvas name
string canvasTextName = "TextCanvas";   //--- Set text canvas name
string canvasTextHighResName = "TextCanvasHighRes"; //--- Set text high-res name

//+------------------------------------------------------------------+
//| Inputs                                                           |
//+------------------------------------------------------------------+
sinput group "=== GENERAL CANVAS SETTINGS ==="
input int CanvasX                     = 30;                             // Canvas X Position
input int CanvasY                     = 50;                             // Canvas Y Position
input int CanvasWidth                 = 400;                            // Canvas Width
input int CanvasHeight                = 300;                            // Canvas Height
input bool EnableStatsPanel           = true;                           // Enable Stats Panel
input int PanelGap                    = 10;                             // Panel Gap
input bool EnableTextPanel            = true;                           // Enable Text Panel
input int TextPanelHeight             = 200;                            // Text Panel Height
input double TextBackgroundOpacityPercent = 85.0;                       // Text Background Opacity Percent

sinput group "=== HEADER CANVAS SETTINGS ==="
input color HeaderShadowColor         = clrDodgerBlue;                  // Header Shadow Color
input double HeaderShadowOpacityPercent = 90.0;                         // Header Shadow Opacity Percent
input int HeaderShadowDistance        = 0;                              // Header Shadow Distance
input int HeaderShadowBlurRadius      = 3;                              // Header Shadow Blur Radius

sinput group "=== GRAPH PANEL SETTINGS ==="
input int graphBars                   = 50;                             // Graph Bars
input color borderColor               = clrBlack;                      // Border Color
input color borderHoverColor          = clrRed;                         // Border Hover Color
input bool UseBackground              = true;                           // Use Background
input double FogOpacity               = 0.5;                            // Fog Opacity
input bool BlendFog                   = true;                           // Blend Fog

sinput group "=== STATS PANEL SETTINGS ==="
input int StatsFontSize               = 12;                             // Stats Font Size
input color StatsLabelColor           = clrDodgerBlue;                  // Stats Label Color
input color StatsValueColor           = clrWhite;                       // Stats Value Color
input color StatsHeaderColor          = clrDodgerBlue;                  // Stats Header Color
input int StatsHeaderFontSize         = 14;                             // Stats Header Font Size
input double BorderOpacityPercentReduction = 20.0;                      // Border Opacity Percent Reduction
input double BorderDarkenPercent      = 30.0;                           // Border Darken Percent
input double StatsHeaderBgOpacityPercent = 20.0;                        // Stats Header Bg Opacity Percent
input int StatsHeaderBgRadius         = 8;                              // Stats Header Bg Radius
input ENUM_BACKGROUND_MODE StatsBackgroundMode = GradientTwoColors;     // Stats Background Mode
input color TopColor                   = clrBlack;                      // Top Color
input color BottomColor                = clrRed;                        // Bottom Color
input double BackgroundOpacity         = 0.7;                           // Background Opacity

sinput group "=== TEXT PANEL SETTINGS ==="
input int TriangleRoundRadius         = 1;                              // Triangle Round Radius
input double TriangleBaseWidthPercent = 65.0;                           // Triangle Base Width Percent (of button size)
input double TriangleHeightPercent    = 70.0;                           // Triangle Height Percent (of base width)
input bool ShowUpDownButtons          = false;                          // Show Up/Down Buttons
input int TextFontSize = 17;                                            // Text Font Size
//+------------------------------------------------------------------+
//| Global variables                                                 |
//+------------------------------------------------------------------+
uint original_bg_pixels[];              //--- Store original background pixels
uint orig_w = 0, orig_h = 0;            //--- Store original background dimensions
uint bg_pixels_graph[];                 //--- Store graph background pixels
uint bg_pixels_stats[];                 //--- Store stats background pixels
uint bg_pixels_text[];                  //--- Store text background pixels

int currentCanvasX = CanvasX;           //--- Set current canvas X position
int currentCanvasY = CanvasY;           //--- Set current canvas Y position
int currentWidth = CanvasWidth;         //--- Set current canvas width
int currentHeight = CanvasHeight;       //--- Set current canvas height
bool panel_dragging = false;            //--- Track panel dragging state
int panel_drag_x = 0, panel_drag_y = 0; //--- Store panel drag coordinates
int panel_start_x = 0, panel_start_y = 0; //--- Store panel start coordinates
bool resizing = false;                  //--- Track resizing state
ENUM_RESIZE_MODE resize_mode = NONE;    //--- Set resize mode
ENUM_RESIZE_MODE hover_mode = NONE;     //--- Set hover mode
int resize_start_x = 0, resize_start_y = 0; //--- Store resize start coordinates
int start_width = 0, start_height = 0;  //--- Store start dimensions
const int resize_thickness = 5;         //--- Define resize thickness
const int min_width = 200;              //--- Define minimum width
const int min_height = 150;             //--- Define minimum height
int hover_mouse_local_x = 0;            //--- Store hover local X
int hover_mouse_local_y = 0;            //--- Store hover local Y
bool header_hovered = false;            //--- Track header hover state
bool minimize_hovered = false;          //--- Track minimize hover state
bool close_hovered = false;             //--- Track close hover state
bool theme_hovered = false;             //--- Track theme hover state
bool resize_hovered = false;            //--- Track resize hover state
int prev_mouse_state = 0;               //--- Store previous mouse state
int last_mouse_x = 0, last_mouse_y = 0; //--- Store last mouse coordinates
int header_height = 27;                 //--- Set header height
int gap_y = 7;                          //--- Set Y gap
int button_size = 25;                   //--- Set button size
int theme_x_offset = -75;               //--- Set theme X offset
int minimize_x_offset = -50;            //--- Set minimize X offset
int close_x_offset = -25;               //--- Set close X offset
bool panels_minimized = false;          //--- Track panels minimized state
color HeaderColor = C'60,60,60';        //--- Set header color
color HeaderHoverColor = clrRed;        //--- Set header hover color
color HeaderDragColor = clrMediumBlue;  //--- Set header drag color

bool is_dark_theme = true;              //--- Set dark theme state

color LightHeaderColor = clrSilver;     //--- Set light header color
color LightHeaderTextColor = clrBlack;  //--- Set light header text color
color LightStatsLabelColor = clrBlue;   //--- Set light stats label color
color LightStatsValueColor = clrBlack;  //--- Set light stats value color
color LightStatsHeaderColor = clrBlue;  //--- Set light stats header color
color LightBorderColor = clrBlack;      //--- Set light border color
color LightTopColor = clrGreen;         //--- Set light top color
color LightBottomColor = clrGold;       //--- Set light bottom color
color LightHeaderHoverColor = clrRed;   //--- Set light header hover color
color LightHeaderDragColor = clrMediumBlue; //--- Set light header drag color

bool graphCreated = false;              //--- Track graph created state
bool statsCreated = false;              //--- Track stats created state
bool textCreated = false;               //--- Track text created state

const int smoothness_factor = 10;  // Higher = smoother drag (e.g., 20 for more)

string text_usage_text =
   "\nCanvas Dashboard Usage Guide\n\n"
   "Welcome to the Canvas Dashboard – Your Interactive Tool for Real-Time Market Monitoring in MetaTrader 5!\n\n"
   "Enhance your trading experience with this dynamic dashboard that visualizes price data, account stats, and interactive controls. Designed for ease of use, it allows customization through dragging, resizing, theme switching, and more, while providing essential market insights at a glance.\n\n"
   "Key Features:\n"
   "- Price Graph Panel: Displays recent bar closes with a fog gradient fill, background image support, and resize indicators.\n"
   "- Stats Panel: Shows account balance, equity, and current bar OHLC values with customizable backgrounds (single color or gradient).\n"
   "- Header Controls: Drag to move the dashboard; buttons for theme toggle (dark/light), minimize/maximize panels, and close.\n"
   "- Text Panel: Scrollable guide (this panel) with hover-expand scrollbar, up/down buttons, and slider for navigation.\n"
   "- Interactivity: Hover for highlights/tooltips; resize via borders (bottom, right, corner); wheel scroll in text area.\n"
   "- Theme Support: Switch between dark and light modes for better visibility on different chart backgrounds.\n"
   "- Background Options: Enable images with fog overlay; blend modes for transparency.\n\n"
   "Usage Instructions:\n"
   "1. Move the Dashboard: Click and drag the header (excluding buttons) to reposition on the chart.\n"
   "2. Resize Panels: Hover near the graph's bottom/right edges; click and drag when the icon appears (arrows for direction).\n"
   "3. Toggle Theme: Click the '[' icon in the header to switch between dark and light modes.\n"
   "4. Minimize/Maximize: Click the 'r' or 'o' icon to hide/show panels (header remains visible).\n"
   "5. Navigate Text: Use the scrollbar (hovers to expand with buttons/slider); click up/down or drag slider; wheel scroll in body.\n"
   "6. Close Dashboard: Click the 'X' icon in the header to remove it from the chart.\n\n"
   "Important Notes:\n"
   "- Risk Disclaimer: This dashboard is for informational purposes only. Always verify data and trade responsibly.\n"
   "- Compatibility Check: Ensure chart settings allow mouse events; test on demo for interactions.\n"
   "- Optimization Tips: Adjust input parameters like graph bars, fonts, colors, and opacity for your setup.\n"
   "- Security Measures: No account modifications; use on trusted platforms.\n"
   "- Legal Notice: No guarantees of accuracy. Consult professionals as needed.\n\n"
   "Contact Methods:\n"
   "NB:\n"
   "********************************************\n"
   " >*** FOR SUPPORT, QUERIES, OR CUSTOMIZATIONS, REACH OUT IMMEDIATELY: ***<\n"
   " __________________________________________\n\n"
   "1. Email: mutiiriallan.forex@gmail.com (Primary Support Channel)\n"
   "2. Telegram Channel: @ForexAlgo-Trader (Updates & Community)\n"
   "3. Telegram Group: https://t.me/Forex_Algo_Trader (Direct Assistance & Discussions)\n\n"
   "********************************************\n\n"
   "Thank you for choosing our Canvas Dashboard solutions. Use wisely, monitor confidently, and elevate your trading journey! 🚀\n"; //--- Define text usage content

int text_scroll_pos = 0;                //--- Set text scroll position
int text_max_scroll = 0;                //--- Set text max scroll
int text_slider_height = 20;            //--- Set text slider height
bool text_movingStateSlider = false;    //--- Track text slider moving state
int text_mlbDownY_Slider = 0;           //--- Store text slider down Y
int text_mlbDown_YD_Slider = 0;         //--- Store text slider down YD
int text_total_height = 0;              //--- Set text total height
int text_visible_height = 0;            //--- Set text visible height
bool text_scroll_visible = false;       //--- Track text scroll visible state
bool text_mouse_in_body = false;        //--- Track text mouse in body state
bool prev_text_mouse_in_body = false;   //--- Track previous text mouse in body state
bool text_scroll_up_hovered = false;    //--- Track text scroll up hover state
bool text_scroll_down_hovered = false;  //--- Track text scroll down hover state
bool text_scroll_slider_hovered = false; //--- Track text scroll slider hover state
bool text_scroll_area_hovered = false;  //--- Track text scroll area hover state
const int TEXT_MAX_LINES = 100;         //--- Define text max lines
int text_adjustedLineHeight = 0;        //--- Set text adjusted line height

int text_scrollbar_full_width = 16;     //--- Set text scrollbar full width
int text_scrollbar_thin_width = 2;      //--- Set text scrollbar thin width
int text_track_width = 16;              //--- Set text track width
int text_scrollbar_margin = 5;          //--- Set text scrollbar margin
int text_button_size = 16;              //--- Set text button size
color text_leader_color_dark = C'45,45,45'; //--- Set dark text leader color
color text_leader_color_light = C'200,200,200'; //--- Set light text leader color
color text_button_bg_dark = C'60,60,60'; //--- Set dark text button bg
color text_button_bg_light = C'220,220,220'; //--- Set light text button bg
color text_button_bg_hover_dark = C'70,70,70'; //--- Set dark text button bg hover
color text_button_bg_hover_light = C'180,180,180'; //--- Set light text button bg hover
color text_arrow_color_dark = C'150,150,150'; //--- Set dark text arrow color
color text_arrow_color_light = C'50,50,50'; //--- Set light text arrow color
color text_arrow_color_disabled_dark = C'80,80,80'; //--- Set dark disabled arrow color
color text_arrow_color_disabled_light = C'150,150,150'; //--- Set light disabled arrow color
color text_arrow_color_hover_dark = C'100,100,100'; //--- Set dark hover arrow color
color text_arrow_color_hover_light = C'0,0,0'; //--- Set light hover arrow color
color text_slider_bg_dark = C'80,80,80'; //--- Set dark slider bg
color text_slider_bg_light = C'150,150,150'; //--- Set light slider bg
color text_slider_bg_hover_dark = C'100,100,100'; //--- Set dark slider bg hover
color text_slider_bg_hover_light = C'100,100,100'; //--- Set light slider bg hover

color text_bg_light = clrWhite;         //--- Set light text bg
color text_bg_dark = clrBlack;          //--- Set dark text bg
color text_base_light = clrBlack;       //--- Set light text base
color text_base_dark = clrWhite;        //--- Set dark text base

const int supersamplingFactor = 4;      // Supersampling for smooth rounds

//+------------------------------------------------------------------+
//| Convert degrees to radians                                       |
//+------------------------------------------------------------------+
double DegreesToRadians(double degrees) { // Convert degrees input to radians
   return((M_PI * degrees) / 180.0);    //--- Compute radians using pi formula
}

//+------------------------------------------------------------------+
//| Initialize expert                                                |
//+------------------------------------------------------------------+
int OnInit() {                          // Initialize expert advisor
   currentWidth = CanvasWidth;          //--- Assign initial width
   currentHeight = CanvasHeight;        //--- Assign initial height
   currentCanvasX = CanvasX;            //--- Assign initial X position
   currentCanvasY = CanvasY;            //--- Assign initial Y position

   if (UseBackground) {                 //--- Check background usage
      if (ResourceReadImage("::1. Transparent MT5 bmp image.bmp", original_bg_pixels, orig_w, orig_h) && orig_w > 0 && orig_h > 0) { //--- Load image if valid
         ArrayCopy(bg_pixels_graph, original_bg_pixels); //--- Copy to graph pixels
         ScaleImage(bg_pixels_graph, (int)orig_w, (int)orig_h, currentWidth, currentHeight); //--- Scale graph image
         if (EnableStatsPanel) {           //--- Check stats panel enabled
            ArrayCopy(bg_pixels_stats, original_bg_pixels); //--- Copy to stats pixels
            ScaleImage(bg_pixels_stats, (int)orig_w, (int)orig_h, currentWidth / 2, currentHeight); //--- Scale stats image
         }
      } else {                          //--- Handle load failure
         Print("Failed to load background image from ::1. Transparent MT5 bmp image.bmp"); //--- Log error message
      }
   }

   int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra space
   int inner_header_width = currentWidth + (EnableStatsPanel ? PanelGap + currentWidth / 2 : 0); //--- Compute inner width
   int header_canvas_width = inner_header_width + 2 * extra; //--- Compute header width
   int header_canvas_height = header_height + 2 * extra; //--- Compute header height

   if (!canvasHeader.CreateBitmapLabel(0, 0, canvasHeaderName, currentCanvasX - extra, currentCanvasY - extra, header_canvas_width, header_canvas_height, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create header canvas
      Print("Failed to create Header Canvas"); //--- Log creation failure
      return(INIT_FAILED);              //--- Return initialization failure
   }

   if (!canvasGraph.CreateBitmapLabel(0, 0, canvasGraphName, currentCanvasX, currentCanvasY + header_height + gap_y, currentWidth, currentHeight, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create graph canvas
      Print("Failed to create Graph Canvas"); //--- Log creation failure
      return(INIT_FAILED);              //--- Return initialization failure
   }
   graphCreated = true;                 //--- Set graph created flag

   if (EnableStatsPanel) {              //--- Check stats panel enabled
      int statsX = currentCanvasX + currentWidth + PanelGap; //--- Compute stats X
      if (!canvasStats.CreateBitmapLabel(0, 0, canvasStatsName, statsX, currentCanvasY + header_height + gap_y, currentWidth / 2, currentHeight, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create stats canvas
         Print("Failed to create Stats Canvas"); //--- Log creation failure
      }
      if (!canvasStatsHighRes.Create(canvasStatsHighResName, (currentWidth / 2) * supersamplingFactor, currentHeight * supersamplingFactor, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create stats high-res
         Print("Failed to create Stats High-Res Canvas"); //--- Log creation failure
      }
      statsCreated = true;              //--- Set stats created flag
   }

   if (EnableTextPanel) {               //--- Check text panel enabled
      int textY = currentCanvasY + header_height + gap_y + currentHeight + PanelGap; //--- Compute text Y
      int text_width = inner_header_width; //--- Set text width
      int text_height = TextPanelHeight; //--- Set text height
      if (!canvasText.CreateBitmapLabel(0, 0, canvasTextName, currentCanvasX, textY, text_width, text_height, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create text canvas
         Print("Failed to create Text Canvas"); //--- Log creation failure
      }
      if (!canvasTextHighRes.Create(canvasTextHighResName, text_width * supersamplingFactor, text_height * supersamplingFactor, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Create text high-res
         Print("Failed to create Text High-Res Canvas"); //--- Log creation failure
      }
      textCreated = true;               //--- Set text created flag
   }

   DrawHeaderOnCanvas();                //--- Draw header
   UpdateGraphOnCanvas();               //--- Update graph
   if (EnableStatsPanel) UpdateStatsOnCanvas(); //--- Update stats if enabled
   if (EnableTextPanel) UpdateTextOnCanvas(); //--- Update text if enabled

   ChartSetInteger(0, CHART_EVENT_MOUSE_MOVE, true); //--- Enable mouse move events
   ChartSetInteger(0, CHART_EVENT_MOUSE_WHEEL, true); //--- Enable mouse wheel events
   ChartRedraw();                       //--- Redraw chart
   return(INIT_SUCCEEDED);              //--- Return initialization success
}

//+------------------------------------------------------------------+
//| Deinitialize expert                                              |
//+------------------------------------------------------------------+
void OnDeinit(const int reason) {       // Deinitialize expert advisor
   canvasHeader.Destroy();              //--- Destroy header canvas
   if (graphCreated) canvasGraph.Destroy(); //--- Destroy graph if created
   if (statsCreated) {
      canvasStats.Destroy();            //--- Destroy stats if created
      canvasStatsHighRes.Destroy();     //--- Destroy stats high-res
   }
   if (textCreated) {
      canvasText.Destroy();             //--- Destroy text if created
      canvasTextHighRes.Destroy();      //--- Destroy text high-res
   }
   ChartRedraw();                       //--- Redraw chart
}

//+------------------------------------------------------------------+
//| Get header color                                                 |
//+------------------------------------------------------------------+
color GetHeaderColor() {                // Retrieve header color
   return is_dark_theme ? HeaderColor : LightHeaderColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get header hover color                                           |
//+------------------------------------------------------------------+
color GetHeaderHoverColor() {           // Retrieve header hover color
   return is_dark_theme ? HeaderHoverColor : LightHeaderHoverColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get header drag color                                            |
//+------------------------------------------------------------------+
color GetHeaderDragColor() {            // Retrieve header drag color
   return is_dark_theme ? HeaderDragColor : LightHeaderDragColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get stats label color                                            |
//+------------------------------------------------------------------+
color GetStatsLabelColor() {            // Retrieve stats label color
   return is_dark_theme ? StatsLabelColor : LightStatsLabelColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get stats value color                                            |
//+------------------------------------------------------------------+
color GetStatsValueColor() {            // Retrieve stats value color
   return is_dark_theme ? StatsValueColor : LightStatsValueColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get stats header color                                           |
//+------------------------------------------------------------------+
color GetStatsHeaderColor() {           // Retrieve stats header color
   return is_dark_theme ? StatsHeaderColor : LightStatsHeaderColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get border color                                                 |
//+------------------------------------------------------------------+
color GetBorderColor() {                // Retrieve border color
   return is_dark_theme ? borderColor : LightBorderColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get top color                                                    |
//+------------------------------------------------------------------+
color GetTopColor() {                   // Retrieve top color
   return is_dark_theme ? TopColor : LightTopColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get bottom color                                                 |
//+------------------------------------------------------------------+
color GetBottomColor() {                // Retrieve bottom color
   return is_dark_theme ? BottomColor : LightBottomColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get header text color                                            |
//+------------------------------------------------------------------+
color GetHeaderTextColor() {            // Retrieve header text color
   return is_dark_theme ? clrWhite : LightHeaderTextColor; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Get icon color                                                   |
//+------------------------------------------------------------------+
color GetIconColor(bool is_drag) {      // Retrieve icon color
   return is_drag ? GetHeaderDragColor() : GetHeaderHoverColor(); //--- Return based on drag state
}

//+------------------------------------------------------------------+
//| Get stats header bg color                                        |
//+------------------------------------------------------------------+
color GetStatsHeaderBgColor() {         // Retrieve stats header bg color
   return is_dark_theme ? C'50,50,50' : C'200,200,200'; //--- Return based on theme
}

//+------------------------------------------------------------------+
//| Draw header on canvas                                            |
//+------------------------------------------------------------------+
void DrawHeaderOnCanvas() {             // Render header elements
   canvasHeader.Erase(0);               //--- Clear header canvas

   int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra space
   int inner_header_width = canvasHeader.Width() - 2 * extra; //--- Compute inner width

   int header_left = extra;             //--- Set header left
   int header_top = extra;              //--- Set header top
   int header_right = header_left + inner_header_width - 1; //--- Set header right
   int header_bottom = header_top + header_height - 1; //--- Set header bottom

   if (HeaderShadowBlurRadius > 0 || HeaderShadowDistance > 0) { //--- Check shadow settings
      int offset_x = HeaderShadowDistance; //--- Set X offset
      int offset_y = HeaderShadowDistance; //--- Set Y offset
      int blur = HeaderShadowBlurRadius; //--- Set blur radius

      for(int layer = blur; layer >= 0; layer--) { //--- Loop through layers
         double factor = (double)layer / (blur + 1.0); //--- Compute factor
         uchar alpha = (uchar)(255 * (HeaderShadowOpacityPercent / 100.0) * (1.0 - factor)); //--- Compute alpha
         uint argb_shadow = ColorToARGB(HeaderShadowColor, alpha); //--- Convert to ARGB

         int s_left = header_left + offset_x - layer; //--- Set shadow left
         int s_top = header_top + offset_y - layer; //--- Set shadow top
         int s_right = header_right + offset_x + layer; //--- Set shadow right
         int s_bottom = header_bottom + offset_y + layer; //--- Set shadow bottom

         int s_width = s_right - s_left + 1; //--- Compute shadow width
         int s_height = s_bottom - s_top + 1; //--- Compute shadow height
         int s_radius = layer;             //--- Set shadow radius

         if (s_width > 0 && s_height > 0) { //--- Check valid dimensions
            FillRoundedRectangle(canvasHeader, s_left, s_top, s_width, s_height, s_radius, argb_shadow); //--- Fill shadow rectangle
         }
      }
   }

   color header_bg = panel_dragging ? GetHeaderDragColor() : (header_hovered ? GetHeaderHoverColor() : GetHeaderColor()); //--- Determine bg color
   uint argb_bg = ColorToARGB(header_bg, 255); //--- Convert to ARGB
   canvasHeader.FillRectangle(header_left, header_top, header_right, header_bottom, argb_bg); //--- Fill header rectangle

   uint argbBorder = ColorToARGB(GetBorderColor(), 255); //--- Convert border to ARGB
   canvasHeader.Line(header_left, header_top, header_right, header_top, argbBorder); //--- Draw top border
   canvasHeader.Line(header_right, header_top, header_right, header_bottom, argbBorder); //--- Draw right border
   canvasHeader.Line(header_right, header_bottom, header_left, header_bottom, argbBorder); //--- Draw bottom border
   canvasHeader.Line(header_left, header_bottom, header_left, header_top, argbBorder); //--- Draw left border

   canvasHeader.FontSet("Arial Bold", 15); //--- Set font for title
   uint argbText = ColorToARGB(GetHeaderTextColor(), 255); //--- Convert text to ARGB
   canvasHeader.TextOut(header_left + 10, header_top + (header_height - 15) / 2, "Price Dashboard", argbText, TA_LEFT); //--- Draw title text

   int theme_x = header_left + inner_header_width + theme_x_offset; //--- Compute theme X
   string theme_symbol = CharToString((uchar)91); //--- Set theme symbol
   color theme_color = theme_hovered ? clrYellow : GetHeaderTextColor(); //--- Determine theme color
   canvasHeader.FontSet("Wingdings", 22); //--- Set font for theme
   uint argb_theme = ColorToARGB(theme_color, 255); //--- Convert to ARGB
   canvasHeader.TextOut(theme_x, header_top + (header_height - 22) / 2, theme_symbol, argb_theme, TA_CENTER); //--- Draw theme symbol

   int min_x = header_left + inner_header_width + minimize_x_offset; //--- Compute minimize X
   string min_symbol = panels_minimized ? CharToString((uchar)111) : CharToString((uchar)114); //--- Set minimize symbol
   color min_color = minimize_hovered ? clrYellow : GetHeaderTextColor(); //--- Determine minimize color
   canvasHeader.FontSet("Wingdings", 22); //--- Set font for minimize
   uint argb_min = ColorToARGB(min_color, 255); //--- Convert to ARGB
   canvasHeader.TextOut(min_x, header_top + (header_height - 22) / 2, min_symbol, argb_min, TA_CENTER); //--- Draw minimize symbol

   int close_x = header_left + inner_header_width + close_x_offset; //--- Compute close X
   string close_symbol = CharToString((uchar)114); //--- Set close symbol
   color close_color = close_hovered ? clrRed : GetHeaderTextColor(); //--- Determine close color
   canvasHeader.FontSet("Webdings", 22); //--- Set font for close
   uint argb_close = ColorToARGB(close_color, 255); //--- Convert to ARGB
   canvasHeader.TextOut(close_x, header_top + (header_height - 22) / 2, close_symbol, argb_close, TA_CENTER); //--- Draw close symbol

   canvasHeader.Update();               //--- Update header canvas
}

//+------------------------------------------------------------------+
//| Check mouse over header                                          |
//+------------------------------------------------------------------+
bool IsMouseOverHeader(int mouse_x, int mouse_y) { // Detect header hover
   int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra space
   int inner_header_width = currentWidth + (EnableStatsPanel && !panels_minimized ? PanelGap + currentWidth / 2 : 0); //--- Compute inner width
   int header_canvas_x = currentCanvasX - extra; //--- Set canvas X
   int header_canvas_y = currentCanvasY - extra; //--- Set canvas Y
   int header_canvas_w = inner_header_width + 2 * extra; //--- Set canvas width
   int header_canvas_h = header_height + 2 * extra; //--- Set canvas height

   if (mouse_x < header_canvas_x || mouse_x > header_canvas_x + header_canvas_w || mouse_y < header_canvas_y || mouse_y > header_canvas_y + header_canvas_h) return false; //--- Return false if outside

   int theme_left = header_canvas_x + extra + inner_header_width + theme_x_offset - button_size / 2; //--- Set theme left
   int theme_right = theme_left + button_size; //--- Set theme right
   int theme_top = header_canvas_y + extra;  //--- Set theme top
   int theme_bottom = theme_top + header_height; //--- Set theme bottom
   if (mouse_x >= theme_left && mouse_x <= theme_right && mouse_y >= theme_top && mouse_y <= theme_bottom) return false; //--- Return false if over theme

   int min_left = header_canvas_x + extra + inner_header_width + minimize_x_offset - button_size / 2; //--- Set minimize left
   int min_right = min_left + button_size;   //--- Set minimize right
   int min_top = header_canvas_y + extra;    //--- Set minimize top
   int min_bottom = min_top + header_height; //--- Set minimize bottom
   if (mouse_x >= min_left && mouse_x <= min_right && mouse_y >= min_top && mouse_y <= min_bottom) return false; //--- Return false if over minimize

   int close_left = header_canvas_x + extra + inner_header_width + close_x_offset - button_size / 2; //--- Set close left
   int close_right = close_left + button_size; //--- Set close right
   int close_top = header_canvas_y + extra;  //--- Set close top
   int close_bottom = close_top + header_height; //--- Set close bottom
   if (mouse_x >= close_left && mouse_x <= close_right && mouse_y >= close_top && mouse_y <= close_bottom) return false; //--- Return false if over close

   return true;                         //--- Return true if over header
}

//+------------------------------------------------------------------+
//| Check mouse over theme                                           |
//+------------------------------------------------------------------+
bool IsMouseOverTheme(int mouse_x, int mouse_y) { // Detect theme hover
   int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra space
   int inner_header_width = currentWidth + (EnableStatsPanel && !panels_minimized ? PanelGap + currentWidth / 2 : 0); //--- Compute inner width
   int header_canvas_x = currentCanvasX - extra; //--- Set canvas X
   int header_canvas_y = currentCanvasY - extra; //--- Set canvas Y
   int theme_left = header_canvas_x + extra + inner_header_width + theme_x_offset - button_size / 2; //--- Set theme left
   int theme_right = theme_left + button_size; //--- Set theme right
   int theme_top = header_canvas_y + extra;  //--- Set theme top
   int theme_bottom = theme_top + header_height; //--- Set theme bottom
   return (mouse_x >= theme_left && mouse_x <= theme_right && mouse_y >= theme_top && mouse_y <= theme_bottom); //--- Return hover state
}

//+------------------------------------------------------------------+
//| Check mouse over minimize                                        |
//+------------------------------------------------------------------+
bool IsMouseOverMinimize(int mouse_x, int mouse_y) { // Detect minimize hover
   int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra space
   int inner_header_width = currentWidth + (EnableStatsPanel && !panels_minimized ? PanelGap + currentWidth / 2 : 0); //--- Compute inner width
   int header_canvas_x = currentCanvasX - extra; //--- Set canvas X
   int header_canvas_y = currentCanvasY - extra; //--- Set canvas Y
   int min_left = header_canvas_x + extra + inner_header_width + minimize_x_offset - button_size / 2; //--- Set minimize left
   int min_right = min_left + button_size;   //--- Set minimize right
   int min_top = header_canvas_y + extra;    //--- Set minimize top
   int min_bottom = min_top + header_height; //--- Set minimize bottom
   return (mouse_x >= min_left && mouse_x <= min_right && mouse_y >= min_top && mouse_y <= min_bottom); //--- Return hover state
}

//+------------------------------------------------------------------+
//| Check mouse over close                                           |
//+------------------------------------------------------------------+
bool IsMouseOverClose(int mouse_x, int mouse_y) { // Detect close hover
   int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra space
   int inner_header_width = currentWidth + (EnableStatsPanel && !panels_minimized ? PanelGap + currentWidth / 2 : 0); //--- Compute inner width
   int header_canvas_x = currentCanvasX - extra; //--- Set canvas X
   int header_canvas_y = currentCanvasY - extra; //--- Set canvas Y
   int close_left = header_canvas_x + extra + inner_header_width + close_x_offset - button_size / 2; //--- Set close left
   int close_right = close_left + button_size; //--- Set close right
   int close_top = header_canvas_y + extra;  //--- Set close top
   int close_bottom = close_top + header_height; //--- Set close bottom
   return (mouse_x >= close_left && mouse_x <= close_right && mouse_y >= close_top && mouse_y <= close_bottom); //--- Return hover state
}

//+------------------------------------------------------------------+
//| Check mouse over resize                                          |
//+------------------------------------------------------------------+
bool IsMouseOverResize(int mx, int my, ENUM_RESIZE_MODE &rmode) { // Detect resize hover
   if (panels_minimized) return false;  //--- Return false if minimized
   int graph_x = currentCanvasX;        //--- Set graph X
   int graph_y = currentCanvasY + header_height + gap_y; //--- Set graph Y
   int graph_right = graph_x + currentWidth; //--- Set graph right
   int graph_bottom = graph_y + currentHeight; //--- Set graph bottom
   bool over_right = (mx >= graph_right - resize_thickness && mx <= graph_right + resize_thickness) && (my >= graph_y && my <= graph_bottom); //--- Check right hover
   bool over_bottom = (my >= graph_bottom - resize_thickness && my <= graph_bottom + resize_thickness) && (mx >= graph_x && mx <= graph_right); //--- Check bottom hover
   if (over_bottom && over_right) {     //--- Check bottom-right
      rmode = BOTTOM_RIGHT;             //--- Set mode to bottom-right
      return true;                      //--- Return true
   } else if (over_bottom) {            //--- Check bottom
      rmode = BOTTOM;                   //--- Set mode to bottom
      return true;                      //--- Return true
   } else if (over_right) {             //--- Check right
      rmode = RIGHT;                    //--- Set mode to right
      return true;                      //--- Return true
   }
   return false;                        //--- Return false
}

//+------------------------------------------------------------------+
//| Toggle theme                                                     |
//+------------------------------------------------------------------+
void ToggleTheme() {                    // Switch theme mode
   is_dark_theme = !is_dark_theme;      //--- Invert theme state
   Print("Switched to ", (is_dark_theme ? "Dark" : "Light"), " theme"); //--- Log theme switch
   DrawHeaderOnCanvas();                //--- Redraw header
   UpdateGraphOnCanvas();               //--- Update graph
   if (EnableStatsPanel) UpdateStatsOnCanvas(); //--- Update stats if enabled
   if (EnableTextPanel) UpdateTextOnCanvas(); //--- Update text if enabled
   ChartRedraw();                       //--- Redraw chart
}

//+------------------------------------------------------------------+
//| Toggle minimize                                                  |
//+------------------------------------------------------------------+
void ToggleMinimize() {                // Switch minimize state
   panels_minimized = !panels_minimized; //--- Invert minimized state
   if (panels_minimized) {              //--- Check minimized
      canvasGraph.Destroy();            //--- Destroy graph canvas
      graphCreated = false;             //--- Reset graph flag
      if (EnableStatsPanel) {           //--- Check stats enabled
         canvasStats.Destroy();         //--- Destroy stats canvas
         canvasStatsHighRes.Destroy();  //--- Destroy stats high-res
         statsCreated = false;          //--- Reset stats flag
      }
      if (EnableTextPanel) {            //--- Check text enabled
         canvasText.Destroy();          //--- Destroy text canvas
         canvasTextHighRes.Destroy();   //--- Destroy text high-res
         textCreated = false;           //--- Reset text flag
      }
   } else {                             //--- Handle maximized
      if (!canvasGraph.CreateBitmapLabel(0, 0, canvasGraphName, currentCanvasX, currentCanvasY + header_height + gap_y, currentWidth, currentHeight, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Recreate graph
         Print("Failed to recreate Graph Canvas"); //--- Log failure
      }
      graphCreated = true;              //--- Set graph flag
      UpdateGraphOnCanvas();            //--- Update graph
      if (EnableStatsPanel) {           //--- Check stats enabled
         int statsX = currentCanvasX + currentWidth + PanelGap; //--- Compute stats X
         if (!canvasStats.CreateBitmapLabel(0, 0, canvasStatsName, statsX, currentCanvasY + header_height + gap_y, currentWidth / 2, currentHeight, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Recreate stats
            Print("Failed to recreate Stats Canvas"); //--- Log failure
         }
         if (!canvasStatsHighRes.Create(canvasStatsHighResName, (currentWidth / 2) * supersamplingFactor, currentHeight * supersamplingFactor, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Recreate stats high-res
            Print("Failed to recreate Stats High-Res Canvas"); //--- Log failure
         }
         statsCreated = true;           //--- Set stats flag
         UpdateStatsOnCanvas();         //--- Update stats
      }
      if (EnableTextPanel) {            //--- Check text enabled
         int textY = currentCanvasY + header_height + gap_y + currentHeight + PanelGap; //--- Compute text Y
         int inner_header_width = currentWidth + (EnableStatsPanel ? PanelGap + currentWidth / 2 : 0); //--- Compute width
         int text_width = inner_header_width; //--- Set text width
         int text_height = TextPanelHeight; //--- Set text height
         if (!canvasText.CreateBitmapLabel(0, 0, canvasTextName, currentCanvasX, textY, text_width, text_height, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Recreate text
            Print("Failed to recreate Text Canvas"); //--- Log failure
         }
         if (!canvasTextHighRes.Create(canvasTextHighResName, text_width * supersamplingFactor, text_height * supersamplingFactor, COLOR_FORMAT_ARGB_NORMALIZE)) { //--- Recreate text high-res
            Print("Failed to recreate Text High-Res Canvas"); //--- Log failure
         }
         textCreated = true;            //--- Set text flag
         UpdateTextOnCanvas();          //--- Update text
      }
   }

   int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra space
   int inner_header_width = currentWidth + (EnableStatsPanel && !panels_minimized ? PanelGap + currentWidth / 2 : 0); //--- Compute inner width
   int header_canvas_width = inner_header_width + 2 * extra; //--- Compute header width
   canvasHeader.Resize(header_canvas_width, header_height + 2 * extra); //--- Resize header
   ObjectSetInteger(0, canvasHeaderName, OBJPROP_XSIZE, header_canvas_width); //--- Set X size
   ObjectSetInteger(0, canvasHeaderName, OBJPROP_YSIZE, header_height + 2 * extra); //--- Set Y size
   DrawHeaderOnCanvas();                //--- Redraw header
   canvasHeader.Update();               //--- Update header
   ChartRedraw();                       //--- Redraw chart
}

//+------------------------------------------------------------------+
//| Close dashboard                                                  |
//+------------------------------------------------------------------+
void CloseDashboard() {                 // Shut down dashboard
   canvasHeader.Destroy();              //--- Destroy header
   canvasGraph.Destroy();               //--- Destroy graph
   if (EnableStatsPanel) {
      canvasStats.Destroy();            //--- Destroy stats if enabled
      canvasStatsHighRes.Destroy();     //--- Destroy stats high-res
   }
   if (EnableTextPanel) {
      canvasText.Destroy();             //--- Destroy text if enabled
      canvasTextHighRes.Destroy();      //--- Destroy text high-res
   }
   ChartRedraw();                       //--- Redraw chart
}

//+------------------------------------------------------------------+
//| Update graph on canvas                                           |
//+------------------------------------------------------------------+
void UpdateGraphOnCanvas() {            // Render graph elements
   canvasGraph.Erase(0);                //--- Clear graph canvas

   if (UseBackground && ArraySize(bg_pixels_graph) == currentWidth * currentHeight) { //--- Check background valid
      for (int y = 0; y < currentHeight; y++) { //--- Loop Y pixels
         for (int x = 0; x < currentWidth; x++) { //--- Loop X pixels
            canvasGraph.PixelSet(x, y, bg_pixels_graph[y * currentWidth + x]); //--- Set pixel
         }
      }
   }

   uint argbBorder = ColorToARGB(GetBorderColor(), 255); //--- Convert border to ARGB
   canvasGraph.Line(0, 0, currentWidth - 1, 0, argbBorder); //--- Draw top border
   canvasGraph.Line(currentWidth - 1, 0, currentWidth - 1, currentHeight - 1, argbBorder); //--- Draw right border
   canvasGraph.Line(currentWidth - 1, currentHeight - 1, 0, currentHeight - 1, argbBorder); //--- Draw bottom border
   canvasGraph.Line(0, currentHeight - 1, 0, 0, argbBorder); //--- Draw left border
   canvasGraph.Line(1, 1, currentWidth - 2, 1, argbBorder); //--- Draw inner top
   canvasGraph.Line(currentWidth - 2, 1, currentWidth - 2, currentHeight - 2, argbBorder); //--- Draw inner right
   canvasGraph.Line(currentWidth - 2, currentHeight - 2, 1, currentHeight - 2, argbBorder); //--- Draw inner bottom
   canvasGraph.Line(1, currentHeight - 2, 1, 1, argbBorder); //--- Draw inner left

   double closePrices[];                //--- Declare close prices array
   ArrayResize(closePrices, graphBars); //--- Resize prices array
   if (CopyClose(_Symbol, _Period, 0, graphBars, closePrices) != graphBars) { //--- Copy closes
      Print("Failed to copy close prices"); //--- Log copy failure
      return;                           //--- Exit function
   }

   datetime timeArr[];                  //--- Declare times array
   ArrayResize(timeArr, graphBars);     //--- Resize times array
   if (CopyTime(_Symbol, _Period, 0, graphBars, timeArr) != graphBars) { //--- Copy times
      Print("Failed to copy times");    //--- Log copy failure
      return;                           //--- Exit function
   }

   double minPrice = closePrices[0];    //--- Set initial min price
   double maxPrice = closePrices[0];    //--- Set initial max price
   for (int i = 1; i < graphBars; i++) { //--- Loop prices
      if (closePrices[i] < minPrice) minPrice = closePrices[i]; //--- Update min
      if (closePrices[i] > maxPrice) maxPrice = closePrices[i]; //--- Update max
   }
   double priceRange = maxPrice - minPrice; //--- Compute range
   if (priceRange == 0) priceRange = _Point; //--- Adjust zero range

   int graphLeft = 2;                   //--- Set graph left
   int graphRight = currentWidth - 3;   //--- Set graph right
   double graphWidth_d = graphRight - graphLeft; //--- Compute width double
   int graphHeight = currentHeight - 4; //--- Set graph height
   int bottomY = 2 + graphHeight;       //--- Set bottom Y

   int x_pos[];                         //--- Declare X positions
   int y_pos[];                         //--- Declare Y positions
   ArrayResize(x_pos, graphBars);       //--- Resize X array
   ArrayResize(y_pos, graphBars);       //--- Resize Y array
   for (int i = 0; i < graphBars; i++) { //--- Loop bars
      double norm = (graphBars > 1) ? (double)i / (graphBars - 1) : 0.0; //--- Normalize index
      x_pos[i] = graphLeft + (int)(norm * graphWidth_d + 0.5); //--- Compute X pos
      double price = closePrices[graphBars - 1 - i]; //--- Get price
      y_pos[i] = 2 + (int)(graphHeight * (maxPrice - price) / priceRange + 0.5); //--- Compute Y pos
   }

   color lineColor = clrBlue;           //--- Set line color
   uint argbLine = ColorToARGB(lineColor, 255); //--- Convert to ARGB
   for (int i = 0; i < graphBars - 1; i++) { //--- Loop segments
      int x1 = (currentWidth - 1) - x_pos[i]; //--- Set X1 flipped
      int y1 = y_pos[i];                //--- Set Y1
      int x2 = (currentWidth - 1) - x_pos[i + 1]; //--- Set X2 flipped
      int y2 = y_pos[i + 1];            //--- Set Y2
      canvasGraph.LineAA(x1, y1, x2, y2, argbLine); //--- Draw anti-aliased line
   }

   int min_flipped_x = (currentWidth - 1) - graphRight; //--- Set min flipped X
   int max_flipped_x = (currentWidth - 1) - graphLeft; //--- Set max flipped X
   for (int colX = min_flipped_x; colX <= max_flipped_x; colX++) { //--- Loop columns
      int logical_colX = (currentWidth - 1) - colX; //--- Compute logical X
      int seg = -1;                     //--- Initialize segment
      for (int j = 0; j < graphBars - 1; j++) { //--- Loop segments
         if (x_pos[j] <= logical_colX && logical_colX <= x_pos[j + 1]) { //--- Check in segment
            seg = j;                    //--- Set segment
            break;                      //--- Exit loop
         }
      }
      if (seg == -1) continue;          //--- Skip if no segment

      double dx = x_pos[seg + 1] - x_pos[seg]; //--- Compute delta X
      double t = (dx > 0) ? (logical_colX - x_pos[seg]) / dx : 0.0; //--- Compute t
      double interpY = y_pos[seg] + t * (y_pos[seg + 1] - y_pos[seg]); //--- Interpolate Y
      int topY = (int)(interpY + 0.5);  //--- Set top Y

      for (int fillY = topY; fillY < bottomY; fillY++) { //--- Loop fill Y
         double fadeFactor = (double)(bottomY - fillY) / (bottomY - topY); //--- Compute fade
         uchar alpha = (uchar)(255 * fadeFactor * FogOpacity); //--- Compute alpha
         uint argbFill = ColorToARGB(lineColor, alpha); //--- Convert fill to ARGB
         if (BlendFog) {                   //--- Check blend fog
            uint currentPixel = canvasGraph.PixelGet(colX, fillY); //--- Get current pixel
            uint blendedPixel = BlendPixels(currentPixel, argbFill); //--- Blend pixels
            canvasGraph.PixelSet(colX, fillY, blendedPixel); //--- Set blended pixel
         } else {                          //--- Handle no blend
            canvasGraph.PixelSet(colX, fillY, argbFill); //--- Set fill pixel
         }
      }
   }

   canvasGraph.FontSet("Arial", 12);    //--- Set font for title
   uint argbText = ColorToARGB(is_dark_theme ? clrBlack : clrGray, 255); //--- Convert text to ARGB
   canvasGraph.TextOut(currentWidth / 2, 10, "Price Graph (" + _Symbol + ")", argbText, TA_CENTER); //--- Draw title

   canvasGraph.FontSet("Arial", 12);    //--- Set font for times

   string newTime = TimeToString(timeArr[0], TIME_DATE | TIME_MINUTES); //--- Format new time
   string oldTime = TimeToString(timeArr[graphBars - 1], TIME_DATE | TIME_MINUTES); //--- Format old time
   canvasGraph.TextOut(10, currentHeight - 15, newTime, argbText, TA_LEFT); //--- Draw new time
   canvasGraph.TextOut(currentWidth - 10, currentHeight - 15, oldTime, argbText, TA_RIGHT); //--- Draw old time

   if (resize_hovered || resizing) {    //--- Check resize state
      ENUM_RESIZE_MODE active_mode = resizing ? resize_mode : hover_mode; //--- Get active mode
      if (active_mode == NONE) {        //--- Check none mode
         canvasGraph.Update();          //--- Update canvas
         return;                        //--- Exit function
      }

      string icon_font = "Wingdings 3"; //--- Set icon font
      int icon_size = 25;               //--- Set icon size
      uchar icon_code;                  //--- Declare icon code
      int angle = 0;                    //--- Set initial angle
      switch (active_mode) {            //--- Switch mode
      case BOTTOM:                      //--- Handle bottom
         icon_code = (uchar)'2';        //--- Set code
         angle = 0;                     //--- Set angle
         break;                         //--- Exit case
      case RIGHT:                       //--- Handle right
         icon_code = (uchar)'1';        //--- Set code
         angle = 0;                     //--- Set angle
         break;                         //--- Exit case
      case BOTTOM_RIGHT:                //--- Handle bottom-right
         icon_code = (uchar)'2';        //--- Set code
         angle = 450;                   //--- Set angle
         break;                         //--- Exit case
      default:                          //--- Handle default
         canvasGraph.Update();          //--- Update canvas
         return;                        //--- Exit function
      }
      string icon_symbol = CharToString(icon_code); //--- Convert to string

      color icon_color = GetIconColor(resizing); //--- Get icon color
      uint argb_icon = ColorToARGB(icon_color, 255); //--- Convert to ARGB

      canvasGraph.FontSet(icon_font, icon_size); //--- Set icon font
      canvasGraph.FontAngleSet(angle);  //--- Set font angle

      int icon_x = 0;                   //--- Set initial X
      int icon_y = 0;                   //--- Set initial Y

      switch (active_mode) {            //--- Switch mode for position
      case BOTTOM:                      //--- Handle bottom
         icon_x = MathMax(0, MathMin(hover_mouse_local_x - (icon_size / 2), currentWidth - icon_size)); //--- Compute X
         icon_y = currentHeight - icon_size - 2; //--- Compute Y
         break;                         //--- Exit case
      case RIGHT:                       //--- Handle right
         icon_y = MathMax(0, MathMin(hover_mouse_local_y - (icon_size / 2), currentHeight - icon_size)); //--- Compute Y
         icon_x = currentWidth - icon_size - 2; //--- Compute X
         break;                         //--- Exit case
      case BOTTOM_RIGHT:                //--- Handle bottom-right
         icon_x = currentWidth - icon_size - 10; //--- Compute X
         icon_y = currentHeight - icon_size; //--- Compute Y
         break;                         //--- Exit case
      default:                          //--- Handle default
         break;                         //--- Exit case
      }

      canvasGraph.TextOut(icon_x, icon_y, icon_symbol, argb_icon, TA_LEFT | TA_TOP); //--- Draw icon
      canvasGraph.FontAngleSet(0);      //--- Reset angle
   }

   canvasGraph.Update();                //--- Update graph canvas
}

//+------------------------------------------------------------------+
//| Fill rounded rectangle                                           |
//+------------------------------------------------------------------+
void FillRoundedRectangle(CCanvas &cvs, int x, int y, int w, int h, int radius, uint argb_color) { // Render rounded fill
   if (radius <= 0) {                   //--- Check zero radius
      cvs.FillRectangle(x, y, x + w - 1, y + h - 1, argb_color); //--- Fill rectangle
      return;                           //--- Exit function
   }
   radius = MathMin(radius, MathMin(w / 2, h / 2)); //--- Adjust radius

   cvs.Arc(x + radius, y + radius, radius, radius, DegreesToRadians(180), DegreesToRadians(90), argb_color); //--- Draw top-left arc
   cvs.Arc(x + w - radius - 1, y + radius, radius, radius, DegreesToRadians(270), DegreesToRadians(90), argb_color); //--- Draw top-right arc
   cvs.Arc(x + w - radius - 1, y + h - radius - 1, radius, radius, DegreesToRadians(0), DegreesToRadians(90), argb_color); //--- Draw bottom-right arc
   cvs.Arc(x + radius, y + h - radius - 1, radius, radius, DegreesToRadians(90), DegreesToRadians(90), argb_color); //--- Draw bottom-left arc

   cvs.FillCircle(x + radius, y + radius, radius, argb_color); //--- Fill top-left circle
   cvs.FillCircle(x + w - radius - 1, y + radius, radius, argb_color); //--- Fill top-right circle
   cvs.FillCircle(x + w - radius - 1, y + h - radius - 1, radius, argb_color); //--- Fill bottom-right circle
   cvs.FillCircle(x + radius, y + h - radius - 1, radius, argb_color); //--- Fill bottom-left circle

   cvs.FillRectangle(x + radius, y, x + w - radius - 1, y + h - 1, argb_color); //--- Fill horizontal body

   cvs.FillRectangle(x, y + radius, x + w - 1, y + h - radius - 1, argb_color); //--- Fill vertical body
}

//+------------------------------------------------------------------+
//| Draw rounded rectangle border                                    |
//+------------------------------------------------------------------+
void DrawRoundedRectangleBorderHiRes(CCanvas &cvs, int positionX, int positionY, int width, int height, int radius, uint borderColorARGB, int thickness) {
   int scaledThickness = thickness;

   DrawRectStraightEdge(cvs, positionX + radius, positionY, positionX + width - radius, positionY, scaledThickness, borderColorARGB);
   DrawRectStraightEdge(cvs, positionX + width - radius, positionY + height - 1, positionX + radius, positionY + height - 1, scaledThickness, borderColorARGB);
   DrawRectStraightEdge(cvs, positionX, positionY + height - radius, positionX, positionY + radius, scaledThickness, borderColorARGB);
   DrawRectStraightEdge(cvs, positionX + width - 1, positionY + radius, positionX + width - 1, positionY + height - radius, scaledThickness, borderColorARGB);

   DrawRectCornerArcPrecise(cvs, positionX + radius, positionY + radius, radius, scaledThickness, borderColorARGB, 
                            M_PI, M_PI * 1.5);
   DrawRectCornerArcPrecise(cvs, positionX + width - radius, positionY + radius, radius, scaledThickness, borderColorARGB,
                            M_PI * 1.5, M_PI * 2.0);
   DrawRectCornerArcPrecise(cvs, positionX + radius, positionY + height - radius, radius, scaledThickness, borderColorARGB,
                            M_PI * 0.5, M_PI);
   DrawRectCornerArcPrecise(cvs, positionX + width - radius, positionY + height - radius, radius, scaledThickness, borderColorARGB,
                            0.0, M_PI * 0.5);
}

//+------------------------------------------------------------------+
//| Draw straight edge for rectangle border                          |
//+------------------------------------------------------------------+
void DrawRectStraightEdge(CCanvas &cvs, double startX, double startY, double endX, double endY, int thickness, uint borderARGB) {
   double deltaX = endX - startX;
   double deltaY = endY - startY;
   double edgeLength = MathSqrt(deltaX*deltaX + deltaY*deltaY);
   if(edgeLength < 1e-6) return;

   double perpendicularX = -deltaY / edgeLength;
   double perpendicularY = deltaX / edgeLength;

   double edgeDirectionX = deltaX / edgeLength;
   double edgeDirectionY = deltaY / edgeLength;

   double halfThickness = (double)thickness / 2.0;
   
   double extensionLength = 1.5;
   double extendedStartX = startX - edgeDirectionX * extensionLength;
   double extendedStartY = startY - edgeDirectionY * extensionLength;
   double extendedEndX = endX + edgeDirectionX * extensionLength;
   double extendedEndY = endY + edgeDirectionY * extensionLength;

   double verticesX[4], verticesY[4];
   verticesX[0] = extendedStartX - perpendicularX * halfThickness;  verticesY[0] = extendedStartY - perpendicularY * halfThickness;
   verticesX[1] = extendedStartX + perpendicularX * halfThickness;  verticesY[1] = extendedStartY + perpendicularY * halfThickness;
   verticesX[2] = extendedEndX + perpendicularX * halfThickness;  verticesY[2] = extendedEndY + perpendicularY * halfThickness;
   verticesX[3] = extendedEndX - perpendicularX * halfThickness;  verticesY[3] = extendedEndY - perpendicularY * halfThickness;

   FillQuadrilateral(cvs, verticesX, verticesY, borderARGB);
}

//+------------------------------------------------------------------+
//| Draw rectangle corner arc with exact boundaries                  |
//+------------------------------------------------------------------+
void DrawRectCornerArcPrecise(CCanvas &cvs, int centerX, int centerY, int radius, int thickness, uint borderARGB,
                              double startAngle, double endAngle) {
   int halfThickness = thickness / 2;
   double outerRadius = (double)radius + halfThickness;
   double innerRadius = (double)radius - halfThickness;
   if(innerRadius < 0) innerRadius = 0;

   int pixelRange = (int)(outerRadius + 2);

   for(int deltaY = -pixelRange; deltaY <= pixelRange; deltaY++) {
      for(int deltaX = -pixelRange; deltaX <= pixelRange; deltaX++) {
         double distance = MathSqrt(deltaX * deltaX + deltaY * deltaY);
         if(distance < innerRadius || distance > outerRadius) continue;

         double angle = MathArctan2((double)deltaY, (double)deltaX);
         
         if(IsAngleBetween(angle, startAngle, endAngle))
            cvs.PixelSet(centerX + deltaX, centerY + deltaY, borderARGB);
      }
   }
}

//+------------------------------------------------------------------+
//| Fill quadrilateral                                               |
//+------------------------------------------------------------------+
void FillQuadrilateral(CCanvas &cvs, double &verticesX[], double &verticesY[], uint fillColor) {
   double minY = verticesY[0], maxY = verticesY[0];
   for(int i = 1; i < 4; i++) {
      if(verticesY[i] < minY) minY = verticesY[i];
      if(verticesY[i] > maxY) maxY = verticesY[i];
   }

   int yStart = (int)MathCeil(minY);
   int yEnd = (int)MathFloor(maxY);

   for(int y = yStart; y <= yEnd; y++) {
      double scanlineY = (double)y + 0.5;
      double xIntersections[8];
      int intersectionCount = 0;

      for(int i = 0; i < 4; i++) {
         int nextIndex = (i + 1) % 4;
         double x0 = verticesX[i], y0 = verticesY[i];
         double x1 = verticesX[nextIndex], y1 = verticesY[nextIndex];

         double edgeMinY = (y0 < y1) ? y0 : y1;
         double edgeMaxY = (y0 > y1) ? y0 : y1;

         if(scanlineY < edgeMinY || scanlineY > edgeMaxY) continue;
         if(MathAbs(y1 - y0) < 1e-12) continue;

         double interpolationFactor = (scanlineY - y0) / (y1 - y0);
         if(interpolationFactor < 0.0 || interpolationFactor > 1.0) continue;

         xIntersections[intersectionCount++] = x0 + interpolationFactor * (x1 - x0);
      }

      for(int a = 0; a < intersectionCount - 1; a++)
         for(int b = a + 1; b < intersectionCount; b++)
            if(xIntersections[a] > xIntersections[b]) {
               double temp = xIntersections[a];
               xIntersections[a] = xIntersections[b];
               xIntersections[b] = temp;
            }

      for(int pairIndex = 0; pairIndex + 1 < intersectionCount; pairIndex += 2) {
         int xLeft = (int)MathCeil(xIntersections[pairIndex]);
         int xRight = (int)MathFloor(xIntersections[pairIndex + 1]);
         for(int x = xLeft; x <= xRight; x++)
            cvs.PixelSet(x, y, fillColor);
      }
   }
}

//+------------------------------------------------------------------+
//| Normalize angle                                                  |
//+------------------------------------------------------------------+
double NormalizeAngle(double angle) {
   double twoPi = 2.0 * M_PI;
   angle = MathMod(angle, twoPi);
   if(angle < 0) angle += twoPi;
   return angle;
}

//+------------------------------------------------------------------+
//| Is angle between                                                 |
//+------------------------------------------------------------------+
bool IsAngleBetween(double angle, double startAngle, double endAngle) {
   angle = NormalizeAngle(angle);
   startAngle = NormalizeAngle(startAngle);
   endAngle = NormalizeAngle(endAngle);
   
   double span = NormalizeAngle(endAngle - startAngle);
   double relativeAngle = NormalizeAngle(angle - startAngle);
   
   return relativeAngle <= span;
}

//+------------------------------------------------------------------+
//| Precompute triangle geometry                                     |
//+------------------------------------------------------------------+
void PrecomputeTriangleGeometry(double &sharpX[], double &sharpY[], int radius, double &arcCentersX[], double &arcCentersY[], double &tangentX[][2], double &tangentY[][2], double &startAngles[], double &endAngles[]) {
   for(int cornerIndex = 0; cornerIndex < 3; cornerIndex++) {
      int previousIndex = (cornerIndex + 2) % 3;
      int nextIndex = (cornerIndex + 1) % 3;

      double edgeA_X = sharpX[cornerIndex] - sharpX[previousIndex], edgeA_Y = sharpY[cornerIndex] - sharpY[previousIndex];
      double edgeA_Length = MathSqrt(edgeA_X*edgeA_X + edgeA_Y*edgeA_Y);
      edgeA_X /= edgeA_Length; edgeA_Y /= edgeA_Length;

      double edgeB_X = sharpX[nextIndex] - sharpX[cornerIndex], edgeB_Y = sharpY[nextIndex] - sharpY[cornerIndex];
      double edgeB_Length = MathSqrt(edgeB_X*edgeB_X + edgeB_Y*edgeB_Y);
      edgeB_X /= edgeB_Length; edgeB_Y /= edgeB_Length;

      double normalA_X = edgeA_Y, normalA_Y = -edgeA_X;
      double normalB_X = edgeB_Y, normalB_Y = -edgeB_X;

      double bisectorX = normalA_X + normalB_X, bisectorY = normalA_Y + normalB_Y;
      double bisectorLength = MathSqrt(bisectorX*bisectorX + bisectorY*bisectorY);
      if(bisectorLength < 1e-12) { bisectorX = normalA_X; bisectorY = normalA_Y; bisectorLength = MathSqrt(bisectorX*bisectorX + bisectorY*bisectorY); }
      bisectorX /= bisectorLength; bisectorY /= bisectorLength;

      double cosInteriorAngle = (-edgeA_X)*edgeB_X + (-edgeA_Y)*edgeB_Y;
      if(cosInteriorAngle > 1.0) cosInteriorAngle = 1.0;
      if(cosInteriorAngle < -1.0) cosInteriorAngle = -1.0;
      double halfAngle = MathArccos(cosInteriorAngle) / 2.0;
      double sinHalfAngle = MathSin(halfAngle);
      if(sinHalfAngle < 1e-12) sinHalfAngle = 1e-12;

      double distanceToCenter = radius / sinHalfAngle;
      arcCentersX[cornerIndex] = sharpX[cornerIndex] + bisectorX * distanceToCenter;
      arcCentersY[cornerIndex] = sharpY[cornerIndex] + bisectorY * distanceToCenter;

      double deltaX_A = sharpX[cornerIndex] - sharpX[previousIndex], deltaY_A = sharpY[cornerIndex] - sharpY[previousIndex];
      double lengthSquared_A = deltaX_A*deltaX_A + deltaY_A*deltaY_A;
      double interpolationFactor_A = ((arcCentersX[cornerIndex] - sharpX[previousIndex])*deltaX_A + (arcCentersY[cornerIndex] - sharpY[previousIndex])*deltaY_A) / lengthSquared_A;
      tangentX[cornerIndex][1] = sharpX[previousIndex] + interpolationFactor_A * deltaX_A;
      tangentY[cornerIndex][1] = sharpY[previousIndex] + interpolationFactor_A * deltaY_A;

      double deltaX_B = sharpX[nextIndex] - sharpX[cornerIndex], deltaY_B = sharpY[nextIndex] - sharpY[cornerIndex];
      double lengthSquared_B = deltaX_B*deltaX_B + deltaY_B*deltaY_B;
      double interpolationFactor_B = ((arcCentersX[cornerIndex] - sharpX[cornerIndex])*deltaX_B + (arcCentersY[cornerIndex] - sharpY[cornerIndex])*deltaY_B) / lengthSquared_B;
      tangentX[cornerIndex][0] = sharpX[cornerIndex] + interpolationFactor_B * deltaX_B;
      tangentY[cornerIndex][0] = sharpY[cornerIndex] + interpolationFactor_B * deltaY_B;

      startAngles[cornerIndex] = MathArctan2(tangentY[cornerIndex][1] - arcCentersY[cornerIndex], tangentX[cornerIndex][1] - arcCentersX[cornerIndex]);
      endAngles[cornerIndex] = MathArctan2(tangentY[cornerIndex][0] - arcCentersY[cornerIndex], tangentX[cornerIndex][0] - arcCentersX[cornerIndex]);
   }
}

//+------------------------------------------------------------------+
//| Angle in arc sweep for triangle                                  |
//+------------------------------------------------------------------+
bool TriangleAngleInArcSweep(double startAngle, double endAngle, double angle) {
   double twoPi = 2.0 * M_PI;
   double startAngleMod = MathMod(startAngle + twoPi, twoPi);
   double endAngleMod = MathMod(endAngle + twoPi, twoPi);
   angle = MathMod(angle + twoPi, twoPi);

   double ccwSpan = MathMod(endAngleMod - startAngleMod + twoPi, twoPi);

   if(ccwSpan <= M_PI) {
      double relativeAngle = MathMod(angle - startAngleMod + twoPi, twoPi);
      return(relativeAngle <= ccwSpan + 1e-6);
   } else {
      double cwSpan = twoPi - ccwSpan;
      double relativeAngle = MathMod(angle - endAngleMod + twoPi, twoPi);
      return(relativeAngle <= cwSpan + 1e-6);
   }
}

//+------------------------------------------------------------------+
//| Fill rounded triangle                                            |
//+------------------------------------------------------------------+
void FillRoundedTriangle(CCanvas &cvs, double &sharpX[], double &sharpY[], int radius, uint fillColor, double &arcCentersX[], double &arcCentersY[], double &tangentX[][2], double &tangentY[][2], double &startAngles[], double &endAngles[]) {
   double minY = sharpY[0], maxY = sharpY[0];
   for(int i = 1; i < 3; i++) {
      if(sharpY[i] < minY) minY = sharpY[i];
      if(sharpY[i] > maxY) maxY = sharpY[i];
   }

   int yStart = (int)MathCeil(minY);
   int yEnd = (int)MathFloor(maxY);

   for(int y = yStart; y <= yEnd; y++) {
      double scanlineY = (double)y + 0.5;

      double xIntersections[12];
      int intersectionCount = 0;

      for(int edgeIndex = 0; edgeIndex < 3; edgeIndex++) {
         int nextIndex = (edgeIndex + 1) % 3;
         double startX = tangentX[edgeIndex][0], startY = tangentY[edgeIndex][0];
         double endX = tangentX[nextIndex][1], endY = tangentY[nextIndex][1];

         double edgeMinY = (startY < endY) ? startY : endY;
         double edgeMaxY = (startY > endY) ? startY : endY;

         if(scanlineY < edgeMinY || scanlineY > edgeMaxY) continue;
         if(MathAbs(endY - startY) < 1e-12) continue;

         double interpolationFactor = (scanlineY - startY) / (endY - startY);
         if(interpolationFactor < 0.0 || interpolationFactor > 1.0) continue;

         xIntersections[intersectionCount++] = startX + interpolationFactor * (endX - startX);
      }

      for(int cornerIndex = 0; cornerIndex < 3; cornerIndex++) {
         double centerX = arcCentersX[cornerIndex], centerY = arcCentersY[cornerIndex];
         double deltaY = scanlineY - centerY;

         if(MathAbs(deltaY) > radius) continue;

         double deltaX = MathSqrt(radius*radius - deltaY*deltaY);

         double candidates[2];
         candidates[0] = centerX - deltaX;
         candidates[1] = centerX + deltaX;

         for(int candidateIndex = 0; candidateIndex < 2; candidateIndex++) {
            double angle = MathArctan2(scanlineY - centerY, candidates[candidateIndex] - centerX);
            if(TriangleAngleInArcSweep(startAngles[cornerIndex], endAngles[cornerIndex], angle))
               xIntersections[intersectionCount++] = candidates[candidateIndex];
         }
      }

      for(int a = 0; a < intersectionCount - 1; a++)
         for(int b = a + 1; b < intersectionCount; b++)
            if(xIntersections[a] > xIntersections[b]) {
               double temp = xIntersections[a];
               xIntersections[a] = xIntersections[b];
               xIntersections[b] = temp;
            }

      for(int pairIndex = 0; pairIndex + 1 < intersectionCount; pairIndex += 2) {
         int xLeft = (int)MathCeil(xIntersections[pairIndex]);
         int xRight = (int)MathFloor(xIntersections[pairIndex + 1]);
         for(int x = xLeft; x <= xRight; x++)
            cvs.PixelSet(x, y, fillColor);
      }
   }
}

//+------------------------------------------------------------------+
//| Draw rounded triangle arrow                                      |
//+------------------------------------------------------------------+
void DrawRoundedTriangleArrow(CCanvas &cvs, int baseX, int baseY, int tri_base_width, int tri_height, bool isUp, uint fillColor) {
   int radius = TriangleRoundRadius * supersamplingFactor;

   double sharpX[3], sharpY[3];

   if (isUp) {
      sharpX[0] = baseX;
      sharpY[0] = baseY;
      sharpX[1] = baseX - tri_base_width / 2.0;
      sharpY[1] = baseY + tri_height;
      sharpX[2] = baseX + tri_base_width / 2.0;
      sharpY[2] = baseY + tri_height;
   } else {
      sharpX[0] = baseX;
      sharpY[0] = baseY + tri_height;
      sharpX[1] = baseX + tri_base_width / 2.0;  // Swapped for consistent winding
      sharpY[1] = baseY;
      sharpX[2] = baseX - tri_base_width / 2.0;
      sharpY[2] = baseY;
   }

   double arcCentersX[3], arcCentersY[3];
   double tangentX[3][2], tangentY[3][2];
   double startAngles[3], endAngles[3];

   PrecomputeTriangleGeometry(sharpX, sharpY, radius, arcCentersX, arcCentersY, tangentX, tangentY, startAngles, endAngles);

   FillRoundedTriangle(cvs, sharpX, sharpY, radius, fillColor, arcCentersX, arcCentersY, tangentX, tangentY, startAngles, endAngles);
}

//+------------------------------------------------------------------+
//| Update stats on canvas                                           |
//+------------------------------------------------------------------+
void UpdateStatsOnCanvas() {            // Render stats elements
   canvasStatsHighRes.Erase(0);         //--- Clear high-res stats canvas

   int statsWidthHighRes = (currentWidth / 2) * supersamplingFactor; //--- Scaled width
   int heightHighRes = currentHeight * supersamplingFactor; //--- Scaled height

   if (UseBackground && ArraySize(bg_pixels_stats) == (currentWidth / 2) * currentHeight) { //--- Check background valid
      uint bg_pixels_stats_high[];      //--- High-res bg
      ArrayResize(bg_pixels_stats_high, statsWidthHighRes * heightHighRes); //--- Resize
      ScaleImage(bg_pixels_stats_high, currentWidth / 2, currentHeight, statsWidthHighRes, heightHighRes); //--- Scale bg
      for (int y = 0; y < heightHighRes; y++) { //--- Loop Y
         for (int x = 0; x < statsWidthHighRes; x++) { //--- Loop X
            canvasStatsHighRes.PixelSet(x, y, bg_pixels_stats_high[y * statsWidthHighRes + x]); //--- Set pixel
         }
      }
   }

   if (StatsBackgroundMode != NoColor) { //--- Check background mode
      for (int y = 0; y < heightHighRes; y++) { //--- Loop rows
         double factor = (double)y / (heightHighRes - 1); //--- Compute factor
         color currentColor = (StatsBackgroundMode == SingleColor) ? GetTopColor() : InterpolateColor(GetTopColor(), GetBottomColor(), factor); //--- Get color
         uchar alpha = (uchar)(255 * BackgroundOpacity); //--- Compute alpha
         uint argbFill = ColorToARGB(currentColor, alpha); //--- Convert to ARGB

         for (int x = 0; x < statsWidthHighRes; x++) { //--- Loop columns
            uint currentPixel = canvasStatsHighRes.PixelGet(x, y); //--- Get pixel
            uint blendedPixel = BlendPixels(currentPixel, argbFill); //--- Blend pixels
            canvasStatsHighRes.PixelSet(x, y, blendedPixel); //--- Set blended pixel
         }
      }
   }

   if (StatsBackgroundMode != NoColor) { //--- Check background mode for borders
      double reduction = BorderOpacityPercentReduction / 100.0; //--- Compute reduction
      double opacity = MathMax(0.0, MathMin(1.0, BackgroundOpacity * (1.0 - reduction))); //--- Compute opacity
      uchar alpha = (uchar)(255 * opacity); //--- Set alpha
      double darkenReduction = BorderDarkenPercent / 100.0; //--- Compute darken reduction
      double darkenFactor = MathMax(0.0, MathMin(1.0, 1.0 - darkenReduction)); //--- Compute darken factor

      for (int y = 0; y < heightHighRes; y++) { //--- Loop vertical borders
         double factor = (StatsBackgroundMode == SingleColor) ? 0.0 : (double)y / (heightHighRes - 1); //--- Get factor
         color baseColor = (StatsBackgroundMode == SingleColor) ? GetTopColor() : InterpolateColor(GetTopColor(), GetBottomColor(), factor); //--- Get base color
         color darkColor = DarkenColor(baseColor, darkenFactor); //--- Darken color
         uint argb = ColorToARGB(darkColor, alpha); //--- Convert to ARGB

         canvasStatsHighRes.PixelSet(0, y, argb); //--- Set left border pixel
         canvasStatsHighRes.PixelSet(1, y, argb); //--- Set inner left pixel

         canvasStatsHighRes.PixelSet(statsWidthHighRes - 1, y, argb); //--- Set right border pixel
         canvasStatsHighRes.PixelSet(statsWidthHighRes - 2, y, argb); //--- Set inner right pixel
      }

      double factorTop = 0.0;           //--- Set top factor
      color baseTop = GetTopColor();     //--- Get top base
      color darkTop = DarkenColor(baseTop, darkenFactor); //--- Darken top
      uint argbTop = ColorToARGB(darkTop, alpha); //--- Convert top to ARGB
      for (int x = 0; x < statsWidthHighRes; x++) { //--- Loop top borders
         canvasStatsHighRes.PixelSet(x, 0, argbTop); //--- Set top pixel
         canvasStatsHighRes.PixelSet(x, 1, argbTop); //--- Set inner top pixel
      }

      double factorBot = (StatsBackgroundMode == SingleColor) ? 0.0 : 1.0; //--- Set bottom factor
      color baseBot = (StatsBackgroundMode == SingleColor) ? GetTopColor() : GetBottomColor(); //--- Get bottom base
      color darkBot = DarkenColor(baseBot, darkenFactor); //--- Darken bottom
      uint argbBot = ColorToARGB(darkBot, alpha); //--- Convert bottom to ARGB
      for (int x = 0; x < statsWidthHighRes; x++) { //--- Loop bottom borders
         canvasStatsHighRes.PixelSet(x, heightHighRes - 1, argbBot); //--- Set bottom pixel
         canvasStatsHighRes.PixelSet(x, heightHighRes - 2, argbBot); //--- Set inner bottom pixel
      }
   } else {                             //--- Handle no background
      uint argbBorder = ColorToARGB(GetBorderColor(), 255); //--- Convert border to ARGB
      canvasStatsHighRes.Line(0, 0, statsWidthHighRes - 1, 0, argbBorder); //--- Draw top border
      canvasStatsHighRes.Line(statsWidthHighRes - 1, 0, statsWidthHighRes - 1, heightHighRes - 1, argbBorder); //--- Draw right border
      canvasStatsHighRes.Line(statsWidthHighRes - 1, heightHighRes - 1, 0, heightHighRes - 1, argbBorder); //--- Draw bottom border
      canvasStatsHighRes.Line(0, heightHighRes - 1, 0, 0, argbBorder); //--- Draw left border
      canvasStatsHighRes.Line(1, 1, statsWidthHighRes - 2, 1, argbBorder); //--- Draw inner top
      canvasStatsHighRes.Line(statsWidthHighRes - 2, 1, statsWidthHighRes - 2, heightHighRes - 2, argbBorder); //--- Draw inner right
      canvasStatsHighRes.Line(statsWidthHighRes - 2, heightHighRes - 2, 1, heightHighRes - 2, argbBorder); //--- Draw inner bottom
      canvasStatsHighRes.Line(1, heightHighRes - 2, 1, 1, argbBorder); //--- Draw inner left
   }

   color labelColor = GetStatsLabelColor(); //--- Get label color
   color valueColor = GetStatsValueColor(); //--- Get value color
   color headerColor = GetStatsHeaderColor(); //--- Get header color

   int yPos = 20 * supersamplingFactor; //--- Scaled Y position for first header
   string headerText = "Account Stats"; //--- Set header text
   int fontSizeHigh = StatsHeaderFontSize * supersamplingFactor; //--- Scaled font
   canvasStatsHighRes.FontSet("Arial Bold", fontSizeHigh); //--- Set header font
   int textW = canvasStatsHighRes.TextWidth(headerText); //--- Get text width
   int textH = canvasStatsHighRes.TextHeight(headerText); //--- Get text height
   int pad = 5 * supersamplingFactor;   //--- Scaled padding
   int rectX = (statsWidthHighRes - textW) / 2 - pad; //--- Compute rect X
   int rectY = yPos - pad / 2;          //--- Compute rect Y
   int rectW = textW + 2 * pad;         //--- Compute rect width
   int rectH = textH + pad;             //--- Compute rect height
   uchar alpha = (uchar)(255 * (StatsHeaderBgOpacityPercent / 100.0)); //--- Compute alpha
   uint argbHeaderBg = ColorToARGB(GetStatsHeaderColor(), alpha); //--- Convert bg to ARGB

   color header_border_color = GetStatsHeaderColor(); //--- Significant color
   uint argbHeaderBorder = ColorToARGB(header_border_color, 255); //--- Convert border to ARGB
   FillRoundedRectangle(canvasStatsHighRes, rectX, rectY, rectW, rectH, StatsHeaderBgRadius * supersamplingFactor, argbHeaderBg); //--- Fill inner rect
   DrawRoundedRectangleBorderHiRes(canvasStatsHighRes, rectX, rectY, rectW, rectH, StatsHeaderBgRadius * supersamplingFactor, argbHeaderBorder, 1 * supersamplingFactor); //--- Draw border

   int yPosSecond = 120 * supersamplingFactor; //--- Scaled Y for second header
   headerText = "Current Bar Stats";    //--- Set bar header
   textW = canvasStatsHighRes.TextWidth(headerText); //--- Get width
   textH = canvasStatsHighRes.TextHeight(headerText); //--- Get height
   rectX = (statsWidthHighRes - textW) / 2 - pad; //--- Compute rect X
   rectY = yPosSecond - pad / 2;        //--- Compute rect Y
   rectW = textW + 2 * pad;             //--- Compute rect width
   rectH = textH + pad;                 //--- Compute rect height

   FillRoundedRectangle(canvasStatsHighRes, rectX, rectY, rectW, rectH, StatsHeaderBgRadius * supersamplingFactor, argbHeaderBg); //--- Fill inner rect
   DrawRoundedRectangleBorderHiRes(canvasStatsHighRes, rectX, rectY, rectW, rectH, StatsHeaderBgRadius * supersamplingFactor, argbHeaderBorder, 2 * supersamplingFactor); //--- Draw border

   DownsampleCanvas(canvasStats, canvasStatsHighRes); //--- Downsample

   canvasStats.FontSet("Arial Bold", StatsHeaderFontSize); //--- Set header font
   uint argbHeader = ColorToARGB(headerColor, 255); //--- Convert header to ARGB
   canvasStats.TextOut((currentWidth / 2) / 2, 20, "Account Stats", argbHeader, TA_CENTER); //--- Draw first header text

   canvasStats.TextOut((currentWidth / 2) / 2, 120, "Current Bar Stats", argbHeader, TA_CENTER); //--- Draw second header text

   canvasStats.FontSet("Arial Bold", StatsFontSize); //--- Set stats font
   uint argbLabel = ColorToARGB(labelColor, 255); //--- Convert label to ARGB
   uint argbValue = ColorToARGB(valueColor, 255); //--- Convert value to ARGB

   int yPosDisplay = 50;                // Name at 50
   canvasStats.TextOut(10, yPosDisplay, "Name:", argbLabel, TA_LEFT); //--- Draw name label
   canvasStats.TextOut((currentWidth / 2) - 10, yPosDisplay, AccountInfoString(ACCOUNT_NAME), argbValue, TA_RIGHT); //--- Draw name value
   yPosDisplay += 20;                   // Balance at 70

   canvasStats.TextOut(10, yPosDisplay, "Balance:", argbLabel, TA_LEFT); //--- Draw balance label
   canvasStats.TextOut((currentWidth / 2) - 10, yPosDisplay, DoubleToString(AccountInfoDouble(ACCOUNT_BALANCE), 2), argbValue, TA_RIGHT); //--- Draw balance value
   yPosDisplay += 20;                   // Equity at 90

   canvasStats.TextOut(10, yPosDisplay, "Equity:", argbLabel, TA_LEFT); //--- Draw equity label
   canvasStats.TextOut((currentWidth / 2) - 10, yPosDisplay, DoubleToString(AccountInfoDouble(ACCOUNT_EQUITY), 2), argbValue, TA_RIGHT); //--- Draw equity value
   yPosDisplay += 30;                   // Second header at 120, then labels

   yPosDisplay = 150;                   // Open at 150 (120 +30)
   canvasStats.TextOut(10, yPosDisplay, "Open:", argbLabel, TA_LEFT); //--- Draw open label
   canvasStats.TextOut((currentWidth / 2) - 10, yPosDisplay, DoubleToString(iOpen(_Symbol, _Period, 0), _Digits), argbValue, TA_RIGHT); //--- Draw open value
   yPosDisplay += 20;                   // High at 170

   canvasStats.TextOut(10, yPosDisplay, "High:", argbLabel, TA_LEFT); //--- Draw high label
   canvasStats.TextOut((currentWidth / 2) - 10, yPosDisplay, DoubleToString(iHigh(_Symbol, _Period, 0), _Digits), argbValue, TA_RIGHT); //--- Draw high value
   yPosDisplay += 20;                   // Low at 190

   canvasStats.TextOut(10, yPosDisplay, "Low:", argbLabel, TA_LEFT); //--- Draw low label
   canvasStats.TextOut((currentWidth / 2) - 10, yPosDisplay, DoubleToString(iLow(_Symbol, _Period, 0), _Digits), argbValue, TA_RIGHT); //--- Draw low value
   yPosDisplay += 20;                   // Close at 210

   canvasStats.TextOut(10, yPosDisplay, "Close:", argbLabel, TA_LEFT); //--- Draw close label
   canvasStats.TextOut((currentWidth / 2) - 10, yPosDisplay, DoubleToString(iClose(_Symbol, _Period, 0), _Digits), argbValue, TA_RIGHT); //--- Draw close value

   canvasStats.Update();                //--- Update stats canvas
}

//+------------------------------------------------------------------+
//| Update text on canvas                                            |
//+------------------------------------------------------------------+
void UpdateTextOnCanvas() {             // Render text elements
   canvasTextHighRes.Erase(0);          //--- Clear high-res text canvas

   int textWidthHighRes = canvasText.Width() * supersamplingFactor; //--- Scaled width
   int textHeightHighRes = canvasText.Height() * supersamplingFactor; //--- Scaled height

   color text_bg = is_dark_theme ? text_bg_dark : text_bg_light; //--- Get bg color
   uint argb_bg = ColorToARGB(text_bg, (uchar)(255 * (TextBackgroundOpacityPercent / 100.0))); //--- Convert bg to ARGB
   canvasTextHighRes.FillRectangle(0, 0, textWidthHighRes - 1, textHeightHighRes - 1, argb_bg); //--- Fill background

   uint argbBorder = ColorToARGB(GetBorderColor(), 255); //--- Convert border to ARGB
   canvasTextHighRes.Line(0, 0, textWidthHighRes - 1, 0, argbBorder); //--- Draw top border
   canvasTextHighRes.Line(textWidthHighRes - 1, 0, textWidthHighRes - 1, textHeightHighRes - 1, argbBorder); //--- Draw right border
   canvasTextHighRes.Line(textWidthHighRes - 1, textHeightHighRes - 1, 0, textHeightHighRes - 1, argbBorder); //--- Draw bottom border
   canvasTextHighRes.Line(0, textHeightHighRes - 1, 0, 0, argbBorder); //--- Draw left border

   int padding = 10 * supersamplingFactor; //--- Scaled padding
   int textAreaX = padding;             //--- Set area X
   int textAreaY = 0;                   //--- Set area Y
   int textAreaWidth = textWidthHighRes - padding * 2; //--- Compute area width
   int textAreaHeight = textHeightHighRes; //--- Set area height
   string font = "Calibri";               //--- Set font
   int fontSize = TextFontSize * supersamplingFactor;  // Scaled font size
   canvasTextHighRes.FontSet(font, fontSize); //--- Apply font
   int lineHeight = canvasTextHighRes.TextHeight("A"); //--- Get line height
   text_adjustedLineHeight = (lineHeight + 3) / supersamplingFactor; //--- Adjust line height (display scale)

   text_visible_height = textAreaHeight / supersamplingFactor; //--- Visible height (display scale)
   static string wrappedLines[];        //--- Declare wrapped lines
   static color wrappedColors[];        //--- Declare wrapped colors
   static bool wrapped = false;         //--- Track wrapped state
   bool need_scroll = false;            //--- Set scroll need
   int reserved_width = 0;              //--- Set reserved width
   if (!wrapped) {                      //--- Check not wrapped
      WrapText(text_usage_text, font, fontSize / supersamplingFactor, textAreaWidth / supersamplingFactor, wrappedLines, wrappedColors); //--- Wrap at display scale
      wrapped = true;                   //--- Set wrapped flag
   }
   int numLines = ArraySize(wrappedLines); //--- Get line count
   text_total_height = numLines * text_adjustedLineHeight; //--- Compute total height
   need_scroll = text_total_height > text_visible_height; //--- Check need scroll
   if (need_scroll) {                   //--- Handle scroll needed
      reserved_width = text_track_width * supersamplingFactor; //--- Reserve scaled width
      textAreaWidth -= reserved_width;  //--- Adjust area width
      WrapText(text_usage_text, font, fontSize / supersamplingFactor, textAreaWidth / supersamplingFactor, wrappedLines, wrappedColors); //--- Rewrap text
      numLines = ArraySize(wrappedLines); //--- Update line count
      text_total_height = numLines * text_adjustedLineHeight; //--- Update total height
   }
   text_max_scroll = MathMax(0, text_total_height - text_visible_height) * smoothness_factor; //--- Compute max scroll
   text_scroll_visible = need_scroll;   //--- Set scroll visible
   text_scroll_pos = MathMax(0, MathMin(text_scroll_pos, text_max_scroll)); //--- Clamp scroll pos
   if (text_scroll_visible) {           //--- Check scroll visible
      int scrollbar_y = 0;              //--- Set scrollbar Y
      int scrollbar_height = textAreaHeight; //--- Set scrollbar height
      int scroll_area_height = scrollbar_height - 2 * (text_button_size * supersamplingFactor); //--- Scaled area height
      text_slider_height = TextCalculateSliderHeight(); //--- Compute slider height
      int scrollbar_x = textWidthHighRes - (text_track_width * supersamplingFactor); //--- Scaled scrollbar X
      color leader_color = is_dark_theme ? text_leader_color_dark : text_leader_color_light; //--- Get leader color
      uint argb_leader = ColorToARGB(leader_color, 255); //--- Convert to ARGB
      canvasTextHighRes.FillRectangle(scrollbar_x, scrollbar_y, scrollbar_x + (text_track_width * supersamplingFactor) - 1, scrollbar_y + scrollbar_height - 1, argb_leader); //--- Fill leader

      int slider_y = scrollbar_y + (text_button_size * supersamplingFactor) + (int)(((double)text_scroll_pos / text_max_scroll) * (scroll_area_height - (text_slider_height * supersamplingFactor))); //--- Scaled slider Y

      if (text_scroll_area_hovered) {   //--- Check area hovered
         color button_bg = is_dark_theme ? text_button_bg_dark : text_button_bg_light; //--- Get button bg
         color button_bg_hover = is_dark_theme ? text_button_bg_hover_dark : text_button_bg_hover_light; //--- Get hover bg
         color up_bg;
         if (ShowUpDownButtons) {
            up_bg = text_scroll_up_hovered ? button_bg_hover : button_bg;
         } else {
            up_bg = leader_color; // Blend with track, no hover
         }
         uint argb_up_bg = ColorToARGB(up_bg, (uchar)255); //--- Convert up bg
         canvasTextHighRes.FillRectangle(scrollbar_x, scrollbar_y, scrollbar_x + (text_track_width * supersamplingFactor) - 1, scrollbar_y + (text_button_size * supersamplingFactor) - 1, argb_up_bg); //--- Fill up button
         color arrow_color = is_dark_theme ? text_arrow_color_dark : text_arrow_color_light; //--- Get arrow color
         color arrow_color_disabled = is_dark_theme ? text_arrow_color_disabled_dark : text_arrow_color_disabled_light; //--- Get disabled arrow
         color arrow_color_hover = is_dark_theme ? text_arrow_color_hover_dark : text_arrow_color_hover_light; //--- Get hover arrow
         color up_arrow = (text_scroll_pos == 0) ? arrow_color_disabled : (text_scroll_up_hovered ? arrow_color_hover : arrow_color); //--- Get up arrow color
         uint argb_up_arrow = ColorToARGB(up_arrow, (uchar)255); //--- Convert up arrow

         // Draw up rounded triangle
         int arrow_x = scrollbar_x + (text_track_width * supersamplingFactor) / 2; //--- Center X
         double base_width = text_button_size * (TriangleBaseWidthPercent / 100.0) * supersamplingFactor; //--- Compute base width
         int tri_height = (int)(base_width * (TriangleHeightPercent / 100.0)); //--- Compute height
         int arrow_y = scrollbar_y + ((text_button_size * supersamplingFactor) - tri_height) / 2; //--- Centered Y
         DrawRoundedTriangleArrow(canvasTextHighRes, arrow_x, arrow_y, (int)base_width, tri_height, true, argb_up_arrow); //--- Up arrow

         int down_y = scrollbar_y + scrollbar_height - (text_button_size * supersamplingFactor); //--- Compute down Y
         color down_bg;
         if (ShowUpDownButtons) {
            down_bg = text_scroll_down_hovered ? button_bg_hover : button_bg;
         } else {
            down_bg = leader_color; // Blend with track, no hover
         }
         uint argb_down_bg = ColorToARGB(down_bg, (uchar)255); //--- Convert down bg
         canvasTextHighRes.FillRectangle(scrollbar_x, down_y, scrollbar_x + (text_track_width * supersamplingFactor) - 1, down_y + (text_button_size * supersamplingFactor) - 1, argb_down_bg); //--- Fill down button
         color down_arrow = (text_scroll_pos >= text_max_scroll) ? arrow_color_disabled : (text_scroll_down_hovered ? arrow_color_hover : arrow_color); //--- Get down arrow color
         uint argb_down_arrow = ColorToARGB(down_arrow, (uchar)255); //--- Convert down arrow

         // Draw down rounded triangle
         int down_arrow_x = scrollbar_x + (text_track_width * supersamplingFactor) / 2; //--- Center X
         int down_arrow_y = down_y + ((text_button_size * supersamplingFactor) - tri_height) / 2; //--- Centered Y
         DrawRoundedTriangleArrow(canvasTextHighRes, down_arrow_x, down_arrow_y, (int)base_width, tri_height, false, argb_down_arrow); //--- Down arrow

         int slider_x = scrollbar_x + (text_scrollbar_margin * supersamplingFactor); //--- Scaled slider X
         int slider_w = (text_track_width * supersamplingFactor) - 2 * (text_scrollbar_margin * supersamplingFactor); //--- Scaled slider width
         int cap_radius = slider_w / 2; //--- Compute cap radius
         color slider_bg_color = is_dark_theme ? text_slider_bg_dark : text_slider_bg_light; //--- Get slider bg
         color slider_bg_hover_color = is_dark_theme ? text_slider_bg_hover_dark : text_slider_bg_hover_light; //--- Get hover bg
         color slider_bg = text_scroll_slider_hovered || text_movingStateSlider ? slider_bg_hover_color : slider_bg_color; //--- Determine slider bg
         uint argb_slider = ColorToARGB(slider_bg, (uchar)255); //--- Convert slider to ARGB

         canvasTextHighRes.Arc(slider_x + cap_radius, slider_y + cap_radius, cap_radius, cap_radius, DegreesToRadians(180), DegreesToRadians(360), argb_slider); //--- Draw top arc
         canvasTextHighRes.FillCircle(slider_x + cap_radius, slider_y + cap_radius, cap_radius, argb_slider); //--- Fill top cap

         canvasTextHighRes.Arc(slider_x + cap_radius, slider_y + (text_slider_height * supersamplingFactor) - cap_radius, cap_radius, cap_radius, DegreesToRadians(0), DegreesToRadians(180), argb_slider); //--- Draw bottom arc
         canvasTextHighRes.FillCircle(slider_x + cap_radius, slider_y + (text_slider_height * supersamplingFactor) - cap_radius, cap_radius, argb_slider); //--- Fill bottom cap

         canvasTextHighRes.FillRectangle(slider_x, slider_y + cap_radius, slider_x + slider_w, slider_y + (text_slider_height * supersamplingFactor) - cap_radius, argb_slider); //--- Fill slider body
      } else {                          //--- Handle thin scrollbar
         int thin_w = text_scrollbar_thin_width * supersamplingFactor; //--- Scaled thin width
         int thin_x = scrollbar_x + ((text_track_width * supersamplingFactor) - thin_w) / 2; //--- Compute thin X
         int cap_radius = thin_w / 2;   //--- Compute cap radius
         color slider_bg_color = is_dark_theme ? text_slider_bg_dark : text_slider_bg_light; //--- Get slider bg
         uint argb_slider = ColorToARGB(slider_bg_color, (uchar)255); //--- Convert to ARGB

         canvasTextHighRes.Arc(thin_x + cap_radius, slider_y + cap_radius, cap_radius, cap_radius, DegreesToRadians(180), DegreesToRadians(360), argb_slider); //--- Draw top arc
         canvasTextHighRes.FillCircle(thin_x + cap_radius, slider_y + cap_radius, cap_radius, argb_slider); //--- Fill top cap

         canvasTextHighRes.Arc(thin_x + cap_radius, slider_y + (text_slider_height * supersamplingFactor) - cap_radius, cap_radius, cap_radius, DegreesToRadians(0), DegreesToRadians(180), argb_slider); //--- Draw bottom arc
         canvasTextHighRes.FillCircle(thin_x + cap_radius, slider_y + (text_slider_height * supersamplingFactor) - cap_radius, cap_radius, argb_slider); //--- Fill bottom cap

         canvasTextHighRes.FillRectangle(thin_x, slider_y + cap_radius, thin_x + thin_w, slider_y + (text_slider_height * supersamplingFactor) - cap_radius, argb_slider); //--- Fill thin body
      }
   }

   DownsampleCanvas(canvasText, canvasTextHighRes); //--- Downsample

   color text_base = is_dark_theme ? text_base_dark : text_base_light; //--- Get base color
   canvasText.FontSet("Calibri", TextFontSize);     //--- Normal font
   for (int line = 0; line < numLines; line++) { //--- Loop lines
      string lineText = wrappedLines[line]; //--- Get line text
      if (StringLen(lineText) == 0) continue; //--- Skip empty
      color lineColor = wrappedColors[line]; //--- Get line color
      if (is_dark_theme) lineColor = (lineColor == clrWhite) ? clrWhite : LightenColor(lineColor, 1.5); //--- Adjust dark color
      else lineColor = (lineColor == clrWhite) ? clrBlack : DarkenColor(lineColor, 0.7); //--- Adjust light color
      int line_y = (textAreaY / supersamplingFactor) + line * text_adjustedLineHeight - (text_scroll_pos/smoothness_factor); //--- Compute line Y (display scale)
      if (line_y + text_adjustedLineHeight < 0 || line_y > text_visible_height) continue; //--- Skip out of view
      if (IsHeading(lineText) ) {        //--- Check heading
         canvasText.FontSet("Calibri Bold", TextFontSize); //--- Set bold font
         lineColor = clrDodgerBlue;     //--- Set heading color
      } else canvasText.FontSet("Calibri", TextFontSize); //--- Set normal font
      uint argbText = ColorToARGB(lineColor, 255); //--- Convert to ARGB
      canvasText.TextOut(textAreaX / supersamplingFactor, line_y, lineText, argbText, TA_LEFT); //--- Draw line text
   }

   canvasText.Update();                 //--- Update text canvas
}

//+------------------------------------------------------------------+
//| Lighten color                                                    |
//+------------------------------------------------------------------+
color LightenColor(color colorValue, double factor) { // Adjust color brighter
   int blue = (int)MathMin(255, (colorValue & 0xFF) * factor); //--- Compute blue
   int green = (int)MathMin(255, ((colorValue >> 8) & 0xFF) * factor); //--- Compute green
   int red = (int)MathMin(255, ((colorValue >> 16) & 0xFF) * factor); //--- Compute red
   return (color)(blue | (green << 8) | (red << 16)); //--- Return lightened color
}

//+------------------------------------------------------------------+
//| Calculate slider height                                          |
//+------------------------------------------------------------------+
int TextCalculateSliderHeight() {       // Compute text slider height
   int scroll_area_height = TextPanelHeight - 2 - 2 * text_button_size; //--- Compute area height
   int slider_min_height = 20;          //--- Set min height
   if (text_total_height <= text_visible_height) return scroll_area_height; //--- Return full if no scroll
   double visible_ratio = (double)text_visible_height / text_total_height; //--- Compute ratio
   int height = (int)MathFloor(scroll_area_height * visible_ratio); //--- Compute height
   return MathMax(slider_min_height, height); //--- Return clamped height
}

//+------------------------------------------------------------------+
//| Scroll text up                                                   |
//+------------------------------------------------------------------+
void TextScrollUp() {                   // Move text scroll up
   if (text_adjustedLineHeight > 0 && text_scroll_pos > 0) { //--- Check valid scroll
      text_scroll_pos = MathMax(0, text_scroll_pos - text_adjustedLineHeight * smoothness_factor); //--- Adjust position
      UpdateTextOnCanvas();             //--- Update canvas
   }
}

//+------------------------------------------------------------------+
//| Scroll text down                                                 |
//+------------------------------------------------------------------+
void TextScrollDown() {                 // Move text scroll down
   if (text_adjustedLineHeight > 0 && text_scroll_pos < text_max_scroll) { //--- Check valid scroll
      text_scroll_pos = MathMin(text_max_scroll, text_scroll_pos + text_adjustedLineHeight * smoothness_factor); //--- Adjust position
      UpdateTextOnCanvas();             //--- Update canvas
   }
}

//+------------------------------------------------------------------+
//| Update text hover effects                                        |
//+------------------------------------------------------------------+
void TextUpdateHoverEffects(int local_x, int local_y) { // Apply text hovers
   if (!text_scroll_visible) return;    //--- Exit if no scroll
   int scrollbar_x = canvasText.Width() - text_track_width; //--- Set scrollbar X
   int scrollbar_y = 0;                 //--- Set scrollbar Y
   int scrollbar_height = canvasText.Height(); //--- Set height
   text_scroll_up_hovered = text_scroll_area_hovered && (local_x >= scrollbar_x && local_x <= scrollbar_x + text_track_width - 1 &&
                            local_y >= scrollbar_y && local_y <= scrollbar_y + text_button_size - 1); //--- Set up hover
   int down_y = scrollbar_y + scrollbar_height - text_button_size; //--- Compute down Y
   text_scroll_down_hovered = text_scroll_area_hovered && (local_x >= scrollbar_x && local_x <= scrollbar_x + text_track_width - 1 &&
                              local_y >= down_y && local_y <= down_y + text_button_size - 1); //--- Set down hover
   int scroll_area_height = scrollbar_height - 2 * text_button_size; //--- Compute area height
   int slider_y = scrollbar_y + text_button_size + (int)(((double)text_scroll_pos / text_max_scroll) * (scroll_area_height - text_slider_height)); //--- Compute slider Y
   text_scroll_slider_hovered = text_scroll_area_hovered && (local_x >= scrollbar_x && local_x <= scrollbar_x + text_track_width - 1 &&
                                local_y >= slider_y && local_y <= slider_y + text_slider_height - 1); //--- Set slider hover
}

//+------------------------------------------------------------------+
//| Get line color                                                   |
//+------------------------------------------------------------------+
color GetLineColor(string lineText) {   // Retrieve line color
   if (StringLen(lineText) == 0 || lineText == " ") return C'25,25,25'; //--- Return default for empty
   if (StringFind(lineText, "mutiiriallan.forex@gmail.com") >= 0) return C'255,100,100'; //--- Return email color
   if (StringFind(lineText, "https://t.me/Forex_Algo_Trader") >= 0) return C'150,100,200'; //--- Return group color
   if (StringFind(lineText, "@ForexAlgo-Trader") >= 0) return C'100,150,255'; //--- Return channel color
   if (StringFind(lineText, "http") >= 0 || StringFind(lineText, "t.me") >= 0) return C'100,150,255'; //--- Return link color
   string start3 = StringSubstr(lineText, 0, 3); //--- Get start substring
   if ((start3 == "1. " || start3 == "2. " || start3 == "3. " || start3 == "4. " || start3 == "5. ") &&
         StringFind(lineText, "Initial Setup Instructions") < 0) return C'255,200,100'; //--- Return numbered color
   return clrWhite;                     //--- Return default white
}

//+------------------------------------------------------------------+
//| Check if heading                                                 |
//+------------------------------------------------------------------+
bool IsHeading(string lineText) {       // Detect heading text
   if (StringLen(lineText) == 0) return false; //--- Return false for empty
   if (StringGetCharacter(lineText, StringLen(lineText) - 1) == ':') return true; //--- Return true for colon end
   if (StringFind(lineText, "Canvas Dashboard Usage Guide") >= 0) return true; //--- Return true for guide
   if (StringFind(lineText, "Key Features") >= 0) return true; //--- Return true for features
   if (StringFind(lineText, "Usage Instructions") >= 0) return true; //--- Return true for instructions
   if (StringFind(lineText, "Important Notes") >= 0) return true; //--- Return true for notes
   if (StringFind(lineText, "Contact Methods") >= 0) return true; //--- Return true for contacts
   if (StringFind(lineText, "NB:") >= 0) return true; //--- Return true for NB
   return false;                        //--- Return false otherwise
}

//+------------------------------------------------------------------+
//| Wrap text                                                        |
//+------------------------------------------------------------------+
void WrapText(const string inputText, const string font, const int fontSize, const int maxWidth, string &wrappedLines[], color &wrappedColors[]) { // Process text wrapping
   ArrayResize(wrappedLines, 0);        //--- Reset lines array
   ArrayResize(wrappedColors, 0);       //--- Reset colors array
   string paragraphs[];                 //--- Declare paragraphs
   int numParagraphs = StringSplit(inputText, '\n', paragraphs); //--- Split by newline
   for (int p = 0; p < numParagraphs; p++) { //--- Loop paragraphs
      string para = paragraphs[p];      //--- Get paragraph
      color paraColor = GetLineColor(para); //--- Get color
      if (StringLen(para) == 0) {       //--- Check empty
         int size = ArraySize(wrappedLines); //--- Get size
         ArrayResize(wrappedLines, size + 1); //--- Resize lines
         wrappedLines[size] = " ";      //--- Add space line
         ArrayResize(wrappedColors, size + 1); //--- Resize colors
         wrappedColors[size] = C'25,25,25'; //--- Set dim color
         continue;                      //--- Skip to next
      }
      string words[];                   //--- Declare words
      int numWords = StringSplit(para, ' ', words); //--- Split by space
      string currentLine = "";          //--- Initialize line
      for (int w = 0; w < numWords; w++) { //--- Loop words
         string testLine = currentLine + (StringLen(currentLine) > 0 ? " " : "") + words[w]; //--- Build test line
         canvasText.FontSet(font, fontSize); //--- Set font
         int textW = canvasText.TextWidth(testLine); //--- Get width
         if (textW <= maxWidth) {       //--- Check fits
            currentLine = testLine;     //--- Update line
         } else {                       //--- Handle overflow
            if (StringLen(currentLine) > 0) { //--- Check current line
               int size = ArraySize(wrappedLines); //--- Get size
               ArrayResize(wrappedLines, size + 1); //--- Resize lines
               wrappedLines[size] = currentLine; //--- Add line
               ArrayResize(wrappedColors, size + 1); //--- Resize colors
               wrappedColors[size] = paraColor; //--- Set color
            }
            currentLine = words[w];    //--- Start new line
         }
      }
      if (StringLen(currentLine) > 0) { //--- Check remaining line
         int size = ArraySize(wrappedLines); //--- Get size
         ArrayResize(wrappedLines, size + 1); //--- Resize lines
         wrappedLines[size] = currentLine; //--- Add line
         ArrayResize(wrappedColors, size + 1); //--- Resize colors
         wrappedColors[size] = paraColor; //--- Set color
      }
   }
}

//+------------------------------------------------------------------+
//| Interpolate color                                                |
//+------------------------------------------------------------------+
color InterpolateColor(color start, color end, double factor) { // Blend colors
   uchar r1 = (uchar)((start >> 16) & 0xFF); //--- Extract start red
   uchar g1 = (uchar)((start >> 8) & 0xFF); //--- Extract start green
   uchar b1 = (uchar)(start & 0xFF);    //--- Extract start blue

   uchar r2 = (uchar)((end >> 16) & 0xFF); //--- Extract end red
   uchar g2 = (uchar)((end >> 8) & 0xFF); //--- Extract end green
   uchar b2 = (uchar)(end & 0xFF);      //--- Extract end blue

   uchar r = (uchar)(r1 + factor * (r2 - r1)); //--- Interpolate red
   uchar g = (uchar)(g1 + factor * (g2 - g1)); //--- Interpolate green
   uchar b = (uchar)(b1 + factor * (b2 - b1)); //--- Interpolate blue

   return (r << 16) | (g << 8) | b;     //--- Return interpolated color
}

//+------------------------------------------------------------------+
//| Darken color                                                     |
//+------------------------------------------------------------------+
color DarkenColor(color colorValue, double factor) { // Adjust color darker
   int blue = int((colorValue & 0xFF) * factor); //--- Compute blue
   int green = int(((colorValue >> 8) & 0xFF) * factor); //--- Compute green
   int red = int(((colorValue >> 16) & 0xFF) * factor); //--- Compute red
   return (color)(blue | (green << 8) | (red << 16)); //--- Return darkened color
}

//+------------------------------------------------------------------+
//| Blend pixels                                                     |
//+------------------------------------------------------------------+
uint BlendPixels(uint bg, uint fg) {    // Combine pixels
   uchar bgA = (uchar)((bg >> 24) & 0xFF); //--- Extract bg alpha
   uchar bgR = (uchar)((bg >> 16) & 0xFF); //--- Extract bg red
   uchar bgG = (uchar)((bg >> 8) & 0xFF); //--- Extract bg green
   uchar bgB = (uchar)(bg & 0xFF);      //--- Extract bg blue

   uchar fgA = (uchar)((fg >> 24) & 0xFF); //--- Extract fg alpha
   uchar fgR = (uchar)((fg >> 16) & 0xFF); //--- Extract fg red
   uchar fgG = (uchar)((fg >> 8) & 0xFF); //--- Extract fg green
   uchar fgB = (uchar)(fg & 0xFF);      //--- Extract fg blue

   if (fgA == 0) return bg;             //--- Return bg if fg transparent
   if (fgA == 255) return fg;           //--- Return fg if opaque

   double alphaFg = fgA / 255.0;        //--- Compute fg alpha
   double alphaBg = 1.0 - alphaFg;      //--- Compute bg alpha

   uchar outR = (uchar)(fgR * alphaFg + bgR * alphaBg); //--- Blend red
   uchar outG = (uchar)(fgG * alphaFg + bgG * alphaBg); //--- Blend green
   uchar outB = (uchar)(fgB * alphaFg + bgB * alphaBg); //--- Blend blue
   uchar outA = (uchar)(fgA + bgA * alphaBg); //--- Blend alpha

   return (((uint)outA) << 24) | (((uint)outR) << 16) | (((uint)outG) << 8) | ((uint)outB); //--- Return blended pixel
}

//+------------------------------------------------------------------+
//| Downsample canvas (average for AA)                               |
//+------------------------------------------------------------------+
void DownsampleCanvas(CCanvas &targetCanvas, CCanvas &highResCanvas) {
   int targetWidth  = targetCanvas.Width();
   int targetHeight = targetCanvas.Height();

   for(int pixelY = 0; pixelY < targetHeight; pixelY++) {
      for(int pixelX = 0; pixelX < targetWidth; pixelX++) {
         double sourceX = pixelX * supersamplingFactor;
         double sourceY = pixelY * supersamplingFactor;

         double sumAlpha = 0, sumRed = 0, sumGreen = 0, sumBlue = 0;
         double weightSum = 0;

         for(int deltaY = 0; deltaY < supersamplingFactor; deltaY++) {
            for(int deltaX = 0; deltaX < supersamplingFactor; deltaX++) {
               int sourcePixelX = (int)(sourceX + deltaX);
               int sourcePixelY = (int)(sourceY + deltaY);

               if(sourcePixelX >= 0 && sourcePixelX < highResCanvas.Width() && sourcePixelY >= 0 && sourcePixelY < highResCanvas.Height()) {
                  uint pixelValue = highResCanvas.PixelGet(sourcePixelX, sourcePixelY);

                  uchar alpha = (uchar)((pixelValue >> 24) & 0xFF);
                  uchar red = (uchar)((pixelValue >> 16) & 0xFF);
                  uchar green = (uchar)((pixelValue >> 8)  & 0xFF);
                  uchar blue = (uchar)(pixelValue         & 0xFF);

                  double weight = 1.0;
                  sumAlpha += alpha * weight;
                  sumRed += red * weight;
                  sumGreen += green * weight;
                  sumBlue += blue * weight;
                  weightSum += weight;
               }
            }
         }

         if(weightSum > 0) {
            uchar finalAlpha = (uchar)(sumAlpha / weightSum);
            uchar finalRed = (uchar)(sumRed / weightSum);
            uchar finalGreen = (uchar)(sumGreen / weightSum);
            uchar finalBlue = (uchar)(sumBlue / weightSum);

            uint finalColor = ((uint)finalAlpha << 24) | ((uint)finalRed << 16) |
                              ((uint)finalGreen << 8)  | (uint)finalBlue;
            targetCanvas.PixelSet(pixelX, pixelY, finalColor);
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Handle chart events                                              |
//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam) { // Process chart events
   if (id == CHARTEVENT_CHART_CHANGE) { //--- Check chart change
      DrawHeaderOnCanvas();             //--- Redraw header
      UpdateGraphOnCanvas();            //--- Update graph
      if (EnableStatsPanel) UpdateStatsOnCanvas(); //--- Update stats
      if (EnableTextPanel) UpdateTextOnCanvas(); //--- Update text
      ChartRedraw();                    //--- Redraw chart
   } else if (id == CHARTEVENT_MOUSE_MOVE) { //--- Check mouse move
      int mouse_x = (int)lparam;        //--- Get mouse X
      int mouse_y = (int)dparam;        //--- Get mouse Y
      int mouse_state = (int)sparam;    //--- Get mouse state

      bool prev_header_hovered = header_hovered; //--- Store prev header hover
      bool prev_min_hovered = minimize_hovered; //--- Store prev minimize hover
      bool prev_close_hovered = close_hovered; //--- Store prev close hover
      bool prev_theme_hovered = theme_hovered; //--- Store prev theme hover
      bool prev_resize_hovered = resize_hovered; //--- Store prev resize hover

      header_hovered = IsMouseOverHeader(mouse_x, mouse_y); //--- Check header hover
      theme_hovered = IsMouseOverTheme(mouse_x, mouse_y); //--- Check theme hover
      minimize_hovered = IsMouseOverMinimize(mouse_x, mouse_y); //--- Check minimize hover
      close_hovered = IsMouseOverClose(mouse_x, mouse_y); //--- Check close hover
      resize_hovered = IsMouseOverResize(mouse_x, mouse_y, hover_mode); //--- Check resize hover

      if (resize_hovered || resizing) { //--- Check resize state
         hover_mouse_local_x = mouse_x - currentCanvasX; //--- Set local X
         hover_mouse_local_y = mouse_y - (currentCanvasY + header_height + gap_y); //--- Set local Y
      }

      bool hover_changed = (prev_header_hovered != header_hovered || prev_min_hovered != minimize_hovered ||
                            prev_close_hovered != close_hovered || prev_theme_hovered != theme_hovered ||
                            prev_resize_hovered != resize_hovered); //--- Check hover change

      if (hover_changed) {              //--- Handle change
         DrawHeaderOnCanvas();          //--- Redraw header
         UpdateGraphOnCanvas();         //--- Update graph
         ChartRedraw();                 //--- Redraw chart
      } else if ((resize_hovered || resizing) && (mouse_x != last_mouse_x || mouse_y != last_mouse_y)) { //--- Check position change
         UpdateGraphOnCanvas();         //--- Update graph
         ChartRedraw();                 //--- Redraw chart
      }

      string header_tooltip = "";       //--- Initialize tooltip
      if (theme_hovered) header_tooltip = "Toggle Theme (Dark/Light)"; //--- Set theme tooltip
      else if (minimize_hovered) header_tooltip = panels_minimized ? "Maximize Panels" : "Minimize Panels"; //--- Set minimize tooltip
      else if (close_hovered) header_tooltip = "Close Dashboard"; //--- Set close tooltip
      ObjectSetString(0, canvasHeaderName, OBJPROP_TOOLTIP, header_tooltip); //--- Apply header tooltip

      string resize_tooltip = "";       //--- Initialize resize tooltip
      if (resize_hovered || resizing) { //--- Check resize state
         ENUM_RESIZE_MODE active_mode = resizing ? resize_mode : hover_mode; //--- Get mode
         switch (active_mode) {         //--- Switch mode
         case BOTTOM:                   //--- Handle bottom
            resize_tooltip = "Resize Bottom"; //--- Set tooltip
            break;                      //--- Exit case
         case RIGHT:                    //--- Handle right
            resize_tooltip = "Resize Right"; //--- Set tooltip
            break;                      //--- Exit case
         case BOTTOM_RIGHT:             //--- Handle bottom-right
            resize_tooltip = "Resize Bottom-Right"; //--- Set tooltip
            break;                      //--- Exit case
         default:                       //--- Handle default
            break;                      //--- Exit case
         }
      }
      ObjectSetString(0, canvasGraphName, OBJPROP_TOOLTIP, resize_tooltip); //--- Apply graph tooltip

      if (EnableTextPanel && !panels_minimized) { //--- Check text enabled
         int text_canvas_x = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_XDISTANCE); //--- Get text X
         int text_canvas_y = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_YDISTANCE); //--- Get text Y
         int text_canvas_w = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_XSIZE); //--- Get text width
         int text_canvas_h = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_YSIZE); //--- Get text height
         bool is_over_text = (mouse_x >= text_canvas_x && mouse_x <= text_canvas_x + text_canvas_w &&
                              mouse_y >= text_canvas_y && mouse_y <= text_canvas_y + text_canvas_h); //--- Check over text
         bool prev_scroll_hovered = text_scroll_area_hovered; //--- Store prev scroll hover
         text_scroll_area_hovered = false; //--- Reset area hover
         if (is_over_text) {            //--- Handle over text
            int local_x = mouse_x - text_canvas_x; //--- Compute local X
            int local_y = mouse_y - text_canvas_y; //--- Compute local Y
            if (local_x >= text_canvas_w - text_track_width) { //--- Check in scroll area
               text_scroll_area_hovered = true; //--- Set area hover
            }
            bool prev_up = text_scroll_up_hovered; //--- Store prev up
            bool prev_down = text_scroll_down_hovered; //--- Store prev down
            bool prev_slider = text_scroll_slider_hovered; //--- Store prev slider
            TextUpdateHoverEffects(local_x, local_y); //--- Update hovers
            if (prev_scroll_hovered != text_scroll_area_hovered || prev_up != text_scroll_up_hovered || prev_down != text_scroll_down_hovered || prev_slider != text_scroll_slider_hovered) { //--- Check change
               UpdateTextOnCanvas();    //--- Update text
               ChartRedraw();           //--- Redraw chart
            }
            text_mouse_in_body = (local_x < text_canvas_w - text_track_width); //--- Set body mouse
         } else {                       //--- Handle not over text
            bool need_redraw = prev_scroll_hovered || text_scroll_up_hovered || text_scroll_down_hovered || text_scroll_slider_hovered; //--- Check need redraw
            if (need_redraw) {          //--- Handle redraw
               text_scroll_area_hovered = false; //--- Reset area
               text_scroll_up_hovered = false; //--- Reset up
               text_scroll_down_hovered = false; //--- Reset down
               text_scroll_slider_hovered = false; //--- Reset slider
               UpdateTextOnCanvas();    //--- Update text
               ChartRedraw();           //--- Redraw chart
            }
            text_mouse_in_body = false; //--- Reset body mouse
         }
         if (text_mouse_in_body != prev_text_mouse_in_body) { //--- Check body change
            ChartSetInteger(0, CHART_MOUSE_SCROLL, !text_mouse_in_body); //--- Set mouse scroll
            prev_text_mouse_in_body = text_mouse_in_body; //--- Update prev
         }
      }

      if (mouse_state == 1 && prev_mouse_state == 0) { //--- Check mouse down
         if (header_hovered) {          //--- Handle header click
            panel_dragging = true;      //--- Set dragging
            panel_drag_x = mouse_x;     //--- Set drag X
            panel_drag_y = mouse_y;     //--- Set drag Y
            panel_start_x = currentCanvasX; //--- Set start X
            panel_start_y = currentCanvasY; //--- Set start Y
            ChartSetInteger(0, CHART_MOUSE_SCROLL, false); //--- Disable scroll
            DrawHeaderOnCanvas();       //--- Redraw header
            ChartRedraw();              //--- Redraw chart
         } else if (theme_hovered) {    //--- Handle theme click
            ToggleTheme();              //--- Toggle theme
         } else if (minimize_hovered) { //--- Handle minimize click
            ToggleMinimize();          //--- Toggle minimize
         } else if (close_hovered) {    //--- Handle close click
            CloseDashboard();           //--- Close dashboard
         } else {                       //--- Handle other clicks
            ENUM_RESIZE_MODE temp_mode = NONE; //--- Set temp mode
            if (!panel_dragging && !resizing && IsMouseOverResize(mouse_x, mouse_y, temp_mode)) { //--- Check resize start
               resizing = true;         //--- Set resizing
               resize_mode = temp_mode; //--- Set mode
               resize_start_x = mouse_x; //--- Set start X
               resize_start_y = mouse_y; //--- Set start Y
               start_width = currentWidth; //--- Set start width
               start_height = currentHeight; //--- Set start height
               ChartSetInteger(0, CHART_MOUSE_SCROLL, false); //--- Disable scroll
               UpdateGraphOnCanvas();   //--- Update graph
               ChartRedraw();           //--- Redraw chart
            }
         }
         if (EnableTextPanel && !panels_minimized && text_scroll_visible && text_scroll_area_hovered) { //--- Check text click
            int text_canvas_x = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_XDISTANCE); //--- Get text X
            int text_canvas_y = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_YDISTANCE); //--- Get text Y
            int local_x = mouse_x - text_canvas_x; //--- Compute local X
            int local_y = mouse_y - text_canvas_y; //--- Compute local Y
            int scrollbar_x = canvasText.Width() - text_track_width; //--- Set scrollbar X
            int scrollbar_y = 0;      //--- Set scrollbar Y
            int scrollbar_height = canvasText.Height(); //--- Set height
            int scroll_area_y = scrollbar_y + text_button_size; //--- Set area Y
            int scroll_area_height = scrollbar_height - 2 * text_button_size; //--- Compute area height
            int slider_y = scrollbar_y + text_button_size + (int)(((double)text_scroll_pos / text_max_scroll) * (scroll_area_height - text_slider_height)); //--- Compute slider Y
            if (local_x >= scrollbar_x && local_x <= scrollbar_x + text_track_width - 1) { //--- Check in scrollbar
               if (local_y >= scrollbar_y && local_y <= scrollbar_y + text_button_size - 1) { //--- Check up button
                  TextScrollUp();       //--- Scroll up
               } else if (local_y >= scrollbar_y + scrollbar_height - text_button_size && local_y <= scrollbar_y + scrollbar_height - 1) { //--- Check down button
                  TextScrollDown();     //--- Scroll down
               } else if (local_y >= scroll_area_y && local_y <= scroll_area_y + scroll_area_height - 1) { //--- Check slider area
                  if (local_y >= slider_y && local_y <= slider_y + text_slider_height - 1) { //--- Check on slider
                     text_movingStateSlider = true; //--- Set moving state
                     text_mlbDownY_Slider = local_y; //--- Set down Y
                     text_mlbDown_YD_Slider = slider_y; //--- Set down YD
                     ChartSetInteger(0, CHART_MOUSE_SCROLL, false); //--- Disable scroll
                  } else {               //--- Handle track click
                     int new_slider_y = local_y - text_slider_height / 2; //--- Compute new Y
                     new_slider_y = MathMax(scroll_area_y, MathMin(new_slider_y, scroll_area_y + scroll_area_height - text_slider_height)); //--- Clamp Y
                     double ratio = (double)(new_slider_y - scroll_area_y) / (scroll_area_height - text_slider_height); //--- Compute ratio
                     text_scroll_pos = (int)MathRound(ratio * text_max_scroll); //--- Set scroll pos
                  }
                  UpdateTextOnCanvas();  //--- Update text
                  ChartRedraw();         //--- Redraw chart
               }
            }
         }
      } else if (panel_dragging && mouse_state == 1) { //--- Check dragging
         int dx = mouse_x - panel_drag_x; //--- Compute delta X
         int dy = mouse_y - panel_drag_y; //--- Compute delta Y
         int new_x = panel_start_x + dx; //--- Compute new X
         int new_y = panel_start_y + dy; //--- Compute new Y

         int chart_w = (int)ChartGetInteger(0, CHART_WIDTH_IN_PIXELS); //--- Get chart width
         int chart_h = (int)ChartGetInteger(0, CHART_HEIGHT_IN_PIXELS); //--- Get chart height
         int full_w = currentWidth + (EnableStatsPanel && !panels_minimized ? PanelGap + currentWidth / 2 : 0); //--- Compute full width
         int full_h = header_height + gap_y + (panels_minimized ? 0 : currentHeight) + (EnableTextPanel && !panels_minimized ? PanelGap + TextPanelHeight : 0); //--- Compute full height
         new_x = MathMax(0, MathMin(chart_w - full_w, new_x)); //--- Clamp new X
         new_y = MathMax(0, MathMin(chart_h - full_h, new_y)); //--- Clamp new Y

         currentCanvasX = new_x;        //--- Update canvas X
         currentCanvasY = new_y;        //--- Update canvas Y

         int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra
         ObjectSetInteger(0, canvasHeaderName, OBJPROP_XDISTANCE, new_x - extra); //--- Set header X
         ObjectSetInteger(0, canvasHeaderName, OBJPROP_YDISTANCE, new_y - extra); //--- Set header Y
         if (!panels_minimized) {       //--- Check not minimized
            ObjectSetInteger(0, canvasGraphName, OBJPROP_XDISTANCE, new_x); //--- Set graph X
            ObjectSetInteger(0, canvasGraphName, OBJPROP_YDISTANCE, new_y + header_height + gap_y); //--- Set graph Y
            if (EnableStatsPanel) {     //--- Check stats enabled
               int stats_x = new_x + currentWidth + PanelGap; //--- Compute stats X
               ObjectSetInteger(0, canvasStatsName, OBJPROP_XDISTANCE, stats_x); //--- Set stats X
               ObjectSetInteger(0, canvasStatsName, OBJPROP_YDISTANCE, new_y + header_height + gap_y); //--- Set stats Y
            }
            if (EnableTextPanel) {      //--- Check text enabled
               int textY = new_y + header_height + gap_y + currentHeight + PanelGap; //--- Compute text Y
               ObjectSetInteger(0, canvasTextName, OBJPROP_XDISTANCE, new_x); //--- Set text X
               ObjectSetInteger(0, canvasTextName, OBJPROP_YDISTANCE, textY); //--- Set text Y
            }
         }
         ChartRedraw();                 //--- Redraw chart
      } else if (resizing && mouse_state == 1) { //--- Check resizing
         int dx = mouse_x - resize_start_x; //--- Compute delta X
         int dy = mouse_y - resize_start_y; //--- Compute delta Y
         int new_width = currentWidth;  //--- Set new width
         int new_height = currentHeight; //--- Set new height
         if (resize_mode == RIGHT || resize_mode == BOTTOM_RIGHT) { //--- Check right modes
            new_width = MathMax(min_width, start_width + dx); //--- Adjust width
         }
         if (resize_mode == BOTTOM || resize_mode == BOTTOM_RIGHT) { //--- Check bottom modes
            new_height = MathMax(min_height, start_height + dy); //--- Adjust height
         }

         if (new_width != currentWidth || new_height != currentHeight) { //--- Check dimension change
            int chart_w = (int)ChartGetInteger(0, CHART_WIDTH_IN_PIXELS); //--- Get chart width
            int chart_h = (int)ChartGetInteger(0, CHART_HEIGHT_IN_PIXELS); //--- Get chart height
            int avail_w = chart_w - currentCanvasX; //--- Compute available width
            int avail_h = chart_h - (currentCanvasY + header_height + gap_y); //--- Compute available height
            new_height = MathMin(new_height, avail_h - (EnableTextPanel ? PanelGap + TextPanelHeight : 0)); //--- Clamp height
            if (EnableStatsPanel) {     //--- Check stats enabled
               double max_w_d = (avail_w - PanelGap) / 1.5; //--- Compute max width double
               int max_w = (int)MathFloor(max_w_d); //--- Floor max width
               new_width = MathMin(new_width, max_w); //--- Clamp width
            } else {                    //--- Handle no stats
               new_width = MathMin(new_width, avail_w); //--- Clamp width
            }

            currentWidth = new_width;  //--- Update width
            currentHeight = new_height; //--- Update height

            if (UseBackground && ArraySize(original_bg_pixels) > 0) { //--- Check background
               ArrayCopy(bg_pixels_graph, original_bg_pixels); //--- Copy graph pixels
               ScaleImage(bg_pixels_graph, (int)orig_w, (int)orig_h, currentWidth, currentHeight); //--- Scale graph
               if (EnableStatsPanel) { //--- Check stats
                  ArrayCopy(bg_pixels_stats, original_bg_pixels); //--- Copy stats pixels
                  ScaleImage(bg_pixels_stats, (int)orig_w, (int)orig_h, currentWidth / 2, currentHeight); //--- Scale stats
               }
            }

            canvasGraph.Resize(currentWidth, currentHeight); //--- Resize graph
            ObjectSetInteger(0, canvasGraphName, OBJPROP_XSIZE, currentWidth); //--- Set graph X size
            ObjectSetInteger(0, canvasGraphName, OBJPROP_YSIZE, currentHeight); //--- Set graph Y size

            if (EnableStatsPanel) {    //--- Check stats
               int stats_width = currentWidth / 2; //--- Compute stats width
               canvasStats.Resize(stats_width, currentHeight); //--- Resize stats
               ObjectSetInteger(0, canvasStatsName, OBJPROP_XSIZE, stats_width); //--- Set stats X size
               ObjectSetInteger(0, canvasStatsName, OBJPROP_YSIZE, currentHeight); //--- Set stats Y size
               canvasStatsHighRes.Resize(stats_width * supersamplingFactor, currentHeight * supersamplingFactor); //--- Resize high-res
               int stats_x = currentCanvasX + currentWidth + PanelGap; //--- Compute stats X
               ObjectSetInteger(0, canvasStatsName, OBJPROP_XDISTANCE, stats_x); //--- Set stats X distance
            }

            int extra = HeaderShadowBlurRadius + HeaderShadowDistance; //--- Compute extra
            int inner_header_width = currentWidth + (EnableStatsPanel ? PanelGap + currentWidth / 2 : 0); //--- Compute inner width
            int header_canvas_width = inner_header_width + 2 * extra; //--- Compute header width
            canvasHeader.Resize(header_canvas_width, header_height + 2 * extra); //--- Resize header
            ObjectSetInteger(0, canvasHeaderName, OBJPROP_XSIZE, header_canvas_width); //--- Set header X size
            ObjectSetInteger(0, canvasHeaderName, OBJPROP_YSIZE, header_height + 2 * extra); //--- Set header Y size
            if (EnableTextPanel) {     //--- Check text
               int text_width = inner_header_width; //--- Set text width
               canvasText.Resize(text_width, TextPanelHeight); //--- Resize text
               ObjectSetInteger(0, canvasTextName, OBJPROP_XSIZE, text_width); //--- Set text X size
               ObjectSetInteger(0, canvasTextName, OBJPROP_YSIZE, TextPanelHeight); //--- Set text Y size
               canvasTextHighRes.Resize(text_width * supersamplingFactor, TextPanelHeight * supersamplingFactor); //--- Resize high-res
               int textY = currentCanvasY + header_height + gap_y + currentHeight + PanelGap; //--- Compute text Y
               ObjectSetInteger(0, canvasTextName, OBJPROP_YDISTANCE, textY); //--- Set text Y distance
            }

            DrawHeaderOnCanvas();      //--- Redraw header
            UpdateGraphOnCanvas();     //--- Update graph
            if (EnableStatsPanel) UpdateStatsOnCanvas(); //--- Update stats
            if (EnableTextPanel) UpdateTextOnCanvas(); //--- Update text
            ChartRedraw();             //--- Redraw chart
         }
      } else if (text_movingStateSlider && mouse_state == 1) { //--- Check slider moving
         int text_canvas_y = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_YDISTANCE); //--- Get text Y
         int local_y = mouse_y - text_canvas_y; //--- Compute local Y
         int delta_y = local_y - text_mlbDownY_Slider; //--- Compute delta Y
         int new_slider_y = text_mlbDown_YD_Slider + delta_y; //--- Compute new Y
         int scrollbar_y = 0;       //--- Set scrollbar Y
         int scrollbar_height = canvasText.Height(); //--- Set height
         int slider_min_y = scrollbar_y + text_button_size; //--- Set min Y
         int slider_max_y = scrollbar_y + scrollbar_height - text_button_size - text_slider_height; //--- Set max Y
         new_slider_y = MathMax(slider_min_y, MathMin(new_slider_y, slider_max_y)); //--- Clamp Y
         double scroll_ratio = (double)(new_slider_y - slider_min_y) / (slider_max_y - slider_min_y); //--- Compute ratio
         int new_scroll_pos = (int)MathRound(scroll_ratio * text_max_scroll); //--- Compute new pos
         if (new_scroll_pos != text_scroll_pos) { //--- Check change
            text_scroll_pos = new_scroll_pos; //--- Update pos
            UpdateTextOnCanvas();      //--- Update text
            ChartRedraw();             //--- Redraw chart
         }
      } else if (mouse_state == 0 && prev_mouse_state == 1) { //--- Check mouse up
         if (panel_dragging) {          //--- Handle drag end
            panel_dragging = false;     //--- Reset dragging
            ChartSetInteger(0, CHART_MOUSE_SCROLL, true); //--- Enable scroll
            DrawHeaderOnCanvas();       //--- Redraw header
            ChartRedraw();              //--- Redraw chart
         }
         if (resizing) {                //--- Handle resize end
            resizing = false;           //--- Reset resizing
            ChartSetInteger(0, CHART_MOUSE_SCROLL, true); //--- Enable scroll
            UpdateGraphOnCanvas();      //--- Update graph
            ChartRedraw();              //--- Redraw chart
         }
         if (text_movingStateSlider) {  //--- Handle slider end
            text_movingStateSlider = false; //--- Reset moving
            ChartSetInteger(0, CHART_MOUSE_SCROLL, true); //--- Enable scroll
            UpdateTextOnCanvas();       //--- Update text
            ChartRedraw();              //--- Redraw chart
         }
      }

      last_mouse_x = mouse_x;           //--- Update last X
      last_mouse_y = mouse_y;           //--- Update last Y
      prev_mouse_state = mouse_state;   //--- Update prev state
   } else if (id == CHARTEVENT_MOUSE_WHEEL) { //--- Check mouse wheel
      int flg_keys = (int)(lparam >> 32); //--- Get keys
      int mx = (int)(short)lparam;      //--- Get X
      int my = (int)(short)(lparam >> 16); //--- Get Y
      int delta = (int)dparam;          //--- Get delta

      if (EnableTextPanel && !panels_minimized && text_scroll_visible) { //--- Check text wheel
         int text_canvas_x = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_XDISTANCE); //--- Get text X
         int text_canvas_y = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_YDISTANCE); //--- Get text Y
         int text_canvas_w = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_XSIZE); //--- Get width
         int text_canvas_h = (int)ObjectGetInteger(0, canvasTextName, OBJPROP_YSIZE); //--- Get height
         bool is_over_text_body = (mx >= text_canvas_x && mx <= text_canvas_x + text_canvas_w - text_track_width &&
                                   my >= text_canvas_y && my <= text_canvas_y + text_canvas_h); //--- Check over body
         if (is_over_text_body) {       //--- Handle over body
            int scroll_step = 20 * smoothness_factor;       //--- Set step
            text_scroll_pos += (delta > 0 ? -scroll_step : scroll_step); //--- Adjust pos
            text_scroll_pos = MathMax(0, MathMin(text_scroll_pos, text_max_scroll)); //--- Clamp pos
            UpdateTextOnCanvas();       //--- Update text
            ChartRedraw();              //--- Redraw chart
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Handle tick events                                               |
//+------------------------------------------------------------------+
void OnTick() {                         // Process tick
   static datetime lastBarTime = 0;     //--- Store last bar time
   datetime currentBarTime = iTime(_Symbol, _Period, 0); //--- Get current time
   if (currentBarTime > lastBarTime) {  //--- Check new bar
      UpdateGraphOnCanvas();            //--- Update graph
      if (EnableStatsPanel) UpdateStatsOnCanvas(); //--- Update stats
      if (EnableTextPanel) UpdateTextOnCanvas(); //--- Update text
      ChartRedraw();                    //--- Redraw chart
      lastBarTime = currentBarTime;     //--- Update last time
   }
}

//+------------------------------------------------------------------+
//| Scale image                                                      |
//+------------------------------------------------------------------+
void ScaleImage(uint &pixels[], int original_width, int original_height, int new_width, int new_height) { // Resize image pixels
   uint scaled_pixels[];                //--- Declare scaled array
   ArrayResize(scaled_pixels, new_width * new_height); //--- Resize scaled
   for (int y = 0; y < new_height; y++) { //--- Loop new Y
      for (int x = 0; x < new_width; x++) { //--- Loop new X
         double original_x = (double)x * original_width / new_width; //--- Compute orig X
         double original_y = (double)y * original_height / new_height; //--- Compute orig Y
         uint pixel = BicubicInterpolate(pixels, original_width, original_height, original_x, original_y); //--- Interpolate pixel
         scaled_pixels[y * new_width + x] = pixel; //--- Set scaled pixel
      }
   }
   ArrayResize(pixels, new_width * new_height); //--- Resize original
   ArrayCopy(pixels, scaled_pixels);    //--- Copy scaled back
}

//+------------------------------------------------------------------+
//| Bicubic interpolate                                              |
//+------------------------------------------------------------------+
uint BicubicInterpolate(uint &pixels[], int width, int height, double x, double y) { // Perform bicubic interpolation
   int x0 = (int)x;                     //--- Set integer X
   int y0 = (int)y;                     //--- Set integer Y
   double fractional_x = x - x0;        //--- Compute fractional X
   double fractional_y = y - y0;        //--- Compute fractional Y
   int x_indices[4], y_indices[4];      //--- Declare indices
   for (int i = -1; i <= 2; i++) {      //--- Loop indices
      x_indices[i + 1] = MathMin(MathMax(x0 + i, 0), width - 1); //--- Clamp X index
      y_indices[i + 1] = MathMin(MathMax(y0 + i, 0), height - 1); //--- Clamp Y index
   }
   uint neighborhood_pixels[16];        //--- Declare neighborhood
   for (int j = 0; j < 4; j++) {        //--- Loop Y neighborhood
      for (int i = 0; i < 4; i++) {     //--- Loop X neighborhood
         neighborhood_pixels[j * 4 + i] = pixels[y_indices[j] * width + x_indices[i]]; //--- Get pixel
      }
   }
   uchar alpha_components[16], red_components[16], green_components[16], blue_components[16]; //--- Declare components
   for (int i = 0; i < 16; i++) {       //--- Loop pixels
      GetArgb(neighborhood_pixels[i], alpha_components[i], red_components[i], green_components[i], blue_components[i]); //--- Extract ARGB
   }
   uchar alpha_out = (uchar)BicubicInterpolateComponent(alpha_components, fractional_x, fractional_y); //--- Interpolate alpha
   uchar red_out = (uchar)BicubicInterpolateComponent(red_components, fractional_x, fractional_y); //--- Interpolate red
   uchar green_out = (uchar)BicubicInterpolateComponent(green_components, fractional_x, fractional_y); //--- Interpolate green
   uchar blue_out = (uchar)BicubicInterpolateComponent(blue_components, fractional_x, fractional_y); //--- Interpolate blue
   return (((uint)alpha_out) << 24) | (((uint)red_out) << 16) | (((uint)green_out) << 8) | ((uint)blue_out); //--- Return interpolated pixel
}

//+------------------------------------------------------------------+
//| Bicubic interpolate component                                    |
//+------------------------------------------------------------------+
double BicubicInterpolateComponent(uchar &components[], double fractional_x, double fractional_y) { // Interpolate single component
   double weights_x[4];                 //--- Declare X weights
   double t = fractional_x;             //--- Set t X
   weights_x[0] = (-0.5 * t * t * t + t * t - 0.5 * t); //--- Compute weight 0
   weights_x[1] = (1.5 * t * t * t - 2.5 * t * t + 1); //--- Compute weight 1
   weights_x[2] = (-1.5 * t * t * t + 2 * t * t + 0.5 * t); //--- Compute weight 2
   weights_x[3] = (0.5 * t * t * t - 0.5 * t * t); //--- Compute weight 3
   double y_values[4];                  //--- Declare Y values
   for (int j = 0; j < 4; j++) {        //--- Loop Y
      y_values[j] = weights_x[0] * components[j * 4 + 0] + weights_x[1] * components[j * 4 + 1] + weights_x[2] * components[j * 4 + 2] + weights_x[3] * components[j * 4 + 3]; //--- Compute Y value
   }
   double weights_y[4];                 //--- Declare Y weights
   t = fractional_y;                    //--- Set t Y
   weights_y[0] = (-0.5 * t * t * t + t * t - 0.5 * t); //--- Compute weight 0
   weights_y[1] = (1.5 * t * t * t - 2.5 * t * t + 1); //--- Compute weight 1
   weights_y[2] = (-1.5 * t * t * t + 2 * t * t + 0.5 * t); //--- Compute weight 2
   weights_y[3] = (0.5 * t * t * t - 0.5 * t * t); //--- Compute weight 3
   double result = weights_y[0] * y_values[0] + weights_y[1] * y_values[1] + weights_y[2] * y_values[2] + weights_y[3] * y_values[3]; //--- Compute result
   return MathMax(0, MathMin(255, result)); //--- Return clamped result
}

//+------------------------------------------------------------------+
//| Get ARGB components                                              |
//+------------------------------------------------------------------+
void GetArgb(uint pixel, uchar &alpha, uchar &red, uchar &green, uchar &blue) { // Extract pixel components
   alpha = (uchar)((pixel >> 24) & 0xFF); //--- Set alpha
   red = (uchar)((pixel >> 16) & 0xFF); //--- Set red
   green = (uchar)((pixel >> 8) & 0xFF); //--- Set green
   blue = (uchar)(pixel & 0xFF);        //--- Set blue
}
//+------------------------------------------------------------------+