//+------------------------------------------------------------------+
//|                                                 Multi Agents.mq5 |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"

#include <Trade/Trade.mqh>
#include "Json.mqh"  // Include our JSON parser

//--- Input parameters
input string   PythonServer = "http://127.0.0.1:5000";  // Python server address
input double   RiskPercent  = 1.0;                      // Risk percentage
input int      SignalCheckInterval = 30;                // Check signal every N seconds
input bool     EnableTrading = true;                    // Enable trading
input string   CommentText   = "Multi-Agent";           // Order comment

//--- Global variables
CTrade trade;
datetime lastSignalCheck = 0;
string currentSignal = "";
double lastSignalPrice = 0;
int magicNumber = 234000;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   // Set magic number for order identification
   trade.SetExpertMagicNumber(magicNumber);
   
   // Set asynchronous mode
   trade.SetAsyncMode(true);
   
   // Print initialization message
   Print("Multi-Agent EA Initialized - Simple JSON Version");
   Print("Python Server: ", PythonServer);
   Print("Risk: ", RiskPercent, "%");
   Print("Magic Number: ", magicNumber);
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   Print("Multi-Agent EA Deinitialized");
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   // Check if it's time to check for new signals
   if(TimeCurrent() - lastSignalCheck >= SignalCheckInterval)
   {
      CheckSignal();
      lastSignalCheck = TimeCurrent();
   }
   
   // Manage existing positions
   ManagePositions();
}

//+------------------------------------------------------------------+
//| Check signal from Python server                                  |
//+------------------------------------------------------------------+
void CheckSignal()
{  
   string url = PythonServer + "/signal";
   
   string payload = "{}";  // empty body is fine
   char data[];
   StringToCharArray(payload, data);
   
   char result[];
   string headers = "Content-Type: application/json\r\n";
   string result_headers;
   
   int res = WebRequest(
      "POST",
      url,
      headers,
      5000,
      data,
      result,
      result_headers
   );
   
   if(res != 200)
   {
      Print("HTTP request failed: ", res);
      return;
   }
   Print("LastError=", GetLastError());
   
   if(res == 200) // HTTP OK
   {
      string jsonStr = CharArrayToString(result);
      Print("Received JSON: ", jsonStr);
      
      // Parse using our simple JSON parser
      string symbol = CJsonSimple::ParseString(jsonStr, "symbol", "");
      int action = CJsonSimple::ParseInteger(jsonStr, "action", 0);
      double confidence = CJsonSimple::ParseNumber(jsonStr, "confidence", 0);
      string reason = CJsonSimple::ParseString(jsonStr, "reason", "");
      double price = CJsonSimple::ParseNumber(jsonStr, "price", 0);
      
      Print("Parsed Signal: Symbol=", symbol, " Action=", action, 
            " Confidence=", confidence, " Reason=", reason);
      
      // Only trade if confidence is high enough
      if(confidence >= 0.5 && EnableTrading)
      {
         // Check if we should trade this symbol
         if(symbol == _Symbol || symbol == "")
         {
            ExecuteSignal(action, confidence, reason, price);
         }
      }
      
      // Update current signal display
      currentSignal = reason;
   }
   else if(res == -1)
   {
      Print("WebRequest failed. Error: ", GetLastError());
      Print("Make sure to add URL to allowed list: ", url);
      
      // To add URL to allowed list in MT5:
      // 1. Go to Tools -> Options -> Expert Advisors
      // 2. Click "Add" under "Allowed URLs"
      // 3. Add: http://localhost:5000
   }
   else
   {
      Print("HTTP request failed: ", res);
   }
}

//+------------------------------------------------------------------+
//| Execute trading signal                                           |
//+------------------------------------------------------------------+
void ExecuteSignal(int action, double confidence, string reason, double signalPrice = 0)
{
   // Get current price
   double ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   double bid = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   double price = (action == 1) ? ask : bid;
   
   // Use signal price if provided and valid
   if(signalPrice > 0 && MathAbs(signalPrice - price) / price < 0.05) // Within 5%
   {
      price = signalPrice;
   }
   
   // Check if price has changed significantly
   if(MathAbs(price - lastSignalPrice) / price < 0.001) // Less than 0.1% change
   {
      Print("Price hasn't changed enough, skipping trade");
      return;
   }
   
   lastSignalPrice = price;
   
   // Close opposite positions first
   if(action == 1) // Buy signal
   {
      CloseSellPositions();
   }
   else if(action == -1) // Sell signal
   {
      CloseBuyPositions();
   }
   
   // Calculate position size
   double volume = CalculateVolume(confidence);
   
   if(volume <= 0)
   {
      Print("Volume too small, skipping trade");
      return;
   }
   
   // Calculate stop loss and take profit based on ATR
   double atr = CalculateATR(14);
   double stopLoss = 0;
   double takeProfit = 0;
   
   if(action == 1) // Buy
   {
      stopLoss = price - (atr * 1.5);
      takeProfit = price + (atr * 3.0);
   }
   else // Sell
   {
      stopLoss = price + (atr * 1.5);
      takeProfit = price - (atr * 3.0);
   }
   
   // Normalize SL/TP to tick size
   stopLoss = NormalizePrice(stopLoss);
   takeProfit = NormalizePrice(takeProfit);
   
   // Place order
   if(action == 1)
   {
      if(trade.Buy(volume, _Symbol, price, stopLoss, takeProfit, reason))
      {
         Print("BUY order placed: ", DoubleToString(volume, 2), " ", _Symbol, 
               " at ", DoubleToString(price, 5));
         Print("SL: ", DoubleToString(stopLoss, 5), 
               " TP: ", DoubleToString(takeProfit, 5));
      }
      else
      {
         Print("Failed to place BUY order: ", trade.ResultRetcodeDescription());
      }
   }
   else if(action == -1)
   {
      if(trade.Sell(volume, _Symbol, price, stopLoss, takeProfit, reason))
      {
         Print("SELL order placed: ", DoubleToString(volume, 2), " ", _Symbol, 
               " at ", DoubleToString(price, 5));
         Print("SL: ", DoubleToString(stopLoss, 5), 
               " TP: ", DoubleToString(takeProfit, 5));
      }
      else
      {
         Print("Failed to place SELL order: ", trade.ResultRetcodeDescription());
      }
   }
}

//+------------------------------------------------------------------+
//| Calculate ATR                                                    |
//+------------------------------------------------------------------+
double CalculateATR(int period)
{
   double atr = 0;
   
   // Try to get ATR from indicator
   int atrHandle = iATR(_Symbol, PERIOD_M5, period);
   if(atrHandle != INVALID_HANDLE)
   {
      double atrArray[];
      if(CopyBuffer(atrHandle, 0, 0, 1, atrArray) > 0)
      {
         atr = atrArray[0];
      }
      IndicatorRelease(atrHandle);
   }
   
   if(atr <= 0)
   {
      // Fallback: use percentage of price
      atr = SymbolInfoDouble(_Symbol, SYMBOL_BID) * 0.002; // 0.2%
   }
   
   return atr;
}

//+------------------------------------------------------------------+
//| Normalize price to tick size                                     |
//+------------------------------------------------------------------+
double NormalizePrice(double price)
{
   double tickSize = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_SIZE);
   if(tickSize > 0)
   {
      price = MathRound(price / tickSize) * tickSize;
   }
   return NormalizeDouble(price, (int)SymbolInfoInteger(_Symbol, SYMBOL_DIGITS));
}

//+------------------------------------------------------------------+
//| Calculate position size based on risk                            |
//+------------------------------------------------------------------+
double CalculateVolume(double confidence)
{
   double accountBalance = AccountInfoDouble(ACCOUNT_BALANCE);
   double riskAmount = accountBalance * (RiskPercent / 100) * confidence;
   
   // Get symbol information
   double tickSize = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_SIZE);
   double tickValue = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_VALUE);
   
   // Calculate stop distance in points
   double atr = CalculateATR(14);
   double stopPoints = atr / tickSize;
   
   if(stopPoints <= 0 || tickValue <= 0)
   {
      Print("Error calculating volume: stopPoints=", stopPoints, " tickValue=", tickValue);
      return 0;
   }
   
   // Calculate volume
   double volume = riskAmount / (stopPoints * tickValue);
   
   // Normalize volume
   double minLot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
   double maxLot = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX);
   double lotStep = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
   
   volume = MathMax(minLot, MathMin(volume, maxLot));
   volume = MathRound(volume / lotStep) * lotStep;
   
   return NormalizeDouble(volume, 2);
}

//+------------------------------------------------------------------+
//| Close all buy positions                                          |
//+------------------------------------------------------------------+
void CloseBuyPositions()
{
   int total = PositionsTotal();
   for(int i = total - 1; i >= 0; i--)
   {
      ulong ticket = PositionGetTicket(i);
      if(PositionSelectByTicket(ticket))
      {
         if(PositionGetString(POSITION_SYMBOL) == _Symbol && 
            PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY &&
            PositionGetInteger(POSITION_MAGIC) == magicNumber)
         {
            trade.PositionClose(ticket);
            Print("Closed BUY position: ", ticket);
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Close all sell positions                                         |
//+------------------------------------------------------------------+
void CloseSellPositions()
{
   int total = PositionsTotal();
   for(int i = total - 1; i >= 0; i--)
   {
      ulong ticket = PositionGetTicket(i);
      if(PositionSelectByTicket(ticket))
      {
         if(PositionGetString(POSITION_SYMBOL) == _Symbol && 
            PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL &&
            PositionGetInteger(POSITION_MAGIC) == magicNumber)
         {
            trade.PositionClose(ticket);
            Print("Closed SELL position: ", ticket);
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Manage existing positions                                        |
//+------------------------------------------------------------------+
void ManagePositions()
{
   // Check for stop loss and take profit hits
   // MT5 handles this automatically, but we can add custom logic here
  
   int total = PositionsTotal();
   if(total > 0)
   {
      Print("Managing ", total, " open positions");
   }
}

//+------------------------------------------------------------------+
//| Chart event handler                                              |
//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
{
   // Display current signal on chart
   static bool labelCreated = false;
   
   if(!labelCreated)
   {
      ObjectCreate(0, "SignalLabel", OBJ_LABEL, 0, 0, 0);
      ObjectSetInteger(0, "SignalLabel", OBJPROP_CORNER, CORNER_RIGHT_UPPER);
      ObjectSetInteger(0, "SignalLabel", OBJPROP_XDISTANCE, 10);
      ObjectSetInteger(0, "SignalLabel", OBJPROP_YDISTANCE, 20);
      ObjectSetInteger(0, "SignalLabel", OBJPROP_FONTSIZE, 10);
      ObjectSetInteger(0, "SignalLabel", OBJPROP_COLOR, clrWhite);
      labelCreated = true;
   }
   
   string text = "Multi-Agent System\n";
   text += "Signal: " + currentSignal + "\n";
   text += "Last Price: " + DoubleToString(lastSignalPrice, 5);
   ObjectSetString(0, "SignalLabel", OBJPROP_TEXT, text);
}
