//+------------------------------------------------------------------+
//|                                                         Json.mqh |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
// JsonSimple.mqh - ULTRA SIMPLE VERSION
// For trading signals only - handles simple key-value pairs

#ifndef JSON_SIMPLE_MQH
#define JSON_SIMPLE_MQH

class CJsonSimple
{
private:
   string m_json;
   
   string ExtractValue(string key)
   {
      int keyPos = StringFind(m_json, "\"" + key + "\"");
      if(keyPos == -1) return "";
      
      int colonPos = StringFind(m_json, ":", keyPos);
      if(colonPos == -1) return "";
      
      // Find the start of the value
      int valueStart = colonPos + 1;
      while(valueStart < StringLen(m_json) && 
            (m_json[valueStart] == ' ' || m_json[valueStart] == '\t' || 
             m_json[valueStart] == '\n' || m_json[valueStart] == '\r'))
      {
         valueStart++;
      }
      
      if(valueStart >= StringLen(m_json)) return "";
      
      char firstChar = m_json[valueStart];
      
      // String value
      if(firstChar == '\"')
      {
         int endQuote = StringFind(m_json, "\"", valueStart + 1);
         if(endQuote == -1) return "";
         return StringSubstr(m_json, valueStart + 1, endQuote - valueStart - 1);
      }
      
      // Number or boolean value
      int valueEnd = valueStart;
      while(valueEnd < StringLen(m_json) && 
            ((m_json[valueEnd] >= '0' && m_json[valueEnd] <= '9') || 
             m_json[valueEnd] == '-' || m_json[valueEnd] == '.' || 
             m_json[valueEnd] == 'e' || m_json[valueEnd] == 'E' ||
             m_json[valueEnd] == 't' || m_json[valueEnd] == 'r' ||
             m_json[valueEnd] == 'u' || m_json[valueEnd] == 'e' ||
             m_json[valueEnd] == 'f' || m_json[valueEnd] == 'a' ||
             m_json[valueEnd] == 'l' || m_json[valueEnd] == 's' ||
             m_json[valueEnd] == 'n' || m_json[valueEnd] == 'u' ||
             m_json[valueEnd] == 'l'))
      {
         valueEnd++;
      }
      
      return StringSubstr(m_json, valueStart, valueEnd - valueStart);
   }
   
public:
   void SetJson(string json)
   {
      m_json = json;
   }
   
   string GetString(string key, string defaultValue = "")
   {
      string value = ExtractValue(key);
      if(value == "") return defaultValue;
      
      // Check if it's actually a string (starts with quote)
      if(StringGetCharacter(value, 0) == '\"')
      {
         return StringSubstr(value, 1, StringLen(value) - 2);
      }
      
      return value;
   }
   
   double GetNumber(string key, double defaultValue = 0)
   {
      string value = ExtractValue(key);
      if(value == "") return defaultValue;
      
      // Check for true/false
      if(value == "true") return 1;
      if(value == "false") return 0;
      
      return StringToDouble(value);
   }
   
   int GetInteger(string key, int defaultValue = 0)
   {
      string value = ExtractValue(key);
      if(value == "") return defaultValue;
      
      // Check for true/false
      if(value == "true") return 1;
      if(value == "false") return 0;
      
      return (int)StringToInteger(value);
   }
   
   bool GetBool(string key, bool defaultValue = false)
   {
      string value = ExtractValue(key);
      if(value == "") return defaultValue;
      
      if(value == "true") return true;
      if(value == "false") return false;
      if(value == "1") return true;
      if(value == "0") return false;
      
      return defaultValue;
   }
   
   // Static helper methods for quick parsing
   static string ParseString(string json, string key, string defaultValue = "")
   {
      CJsonSimple parser;
      parser.SetJson(json);
      return parser.GetString(key, defaultValue);
   }
   
   static double ParseNumber(string json, string key, double defaultValue = 0)
   {
      CJsonSimple parser;
      parser.SetJson(json);
      return parser.GetNumber(key, defaultValue);
   }
   
   static int ParseInteger(string json, string key, int defaultValue = 0)
   {
      CJsonSimple parser;
      parser.SetJson(json);
      return parser.GetInteger(key, defaultValue);
   }
   
   static bool ParseBool(string json, string key, bool defaultValue = false)
   {
      CJsonSimple parser;
      parser.SetJson(json);
      return parser.GetBool(key, defaultValue);
   }
};

#endif // JSON_SIMPLE_MQH
